from .eris_utils import ErisSetupInfo
from adari_core.data_libs.master_raw_cuts import MasterRawCutsReport
from adari_core.data_libs.master_ifu_flat import MasterIfuFlatReport
from adari_core.plots.panel import Panel
from adari_core.plots.text import TextPlot
from adari_core.plots.images import ImagePlot
from adari_core.utils.utils import fetch_kw_or_default
import numpy as np

import os

from . import ErisReportMixin


class ErisIfuFlatReport(ErisReportMixin, MasterIfuFlatReport):

    def __init__(self):
        super().__init__("eris_ifu_flat")
        self.center_size = 200
        self.histmin = 0.8
        self.histmax = 1.2
        self.xcut = 1060
        self.xcut_center = 136

    def parse_sof(self):

        master_flat = None
        bpm_flat = None

        for filename, catg in self.inputs:
            if catg == "MASTER_FLAT":
                master_flat = filename
            if catg == "BPM_FLAT":
                bpm_flat = filename

        file_lists = []
        if master_flat is not None:
            if bpm_flat is not None:
                file_lists.append(
                    {
                        "master_im": master_flat,
                        "bpm_flat": bpm_flat,
                    }
                )

        return file_lists

    def generate_first_panel(self, **kwargs):
        """Create both single and multiple extension panels."""
        panels = {}

        ext = "DATA"
        for i in range(len(self.hdus)):

            hdr = self.hdus[i]["master_im"]["PRIMARY"].header

            p = self.single_panel(
                self.hdus[i]["master_im"][ext],
                master_im_limits=None,
                cutout_limits=None,
            )

            px, py = 0, 0
            vspace = 0.3
            t1 = TextPlot(columns=1, v_space=vspace)
            fname = os.path.basename(str(self.hdus[i]["master_im"].filename()))

            col1 = (
                str(hdr.get("INSTRUME")),
                "EXTNAME: " + ext,
                "PRO CATG: " + str(hdr.get("HIERARCH ESO PRO CATG")),
                "FILE NAME: " + fname,
                "RAW1 NAME: " + str(hdr.get("HIERARCH ESO PRO REC1 RAW1 NAME")),
            )
            t1.add_data(col1)
            p.assign_plot(t1, px, py, xext=2)

            px = px + 2
            t2 = TextPlot(columns=1, v_space=vspace, xext=1)
            col2 = ErisSetupInfo.ifu_flat(self.hdus[i]["master_im"])
            t2.add_data(col2)
            p.assign_plot(t2, px, py, xext=1)

            master_plot = p.retrieve(0, 1)
            master_center = p.retrieve(0, 2)
            cut = p.retrieve(1, 1)
            cut_cent = p.retrieve(1, 2)
            cut.add_data(
                master_plot,
                master_plot.get_data_coord(self.xcut, "x"),
                label="master",
                color="red",
            )
            cut_cent.add_data(
                master_center,
                master_center.get_data_coord(self.xcut_center, "x"),
                label="master",
                color="red",
            )

            master_procatg = str(hdr.get("HIERARCH ESO PRO CATG"))

            addme = {  # "ext": ext,
                "report_name": f"eris_master_flat_multi_panel_{i}",
                "report_description": f"ERIS flat multi panel {master_procatg}",
                "report_tags": [],
            }

            panels[p] = addme
        return panels

    def generate_second_panel(self, **kwargs):

        panels = {}

        p = Panel(2, 2, height_ratios=[1, 4], x_stretch=1.4)

        procatg = fetch_kw_or_default(
            self.hdus[0]["bpm_flat"]["PRIMARY"], "HIERARCH ESO PRO CATG", "UNKNOWN"
        )
        filepath = os.path.basename(self.hdus[0]["bpm_flat"].filename())

        data_bpm = self.hdus[0]["bpm_flat"]["PRIMARY"].data
        dqi = self.hdus[0]["master_im"]["DQI"].data
        procatg_bpm = fetch_kw_or_default(
            self.hdus[0]["bpm_flat"]["PRIMARY"], "HIERARCH ESO PRO CATG", "UNKNOWN"
        )
        procatg_dqi = fetch_kw_or_default(
            self.hdus[0]["master_im"]["PRIMARY"], "HIERARCH ESO PRO CATG", "UNKNOWN"
        )
        vspace = 0.3

        t1 = TextPlot(columns=1, v_space=vspace)
        fname = os.path.basename(str(self.hdus[0]["master_im"].filename()))
        hdr = self.hdus[0]["bpm_flat"]["PRIMARY"].header

        col1 = (
            str(hdr.get("INSTRUME")),
            "EXTNAME: " + str(hdr.get("EXTNAME", "N/A")),
            "PRO CATG: " + str(hdr.get("HIERARCH ESO PRO CATG")),
            "FILE NAME: " + fname,
            "RAW1 NAME: " + str(hdr.get("HIERARCH ESO PRO REC1 RAW1 NAME")),
        )
        t1.add_data(col1)
        p.assign_plot(t1, 0, 0, xext=2)

        t2 = TextPlot(columns=1, v_space=vspace, xext=1)
        col2 = ErisSetupInfo.ifu_flat(self.hdus[0]["bpm_flat"])
        t2.add_data(col2)
        p.assign_plot(t2, 1, 0, xext=1)

        dqi_im = ImagePlot(
            dqi,
            v_clip="val",
            title=procatg_dqi + " DQI",
            v_clip_kwargs={"low": 0, "high": np.max(dqi)},
            interpolation="nearest",
        )

        p.assign_plot(dqi_im, 0, 1)

        bpm_im = ImagePlot(
            data_bpm,
            v_clip="val",
            title=procatg_bpm + " DATA",
            v_clip_kwargs={"low": 0, "high": 1},
            interpolation="nearest",
        )
        p.assign_plot(bpm_im, 1, 1)

        addme = {  # "ext": ext,
            "report_name": f"eris_master_flat_multi_panel_1",
            "report_description": f"ERIS flat multi panel {procatg_bpm}",
            "report_tags": [],
        }

        panels[p] = addme
        return panels

    def generate_panels(self, **kwargs):

        panels = {
            **self.generate_first_panel(),
            **self.generate_second_panel(),
        }

        return panels


rep = ErisIfuFlatReport()
