from .eris_utils import ErisSetupInfo
from adari_core.data_libs.master_raw_cuts import MasterRawCutsReport
from adari_core.data_libs.master_sky_flat import MasterSkyFlatReport
from adari_core.plots.panel import Panel
from adari_core.plots.text import TextPlot
from adari_core.plots.images import ImagePlot

import os

from . import ErisReportMixin


class ErisNixFlatReport(ErisReportMixin, MasterRawCutsReport, MasterSkyFlatReport):

    def __init__(self):
        super().__init__("eris_nix_flat")
        self.files_needed = [
            "MASTER_FLAT_LAMP_LOFREQ",
            "MASTER_FLAT_LAMP_HIFREQ",
            "MASTER_FLAT_SKY_HIFREQ",
            "MASTER_FLAT_SKY_LOFREQ",
            "MASTER_FLAT_TWILIGHT_LOFREQ",
        ]
        self.files_needed_bpm = ["MASTER_BPM_LAMP", "MASTER_BPM_SKY"]
        self.center_size = 200
        self.min = 0.8
        self.max = 1.2
        self.multi = False
        self.twilight = False

    def parse_sof(self):

        master_flat = []
        master_bpm = []
        twilight_im = None

        for filename, catg in self.inputs:
            if catg in self.files_needed:
                master_flat.append(filename)
            if catg == "MASTER_FLAT_LAMP_LOFREQ" or catg == "MASTER_FLAT_SKY_LOFREQ":
                self.multi = True
            if catg == "MASTER_FLAT_TWILIGHT_LOFREQ":
                twilight_im = filename
                self.twilight = True
                self.multi = False

        for filename, catg in self.inputs:
            if catg in self.files_needed_bpm:
                master_bpm.append(filename)

        # Build and return the file name list
        file_lists = []
        if self.twilight == True:
            file_lists.append(
                {
                    "master_im": twilight_im,
                }
            )
        else:
            for i in master_flat:
                file_lists.append(
                    {
                        "master_im": i,
                        "master_bpm": master_bpm[0],
                    }
                )

        return file_lists

    def generate_first_panel(self, **kwargs):
        panels = {}
        ext = "DATA"
        vspace = 0.3
        new_panels = super().generate_panels(
            master_im_ext=ext,
            master_im_clipping="val",
            master_im_n_clipping=[0.9, 1.1],
            master_im_zoom_clipping="val",
            master_im_zoom_n_clipping=[0.9, 1.1],
            interpolation="nearest",
        )

        for i, (panel, panel_descr) in enumerate(new_panels.items()):
            panel_descr["report_description"] = (
                f"ERIS img flat panel - "
                f"{os.path.basename(panel_descr['master_im'])}, "
                f"{panel_descr['master_im_ext']}"
            )

            master_im = self.hdus[i]["master_im"]
            hdr = master_im["PRIMARY"].header
            procatg = str(hdr.get("HIERARCH ESO PRO CATG"))

            # Text Plot
            fname = os.path.basename(str(master_im.filename()))
            t1 = TextPlot(columns=1, v_space=vspace)

            col1 = (
                str(master_im["PRIMARY"].header.get("INSTRUME")),
                "EXTNAME: " + ext,
                "PRO CATG: " + procatg,
                "FILE NAME: " + fname,
                "RAW1 NAME: "
                + str(
                    master_im["PRIMARY"].header.get("HIERARCH ESO PRO REC1 RAW1 NAME")
                ),
            )
            t1.add_data(col1)
            panel.assign_plot(t1, 0, 0, xext=2)

            t2 = TextPlot(columns=1, v_space=vspace, xext=1)
            if self.multi:
                self.metadata = ErisSetupInfo.nix_flat_lamp(master_im)
            else:
                self.metadata = ErisSetupInfo.nix_flat_sky_twi(master_im)
            col2 = self.metadata
            t2.add_data(col2)
            panel.assign_plot(t2, 2, 0, xext=1)

            if "HIFREQ" in procatg:
                im1 = panel.retrieve(0, 1)
                im1.set_vlim(0.8, 1.2)
                im2 = panel.retrieve(0, 2)
                im2.set_vlim(0.8, 1.2)

            collapse = panel.retrieve(3, 2)
            collapse.y_label = "ADU"

        panels = {**panels, **new_panels}
        return panels

    def generate_extra_panel(self, **kwargs):

        panels = {}

        p = Panel(4, 3, height_ratios=[1, 4, 4])

        hdr = self.hdus[0]["master_im"]["PRIMARY"].header
        # Text Plot
        px, py = 0, 0
        # which hdul and ext to use
        vspace = 0.3
        t1 = TextPlot(columns=1, v_space=vspace)
        fname = os.path.basename(str(self.hdus[0]["master_im"].filename()))
        extensions = [hdu.name for hdu in self.hdus[0]["master_im"]]

        col1 = (
            str(hdr.get("INSTRUME")),
            "EXTNAME: " + str(hdr.get("EXTNAME", "N/A")),
            "PRO CATG: " + str(hdr.get("HIERARCH ESO PRO CATG")),
            "FILE NAME: " + fname,
            "RAW1 NAME: " + str(hdr.get("HIERARCH ESO PRO REC1 RAW1 NAME")),
        )
        t1.add_data(col1)
        p.assign_plot(t1, px, py, xext=1)

        px = px + 2
        t2 = TextPlot(columns=1, v_space=vspace, xext=1)
        col2 = ErisSetupInfo.nix_flat_lamp(self.hdus[0]["master_im"])
        t2.add_data(col2)
        p.assign_plot(t2, px, py)

        master_procatg = str(hdr.get("HIERARCH ESO PRO CATG"))
        xpos = [0, 1, 0, 1]
        ypos = [1, 1, 2, 2]

        im1 = ImagePlot(
            self.hdus[0]["master_im"][1].data,
            title=extensions[1],
            v_clip="val",
            v_clip_kwargs={"low": 0.8, "high": 1.2},
            interpolation="nearest",
        )
        p.assign_plot(im1, xpos[0], ypos[0], xext=1)

        im2 = ImagePlot(
            self.hdus[0]["master_im"][2].data,
            title=extensions[2],
            v_clip="val",
            v_clip_kwargs={"low": 0, "high": 1},
            interpolation="nearest",
        )
        p.assign_plot(im2, xpos[1], ypos[1], xext=1)

        im3 = ImagePlot(
            self.hdus[0]["master_im"][3].data,
            title=extensions[3],
            v_clip="val",
            v_clip_kwargs={"low": 0, "high": 1},
        )
        p.assign_plot(im3, xpos[2], ypos[2], xext=1)

        if self.twilight == False:

            im4 = ImagePlot(
                self.hdus[0]["master_im"][4].data,
                title=extensions[4],
                v_clip="val",
                v_clip_kwargs={"low": 0, "high": 0.2},
            )
            p.assign_plot(im4, xpos[3], ypos[3], xext=1)

            master_bpm = self.hdus[0]["master_bpm"]

            px = px + 1
            t3 = TextPlot(columns=1, v_space=vspace, xext=1)
            col3 = ErisSetupInfo.nix_flat_2(
                self.hdus[0]["master_im"], self.hdus[0]["master_bpm"]
            )
            t3.add_data(col3)
            p.assign_plot(t3, px, py)

            bpm_procatg = str(master_bpm["PRIMARY"].header.get("HIERARCH ESO PRO CATG"))
            bpm_im = ImagePlot(
                master_bpm["PRIMARY"].data,
                title=bpm_procatg,
                v_clip="val",
                v_clip_kwargs={"low": 0, "high": 1000},
            )
            p.assign_plot(bpm_im, 2, 1, xext=2, yext=2)

            im4 = p.retrieve(1, 2)
            im4.set_vlim(0, None)

        im2 = p.retrieve(1, 1)
        im2.set_vlim(0, None)

        im3 = p.retrieve(0, 2)
        im3.set_vlim(0, None)

        panels[p] = {
            "master_im": self.hdus[0]["master_im"].filename(),
            "master_im_ext": "DATA",
            "master_procatg": master_procatg,
            "report_name": f"{hdr.get('INSTRUME', 'INSTRUMENT')}_{master_procatg.lower()}_2",
            "report_description": f"nix flat panel -  ({self.hdus[0]['master_im'].filename()}_2)",
            "report_tags": [],
        }

        return panels

    def generate_panels(self, **kwargs):

        panels = {
            **self.generate_first_panel(),
            **self.generate_extra_panel(),
        }

        return panels


rep = ErisNixFlatReport()
