import astropy.io.fits as fits
from adari_core.utils.utils import fetch_kw_or_default


def data_reader(filename):
    hdu = fits.open(filename, mode="readonly")
    rext = "PRIMARY"
    if len(hdu[rext].data.shape) > 2:
        hdu[rext].data = hdu[rext].data[0]
    return hdu


def met1(hdul):
    m = [
        "DET.READ.CURNAME: "
        + str(
            fetch_kw_or_default(
                hdul["PRIMARY"], "HIERARCH ESO DET READ CURNAME", default="N/A"
            )
        ),
        "INS2.NXFW.NAME: "
        + str(
            fetch_kw_or_default(
                hdul["PRIMARY"], "HIERARCH ESO INS2 NXFW NAME", default="N/A"
            )
        ),
        "INS2.NXCW.NAME: "
        + str(
            fetch_kw_or_default(
                hdul["PRIMARY"], "HIERARCH ESO INS2 NXCW NAME", default="N/A"
            )
        ),
    ]
    return m


def met2(hdul):
    m = [
        "INS3.SPXW.NAME: "
        + str(
            fetch_kw_or_default(
                hdul["PRIMARY"], "HIERARCH ESO INS3 SPXW NAME", default="N/A"
            )
        ),
        "INS3.SPGW.NAME: "
        + str(
            fetch_kw_or_default(
                hdul["PRIMARY"], "HIERARCH ESO INS3 SPGW NAME", default="N/A"
            )
        ),
    ]
    return m


def basic(hdul):
    metadata = [
        "DET.SEQ1.DIT: "
        + str(hdul["PRIMARY"].header.get("HIERARCH ESO DET SEQ1 DIT", "N/A")),
        "DET.NDIT: " + str(hdul["PRIMARY"].header.get("HIERARCH ESO DET NDIT", "N/A")),
    ]
    return metadata


class ErisSetupInfo(object):

    @staticmethod
    def ifu_flat(hdul):
        metadata = met2(hdul)
        return metadata

    @staticmethod
    def nix_flat_lamp(hdul):
        m1 = basic(hdul)
        m2 = met1(hdul)
        metadata = m1 + m2
        return metadata

    @staticmethod
    def nix_flat_2(hdul, hdul2):
        metadata = [
            "QC.NUMBER.COLD.PIXELS: "
            + str(
                hdul["PRIMARY"].header.get("HIERARCH ESO QC NUMBER COLD PIXELS", "N/A")
            ),
            "QC.NUMBER.BAD.PIXELS: "
            + str(
                hdul2["PRIMARY"].header.get("HIERARCH ESO QC NUMBER BAD PIXELS", "N/A")
            ),
            "QC.FRACTION.BAD.PIXELS: "
            + str(
                hdul2["PRIMARY"].header.get(
                    "HIERARCH ESO QC FRACTION BAD PIXELS", "N/A"
                )
            ),
        ]
        return metadata

    @staticmethod
    def nix_flat_sky_twi(hdul):
        meta1 = ErisSetupInfo.nix_flat_lamp(hdul)
        meta2 = [
            "INS2.NXAW.NAME: "
            + str(
                fetch_kw_or_default(
                    hdul["PRIMARY"], "HIERARCH ESO INS2 NXAW NAME", default="N/A"
                )
            ),
            "INS2.NXPW.NAME: "
            + str(
                fetch_kw_or_default(
                    hdul["PRIMARY"], "HIERARCH ESO INS2 NXPW NAME", default="N/A"
                )
            ),
        ]
        metadata = meta1 + meta2
        return metadata

    @staticmethod
    def detmon(hdul):
        catg = str(
            fetch_kw_or_default(hdul["PRIMARY"], "HIERARCH ESO PRO CATG", default="N/A")
        )
        if catg == "BP_MAP_NL":
            metadata = met1(hdul)
        else:
            metadata = met2(hdul)
        return metadata

    @staticmethod
    def ifu_dark(hdul):
        metadata = [
            "DET.SEQ1.DIT: "
            + str(hdul["PRIMARY"].header.get("HIERARCH ESO DET SEQ1 DIT", "N/A")),
            "DET.NDIT: "
            + str(hdul["PRIMARY"].header.get("HIERARCH ESO DET NDIT", "N/A")),
            "DET.FRAM.FORMAT: "
            + str(hdul["PRIMARY"].header.get("HIERARCH ESO DET FRAM FORMAT", "N/A")),
        ]
        return metadata

    @staticmethod
    def nix_dark(hdul):
        metadata = ErisSetupInfo.ifu_dark(hdul)
        metadata = metadata + [
            "DET.READ.CURNAME: "
            + str(hdul["PRIMARY"].header.get("HIERARCH ESO DET READ CURNAME", "N/A")),
            "DET.SEQ1.WIN.NAME: "
            + str(hdul["PRIMARY"].header.get("HIERARCH ESO DET SEQ1 WIN NAME", "N/A")),
            "TPL.NEXP: "
            + str(hdul["PRIMARY"].header.get("HIERARCH ESO TPL NEXP", "N/A")),
        ]
        return metadata

    @staticmethod
    def distortion(hdul):
        metadata = met2(hdul)
        return metadata

    @staticmethod
    def wavelength(hdul):
        metadata = met2(hdul)
        return metadata

    @staticmethod
    def ifu_stdstar(hdul):
        metadata = met2(hdul)
        return metadata

    @staticmethod
    def dark(hdul):
        filename = str(fetch_kw_or_default(hdul["PRIMARY"], "ORIGFILE", default="N/A"))
        if "NIX" in filename:
            metadata = ErisSetupInfo.nix_dark(hdul)
        elif filename == "N/A":
            metadata = ErisSetupInfo.nix_dark(hdul)
        else:
            metadata = ErisSetupInfo.ifu_dark(hdul)
        return metadata

    @staticmethod
    def persistence(hdul):
        metadata = [
            "DET.READ.CURNAME: "
            + str(hdul["PRIMARY"].header.get("HIERARCH ESO DET READ CURNAME", "N/A")),
        ]
        return metadata

    @staticmethod
    def pupil_nix(hdul):
        m1 = basic(hdul)
        m2 = met1(hdul)
        m3 = [
            "INS2.NXIS.NAME: "
            + str(
                fetch_kw_or_default(
                    hdul["PRIMARY"], "HIERARCH ESO INS2 NXIS NAME", default="N/A"
                )
            ),
            "INS2.NXAW.NAME: "
            + str(
                fetch_kw_or_default(
                    hdul["PRIMARY"], "HIERARCH ESO INS2 NXAW NAME", default="N/A"
                )
            ),
            "INS2.NXPW.NAME: "
            + str(
                fetch_kw_or_default(
                    hdul["PRIMARY"], "HIERARCH ESO INS2 NXPW NAME", default="N/A"
                )
            ),
        ]

        return m1 + m2 + m3

    @staticmethod
    def pupil_spiffier(hdul):
        metadata = met2(hdul)
        return metadata

    @staticmethod
    def pupil(hdul):
        filename = str(fetch_kw_or_default(hdul["PRIMARY"], "ORIGFILE", default="N/A"))
        if "NIX" in filename:
            metadata = ErisSetupInfo.pupil_nix(hdul)
        else:
            metadata = ErisSetupInfo.pupil_spiffier(hdul)
        return metadata

    @staticmethod
    def linearity(hdul):
        metadata = met2(hdul)
        return metadata

    @staticmethod
    def nix_on_sky(hdul):
        m1 = basic(hdul)
        m2 = met1(hdul)
        m3 = [
            "INS2.NXAW.NAME: "
            + str(
                fetch_kw_or_default(
                    hdul["PRIMARY"], "HIERARCH ESO INS2 NXAW NAME", default="N/A"
                )
            ),
            "INS2.NXPW.NAME: "
            + str(
                fetch_kw_or_default(
                    hdul["PRIMARY"], "HIERARCH ESO INS2 NXPW NAME", default="N/A"
                )
            ),
            "DET.SEQ1.WIN.NAME: "
            + str(
                fetch_kw_or_default(
                    hdul["PRIMARY"], "HIERARCH ESO DET SEQ1 WIN NAME", default="N/A"
                )
            ),
        ]
        return m1 + m2 + m3
