from edps import data_source, match_rules
from edps.generator.time_range import RelativeTimeRange, ONE_DAY, FOUR_DAYS, ONE_WEEK, ONE_MONTH, \
    SAME_NIGHT, UNLIMITED, TWO_WEEKS

from .eris_classification import *

# Convention for Data sources Association rule levels:
# Each data source can have several match function which correspond to different
# quality levels for the selected data. The level is specified as a number that
# follows this convention:
#   level < 0: more restrictive than the calibration plan
#   level = 0 follows the calibration plan
#   level = 1 quality sufficient for QC1 certification
#   level = 2 probably still acceptable quality
#   level = 3 significant risk of bad quality results

# Association and grouping keywords
dit_kwd = [kwd.det_dit, kwd.det_ndit]
dit_read = dit_kwd + [kwd.det_read_curname]
spiffier_keywords = [kwd.seq_arm, kwd.ins3_spxw_name, kwd.ins3_spgw_name]

nix_keywords = [kwd.seq_arm, kwd.ins2_nxfw_name, kwd.ins2_nxcw_name]
nix_sky_keywords = [kwd.ins2_nxaw_name, kwd.ins2_nxpw_name]

dark_spiffier_setup = dit_kwd + [kwd.det_fram_format, kwd.seq_arm]
dark_nix_setup = dark_spiffier_setup + [kwd.seq_arm, kwd.det_read_curname, kwd.det_seq1_win_name]
match_spiffier_dark = dit_read + [kwd.seq_arm]

nix_setup = dit_read + nix_keywords
sky_nix_setup = dit_read + nix_keywords + nix_sky_keywords
match_nix = [kwd.det_read_curname] + nix_keywords

std_nix_jitter_setup = sky_nix_setup + [kwd.det_seq1_win_name]
persistence_spiffier_setup = [kwd.seq_arm, kwd.det_read_curname, kwd.ins3_spxw_name, kwd.ins3_spgw_name, kwd.tpl_id]
persistence_nix_setup = [kwd.seq_arm, kwd.det_read_curname, kwd.tpl_id]
night_calib_nix_setup = nix_keywords + nix_sky_keywords + [kwd.det_read_curname, kwd.det_seq1_win_name, kwd.dpr_type]

# --- RAW types datasources -----------------------------------------------------------------------

# - Linearity
raw_linearity_ifu = (data_source("LINEARITY_IFU")
                     .with_classification_rule(linearity_class)
                     .with_grouping_keywords(spiffier_keywords + [kwd.tpl_start])
                     .with_min_group_size(32)
                     .with_setup_keywords(spiffier_keywords)
                     .with_match_keywords([kwd.seq_arm], time_range=ONE_MONTH, level=0)
                     .with_match_keywords([kwd.seq_arm], time_range=UNLIMITED, level=3)
                     .build())

raw_linearity_nix = (data_source("LINEARITY_NIX")
                     .with_classification_rule(linearity_off_class)
                     .with_classification_rule(linearity_on_class)
                     .with_grouping_keywords([kwd.det_read_curname, kwd.tpl_start])
                     .with_min_group_size(32)
                     .with_setup_keywords(nix_keywords + [kwd.det_read_curname])
                     .with_match_keywords([kwd.seq_arm, kwd.det_read_curname], time_range=ONE_MONTH, level=0)
                     .with_match_keywords([kwd.seq_arm, kwd.det_read_curname], time_range=UNLIMITED, level=3)
                     .build())
# ---

# - Darks ---
raw_dark_ifu = (data_source("DARK_IFU")
                .with_classification_rule(dark_spiffier_class)
                .with_setup_keywords(dark_spiffier_setup)
                .with_grouping_keywords(dark_spiffier_setup + [kwd.tpl_start])
                .with_match_keywords(match_spiffier_dark, time_range=FOUR_DAYS, level=0)
                .with_match_keywords(match_spiffier_dark, time_range=ONE_WEEK, level=1)
                .with_match_keywords(match_spiffier_dark, time_range=UNLIMITED, level=3)
                .build())

# In SPIFFIER mode, the flats and persistence correction uses different rules to associated dark calibrations
match_dark_for_persistence = (match_rules()
                              .with_match_function(rules.is_assoc_spiffier_dark_persistence,
                                                   time_range=FOUR_DAYS, level=0)
                              .with_match_function(rules.is_assoc_spiffier_dark_persistence,
                                                   time_range=ONE_WEEK, level=1)
                              .with_match_function(rules.is_assoc_spiffier_dark_persistence,
                                                   time_range=UNLIMITED, level=3))

match_dark_for_flat = (match_rules()
                       .with_match_function(rules.is_assoc_spiffier_dark_flat, time_range=FOUR_DAYS, level=0)
                       .with_match_function(rules.is_assoc_spiffier_dark_flat, time_range=ONE_WEEK, level=1)
                       .with_match_function(rules.is_assoc_spiffier_dark_flat, time_range=UNLIMITED, level=3))

raw_nix_dark = (data_source("DARK_NIX")
                .with_classification_rule(nix_dark_class)
                .with_min_group_size(3)
                .with_setup_keywords(dark_nix_setup)
                .with_grouping_keywords(dark_nix_setup + [kwd.tpl_start])
                .with_match_function(rules.is_assoc_nix_dark, time_range=ONE_DAY, level=0)
                .with_match_function(rules.is_assoc_nix_dark, time_range=ONE_WEEK, level=1)
                .with_match_function(rules.is_assoc_nix_dark, time_range=UNLIMITED, level=3)
                .build())

# In NIX mode, persistence needs DARKS matching different conditions, depending if the persistence frame is taken
# in fast or slow modes.
match_nix_dark_persistence_fast = (match_rules()
                                   .with_match_function(rules.is_assoc_nix_dark_persistence_fast,
                                                        time_range=FOUR_DAYS, level=0)
                                   .with_match_function(rules.is_assoc_nix_dark_persistence_fast,
                                                        time_range=ONE_WEEK, level=1)
                                   .with_match_function(rules.is_assoc_nix_dark_persistence_fast,
                                                        time_range=UNLIMITED, level=3))

match_nix_dark_persistence_slow = (match_rules()
                                   .with_match_function(rules.is_assoc_nix_dark_persistence_slow,
                                                        time_range=FOUR_DAYS, level=0)
                                   .with_match_function(rules.is_assoc_nix_dark_persistence_slow,
                                                        time_range=ONE_WEEK, level=1)
                                   .with_match_function(rules.is_assoc_nix_dark_persistence_slow,
                                                        time_range=UNLIMITED, level=3))
# ---

raw_persistence_spiffier = (data_source("PERSISTENCE_IFU")
                            .with_classification_rule(persistence_sp_calib_class)
                            .with_grouping_keywords([kwd.tpl_start, kwd.naxis] + persistence_spiffier_setup)
                            .with_setup_keywords(persistence_spiffier_setup)
                            .with_match_keywords([kwd.instrume, kwd.seq_arm], time_range=ONE_DAY, level=0)
                            .build())

raw_persistence_nix = (data_source("PERSISTENCE_NIX")
                       .with_classification_rule(persistence_nix_fast_class)
                       .with_classification_rule(persistence_nix_slow_class)
                       .with_grouping_keywords([kwd.tpl_start, kwd.naxis] + persistence_spiffier_setup)
                       .with_setup_keywords(persistence_nix_setup)
                       .with_match_keywords([kwd.instrume, kwd.seq_arm], time_range=ONE_DAY, level=0)
                       .build())

# - Distortion calibrations (spiffier only) ---
raw_distortion = (data_source("DISTORTION_IFU")
                  .with_classification_rule(ns_slit_class)
                  .with_classification_rule(ns_dark_class)
                  .with_classification_rule(ns_flat_class)
                  .with_classification_rule(ns_wave_class)
                  .with_setup_keywords(spiffier_keywords)
                  .with_grouping_keywords(spiffier_keywords + [kwd.tpl_start])
                  .with_match_keywords(spiffier_keywords, time_range=ONE_MONTH, level=0)
                  .with_match_keywords(spiffier_keywords, time_range=UNLIMITED, level=3)
                  .build())
# ---

# - Raw flats ---
# SPIFFIER raw flats (IFU)
raw_lamp_flat_ifu = (data_source("LAMP_FLAT_IFU")
                     .with_classification_rule(flat_spiffier_class)
                     .with_grouping_keywords(spiffier_keywords + [kwd.tpl_start])
                     .with_setup_keywords(spiffier_keywords)
                     .with_match_keywords(spiffier_keywords, time_range=FOUR_DAYS, level=0)
                     .with_match_keywords(spiffier_keywords, time_range=UNLIMITED, level=3)
                     .build())

# NIX lamp flats (imaging)
raw_lamp_flat_nix = (data_source("LAMP_FLAT_NIX")
                     .with_classification_rule(nix_lamp_flat_on_class)
                     .with_classification_rule(nix_lamp_flat_off_class)
                     .with_setup_keywords(nix_setup)
                     .with_grouping_keywords(nix_setup + [kwd.tpl_start])
                     .with_min_group_size(2)
                     .with_match_keywords(match_nix + [kwd.det_seq1_dit], time_range=ONE_DAY, level=-1)
                     .with_match_keywords(match_nix, time_range=ONE_DAY, level=0)
                     .with_match_keywords(match_nix, time_range=UNLIMITED, level=3)
                     .build())

# NIX twilight flats
raw_twilight_flat = (data_source("TWILIGHT_FLATS_NIX")
                     .with_classification_rule(nix_twilight_flat_class)
                     .with_setup_keywords(sky_nix_setup)
                     .with_grouping_keywords(sky_nix_setup + [kwd.tpl_start])
                     .with_min_group_size(2)
                     .with_match_function(rules.is_assoc_twflat_dit, time_range=TWO_WEEKS, level=-1)
                     .with_match_function(rules.is_assoc_twflat, time_range=TWO_WEEKS, level=0)
                     .with_match_function(rules.is_assoc_twflat, time_range=UNLIMITED, level=3)
                     .build())

# NIX sky flats (imaging, coronagraphy and long-slit)
raw_sky_flat = (data_source("SKY_FLAT_NIX")
                .with_classification_rule(nix_sky_flat_class)
                .with_classification_rule(nix_sky_flat_lss_class)
                .with_classification_rule(flat_coronography_class)
                .with_setup_keywords(sky_nix_setup)
                .with_grouping_keywords(sky_nix_setup + [kwd.tpl_start, kwd.dpr_tech])
                .with_min_group_size(2)
                .build())

# different matching rules for different types of flats and observations:
match_skyflat_long_bra = (match_rules()
                          .with_match_function(rules.match_long_bra_dit, time_range=TWO_WEEKS, level=-1)
                          .with_match_function(rules.match_long_bra, time_range=TWO_WEEKS, level=0)
                          .with_match_function(rules.match_long_bra, time_range=UNLIMITED, level=3))

match_coronography_flat = (match_rules()
                           .with_match_function(rules.match_coronography_flat, time_range=TWO_WEEKS, level=0)
                           .with_match_function(rules.match_coronography_flat, time_range=UNLIMITED, level=3))

match_nix_skyflat = (match_rules()
                     .with_match_function(rules.match_nix_skyflat_dit, time_range=TWO_WEEKS, level=-1)
                     .with_match_function(rules.match_nix_skyflat, time_range=TWO_WEEKS, level=0)
                     .with_match_function(rules.match_nix_skyflat, time_range=UNLIMITED, level=3))

match_skyflat_lss = (match_rules()
                     .with_match_function(rules.match_nix_lss_flat_dit, time_range=TWO_WEEKS, level=-1)
                     .with_match_function(rules.match_nix_lss_flat, time_range=TWO_WEEKS, level=0)
                     .with_match_function(rules.match_nix_lss_flat_nocurname, time_range=UNLIMITED, level=3))

# exposures that are classified short-broad (i.e.g, J, H, and Ks) have the following rules:
match_twflat_short_broad = (match_rules()
                            .with_match_keywords(nix_keywords + [kwd.det_seq1_dit], time_range=TWO_WEEKS, level=-1)
                            .with_match_keywords(nix_keywords, time_range=TWO_WEEKS, level=0)
                            .with_match_keywords(nix_keywords, time_range=UNLIMITED, level=3))

# exposures that are classified short (i.e.g, not in J, H, Ks, "L-Broad", "Mp", "Short-Lp", "Lp", "Br-a" , "Br-a-cont")
# have the following rules:
match_twflat_short = (match_rules()
                      .with_match_function(rules.is_assoc_twflat_short_dit, time_range=TWO_WEEKS, level=-1)
                      .with_match_function(rules.is_assoc_twflat_short, time_range=TWO_WEEKS, level=0)
                      .with_match_function(rules.is_assoc_twflat_short, time_range=UNLIMITED, level=3))

# ---
# Wavelength calibrations (only spiffier)
raw_wave_ifu = (data_source("WAVELENGTH_IFU")
                .with_classification_rule(wave_class)
                .with_grouping_keywords(spiffier_keywords + [kwd.tpl_start])
                .with_setup_keywords(spiffier_keywords)
                .with_match_keywords(spiffier_keywords, time_range=ONE_DAY, level=0)
                .with_match_keywords(spiffier_keywords, time_range=UNLIMITED, level=3)
                .build())

# On sky observations
# Standard star spiffier
TWO_MONTHS = RelativeTimeRange(-60, 60)
raw_telluric_std_ifu = (data_source("TELLURIC_STD_IFU")
                        .with_classification_rule(obj_std_nodding_class)
                        .with_classification_rule(sky_std_nodding_class)
                        .with_grouping_keywords(spiffier_keywords + [kwd.tpl_start])
                        .with_setup_keywords(spiffier_keywords)
                        .with_match_keywords([kwd.tpl_start, kwd.night], time_range=SAME_NIGHT, level=-1)
                        .with_match_keywords([kwd.tpl_start], time_range=ONE_MONTH, level=0)
                        .with_match_keywords([kwd.tpl_start], time_range=TWO_MONTHS, level=2)
                        .build())

raw_flux_std_ifu = (data_source("FLUX_STD_IFU")
                    .with_classification_rule(obj_std_flux_nodding_class)
                    .with_classification_rule(sky_std_flux_nodding_class)
                    .with_grouping_keywords(spiffier_keywords + [kwd.tpl_start])
                    .with_setup_keywords(spiffier_keywords)
                    .with_match_function(rules.is_assoc_flux_standard_same_night, time_range=ONE_DAY, level=-1)
                    .with_match_function(rules.is_assoc_flux_standard, time_range=ONE_MONTH, level=0)
                    .with_match_function(rules.is_assoc_flux_standard, time_range=TWO_MONTHS, level=2)
                    .build())

raw_psf = (data_source("PSF_IFU")
           .with_classification_rule(psf_object_class)
           .with_classification_rule(psf_sky_class)
           .with_setup_keywords(spiffier_keywords)
           .with_grouping_keywords(spiffier_keywords + [kwd.tpl_start])
           .build())

raw_strehl = (data_source("STREHL_IFU")
              .with_classification_rule(strehl_object_class)
              .with_classification_rule(strehl_sky_class)
              .with_setup_keywords(spiffier_keywords)
              .with_grouping_keywords(spiffier_keywords + [kwd.tpl_start])
              .build())

# RAW science SPIFFIER observations
raw_object_spiffier = (data_source("OBJECT_IFU")
                       .with_classification_rule(obj_nodding_class)
                       .with_classification_rule(obj_jitter_class)
                       .with_setup_keywords(spiffier_keywords)
                       .with_grouping_keywords(spiffier_keywords + [kwd.dpr_tech, kwd.tpl_start])
                       .build())

raw_sky_spiffier = (data_source("SKY_IFU")
                    .with_classification_rule(sky_ifu_class)
                    .with_grouping_keywords([kwd.tpl_start])
                    .with_match_keywords(spiffier_keywords + [kwd.dpr_tech, kwd.tpl_start])
                    .build())

# Raw science NIX observations
raw_object_nix_image = (data_source("OBJECT_NIX_IMG")
                        .with_classification_rule(nix_object_jitter_short_class)
                        .with_classification_rule(nix_object_jitter_long_class)
                        .with_setup_keywords(std_nix_jitter_setup)
                        .with_grouping_keywords(std_nix_jitter_setup + [kwd.tpl_start])
                        .with_match_keywords(nix_keywords + ["$combination_nix"])
                        .build())

raw_object_nix_ifu = (data_source("OBJECT_NIX_CUBE")
                      .with_classification_rule(nix_std_jitter_short_cube_class)
                      .with_classification_rule(nix_std_jitter_long_cube_class)
                      .with_classification_rule(nix_object_jitter_short_cube_class)
                      .with_classification_rule(nix_object_jitter_long_cube_class)
                      .with_setup_keywords(std_nix_jitter_setup)
                      .with_grouping_keywords(std_nix_jitter_setup + [kwd.tpl_start])
                      .build())

raw_object_nix_lss = (data_source("OBJECT_NIX_LSS")
                      .with_classification_rule(nix_object_jitter_lss)
                      .with_setup_keywords(std_nix_jitter_setup)
                      .with_grouping_keywords(std_nix_jitter_setup + [kwd.tpl_start])
                      .build())

raw_coronagraphy = (data_source("CORONAGRAPHY_NIX")
                    .with_classification_rule(object_coronagraphy_class)
                    .with_classification_rule(sky_coronagraphy_class)
                    .with_classification_rule(object_fpc_class)
                    .with_classification_rule(sky_fpc_class)
                    .with_setup_keywords(std_nix_jitter_setup + [kwd.dpr_tech])
                    .with_grouping_keywords(std_nix_jitter_setup + [kwd.tpl_start, kwd.dpr_tech])
                    .build())

raw_sam = (data_source("SPARSE_APERTURE_MASK_NIX")
           .with_classification_rule(object_sam_class)
           .with_setup_keywords(std_nix_jitter_setup)
           .with_grouping_keywords(std_nix_jitter_setup + [kwd.tpl_start])
           .build())

# SKY_JITTER
raw_sky_jitter_image = (data_source("SKY_JITTER_NIX_IMAGE")
                        .with_classification_rule(nix_sky_jitter_long_class)
                        .with_classification_rule(nix_sky_jitter_short_class)
                        .with_classification_rule(nix_sky_jitter_long_cube_class)
                        .with_classification_rule(nix_sky_jitter_short_cube_class)
                        .with_classification_rule(nix_sky_astrometry_long_class)
                        .with_classification_rule(nix_sky_astrometry_short_class)
                        .with_grouping_keywords(std_nix_jitter_setup + [kwd.tpl_start, kwd.dpr_type])
                        .with_match_keywords(nix_keywords + [kwd.tpl_start])
                        .build())

raw_sky_jitter_lss = (data_source("SKY_JITTER_NIX_LSS")
                      .with_classification_rule(nix_sky_jitter_lss)
                      .with_classification_rule(nix_sky_std_jitter_lss)
                      .with_grouping_keywords(std_nix_jitter_setup + [kwd.tpl_start])
                      .with_match_keywords(nix_keywords + [kwd.tpl_start])
                      .build())

raw_sky_jitter_cube = (data_source("SKY_JITTER_NIX_CUBE")
                       .with_classification_rule(nix_sky_jitter_short_cube_class)
                       .with_classification_rule(nix_sky_jitter_long_cube_class)
                       .with_classification_rule(nix_sky_astrometry_short_cube_class)
                       .with_classification_rule(nix_sky_astrometry_long_cube_class)
                       .with_match_keywords(nix_keywords + [kwd.tpl_start])
                       .build())

# ON-sky NIX calibrations (Standard stars and astrometry)
raw_on_sky_calib_nix_image = (data_source("ON-SKY_CALIBRATION_NIX_IMG")
                              .with_classification_rule(nix_std_astrometry_short_class)
                              .with_classification_rule(nix_std_astrometry_long_class)
                              .with_classification_rule(nix_object_astrometry_jitter_short_class)
                              .with_classification_rule(nix_object_astrometry_jitter_long_class)
                              .with_classification_rule(nix_std_astrometry_long_class)
                              .with_classification_rule(nix_std_jitter_short_class)
                              .with_classification_rule(nix_std_jitter_long_class)
                              .with_setup_keywords(night_calib_nix_setup)
                              .with_grouping_keywords(std_nix_jitter_setup + [kwd.tpl_start])
                              .with_match_keywords(nix_keywords + [kwd.tpl_start])
                              .build())

raw_on_sky_calib_nix_cube = (data_source("ON-SKY_CALIBRATION_NIX_CUBE")
                             .with_classification_rule(nix_std_astrometry_short_cube_class)
                             .with_classification_rule(nix_std_astrometry_long_cube_class)
                             .with_classification_rule(nix_object_astrometry_jitter_short_cube_class)
                             .with_classification_rule(nix_object_astrometry_jitter_long_cube_class)
                             .with_classification_rule(nix_std_jitter_short_cube_class)
                             .with_classification_rule(nix_std_jitter_long_cube_class)
                             .with_setup_keywords(night_calib_nix_setup)
                             .with_grouping_keywords(std_nix_jitter_setup + [kwd.tpl_start])
                             .with_match_keywords(nix_keywords + [kwd.tpl_start])
                             .build())

raw_on_sky_calib_nix_lss = (data_source("ON-SKY_CALIBRATION_NIX_LSS")
                            .with_classification_rule(nix_std_jitter_lss)
                            .with_setup_keywords(night_calib_nix_setup)
                            .with_grouping_keywords(std_nix_jitter_setup + [kwd.tpl_start])
                            .with_match_keywords(nix_keywords + ["$combination_nix"])
                            .build())
# PUPIL datasources
raw_pupil_spiffier = (data_source("PUPIL_IFU")
                      .with_classification_rule(pupil_spiffier_lamp_class)
                      .with_classification_rule(pupil_spiffier_sky_class)
                      .with_setup_keywords(spiffier_keywords)
                      .with_grouping_keywords([kwd.arcfile])
                      .build())

raw_pupil_nix_background = (data_source("PUPIL_NIX_BACKGROUND")
                            .with_classification_rule(pupil_nix_background_class)
                            .with_match_keywords([kwd.tpl_start, kwd.ins2_nxcw_name])
                            .build())

raw_pupil_nix = (data_source("PUPIL_NIX")
                 .with_classification_rule(pupil_nix_lamp_class)
                 .with_classification_rule(pupil_nix_sky_class)
                 .with_setup_keywords(night_calib_nix_setup)
                 .with_grouping_keywords([kwd.tpl_start, kwd.ins2_nxcw_name, kwd.dpr_type])
                 .build())

nix_pupil_reference = (data_source("PUPIL_REFERENCE")
                       .with_classification_rule(pupil_ref_class)
                       .with_match_keywords([kwd.instrume, kwd.ins2_nxcw_name])
                       .build())

nix_sam_info = (data_source("SAM_INFO")
                .with_classification_rule(sam_info_class)
                .with_match_keywords([kwd.instrume, kwd.ins2_nxcw_name])
                .build())

# ---------------- STATIC DATA SOURCES -----------------
# SPIFFIER
ref_line_arc = (data_source()
                .with_classification_rule(ref_line_arc_class)
                .with_match_function(rules.is_assoc_ref_line_arc)
                .build())

efficiency_windows = (data_source()
                      .with_classification_rule(efficiency_windows_class)
                      .with_match_function(rules.is_assoc_wavelength_windows, time_range=UNLIMITED, level=0)
                      .build())

response_windows = (data_source()
                    .with_classification_rule(response_windows_class)
                    .with_match_function(rules.is_assoc_wavelength_windows, time_range=UNLIMITED, level=0)
                    .build())

extcoeff_table = (data_source()
                  .with_classification_rule(extcoeff_table_class)
                  .with_match_keywords([kwd.instrume], time_range=UNLIMITED, level=0)
                  .build())

flux_std_catalog = (data_source()
                    .with_classification_rule(flux_std_catalog_class)
                    .with_match_keywords([kwd.instrume], time_range=UNLIMITED, level=0)
                    .build())

tell_mod_catalog = (data_source()
                    .with_classification_rule(tell_mod_catalog_class)
                    .with_match_keywords([kwd.instrume], time_range=UNLIMITED, level=0)
                    .build())

fit_areas = (data_source()
             .with_classification_rule(fit_areas_class)
             .with_match_function(rules.is_assoc_wavelength_windows, time_range=UNLIMITED, level=0)
             .build())

quality_areas = (data_source()
                 .with_classification_rule(quality_areas_class)
                 .with_match_function(rules.is_assoc_wavelength_windows, time_range=UNLIMITED, level=0)
                 .build())

resp_fit_points_catalog = (data_source()
                           .with_classification_rule(resp_fit_points_catalog_class)
                           .with_match_keywords([kwd.instrume, kwd.ins3_spgw_id], time_range=UNLIMITED, level=0)
                           .build())

high_abs_regions = (data_source()
                    .with_classification_rule(high_abs_regions_class)
                    .with_match_function(rules.is_assoc_wavelength_windows, time_range=UNLIMITED, level=0)
                    .build())

oh_spectrum = (data_source()
               .with_classification_rule(oh_spec_class)
               .with_match_keywords([kwd.instrume], time_range=UNLIMITED, level=0)
               .build())

atm_ref_corr = (data_source()
                .with_classification_rule(atm_ref_corr_class)
                .with_match_keywords([kwd.instrume, kwd.ins3_spgw_id], time_range=UNLIMITED, level=0)
                .build())

first_wave_fit = (data_source()
                  .with_classification_rule(first_wave_fit_class)
                  .with_match_keywords([kwd.instrume], time_range=UNLIMITED, level=0)
                  .build())

wave_setup = (data_source()
              .with_classification_rule(wave_setup_class)
              .with_match_keywords([kwd.instrume], time_range=UNLIMITED, level=0)
              .build())

# NIX
nix_wcs_matched_catalogue = (data_source()
                             .with_classification_rule(nix_wcs_matched_catalogue_class)
                             .with_grouping_keywords([kwd.instrume])
                             .with_match_keywords([kwd.instrume])
                             .build())

nix_wcs_refine = (data_source()
                  .with_classification_rule(nix_wcs_refine_class)
                  .with_grouping_keywords([kwd.arcfile])
                  .with_match_keywords([kwd.instrume])
                  .build())

nix_phot_data = (data_source("PHOTOMETRIC_DATA")
                 .with_classification_rule(nix_phot_data_class)
                 .with_grouping_keywords([kwd.arcfile])
                 .with_match_keywords([kwd.instrume])
                 .build())

nix_master_startrace = (data_source("STARTRACE")
                        .with_classification_rule(nix_master_startrace_class)
                        .with_match_keywords([kwd.instrume])
                        .build())
