from edps import List, ClassifiedFitsFile, JobParameters, get_parameter, Job

from . import eris_keywords as kwd


def set_detmon_thresholds(job: Job):
    saturation_limit = f'eris.{job.command}.saturation_limit'
    gain_threshold = f'eris.{job.command}.gain_threshold'
    ref_level = f'eris.{job.command}.ref_level'
    autocorr = f'eris.{job.command}.autocorr'
    job.parameters.recipe_parameters[autocorr] = "true"

    curname = job.input_files[0].get_keyword_value(kwd.det_read_curname, None)
    if curname == "FAST_UNCORR":
        job.parameters.recipe_parameters[saturation_limit] = 49000
        job.parameters.recipe_parameters[gain_threshold] = 35000
        job.parameters.recipe_parameters[ref_level] = 35000
    else:
        job.parameters.recipe_parameters[saturation_limit] = 11000
        job.parameters.recipe_parameters[gain_threshold] = 8000
        job.parameters.recipe_parameters[ref_level] = 8000


# Functions to set recipe parameters depending on the properties of input files.
def set_persistence(job: Job):
    speed = job.input_files[0].get_keyword_value(kwd.det_read_curname, None)
    mode = job.input_files[0].get_keyword_value(kwd.seq_arm, None)

    # recipe default
    saturation = 60000
    threshold = 300
    # NIX
    if mode == "NIX":
        if "FAST" in speed:  # FAST IMAGE MODE
            saturation = 49000
            threshold = 37000
        elif "SLOW" in speed:  # SLOW IMAGE MODE
            saturation = 15000
            threshold = 12000
    # SPIFFIER
    elif mode == "SPIFFIER":
        saturation = 60000
        threshold = 45000

    job.parameters.recipe_parameters["eris.eris_persistence_monitor.saturation"] = saturation
    job.parameters.recipe_parameters["eris.eris_persistence_monitor.threshold"] = threshold


def set_cube_collapse(job: Job):
    format = job.input_files[0].get_keyword_value(kwd.det_fram_format, None)
    if format == "cube":
        job.parameters.recipe_parameters["eris.eris_nix_cal_det.collapse_cube"] = 1
    else:
        job.parameters.recipe_parameters["eris.eris_nix_cal_det.collapse_cube"] = 0


# Functions to assign a value to a dynamic parameter based on the
# properties of the main input files

def which_mode(files: List[ClassifiedFitsFile]):
    return files[0].get_keyword_value(kwd.seq_arm, None)


def which_speed(files: List[ClassifiedFitsFile]):
    return files[0].get_keyword_value(kwd.det_read_curname, 'NONE')


def which_target(files: List[ClassifiedFitsFile]):
    dpr_types = [f.get_keyword_value(kwd.dpr_type, None) for f in files]
    return 'STD' if 'STD' in dpr_types else 'SKY_or_OBJECT'


def which_band(files: List[ClassifiedFitsFile]):
    # Determines if the observation was done on the NIX short or long band,
    # and set the return value accordingly.
    band = files[0].get_keyword_value(kwd.ins2_nxfw_name, None)
    if band in ["J", "H", "Ks"]:
        return "short_broad"
    elif band not in ["L-Broad", "Mp", "Short-Lp", "Lp", "Br-a", "Br-a-cont", "J", "H", "Ks"]:
        return "short"
    elif band in ["L-Broad", "Mp", "Lp", "Short-Lp"]:
        return "long"
    elif band in ["Br-a", "Br-a-cont"]:
        return "long_bra"
    else:
        return None


def which_nxcw(files: List[ClassifiedFitsFile]):
    return files[0].get_keyword_value(kwd.ins2_nxcw_name, None)


def which_nxfw(files: List[ClassifiedFitsFile]):
    return files[0].get_keyword_value(kwd.ins2_nxfw_name, None)


def which_pupil(files: List[ClassifiedFitsFile]):
    return files[0].get_keyword_value(kwd.dpr_type, None)


def which_nix_mode(files: List[ClassifiedFitsFile]):
    # Determines if the observation was done on the NIX img or lss or fpc,
    # and set the return value accordingly.
    nix_mode = files[0].get_keyword_value(kwd.ins1_mode, None)
    return nix_mode if nix_mode in ["nixLSS", "nixIMG"] else "None"


def which_frame_format(files: List[ClassifiedFitsFile]):
    return files[0].get_keyword_value(kwd.det_fram_format, None)


def is_pupil_tracking(files: List[ClassifiedFitsFile]):
    dpr_tech = files[0].get_keyword_value(kwd.dpr_tech, "None")
    return True if "PT" in dpr_tech else False


def is_astrometry(files: List[ClassifiedFitsFile]):
    dpr_type = files[0].get_keyword_value(kwd.dpr_type, "None")
    return True if "ASTROMETRY" in dpr_type else False


# Functions that define conditions depending on the
# values of the dynamic parameters
def not_pupil_tracking_and_not_astrometry(params: JobParameters) -> bool:
    return get_parameter(params, "is_pupil_tracking") is False and get_parameter(params, "is_astrometry") is False


def is_short_broad(params: JobParameters) -> bool:
    return get_parameter(params, "band_used") == "short_broad" and get_parameter(params, "nix_mode") != "nixLSS"


def is_short(params: JobParameters) -> bool:
    return get_parameter(params, "band_used") == "short" and get_parameter(params, "nix_mode") != "nixLSS"


def is_short_general(params: JobParameters) -> bool:
    return get_parameter(params, "band_used") in ["short", "short_broad"] and \
        get_parameter(params, "nix_mode") != "nixLSS"


def is_long(params: JobParameters) -> bool:
    return get_parameter(params, "band_used") == "long" and get_parameter(params, "nix_mode") != "nixLSS"


def is_long_bra(params: JobParameters) -> bool:
    return get_parameter(params, "band_used") == "long_bra" and get_parameter(params, "nix_mode") != "nixLSS"


def is_long_general(params: JobParameters) -> bool:
    return get_parameter(params, "band_used") in ["long", "long_bra"] and get_parameter(params, "nix_mode") != "nixLSS"


def is_nix_lss(params: JobParameters) -> bool:
    return get_parameter(params, "nix_mode") == "nixLSS"


def is_nix_img(params: JobParameters) -> bool:
    return get_parameter(params, "nix_mode") == "nixIMG"


def is_nix_slow(params: JobParameters) -> bool:
    return get_parameter(params, "which_mode") == "NIX" and "SLOW" in get_parameter(params, "which_speed")


def is_nix_fast(params: JobParameters) -> bool:
    return get_parameter(params, "which_mode") == "NIX" and "FAST" in get_parameter(params, "which_speed")


def is_spiffier(params: JobParameters) -> bool:
    return get_parameter(params, "which_mode") == "SPIFFIER"


def is_standard_star(params: JobParameters) -> bool:
    return get_parameter(params, "std_or_science") == "STD"


def is_science_observation(params: JobParameters) -> bool:
    return get_parameter(params, "std_or_science") == "SKY_or_OBJECT"


def is_image(params: JobParameters) -> bool:
    return get_parameter(params, "frame_format") == "image"


def is_cube(params: JobParameters) -> bool:
    return get_parameter(params, "frame_format") == "cube"


def is_LM_short(params: JobParameters) -> bool:
    nxcw = get_parameter(params, "NXCW")
    nxfw = get_parameter(params, "NXFW")
    return nxcw == "13mas-LM" and \
        nxfw in ["J", "H", "Ks", "Pa-b", "Fe-II", "H2-cont", "H2-1-0S", "Br-g", "K-peak", "IB-2.42", "IB-2.48"]


def is_JHK_short(params: JobParameters) -> bool:
    nxcw = get_parameter(params, "NXCW")
    nxfw = get_parameter(params, "NXFW")
    return nxcw in ["13mas-JHK", "27mas-JHK"] and \
        nxfw in ["J", "H", "Ks", "Pa-b", "Fe-II", "H2-cont", "H2-1-0S", "Br-g", "K-peak", "IB-2.42", "IB-2.48"]


def force_linearity(params: JobParameters) -> bool:
    return get_parameter(params, "force_linearity") == "TRUE"


def force_dark(params: JobParameters) -> bool:
    return get_parameter(params, "force_dark") == "TRUE"


def is_pupil_nix_lamp(params: JobParameters) -> bool:
    return get_parameter(params, "which_pupil") == "PUPIL,LAMP" and get_parameter(params, "which_mode") == "NIX"
