/* $Id$
 *
 * This file is part of the ERIS Pipeline
 * Copyright (C) 2002,2003 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif
#include <string.h>
#include <gsl/gsl_version.h>
#include <gsl/gsl_multifit.h>
#include <eris_ifu_efficiency_response.h>
#include <eris_ifu_star_index.h>
#include <eris_utils.h>
#include <eris_ifu_dfs.h>
#include <eris_ifu_utils.h>
#include <eris_utils.h>
#include "eris_pfits.h"
#include <math.h>
// TODO: next include is to get a cube saving function. That should be elsewhere
#include <eris_ifu_resample.h>
const hdrl_spectrum1D_wave_scale
global_scale = hdrl_spectrum1D_wave_scale_linear;



#define KEY_VALUE_HPRO_DID                 "PRO-1.15"
#define TELESCOPE_SURFACE                  52.8101279
#define FILE_NAME_SZ 256
#define KEY_NAME_FILT_NAME                 "ESO INS3 SPGW ID"
#define KEY_NAME_FILT_NAME2                 "ESO INS FILT1 NAME"

#define RESPONSE_FILENAME "out_response.fits"
#define DISPERSION_K_DITH  0.000245
#define DISPERSION_H_DITH  0.000195
#define DISPERSION_J_DITH  0.000145
#define DISPERSION_HK_DITH 0.0005
/*
 * Band    lam_c  lam-range  Resol    Disp
 * --------------------------------------
 * J_low    1.25  1.09-1.42  5000     0.00025
 * H_low    1.66  1.45-1.87  5200     0.000319230769231
 * K_low    2.21  1.93-2.48  5600     0.000394642857143
 * ------------------------------------------
 * J_short  1.18  1.10-1.27 10000     0.000118
 * J_middle 1.26  1.18-1.35 10000     0.000126
 * J_long   1.34  1.26-1.43 10000     0.000134
 * --------------------------------------------
 * H_short  1.56  1.46-1.67 10000     0.000156
 * H_middle 1.67  1.56-1.77 10000     0.000167
 * H_long   1.76  1.66-1.87 10000     0.000176
 * --------------------------------------------
 * K_short  2.07  1.93-2.22 11200     0.000207
 * K_middle 2.20  2.06-2.34 11200     0.000220
 * K_long   2.33  2.19-2.47 11200     0.000233
 * -------------------------------------------
 */
#define ERIS_GAIN 2.0
#define SINFONI_GAIN 2.42
#define DISPERSION_J_low    0.00025
#define DISPERSION_J_short  0.000118
#define DISPERSION_J_middle 0.000126
#define DISPERSION_J_long   0.000134

#define DISPERSION_H_low    0.000319230769231
#define DISPERSION_H_short  0.000156
#define DISPERSION_H_middle 0.000167
#define DISPERSION_H_long   0.000176

#define DISPERSION_K_low    0.000394642857143
#define DISPERSION_K_short  0.000207
#define DISPERSION_K_middle 0.000220
#define DISPERSION_K_long   0.000233


typedef struct
{
    double *x, *y, *err;
    int n;
} eris_poly_data;

static double
eris_ifu_get_airmass_mean(cpl_propertylist* plist) {
	double airmass_start = cpl_propertylist_get_double(plist, "ESO TEL AIRM START");
	double airmass_end = cpl_propertylist_get_double(plist, "ESO TEL AIRM END");
	double airmass = 0.5 * (airmass_start + airmass_end);
	return airmass;
}

static cpl_error_code
eris_get_wcs_cube(const cpl_propertylist* plist, double *clambda,
         double *dis, double *cpix, double *cx, double *cy)
{

  *clambda = eris_pfits_get_crval3(plist);
  *dis = eris_pfits_get_cd33(plist);
  *cpix = eris_pfits_get_crpix3(plist);
  *cx = eris_pfits_get_crpix1(plist);
  *cy = eris_pfits_get_crpix2(plist);

  eris_check_error_code("eris_get_wcs_cube");
  return cpl_error_get_code();

}
/*---------------------------------------------------------------------------*/
/**
  @brief    get dispersion at a given band

  @param    band       input wavelength band
  @return   dispersion
 */
/*---------------------------------------------------------------------------*/

static double
sinfo_get_dispersion(const char* band)
{
	double disp = 0.;
	if (strcmp(band,"H+K") == 0) {
		disp = DISPERSION_HK_DITH;
	} else if (strcmp(band,"K") == 0) {
		disp = DISPERSION_K_DITH;
	} else if (strcmp(band,"J") == 0) {
		disp = DISPERSION_J_DITH;
	} else if (strcmp(band,"H") == 0) {
		disp = DISPERSION_H_DITH;
	}
	return disp;

}
/**
   @brief    Set dispersion as a function of band
   @param    band    input band
   @return   dispersion value

 */

static double eris_get_dispersion(const char* band)
{
	cpl_ensure(band != NULL, CPL_ERROR_NULL_INPUT, 0);
	double disp = 0.;
	if (strcmp(band, "J_low") == 0) {
		disp = DISPERSION_J_low;
	} else if (strcmp(band, "J_short") == 0) {
		disp = DISPERSION_J_short;
	} else if (strcmp(band, "J_middle") == 0) {
		disp = DISPERSION_J_middle;
	} else if (strcmp(band, "J_long") == 0) {
		disp = DISPERSION_J_long;
	} else if (strcmp(band, "H_low") == 0) {
		disp = DISPERSION_J_low;
	} else if (strcmp(band, "H_short") == 0) {
		disp = DISPERSION_H_short;
	} else if (strcmp(band, "H_middle") == 0) {
		disp = DISPERSION_H_middle;
	} else if (strcmp(band, "H_long") == 0) {
		disp = DISPERSION_H_long;
	} else if (strcmp(band, "K_low") == 0) {
		disp = DISPERSION_K_low;
	} else if (strcmp(band, "K_short") == 0) {
		disp = DISPERSION_K_short;
	} else if (strcmp(band, "K_middle") == 0) {
		disp = DISPERSION_K_middle;
	} else if (strcmp(band, "K_long") == 0) {
		disp = DISPERSION_K_long;
	}
	eris_check_error_code("eris_get_dispersion");
	return disp;

}
/*---------------------------------------------------------------------------*/
/**
   @brief    Add PRO* keywords to a SINFONI FITS header
   @param    ref_frame   input reference frame
   @param    band        input wavelength range band
   @return   error code

 */
/*---------------------------------------------------------------------------*/

static cpl_error_code
eris_get_band(cpl_frame * ref_frame, char * band)
{
	cpl_msg_debug(cpl_func,"ref_frame: %p filename: %s",
                      (const void*)ref_frame, cpl_frame_get_filename(ref_frame));

	cpl_ensure(ref_frame != NULL, CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT);
	cpl_ensure(band != NULL, CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT);

	char* ref_file = NULL;
	cpl_propertylist* plist = NULL;

	ref_file = cpl_strdup(cpl_frame_get_filename(ref_frame)) ;

	if ((cpl_error_code)((plist = cpl_propertylist_load(ref_file, 0)) == NULL)) {
		cpl_msg_error(cpl_func, "getting header from reference frame %s",
				ref_file);
		cpl_propertylist_delete(plist) ;
		return cpl_error_get_code() ;
	}

	if (cpl_propertylist_has(plist, KEY_NAME_FILT_NAME)) {
		strcpy(band, cpl_propertylist_get_string(plist, KEY_NAME_FILT_NAME));
		/* cpl_msg_info(cpl_func, "%s value is %s", KEY_NAME_FILT_NAME, band); */

	} else {
		cpl_msg_warning(cpl_func, "keyword %s does not exist",
				KEY_NAME_FILT_NAME);
		return cpl_error_get_code() ;
	}

	cpl_free(ref_file);
	cpl_propertylist_delete(plist);
	eris_check_error_code("eris_get_band");
	return cpl_error_get_code() ;
}

/*---------------------------------------------------------------------------*/
/**
   @brief    Add PRO* keywords to a SINFONI FITS header
   @param    ref_frame   input reference frame
   @param    band        input wavelength range band
   @return   error code

 */
/*---------------------------------------------------------------------------*/

static int
sinfo_get_band(cpl_frame * ref_frame, char * band)
{
	cpl_ensure(ref_frame != NULL, CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT);
	cpl_ensure(band != NULL, CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT);

	char* ref_file = NULL;
	cpl_propertylist* plist = NULL;

	ref_file = cpl_strdup(cpl_frame_get_filename(ref_frame)) ;
	if ((cpl_error_code)((plist = cpl_propertylist_load(ref_file, 0)) == NULL)) {
		cpl_msg_error(cpl_func, "getting header from reference frame %s",
				ref_file);
		cpl_propertylist_delete(plist) ;
		return cpl_error_get_code();
	}

	if (cpl_propertylist_has(plist, KEY_NAME_FILT_NAME2)) {
		strcpy(band, cpl_propertylist_get_string(plist, KEY_NAME_FILT_NAME2));
		/* cpl_msg_info(cpl_func, "%s value is %s", KEY_NAME_FILT_NAME, band); */

	} else {
		cpl_msg_warning(cpl_func, "keyword %s does not exist",
				KEY_NAME_FILT_NAME2);
		return cpl_error_get_code();
	}

	cpl_free(ref_file);
	cpl_propertylist_delete(plist);
	return cpl_error_get_code();
}
cpl_boolean eris_can_extract(cpl_frameset* sof) {

	cpl_boolean has_cube_coadd = CPL_TRUE;

	if (
			NULL == cpl_frameset_find(sof, ERIS_IFU_PRO_JITTER_TWK_CUBE_COADD) &&
			NULL == cpl_frameset_find(sof, ERIS_IFU_PRO_JITTER_DAR_CUBE_COADD) &&
			NULL == cpl_frameset_find(sof, ERIS_IFU_PRO_JITTER_OBJ_CUBE_COADD) &&
			NULL == cpl_frameset_find(sof, ERIS_IFU_PRO_JITTER_OBJ_DAR_CUBE_COADD) &&
			NULL == cpl_frameset_find(sof, ERIS_IFU_PRO_JITTER_STD_CUBE_COADD) &&
			NULL == cpl_frameset_find(sof, ERIS_IFU_PRO_JITTER_STD_FLUX_CUBE_COADD) &&
			NULL == cpl_frameset_find(sof, ERIS_IFU_PRO_JITTER_STD_FLUX_CUBE_COADD_NOFLAT) &&
			NULL == cpl_frameset_find(sof, ERIS_IFU_PRO_JITTER_PSF_CUBE_COADD)
					){
		cpl_msg_warning(cpl_func,"Provide %s or %s or %s or %s or %s or %s or %s or %s"
				"to extract..",
				ERIS_IFU_PRO_JITTER_TWK_CUBE_COADD,
				ERIS_IFU_PRO_JITTER_DAR_CUBE_COADD,
				ERIS_IFU_PRO_JITTER_OBJ_CUBE_COADD,
				ERIS_IFU_PRO_JITTER_OBJ_DAR_CUBE_COADD,
				ERIS_IFU_PRO_JITTER_STD_CUBE_COADD,
				ERIS_IFU_PRO_JITTER_STD_FLUX_CUBE_COADD,
				ERIS_IFU_PRO_JITTER_PSF_CUBE_COADD,
				ERIS_IFU_PRO_JITTER_STD_FLUX_CUBE_COADD_NOFLAT
		);
		has_cube_coadd = CPL_FALSE;
	}
	return has_cube_coadd;
}
int eris_can_flux_calibrate(cpl_frameset* sof) {

	int has_extcoeff_table=1;
	int has_response = 1;
  

	if (NULL == cpl_frameset_find(sof, ERIS_IFU_PRO_JITTER_RESPONSE) ){
		cpl_msg_warning(cpl_func,"Provide %s to flux calibrate.",
				ERIS_IFU_PRO_JITTER_RESPONSE);
		has_response = 0;
	}

	if (NULL == cpl_frameset_find(sof, ERIS_IFU_CALIB_EXTCOEFF_TABLE) ){
		cpl_msg_warning(cpl_func,"Provide %s to flux calibrate.",
				ERIS_IFU_CALIB_EXTCOEFF_TABLE);
		has_extcoeff_table = 0;
	}


	eris_check_error_code("eris_can_flux_calibrate");
	return  has_response *
			has_extcoeff_table;

}


int eris_can_compute_efficiency(cpl_frameset* sof) {

	int has_std_star_spectra=1;
	int has_extcoeff_table=1;
	int has_response = 1;

	if (NULL == cpl_frameset_find(sof, ERIS_IFU_PRO_JITTER_SPECTRUM_NOFLAT)){
		cpl_msg_warning(cpl_func,"Provide %s to compute efficiency.",
				ERIS_IFU_PRO_JITTER_SPECTRUM_NOFLAT);
		has_std_star_spectra=0;
	}

	if (NULL == cpl_frameset_find(sof, ERIS_IFU_CALIB_EFFICIENCY_WINDOWS) ){
		cpl_msg_warning(cpl_func,"Provide %s to compute efficiency.",
				ERIS_IFU_CALIB_EFFICIENCY_WINDOWS);
		has_response = 0;
	}

	if (NULL == cpl_frameset_find(sof, ERIS_IFU_CALIB_EXTCOEFF_TABLE) ){
		cpl_msg_warning(cpl_func,"Provide %s to compute efficiency.",
				ERIS_IFU_CALIB_EXTCOEFF_TABLE);
		has_extcoeff_table = 0;
	}


	eris_check_error_code("eris_can_compute_efficiency");
	return  has_std_star_spectra *
			has_response *
			has_extcoeff_table;

}


int eris_can_compute_response(cpl_frameset* sof) {

	int has_std_star_spectra=1;
	int has_flux_std_catalog=1;
	int has_tel_mod_catalog=1;
	int has_resp_fit_points_cat=1;
	int has_extcoeff_table=1;
	int has_high_abs_regions=1;
	int has_quality_areas=1;
	int has_fit_areas=1;
	int has_resp_windows=1;

	if (NULL == cpl_frameset_find(sof, ERIS_IFU_PRO_JITTER_SPECTRUM) ){
		cpl_msg_warning(cpl_func,"Provide %s to compute response.",
				ERIS_IFU_PRO_JITTER_SPECTRUM);
		has_std_star_spectra=0;
	}

	if (NULL == cpl_frameset_find(sof, ERIS_IFU_CALIB_FLUX_STD_CATALOG) ){
		cpl_msg_warning(cpl_func,"Provide %s to compute response.",
				ERIS_IFU_CALIB_FLUX_STD_CATALOG);
		has_flux_std_catalog=0;
	}
	if (NULL == cpl_frameset_find(sof, ERIS_IFU_CALIB_TELL_MOD_CATALOG) ){
		cpl_msg_warning(cpl_func,"Provide %s to compute response.",
				ERIS_IFU_CALIB_TELL_MOD_CATALOG);
		has_tel_mod_catalog=0;
	}
	if (NULL == cpl_frameset_find(sof, ERIS_IFU_CALIB_RESP_FIT_POINTS_CATALOG) ){
		cpl_msg_warning(cpl_func,"Provide %s to compute response.",
				ERIS_IFU_CALIB_RESP_FIT_POINTS_CATALOG);
		has_resp_fit_points_cat=0;
	}

	if (NULL == cpl_frameset_find(sof, ERIS_IFU_CALIB_EXTCOEFF_TABLE) ){
		cpl_msg_warning(cpl_func,"Provide %s to compute response.",
				ERIS_IFU_CALIB_EXTCOEFF_TABLE);
		has_extcoeff_table=0;
	}

	/*
	if (NULL == cpl_frameset_find(sof, ERIS_IFU_CALIB_HIGH_ABS_REGIONS) ){
		cpl_msg_warning(cpl_func,"Frame %s not provided to compute response.",
				ERIS_IFU_CALIB_HIGH_ABS_REGIONS);
		has_high_abs_regions=1;
		//return 0;
	}
	*/

	if (NULL == cpl_frameset_find(sof, ERIS_IFU_CALIB_QUALITY_AREAS) ){
		cpl_msg_warning(cpl_func,"Provide %s to compute response.",
				ERIS_IFU_CALIB_QUALITY_AREAS);
		has_quality_areas=0;
	}

	if (NULL == cpl_frameset_find(sof, ERIS_IFU_CALIB_FIT_AREAS) ){
		cpl_msg_warning(cpl_func,"Provide %s to compute response.",
				ERIS_IFU_CALIB_FIT_AREAS);
		has_fit_areas=0;
	}

	if (NULL == cpl_frameset_find(sof, ERIS_IFU_CALIB_RESPONSE_WINDOWS) ){
		cpl_msg_warning(cpl_func,"Provide %s to compute response.",
				ERIS_IFU_CALIB_RESPONSE_WINDOWS);
		has_resp_windows=0;
	}
	eris_check_error_code("eris_can_compute_response");
	return has_std_star_spectra *
			has_flux_std_catalog *
			has_tel_mod_catalog *
			has_resp_fit_points_cat *
			has_extcoeff_table *
			has_high_abs_regions *
			has_quality_areas *
			has_fit_areas *
			has_resp_windows;

}



static cpl_error_code
eris_eff_qc(cpl_table* tot_eff, cpl_frame* frm_eff_wind_qc, cpl_table** qclog_tbl )
{
    const char* ew_name=cpl_frame_get_filename(frm_eff_wind_qc);
    cpl_table* ew_tab=cpl_table_load(ew_name,1,0);
    int nrow = cpl_table_get_nrow(ew_tab);

    float* pwmin = cpl_table_get_data_float(ew_tab,"WMIN");
    float* pwmax = cpl_table_get_data_float(ew_tab,"WMAX");
    cpl_table* tmp_eff=NULL;
    cpl_table* sto_eff=NULL;
    double wmin;
    double wmax;
    double eff_avg;
    double eff_med;
    double eff_min;
    double eff_max;
    double eff_std;
    char* key_name;

    int nrow_tmp=0;
    sto_eff = cpl_table_new(0);
    cpl_table_copy_structure(sto_eff, tot_eff);

    int ncontributes = 0;
    int nrow_sto = 0;

    for(int i=0;i<nrow;i++) {
        //cpl_msg_info(cpl_func, "i=%d wmin=%g wmax=%g",i,pwmin[i],pwmax[i]);
        wmin=pwmin[i];
        wmax=pwmax[i];

        cpl_table_and_selected_double(tot_eff,"WAVE",CPL_NOT_LESS_THAN,wmin);
        cpl_table_and_selected_double(tot_eff,"WAVE",CPL_NOT_GREATER_THAN,wmax);
        tmp_eff=cpl_table_extract_selected(tot_eff);
        nrow_tmp = cpl_table_get_nrow(tmp_eff);
        if(nrow_tmp>0) {
        	ncontributes++;
        	cpl_table_insert(sto_eff,tmp_eff, nrow_sto);
        	nrow_sto = cpl_table_get_nrow(sto_eff);
        	eff_avg=cpl_table_get_column_mean(tmp_eff,"EFF");
        	eff_med=cpl_table_get_column_median(tmp_eff,"EFF");
        	eff_min=cpl_table_get_column_min(tmp_eff,"EFF");
        	eff_max=cpl_table_get_column_max(tmp_eff,"EFF");
        	eff_std=cpl_table_get_column_stdev(tmp_eff,"EFF");


        	key_name = cpl_sprintf("QC EFF WIN%d WLMIN",i);
        	eris_qclog_add_double(*qclog_tbl, key_name, wmin,
        			"[um] Min window wavelength for eff comp");
            cpl_free(key_name);

            key_name = cpl_sprintf("QC EFF WIN%d WLMAX",i);
        	eris_qclog_add_double(*qclog_tbl, key_name, wmax,
        			"[um] Max window wavelength for eff comp");
        	cpl_free(key_name);

        	key_name = cpl_sprintf("QC EFF WIN%d MEAN",i);
        	eris_qclog_add_double(*qclog_tbl, key_name, eff_avg,
        			"Mean efficiency on window");
        	cpl_free(key_name);

        	key_name = cpl_sprintf("QC EFF WIN%d MEDIAN",i);
        	eris_qclog_add_double(*qclog_tbl, key_name, eff_med,
        			"Median efficiency on window");
        	cpl_free(key_name);

        	key_name = cpl_sprintf("QC EFF WIN%d MIN",i);
        	eris_qclog_add_double(*qclog_tbl, key_name, eff_min,
        			"Min efficiency on window");
        	cpl_free(key_name);

        	key_name = cpl_sprintf("QC EFF WIN%d MAX",i);
            eris_qclog_add_double(*qclog_tbl, key_name, eff_max,
        			"Max efficiency on window");
            cpl_free(key_name);

            key_name = cpl_sprintf("QC EFF WIN%d RMS",i);
        	eris_qclog_add_double(*qclog_tbl, key_name, eff_std,
        			"RMS efficiency on window");
        	cpl_free(key_name);
        }
        cpl_table_select_all(tot_eff);
        cpl_table_delete(tmp_eff);

    }
    //cpl_table_save (ew_tab, NULL, NULL, "ew_tbl.fits", CPL_IO_CREATE);
    wmin=cpl_table_get_column_min(sto_eff,"WAVE");
    wmax=cpl_table_get_column_max(sto_eff,"WAVE");

    eff_avg=cpl_table_get_column_mean(sto_eff,"EFF");
    eff_med=cpl_table_get_column_median(sto_eff,"EFF");
    eff_min=cpl_table_get_column_min(sto_eff,"EFF");
    eff_max=cpl_table_get_column_max(sto_eff,"EFF");
    eff_std=cpl_table_get_column_stdev(sto_eff,"EFF");


    key_name = cpl_sprintf("QC EFF NWIN");
        eris_qclog_add_int(*qclog_tbl, key_name, ncontributes,
                                     "Number of windows used for eff comp");
        cpl_free(key_name);
    key_name = cpl_sprintf("QC EFF WLMIN");
    eris_qclog_add_double(*qclog_tbl, key_name, wmin,
                                 "[um] Min wavelength for eff comp");
    cpl_free(key_name);

    key_name = cpl_sprintf("QC EFF WLMAX");
    eris_qclog_add_double(*qclog_tbl, key_name, wmax,
                                  "[um] Max wavelength for eff comp");
    cpl_free(key_name);

    key_name = cpl_sprintf("QC EFF MEAN");
    eris_qclog_add_double(*qclog_tbl, key_name, eff_avg,
                           "Mean efficiency");
    cpl_free(key_name);

    key_name = cpl_sprintf("QC EFF MEDIAN");
    eris_qclog_add_double(*qclog_tbl, key_name, eff_med,
                           "Median efficiency");
    cpl_free(key_name);

    key_name = cpl_sprintf("QC EFF MIN");
    eris_qclog_add_double(*qclog_tbl, key_name, eff_min,
                           "Min efficiency");
    cpl_free(key_name);

    key_name = cpl_sprintf("QC EFF MAX");
    eris_qclog_add_double(*qclog_tbl, key_name, eff_max,
                           "Max efficiency");
    cpl_free(key_name);

    key_name = cpl_sprintf("QC EFF RMS");
    eris_qclog_add_double(*qclog_tbl, key_name, eff_std,
                           "RMS efficiency");

    //cpl_table_save (*qclog_tbl, NULL, NULL, "qclog_tbl.fits", CPL_IO_CREATE);

    cpl_free(key_name);
    cpl_table_delete(ew_tab);
    cpl_table_delete(sto_eff);
    eris_check_error_code("eris_eff_qc");
    return cpl_error_get_code();
}

static cpl_table*
eris_ifu_table_from_sdp_to_normal_format(cpl_frame** frm_sci)
{
	cpl_table* sci_tab = NULL;
	const cpl_array* array_wave = NULL;
	const char* name_sci = NULL;
	cpl_propertylist* plist = NULL;
	name_sci = cpl_frame_get_filename(*frm_sci);
	plist = cpl_propertylist_load(name_sci,0);
	sci_tab = cpl_table_load(name_sci, 1, 0);
	if(NULL != (array_wave = cpl_table_get_array(sci_tab, "WAVE", 0))) {
		/* convert from SDP table data format to normal table format */
		const cpl_array* array_flux = NULL;
		const cpl_array* array_err = NULL;
		cpl_size nrows = cpl_array_get_size(array_wave);

		cpl_table* spc_tab = cpl_table_new(nrows);
		cpl_table_new_column(spc_tab, "WAVE", CPL_TYPE_DOUBLE);
		cpl_table_new_column(spc_tab, "FLUX", CPL_TYPE_DOUBLE);
		cpl_table_new_column(spc_tab, "ERR", CPL_TYPE_DOUBLE);
		array_flux = cpl_table_get_array(sci_tab, "TOT_FLUX", 0);
		array_err = cpl_table_get_array(sci_tab, "ERR", 0);

		cpl_table_copy_data_double(spc_tab, "WAVE",
				cpl_array_get_data_double_const(array_wave));
		cpl_table_copy_data_double(spc_tab, "FLUX",
				cpl_array_get_data_double_const(array_flux));
		cpl_table_copy_data_double(spc_tab, "ERR",
				cpl_array_get_data_double_const(array_err));

		cpl_table_save(spc_tab,plist,NULL,"spc_tab.fits", CPL_IO_CREATE);
        cpl_table_delete(spc_tab);
		cpl_table_delete(sci_tab);

		sci_tab = cpl_table_load("spc_tab.fits", 1, 0);
		cpl_frame_set_filename(*frm_sci, "spc_tab.fits");

	}
	cpl_propertylist_delete(plist);
	eris_check_error_code("eris_ifu_table_from_sdp_to_normal_format");
	return sci_tab;
}

static void
eris_remove_table_normal_format(void) {
	char* cmd = cpl_sprintf("rm spc_tab.fits");
	int status = system(cmd);
	if(status == -1) {
		cpl_msg_warning(cpl_func,"call to system failed");
	}
	cpl_free(cmd);
	return;
}
/*---------------------------------------------------------------------------*/
/**
   @brief    compute efficiency
   @param    frm_sci  input science frame
   @param    frm_cat  input reference std star catalog
   @param    frm_atmext input atmospheric extinction table
   @return   cpl_table* output table with computed efficiency

   DFS only. See the DICB dictionaries to have details on the keywords.
   @note: currently support both SINFONI and ERIS data to enable comparisons
   on efficiency with ETC. Save extra products for checks. gain is set to
   compensate bug on cubes flux calibration and the fact ERIS input comes from
   1d spectrum from python script where gain is already applied
 */
/*---------------------------------------------------------------------------*/

cpl_table*
eris_efficiency_compute(cpl_frame* frm_sci, cpl_frame* frm_cat,
		cpl_frame* frm_atmext, cpl_frameset* frameset,
		const cpl_parameterlist* parlist, const char* pipefile_prefix)

{
	cpl_ensure(frm_sci != NULL, CPL_ERROR_NULL_INPUT, NULL);
	cpl_ensure(frm_cat != NULL, CPL_ERROR_NULL_INPUT, NULL);
	cpl_ensure(frm_atmext != NULL, CPL_ERROR_NULL_INPUT, NULL);
	cpl_frame* frm_eff_wind_qc = cpl_frameset_find(frameset, ERIS_IFU_CALIB_EFFICIENCY_WINDOWS);
	cpl_propertylist* plist = NULL;

	cpl_table* tbl_eff = NULL;
	cpl_table* tbl_ref = NULL;
	cpl_table* tbl_atmext = NULL;
	cpl_table* sci_tab = NULL;



	double dRA = 0;
	double dDEC = 0;
	double gain = 0;
	//double aimprim = 0;
	double dEpsilon = 0.1;
	//double um2AA = 1.e4;
	double um2nm = 1000.;
	double nm2um = 0.001;
	double m2tocm2 = 1.e4;

	/* int nrow=0; */

	cpl_boolean is_sinfoni = CPL_FALSE;

	const char* name_sci = NULL;
	const char* name_atm = NULL;

	name_sci = cpl_frame_get_filename(frm_sci);

	plist = cpl_propertylist_load(name_sci, 0);
	const char* instrume = cpl_propertylist_get_string(plist, "INSTRUME");
	//cpl_msg_warning(cpl_func, "fname: %s instrume: %s", name_sci, instrume);
	if(strcmp(instrume, "SINFONI") == 0) {
		is_sinfoni = CPL_TRUE;
	}
	//sci_tab = cpl_table_load(name_sci, 1, 0);

	dRA = cpl_propertylist_get_double(plist, "RA");
	dDEC = cpl_propertylist_get_double(plist, "DEC");
    cpl_frame* frm_sci_dup = cpl_frame_duplicate(frm_sci);
	sci_tab = eris_ifu_table_from_sdp_to_normal_format(&frm_sci_dup);
	cpl_frame_delete(frm_sci_dup);
	double airmass = eris_ifu_get_airmass_mean(plist);
	if(is_sinfoni) {
		gain = SINFONI_GAIN;
	} else {
		gain = ERIS_GAIN; /* value to be adopted if input spectrum is in ADU */
		/* gain = 1.0; value to be adopted if input spectrum is in e- (as py script)
	                   or input spectrum is in ADU and CUBES are not flux calibrated */
		//gain = 0.5; /* value to be adopted taking in count that CUBES are not flux
		//calibrated when making a rectangular spaxel square */
	}
	//int biny=1;

	double exptime = 0;
	if(is_sinfoni) {
		exptime =  cpl_propertylist_get_double(plist, FHDR_S_DIT);
	} else {
		exptime = cpl_propertylist_get_double(plist, FHDR_E_DIT);
	}
	cpl_msg_debug(cpl_func, "gain=%g airm=%g exptime=%g ra=%g dec=%g dEpsilon=%g",
			gain, airmass, exptime, dRA, dDEC, dEpsilon);

	//cpl_msg_info(cpl_func, "table sci spectra=%s", name_sci);

	name_atm = cpl_frame_get_filename(frm_atmext);
	tbl_atmext = cpl_table_load(name_atm, 1, 0);
	//cpl_msg_info(cpl_func,"frm_atm: %s",cpl_frame_get_filename(frm_atmext));
    //cpl_msg_info(cpl_func,"frm_cat: %s",cpl_frame_get_filename(frm_cat));

	eris_parse_catalog_std_stars(frm_cat, dRA, dDEC, dEpsilon, &tbl_ref);

	if(tbl_ref == NULL) {
		cpl_msg_error(cpl_func, "Provide std sar catalog frame");
		cpl_msg_warning(cpl_func, "STD star not found in catalogue. Skip efficiency computation");
		cpl_error_set(cpl_func, CPL_ERROR_NONE);

		return NULL;
	}

	//cpl_table_save(sci_tab, NULL, NULL, "sci.fits", CPL_IO_DEFAULT);


	hdrl_spectrum1D * sci_s1d;
	hdrl_spectrum1D * std_s1d;
	hdrl_spectrum1D * ext_s1d;
	hdrl_spectrum1D * eff_s1d;
	hdrl_spectrum1D_wave_scale scale = hdrl_spectrum1D_wave_scale_linear;


	if(is_sinfoni) {
		sci_s1d = hdrl_spectrum1D_convert_from_table(sci_tab, "counts_bkg",
				"wavelength", NULL, NULL, scale);

	} else {
		sci_s1d = hdrl_spectrum1D_convert_from_table(sci_tab, "FLUX",
				"WAVE", NULL, NULL, scale);

	}

	//hdrl_spectrum1D_save(sci_s1d, "sci_s1d.fits");
	/* HDRL expects results in nm */
	hdrl_spectrum1D_wavelength_mult_scalar_linear(sci_s1d, um2nm);

	//hdrl_spectrum1D_save(sci_s1d, "sci_s1d_nm.fits");
	/* because the reference flux std assumes the flux in units of ergs/cm2/s/A
	 * we need to convert the spectrum to Angstroms and know the size of a bin
	 * in angstroms
	 */

	char band[FILE_NAME_SZ];
	if(is_sinfoni) {
		sinfo_get_band(frm_sci, band);
	} else {
		eris_get_band(frm_sci, band);
	}

	cpl_msg_debug(cpl_func, "band: %s", band);

	double conv_factor = 0;
	double nm2AA = 10.;
	//double um2AA = um2nm * nm2AA;
	/* the following is the bin size in um */
	cpl_size size = cpl_table_get_nrow(sci_tab);

	int status;
	double cdelt1 =0;
	if(is_sinfoni) {
		cdelt1 = cpl_table_get_float(sci_tab, "wavelength", size / 2,
				&status) - cpl_table_get_float(sci_tab, "wavelength",
						(size / 2 - 1), &status);
	} else {
		cdelt1 = cpl_table_get_double(sci_tab, "WAVE", size / 2,
				&status) - cpl_table_get_double(sci_tab, "WAVE",
						(size / 2 - 1), &status);
	}

	//cpl_msg_info(cpl_func, "cdelt1: %g", cdelt1);
	double bin_size = 0;
	if(is_sinfoni) {
		bin_size = sinfo_get_dispersion(band);
	} else {
		bin_size = eris_get_dispersion(band);
	}
	//cpl_msg_info(cpl_func, "bin_size: %g", bin_size);
	bin_size = cdelt1;
	//cpl_msg_info(cpl_func, "bin_size: %g", bin_size);
	//bin_size = 0.000394643;

	conv_factor = bin_size * um2nm * nm2AA;
	hdrl_value operator = {conv_factor, 0.};
	//cpl_msg_info(cpl_func,"conv factor: %g", conv_factor);
	hdrl_spectrum1D_div_scalar(sci_s1d, operator);
	//hdrl_spectrum1D_save(sci_s1d, "sci_s1d_conv_factor.fits");
	std_s1d = hdrl_spectrum1D_convert_from_table(tbl_ref, "FLUX",
			"LAMBDA", NULL, NULL, scale);
	//hdrl_spectrum1D_save(std_s1d, "std_s1d.fits");
	ext_s1d = hdrl_spectrum1D_convert_from_table(tbl_atmext, "EXTINCTION",
			"LAMBDA", NULL, NULL, scale);
	//hdrl_spectrum1D_save(ext_s1d, "ext_s1d.fits");

	hdrl_parameter* eff_pars;

	hdrl_value Ap = {0,0};
	hdrl_value Am = {airmass,0};
	hdrl_value G = {gain,0};
	hdrl_value Tex = {exptime,0};

	/*HDRL expects telescope area in cm2 */
	hdrl_value Atel = {TELESCOPE_SURFACE * m2tocm2,0};
	//cpl_msg_info(cpl_func, "Atel: %g", TELESCOPE_SURFACE * m2tocm2);
	eff_pars = hdrl_efficiency_parameter_create(Ap, Am, G, Tex, Atel);

	eff_s1d = hdrl_efficiency_compute(sci_s1d, std_s1d, ext_s1d, eff_pars);
	/* HDRL expects results in nm */
	hdrl_spectrum1D_wavelength_mult_scalar_linear(eff_s1d, nm2um);

	tbl_eff = hdrl_spectrum1D_convert_to_table(eff_s1d, "EFF", "WAVE", NULL,
			NULL);

	if( tbl_eff != NULL && frm_eff_wind_qc != NULL) {
		 cpl_table* qclog_tbl = eris_qclog_init();
		 eris_eff_qc(tbl_eff, frm_eff_wind_qc, &qclog_tbl);
		 eris_pfits_put_qc(plist, qclog_tbl);
		 cpl_table_delete(qclog_tbl);
	}


	cpl_propertylist_update_string(plist, CPL_DFS_PRO_CATG,
	                			ERIS_IFU_PRO_JITTER_EFFICIENCY);
    cpl_propertylist_erase(plist,"BUNIT");

    char* fname = cpl_sprintf("%s_efficiency.fits", pipefile_prefix);
	cpl_dfs_save_table(frameset, NULL, parlist, frameset, NULL,
	                tbl_eff, plist, "eris_ifu_stdstar", plist,
	                NULL, PACKAGE "/" PACKAGE_VERSION, fname);
    cpl_free(fname);


	hdrl_spectrum1D_delete(&std_s1d);
	hdrl_spectrum1D_delete(&sci_s1d);
	hdrl_spectrum1D_delete(&ext_s1d);
	hdrl_spectrum1D_delete(&eff_s1d);
	hdrl_parameter_delete(eff_pars);
	eris_remove_table_normal_format();
	cpl_table_delete(sci_tab);
	cpl_table_delete(tbl_ref);
	cpl_table_delete(tbl_atmext);
	cpl_propertylist_delete(plist);

	eris_check_error_code("eris_efficiency_compute");
	return tbl_eff;

}



static hdrl_spectrum1D *
eris_std_cat_frame_to_hdrl_spectrum1D(cpl_frame* frm_std_cat,const double dRA,
		const double dDEC)
{
	cpl_ensure(frm_std_cat != NULL, CPL_ERROR_NULL_INPUT, NULL);
	hdrl_spectrum1D * std_s1d;

	//const char* name = cpl_frame_get_filename(frm_std_cat);
	cpl_table* tbl_ref = NULL;

	double dEpsilon=0.1;

	eris_parse_catalog_std_stars(frm_std_cat,dRA,dDEC,dEpsilon,&tbl_ref);

	if(tbl_ref == NULL) {
		cpl_msg_error(cpl_func, "Provide std sar catalog frame");
		return NULL;
	}

	std_s1d=hdrl_spectrum1D_convert_from_table(tbl_ref, "FLUX",
			"LAMBDA", NULL, NULL, global_scale);
	cpl_table_delete(tbl_ref);
	eris_check_error_code("eris_std_cat_frame_to_hdrl_spectrum1D");
	return std_s1d;
}



static hdrl_spectrum1D *
eris_sci_frame_to_hdrl_spectrum1D(cpl_frame* frm_sci)
{
	cpl_ensure(frm_sci != NULL, CPL_ERROR_NULL_INPUT, NULL);
	hdrl_spectrum1D * sci_s1d;
	const char* name_sci=NULL;
	name_sci=cpl_frame_get_filename(frm_sci);
	cpl_table* sci_tab=NULL;
	sci_tab=cpl_table_load(name_sci,1,0);
	cpl_ensure(sci_tab != NULL, CPL_ERROR_NULL_INPUT, NULL);


	sci_s1d=hdrl_spectrum1D_convert_from_table(sci_tab, "FLUX",
			"WAVE", NULL, NULL, global_scale);
	cpl_table_delete(sci_tab);
	eris_check_error_code("eris_sci_frame_to_hdrl_spectrum1D");
	return sci_s1d;

}

static hdrl_spectrum1D *
eris_atmext_frame_to_hdrl_spectrum1D(cpl_frame* frm_atmext)
{
	cpl_ensure(frm_atmext != NULL, CPL_ERROR_NULL_INPUT, NULL);
	hdrl_spectrum1D * ext_s1d;
	const char* name_atm = cpl_frame_get_filename(frm_atmext);

	cpl_table* tbl_atmext = cpl_table_load(name_atm,1,0);


	ext_s1d=hdrl_spectrum1D_convert_from_table(tbl_atmext, "EXTINCTION",
			"LAMBDA", NULL, NULL, global_scale);
	cpl_table_delete(tbl_atmext);
	eris_check_error_code("eris_atmext_frame_to_hdrl_spectrum1D");
	return ext_s1d;
}

static hdrl_spectrum1Dlist *
eris_get_telluric_models(const cpl_frame * telluric_cat){

	cpl_ensure(telluric_cat != NULL, CPL_ERROR_NULL_INPUT, NULL);
	const char * cat_name = cpl_frame_get_filename(telluric_cat);
	const cpl_size next = cpl_frame_get_nextensions(telluric_cat);

	hdrl_spectrum1Dlist * list = hdrl_spectrum1Dlist_new();

	for(cpl_size i = 0; i < next; ++i){
		cpl_table * tab = cpl_table_load(cat_name, 1 + i, 1);

		hdrl_spectrum1D * s =
				hdrl_spectrum1D_convert_from_table(tab,
						"flux",
						"lam",
						NULL,
						NULL,
						global_scale);
		cpl_table_delete(tab);
		/*um to nm*/
		hdrl_spectrum1D_wavelength_mult_scalar_linear(s, 1e3);
		hdrl_spectrum1Dlist_set(list, s, i);
	}
	eris_check_error_code("eris_get_telluric_models");

	return list;
}

static cpl_bivector *
eris_read_wlen_windows(const cpl_frame * fit_areas_frm){

	cpl_ensure(fit_areas_frm != NULL, CPL_ERROR_NULL_INPUT, NULL);
	cpl_table * tab = NULL;

	tab = cpl_table_load(cpl_frame_get_filename(fit_areas_frm),1,0);


	const cpl_size nrow = cpl_table_get_nrow(tab);

	double * pwmin = cpl_table_unwrap(tab,"LAMBDA_MIN");
	double * pwmax = cpl_table_unwrap(tab,"LAMBDA_MAX");

	cpl_vector * wmin = cpl_vector_wrap(nrow, pwmin);
	cpl_vector * wmax = cpl_vector_wrap(nrow, pwmax);
	cpl_bivector * to_ret = cpl_bivector_wrap_vectors(wmin, wmax);
	cpl_table_delete(tab);

	eris_check_error_code("eris_read_wlen_windows");
	return to_ret;
}


static hdrl_parameter*
eris_response_parameters_calc_par(cpl_propertylist* plist)
{
	cpl_ensure(plist != NULL, CPL_ERROR_NULL_INPUT, NULL);
	hdrl_value Ap = {0, 0};


	double airmass = eris_ifu_get_airmass_mean(plist);
	hdrl_value Am = {airmass, 0.};



	double gain=ERIS_GAIN; // SINFONI had value of SINFONI_GAIN

	hdrl_value G= {gain, 0.};
	cpl_msg_debug(cpl_func, "resp gain=%g",gain);
	double dit = cpl_propertylist_get_double(plist, FHDR_E_DIT);
	double exptime = dit;
	cpl_msg_debug(cpl_func, "resp exptime=%g",exptime);

	hdrl_value Tex= {exptime, 0.};
	hdrl_parameter* calc_par = hdrl_response_parameter_create(Ap, Am, G, Tex);
	eris_check_error_code("eris_response_parameters_calc_par");
	return calc_par;

}


static cpl_array *
eris_read_fit_points(const cpl_frame * frm_fit_points,
		const double ra, const double dec, const double ra_dec_tolerance){

	cpl_ensure(frm_fit_points != NULL, CPL_ERROR_NULL_INPUT, NULL);
	const char * fname = cpl_frame_get_filename(frm_fit_points);

	cpl_table * index_table = cpl_table_load(fname, 1, CPL_FALSE);
	const cpl_size sz = cpl_table_get_nrow(index_table);

	cpl_size selected_ext = -1;

	for(cpl_size i = 0; i < sz; ++i){
		int rej;
		const int ext_id = cpl_table_get_int(index_table, "ext_id", i ,&rej);
		const double curr_ra = cpl_table_get(index_table, "ra", i, &rej);
		const double curr_dec = cpl_table_get(index_table, "dec", i, &rej);
		if ((ext_id > 0) && (fabs(curr_ra - ra) < ra_dec_tolerance) &&
				(fabs(curr_dec - dec) < ra_dec_tolerance)){
			selected_ext = ext_id;
		}
	}

	cpl_table_delete(index_table);
	cpl_ensure(selected_ext >= 0, CPL_ERROR_ILLEGAL_OUTPUT, NULL);


	cpl_table * tb = cpl_table_load(fname, selected_ext, CPL_FALSE);

	const cpl_size sz_points = cpl_table_get_nrow(tb);
	cpl_array * fit_points = cpl_array_new(sz_points, CPL_TYPE_DOUBLE);

	for(cpl_size i = 0; i < sz_points; ++i){
		int rej = 0;
		const double d = cpl_table_get(tb, "LAMBDA", i, &rej);
		cpl_array_set(fit_points, i, d);
	}

	cpl_table_delete(tb);
	eris_check_error_code("eris_read_fit_points");
	return fit_points;
}

static inline cpl_size
get_before(const double wmin_spectrum, const double wmin_interval, const double step){
	const double num = (wmin_spectrum - wmin_interval) / step;
	if(num <= 0) return 0;
	return (cpl_size) ceil(num);
}

static inline cpl_size
get_after(const double wmax_spectrum, const double wmax_interval, const double step){
	const double num = (wmax_interval - wmax_spectrum) / step;
	if(num <= 0) return 0;
	return (cpl_size) ceil(num);
}
static inline
hdrl_spectrum1D * extend_hdrl_spectrum(const double wmin, const double wmax,
		const hdrl_spectrum1D * s){

	cpl_ensure(wmin < wmax, CPL_ERROR_ILLEGAL_INPUT, NULL);
	cpl_ensure(s != NULL, CPL_ERROR_NULL_INPUT, NULL);

	const cpl_array * lambadas_s = hdrl_spectrum1D_get_wavelength(s).wavelength;
	const double s_wmin = cpl_array_get_min(lambadas_s);
	const double s_wmax = cpl_array_get_max(lambadas_s);

	const cpl_size sz_s = cpl_array_get_size(lambadas_s);
	const double step = (s_wmax - s_wmin)/sz_s;

	cpl_msg_debug(cpl_func, "min=%g max=%g step=%g",s_wmin,s_wmax,step);
	cpl_size n_before = get_before(s_wmin, wmin, step);
	cpl_size n_after = get_after(s_wmax, wmax, step);

	const cpl_size new_size = n_before + n_after + sz_s;
	cpl_array * new_lambdas = cpl_array_new(new_size,
			cpl_array_get_type(lambadas_s));
	hdrl_image * new_flux = hdrl_image_new(new_size, 1);

	double lambda = s_wmin;
	for(cpl_size i = n_before - 1; i >= 0; i--){
		lambda -= step;
		cpl_array_set(new_lambdas, i, lambda);
		hdrl_image_reject(new_flux, i + 1, 1);
	}

	lambda = s_wmax;
	for(cpl_size i = n_before + sz_s; i < new_size; i++){
		lambda += step;
		cpl_array_set(new_lambdas, i, lambda);
		hdrl_image_reject(new_flux, i + 1, 1);
	}

	for(cpl_size i = n_before; i < n_before + sz_s; i++){
		const cpl_size idx_ori = i - n_before;
		int rej = 0;
		lambda = hdrl_spectrum1D_get_wavelength_value(s, idx_ori, &rej);
		cpl_array_set(new_lambdas, i, lambda);

		if(rej) {
			hdrl_image_reject(new_flux, i + 1, 1);
			continue;
		}

		hdrl_value val = hdrl_spectrum1D_get_flux_value(s, idx_ori, NULL);
		hdrl_image_set_pixel(new_flux, i + 1, 1, val);
	}

	const hdrl_spectrum1D_wave_scale scale = hdrl_spectrum1D_get_scale(s);

	hdrl_spectrum1D * to_ret =
			hdrl_spectrum1D_create(hdrl_image_get_image(new_flux),
					hdrl_image_get_error(new_flux),
					new_lambdas,
					scale);


	cpl_array_delete(new_lambdas);
	hdrl_image_delete(new_flux);

	eris_check_error_code("hdrl_response_fit_parameter_create");
	return to_ret;
}

static cpl_error_code
eris_resp_qc(cpl_table* resp, cpl_frame* frm_eff_wind_qc, const char* col1,
		const char* col2, const double wmin_g,
		const double wmax_g, cpl_table** qclog_tbl )
{

	cpl_ensure(resp != NULL, CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT);
	cpl_ensure(frm_eff_wind_qc != NULL, CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT);
	cpl_ensure(wmin_g < wmax_g, CPL_ERROR_ILLEGAL_INPUT, CPL_ERROR_ILLEGAL_INPUT);
	cpl_ensure(*qclog_tbl != NULL, CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT);

	const char* ew_name=cpl_frame_get_filename(frm_eff_wind_qc);
	cpl_table* ew_tab=cpl_table_load(ew_name,1,0);
	int nrow = cpl_table_get_nrow(ew_tab);

	float* pwmin = cpl_table_get_data_float(ew_tab,"WMIN");
	float* pwmax = cpl_table_get_data_float(ew_tab,"WMAX");
	cpl_table* tmp_resp=NULL;
	double wmin;
	double wmax;
	double resp_avg;
	double resp_med;
	double resp_min;
	double resp_max;
	double resp_std;
	char* key_name;
	//double um2nm=1000;
	cpl_table* sto_resp = cpl_table_new(0);
	cpl_table_copy_structure(sto_resp, resp);
	int nrow_tmp=0;
	int ncontributes = 0;
	int nrow_sto = 0;
	for(int i=0;i<nrow;i++) {
		cpl_table_select_all(resp);
		wmin=pwmin[i];
		wmax=pwmax[i];
		cpl_table_and_selected_double(resp, col1, CPL_NOT_LESS_THAN, wmin_g);
		cpl_table_and_selected_double(resp, col1, CPL_NOT_LESS_THAN, wmin);
		cpl_table_and_selected_double(resp, col1, CPL_NOT_GREATER_THAN, wmax);
		cpl_table_and_selected_double(resp, col1, CPL_NOT_GREATER_THAN, wmax_g);
		tmp_resp=cpl_table_extract_selected(resp);
		nrow_tmp = cpl_table_get_nrow(tmp_resp);
		if(nrow_tmp>0) {
			ncontributes++;
		    cpl_table_insert(sto_resp,tmp_resp, nrow_sto);
		    nrow_sto = cpl_table_get_nrow(sto_resp);
			resp_avg=cpl_table_get_column_mean(tmp_resp, col2);
			resp_med=cpl_table_get_column_median(tmp_resp, col2);
			resp_min=cpl_table_get_column_min(tmp_resp, col2);
			resp_max=cpl_table_get_column_max(tmp_resp, col2);
			resp_std=cpl_table_get_column_stdev(tmp_resp, col2);
			/*
            cpl_msg_info(cpl_func, "wmin=%g wmax=%g avg=%g med=%g min=%g max=%g std=%g",
            		wmin, wmax, resp_avg,resp_med,resp_min,resp_max,resp_std);
			 */
			key_name = cpl_sprintf("QC RESP WIN%d WLMIN",i);
			eris_qclog_add_double(*qclog_tbl, key_name, wmin,
					"[um] Min window wavelength for resp comp");
			cpl_free(key_name);

			key_name = cpl_sprintf("QC RESP WIN%d WLMAX",i);
			eris_qclog_add_double(*qclog_tbl, key_name, wmax,
					"[um] Max window wavelength for resp comp");
			cpl_free(key_name);


			key_name = cpl_sprintf("QC RESP WIN%d MEAN",i);
			eris_qclog_add_double(*qclog_tbl, key_name, resp_avg,
					"Mean response on window");
			cpl_free(key_name);

			key_name = cpl_sprintf("QC RESP WIN%d MEDIAN",i);
			eris_qclog_add_double(*qclog_tbl, key_name, resp_med,
					"Median response on window");
			cpl_free(key_name);

			key_name = cpl_sprintf("QC RESP WIN%d MIN",i);
			eris_qclog_add_double(*qclog_tbl, key_name, resp_min,
					"Min response on window");
			cpl_free(key_name);

			key_name = cpl_sprintf("QC RESP WIN%d MAX",i);
			eris_qclog_add_double(*qclog_tbl, key_name, resp_max,
					"Max response on window");
			cpl_free(key_name);

			key_name = cpl_sprintf("QC RESP WIN%d RMS",i);
			eris_qclog_add_double(*qclog_tbl, key_name, resp_std,
					"RMS response on window");
			cpl_free(key_name);
		}
		cpl_table_select_all(resp);
		cpl_table_delete(tmp_resp);

	}
	cpl_table_select_all(resp);
	wmin=cpl_table_get_column_min(sto_resp, col1);
	wmax=cpl_table_get_column_max(sto_resp, col1);
	cpl_table_and_selected_double(sto_resp, col1, CPL_NOT_LESS_THAN, wmin_g);
	cpl_table_and_selected_double(sto_resp, col1, CPL_NOT_GREATER_THAN, wmax_g);
	tmp_resp=cpl_table_extract_selected(sto_resp);

	resp_avg=cpl_table_get_column_mean(sto_resp, col2);
	resp_med=cpl_table_get_column_median(sto_resp, col2);
	resp_min=cpl_table_get_column_min(sto_resp, col2);
	resp_max=cpl_table_get_column_max(sto_resp, col2);
	resp_std=cpl_table_get_column_stdev(sto_resp, col2);
	/*
    cpl_msg_info(cpl_func,"wmin=%g wmax=%g avg=%g med=%g min=%g max=%g std=%g",
               wmin, wmax,resp_avg,resp_med,resp_min,resp_max,resp_std);
	 */
	 key_name = cpl_sprintf("QC RESP NWIN");
	        eris_qclog_add_int(*qclog_tbl, key_name, ncontributes,
	                                     "Number of windows used for resp comp");
	cpl_free(key_name);
	key_name = cpl_sprintf("QC RESP WLMIN");
	eris_qclog_add_double(*qclog_tbl, key_name, wmin,
			"[um] Min wavelength for resp comp");
	cpl_free(key_name);

	key_name = cpl_sprintf("QC RESP WLMAX");
	eris_qclog_add_double(*qclog_tbl, key_name, wmax,
			"[um] Max wavelength for resp comp");
	cpl_free(key_name);

	key_name = cpl_sprintf("QC RESP MEAN");
	eris_qclog_add_double(*qclog_tbl, key_name, resp_avg,
			"Mean response");
	cpl_free(key_name);

	key_name = cpl_sprintf("QC RESP MEDIAN");
	eris_qclog_add_double(*qclog_tbl, key_name, resp_med,
			"Median response");
	cpl_free(key_name);

	key_name = cpl_sprintf("QC RESP MIN");
	eris_qclog_add_double(*qclog_tbl, key_name, resp_min,
			"Min response");
	cpl_free(key_name);

	key_name = cpl_sprintf("QC RESP MAX");
	eris_qclog_add_double(*qclog_tbl, key_name, resp_max,
			"Max response");
	cpl_free(key_name);

	key_name = cpl_sprintf("QC RESP RMS");
	eris_qclog_add_double(*qclog_tbl, key_name, resp_std,
			"RMS response");
	cpl_free(key_name);

	cpl_table_delete(tmp_resp);
	cpl_table_delete(ew_tab);
	cpl_table_delete(sto_resp);
	eris_check_error_code("eris_resp_qc");
	return cpl_error_get_code();
}


static void
eris_log_pro(char* name_o,
		const char* pro_catg,
		int frm_type,
		cpl_frameset* ref_set,
		cpl_frameset** out_set,
		cpl_propertylist** plist,
		const cpl_parameterlist* parlist,
		const char* recid)
{
	cpl_frame* product_frame = NULL ;
	char * pipe_id=NULL;
	cpl_errorstate initial_errorstate = cpl_errorstate_get();

    pipe_id = cpl_sprintf("%s%s","eris/",PACKAGE_VERSION);
	product_frame = cpl_frame_new() ;

	cpl_frame_set_filename(product_frame, name_o) ;
	cpl_frame_set_tag(product_frame, pro_catg) ;
	cpl_frame_set_type(product_frame, frm_type);
	cpl_frame_set_group(product_frame, CPL_FRAME_GROUP_PRODUCT);
	cpl_frame_set_level(product_frame, CPL_FRAME_LEVEL_FINAL);


	if(cpl_dfs_setup_product_header(*plist,product_frame,ref_set,parlist,recid,
			pipe_id,KEY_VALUE_HPRO_DID,NULL) != CPL_ERROR_NONE) {
		cpl_msg_warning(cpl_func, "Problem in the product DFS-compliance");
		cpl_msg_warning(cpl_func, "%s", (char* ) cpl_error_get_message());
		cpl_errorstate_dump(initial_errorstate, CPL_FALSE, NULL);
		cpl_error_reset();
	}

	cpl_frameset_insert(*out_set, product_frame);

	cpl_free(pipe_id);
	eris_check_error_code("eris_log_pro");
	return;
}



static char * eris_new_get_rootname(const char * filename)
{
	static char path[MAX_STR_SIZE+1];
	char * lastdot ;

	if (strlen(filename)>MAX_STR_SIZE) return NULL ;
	memset(path, 0, MAX_STR_SIZE);
	strcpy(path, filename);
	lastdot = strrchr(path, '.');
	if (lastdot == NULL) return path ;
	if ((!strcmp(lastdot, ".fits")) || (!strcmp(lastdot, ".FITS"))
			|| (!strcmp(lastdot, ".paf")) || (!strcmp(lastdot, ".PAF"))
			|| (!strcmp(lastdot, ".dat")) || (!strcmp(lastdot, ".DAT"))
			|| (!strcmp(lastdot, ".fits")) || (!strcmp(lastdot, ".TFITS"))
			|| (!strcmp(lastdot, ".ascii"))
			|| (!strcmp(lastdot, ".ASCII")))
	{
		lastdot[0] = (char)0;
	}
	return path ;
}




static void
eris_check_name(const char* in, char** ou, /*int type, */char** paf) {

	char*  name_b;
	if (strstr(in, "." ) != NULL ) {
	  char* tmp = eris_new_get_rootname(in);
	  name_b= cpl_sprintf("%s",tmp);
	} else {
	  name_b = cpl_sprintf("%s",in);
	}
	*ou = cpl_sprintf("%s.fits",name_b);
	/*
	if (type == CPL_FRAME_TYPE_TABLE) {
	  strcat(*ou,".fits");
	} else {
	  strcat(*ou,".fits");
	}
	*/
	*paf = cpl_sprintf("%s.paf",name_b);
    cpl_free(name_b);

}


static int
eris_pro_save_tbl(
		cpl_table       *   table,
		cpl_frameset    *   ref,
		cpl_frameset    *   set,
		const char      *   out_file,
		const char      *   pro_catg,
		cpl_table       *   qclog,
		const char      *   recid,
		const cpl_parameterlist* parlist)

{
	char *    name_o =NULL;
	char *    name_p =NULL;
	cpl_propertylist *   plist=NULL ;
	char* ref_file=NULL;
	/* Get the reference file  */

	cpl_frameset_iterator* it = cpl_frameset_iterator_new(ref);
	cpl_frame *first_frame = cpl_frameset_iterator_get(it);
	ref_file = cpl_strdup(cpl_frame_get_filename(first_frame)) ;

	//name_o = cpl_malloc(FILE_NAME_SZ * sizeof(char));
	//name_p = cpl_malloc(FILE_NAME_SZ * sizeof(char));
    eris_check_name(out_file, &name_o, /*CPL_FRAME_TYPE_TABLE, */&name_p);
	cpl_msg_info(cpl_func, "Writing tbl %s pro catg %s" , name_o, pro_catg) ;

	/* Get FITS header from reference file */
	if ((cpl_error_code)((plist = cpl_propertylist_load(ref_file,0)) == NULL))
	{
		cpl_msg_error(cpl_func, "getting header from tbl frame %s",ref_file);
		cpl_propertylist_delete(plist) ;
		cpl_free(ref_file);
		cpl_free(name_o);
		cpl_free(name_p);
		cpl_frameset_iterator_delete(it);
		return -1 ;
	}

	//Remove CHECKSUM & ESO PRO keywords: Why?
	//cpl_propertylist_erase_regexp(plist, "CHECKSUM",0);
	//cpl_propertylist_erase_regexp(plist, "^ESO PRO .*",0);

	/* Add DataFlow keywords and log the saved file in the input frameset */
	eris_log_pro(name_o, pro_catg, CPL_FRAME_TYPE_TABLE,
			ref,&set,&plist,parlist,recid);

	if(qclog != NULL) {
		eris_pfits_put_qc(plist, qclog) ;
	}
	cpl_propertylist_append_string(plist, "PRODCATG", "SCIENCE.SPECTRUM");
	/* Save the file */
	if (cpl_table_save(table, plist, NULL, name_o, CPL_IO_DEFAULT)
			!= CPL_ERROR_NONE) {
		cpl_msg_error(cpl_func, "Cannot save the product: %s", name_o);
		cpl_propertylist_delete(plist) ;
		cpl_free(ref_file);
		cpl_free(name_o);
		cpl_free(name_p);
		cpl_frameset_iterator_delete(it);

		return -1 ;
	}

	cpl_propertylist_delete(plist) ;
	cpl_msg_indent_less() ;
	cpl_free(name_o);
	cpl_free(name_p);
	cpl_free(ref_file);
	cpl_frameset_iterator_delete(it);

	eris_check_error_code("eris_pro_save_tbl");
	return 0 ;
}


/*----------------------------------------------------------------------------*/
/**
 @brief     polynomial fit (wrap for the gsl functions)
 @param         data            data to be fitted
 @param         deg             polynomial degree
 @param[out]    fit_RE          fit result
 @param[out]    coeffs_RE       coeffs of the fit
 @param[out]    coeffs_err_RE   error on coeffs of the fit
 @param[out]    chisq_RE        CHI2 of the fit
 @param         print_flag      1 - print, 0 - don't
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

static cpl_error_code
eris_fit_poly (eris_poly_data *data, const int deg, double *fit_RE,
		      double *coeffs_RE, double *coeffs_err_RE, double *chisq_RE,
			  const int print_flag) {

	cpl_ensure(data != NULL, CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT);
	cpl_ensure(deg > 1, CPL_ERROR_ILLEGAL_INPUT, CPL_ERROR_ILLEGAL_INPUT);

    int i, j, n;
    double err, chisq;
    gsl_vector *x, *y, *w, *p;
    gsl_matrix *X, *covar;
    //double *fit;
    gsl_multifit_linear_workspace *work;

    n = data->n;
    x = gsl_vector_alloc (n);
    y = gsl_vector_alloc (n);
    w = gsl_vector_alloc (n);
    p = gsl_vector_alloc (deg);
    X = gsl_matrix_alloc (n, deg);
    covar = gsl_matrix_alloc (deg, deg);

    if (print_flag) {
        cpl_msg_info(cpl_func,"n = %d, deg = %d", n, deg);
    }

    for (i = 0; i < n; i++) {
        gsl_vector_set (x, i, data->x[i]);
        gsl_vector_set (y, i, data->y[i]);
        err = data->err[i];
        gsl_vector_set (w, i, 1.0/err/err);

        for (j = 0; j < deg; j++)
            gsl_matrix_set (X, i, j, gsl_pow_int(gsl_vector_get(x, i), j));
    }

    if (print_flag) {
        cpl_msg_info(cpl_func, "vectors set");
    }

    work = gsl_multifit_linear_alloc (n, deg);
    gsl_multifit_wlinear (X, w, y, p, covar, &chisq, work);
    gsl_multifit_linear_free (work);

    if (print_flag) {
        cpl_msg_info(cpl_func, "Fit done");
    }

    for (i = 0; i < n; i++) {
        fit_RE[i] = 0.;
        for (j = 0; j < deg; j++)
            fit_RE[i] += gsl_matrix_get (X, i, j) * gsl_vector_get (p, j);
    }

    chisq = chisq/(n-deg);

    for (j = 0; j < deg; j++) {
        coeffs_RE[j] = gsl_vector_get (p, j);
        coeffs_err_RE[j] = sqrt(gsl_matrix_get (covar, j, j));
    }

    *chisq_RE = chisq;

    gsl_vector_free(x);
    gsl_vector_free(y);
    gsl_vector_free(w);
    gsl_vector_free(p);
    gsl_matrix_free(X);
    gsl_matrix_free(covar);

    eris_check_error_code("eris_fit_poly");
    return cpl_error_get_code();
}

static cpl_table*
eris_fit_poly_and_clip_response(const cpl_table* resp_pts,
		const cpl_table* resp_smo, const int deg, const double kappa,
		const int niter)
{

	cpl_ensure(resp_pts != NULL, CPL_ERROR_NULL_INPUT, NULL);
	cpl_ensure(resp_smo != NULL, CPL_ERROR_NULL_INPUT, NULL);

	/* polynomial fit */
	eris_poly_data *p_data = (eris_poly_data *)cpl_malloc(sizeof(eris_poly_data));

	cpl_size pol_deg = deg;
	double chisq = 0;
	cpl_table* resp_pts_tmp = cpl_table_duplicate(resp_pts);
	double* fit =NULL;
	double* coeffs = NULL;
	double* coeffs_err = NULL;
	double y2, x2;
	//double y2err;
	cpl_table* xtab = NULL;
	for(cpl_size k= 0; k < niter; k++) {
		cpl_size nrows = cpl_table_get_nrow(resp_pts_tmp);
		cpl_msg_debug(cpl_func,"k: %lld nrows: %lld", k,nrows);
		fit = (double *) cpl_calloc (nrows , sizeof(double));
		coeffs = (double *) cpl_calloc (pol_deg , sizeof(double));
		coeffs_err = (double *) cpl_calloc (pol_deg , sizeof(double));

		p_data->x = (double *) cpl_calloc(nrows,sizeof(double));
		p_data->y = (double *) cpl_calloc(nrows,sizeof(double));
		p_data->err = (double *) cpl_calloc(nrows,sizeof(double));
		p_data->n = nrows;

		for (cpl_size i = 0; i < nrows; i++) {
			p_data->x[i] = cpl_table_get(resp_pts_tmp,"wavelength",i,NULL);
			p_data->y[i] = cpl_table_get(resp_pts_tmp,"response_fit_pts",i,NULL);
			p_data->err[i] = 1.;
		}

		eris_fit_poly(p_data, pol_deg, fit, coeffs, coeffs_err, &chisq, 0);

		// determine residuals to poly fit
		cpl_table_new_column(resp_pts_tmp, "res", CPL_TYPE_DOUBLE);
		for (int j = 0; j < nrows; j++) {
			x2 = cpl_table_get(resp_pts_tmp, "wavelength", j, NULL);
			y2 = 0.;
			for (int c = pol_deg-1; c >= 0; c--) {
				y2 = y2 * x2 + coeffs[c];
			}
			cpl_table_set_double(resp_pts_tmp, "res",j, y2-p_data->y[j]);
		}
		//cpl_table_save (resp_pts_tmp, NULL, NULL, "resp_pts_tmp.fits", CPL_IO_CREATE);

		// clip data for which residual is outside 3 sigma from fit
		double stddev = cpl_table_get_column_stdev(resp_pts_tmp, "res");
		double mean = cpl_table_get_column_mean(resp_pts_tmp, "res");
		cpl_table_duplicate_column(resp_pts_tmp,"dist",resp_pts_tmp,"res");
		cpl_table_subtract_scalar(resp_pts_tmp,"dist",mean);
		cpl_table_abs_column(resp_pts_tmp,"dist");
		cpl_msg_debug(cpl_func,"mean: %g, stdev: %g",mean, stddev);
		cpl_table_and_selected_double(resp_pts_tmp, "dist", CPL_LESS_THAN,
				kappa * stddev);
		xtab = cpl_table_extract_selected(resp_pts_tmp);
		//cpl_table_save (xtab, NULL, NULL, "xtab.fits", CPL_IO_CREATE);
		cpl_table_delete(resp_pts_tmp);
		cpl_table_erase_column(xtab,"res");
		cpl_table_erase_column(xtab,"dist");
		resp_pts_tmp = cpl_table_duplicate(xtab);

		cpl_free(coeffs);
		cpl_free(coeffs_err);
		cpl_free(fit);
		cpl_table_delete(xtab);
		cpl_free(p_data->x);
		cpl_free(p_data->y);
		cpl_free(p_data->err);
	}

	cpl_size next = cpl_table_get_nrow(resp_pts_tmp);

	/* now repeat fit after clipping of outliers */

	//cpl_free(p_data);
	p_data->x = (double *) cpl_calloc(next,sizeof(double));
	p_data->y = (double *) cpl_calloc(next,sizeof(double));
	p_data->err = (double *) cpl_calloc(next,sizeof(double));
	p_data->n = next;

	for (cpl_size i = 0; i < next; i++) {
		p_data->x[i] = cpl_table_get(resp_pts_tmp,"wavelength",i,NULL);
		p_data->y[i] = cpl_table_get(resp_pts_tmp,"response_fit_pts",i,NULL);
		p_data->err[i] = 1.;
	}
	//cpl_free(fit);
	//cpl_free(coeffs);
	//cpl_free(coeffs_err);

	/* repeat fit after kappa-sigma clip of data */
	fit = (double *) cpl_calloc (next , sizeof(double));
	coeffs = (double *) cpl_calloc (pol_deg , sizeof(double));
	coeffs_err = (double *) cpl_calloc (pol_deg , sizeof(double));
	eris_fit_poly(p_data, pol_deg, fit, coeffs, coeffs_err, &chisq, 0);
    //cpl_msg_warning(cpl_func,"chisq: %g",chisq);

	/* now create final result */
	cpl_table* resp_poly = cpl_table_duplicate(resp_smo);
	cpl_size poly_nval = cpl_table_get_nrow(resp_poly);
	cpl_table_name_column(resp_poly, "response_smo", "response_poly");
	cpl_table_name_column(resp_poly, "response_smo_error", "response_poly_error");
	cpl_table_fill_column_window_double(resp_poly,"response_poly_error",0, poly_nval, 0);
	cpl_table_name_column(resp_poly, "response_smo_bpm", "response_poly_bpm");

	for (int j = 0; j < poly_nval; j++) {
		x2 = cpl_table_get(resp_poly, "wavelength", j, NULL);
		//cpl_table_set_double(*s1d_eff_RE, "wavelength", j, x2);
		y2 = 0.;
		for (int c = pol_deg-1; c >= 0; c--) {
			y2 = y2 * x2 + coeffs[c];
		}
		/*
		y2err = 0.;
		for (int c = pol_deg-1; c >= 0; c--) {
			y2err = y2err * x2 + coeffs_err[c];
		}
		*/

		cpl_table_set_double(resp_poly, "response_poly",j, y2);
		/* estimate error as 0.1*value */
		cpl_table_set_double(resp_poly, "response_poly_error",j, 0.1*y2 );
	}

	/* free memory */
	cpl_free(fit);
	cpl_free(coeffs);
	cpl_free(coeffs_err);
	cpl_table_delete(resp_pts_tmp);
	cpl_free(p_data->x);
    cpl_free(p_data->y);
	cpl_free(p_data->err);
	cpl_free(p_data);

	eris_check_error_code("eris_fit_poly_and_clip_response");
	return resp_poly;
}

static cpl_error_code
eris_get_wave_shift_param(const char* band, double* wmin, double* wmax,
		cpl_boolean* enable)
{

	cpl_msg_info(cpl_func, "observed band: %s", band);
	if ((strcmp(band, "J_low") == 0) ||
			(strcmp(band, "J_short") == 0) ||
			(strcmp(band, "J_middle") == 0) ||
			(strcmp(band, "J_long") == 0)) {
		//J
		*wmin = 7.001; //log scale
		*wmax = 7.258;
		*enable = CPL_FALSE; // was CPL_TRUE;
	} else if ((strcmp(band, "H_low") == 0) ||
			(strcmp(band, "H_short") == 0) ||
			(strcmp(band, "H_middle") == 0) ||
			(strcmp(band, "H_long") == 0)) {
		//H
		*wmin = 7.268; //log scale
		*wmax = 7.531;
		*enable = CPL_FALSE;
	} else if ((strcmp(band, "K_low") == 0) ||
			(strcmp(band, "K_short") == 0) ||
			(strcmp(band, "K_middle") == 0) ||
			(strcmp(band, "K_long") == 0)) {
		//K
		*wmin = 7.565; //log scale
		*wmax = 7.812;
		*enable = CPL_FALSE;
	} else if (strcmp(band, "H+K") == 0) {
		//HK
		*wmin = 7.268; //log scale
		*wmax = 7.812;
		*enable = CPL_FALSE;
	}

	eris_check_error_code("eris_get_wave_shift_param");
	return cpl_error_get_code();
}

/*---------------------------------------------------------------------------*/
/**
   @brief    compute response
   @param    plugin_id  input recipe plugin id
   @param    config input parameterlist
   @param    frame_set input reference set of frames
   @param    set input set of frames
   @return   cpl_table* output table with computed efficiency

   DFS only. See the DICB dictionaries to have details on the keywords.
   @note: currently support both SINFONI and ERIS data to enable comparisons
   on efficiency with ETC. Save extra products for checks. gain is set to
   compensate bug on cubes flux calibration and the fact ERIS input comes from
   1d spectrum from python script where gain is already applied
 */
/*---------------------------------------------------------------------------*/


cpl_error_code
eris_response_compute(const char* plugin_id, const cpl_parameterlist* config,
		cpl_frameset* ref_set,cpl_frameset* sof)
{

    cpl_ensure_code(plugin_id, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(config, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(ref_set, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(sof, CPL_ERROR_NULL_INPUT);

	cpl_frame* frm_sci = cpl_frameset_find(sof, ERIS_IFU_PRO_JITTER_SPECTRUM);
	cpl_frame* frm_std_cat = cpl_frameset_find(sof, ERIS_IFU_CALIB_FLUX_STD_CATALOG);
	cpl_frame* frm_atmext = cpl_frameset_find(sof, ERIS_IFU_CALIB_EXTCOEFF_TABLE);
	cpl_frame* frm_tell_cat = cpl_frameset_find(sof, ERIS_IFU_CALIB_TELL_MOD_CATALOG);
	cpl_frame* frm_resp_fit_points = cpl_frameset_find(sof, ERIS_IFU_CALIB_RESP_FIT_POINTS_CATALOG);
	//cpl_frame* frm_high_abs_reg = cpl_frameset_find(sof, ERIS_IFU_CALIB_HIGH_ABS_REGIONS);
	cpl_frame* frm_resp_quality_areas = cpl_frameset_find(sof, ERIS_IFU_CALIB_QUALITY_AREAS);
	cpl_frame* frm_resp_fit_areas = cpl_frameset_find(sof, ERIS_IFU_CALIB_FIT_AREAS);

	cpl_ensure(frm_sci != NULL, CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT);
	cpl_ensure(frm_std_cat != NULL, CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT);
	cpl_ensure(frm_atmext != NULL, CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT);
	cpl_ensure(frm_tell_cat != NULL, CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT);
	cpl_ensure(frm_resp_fit_points != NULL, CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT);
	/* HIGH_ABS_REGIONS is an optional input
	cpl_ensure(frm_high_abs_reg != NULL, CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT);
	 */
	cpl_ensure(frm_resp_quality_areas != NULL, CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT);
	cpl_ensure(frm_resp_fit_areas != NULL, CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT);

	const char* name_sci = NULL;
	cpl_propertylist* plist = NULL;
	cpl_table* sci_tab = NULL;
	name_sci = cpl_frame_get_filename(frm_sci);
	cpl_msg_debug(cpl_func,"fname: %s",name_sci);
	plist = cpl_propertylist_load(name_sci,0);

	double dRA = cpl_propertylist_get_double(plist, "RA");

	double  dDEC = cpl_propertylist_get_double(plist, "DEC");
	//sci_tab = cpl_table_load(name_sci, 1, 0);



	double sci_wmin = 0;
	double sci_wmax = 0;
	cpl_frame* frm_sci_dup = cpl_frame_duplicate(frm_sci);
	//scpl_table_delete(sci_tab);
	sci_tab = eris_ifu_table_from_sdp_to_normal_format(&frm_sci_dup);

	sci_wmin = cpl_table_get_column_min(sci_tab,"WAVE");
	sci_wmax = cpl_table_get_column_max(sci_tab, "WAVE");

	double um2nm = 1.e3;

	hdrl_spectrum1D * sci_s1d = eris_sci_frame_to_hdrl_spectrum1D(frm_sci_dup);

	cpl_frame_delete(frm_sci_dup);
	/* HDRL expects results in nm */
	hdrl_spectrum1D_wavelength_mult_scalar_linear(sci_s1d, um2nm);

	/* because the reference flux std assumes the flux in units of ergs/cm2/s/A
	 * we need to convert the spectrum to Angstroms and know the size of a bin
	 * in angstroms
	 */
	char band[FILE_NAME_SZ];
	eris_get_band(frm_sci,band);

	double conv_factor = 0;
	double nm2AA = 10.;
	/* the following is the bin size in um */
	double bin_size=eris_get_dispersion(band);

	cpl_size size = cpl_table_get_nrow(sci_tab);
	int status;
	double cdelt1 = cpl_table_get_double(sci_tab, "WAVE", size / 2, &status) -
			cpl_table_get_double(sci_tab, "WAVE", (size / 2 - 1), &status);
	cpl_table_delete(sci_tab);
	//cpl_msg_info(cpl_func, "cdelt1: %g", cdelt1);
	bin_size = cdelt1;
	//cpl_msg_info(cpl_func, "bin_size=%g", bin_size);
	conv_factor = bin_size * um2nm * nm2AA;

	hdrl_spectrum1D_div_scalar(sci_s1d, (hdrl_value){conv_factor,0.});

	hdrl_spectrum1D * std_s1d =
			eris_std_cat_frame_to_hdrl_spectrum1D(frm_std_cat, dRA, dDEC);

	if(std_s1d == NULL) {
		cpl_msg_warning(cpl_func,"Ref flux STD star not found in catalogue. Skip response determination");
		cpl_error_set(cpl_func,CPL_ERROR_NONE);
		return CPL_ERROR_NONE;
	}

	hdrl_spectrum1D * ext_s1d = eris_atmext_frame_to_hdrl_spectrum1D(frm_atmext);

	//hdrl_parameter* eff_pars;
	/*HDRL expects telescope area in cm2 */

	hdrl_spectrum1Dlist * telluric_models =
			eris_get_telluric_models(frm_tell_cat);

	cpl_bivector * fit_areas = eris_read_wlen_windows(frm_resp_fit_areas);

	cpl_bivector * quality_areas = eris_read_wlen_windows(frm_resp_quality_areas);

	/*
	cpl_bivector * high_abs_regions = NULL;
	if(frm_high_abs_reg!= NULL)
		high_abs_regions = eris_read_wlen_windows(frm_high_abs_reg);
		*/

	/* set [parameters */

	double wmin = 0;
	double wmax = 0;
	cpl_msg_info(cpl_func, "observed band: %s", band);
	cpl_boolean velocity_enable = CPL_TRUE;
	eris_get_wave_shift_param(band, &wmin, &wmax, &velocity_enable);


	cpl_msg_debug(cpl_func, "band=%s wmin=%g wmax=%g", band, wmin, wmax);

	const hdrl_data_t lmin = (hdrl_data_t)wmin;
	const hdrl_data_t lmax = (hdrl_data_t)wmax;


	const cpl_boolean xcorr_normalize = (cpl_boolean) CPL_TRUE;
	const cpl_boolean shift_in_log_scale = (cpl_boolean) CPL_TRUE;

	const hdrl_data_t wguess = 1094.12; // um
	const hdrl_data_t range_wmin = 1000;
	const hdrl_data_t range_wmax = 1200;
	const hdrl_data_t fit_wmin = 1090;
	const hdrl_data_t fit_wmax = 1100;
	const hdrl_data_t fit_half_win = 1.5;

	hdrl_parameter* shift_par = NULL;
	if(velocity_enable) {
		shift_par = hdrl_spectrum1D_shift_fit_parameter_create(wguess,
				range_wmin, range_wmax, fit_wmin, fit_wmax, fit_half_win);
	}

	hdrl_parameter* resp_calc_par = eris_response_parameters_calc_par(plist);

	double ra_dec_tolerance = 0.1;
	cpl_array 	* fit_points = eris_read_fit_points(frm_resp_fit_points,
			dRA, dDEC, ra_dec_tolerance);

	hdrl_parameter * telluric_par = NULL;
	if(telluric_models != NULL && quality_areas != NULL && fit_areas != NULL && lmin < lmax) {
		const cpl_size xcorr_half_win = 200;
		hdrl_data_t xcorr_w_step = 1e-5;
		telluric_par = hdrl_response_telluric_evaluation_parameter_create(telluric_models,
				xcorr_w_step, xcorr_half_win, xcorr_normalize, shift_in_log_scale, quality_areas,
				fit_areas, lmin, lmax);
	}

	const cpl_size post_smoothing_radius = 11;

	//TODO: fill following params
	const hdrl_data_t fit_wrange = 4.0;
	hdrl_parameter* resp_par =
			hdrl_response_fit_parameter_create(post_smoothing_radius, fit_points,
					fit_wrange, NULL);

	hdrl_spectrum1D * std_s1d_e = extend_hdrl_spectrum(sci_wmin * um2nm,
			sci_wmax * um2nm, std_s1d);
	hdrl_spectrum1D * ext_s1d_e = extend_hdrl_spectrum(sci_wmin * um2nm,
			sci_wmax * um2nm, ext_s1d);

	cpl_msg_info(cpl_func,"compute response");
	hdrl_response_result * response =
			hdrl_response_compute(sci_s1d, std_s1d_e, ext_s1d_e, telluric_par,
					shift_par, resp_calc_par, resp_par);
        /*
    hdrl_spectrum1D_save(sci_s1d, "sci_s1d_resp.fits");
    hdrl_spectrum1D_save(std_s1d_e, "std_s1d_e_resp.fits");
    hdrl_spectrum1D_save(ext_s1d_e, "ext_s1d_e_resp.fits");
        */


	hdrl_spectrum1D_delete(&std_s1d_e);
	hdrl_spectrum1D_delete(&ext_s1d_e);
	/* extends response to same wavelength range as observed sci to flux-cal  */


	if(cpl_error_get_code() == CPL_ERROR_ILLEGAL_OUTPUT) {
		cpl_msg_error(cpl_func,"Problems computing response. Skip this part");
		cpl_error_set(cpl_func,CPL_ERROR_NONE);
	} else if(response != NULL && cpl_error_get_code() == CPL_ERROR_NONE){
		cpl_msg_debug(cpl_func,"size sci = %lld size resp = %lld",
				hdrl_spectrum1D_get_size(sci_s1d),
				hdrl_spectrum1D_get_size(response->final_response));
		const hdrl_spectrum1D * resp_s1d_smo =
				hdrl_response_result_get_final_response(response);

		const hdrl_spectrum1D * resp_s1d_raw =
				hdrl_response_result_get_raw_response(response);
		//cpl_msg_info(cpl_func,"wmin=%g wmax=%g",sci_wmin,sci_wmax);
		//hdrl_spectrum1D_save(resp_s1d_raw, "resp_s1d_raw.fits");
		cpl_table* tab=hdrl_spectrum1D_convert_to_table(resp_s1d_smo,
				"response", "wavelength", "response_error", "response_bpm");

		sci_wmin=cpl_table_get_column_min(tab, "wavelength");
		sci_wmax=cpl_table_get_column_max(tab, "wavelength");
		//cpl_msg_info(cpl_func, "wmin=%g wmax=%g", sci_wmin, sci_wmax);

		cpl_table* resp_smo = NULL;
		cpl_table* resp_raw = NULL;
		cpl_table* resp_pts = NULL;
		resp_smo = hdrl_spectrum1D_convert_to_table(resp_s1d_smo,
				"response_smo", "wavelength", "response_smo_error",
				"response_smo_bpm");

		resp_raw = hdrl_spectrum1D_convert_to_table(resp_s1d_raw,
				"response_raw", "wavelength", "response_raw_error",
				"response_raw_bpm");

		resp_pts = hdrl_spectrum1D_convert_to_table(
				hdrl_response_result_get_selected_response(response),
				"response_fit_pts", "wavelength", "response_fit_pts_error",
				"response_fit_pts_bpm");
		cpl_table_delete(tab);


		um2nm = 1e3;
		cpl_table_divide_scalar(resp_smo, "wavelength", um2nm);
		cpl_table_divide_scalar(resp_raw, "wavelength", um2nm);
		cpl_table_divide_scalar(resp_pts, "wavelength", um2nm);

		cpl_table* qclog_tbl = eris_qclog_init();
		cpl_frame* frm_resp_wind_qc = cpl_frameset_find(sof, ERIS_IFU_CALIB_RESPONSE_WINDOWS);

		const hdrl_spectrum1D * selected_points = hdrl_response_result_get_selected_response(response);
		const cpl_array * waves = hdrl_spectrum1D_get_wavelength(selected_points).wavelength;
		const double wmin_g = cpl_array_get_min(waves) / um2nm;
		const double wmax_g = cpl_array_get_max(waves) / um2nm;
		//cpl_msg_info(cpl_func, "good wavelengths wmin=%g wmax=%g", wmin_g, wmax_g);

		/* polynomial fit */
		int niter = 3;
		int deg = 0;
		double kappa = 3.0;
		//cpl_parameterlist_dump(config,stdout);
		const cpl_parameter* p;
		char* param_name;
		const char* context = "eris.eris_ifu_stdstar";

        eris_print_rec_status(100);
		param_name = cpl_sprintf("%s.response-polyfit-deg", context);
		p = cpl_parameterlist_find_const(config, param_name);
		deg = cpl_parameter_get_int(p);
		cpl_free(param_name);
		eris_print_rec_status(101);
		param_name = cpl_sprintf("%s.response-polyfit-ksigma-kappa", context);
	    p = cpl_parameterlist_find_const(config, param_name);
		kappa = cpl_parameter_get_double(p);
		cpl_free(param_name);
		eris_print_rec_status(102);
		param_name = cpl_sprintf("%s.response-polyfit-ksigma-niter", context);
	    p = cpl_parameterlist_find_const(config, param_name);
		niter = cpl_parameter_get_int(p);
		cpl_free(param_name);
		eris_print_rec_status(103);

		cpl_table* resp_poly =  eris_fit_poly_and_clip_response(resp_pts,
				resp_smo, deg, kappa, niter);
		eris_print_rec_status(104);
		if( resp_poly != NULL && frm_resp_wind_qc != NULL) {

			eris_resp_qc(resp_poly, frm_resp_wind_qc, "wavelength",
					"response_poly", wmin_g, wmax_g, &qclog_tbl);
			eris_print_rec_status(105);
		}
		eris_print_rec_status(106);
		char* fname = cpl_sprintf("%s_response.fits",plugin_id);
		eris_pro_save_tbl(resp_smo, ref_set, sof, fname,
				ERIS_IFU_PRO_JITTER_RESPONSE, qclog_tbl, plugin_id, config);

		cpl_table_save(resp_raw, NULL, NULL, fname, CPL_IO_EXTEND);
		cpl_table_save(resp_pts, NULL, NULL, fname, CPL_IO_EXTEND);
		cpl_table_save(resp_poly, NULL, NULL, fname, CPL_IO_EXTEND);

		cpl_free(fname);
		cpl_table_delete(resp_poly);
		cpl_table_delete(qclog_tbl);
		cpl_table_delete(resp_smo);
		cpl_table_delete(resp_raw);
		cpl_table_delete(resp_pts);

	}

	cpl_bivector_delete(quality_areas);
	cpl_bivector_delete(fit_areas);
	//cpl_bivector_delete(high_abs_regions);

	hdrl_parameter_delete(telluric_par);
	hdrl_parameter_delete(shift_par);
	hdrl_parameter_delete(resp_calc_par);
	hdrl_parameter_delete(resp_par);

	hdrl_spectrum1D_delete(&sci_s1d);
	hdrl_spectrum1D_delete(&std_s1d);
	hdrl_spectrum1D_delete(&ext_s1d);
	cpl_propertylist_delete(plist);
	hdrl_spectrum1Dlist_delete(telluric_models);

	eris_remove_table_normal_format();
	cpl_array_delete(fit_points);
	hdrl_response_result_delete(response);
	eris_check_error_code("eris_response_compute");
	if(cpl_error_get_code() == CPL_ERROR_ILLEGAL_OUTPUT) {
		/* occasionally, in long setting, response fail with CPL_ERROR_ILLEGAL_OUTPUT
                for the moment we exit.
                TODO: we should better handle those residual cases */
		cpl_error_set(cpl_func,CPL_ERROR_NONE);
		return CPL_ERROR_NONE;
	}
	return cpl_error_get_code();
}


//static void  hdrl_spectrum1D_save(const hdrl_spectrum1D * s, const char * fname){
//	if(s == NULL) return;
//	cpl_table * tb = hdrl_spectrum1D_convert_to_table(s, "FLX", "WLN", "FLX_E",
//			"FLX_BPM");
//	cpl_table_save(tb, NULL, NULL, fname, CPL_IO_CREATE);
//	cpl_table_delete(tb);
//}

cpl_error_code
eris_flux_calibrate_spectra(const char* pipefile_prefix, const char* plugin_id,
		const cpl_parameterlist* config, cpl_frameset* ref_set, cpl_frameset* sof){



	cpl_ensure_code(pipefile_prefix, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(plugin_id, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(config, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(ref_set, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(sof, CPL_ERROR_NULL_INPUT);

	cpl_size ref_set_sz = cpl_frameset_get_size(ref_set);
	cpl_size sof_sz = cpl_frameset_get_size(sof);
	cpl_ensure_code(ref_set_sz >0, CPL_ERROR_DATA_NOT_FOUND);
	cpl_ensure_code(sof_sz >0, CPL_ERROR_DATA_NOT_FOUND);


	/* Flux calibrate the spectrum */
	const char* fname;
    cpl_msg_info(cpl_func,"Flux calibrate the spectrum");
	hdrl_spectrum1D_wave_scale scale = hdrl_spectrum1D_wave_scale_linear;
	//cpl_frame* response=cpl_frameset_find(sof, PRO_RESPONSE);
	cpl_frame* spectra_frm = cpl_frameset_find(sof, ERIS_IFU_PRO_JITTER_SPECTRUM);
	fname = cpl_frame_get_filename(spectra_frm);


	//cpl_table* spectra_tab = cpl_table_load(fname, 1, 0);
	cpl_frame* frm_sci_dup = cpl_frame_duplicate(spectra_frm);
	cpl_table* spectra_tab = eris_ifu_table_from_sdp_to_normal_format(&frm_sci_dup);
	cpl_frame_delete(frm_sci_dup);
	char band[FILE_NAME_SZ];
	eris_get_band(spectra_frm, band);
	double bin_size = eris_get_dispersion(band);
	//cpl_msg_info(cpl_func, "bin_size=%g", bin_size);
	double um2nm = 1.e3;
	double nm2AA = 10;
	int status;
	double sci_wmin = 0;
	double sci_wmax = 0;
	double resp_wmin = 0;
	double resp_wmax = 0;

	cpl_size size = cpl_table_get_nrow(spectra_tab);
	cpl_size half_size = size / 2;
	double cdelt1 = cpl_table_get_double(spectra_tab, "WAVE", half_size, &status) -
			cpl_table_get_double(spectra_tab, "WAVE", (half_size - 1), &status);
	//cpl_table_delete(spectra_tab);
	bin_size = cdelt1;
	double bin_size_scale_factor = bin_size * um2nm * nm2AA;

	sci_wmin = cpl_table_get_column_min(spectra_tab,"WAVE");
	sci_wmax = cpl_table_get_column_max(spectra_tab,"WAVE");
	cpl_msg_debug(cpl_func,"Sci table wmin: %f wmax: %f",sci_wmin, sci_wmax);

	cpl_propertylist* plist = cpl_propertylist_load(fname,0);
	double dit = cpl_propertylist_get_double(plist, FHDR_E_DIT);

	//int ndit = eris_pfits_get_ndit(plist);
	double exptime = dit; // for NIR (SINFONI data) we use only DIT
	double gain = ERIS_GAIN; // eris_pfits_get_gain(plist);


	double airmass = eris_ifu_get_airmass_mean(plist);
    cpl_msg_info(cpl_func,"airmass: %g", airmass);
    cpl_propertylist_delete(plist);

    double factor = 1. / (gain * exptime * bin_size_scale_factor);

    /* we load response table. As response is computed in low resolution but
     * may be applied also to high resolution bands, we need to trim first the
     * response to the same range of the science spectrum, and in case the
     * science spectrum is defined over a wider range than the response we also
     * trim the observed spectrum so that both the observed spectrum and the
     * response have the same wavelength ranges
     */
    cpl_frame* resp_frm = cpl_frameset_find(sof, ERIS_IFU_PRO_JITTER_RESPONSE);
    fname = cpl_frame_get_filename(resp_frm);
    cpl_size next = cpl_frame_get_nextensions(resp_frm);
    cpl_table* resp_tab = NULL;
    if(next == 4) {
       resp_tab = cpl_table_load(fname, 4, 0);
    } else {
       resp_tab = cpl_table_load(fname, 1, 0);
    }
    resp_wmin = cpl_table_get_column_min(resp_tab,"wavelength");
    resp_wmax = cpl_table_get_column_max(resp_tab,"wavelength");
    cpl_msg_debug(cpl_func,"Response table wmin: %f wmax: %f",resp_wmin, resp_wmax);
    //cpl_table_save(resp_tab,NULL,NULL,"resp_tab_before.fits",CPL_IO_DEFAULT);
    cpl_table_and_selected_double(resp_tab, "wavelength", CPL_NOT_LESS_THAN, sci_wmin);
    cpl_table_and_selected_double(resp_tab, "wavelength", CPL_NOT_GREATER_THAN, sci_wmax);
    cpl_table* resp_xtab = cpl_table_extract_selected(resp_tab);
    //cpl_table_save(resp_xtab,NULL,NULL,"resp_tab_after.fits",CPL_IO_DEFAULT);

    /* trim also observed spectrum to same range as the one of the response */
    //cpl_table_save(spectra_tab,NULL,NULL,"sci_tab_before.fits",CPL_IO_DEFAULT);
    resp_wmin = cpl_table_get_column_min(resp_xtab,"wavelength");
    resp_wmax = cpl_table_get_column_max(resp_xtab,"wavelength");
    cpl_msg_info(cpl_func,"Response table after selection wmin: %f wmax: %f",resp_wmin, resp_wmax);
    cpl_table_and_selected_double(spectra_tab, "WAVE", CPL_NOT_LESS_THAN, resp_wmin);
    cpl_table_and_selected_double(spectra_tab, "WAVE", CPL_NOT_GREATER_THAN, resp_wmax);
    cpl_table* spectra_xtab = cpl_table_extract_selected(spectra_tab);
    //cpl_table_save(spectra_xtab,NULL,NULL,"sci_tab_after.fits",CPL_IO_DEFAULT);
    sci_wmin = cpl_table_get_column_min(spectra_xtab,"WAVE");
    sci_wmax = cpl_table_get_column_max(spectra_xtab,"WAVE");
    cpl_msg_debug(cpl_func,"Sci table after selection wmin: %f wmax: %f",sci_wmin, sci_wmax);
	hdrl_spectrum1D * obs_s1d =
			hdrl_spectrum1D_convert_from_table(spectra_xtab, "FLUX",
					"WAVE", "ERR", NULL, scale);
	cpl_table_delete(spectra_xtab);
	//hdrl_spectrum1D_save(obs_s1d, "obs_s1d_spct.fits");


	//cpl_msg_info(cpl_func,"Response table wmin: %g",cpl_table_get_column_min(resp_tab,"wavelength"));
	//cpl_msg_info(cpl_func,"Response table wmax: %g",cpl_table_get_column_max(resp_tab,"wavelength"));
    /* As response curve may have a slightly different wavelength span than the observed 
       spectrum, we extract the relevant common region from the response table */

	hdrl_spectrum1D * resp_s1d = NULL;
	if(next == 4) {
       resp_s1d = hdrl_spectrum1D_convert_from_table(resp_xtab, "response_poly",
				"wavelength", "response_poly_error", "response_poly_bpm", scale);
	} else {
	   resp_s1d = hdrl_spectrum1D_convert_from_table(resp_xtab, "response_smo",
				"wavelength", "response_smo_error", "response_smo_bpm", scale);
	}
    //hdrl_spectrum1D_save(resp_s1d, "resp_s1d_spct.fits");
    hdrl_spectrum1D * resp_s1d_e = extend_hdrl_spectrum(sci_wmin, sci_wmax, resp_s1d);
    hdrl_spectrum1D * obs_s1d_e = extend_hdrl_spectrum(resp_wmin, resp_wmax, obs_s1d);
    //hdrl_spectrum1D_save(obs_s1d_e, "obs_s1d_e_spct.fits");

    hdrl_parameter * res_par =
    		hdrl_spectrum1D_resample_interpolate_parameter_create(hdrl_spectrum1D_interp_akima);

    hdrl_spectrum1D_wavelength wav = hdrl_spectrum1D_get_wavelength(obs_s1d_e);

    hdrl_spectrum1D * resp_s1d_r = hdrl_spectrum1D_resample(resp_s1d_e, &wav, res_par);
    //hdrl_spectrum1D_save(resp_s1d_r, "resp_s1d_r_spct.fits");
    cpl_msg_debug(cpl_func,"obs_e wmin: %f wmax: %f", cpl_array_get_min(hdrl_spectrum1D_get_wavelength(obs_s1d_e).wavelength), cpl_array_get_max(hdrl_spectrum1D_get_wavelength(obs_s1d_e).wavelength));
    cpl_msg_debug(cpl_func,"rsp_r wmin: %f wmax: %f", cpl_array_get_min(hdrl_spectrum1D_get_wavelength(resp_s1d_r).wavelength), cpl_array_get_max(hdrl_spectrum1D_get_wavelength(resp_s1d_r).wavelength));

    /* now response and observed spectrum have same range and sampling step
     * we can multiply one by the other
     */


	cpl_frame* atm_ext_frm = cpl_frameset_find(sof, ERIS_IFU_CALIB_EXTCOEFF_TABLE);
	fname = cpl_frame_get_filename(atm_ext_frm);
	cpl_table* atm_ext_tab = cpl_table_load(fname,1,0);

    hdrl_spectrum1D * ext_s1d =
				hdrl_spectrum1D_convert_from_table(atm_ext_tab, "EXTINCTION",
						"LAMBDA", NULL, NULL, scale);
    //hdrl_spectrum1D_save(ext_s1d, "ext_s1d_spct.fits");
    hdrl_spectrum1D * ext_s1d_e = extend_hdrl_spectrum(sci_wmin * um2nm, sci_wmax * um2nm, ext_s1d);
    //hdrl_spectrum1D_save(ext_s1d_e, "ext_s1d_e.fits");


    //hdrl_spectrum1D_save(obs_s1d_e, "obs_s1d_e.fits");
    //hdrl_spectrum1D_save(resp_s1d_e, "resp_s1d_e_spct.fits");
    hdrl_spectrum1D* flux_s1d = hdrl_spectrum1D_mul_spectrum_create(obs_s1d_e, resp_s1d_r);

    //hdrl_spectrum1D_save(flux_s1d, "flux_s1d_0_spct.fits");
    hdrl_spectrum1D_mul_scalar(flux_s1d,(hdrl_value){factor,0.});

    //hdrl_spectrum1D_save(flux_s1d, "flux_s1d_01_spct.fits");

    double* pflux = cpl_image_get_data_double(hdrl_image_get_image(flux_s1d->flux));
    double* perr = cpl_image_get_data_double(hdrl_image_get_error(flux_s1d->flux));
    int sx = hdrl_image_get_size_x(flux_s1d->flux);


    double* pext = cpl_image_get_data_double(hdrl_image_get_image(ext_s1d->flux));
    cpl_msg_warning(cpl_func,"sx: %d",sx);
    for(int i = 0; i < sx; i++) {
    	double exp = -0.4 * airmass * pext[i];
    	pflux[i] *= pow(10., exp);
    	//cpl_msg_warning(cpl_func,"flux: %g err: %g corr: %g",pflux[i], perr[i], pow(10., exp));
    	perr[i] *= pow(10., exp);
    }
    //hdrl_spectrum1D_save(flux_s1d, "flux_s1d_1_spct.fits");

    cpl_table* flux_tab = hdrl_spectrum1D_convert_to_table(flux_s1d,
    		"flux", "wavelength", "flux_error", NULL);
    //"flux", "wavelength", "flux_error", "flux_bpm");
    //cpl_table_save(flux_tab,NULL,NULL,"spectrum_flux_spct.fits",CPL_IO_DEFAULT);

    cpl_table* qclog_tbl = eris_qclog_init();
    char* file_name = cpl_sprintf("%s_spectrum_fluxcal",pipefile_prefix);
    eris_pro_save_tbl(flux_tab, ref_set, sof, file_name,
			ERIS_IFU_PRO_JITTER_SPECTRUM_FLUXCAL, qclog_tbl, plugin_id, config);
    cpl_free(file_name);
    cpl_table_delete(qclog_tbl);
    eris_remove_table_normal_format();

    hdrl_spectrum1D_delete(&resp_s1d_r);
    hdrl_spectrum1D_delete(&obs_s1d_e);
    hdrl_spectrum1D_delete(&obs_s1d);
    hdrl_spectrum1D_delete(&resp_s1d);
    hdrl_spectrum1D_delete(&resp_s1d_e);
    hdrl_spectrum1D_delete(&ext_s1d);
    hdrl_spectrum1D_delete(&ext_s1d_e);
    hdrl_spectrum1D_delete(&flux_s1d);
    cpl_table_delete(atm_ext_tab);
    cpl_table_delete(flux_tab);
    cpl_table_delete(spectra_tab);
	cpl_table_delete(resp_tab);
	cpl_table_delete(resp_xtab);
	hdrl_parameter_delete(res_par);
	eris_check_error_code("eris_flux_calibrate_spectra");

	return cpl_error_get_code();
}

//static cpl_error_code
//eris_ifu_apply_response_to_cube(hdrl_imagelist* hlist,
//		hdrl_spectrum1D* resp_s1d_r, hdrl_spectrum1D* resp_s1d_e,
//		hdrl_spectrum1D * ext_s1d, const double airm, const double factor)
//{


//	//hdrl_spectrum1D_save(ext_s1d, "ext_s1d_cube.fits");
//	double* presp = cpl_image_get_data_double(hdrl_image_get_image(resp_s1d_r->flux));
//	double* pext = cpl_image_get_data_double(hdrl_image_get_image(ext_s1d->flux));
//	cpl_size sx = hdrl_imagelist_get_size_x(hlist);
//	cpl_size sy = hdrl_imagelist_get_size_y(hlist);
//	cpl_size sz = hdrl_imagelist_get_size(hlist);

//	int nx=hdrl_image_get_size_x(resp_s1d_e->flux);
//	cpl_msg_warning(cpl_func,"nx: %d sz: %lld",nx, sz);
//	cpl_size kmax = (nx < sz) ? nx : sz;
//	kmax = sz;
//	double resp=0;
//	hdrl_value resp_fct;
//	hdrl_value factor_value = {factor, 0};

//	cpl_msg_warning(cpl_func,"kmax: %lld",kmax);
//	for (cpl_size k = 0; k < kmax; k++) {
//		double exp = -0.4 * airm * pext[k];

//		hdrl_image* hima = hdrl_imagelist_get(hlist, k);
//		hdrl_value atm_ext = {pow(10., exp), 0.};
//		resp = presp[k];
//		resp_fct.data = resp;
//		//if(isfinite(resp) && resp != 0) {
//	    if(isfinite(resp) && resp != 0) {
//			hdrl_image_mul_scalar(hima, resp_fct);
//		} else if(resp == 0) {
//			hdrl_image_mul_scalar(hima, resp_fct);
//			/* Flag bad pixel in result */
//			//cpl_msg_warning(cpl_func,"flag bad plane: %lld",k);
//			for(cpl_size j = 1; j <= sy; j++ ) {
//				for(cpl_size i = 1; i <= sx; i++ ) {
//					hdrl_image_reject(hima, i, j);
//				}
//			}
//		} else if(!isfinite(resp)) {
//			// Flag bad pixel in result
//			//cpl_msg_warning(cpl_func,"flag bad plane: %lld",k);
//			for(cpl_size j = 1; j <= sy; j++ ) {
//				for(cpl_size i = 1; i <= sx; i++ ) {
//					hdrl_image_reject(hima, i, j);
//				}
//			}

//		}

//		hdrl_image_mul_scalar(hima, atm_ext);
//		hdrl_image_mul_scalar(hima, factor_value);
//	}

//	eris_check_error_code("eris_ifu_apply_response_to_cube");
//	return cpl_error_get_code();
//}

//TODO Here the wmin/wmax should be taken from the cube and not from the extracted spectrum.
cpl_error_code
eris_flux_calibrate_cube2(const char* procatg, const char* pipefile_prefix,
		const char* plugin_id, const cpl_parameterlist* parlist,
        cpl_frameset* sof){

	cpl_ensure_code(procatg, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(pipefile_prefix, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(plugin_id, CPL_ERROR_NULL_INPUT);

	cpl_ensure_code(parlist, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(sof, CPL_ERROR_NULL_INPUT);

    cpl_size sof_sz = cpl_frameset_get_size(sof);
	cpl_ensure_code(sof_sz >0, CPL_ERROR_DATA_NOT_FOUND);




	cpl_msg_info(cpl_func,"Flux calibrate the cube %s", procatg);

	const char* fname;
	hdrl_spectrum1D_wave_scale scale = hdrl_spectrum1D_wave_scale_linear;
	/* load input science cube frame */
	cpl_frame* coadd_frm = cpl_frameset_find(sof, procatg);
	fname = cpl_frame_get_filename(coadd_frm);
	cpl_msg_debug(cpl_func, "fname=%s",fname);
	cpl_imagelist* cube = cpl_imagelist_load(fname, CPL_TYPE_FLOAT,1);
	cpl_imagelist* errs = cpl_imagelist_load(fname, CPL_TYPE_FLOAT,2);
	cpl_propertylist* phead = cpl_propertylist_load(fname, 0);
	hdrl_imagelist* hlist = hdrl_imagelist_create(cube, errs);
	cpl_imagelist_delete(cube);
	cpl_imagelist_delete(errs);
	double airm = eris_ifu_get_airmass_mean(phead);

    /* define bin scale factor */
	char band[FILE_NAME_SZ];
	eris_get_band(coadd_frm, band);
	double bin_size = eris_get_dispersion(band);
	cpl_msg_debug(cpl_func, "airm=%g", airm);
	double um2nm = 1.e3;
	double nm2AA = 10;

	double sci_wmin = 0;
	double sci_wmax = 0;
	double resp_wmin = 0;
	double resp_wmax = 0;
	cpl_propertylist* data_head = cpl_propertylist_load(fname, 1);
	cpl_propertylist*  head_wcs = eris_ifu_plist_extract_wcs(data_head);
	cpl_propertylist_delete(data_head);
	cpl_size size = hdrl_imagelist_get_size(hlist);
	double cdelt3 = cpl_propertylist_get_double(head_wcs, "CD3_3");
	double crval3 = cpl_propertylist_get_double(head_wcs, "CRVAL3");
    bin_size = cdelt3;
	double bin_size_scale_factor = bin_size * um2nm * nm2AA;
	double dit = cpl_propertylist_get_double(phead, FHDR_E_DIT);
    int ndit = cpl_propertylist_get_int(phead, "ESO DET NDIT");
    double exptime = dit * ndit;
    cpl_msg_debug(cpl_func, "exptime=%g", exptime);  //TODO: DIT OR DIT * NDIT??
    double gain = ERIS_GAIN;   //SINFONI_GAIN; // eris_pfits_get_gain(plist);
    cpl_msg_debug(cpl_func, "gain=%g", gain);

    //cpl_propertylist_delete(phead);

    double factor = 1. / (gain * exptime * bin_size_scale_factor);
	sci_wmin = crval3;
	sci_wmax = crval3+cdelt3*(size-1);
	cpl_msg_warning(cpl_func,"Sci table wmin: %f wmax: %f",sci_wmin, sci_wmax);


	/* we load response table. As response is computed in low resolution but
	 * may be applied also to high resolution bands, we need to trim first the
	 * response to the same range of the science spectrum, and in case the
	 * science spectrum is defined over a wider range than the response we also
	 * trim the observed spectrum so that both the observed spectrum and the
	 * response have the same wavelength ranges
	 */
	cpl_frame* resp_frm = cpl_frameset_find(sof, ERIS_IFU_PRO_JITTER_RESPONSE);
	cpl_size next = cpl_frame_get_nextensions(resp_frm);
	fname = cpl_frame_get_filename(resp_frm);
	cpl_table* resp_tab = NULL;
	if(next == 4) {
		resp_tab = cpl_table_load(fname, 4, 0);
	} else {
		resp_tab = cpl_table_load(fname, 1, 0);
	}

	resp_wmin = cpl_table_get_column_min(resp_tab,"wavelength");
	resp_wmax = cpl_table_get_column_max(resp_tab,"wavelength");
	cpl_msg_warning(cpl_func,"Response table wmin: %f wmax: %f",resp_wmin, resp_wmax);
	//cpl_table_save(resp_tab,NULL,NULL,"resp_tab_before.fits",CPL_IO_DEFAULT);
	cpl_table_and_selected_double(resp_tab, "wavelength", CPL_NOT_LESS_THAN, sci_wmin);
	cpl_table_and_selected_double(resp_tab, "wavelength", CPL_NOT_GREATER_THAN, sci_wmax);
	cpl_table* resp_xtab = cpl_table_extract_selected(resp_tab);
	//cpl_table_save(resp_xtab,NULL,NULL,"resp_tab_after.fits",CPL_IO_DEFAULT);

	/* trim also observed spectrum to same range as the one of the response */
	//cpl_table_save(spectra_tab,NULL,NULL,"sci_tab_before.fits",CPL_IO_DEFAULT);
	resp_wmin = cpl_table_get_column_min(resp_xtab,"wavelength");
	resp_wmax = cpl_table_get_column_max(resp_xtab,"wavelength");
	cpl_msg_info(cpl_func,"Response table after selection wmin: %f wmax: %f",resp_wmin, resp_wmax);

	/* FUNCTION TO TRIM CUBE OVER SAME RANGE AS ONE OF RESPONSE */
	cpl_size klow_skip = 0;
	klow_skip = (resp_wmin > sci_wmin) ? (resp_wmin - sci_wmin) / cdelt3 : 0;
	cpl_size kupp_skip = 0;
	kupp_skip = (sci_wmax > resp_wmax) ? (sci_wmax - resp_wmax) / cdelt3 : 0;
	cpl_msg_info(cpl_func,"klow_skip: %lld kupp_skip: %lld",klow_skip, kupp_skip);
	cpl_size z_min = klow_skip;
	cpl_size z_max = size - kupp_skip;
	cpl_msg_info(cpl_func,"z_min: %lld: z_max: %lld",z_min, z_max);
	cpl_imagelist* bpm = cpl_imagelist_new();
	
	for(cpl_size k=0; k< size; k++) {
		//cpl_msg_info(cpl_func,"k=%lld",k);
		cpl_imagelist_set(bpm, hdrl_image_get_image(hdrl_imagelist_get(hlist,k)), k);
	}

	//cpl_propertylist* xhead = cpl_propertylist_load(fname, 1);
	cpl_propertylist_append_int(head_wcs,"NAXIS3",size);
	eris_ifu_cube_trim_nans(z_min, z_max,&hlist, &bpm, head_wcs);
    //cpl_imagelist_delete(bpm);
	/* updated sci_wmin, sci_wmax */
	sci_wmin = crval3;
	sci_wmax = crval3+cdelt3*(size-1);
	cpl_msg_warning(cpl_func,"Sci table wmin: %f wmax: %f",sci_wmin, sci_wmax);

	hdrl_spectrum1D * resp_s1d = NULL;
	if(next == 4) {
		resp_s1d = hdrl_spectrum1D_convert_from_table(resp_xtab, "response_poly",
					"wavelength", "response_poly_error", "response_poly_bpm", scale);
	} else {
		resp_s1d = hdrl_spectrum1D_convert_from_table(resp_xtab, "response_smo",
					"wavelength", "response_smo_error", "response_smo_bpm", scale);
	}
	cpl_table_delete(resp_xtab);
	//hdrl_spectrum1D_save(resp_s1d, "resp_s1d_cube.fits");
	hdrl_spectrum1D * resp_s1d_e = extend_hdrl_spectrum(sci_wmin, sci_wmax, resp_s1d);
	//hdrl_spectrum1D_save(resp_s1d_e, "resp_s1d_e_cube.fits");

	/* creates a wavelength sampling array to re-sample the response properly
	 * on the same range and with same sampling as the science data cube */
	hdrl_parameter * res_par =
			hdrl_spectrum1D_resample_interpolate_parameter_create(hdrl_spectrum1D_interp_akima);
	
	 cpl_frame* spectra_frm = NULL;

	if( NULL == (spectra_frm = cpl_frameset_find(sof, ERIS_IFU_PRO_JITTER_SPECTRUM))){
		cpl_msg_error(cpl_func,"To flux calibrate a data cube you need to set extract-source=TRUE");
		cpl_msg_error(cpl_func,"as the recipe uses the min/max wavelength of the extracted spectrum");
		cpl_msg_error(cpl_func,"to determine the range where to flux calibrate the data cube.");
		eris_check_error_code("eris_flux_calibrate_cube2");

		hdrl_spectrum1D_delete(&resp_s1d);
		hdrl_spectrum1D_delete(&resp_s1d_e);
		cpl_table_delete(resp_tab);
		hdrl_parameter_delete(res_par);
		hdrl_imagelist_delete(hlist);
		cpl_propertylist_delete(head_wcs);
		//cpl_imagelist_delete(bpm);
		cpl_propertylist_delete(phead);

		return cpl_error_get_code();
	}
	
	fname = cpl_frame_get_filename(spectra_frm);

	//cpl_table* spectra_tab = cpl_table_load(fname, 1, 0);
	cpl_frame* frm_sci_dup = cpl_frame_duplicate(spectra_frm);
	cpl_table* spectra_tab = eris_ifu_table_from_sdp_to_normal_format(&frm_sci_dup);
	cpl_frame_delete(frm_sci_dup);

	cpl_table_and_selected_double(spectra_tab, "WAVE", CPL_NOT_LESS_THAN, sci_wmin);
	cpl_table_and_selected_double(spectra_tab, "WAVE", CPL_NOT_GREATER_THAN, sci_wmax);
	cpl_table* spectra_xtab = cpl_table_extract_selected(spectra_tab);
	cpl_table_delete(spectra_tab);
	//cpl_table_save(spectra_xtab,NULL,NULL,"sci_tab_after.fits",CPL_IO_DEFAULT);
	sci_wmin = cpl_table_get_column_min(spectra_xtab,"WAVE");
	sci_wmax = cpl_table_get_column_max(spectra_xtab,"WAVE");
	cpl_msg_debug(cpl_func,"Sci table after selection wmin: %f wmax: %f",sci_wmin, sci_wmax);
	hdrl_spectrum1D * obs_s1d = hdrl_spectrum1D_convert_from_table(spectra_xtab, "FLUX",
					"WAVE", NULL, NULL, scale);
	cpl_table_delete(spectra_xtab);

	hdrl_spectrum1D_wavelength wav = hdrl_spectrum1D_get_wavelength(obs_s1d);
	hdrl_spectrum1D * resp_s1d_r = hdrl_spectrum1D_resample(resp_s1d_e, &wav, res_par);
	//hdrl_spectrum1D_save(resp_s1d_r, "resp_s1d_r_cube.fits");
	cpl_msg_debug(cpl_func,"obs_e wmin: %f wmax: %f", sci_wmin, sci_wmax);
	//cpl_msg_debug(cpl_func,"rsp_r wmin: %f wmax: %f", cpl_array_get_min(hdrl_spectrum1D_get_wavelength(resp_s1d_r).wavelength), cpl_array_get_max(hdrl_spectrum1D_get_wavelength(resp_s1d_r).wavelength));
	
	/* now response and observed spectrum have same range and sampling step
	 * we can multiply one by the other
	 */


	cpl_frame* atm_ext_frm = cpl_frameset_find(sof, ERIS_IFU_CALIB_EXTCOEFF_TABLE);
	fname = cpl_frame_get_filename(atm_ext_frm);
	cpl_table* atm_ext_tab = cpl_table_load(fname,1,0);

	hdrl_spectrum1D * ext_s1d =
			hdrl_spectrum1D_convert_from_table(atm_ext_tab, "EXTINCTION",
					"LAMBDA", NULL, NULL, scale);
	//hdrl_spectrum1D_save(ext_s1d, "ext_s1d_cube.fits");
    /*
	eris_ifu_apply_response_to_cube(hlist, resp_s1d_r, resp_s1d_e, ext_s1d,
			airm, factor);
			*/



	double* presp = cpl_image_get_data_double(hdrl_image_get_image(resp_s1d_r->flux));
	double* pext = cpl_image_get_data_double(hdrl_image_get_image(ext_s1d->flux));
	cpl_size sx = hdrl_imagelist_get_size_x(hlist);
	cpl_size sy = hdrl_imagelist_get_size_y(hlist);
	cpl_size sz = hdrl_imagelist_get_size(hlist);

	int nx=hdrl_image_get_size_x(resp_s1d_e->flux);
	cpl_msg_warning(cpl_func,"nx: %d sz: %lld",nx, sz);
	cpl_size kmax = (nx < sz) ? nx : sz;
	kmax = sz;
	double resp=0;
	hdrl_value resp_fct;
	hdrl_value factor_value = {factor, 0};
	
	cpl_msg_warning(cpl_func,"kmax: %lld",kmax);
	cpl_size resp_sz = hdrl_image_get_size_x(resp_s1d_r->flux);
	for (cpl_size k = 0; k < kmax; k++) {
		double exp = -0.4 * airm * pext[k];

		hdrl_image* hima = hdrl_imagelist_get(hlist, k);
		hdrl_value atm_ext = {pow(10., exp), 0.};
		if(k < resp_sz) {
		    resp = presp[k];
		}
		resp_fct.data = resp;
		//if(isfinite(resp) && resp != 0) {
	    if(isfinite(resp) && resp != 0) {
			hdrl_image_mul_scalar(hima, resp_fct);
		} else if(resp == 0) {
			hdrl_image_mul_scalar(hima, resp_fct);
			// Flag bad pixel in result
			//cpl_msg_warning(cpl_func,"flag bad plane: %lld",k);
			for(cpl_size j = 1; j <= sy; j++ ) {
				for(cpl_size i = 1; i <= sx; i++ ) {
					hdrl_image_reject(hima, i, j);
				}
			}
		} else if(!isfinite(resp)) {
			// Flag bad pixel in result
			//cpl_msg_warning(cpl_func,"flag bad plane: %lld",k);
			for(cpl_size j = 1; j <= sy; j++ ) {
				for(cpl_size i = 1; i <= sx; i++ ) {
					hdrl_image_reject(hima, i, j);
				}
			}

		}

		hdrl_image_mul_scalar(hima, atm_ext);
		hdrl_image_mul_scalar(hima, factor_value);
	}



	/* save results */
	char* fcal_cube_ptag = cpl_sprintf("%s_FLUXCAL", procatg);
	char* cube_file_name = cpl_sprintf("%s_cube_fluxcal.fits", pipefile_prefix);
	//cpl_error_set(cpl_func, CPL_ERROR_ILLEGAL_INPUT);
	//if(cpl_error_get_code() != CPL_ERROR_NONE) exit(0);
	cpl_propertylist_set_string(phead, FHDR_PRO_CATG, fcal_cube_ptag);

	hdrl_resample_result* res = cpl_calloc(1, sizeof(hdrl_resample_result));
	//res->himlist = hdrl_imagelist_duplicate(hlist);
	res->himlist = hlist;
	res->header = cpl_propertylist_duplicate(phead);
	cpl_propertylist_delete(phead);
	cpl_propertylist_append(res->header, head_wcs);
	cpl_propertylist_delete(head_wcs);
	cpl_propertylist_update_string(res->header, "BUNIT", UNIT_FLUXCAL);


	/* collapse data cube */
	
	cpl_propertylist* phead2D = cpl_propertylist_duplicate(res->header);
	eris_ifu_plist_erase_wcs3D(phead2D);
	char* mean_cube_pctg = cpl_sprintf("%s_%s",fcal_cube_ptag,"MEAN");
	const char *filenameSpec ="cube_fluxcal";
	char* ima_file_name = cpl_sprintf("%s_%s_mean.fits", pipefile_prefix, filenameSpec);
	cpl_propertylist_append_string(phead2D, "PRODCATG", PRODCATG_WHITELIGHT);
	cpl_propertylist_update_string(phead2D, "BUNIT", UNIT_FLUXCAL);
	hdrl_image* cube_collapsed = NULL;
	cpl_image* cube_cmap = NULL;
	//TODO: AMO checking why collapsed mean of flux calibrated result is full of NANs.
	//hdrl_value fct = {1000000.,0};
	//hdrl_imagelist_mul_scalar(res->himlist, fct);
	hdrl_imagelist_collapse_mean(res->himlist, &cube_collapsed, &cube_cmap);
	cpl_propertylist_update_string(phead2D, CPL_DFS_PRO_CATG, mean_cube_pctg);
	cpl_image* mask_ima = cpl_image_new_from_mask(hdrl_image_get_mask(cube_collapsed));
	eris_ifu_save_deq_image(sof, NULL, parlist,sof, NULL,
			plugin_id, phead2D, "RADECSYS", ima_file_name,
			hdrl_image_get_image(cube_collapsed),
			hdrl_image_get_error(cube_collapsed),
			rmse, mask_ima, flag16bit, UNIT_FLUXCAL);
	cpl_image_delete(mask_ima);
	cpl_free(mean_cube_pctg);
	cpl_free(ima_file_name);
	cpl_propertylist_delete(phead2D);

	if(strcmp(plugin_id,"eris_ifu_jitter") == 0) {
		char* param_name = cpl_sprintf("%s.crea-phase3", "eris.eris_ifu_jitter");
		const cpl_parameter* p = cpl_parameterlist_find_const(parlist, param_name);

		cpl_boolean gen_phase3 = cpl_parameter_get_bool(p);
		if(gen_phase3) {
			cpl_msg_info(cpl_func,"Save flux calibrated COADDED OBJECT cube with phase3 data format");
			eris_ifu_resample_save_cube(res, fcal_cube_ptag, plugin_id,
					cube_file_name, parlist, sof, CPL_TRUE);
		} else {
			cpl_msg_info(cpl_func,"Save flux calibrated COADDED OBJECT cube");
			eris_ifu_resample_save_cube(res, fcal_cube_ptag, plugin_id,
					cube_file_name, parlist, sof, CPL_FALSE);
		}
		cpl_free(param_name);
	} else {
		cpl_msg_info(cpl_func,"Save flux calibrated STD STAR cube");
		eris_ifu_resample_save_cube(res, fcal_cube_ptag, plugin_id,
				cube_file_name, parlist, sof, CPL_FALSE);
	}

	cpl_free(cube_file_name);

	cpl_table* qclog = eris_qclog_init();
	hdrl_image_delete(cube_collapsed);
	cpl_image_delete(cube_cmap);

	cpl_table_delete(qclog);


	cpl_free(fcal_cube_ptag);
	hdrl_spectrum1D_delete(&resp_s1d);
	hdrl_spectrum1D_delete(&ext_s1d);
	hdrl_spectrum1D_delete(&resp_s1d_e);
	hdrl_spectrum1D_delete(&obs_s1d);
	hdrl_spectrum1D_delete(&resp_s1d_r);
	cpl_table_delete(atm_ext_tab);
	cpl_table_delete(resp_tab);
	eris_remove_table_normal_format();
	hdrl_parameter_delete(res_par);
	hdrl_imagelist_delete(hlist);
	//hdrl_resample_result_delete(res);
	cpl_free(res);
	eris_check_error_code("eris_flux_calibrate_cube2");

	return cpl_error_get_code();
}

//TODO: missing error and bad pixel propagation
cpl_error_code
eris_flux_calibrate_cube(const char* procatg, const char* pipefile_prefix,
		const char* plugin_id, const cpl_parameterlist* parlist,
        /*cpl_frameset* ref_set, */cpl_frameset* sof){


	cpl_ensure_code(procatg, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(pipefile_prefix, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(plugin_id, CPL_ERROR_NULL_INPUT);

	cpl_ensure_code(parlist, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(sof, CPL_ERROR_NULL_INPUT);

	cpl_size sof_sz = cpl_frameset_get_size(sof);
	cpl_ensure_code(sof_sz >0, CPL_ERROR_DATA_NOT_FOUND);

	cpl_msg_info(cpl_func,"Flux calibrate the cube %s", procatg);
	// Flux calibrate the spectrum
	const char* fname;
	hdrl_spectrum1D_wave_scale scale = hdrl_spectrum1D_wave_scale_linear;

	cpl_frame* coadd_frm = cpl_frameset_find(sof, procatg);
	fname = cpl_frame_get_filename(coadd_frm);
	cpl_msg_debug(cpl_func, "fname=%s",fname);
	cpl_imagelist* cube = cpl_imagelist_load(fname, CPL_TYPE_FLOAT,1);
	cpl_imagelist* errs = cpl_imagelist_load(fname, CPL_TYPE_FLOAT,2);
	cpl_propertylist* phead = cpl_propertylist_load(fname, 0);
	hdrl_imagelist* hlist = hdrl_imagelist_create(cube, errs);
    cpl_imagelist_delete(cube);
    cpl_imagelist_delete(errs);
	double dit = cpl_propertylist_get_double(phead, FHDR_E_DIT);
	int ndit = cpl_propertylist_get_int(phead, "ESO DET NDIT");

	double centralLambda = 0;
	double dis = 0;
	double centralpix = 0;
	double newcenter_x = 0;
	double newcenter_y = 0;

	cpl_propertylist* data_head = cpl_propertylist_load(fname, 1);

	eris_get_wcs_cube(data_head, &centralLambda, &dis, &centralpix, &newcenter_x,
			&newcenter_y);
	cpl_propertylist*  head_wcs = eris_ifu_plist_extract_wcs(data_head);
	cpl_propertylist_delete(data_head);

	double exptime = dit * ndit;
	cpl_msg_debug(cpl_func, "exptime=%g", exptime);  //TODO: DIT OR DIT * NDIT??
	double gain = ERIS_GAIN;   //SINFONI_GAIN; // eris_pfits_get_gain(plist);
	cpl_msg_debug(cpl_func, "gain=%g", gain);

	double airm = eris_ifu_get_airmass_mean(phead);
	cpl_msg_debug(cpl_func, "airm=%g", airm);
	double um2nm = 1000.;
	double nm2AA = 10.;
	char band[FILE_NAME_SZ];

	eris_get_band(coadd_frm,band);

	double bin_size = eris_get_dispersion(band);

	double bin_size_scale_factor = bin_size * um2nm * nm2AA;
	hdrl_value factor = {1. / (gain * exptime * bin_size_scale_factor), 0};
	cpl_msg_debug(cpl_func, "factor=%g", factor.data);
	cpl_frame* resp_frm = cpl_frameset_find(sof, ERIS_IFU_PRO_JITTER_RESPONSE);
	fname = cpl_frame_get_filename(resp_frm);
	cpl_table* resp_tab = cpl_table_load(fname,1,0);

	hdrl_spectrum1D * resp_s1d =
			hdrl_spectrum1D_convert_from_table(resp_tab, "response_smo",
					"wavelength", "response_smo_error", "response_smo_bpm", scale);
	//hdrl_spectrum1D_save(resp_s1d, "resp_s1d_cube.fits");

	cpl_frame* atm_ext_frm = cpl_frameset_find(sof, ERIS_IFU_CALIB_EXTCOEFF_TABLE);
	fname = cpl_frame_get_filename(atm_ext_frm);
	cpl_table* atm_ext_tab = cpl_table_load(fname, 1, 0);

	hdrl_spectrum1D * ext_s1d =
			hdrl_spectrum1D_convert_from_table(atm_ext_tab, "EXTINCTION",
					"LAMBDA", NULL, NULL, scale);
	//hdrl_spectrum1D_save(ext_s1d, "ext_s1d_cube.fits");

	hdrl_parameter * res_par =
			hdrl_spectrum1D_resample_interpolate_parameter_create(hdrl_spectrum1D_interp_akima);

	hdrl_spectrum1D_wavelength wav = hdrl_spectrum1D_get_wavelength(resp_s1d);

	hdrl_spectrum1D * ext_s1d_e = hdrl_spectrum1D_resample(ext_s1d, &wav, res_par);

	hdrl_parameter_delete(res_par);
	//BAD

	//hdrl_spectrum1D_save(ext_s1d_e, "resp_s1d_e_cube.fits");

	int nx=hdrl_image_get_size_x(resp_s1d->flux);
	double* pext = cpl_image_get_data_double(hdrl_image_get_image(ext_s1d->flux));

	// Flux calibrate the cube
	double* presp = cpl_image_get_data_double(hdrl_image_get_image(resp_s1d->flux));



	/* TODO:
	 * here we should also check that hlist and the other arrays are aligned
	 * in wavelength
	 */

	cpl_size sx = hdrl_imagelist_get_size_x(hlist);
    cpl_size sy = hdrl_imagelist_get_size_y(hlist);
	cpl_size sz = hdrl_imagelist_get_size(hlist);

	cpl_size kmax = (nx < sz) ? nx : sz;
	double resp=0;
	hdrl_value resp_fct;

    for (cpl_size k = 0; k < kmax; k++) {
		double exp = -0.4 * airm * pext[k];

		hdrl_image* hima = hdrl_imagelist_get(hlist, k);
		hdrl_value atm_ext = {pow(10., exp), 0.};
		resp = presp[k];
		resp_fct.data = resp;
		if(isfinite(resp)) {
			hdrl_image_mul_scalar(hima, resp_fct);
		} else {
			/* Flag bad pixel in result */
			for(cpl_size j = 1; j <= sy; j++ ) {
				for(cpl_size i = 1; i <= sx; i++ ) {
					hdrl_image_reject(hima, i, j);
				}
			}

		}

		hdrl_image_mul_scalar(hima, atm_ext);
		hdrl_image_mul_scalar(hima, factor);
	}

	char* fcal_cube_ptag = cpl_sprintf("%s_FLUXCAL", procatg);
	char* cube_file_name = cpl_sprintf("%s_cube_fluxcal.fits", pipefile_prefix);
	cpl_propertylist_set_string(phead, FHDR_PRO_CATG, fcal_cube_ptag);

	hdrl_resample_result* res = cpl_calloc(1, sizeof(hdrl_resample_result));
	res->himlist = hlist;
	res->header = phead;
	cpl_propertylist_append(res->header, head_wcs);
	cpl_propertylist_delete(head_wcs);
	cpl_propertylist_update_string(res->header, "BUNIT", UNIT_FLUXCAL);

	/* collapse data cube */
	eris_print_rec_status(0);
	cpl_propertylist* phead2D = cpl_propertylist_duplicate(res->header);
	eris_ifu_plist_erase_wcs3D(phead2D);
	char* mean_cube_pctg = cpl_sprintf("%s_%s",fcal_cube_ptag,"MEAN");
	const char *filenameSpec ="cube_fluxcal";
	char* ima_file_name = cpl_sprintf("%s_%s_mean.fits", pipefile_prefix, filenameSpec);
	cpl_propertylist_append_string(phead2D, "PRODCATG", PRODCATG_WHITELIGHT);
	cpl_propertylist_update_string(phead2D, "BUNIT", UNIT_FLUXCAL);
	hdrl_image* cube_collapsed = NULL;
	cpl_image* cube_cmap = NULL;
	//TODO: AMO checking why collapsed mean of flux calibrated result is full of NANs.
	//hdrl_value fct = {1000000.,0};
	//hdrl_imagelist_mul_scalar(res->himlist, fct);
	hdrl_imagelist_collapse_mean(res->himlist, &cube_collapsed, &cube_cmap);
	cpl_propertylist_update_string(phead2D, CPL_DFS_PRO_CATG, mean_cube_pctg);
	cpl_image* mask_ima = cpl_image_new_from_mask(hdrl_image_get_mask(cube_collapsed));
	eris_ifu_save_deq_image(sof, NULL, parlist,sof, NULL,
			plugin_id, phead2D, "RADECSYS", ima_file_name,
			hdrl_image_get_image(cube_collapsed),
			hdrl_image_get_error(cube_collapsed),
			rmse, mask_ima, flag16bit, UNIT_FLUXCAL);
	cpl_image_delete(mask_ima);
	cpl_free(mean_cube_pctg);
	cpl_free(ima_file_name);
    cpl_propertylist_delete(phead2D);

	if(strcmp(plugin_id,"eris_ifu_jitter") == 0) {
		char* param_name = cpl_sprintf("%s.crea-phase3", "eris.eris_ifu_jitter");
		const cpl_parameter* p = cpl_parameterlist_find_const(parlist, param_name);

		cpl_boolean gen_phase3 = cpl_parameter_get_bool(p);
		if(gen_phase3) {
			cpl_msg_info(cpl_func,"Save flux calibrated COADDED OBJECT cube with phase3 data format");
			eris_ifu_resample_save_cube(res, fcal_cube_ptag, plugin_id,
					cube_file_name, parlist, sof, CPL_TRUE);
		} else {
			cpl_msg_info(cpl_func,"Save flux calibrated COADDED OBJECT cube");
			eris_ifu_resample_save_cube(res, fcal_cube_ptag, plugin_id,
					cube_file_name, parlist, sof, CPL_FALSE);
		}
		cpl_free(param_name);
	} else {
		cpl_msg_info(cpl_func,"Save flux calibrated STD STAR cube");
		eris_ifu_resample_save_cube(res, fcal_cube_ptag, plugin_id,
				cube_file_name, parlist, sof, CPL_FALSE);
	}

	cpl_free(cube_file_name);

	cpl_table* qclog = eris_qclog_init();
	hdrl_image_delete(cube_collapsed);
	cpl_image_delete(cube_cmap);

	cpl_table_delete(qclog);

	//hdrl_resample_result_delete(res);
    cpl_free(fcal_cube_ptag);
	hdrl_spectrum1D_delete(&resp_s1d);
	hdrl_spectrum1D_delete(&ext_s1d);
	hdrl_spectrum1D_delete(&ext_s1d_e);
	cpl_table_delete(atm_ext_tab);
	cpl_table_delete(resp_tab);

	eris_check_error_code("eris_flux_calibrate_cube");

	return cpl_error_get_code();
}
