#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <math.h>

#include "eris_ifu_skycorr.h"
#include "eris_ifu_error.h"
#include "eris_ifu_lambda_corr.h"
#include "eris_ifu_error.h"
#include "skycorr/sc_basic.h"

/**
 * @brief eris_ifu_extract_spectrum4
 * @param cube   obj or sky cube
 * @param header FITS header of the cube
 * @param x      x-Pos to get the spectrum from
 * @param y      y-Pos to get the spectrum from
 * @return       Table with columns: lambda, data, error
 *
 * This function extracts the spectrum at a specific position (x,y) and returns
 * the values as needed by skycorr
 */
cpl_table* eris_ifu_skycorr_extract_spectrum(const hdrl_imagelist *cube,
                                             const cpl_propertylist *header,
                                             cpl_size x,
                                             cpl_size y)
{
    cpl_table   *table      = NULL;
    cpl_vector  *lambda     = NULL;
    cpl_image   *img_data   = NULL,
                *img_err    = NULL;
    hdrl_image  *hdrl_img   = NULL;
    double      *plambda    = NULL,
                *pdata      = NULL,
                *perror     = NULL,
                *pweight    = NULL,
                val         = 0;
    cpl_size    nz          = 0;
    int         rej         = 0,
                *pmask      = NULL,
                *pmask_i    = NULL;

    cpl_ensure(cube   != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(header != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(x > 0,          CPL_ERROR_ILLEGAL_INPUT, NULL);
    cpl_ensure(y > 0,          CPL_ERROR_ILLEGAL_INPUT, NULL);

    TRY {
        BRK_IF_NULL(
            lambda = eris_ifu_lcorr_create_lambda_vector(header));
        BRK_IF_NULL(
            plambda = cpl_vector_get_data(lambda));
        nz = cpl_vector_get_size(lambda);

        BRK_IF_NULL(
            table = cpl_table_new(nz));

        BRK_IF_NULL(
            pdata   = cpl_malloc(nz*sizeof(double)));
        BRK_IF_NULL(
            perror  = cpl_malloc(nz*sizeof(double)));
        BRK_IF_NULL(
            pmask   = cpl_malloc(nz*sizeof(int)));
        BRK_IF_NULL(
            pmask_i = cpl_malloc(nz*sizeof(int)));
        BRK_IF_NULL(
            pweight = cpl_malloc(nz*sizeof(double)));

        // loop over all lambda slices
        for (cpl_size z = 0; z < nz; z++) {
            BRK_IF_NULL(
                hdrl_img = hdrl_imagelist_get(cube, z));
            BRK_IF_NULL(
                img_data = hdrl_image_get_image(hdrl_img));
            BRK_IF_NULL(
                img_err = hdrl_image_get_error(hdrl_img));

            val = cpl_image_get(img_data, x, y, &rej);

            if (rej == 1) {
                pdata[z] = 0.;
                perror[z] = 0.;
                pmask[z] = 0;   // pixel is bad
                pmask_i[z] = 0;   // pixel is bad
                pweight[z] = 0.;
            } else {
                pdata[z] = val;
                perror[z] = cpl_image_get(img_err, x, y, &rej);
                pmask[z] = 1;   // pixel is good
                pmask_i[z] = 1;   // pixel is good
                pweight[z] = 1./perror[z];
            }
        }

        BRK_IF_ERROR(
            cpl_table_wrap_double(table, plambda, "lambda"));
        BRK_IF_ERROR(
            cpl_table_wrap_double(table, pdata,   "flux"));
        BRK_IF_ERROR(
            cpl_table_wrap_double(table, perror,  "dflux"));
        BRK_IF_ERROR(
            cpl_table_wrap_int(table, pmask,      "mask"));
        BRK_IF_ERROR(
            cpl_table_wrap_double(table, pweight, "weight"));
        BRK_IF_ERROR(
            cpl_table_wrap_int(table, pmask_i,      "mask_I"));
    } CATCH {
        eris_ifu_free_table(&table);
    }

    return table;
}
/**
 * @brief generate a spectrum in a table filled by NANs
 * @param cube  input data cube
 * @return Table with columns: "scflux","scdflux", "scmask", filled by NANs
 */
cpl_table* eris_ifu_skycorr_create_nan_spectrum(const hdrl_imagelist *cube)
{
    cpl_size    nz          = 0;
    cpl_table   *table      = NULL;
    double      *pscflux    = NULL,
                *pscdflux   = NULL;
    int         *pscmask    = NULL;

    cpl_ensure(cube   != NULL, CPL_ERROR_NULL_INPUT, NULL);

    TRY {
        nz = hdrl_imagelist_get_size(cube);

        BRK_IF_NULL(
            table = cpl_table_new(nz));

        BRK_IF_NULL(
            pscflux   = cpl_malloc(nz*sizeof(double)));
        BRK_IF_NULL(
            pscdflux  = cpl_malloc(nz*sizeof(double)));
        BRK_IF_NULL(
            pscmask   = cpl_malloc(nz*sizeof(int)));

        // loop over all lambda slices
        for (cpl_size z = 0; z < nz; z++) {
            pscflux[z] = NAN;
            pscdflux[z] = NAN;
            pscmask[z] = 0;
        }

        BRK_IF_ERROR(
            cpl_table_wrap_double(table, pscflux, "scflux"));
        BRK_IF_ERROR(
            cpl_table_wrap_double(table, pscdflux, "scdflux"));
        BRK_IF_ERROR(
            cpl_table_wrap_int(table, pscmask,  "scmask"));
    } CATCH {
        eris_ifu_free_table(&table);
    }

    return table;
}
/**
 * @brief eris_ifu_insert_spectrum
 * @param cube   Cube to manipulate
 * @param scspec Spectrum to insert into cube
 * @param x      x-Pos to insert the spectrum at
 * @param y      y-Pos to insert the spectrum at
 * @return CPL_ERROR_NONE in case of success
 *
 * Since the cube is already blown up to 64x64 pixels, the value to be added is
 * not only inserted at position (x/y) but also at (x/y+1)
 */
cpl_error_code eris_ifu_skycorr_insert_spectrum(hdrl_imagelist *cube,
                                                cpl_table *scspec,
                                                cpl_size x,
                                                cpl_size y)
{
    cpl_image       *img_data   = NULL,
        *img_err    = NULL;
    hdrl_image      *hdrl_img   = NULL;
    const double    *scflux     = NULL,
        *scdflux    = NULL;
    const int       *scmask     = NULL;
    cpl_size        nz          = 0;
    cpl_mask        *img_mask   = NULL;
    cpl_error_code  ret         = CPL_ERROR_NONE;

    cpl_ensure_code(cube   != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(scspec != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(x > 0,          CPL_ERROR_ILLEGAL_INPUT);
    cpl_ensure_code(y > 0,          CPL_ERROR_ILLEGAL_INPUT);

    TRY {
        nz = hdrl_imagelist_get_size(cube);

        BRK_IF_NULL(
            scflux  = cpl_table_get_data_double_const(scspec, "scflux"));
        BRK_IF_NULL(
            scdflux = cpl_table_get_data_double_const(scspec, "scdflux"));
        BRK_IF_NULL(
            scmask  = cpl_table_get_data_int_const(scspec, "scmask"));

        // loop over all lambda slices
        for (cpl_size z = 0; z < nz; z++) {
            BRK_IF_NULL(
                hdrl_img = hdrl_imagelist_get(cube, z));

            BRK_IF_NULL(
                img_data = hdrl_image_get_image(hdrl_img));
            BRK_IF_NULL(
                img_err  = hdrl_image_get_error(hdrl_img));
            BRK_IF_NULL(
                img_mask = hdrl_image_get_mask(hdrl_img));

            if (scmask[z] != 0) {
                // value is valid
                BRK_IF_ERROR(
                    cpl_mask_set(img_mask, x, y, GOOD_PIX));
                BRK_IF_ERROR(
                    cpl_image_set(img_data, x, y, scflux[z]));
                BRK_IF_ERROR(
                    cpl_image_set(img_err, x, y, scdflux[z]));

                BRK_IF_ERROR(
                    cpl_mask_set(img_mask, x, y+1, GOOD_PIX));
                BRK_IF_ERROR(
                    cpl_image_set(img_data, x, y+1, scflux[z]));
                BRK_IF_ERROR(
                    cpl_image_set(img_err, x, y+1, scdflux[z]));
            } else {
                // assert that pixel is set invalid
                // scmask[z] == 0          -> bad pixel
                // img_err == BAD_PIX      -> bad pixel
                BRK_IF_ERROR(
                    cpl_mask_set(img_mask, x, y, BAD_PIX));
                BRK_IF_ERROR(
                    cpl_image_set(img_data, x, y, NAN));
                BRK_IF_ERROR(
                    cpl_image_set(img_err, x, y, NAN));

                BRK_IF_ERROR(
                    cpl_mask_set(img_mask, x, y+1, BAD_PIX));
                BRK_IF_ERROR(
                    cpl_image_set(img_data, x, y+1, NAN));
                BRK_IF_ERROR(
                    cpl_image_set(img_err, x, y+1, NAN));
            }
        }
        CHECK_ERROR_STATE();
    } CATCH {
        ret = cpl_error_get_code();
    }

    return ret;
}

/**
 * @brief eris_ifu_skycorr_flatten_outliers
 * @param objspec table with object spectrum
 * @param skyspec table with sky spectrum
 * @param f  scaling factor
 * @return
 *
 * Outliers that are bigger than a factor of f will be set to the overall mean value
 */
cpl_error_code eris_ifu_skycorr_flatten_outliers(cpl_table *objspec,
                                                 cpl_table *skyspec,
                                                 int f)
{
    double   mean_of    = 0.,
             mean_sf    = 0.,
             mean_odf   = 0.,
             mean_sdf   = 0.,
             limit_of   = 0.,
             limit_sf   = 0.,
             limit_odf  = 0.,
             limit_sdf  = 0.,
            *data_of    = NULL,
            *data_sf    = NULL,
            *data_odf   = NULL,
            *data_sdf   = NULL;
    cpl_error_code  ret = CPL_ERROR_NONE;

    cpl_ensure_code(objspec != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(skyspec != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(f > 0,      CPL_ERROR_ILLEGAL_INPUT);

    TRY {
        mean_of =  cpl_table_get_column_mean(objspec, "flux");
        mean_sf =  cpl_table_get_column_mean(skyspec, "flux");
        mean_odf = cpl_table_get_column_mean(objspec, "dflux");
        mean_sdf = cpl_table_get_column_mean(skyspec, "dflux");

        limit_of = mean_of * f;
        limit_sf = mean_sf * f;
        limit_odf = mean_odf * f;
        limit_sdf = mean_sdf * f;

        data_of =  cpl_table_get_data_double(objspec, "flux");
        data_sf =  cpl_table_get_data_double(skyspec, "flux");
        data_odf = cpl_table_get_data_double(objspec, "dflux");
        data_sdf = cpl_table_get_data_double(skyspec, "dflux");

        CHECK_ERROR_STATE();

        for (int i = 0; i < cpl_table_get_nrow(objspec); i++) {
            if (data_of[i] > limit_of) {
                data_of[i] = mean_of;
            }
            if (data_sf[i] > limit_sf) {
                data_sf[i] = mean_sf;
            }
            if (data_odf[i] > limit_odf) {
                data_odf[i] = mean_odf;
            }
            if (data_sdf[i] > limit_sdf) {
                data_sdf[i] = mean_sdf;
            }
        }
    } CATCH {
        ret = cpl_error_get_code();
    }

    return ret;
}

/**
 * @brief eris_ifu_skycorr_flatten_outliers
 * @param objspec table with object spectrum
 * @param skyspec table with object spectrum
 * @param min_frac ???   TODO: document this
 * @return
 *
 * Outliers are identified using eris_ifu_lcorr_get_peak_lambdas and masked
 */
cpl_error_code eris_ifu_skycorr_flatten_outliers2(cpl_table *objspec,
                                                 cpl_table *skyspec,
                                                 double min_frac)
{
    cpl_error_code  ret             = CPL_ERROR_NONE;
    cpl_size        nz              = 0;
    cpl_bivector    *spectrum       = NULL;
//    cpl_vector      *outliers       = NULL;
    cpl_array       *positions      = NULL;
    double          *pobjlambda     = NULL,
                    *pobjflux       = NULL,
                    *pobjdflux      = NULL,
                    *pobjweight     = NULL,
                    *pskyflux       = NULL,
                    *pskydflux      = NULL,
                    *pskyweight     = NULL,
                    *pbiveclambda   = NULL,
                    *pbivecflux     = NULL/*,
                    *poutliers      = NULL*/;
    int             *pobjmask       = NULL,
                    *pskymask       = NULL,
                    *pobjmaskI      = NULL,
                    *pskymaskI      = NULL,
                    *ppositions     = NULL;

    cpl_ensure_code(objspec != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(skyspec != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code((min_frac > 0) && (min_frac <= 1.0), CPL_ERROR_ILLEGAL_INPUT);

    TRY {
        // analyze outliers on obj spec
        nz = cpl_table_get_nrow(objspec);
        BRK_IF_NULL(
            spectrum    = cpl_bivector_new(nz));

        BRK_IF_NULL(
            pobjlambda  = cpl_table_get_data_double(objspec, "lambda"));

        BRK_IF_NULL(
            pobjflux    = cpl_table_get_data_double(objspec, "flux"));
        BRK_IF_NULL(
            pobjdflux   = cpl_table_get_data_double(objspec, "dflux"));
        BRK_IF_NULL(
            pobjmask    = cpl_table_get_data_int(objspec, "mask"));
        BRK_IF_NULL(
            pobjweight  = cpl_table_get_data_double(objspec, "weight"));
        BRK_IF_NULL(
            pobjmaskI   = cpl_table_get_data_int(objspec, "mask_I"));

        BRK_IF_NULL(
            pskyflux    = cpl_table_get_data_double(skyspec, "flux"));
        BRK_IF_NULL(
            pskydflux   = cpl_table_get_data_double(skyspec, "dflux"));
        BRK_IF_NULL(
            pskymask    = cpl_table_get_data_int(skyspec, "mask"));
        BRK_IF_NULL(
            pskyweight  = cpl_table_get_data_double(skyspec, "weight"));
        BRK_IF_NULL(
            pskymaskI   = cpl_table_get_data_int(skyspec, "mask_I"));

        BRK_IF_NULL(
            pbiveclambda = cpl_bivector_get_x_data(spectrum));
        BRK_IF_NULL(
            pbivecflux   = cpl_bivector_get_y_data(spectrum));

        for (cpl_size z = 0; z < nz; z++) {
            pbiveclambda[z] = pobjlambda[z];
            pbivecflux[z]   = pobjflux[z];
        }

        BRK_IF_NULL(
            positions = eris_ifu_lcorr_get_peak_positions(spectrum, min_frac, NULL));
        BRK_IF_NULL(
            ppositions = cpl_array_get_data_int(positions));

        // mask outliers on obj & sky spec
        for (cpl_size o = 0; o < cpl_array_get_size(positions); o++) {
            int pos = ppositions[o];
            pobjflux[pos]     = 0.;
            pobjdflux[pos]    = 0.;
            pobjmask[pos]     = 0;
            pobjweight[pos]   = 0.;
            pobjmaskI[pos]    = 0;

            pskyflux[pos]     = 0.;
            pskydflux[pos]    = 0.;
            pskymask[pos]     = 0;
            pskyweight[pos]   = 0.;
            pskymaskI[pos]    = 0;
        }
    } CATCH {
        ret = cpl_error_get_code();

    }
    cpl_array_delete(positions);
    eris_ifu_free_bivector(&spectrum);

    return ret;
}
/**
 * @brief generates a parameter list to control skycorr functionality
 * @param sky_header generates a parameter list
 * @return the parameter list
 *
 * Creates the parameter list needed for skycorr. Here the default values are set
 */
cpl_parameterlist* eris_ifu_skycorr_create_parlist(const cpl_propertylist *sky_header)
{
    cpl_parameterlist   *parlist    = NULL;
    cpl_parameter       *p          = NULL;
    const cpl_property  *prop       = NULL;

    TRY {
        BRK_IF_NULL(
            parlist = cpl_parameterlist_new());

        prop = cpl_propertylist_get_property_const(sky_header, "MJD-OBS");
        double mjd = cpl_property_get_double(prop);
        double val = sc_basic_mjd2fracyear(mjd);
        prop = cpl_propertylist_get_property_const(sky_header, "UTC");
        double utc = cpl_property_get_double(prop);
        prop = cpl_propertylist_get_property_const(sky_header, "ESO TEL ALT");
        double telalt = cpl_property_get_double(prop);

        /* Directories and files */
        // INST_DIR
        p = cpl_parameter_new_value("inst_dir", CPL_TYPE_STRING, "", "", ".");
        cpl_parameterlist_append(parlist, p);
        //
        p = cpl_parameter_new_value("data_dir", CPL_TYPE_STRING, "", "", "sysdata/");
        cpl_parameterlist_append(parlist, p);  /* not to be changed by user */
        //
        p = cpl_parameter_new_value("config_dir", CPL_TYPE_STRING, "", "", "config/");
        cpl_parameterlist_append(parlist, p);  /* not to be changed by user */
        //
        p = cpl_parameter_new_value("spectype", CPL_TYPE_STRING, "", "", "SKY");
        cpl_parameterlist_append(parlist, p);
        // INPUT_OBJECT_SPECTRUM
        p = cpl_parameter_new_value("scispec", CPL_TYPE_STRING, "", "", "eris-obj-spec.fits");
        cpl_parameterlist_append(parlist, p);
        // INPUT_SKY_SPECTRUM
        p = cpl_parameter_new_value("skyspec", CPL_TYPE_STRING, "", "", "eris-sky-spec.fits");
        cpl_parameterlist_append(parlist, p);
        // OUTPUT_DIR
        p = cpl_parameter_new_value("output_dir", CPL_TYPE_STRING, "", "", ".");
        cpl_parameterlist_append(parlist, p);
        // OUTPUT_NAME
        p = cpl_parameter_new_value("output_name", CPL_TYPE_STRING, "", "", "TEST-ERIS");
        cpl_parameterlist_append(parlist, p);

        /* Input structure */
        // COL_NAMES
        p = cpl_parameter_new_value("col_lam", CPL_TYPE_STRING, "", "", "lambda");
        cpl_parameterlist_append(parlist, p);
        p = cpl_parameter_new_value("col_flux", CPL_TYPE_STRING, "", "", "flux");
        cpl_parameterlist_append(parlist, p);
        p = cpl_parameter_new_value("col_dflux", CPL_TYPE_STRING, "", "", "dflux");
        cpl_parameterlist_append(parlist, p);
        p = cpl_parameter_new_value("col_mask", CPL_TYPE_STRING, "", "", "mask");
        cpl_parameterlist_append(parlist, p);
        // DEFAULT_ERROR
        p = cpl_parameter_new_value("default_error", CPL_TYPE_DOUBLE, "", "", 0.01);
        cpl_parameterlist_append(parlist, p);
        // WLG_TO_MICRON
        p = cpl_parameter_new_value("wlgtomicron", CPL_TYPE_DOUBLE, "", "", 1.);
        cpl_parameterlist_append(parlist, p);
        // VAC_AIR
        p = cpl_parameter_new_value("vac_air", CPL_TYPE_STRING, "", "", "vac");
        cpl_parameterlist_append(parlist, p);

        /* FITS header keywords */
        // DATE_KEY
        p = cpl_parameter_new_value("date_key", CPL_TYPE_STRING, "", "", "MJD-OBS");
        cpl_parameterlist_append(parlist, p);
        // DATE_VAL
        p = cpl_parameter_new_value("date_val", CPL_TYPE_DOUBLE, "", "", val);
        cpl_parameterlist_append(parlist, p);
        //
        p = cpl_parameter_new_value("mjd", CPL_TYPE_DOUBLE, "", "", mjd);
        cpl_parameterlist_append(parlist, p);
        // TIME_KEY
        p = cpl_parameter_new_value("time_key", CPL_TYPE_STRING, "", "", "UTC");
        cpl_parameterlist_append(parlist, p);
        // TIME_VAL
        p = cpl_parameter_new_value("time_val", CPL_TYPE_DOUBLE, "", "", utc);
        cpl_parameterlist_append(parlist, p);
        // TELALT_KEY
        p = cpl_parameter_new_value("telalt_key", CPL_TYPE_STRING, "", "", "ESO TEL ALT");
        cpl_parameterlist_append(parlist, p);
        // TELALT_VAL
        p = cpl_parameter_new_value("telalt_val", CPL_TYPE_DOUBLE, "", "", telalt);
        cpl_parameterlist_append(parlist, p);

        /* Required input data */
        // LINETABNAME
        p = cpl_parameter_new_value("linetabname", CPL_TYPE_STRING, "", "", "airglow_groups.dat");
        cpl_parameterlist_append(parlist, p);
        // VARDATNAME
        p = cpl_parameter_new_value("vardatname", CPL_TYPE_STRING, "", "", "airglow_var.dat");
        cpl_parameterlist_append(parlist, p);
        // SOLDATURL
        p = cpl_parameter_new_value("soldaturl", CPL_TYPE_STRING, "", "", "ftp://ftp.seismo.nrcan.gc.ca/spaceweather/solar_flux/monthly_averages");
        cpl_parameterlist_append(parlist, p);
        // SOLDATNAME
        p = cpl_parameter_new_value("soldatname", CPL_TYPE_STRING, "", "", "solflux_monthly_average.txt");
        cpl_parameterlist_append(parlist, p);
        //
        p = cpl_parameter_new_value("soldatsource", CPL_TYPE_STRING, "", "", "NONE");
        cpl_parameterlist_append(parlist, p);
        // SOLFLUX
        p = cpl_parameter_new_value("solflux", CPL_TYPE_DOUBLE, "", "", -1.);
        cpl_parameterlist_append(parlist, p);

        /* Line identification */
        // FWHM
        p = cpl_parameter_new_value("fwhm", CPL_TYPE_DOUBLE, "", "", 5.);
        cpl_parameterlist_append(parlist, p);
        // VARFWHM
        p = cpl_parameter_new_range("varfwhm", CPL_TYPE_INT, "", "", 0, 0, 0);
        cpl_parameterlist_append(parlist, p);
        //
        p = cpl_parameter_new_value("meanlam", CPL_TYPE_DOUBLE, "", "", 1.);
        cpl_parameterlist_append(parlist, p);
        // LTOL
        p = cpl_parameter_new_value("ltol", CPL_TYPE_DOUBLE, "", "", 0.01);
        cpl_parameterlist_append(parlist, p);
        //
        p = cpl_parameter_new_value("min_line_width_fac", CPL_TYPE_DOUBLE, "", "", 0.);
        cpl_parameterlist_append(parlist, p);
        // MIN_LINE_DIST
        p = cpl_parameter_new_value("min_line_dist_fac", CPL_TYPE_DOUBLE, "", "", 2.5);
        cpl_parameterlist_append(parlist, p);
        // MIN_LINE_FLUX
        p = cpl_parameter_new_value("min_line_flux_fac", CPL_TYPE_DOUBLE, "", "", 0.);
        cpl_parameterlist_append(parlist, p);
        // FLUXLIM
        p = cpl_parameter_new_value("fluxlim", CPL_TYPE_DOUBLE, "", "", -1.);
        cpl_parameterlist_append(parlist, p);
        //
        p = cpl_parameter_new_value("iteration",CPL_TYPE_INT, "", "", 0);
        cpl_parameterlist_append(parlist, p);

        /* Fitting of sky lines */
        // FTOL
        p = cpl_parameter_new_value("ftol", CPL_TYPE_DOUBLE, "", "", 1e-3);
        cpl_parameterlist_append(parlist, p);
        // XTOL
        p = cpl_parameter_new_value("xtol", CPL_TYPE_DOUBLE, "", "", 1e-3);
        cpl_parameterlist_append(parlist, p);
        // WTOL
        p = cpl_parameter_new_value("wtol", CPL_TYPE_DOUBLE, "", "", 1e-3);
        cpl_parameterlist_append(parlist, p);
        // CHEBY_MAX
        p = cpl_parameter_new_value("cheby_max", CPL_TYPE_INT, "", "", 7);
        cpl_parameterlist_append(parlist, p);
        // CHEBY_MIN
        p = cpl_parameter_new_value("cheby_min", CPL_TYPE_INT, "", "", 3);
        cpl_parameterlist_append(parlist, p);
        // CHEBY_CONST
        p = cpl_parameter_new_value("cheby_const", CPL_TYPE_DOUBLE, "", "", 0.);
        cpl_parameterlist_append(parlist, p);
        // REBINTYPE
        p = cpl_parameter_new_value("rebintype", CPL_TYPE_INT, "", "", 1);
        cpl_parameterlist_append(parlist, p);
        // WEIGHTLIM
        p = cpl_parameter_new_value("weightlim", CPL_TYPE_DOUBLE, "", "", 0.67);
        cpl_parameterlist_append(parlist, p);
        // SIGLIM
        p = cpl_parameter_new_value("siglim", CPL_TYPE_DOUBLE, "", "", 15.);
        cpl_parameterlist_append(parlist, p);
        // FITLIM
        p = cpl_parameter_new_value("fitlim", CPL_TYPE_DOUBLE, "", "", 0.);
        cpl_parameterlist_append(parlist, p);

        CHECK_ERROR_STATE();
    } CATCH {
        eris_ifu_free_parameterlist(&parlist);
    }
    return parlist;
}
