/*
 *  This file is part of the SKYCORR software package.
 *  Copyright (C) 2009-2013 European Southern Observatory
 *
 *  This programme is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This programme is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this programme. If not, see <http://www.gnu.org/licenses/>.
 */


/*!
 * \ingroup sky_correction
 */

/**@{*/

/*!
 * \file sc_par.c
 *
 * Routines for handling parameter files
 *
 * \author  Wolfgang Kausch, Stefan Noll, & ESO In-Kind Team Innsbruck
 * \since   02 Feb 2011
 * \date    28 Aug 2013
 *
 */

/*****************************************************************************
 *                                  INCLUDES                                 *
 ****************************************************************************/

#include <sc_par.h>

/*****************************************************************************
 *                                    CODE                                   *
 ****************************************************************************/

cpl_error_code sc_par_addkeywords(cpl_propertylist *header,
                                  const cpl_parameterlist *parlist)
{
    /*!
     * Write parameters of the general CPL parameter list to a CPL property
     * list.
     *
     * \b INPUT:
     * \param header   input CPL property list
     * \param parlist  general CPL parameter list
     *
     * \b OUTPUT:
     * \param header   extended CPL property list
     *
     * \b ERRORS:
     * - none
     */

    const cpl_parameter *p;
    char * skyfile = NULL;
    char *skyspec = NULL;

    /* Add name of reference sky spectrum */
    p = cpl_parameterlist_find_const(parlist, "skyspec");
    skyfile = cpl_sprintf("%s", cpl_parameter_get_string(p));
    skyspec = strrchr(skyfile, '/');
    if (skyspec == NULL) {
        skyspec = skyfile;
    }
    cpl_propertylist_append_string(header,"ESO SC REFSKY NAME", (skyspec+1));
    cpl_propertylist_set_comment(header, "ESO SC REFSKY NAME",
                                 "Name of reference sky");

    /* Add MJD of reference sky spectrum if given */
    p = cpl_parameterlist_find_const(parlist, "mjd");
    if (cpl_parameter_get_double(p) > 0.) {
        cpl_propertylist_append_double(header,"ESO SC REFSKY MJD",
                                       cpl_parameter_get_double(p));
        cpl_propertylist_set_comment(header, "ESO SC REFSKY MJD",
                                     "MJD [d] for reference sky");
    }

    /* Add date of reference sky spectrum in years */
    p = cpl_parameterlist_find_const(parlist, "date_val");
    cpl_propertylist_append_double(header,"ESO SC REFSKY DATE",
                                   cpl_parameter_get_double(p));
    cpl_propertylist_set_comment(header, "ESO SC REFSKY DATE",
                                 "Date [yr] for reference sky");

    /* Add time of reference sky spectrum in seconds */
    p = cpl_parameterlist_find_const(parlist, "time_val");
    cpl_propertylist_append_double(header,"ESO SC REFSKY TIME",
                                   cpl_parameter_get_double(p));
    cpl_propertylist_set_comment(header, "ESO SC REFSKY TIME",
                                 "Start time [s] for reference sky");

    /* Add solar radio flux */
    p = cpl_parameterlist_find_const(parlist, "solflux");
    cpl_propertylist_append_float(header,"ESO SC SOLRFLUX",
                                  cpl_parameter_get_double(p));
    cpl_propertylist_set_comment(header, "ESO SC SOLRFLUX",
                                 "Solar radio flux [s.f.u.]");

    /* Add estimated FWHM of sky lines */
    p = cpl_parameterlist_find_const(parlist, "fwhm");
    cpl_propertylist_append_float(header,"ESO SC FWHM",
                                  cpl_parameter_get_double(p));
    cpl_propertylist_set_comment(header, "ESO SC FWHM",
                                 "FWHM of sky lines [px]");

    cpl_free(skyfile);
    return CPL_ERROR_NONE;
}

/**@}*/
