/* 
 * This file is part of the ERIS Pipeline
 * Copyright (C) 2002,2003 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <cpl.h>
#include <math.h>
#include <stdio.h>
#include <string.h>

#include "eris_ifu_lambda_corr.h"
#include "eris_ifu_utils.h"
#include "eris_ifu_error.h"

/* Helper functions for test data creation */
static cpl_bivector* create_test_spectrum(double start_lambda, double delta_lambda, int size) {
    cpl_bivector* spectrum = cpl_bivector_new(size);
    cpl_vector* lambda = cpl_bivector_get_x(spectrum);
    cpl_vector* flux = cpl_bivector_get_y(spectrum);
    
    for (int i = 0; i < size; i++) {
        double x = start_lambda + i * delta_lambda;
        cpl_vector_set(lambda, i, x);
        // Create a spectrum with Gaussian peaks
        double y = 0;
        // Add three Gaussian peaks
        y += 100 * exp(-pow((x - 2.1), 2) / (2 * 0.01));
        y += 80 * exp(-pow((x - 2.3), 2) / (2 * 0.01));
        y += 120 * exp(-pow((x - 2.5), 2) / (2 * 0.01));
        cpl_vector_set(flux, i, y);
    }
    
    return spectrum;
}

static cpl_propertylist* create_test_header(void) {
	cpl_propertylist* header = cpl_propertylist_new();
	cpl_propertylist_append_string(header, "INSTRUME", "ERIS");
	cpl_propertylist_append_string(header, FHDR_E_ARM, "SPIFFIER");
	cpl_propertylist_append_string(header, "ESO DPR TYPE", "OBJECT");
	cpl_propertylist_append_string(header, "ESO INS MODE", "IFU");
	cpl_propertylist_append_string(header, "ESO INS OPTI3 NAME", "13mas");
	cpl_propertylist_append_string(header, "ESO INS3 SPGW ID", "K_low");
	cpl_propertylist_append_string(header, "ESO INS3 SPXW ID", "25mas");
	cpl_propertylist_append_int(header, "NAXIS3", 100);
	cpl_propertylist_append_double(header, "CRVAL3", 2.0);
	cpl_propertylist_append_double(header, "CDELT3", 0.01);
	cpl_propertylist_append_double(header, "CRPIX3", 1.0);
    return header;
}

/* Test functions */
static void test_gauss1d_functions(void) {
    double x[1] = {2.0};
    double a[4] = {100.0, 2.0, 0.1, 10.0}; // peak, mean, sigma, offset
    double result = 0.0;
    double derivatives[4] = {0.0};
    
    // Test gauss1d_fnc
    int ret = gauss1d_fnc(x, a, &result);
    cpl_test_zero(ret);
    cpl_test_abs(result, 110.0, 1e-10); // Expected: offset + peak
    
    // Test gauss1d_fncd
    ret = gauss1d_fncd(x, a, derivatives);
    cpl_test_zero(ret);
    cpl_test_abs(derivatives[0], 1.0, 1e-10); // d/d(peak)
    cpl_test_abs(derivatives[1], 0.0, 1e-10); // d/d(mean) at x=mean
    cpl_test_abs(derivatives[2], 0.0, 1e-10); // d/d(sigma) at x=mean
    cpl_test_abs(derivatives[3], 1.0, 1e-10); // d/d(offset)
}

static void test_fit_peak(void) {
	cpl_size size = 100;
	double start_lambda = 2.0;
	double end_lambda = 3.0;
	double delta_lambda = (end_lambda - start_lambda) / (double)size;
    cpl_bivector* spectrum = create_test_spectrum(start_lambda, delta_lambda, size);
    double sigma = 0.0;
    double peak_pos = fit_peak(spectrum, size, 2.1, 5, &sigma);
    
    cpl_test_abs(peak_pos, 2.1, 1e-3); // Should find the peak at 2.1
    cpl_test_abs(sigma, 0.1, 1e-2); // Should find sigma ≈ 0.1
    
    cpl_bivector_delete(spectrum);
}

static void test_create_lambda_vector(void) {
    cpl_propertylist* header = create_test_header();
    cpl_vector* lambda = eris_ifu_lcorr_create_lambda_vector(header);
    
    cpl_test_nonnull(lambda);
    cpl_test_eq(cpl_vector_get_size(lambda), 100);
    cpl_test_abs(cpl_vector_get(lambda, 0), 2.0, 1e-10); // First value
    cpl_test_abs(cpl_vector_get(lambda, 99), 2.99, 1e-10); // Last value
    
    cpl_vector_delete(lambda);
    cpl_propertylist_delete(header);
}

static void test_create_object_mask(void) {
    // Create a test cube with a bright region
    cpl_imagelist* cube = cpl_imagelist_new();
    for (int i = 0; i < 100; i++) {
        cpl_image* img = cpl_image_new(10, 10, CPL_TYPE_FLOAT);
        // Set central pixels brighter
        for (int y = 3; y < 7; y++) {
            for (int x = 3; x < 7; x++) {
                cpl_image_set(img, x+1, y+1, 100.0);
            }
        }
        cpl_imagelist_set(cube, img, i);
    }
    cpl_imagelist_save(cube, "test_cube.fits", CPL_TYPE_DOUBLE, NULL, CPL_IO_CREATE);
    cpl_image* mask;
    mask = eris_ifu_lcorr_create_object_mask(cube, 0.8, NULL, NULL);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_nonnull(mask);
    cpl_image_save(mask, "mask.fits", CPL_TYPE_DOUBLE, NULL, CPL_IO_CREATE);

    mask = eris_ifu_lcorr_create_object_mask(cube, 0.99, NULL, NULL);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_nonnull(mask);
    cpl_image_save(mask, "mask1.fits", CPL_TYPE_DOUBLE, NULL, CPL_IO_CREATE);
    // Test that bright central region is masked (value = 1)
    int status;
    cpl_test_eq(cpl_image_get(mask, 5, 5, &status), 1.0);
    // Test that dark corner is not masked (value = 0)
    cpl_test_eq(cpl_image_get(mask, 1, 1, &status), 0.0);
    
    cpl_image_delete(mask);
    cpl_imagelist_delete(cube);
}

static void test_extract_spectrum(void) {
    // Create test data
    cpl_imagelist* cube = cpl_imagelist_new();
    cpl_propertylist* header = create_test_header();
    
    // Create images with a Gaussian peak moving across the cube
    for (int i = 0; i < 100; i++) {
        cpl_image* img = cpl_image_new(10, 10, CPL_TYPE_DOUBLE);
        double peak_x = 5 + 2 * sin(i * 0.1);
        double peak_y = 5 + 2 * cos(i * 0.1);
        
        for (int y = 0; y < 10; y++) {
            for (int x = 0; x < 10; x++) {
                double val = 100 * exp(-((x-peak_x)*(x-peak_x) + (y-peak_y)*(y-peak_y))/2.0);
                cpl_image_set(img, x+1, y+1, val);
            }
        }
        cpl_imagelist_set(cube, img, i);
    }
    
    cpl_bivector* spectrum = eris_ifu_lcorr_extract_spectrum(cube, header, 0.8, NULL);
    cpl_test_nonnull(spectrum);
    
    // Test spectrum properties
    cpl_test_eq(cpl_bivector_get_size(spectrum), 100);
    
    cpl_bivector_delete(spectrum);
    cpl_imagelist_delete(cube);
    cpl_propertylist_delete(header);
}

static void test_crosscorrelate_spectra(void) {
    // Create test spectra
    cpl_bivector* object = create_test_spectrum(2.0, 0.01, 100);
    cpl_vector* peaks = cpl_vector_new(3);
    cpl_vector_set(peaks, 0, 2.1);
    cpl_vector_set(peaks, 1, 2.3);
    cpl_vector_set(peaks, 2, 2.5);
    
    cpl_polynomial* poly = eris_ifu_lcorr_crosscorrelate_spectra(object, peaks, K_SPIFFI, 2);
    cpl_test_nonnull(poly);
    
    cpl_polynomial_delete(poly);
    cpl_bivector_delete(object);
    cpl_vector_delete(peaks);
}

static void test_lcorr_get(void) {
    // Create test data
    cpl_imagelist* cube = cpl_imagelist_new();
    cpl_propertylist* header = create_test_header();
    cpl_vector* peaks = cpl_vector_new(3);
    cpl_vector_set(peaks, 0, 2.1);
    cpl_vector_set(peaks, 1, 2.3);
    cpl_vector_set(peaks, 2, 2.5);
    
    // Create cube with Gaussian peaks
    for (int i = 0; i < 100; i++) {
        cpl_image* img = cpl_image_new(10, 10, CPL_TYPE_DOUBLE);
        double lambda = 2.0 + i * 0.01;
        for (int y = 0; y < 10; y++) {
            for (int x = 0; x < 10; x++) {
                double val = 0;
                val += 100 * exp(-pow((lambda - 2.1), 2) / (2 * 0.01));
                val += 80 * exp(-pow((lambda - 2.3), 2) / (2 * 0.01));
                val += 120 * exp(-pow((lambda - 2.5), 2) / (2 * 0.01));
                cpl_image_set(img, x+1, y+1, val);
            }
        }
        cpl_imagelist_set(cube, img, i);
    }
    cpl_test_error(CPL_ERROR_NONE);
    cpl_polynomial* poly = eris_ifu_lcorr_get(cube, NULL, header, peaks, 2);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_nonnull(poly);
    
    cpl_polynomial_delete(poly);
    //cpl_imagelist_delete(cube);
    cpl_propertylist_delete(header);
    cpl_vector_delete(peaks);
}

int main(void) {
	cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_WARNING);
    
    test_gauss1d_functions();
    //test_fit_peak(); //fix errors
    test_create_lambda_vector();
    //test_create_object_mask(); //fix errors
    test_extract_spectrum();
    test_crosscorrelate_spectra();
    test_lcorr_get();
    
    return cpl_test_end(0);
}
