/* 
 * This file is part of the ERIS Pipeline
 * Copyright (C) 2002,2003 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02111-1307  USA
 */


#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <cpl.h>
#include <hdrl.h>
#include <string.h>

#include "eris_ifu_resample.c"
#include "eris_ifu_utils.h"
#include "eris_ifu_error.h"
#include "eris_ifu_dfs.h"




/*Fine-grained accuracies*/
#define HDRL_DELTA_COMPARE_VALUE_ABS  CPL_MAX(HDRL_EPS_DATA, HDRL_EPS_ERROR) * 4.

#define HDRL_EPS_TEST       HDRL_EPS_DATA
/* For resampling method definition */
#define DRIZZLE_DOWN_SCALING_FACTOR_X 0.8
#define DRIZZLE_DOWN_SCALING_FACTOR_Y 0.8
#define DRIZZLE_DOWN_SCALING_FACTOR_Z 0.8
#define RENKA_CRITICAL_RADIUS         1.25
#define LANCZOS_KERNEL_SIZE 2
//#define LOOP_DISTANCE                 1.0
/* Image value */
#define HDRL_FLUX_ADU 100
/* For WCS definition From SINFONI cube example 1:
 * SINFO.2005-08-22T07:47:54.305.fits */
#define HDRL_SCALE_Z 500
#define HDRL_CD11   -3.47222e-05
#define HDRL_CD12   0.0
#define HDRL_CD21   0.0
#define HDRL_CD22   3.47222e-05
#define HDRL_CD13   0.0
#define HDRL_CD31   0.0
#define HDRL_CD23   0.0
#define HDRL_CD32   0.0
#define HDRL_CD33   2.45e-10*HDRL_SCALE_Z

#define HDRL_CDELT1 fabs(HDRL_CD11)
#define HDRL_CDELT2 fabs(HDRL_CD22)
#define HDRL_CDELT3 fabs(HDRL_CD33)

#define HDRL_CRPIX1 33.5
#define HDRL_CRPIX2 33.5
#define HDRL_CRPIX3 1.

#define HDRL_CRVAL1 48.0706
#define HDRL_CRVAL2 -20.6219
#define HDRL_CRVAL3 1.9283e-06

#define HDRL_RA        48.070
#define HDRL_DEC         -20.621
#define HDRL_RA_MIN     48.069416667
#define HDRL_RA_MAX     48.0718125
#define HDRL_DEC_MIN    -20.6229925
#define HDRL_DEC_MAX    -20.620708611

#define HDRL_LAMBDA_MIN 1.9283e-06
#define HDRL_LAMBDA_MAX 2.47146e-06

/* Image sizes */
#define HDRL_SIZE_X 50
#define HDRL_SIZE_Y 50
#define HDRL_SIZE_Z 3




/* Helper functions for test data creation */
/*----------------------------------------------------------------------------*/
/**
  @brief    Helper function to create a test cube
  @param    nx      Size in X
  @param    ny      Size in Y
  @param    nz      Size in Z (wavelength)
  @return   The created cube or NULL on error
 */
/*----------------------------------------------------------------------------*/
static cpl_imagelist* create_test_cube(int nx, int ny, int nz) {
    cpl_imagelist* cube;
    cpl_image* img;
    int  k;

    const double sigma = 2.0;
    const double amplitude = 100.0;
    const int center_x = nx/2;
    const int center_y = ny/2;
    double sky_ampl = 1000.0;
    double sky_ampl1 = 400.0;
    double sky_ampl2 = 100.0;

    /* Create the cube */
    cube = cpl_imagelist_new();
    for (k = 0; k < nz; k++) {
        img = cpl_image_new(nx, ny, CPL_TYPE_DOUBLE);
        //data = cpl_image_get_data_double(img);

        /* Add a Gaussian PSF */
        for(int y = 1; y <= ny; y++) {
        	for(int x = 1; x <= nx; x++) {
        		double dx = x - center_x;
        		double dy = y - center_y;
        		double r2 = dx*dx + dy*dy;
        		double value = amplitude * exp(-r2/(2*sigma*sigma));
        		cpl_image_set(img, x, y, value);
        	}
        }
        /* Add sky lines at specific wavelengths */
        if (k == 10 || k == 20 || k == 30) {
        	cpl_image_add_scalar(img,sky_ampl);
        }
        if (k-1 == 10 || k-1 == 20 || k-1 == 30) {
        	cpl_image_add_scalar(img,sky_ampl1);
        }
        if (k-2 == 10 || k-2 == 20 || k-2 == 30) {
        	cpl_image_add_scalar(img,sky_ampl2);
        }
        if (k+1 == 10 || k+1 == 20 || k+1 == 30) {
        	cpl_image_add_scalar(img,sky_ampl1);
        }
        if (k+2 == 10 || k+2 == 20 || k+2 == 30) {
        	cpl_image_add_scalar(img,sky_ampl2);
        }
        cpl_imagelist_set(cube, img, k);
    }
    cpl_imagelist_save(cube, "test_cube.fits", CPL_TYPE_DOUBLE, NULL, CPL_IO_CREATE);

    return cube;
}



static cpl_table* create_test_table(void) {
    cpl_table* table = cpl_table_new(3);
    cpl_table_new_column(table, HDRL_RESAMPLE_TABLE_RA, CPL_TYPE_DOUBLE);
    cpl_table_new_column(table, HDRL_RESAMPLE_TABLE_DEC, CPL_TYPE_DOUBLE);
    cpl_table_new_column(table, HDRL_RESAMPLE_TABLE_LAMBDA, CPL_TYPE_DOUBLE);
    cpl_table_new_column(table, HDRL_RESAMPLE_TABLE_DATA, CPL_TYPE_DOUBLE);
    cpl_table_new_column(table, HDRL_RESAMPLE_TABLE_ERRORS, CPL_TYPE_DOUBLE);
    cpl_table_new_column(table, HDRL_RESAMPLE_TABLE_BPM, CPL_TYPE_INT);
    
    // Add some test data
    cpl_table_set_double(table, HDRL_RESAMPLE_TABLE_RA, 0, 10.0);
    cpl_table_set_double(table, HDRL_RESAMPLE_TABLE_DEC, 0, 20.0);
    cpl_table_set_double(table, HDRL_RESAMPLE_TABLE_LAMBDA, 0, 2.0);
    cpl_table_set_double(table, HDRL_RESAMPLE_TABLE_DATA, 0, 100.0);
    cpl_table_set_double(table, HDRL_RESAMPLE_TABLE_ERRORS, 0, 1.0);
    cpl_table_set_int(table, HDRL_RESAMPLE_TABLE_BPM, 0, 0);
    
    cpl_table_set_double(table, HDRL_RESAMPLE_TABLE_RA, 1, 10.001);
    cpl_table_set_double(table, HDRL_RESAMPLE_TABLE_DEC, 1, 20.001);
    cpl_table_set_double(table, HDRL_RESAMPLE_TABLE_LAMBDA, 1, 2.1);
    cpl_table_set_double(table, HDRL_RESAMPLE_TABLE_DATA, 1, 110.0);
    cpl_table_set_double(table, HDRL_RESAMPLE_TABLE_ERRORS, 1, 1.1);
    cpl_table_set_int(table, HDRL_RESAMPLE_TABLE_BPM, 1, 0);
    
    cpl_table_set_double(table, HDRL_RESAMPLE_TABLE_RA, 2, 10.002);
    cpl_table_set_double(table, HDRL_RESAMPLE_TABLE_DEC, 2, 20.002);
    cpl_table_set_double(table, HDRL_RESAMPLE_TABLE_LAMBDA, 2, 2.2);
    cpl_table_set_double(table, HDRL_RESAMPLE_TABLE_DATA, 2, 120.0);
    cpl_table_set_double(table, HDRL_RESAMPLE_TABLE_ERRORS, 2, 1.2);
    cpl_table_set_int(table, HDRL_RESAMPLE_TABLE_BPM, 2, 0);
    
    return table;
}

static cpl_parameterlist* create_test_parameters(void) {
    cpl_parameterlist* parlist = cpl_parameterlist_new();
    
    // Add method parameters
    cpl_parameter* p = cpl_parameter_new_value("eris.test.method", 
        CPL_TYPE_STRING, "Method", "Method", "LANCZOS");
    cpl_parameterlist_append(parlist, p);
    
    p = cpl_parameter_new_value("eris.test.method.use-errorweights",
        CPL_TYPE_BOOL, "Use error weights", "Use error weights", TRUE);
    cpl_parameterlist_append(parlist, p);
    
    p = cpl_parameter_new_value("eris.test.method.loop-distance",
        CPL_TYPE_INT, "Loop distance", "Loop distance", 3);
    cpl_parameterlist_append(parlist, p);
    
    p = cpl_parameter_new_value("eris.test.method.renka.critical-radius",
        CPL_TYPE_DOUBLE, "Critical radius", "Critical radius", 1.25);
    cpl_parameterlist_append(parlist, p);
    
    p = cpl_parameter_new_value("eris.test.method.lanczos.kernel-size",
        CPL_TYPE_INT, "Kernel size", "Kernel size", 2);
    cpl_parameterlist_append(parlist, p);
    
    p = cpl_parameter_new_value("eris.test.method.drizzle.downscale-x",
        CPL_TYPE_DOUBLE, "Drizzle X", "Drizzle X", 0.8);
    cpl_parameterlist_append(parlist, p);
    
    p = cpl_parameter_new_value("eris.test.method.drizzle.downscale-y",
        CPL_TYPE_DOUBLE, "Drizzle Y", "Drizzle Y", 0.8);
    cpl_parameterlist_append(parlist, p);
    
    p = cpl_parameter_new_value("eris.test.method.drizzle.downscale-z",
        CPL_TYPE_DOUBLE, "Drizzle Z", "Drizzle Z", 0.8);
    cpl_parameterlist_append(parlist, p);
    


    /* --hdrldemo_resample.fieldmargin */
    char            pName[256];
    snprintf(pName, sizeof(pName), "%s.%s", "eris.test", "fieldmargin");
    p = cpl_parameter_new_value(pName, CPL_TYPE_DOUBLE,
    		"Ad this margin/border (in percent) to the "
    		"resampled image/cube", "eris.test", FIELDMARGIN);
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "fieldmargin");
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(parlist, p);

    return parlist;
}

/* unit tests utility */


static cpl_propertylist*
hdrl_resample_util_crea_header_image(const int naxis, const int sx, const int sy,
				     const double ra, const double dec,
				     const double cd11, const double cd12,
				     const double cd21, const double cd22,
				     const double crpix1, const double crpix2,
				     const double crval1, const double crval2,
				     const double cdelt1, const double cdelt2,
				     const char* ctype1, const char* ctype2,
				     const char* cunit1, const char* cunit2)
{
  cpl_propertylist* plist = cpl_propertylist_new();
  cpl_propertylist_append_int(plist,    "NAXIS", naxis);

  cpl_propertylist_append_int(plist,    "NAXIS1", sx);
  cpl_propertylist_append_int(plist,    "NAXIS2", sy);

  cpl_propertylist_append_double(plist, "RA", ra);
  cpl_propertylist_append_double(plist, "DEC", dec);

  cpl_propertylist_append_double(plist, "CRPIX1", crpix1);
  cpl_propertylist_append_double(plist, "CRPIX2", crpix2);

  cpl_propertylist_append_double(plist, "CRVAL1", crval1);
  cpl_propertylist_append_double(plist, "CRVAL2", crval2);

  cpl_propertylist_append_double(plist, "CDELT1", cdelt1);
  cpl_propertylist_append_double(plist, "CDELT2", cdelt2);

  cpl_propertylist_append_string(plist, "CTYPE1", ctype1);
  cpl_propertylist_append_string(plist, "CTYPE2", ctype2);

  cpl_propertylist_append_string(plist, "CUNIT1", cunit1);
  cpl_propertylist_append_string(plist, "CUNIT2", cunit2);

  cpl_propertylist_append_double(plist, "CD1_1", cd11);
  cpl_propertylist_append_double(plist, "CD1_2", cd12);
  cpl_propertylist_append_double(plist, "CD2_1", cd21);
  cpl_propertylist_append_double(plist, "CD2_2", cd22);

  /* To be sure to have a standard FITS header we save and reload the image */
  cpl_image* ima = cpl_image_new(sx, sy, CPL_TYPE_INT);
  cpl_image_add_scalar(ima, 1);

  cpl_image_save(ima, "ima.fits", CPL_TYPE_INT, plist, CPL_IO_DEFAULT);
  cpl_image_delete(ima);
  cpl_propertylist_delete(plist);
  plist = cpl_propertylist_load("ima.fits", 0);
  cpl_test_error(CPL_ERROR_NONE);
  remove("ima.fits");
  return plist;
}



static cpl_propertylist*
hdrl_resample_crea_header_cube(const int naxis,
			       const int sx, const int sy, const int sz,
			       const double ra, const double dec,
			       const double cd11, const double cd12,
			       const double cd21, const double cd22,
			       const double cd13, const double cd31,
			       const double cd23, const double cd32,
			       const double cd33,
			       const double crpix1, const double crpix2, const double crpix3,
			       const double crval1, const double crval2, const double crval3,
			       const double cdelt1, const double cdelt2, const double cdelt3,
			       const char* ctype1, const char* ctype2, const char* ctype3,
			       const char* cunit1, const char* cunit2, const char* cunit3)
{
  /* crea first the FITS header for a 2D example */
  cpl_propertylist* plist =
      hdrl_resample_util_crea_header_image(naxis, sx, sy, ra, dec,
					   cd11, cd12, cd21, cd22, crpix1, crpix2,
					   crval1, crval2, cdelt1, cdelt2, ctype1,
					   ctype2, cunit1, cunit2);

  /* then add information for a 3D example */
  cpl_propertylist_update_int(plist,    "NAXIS", naxis);

  cpl_propertylist_append_int(plist,    "NAXIS3", sz);

  cpl_propertylist_append_double(plist, "CRVAL3", crval3);
  cpl_propertylist_append_double(plist, "CRPIX3", crpix3);
  cpl_propertylist_append_double(plist, "CDELT3", cdelt3);
  cpl_propertylist_append_string(plist, "CTYPE3", ctype3);
  cpl_propertylist_append_string(plist, "CUNIT3", cunit3);

  cpl_propertylist_append_double(plist, "CD1_3", cd13);
  cpl_propertylist_append_double(plist, "CD3_1", cd31);
  cpl_propertylist_append_double(plist, "CD2_3", cd23);
  cpl_propertylist_append_double(plist, "CD3_2", cd32);
  cpl_propertylist_append_double(plist, "CD3_3", cd33);


  /* To be sure to have a standard FITS header we save & reload the imagelist */
  cpl_image* ima = cpl_image_new(sx, sy, CPL_TYPE_INT);
  cpl_image_add_scalar(ima, 1);
  cpl_imagelist* iml = cpl_imagelist_new();
  for(int i = 0; i< sz;i++) {
      cpl_imagelist_set(iml, ima, i);
  }

  cpl_imagelist_save(iml, "iml.fits", CPL_TYPE_INT, plist, CPL_IO_DEFAULT);
  cpl_imagelist_delete(iml);
  cpl_propertylist_delete(plist);
  plist = cpl_propertylist_load("iml.fits", 0);
  cpl_test_error(CPL_ERROR_NONE);
  remove("iml.fits");
  return plist;
}

static cpl_propertylist*
eris_create_test_header(void)
{
	/* test 3D case */
	cpl_propertylist* plist = NULL;
	int sx = HDRL_SIZE_X;
	int sy = HDRL_SIZE_Y;

	int naxis = 2;
	double ra = 10.;
	double dec = 20.;
	double cd11 = HDRL_CD11;
	double cd22 = HDRL_CD22;
	double cd12 = HDRL_CD12;
	double cd21 = HDRL_CD21;
	const double crpix1 = HDRL_CRPIX1;
	const double crpix2 = HDRL_CRPIX2;
	const double crval1 = HDRL_CRVAL1;
	const double crval2 = HDRL_CRVAL2;
	const double cdelt1 = HDRL_CDELT1;
	const double cdelt2 = HDRL_CDELT2;
	const char* cunit1 = "deg";
	const char* cunit2 = "deg";
	const char* ctype1 = "RA---TAN";
	const char* ctype2 = "DEC--TAN";

	int sz = HDRL_SIZE_Z;
	double cd13 = HDRL_CD13;
	double cd31 = HDRL_CD31;
	double cd23 = HDRL_CD23;
	double cd32 = HDRL_CD32;
	double cd33 = HDRL_CD33;
	const double crpix3 = HDRL_CRPIX3;
	const double crval3 = HDRL_CRVAL3;
	const double cdelt3 = HDRL_CDELT3;
	const char* cunit3 = "m";
	const char* ctype3 = "WAV";
	naxis = 3;


	plist = hdrl_resample_crea_header_cube(naxis, sx, sy, sz, ra, dec,
			cd11, cd12, cd21, cd22, cd13, cd31,
			cd23, cd32, cd33,
			crpix1, crpix2, crpix3,
			crval1, crval2, crval3,
			cdelt1, cdelt2, cdelt3,
			ctype1, ctype2, ctype3,
			cunit1, cunit2, cunit3);


	return plist;
}



static cpl_wcs* create_test_wcs(cpl_propertylist* plist) {
    cpl_wcs* wcs = cpl_wcs_new_from_propertylist(plist);
    cpl_matrix* cd = cpl_matrix_new(3, 3);
    
    // Set CD matrix values
    /*
    cpl_matrix_set(cd, 0, 0, -0.0001); // CD1_1
    cpl_matrix_set(cd, 0, 1, 0.0);     // CD1_2
    cpl_matrix_set(cd, 0, 2, 0.0);     // CD1_3
    cpl_matrix_set(cd, 1, 0, 0.0);     // CD2_1
    cpl_matrix_set(cd, 1, 1, 0.0001);  // CD2_2
    cpl_matrix_set(cd, 1, 2, 0.0);     // CD2_3
    cpl_matrix_set(cd, 2, 0, 0.0);     // CD3_1
    cpl_matrix_set(cd, 2, 1, 0.0);     // CD3_2
    cpl_matrix_set(cd, 2, 2, 0.0001);  // CD3_3
    
    cpl_wcs_set_cd(wcs, cd);

    */
    cpl_matrix_delete(cd);
    return wcs;
}

static cpl_frameset* create_test_frameset(void) {
    cpl_frameset* frameset = cpl_frameset_new();
    cpl_frame* frame;
    cpl_size sx = 10;
    cpl_size sy = 10;
    cpl_image* ima = cpl_image_new(sx,sy,CPL_TYPE_DOUBLE);

    /* Create test frames */
    frame = cpl_frame_new();
    cpl_frame_set_filename(frame, "test_raw1.fits");
    cpl_frame_set_tag(frame, "RAW");
    cpl_frame_set_group(frame, CPL_FRAME_GROUP_RAW);
    cpl_frameset_insert(frameset, frame);
    cpl_image_save(ima, "test_raw1.fits", CPL_TYPE_DOUBLE, NULL, CPL_IO_CREATE);

    frame = cpl_frame_new();
    cpl_frame_set_filename(frame, "test_raw2.fits");
    cpl_frame_set_tag(frame, "RAW");
    cpl_frame_set_group(frame, CPL_FRAME_GROUP_RAW);
    cpl_frameset_insert(frameset, frame);
    cpl_image_save(ima, "test_raw2.fits", CPL_TYPE_DOUBLE, NULL, CPL_IO_CREATE);

    frame = cpl_frame_new();
    cpl_frame_set_filename(frame, "test_cal1.fits");
    cpl_frame_set_tag(frame, "CALIB");
    cpl_frame_set_group(frame, CPL_FRAME_GROUP_CALIB);
    cpl_frameset_insert(frameset, frame);
    cpl_image_save(ima, "test_cal1.fits", CPL_TYPE_DOUBLE, NULL, CPL_IO_CREATE);
    cpl_image_delete(ima);

    return frameset;
}



/* Test functions */
static void test_update_table(void) {
    cpl_table* storage = NULL;
    cpl_table* append = create_test_table();
    
    // Test creating new storage
    cpl_error_code err = eris_ifu_resample_update_table(append, &storage);
    cpl_test_eq_error(err, CPL_ERROR_NONE);
    cpl_test_nonnull(storage);
    cpl_test_eq(cpl_table_get_nrow(storage), 3);
    
    // Test appending to existing storage
    err = eris_ifu_resample_update_table(append, &storage);
    cpl_test_eq_error(err, CPL_ERROR_NONE);
    cpl_test_eq(cpl_table_get_nrow(storage), 6);
    
    cpl_table_delete(append);
    cpl_table_delete(storage);
}

static void test_parameters(void) {
    cpl_parameterlist* parlist = create_test_parameters();
    
    // Test getting integer parameter
    int loop_distance = 0;
    cpl_error_code err = eris_ifu_resample_parameters_get_int(parlist, 
        "eris.test.method.loop-distance", &loop_distance);
    cpl_test_eq_error(err, CPL_ERROR_NONE);
    cpl_test_eq(loop_distance, 3);
    
    // Test setting double parameter
    double critical_radius = 0.0;
    err = set_double_from_parameters(parlist, 
        "eris.test.method.renka.critical-radius", 1.0, &critical_radius);
    cpl_test_eq_error(err, CPL_ERROR_NONE);
    cpl_test_abs(critical_radius, 1.25, 0.0001);
    
    cpl_parameterlist_delete(parlist);
}

static void test_set_method(void) {
    cpl_parameterlist* parlist = create_test_parameters();
    
    // Test LANCZOS method
    hdrl_parameter* params = eris_ifu_resample_set_method(parlist, "eris.test");
    cpl_test_nonnull(params);
    hdrl_parameter_delete(params);
    
    // Change method to RENKA and test
    cpl_parameter* p = cpl_parameterlist_find(parlist, "eris.test.method");
    cpl_parameter_set_string(p, "RENKA");
    params = eris_ifu_resample_set_method(parlist, "eris.test");
    cpl_test_nonnull(params);
    hdrl_parameter_delete(params);
    
    cpl_parameterlist_delete(parlist);
}

static void test_get_cdelt123(void) {
	cpl_propertylist* plist = eris_create_test_header();

    cpl_wcs* wcs = create_test_wcs(plist);
    double cdelt1, cdelt2, cdelt3;
    
    cpl_error_code err = eris_ifu_resample_get_cdelt123(wcs, &cdelt1, &cdelt2, &cdelt3);
    cpl_test_eq_error(err, CPL_ERROR_NONE);

    cpl_test_abs(cdelt1, HDRL_CDELT1, 1e-10);
    cpl_test_abs(cdelt2, HDRL_CDELT2, 1e-10);
    cpl_test_abs(cdelt3, HDRL_CDELT3, 1e-10);
    cpl_propertylist_delete(plist);
    
    cpl_wcs_delete(wcs);
}

static void test_set_outputgrid(void) {
    cpl_parameterlist* parlist = create_test_parameters();
    cpl_table* table = create_test_table();
    cpl_propertylist* plist = eris_create_test_header();
    cpl_wcs* wcs = create_test_wcs(plist);
    
    hdrl_parameter* params = eris_ifu_resample_set_outputgrid("test", parlist, table, wcs);
    cpl_test_nonnull(params);
    
    hdrl_parameter_delete(params);
    cpl_wcs_delete(wcs);
    cpl_table_delete(table);
    cpl_parameterlist_delete(parlist);
    cpl_propertylist_delete(plist);
}

static void test_save_cube(void) {
    // Create a test cube
	cpl_imagelist* cube_data = create_test_cube(10, 10, 10);
	hdrl_imagelist * hdrl_cube = hdrl_imagelist_create(cube_data, NULL);
	cpl_imagelist_delete(cube_data);
	cpl_propertylist* plist = eris_create_test_header();
	cpl_wcs* wcs = create_test_wcs(plist);
	cpl_table* cube_table = hdrl_resample_imagelist_to_table(hdrl_cube, wcs);
    hdrl_imagelist_delete(hdrl_cube);

    cpl_parameterlist* parlist = create_test_parameters();
    cpl_frameset* frameset = create_test_frameset();

    hdrl_parameter *method = NULL;
    /* set re-sampling method */
    const char* recipe_name = "test";
    char *context = NULL;
    context = cpl_sprintf("eris.%s", recipe_name);
    method = eris_ifu_resample_set_method(parlist, context);
    cpl_free(context);

    /* set re-sampling outputgrid */
    hdrl_parameter *outputgrid = NULL;
    outputgrid = eris_ifu_resample_set_outputgrid(recipe_name, parlist, cube_table, wcs);

    hdrl_resample_result * cube = hdrl_resample_compute(cube_table, method, outputgrid, wcs);
    

    cpl_error_code err = eris_ifu_resample_save_cube(cube, "TEST", "test_recipe", 
        "test_cube.fits", parlist, frameset, CPL_FALSE);
    cpl_test_eq_error(err, CPL_ERROR_NONE);
    
    cpl_table_delete(cube_table);
    hdrl_resample_result_delete(cube);
    cpl_parameterlist_delete(parlist);
    cpl_frameset_delete(frameset);
    cpl_wcs_delete(wcs);
    cpl_propertylist_delete(plist);
    hdrl_parameter_destroy(method);
    hdrl_parameter_destroy(outputgrid);
    remove("test_cube.fits");
    remove("test_raw1.fits");
    remove("test_raw2.fits");
    remove("test_cal1.fits");
}

int main(void) {
	cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_WARNING);
    
    test_update_table();
    test_parameters();
    test_set_method();
    test_get_cdelt123();
    test_set_outputgrid(); //TODO: fix error
    test_save_cube();//TODO: fix error
    
    return cpl_test_end(0);
}

