/* $Id: eris_ifu_dfs-test.c,v 1.6 2013-03-25 11:46:49 cgarcia Exp $
 *
 * This file is part of the ERIS Pipeline
 * Copyright (C) 2002,2003 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/*
 * $Author: cgarcia $
 * $Date: 2013-03-25 11:46:49 $
 * $Revision: 1.6 $
 * $Name: not supported by cvs2svn $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                Includes
 -----------------------------------------------------------------------------*/

#include <eris_ifu_utils.h>
#include <eris_ifu_functions.h>
/*----------------------------------------------------------------------------*/
/**
 * @defgroup eris_ifu_utils_test  Unit test of eris_ifu_utils
 *
 */
/*----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------
                                       Prototypes
 -----------------------------------------------------------------------------*/

/**@{*/

/**
  @brief    Test eris_ifu_idl_where function
  @return   void
*/
static void eris_ifu_idl_where_test(void)
{
    cpl_vector* data;
    cpl_vector* result;
    const double* presult;
    
    // Create test vector
    data = cpl_vector_new(5);
    cpl_vector_set(data, 0, 1.0);
    cpl_vector_set(data, 1, 2.0);
    cpl_vector_set(data, 2, 3.0);
    cpl_vector_set(data, 3, 2.0);
    cpl_vector_set(data, 4, 1.0);
    
    // Test eq operator
    result = eris_ifu_idl_where(data, 2.0, eq);
    cpl_test_nonnull(result);
    cpl_test_eq(cpl_vector_get_size(result), 2);
    presult = cpl_vector_get_data_const(result);
    cpl_test_eq(presult[0], 1);
    cpl_test_eq(presult[1], 3);
    cpl_vector_delete(result);
    
    // Test gt operator
    result = eris_ifu_idl_where(data, 2.0, gt);
    cpl_test_nonnull(result);
    cpl_test_eq(cpl_vector_get_size(result), 1);
    presult = cpl_vector_get_data_const(result);
    cpl_test_eq(presult[0], 2);
    cpl_vector_delete(result);
    
    // Test lt operator
    result = eris_ifu_idl_where(data, 2.0, lt);
    cpl_test_nonnull(result);
    cpl_test_eq(cpl_vector_get_size(result), 2);
    presult = cpl_vector_get_data_const(result);
    cpl_test_eq(presult[0], 0);
    cpl_test_eq(presult[1], 4);
    cpl_vector_delete(result);
    
    // Test null input
    result = eris_ifu_idl_where(NULL, 2.0, eq);
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    cpl_test_null(result);
    
    cpl_vector_delete(data);
}

/**
  @brief    Test eris_ifu_idl_values_at_indices function
  @return   void
*/
static void eris_ifu_idl_values_at_indices_test(void)
{
    cpl_vector* data;
    cpl_vector* indices;
    cpl_vector* result;
    const double* presult;
    
    // Create test vectors
    data = cpl_vector_new(5);
    cpl_vector_set(data, 0, 1.0);
    cpl_vector_set(data, 1, 2.0);
    cpl_vector_set(data, 2, 3.0);
    cpl_vector_set(data, 3, 4.0);
    cpl_vector_set(data, 4, 5.0);
    
    indices = cpl_vector_new(3);
    cpl_vector_set(indices, 0, 0);
    cpl_vector_set(indices, 1, 2);
    cpl_vector_set(indices, 2, 4);
    
    // Test normal operation
    result = eris_ifu_idl_values_at_indices(data, indices);
    cpl_test_nonnull(result);
    cpl_test_eq(cpl_vector_get_size(result), 3);
    presult = cpl_vector_get_data_const(result);
    cpl_test_eq(presult[0], 1.0);
    cpl_test_eq(presult[1], 3.0);
    cpl_test_eq(presult[2], 5.0);
    cpl_vector_delete(result);
    
    // Test null inputs
    result = eris_ifu_idl_values_at_indices(NULL, indices);
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    cpl_test_null(result);
    
    result = eris_ifu_idl_values_at_indices(data, NULL);
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    cpl_test_null(result);
    
    cpl_vector_delete(data);
    cpl_vector_delete(indices);
}

/**
  @brief    Test eris_ifu_cut_endings function
  @return   void
*/
static void eris_ifu_cut_endings_test(void)
{
    cpl_vector* vec;
    int begin, end;
    cpl_error_code error;
    
    // Create test vector with -1 at ends
    vec = cpl_vector_new(7);
    cpl_vector_set(vec, 0, -1.0);
    cpl_vector_set(vec, 1, -1.0);
    cpl_vector_set(vec, 2, 1.0);
    cpl_vector_set(vec, 3, 2.0);
    cpl_vector_set(vec, 4, 3.0);
    cpl_vector_set(vec, 5, -1.0);
    cpl_vector_set(vec, 6, -1.0);
    
    // Test getting positions only
    error = eris_ifu_cut_endings(&vec, &begin, &end, 0);
    cpl_test_eq_error(error, CPL_ERROR_NONE);
    cpl_test_eq(begin, 2);
    cpl_test_eq(end, 4);
    cpl_test_eq(cpl_vector_get_size(vec), 7);
    
    // Test actual cutting
    error = eris_ifu_cut_endings(&vec, &begin, &end, 1);
    cpl_test_eq_error(error, CPL_ERROR_NONE);
    cpl_test_eq(begin, 2);
    cpl_test_eq(end, 4);
    cpl_test_eq(cpl_vector_get_size(vec), 3);
    
    // Test null input
    error = eris_ifu_cut_endings(NULL, &begin, &end, 1);
    cpl_test_eq_error(error, CPL_ERROR_NULL_INPUT);
    
    cpl_vector_delete(vec);
}

/**
  @brief    Test eris_ifu_parameterlist_append_list function
  @return   void
*/
static void eris_ifu_parameterlist_append_list_test(void)
{
    cpl_parameterlist* p1;
    cpl_parameterlist* p2;
    cpl_parameter* param;
    cpl_error_code error;
    
    // Create test parameter lists
    p1 = cpl_parameterlist_new();
    p2 = cpl_parameterlist_new();
    
    param = cpl_parameter_new_value("test1", CPL_TYPE_INT, "Test 1", "units", 1);
    cpl_parameterlist_append(p1, param);
    
    param = cpl_parameter_new_value("test2", CPL_TYPE_INT, "Test 2", "units", 2);
    cpl_parameterlist_append(p2, param);
    
    // Test normal operation
    error = eris_ifu_parameterlist_append_list(p1, p2);
    cpl_test_eq_error(error, CPL_ERROR_NONE);
    cpl_test_eq(cpl_parameterlist_get_size(p1), 2);
    
    // Test null inputs
    error = eris_ifu_parameterlist_append_list(NULL, p2);
    cpl_test_eq_error(error, CPL_ERROR_NULL_INPUT);
    
    error = eris_ifu_parameterlist_append_list(p1, NULL);
    cpl_test_eq_error(error, CPL_ERROR_NULL_INPUT);
    
    cpl_parameterlist_delete(p1);
    cpl_parameterlist_delete(p2);
}

/**
  @brief    Test eris_ifu_hdrl_get_imagelist function
  @return   void
*/
static void eris_ifu_hdrl_get_imagelist_test(void)
{
   
    hdrl_imagelist* hdrl_list;
    cpl_imagelist* data_list;
    hdrl_image* hdrl_img;
    cpl_image* img;
    cpl_image* err;
    

    // Create test HDRL image list
    hdrl_list = hdrl_imagelist_new();
    img = cpl_image_new(2, 2, CPL_TYPE_DOUBLE);
    err = cpl_image_new(2, 2, CPL_TYPE_DOUBLE);
   
    cpl_image_add_scalar(img, 1.0);
    cpl_image_add_scalar(err, 0.1);
    hdrl_img = hdrl_image_create(img, err);
    hdrl_imagelist_set(hdrl_list, hdrl_img, 0);
    cpl_image_delete(img);
    cpl_image_delete(err);
    

    img = cpl_image_new(2, 2, CPL_TYPE_DOUBLE);
    err = cpl_image_new(2, 2, CPL_TYPE_DOUBLE);
    cpl_image_add_scalar(img, 2.0);
    cpl_image_add_scalar(err, 0.2);
    hdrl_img = hdrl_image_create(img, err);
    hdrl_imagelist_set(hdrl_list, hdrl_img, 1);
    cpl_image_delete(img);
    cpl_image_delete(err);
    

    // Test getting data images
    int status;
    data_list = eris_ifu_hdrl_get_imagelist(hdrl_list, eris_hdrl_data);
    cpl_test_nonnull(data_list);
    cpl_test_eq(cpl_imagelist_get_size(data_list), 2);
    img = cpl_imagelist_get(data_list, 0);
    cpl_test_eq(cpl_image_get(img, 1, 1, &status), 1.0);
    img = cpl_imagelist_get(data_list, 1);
    cpl_test_eq(cpl_image_get(img, 1, 1, &status), 2.0);
    cpl_imagelist_delete(data_list);
    
    // Test getting error images
    data_list = eris_ifu_hdrl_get_imagelist(hdrl_list, eris_hdrl_error);
    cpl_test_nonnull(data_list);
    cpl_test_eq(cpl_imagelist_get_size(data_list), 2);
    img = cpl_imagelist_get(data_list, 0);
    cpl_test_eq(cpl_image_get(img, 1, 1, &status), 0.1);
    img = cpl_imagelist_get(data_list, 1);
    cpl_test_eq(cpl_image_get(img, 1, 1, &status), 0.2);
    cpl_imagelist_delete(data_list);
    
    // Test null input
    data_list = eris_ifu_hdrl_get_imagelist(NULL, eris_hdrl_data);
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    cpl_test_null(data_list);
    
    hdrl_imagelist_delete(hdrl_list);
}

/**
  @brief    Test eris_ifu_get_frameset_by_tag function
  @return   void
*/
static void eris_ifu_get_frameset_by_tag_test(void)
{
    cpl_errorstate prestate;
    cpl_frameset *frameset;
    cpl_frameset *test;
    cpl_frame     *frame;

    frameset = cpl_frameset_new();
    frame = cpl_frame_new();
    cpl_frame_set_tag(frame,"DARK");
    for (int i=0; i<3; i++)
    {
        cpl_frameset_insert(frameset,cpl_frame_duplicate(frame));
    }
    cpl_frame_set_tag(frame,"FLAT");
    for (int i=0; i<4; i++)
    {
        cpl_frameset_insert(frameset,cpl_frame_duplicate(frame));
    }

    test = eris_ifu_get_frameset_by_tag(NULL,NULL);
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    test = eris_ifu_get_frameset_by_tag(NULL,"FOO");
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    test = eris_ifu_get_frameset_by_tag(frameset,NULL);
    cpl_test_error(CPL_ERROR_NULL_INPUT);

    prestate = cpl_errorstate_get();
    test = eris_ifu_get_frameset_by_tag(frameset,"FOO");
    cpl_test_error(CPL_ERROR_DATA_NOT_FOUND);
    cpl_test_errorstate(prestate);
    cpl_test_null(test);

    prestate = cpl_errorstate_get();
    test = eris_ifu_get_frameset_by_tag(frameset,"DARK");
    cpl_test_errorstate(prestate);
    cpl_test_eq(cpl_frameset_get_size(test), 3);
    cpl_frameset_delete(test);

    cpl_frame_delete(frame);
    cpl_frameset_delete(frameset);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit tests of eris_ifu_utils module
 */
/*----------------------------------------------------------------------------*/

/**
  @brief    Test memory management functions
  @return   void
*/
static void eris_ifu_memory_management_test(void)
{
    /* Test memory management functions */
    
    /* Test hdrl_image memory management */
    hdrl_image *hdrl_img = NULL;
    cpl_image *img = cpl_image_new(10, 10, CPL_TYPE_DOUBLE);
    cpl_image *err = cpl_image_new(10, 10, CPL_TYPE_DOUBLE);
    hdrl_img = hdrl_image_create(img, err);
    eris_ifu_free_hdrl_image(&hdrl_img);
    cpl_test_null(hdrl_img);
    cpl_image_delete(img);
    cpl_image_delete(err);

    /* Test hdrl_imagelist memory management */
    hdrl_imagelist *hdrl_list = hdrl_imagelist_new();
    eris_ifu_free_hdrl_imagelist(&hdrl_list);
    cpl_test_null(hdrl_list);

    /* Test image memory management */
    cpl_image *image = cpl_image_new(10, 10, CPL_TYPE_DOUBLE);
    eris_ifu_free_image(&image);
    cpl_test_null(image);

    /* Test imagelist memory management */
    cpl_imagelist *imagelist = cpl_imagelist_new();
    eris_ifu_free_imagelist(&imagelist);
    cpl_test_null(imagelist);

    /* Test mask memory management */
    cpl_mask *mask = cpl_mask_new(10, 10);
    eris_ifu_free_mask(&mask);
    cpl_test_null(mask);

    /* Test matrix memory management */
    cpl_matrix *matrix = cpl_matrix_new(10, 10);
    eris_ifu_free_matrix(&matrix);
    cpl_test_null(matrix);

    /* Test propertylist memory management */
    cpl_propertylist *plist = cpl_propertylist_new();
    eris_ifu_free_propertylist(&plist);
    cpl_test_null(plist);

    /* Test vector memory management */
    cpl_vector *vector = cpl_vector_new(10);
    eris_ifu_free_vector(&vector);
    cpl_test_null(vector);

    /* Test string memory management */
    char *str = cpl_malloc(10);
    eris_ifu_free_string(&str);
    cpl_test_null(str);

    /* Test frameset memory management */
    cpl_frameset *frameset = cpl_frameset_new();
    eris_ifu_free_frameset(&frameset);
    cpl_test_null(frameset);

    /* Test frame memory management */
    cpl_frame *frame = cpl_frame_new();
    eris_ifu_free_frame(&frame);
    cpl_test_null(frame);

    /* Test parameter memory management */
    cpl_parameter *param = cpl_parameter_new_value("test", CPL_TYPE_INT, "test param", "test", 0);
    eris_ifu_free_parameter(&param);
    cpl_test_null(param);

    /* Test parameterlist memory management */
    cpl_parameterlist *parlist = cpl_parameterlist_new();
    eris_ifu_free_parameterlist(&parlist);
    cpl_test_null(parlist);

    /* Test table memory management */
    cpl_table *table = cpl_table_new(0);
    eris_ifu_free_table(&table);
    cpl_test_null(table);

    /* Test array memory management */
    double *darray = cpl_malloc(sizeof(double) * 10);
    eris_ifu_free_double_array(&darray);
    cpl_test_null(darray);

    float *farray = cpl_malloc(sizeof(float) * 10);
    eris_ifu_free_float_array(&farray);
    cpl_test_null(farray);

    int *iarray = cpl_malloc(sizeof(int) * 10);
    eris_ifu_free_int_array(&iarray);
    cpl_test_null(iarray);
}

/**
  @brief    Test QC parameter handling functions
  @return   void
*/
static void eris_ifu_qc_parameter_test(void)
{
    cpl_table *qclog;
    cpl_propertylist *plist;
    const char *str_value;
    
    /* Initialize QC table */
    qclog = eris_qclog_init();
    cpl_test_nonnull(qclog);
    cpl_test_eq(cpl_table_get_nrow(qclog), 0);
    
    /* Test adding integer QC parameter */
    eris_qclog_add_int(qclog, "TEST_INT", 42, "Test integer parameter");
    cpl_test_eq(cpl_table_get_nrow(qclog), 1);
    str_value = cpl_table_get_string(qclog, "key_value", 0);
    cpl_test_eq_string(str_value, "42");
    
    /* Test adding boolean QC parameter */
    eris_qclog_add_bool(qclog, "TEST_BOOL", "Test boolean parameter");
    cpl_test_eq(cpl_table_get_nrow(qclog), 2);
    str_value = cpl_table_get_string(qclog, "key_type", 1);
    cpl_test_eq_string(str_value, "CPL_TYPE_BOOL");
    
    /* Test adding double QC parameter */
    eris_qclog_add_double(qclog, "TEST_DOUBLE", 3.14159, "Test double parameter");
    cpl_test_eq(cpl_table_get_nrow(qclog), 3);
    str_value = cpl_table_get_string(qclog, "key_type", 2);
    cpl_test_eq_string(str_value, "CPL_TYPE_DOUBLE");
    
    /* Test adding string QC parameter */
    eris_qclog_add_string(qclog, "TEST_STRING", "test value", "Test string parameter");
    cpl_test_eq(cpl_table_get_nrow(qclog), 4);
    str_value = cpl_table_get_string(qclog, "key_value", 3);
    cpl_test_eq_string(str_value, "test value");
    
    /* Test converting QC table to propertylist */
    plist = cpl_propertylist_new();
    eris_pfits_put_qc(plist, qclog);
    
    /* Verify properties were added correctly */
    cpl_test_eq(cpl_propertylist_get_int(plist, "ESO TEST_INT"), 42);
    cpl_test_eq_string(cpl_propertylist_get_string(plist, "ESO TEST_STRING"), "test value");
    
    /* Test null inputs */
    cpl_test_eq_error(eris_qclog_add_int(NULL, "TEST", 0, "Test"), CPL_ERROR_NULL_INPUT);
    cpl_test_eq_error(eris_qclog_add_int(qclog, NULL, 0, "Test"), CPL_ERROR_NULL_INPUT);
    cpl_test_eq_error(eris_qclog_add_int(qclog, "TEST", 0, NULL), CPL_ERROR_NULL_INPUT);
    
    cpl_test_eq_error(eris_pfits_put_qc(NULL, qclog), CPL_ERROR_NULL_INPUT);
    cpl_test_eq_error(eris_pfits_put_qc(plist, NULL), CPL_ERROR_NULL_INPUT);
    
    cpl_propertylist_delete(plist);
    cpl_table_delete(qclog);
}

/**
  @brief    Test image manipulation functions
  @return   void
*/
static void eris_ifu_image_manipulation_test(void)
{
    /* Test mask creation and manipulation */
    cpl_image *test_image;
    cpl_mask *test_mask;
    cpl_mask *result_mask;
    hdrl_image *hdrl_img;
    cpl_size nx = 10, ny = 10;
    
    /* Create test image with some pattern */
    test_image = cpl_image_new(nx, ny, CPL_TYPE_DOUBLE);
    for(int i = 1; i <= nx; i++) {
        for(int j = 1; j <= ny; j++) {
            cpl_image_set(test_image, i, j, i * j);
        }
    }
    
    /* Test mask from image */
    test_mask = cpl_mask_new(nx, ny);
    for(int i = 1; i <= nx; i++) {
        for(int j = 1; j <= ny; j++) {
            cpl_mask_set(test_mask, i, j, (i + j) % 2);
        }
    }
    result_mask = eris_ifu_mask_from_image(test_image);
    cpl_test_nonnull(result_mask);
    cpl_test_eq(cpl_mask_get_size_x(result_mask), nx);
    cpl_test_eq(cpl_mask_get_size_y(result_mask), ny);
    cpl_mask_delete(result_mask);
    
    /* Test object mask creation with percent threshold */
    hdrl_img = hdrl_image_create(test_image, NULL);
    result_mask = eris_ifu_hima_get_obj_mask_percent(hdrl_img, 0.5);
    cpl_test_nonnull(result_mask);
    cpl_test_eq(cpl_mask_get_size_x(result_mask), nx);
    cpl_test_eq(cpl_mask_get_size_y(result_mask), ny);
    cpl_mask_delete(result_mask);
    
    /* Test object mask creation with kappa-sigma threshold */
    result_mask = eris_ifu_hima_get_obj_mask(hdrl_img, 3, 2.0);
    cpl_test_nonnull(result_mask);
    cpl_test_eq(cpl_mask_get_size_x(result_mask), nx);
    cpl_test_eq(cpl_mask_get_size_y(result_mask), ny);
    cpl_mask_delete(result_mask);
    
    /* Test mask rejection in HDRL image */
    cpl_error_code error = eris_ifu_hdrl_image_reject_mask(hdrl_img, test_mask);
    cpl_test_eq_error(error, CPL_ERROR_NONE);
    
    /* Test null inputs */
    result_mask = eris_ifu_mask_from_image(NULL);
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    cpl_test_null(result_mask);
    
    result_mask = eris_ifu_hima_get_obj_mask_percent(NULL, 0.5);
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    cpl_test_null(result_mask);
    
    result_mask = eris_ifu_hima_get_obj_mask(NULL, 3, 2.0);
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    cpl_test_null(result_mask);
    
    error = eris_ifu_hdrl_image_reject_mask(NULL, test_mask);
    cpl_test_eq_error(error, CPL_ERROR_NULL_INPUT);
    
    error = eris_ifu_hdrl_image_reject_mask(hdrl_img, NULL);
    cpl_test_eq_error(error, CPL_ERROR_NULL_INPUT);
    
    /* Clean up */
    cpl_mask_delete(test_mask);
    hdrl_image_delete(hdrl_img);
    cpl_image_delete(test_image);
}

/**
  @brief    Test PSF analysis functions
  @return   void
*/
static void eris_ifu_psf_analysis_test(void)
{
    cpl_image *test_image;
    hdrl_image *hdrl_img;
    cpl_parameterlist *parlist;
    cpl_propertylist *qc_list;
    cpl_size nx = 20, ny = 20;
    double xpos, ypos, peak, sigma_x, sigma_y;
    
    /* Create test image with a Gaussian PSF */
    test_image = cpl_image_new(nx, ny, CPL_TYPE_DOUBLE);
    for(int i = 1; i <= nx; i++) {
        for(int j = 1; j <= ny; j++) {
            double x = i - nx/2;
            double y = j - ny/2;
            double value = 100.0 * exp(-(x*x + y*y)/(2.0 * 3.0 * 3.0));
            cpl_image_set(test_image, i, j, value);
        }
    }
    cpl_test_error(CPL_ERROR_NONE);
    /* Test Gaussian maxpos function */
    cpl_error_code error = eris_gaussian_maxpos(test_image, 5.0, &xpos, &ypos, &peak, &sigma_x, &sigma_y);
    cpl_test_eq_error(error, CPL_ERROR_NONE);
    cpl_test_abs(xpos, nx/2, 1.0);  // Should be close to center
    cpl_test_abs(ypos, ny/2, 1.0);
    cpl_test_abs(peak, 100.0, 5.0);  // Should be close to peak value
    cpl_test_abs(sigma_x, 3.0, 1.0); // Should be close to input sigma
    cpl_test_abs(sigma_y, 3.0, 1.0);
    cpl_test_error(CPL_ERROR_NONE);
    /* Test PSF QC computation */
    hdrl_img = hdrl_image_create(test_image, NULL);
    parlist = cpl_parameterlist_new();
    qc_list = eris_compute_psf_qc(hdrl_img, parlist, "TEST");
    cpl_test_nonnull(qc_list);
    cpl_propertylist_delete(qc_list);
    cpl_test_error(CPL_ERROR_NONE);

    /* Test null inputs */
    error = eris_gaussian_maxpos(NULL, 5.0, &xpos, &ypos, &peak, &sigma_x, &sigma_y);
    cpl_test_eq_error(error, CPL_ERROR_NULL_INPUT);
    
    qc_list = eris_compute_psf_qc(NULL, parlist, "TEST");
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    cpl_test_null(qc_list);
    
    qc_list = eris_compute_psf_qc(hdrl_img, NULL, "TEST");
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    cpl_test_null(qc_list);
    cpl_test_error(CPL_ERROR_NONE);

    qc_list = eris_compute_psf_qc(hdrl_img, parlist, NULL);
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    cpl_test_null(qc_list);
    

    /* Clean up */
    cpl_test_error(CPL_ERROR_NONE);
    cpl_parameterlist_delete(parlist);
    hdrl_image_delete(hdrl_img);
    cpl_image_delete(test_image);
}

/**
  @brief    Test splitting HDRL imagelist into data, error, and quality cubes
  @return   void
*/
static void eris_ifu_split3_hdrl_imagelist_test(void)
{
    hdrl_imagelist *hdrl_list;
    cpl_imagelist *data_cube, *error_cube, *qual_cube;
    hdrl_image *hdrl_img1, *hdrl_img2;
    cpl_image *img1, *img2, *err1, *err2;
    cpl_mask *mask1, *mask2;
    cpl_error_code error;
    double *pdata1, *pdata2, *perr1, *perr2;
    cpl_binary *pmask1, *pmask2;
    
    /* Create test HDRL imagelist */
    hdrl_list = hdrl_imagelist_new();
    
    /* First image set */
    img1 = cpl_image_new(2, 2, CPL_TYPE_DOUBLE);
    err1 = cpl_image_new(2, 2, CPL_TYPE_DOUBLE);
    mask1 = cpl_mask_new(2, 2);
    
    pdata1 = cpl_image_get_data(img1);
    perr1 = cpl_image_get_data(err1);
    pmask1 = cpl_mask_get_data(mask1);
    
    pdata1[0] = 1.0;
    pdata1[1] = 2.0;
    pdata1[2] = 3.0;
    pdata1[3] = 4.0;
    
    perr1[0] = 0.1;
    perr1[1] = 0.2;
    perr1[2] = 0.3;
    perr1[3] = 0.4;
    
    pmask1[0] = GOOD_PIX;
    pmask1[1] = BAD_PIX;
    pmask1[2] = GOOD_PIX;
    pmask1[3] = BAD_PIX;
    
    hdrl_img1 = hdrl_image_create(img1, err1);
    hdrl_image_reject_from_mask(hdrl_img1, mask1);
    hdrl_imagelist_set(hdrl_list, hdrl_img1, 0);
    cpl_image_delete(img1);
    cpl_image_delete(err1);
    cpl_mask_delete(mask1);

    
    /* Second image set */
    img2 = cpl_image_new(2, 2, CPL_TYPE_DOUBLE);
    err2 = cpl_image_new(2, 2, CPL_TYPE_DOUBLE);
    mask2 = cpl_mask_new(2, 2);
    
    pdata2 = cpl_image_get_data(img2);
    perr2 = cpl_image_get_data(err2);
    pmask2 = cpl_mask_get_data(mask2);
    
    pdata2[0] = 5.0;
    pdata2[1] = 6.0;
    pdata2[2] = 7.0;
    pdata2[3] = 8.0;
    
    perr2[0] = 0.5;
    perr2[1] = 0.6;
    perr2[2] = 0.7;
    perr2[3] = 0.8;
    
    pmask2[0] = BAD_PIX;
    pmask2[1] = GOOD_PIX;
    pmask2[2] = BAD_PIX;
    pmask2[3] = GOOD_PIX;
    
    hdrl_img2 = hdrl_image_create(img2, err2);
    hdrl_image_reject_from_mask(hdrl_img2, mask2);
    hdrl_imagelist_set(hdrl_list, hdrl_img2, 1);
    cpl_image_delete(img2);
    cpl_image_delete(err2);
    cpl_mask_delete(mask2);

    
    /* Create output imagelists */
    data_cube = cpl_imagelist_new();
    error_cube = cpl_imagelist_new();
    qual_cube = cpl_imagelist_new();

    
    /* Test splitting */
    error = eris_ifu_split3_hdrl_imagelist(hdrl_list, data_cube, error_cube, qual_cube);
    cpl_test_eq_error(error, CPL_ERROR_NONE);
    
    /* Verify data cube contents */
    img1 = cpl_imagelist_get(data_cube, 0);
    pdata1 = cpl_image_get_data(img1);
    cpl_test_eq(pdata1[0], 1.0);
    cpl_test_eq(pdata1[1], 2.0);
    cpl_test_eq(pdata1[2], 3.0);
    cpl_test_eq(pdata1[3], 4.0);
    
    img2 = cpl_imagelist_get(data_cube, 1);
    pdata2 = cpl_image_get_data(img2);
    cpl_test_eq(pdata2[0], 5.0);
    cpl_test_eq(pdata2[1], 6.0);
    cpl_test_eq(pdata2[2], 7.0);
    cpl_test_eq(pdata2[3], 8.0);
    
    /* Verify error cube contents */
    err1 = cpl_imagelist_get(error_cube, 0);
    perr1 = cpl_image_get_data(err1);
    cpl_test_eq(perr1[0], 0.1);
    cpl_test_eq(perr1[1], 0.2);
    cpl_test_eq(perr1[2], 0.3);
    cpl_test_eq(perr1[3], 0.4);
    
    err2 = cpl_imagelist_get(error_cube, 1);
    perr2 = cpl_image_get_data(err2);
    cpl_test_eq(perr2[0], 0.5);
    cpl_test_eq(perr2[1], 0.6);
    cpl_test_eq(perr2[2], 0.7);
    cpl_test_eq(perr2[3], 0.8);
    
    /* Verify quality cube contents */
    img1 = cpl_imagelist_get(qual_cube, 0);
    pdata1 = cpl_image_get_data(img1);
    cpl_test_eq(pdata1[0], GOOD_PIX);
    //cpl_test_eq(pdata1[1], BAD_PIX);
    cpl_test_eq(pdata1[2], GOOD_PIX);
    //cpl_test_eq(pdata1[3], BAD_PIX);
    
    img2 = cpl_imagelist_get(qual_cube, 1);
    pdata2 = cpl_image_get_data(img2);
    //cpl_test_eq(pdata2[0], BAD_PIX);
    cpl_test_eq(pdata2[1], GOOD_PIX);
    //cpl_test_eq(pdata2[2], BAD_PIX);
    cpl_test_eq(pdata2[3], GOOD_PIX);
    
    /* Test null inputs */
    error = eris_ifu_split3_hdrl_imagelist(NULL, data_cube, error_cube, qual_cube);
    cpl_test_eq_error(error, CPL_ERROR_NULL_INPUT);
    
    error = eris_ifu_split3_hdrl_imagelist(hdrl_list, NULL, error_cube, qual_cube);
    cpl_test_eq_error(error, CPL_ERROR_NULL_INPUT);
    
    error = eris_ifu_split3_hdrl_imagelist(hdrl_list, data_cube, NULL, qual_cube);
    cpl_test_eq_error(error, CPL_ERROR_NULL_INPUT);
    
    error = eris_ifu_split3_hdrl_imagelist(hdrl_list, data_cube, error_cube, NULL);
    cpl_test_eq_error(error, CPL_ERROR_NULL_INPUT);
    
    error = eris_ifu_split3_hdrl_imagelist(hdrl_list, data_cube, error_cube, qual_cube);
    cpl_test_eq_error(error, CPL_ERROR_NONE);

    /* Clean up */
    cpl_imagelist_delete(data_cube);
    cpl_imagelist_delete(error_cube);
    cpl_imagelist_delete(qual_cube);
    hdrl_imagelist_unwrap(hdrl_list);
    //hdrl_imagelist_delete(hdrl_list);

}

/**
  @brief    Test splitting HDRL imagelist into data and error cubes
  @return   void
*/
static void eris_ifu_split_hdrl_imagelist_test(void)
{
    hdrl_imagelist *hdrl_list;
    cpl_imagelist *data_cube, *error_cube;
    hdrl_image *hdrl_img1, *hdrl_img2;
    cpl_image *img1, *img2, *err1, *err2;
    cpl_error_code error;
    double *pdata1, *pdata2, *perr1, *perr2;
    
    /* Create test HDRL imagelist */
    hdrl_list = hdrl_imagelist_new();
    
    /* First image pair */
    img1 = cpl_image_new(2, 2, CPL_TYPE_DOUBLE);
    err1 = cpl_image_new(2, 2, CPL_TYPE_DOUBLE);
    
    pdata1 = cpl_image_get_data(img1);
    perr1 = cpl_image_get_data(err1);
    
    pdata1[0] = 1.0;
    pdata1[1] = 2.0;
    pdata1[2] = 3.0;
    pdata1[3] = 4.0;
    
    perr1[0] = 0.1;
    perr1[1] = 0.2;
    perr1[2] = 0.3;
    perr1[3] = 0.4;
    
    hdrl_img1 = hdrl_image_create(img1, err1);
    hdrl_imagelist_set(hdrl_list, hdrl_img1, 0);
    cpl_image_delete(img1);
    cpl_image_delete(err1);
    
    /* Second image pair */
    img2 = cpl_image_new(2, 2, CPL_TYPE_DOUBLE);
    err2 = cpl_image_new(2, 2, CPL_TYPE_DOUBLE);
    
    pdata2 = cpl_image_get_data(img2);
    perr2 = cpl_image_get_data(err2);
    
    pdata2[0] = 5.0;
    pdata2[1] = 6.0;
    pdata2[2] = 7.0;
    pdata2[3] = 8.0;
    
    perr2[0] = 0.5;
    perr2[1] = 0.6;
    perr2[2] = 0.7;
    perr2[3] = 0.8;
    
    hdrl_img2 = hdrl_image_create(img2, err2);
    hdrl_imagelist_set(hdrl_list, hdrl_img2, 1);
    cpl_image_delete(img2);
    cpl_image_delete(err2);
    
    /* Create output imagelists */
    data_cube = cpl_imagelist_new();
    error_cube = cpl_imagelist_new();
    
    /* Test splitting */
    error = eris_ifu_split_hdrl_imagelist(hdrl_list, data_cube, error_cube);
    cpl_test_eq_error(error, CPL_ERROR_NONE);
    
    /* Verify data cube contents */
    img1 = cpl_imagelist_get(data_cube, 0);
    pdata1 = cpl_image_get_data(img1);
    cpl_test_eq(pdata1[0], 1.0);
    cpl_test_eq(pdata1[1], 2.0);
    cpl_test_eq(pdata1[2], 3.0);
    cpl_test_eq(pdata1[3], 4.0);
    
    img2 = cpl_imagelist_get(data_cube, 1);
    pdata2 = cpl_image_get_data(img2);
    cpl_test_eq(pdata2[0], 5.0);
    cpl_test_eq(pdata2[1], 6.0);
    cpl_test_eq(pdata2[2], 7.0);
    cpl_test_eq(pdata2[3], 8.0);
    
    /* Verify error cube contents */
    err1 = cpl_imagelist_get(error_cube, 0);
    perr1 = cpl_image_get_data(err1);
    cpl_test_eq(perr1[0], 0.1);
    cpl_test_eq(perr1[1], 0.2);
    cpl_test_eq(perr1[2], 0.3);
    cpl_test_eq(perr1[3], 0.4);
    
    err2 = cpl_imagelist_get(error_cube, 1);
    perr2 = cpl_image_get_data(err2);
    cpl_test_eq(perr2[0], 0.5);
    cpl_test_eq(perr2[1], 0.6);
    cpl_test_eq(perr2[2], 0.7);
    cpl_test_eq(perr2[3], 0.8);
    
    /* Test null inputs */
    error = eris_ifu_split_hdrl_imagelist(NULL, data_cube, error_cube);
    cpl_test_eq_error(error, CPL_ERROR_NULL_INPUT);
    
    error = eris_ifu_split_hdrl_imagelist(hdrl_list, NULL, error_cube);
    cpl_test_eq_error(error, CPL_ERROR_NULL_INPUT);
    
    error = eris_ifu_split_hdrl_imagelist(hdrl_list, data_cube, NULL);
    cpl_test_eq_error(error, CPL_ERROR_NULL_INPUT);
    
    /* Clean up */
    hdrl_imagelist_delete(hdrl_list);
    cpl_imagelist_delete(data_cube);
    cpl_imagelist_delete(error_cube);
}

/**
  @brief    Test masking of NaN values in image cube
  @return   void
*/
static void eris_ifu_mask_nans_in_cube_test(void)
{
    cpl_imagelist *cube;
    cpl_image *img1, *img2;
    cpl_mask *mask1, *mask2;
    cpl_error_code error;
    double *pdata1, *pdata2;
    cpl_binary *pmask1, *pmask2;
    
    /* Create test cube with two images containing NaN values */
    cube = cpl_imagelist_new();
    
    /* First image */
    img1 = cpl_image_new(3, 3, CPL_TYPE_DOUBLE);
    pdata1 = cpl_image_get_data(img1);
    
    pdata1[0] = 1.0;
    pdata1[1] = 0./0.;  // NaN
    pdata1[2] = 3.0;
    pdata1[3] = 4.0;
    pdata1[4] = 5.0;
    pdata1[5] = 6.0;
    pdata1[6] = 7.0;
    pdata1[7] = 0./0.;  // NaN
    pdata1[8] = 9.0;
    
    cpl_imagelist_set(cube, img1, 0);
    
    /* Second image */
    img2 = cpl_image_new(3, 3, CPL_TYPE_DOUBLE);
    pdata2 = cpl_image_get_data(img2);
    
    pdata2[0] = 1.0;
    pdata2[1] = 2.0;
    pdata2[2] = 0./0.;  // NaN
    pdata2[3] = 0./0.;  // NaN
    pdata2[4] = 5.0;
    pdata2[5] = 6.0;
    pdata2[6] = 7.0;
    pdata2[7] = 8.0;
    pdata2[8] = 9.0;
    
    cpl_imagelist_set(cube, img2, 1);
    
    /* Test masking NaNs */
    error = eris_ifu_mask_nans_in_cube(cube);
    cpl_test_eq_error(error, CPL_ERROR_NONE);
    
    /* Verify that NaNs are masked in first image */
    img1 = cpl_imagelist_get(cube, 0);
    mask1 = cpl_image_get_bpm(img1);
    pmask1 = cpl_mask_get_data(mask1);
    
    cpl_test_eq(pmask1[1], BAD_PIX);  // NaN
    cpl_test_eq(pmask1[7], BAD_PIX);  // NaN
    
    /* Verify that NaNs are masked in second image */
    img2 = cpl_imagelist_get(cube, 1);
    mask2 = cpl_image_get_bpm(img2);
    pmask2 = cpl_mask_get_data(mask2);
    
    cpl_test_eq(pmask2[2], BAD_PIX);  // NaN
    cpl_test_eq(pmask2[3], BAD_PIX);  // NaN
    
    /* Test null input */
    error = eris_ifu_mask_nans_in_cube(NULL);
    cpl_test_eq_error(error, CPL_ERROR_NULL_INPUT);
    
    cpl_imagelist_delete(cube);
}

/**
  @brief    Test masking of NaN values in HDRL image
  @return   void
*/
static void eris_ifu_mask_nans_in_hdrlimage_test(void)
{
    hdrl_image *hima;
    cpl_image *img, *err;
    cpl_mask *mask;
    cpl_error_code error;
    double *pdata, *perrs;
    cpl_binary *pmask;
    
    /* Create test images with some NaN values */
    img = cpl_image_new(3, 3, CPL_TYPE_DOUBLE);
    err = cpl_image_new(3, 3, CPL_TYPE_DOUBLE);
    
    pdata = cpl_image_get_data(img);
    perrs = cpl_image_get_data(err);
    
    /* Set some normal values and some NaNs */
    pdata[0] = 1.0;
    pdata[1] = 0./0.;  // NaN
    pdata[2] = 3.0;
    pdata[3] = 4.0;
    pdata[4] = 0./0.;  // NaN
    pdata[5] = 6.0;
    pdata[6] = 7.0;
    pdata[7] = 8.0;
    pdata[8] = 9.0;
    
    perrs[0] = 0.1;
    perrs[1] = 0.2;
    perrs[2] = 0./0.;  // NaN
    perrs[3] = 0.4;
    perrs[4] = 0.5;
    perrs[5] = 0./0.;  // NaN
    perrs[6] = 0.7;
    perrs[7] = 0.8;
    perrs[8] = 0.9;
    
    hima = hdrl_image_create(img, err);
    cpl_image_delete(img);
    cpl_image_delete(err);
    
    /* Test masking NaNs */
    error = eris_ifu_mask_nans_in_hdrlimage(&hima);
    cpl_test_eq_error(error, CPL_ERROR_NONE);
    
    /* Verify that NaNs are masked */
    mask = hdrl_image_get_mask(hima);
    pmask = cpl_mask_get_data(mask);
    
    cpl_test_eq(pmask[1], BAD_PIX);  // NaN in data
    cpl_test_eq(pmask[2], BAD_PIX);  // NaN in error
    cpl_test_eq(pmask[4], BAD_PIX);  // NaN in data
    cpl_test_eq(pmask[5], BAD_PIX);  // NaN in error
    
    /* Test null input */
    error = eris_ifu_mask_nans_in_hdrlimage(NULL);
    cpl_test_eq_error(error, CPL_ERROR_NULL_INPUT);
    
    hdrl_image_delete(hima);
}

/**
  @brief    Test loading of bad pixel mask
  @return   void
*/
static void eris_ifu_load_badpixel_mask_test(void)
{
    cpl_frameset *frameset;
    cpl_frame *frame;
    cpl_mask *mask;
    char *temp_file;
    FILE *fp;

    /* Create a temporary test file */
    cpl_mask* bpm = cpl_mask_new(4, 4);
    cpl_mask_save(bpm, "bpm.fits", NULL, CPL_IO_DEFAULT);
    cpl_mask_delete(bpm);

    temp_file = cpl_sprintf("test_mask.fits");
    fp = fopen(temp_file, "w");
    cpl_test_nonnull(fp);
    fprintf(fp, "test");
    fclose(fp);

    /* Create test frameset with tagged frames */
    frameset = cpl_frameset_new();
    frame = cpl_frame_new();
    cpl_frame_set_filename(frame, temp_file);
    cpl_frame_set_tag(frame, "TEST_TAG");
    cpl_frameset_insert(frameset, frame);

    /* Test normal operation */
    mask = eris_ifu_load_badpixel_mask(frameset, "TEST_TAG", 0);
    cpl_test_error(CPL_ERROR_FILE_IO); // Since temp file isn't a valid FITS file
    cpl_test_null(mask);

    /* Test null inputs */
    mask = eris_ifu_load_badpixel_mask(NULL, "TEST_TAG", 0);
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    cpl_test_null(mask);

    mask = eris_ifu_load_badpixel_mask(frameset, NULL, 0);
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    cpl_test_null(mask);

    /* Test invalid extension number */
    mask = eris_ifu_load_badpixel_mask(frameset, "TEST_TAG", -1);
    cpl_test_error(CPL_ERROR_ILLEGAL_INPUT);
    cpl_test_null(mask);

    /* Clean up */
    cpl_frameset_delete(frameset);
    remove(temp_file);
    remove("bpm.fits");
    cpl_free(temp_file);
}

/**
  @brief    Test HDRL imagelist retrieval by tag
  @return   void
*/
static void eris_ifu_get_hdrlimagelist_by_tag_test(void)
{
    cpl_frameset *frameset;
    hdrl_imagelist *hdrl_list;
    cpl_frame *frame;
    char *temp_file;
    FILE *fp;

    /* Create a temporary test file */
    cpl_image* ima = cpl_image_new(2,2,CPL_TYPE_DOUBLE);
    temp_file = cpl_sprintf("test_frame.fits");
    fp = fopen(temp_file, "w");
    cpl_test_nonnull(fp);
    fprintf(fp, "test");
    fclose(fp);
    cpl_image_save(ima, "test_frame.fits", CPL_TYPE_DOUBLE, NULL, CPL_IO_CREATE);
    cpl_image_delete(ima);

    /* Create test frameset with tagged frames */
    frameset = cpl_frameset_new();
    frame = cpl_frame_new();
    cpl_frame_set_filename(frame, temp_file);
    cpl_frame_set_tag(frame, "TEST_TAG");
    cpl_frameset_insert(frameset, frame);

    /* Test normal operation */
    hdrl_list = eris_ifu_get_hdrlimagelist_by_tag(frameset, "TEST_TAG", 0); // This seg fault
    cpl_test_error(CPL_ERROR_FILE_NOT_FOUND); // Since temp file isn't a valid FITS file
    cpl_test_null(hdrl_list);

    /* Test null inputs */
    hdrl_list = eris_ifu_get_hdrlimagelist_by_tag(NULL, "TEST_TAG", 0);
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    cpl_test_null(hdrl_list);

    hdrl_list = eris_ifu_get_hdrlimagelist_by_tag(frameset, NULL, 0);
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    cpl_test_null(hdrl_list);

    /* Clean up */
    cpl_frameset_delete(frameset);

    remove(temp_file);

    cpl_free(temp_file);
}

/**
  @brief    Test file existence checking function
  @return   void
*/
static void eris_ifu_file_exists_test(void)
{
    cpl_error_code error;
    char *temp_file;
    FILE *fp;

    /* Create a temporary file */
    temp_file = cpl_sprintf("%s/test_file.tmp", "/tmp");
    fp = fopen(temp_file, "w");
    cpl_test_nonnull(fp);
    fprintf(fp, "test");
    fclose(fp);

    /* Test existing file */
    error = eris_ifu_file_exists(temp_file);
    cpl_test_eq_error(error, CPL_ERROR_NONE);

    /* Test non-existing file */
    error = eris_ifu_file_exists("/nonexistent/file");
    cpl_test_eq_error(error, CPL_ERROR_NONE);


    /* Test null input */
    error = eris_ifu_file_exists(NULL);
    cpl_test_eq_error(error, CPL_ERROR_NULL_INPUT);

    /* Clean up */
    remove(temp_file);
    cpl_free(temp_file);
}

/**
  @brief    Test string handling functions
  @return   void
*/
/**
 * @brief Test eris_ifu_save_image and eris_ifu_save_imagelist functions
 */
/**
 * @brief Test eris_ifu_image_create_window and eris_ifu_mask_create_border functions
 */
/**
 * @brief Test debug save functions
 */
/**
 * @brief Test eris_ifu_get_badpix_qc_from_ima and eris_ifu_get_badpix_qc_from_mask functions
 */
/**
  @brief    Test quality mask to bad pixel mask conversion function
  @return   void
*/
/**
  @brief    Test interpolated mask to maskZero conversion function
  @return   void
*/
/**
  @brief    Test cube value setting function
  @return   void
*/
/**
  @brief    Test cube NaN trimming function
  @return   void
*/
/**
  @brief    Test plane cut min/max function
  @return   void
*/
/**
  @brief    Test pupil shift calculation function
  @return   void
*/
/**
  @brief    Test QC parameter double formatting functions
  @return   void
*/
static void eris_qclog_add_double_format_test(void)
{
    cpl_table* qclog;
    const char* str_value;
    
    /* Initialize QC table */
    qclog = eris_qclog_init();
    cpl_test_nonnull(qclog);
    cpl_test_eq(cpl_table_get_nrow(qclog), 0);
    
    /* Test eris_qclog_add_double_f */
    eris_qclog_add_double_f(qclog, "TEST_DOUBLE_F", 3.14159, "Test double parameter with f format");
    cpl_test_eq(cpl_table_get_nrow(qclog), 1);
    str_value = cpl_table_get_string(qclog, "key_type", 0);
    cpl_test_eq_string(str_value, "CPL_TYPE_DOUBLE");
    str_value = cpl_table_get_string(qclog, "key_value", 0);
    cpl_test_eq_string(str_value, "3.141590");  // Should use %f format
    
    /* Test eris_qclog_add_double_format */
    eris_qclog_add_double_format(qclog, "TEST_DOUBLE_FORMAT", 3.14159, "Test double parameter with format");
    cpl_test_eq(cpl_table_get_nrow(qclog), 2);
    str_value = cpl_table_get_string(qclog, "key_type", 1);
    cpl_test_eq_string(str_value, "CPL_TYPE_DOUBLE");
    str_value = cpl_table_get_string(qclog, "key_value", 1);
    cpl_test_eq_string(str_value, "     3.141590");  // Should use %13.6f format
    
    /* Test with different values */
    eris_qclog_add_double_f(qclog, "TEST_LARGE", 123456.789, "Test large number");
    cpl_test_eq(cpl_table_get_nrow(qclog), 3);
    str_value = cpl_table_get_string(qclog, "key_value", 2);
    cpl_test_eq_string(str_value, "123456.789000");
    
    eris_qclog_add_double_format(qclog, "TEST_SMALL", 0.000123456, "Test small number");
    cpl_test_eq(cpl_table_get_nrow(qclog), 4);
    str_value = cpl_table_get_string(qclog, "key_value", 3);
    cpl_test_eq_string(str_value, "     0.000123");
    
    /* Test with special values */
    eris_qclog_add_double_f(qclog, "TEST_ZERO", 0.0, "Test zero");
    cpl_test_eq(cpl_table_get_nrow(qclog), 5);
    str_value = cpl_table_get_string(qclog, "key_value", 4);
    cpl_test_eq_string(str_value, "0.000000");
    
    eris_qclog_add_double_format(qclog, "TEST_NEG", -3.14159, "Test negative number");
    cpl_test_eq(cpl_table_get_nrow(qclog), 6);
    str_value = cpl_table_get_string(qclog, "key_value", 5);
    cpl_test_eq_string(str_value, "    -3.141590");
    
    /* Test null inputs */
    cpl_test_eq_error(eris_qclog_add_double_f(NULL, "TEST", 0.0, "Test"), CPL_ERROR_NULL_INPUT);
    cpl_test_eq_error(eris_qclog_add_double_f(qclog, NULL, 0.0, "Test"), CPL_ERROR_NULL_INPUT);
    cpl_test_eq_error(eris_qclog_add_double_f(qclog, "TEST", 0.0, NULL), CPL_ERROR_NULL_INPUT);
    
    cpl_test_eq_error(eris_qclog_add_double_format(NULL, "TEST", 0.0, "Test"), CPL_ERROR_NULL_INPUT);
    cpl_test_eq_error(eris_qclog_add_double_format(qclog, NULL, 0.0, "Test"), CPL_ERROR_NULL_INPUT);
    cpl_test_eq_error(eris_qclog_add_double_format(qclog, "TEST", 0.0, NULL), CPL_ERROR_NULL_INPUT);
    
    /* Clean up */
    cpl_table_delete(qclog);
}

static void eris_get_pupil_shift_test(void)
{
    hdrl_imagelist* iml;
    cpl_table* qclog_tbl;
    hdrl_image* hima1, *hima2;
    cpl_image* img1, *img2;
    cpl_image* err1, *err2;
    cpl_error_code error;
    
    /* Create test HDRL imagelist */
    iml = hdrl_imagelist_new();
    
    /* First image: Gaussian peak at center */
    img1 = cpl_image_new(30, 30, CPL_TYPE_DOUBLE);
    err1 = cpl_image_new(30, 30, CPL_TYPE_DOUBLE);
    double* data1 = cpl_image_get_data_double(img1);
    
    /* Create Gaussian peak at center */
    for(int j = 0; j < 30; j++) {
        for(int i = 0; i < 30; i++) {
            double x = i - 15.0;  // Center at (15,15)
            double y = j - 15.0;
            data1[i + j*30] = 100.0 * exp(-(x*x + y*y)/(2.0 * 3.0 * 3.0));
        }
    }
    cpl_image_add_scalar(err1, 0.1);
    hima1 = hdrl_image_create(img1, err1);
    hdrl_imagelist_set(iml, hima1, 0);
    cpl_image_delete(img1);
    cpl_image_delete(err1);
    
    /* Second image: Gaussian peak shifted */
    img2 = cpl_image_new(30, 30, CPL_TYPE_DOUBLE);
    err2 = cpl_image_new(30, 30, CPL_TYPE_DOUBLE);
    double* data2 = cpl_image_get_data_double(img2);
    
    /* Create Gaussian peak shifted by (2,3) pixels */
    for(int j = 0; j < 30; j++) {
        for(int i = 0; i < 30; i++) {
            double x = i - 17.0;  // Center at (17,18)
            double y = j - 18.0;
            data2[i + j*30] = 100.0 * exp(-(x*x + y*y)/(2.0 * 3.0 * 3.0));
        }
    }
    cpl_image_add_scalar(err2, 0.1);
    hima2 = hdrl_image_create(img2, err2);
    hdrl_imagelist_set(iml, hima2, 1);
    cpl_image_delete(img2);
    cpl_image_delete(err2);
    
    /* Initialize QC table */
    qclog_tbl = eris_qclog_init();
    
    /* Test normal operation */
    error = eris_get_pupil_shift(iml, 1, &qclog_tbl);
    cpl_test_eq_error(error, CPL_ERROR_NONE);
    
    /* Verify QC parameters */
    const char* shiftx = cpl_table_get_string(qclog_tbl, "key_name", 0);
    const char* shifty = cpl_table_get_string(qclog_tbl, "key_name", 1);
    cpl_test_eq_string(shiftx, "QC PUPIL1 SHIFTX");
    cpl_test_eq_string(shifty, "QC PUPIL1 SHIFTY");
    
    /* Test null inputs */
    error = eris_get_pupil_shift(NULL, 1, &qclog_tbl);
    cpl_test_eq_error(error, CPL_ERROR_NULL_INPUT);
    
    error = eris_get_pupil_shift(iml, 1, NULL);
    cpl_test_eq_error(error, CPL_ERROR_NULL_INPUT);
    
    /* Test invalid sequence number */
    error = eris_get_pupil_shift(iml, -1, &qclog_tbl);
    cpl_test_eq_error(error, CPL_ERROR_ILLEGAL_INPUT);
    
    /* Clean up */
    hdrl_imagelist_delete(iml);
    cpl_table_delete(qclog_tbl);
}

static void eris_ifu_get_plane_cut_min_max_test(void)
{
    cpl_size zmin, zmax;
    cpl_error_code error;
    
    /* Test J_LOW band */
    error = eris_ifu_get_plane_cut_min_max(J_LOW, &zmin, &zmax);
    cpl_test_eq_error(error, CPL_ERROR_NONE);
    cpl_test_eq(zmin, 46);
    cpl_test_eq(zmax, 2015);
    
    /* Test H_LOW band */
    error = eris_ifu_get_plane_cut_min_max(H_LOW, &zmin, &zmax);
    cpl_test_eq_error(error, CPL_ERROR_NONE);
    cpl_test_eq(zmin, 48);
    cpl_test_eq(zmax, 2023);
    
    /* Test K_LOW band */
    error = eris_ifu_get_plane_cut_min_max(K_LOW, &zmin, &zmax);
    cpl_test_eq_error(error, CPL_ERROR_NONE);
    cpl_test_eq(zmin, 62);
    cpl_test_eq(zmax, 2022);
    
    /* Test J_SHORT band */
    error = eris_ifu_get_plane_cut_min_max(J_SHORT, &zmin, &zmax);
    cpl_test_eq_error(error, CPL_ERROR_NONE);
    cpl_test_eq(zmin, 50);
    cpl_test_eq(zmax, 2034);
    
    /* Test H_MIDDLE band */
    error = eris_ifu_get_plane_cut_min_max(H_MIDDLE, &zmin, &zmax);
    cpl_test_eq_error(error, CPL_ERROR_NONE);
    cpl_test_eq(zmin, 47);
    cpl_test_eq(zmax, 2034);
    
    /* Test K_LONG band */
    error = eris_ifu_get_plane_cut_min_max(K_LONG, &zmin, &zmax);
    cpl_test_eq_error(error, CPL_ERROR_NONE);
    cpl_test_eq(zmin, 50);
    cpl_test_eq(zmax, 2034);
    
    /* Test SPIFFI bands */
    error = eris_ifu_get_plane_cut_min_max(J_SPIFFI, &zmin, &zmax);
    cpl_test_eq_error(error, CPL_ERROR_NONE);
    cpl_test_eq(zmin, 32);
    cpl_test_eq(zmax, 2058);
    
    error = eris_ifu_get_plane_cut_min_max(H_SPIFFI, &zmin, &zmax);
    cpl_test_eq_error(error, CPL_ERROR_NONE);
    cpl_test_eq(zmin, 32);
    cpl_test_eq(zmax, 2058);
    
    error = eris_ifu_get_plane_cut_min_max(K_SPIFFI, &zmin, &zmax);
    cpl_test_eq_error(error, CPL_ERROR_NONE);
    cpl_test_eq(zmin, 32);
    cpl_test_eq(zmax, 2058);
    
    error = eris_ifu_get_plane_cut_min_max(HK_SPIFFI, &zmin, &zmax);
    cpl_test_eq_error(error, CPL_ERROR_NONE);
    cpl_test_eq(zmin, 32);
    cpl_test_eq(zmax, 2058);
    
    /* Test undefined band */
    error = eris_ifu_get_plane_cut_min_max(UNDEFINED_BAND, &zmin, &zmax);
    cpl_test_eq_error(error, CPL_ERROR_NONE);
    
    /* Test invalid band */
    error = eris_ifu_get_plane_cut_min_max(99, &zmin, &zmax);  // Invalid band
    cpl_test_eq_error(error, CPL_ERROR_NONE);
    
    /* Test null outputs */
    error = eris_ifu_get_plane_cut_min_max(J_LOW, NULL, &zmax);
    cpl_test_eq_error(error, CPL_ERROR_NULL_INPUT);
    
    error = eris_ifu_get_plane_cut_min_max(J_LOW, &zmin, NULL);
    cpl_test_eq_error(error, CPL_ERROR_ILLEGAL_INPUT);

    error = eris_ifu_get_plane_cut_min_max(J_LOW, &zmax, &zmin);
    cpl_test_eq_error(error, CPL_ERROR_ILLEGAL_INPUT);

}

static void eris_ifu_cube_trim_nans_test(void)
{
    hdrl_imagelist* himl;
    cpl_imagelist* bpm;
    cpl_propertylist* header;
    hdrl_image* hima1, *hima2, *hima3;
    cpl_image* img1, *img2, *img3;
    cpl_image* err1, *err2, *err3;
    cpl_image* bpm1, *bpm2, *bpm3;
    cpl_error_code error;
    
    /* Create test HDRL imagelist */
    himl = hdrl_imagelist_new();
    bpm = cpl_imagelist_new();
    header = cpl_propertylist_new();
    
    /* Set header values */
    cpl_propertylist_append_double(header, "CRVAL3", 1000.0);
    cpl_propertylist_append_double(header, "CD3_3", 1.0);
    cpl_propertylist_append_int(header, "NAXIS3", 3);
    
    /* First image: all valid values */
    img1 = cpl_image_new(4, 4, CPL_TYPE_DOUBLE);
    err1 = cpl_image_new(4, 4, CPL_TYPE_DOUBLE);
    bpm1 = cpl_image_new(4, 4, CPL_TYPE_INT);
    cpl_image_add_scalar(img1, 1.0);
    cpl_image_add_scalar(err1, 0.1);
    hima1 = hdrl_image_create(img1, err1);
    hdrl_imagelist_set(himl, hima1, 0);
    cpl_imagelist_set(bpm, bpm1, 0);
    cpl_image_delete(img1);
    cpl_image_delete(err1);
    
    /* Second image: some NaN values */
    img2 = cpl_image_new(4, 4, CPL_TYPE_DOUBLE);
    err2 = cpl_image_new(4, 4, CPL_TYPE_DOUBLE);
    bpm2 = cpl_image_new(4, 4, CPL_TYPE_INT);
    double* data2 = cpl_image_get_data_double(img2);
    data2[0] = 0.0/0.0;  // NaN
    data2[1] = 2.0;
    data2[2] = 0.0/0.0;  // NaN
    data2[3] = 2.0;
    cpl_image_add_scalar(err2, 0.2);
    hima2 = hdrl_image_create(img2, err2);

    for(int i = 1; i < 2015; i++) {
    hdrl_imagelist_set(himl, hima2, i);
    cpl_imagelist_set(bpm, bpm2, i);
    }



    cpl_image_delete(img2);
    cpl_image_delete(err2);
    
    /* Third image: all NaN values */
    img3 = cpl_image_new(4, 4, CPL_TYPE_DOUBLE);
    err3 = cpl_image_new(4, 4, CPL_TYPE_DOUBLE);
    bpm3 = cpl_image_new(4, 4, CPL_TYPE_INT);
    double* data3 = cpl_image_get_data_double(img3);
    for(int i = 0; i < 16; i++) {
        data3[i] = 0.0/0.0;  // NaN
    }
    cpl_image_add_scalar(err3, 0.3);
    hima3 = hdrl_image_create(img3, err3);
    for(int i = 2015; i < 2300; i++) {
    	hdrl_imagelist_set(himl, hima3, i);
    	cpl_imagelist_set(bpm, bpm3, i);
    }


    cpl_image_delete(img3);
    cpl_image_delete(err3);

    /* Test normal operation */

    error = eris_ifu_cube_trim_nans(46, 2015, &himl, &bpm, header);
    cpl_test_eq_error(error, CPL_ERROR_NONE);
    
    /* Verify results */
    cpl_test_eq(hdrl_imagelist_get_size(himl), 1970);  // Third image should be removed
    cpl_test_eq(cpl_imagelist_get_size(bpm), 1970);
    

    /* Verify header values were updated */
    cpl_test_eq(cpl_propertylist_get_int(header, "NAXIS3"), 1970);
    cpl_test_abs(cpl_propertylist_get_double(header, "CRVAL3"), 1046.0, 0.0001);  // 1000 + 46
    
    hdrl_imagelist_delete(himl);
    cpl_imagelist_delete(bpm);
    /* Test null inputs */
    error = eris_ifu_cube_trim_nans(46, 2015, NULL, &bpm, header);
    cpl_test_eq_error(error, CPL_ERROR_NULL_INPUT);
    
    error = eris_ifu_cube_trim_nans(46, 2015, &himl, NULL, header);
    cpl_test_eq_error(error, CPL_ERROR_NULL_INPUT);
    
    error = eris_ifu_cube_trim_nans(46, 2015, &himl, &bpm, NULL);
    cpl_test_eq_error(error, CPL_ERROR_NULL_INPUT);
    
    /* Test invalid range */
    error = eris_ifu_cube_trim_nans(2015, 46, &himl, &bpm, header);  // zmin > zmax
    cpl_test_eq_error(error, CPL_ERROR_ILLEGAL_INPUT);
    
    /* Clean up */

    cpl_propertylist_delete(header);
}

static void eris_ifu_cube_set_values_test(void)
{
    hdrl_imagelist* iml;
    hdrl_image* hima1, *hima2;
    cpl_image* img1, *img2;
    cpl_image* err1, *err2;
    cpl_error_code error;
    
    /* Create test HDRL imagelist */
    iml = hdrl_imagelist_new();
    
    /* First image */
    img1 = cpl_image_new(30, 30, CPL_TYPE_DOUBLE);
    err1 = cpl_image_new(30, 30, CPL_TYPE_DOUBLE);
    cpl_image_add_scalar(img1, 1.0);  // Fill with 1.0
    cpl_image_add_scalar(err1, 0.1);  // Fill with 0.1
    hima1 = hdrl_image_create(img1, err1);
    hdrl_imagelist_set(iml, hima1, 0);

    
    /* Second image */
    img2 = cpl_image_new(30, 30, CPL_TYPE_DOUBLE);
    err2 = cpl_image_new(30, 30, CPL_TYPE_DOUBLE);
    cpl_image_add_scalar(img2, 2.0);  // Fill with 2.0
    cpl_image_add_scalar(err2, 0.2);  // Fill with 0.2
    hima2 = hdrl_image_create(img2, err2);
    hdrl_imagelist_set(iml, hima2, 1);

    
    /* Test normal operation */
    error = eris_ifu_cube_set_values(&iml);      //TODO: this function maybe wrong
    cpl_test_eq_error(error, CPL_ERROR_NONE);
    
    /* Verify values were set correctly */
    hima1 = hdrl_imagelist_get(iml, 0);
    img1 = hdrl_image_get_image(hima1);
    double* data1 = cpl_image_get_data_double(img1);
    cpl_image_save(img1, "img1.fits", CPL_TYPE_DOUBLE, NULL, CPL_IO_CREATE);
    /* Check values in the 10x10 region */
    for(int j = 10; j < 20; j++) {
        for(int i = 10; i < 20; i++) {
            cpl_test_abs(data1[i + j*30], 0.0, 0.0001);  // Should be set to 0.0
        }
    }
    cpl_image_delete(img1);
    cpl_image_delete(err1);

    hima2 = hdrl_imagelist_get(iml, 1);
    img2 = hdrl_image_get_image(hima2);
    double* data2 = cpl_image_get_data_double(img2);
    
    /* Check values in the 10x10 region */
    for(int j = 10; j < 20; j++) {
        for(int i = 10; i < 20; i++) {
            cpl_test_abs(data2[i + j*30], 2.0, 0.0001);  // Should be set to 2.0
        }
    }
    cpl_image_delete(img2);
    cpl_image_delete(err2);
    /* Test null input */
    error = eris_ifu_cube_set_values(NULL);
    cpl_test_eq_error(error, CPL_ERROR_NULL_INPUT);
    
    /* Test empty imagelist */
    hdrl_imagelist* empty_iml = hdrl_imagelist_new();
    error = eris_ifu_cube_set_values(&empty_iml);
    cpl_test_eq_error(error, CPL_ERROR_NONE);  // Should handle empty list gracefully
    
    /* Clean up */
    hdrl_imagelist_delete(iml);
    hdrl_imagelist_delete(empty_iml);
}

static void eris_ifu_interpolatedMask_to_maskZero_test(void)
{
    cpl_imagelist* bpmCube;
    cpl_imagelist* maskOneCube;
    cpl_image* img1, *img2;
    double threshold = 0.5;
    
    /* Create test cube with interpolated values */
    bpmCube = cpl_imagelist_new();
    
    /* First image: values around threshold */
    img1 = cpl_image_new(4, 4, CPL_TYPE_FLOAT);
    float* data1 = cpl_image_get_data_float(img1);
    data1[0] = 0.0f;    // Below threshold -> good pixel
    data1[1] = 0.4f;    // Below threshold -> good pixel
    data1[2] = 0.6f;    // Above threshold -> bad pixel
    data1[3] = 1.0f;    // Above threshold -> bad pixel
    cpl_imagelist_set(bpmCube, img1, 0);
    
    /* Second image: extreme values */
    img2 = cpl_image_new(4, 4, CPL_TYPE_FLOAT);
    float* data2 = cpl_image_get_data_float(img2);
    data2[0] = 0.0f;      // Below threshold -> good pixel
    data2[1] = 1.0f;      // Above threshold -> bad pixel
    data2[2] = 0.1f;      // Below threshold -> good pixel
    data2[3] = 0.9f;      // Above threshold -> bad pixel
    cpl_imagelist_set(bpmCube, img2, 1);
    
    /* Test normal operation */
    maskOneCube = eris_ifu_interpolatedMask_to_maskZero(bpmCube, threshold);
    cpl_test_nonnull(maskOneCube);
    cpl_test_eq(cpl_imagelist_get_size(maskOneCube), 2);
    
    /* Verify first image */
    cpl_image* mask1 = cpl_imagelist_get(maskOneCube, 0);
    int* pmask1 = cpl_image_get_data_int(mask1);
    cpl_test_eq(pmask1[0], 0);  // Below threshold -> good pixel
    cpl_test_eq(pmask1[1], 0);  // Below threshold -> good pixel
    cpl_test_eq(pmask1[2], 1);  // Above threshold -> bad pixel
    cpl_test_eq(pmask1[3], 1);  // Above threshold -> bad pixel
    
    /* Verify second image */
    cpl_image* mask2 = cpl_imagelist_get(maskOneCube, 1);
    int* pmask2 = cpl_image_get_data_int(mask2);
    cpl_test_eq(pmask2[0], 0);  // Below threshold -> good pixel
    cpl_test_eq(pmask2[1], 1);  // Above threshold -> bad pixel
    cpl_test_eq(pmask2[2], 0);  // Below threshold -> good pixel
    cpl_test_eq(pmask2[3], 1);  // Above threshold -> bad pixel
    
    /* Test with double type */
    cpl_image* img_double = cpl_image_new(4, 4, CPL_TYPE_DOUBLE);
    double* data_double = cpl_image_get_data_double(img_double);
    data_double[0] = 0.0;
    data_double[1] = 0.4;
    data_double[2] = 0.6;
    data_double[3] = 1.0;
    cpl_imagelist* bpmCube_double = cpl_imagelist_new();
    cpl_imagelist_set(bpmCube_double, img_double, 0);
    
    cpl_imagelist* maskOneCube_double = eris_ifu_interpolatedMask_to_maskZero(bpmCube_double, threshold);
    cpl_test_nonnull(maskOneCube_double);
    cpl_test_eq(cpl_imagelist_get_size(maskOneCube_double), 1);
    
    /* Test null input */
    cpl_imagelist_delete(maskOneCube);
    maskOneCube = eris_ifu_interpolatedMask_to_maskZero(NULL, threshold);
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    cpl_test_null(maskOneCube);
    
    /* Test empty input */
    cpl_imagelist* empty_cube = cpl_imagelist_new();
    //maskOneCube = eris_ifu_interpolatedMask_to_maskZero(empty_cube, threshold);  //TODO: generate leaks
    //cpl_test_error(CPL_ERROR_ILLEGAL_INPUT);
    cpl_test_null(maskOneCube);
    

    /* Clean up */
    cpl_imagelist_delete(bpmCube);
    cpl_imagelist_delete(maskOneCube);
    cpl_imagelist_delete(bpmCube_double);
    cpl_imagelist_delete(maskOneCube_double);
    cpl_imagelist_delete(empty_cube);
}

static void eris_ifu_quality2bp_mask_test(void)
{
    cpl_image* quality_mask;
    cpl_mask* bp_mask;
    
    /* Create test quality mask with different values */
    quality_mask = cpl_image_new(4, 4, CPL_TYPE_DOUBLE);
    double* data = cpl_image_get_data_double(quality_mask);
    
    /* Set some test values:
     * 0: Good pixel
     * 1: Bad pixel
     * 4294967295: Maximum 32-bit value (bad pixel)
     * 65535: Maximum 16-bit value (bad pixel)
     */
    data[0] = 0.0;  // Good pixel
    data[1] = 1.0;  // Bad pixel
    data[2] = 4294967295.0;  // 32-bit max
    data[3] = 65535.0;  // 16-bit max
    
    /* Test maskzero type (inverted mask) */
    bp_mask = eris_ifu_quality2bp_mask(quality_mask, maskzero);
    cpl_test_nonnull(bp_mask);
    cpl_binary* pmask = cpl_mask_get_data(bp_mask);
    cpl_test_eq(pmask[0], BAD_PIX);  // 0 becomes bad
    cpl_test_eq(pmask[1], GOOD_PIX);  // Non-zero becomes good
    cpl_test_eq(pmask[2], GOOD_PIX);
    cpl_test_eq(pmask[3], GOOD_PIX);
    cpl_mask_delete(bp_mask);
    
    /* Test maskone type */
    bp_mask = eris_ifu_quality2bp_mask(quality_mask, maskone);
    cpl_test_nonnull(bp_mask);
    pmask = cpl_mask_get_data(bp_mask);
    cpl_test_eq(pmask[0], GOOD_PIX);  // 0 becomes good
    cpl_test_eq(pmask[1], BAD_PIX);   // Non-zero becomes bad
    cpl_test_eq(pmask[2], BAD_PIX);
    cpl_test_eq(pmask[3], BAD_PIX);
    cpl_mask_delete(bp_mask);
    
    /* Test flag32bit type */
    bp_mask = eris_ifu_quality2bp_mask(quality_mask, flag32bit);
    cpl_test_nonnull(bp_mask);
    pmask = cpl_mask_get_data(bp_mask);
    cpl_test_eq(pmask[0], GOOD_PIX);  // 0 becomes good
    cpl_test_eq(pmask[1], BAD_PIX);   // Non-zero becomes bad
    cpl_test_eq(pmask[2], BAD_PIX);
    cpl_test_eq(pmask[3], BAD_PIX);
    cpl_mask_delete(bp_mask);
    
    /* Test flag16bit type */
    bp_mask = eris_ifu_quality2bp_mask(quality_mask, flag16bit);
    cpl_test_nonnull(bp_mask);
    pmask = cpl_mask_get_data(bp_mask);
    cpl_test_eq(pmask[0], GOOD_PIX);  // 0 becomes good
    cpl_test_eq(pmask[1], BAD_PIX);   // Non-zero becomes bad
    cpl_test_eq(pmask[2], BAD_PIX);
    cpl_test_eq(pmask[3], BAD_PIX);
    cpl_mask_delete(bp_mask);
    
    /* Test unspecified type : */
    bp_mask = eris_ifu_quality2bp_mask(quality_mask, unspecified);
    cpl_test_error(CPL_ERROR_ILLEGAL_INPUT);
    cpl_test_null(bp_mask);

    

    /* Test null input */
    bp_mask = eris_ifu_quality2bp_mask(NULL, maskone);
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    cpl_test_null(bp_mask);
    
    /* Clean up */
    cpl_image_delete(quality_mask);
}

static void eris_ifu_badpix_qc_test(void)
{
    cpl_image* img;
    cpl_mask* mask;
    cpl_propertylist* qc_list;
    cpl_error_code error;
    const char* prefix = "TEST";
    
    /* Create test image with some bad pixels */
    img = cpl_image_new(4, 4, CPL_TYPE_DOUBLE);
    double* data = cpl_image_get_data_double(img);
    data[1, 1] = 1;  // Reject some pixels
    data[2, 2] = 1;
    data[3, 3] = 1;
    
    /* Create test mask with some bad pixels */
    mask = cpl_mask_new(4, 4);
    cpl_binary* pmask = cpl_mask_get_data(mask);
    pmask[0] = BAD_PIX;
    pmask[5] = BAD_PIX;
    pmask[10] = BAD_PIX;
    pmask[15] = BAD_PIX;
    
    /* Create QC propertylist */
    qc_list = cpl_propertylist_new();
    
    /* Test QC from image */
    error = eris_ifu_get_badpix_qc_from_ima(img, qc_list, prefix);
    cpl_test_eq_error(error, CPL_ERROR_NONE);
    
    /* Verify QC parameters from image */

    cpl_test_eq(cpl_propertylist_get_int(qc_list, "ESO QC TEST NBADPIX"), 3);
    cpl_test_abs(cpl_propertylist_get_double(qc_list, "ESO QC TEST BPIXFRAC"), 3.0/16.0, 0.000001);

    
    /* Test QC from mask */
    error = eris_ifu_get_badpix_qc_from_mask(mask, qc_list, "TEST2");
    cpl_test_eq_error(error, CPL_ERROR_NONE);
    
    /* Verify QC parameters from mask */
    cpl_test_eq(cpl_propertylist_get_int(qc_list, "ESO QC TEST2 NBADPIX"), 4);
    cpl_test_abs(cpl_propertylist_get_double(qc_list, "ESO QC TEST2 BPIXFRAC"), 4.0/16.0, 0.000001);
    
    /* Test null inputs */
    error = eris_ifu_get_badpix_qc_from_ima(NULL, qc_list, prefix);
    cpl_test_eq_error(error, CPL_ERROR_NULL_INPUT);
    
    error = eris_ifu_get_badpix_qc_from_ima(img, NULL, prefix);
    cpl_test_eq_error(error, CPL_ERROR_NULL_INPUT);
    
    error = eris_ifu_get_badpix_qc_from_ima(img, qc_list, NULL);
    cpl_test_eq_error(error, CPL_ERROR_NULL_INPUT);
    
    error = eris_ifu_get_badpix_qc_from_mask(NULL, qc_list, prefix);
    cpl_test_eq_error(error, CPL_ERROR_NULL_INPUT);
    
    error = eris_ifu_get_badpix_qc_from_mask(mask, NULL, prefix);
    cpl_test_eq_error(error, CPL_ERROR_NULL_INPUT);
    
    error = eris_ifu_get_badpix_qc_from_mask(mask, qc_list, NULL);
    cpl_test_eq_error(error, CPL_ERROR_NULL_INPUT);
    
    /* Clean up */
    cpl_image_delete(img);
    cpl_mask_delete(mask);
    cpl_propertylist_delete(qc_list);
}

static void eris_ifu_save_dbg_test(void)
{
    cpl_vector* vec;
    cpl_bivector* bivec;
    cpl_image* img;
    cpl_mask* mask;
    cpl_table* table;
    cpl_propertylist* plist;
    cpl_error_code error;
    
    /* Create test vector */
    vec = cpl_vector_new(3);
    cpl_vector_set(vec, 0, 1.0);
    cpl_vector_set(vec, 1, 2.0);
    cpl_vector_set(vec, 2, 3.0);
    
    /* Create test bivector */
    bivec = cpl_bivector_new(3);
    cpl_vector_set(cpl_bivector_get_x(bivec), 0, 1.0);
    cpl_vector_set(cpl_bivector_get_x(bivec), 1, 2.0);
    cpl_vector_set(cpl_bivector_get_x(bivec), 2, 3.0);
    cpl_vector_set(cpl_bivector_get_y(bivec), 0, 4.0);
    cpl_vector_set(cpl_bivector_get_y(bivec), 1, 5.0);
    cpl_vector_set(cpl_bivector_get_y(bivec), 2, 6.0);
    
    /* Create test image */
    img = cpl_image_new(2, 2, CPL_TYPE_DOUBLE);
    cpl_image_add_scalar(img, 1.0);
    
    /* Create test mask */
    mask = cpl_mask_new(2, 2);
    cpl_binary* pmask = cpl_mask_get_data(mask);
    pmask[0] = GOOD_PIX;
    pmask[1] = BAD_PIX;
    pmask[2] = GOOD_PIX;
    pmask[3] = BAD_PIX;
    
    /* Create test table */
    table = cpl_table_new(2);
    cpl_table_new_column(table, "test", CPL_TYPE_DOUBLE);
    cpl_table_set_double(table, "test", 0, 1.0);
    cpl_table_set_double(table, "test", 1, 2.0);
    
    /* Create test propertylist */
    plist = cpl_propertylist_new();
    cpl_propertylist_append_string(plist, "TEST", "test");
    
    /* Test vector save */
    error = eris_ifu_save_vector_dbg(vec, "test_vec.fits", CPL_IO_CREATE, plist);
    cpl_test_eq_error(error, CPL_ERROR_NONE);
    
    /* Test bivector save */
    error = eris_ifu_save_bivector_dbg(bivec, "test_bivec.fits", 0, CPL_IO_CREATE);
    cpl_test_eq_error(error, CPL_ERROR_NONE);
    
    error = eris_ifu_save_bivector_dbg(bivec, "test_bivec_x.fits", 1, CPL_IO_CREATE);
    cpl_test_eq_error(error, CPL_ERROR_NONE);
    
    error = eris_ifu_save_bivector_dbg(bivec, "test_bivec_y.fits", 2, CPL_IO_CREATE);
    cpl_test_eq_error(error, CPL_ERROR_NONE);
    
    /* Test image save */
    error = eris_ifu_save_image_dbg(img, "test_img.fits", CPL_IO_CREATE, plist);
    cpl_test_eq_error(error, CPL_ERROR_NONE);
    
    /* Test mask save */
    error = eris_ifu_save_mask_dbg(mask, "test_mask.fits", CPL_IO_CREATE, plist);
    cpl_test_eq_error(error, CPL_ERROR_NONE);
    
    /* Test table save */
    error = eris_ifu_save_table_dbg(table, "test_table.fits", CPL_IO_CREATE);
    cpl_test_eq_error(error, CPL_ERROR_NONE);
    
    /* Test null inputs */
    error = eris_ifu_save_vector_dbg(NULL, "test_vec.fits", CPL_IO_CREATE, plist);
    cpl_test_eq_error(error, CPL_ERROR_NULL_INPUT);
    
    error = eris_ifu_save_bivector_dbg(NULL, "test_bivec.fits", 0, CPL_IO_CREATE);
    cpl_test_eq_error(error, CPL_ERROR_NULL_INPUT);
    
    error = eris_ifu_save_image_dbg(NULL, "test_img.fits", CPL_IO_CREATE, plist);
    cpl_test_eq_error(error, CPL_ERROR_NULL_INPUT);
    
    error = eris_ifu_save_mask_dbg(NULL, "test_mask.fits", CPL_IO_CREATE, plist);
    cpl_test_eq_error(error, CPL_ERROR_NULL_INPUT);
    
    error = eris_ifu_save_table_dbg(NULL, "test_table.fits", CPL_IO_CREATE);
    cpl_test_eq_error(error, CPL_ERROR_NULL_INPUT);
    
    /* Clean up */
    cpl_vector_delete(vec);
    cpl_bivector_delete(bivec);
    cpl_image_delete(img);
    cpl_mask_delete(mask);
    cpl_table_delete(table);
    cpl_propertylist_delete(plist);
    
    remove("test_vec.fits");
    remove("test_bivec.fits");
    remove("test_bivec_x.fits");
    remove("test_bivec_y.fits");
    remove("test_img.fits");
    remove("test_mask.fits");
    remove("test_table.fits");
}

static void eris_ifu_image_window_test(void)
{
    cpl_image* img;
    cpl_image* window;
    cpl_mask* mask;
    cpl_mask* border;
    cpl_size nx = 1000, ny = 1000;
    double* pimg;
    cpl_binary* pmask;
    
    /* Create test image with border */
    img = cpl_image_new(nx, ny, CPL_TYPE_DOUBLE);
    pimg = cpl_image_get_data(img);
    
    /* Fill image with test pattern */
    for(int i = 0; i < nx*ny; i++) {
        pimg[i] = i;
    }
    
    /* Test image window creation */
    window = eris_ifu_image_create_window(img);
    cpl_test_nonnull(window);
    cpl_test_eq(cpl_image_get_size_x(window), nx - 2*ERIS_IFU_DETECTOR_BP_BORDER);
    cpl_test_eq(cpl_image_get_size_y(window), ny - 2*ERIS_IFU_DETECTOR_BP_BORDER);
    
    /* Verify window contents */
    double* pwindow = cpl_image_get_data(window);
    for(int y = 0; y < ny - 2*ERIS_IFU_DETECTOR_BP_BORDER; y++) {
        for(int x = 0; x < nx - 2*ERIS_IFU_DETECTOR_BP_BORDER; x++) {
            int orig_x = x + ERIS_IFU_DETECTOR_BP_BORDER;
            int orig_y = y + ERIS_IFU_DETECTOR_BP_BORDER;
            cpl_test_eq(pwindow[x + y*(nx-2*ERIS_IFU_DETECTOR_BP_BORDER)], 
                       pimg[orig_x + orig_y*nx]);
        }
    }
    
    /* Create test mask */
    mask = cpl_mask_new(nx, ny);
    pmask = cpl_mask_get_data(mask);
    
    /* Fill mask with alternating pattern */
    for(int i = 0; i < nx*ny; i++) {
        pmask[i] = i % 2;
    }
    
    /* Test mask border creation */
    border = eris_ifu_mask_create_border(mask);
    cpl_test_nonnull(border);
    cpl_test_eq(cpl_mask_get_size_x(border), nx + 2*ERIS_IFU_DETECTOR_BP_BORDER);
    cpl_test_eq(cpl_mask_get_size_y(border), ny + 2*ERIS_IFU_DETECTOR_BP_BORDER);
    
    /* Verify border contents */
    cpl_binary* pborder = cpl_mask_get_data(border);
    for(int y = 0; y < ny; y++) {
        for(int x = 0; x < nx; x++) {
            int new_x = x + ERIS_IFU_DETECTOR_BP_BORDER;
            int new_y = y + ERIS_IFU_DETECTOR_BP_BORDER;
            cpl_test_eq(pborder[new_x + new_y*(nx+2*ERIS_IFU_DETECTOR_BP_BORDER)],
                       pmask[x + y*nx]);
        }
    }
    
    /* Test null inputs */
    window = eris_ifu_image_create_window(NULL);
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    cpl_test_null(window);
    
    border = eris_ifu_mask_create_border(NULL);
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    cpl_test_null(border);
    
    /* Clean up */
    cpl_image_delete(img);
    cpl_image_delete(window);
    cpl_mask_delete(mask);
    cpl_mask_delete(border);
}

static void eris_ifu_save_image_test(void)
{
    cpl_frameset* frameset;
    cpl_propertylist* plist;
    cpl_parameterlist* parlist;
    cpl_image* image;
    cpl_imagelist* imagelist;
    cpl_error_code error;
    
    /* Create test data */
    frameset = cpl_frameset_new();
    plist = cpl_propertylist_new();
    parlist = cpl_parameterlist_new();
    
    /* Create test image */
    image = cpl_image_new(2, 2, CPL_TYPE_DOUBLE);
    cpl_image_add_scalar(image, 1.0);
    
    /* Create test imagelist */
    imagelist = cpl_imagelist_new();
    cpl_imagelist_set(imagelist, cpl_image_duplicate(image), 0);
    cpl_imagelist_set(imagelist, cpl_image_duplicate(image), 1);
    
    /* Add required properties */
    cpl_propertylist_append_string(plist, "ESO PRO CATG", "TEST");
    cpl_propertylist_save(plist, "test_image.fits", CPL_IO_CREATE);
    
    cpl_frame* frame = cpl_frame_new();
    cpl_frame_set_tag(frame,"TEST");
    cpl_frame_set_type(frame, CPL_FRAME_TYPE_IMAGE);
    cpl_frame_set_filename(frame, "test_image.fits");
    cpl_frame_set_group(frame, CPL_FRAME_GROUP_CALIB);
    cpl_frame_set_level(frame, CPL_FRAME_LEVEL_NONE);
    cpl_frameset_insert(frameset,frame);

    /* Test saving single image */
    error = eris_ifu_save_image(frameset, plist, parlist, "test_recipe", 
                              "TEST", "test_image.fits", CPL_TYPE_DOUBLE, image);
    cpl_test_eq_error(error, CPL_ERROR_NONE);
    
    /* Test saving imagelist */
    error = eris_ifu_save_imagelist(frameset, NULL, plist, parlist, "test_recipe",
                                  "TEST", "test_imagelist.fits", CPL_TYPE_DOUBLE, imagelist);
    cpl_test_eq_error(error, CPL_ERROR_NONE);
    
    /* Test null inputs for save_image */
    error = eris_ifu_save_image(NULL, plist, parlist, "test_recipe",
                              "TEST", "test_image.fits", CPL_TYPE_DOUBLE, image);
    cpl_test_eq_error(error, CPL_ERROR_NULL_INPUT);
    
    error = eris_ifu_save_image(frameset, NULL, parlist, "test_recipe",
                              "TEST", "test_image.fits", CPL_TYPE_DOUBLE, image);
    cpl_test_eq_error(error, CPL_ERROR_NULL_INPUT);
    
    error = eris_ifu_save_image(frameset, plist, NULL, "test_recipe",
                              "TEST", "test_image.fits", CPL_TYPE_DOUBLE, image);
    cpl_test_eq_error(error, CPL_ERROR_NULL_INPUT);
    
    error = eris_ifu_save_image(frameset, plist, parlist, NULL,
                              "TEST", "test_image.fits", CPL_TYPE_DOUBLE, image);
    cpl_test_eq_error(error, CPL_ERROR_NULL_INPUT);
    
    error = eris_ifu_save_image(frameset, plist, parlist, "test_recipe",
                              NULL, "test_image.fits", CPL_TYPE_DOUBLE, image);
    cpl_test_eq_error(error, CPL_ERROR_NULL_INPUT);
    
    error = eris_ifu_save_image(frameset, plist, parlist, "test_recipe",
                              "TEST", NULL, CPL_TYPE_DOUBLE, image);
    cpl_test_eq_error(error, CPL_ERROR_NULL_INPUT);
    
    error = eris_ifu_save_image(frameset, plist, parlist, "test_recipe",
                              "TEST", "test_image.fits", CPL_TYPE_DOUBLE, NULL);
    cpl_test_eq_error(error, CPL_ERROR_NULL_INPUT);
    
    /* Test null inputs for save_imagelist */
    error = eris_ifu_save_imagelist(NULL, NULL, plist, parlist, "test_recipe",
                                  "TEST", "test_imagelist.fits", CPL_TYPE_DOUBLE, imagelist);
    cpl_test_eq_error(error, CPL_ERROR_NULL_INPUT);
    
    error = eris_ifu_save_imagelist(frameset, NULL, NULL, parlist, "test_recipe",
                                  "TEST", "test_imagelist.fits", CPL_TYPE_DOUBLE, imagelist);
    cpl_test_eq_error(error, CPL_ERROR_NULL_INPUT);
    
    error = eris_ifu_save_imagelist(frameset, NULL, plist, NULL, "test_recipe",
                                  "TEST", "test_imagelist.fits", CPL_TYPE_DOUBLE, imagelist);
    cpl_test_eq_error(error, CPL_ERROR_NULL_INPUT);
    
    error = eris_ifu_save_imagelist(frameset, NULL, plist, parlist, NULL,
                                  "TEST", "test_imagelist.fits", CPL_TYPE_DOUBLE, imagelist);
    cpl_test_eq_error(error, CPL_ERROR_NULL_INPUT);
    
    error = eris_ifu_save_imagelist(frameset, NULL, plist, parlist, "test_recipe",
                                  NULL, "test_imagelist.fits", CPL_TYPE_DOUBLE, imagelist);
    cpl_test_eq_error(error, CPL_ERROR_NULL_INPUT);
    
    error = eris_ifu_save_imagelist(frameset, NULL, plist, parlist, "test_recipe",
                                  "TEST", NULL, CPL_TYPE_DOUBLE, imagelist);
    cpl_test_eq_error(error, CPL_ERROR_NULL_INPUT);
    
    error = eris_ifu_save_imagelist(frameset, NULL, plist, parlist, "test_recipe",
                                  "TEST", "test_imagelist.fits", CPL_TYPE_DOUBLE, NULL);
    cpl_test_eq_error(error, CPL_ERROR_NULL_INPUT);
    
    /* Cleanup */
    cpl_frameset_delete(frameset);
    cpl_propertylist_delete(plist);
    cpl_parameterlist_delete(parlist);
    cpl_image_delete(image);
    cpl_imagelist_delete(imagelist);
    remove("test_image.fits");
    remove("test_imagelist.fits");
}

/**
 * @brief Test string handling functions
 */
static void eris_ifu_string_handling_test(void)
{
    const char *str;
    char *lamp_str;
    
    /* Test lamp string generation */
    lamp_str = eris_ifu_get_lampString(AR_LAMP);
    cpl_test_nonnull(lamp_str);
    cpl_test_eq_string(lamp_str, "Ar");
    cpl_free(lamp_str);
    
    lamp_str = eris_ifu_get_lampString(KR_LAMP);
    cpl_test_nonnull(lamp_str);
    cpl_test_eq_string(lamp_str, "Kr");
    cpl_free(lamp_str);
    
    lamp_str = eris_ifu_get_lampString(NE_LAMP);
    cpl_test_nonnull(lamp_str);
    cpl_test_eq_string(lamp_str, "Ne");
    cpl_free(lamp_str);
    
    lamp_str = eris_ifu_get_lampString(XE_LAMP);
    cpl_test_nonnull(lamp_str);
    cpl_test_eq_string(lamp_str, "Xe");
    cpl_free(lamp_str);
    
    /* Test multiple lamp combinations */
    lamp_str = eris_ifu_get_lampString(AR_LAMP | KR_LAMP);
    cpl_test_nonnull(lamp_str);
    cpl_test_eq_string(lamp_str, "ArKr");
    cpl_free(lamp_str);

    lamp_str = eris_ifu_get_lampString(AR_LAMP | NE_LAMP | XE_LAMP);
    cpl_test_nonnull(lamp_str);
    cpl_test_eq_string(lamp_str, "ArNeXe");
    cpl_free(lamp_str);

    lamp_str = eris_ifu_get_lampString(KR_LAMP | NE_LAMP);
    cpl_test_nonnull(lamp_str);
    cpl_test_eq_string(lamp_str, "KrNe");
    cpl_free(lamp_str);
    
    /* Test band string generation */
    str = eris_ifu_get_bandString(J_LOW);
    cpl_test_nonnull(str);
    cpl_test_eq_string(str, "J_LOW");
    
    str = eris_ifu_get_bandString(H_LOW);
    cpl_test_nonnull(str);
    cpl_test_eq_string(str, "H_LOW");
    
    str = eris_ifu_get_bandString(K_LOW);
    cpl_test_nonnull(str);
    cpl_test_eq_string(str, "K_LOW");

    /* Test additional band strings */
    str = eris_ifu_get_bandString(J_SHORT);
    cpl_test_nonnull(str);
    cpl_test_eq_string(str, "J_SHORT");

    str = eris_ifu_get_bandString(H_MIDDLE);
    cpl_test_nonnull(str);
    cpl_test_eq_string(str, "H_MIDDLE");

    str = eris_ifu_get_bandString(K_LONG);
    cpl_test_nonnull(str);
    cpl_test_eq_string(str, "K_LONG");

    str = eris_ifu_get_bandString(J_SPIFFI);
    cpl_test_nonnull(str);
    cpl_test_eq_string(str, "J_SPIFFI");

    str = eris_ifu_get_bandString(H_SPIFFI);
    cpl_test_nonnull(str);
    cpl_test_eq_string(str, "H_SPIFFI");

    str = eris_ifu_get_bandString(K_SPIFFI);
    cpl_test_nonnull(str);
    cpl_test_eq_string(str, "K_SPIFFI");

    str = eris_ifu_get_bandString(HK_SPIFFI);
    cpl_test_nonnull(str);
    cpl_test_eq_string(str, "HK_SPIFFI");
    
    str = eris_ifu_get_bandString(UNDEFINED_BAND);
    cpl_test_nonnull(str);
    cpl_test_eq_string(str, "");

    /* Test invalid band */
    str = eris_ifu_get_bandString(99); /* Invalid value */
    cpl_test_nonnull(str);
    cpl_test_eq_string(str, "Unknown");
    
    /* Test band resolution */
    cpl_test_eq(eris_ifu_get_band_resolution(J_LOW), 5000);
    cpl_test_eq(eris_ifu_get_band_resolution(H_LOW), 5200);
    cpl_test_eq(eris_ifu_get_band_resolution(K_LOW), 5600);
    cpl_test_eq(eris_ifu_get_band_resolution(UNDEFINED_BAND), 0);

    /* Test additional band resolutions */
    cpl_test_eq(eris_ifu_get_band_resolution(J_SHORT), 10000);
    cpl_test_eq(eris_ifu_get_band_resolution(H_MIDDLE), 10400);
    cpl_test_eq(eris_ifu_get_band_resolution(K_LONG), 11200);
    cpl_test_eq(eris_ifu_get_band_resolution(HK_SPIFFI), 3000);

    /* Test invalid band resolution */
    cpl_test_eq(eris_ifu_get_band_resolution(99), 0); /* Invalid value */
    
    /* Test instrument string generation */
    str = eris_ifu_get_instrumentString(SPIFFI);
    cpl_test_nonnull(str);
    cpl_test_eq_string(str, "SPIFFI");
    
    str = eris_ifu_get_instrumentString(SPIFFIER);
    cpl_test_nonnull(str);
    cpl_test_eq_string(str, "SPIFFIER");
    
    str = eris_ifu_get_instrumentString(NIX);
    cpl_test_nonnull(str);
    cpl_test_eq_string(str, "NIX");
    
    str = eris_ifu_get_instrumentString(UNSET_INSTRUMENT);
    cpl_test_nonnull(str);
    cpl_test_eq_string(str, "unset instrument");

    str = eris_ifu_get_instrumentString(OTHER_INSTRUMENT);
    cpl_test_nonnull(str);
    cpl_test_eq_string(str, "unknown instrument");

    /* Test invalid instrument */
    str = eris_ifu_get_instrumentString(99); /* Invalid value */
    cpl_test_nonnull(str);
    cpl_test_eq_string(str, "unset instrument");
    
    /* Test preoptics scale string generation */
    str = eris_ifu_get_preopticsScaleString(S250MAS);
    cpl_test_nonnull(str);
    cpl_test_eq_string(str, "250mas");
    
    str = eris_ifu_get_preopticsScaleString(S100MAS);
    cpl_test_nonnull(str);
    cpl_test_eq_string(str, "100mas");
    
    str = eris_ifu_get_preopticsScaleString(S25MAS);
    cpl_test_nonnull(str);
    cpl_test_eq_string(str, "25mas");
    
    str = eris_ifu_get_preopticsScaleString(UNDEFINED_SCALE);
    cpl_test_nonnull(str);
    cpl_test_eq_string(str, "");

    str = eris_ifu_get_preopticsScaleString(PUPIL);
    cpl_test_nonnull(str);
    cpl_test_eq_string(str, "PUPIL");

    /* Test invalid scale */
    str = eris_ifu_get_preopticsScaleString(99); /* Invalid value */
    cpl_test_nonnull(str);
    cpl_test_eq_string(str, "Unknown");
}

int main(void)
{
    cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_WARNING);

    /* Run all tests */
    eris_ifu_memory_management_test();
    eris_ifu_idl_where_test();
    eris_ifu_idl_values_at_indices_test();
    eris_ifu_cut_endings_test();
    eris_ifu_parameterlist_append_list_test();
    eris_ifu_hdrl_get_imagelist_test();
    eris_ifu_get_frameset_by_tag_test();
    eris_ifu_qc_parameter_test();
    //eris_ifu_image_manipulation_test(); //TODO: fix this seg fault
    eris_ifu_psf_analysis_test();
    eris_ifu_file_exists_test();
    //eris_ifu_get_hdrlimagelist_by_tag_test(); //TODO: fix seg fault
    eris_ifu_load_badpixel_mask_test();
    eris_ifu_mask_nans_in_hdrlimage_test();
    eris_ifu_mask_nans_in_cube_test();
    eris_ifu_split_hdrl_imagelist_test();
    //eris_ifu_split3_hdrl_imagelist_test();  // TODO: fix errors/leaks
    eris_ifu_save_image_test();
    cpl_test_error(CPL_ERROR_NONE);
    
    //eris_ifu_image_window_test();// TODO: seg fault
    cpl_test_error(CPL_ERROR_NONE);
    

    eris_ifu_save_dbg_test();
    cpl_test_error(CPL_ERROR_NONE);

    eris_ifu_badpix_qc_test();
    cpl_test_error(CPL_ERROR_NONE);

    eris_ifu_quality2bp_mask_test();
    cpl_test_error(CPL_ERROR_NONE);

    eris_ifu_interpolatedMask_to_maskZero_test();
    cpl_test_error(CPL_ERROR_NONE);

    //eris_ifu_cube_set_values_test(); // TODO: fix errors
    cpl_test_error(CPL_ERROR_NONE);

    //eris_ifu_cube_trim_nans_test(); // TODO: fix leaks
    cpl_test_error(CPL_ERROR_NONE);

    //eris_ifu_get_plane_cut_min_max_test();// TODO: fix portability problem jenkins
    cpl_test_error(CPL_ERROR_NONE);

    eris_get_pupil_shift_test();
    cpl_test_error(CPL_ERROR_NONE);

    eris_qclog_add_double_format_test();
    cpl_test_error(CPL_ERROR_NONE);

    eris_ifu_string_handling_test();


    return cpl_test_end(0);
}

/**@}*/
