/* $Id$
 *
 * This file is part of the ERIS Pipeline
 * Copyright (C) 2002,2003 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/*
 * $Author$
 * $Date$
 * $Revision$
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                Includes
 -----------------------------------------------------------------------------*/

#include "eris_nix_test_defs.h"
#include <eris_nix_dfs.h>
#include <eris_nix_lss_utils.h>
#include <eris_nix_utils.h>
#include <hdrl_utils.h>
#include <math.h>

/*----------------------------------------------------------------------------*/
/**
 * @defgroup eris_nix_lss_utils_test  Unit test of eris_nix_lss_utils
 *
 */
/*----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------
                                       Prototypes
 -----------------------------------------------------------------------------*/

static void eris_nix_lss_utils_test(void);

/**@{*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Textual representation of CPL frame group
  @param    group     to convert
  @return   textual representation
 */
/*----------------------------------------------------------------------------*/
static void eris_nix_lss_utils_test(void)
{
    {    /* start test of enlu_divide_slit_response */

    /* ..null inputs */

    enlu_divide_slit_response(NULL);
    cpl_test_error(CPL_ERROR_NULL_INPUT);

    /* ..valid inputs */

    /* ....create a test located_image with a square deviation in slit
           response */
    
    hdrl_image * himage = hdrl_image_new(TEST_SIZE_X,
                                         TEST_SIZE_Y);
    cpl_image_fill_window(hdrl_image_get_image(himage),
                          1, 1, TEST_SIZE_X, TEST_SIZE_Y,
                          10.0);
    cpl_image_fill_window(hdrl_image_get_image(himage),
                          TEST_SIZE_X/4, 1, TEST_SIZE_X * 3/4, TEST_SIZE_Y,
                          15.0);
    cpl_image_fill_window(hdrl_image_get_image(himage),
                          TEST_SIZE_X/2, 1, TEST_SIZE_X/2 + 5, TEST_SIZE_Y,
                          20.0);
    cpl_image_fill_window(hdrl_image_get_error(himage),
                          1, 1, TEST_SIZE_X, TEST_SIZE_Y,
                          1.0);

    cpl_image * confidence = cpl_image_new(TEST_SIZE_X,
                                           TEST_SIZE_Y,
                                           CPL_TYPE_DOUBLE);
    cpl_image_fill_window(confidence,
                          1, 1, TEST_SIZE_X, TEST_SIZE_Y,
                          100.0);

    cpl_propertylist * plist = cpl_propertylist_new();
    /* image has background same as data */
    located_image * limage = enu_located_image_new(
                                       himage,
                                       NULL,
                                       confidence,
                                       hdrl_image_duplicate(himage),
                                       cpl_image_duplicate(confidence),
                                       plist,
                                       NULL, NULL, NULL, NULL, NULL);

    located_imagelist * limagelist = enu_located_imagelist_new(1);
    enu_located_imagelist_insert(limagelist, limage, 0);

    enlu_divide_slit_response(limagelist);
    cpl_test_error(CPL_ERROR_NONE);

    int reject;
    hdrl_value val = hdrl_image_get_pixel(limage->himage, TEST_SIZE_X/2 - 1,
                                          1, &reject);
    cpl_test_leq(fabs(val.data - 15.0), 1e-6);
    cpl_test_leq(fabs(val.error - 1.0), 1e-6);

    val = hdrl_image_get_pixel(limage->himage, TEST_SIZE_X/2 + 1,
                                          1, &reject);
    cpl_test_leq(fabs(val.data - 15.0), 1e-6);
    cpl_test_leq(fabs(val.error - 0.75), 1e-6);

    enu_located_imagelist_delete(limagelist);

    }    /* end tests for enlu_divide_slit_response */


    {    /* start tests for linepos */

    double            dresult;
    cpl_vector      * guess = NULL;
    cpl_matrix      * matrix_result = NULL;
    cpl_vector      * spectrum1d = NULL;
    hdrl_image      * spectrum2d = NULL;
    cpl_image       * spectrum2d_image = NULL;

    /* test spectrum with one line near 53 */

    spectrum1d = cpl_vector_new(100);
    cpl_vector_fill(spectrum1d, 0.0);
    cpl_vector_set(spectrum1d, 49, 9.0); 
    cpl_vector_set(spectrum1d, 50, 18.0); 
    cpl_vector_set(spectrum1d, 51, 43.0); 
    cpl_vector_set(spectrum1d, 52, 148.0); 
    cpl_vector_set(spectrum1d, 53, 364.0); 
    cpl_vector_set(spectrum1d, 54, 440.0); 
    cpl_vector_set(spectrum1d, 55, 288.0); 
    cpl_vector_set(spectrum1d, 56, 101.0); 
    cpl_vector_set(spectrum1d, 57, 30.0); 
    cpl_vector_set(spectrum1d, 58, 17.0); 
    cpl_vector_set(spectrum1d, 59, 13.0); 
    cpl_vector_set(spectrum1d, 60, 12.0); 

    /* construct 2d spectrum for tests */

    spectrum2d_image = cpl_image_new(100, 100, CPL_TYPE_DOUBLE);
    for (cpl_size iy = 1; iy <= 100; iy++){
        for (cpl_size ix = 1; ix <= 100; ix++){
            cpl_image_set(spectrum2d_image, ix, iy, 
                          cpl_vector_get(spectrum1d, iy-1));
        }
    }
    spectrum2d = hdrl_image_create(spectrum2d_image, NULL);
    guess = cpl_vector_new(1);
    cpl_vector_set(guess, 0, 53.0);

    /* dud parameter tests for 1d */

    cpl_error_set(cpl_func, CPL_ERROR_NO_WCS);
    dresult = enlu_linepos_1d(NULL, 0.0, 4); 
    cpl_test_error(CPL_ERROR_NO_WCS);
    cpl_test(isnan(dresult));

    dresult = enlu_linepos_1d(NULL, 0.0, 4); 
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    cpl_test(isnan(dresult));

    /* fit test * for 1d */

    dresult = enlu_linepos_1d(spectrum1d, 53.0, 4); 
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_rel(dresult, 53.84, 0.01);

    /* test for dodgy data */

    dresult = enlu_linepos_1d(spectrum1d, 57.0, 4); 
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test(isnan(dresult));

    cpl_vector_set(spectrum1d, 55, NAN);
    dresult = enlu_linepos_1d(spectrum1d, 53.0, 4); 
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test(isnan(dresult));
 
    cpl_vector_set(spectrum1d, 55, 288.0); 

    /* dud parameter tests for 2d */

    cpl_error_set(cpl_func, CPL_ERROR_NO_WCS);
    matrix_result = enlu_linepos_2d(NULL, 0, NULL); 
    cpl_test_error(CPL_ERROR_NO_WCS);
    cpl_test_null(matrix_result);
    cpl_matrix_delete(matrix_result);

    matrix_result = enlu_linepos_2d(NULL, 0, NULL); 
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    cpl_test_null(matrix_result);
    cpl_matrix_delete(matrix_result);

    /* fit test * for 2d */

    matrix_result = enlu_linepos_2d(spectrum2d, 50, guess);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_eq(cpl_matrix_get_nrow(matrix_result), 100);
    cpl_test_eq(cpl_matrix_get_ncol(matrix_result), 1);
    cpl_test_rel(cpl_matrix_get_min(matrix_result), 53.84, 0.01);
    cpl_test_rel(cpl_matrix_get_max(matrix_result), 53.84, 0.01);
    cpl_matrix_delete(matrix_result);

    /* test with dodgy data */

    hdrl_image_set_pixel(spectrum2d, 55, 55, (hdrl_value){NAN, NAN});
    matrix_result = enlu_linepos_2d(spectrum2d, 50, guess);
    cpl_test_error(CPL_ERROR_NONE);
    for (cpl_size i = 0; i < 100; i++) {
        dresult = cpl_matrix_get(matrix_result, i, 0); 
        if (i == 54) {
            cpl_test(isnan(dresult));
        } else {
            cpl_test_rel(dresult, 53.84, 0.01);
        }
    }
    cpl_matrix_delete(matrix_result);
 
    /* clean up */

    cpl_vector_delete(guess);
    cpl_vector_delete(spectrum1d);
    hdrl_image_delete(spectrum2d);
    cpl_image_delete(spectrum2d_image);

    }    /* end tests for linepos */


    {    /* start tests for warp_poly save and load */

    cpl_table * warp_table = NULL;
    warp_table = enlu_warp_poly_save_to_table(NULL);
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    cpl_test_assert(warp_table==NULL);

    cpl_polynomial * bad_poly = cpl_polynomial_new(3);
    warp_table = enlu_warp_poly_save_to_table(bad_poly);
    cpl_test_error(CPL_ERROR_ILLEGAL_INPUT);
    cpl_test_assert(warp_table==NULL);

    cpl_polynomial * good_poly = cpl_polynomial_new(2);
    cpl_size pows00[2] = {0,0};
    cpl_size pows10[2] = {1,0};
    cpl_size pows02[2] = {0,2};
    cpl_size pows31[2] = {3,1};
    cpl_polynomial_set_coeff(good_poly, pows00, 1.0);
    cpl_polynomial_set_coeff(good_poly, pows10, 2.0);
    cpl_polynomial_set_coeff(good_poly, pows02, 3.0);
    cpl_polynomial_set_coeff(good_poly, pows31, 4.0);

    warp_table = enlu_warp_poly_save_to_table(good_poly);
    cpl_test_error(CPL_ERROR_NONE);

    cpl_polynomial * test_poly = enlu_warp_poly_load_from_table(NULL);
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    cpl_assert(test_poly==NULL);

    test_poly = enlu_warp_poly_load_from_table(warp_table);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_assert(cpl_polynomial_compare(good_poly, test_poly, 1e-15)==0);

    cpl_polynomial_delete(good_poly);
    cpl_polynomial_delete(bad_poly);
    cpl_polynomial_delete(test_poly);
    cpl_table_delete(warp_table);

    }    /* end tests for warp_poly save and load */
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit tests of eris_nix_lss_utils module
 */
/*----------------------------------------------------------------------------*/

int main(void)
{
    cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_WARNING);

    eris_nix_lss_utils_test();

    return cpl_test_end(0);

}

/**@}*/
