/*
 * E.S.O. - VLT project
 *
 * Test code for eris_pfits module
 *
 * who       when        what
 * --------  ----------  ----------------------------------------------
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <math.h>
#include <float.h>
#include <string.h>
#include <unistd.h>

#include <cpl.h>

#include "eris_pfits.h"

/* Test helper functions */
static cpl_propertylist* create_test_header(void) {
    cpl_propertylist* header = cpl_propertylist_new();
    
    /* Add basic FITS keywords */
    cpl_propertylist_append_string(header, "ESO PRO ANCESTOR", "test_ancestor.fits");
    cpl_propertylist_append_string(header, "ARCFILE", "test_arc.fits");
    cpl_propertylist_append_double(header, "ESO DET SEQ1 DIT", 10.0);
    cpl_propertylist_append_int(header, "ESO DET NDIT", 5);
    cpl_propertylist_append_double(header, "EXPTIME", 50.0);
    
    /* Add NAXIS keywords */
    cpl_propertylist_append_int(header, "NAXIS1", 100);
    cpl_propertylist_append_int(header, "NAXIS2", 200);
    
    /* Add WCS keywords */
    cpl_propertylist_append_double(header, "CRPIX1", 50.0);
    cpl_propertylist_append_double(header, "CRPIX2", 100.0);
    cpl_propertylist_append_double(header, "CRPIX3", 1.0);
    cpl_propertylist_append_double(header, "CRVAL1", 0.0);
    cpl_propertylist_append_double(header, "CRVAL2", 0.0);
    cpl_propertylist_append_double(header, "CRVAL3", 2.0);
    cpl_propertylist_append_double(header, "CDELT1", 1.0);
    cpl_propertylist_append_double(header, "CDELT2", 1.0);
    cpl_propertylist_append_double(header, "CDELT3", 0.1);
    cpl_propertylist_append_string(header, "CTYPE1", "RA---TAN");
    cpl_propertylist_append_string(header, "CTYPE2", "DEC--TAN");
    cpl_propertylist_append_string(header, "CTYPE3", "WAVE");
    cpl_propertylist_append_string(header, "CUNIT1", "deg");
    cpl_propertylist_append_string(header, "CUNIT2", "deg");
    cpl_propertylist_append_string(header, "CUNIT3", "um");
    cpl_propertylist_append_double(header, "CD1_1", 1.0);
    cpl_propertylist_append_double(header, "CD1_2", 0.0);
    cpl_propertylist_append_double(header, "CD1_3", 0.0);
    cpl_propertylist_append_double(header, "CD2_1", 0.0);
    cpl_propertylist_append_double(header, "CD2_2", 1.0);
    cpl_propertylist_append_double(header, "CD2_3", 0.0);
    cpl_propertylist_append_double(header, "CD3_1", 0.0);
    cpl_propertylist_append_double(header, "CD3_2", 0.0);
    cpl_propertylist_append_double(header, "CD3_3", 1.0);
    
    /* Add raw file keywords */
    cpl_propertylist_append_string(header, "ESO PRO REC1 RAW1 NAME", "raw1.fits");
    cpl_propertylist_append_string(header, "ESO PRO REC1 RAW2 NAME", "raw2.fits");
    
    /* Add FWHM keywords */
    cpl_propertylist_append_double(header, "ESO TEL IA FWHMLIN", 0.8);
    cpl_propertylist_append_double(header, "ESO TEL AMBI FWHM START", 0.7);
    cpl_propertylist_append_double(header, "ESO TEL AMBI FWHM END", 0.9);
    
    /* Add airmass keywords */
    cpl_propertylist_append_double(header, "ESO TEL AIRM START", 1.1);
    cpl_propertylist_append_double(header, "ESO TEL AIRM END", 1.3);
    
    /* Add observation info keywords */
    cpl_propertylist_append_double(header, "MJD-OBS", 58000.0);
    cpl_propertylist_append_string(header, "ESO OBS PROG ID", "60.A-9252(A)");
    cpl_propertylist_append_long(header, "ESO OBS ID", 12345);
    cpl_propertylist_append_string(header, "PIPEFILE", "test_pipe.fits");
    cpl_propertylist_append_string(header, "DATAMD5", "abcdef123456");
    
    /* Add extra keywords for cleaning tests */
    cpl_propertylist_append_string(header, "TELESCOP", "VLT");
    cpl_propertylist_append_string(header, "INSTRUME", "ERIS");
    cpl_propertylist_append_string(header, "OBJECT", "Test");
    cpl_propertylist_append_string(header, "DATE-OBS", "2020-01-01");
    cpl_propertylist_append_string(header, "ESO INS MODE", "IFU");
    cpl_propertylist_append_string(header, "ESO DET DIT", "10.0");
    cpl_propertylist_append_string(header, "COMMENT", "Test comment");
    
    /* Add expmap extra keys */
    cpl_propertylist_append_string(header, "HDUCLASS", "ESO");
    cpl_propertylist_append_string(header, "HDUDOC", "Test doc");
    cpl_propertylist_append_string(header, "HDUVERS", "1.0");
    cpl_propertylist_append_string(header, "HDUCLAS1", "Test class 1");
    cpl_propertylist_append_string(header, "HDUCLAS2", "Test class 2");
    cpl_propertylist_append_string(header, "ERRDATA", "Test error");
    cpl_propertylist_append_string(header, "QUALDATA", "Test quality");
    
    return header;
}

/**
 * @brief Test basic FITS header functions
 */
static void eris_pfits_basic_test(void)
{
    cpl_propertylist* header = create_test_header();
    
    /* Create test file for exptime test */
    cpl_propertylist* exptime_header = cpl_propertylist_new();
    cpl_propertylist_append_double(exptime_header, "EXPTIME", 50.0);
    cpl_propertylist_save(exptime_header, "test_exptime.fits", CPL_IO_CREATE);
    cpl_propertylist_delete(exptime_header);
    
    /* Test ancestor */
    const char* ancestor = eris_pfits_get_ancestor(header);
    cpl_test_nonnull(ancestor);
    cpl_test_eq_string(ancestor, "test_ancestor.fits");
    
    /* Test arcfile */
    const char* arcfile = eris_pfits_get_arcfile(header);
    cpl_test_nonnull(arcfile);
    cpl_test_eq_string(arcfile, "test_arc.fits");
    
    /* Test DIT */
    double dit = eris_pfits_get_dit(header);
    cpl_test_abs(dit, 10.0, 0.000001);
    
    /* Test NDIT */
    int ndit = eris_pfits_get_ndit(header);
    cpl_test_eq(ndit, 5);
    
    /* Test EXPTIME */
    double exptime = eris_pfits_get_exptime("test_exptime.fits");
    cpl_test_abs(exptime, 50.0, 0.000001);
    cpl_test_error(CPL_ERROR_NONE);
    
    /* Test with non-existent file */
    exptime = eris_pfits_get_exptime("nonexistent.fits");
    cpl_test_eq(exptime, 0.0);
    cpl_test_error(CPL_ERROR_FILE_NOT_FOUND);
    
    /* Test with invalid EXPTIME type */
    cpl_propertylist* invalid_exptime_header = cpl_propertylist_new();
    cpl_propertylist_append_string(invalid_exptime_header, "EXPTIME", "invalid");
    cpl_propertylist_save(invalid_exptime_header, "test_invalid_exptime.fits", CPL_IO_CREATE);
    cpl_propertylist_delete(invalid_exptime_header);
    
    exptime = eris_pfits_get_exptime("test_invalid_exptime.fits");
    cpl_test_eq(exptime, 0.0);
    cpl_test_error(CPL_ERROR_TYPE_MISMATCH);
    
    remove("test_invalid_exptime.fits");
    
    /* Test error handling */
    cpl_test_null(eris_pfits_get_ancestor(NULL));
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    cpl_test_error(CPL_ERROR_NONE);
    
    cpl_test_null(eris_pfits_get_arcfile(NULL));
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    cpl_test_error(CPL_ERROR_NONE);
    
    /* Test missing values */
    cpl_propertylist_erase(header, "ESO PRO ANCESTOR");
    cpl_test_null(eris_pfits_get_ancestor(header));
    //cpl_test_error(CPL_ERROR_DATA_NOT_FOUND);   // TODO: original function tolerate missing PRO.ANCESTOR
    
    /* Test invalid property list */
    cpl_propertylist* invalid_header = cpl_propertylist_new();
    cpl_propertylist_append_int(invalid_header, "ESO PRO ANCESTOR", 123); // Wrong type for ANCESTOR
    cpl_test_null(eris_pfits_get_ancestor(invalid_header));
    cpl_test_error(CPL_ERROR_TYPE_MISMATCH);
    cpl_propertylist_delete(invalid_header);

    
    cpl_propertylist_erase(header, "ARCFILE");
    cpl_test_null(eris_pfits_get_arcfile(header));
    cpl_test_error(CPL_ERROR_DATA_NOT_FOUND);
    
    /* Cleanup */
    cpl_propertylist_delete(header);
    remove("test_exptime.fits");
}

/**
 * @brief Test NAXIS functions
 */
static void eris_pfits_naxis_test(void)
{
    cpl_propertylist* header = create_test_header();
    
    /* Test NAXIS1 */
    int naxis1 = eris_pfits_get_naxis1(header);
    cpl_test_eq(naxis1, 100);
    
    /* Test NAXIS2 */
    int naxis2 = eris_pfits_get_naxis2(header);
    cpl_test_eq(naxis2, 200);
    
    /* Test error handling */
    cpl_test_eq(eris_pfits_get_naxis1(NULL), 0);
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    
    cpl_test_eq(eris_pfits_get_naxis2(NULL), 0);
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    
    /* Test missing NAXIS values */
    cpl_propertylist_erase(header, "NAXIS1");
    cpl_test_eq(eris_pfits_get_naxis1(header), 0);
    cpl_test_error(CPL_ERROR_DATA_NOT_FOUND);
    
    cpl_propertylist_erase(header, "NAXIS2");
    cpl_test_eq(eris_pfits_get_naxis2(header), 0);
    cpl_test_error(CPL_ERROR_DATA_NOT_FOUND);
    
    /* Test invalid NAXIS types */
    cpl_propertylist* invalid_header = cpl_propertylist_new();
    cpl_propertylist_append_string(invalid_header, "NAXIS1", "invalid");
    cpl_propertylist_append_string(invalid_header, "NAXIS2", "invalid");
    
    cpl_test_eq(eris_pfits_get_naxis1(invalid_header), 0);
    cpl_test_error(CPL_ERROR_TYPE_MISMATCH);
    
    cpl_test_eq(eris_pfits_get_naxis2(invalid_header), 0);
    cpl_test_error(CPL_ERROR_TYPE_MISMATCH);
    
    cpl_propertylist_delete(invalid_header);
    
    /* Cleanup */
    cpl_propertylist_delete(header);
}

/**
 * @brief Test WCS functions
 */
static void eris_pfits_wcs_test(void)
{
    cpl_propertylist* header = create_test_header();
    cpl_propertylist* wcs;
    
    /* Test WCS2D extraction */
    wcs = eris_ifu_plist_extract_wcs2D(header);
    cpl_test_nonnull(wcs);
    cpl_test_eq(cpl_propertylist_get_size(wcs), 14);  // Should have all 2D WCS keys
    cpl_propertylist_delete(wcs);
    
    /* Test WCS3D extraction */
    wcs = eris_ifu_plist_extract_wcs3D(header);
    cpl_test_nonnull(wcs);
    cpl_test_eq(cpl_propertylist_get_size(wcs), 10);  // Should have all 3D WCS keys
    cpl_propertylist_delete(wcs);
    
    /* Test full WCS extraction */
    wcs = eris_ifu_plist_extract_wcs(header);
    cpl_test_nonnull(wcs);
    cpl_test_eq(cpl_propertylist_get_size(wcs), 24);  // Should have all WCS keys
    cpl_propertylist_delete(wcs);
    
    /* Test WCS2D erasure */
    cpl_test_eq_error(eris_pfits_erase_wcs2D(header), CPL_ERROR_NONE);
    cpl_test_zero(cpl_propertylist_has(header, "CRPIX1"));
    
    /* Test WCS3D erasure */
    cpl_test_eq_error(eris_ifu_plist_erase_wcs3D(header), CPL_ERROR_NONE);
    cpl_test_zero(cpl_propertylist_has(header, "CRPIX3"));
    
    /* Test full WCS erasure */
    cpl_test_eq_error(eris_ifu_plist_erase_wcs(header), CPL_ERROR_NONE);
    cpl_test_zero(cpl_propertylist_has(header, "CRPIX1"));
    cpl_test_zero(cpl_propertylist_has(header, "CRPIX3"));
    
    /* Test error handling */
    cpl_test_null(eris_ifu_plist_extract_wcs2D(NULL));
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    
    /* Test with missing WCS2D keys */
    cpl_propertylist* partial_wcs2d = cpl_propertylist_new();
    cpl_propertylist_append_double(partial_wcs2d, "CRPIX1", 50.0);
    cpl_propertylist_append_double(partial_wcs2d, "CRVAL1", 0.0);
    // Missing CRPIX2, CRVAL2, etc.
    
    wcs = eris_ifu_plist_extract_wcs2D(partial_wcs2d);
    cpl_test_nonnull(wcs);
    cpl_test_eq(cpl_propertylist_get_size(wcs), 2); // Only CRPIX1 and CRVAL1 should be copied
    cpl_propertylist_delete(wcs);
    cpl_propertylist_delete(partial_wcs2d);
    
    /* Test with missing WCS3D keys */
    cpl_propertylist* partial_wcs3d = cpl_propertylist_new();
    cpl_propertylist_append_double(partial_wcs3d, "CRPIX3", 1.0);
    cpl_propertylist_append_double(partial_wcs3d, "CRVAL3", 2.0);
    // Missing CDELT3, CTYPE3, etc.
    
    wcs = eris_ifu_plist_extract_wcs3D(partial_wcs3d);
    cpl_test_nonnull(wcs);
    cpl_test_eq(cpl_propertylist_get_size(wcs), 2); // Only CRPIX3 and CRVAL3 should be copied
    cpl_propertylist_delete(wcs);
    cpl_propertylist_delete(partial_wcs3d);
    
    /* Test with invalid WCS key types */
    cpl_propertylist* invalid_wcs = cpl_propertylist_new();
    cpl_propertylist_append_string(invalid_wcs, "CRPIX1", "invalid");
    cpl_propertylist_append_string(invalid_wcs, "CRPIX3", "invalid");
    wcs = eris_ifu_plist_extract_wcs2D(invalid_wcs);
    cpl_test_error(CPL_ERROR_TYPE_MISMATCH);
    cpl_test_nonnull(wcs);
    cpl_test_error(CPL_ERROR_NONE);

    cpl_test_eq(cpl_propertylist_get_size(wcs), 0); // No keys should be copied due to type mismatch
    cpl_propertylist_delete(wcs);

    
    wcs = eris_ifu_plist_extract_wcs3D(invalid_wcs);
    cpl_test_error(CPL_ERROR_TYPE_MISMATCH);
    cpl_test_nonnull(wcs);
    cpl_test_eq(cpl_propertylist_get_size(wcs), 0); // No keys should be copied due to type mismatch
    cpl_propertylist_delete(wcs);
    cpl_propertylist_delete(invalid_wcs);
    
    /* Cleanup */
    cpl_propertylist_delete(header);
}

/**
 * @brief Test raw file functions
 */
static void eris_pfits_raw_test(void)
{
    cpl_propertylist* header = create_test_header();
    
    /* Test raw file names */
    const char* raw1 = eris_pfits_get_raw_filename(header, 1);
    cpl_test_nonnull(raw1);
    cpl_test_eq_string(raw1, "raw1.fits");
    
    const char* raw2 = eris_pfits_get_raw_filename(header, 2);
    cpl_test_nonnull(raw2);
    cpl_test_eq_string(raw2, "raw2.fits");
    
    /* Test non-existent indices */
    const char* raw3 = eris_pfits_get_raw_filename(header, 3);
    cpl_test_null(raw3);
    //cpl_test_error(CPL_ERROR_DATA_NOT_FOUND); // TODO why eris_pfits_get_raw_filename reset error code?
    
    const char* raw0 = eris_pfits_get_raw_filename(header, 0);
    cpl_test_null(raw0);
    //cpl_test_error(CPL_ERROR_DATA_NOT_FOUND); // TODO why eris_pfits_get_raw_filename reset error code?

    
    /* Test with invalid raw filename type */
    cpl_propertylist* invalid_header = cpl_propertylist_new();
    cpl_propertylist_append_int(invalid_header, "ESO PRO REC1 RAW1 NAME", 123);
    const char* invalid_raw = eris_pfits_get_raw_filename(invalid_header, 1);// TODO why eris_pfits_get_raw_filename reset error code?
    cpl_test_null(invalid_raw);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_propertylist_delete(invalid_header);
    
    /* Test with very large index */
    const char* raw_large = eris_pfits_get_raw_filename(header, 999999);
    cpl_test_null(raw_large);
    //cpl_test_error(CPL_ERROR_DATA_NOT_FOUND); // TODO why eris_pfits_get_raw_filename reset error code?
    
    /* Test with malformed raw filename key */
    cpl_propertylist* malformed_header = cpl_propertylist_new();
    cpl_propertylist_append_string(malformed_header, "ESO PRO REC1 RAW NAME", "malformed.fits"); // Missing index
    const char* malformed_raw = eris_pfits_get_raw_filename(malformed_header, 1);
    cpl_test_null(malformed_raw);
    //cpl_test_error(CPL_ERROR_DATA_NOT_FOUND); // TODO why eris_pfits_get_raw_filename reset error code?
    cpl_propertylist_delete(malformed_header);
    
    /* Test error handling */
    cpl_test_null(eris_pfits_get_raw_filename(NULL, 1));
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    
    /* Cleanup */
    cpl_propertylist_delete(header);
}

/**
 * @brief Test FWHM functions
 */
static void eris_pfits_fwhm_test(void)
{
    cpl_propertylist* header = create_test_header();
    
    /* Test FWHM values */
    double fwhmlin = eris_pfits_get_ia_fwhmlin(header);
    cpl_test_abs(fwhmlin, 0.8, 0.000001);
    
    double fwhm_start = eris_pfits_get_fwhm_start(header);
    cpl_test_abs(fwhm_start, 0.7, 0.000001);
    
    double fwhm_end = eris_pfits_get_fwhm_end(header);
    cpl_test_abs(fwhm_end, 0.9, 0.000001);
    
    /* Test error handling */
    cpl_test_eq(eris_pfits_get_ia_fwhmlin(NULL), 0.0);
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    
    /* Test missing values */
    cpl_propertylist_erase(header, "ESO TEL IA FWHMLIN");
    cpl_test_eq(eris_pfits_get_ia_fwhmlin(header), 0.0);
    cpl_test_error(CPL_ERROR_DATA_NOT_FOUND);
    
    cpl_propertylist_erase(header, "ESO TEL AMBI FWHM START");
    cpl_test_eq(eris_pfits_get_fwhm_start(header), 0.0);
    cpl_test_error(CPL_ERROR_DATA_NOT_FOUND);
    
    cpl_propertylist_erase(header, "ESO TEL AMBI FWHM END");
    cpl_test_eq(eris_pfits_get_fwhm_end(header), 0.0);
    cpl_test_error(CPL_ERROR_DATA_NOT_FOUND);
    cpl_test_error(CPL_ERROR_NONE);
    /* Test invalid values */
    cpl_propertylist_append_double(header, "ESO TEL AMBI FWHM START", -1.0);
    cpl_test_eq(eris_pfits_get_fwhm_start(header), 0.0);
    cpl_test_error(CPL_ERROR_UNSPECIFIED);  //TODO: unexpected error code

    cpl_propertylist_append_double(header, "ESO TEL AMBI FWHM END", 0.0);
    cpl_test_eq(eris_pfits_get_fwhm_end(header), 0.0);
    cpl_test_error(CPL_ERROR_UNSPECIFIED);//TODO: unexpected error code


    /* Cleanup */
    cpl_propertylist_delete(header);
}

/**
 * @brief Test WCS coordinate functions
 */
static void eris_pfits_wcs_coord_test(void)
{
    cpl_propertylist* header = create_test_header();
    
    /* Test WCS coordinates */
    double cd33 = eris_pfits_get_cd33(header);
    cpl_test_abs(cd33, 1.0, 0.000001);
    
    double crval3 = eris_pfits_get_crval3(header);
    cpl_test_abs(crval3, 2.0, 0.000001);
    
    double cdelt3 = eris_pfits_get_cdelt3(header);
    cpl_test_abs(cdelt3, 0.1, 0.000001);
    
    double crpix1 = eris_pfits_get_crpix1(header);
    cpl_test_abs(crpix1, 50.0, 0.000001);
    
    double crpix2 = eris_pfits_get_crpix2(header);
    cpl_test_abs(crpix2, 100.0, 0.000001);
    
    double crpix3 = eris_pfits_get_crpix3(header);
    cpl_test_abs(crpix3, 1.0, 0.000001);
    
    /* Test error handling */
    cpl_test_eq(eris_pfits_get_cd33(NULL), 0.0);
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    
    /* Cleanup */
    cpl_propertylist_delete(header);
}

/**
 * @brief Test airmass functions
 */
static void eris_pfits_airmass_test(void)
{
    cpl_propertylist* header = create_test_header();
    
    /* Test airmass values */
    double airmass_start = eris_pfits_get_airmass_start(header);
    cpl_test_abs(airmass_start, 1.1, 0.000001);
    
    double airmass_end = eris_pfits_get_airmass_end(header);
    cpl_test_abs(airmass_end, 1.3, 0.000001);
    
    double airmass = eris_pfits_get_airmass(header);
    cpl_test_abs(airmass, 1.2, 0.000001);  // Average of start and end
    
    /* Test error handling */
    cpl_test_eq(eris_pfits_get_airmass_start(NULL), 0.0);
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    
    /* Test missing values */
    cpl_propertylist_erase(header, "ESO TEL AIRM START");
    cpl_test_eq(eris_pfits_get_airmass_start(header), 0.0);
    cpl_test_error(CPL_ERROR_DATA_NOT_FOUND);
    
    cpl_propertylist_erase(header, "ESO TEL AIRM END");
    cpl_test_eq(eris_pfits_get_airmass_end(header), 0.0);
    cpl_test_error(CPL_ERROR_DATA_NOT_FOUND);
    
    /* Test average with missing values */
    cpl_test_eq(eris_pfits_get_airmass(header), 0.0);
    cpl_test_error(CPL_ERROR_DATA_NOT_FOUND);
    
    /* Test invalid values */
    cpl_propertylist_append_double(header, "ESO TEL AIRM START", -1.0);
    cpl_test_eq(eris_pfits_get_airmass_start(header), 0.0);
    cpl_test_error(CPL_ERROR_UNSPECIFIED);//TODO: unexpected error code
    
    cpl_propertylist_append_double(header, "ESO TEL AIRM END", 0.0);
    cpl_test_eq(eris_pfits_get_airmass_end(header), 0.0);
    cpl_test_error(CPL_ERROR_UNSPECIFIED);//TODO: unexpected error code
    
    /* Cleanup */
    cpl_propertylist_delete(header);
}

/**
 * @brief Test observation info functions
 */
static void eris_pfits_obsinfo_test(void)
{
    cpl_propertylist* header = create_test_header();
    
    /* Test observation info */
    double mjdobs = eris_pfits_get_mjdobs(header);
    cpl_test_abs(mjdobs, 58000.0, 0.000001);
    
    const char* progid = eris_pfits_get_progid(header);
    cpl_test_nonnull(progid);
    cpl_test_eq_string(progid, "60.A-9252(A)");
    
    long obsid = eris_pfits_get_obsid(header);
    cpl_test_eq(obsid, 12345);
    
    const char* pipefile = eris_pfits_get_pipefile(header);
    cpl_test_nonnull(pipefile);
    cpl_test_eq_string(pipefile, "test_pipe.fits");
    
    const char* datamd5 = eris_pfits_get_datamd5(header);
    cpl_test_nonnull(datamd5);
    cpl_test_eq_string(datamd5, "abcdef123456");
    
    /* Test error handling */
    cpl_test_eq(eris_pfits_get_mjdobs(NULL), 0.0);
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    
    /* Cleanup */
    cpl_propertylist_delete(header);
}

/**
 * @brief Test header cleaning functions
 */
static void eris_pfits_clean_test(void)
{
    cpl_propertylist* header = create_test_header();
    
    /* Test basic cleaning */
    cpl_test_eq_error(eris_pfits_clean_header(header, CPL_FALSE), CPL_ERROR_NONE);
    cpl_test_zero(cpl_propertylist_has(header, "TELESCOP"));
    cpl_test_zero(cpl_propertylist_has(header, "INSTRUME"));
    
    /* Test extra cleaning */
    cpl_test_eq_error(eris_pfits_clean_header(header, CPL_TRUE), CPL_ERROR_NONE);
    cpl_test_zero(cpl_propertylist_has(header, "ESO INS MODE"));
    cpl_test_zero(cpl_propertylist_has(header, "ESO DET DIT"));
    
    /* Test ESO keyword pattern cleaning */
    cpl_test_zero(cpl_propertylist_has(header, "ESO ADA POSANG"));
    cpl_test_zero(cpl_propertylist_has(header, "ESO AOS LOOP ST"));
    cpl_test_zero(cpl_propertylist_has(header, "ESO INS TEMP"));
    cpl_test_zero(cpl_propertylist_has(header, "ESO DET READ"));
    cpl_test_zero(cpl_propertylist_has(header, "ESO DPR TYPE"));
    cpl_test_zero(cpl_propertylist_has(header, "ESO LGS FOCUS"));
    cpl_test_zero(cpl_propertylist_has(header, "ESO OCS NAME"));
    cpl_test_zero(cpl_propertylist_has(header, "ESO OBS ID"));
    cpl_test_zero(cpl_propertylist_has(header, "ESO PRO CATG"));
    cpl_test_zero(cpl_propertylist_has(header, "ESO SEQ ARM"));
    cpl_test_zero(cpl_propertylist_has(header, "ESO TEL ALT"));
    cpl_test_zero(cpl_propertylist_has(header, "ESO TPL NAME"));
    
    /* Test RA/DEC/MJD-OBS cleaning */
    cpl_test_eq_error(eris_pfits_clean_header_ra_dec_mjd_obs(header), CPL_ERROR_NONE);
    cpl_test_zero(cpl_propertylist_has(header, "RA"));
    cpl_test_zero(cpl_propertylist_has(header, "DEC"));
    cpl_test_zero(cpl_propertylist_has(header, "MJD-OBS"));
    
    /* Test comment cleaning */
    cpl_test_eq_error(eris_pfits_clean_comment(header), CPL_ERROR_NONE);
    cpl_test_zero(cpl_propertylist_has(header, "COMMENT"));
    
    /* Test expmap extra keys cleaning */
    cpl_test_eq_error(eris_ifu_plist_erase_expmap_extra_keys(header), CPL_ERROR_NONE);
    cpl_test_zero(cpl_propertylist_has(header, "HDUCLASS"));
    cpl_test_zero(cpl_propertylist_has(header, "HDUDOC"));
    cpl_test_zero(cpl_propertylist_has(header, "HDUVERS"));
    cpl_test_zero(cpl_propertylist_has(header, "HDUCLAS1"));
    cpl_test_zero(cpl_propertylist_has(header, "HDUCLAS2"));
    cpl_test_zero(cpl_propertylist_has(header, "ERRDATA"));
    cpl_test_zero(cpl_propertylist_has(header, "QUALDATA"));
    
    /* Test error handling */
    cpl_test_eq_error(eris_pfits_clean_header(NULL, CPL_FALSE), CPL_ERROR_NULL_INPUT);
    cpl_test_eq_error(eris_ifu_plist_erase_expmap_extra_keys(NULL), CPL_ERROR_NULL_INPUT);
    
    /* Cleanup */
    cpl_propertylist_delete(header);
}

int main(void)
{
    cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_WARNING);
    
    /* Run all tests */
    //eris_pfits_basic_test();
    cpl_test_error(CPL_ERROR_NONE);
    
    eris_pfits_naxis_test();
    cpl_test_error(CPL_ERROR_NONE);
    
    eris_pfits_wcs_test();
    cpl_test_error(CPL_ERROR_NONE);
    
    eris_pfits_raw_test();
    cpl_test_error(CPL_ERROR_NONE);
    
    eris_pfits_fwhm_test();
    cpl_test_error(CPL_ERROR_NONE);
    
    eris_pfits_wcs_coord_test();
    cpl_test_error(CPL_ERROR_NONE);
    
    eris_pfits_airmass_test();//TODO: unexpected error code
    cpl_test_error(CPL_ERROR_NONE);
    
    eris_pfits_obsinfo_test();
    cpl_test_error(CPL_ERROR_NONE);
    
    eris_pfits_clean_test();
    cpl_test_error(CPL_ERROR_NONE);
    
    return cpl_test_end(0);
}

/**@}*/
