from __future__ import with_statement
from __future__ import absolute_import
from __future__ import print_function
import sys


try:
    import numpy
    import reflex
    from pipeline_product import PipelineProduct
    from reflex_plot_widgets import *
    import pipeline_display

    import_success = True
    try:
        from astropy.io import fits as pyfits
    except ImportError:
        import pyfits
       
except ImportError:
    import_success = False
    print("Error importing modules pyfits, wx, matplotlib, numpy")


def paragraph(text, width=None):
    """ wrap text string into paragraph
       text:  text to format, removes leading space and newlines
       width: if not None, wraps text, not recommended for tooltips as
              they are wrapped by wxWidgets by default
    """
    import textwrap

    if width is None:
        return textwrap.dedent(text).replace("\n", " ").strip()
    else:
        return textwrap.fill(textwrap.dedent(text), width=width)


class DataPlotterManager(object):
    """
    This class must be added to the PipelineInteractiveApp with setPlotManager
    It must have following member functions which will be called by the app:
     - setInteractiveParameters(self)
     - readFitsData(self, fits_files):
     - addSubplots(self, figure):
     - plotProductsGraphics(self)
    Following members are optional:
     - setWindowHelp(self)
     - setWindowTitle(self)
     - setCurrentParameterHelper(self, helper)
    """

    # static members
    recipe_name = "eris_nix_detmon_ir_lg"
    img_cat_list = ["BP_MAP_NL"]

    def setInteractiveParameters(self):
        """
        This function specifies which are the parameters that should be presented
        in the window to be edited.  Note that the parameter has to also be in the
        in_sop port (otherwise it won't appear in the window). The descriptions are
        used to show a tooltip. They should match one to one with the parameter
        list.
        """
        return [
            reflex.RecipeParameter(
                recipe=self.recipe_name,
                displayName="saturation_limit",
                group="eris_nix_detmon_ir_lg",
                description="Limit of mean signal (ADU) above which to ignore data",
            ),
            reflex.RecipeParameter(
                recipe=self.recipe_name,
                displayName="gain_threshold",
                group="eris_nix_detmon_ir_lg",
                description="Limit of mean signal (ADU) for frames contributing to gain",
            ),
        ]

    def readFitsData(self, fits_files):
        """
        This function should be used to read and organize the raw fits files
        produced by the recipes.
        It receives as input a list of reflex.fits_files
        """
        # organize the files into a dictionary, here we assume we only have
        # one file per category if there are more, one must use a
        # dictionary of lists
        self.frames = dict()
        for f in fits_files:
            self.frames[f.category] = PipelineProduct(f)
            header = pyfits.open(f.name)

            if 'ESO PRO REC1 ID' in header[0].header:
                rec_id = header[0].header['ESO PRO REC1 ID']
                if rec_id == self.recipe_name:
                    try:
                        self.read_curname = header[0].header['ESO DET READ CURNAME']
                    except:
                        self.read_curname = header[0].header['ESO DET NCORRS NAME']

        # we only have two states, we have data or we don't
        # define the plotting functions we want to use for each
        # the method can handle multiple 'CAL_DET' categories,
        # adopt the first to match
        self.img_cat = None
        for cat in self.img_cat_list:
            if cat in self.frames:
                self.img_cat = cat
                break
        if self.img_cat is not None:
            self._add_subplots = self._add_data_subplots
            self._plot = self._data_plot
        else:
            self._add_subplots = self._add_nodata_subplots
            self._plot = self._nodata_plot

    def addSubplots(self, figure):
        """
        This function should be used to setup the subplots of the gui.  The the
        matplotlib documentation for a description of subplots.
        """
        self._add_subplots(figure)

    def plotProductsGraphics(self):
        """
        This function should be used to plot the data onto the subplots.
        """
        self._plot()

    @staticmethod
    def setWindowHelp():
        return "Help for eris_nix_detmon_ir_lg interactive window"

    @staticmethod
    def setWindowTitle():
        return "eris_nix_detmon_ir_lg interactive window"

    def _add_nodata_subplots(self, figure):
        self.txt_plot = figure.add_subplot(111)

    def _add_data_subplots(self, figure):
        self.img_data_plot = figure.add_subplot(221)
        self.img_err_plot = figure.add_subplot(222)
        self.img_dq_plot = figure.add_subplot(223)
        self.img_text_plot = figure.add_subplot(224)

    def _data_plot(self):
        # get the right category file from our dictionary

        p = self.frames[self.img_cat]
        p.readImage()
        # setup the image display
        imgdisp = pipeline_display.ImageDisplay()
        imgdisp.setLabels("X", "Y")
        tooltip = paragraph(
            """\
                            Bad (non-linear) pixel map."""
        )
        imgdisp.setCmap("hot")

        if self.img_cat == "BP_MAP_NL":
            catg = "BP_MAP_NL ("+self.read_curname+")"
        else:
            catg = "UNKNOWN"

        imgdisp.z_lim = 0, 1
        imgdisp.display(self.img_data_plot, "%s data" % catg, tooltip, p.image)

        # write warning
        self.img_err_plot.cla()
        self.img_err_plot.set_axis_off()
        self.img_dq_plot.cla()
        self.img_dq_plot.set_axis_off()
        self.img_text_plot.cla()
        self.img_text_plot.set_axis_off()
        self.img_text_plot.text(
                    -0.8,
                    0.5,
                    """
                    Current version set proper values of 
                    saturation_threshold and gain_threshold as a 
                    function of the read-out speed 
                    (DET.READ.CURNAME). They are:
                    parameter                  value    det.read.curname  
                    saturation_threshold   49000     FAST_UNCORR
                    gain_threshold            30000     FAST_UNCORR
                    saturation_threshold   15000     SLOW_GR_UTR
                    gain_threshold            15000     SLOW_GR_UTR
                    """,
                    color="black",
                    fontsize=10,
                    ha="left",
                    va="center",
                    alpha=0.7)
        
    #        p.readImage('ERR')
    #        imgdisp = pipeline_display.ImageDisplay()
    #        imgdisp.setLabels('X', 'Y')
    #        tooltip = paragraph("""\
    #                            Error on reduced data.""")
    #        imgdisp.setCmap('hot')
    #        imgdisp.display(self.img_err_plot, "%s error" % catg,
    #                        tooltip, p.image)

    #        p.readImage('DQ')
    #        imgdisp = pipeline_display.ImageDisplay()
    #        imgdisp.setLabels('X', 'Y')
    #        tooltip = paragraph("""\
    #                            Data quality on reduced data.""")
    #        imgdisp.z_lim = 0, 1
    #        imgdisp.setCmap('hot')
    #        imgdisp.display(self.img_dq_plot, "%s quality" % catg,
    #                        tooltip, p.image)

    def _nodata_plot(self):
        # could be moved to reflex library?
        self.txt_plot.set_axis_off()
        text_nodata = "Data not found. Input files should contain these" " types:\n"
        for cat in self.img_cat_list:
            text_nodata += "%s\n" % cat
        self.txt_plot.text(
            0.1,
            0.6,
            text_nodata,
            color="#11557c",
            fontsize=18,
            ha="left",
            va="center",
            alpha=1.0,
        )
        self.txt_plot.tooltip = "No data found"

    @staticmethod
    def plotWidgets():
        widgets = list()
        #        if hasattr(self, 'spec_plot') {
        # Updating number
        #            self.clickablespec = InteractiveClickableSubplot(
        #                self.spec_plot, self.increaseFloatNumber)
        #            widgets.append(self.clickablespec)
        #        }
        return widgets

    def increaseFloatNumber(self, point):
        floatpoint = self.getCurrentParameterHelper("floatopt") + 1
        new_params = list()
        new_params.append(
            reflex.RecipeParameter(self.recipe_name, "floatopt", value=str(floatpoint))
        )
        return new_params

    def setCurrentParameterHelper(self, helper):
        self.getCurrentParameterHelper = helper


# This is the 'main' function
if __name__ == "__main__":
    from reflex_interactive_app import PipelineInteractiveApp

    # Create interactive application
    interactive_app = PipelineInteractiveApp(enable_init_sop=True)

    # get inputs from the command line
    interactive_app.parse_args()

    # Check if import failed or not
    if not import_success:
        interactive_app.setEnableGUI(False)

    # Open the interactive window if enabled
    if interactive_app.isGUIEnabled():
        # Get the specific functions for this window
        dataPlotManager = DataPlotterManager()

        interactive_app.setPlotManager(dataPlotManager)
        interactive_app.showGUI()
    else:
        interactive_app.set_continue_mode()

    # Print outputs. This is parsed by the Reflex python actor to
    # get the results. Do not remove
    interactive_app.print_outputs()
    sys.exit()
