/* $Id$
 *
 * This file is part of the ERIS Pipeline
 * Copyright (C) 2002,2003 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdbool.h>
#include <string.h>
#include <strings.h>
#include <math.h>

#include "eris_ifu_error.h"
#include "eris_ifu_utils.h"
#include "eris_ifu_functions.h"
#include "eris_ifu_dfs.h"
#include "eris_ifu_jitter_interface.h"
#include "eris_ifu_wavecal_static.h"
#include "eris_ifu_distortion_static.h"
#include "eris_ifu_lambda_corr.h"
#include "eris_ifu_sky_tweak.h"
#include "eris_ifu_efficiency_response.h"
#include "eris_utils.h"
#include <eris_ifu_resample.h>
cpl_error_code eris_ifu_stdstar_fill_common_parameterlist(
		const char *recipeName,
		jitterModes jitterMode,
		cpl_parameterlist *pl)
{

	cpl_ensure_code(recipeName, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(jitterMode >= M_UNSET && jitterMode <= M_TELLURIC, CPL_ERROR_ILLEGAL_INPUT);
	cpl_ensure_code(pl, CPL_ERROR_NULL_INPUT);

	cpl_error_code  err     = CPL_ERROR_NONE;
	cpl_parameter   *p      = NULL;
	char            pName[256];
	char            *context = NULL;

	context = cpl_sprintf("eris.%s", recipeName);

	/* efficiency params */
	snprintf(pName, sizeof(pName), "%s.%s", context, "compute-efficiency");
	p = cpl_parameter_new_value(pName, CPL_TYPE_BOOL,
			"If True try to compute the efficiency from a STD star spectrum",
			context, CPL_FALSE);
	if (jitterMode == M_SCIENCE) {
		cpl_parameter_set_default_bool(p, CPL_FALSE);
	}
	cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "compute-efficiency");
	cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
	cpl_parameterlist_append(pl, p);

	/* response params */
	snprintf(pName, sizeof(pName), "%s.%s", context, "compute-response");
	p = cpl_parameter_new_value(pName, CPL_TYPE_BOOL,
			"If True try to compute the response from a STD star spectrum",
			context, CPL_FALSE);
	if (jitterMode == M_SCIENCE) {
		cpl_parameter_set_default_bool(p, CPL_FALSE);
	}
	cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "compute-response");
	cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
	cpl_parameterlist_append(pl, p);

	/* strehl params */
	snprintf(pName, sizeof(pName), "%s.%s", context, "compute-strehl");
	p = cpl_parameter_new_value(pName, CPL_TYPE_BOOL,
			"If True try to compute the response from a PSF star spectrum",
			context, CPL_FALSE);
	if (jitterMode == M_SCIENCE) {
		cpl_parameter_set_default_bool(p, CPL_FALSE);
	}
	cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "compute-strehl");
	cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
	cpl_parameterlist_append(pl, p);



	snprintf(pName, sizeof(pName), "%s.%s", context, "strehl_flux_radius");
	p = cpl_parameter_new_value(pName, CPL_TYPE_DOUBLE,
			"PSF Flux integration radius [pixels]. If -1 uses 3 times object PSF FWHM.",
			context, -1.);

	cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "strehl_flux_radius");
	cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
	cpl_parameterlist_append(pl, p);

	snprintf(pName, sizeof(pName), "%s.%s", context, "strehl_bkg-radius-low");
	p = cpl_parameter_new_value(pName, CPL_TYPE_DOUBLE,
			"PSF background inner radii [pixels]. If -1 uses 1.5 times strehl_flux_radius.",
			context, -1.);

	cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "strehl_bkg-radius-low");
	cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
	cpl_parameterlist_append(pl, p);

	snprintf(pName, sizeof(pName), "%s.%s", context, "strehl_bkg-radius-high");
	p = cpl_parameter_new_value(pName, CPL_TYPE_DOUBLE,
			"PSF background outer radius [pixels]. If -1 uses 2.0 times strehl_flux_radius.",
			context, -1.);

	cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "strehl_bkg-radius-high");
	cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
	cpl_parameterlist_append(pl, p);


	snprintf(pName, sizeof(pName), "%s.%s", context, "response-polyfit-deg");
	p = cpl_parameter_new_range(pName, CPL_TYPE_INT,
			"Degree of polynomial fit of response data points.", context,
			3, 3, 5);

	cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "response-polyfit-deg");
	cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
	cpl_parameterlist_append(pl, p);

	snprintf(pName, sizeof(pName), "%s.%s", context, "response-polyfit-ksigma-kappa");
	p = cpl_parameter_new_range(pName, CPL_TYPE_DOUBLE,
				"Kappa value used to clip data points in polynomial fit of response data points.", context,
				3.0, 0.0, 10.);

    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "response-polyfit-ksigma-kappa");
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
	cpl_parameterlist_append(pl, p);




	snprintf(pName, sizeof(pName), "%s.%s", context, "response-polyfit-ksigma-niter");
	p = cpl_parameter_new_range(pName, CPL_TYPE_INT,
			"Degree of polynomial fit of response data points.", context,
			3, 1, 10);

	cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "response-polyfit-ksigma-niter");
	cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
	cpl_parameterlist_append(pl, p);


	eris_ifu_free_string(&context);
	eris_check_error_code("eris_ifu_stdstar_fill_common_parameterlist");
	return err;
}
static cpl_error_code eris_ifu_jitter_fill_resample_parameterlist(char* context,
		cpl_parameterlist *pl)
{

	cpl_parameter   *p      = NULL;
	//    char            *pName = NULL;
	char            pName[256];


	/* --hdrldemo_resample.method */
	snprintf(pName, sizeof(pName), "%s.%s", context, "method");
	p = cpl_parameter_new_enum(pName, CPL_TYPE_STRING,
			"Resampling method", context, "DRIZZLE",6,
			"NEAREST", "LINEAR", "QUADRATIC", "RENKA",
			"DRIZZLE","LANCZOS");
	cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "method");
	cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
	cpl_parameterlist_append(pl, p);

	/* --hdrldemo_resample.method.loop-distance */
	snprintf(pName, sizeof(pName), "%s.%s", context, "method.loop-distance");
	p = cpl_parameter_new_value(pName, CPL_TYPE_INT,
			"Loop distance used by all (but NEAREST) "
			"methods to control the number of surrounding "
			"voxels that are taken into account. "
			"A small value allow faster re-sampling but may not give good quality",
			context, LOOP_DISTANCE);
	cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "method.loop-distance");
	cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
	cpl_parameterlist_append(pl, p);

	/* --hdrldemo_resample.method.use-errorweights*/
	snprintf(pName, sizeof(pName), "%s.%s", context, "method.use-errorweights");
	p = cpl_parameter_new_value(pName,
			CPL_TYPE_BOOL,
			"Use additional weights of 1/err^2", context,
			CPL_FALSE);
	cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI,
			"method.use-errorweights");
	cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
	cpl_parameterlist_append(pl, p);

	/* --hdrldemo_resample.method.renka.critical-radius */
	snprintf(pName, sizeof(pName), "%s.%s", context, "method.renka.critical-radius");
	p = cpl_parameter_new_value(pName,
			CPL_TYPE_DOUBLE,"Critical radius of the Renka "
			"method",
			context, RENKA_CRITICAL_RADIUS);
	cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI,
			"method.renka.critical-radius");
	cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
	cpl_parameterlist_append(pl, p);

	/* --hdrldemo_resample.method.lanczos.kernel-size */
	snprintf(pName, sizeof(pName), "%s.%s", context, "method.lanczos.kernel-size");
	p = cpl_parameter_new_value(pName,
			CPL_TYPE_INT,"Kernel size of the Lanczos "
			"method",
			context, LANCZOS_KERNEL_SIZE);
	cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI,
			"method.lanczos.kernel-size");
	cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
	cpl_parameterlist_append(pl, p);

	/* --hdrldemo_resample.method.drizzle.downscale-x */
	snprintf(pName, sizeof(pName), "%s.%s", context, "method.drizzle.downscale-x");
	p = cpl_parameter_new_value(pName,
			CPL_TYPE_DOUBLE, "Drizzle down-scaling factor "
			"in x direction",
			context, DRIZZLE_DOWN_SCALING_FACTOR_X);
	cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI,
			"method.drizzle.downscale-x");
	cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
	cpl_parameterlist_append(pl, p);

	/* --hdrldemo_resample.method.drizzle.downscale-y */
	snprintf(pName, sizeof(pName), "%s.%s", context, "method.drizzle.downscale-y");
	p = cpl_parameter_new_value(pName,
			CPL_TYPE_DOUBLE, "Drizzle down-scaling factor "
			"in y direction",
			context, DRIZZLE_DOWN_SCALING_FACTOR_Y);
	cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI,
			"method.drizzle.downscale-y");
	cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
	cpl_parameterlist_append(pl, p);

	/* --hdrldemo_resample.method.drizzle.downscale-z*/
	snprintf(pName, sizeof(pName), "%s.%s", context, "method.drizzle.downscale-z");
	p = cpl_parameter_new_value(pName,
			CPL_TYPE_DOUBLE, "Drizzle down-scaling factor "
			"in wavelength direction",
			context, DRIZZLE_DOWN_SCALING_FACTOR_Z);
	cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI,
			"method.drizzle.downscale-z");
	cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
	cpl_parameterlist_append(pl, p);

	eris_check_error_code("eris_ifu_jitter_fill_resample_parameterlist");
	return cpl_error_get_code();


}

//static cpl_error_code eris_ifu_jitter_fill_outgrid_parameterlist(char* context,
//		cpl_parameterlist *pl)
//{
//	cpl_parameter   *p      = NULL;
//	//    char            *pName = NULL;
//	char            pName[256];


//	/* --hdrldemo_resample.outgrid.ra-min */
//	snprintf(pName, sizeof(pName), "%s.%s", context, "outgrid.ra-min");
//	p = cpl_parameter_new_value(pName, CPL_TYPE_DOUBLE,
//			"Minimum right ascension of the output "
//			"image/cube in degree", context, -1.0);
//	cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "outgrid.ra-min");
//	cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
//	cpl_parameterlist_append(pl, p);


//	/* --hdrldemo_resample.outgrid.ra-max */
//	snprintf(pName, sizeof(pName), "%s.%s", context, "outgrid.ra-max");
//	p = cpl_parameter_new_value(pName, CPL_TYPE_DOUBLE,
//			"Maximum right ascension of the output "
//			"image/cube in degree", context, -1.0);
//	cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "outgrid.ra-max");
//	cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
//	cpl_parameterlist_append(pl, p);


//	/* --hdrldemo_resample.outgrid.dec-min */
//	snprintf(pName, sizeof(pName), "%s.%s", context, "outgrid.dec-min");
//	p = cpl_parameter_new_value(pName, CPL_TYPE_DOUBLE,
//			"Minimum declination of the output "
//			"image/cube in degree", context, -1.0);
//	cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "outgrid.dec-min");
//	cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
//	cpl_parameterlist_append(pl, p);


//	/* --hdrldemo_resample.outgrid.dec-max */
//	snprintf(pName, sizeof(pName), "%s.%s", context, "outgrid.dec-max");
//	p = cpl_parameter_new_value(pName, CPL_TYPE_DOUBLE,
//			"Maximum declination of the output "
//			"image/cube in degree", context, -1.0);
//	cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "outgrid.dec-max");
//	cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
//	cpl_parameterlist_append(pl, p);


//	/* --hdrldemo_resample.lambda */
//	snprintf(pName, sizeof(pName), "%s.%s", context, "outgrid.lambda-min");
//	p = cpl_parameter_new_value(pName,
//			CPL_TYPE_DOUBLE, "Minimum wavelength of the "
//			"output cube in meter", context, -1.);
//	cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "outgrid.lambda-min");
//	cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
//	cpl_parameterlist_append(pl, p);


//	/* --hdrldemo_resample.lambda */
//	snprintf(pName, sizeof(pName), "%s.%s", context, "outgrid.lambda-max");
//	p = cpl_parameter_new_value(pName,
//			CPL_TYPE_DOUBLE, "Maximum wavelength of the "
//			"output cube in meter", context, -1.);
//	cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "outgrid.lambda-max");
//	cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
//	cpl_parameterlist_append(pl, p);


//	/* --hdrldemo_resample.outgrid.delta-ra */
//	snprintf(pName, sizeof(pName), "%s.%s", context, "outgrid.delta-ra");
//	p = cpl_parameter_new_value(pName,CPL_TYPE_DOUBLE,
//			"Output step-size in right ascension in degree",
//			context, -1.);
//	cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "outgrid.delta-ra");
//	cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
//	cpl_parameterlist_append(pl, p);


//	/* --hdrldemo_resample.outgrid.delta-dec */
//	snprintf(pName, sizeof(pName), "%s.%s", context, "outgrid.delta-dec");
//	p = cpl_parameter_new_value(pName,
//			CPL_TYPE_DOUBLE, "Output step-size in "
//			"declination in degree",
//			context, -1.);
//	cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "outgrid.delta-dec");
//	cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
//	cpl_parameterlist_append(pl, p);


//	/* --hdrldemo_resample.outgrid.delta-lambda */
//	snprintf(pName, sizeof(pName), "%s.%s", context, "outgrid.delta-lambda");
//	p = cpl_parameter_new_value(pName,
//			CPL_TYPE_DOUBLE, "Output step-size in "
//			"wavelength in meter",
//			context, -1.);
//	cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "outgrid.delta-lambda");
//	cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
//	cpl_parameterlist_append(pl, p);


//	/* --hdrldemo_resample.save-table */
//	snprintf(pName, sizeof(pName), "%s.%s", context, "save-table");
//	p = cpl_parameter_new_value(pName, CPL_TYPE_BOOL,
//			"Save the table before resampling", context,
//			CPL_FALSE);
//	cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "save-table");
//	cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
//	cpl_parameterlist_append(pl, p);



//	eris_check_error_code("eris_ifu_jitter_fill_outgrid_parameterlist");
//	return cpl_error_get_code();
//}

static cpl_error_code eris_ifu_jitter_fill_dar_parameterlist(char* context,
		jitterModes jitterMode,
		cpl_parameterlist *pl)
{
	cpl_parameter   *p      = NULL;
	//    char            *pName = NULL;
	char            pName[256];


	/* DAR */
	snprintf(pName, sizeof(pName), "%s.%s", context, "dar-corr");
	p = cpl_parameter_new_value(pName, CPL_TYPE_BOOL,
			"Correct Differential Atmospheric Refraction (DAR)",
			context, CPL_FALSE);
	if (jitterMode == M_SCIENCE) {
		cpl_parameter_set_default_bool(p, CPL_FALSE);
	}
	cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "dar-corr");
	cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
	cpl_parameterlist_append(pl, p);

	snprintf(pName, sizeof(pName), "%s.%s", context, "dar-shift-method");
	p = cpl_parameter_new_range(pName, CPL_TYPE_INT,
			"The method to shift images for DAR correction" , context,
			0, 0, 8);
	cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "dar-shift-method");
	cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
	cpl_parameterlist_append(pl, p);

	snprintf(pName, sizeof(pName), "%s.%s", context, "dar-shift-length");
	p = cpl_parameter_new_value(pName, CPL_TYPE_INT,
			"Kernel length for DAR shift image interpolation",
			context, 0);
	cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "dar-shift-length");
	cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
	cpl_parameterlist_append(pl, p);

	snprintf(pName, sizeof(pName), "%s.%s", context, "dar-shift-radius");
	p = cpl_parameter_new_value(pName, CPL_TYPE_DOUBLE,
			"Kernel radius for DAR shift image interpolation",
			context, 0.);
	cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "dar-shift-radius");
	cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
	cpl_parameterlist_append(pl, p);

	eris_check_error_code("eris_ifu_jitter_fill_dar_parameterlist");
	return cpl_error_get_code();;
}
cpl_error_code eris_ifu_jitter_fill_extract_parameterlist(char* context,
		jitterModes jitterMode,
		cpl_parameterlist *pl)
{

	cpl_ensure_code(context, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code((jitterMode >= M_UNSET) && (jitterMode <= M_TELLURIC), CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(pl, CPL_ERROR_NULL_INPUT);

	cpl_parameter   *p      = NULL;
	//    char            *pName = NULL;
	char            pName[256];

	/* extraction params */
	snprintf(pName, sizeof(pName), "%s.%s", context, "extract-source");
	p = cpl_parameter_new_value(pName, CPL_TYPE_BOOL,
			"If True try to extract a point souce",
			context, CPL_TRUE);
	if (jitterMode == M_SCIENCE || jitterMode == M_PUPIL) {
		cpl_parameter_set_default_bool(p, CPL_FALSE);
	}
	cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "extract-source");
	cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
	cpl_parameterlist_append(pl, p);

	/* --mask_method */
	snprintf(pName, sizeof(pName), "%s.%s", context, "mask_method");
	p = cpl_parameter_new_value(pName, CPL_TYPE_STRING,
            "Method to specify extraction mask : mask, position, max, fit or optimal",
			context, "max");
	cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "mask_method");
	cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
	cpl_parameterlist_append(pl, p);

	/* --centre */
	snprintf(pName, sizeof(pName), "%s.%s", context, "center");
	p = cpl_parameter_new_value(pName, CPL_TYPE_STRING,
			"The centre of the circular mask (pixel)",
			context, "32,32");
	cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "center");
	cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
	cpl_parameterlist_append(pl, p);

	/* --radius */
	double radius = 4.0;
	if(strstr(context, "stdstar")) {
		radius = 11.0;
	}
	snprintf(pName, sizeof(pName), "%s.%s", context, "radius");
	p = cpl_parameter_new_value(pName, CPL_TYPE_DOUBLE,
			"The radius of the circular mask (pixel)",
			context, radius);
	cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "radius");
	cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
	cpl_parameterlist_append(pl, p);


	eris_check_error_code("eris_ifu_jitter_fill_extract_parameterlist");
	return cpl_error_get_code();;
}

cpl_error_code eris_ifu_jitter_fill_common_parameterlist(
		const char *recipeName,
		jitterModes jitterMode,
		cpl_parameterlist *pl)
{

	cpl_ensure_code(recipeName, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(jitterMode >= M_UNSET && jitterMode <= M_TELLURIC, CPL_ERROR_ILLEGAL_INPUT);
	cpl_ensure_code(pl, CPL_ERROR_NULL_INPUT);


	cpl_error_code  err     = CPL_ERROR_NONE;
	cpl_parameter   *p      = NULL;
	//    char            *pName = NULL;
	char            pName[256];
	char* context = NULL;
	TRY
	{
		context = cpl_sprintf("eris.%s", recipeName);

		eris_ifu_add_std_params(pl, recipeName);

		/* --sky_tweak */
		snprintf(pName, sizeof(pName), "%s.%s", context, "sky_tweak");
		p = cpl_parameter_new_value(pName, CPL_TYPE_INT,
				"Use modified sky cube for sky subtraction."
                "0: don't apply, 1: Davies' method)",
				context, 0);
		cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "sky_tweak");
		cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
		cpl_parameterlist_append(pl, p);

		/* --skip_sky_oh_align */
		snprintf(pName, sizeof(pName), "%s.%s", context, "skip_sky_oh_align");

		p = cpl_parameter_new_value(pName, CPL_TYPE_BOOL,
				"Skip the OH alignment for the SKY",
				context, FALSE);
		cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "skip_sky_oh_align");
		cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
		cpl_parameterlist_append(pl, p);

        /* --skip_oh_align */
        snprintf(pName, sizeof(pName), "%s.%s", context, "skip_oh_align");

        p = cpl_parameter_new_value(pName, CPL_TYPE_BOOL,
                                    "Skip the OH alignment",
                                    context, FALSE);
        cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "skip_oh_align");
        cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
        cpl_parameterlist_append(pl, p);
       
        /* order of OH poly fit */
        snprintf(pName, sizeof(pName), "%s.%s", context, "oh_align_poly_order");

        p = cpl_parameter_new_range(pName, CPL_TYPE_INT,
        		"Order of polynomial fit used in the OH alignment for the SKY: [0,3]."
        		" 0 is recommended to prevent extrapolation problems at band edges and in band K."
        		" A higher value may recover a failed correction in case of large flexures, "
        		" but is less accurate at band edges due to polynomial extrapolation",
				context, OH_POLY_DEG, 0, 3);
        cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "oh_align_poly_order");
        cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
        cpl_parameterlist_append(pl, p);


		/* --discard_subband */
		snprintf(pName, sizeof(pName), "%s.%s", context, "discard_subband");
		p = cpl_parameter_new_value(pName, CPL_TYPE_BOOL,
				"Ignore last sub-band in the sky tweaking",
				context, FALSE);
		cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "discard_subband");
		cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
		cpl_parameterlist_append(pl, p);

		/* --stretch_sky */
		snprintf(pName, sizeof(pName), "%s.%s", context, "stretch_sky");
		p = cpl_parameter_new_value(pName, CPL_TYPE_BOOL,
				"Stretch sky in the sky tweaking",
				"kmos.kmos_sci_red", FALSE);
		cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "stretch");
		cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
		cpl_parameterlist_append(pl, p);

		/* --stretch_degree */
		snprintf(pName, sizeof(pName), "%s.%s", context, "stretch_degree");
		p = cpl_parameter_new_value(pName, CPL_TYPE_INT,
				"Stretch polynomial degree",
				context, 8);
		cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "stretch_degree");
		cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
		cpl_parameterlist_append(pl, p);

		/* --stretch_resampling */
		snprintf(pName, sizeof(pName), "%s.%s", context, "stretch_resampling");
		p = cpl_parameter_new_value(pName, CPL_TYPE_STRING,
				"Stretch resampling method (linear/spline)",
				context, "spline");
		cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "stretch_resampling");
		cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
		cpl_parameterlist_append(pl, p);

		/* --tbsub */
		snprintf(pName, sizeof(pName), "%s.%s", context, "tbsub");
		p = cpl_parameter_new_value(pName, CPL_TYPE_BOOL,
				"Subtract thermal background from input cube."
				"(TRUE (apply) or FALSE (don't apply)",
				context, TRUE);
		cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "tbsub");
		cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
		cpl_parameterlist_append(pl, p);

		/* --velocity_offset */
		snprintf(pName, sizeof(pName), "%s.%s", context, "velocity_offset");
		p = cpl_parameter_new_value(pName, CPL_TYPE_DOUBLE,
				"Specify velocity offset correction in km/s for lambda scale",
				context, 0.0);
		cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "velocity_offset");
		cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
		cpl_parameterlist_append(pl, p);

		/* --bpm_threshold */
		snprintf(pName, sizeof(pName), "%s.%s", context, "bpm_threshold");
		p = cpl_parameter_new_value(pName, CPL_TYPE_DOUBLE,
				"Specify a theshold for the interpolated BPM values",
				context, 0.5);
		cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "bpm_threshold");
		cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
		cpl_parameterlist_append(pl, p);

		snprintf(pName, sizeof(pName), "%s.%s", context, "cube.slitlet-detection");
		p = cpl_parameter_new_enum(pName, CPL_TYPE_STRING,
				"Specifies the slitlet detection:"
				" \"DIST\" slitlet distances as detected by distortion recipe "
				"or \"EDGE\" slitlet edges as detected be wavecalrecipe",
				context, "DIST", 3, "DIST", "EDGE", "GRID");
		cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI,
				"cube.slitlet-detection");
		cpl_parameterlist_append(pl, p);

		snprintf(pName, sizeof(pName), "%s.%s", context, "cube.first-col");
		p = cpl_parameter_new_value(pName, CPL_TYPE_DOUBLE,
				"Specifies the first column offset in case the"
				" cube.slitlet-detection parameter is set to \"DIST\"",
				context, 1.0);
		cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI,
				"cube.first-col");
		cpl_parameterlist_append(pl, p);

		snprintf(pName, sizeof(pName), "%s.%s", context, "cube.fine-tune");
		p = cpl_parameter_new_value(pName, CPL_TYPE_INT,
				"Specifies the row interpolation mode: "
				"0 -> no interpolation, otherwise see eris_ifu_1d_interpolation"
				,
				context, -1);
		cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI,
				"cube.fine-tune");
		cpl_parameterlist_append(pl, p);

		/* resample params */
		eris_ifu_jitter_fill_resample_parameterlist(context, pl);

		//eris_ifu_jitter_fill_outgrid_parameterlist(context, pl);

		/* --hdrldemo_resample.subtract-background */
		snprintf(pName, sizeof(pName), "%s.%s", context, "subtract-background");
		p = cpl_parameter_new_value(pName, CPL_TYPE_BOOL,
				"Subtract median of the images in 2D only", context,
				CPL_FALSE);
		cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "subtract-background");
		cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
		cpl_parameterlist_append(pl, p);

		/* --hdrldemo_resample.fieldmargin */
		snprintf(pName, sizeof(pName), "%s.%s", context, "fieldmargin");
		p = cpl_parameter_new_value(pName, CPL_TYPE_DOUBLE,
				"Ad this margin/border (in percent) to the "
				"resampled image/cube", context, FIELDMARGIN);
		cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "fieldmargin");
		cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
		cpl_parameterlist_append(pl, p);

		/* --hdrldemo_resample.edge-trim */
		snprintf(pName, sizeof(pName), "%s.%s", context, "edge-trim");
		p = cpl_parameter_new_range(pName, CPL_TYPE_INT,
				"Number or pixels to trim for each plane of the input frames. "
				"It should be smaller in the range [0,5]", context, EDGETRIM,
				0, 5);
		cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "edge-trim");
		cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
		cpl_parameterlist_append(pl, p);

		/* extraction params */
		eris_ifu_jitter_fill_extract_parameterlist(context, jitterMode, pl);

		/* flux calibration params */
		snprintf(pName, sizeof(pName), "%s.%s", context, "flux-calibrate");
		p = cpl_parameter_new_value(pName, CPL_TYPE_BOOL,
				"If True flux calibrate the extracted spectrum and data cube",
				context, CPL_FALSE);
		if (jitterMode == M_SCIENCE) {
			cpl_parameter_set_default_bool(p, CPL_FALSE);
		}
		cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "flux-calibrate");
		cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
		cpl_parameterlist_append(pl, p);


		/* DAR */
		eris_ifu_jitter_fill_dar_parameterlist(context, jitterMode, pl);


		snprintf(pName, sizeof(pName), "%s.%s", context, "fwhm-factor");
		p = cpl_parameter_new_value(pName, CPL_TYPE_DOUBLE,
				"Factor to find 2D-Gauss FWHM. The extraction box is: "
				"halfbox_x=halfbox_y=fwhm_factor*(fwhm_x+fwhm_y)*0.5.",
				context, 5.);
		cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "fwhm-factor");
		cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
		cpl_parameterlist_append(pl, p);


		snprintf(pName, sizeof(pName), "%s.%s", context, "max-cubes-centres-dist");
		p = cpl_parameter_new_range(pName, CPL_TYPE_INT,
				"Maximum distance between cube centers to build a mosaic. Mosaic creation "
				"requires a lot of RAM. Users may trim this value to fit RAM resources",
				context, 240, 20, 10000);
		cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "max-cubes-centres-dist");
		cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
		cpl_parameterlist_append(pl, p);


		snprintf(pName, sizeof(pName), "%s.%s", context, "cube.combine");
		p = cpl_parameter_new_value(pName, CPL_TYPE_BOOL,
				"With multi-pointing observations combine cubes into a mosaic",
				context, CPL_TRUE);
		cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI,
				"cube.combine");
		cpl_parameterlist_append(pl, p);

		snprintf(pName, sizeof(pName), "%s.%s", context, "derot_corr");
		p = cpl_parameter_new_value(pName, CPL_TYPE_STRING,
				"The first column offset to be applied on the science data for distortion correction. "
				"This effect is visible when the spectrum on one side of the cube wraps to the other side. "
				"Typical value is within +/- 2 pixels. "
				"A default correction based on altitude and rotation angle will be applied if this parameter "
				"set to auto; otherwise the user-speficied number of the shift would be used",
				context, "auto");
		cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "derot_corr");
		cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
		cpl_parameterlist_append(pl, p);

		if(strstr(context,"jitter") != NULL) {
			snprintf(pName, sizeof(pName), "%s.%s", context, "aj-method");
			p = cpl_parameter_new_range(pName, CPL_TYPE_INT,
					"Method used to estimate the sky in case of missing sky frames. "
					"0: no sky subtraction; "
					"1: sky is taken from the next in MJD-OBS object frame; "
					"2: sky is taken from the median of all input sky frames; "
					"3: sky is taken from the mean of all input sky frames; "
					"4: sky is taken from the median of user spec. rect. box; "
					"5: sky is taken from the median of 4 regions at FOV edges; "
					"6: sky is taken from the median of 8 regions at FOV edges; "
					"7: sky is taken from each cube slice based on ks-clip based mask computed on collapsed cube."
					"8: sky is taken from each cube slice based on 25% clip based mask computed on collapsed cube.",
					context, 7, 0, 8);
			cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "aj-method");
			cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
			cpl_parameterlist_append(pl, p);


			/* --sky box centre */
			snprintf(pName, sizeof(pName), "%s.%s", context, "sky-box-center");
			p = cpl_parameter_new_value(pName, CPL_TYPE_STRING,
					"The centre of the rectangular sky region (pixel). "
					"two integer numbers each in the range [1,64]. ",
					context, "32,32");
			cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "sky-box-center");
			cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
			cpl_parameterlist_append(pl, p);

			/* --sky box width */
			snprintf(pName, sizeof(pName), "%s.%s", context, "sky-box-width");
			p = cpl_parameter_new_value(pName, CPL_TYPE_STRING,
					"The rectangular (small) sky region x,y half widths [pixel]."
					"Two numbers each in the range [0,10].",
					context, "1,1");
			cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "sky-box-width");
			cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
			cpl_parameterlist_append(pl, p);


			/* --sky box edge margin */
			snprintf(pName, sizeof(pName), "%s.%s", context, "sky-box-edges-margin");
			p = cpl_parameter_new_value(pName, CPL_TYPE_STRING,
					"The (small)number of x, y pixels taken as marging to the "
					"FOV edges so that sky regions (aj-method=5,6) do not "
					"include pixels at the edges of the re-sampled image [pixel]."
					"Two numbers each in the range [0,5].",
					context, "1,1");
			cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "sky-box-edges-margin");
			cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
			cpl_parameterlist_append(pl, p);


			/* --sky est method */
			snprintf(pName, sizeof(pName), "%s.%s", context, "sky-est-method");
			p = cpl_parameter_new_range(pName, CPL_TYPE_INT,
					"If the user has no input sky frame and the sky need to be "
					"estimated from user defined pixels set by aj-method "
					"this parameter allow to use the median (0) or the mean (1) [0].",
					context, 0, 0, 1);
			cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "sky-est-method");
			cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
			cpl_parameterlist_append(pl, p);


			/* --sky est kappa */
			snprintf(pName, sizeof(pName), "%s.%s", context, "sky-est-kappa");
			p = cpl_parameter_new_range(pName, CPL_TYPE_DOUBLE,
					"If the user has no input sky frame and the sky need to be "
					"estimated from user defined pixels set by aj-method "
					"this parameter allow to use the kappa of the kappa sigma clip iteration [3].",
					context, 3., 0., 100.);
			cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "sky-est-kappa");
			cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
			cpl_parameterlist_append(pl, p);

			/* --sky est niter */
			snprintf(pName, sizeof(pName), "%s.%s", context, "sky-est-niter");
			p = cpl_parameter_new_range(pName, CPL_TYPE_INT,
					"If the user has no input sky frame and the sky need to be "
					"estimated from user defined pixels set by aj-method "
					"this parameter allow to set the number of the kappa sigma clip iterations [5].",
					context, 5, 0, 10);
			cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "sky-est-niter");
			cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
			cpl_parameterlist_append(pl, p);

			snprintf(pName, sizeof(pName), "%s.%s", context, "crea-phase3");
			p = cpl_parameter_new_value(pName, CPL_TYPE_BOOL,
					"If true crea phase3 compliant data products",
					context, CPL_FALSE);
			cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "crea-phase3");
			cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
			cpl_parameterlist_append(pl, p);


		}
		snprintf(pName, sizeof(pName), "%s.%s", context, "bpc_iter");
		p = cpl_parameter_new_value(pName, CPL_TYPE_INT,
				"No. of iterations for bad pixel correction",
				context, 1);
		cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "bpc_iter");
		cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
		cpl_parameterlist_append(pl, p);


		snprintf(pName, sizeof(pName), "%s.%s", context, "chop-nan");
		p = cpl_parameter_new_value(pName, CPL_TYPE_BOOL,
				"If true chop cube planes with more than 50% NAN pixels",
				context, CPL_FALSE);
		cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "chop-nan");
		cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
		cpl_parameterlist_append(pl, p);




		CHECK_ERROR_STATE();

	} CATCH {
		err = cpl_error_get_code();
	}
	eris_ifu_free_string(&context);
	eris_check_error_code("eris_ifu_jitter_fill_common_parameterlist");
	return err;
}


cpl_error_code eris_ifu_jitter_fetch_params(
		const char* context,
		const char* recipe_name,
		const cpl_parameterlist * parlist,
		struct stdParamStruct *stdParams,
		struct paramStruct *params)
{
	char* param_name = NULL;

	cpl_ensure_code(context, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(recipe_name, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(parlist, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(stdParams, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(params, CPL_ERROR_NULL_INPUT);


	TRY
	{
		eris_ifu_fetch_std_param(parlist, recipe_name, stdParams);

		param_name = cpl_sprintf("%s.sky_tweak", context);
		params->skyTweak = cpl_parameter_get_int(
				cpl_parameterlist_find_const(parlist, param_name));
		cpl_free(param_name);

		param_name = cpl_sprintf("%s.skip_sky_oh_align", context);
		params->skip_sky_oh_align = cpl_parameter_get_bool(
				cpl_parameterlist_find_const(parlist, param_name));
		cpl_free(param_name);

        param_name = cpl_sprintf("%s.skip_oh_align", context);
        params->skip_oh_align = cpl_parameter_get_bool(
            cpl_parameterlist_find_const(parlist, param_name));
        cpl_free(param_name);

		param_name = cpl_sprintf("%s.discard_subband", context);
		params->discard_subband = cpl_parameter_get_bool(
				cpl_parameterlist_find_const(parlist, param_name));
		cpl_free(param_name);

		param_name = cpl_sprintf("%s.stretch_sky", context);
		params->stretch_sky = cpl_parameter_get_bool(
				cpl_parameterlist_find_const(parlist, param_name));
		cpl_free(param_name);

		param_name = cpl_sprintf("%s.stretch_degree", context);
		params->stretch_degree = cpl_parameter_get_int(
				cpl_parameterlist_find_const(parlist, param_name));
		cpl_free(param_name);

		param_name = cpl_sprintf("%s.stretch_resampling", context);
		const char *resamplingInput = cpl_parameter_get_string(
				cpl_parameterlist_find_const(parlist, param_name));
		cpl_free(param_name);


		if (strncasecmp(resamplingInput, "linear", strlen("linear")) == 0) {
			params->stretch_resampling = LINEAR;
		} else if (strncasecmp(resamplingInput, "spline", strlen("spline")) == 0) {
			params->stretch_resampling = SPLINE;
		} else {
			BRK_WITH_ERROR_MSG(CPL_ERROR_ILLEGAL_INPUT,
					"Error reading recipe parameter, unknown stretch resampling method %s",
					resamplingInput);
		}


		param_name = cpl_sprintf("%s.tbsub", context);
		params->tbsub = cpl_parameter_get_bool(
				cpl_parameterlist_find_const(parlist, param_name));
		cpl_free(param_name);

		param_name = cpl_sprintf("%s.velocity_offset", context);
		params->velocityOffset = cpl_parameter_get_double(
				cpl_parameterlist_find_const(parlist, param_name));
		cpl_free(param_name);

		param_name = cpl_sprintf("%s.bpm_threshold", context);
		params->bpmThreshold = cpl_parameter_get_double(
				cpl_parameterlist_find_const(parlist, param_name));
		cpl_free(param_name);

		param_name = cpl_sprintf("%s.cube.first-col", context);
		params->firstCol = cpl_parameter_get_double(
				cpl_parameterlist_find_const(parlist, param_name));
		cpl_free(param_name);

		param_name = cpl_sprintf("%s.cube.fine-tune", context);
		params->fineTuneMode = cpl_parameter_get_int(
				cpl_parameterlist_find_const(parlist, param_name));
		cpl_free(param_name);

		param_name = cpl_sprintf("%s.cube.combine", context);
		params->combine = cpl_parameter_get_bool(
				cpl_parameterlist_find_const(parlist, param_name));
		cpl_free(param_name);

		param_name = cpl_sprintf("%s.cube.slitlet-detection", context);
		const char*  slitletMode = cpl_parameter_get_string(
				cpl_parameterlist_find_const(parlist, param_name));
		cpl_free(param_name);

		if (strcmp(slitletMode,"DIST") == 0) {
			params->slitletDetectionMode = DISTANCES;
		} else if (strcmp(slitletMode,"EDGE") == 0) {
			params->slitletDetectionMode = EDGES;
		} else if (strcmp(slitletMode,"GRID") == 0) {
			params->slitletDetectionMode = GRID;
		} else {
			params->slitletDetectionMode = UNSET;
		}

		param_name = cpl_sprintf("%s.cube.first-col", context);
		params->firstCol = cpl_parameter_get_double(
				cpl_parameterlist_find_const(parlist, param_name));
		cpl_free(param_name);

		param_name = cpl_sprintf("%s.extract-source", context);
		params->extractSource = cpl_parameter_get_bool(
				cpl_parameterlist_find_const(parlist, param_name));
		cpl_free(param_name);

		param_name = cpl_sprintf("%s.dar-corr", context);
		params->darCorrection = cpl_parameter_get_bool(
				cpl_parameterlist_find_const(parlist, param_name));
		cpl_free(param_name);

		param_name = cpl_sprintf("%s.dar-shift-method", context);
		params->darShiftMethod = cpl_parameter_get_int(
				cpl_parameterlist_find_const(parlist, param_name));
		cpl_free(param_name);

		param_name = cpl_sprintf("%s.dar-shift-radius", context);
		params-> darShiftWidth = cpl_parameter_get_double(
				cpl_parameterlist_find_const(parlist, param_name));
		cpl_free(param_name);

		param_name = cpl_sprintf("%s.dar-shift-length", context);
		params->darShiftLength = cpl_parameter_get_int(
				cpl_parameterlist_find_const(parlist, param_name));
		cpl_free(param_name);

		param_name = cpl_sprintf("%s.derot_corr", context);
		const char* derot_opt = cpl_parameter_get_string(
				cpl_parameterlist_find_const(parlist, param_name));
		cpl_free(param_name);
		if (!strcmp(derot_opt, "auto"))
			params->derot_corr = nan("");
		else
			sscanf(derot_opt, "%lf", &(params->derot_corr));

		param_name = cpl_sprintf("%s.bpc_iter", context);
		params->bpc_iter = cpl_parameter_get_int(
				cpl_parameterlist_find_const(parlist, param_name));
		cpl_free(param_name);

		CHECK_ERROR_STATE();

	} CATCH
	{
	}
	eris_check_error_code("eris_ifu_jitter_fetch_params");
	return cpl_error_get_code();
}


cpl_error_code eris_ifu_jitter_processSof(
		cpl_frameset* frames,
		struct stdParamStruct stdParams,
		struct paramStruct params,
		struct sofStruct *sof)
{
	cpl_image           *qualityMask = NULL;
	cpl_mask            *flatBpm = NULL;
	cpl_mask            *darkBpm = NULL;
	cpl_mask            *detlinBpm = NULL;
	cpl_mask            *masterBpm = NULL;
	cpl_vector          *lambdaRange = NULL;
	cpl_bivector        *ref_spectrum   = NULL;

	cpl_ensure_code(frames != NULL, CPL_ERROR_NULL_INPUT);
	//cpl_ensure_code(stdParams != NULL, CPL_ERROR_NULL_INPUT);
	//cpl_ensure_code(params != NULL, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(sof != NULL, CPL_ERROR_NULL_INPUT);

	TRY
	{
		const cpl_frame     *frame;
		deqQualityType      qualityType = unspecified;
		ifsBand             band = UNDEFINED_BAND;
		ifsPreopticsScale   scale = UNDEFINED_SCALE;
		ifsInstrument       instrument = UNSET_INSTRUMENT;
        cpl_error_code error0 = cpl_error_get_code();

		if (frames == NULL) {
			BRK_WITH_ERROR_MSG(CPL_ERROR_NULL_INPUT,
					"missing frameset");
		}
		if (cpl_frameset_is_empty(frames)) {
			BRK_WITH_ERROR_MSG(CPL_ERROR_NULL_INPUT,
					"SOF file is empty or missing");
		}
		CHECK_ERROR_STATE();

		eris_ifu_dfs_set_groups(frames);

		// get instrument settings
		eris_ifu_jitter_get_instrument_settings(
				frames, &instrument, &band, &scale);
		sof->band = band;
		sof->scale = scale;
		sof->instrument = instrument;

		sof->dqi = cpl_image_new(ERIS_IFU_DETECTOR_SIZE_X,
				ERIS_IFU_DETECTOR_SIZE_Y, CPL_TYPE_INT);

		// get "dark" BPM (which is not required)
		frame = cpl_frameset_find(frames, ERIS_IFU_CALIB_DARK_BP_MAP);
		if (frame != NULL) {
			darkBpm = cpl_mask_load(cpl_frame_get_filename(frame), 0, 0);
		} else {
			cpl_error_set(cpl_func, error0);
		}
		// get "flat" BPM (which is not required)
		frame = cpl_frameset_find(frames, ERIS_IFU_PRO_FLAT_BPM);
		if (frame != NULL) {
			flatBpm = cpl_mask_load(cpl_frame_get_filename(frame), 0, 0);
		} else {
			cpl_error_set(cpl_func, error0);
		}
		// get "detlin" BPM (which is not required)
		frame = cpl_frameset_find(frames, ERIS_IFU_CALIB_DETLIN_BP_MAP);
		if (frame != NULL) {
			detlinBpm = cpl_mask_load(cpl_frame_get_filename(frame), 0, 0);
		} else {
			cpl_error_set(cpl_func, error0);
		}

		// get master dark file  (which is not required)
		frame = cpl_frameset_find(frames, ERIS_IFU_CALIB_DARK);
		if (frame != NULL) {
			// agudo: why needed?
			qualityType = unspecified;
			ifsBand darkBand = UNDEFINED_BAND;
			ifsPreopticsScale darkScale = UNDEFINED_SCALE;
			sof->masterDark = eris_ifu_load_cal_image_frame(frame,
					darkBand, darkScale, &qualityMask, &qualityType);
			if (darkBpm == NULL) {
				darkBpm = eris_ifu_quality2bp_mask(qualityMask, qualityType);
				if (qualityType == flag16bit || qualityType == flag32bit) {
					cpl_image_add(sof->dqi, qualityMask);
				}
			}
		} else {
			sof->masterDark = NULL;
			cpl_error_set(cpl_func, error0);
		}

		// get flat field calibration file (which is not required)
		frame = cpl_frameset_find(frames, ERIS_IFU_CALIB_FLATFIELD);
		if (frame != NULL) {
			// agudo: why needed?
			qualityType = unspecified;
			sof->masterFlat = eris_ifu_load_cal_image_frame(frame,
					band, scale, &qualityMask, &qualityType);
			if (flatBpm == NULL) {
				flatBpm = eris_ifu_quality2bp_mask(qualityMask, qualityType);
				if (qualityType == flag16bit || qualityType == flag32bit) {
					cpl_image_add(sof->dqi, qualityMask);
				}
			}
		} else {
			sof->masterFlat = NULL;
			cpl_error_set(cpl_func, error0);
		}

		//compute master BPM
		masterBpm = cpl_mask_new(ERIS_IFU_DETECTOR_SIZE_X,
				ERIS_IFU_DETECTOR_SIZE_Y);
		if (darkBpm != NULL) {
			cpl_mask_or(masterBpm, darkBpm);
		}
		if (flatBpm != NULL) {
			cpl_mask_or(masterBpm, flatBpm);
		}
		if (detlinBpm != NULL) {
			cpl_mask_or(masterBpm, detlinBpm);
		}
		sof->badPixelMask = cpl_mask_duplicate(masterBpm);
		cpl_mask_delete(masterBpm);
		masterBpm = NULL;
		cpl_msg_debug(__func__,"Number of bad pixels in master bad pixel map %lld",
				cpl_mask_count(sof->badPixelMask));

		// get distortion polynomials
		frame = cpl_frameset_find(frames, ERIS_IFU_CALIB_DISTORTION);
		CHECK_ERROR_STATE();
		if (frame == NULL) {
			BRK_WITH_ERROR_MSG(CPL_ERROR_NULL_INPUT,
					"missing \"%s\" tag in the SOF, distortion polynomials",
					ERIS_IFU_CALIB_DISTORTION);
		}
		if (cpl_fits_count_extensions(cpl_frame_get_filename(frame)) > 2) {
			eris_ifu_load_distortion_polynomials(
					cpl_frame_get_filename(frame),
					&sof->distortion, &sof->borders);

			// // apply derotator correction
			// for (int i = 0; i < SLITLET_CNT; i++) {
			//     if (sof->distortion[i] != NULL) {
			//         BRK_IF_ERROR(
			//             cpl_polynomial_shift_1d(sof->distortion[i], 0, params.derot_corr));
			//     }
			// }

			sof->poly_u = NULL;
			sof->poly_v = NULL;
			sof->distances = NULL;
			sof->positions = NULL;
		} else {
			eris_ifu_load_distortion_polynomials_old(
					cpl_frame_get_filename(frame), &sof->poly_u, &sof->poly_v);
			//        cpl_polynomial_dump(sof->poly_u,stdout);
			//        cpl_polynomial_dump(sof->poly_v,stdout);
			sof->distortion = NULL;
			sof->borders = NULL;

			// get slitlet distances
			frame = cpl_frameset_find(frames, ERIS_IFU_CALIB_DISTANCES);
			CHECK_ERROR_STATE();
			if (frame == NULL) {
				if (params.slitletDetectionMode == DISTANCES) {
					BRK_WITH_ERROR_MSG(CPL_ERROR_NULL_INPUT,
							"missing \"%s\" tag in the SOF, slitlet distances",
							ERIS_IFU_CALIB_DISTANCES);
				}
			}
			if (params.slitletDetectionMode != DISTANCES) {
				cpl_msg_warning(cpl_func,
						"SOF entry with tag %s will be ignored as %s recipe parameter"
						" is not set to \"DIST\"", ERIS_IFU_CALIB_DISTANCES,
						"cube.slitlet-detection");
			}
			sof->distances = eris_ifu_load_distances(
					cpl_frame_get_filename(frame));
			//        cpl_vector_dump(sof->distances, stdout);


			// get slitlet edges
			frame = cpl_frameset_find(frames, ERIS_IFU_PRO_DIST_SLIT_POS);
			CHECK_ERROR_STATE();
			if (frame == NULL) {
				if (params.slitletDetectionMode == EDGES) {
					BRK_WITH_ERROR_MSG(CPL_ERROR_NULL_INPUT,
							"missing \"%s\" tag in the SOF, slitlet positions",
							ERIS_IFU_PRO_DIST_SLIT_POS);
				}
			}
			if (params.slitletDetectionMode != EDGES) {
				cpl_msg_warning(cpl_func,
						"SOF entry with tag %s will be ignored as %s recipe parameter"
						" is not set to \"EDGE\"", ERIS_IFU_PRO_DIST_SLIT_POS,
						"cube.slitlet-detection");
			}
			sof->positions = eris_ifu_load_slit_positions(
					cpl_frame_get_filename(frame));
			if (cpl_msg_get_level() == CPL_MSG_DEBUG) {
				cpl_bivector_dump(sof->positions, stdout);
			}
		}

		// get wavelength calibration map
		frame = cpl_frameset_find(frames, ERIS_IFU_CALIB_WAVE_MAP);
		CHECK_ERROR_STATE();
		if (frame == NULL) {
			BRK_WITH_ERROR_MSG(CPL_ERROR_NULL_INPUT,
					"missing \"%s\" tag in the SOF, wavelength map",
					ERIS_IFU_CALIB_WAVE_MAP);
		}
		sof->waveMap = cpl_image_load(
				cpl_frame_get_filename(frame),
				CPL_TYPE_UNSPECIFIED, 0, 0);

		frame = cpl_frameset_find(frames, ERIS_IFU_CALIB_OH_SPEC);
		CHECK_ERROR_STATE();
		if (frame != NULL) {
			const char *ref_filename   = NULL;
			cpl_size    ext_cnt;

			sof->oh_ref_frame = cpl_frame_duplicate(frame);
			lambdaRange = cpl_vector_new(2);
			cpl_vector_set(lambdaRange, 0, cpl_image_get_min(sof->waveMap));
			cpl_vector_set(lambdaRange, 1, cpl_image_get_max(sof->waveMap));
			BRK_IF_NULL(
					ref_filename = cpl_frame_get_filename(frame));
			ext_cnt = cpl_fits_count_extensions(ref_filename);
			if (ext_cnt != 0) {
				const char *ext_name = NULL;

				switch(sof->band) {
				case J_LOW: case J_SHORT: case J_MIDDLE: case J_LONG: case J_SPIFFI:
					ext_name = OH_SPECTRA_J;
					break;
				case H_LOW: case H_SHORT: case H_MIDDLE: case H_LONG: case H_SPIFFI:
					ext_name = OH_SPECTRA_H;
					break;
				case K_LOW: case K_SHORT: case K_MIDDLE: case K_LONG: case K_SPIFFI:
					ext_name = OH_SPECTRA_K;
					break;
				case HK_SPIFFI:
					ext_name = OH_SPECTRA_HK;
					break;
				default:
					BRK_WITH_ERROR_MSG(CPL_ERROR_UNSUPPORTED_MODE,
							"unidentified SPIFFIER band selection: %s",
							eris_ifu_get_bandString(sof->band));
				}
				ext_cnt = cpl_fits_find_extension(ref_filename, ext_name);
				CHECK_ERROR_STATE();
			}
			ref_spectrum = eris_ifu_lcorr_read_OH_reference_spectrum(
					ref_filename, ext_cnt);
			sof->oh_ref_peaks =
					eris_ifu_lcorr_get_peak_lambdas(ref_spectrum, 0.2, lambdaRange);

		} else {
			sof->oh_ref_frame = NULL;
			sof->oh_ref_peaks = NULL;
		}

		// get all object/sky exposures
		eris_ifu_jitter_get_objsky_exposures( frames, stdParams, sof);

		// get all object/sky pairs
		eris_ifu_jitter_get_objsky_pairs(sof);

	} CATCH
	{
		//        CATCH_MSGS();
	}

	cpl_image_delete(qualityMask);
	cpl_mask_delete(flatBpm);
	cpl_mask_delete(darkBpm);
	cpl_mask_delete(detlinBpm);
	cpl_mask_delete(masterBpm);
	cpl_vector_delete(lambdaRange);
	cpl_bivector_delete(ref_spectrum);
	eris_check_error_code("eris_ifu_jitter_processSof");
	return cpl_error_get_code();

}

cpl_error_code eris_ifu_jitter_get_instrument_settings(
		cpl_frameset *frames,
		ifsInstrument *instrument,
		ifsBand *band,
		ifsPreopticsScale *scale)
{


	cpl_ensure_code(frames != NULL, CPL_ERROR_NULL_INPUT);


	cpl_error_code  retVal  = CPL_ERROR_NONE;
	cpl_frameset_iterator *iter = NULL;
	cpl_propertylist *header = NULL;
	cpl_frame *frame = NULL;
	cpl_error_code iterError;

	TRY
	{
		// loop through frame set to get first RAW image to preset instrument,
		//  band and pre-optics variables
		iter = cpl_frameset_iterator_new(frames);
		bool found = FALSE;
		while (!found) {
			frame = cpl_frameset_iterator_get(iter);
			if (frame != NULL) {
				if (cpl_frame_get_group(frame) == CPL_FRAME_GROUP_RAW) {
					*instrument = eris_ifu_get_instrument_frame(frame);
					header = cpl_propertylist_load(
							cpl_frame_get_filename(frame), 0);
					*band = eris_ifu_get_band(header);
					*scale = eris_ifu_get_preopticsScale(header);
					cpl_propertylist_delete(header);
					found = TRUE;
				} else {
					iterError = cpl_frameset_iterator_advance(iter, 1);
					if (iterError == CPL_ERROR_ACCESS_OUT_OF_RANGE) {
						SET_ERROR_MSG(CPL_ERROR_DATA_NOT_FOUND,
								"Found no raw data frames in SOF");
						break;
					}
				}
			} else {
				SET_ERROR_MSG(CPL_ERROR_DATA_NOT_FOUND,
						"Found no raw data frames in SOF");
				break;
			}
		}
		CHECK_ERROR_STATE();
	}
	CATCH
	{
		retVal = cpl_error_get_code();
	}
	cpl_frameset_iterator_delete(iter);
	eris_check_error_code("eris_ifu_jitter_get_instrument_settings");
	return retVal;
}


cpl_error_code eris_ifu_jitter_get_objsky_exposures(
		cpl_frameset *frames,
		struct stdParamStruct params,
		struct sofStruct *sof)
{

	cpl_ensure_code(frames != NULL, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(params.crh_detection != NULL, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(sof != NULL, CPL_ERROR_NULL_INPUT);


	cpl_error_code  retVal  = CPL_ERROR_NONE;
	cpl_frameset_iterator *iter = NULL;
	cpl_frame *frame = NULL;
	cpl_error_code iterError;
	const char* tag = NULL;
	cpl_size frameSetSize;
	sofModes mode = NODATA;
	TRY
	{
		frameSetSize = cpl_frameset_get_size(frames);
		sof->exposureTableSize = (int) frameSetSize;
		sof->exposureTable = cpl_calloc(sof->exposureTableSize,
				sizeof(struct exposureEntry));

		sof->exposureTableCnt = 0;
		sof->objectCnt = 0;
		iter = cpl_frameset_iterator_new(frames);
		frame = cpl_frameset_iterator_get(iter);

		sof->mode = NODATA;
		while (frame != NULL) {
			bool exposureFound = false;
			int ix = sof->exposureTableCnt;
			struct exposureEntry *expEntry = &sof->exposureTable[ix];

			mode = NODATA;
			tag = cpl_frame_get_tag(frame);
			if (strcmp(tag, ERIS_IFU_RAW_OBJ) == 0 ||
					strcmp(tag, ERIS_IFU_RAW_STD) == 0 ||
					strcmp(tag, ERIS_IFU_RAW_PUPIL_LAMP) == 0 ||
					strcmp(tag, ERIS_IFU_RAW_PUPIL_SKY) == 0 ||
					strcmp(tag, ERIS_IFU_RAW_STD_FLUX) == 0 ||
					strcmp(tag, ERIS_IFU_RAW_PSF) == 0) {
				expEntry->isObject = true;
				sof->objectCnt++;
				exposureFound = true;
				if (strcmp(tag, ERIS_IFU_RAW_OBJ) == 0) {
					mode = SCIENCE;
				} else if (strcmp(tag, ERIS_IFU_RAW_STD) == 0) {
					mode = STD;
				} else if (strcmp(tag, ERIS_IFU_RAW_STD_FLUX) == 0) {
					mode = STD_FLUX;
				} else if (strcmp(tag, ERIS_IFU_RAW_PSF) == 0) {
					mode = PSF;
				} else if (strcmp(tag, ERIS_IFU_RAW_PUPIL_LAMP) == 0||
                                           strcmp(tag, ERIS_IFU_RAW_PUPIL_SKY) == 0) {
					mode = MODE_PUPIL;
				}
			}
			if (strcmp(tag, ERIS_IFU_RAW_SKY) == 0 ||
					strcmp(tag, ERIS_IFU_RAW_OBJ_SKY) == 0 ||
					strcmp(tag, ERIS_IFU_RAW_STD_SKY) ==0 ||
					strcmp(tag, ERIS_IFU_RAW_STD_FLUX_SKY) ==0 ||
					strcmp(tag, ERIS_IFU_RAW_PSF_SKY) == 0) {
				expEntry->isObject = false;
				exposureFound = true;
				if (strcmp(tag, ERIS_IFU_RAW_SKY) == 0) {
					mode = IGNORE_SOF_MODE;
				} else if (strcmp(tag, ERIS_IFU_RAW_OBJ_SKY) == 0) {
					mode = SCIENCE;
				} else if (strcmp(tag, ERIS_IFU_RAW_STD_SKY) == 0) {
					mode = STD;
				} else if (strcmp(tag, ERIS_IFU_RAW_STD_FLUX_SKY) == 0) {
					mode = STD_FLUX;
				} else if (strcmp(tag, ERIS_IFU_RAW_PSF_SKY) == 0) {
					mode = PSF;
				} else if (strcmp(tag, ERIS_IFU_RAW_PUPIL_SKY) == 0) {
					mode = MODE_PUPIL;
				}
			}
			if (exposureFound) {
				if (sof->mode == NODATA) {
					if (mode != NODATA) {
						if (mode != IGNORE_SOF_MODE) {
							sof->mode = mode;
						}
					} else {
						BRK_WITH_ERROR_MSG(CPL_ERROR_ILLEGAL_INPUT,
								"Cannot interpret tag %s", tag);
					}
				} else {
					if (mode != IGNORE_SOF_MODE) {
						if (mode != sof->mode) {
							BRK_WITH_ERROR_MSG(CPL_ERROR_ILLEGAL_INPUT,
									"Do not mix different input raw data types like"
									" SCIENCE, PSF, STD STAR ...");
						}
					}
				}
				expEntry->dqiImage = cpl_image_new(ERIS_IFU_DETECTOR_SIZE_X,
						ERIS_IFU_DETECTOR_SIZE_Y, CPL_TYPE_INT);
				const char *filename = cpl_frame_get_filename(frame);
				expEntry->frame = frame;
				expEntry->rawImage = eris_ifu_load_exposure_frame(
						frame, params.rawImageCorrectionMask, expEntry->dqiImage);
				expEntry->hdr = cpl_propertylist_load( filename, 0);
				expEntry->dit = eris_ifu_get_dit(expEntry->hdr);
				expEntry->obsDate = cpl_propertylist_get_double(
						expEntry->hdr, FHDR_MJD_OBS);
				expEntry->centeredObsDate =
						expEntry->obsDate + (expEntry->dit /2.)/86400.;
				if(cpl_propertylist_has(expEntry->hdr,"ESO TEL ALT")) {
					expEntry->alt = cpl_propertylist_get_double(
							expEntry->hdr, "ESO TEL ALT");
				}
				if(cpl_propertylist_has(expEntry->hdr,"ESO ADA ABSROT START")) {
					expEntry->rot = cpl_propertylist_get_double(
							expEntry->hdr, "ESO ADA ABSROT START");
				}
				expEntry->derot_corr = eris_ifu_auto_derot_corr(expEntry->alt, expEntry->rot);
				expEntry->skyIndex = -1;
				expEntry->cube = NULL;
				expEntry->cubeHdr = NULL;
				expEntry->darkSubtrImage  = NULL;
				expEntry->skySubtrImage  = NULL;
				expEntry->badPixelMask = cpl_mask_duplicate(hdrl_image_get_mask(expEntry->rawImage));
				cpl_msg_debug(__func__,"Number of bad pixels in master bad pixel map %lld",
						cpl_mask_count(expEntry->badPixelMask));
				cpl_mask_or(expEntry->badPixelMask, sof->badPixelMask);
				hdrl_image_reject_from_mask(expEntry->rawImage,
						expEntry->badPixelMask);
				cpl_image_add(expEntry->dqiImage, sof->dqi);

				if (params.productDepth >= PD_DEBUG) {
					char *filename2 = NULL;
					filename2 = cpl_sprintf("%s_%3.3d",
							ERIS_IFU_PRO_JITTER_DBG_RAW_FN, ix);
					eris_ifu_save_hdrl_image_dbg(expEntry->rawImage,
							filename2, 1, NULL);
					eris_ifu_free_string(&filename2);
					filename2 = cpl_sprintf("%s_%3.3d.fits",
							ERIS_IFU_PRO_JITTER_DBG_RAW_FN, ix);
					cpl_image_save(expEntry->dqiImage, filename2,
							CPL_TYPE_INT, NULL, CPL_IO_EXTEND);
					eris_ifu_free_string(&filename2);
				}

				//cpl_image_delete(expEntry->dqiImage);
				sof->exposureTableCnt++;

			}
			iterError = cpl_frameset_iterator_advance(iter, 1);
			if (iterError == CPL_ERROR_ACCESS_OUT_OF_RANGE) {
				break;
			}
			frame = cpl_frameset_iterator_get(iter);
		}

		if (sof->objectCnt == 0) {
			BRK_WITH_ERROR_MSG(CPL_ERROR_ILLEGAL_INPUT,
					"No object frame tag found in SOF file, either"
					" OBJ, STD, STD_FLUX, PSF_CALIBRATOR, PUPIL_LAMP is missing");
		} else {
			cpl_msg_info(cpl_func, "Number of object frames to be processed: %d", sof->objectCnt);
		}
	}
	CATCH
	{
		retVal = cpl_error_get_code();
	}
	cpl_frameset_iterator_delete(iter);
	eris_check_error_code("eris_ifu_jitter_get_objsky_exposures");
	return retVal;
}

cpl_error_code eris_ifu_jitter_get_objsky_pairs(struct sofStruct *sof)
{
	cpl_error_code  retVal  = CPL_ERROR_NONE;
	int skyCnt;
	int skyIdx;
	int *skyIndex = NULL;
	double *skyDate = NULL;
	double obsDate;
	double minDiff;

	cpl_ensure_code(sof, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code((sof->objectCnt <= sof->exposureTableCnt),
			CPL_ERROR_ILLEGAL_INPUT);
	TRY
	{
		// when sof->objectCnt equals sof->exposureTableCnt then there are
		// no sky exposures given hence no pairing expEntry.skyIndex = -1
		if  (sof->objectCnt != sof->exposureTableCnt) {
			// allocate array to store sky dates and correspinding indexes
			skyCnt = sof->exposureTableCnt - sof->objectCnt;
			skyIndex = cpl_calloc(skyCnt, sizeof(int));
			skyDate = cpl_calloc(skyCnt, sizeof(double));

			// get all sky dates
			skyIdx = 0;
			for (int ix=0; ix < sof->exposureTableCnt; ix++) {
				struct exposureEntry *expEntry = &sof->exposureTable[ix];
				if (expEntry->isObject == false) {
					skyIndex[skyIdx] = ix;
					skyDate[skyIdx] = expEntry->centeredObsDate;
					skyIdx++;
				}
			}

			// find closest sky
			for (int ix=0; ix < sof->exposureTableCnt; ix++) {
				struct exposureEntry *expEntry = &sof->exposureTable[ix];
				if (expEntry->isObject == true) {
					obsDate = expEntry->centeredObsDate;
					skyIdx = 0;
					minDiff = fabs(obsDate - skyDate[0]);
					for (int sx = 1; sx<skyCnt; sx++) {
						double diff = fabs(obsDate - skyDate[sx]);
						if (diff < minDiff) {
							skyIdx = sx;
							minDiff = diff;
						}
					}
					expEntry->skyIndex = skyIndex[skyIdx];

				}
			}
		}
	}
	CATCH
	{
		retVal = cpl_error_get_code();
	}
	eris_ifu_free_int_array(&skyIndex);
	eris_ifu_free_double_array(&skyDate);
	eris_check_error_code("eris_ifu_jitter_get_objsky_pairs");
	return retVal;
}

void eris_free_exposureEntry(struct exposureEntry* ee) {

	if(ee != NULL) {
		if(ee->frame) cpl_frame_delete(ee->frame);
		if(ee->hdr) cpl_propertylist_delete(ee->hdr);
		if(ee->rawImage) hdrl_image_delete(ee->rawImage);
		if(ee->dqiImage) cpl_image_delete(ee->dqiImage);
		if(ee->cube) hdrl_imagelist_delete(ee->cube);
		if(ee->cubeBpm) cpl_imagelist_delete(ee->cubeBpm);
		if(ee->cubeHdr) cpl_propertylist_delete(ee->cubeHdr);
		if(ee->badPixelMask) cpl_mask_delete(ee->badPixelMask);
		if(ee->darkSubtrImage) hdrl_image_delete(ee->darkSubtrImage);
		if(ee->skySubtrImage) hdrl_image_delete(ee->skySubtrImage);
	}
	eris_check_error_code("eris_free_exposureEntry");
	return;
}
