/* $Id: eris_resample.c,v 1.9 2013-03-26 17:00:44 jtaylor Exp $
 *
 * This file is part of the ERIS Pipeline
 * Copyright (C) 2002,2003 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */


#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/



#include <string.h>
#include <hdrl.h>
#include <eris_ifu_resample.h>
#include <eris_ifu_dfs.h>
#include <eris_ifu_utils.h>
#include <eris_utils.h>
#include <eris_pfits.h>
#include <eris_ifu_sdp.h>
/*----------------------------------------------------------------------------*/
/**
 * @defgroup eris_ifu_resample  Cube resampling related functions
 *
 * TBD
 */
/*----------------------------------------------------------------------------*/
#define MAX_COMBINE_CUBE_PLANE_DIAGONAL 1000
/**@{*/


/**
 * @brief This function does the actual re-sampling.
 * @param append input table to append
 * @param storage output table
 * @return cpl_error_code
 */
static cpl_error_code
eris_ifu_resample_update_table(const cpl_table* append, cpl_table** storage) {

  cpl_ensure_code(append, CPL_ERROR_NULL_INPUT);

  if(*storage == NULL) {
      *storage = cpl_table_new(0);
      cpl_table_copy_structure(*storage, append);
      cpl_table_insert(*storage, append, 0);
  } else {
      cpl_size nrow = cpl_table_get_nrow(*storage);
      cpl_table_insert(*storage, append, nrow);
  }
  eris_check_error_code("eris_ifu_resample_update_table");
  return cpl_error_get_code();

}


/**
 * @brief save table with ra, dec,lambda information

 * @param par input parameters
 * @param parlist input parameters
 * @param frameset input frameset
 * @param tab_final input table
 * @return cpl_error_code
 * @note: TODO to be removed, only for debugging purposes
 * */
static cpl_error_code
eris_ifu_resample_tablesave (const char* recipe_name, const cpl_parameter *par,
        const cpl_parameterlist *parlist, cpl_frameset *frameset,
        cpl_table *tab_final, const char* pipefile_prefix)
{
  /* Save the final table if required */
  char* param_name = cpl_sprintf("eris.%s.save-table", recipe_name);
  par = cpl_parameterlist_find_const (parlist, param_name);
  cpl_free(param_name);

  const cpl_boolean save_table = cpl_parameter_get_bool (par);

  if (save_table)
    {
      cpl_msg_info (cpl_func, "Saving the table before resampling");
      cpl_propertylist *applist = cpl_propertylist_new ();
      cpl_propertylist_update_string (applist, CPL_DFS_PRO_CATG,
                      "RESAMPLE_TABLE");
      char* fname = cpl_sprintf("%s_resample_table.fits", pipefile_prefix);
      cpl_dfs_save_table (frameset, NULL, parlist, frameset, NULL, tab_final,
              NULL, recipe_name, applist, NULL, PACKAGE "/" PACKAGE_VERSION,
              fname);
      cpl_free(fname);
      cpl_propertylist_delete (applist);
    }
  eris_check_error_code("eris_ifu_resample_tablesave");
  return cpl_error_get_code();
}



/*----------------------------------------------------------------------------*/
/**
 @brief   get int parameter value if changed by the user
 @param   parlist list of input recipe parameters
 @param   pname   recipe parameter name
 @param   pvalue  recipe parameter value
 @return  cpl_error_code
 */
/*----------------------------------------------------------------------------*/

/* set propertylist (int) value */
static cpl_error_code
eris_ifu_resample_parameters_get_int(const cpl_parameterlist* parlist,
                    const char* pname, int *pvalue)
{
  const cpl_parameter* p = cpl_parameterlist_find_const(parlist, pname);
  cpl_boolean p_has_changed = eris_param_has_changed(p);

  if  ( cpl_parameter_get_default_flag(p) && p_has_changed != CPL_FALSE) {
      *pvalue = cpl_parameter_get_int(p);
  } else {
      *pvalue = cpl_parameter_get_default_int(p);
  }
  eris_check_error_code("eris_ifu_resample_parameters_get_int");
  return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
 @brief   set double variable from parameter value if changed by the user
 @param   parlist list of input recipe parameters
 @param   pname   recipe parameter name
 @param   table_value  table value (value to be used if recipe parameter has
                       default value)
 @param   value  variable value
 @return  cpl_error_code
 */
/*----------------------------------------------------------------------------*/


static cpl_error_code
set_double_from_parameters(const cpl_parameterlist* parlist,
               const char* pname,
               const double table_value,
               double *value) {

  double temp_value;
  eris_parameters_get_double (parlist, pname, &temp_value);
  if (temp_value != -1)
    {
      *value = temp_value;
    }
  else
    {
      *value = table_value;
    }
  eris_check_error_code("set_double_from_parameters");
  return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
 @brief   set int variable from parameter value if changed by the user
 @param   parlist list of input recipe parameters
 @param   pname   recipe parameter name
 @param   table_value  table value (value to be used if recipe parameter has
                       default value)
 @param   value  variable value
 @return  cpl_error_code
 */
/*----------------------------------------------------------------------------*/


static cpl_error_code
set_int_from_parameters(const cpl_parameterlist* parlist,
               const char* pname,
               const int table_value,
               int *value) {

  int temp_value;
  eris_ifu_resample_parameters_get_int (parlist, pname, &temp_value);
  if (temp_value != -1)
    {
      *value = temp_value;
    }
  else
    {
      *value = table_value;
    }
  eris_check_error_code("set_int_from_parameters");
  return cpl_error_get_code();
}



/*----------------------------------------------------------------------------*/
/**
 @brief   set values into hdrl_parameter* structure
 @param   parlist input recipe parameters
 @return  cpl_error_code
 */
/*----------------------------------------------------------------------------*/

static hdrl_parameter *
eris_ifu_resample_set_method (const cpl_parameterlist *parlist,
        const char* context)
{
  hdrl_parameter      *aParams_method = NULL;
  const cpl_parameter *p = NULL;
  cpl_boolean p_has_changed = 0;
  int loop_distance = 0;
  double critical_radius_renka = 0.;
  int kernel_size_lanczos = 0;
  double pix_frac_drizzle_x = 0.;
  double pix_frac_drizzle_y= 0.;
  double pix_frac_drizzle_l= 0.;
  cpl_boolean use_errorweights = FALSE;

  const cpl_parameter *par = NULL;
  char* param_name = NULL;
  /* Apply additional weights based on the error if required */

  param_name = cpl_sprintf("%s.method.use-errorweights",context);
  par = cpl_parameterlist_find_const (parlist, param_name);
  use_errorweights = cpl_parameter_get_bool (par);
  cpl_free(param_name);

  /** loop distance, used for all weighted resampling schemes */
  param_name = cpl_sprintf("%s.method.loop-distance",context);
  p = cpl_parameterlist_find_const (parlist, param_name);

  p_has_changed = eris_param_has_changed (p);
  if (cpl_parameter_get_default_flag (p) && p_has_changed != CPL_FALSE)
    {
      loop_distance = cpl_parameter_get_int (p);
    }
  else
    {
      loop_distance = LOOP_DISTANCE;
    }
  cpl_free(param_name);

  /** critical radius of the Renka-weighted method */
  param_name = cpl_sprintf("%s.method.renka.critical-radius",context);
  set_double_from_parameters(parlist, param_name,
                 RENKA_CRITICAL_RADIUS, &critical_radius_renka);
  cpl_free(param_name);

  /** Kernel size of the Lanczos-weighted method */
  param_name = cpl_sprintf("%s.method.lanczos.kernel-size",context);
  set_int_from_parameters(parlist, param_name,
                 LANCZOS_KERNEL_SIZE, &kernel_size_lanczos);
  cpl_free(param_name);

  /** the pixfrac parameter of the drizzle method: down-scaling factor  *
   *  of input pixel size before computing drizzling weights; different *
   *  values for x-, y-, and lambda directions are possible
   *
   *              */
  param_name = cpl_sprintf("%s.method.drizzle.downscale-x",context);
  set_double_from_parameters(parlist, param_name, DRIZZLE_DOWN_SCALING_FACTOR_X,
          &pix_frac_drizzle_x);
  cpl_free(param_name);

  param_name = cpl_sprintf("%s.method.drizzle.downscale-y",context);
  set_double_from_parameters(parlist, param_name, DRIZZLE_DOWN_SCALING_FACTOR_Y,
          &pix_frac_drizzle_y);
  cpl_free(param_name);

  param_name = cpl_sprintf("%s.method.drizzle.downscale-z",context);
  set_double_from_parameters(parlist, param_name, DRIZZLE_DOWN_SCALING_FACTOR_Z,
          &pix_frac_drizzle_l);
  cpl_free(param_name);

  /* Create the right re-sampling parameter */
  param_name = cpl_sprintf("%s.method",context);
  par = cpl_parameterlist_find_const (parlist, param_name);
  const char *method = cpl_parameter_get_string (par);

  if (strcmp (method, "NEAREST") == 0)
    {
      aParams_method = hdrl_resample_parameter_create_nearest();
    }
  else if (strcmp (method, "RENKA") == 0)
    {
      aParams_method = hdrl_resample_parameter_create_renka(loop_distance,
                                use_errorweights,
                                critical_radius_renka);
    }
  else if (strcmp (method, "LINEAR") == 0)
    {
      aParams_method = hdrl_resample_parameter_create_linear(loop_distance,
                                 use_errorweights);
    }
  else if (strcmp (method, "QUADRATIC") == 0)
    {
      aParams_method = hdrl_resample_parameter_create_quadratic(loop_distance,
                                use_errorweights);
    }
  else if (strcmp (method, "DRIZZLE") == 0)
    {
      aParams_method = hdrl_resample_parameter_create_drizzle(loop_distance,
                                  use_errorweights,
                                  pix_frac_drizzle_x,
                                  pix_frac_drizzle_y,
                                  pix_frac_drizzle_l);
    }
  else if (strcmp (method, "LANCZOS") == 0)
    {
      aParams_method = hdrl_resample_parameter_create_lanczos(loop_distance,
                                  use_errorweights,
                                  kernel_size_lanczos);
    }
  else
    {
      aParams_method = hdrl_resample_parameter_create_lanczos(loop_distance,
                                  use_errorweights,
                                  kernel_size_lanczos);
      cpl_msg_warning (cpl_func,
               "%s is an unsupported method! Default to LANCZOS",
               method);
    }
  cpl_free(param_name);
  //eris_ifu_resample_print_method_params(aParams_method);
  eris_check_error_code("eris_ifu_resample_set_method");
  return aParams_method;
}

/*----------------------------------------------------------------------------*/
/**
 @brief   get WCS and put into output cdij, crpixj, crvalj, cdeltj (j=1,2,3)
 variables
 @param   wcs input wcs
 @param   cdelt1
 @param   cdelt2
 @param   cdelt3
 @return  cpl_error_code
 */
/*----------------------------------------------------------------------------*/

static cpl_error_code
eris_ifu_resample_get_cdelt123(cpl_wcs *wcs, double *cdelt1, double *cdelt2,
                   double *cdelt3)
{
  cpl_ensure_code(wcs, CPL_ERROR_NULL_INPUT);
  
  
  double cd1_1 = 0;
  double cd1_2 = 0;
  double cd2_1 = 0;
  double cd2_2 = 0;
  double cd3_3 = 0;

  double dx=0;
  double dy=0;

  const cpl_matrix *cd = cpl_wcs_get_cd(wcs);
  cd1_1 = cpl_matrix_get(cd, 0, 0); 
  cd1_2 = cpl_matrix_get(cd, 1, 0);
  cd2_1 = cpl_matrix_get(cd, 0, 1);
  cd2_2 = cpl_matrix_get(cd, 1, 1);

  dx = sqrt(pow(cd1_1,2) + pow(cd2_1,2));
  dy = sqrt(pow(cd1_2,2)+ pow(cd2_2,2));

//   double det = cd1_1 * cd2_2 - cd1_2 * cd2_1; 
//   if (det < 0)
//     dx *= -1.0; 

//   *cdelt1 = fabs(cpl_matrix_get(cd, 0, 0)); /* CD1_1 */
//   *cdelt2 = fabs(cpl_matrix_get(cd, 1, 1)); /* CD2_2 */
  *cdelt1 = dx;
  *cdelt2 = dy;

  /* Changes for Cube and image */
  cpl_size cdncol = cpl_matrix_get_ncol(cd);
  if (cdncol == 3) {
      cd3_3= fabs(cpl_matrix_get(cd, 2, 2)); /* CD3_3 */
  }
  else {
      cd3_3 = 1.; /* CD3_3 */
  }

  *cdelt3 = cd3_3; 

  /* Check keys (for debug) */
  cpl_msg_debug(cpl_func, "cdelt1: %g, cdelt2: %g, cdelt3: %g",
        *cdelt1, *cdelt2, *cdelt3);
  eris_check_error_code("eris_ifu_resample_get_cdelt123");
  return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
 @brief   set values into hdrl_parameter* structure
 @param   parlist input recipe parameters
 @param   muse_table input table with x,y,lambda values
 @param   wcs input wcs
 @return  cpl_error_code
 */
/*----------------------------------------------------------------------------*/

static hdrl_parameter *
eris_ifu_resample_set_outputgrid (const char* recipe_name,
    const cpl_parameterlist *parlist, cpl_table *muse_table, cpl_wcs *wcs)
{
  /* Should be done in a dedicated _new function */
  cpl_error_ensure(parlist != NULL, CPL_ERROR_NULL_INPUT,
           return NULL, "NULL Input Parameters");
  cpl_error_ensure(muse_table != NULL, CPL_ERROR_NULL_INPUT,
           return NULL, "NULL Input table");
  cpl_error_ensure(wcs != NULL, CPL_ERROR_NULL_INPUT,
           return NULL, "NULL Input wcs");

  hdrl_parameter      *aParams_outputgrid = NULL;
  char* param_name = NULL;
  char* context = NULL;
  double ra_min =  0.;
  double ra_max =  0.;
  double dec_min =  0.;
  double dec_max =  0.;
  double lambda_min =  0.;
  double lambda_max =  0.;

  double dx =  0.;
  double dy =  0.;
  double dlambda =  0.;

  context = cpl_sprintf("eris.%s", recipe_name);

  /* init relevant parameters */
  double ra_min_tmp =
      cpl_table_get_column_min (muse_table, HDRL_RESAMPLE_TABLE_RA);
  double ra_max_tmp =
      cpl_table_get_column_max (muse_table, HDRL_RESAMPLE_TABLE_RA);
  double dec_min_tmp =
      cpl_table_get_column_min (muse_table, HDRL_RESAMPLE_TABLE_DEC);
  double dec_max_tmp =
      cpl_table_get_column_max (muse_table, HDRL_RESAMPLE_TABLE_DEC);
  double lambda_min_tmp =
      cpl_table_get_column_min (muse_table, HDRL_RESAMPLE_TABLE_LAMBDA);
  double lambda_max_tmp =
      cpl_table_get_column_max (muse_table, HDRL_RESAMPLE_TABLE_LAMBDA);

  /* We have the rare case that the image spans over ra = 0.*/
  if(ra_max_tmp - ra_min_tmp > 180){
      const double *ra = cpl_table_get_data_double_const(muse_table,
                             HDRL_RESAMPLE_TABLE_RA);
      /* Should we also take the bpm into account ?
  const int    *bpm = cpl_table_get_data_int_const(ResTable,
                           HDRL_RESAMPLE_TABLE_BPM);
       */

      /* set to extreme values for a start */
      ra_min_tmp = 0.;
      ra_max_tmp = 360.;
      cpl_size nrow = cpl_table_get_nrow(muse_table);

      for (cpl_size i = 0; i < nrow; i++) {
      if (ra[i] > ra_min_tmp && ra[i] <= 180.) ra_min_tmp = ra[i]; /* get the maximum */
      if (ra[i] < ra_max_tmp && ra[i] >  180.) ra_max_tmp = ra[i]; /* get the minimum */
      }
  }

  /* Check output (for debug) */
  cpl_msg_debug (cpl_func, "min x %10.7f", ra_min_tmp);
  cpl_msg_debug (cpl_func, "max x %10.7f", ra_max_tmp);
  cpl_msg_debug (cpl_func, "min y %10.7f", dec_min_tmp);
  cpl_msg_debug (cpl_func, "max y %10.7f", dec_max_tmp);
  cpl_msg_debug (cpl_func, "min lambda %10.7f", lambda_min_tmp);
  cpl_msg_debug (cpl_func, "max lambda %10.7f", lambda_max_tmp);
  /*
  param_name = cpl_sprintf("%s.outgrid.ra-min", context);
  set_double_from_parameters(parlist, param_name, ra_min_tmp, &ra_min);
  cpl_free(param_name);
  */
  ra_min = ra_min_tmp;
  /*
  param_name = cpl_sprintf("%s.outgrid.ra-max", context);
  set_double_from_parameters(parlist, param_name, ra_max_tmp, &ra_max);
  cpl_free(param_name);
  */
  ra_max = ra_max_tmp;

  /*
  param_name = cpl_sprintf("%s.outgrid.dec-min", context);
  set_double_from_parameters(parlist, param_name, dec_min_tmp, &dec_min);
  cpl_free(param_name);
  */
  dec_min = dec_min_tmp;

  /*
  param_name = cpl_sprintf("%s.outgrid.dec-max", context);
  set_double_from_parameters(parlist, param_name, dec_max_tmp, &dec_max);
  cpl_free(param_name);
  */
  dec_max = dec_max_tmp;

  /*
  param_name = cpl_sprintf("%s.outgrid.lambda-min", context);
  set_double_from_parameters(parlist, param_name, lambda_min_tmp, &lambda_min);
  cpl_free(param_name);

  param_name = cpl_sprintf("%s.outgrid.lambda-max", context);
  set_double_from_parameters(parlist, param_name, lambda_max_tmp, &lambda_max);
  cpl_free(param_name);
  */
  lambda_max = lambda_max_tmp;


  /* Reset all delta values */
  double cdelt1 = 0., cdelt2 = 0., cdelt3 = 0.;
  eris_ifu_resample_get_cdelt123 (wcs, &cdelt1, &cdelt2, &cdelt3);
  /*
  param_name = cpl_sprintf("%s.outgrid.delta-ra", context);
  set_double_from_parameters(parlist, param_name, cdelt1, &dx);
  cpl_free(param_name);
  */
  dx = cdelt1;

  /*
  param_name = cpl_sprintf("%s.outgrid.delta-dec", context);
  set_double_from_parameters(parlist, param_name, cdelt2, &dy);
  cpl_free(param_name);
  */
  dy = cdelt2;

  /*
  param_name = cpl_sprintf("%s.outgrid.delta-lambda", context);
  set_double_from_parameters(parlist, param_name, cdelt3, &dlambda);
  cpl_free(param_name);
  */
  dlambda =  cdelt3;


  /* Assign the field margin */
  const cpl_parameter *par = NULL;
  param_name = cpl_sprintf("%s.fieldmargin", context);
  par = cpl_parameterlist_find_const (parlist, param_name);
  cpl_free(param_name);

  double fieldmargin = 0.;
  fieldmargin = cpl_parameter_get_double(par);

  /* create final outgrid parameter structure */
  int naxis = cpl_wcs_get_image_naxis(wcs);

  if(naxis == 2) {
      aParams_outputgrid =
      hdrl_resample_parameter_create_outgrid2D_userdef(dx, dy,
                               ra_min, ra_max,
                               dec_min, dec_max,
                               fieldmargin);
  } else {
      //cpl_msg_info(cpl_func,"dx: %g dy: %g dlambda: %g", dx, dy, dlambda);
      //cpl_msg_info(cpl_func,"ra_min: %g ra_max: %g, dec_min: %g dec_max: %g, lambda_min: %g, lambda_max: %g fieldmarging: %g",
        //      ra_min, ra_max, dec_min, dec_max, lambda_min, lambda_max, fieldmargin);
      aParams_outputgrid =
      hdrl_resample_parameter_create_outgrid3D_userdef(dx, dy, dlambda,
                               ra_min, ra_max,
                               dec_min, dec_max,
                               lambda_min, lambda_max,
                               fieldmargin);
  }

  double dist_x = (ra_max-ra_min)/dx;
  double dist_y = (dec_max-dec_min)/dy;

  cpl_msg_debug(cpl_func,"size1: %g size2: %g",dist_x, dist_y);
  double dist = sqrt(dist_x * dist_x + dist_y * dist_y);
  cpl_msg_info(cpl_func,"Output cube diagonal size: %g", dist);
  if ( dist > MAX_COMBINE_CUBE_PLANE_DIAGONAL) {
        cpl_msg_info(cpl_func,"Internal threshold on max cube plane diagonal size: %d",MAX_COMBINE_CUBE_PLANE_DIAGONAL);
     	cpl_msg_warning(cpl_func,"Resampled cube plane diagonal greater than threshold. Skip cube combination.");
     	cpl_error_set(cpl_func,CPL_ERROR_INCOMPATIBLE_INPUT);
     	cpl_free(context);
     	eris_check_error_code("eris_ifu_resample_set_outputgrid");
     	return aParams_outputgrid;
  }
  cpl_free(context);
  eris_check_error_code("eris_ifu_resample_set_outputgrid");

  return aParams_outputgrid;
}

/*----------------------------------------------------------------------------*/
/**
 @brief   set values into hdrl_parameter* structure 
 @param   parlist input recipe parameters
 @param   muse_table input table with x,y values
 @param   wcs input wcs in 3D
 @return  cpl_error_code
 */
/*----------------------------------------------------------------------------*/

static hdrl_parameter *
eris_ifu_resample_set_outputgrid2D (const char* recipe_name,
    const cpl_parameterlist *parlist, cpl_table *muse_table, cpl_wcs *wcs)
{
  /* Should be done in a dedicated _new function */
  cpl_error_ensure(parlist != NULL, CPL_ERROR_NULL_INPUT,
           return NULL, "NULL Input Parameters");
  cpl_error_ensure(muse_table != NULL, CPL_ERROR_NULL_INPUT,
           return NULL, "NULL Input table");
  cpl_error_ensure(wcs != NULL, CPL_ERROR_NULL_INPUT,
           return NULL, "NULL Input wcs");

  hdrl_parameter      *aParams_outputgrid = NULL;
  char* param_name = NULL;
  char* context = NULL;
  double ra_min =  0.;
  double ra_max =  0.;
  double dec_min =  0.;
  double dec_max =  0.;

  double dx =  0.;
  double dy =  0.;

  context = cpl_sprintf("eris.%s", recipe_name);
  //cpl_table_save (muse_table, NULL, NULL, "muse_table.fits", CPL_IO_CREATE);
  /* init relevant parameters */
  double ra_min_tmp =
      cpl_table_get_column_min (muse_table, HDRL_RESAMPLE_TABLE_RA);
  double ra_max_tmp =
      cpl_table_get_column_max (muse_table, HDRL_RESAMPLE_TABLE_RA);
  double dec_min_tmp =
      cpl_table_get_column_min (muse_table, HDRL_RESAMPLE_TABLE_DEC);
  double dec_max_tmp =
      cpl_table_get_column_max (muse_table, HDRL_RESAMPLE_TABLE_DEC);

  //cpl_msg_info(cpl_func,"ra_max_tmp: %g ra_min_tmp: %g",ra_max_tmp, ra_min_tmp);
  //cpl_msg_info(cpl_func,"dec_max_tmp: %g dec_min_tmp: %g",dec_max_tmp, dec_min_tmp);
  /* We have the rare case that the image spans over ra = 0.*/
  if(ra_max_tmp - ra_min_tmp > 180){
      const double *ra = cpl_table_get_data_double_const(muse_table,
                             HDRL_RESAMPLE_TABLE_RA);
      /* Should we also take the bpm into account ?
  const int    *bpm = cpl_table_get_data_int_const(ResTable,
                           HDRL_RESAMPLE_TABLE_BPM);
       */

      /* set to extreme values for a start */
      ra_min_tmp = 0.;
      ra_max_tmp = 360.;
      cpl_size nrow = cpl_table_get_nrow(muse_table);

      for (cpl_size i = 0; i < nrow; i++) {
      if (ra[i] > ra_min_tmp && ra[i] <= 180.) ra_min_tmp = ra[i]; /* get the maximum */
      if (ra[i] < ra_max_tmp && ra[i] >  180.) ra_max_tmp = ra[i]; /* get the minimum */
      }
  }

  /* Check output (for debug) */
  cpl_msg_debug (cpl_func, "min x %10.7f", ra_min_tmp);
  cpl_msg_debug (cpl_func, "max x %10.7f", ra_max_tmp);
  cpl_msg_debug (cpl_func, "min y %10.7f", dec_min_tmp);
  cpl_msg_debug (cpl_func, "max y %10.7f", dec_max_tmp);

  /*
  param_name = cpl_sprintf("%s.outgrid.ra-min", context);
  set_double_from_parameters(parlist, param_name, ra_min_tmp, &ra_min);
  cpl_free(param_name);
  */
  ra_min = ra_min_tmp;
  /*
  param_name = cpl_sprintf("%s.outgrid.ra-max", context);
  set_double_from_parameters(parlist, param_name, ra_max_tmp, &ra_max);
  cpl_free(param_name);
  */
  ra_max = ra_max_tmp;
  /*
  param_name = cpl_sprintf("%s.outgrid.dec-min", context);
  set_double_from_parameters(parlist, param_name, dec_min_tmp, &dec_min);
  cpl_free(param_name);
  */
  dec_min = dec_min_tmp;

  /*
  param_name = cpl_sprintf("%s.outgrid.dec-max", context);
  set_double_from_parameters(parlist, param_name, dec_max_tmp, &dec_max);
  cpl_free(param_name);
  */
  dec_max = dec_max_tmp;



  /* Reset all delta values */
  double cdelt1 = 0., cdelt2 = 0., cdelt3 = 0.;
  eris_ifu_resample_get_cdelt123 (wcs, &cdelt1, &cdelt2, &cdelt3);
  /*
  param_name = cpl_sprintf("%s.outgrid.delta-ra", context);
  set_double_from_parameters(parlist, param_name, cdelt1, &dx);
  cpl_free(param_name);
  */
  dx = cdelt1;
  /*
  param_name = cpl_sprintf("%s.outgrid.delta-dec", context);
  set_double_from_parameters(parlist, param_name, cdelt2, &dy);
  cpl_free(param_name);
  */
  dy = cdelt2;

  /* Assign the field margin */
  const cpl_parameter *par = NULL;
  param_name = cpl_sprintf("%s.fieldmargin", context);
  par = cpl_parameterlist_find_const (parlist, param_name);
  cpl_free(param_name);

  double fieldmargin = 0.;
  fieldmargin = cpl_parameter_get_double(par);

  /* create final outgrid parameter structure */
  //int naxis = cpl_wcs_get_image_naxis(wcs);

    aParams_outputgrid =
        hdrl_resample_parameter_create_outgrid2D_userdef(dx, dy,
                                ra_min, ra_max,
                                dec_min, dec_max,
                                fieldmargin);
  //cpl_msg_info(cpl_func,"ra_max: %g ra_min: %g dx: %g",ra_max, ra_min, dx);
  //cpl_msg_info(cpl_func,"dec_max: %g dec_min: %g dy: %g",dec_max, dec_min, dy);
  double dist_x = (ra_max-ra_min)/dx;
  double dist_y = (dec_max-dec_min)/dy;

  cpl_msg_debug(cpl_func,"size1: %g size2: %g",dist_x, dist_y);
  double dist = sqrt(dist_x * dist_x + dist_y * dist_y);
  cpl_msg_debug(cpl_func,"Output cube diagonal size: %g", dist);
  if ( dist > MAX_COMBINE_CUBE_PLANE_DIAGONAL) {
        cpl_msg_info(cpl_func,"Internal threshold on max cube plane diagonal size: %d",MAX_COMBINE_CUBE_PLANE_DIAGONAL);
       	cpl_msg_warning(cpl_func,"Resampled cube plane diagonal %g greater than threshold. Skip cube combination.", dist);
       	cpl_error_set(cpl_func,CPL_ERROR_INCOMPATIBLE_INPUT);
       	eris_check_error_code("eris_ifu_resample_set_outputgrid2D");
       	return aParams_outputgrid;
    }
  cpl_free(context);
  eris_check_error_code("eris_ifu_resample_set_outputgrid2D");

  return aParams_outputgrid;
}


/*----------------------------------------------------------------------------*/
/**
 * @brief   save an cube (imagelist, product of pixel resampling) on a FITS file
 * with proper FITS header
 *
 * @param   aCube       cube to save
 * @param   procatg     FITS product category
 * @param   recipe      recipe name
 * @param   filename    product filename
 * @param   parlist     input recipe parameters
 * @param   frameset    product set of frames

 *
 * @return  cpl_error_code
 */
cpl_error_code
eris_ifu_resample_save_cube(hdrl_resample_result *aCube,
                       const char              *procatg,
                       const char              *recipe,
                       const char              *filename,
                       const cpl_parameterlist *parlist,
                       cpl_frameset            *frameset,
					   cpl_boolean             gen_phase3)
{

    cpl_ensure_code(aCube && aCube->header, CPL_ERROR_NULL_INPUT);

//    cpl_error_code rc;
    //cpl_propertylist *header = NULL;
    cpl_propertylist* dataHeader;
    cpl_propertylist* 	errsHeader;
    cpl_propertylist* 	qualHeader;


    deqErrorType errorType = rmse;
	deqQualityType qualityType = flag16bit;
	char* pipe_id = cpl_sprintf("%s%s%s", PACKAGE, "/", PACKAGE_VERSION);

	/* Create FITS headers for the extensions */
	if(cpl_propertylist_has(aCube->header, "CDELT3")) {
		cpl_propertylist_erase(aCube->header, "CDELT3");
	}

    if(!cpl_propertylist_has(aCube->header, CUNIT3)) {
		cpl_propertylist_append_string(aCube->header,"CUNIT3","um");
	}
	if(!cpl_propertylist_has(aCube->header, "BUNIT")) {
		cpl_propertylist_append_string(aCube->header,"BUNIT", UNIT_ADU);
	}


	dataHeader = cpl_propertylist_duplicate(aCube->header);
	errsHeader = cpl_propertylist_duplicate(aCube->header);
	qualHeader = cpl_propertylist_duplicate(aCube->header);
	cpl_propertylist_update_string(qualHeader,"BUNIT", "");

	eris_ifu_heades_add_hduclass_common(dataHeader, "IMAGE");
	eris_ifu_heades_add_hduclass_data(dataHeader);

	eris_ifu_heades_add_hduclass_common(errsHeader, "IMAGE");
	eris_ifu_heades_add_hduclass_errs(errsHeader, errorType);

	eris_ifu_heades_add_hduclass_common(qualHeader, "IMAGE");
	eris_ifu_heades_add_hduclass_qual(qualHeader, qualityType);

	if(cpl_propertylist_has(dataHeader,"CTYPE1")) {
		const char* ctype1 = cpl_propertylist_get_string(dataHeader,"CTYPE1");
        if (!cpl_propertylist_has(errsHeader,"CTYPE1"))
		    cpl_propertylist_append_string(errsHeader, "CTYPE1", ctype1);
        if (!cpl_propertylist_has(qualHeader,"CTYPE1"))
		    cpl_propertylist_append_string(qualHeader, "CTYPE1", ctype1);
	}
	if(cpl_propertylist_has(dataHeader,"CTYPE2")) {
		const char* ctype2 = cpl_propertylist_get_string(dataHeader,"CTYPE2");
        if (!cpl_propertylist_has(errsHeader,"CTYPE2"))
			cpl_propertylist_append_string(errsHeader, "CTYPE2",ctype2);
        if (!cpl_propertylist_has(errsHeader,"CTYPE2"))
			cpl_propertylist_append_string(qualHeader, "CTYPE2", ctype2);
	}

	 cpl_propertylist* applist = cpl_propertylist_duplicate(aCube->header);
	 if(gen_phase3)  {
	    	/*
	    	if(cpl_propertylist_has(aCube->header,"MJD-OBS")) {
	    	   double mjd_obs = cpl_propertylist_get_double(aCube->header, "MJD-OBS");
	    	   cpl_propertylist_append_double(header, "MJD-OBS",mjd_obs);
	    	}
	    	*/
	    	eris_ifu_sdp_properties * prop = eris_ifu_sdp_properties_collect(aCube,
	    			frameset, parlist, recipe);

	    	eris_ifu_sdp_properties_update(applist, prop);
	    	//cpl_propertylist_dump(applist,stdout);
	    	eris_ifu_sdp_properties_delete(prop);
	    }
	 
	/* Create an empty primary HDU with only the header information*/
    eris_ifu_plist_erase_wcs(applist);
    eris_pfits_clean_header(dataHeader, CPL_TRUE);
    eris_pfits_clean_header(errsHeader, CPL_TRUE);
    eris_pfits_clean_header(qualHeader, CPL_TRUE);
	cpl_propertylist_append_string(applist, "PRODCATG", "SCIENCE.CUBE.IFS");

	cpl_propertylist_update_string(applist, CPL_DFS_PRO_CATG, procatg);
	if(cpl_propertylist_has(applist,"BUNIT")) {
			cpl_propertylist_erase(applist,"BUNIT");
	}
	if(cpl_propertylist_has(applist,"COMMENT")) {
		cpl_msg_warning(cpl_func,"remove comments");
		cpl_propertylist_erase(applist,"COMMENT");
	}
	if(cpl_propertylist_has(dataHeader,"COMMENT")) {
			cpl_msg_warning(cpl_func,"remove comments");
			cpl_propertylist_erase(dataHeader,"COMMENT");
		}
	cpl_dfs_save_propertylist(frameset, NULL, parlist, frameset,
			NULL, recipe, applist, "RADECSYS", pipe_id, filename);
	cpl_free(pipe_id);
	cpl_imagelist* data = cpl_imagelist_new();
	cpl_imagelist* errs = cpl_imagelist_new();
	cpl_imagelist* qual = cpl_imagelist_new();

	eris_ifu_split3_hdrl_imagelist(aCube->himlist, data, errs, qual);
	
	/* Write the extension units */
	cpl_type data_type = cpl_image_get_type(cpl_imagelist_get(data,0));
	
	cpl_imagelist_save(data, filename, data_type, dataHeader, CPL_IO_EXTEND);
	cpl_imagelist_save(errs, filename, data_type, errsHeader, CPL_IO_EXTEND);
	cpl_imagelist_save(qual, filename, CPL_TYPE_INT, qualHeader, CPL_IO_EXTEND);
	cpl_size ix_max = cpl_imagelist_get_size(data);
	cpl_image * q;
	for (cpl_size ix=ix_max-1; ix >= 0; ix--) {
	     cpl_imagelist_unset(data, ix);
	     cpl_imagelist_unset(errs, ix);
	     q=cpl_imagelist_unset(qual, ix);
	     cpl_image_delete(q);
	}

    cpl_imagelist_delete(data);
    cpl_imagelist_delete(errs);
    cpl_imagelist_delete(qual);

    cpl_propertylist_delete(applist);
    /* changed up to here */

    /* this creates a problem */
//    eris_ifu_plist_erase_wcs(header);


//    if (rc != CPL_ERROR_NONE) {
//    	return cpl_error_get_code();
//    }


//    cpl_size planes = hdrl_imagelist_get_size(aCube->himlist);

   
    //cpl_propertylist_delete(header);
    eris_ifu_free_propertylist(&dataHeader);
    eris_ifu_free_propertylist(&errsHeader);
    eris_ifu_free_propertylist(&qualHeader);
    eris_check_error_code("eris_ifu_resample_save_cube");
    return cpl_error_get_code();
}


/**
 * @brief This function determines WCS from input data
 *
 * @param frameset input set of frames
 * @return filled cpl_wcs object
 *
 * @doc
 * Generate a FITS header to hold WCS that define the output re-sampled image
 * This is INSTRUMENT mode dependent, because the information on the WCS may
 * come from the primary header or from a FITS extensions in case of different
 * instruments
 *
 */
static cpl_wcs *
eris_ifu_resample_get_wcs_from_frameset(cpl_frameset* frameset,
        const char* procatg) {

  cpl_ensure(frameset, CPL_ERROR_NULL_INPUT, NULL);

  /* Read wcs from firts raw image */
  cpl_frameset* in_set = NULL;

  if ((in_set = eris_ifu_extract_frameset(frameset, procatg)) == NULL){
      cpl_error_set_message(cpl_func, CPL_ERROR_FILE_NOT_FOUND, "Missing RAW "
                "files");
      return NULL;
  }

  cpl_frame* frame = cpl_frameset_get_position(in_set, 0);
  const char* fname = cpl_frame_get_filename(frame);


  cpl_wcs *wcs = NULL;
  cpl_propertylist* head = NULL;

  cpl_errorstate prestate = cpl_errorstate_get();
  head = cpl_propertylist_load(fname, 0);
  wcs = cpl_wcs_new_from_propertylist(head);
  if (wcs == NULL) {
      /* Not possible to read wcs - trying from extension */
      cpl_errorstate_set(prestate);
      cpl_propertylist_delete(head);
  } else {
      cpl_propertylist_delete(head);
      cpl_frameset_delete(in_set);
      return wcs;
  }

  prestate = cpl_errorstate_get();
  head = cpl_propertylist_load(fname, 1);
  wcs = cpl_wcs_new_from_propertylist(head);
  cpl_propertylist_delete(head);
  cpl_frameset_delete(in_set);
  eris_check_error_code("eris_ifu_resample_get_wcs_from_frameset");
  return wcs ;

}


/**
 * @brief transform and insert the data (pixel flux, error, quality) into a
 * cpl_table. Parallelised code
 * @param data_frm input frame with data
 * @param data_ext_id input frame extension containing data
 * @param errs_frm input frame with errors
 * @param errs_ext_id input frame extension containing errors
 * @param qual_frm input frame with pixel quality
 * @param qual_ext_id input frame extension containing pixel quality
 * @param is_variance is input an error or a variance?
 * @param subtract_bkg should be the background be subtracted?
 * @param edge_trim how many pixels are trimmed from resampled plane(s) edges
 * @doc
 * @return cpl_table with filled information
 * */

static cpl_table *
eris_ifu_resample_frameset_to_table(const cpl_frame *data_frm,
                    const cpl_size data_ext_id,
                    const cpl_frame *errs_frm,
                    const cpl_size errs_ext_id,
                    const cpl_frame *qual_frm,
                    const cpl_size qual_ext_id,
                    const cpl_boolean is_variance,
                    const cpl_boolean subtract_bkg,
                    const int edgetrim)
{

  cpl_ensure(data_frm, CPL_ERROR_NULL_INPUT, NULL);
  cpl_msg_severity level = cpl_msg_get_level();
  cpl_msg_set_level(CPL_MSG_INFO);
  const char *name = cpl_frame_get_filename(data_frm);
  cpl_imagelist *dlist = NULL;
  cpl_imagelist *elist = NULL;
  cpl_imagelist *qlist = NULL;

  cpl_table *tab_ext = NULL;

  cpl_errorstate  prestate = cpl_errorstate_get();

  dlist = cpl_imagelist_load(name, CPL_TYPE_DOUBLE, data_ext_id);

  if (dlist == NULL) {
      /*It was not an image nor a imagelist - reset cpl error and return NULL*/
      if (!cpl_errorstate_is_equal(prestate)) {
      cpl_errorstate_set(prestate);
      }
      return NULL;
  }

  cpl_propertylist *xheader_data = cpl_propertylist_load(name,
                             data_ext_id);
  cpl_wcs *wcs = cpl_wcs_new_from_propertylist(xheader_data);

  if (errs_frm != NULL) {
      name = cpl_frame_get_filename(errs_frm);
      elist = cpl_imagelist_load(name, CPL_TYPE_DOUBLE, errs_ext_id);
      if(is_variance) {
      /* transform variance into error */
      cpl_imagelist_power(elist, 0.5);
      }
  }
  if (qual_frm != NULL) {
      name = cpl_frame_get_filename(qual_frm);
      qlist = cpl_imagelist_load(name, CPL_TYPE_INT, qual_ext_id);
  }

  cpl_size size = cpl_imagelist_get_size(dlist);
  /* ingest pixel quality in data */

  if (qual_frm != NULL && size > 0){
      for(cpl_size k = 0; k < size; k++) {
         cpl_image* data = cpl_imagelist_get(dlist, k);
         cpl_image* qual = cpl_imagelist_get(qlist, k);

         /*we use INT_MAX instead of 1.1 as some pipeline
          * may use pixel codes as qualifier */
         cpl_mask* mask = cpl_mask_threshold_image_create(qual, 0, INT_MAX);

         cpl_image_reject_from_mask(data, mask);
         cpl_mask_delete(mask);
         cpl_imagelist_set(dlist, data, k);
      }
  }

  /* In case the error is passed as variance we take the square root. This could
   * cause on some data to add a bad pixel mask - thus we have to synchronize it
   * */
  if(elist != NULL && is_variance){
      for(cpl_size k = 0; k < size; k++) {
      cpl_image* data = cpl_imagelist_get(dlist, k);
      cpl_mask*  data_mask = cpl_image_get_bpm(data);

      cpl_image* error = cpl_imagelist_get(elist, k);
      cpl_mask*  error_mask = cpl_image_get_bpm(error);

      cpl_mask* merged = cpl_mask_duplicate(data_mask);

      /*Add original bad pixels to previous iteration*/
      cpl_mask_or(merged, error_mask);
      cpl_image_reject_from_mask(data, merged);
      cpl_image_reject_from_mask(error, merged);
      cpl_mask_delete(merged);

      }
  }

  hdrl_imagelist* hlist = NULL;
  if (size > 0) {
      hlist = hdrl_imagelist_create(dlist, elist);
  }
  int edge_trim = edgetrim;
  if (edge_trim > 0) {
      cpl_msg_info(cpl_func, "Trim input image edges of %d pixels", edge_trim);
      /* trim each product plane edge of edgetrim pixels */
      cpl_size sx = hdrl_imagelist_get_size_x(hlist);
      cpl_size sy = hdrl_imagelist_get_size_y(hlist);
      cpl_size sz = hdrl_imagelist_get_size(hlist);
      if (edge_trim >= 0.5* sx) {
          edge_trim = 0;
          cpl_msg_warning(cpl_func, "edge-trim must be smaller than half image "
                  "size. Reset to 0");
      }
      if (edge_trim >= 0.5* sy) {
          edge_trim = 0;
          cpl_msg_warning(cpl_func, "edge-trim must be smaller than half image "
                  "size. Reset to 0");
      }
      for(cpl_size k = 0; k < sz; k++) {

          hdrl_image* hima = hdrl_imagelist_get(hlist, k);
          /* Note FITS convention to loop over pixels */
          /* trim lower image border along X direction */
          for(cpl_size j = 1; j <= edge_trim; j++) {
              for(cpl_size i = 1; i <= sx; i++) {
                  hdrl_image_reject(hima, i, j);
              }
          }
          /* trim upper image border along X direction */
          for(cpl_size j = sy - edge_trim + 1; j <= sy; j++) {
              for(cpl_size i = 1; i <= sx; i++) {
                  hdrl_image_reject(hima, i, j);
              }
          }
          /* trim left image border along Y direction */
          for(cpl_size j = 1; j <= sy; j++) {
              for(cpl_size i = 1; i <= edge_trim; i++) {
                  hdrl_image_reject(hima, i, j);
              }
          }
          /* trim right image border along Y direction */
          for(cpl_size j = 1; j <= sy; j++) {
              for(cpl_size i = sx - edge_trim + 1; i <= sx; i++) {
                  hdrl_image_reject(hima, i, j);
              }
          }
      }
  }

  /* single list are not anymore needed as data are copied to the hdrl_imagelist
   * therefore we free them */
  cpl_imagelist_delete(dlist);
  if (qual_frm != NULL)
    cpl_imagelist_delete(qlist);
  if (errs_frm != NULL)
    cpl_imagelist_delete(elist);

  /* Do the calculation */
  if (size == 1){
      /* Subtract the background if we have an image and not a cube */
      cpl_msg_info(cpl_func, "Reading the image ...");
      hdrl_image* hima = hdrl_imagelist_get(hlist, 0);

      /* Subtract the background on request */
      if(subtract_bkg == CPL_TRUE) {
          cpl_msg_info(cpl_func, "Subtracting the median as requested ...");
          hdrl_image_sub_scalar(hima, hdrl_image_get_median(hima));
      }

      /*!!!!!!!!! Interface with the hdrl library function !!!!!!!!!! */
      tab_ext = hdrl_resample_image_to_table(hima, wcs);
  } else {
      cpl_msg_info(cpl_func, "Converting imagelist to table with hdrl function");
      tab_ext = hdrl_resample_imagelist_to_table(hlist, wcs);
  }

  /*Cleanup the memory */
  if (hlist != NULL)
      hdrl_imagelist_delete(hlist);
  cpl_wcs_delete(wcs);
  cpl_propertylist_delete(xheader_data);
  eris_check_error_code("eris_ifu_resample_frameset_to_table");
  cpl_msg_set_level(level);
  return tab_ext;

}


/*----------------------------------------------------------------------------*/
/**
 @brief  print content of the wcs structure
 @param  wcs cpl_wcs to be printed
 @return cpl_error_code
 */
/*----------------------------------------------------------------------------*/

static cpl_error_code
eris_ifu_resample_wcs_print(cpl_wcs *wcs)
{
  cpl_ensure_code(wcs, CPL_ERROR_NULL_INPUT);

  const cpl_array *crval = cpl_wcs_get_crval(wcs);
  const cpl_array *crpix = cpl_wcs_get_crpix(wcs);
  const cpl_array *ctype = cpl_wcs_get_ctype(wcs);
  const cpl_array *cunit = cpl_wcs_get_cunit(wcs);

  const cpl_matrix *cd = cpl_wcs_get_cd(wcs);
  const cpl_array *dims = cpl_wcs_get_image_dims(wcs);
  cpl_size naxis = cpl_wcs_get_image_naxis(wcs);

  cpl_msg_info(cpl_func, "NAXIS:  %lld", naxis);
  int testerr = 0;

  cpl_msg_indent_more();
  /* Check NAXIS */
  for (cpl_size i = 0; i < naxis; i++) {
      cpl_msg_info(cpl_func, "NAXIS%lld: %d", i + 1,
           cpl_array_get_int(dims, i, &testerr));
  }
  cpl_msg_indent_less();

  double cd11 = cpl_matrix_get(cd, 0, 0);
  double cd12 = cpl_matrix_get(cd, 0, 1);
  double cd21 = cpl_matrix_get(cd, 1, 0);
  double cd22 = cpl_matrix_get(cd, 1, 1);
  double crpix1 = cpl_array_get_double(crpix, 0, &testerr);
  double crpix2 = cpl_array_get_double(crpix, 1, &testerr);
  double crval1 = cpl_array_get_double(crval, 0, &testerr);
  double crval2 = cpl_array_get_double(crval, 1, &testerr);

  cpl_msg_info(cpl_func, "1st and 2nd dimension");
  cpl_msg_indent_more();
  cpl_msg_info(cpl_func, "CD1_1:  %g", cd11);
  cpl_msg_info(cpl_func, "CD1_2:  %g", cd12);
  cpl_msg_info(cpl_func, "CD2_1:  %g", cd21);
  cpl_msg_info(cpl_func, "CD2_2:  %g", cd22);

  cpl_msg_info(cpl_func, "CRPIX1: %g", crpix1);
  cpl_msg_info(cpl_func, "CRPIX2: %g", crpix2);
  cpl_msg_info(cpl_func, "CRVAL1: %f", crval1);
  cpl_msg_info(cpl_func, "CRVAL2: %f", crval2);
  if (ctype) {
      cpl_msg_info(cpl_func, "CTYPE1: %s", cpl_array_get_string(ctype, 0));
      cpl_msg_info(cpl_func, "CTYPE2: %s", cpl_array_get_string(ctype, 1));
  }

  if (cunit) {
      cpl_msg_info(cpl_func, "CUNIT1: %s", cpl_array_get_string(cunit, 0));
      cpl_msg_info(cpl_func, "CUNIT2: %s", cpl_array_get_string(cunit, 1));
  }
  cpl_msg_indent_less();

  /* Is it a 3D cube or a 2D image */
  cpl_size cdncol = cpl_matrix_get_ncol(cd);
  if (cdncol == 3) {

      double cd13 = cpl_matrix_get(cd, 0, 2);
      double cd23 = cpl_matrix_get(cd, 1, 2);
      double cd31 = cpl_matrix_get(cd, 2, 0);
      double cd32 = cpl_matrix_get(cd, 2, 1);
      double cd33 = cpl_matrix_get(cd, 2, 2);
      double crval3 = cpl_array_get_double(crval, 2, &testerr);
      double crpix3 = cpl_array_get_double(crpix, 2, &testerr);

      cpl_msg_info(cpl_func, "3rd dimension");
      cpl_msg_indent_more();
      cpl_msg_info(cpl_func, "CD1_3:  %g", cd13);
      cpl_msg_info(cpl_func, "CD2_3:  %g", cd23);
      cpl_msg_info(cpl_func, "CD3_1:  %g", cd31);
      cpl_msg_info(cpl_func, "CD3_2:  %g", cd32);
      cpl_msg_info(cpl_func, "CD3_3:  %g", cd33);

      cpl_msg_info(cpl_func, "CRPIX3: %g", crpix3);
      cpl_msg_info(cpl_func, "CRVAL3: %g", crval3);

      if (ctype) {
          cpl_msg_info(cpl_func, "CTYPE3: %s", cpl_array_get_string(ctype, 2));
      }

      if (cunit) {
          cpl_msg_info(cpl_func, "CUNIT3: %s", cpl_array_get_string(cunit, 2));
      }

      cpl_msg_indent_less();
  }
  eris_check_error_code("eris_ifu_resample_wcs_print");
  return cpl_error_get_code();
}


/**
 * @brief transform and insert the data (pixel flux, error, quality) into a
 * cpl_table. Parallelised code
 * @param data_frm input frame with data
 * @param errs_frm input frame with errors
 * @param qual_frm input frame with pixel quality
 * @param data_ext_id input frame extension containing data
 * @param errs_ext_id input frame extension containing errors
 * @param qual_ext_id input frame extension containing pixel quality
 * @param is_variance  boolean to indicate if errs_frm contains variance or error
 * @param subtract_bkg boolean to indicate if background need to be corrected
 * @param edgetrim     indicates how many pixels need to be trimmed from each
 *                     resampled plane
 * @return cpl_table with filled information
 * @doc
 * */
static cpl_table*
eris_ifu_resample_get_table_from_frameset(const cpl_frame* data_frm,
        const cpl_frame* errs_frm,
        const cpl_frame* qual_frm,
        const cpl_size data_ext_id,
        const cpl_size errs_ext_id,
        const cpl_size qual_ext_id,
        const cpl_boolean is_variance,
        const cpl_boolean subtract_bkg,
        const int edge_trim) {

    cpl_ensure(data_frm, CPL_ERROR_NULL_INPUT, NULL);
    /* Assumption - all have the same number of extensions */
    cpl_size next = cpl_frame_get_nextensions(data_frm);
    cpl_msg_info(cpl_func, "Analysing and processing file %s",
            cpl_frame_get_filename(data_frm));

    /* Assumption
     * if data_ext_id == -1 the loop can be done from 0 to next for data, error,
     * and quality; only the filename changes
     * if data_ext_id != -1 no loop is needed and is difficult to do in the loop
     *  */

    cpl_msg_indent_more();
    cpl_table* table = NULL;
    if(data_ext_id != -1) {
        cpl_msg_info(cpl_func, "Extension: %02lld", data_ext_id);
        /* No Loop, only use the extension given by the user */
        table = eris_ifu_resample_frameset_to_table(data_frm, data_ext_id,
                errs_frm, errs_ext_id,
                qual_frm, qual_ext_id,
                is_variance, subtract_bkg, edge_trim);
    } else {
        /* Loop */
        for (cpl_size i = 0; i < next + 1; i++ ) {
            cpl_table * table_local = NULL;
            cpl_msg_info(cpl_func,"Extension: %02lld", i);
            table_local = eris_ifu_resample_frameset_to_table(data_frm, i,
                    errs_frm, i,
                    qual_frm, i,
                    is_variance, subtract_bkg, edge_trim);

            /* If table_local == NULL go to the next extension */
            if (table_local == NULL) {
                cpl_msg_info(cpl_func, "No siutable data found - continuing");
                continue;
            }

            /* Now table_local is always != NULL */
            if (table == NULL){
                table = table_local;
                continue;
            } else {
                cpl_size nrow = cpl_table_get_nrow(table);
                cpl_table_insert(table, table_local, nrow);
                cpl_table_delete(table_local);
                table_local = NULL;
            }
        }
    }
    cpl_msg_indent_less();
    eris_check_error_code("eris_ifu_resample_get_table_from_frameset");

    return table;
}

///*----------------------------------------------------------------------------*/
///**
// * @internal
// * @brief    Find the aperture(s) with the greatest flux
// * @param    self   The aperture object
// * @param    ind  The aperture-indices in order of decreasing flux
// * @param    nfind  Number of indices to find
// * @return   CPL_ERROR_NONE or the relevant _cpl_error_code_ on error
// *
// * nfind must be at least 1 and at most the size of the aperture object.
// *
// * The ind array must be able to hold (at least) nfind integers.
// * On success the first nfind elements of ind point to indices of the
// * aperture object.
// *
// * To find the single ind of the aperture with the maximum flux use simply:
// * int ind;
// * eris_apertures_find_max_flux(self, &ind, 1);
// *
// */
///*----------------------------------------------------------------------------*/
//static cpl_error_code eris_apertures_find_max_flux(const cpl_apertures * self,
//                                              int * ind, int nfind)
//{
//    const int    nsize = cpl_apertures_get_size(self);
//    int          ifind;


//    cpl_ensure_code(nsize > 0,      cpl_error_get_code());
//    cpl_ensure_code(ind,          CPL_ERROR_NULL_INPUT);
//    cpl_ensure_code(nfind > 0,      CPL_ERROR_ILLEGAL_INPUT);
//    cpl_ensure_code(nfind <= nsize, CPL_ERROR_ILLEGAL_INPUT);

//    for (ifind=0; ifind < nfind; ifind++) {
//        double maxflux = -1;
//        int maxind = -1;
//        int i;
//        for (i=1; i <= nsize; i++) {
//            int k;

//            /* The flux has to be the highest among those not already found */
//            for (k=0; k < ifind; k++) if (ind[k] == i) break;

//            if (k == ifind) {
//                /* i has not been inserted into ind */
//                const double flux = cpl_apertures_get_flux(self, i);

//                if (maxind < 0 || flux > maxflux) {
//                    maxind = i;
//                    maxflux = flux;
//                }
//            }
//        }
//        ind[ifind] = maxind;
//    }
//    eris_check_error_code("eris_apertures_find_max_flux");
//    return CPL_ERROR_NONE;

//}

///**
// * @brief filter input mask by kernel of given sizes and mode
// *
// * @param input_mask  input mask
// * @param kernel_nx   kernel X size
// * @param kernel_ny   kernel X size
// * @param filter      filter mode
// * @return mask corresponding to what filtered
// * @doc
// *
// *  */
//static cpl_mask *
//eris_bpm_filter(
//        const cpl_mask    *   input_mask,
//        cpl_size        kernel_nx,
//        cpl_size        kernel_ny,
//        cpl_filter_mode filter)
//{
//    cpl_mask * kernel = NULL;
//    cpl_mask * filtered_mask = NULL;
//    cpl_mask * expanded_mask = NULL;
//    cpl_mask * expanded_filtered_mask = NULL;

//    /* Check Entries */
//    cpl_ensure(input_mask != NULL, CPL_ERROR_NULL_INPUT, NULL);
//    cpl_ensure(kernel_nx >= 1, CPL_ERROR_ILLEGAL_INPUT, NULL);
//    cpl_ensure(kernel_ny >= 1, CPL_ERROR_ILLEGAL_INPUT, NULL);
//    cpl_ensure(filter == CPL_FILTER_EROSION || filter == CPL_FILTER_DILATION ||
//            filter == CPL_FILTER_OPENING || filter == CPL_FILTER_CLOSING,
//            CPL_ERROR_ILLEGAL_INPUT, NULL);

//    /* Only odd-sized masks allowed */
//    cpl_ensure((kernel_nx&1) == 1, CPL_ERROR_ILLEGAL_INPUT, NULL);
//    cpl_ensure((kernel_ny&1) == 1, CPL_ERROR_ILLEGAL_INPUT, NULL);

//    kernel = cpl_mask_new(kernel_nx, kernel_ny);
//    cpl_mask_not(kernel); /* All values set to unity*/

//    /* Enlarge the original mask with the kernel size and assume that outside
//     * all pixels are good */
//    expanded_mask = cpl_mask_new(
//            cpl_mask_get_size_x(input_mask) + 2 * kernel_nx,
//            cpl_mask_get_size_y(input_mask) + 2 * kernel_ny);

//    cpl_mask_copy(expanded_mask, input_mask, kernel_nx + 1, kernel_ny +1 );

//    expanded_filtered_mask = cpl_mask_new(cpl_mask_get_size_x(expanded_mask),
//            cpl_mask_get_size_y(expanded_mask));

//    if(cpl_mask_filter(expanded_filtered_mask, expanded_mask, kernel, filter,
//                    CPL_BORDER_ZERO) != CPL_ERROR_NONE) {

//        cpl_mask_delete(kernel);
//        cpl_mask_delete(expanded_filtered_mask);
//        cpl_mask_delete(expanded_mask);
//        return NULL;
//    }

//    /* Extract the original mask from the expanded mask */
//    filtered_mask = cpl_mask_extract(expanded_filtered_mask,
//            kernel_nx+1, kernel_ny + 1,
//            cpl_mask_get_size_x(input_mask) + kernel_nx,
//            cpl_mask_get_size_y(input_mask) + kernel_ny);


//    /* Free memory */
//    cpl_mask_delete(kernel);
//    cpl_mask_delete(expanded_filtered_mask);
//    cpl_mask_delete(expanded_mask);
//    eris_check_error_code("eris_bpm_filter");
//    return filtered_mask;
//}

///**
// * @brief generate mask corresponding to post-filtered image according to method
// *
// * @param std_med_ima  input image
// * @param pfm  post filter mask method
// * @param pfx  post filer x size
// * @param pfy  post filer x size
// * @param kappa value used in kappa-sigma clipping
// * @return cpl_error_code
// * @doc
// *
// *  */
//static cpl_mask*
//eris_object_mask(cpl_image* std_med_ima, const char* pfm, const int pfx,
//                  const int pfy, const double kappa)
//{
//    cpl_mask* object_mask;
//    double mad;
//    double median;
//    cpl_mask* nan_mask;
//    /* flag NAN in image (due to SPIFFIER BP conventions) before computing mad */
//    cpl_image_reject_value(std_med_ima,CPL_VALUE_NAN);
//    /* extract mask of NANs to be used later */
//    nan_mask = cpl_image_get_bpm(std_med_ima);
//    //cpl_mask_save(nan_mask,"nan_mask.fits",NULL,CPL_IO_CREATE);
//    median = cpl_image_get_mad(std_med_ima, &mad);
//    double low_cut = median - 10 * kappa * mad * CPL_MATH_STD_MAD;
//    double hi_cut = median + kappa * mad * CPL_MATH_STD_MAD;
//    /* takes as low cut a very small value to be sure to get all negative pixels
//     * as good */
//    low_cut = -1.e10;
//    //cpl_msg_info(cpl_func, "median: %g mad: %g low: %g hi : %g", median, mad, low_cut, hi_cut);
//    /* detect object (and other previously flagged values) as good pixels */
//    object_mask = cpl_mask_threshold_image_create(std_med_ima, low_cut, hi_cut);
//    //cpl_mask_not(object_mask);

//    //cpl_mask_save(object_mask,"object_mask.fits",NULL,CPL_IO_CREATE);
//    /* to detect only object pixels as good one make or with previously flagged
//     * NAN values
//     */

//    cpl_mask_or(object_mask,nan_mask);
//    //cpl_mask_save(object_mask,"or_mask.fits",NULL,CPL_IO_CREATE);
//    /* increase the mask of a few pixels to be sure to include all object values
//     * for this reason we use method dilation and we make a NOT on the image
//     * because the DILATION enlarge the area of BAD pixels, not the GOOD ones
//     * but we want to increase the area covered by the object (flagged as good)
//     */
//    cpl_filter_mode filter_mode = CPL_FILTER_EROSION ;
//    cpl_msg_warning(cpl_func, "pfm = %s", pfm);
//    if( strcmp(pfm,"erosion") == 0 ) {
//        cpl_msg_info(cpl_func, "Filter erosion");
//        filter_mode = CPL_FILTER_EROSION ;
//    } else if( strcmp(pfm, "dilation") == 0 ) {
//        cpl_msg_info(cpl_func, "Filter dilation");
//        filter_mode = CPL_FILTER_DILATION ;
//    } else if( strcmp(pfm,"closing") == 0 ) {
//        cpl_msg_info(cpl_func, "Filter closing");
//        filter_mode = CPL_FILTER_CLOSING ;
//    }
//    //cpl_filter_mode filter_mode = CPL_FILTER_DILATION ;

//    cpl_mask_not(object_mask);
//    cpl_mask*  obj_mask_filtered = eris_bpm_filter(object_mask, pfx, pfy, filter_mode);
//    /* To have again a proper mask with object flagged as good we do a NOT */
//    cpl_mask_not(obj_mask_filtered);
//    //cpl_mask_save(obj_mask_filtered,"filtered_mask.fits",NULL,CPL_IO_CREATE);
//    /* clean-up memory */
//    cpl_mask_delete(object_mask);
//    //cpl_mask_delete(nan_mask);
//    eris_check_error_code("eris_object_mask");
//    return obj_mask_filtered;
//}

/**
 * @brief This function mean-collapse and save a cube along wavelength (Z) azis
 *
 * @param pro_catg  cube product category
 * @param frameset  input set of frames
 * @param parlist   input recipe parameters
 * @param recipe_name recipe name
 * @param apply_flat do flat field
 * @param is_pupil   is a pupil frame
 * @return cpl_error_code
 *  */
cpl_error_code
eris_ifu_cube_collapse_mean_and_save(const char* pro_catg,
                                     cpl_frameset* frameset,
                                     const cpl_parameterlist* parlist,
                                     const char* recipe_name,
//									 const char* filenameSpec,
									 cpl_boolean apply_flat,
									 cpl_boolean is_pupil)
{
    hdrl_image* cube_collapsed = NULL;
    cpl_image* cube_cmap = NULL;

    cpl_frame* frame = cpl_frameset_find(frameset, pro_catg);
    const char* cube_fname = cpl_frame_get_filename(frame);
    cpl_propertylist* pheader = cpl_propertylist_load(cube_fname, 0);
    cpl_propertylist* head_wcs_2d = eris_ifu_plist_extract_wcs2D(pheader);
    cpl_propertylist* dheader = cpl_propertylist_load(cube_fname, 1);
    cpl_propertylist* eheader = cpl_propertylist_load(cube_fname, 2);
    cpl_propertylist* qheader = cpl_propertylist_load(cube_fname, 3);

    cpl_imagelist* iml_data = cpl_imagelist_load(cube_fname, CPL_TYPE_DOUBLE, 1);
    cpl_imagelist* iml_errs = cpl_imagelist_load(cube_fname, CPL_TYPE_DOUBLE, 2);
    cpl_imagelist* iml_qual = cpl_imagelist_load(cube_fname, CPL_TYPE_DOUBLE, 3);

    cpl_propertylist_append(dheader, head_wcs_2d);
    cpl_propertylist_append(eheader, head_wcs_2d);
    cpl_propertylist_append(qheader, head_wcs_2d);
    cpl_propertylist_delete(head_wcs_2d);
    double dit = eris_pfits_get_dit(pheader);
    int ndit = eris_pfits_get_ndit(pheader);
    double exptime = dit * ndit;

    cpl_size sz = cpl_imagelist_get_size(iml_data);
    cpl_image* data = NULL;
    cpl_image* errs = NULL;
    cpl_image* qual = NULL;
    cpl_mask* bpm_data = NULL;
    cpl_mask* bpm_errs = NULL;
    cpl_boolean edge_trim = CPL_TRUE;
    cpl_size k_center = 0.5 * sz;
    for(cpl_size k = 0; k < sz; k++) {
        data = cpl_imagelist_get(iml_data, k);
        errs = cpl_imagelist_get(iml_errs, k);
        qual = cpl_imagelist_get(iml_qual, k);
        cpl_image_reject_value(data, CPL_VALUE_NAN);
        cpl_image_reject_value(errs, CPL_VALUE_NAN);
        bpm_data = cpl_image_get_bpm(data);
        bpm_errs = cpl_image_get_bpm(errs);
        cpl_mask * bpm_mask = cpl_mask_threshold_image_create(qual, -0.5, 0.5) ;
        cpl_mask_not(bpm_mask) ;
        cpl_mask_or(bpm_mask,bpm_data);
        cpl_mask_or(bpm_mask,bpm_errs);
        cpl_image_reject_from_mask(data, bpm_mask);
        cpl_image_reject_from_mask(errs, bpm_mask);

        //EXPOSURE MAP CREATION


          /* exposure map: use the plane at the center of the wavelength range
           * then replace intensity with EXPTIME value and error with 1 and
           * generate HDRL image. This will be re-sampled as the cube and give
           * the exposure map associated to the combined cube
           */

          cpl_image* ima_time = NULL;
          cpl_image* err_time = NULL;
          hdrl_image *himg_exptime = NULL;
          if(apply_flat && k ==  k_center) {
          	ima_time = cpl_image_new(cpl_image_get_size_x(data),
          			cpl_image_get_size_y(data), CPL_TYPE_DOUBLE);
          	err_time = cpl_image_new(cpl_image_get_size_x(data),
          	         cpl_image_get_size_y(data), CPL_TYPE_DOUBLE);


          	cpl_image_add_scalar(ima_time, exptime);
          	cpl_image_add_scalar(err_time, sqrt(exptime));
          	himg_exptime = hdrl_image_create(ima_time, err_time);
          	hdrl_image_reject_from_mask(himg_exptime, bpm_mask);

          	if (edge_trim > 0) {
          		eris_ifu_resample_trim_edge(himg_exptime, edge_trim);
          	}
          	/*
          	char* fname = cpl_sprintf("ima_time_%lld.fits",i);
          	cpl_image_save(ima_time,fname,CPL_TYPE_DOUBLE,NULL,CPL_IO_DEFAULT);
          	cpl_free(fname);
          	fname = cpl_sprintf("err_time_%lld.fits",i);
          	cpl_image_save(err_time,fname,CPL_TYPE_DOUBLE,NULL,CPL_IO_DEFAULT);
          	cpl_free(fname);
          	*/
          	cpl_image_delete(ima_time);
            cpl_image_delete(err_time);

            /* SAVE EXPOSURE MAP */

            cpl_propertylist* phead2D = cpl_propertylist_duplicate(dheader);
            char* fname = cpl_sprintf("%s_exposure_map.fits", recipe_name);
            eris_ifu_plist_erase_wcs3D(phead2D);
            eris_ifu_plist_erase_expmap_extra_keys(phead2D);
            cpl_propertylist_update_string(phead2D, CPL_DFS_PRO_CATG, ERIS_IFU_PRO_JITTER_EXPOSURE_MAP);
            cpl_propertylist_update_string(phead2D, "PRODCATG", PRODCATG_EXPOSUREMAP);
            eris_ifu_save_image_phase3(frameset, /*NULL, */parlist,frameset, /*NULL,*/
                   		recipe_name, phead2D, "RADECSYS", fname,
						hdrl_image_get_image(himg_exptime), CPL_TYPE_DOUBLE,
						UNIT_TIME);
            hdrl_image_delete(himg_exptime);
            cpl_propertylist_delete(phead2D);
            cpl_free(fname);
          }



        cpl_mask_delete(bpm_mask) ;
    }

    hdrl_imagelist* himlist = hdrl_imagelist_create(iml_data, iml_errs);

    hdrl_imagelist_collapse_mean(himlist, &cube_collapsed, &cube_cmap);
    hdrl_imagelist_delete(himlist);

    /* TODO: check header is correct */
    cpl_propertylist* phead2D = cpl_propertylist_duplicate(pheader);
    cpl_propertylist_erase_regexp(phead2D,NAXIS3,0);
    cpl_propertylist_erase_regexp(phead2D,CTYPE3,0);
    cpl_propertylist_erase_regexp(phead2D,CRVAL3,0);
    cpl_propertylist_erase_regexp(phead2D,CRPIX3,0);
    cpl_propertylist_erase_regexp(phead2D,CUNIT3, 0);
    cpl_propertylist_erase_regexp(phead2D,"CD3_*",0);
    cpl_propertylist_erase_regexp(phead2D,CD1_3,0);
    cpl_propertylist_erase_regexp(phead2D,CD2_3,0);

    char* fname = NULL;
    char* prefix = NULL;
    char* suffix = NULL;
    if(apply_flat) {
    	suffix = cpl_sprintf("%s","");
    } else {
    	suffix = cpl_sprintf("%s","_no_flat");
    }
    if(strstr(recipe_name, "jitter") != NULL ) {
        prefix = cpl_sprintf("eris_ifu_jitter%s", suffix);
    } else {
        prefix = cpl_sprintf("eris_ifu_stdstar%s", suffix);
    }
    cpl_free(suffix);
    const char* pcatg = NULL;
    if(strstr(pro_catg, "OBJ") != NULL ) {
        fname = cpl_sprintf("%s_obj_cube_mean.fits", prefix);
        if(apply_flat) {
            pcatg = ERIS_IFU_PRO_JITTER_OBJ_CUBE_MEAN;
        } else {
        	pcatg = ERIS_IFU_PRO_JITTER_OBJ_CUBE_NOFLAT_MEAN;
        }
    } else if(strstr(pro_catg, ERIS_IFU_PRO_JITTER_DAR_CUBE) != NULL ) {
        fname = cpl_sprintf("%s_dar_cube_mean.fits", prefix);
        if(apply_flat) {
            pcatg = ERIS_IFU_PRO_JITTER_OBJ_DAR_CUBE_MEAN;
        } else {
        	pcatg = ERIS_IFU_PRO_JITTER_OBJ_CUBE_NOFLAT_MEAN;
        }
    } else if(strstr(pro_catg, ERIS_IFU_PRO_JITTER_TWK_CUBE) != NULL ) {
        fname = cpl_sprintf("%s_twk_cube_mean.fits", prefix);
        if(apply_flat) {
            pcatg = ERIS_IFU_PRO_JITTER_TWK_CUBE_MEAN;
        } else {
        	pcatg = ERIS_IFU_PRO_JITTER_OBJ_CUBE_NOFLAT_MEAN;
        }
    } else if(strstr(pro_catg, "STD") != NULL ) {
        fname = cpl_sprintf("%s_std_cube_mean.fits", prefix);
        if(apply_flat) {
            pcatg = ERIS_IFU_PRO_JITTER_STD_CUBE_MEAN;
        } else {
        	pcatg = ERIS_IFU_PRO_JITTER_STD_CUBE_NOFLAT_MEAN;
        }
    } else if(strstr(pro_catg, "PSF") != NULL ) {
        fname = cpl_sprintf("%s_psf_cube_mean.fits", prefix);
        if(apply_flat) {
            pcatg = ERIS_IFU_PRO_JITTER_PSF_CUBE_MEAN;
        } else {
        	pcatg = ERIS_IFU_PRO_JITTER_PSF_CUBE_NOFLAT_MEAN;
        }
    }
    cpl_free(prefix);
    cpl_propertylist_update_string(phead2D, CPL_DFS_PRO_CATG, pcatg);
    /* remove WCS from primary header as data go into extensions */
//    eris_ifu_plist_erase_wcs(phead2D);


    if( is_pupil) {
        /* compute PUPIL X/Y SHIFT for QC */
//    	hdrl_imagelist* himl = NULL;

    	cpl_table* qclog_tbl = eris_qclog_init();
    	cpl_image* img = hdrl_image_get_image(cube_collapsed);
    	cpl_size max_ima_x = 0;
    	cpl_size max_ima_y = 0;
    	cpl_image_get_maxpos(img, &max_ima_x, &max_ima_y);
    	cpl_size sx = hdrl_image_get_size_x(cube_collapsed);
    	cpl_size sy = hdrl_image_get_size_y(cube_collapsed);
    	double max_ima_cx = cpl_image_get_centroid_x_window(img, 1, 1, sx, sy);
    	double max_ima_cy = cpl_image_get_centroid_y_window(img, 1, 1, sx, sy);
    	double xshift = max_ima_cx - (double) sx * 0.5;
    	double yshift = max_ima_cy - (double) sy * 0.5;
    	char* key_name = cpl_sprintf("QC PUPIL%d SHIFTX", 0);

    	eris_qclog_add_double_f(qclog_tbl, key_name, xshift,
    			"[pix] X shift centroid - center image");
    	cpl_free(key_name);

    	key_name = cpl_sprintf("QC PUPIL%d SHIFTY", 0);
    	eris_qclog_add_double_f(qclog_tbl, key_name, yshift,
    			"[pix] Y shift centroid - center image");
    	cpl_msg_info(cpl_func, "xshift: %g yshift: %g", xshift, yshift);

    	cpl_free(key_name);
    	eris_pfits_put_qc(phead2D, qclog_tbl);
    	cpl_free(qclog_tbl);
    }

    cpl_image* mask_ima = cpl_image_new_from_mask(hdrl_image_get_mask(cube_collapsed));
    eris_ifu_save_deq_image(frameset, NULL, parlist,frameset, NULL,
    		recipe_name, phead2D, "RADECSYS", fname,
                    hdrl_image_get_image(cube_collapsed),
                    hdrl_image_get_error(cube_collapsed),
                    rmse, mask_ima, flag16bit, UNIT_ADU);
    
    cpl_image_delete(mask_ima);
    cpl_free(fname);
    eris_ifu_free_propertylist(&phead2D);
    cpl_propertylist_delete(dheader);
    cpl_propertylist_delete(eheader);
    cpl_propertylist_delete(qheader);
    cpl_propertylist_delete(pheader);
    cpl_imagelist_delete(iml_data);
    cpl_imagelist_delete(iml_errs);
    cpl_imagelist_delete(iml_qual);
    hdrl_image_delete(cube_collapsed);
    cpl_image_delete(cube_cmap);
    eris_check_error_code("eris_ifu_cube_collapse_mean_and_save");

    return cpl_error_get_code();
}

static double
eris_ifu_compute_max_cubes_dist(cpl_frameset* cube_set)
{
	/* First check if cube size is too large NOT YET VERIFIED */
	double dist_max = -1;
	double dist_sqr = -1;
	cpl_size ncubes = cpl_frameset_get_size(cube_set);
	cpl_vector* vec_ra = cpl_vector_new(ncubes);
	cpl_vector* vec_dec = cpl_vector_new(ncubes);
	cpl_propertylist* plist = NULL;
	const char* name = NULL;
	double ra = 0, dec = 0;
	double ra1 = 0, dec1 = 0;
	double ra2 = 0, dec2 = 0;
	double dist = 0;
	//double cd1_1 = 0;
	//double cd2_2 = 0;
	double pix_size = 0;
	//const char* scale = NULL;
	cpl_propertylist* phead = NULL;
	//TODO: why cd3_3 is set and not used?
//	double cd3_3 = 0;
	 int             extnum_raw = 1;
	cpl_frame* data_frame = NULL;

	data_frame = cpl_frameset_get_position(cube_set, 0);
	name = cpl_frame_get_filename(data_frame);
	plist = cpl_propertylist_load(name, extnum_raw);
	phead = cpl_propertylist_load(name, 0);
//	cd1_1 = cpl_propertylist_get_double(plist, "CD1_1");
//	cd2_2 = cpl_propertylist_get_double(plist, "CD2_2");
//	cd3_3 = cpl_propertylist_get_double(plist, "CD3_3");
	
	ifsPreopticsScale scale_mas = eris_ifu_get_preopticsScale(phead);
	
	cpl_propertylist_delete(plist);
	cpl_propertylist_delete(phead);
	
	switch(scale_mas){
	case S250MAS: pix_size = 0.250; break;
	case S100MAS: pix_size = 0.100; break;
	case S25MAS:  pix_size = 0.025; break;
	case PUPIL:  pix_size = 0.025; break;
	case UNDEFINED_SCALE: pix_size = 0.100; break;
	}
	
	pix_size /= 3600;
	
	/*
	cpl_msg_warning(cpl_func, "cd1_1: %g cd2_2: %g cd3_3: %g",
			cd1_1, cd2_2, cd3_3);
			*/
	for (cpl_size i = 0; i < ncubes ; i++) {

		data_frame = cpl_frameset_get_position(cube_set, i);
		name = cpl_frame_get_filename(data_frame);
		plist = cpl_propertylist_load(name, extnum_raw);

		ra = cpl_propertylist_get_double(plist, "RA");
		dec = cpl_propertylist_get_double(plist, "DEC");
		//cpl_msg_warning(cpl_func, "ra: %13.10g dec: %13.10g",ra,dec);
		cpl_vector_set(vec_ra, i, ra);
		cpl_vector_set(vec_dec, i, dec);

		cpl_propertylist_delete(plist);
	}
	for (cpl_size i = 0; i < ncubes; i++) {
		ra1 = cpl_vector_get(vec_ra, i);
		dec1 = cpl_vector_get(vec_dec, i);
		for (cpl_size j = 0; j < ncubes && i != j; j++) {
			ra2 = cpl_vector_get(vec_ra, j);
			dec2 = cpl_vector_get(vec_dec, j);
			/*
			cpl_msg_warning(cpl_func, "ra1: %13.10g ra2: %13.10g",ra1, ra2);
			cpl_msg_warning(cpl_func, "dist_ra: %13.10g",(ra2 - ra1)/pix_size);
			cpl_msg_warning(cpl_func, "dec1: %13.10g dec2: %13.10g",dec1, dec2);
			cpl_msg_warning(cpl_func, "dist_dec: %13.10g",(dec2 - dec1)/pix_size);
                        */
			dist = (ra2 - ra1) * (ra2 - ra1) / pix_size / pix_size +
					(dec2 - dec1) * (dec2 - dec1) / pix_size / pix_size;
			//cpl_msg_warning(cpl_func, "dist: %13.10g",dist);
			if(dist > dist_sqr) {
				dist_sqr = dist;
			}
		}
	}
	cpl_vector_delete(vec_ra);
	cpl_vector_delete(vec_dec);
	dist_max = sqrt(dist_sqr);
	cpl_msg_info(cpl_func, "Max distance between contributing cubes centers: %g",dist_max);

	eris_check_error_code("eris_ifu_compute_max_cubes_dist");
	return dist_max;

}

/*
static cpl_error_code
eris_resample_compute_size(hdrl_parameter *aParams_outputgrid,
			     int *aX, int *aY, int *aZ)
{
  cpl_ensure_code(aParams_outputgrid && aX && aY && aZ, CPL_ERROR_NULL_INPUT);
  const char func[] = "hdrl_resample_compute_size";
  double x1, y1, x2, y2;

  double ramin = aParams_outputgrid->ra_min;
  double ramax = aParams_outputgrid->ra_max;
  double decmin = aParams_outputgrid->dec_min;
  double decmax = aParams_outputgrid->dec_max;
  hdrl_resample_wcs_projplane_from_celestial(aParams_outputgrid, ramin, decmin,
		  &x1, &y1);
  hdrl_resample_wcs_projplane_from_celestial(aParams_outputgrid, ramax, decmax,
		  &x2, &y2);
  *aX = lround(fabs(x2 - x1) / aParams_outputgrid->delta_ra) + 1;
  *aY = lround(fabs(y2 - y1) / aParams_outputgrid->delta_dec) + 1;

  double lmin = aParams_outputgrid->lambda_min;
  double lmax = aParams_outputgrid->lambda_max;

  *aZ = (int)ceil((lmax - lmin) / aParams_outputgrid->delta_lambda) + 1;

  cpl_msg_debug(func, "Output cube size %d x %d x %d (fit to data)",
	       *aX, *aY, *aZ);
  return CPL_ERROR_NONE;
}
*/
/**
 * @brief This function resample input cubes using HDRL
 *
 * @param obj_type  input object type
 * @param frameset  input set of frames
 * @param parlist   input recipe parameters
 * @param recipe_name recipe name
 * @param pipefile_prefix product filename prefix
 * @return cpl_error_code
 *
 *
 *  */
cpl_error_code
eris_ifu_combine(cubeType obj_type, cpl_frameset* frameset,
        const cpl_parameterlist * parlist,
        const char* recipe_name,
        const char* pipefile_prefix)
{
    cpl_size nframes = cpl_frameset_get_size(frameset);

    cpl_frameset* cube_set = NULL;
    char *proCatg = NULL;
    char    *filenameSpec = NULL;
    cpl_msg_info(cpl_func,"Combine cubes into a single one, compute mean, median along wavelength axis");
    eris_ifu_jitter_get_procatg_and_filename(obj_type, &proCatg, &filenameSpec);

    if ((cube_set = eris_ifu_extract_frameset(frameset, proCatg )) == NULL){

        cpl_msg_warning(cpl_func,"No %s files", proCatg);
        cpl_error_set_message(cpl_func, CPL_ERROR_FILE_NOT_FOUND, "Missing CUBE "
                        "files");
        return cpl_error_get_code();
    } else {
        cpl_msg_info(cpl_func, "%02d file(s) of type %s",
                (int)cpl_frameset_get_size(cube_set), proCatg);
    }

    /*convert the images in a table of predefined structure*/
    cpl_table* restable = NULL; /*Final table to resample*/
    cpl_table* tab_tmp = NULL;
    const cpl_parameter     *   par = NULL;
    cpl_frame   *   data_frame = NULL;
    cpl_frame   *   errs_frame = NULL;
    cpl_frame   *   qual_frame = NULL;
    int             extnum_raw = 1;
    int             extnum_err = 2;
    int             extnum_bpm = 3;
    cpl_boolean  is_variance = CPL_TRUE;
    cpl_boolean  subtract_bkg = CPL_FALSE;
    int                         edge_trim = 0;
    /*Loops over all frames and adds the images/imagelists to the final table */
    cpl_size ncubes = cpl_frameset_get_size(cube_set);
	char* pname = cpl_sprintf("eris.%s.max-cubes-centres-dist",recipe_name);
	int max_cubes_dist = 20;
	//cpl_parameterlist_dump(parlist,stdout);
	
	max_cubes_dist = cpl_parameter_get_int(
	                 cpl_parameterlist_find_const(parlist, pname));

    if ( max_cubes_dist < eris_ifu_compute_max_cubes_dist(cube_set)) {
        cpl_msg_info(cpl_func,"max-cubes-centres-dist: %d",max_cubes_dist);
    	cpl_msg_warning(cpl_func,"max distance between cube centres greater than threshold. Skip cube combination.");
    	    	return CPL_ERROR_INCOMPATIBLE_INPUT;
    }

    for (cpl_size i = 0; i < ncubes ; i++) {

        data_frame = cpl_frameset_get_position(cube_set, i);
        errs_frame = data_frame;
        qual_frame = data_frame;

        tab_tmp = eris_ifu_resample_get_table_from_frameset(data_frame,
                errs_frame,
                qual_frame, extnum_raw,
                extnum_err, extnum_bpm,
                is_variance, subtract_bkg, edge_trim);


        if(nframes == 1) {
            restable = tab_tmp;
        } else {
            eris_ifu_resample_update_table(tab_tmp, &restable);
            cpl_table_delete(tab_tmp);
        }


    }
    cpl_ensure_code(restable, CPL_ERROR_NULL_INPUT);

    /* Save the final table if required */
    eris_ifu_resample_tablesave (recipe_name, par, parlist, frameset, restable,
            pipefile_prefix);
    /* Getting the input wcs needed for determine the scaling of output cube */
    cpl_wcs *wcs = eris_ifu_resample_get_wcs_from_frameset(cube_set, proCatg);
    cpl_ensure_code(wcs, CPL_ERROR_NULL_INPUT);

    cpl_msg_info(cpl_func, "WCS used for the output grid definition and passed to "
            "hdrl_resample_compute():");
    cpl_msg_indent_more();
    eris_ifu_resample_wcs_print(wcs);
    cpl_msg_indent_less();

    hdrl_parameter *aParams_method = NULL;
    /* set re-sampling method */

    char *context = NULL;
    context = cpl_sprintf("eris.%s", recipe_name);
    aParams_method = eris_ifu_resample_set_method(parlist, context);
    cpl_free(context);

    cpl_ensure_code(aParams_method, CPL_ERROR_NULL_INPUT);

    /* set re-sampling outputgrid */
    hdrl_parameter *aParams_outputgrid = NULL;

    aParams_outputgrid = eris_ifu_resample_set_outputgrid(recipe_name, parlist,
            restable, wcs);
    if(cpl_error_get_code() != CPL_ERROR_NONE) {

    	eris_check_error_code("eris_ifu_combine");
    	return CPL_ERROR_INCOMPATIBLE_INPUT;
    }

    cpl_ensure_code(aParams_outputgrid, CPL_ERROR_NULL_INPUT);

    hdrl_resample_result *result = NULL;

    /* !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!! */
    /* !!! Calling the hdrl_resample_compute function doing the real work !!! */
    /* !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!! */
    cpl_msg_severity level = cpl_msg_get_level();
    cpl_msg_set_level(CPL_MSG_INFO);
    result = hdrl_resample_compute(restable, aParams_method, aParams_outputgrid,
            wcs);
    cpl_msg_set_level(level);
    /* TO BE REMOVED: this was added to check position of object (verify DAR)
    hdrl_image* hima = NULL;
    double max_x = 0, max_y = 0, peak = 0;
    cpl_size margin = 100;
    for(cpl_size k= margin; k < hdrl_imagelist_get_size(result->himlist) - margin; k++) {
    	hima = hdrl_imagelist_get(result->himlist, k);
    	eris_gaussian_maxpos(hdrl_image_get_image(hima), 5, &max_x, &max_y, &peak);
    	cpl_msg_info(cpl_func,"z=%lld x: %g y: %g",k,max_x,max_y);
    }
    */

    cpl_ensure_code(result, CPL_ERROR_NULL_INPUT);
    char* pro_catg = NULL;
    if(strstr(pipefile_prefix,"no_flat") != NULL) {
        pro_catg = cpl_sprintf("%s_%s",proCatg,"COADD_NOFLAT");
    } else {
        pro_catg = cpl_sprintf("%s_%s",proCatg,"COADD");
    }
    char* fname = cpl_sprintf("%s_%s_coadd.fits", pipefile_prefix, filenameSpec);

    eris_ifu_resample_save_cube(result, pro_catg, recipe_name, fname, parlist,
            frameset, CPL_FALSE);

    eris_ifu_free_string(&fname);
    eris_ifu_free_string(&pro_catg);
    hdrl_image* cube_collapsed = NULL;
    cpl_image* cube_cmap = NULL;
    hdrl_imagelist_collapse_mean(result->himlist, &cube_collapsed, &cube_cmap);

    /* TODO: check header is correct */
    pro_catg = cpl_sprintf("%s_%s",proCatg,"MEAN");
    fname = cpl_sprintf("%s_%s_mean.fits", pipefile_prefix, filenameSpec);
    cpl_propertylist* phead2D = cpl_propertylist_duplicate(result->header);
    /* remove WCS from primary header as data go into extensions */
//    eris_ifu_plist_erase_wcs(phead2D);
    cpl_propertylist_append_string(phead2D, "PRODCATG",PRODCATG_WHITELIGHT);

    eris_ifu_save_image(frameset, phead2D, parlist, recipe_name, pro_catg,
            fname, CPL_TYPE_DOUBLE, hdrl_image_get_image(cube_collapsed));
    cpl_image_save(hdrl_image_get_error(cube_collapsed), fname, CPL_TYPE_DOUBLE,
            NULL, CPL_IO_EXTEND);
    cpl_image_save(cpl_image_new_from_mask(hdrl_image_get_mask(cube_collapsed)),
            fname, CPL_TYPE_INT, NULL, CPL_IO_EXTEND);

    eris_ifu_free_string(&fname);
    eris_ifu_free_string(&pro_catg);
    eris_ifu_free_string(&filenameSpec);
    eris_ifu_free_string(&proCatg);
    eris_ifu_free_propertylist(&phead2D);
    hdrl_image_delete(cube_collapsed);
    cpl_image_delete(cube_cmap);


    /* Printing the wcs after resampling: */

    cpl_wcs_delete(wcs);
    wcs = cpl_wcs_new_from_propertylist(result->header);
    cpl_msg_info(cpl_func, "Final WCS after resampling: ");
    cpl_msg_indent_more();
    eris_ifu_resample_wcs_print(wcs);
    cpl_msg_indent_less();

    /* Cleanup */

    /* Delete the parameters */
    hdrl_parameter_delete(aParams_method);
    hdrl_parameter_delete(aParams_outputgrid);

    hdrl_resample_result_delete(result);
    cpl_table_delete(restable);
    cpl_wcs_delete(wcs);
    cpl_frameset_delete(cube_set);
    //cpl_frameset_delete(var_err_set);
    //cpl_frameset_delete(bpm_set);
    eris_check_error_code("eris_ifu_combine");

    return cpl_error_get_code();
}

/**
 * @brief Duplicate a 2D header from a 3D header
 * 
 * @param header3D 3D header to be duplicated
 * @return cpl_propertylist
 * 
 */
static cpl_propertylist* eris_ifu_resample_get_header2D(cpl_propertylist *header3D){
    cpl_propertylist *phead2D = cpl_propertylist_duplicate(header3D);
    cpl_propertylist_set_int(phead2D,"NAXIS",2);
    cpl_propertylist_erase_regexp(phead2D,"NAXIS3",0);
    cpl_propertylist_erase_regexp(phead2D,"CTYPE3",0);
    cpl_propertylist_erase_regexp(phead2D,"CRVAL3",0);
    cpl_propertylist_erase_regexp(phead2D,"CRPIX3",0);
    cpl_propertylist_erase_regexp(phead2D,"CUNIT3",0);
    cpl_propertylist_erase_regexp(phead2D,"CDELT3",0);
    cpl_propertylist_erase_regexp(phead2D,"CD3_*",0);
    cpl_propertylist_erase_regexp(phead2D,"CD1_3",0);
    cpl_propertylist_erase_regexp(phead2D,"CD2_3",0);
    eris_check_error_code("eris_ifu_resample_get_header2D");
    return phead2D;
}

/**
 * @brief Duplicate a 2D header from a 3D header
 * 
 * @param himg hdrl_image to be trimmed
 * @param edge_trim Number of pixels to be trimmed
 * @return cpl_error_code
 * 
 */
cpl_error_code eris_ifu_resample_trim_edge(hdrl_image *himg, int edge_trim){
    if (edge_trim > 0) {
        //cpl_msg_info(cpl_func, "Trim input image edges of %d pixels", edge_trim);
        /* trim each product plane edge of edgetrim pixels */
        cpl_size sx = hdrl_image_get_size_x(himg);
        cpl_size sy = hdrl_image_get_size_y(himg);
        if (edge_trim >= 0.5* sx) {
            edge_trim = 0;
            cpl_msg_warning(cpl_func, "edge-trim must be smaller than half image "
                    "size. Reset to 0");
        }
        if (edge_trim >= 0.5* sy) {
            edge_trim = 0;
            cpl_msg_warning(cpl_func, "edge-trim must be smaller than half image "
                    "size. Reset to 0");
        }

        /* Note FITS convention to loop over pixels */
        /* trim lower image border along X direction */
        for(cpl_size j = 1; j <= edge_trim; j++) {
            for(cpl_size i = 1; i <= sx; i++) {
                hdrl_image_reject(himg, i, j);
            }
        }
        /* trim upper image border along X direction */
        for(cpl_size j = sy - edge_trim + 1; j <= sy; j++) {
            for(cpl_size i = 1; i <= sx; i++) {
                hdrl_image_reject(himg, i, j);
            }
        }
        /* trim left image border along Y direction */
        for(cpl_size j = 1; j <= sy; j++) {
            for(cpl_size i = 1; i <= edge_trim; i++) {
                hdrl_image_reject(himg, i, j);
            }
        }
        /* trim right image border along Y direction */
        for(cpl_size j = 1; j <= sy; j++) {
            for(cpl_size i = sx - edge_trim + 1; i <= sx; i++) {
                hdrl_image_reject(himg, i, j);
            }
        }
    }
    else {
    cpl_msg_warning(cpl_func, "edge-trim is set to 0");
    }
    eris_check_error_code("eris_ifu_resample_trim_edge");
    return cpl_error_get_code();
}

/**
 * @brief This function resample input cubes plane by plane using HDRL
 *
 * @param frameset  input set of frames
 * @param parlist   input recipe parameters
 * @param input_cube_pro_catg   type of products to be combined
 * @param filenameSpec product filename suffix
 * @param offsetx   user defined X offsets
 * @param offsety   user defined Y offsets
 * @param offunit   user defined offset unit [pix/arcsec]
 * @param recipe_name recipe name
 * @param pipefile_prefix product filename prefix
 * @return cpl_error_code
 *  */
cpl_error_code
eris_ifu_combine_pbp(cpl_frameset* frameset,
        const cpl_parameterlist * parlist,
        const char *input_cube_pro_catg,
        const char *filenameSpec,
        float *offsetx, 
        float *offsety, 
        const char* offunit, 
        const char* recipe_name,
        const char* pipefile_prefix){

    cpl_frameset* cube_set = NULL;
    cpl_msg_info(cpl_func,"Combine cubes into a single one plane-by-plane, compute mean, median along wavelength axis");

    if ((cube_set = eris_ifu_extract_frameset(frameset, input_cube_pro_catg )) == NULL){

        cpl_msg_warning(cpl_func,"No %s files", input_cube_pro_catg);
        //cpl_frameset_dump(frameset, stdout);
        cpl_error_set_message(cpl_func, CPL_ERROR_FILE_NOT_FOUND,
        		"Missing cube files tagged as %s", input_cube_pro_catg);
        return cpl_error_get_code();
    } else {
        cpl_msg_info(cpl_func, "%02d file(s) of type %s",
                (int)cpl_frameset_get_size(cube_set), input_cube_pro_catg);
    }

    /*convert the images in a table of predefined structure*/
    //const cpl_parameter     *   par = NULL;
    const cpl_frame   *   data_frame = NULL;
    hdrl_parameter *aParams_outputgrid = NULL;
    cpl_propertylist *phead2D = NULL; 
    int             extnum_raw = 1;
    int             extnum_err = 2;
    int             extnum_bpm = 3;

    //cpl_boolean  is_variance = CPL_TRUE;
    cpl_boolean  subtract_bkg = CPL_FALSE;
    int                         edge_trim = 0;
    /*Loops over all frames and adds the images/imagelists to the final table */
    cpl_size ncubes = cpl_frameset_get_size(cube_set);
    if (ncubes == 1){
        cpl_error_set_message(cpl_func, CPL_ERROR_FILE_NOT_FOUND, "Only one CUBE "
                        "files");
        return cpl_error_get_code();
    } 

	char* pname = cpl_sprintf("eris.%s.max-cubes-centres-dist",recipe_name);
	int max_cubes_dist = 20;
	//cpl_parameterlist_dump(parlist,stdout);
    if (offsetx !=NULL && offsety !=NULL) // user offset mode
        max_cubes_dist = INT_MAX ;//set to the largest, i.e. ignore this param. 
	else 
        max_cubes_dist = cpl_parameter_get_int(
                        cpl_parameterlist_find_const(parlist, pname));
    cpl_free(pname);

    pname = cpl_sprintf("eris.%s.subtract-background",recipe_name);
    subtract_bkg = cpl_parameter_get_bool(
	                 cpl_parameterlist_find_const(parlist, pname)); 
    if(subtract_bkg == CPL_TRUE) 
        cpl_msg_info(cpl_func, "Median of each plane will be subtracted.");
    cpl_free(pname);

    pname = cpl_sprintf("eris.%s.edge-trim",recipe_name);
    edge_trim = cpl_parameter_get_int(
	            cpl_parameterlist_find_const(parlist, pname));
    if (edge_trim > 0)
        cpl_msg_info(cpl_func, "%d pixel(s) on edges will be trimmed.", edge_trim);

    cpl_free(pname);
    
    
  
    if ( max_cubes_dist < eris_ifu_compute_max_cubes_dist(cube_set)) {
        cpl_msg_info(cpl_func,"max-cubes-centres-dist: %d",max_cubes_dist);
    	cpl_msg_warning(cpl_func,"max distance between cube centres greater than threshold. Skip cube combination.");
    	    	return CPL_ERROR_INCOMPATIBLE_INPUT;
    }

    hdrl_parameter *aParams_method = NULL;
    /* set re-sampling method */
    char *context = NULL;
    context = cpl_sprintf("eris.%s", recipe_name);
    aParams_method = eris_ifu_resample_set_method(parlist, context);
    cpl_free(context);
    cpl_ensure_code(aParams_method, CPL_ERROR_NULL_INPUT);

    hdrl_imagelist *res_cube = hdrl_imagelist_new();
    cpl_wcs *wcs = NULL;
    cpl_wcs **tmp_wcs_array = (cpl_wcs **) cpl_calloc(ncubes, sizeof(wcs)); 
    char const **name_array = (char const **) cpl_calloc(ncubes, sizeof(const char*)); 

    cpl_wcs *wcs3d = NULL;
    double mjd_obs = 0;
    double crpix3 = 0;
    double crval3 = 0;
    // double cdelt3 = 0;
    double cd1_3 = 0;
    double cd2_3 = 0;
    double cd3_1 = 0;
    double cd3_2 = 0;
    double cd3_3 = 0;
    double cd1_1 = 0;
    double cd1_2 = 0;
    double cd2_1 = 0;
    double cd2_2 = 0;
    int naxis3 = 0;
    const char* ctype3 = "WAVE";
    // const char* cunit3 = "um";
    const cpl_matrix *cd = NULL; 
    cpl_vector* exptime_vec = cpl_vector_new(ncubes);
    double exptime = 0;
    double dit = 0;
    int ndit = 0;
    for (cpl_size i = 0; i < ncubes ; i++) {
        data_frame = cpl_frameset_get_position_const(cube_set, i);
        const char *name = cpl_frame_get_filename(data_frame);
        cpl_propertylist *xheader_data = cpl_propertylist_load(name,
                                    extnum_raw);
        cpl_propertylist* phead = cpl_propertylist_load(name, 0);
        dit = eris_pfits_get_dit(phead);
        ndit = eris_pfits_get_ndit(phead);

        cpl_propertylist_delete(phead);
        exptime = dit * ndit;
        cpl_vector_set(exptime_vec, i, exptime);
        cpl_propertylist *header2D = eris_ifu_resample_get_header2D(xheader_data);
        cpl_wcs *tmp_wcs = cpl_wcs_new_from_propertylist(header2D);
        cpl_propertylist_delete(header2D);

        tmp_wcs_array[i] = tmp_wcs;
        name_array[i] = name;

        if (i==0){
            wcs = tmp_wcs;
            cpl_ensure_code(wcs, CPL_ERROR_NULL_INPUT);
            cpl_msg_info(cpl_func, "WCS used for the output grid definition and passed to "
            "hdrl_resample_compute():");
            cpl_msg_indent_more();
            eris_ifu_resample_wcs_print(wcs);
            cpl_msg_indent_less();
            wcs3d = cpl_wcs_new_from_propertylist(xheader_data);
            mjd_obs = cpl_propertylist_get_double(xheader_data, "MJD-OBS");
            crpix3 = cpl_propertylist_get_double(xheader_data, "CRPIX3");
            crval3 = cpl_propertylist_get_double(xheader_data, "CRVAL3");
            naxis3 = cpl_propertylist_get_int(xheader_data, "NAXIS3");
            //cdelt3 = cpl_propertylist_get_double(xheader_data, "CDELT3");
            /*
            if(cpl_propertylist_has(xheader_data,"CTYPE3")) {
            	ctype3 = cpl_propertylist_get_string(xheader_data, "CTYPE3");
            	cpl_msg_warning(cpl_func,"check ctype3: >%s< >%s<",
            			ctype3,cpl_propertylist_get_string(xheader_data, "CTYPE3"));
            } else {
            	ctype3 = "WAVE";
            }
            if(cpl_propertylist_has(xheader_data,"CUNIT3")) {
            	cunit3 = cpl_propertylist_get_string(xheader_data, "CUNIT3");
            } else {
            	cunit3 = "um";
            }
            */
            cd3_3 = cpl_propertylist_get_double(xheader_data, "CD3_3");
            // cdelt3 = cd3_3;

            cd = cpl_wcs_get_cd(wcs);
            cd1_1 = cpl_matrix_get(cd, 0, 0); 
            cd1_2 = cpl_matrix_get(cd, 1, 0);
            cd2_1 = cpl_matrix_get(cd, 0, 1);
            cd2_2 = cpl_matrix_get(cd, 1, 1);

            cpl_ensure_code(wcs3d, CPL_ERROR_NULL_INPUT);            
        } 
        cpl_propertylist_delete(xheader_data);
    }

    /*Convert offset unit to degree*/ 
    double as2deg = 1.0/3600.0; 
    double scalex = 1.0;
    double scaley = 1.0;

    int offcode = -1; 
    if (offunit != NULL && offsetx !=NULL && offsety !=NULL){
        if (strcmp(offunit, "PIXEL") == 0){
            offcode = 0; 
            cpl_msg_info(cpl_func, "User-defined offset unit in PIXEL");
        }else if (strcmp(offunit,  "DEGREE") == 0){
            offcode = 1; 
            cpl_msg_info(cpl_func, "User-defined offset unit is DEGREE");
        }else if (strcmp(offunit,  "ARCSEC") == 0){
            offcode = 2; 
            cpl_msg_info(cpl_func, "User-defined offset unit is ARCSEC");
            scalex =  as2deg;
            scaley =  as2deg;
        }else 
            cpl_msg_error(cpl_func, "User-defined offset unit or offset list is not correct.");
    }


    const cpl_array *dims =  cpl_wcs_get_image_dims(wcs3d);
    int nchan = cpl_array_get_int(dims, 2, NULL);  
    cpl_size ich_center = 0.5 * nchan;
    cpl_msg_info(cpl_func, "Number of planes: %4.4d", nchan);
    //cpl_table* restable_exptime = NULL;
//    hdrl_image *res_himg_exptime = NULL;
    hdrl_imagelist* res_himlist_exptime = hdrl_imagelist_new();

    cpl_msg_severity level = cpl_msg_get_level();
    cpl_msg_set_level(CPL_MSG_INFO);
    for (cpl_size ich = 0; ich < nchan; ich++){
        cpl_table* restable = NULL; /*Final table to resample*/

        for (cpl_size i = 0; i < ncubes ; i++) {
            const char *name = name_array[i];
            cpl_image *im = cpl_image_load(name, CPL_TYPE_DOUBLE, ich, extnum_raw);
            cpl_image *err = cpl_image_load(name, CPL_TYPE_DOUBLE, ich, extnum_err);
            cpl_image* qual = cpl_image_load(name, CPL_TYPE_INT, ich, extnum_bpm);
            cpl_mask *mask = cpl_mask_threshold_image_create(qual, 0, INT_MAX);
            cpl_image_delete(qual);
            // EXPOSURE MAP
            /* exposure map: use the plane at the center of the wavelength range
             * then replace intensity with EXPTIME value and error with 1 and
             * generate HDRL image. This will be re-sampled as the cube and give
             * the exposure map associated to the combined cube
             */

            cpl_image* ima_time = NULL;
            cpl_image* err_time = NULL;
            hdrl_image *himg_exptime = NULL;
            if(ich ==  ich_center) {
            	ima_time = cpl_image_new(cpl_image_get_size_x(im),
            			cpl_image_get_size_y(im), CPL_TYPE_DOUBLE);
            	err_time = cpl_image_new(cpl_image_get_size_x(im),
            	         cpl_image_get_size_y(im), CPL_TYPE_DOUBLE);

            	exptime = cpl_vector_get(exptime_vec, i);
            	cpl_image_add_scalar(ima_time, exptime);
            	cpl_image_add_scalar(err_time, sqrt(exptime));
            	himg_exptime = hdrl_image_create(ima_time, err_time);
            	/* for EXPPOSURE MAP we assume all pixels are good */
            	//hdrl_image_reject_from_mask(himg_exptime, mask);
            	 
            	if (edge_trim > 0) {
            		eris_ifu_resample_trim_edge(himg_exptime, edge_trim);
            	}

            	cpl_image_delete(ima_time);
                cpl_image_delete(err_time);
            }

            hdrl_image *himg = hdrl_image_create(im, err);
            cpl_image_delete(im);
            cpl_image_delete(err);
            hdrl_image_reject_from_mask(himg, mask);
            cpl_mask_delete(mask);

            /* Subtract the background on request */
            if(subtract_bkg == CPL_TRUE) {
                //cpl_msg_info(cpl_func, "Subtracting the median as requested ...");
                hdrl_image_sub_scalar(himg, hdrl_image_get_median(himg));
            }

            if (edge_trim > 0) {
                eris_ifu_resample_trim_edge(himg, edge_trim);
            }

            cpl_table *tab_tmp = NULL;
            cpl_table *tab_exptime_tmp = NULL;
            if (offsetx !=NULL && offsety !=NULL){
                if (offcode==0){
                    tab_tmp = hdrl_resample_image_to_table(himg, tmp_wcs_array[0]);
                    cpl_table_subtract_scalar(tab_tmp, "ra",  offsetx[i]*cd1_1+ offsety[i]*cd1_2);
                    cpl_table_subtract_scalar(tab_tmp, "dec",  offsetx[i]*cd2_1+ offsety[i]*cd2_2);
                    if (ich ==0)
                        cpl_msg_info(__func__, "User-defined offest in RA, DEC (ARCSEC) %g, %g", 
                                    (offsetx[i]*cd1_1+ offsety[i]*cd1_2)*3600, 
                                    (offsetx[i]*cd2_1+ offsety[i]*cd2_2)*3600);
                   
                    if(ich ==  ich_center) {
                    	tab_exptime_tmp = hdrl_resample_image_to_table(himg_exptime,
                    			tmp_wcs_array[0]);
                    	cpl_table_subtract_scalar(tab_exptime_tmp, "ra",  offsetx[i]*cd1_1+ offsety[i]*cd1_2);
                    	cpl_table_subtract_scalar(tab_exptime_tmp, "dec",  offsetx[i]*cd2_1+ offsety[i]*cd2_2);	
                    }
                } else if (offcode==1 || offcode==2 ){
                    tab_tmp = hdrl_resample_image_to_table(himg, tmp_wcs_array[0]);
                    cpl_table_subtract_scalar(tab_tmp, "ra",  offsetx[i]*scalex);
                    cpl_table_subtract_scalar(tab_tmp, "dec",  offsety[i]*scaley);
                    if (ich ==0)
                        cpl_msg_info(__func__, "User-defined offest in RA, DEC (%s) %g, %g", 
                                    offunit, 
                                    offsetx[i], 
                                    offsety[i]);
                   
                    if(ich ==  ich_center) {
                    	tab_exptime_tmp = hdrl_resample_image_to_table(himg_exptime, tmp_wcs_array[0]);
                    	cpl_table_subtract_scalar(tab_exptime_tmp, "ra",  offsetx[i]*scalex);
                    	cpl_table_subtract_scalar(tab_exptime_tmp, "dec",  offsety[i]*scaley);	
                    }
                }
            } else {
                tab_tmp = hdrl_resample_image_to_table(himg, tmp_wcs_array[i]);
                if(ich ==  ich_center)   {
                    tab_exptime_tmp = hdrl_resample_image_to_table(himg_exptime, tmp_wcs_array[i]);
                }
            }
            hdrl_image_delete(himg);
            eris_ifu_resample_update_table(tab_tmp, &restable);
          
            if(ich ==  ich_center) {
            	
            	hdrl_image_delete(himg_exptime);
            	//eris_ifu_resample_update_table(tab_exptime_tmp, &restable_exptime);
                hdrl_resample_result *himl_exptime_tmp = NULL;
            	himl_exptime_tmp = hdrl_resample_compute(tab_exptime_tmp,
            			aParams_method, aParams_outputgrid, wcs);
            	//const char* fname = cpl_sprintf("ima_time_%lld.fits",i);
            	hdrl_image *himg_tmp = hdrl_image_duplicate(
            			hdrl_imagelist_get_const(himl_exptime_tmp->himlist, 0));
            	hdrl_imagelist_set(res_himlist_exptime, himg_tmp, i);
            	hdrl_resample_result_delete(himl_exptime_tmp);

            	/*
            	cpl_image_save(hdrl_image_get_image(himg_tmp), fname,
            			CPL_TYPE_DOUBLE, NULL, CPL_IO_DEFAULT);
            			*/
            	cpl_table_delete(tab_exptime_tmp);

            }
         
            cpl_table_delete(tab_tmp);
            if (ich%100 == 0)
                cpl_msg_debug(__func__, "Created table for channel %d cube %d", (int)ich, (int)i);
        } /* end loop over input data cubes */
        cpl_ensure_code(restable, CPL_ERROR_NULL_INPUT);
        
        if (ich%100 == 0)
            cpl_msg_debug(__func__, "Table of the chanel %d size %lld x %lld", (int)ich,
                                 cpl_table_get_nrow(restable), cpl_table_get_ncol(restable));

        if(ich==0){
            aParams_outputgrid = eris_ifu_resample_set_outputgrid2D(recipe_name, parlist,
                    restable, wcs);
            if ( cpl_error_get_code() != CPL_ERROR_NONE) {
                       	eris_check_error_code("eris_ifu_combine_pbp");
                       	return CPL_ERROR_NONE;
                       }
            cpl_ensure_code(aParams_outputgrid, CPL_ERROR_INCOMPATIBLE_INPUT);

            //int err_code = cpl_table_save(restable, NULL, NULL, "test_table.fits", CPL_IO_CREATE);
        }
        cpl_ensure_code(aParams_outputgrid, CPL_ERROR_NULL_INPUT);

        hdrl_resample_result *result = NULL;
        result = hdrl_resample_compute(restable, aParams_method, aParams_outputgrid, wcs);
        
        /* OLD WAY
        if(ich ==  ich_center) {
        	 hdrl_resample_result *result_exptime = NULL;
        	 result_exptime = hdrl_resample_compute(restable_exptime, aParams_method, aParams_outputgrid, wcs);
        	 res_himg_exptime = hdrl_image_duplicate(hdrl_imagelist_get_const(result_exptime->himlist, 0));

        }
        */
        
        cpl_ensure_code(result, CPL_ERROR_NULL_INPUT);
        cpl_table_delete(restable);

        if(ich==0){
            /* Printing the wcs after resampling: */
            cpl_wcs *res_wcs = cpl_wcs_new_from_propertylist(result->header);
            cpl_msg_info(cpl_func, "Final WCS after resampling: ");
            cpl_msg_indent_more();
            eris_ifu_resample_wcs_print(res_wcs);
            cpl_msg_indent_less();
            cpl_wcs_delete(res_wcs);
            phead2D = cpl_propertylist_duplicate(result->header);
        }

        hdrl_image *res_himg = hdrl_image_duplicate(hdrl_imagelist_get_const(result->himlist, 0));
        hdrl_imagelist_set(res_cube, res_himg, ich); 
        hdrl_resample_result_delete(result);

        if (ich==nchan-1){
            for (int i = 0; i < ncubes; i++){
                cpl_wcs_delete(tmp_wcs_array[i]);
            }
            cpl_free(tmp_wcs_array);
            cpl_free(name_array);
        }

        if (ich%100 == 0)
            cpl_msg_info(__func__, "Coadding cubes: range [%4.4d,%4.4d] of %d",
                            (int)ich,(int)ich+100, nchan);
    }
    cpl_msg_set_level(level);

    hdrl_image* hima_exptime_mean = NULL;
    cpl_image* contrib_exptime_map = NULL;
    hdrl_imagelist_collapse_mean(res_himlist_exptime, &hima_exptime_mean,
    		&contrib_exptime_map);
    hdrl_imagelist_delete(res_himlist_exptime);
    hdrl_image_delete(hima_exptime_mean);
    /*
    cpl_image_save(hdrl_image_get_image(hima_exptime_mean), "exptime_mean.fits",
                			CPL_TYPE_DOUBLE, NULL, CPL_IO_DEFAULT);
                			*/
    cpl_image_multiply_scalar(contrib_exptime_map, cpl_vector_get_median(exptime_vec));
    //cpl_image_save(contrib, "contrib.fits", CPL_TYPE_DOUBLE, NULL, CPL_IO_DEFAULT);
    cpl_ensure_code(res_cube, CPL_ERROR_NULL_INPUT);
    cpl_vector_delete(exptime_vec);

    hdrl_parameter_delete(aParams_method);
    hdrl_parameter_delete(aParams_outputgrid);

    /* Save resample cube*/


    hdrl_resample_result *result = cpl_calloc(1, sizeof(hdrl_resample_result));
    result->himlist = res_cube;
    result->header = phead2D;


    cpl_propertylist_append_double(result->header, "CRPIX3", crpix3);
    cpl_propertylist_append_double(result->header, "CRVAL3", crval3);
    cpl_propertylist_append_int(result->header, "NAXIS3", naxis3);
    //cpl_propertylist_append_double(result->header, "CDELT3", cdelt3);
    cpl_propertylist_append_string(result->header, "CTYPE3", ctype3);
    cpl_propertylist_append_double(result->header, "CD1_3", cd1_3);
    cpl_propertylist_append_double(result->header, "CD2_3", cd2_3);
    cpl_propertylist_append_double(result->header, "CD3_3", cd3_3);
    cpl_propertylist_append_double(result->header, "CD3_1", cd3_1);
    cpl_propertylist_append_double(result->header, "CD3_2", cd3_2);
    cpl_propertylist_append_double(result->header, "MJD-OBS", mjd_obs);
    cpl_propertylist* wcs2D = eris_ifu_plist_extract_wcs2D(result->header);

    hdrl_image* cube_collapsed = NULL;
    cpl_image* cube_cmap = NULL;
    hdrl_imagelist_collapse_mean(res_cube, &cube_collapsed, &cube_cmap);
    /* This makes 2D products not FITS standard 
    cpl_propertylist* wcs2D = eris_ifu_plist_extract_wcs(result->header);
    cpl_propertylist_append(phead2D, wcs2D);
    cpl_propertylist_delete(wcs2D);
     */

    char* pro_catg = NULL;
    if(strstr(pipefile_prefix,"no_flat") != NULL) {
    	pro_catg = cpl_sprintf("%s_%s",input_cube_pro_catg,"COADD_NOFLAT");
    } else {
    	pro_catg = cpl_sprintf("%s_%s",input_cube_pro_catg,"COADD");
    }
    char* fname = cpl_sprintf("%s_%s_coadd.fits", pipefile_prefix, filenameSpec);

    if(strcmp(recipe_name,"eris_ifu_jitter") == 0) {
    	eris_ifu_resample_save_cube(result, pro_catg, recipe_name, fname, parlist,
    			frameset, CPL_FALSE);
    } else {
    	eris_ifu_resample_save_cube(result, pro_catg, recipe_name, fname, parlist,
    			frameset, CPL_FALSE);
    }
    eris_ifu_free_string(&fname);
    eris_ifu_free_string(&pro_catg);


    /* TODO: check header is correct */
    pro_catg = cpl_sprintf("%s_%s",input_cube_pro_catg,"MEAN");
    fname = cpl_sprintf("%s_%s_mean.fits", pipefile_prefix, filenameSpec);
    cpl_propertylist_append_string(phead2D, "PRODCATG", PRODCATG_WHITELIGHT);
    /* remove WCS from primary header as data go into extensions */
//    eris_pfits_erase_wcs2D(phead2D);
    cpl_propertylist_update_string(phead2D, CPL_DFS_PRO_CATG, pro_catg);
    cpl_image* mask_ima = cpl_image_new_from_mask(hdrl_image_get_mask(cube_collapsed));
    eris_ifu_save_deq_image(frameset, NULL, parlist,frameset, NULL,
    		recipe_name, phead2D, "RADECSYS", fname,
			hdrl_image_get_image(cube_collapsed),
			hdrl_image_get_error(cube_collapsed),
			rmse, mask_ima, flag16bit, UNIT_ADU);
    cpl_free(fname);
    fname = cpl_sprintf("%s_%s_exposure_map.fits", pipefile_prefix, filenameSpec);

    cpl_propertylist_update_string(phead2D, CPL_DFS_PRO_CATG, ERIS_IFU_PRO_JITTER_EXPOSURE_MAP);
    cpl_propertylist_update_string(phead2D, "PRODCATG", PRODCATG_EXPOSUREMAP);

       cpl_propertylist_append(phead2D, wcs2D);
       cpl_propertylist_delete(wcs2D);
    eris_ifu_save_image_phase3(frameset, /*NULL, */parlist,frameset, /*NULL,*/
    		recipe_name, phead2D, "RADECSYS", fname, contrib_exptime_map,
			CPL_TYPE_DOUBLE, UNIT_TIME);


    cpl_image_delete(mask_ima);
    cpl_image_delete(contrib_exptime_map);
    eris_ifu_free_string(&fname);
    eris_ifu_free_string(&pro_catg);
    hdrl_image_delete(cube_collapsed);
    cpl_image_delete(cube_cmap);
    cpl_wcs_delete(wcs3d);
    hdrl_resample_result_delete(result);

    cpl_frameset_delete(cube_set);
    eris_check_error_code("eris_ifu_combine_pbp");

    return cpl_error_get_code();
}

/**@}*/
