
/*********************************************************************
 * E.S.O. - VLT project
 *
 * "@(#) $Id: eris_ifu_vector.c,v 1.12 2013-07-31 10:35:34 aagudo Exp $"
 *
 * Functions that are implemented as recipes. They are also useedd in KMOS
 * data reduction pipeline.
 *
 * who       when        what
 * --------  ----------  ----------------------------------------------
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*------------------------------------------------------------------------------
 *                                  Includes, etc
 *----------------------------------------------------------------------------*/
#define _ISOC99_SOURCE

#include <math.h>

#include <cpl.h>

#include "eris_ifu_error.h"
#include "eris_ifu_vector.h"
#include "eris_ifu_utils.h"
//#include "eris_ifu_math.h"

/* CPL_SORT_ASCENDING is introduced only after CPL 5.3.0 */
#ifndef CPL_SORT_ASCENDING
    #define CPL_SORT_ASCENDING 1
#endif

/*------------------------------------------------------------------------------
 *                                  Definitions
 *----------------------------------------------------------------------------*/
/**
    @brief
        Create a new eris_ifu_vector.

    @param n    Number of elements of the eris_ifu_vector

    @return     1 newly allocated eris_ifu_vector or NULL in case of an error

    The returned object must be deallocated using @li eris_ifu_vector_delete().
    In opposite to cpl_vector_new, the data values are initialised to zero,
    and all values are not marked as rejected

    Possible cpl_error_code set in this function:

    @li CPL_ERROR_ILLEGAL_INPUT if n is negative or zero
*/
eris_ifu_vector* eris_ifu_vector_new(int n)
{
    eris_ifu_vector *ev = NULL;

    cpl_ensure(n >= 1, CPL_ERROR_ILLEGAL_INPUT, NULL);

    TRY
    {
        BRK_IF_NULL(
            ev = (eris_ifu_vector*)cpl_malloc(sizeof(eris_ifu_vector)));

        BRK_IF_NULL(
            ev->data = cpl_vector_new(n));

        BRK_IF_ERROR(
            cpl_vector_fill(ev->data, 0));

        BRK_IF_NULL(
            ev->mask = cpl_vector_new(n));

        BRK_IF_ERROR(
            cpl_vector_fill(ev->mask, 1));
    }
    CATCH
    {
        if (ev != NULL) {
            cpl_vector_delete(ev->data); ev->data = NULL;
            cpl_vector_delete(ev->mask); ev->mask = NULL;
        }
        cpl_free(ev); ev = NULL;
    }

    return ev;
}

/**
    @brief
        Create a new eris_ifu_vector from a vector (duplicated).

    @param n    Number of elements of the eris_ifu_vector
    @param data data

    @return     1 newly allocated eris_ifu_vector or NULL in case of an error

    The returned object must be deallocated using @li eris_ifu_vector_delete().
    In opposite to cpl_vector_new, the data values are initialised to zero,
    and all values are not marked as rejected

    Possible cpl_error_code set in this function:

    @li CPL_ERROR_ILLEGAL_INPUT if n is negative or zero
*/
eris_ifu_vector* eris_ifu_vector_new_wrap(int n, const double *data)
{
    eris_ifu_vector *ev     = NULL;
    cpl_vector      *tmp    = NULL;
    double          *pdata  = NULL,
                    *pmask  = NULL;

    cpl_ensure(n >= 1, CPL_ERROR_ILLEGAL_INPUT, NULL);

    TRY
    {
        BRK_IF_NULL(
            ev = (eris_ifu_vector*)cpl_malloc(sizeof(eris_ifu_vector)));

        BRK_IF_NULL(
            tmp = cpl_vector_wrap(n, (double*) data));
        BRK_IF_NULL(
            ev->data = cpl_vector_duplicate(tmp));
        cpl_vector_unwrap(tmp);

        BRK_IF_NULL(
            ev->mask = cpl_vector_new(n));

        BRK_IF_ERROR(
            cpl_vector_fill(ev->mask, 1));

        BRK_IF_NULL(
            pdata = cpl_vector_get_data(ev->data));
        BRK_IF_NULL(
            pmask = cpl_vector_get_data(ev->mask));
        for (int i = 0; i < n; i++) {
            if (eris_ifu_is_nan_or_inf(pdata[i])) {
                /*pdata[i] = NAN;*/
                pmask[i] = 0.;
            }
        }
    }
    CATCH
    {
        if (ev != NULL) {
            cpl_vector_delete(ev->data); ev->data = NULL;
            cpl_vector_delete(ev->mask); ev->mask = NULL;
        }
        cpl_free(ev); ev = NULL;
    }

    return ev;
}

/**
    @brief
        Create a new eris_ifu_vector out of a data cpl_vector.

    @param data    The cpl_vector to wrap.

    @return        1 newly allocated eris_ifu_vector or NULL in case of an error

    The returned object must be deallocated using @li eris_ifu_vector_delete().
    (Do not deallocate data with cpl_vector_delete()!)

    All NaNs and Infs in @c data will be marked as rejected. The rejected data
    value will remain unchanged.

    Possible cpl_error_code set in this function:

    @li CPL_ERROR_NULL_INPUT if @c data is NULL
*/
eris_ifu_vector* eris_ifu_vector_create(const cpl_vector *data)
{
    eris_ifu_vector *ev     = NULL;
    double          *pdata  = NULL,
                    *pmask  = NULL;
    int             n       = 0;

    cpl_ensure(data, CPL_ERROR_NULL_INPUT, NULL);

    TRY
    {
        BRK_IF_NULL(
            ev = (eris_ifu_vector*)cpl_malloc(sizeof(eris_ifu_vector)));

        BRK_IF_NULL(
            ev->data = cpl_vector_duplicate(data));

        n = cpl_vector_get_size(data);

        BRK_IF_NULL(
            ev->mask = cpl_vector_new(n));

        BRK_IF_ERROR(
            cpl_vector_fill(ev->mask, 1));

        BRK_IF_NULL(
            pdata = cpl_vector_get_data(ev->data));
        BRK_IF_NULL(
            pmask = cpl_vector_get_data(ev->mask));
        for (int i = 0; i < cpl_vector_get_size(data); i++)
        {
            if (eris_ifu_is_nan_or_inf(pdata[i])) {
                /*pdata[i] = NAN;*/
                pmask[i] = 0.;
            }
        }
    }
    CATCH
    {
        if (ev != NULL) {
            cpl_vector_delete(ev->data); ev->data = NULL;
            cpl_vector_delete(ev->mask); ev->mask = NULL;
        }
        cpl_free(ev); ev = NULL;
    }

    return ev;
}

/**
    @brief
        Create a new eris_ifu_vector out of a data and mask cpl_vector.

    @param data    The data cpl_vector to wrap.
    @param mask    The mask cpl_vector to wrap.

    @return        1 newly allocated eris_ifu_vector or NULL in case of an error

    The returned object must be deallocated using @li eris_ifu_vector_delete().
    (Do not deallocate data or mask with cpl_vector_delete()!)

    All NaNs and Infs in @c data will be marked as rejected. The rejected data
    value will remain unchanged.
    All NaNs and Infs in @c mask are set rejected.

    Possible cpl_error_code set in this function:

    @li CPL_ERROR_NULL_INPUT if @c data or mask is NULL
    @li CPL_ERROR_ILLEGAL_INPUT if the size of @c data and mask isn't the same.
*/
eris_ifu_vector* eris_ifu_vector_create2(const cpl_vector *data,
                                         const cpl_vector *mask)
{
    eris_ifu_vector *ev     = NULL;
    double          *pdata  = NULL,
                    *pmask  = NULL;

    cpl_ensure(data, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(mask, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(cpl_vector_get_size(data) == cpl_vector_get_size(mask),
               CPL_ERROR_ILLEGAL_INPUT, NULL);

    TRY
    {
        BRK_IF_NULL(
            ev = (eris_ifu_vector*)cpl_malloc(sizeof(eris_ifu_vector)));

        BRK_IF_NULL(
            ev->data = cpl_vector_duplicate(data));
        BRK_IF_NULL(
            ev->mask = cpl_vector_duplicate(mask));

        BRK_IF_NULL(
            pdata = cpl_vector_get_data(ev->data));
        BRK_IF_NULL(
            pmask = cpl_vector_get_data(ev->mask));

        for (int i = 0; i < cpl_vector_get_size(data); i++)
        {
            if (eris_ifu_is_nan_or_inf(pmask[i])) {
                /*pdata[i] = NAN;*/
                pmask[i] = 0.;
            } else if (pmask[i] >= 0.5) {
                if (eris_ifu_is_nan_or_inf(pdata[i])) {
                    /*pdata[i] = NAN;*/
                    pmask[i] = 0.;
                } else {
                    pmask[i] = 1;
                }
            } else {
                /*pdata[i] = NAN;*/
                pmask[i] = 0;
            }
        }
    }
    CATCH
    {
        if (ev != NULL) {
            cpl_vector_delete(ev->data); ev->data = NULL;
            cpl_vector_delete(ev->mask); ev->mask = NULL;
        }
        cpl_free(ev); ev = NULL;
    }

    return ev;
}

/**
    @brief Delete a eris_ifu_vector

    @param ev    eris_ifu_vector to delete.


    If the eris_ifu_vector ev is NULL, nothing is done and no error is set.
*/
void eris_ifu_vector_delete(eris_ifu_vector *ev)
{
    TRY
    {
        if (ev != NULL) {
            cpl_vector_delete(ev->data); ev->data = NULL;
            cpl_vector_delete(ev->mask); ev->mask = NULL;
            cpl_free(ev);
        }
    }
    CATCH
    {
    }
}

/**
    @brief
        This function duplicates an existing eris_ifu_vector and allocates memory.

    @param ev       Input eris_ifu_vector

    @return         A newly allocated eris_ifu_vector or NULL in case of an error

    The returned object must be deallocated using eris_ifu_vector_delete()

    Possible cpl_error_code set in this function:

    @li CPL_ERROR_NULL_INPUT if @c ev is NULL
*/
eris_ifu_vector* eris_ifu_vector_duplicate(const eris_ifu_vector *ev)
{
    eris_ifu_vector *ev_dup = NULL;

    cpl_ensure(ev, CPL_ERROR_NULL_INPUT, NULL);

    TRY
    {
        BRK_IF_NULL(
            ev_dup = (eris_ifu_vector*)cpl_malloc(sizeof(eris_ifu_vector)));

        ev_dup->data = cpl_vector_duplicate(ev->data);
        CHECK_ERROR_STATE();

        ev_dup->mask = cpl_vector_duplicate(ev->mask);
        CHECK_ERROR_STATE();
    }
    CATCH
    {
        if (ev_dup != NULL) {
            cpl_vector_delete(ev_dup->data); ev_dup->data = NULL;
            cpl_vector_delete(ev_dup->mask); ev_dup->mask = NULL;
        }
        cpl_free(ev_dup); ev_dup = NULL;
    }

    return ev_dup;
}

/**
    @brief
        Set an element of the eris_ifu_vector.

    @param ev       Input eris_ifu_vector
    @param pos 	    The index of the element (0 to nelem-1)
    @param val 	    The value to set in the vector

    @return         A newly allocated eris_ifu_vector or NULL in case of an error

    If the value is Nan or Inf the element is marked as rejected.

    Possible cpl_error_code set in this function:

    @li CPL_ERROR_NULL_INPUT if @c ev is NULL
    @li CPL_ERROR_ACCESS_OUT_OF_RANGE if the specified position is out of the
    vector
*/
cpl_error_code eris_ifu_vector_set(eris_ifu_vector *ev, int pos, double val)
{
    cpl_error_code err = CPL_ERROR_NONE;

    cpl_ensure_code(ev, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code((pos >= 0) && (pos < cpl_vector_get_size(ev->data)),
                    CPL_ERROR_ACCESS_OUT_OF_RANGE);

    TRY
    {
        cpl_vector_set(ev->data, pos, val);
        if (eris_ifu_is_nan_or_inf(val)) {
            cpl_vector_set(ev->mask, pos, 0.);
        } else {
            cpl_vector_set(ev->mask, pos, 1.);
        }
    }
    CATCH
    {
        err = cpl_error_get_code();
    }

    return err;
}

/**
    @brief Get an element of the eris_ifu_vector.

    @param ev       Input eris_ifu_vector
    @param pos 	    The index of the element (0 to nelem-1)
    @param rej      1 if the pixel is bad, 0 if good, negative on error

    @return         The element value

    The value is returned regardless if it is rejected or not! By examining
    @c rej the status of the element can be examined.

    Possible cpl_error_code set in this function:

    @li CPL_ERROR_NULL_INPUT if @c ev is NULL
    @li CPL_ERROR_ACCESS_OUT_OF_RANGE if the specified position is out of the
    vector
*/
double eris_ifu_vector_get_rej(const eris_ifu_vector *ev, int pos, int *rej)
{
    double ret = 0.;

    cpl_ensure_code(ev, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code((pos >= 0) && (pos < cpl_vector_get_size(ev->data)),
                    CPL_ERROR_ACCESS_OUT_OF_RANGE);

    TRY
    {
        ret = cpl_vector_get(ev->data, pos);
        if (rej != NULL) {
            if (cpl_vector_get(ev->mask, pos) > 0.5) {
                /* value isn't rejected */
                *rej = 0;
            } else {
                /* value is rejected */
                *rej = 1;
            }
        }
    }
    CATCH
    {
        ret = 0.;
        if (rej != NULL) {
            *rej = -1;
        }
    }

    return ret;
}

/**
    @brief Get an element of the eris_ifu_vector.

    @param ev       Input eris_ifu_vector
    @param pos 	    The index of the element (0 to nelem-1)

    @return         The element value

    The value is returned regardless if it is rejected or not! By examining
    @c rej the status of the element can be examined.

    Possible cpl_error_code set in this function:

    @li CPL_ERROR_NULL_INPUT if @c ev is NULL
    @li CPL_ERROR_ACCESS_OUT_OF_RANGE if the specified position is out of the
    vector
*/
double eris_ifu_vector_get(const eris_ifu_vector *ev, int pos)
{
    int rej = 0;

    return eris_ifu_vector_get_rej(ev, pos, &rej);
}

/**
    @brief Get a copy of the mask of eris_ifu_vector.

    @param ev       Input eris_ifu_vector

    @return         The mask of the input vector

    The output vector contains 1 for non-rejected elements and 0 for rejected
    elements.

    Possible cpl_error_code set in this function:

    @li CPL_ERROR_NULL_INPUT if @c ev is NULL
    @li CPL_ERROR_ACCESS_OUT_OF_RANGE if the specified position is out of the
    vector
*/
cpl_vector* eris_ifu_vector_get_mask(const eris_ifu_vector *ev)
{
    cpl_vector *mask = NULL;

    cpl_ensure(ev, CPL_ERROR_NULL_INPUT, NULL);

    TRY
    {
        BRK_IF_NULL(
            mask = cpl_vector_duplicate(ev->mask));
    }
    CATCH
    {
        cpl_vector_delete(mask); mask = NULL;
    }

    return mask;
}

/**
    @brief Get a copy of the data, rejected values are set to NaN.

    @param ev       Input eris_ifu_vector

    @return         A cpl_vector input vector

    The output vector contains NaN for rejected elements.

    Possible cpl_error_code set in this function:

    @li CPL_ERROR_NULL_INPUT if @c ev is NULL
    @li CPL_ERROR_ACCESS_OUT_OF_RANGE if the specified position is out of the
    vector
*/
cpl_vector* eris_ifu_vector_get_data(const eris_ifu_vector *ev)
{
    cpl_ensure(ev, CPL_ERROR_NULL_INPUT, NULL);

    cpl_vector *data = NULL;

    TRY
    {
        BRK_IF_NULL(
            data = cpl_vector_duplicate(ev->data));
        for (int i = 0; i < eris_ifu_vector_get_size(ev); i++) {
            if (eris_ifu_vector_is_rejected(ev, i)) {
                BRK_IF_ERROR(
                    cpl_vector_set(data, i, NAN));
            }
        }
    }
    CATCH
    {
        cpl_vector_delete(data); data = NULL;
    }
    return data;
}

/**
    @brief Get the pointer to the mask of the eris_ifu_vector.

    @param ev       Input eris_ifu_vector

    @return         Pointer to the mask of the input vector

    The output vector contains 1 for non-rejected elements and 0 for rejected
    elements.

    Possible cpl_error_code set in this function:

    @li CPL_ERROR_NULL_INPUT if @c ev is NULL
    @li CPL_ERROR_ACCESS_OUT_OF_RANGE if the specified position is out of the
    vector
*/
cpl_vector* eris_ifu_vector_get_bpm(eris_ifu_vector *ev)
{
    cpl_vector *mask = NULL;

    cpl_ensure(ev, CPL_ERROR_NULL_INPUT, NULL);

    TRY
    {
        BRK_IF_NULL(
            mask = ev->mask);
    }
    CATCH
    {
        mask = NULL;
    }

    return mask;
}

/**
    @brief Set the rejected elements in an eris_ifu_vector as defined in a mask.

    @param ev       Input eris_ifu_vector
    @param mask     The mask defining the bad elements.
    @param keep     TRUE: keep already rejected values and just update the
                    non-rejected ones. FALSE: apply the supplied @c mask.

    @return         the @li _cpl_error_code_ or CPL_ERROR_NONE

    Possible cpl_error_code set in this function:

    @li CPL_ERROR_NULL_INPUT if @c ev or @c mask is NULL
    @li CPL_ERROR_ILLEGAL_INPUT if @c ev and @c mask don't have same length
*/
cpl_error_code eris_ifu_vector_reject_from_mask(eris_ifu_vector *ev,
                                               const cpl_vector *mask,
                                               int keep)
{
    int             ret         = CPL_ERROR_NONE,
                    size        = 0;
    double          *pkvmask    = NULL;
    const double    *pmask      = NULL;

    cpl_ensure_code(ev, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(mask, CPL_ERROR_NULL_INPUT);

    size = cpl_vector_get_size(ev->data);

    cpl_ensure_code(size == cpl_vector_get_size(mask), CPL_ERROR_ILLEGAL_INPUT);
    cpl_ensure_code((keep == 0) || (keep == 1), CPL_ERROR_ILLEGAL_INPUT);

    TRY
    {
        BRK_IF_NULL(
            pkvmask = cpl_vector_get_data(ev->mask));
        BRK_IF_NULL(
            pmask = cpl_vector_get_data_const(mask));

        for (int i = 0; i < size; i++) {
            if ((!keep) ||
                ((keep) && (pkvmask[i] > 0.5)))
            {
                pkvmask[i] = pmask[i];
            }
        }
        CHECK_ERROR_STATE();
    }
    CATCH
    {
        ret = cpl_error_get_code();
    }

    return ret;
}

/**
    @brief
        Count the number of rejected elements in a eris_ifu_vector.

    @param ev    The input eris_ifu_vector

    @return      The number of rejected values or -1 if the input vector is NULL

    Possible cpl_error_code set in this function:

    @li CPL_ERROR_NULL_INPUT if @c ev is NULL
*/
int eris_ifu_vector_count_rejected(const eris_ifu_vector *ev)
{
    int     cnt     = 0;
    double  *pmask  = NULL;

    cpl_ensure_code(ev, CPL_ERROR_NULL_INPUT);

    TRY
    {
        BRK_IF_NULL(
            pmask = cpl_vector_get_data(ev->mask));
        for (int i = 0; i < cpl_vector_get_size(ev->mask); i++) {
            if (pmask[i] == 0) {
                cnt++;
            }
        }
    }
    CATCH
    {
        cnt = -1;
    }

    return cnt;
}

/**
    @brief
        Count the number of non-rejected elements in a eris_ifu_vector.

    @param ev    The input eris_ifu_vector

    @return      The number of rejected values or -1 if the input vector is NULL

    Possible cpl_error_code set in this function:

    @li CPL_ERROR_NULL_INPUT if @c ev is NULL
*/
int eris_ifu_vector_count_non_rejected(const eris_ifu_vector *ev)
{
    int     cnt     = 0,
            size    = 0;
    double  *pmask  = NULL;

    cpl_ensure_code(ev, CPL_ERROR_NULL_INPUT);

    TRY
    {
        size = cpl_vector_get_size(ev->data);

        BRK_IF_NULL(
            pmask = cpl_vector_get_data(ev->mask));
        for (int i = 0; i < size; i++) {
            if (pmask[i] == 0) {
                cnt++;
            }
        }

        cnt = size - cnt;
    }
    CATCH
    {
        cnt = -1;
    }

    return cnt;
}

/**
    @brief
        Test if a value is good or bad.

    @param ev    The input eris_ifu_vector
    @param n     The position of the value in the vector (first value is 0)

    @return      1 if the value is rejected, 0 if it is good, -1 on error

    Possible cpl_error_code set in this function:

    @li CPL_ERROR_NULL_INPUT if @c ev is NULL
    @li CPL_ERROR_ACCESS_OUT_OF_RANGE if the specified position is out of the
    vector
*/
int eris_ifu_vector_is_rejected(const eris_ifu_vector *ev, int n)
{
    int ret = 0;

    cpl_ensure_code(ev, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code((n >= 0) && (n < cpl_vector_get_size(ev->data)),
                    CPL_ERROR_ACCESS_OUT_OF_RANGE);

    TRY
    {
        if (cpl_vector_get(ev->mask, n) > 0.5) {
            ret = FALSE;
        } else {
            ret = TRUE;
        }
    }
    CATCH
    {
        ret = -1;
    }

    return ret;
}

/**
    @brief
        Set a value as rejected in a eris_ifu_vector.

    @param ev    The input eris_ifu_vector
    @param n     The position of the value in the vector (first value is 0)

    @return      the @li _cpl_error_code_ or CPL_ERROR_NONE

    Possible cpl_error_code set in this function:

    @li CPL_ERROR_NULL_INPUT if @c ev is NULL
    @li CPL_ERROR_ACCESS_OUT_OF_RANGE if the specified position is out of the
    vector
*/
cpl_error_code eris_ifu_vector_reject(eris_ifu_vector *ev, int n)
{
    int ret = CPL_ERROR_NONE;

    cpl_ensure_code(ev, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code((n >= 0) && (n < cpl_vector_get_size(ev->data)),
                    CPL_ERROR_ILLEGAL_INPUT);

    TRY
    {
/*        cpl_vector_set(ev->data, n, NAN);*/
        cpl_vector_set(ev->mask, n, 0);
        CHECK_ERROR_STATE();
    }
    CATCH
    {
        ret = cpl_error_get_code();
    }

    return ret;
}

/**
    @brief
        Extract a sub_vector from a eris_ifu_vector.

    @param ev       The input eris_ifu_vector
    @param istart 	Start index (from 0 to number of elements - 1)
    @param istop 	Stop index (from 0 to number of elements - 1)

    @return      A newly allocated eris_ifu_vector or NULL in case of an error

    Rejected elements are also extracted.

    Possible cpl_error_code set in this function:

    @li CPL_ERROR_NULL_INPUT if @c ev is NULL
    @li CPL_ERROR_ILLEGAL_INPUT if istop <= istart
*/
eris_ifu_vector* eris_ifu_vector_extract(const eris_ifu_vector *ev,
                                       int istart,
                                       int istop)
{
    eris_ifu_vector *ev_out = NULL;

    cpl_ensure(ev, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(istop > istart, CPL_ERROR_ILLEGAL_INPUT, NULL);

    TRY
    {
        BRK_IF_NULL(
            ev_out = (eris_ifu_vector*)cpl_malloc(sizeof(eris_ifu_vector)));

        BRK_IF_NULL(
            ev_out->data = cpl_vector_extract(ev->data, istart, istop, 1));

        BRK_IF_NULL(
            ev_out->mask = cpl_vector_extract(ev->mask, istart, istop, 1));
    }
    CATCH
    {
        if (ev_out != NULL) {
            cpl_vector_delete(ev_out->data); ev_out->data = NULL;
            cpl_vector_delete(ev_out->mask); ev_out->mask = NULL;
        }
        cpl_free(ev_out); ev_out = NULL;
    }

    return ev_out;
}

/**
    @brief
        Creates a cpl_vector out of a eris_ifu_vector with non-rejected values.

    @param ev    The input eris_ifu_vector

    @return      The extracted cpl_vector or NULL.

    The length of the returned cpl_vector may be, depending on the number of
    rejected values shorter than the input eris_ifu_vector!
    If all values are rejected NULL is returned but no error is set.

    Possible cpl_error_code set in this function:

    @li CPL_ERROR_NULL_INPUT if @c ev is NULL
*/
cpl_vector* eris_ifu_vector_create_non_rejected(const eris_ifu_vector *ev)
{
    int             cnt         = 0,
                    j           = 0,
                    n           = 0;
    const double    *pkvmask    = NULL,
                    *pkvdata    = NULL;
    double          *pret       = NULL;
    cpl_vector      *ret        = NULL;

    cpl_ensure(ev, CPL_ERROR_NULL_INPUT, NULL);

    TRY
    {
        n = cpl_vector_get_size(ev->data);
        cnt = eris_ifu_vector_count_rejected(ev);
        CHECK_ERROR_STATE();

        if (n-cnt > 0) {
            BRK_IF_NULL(
                ret = cpl_vector_new(n-cnt));
            BRK_IF_NULL(
                pret = cpl_vector_get_data(ret));
            BRK_IF_NULL(
                pkvdata = cpl_vector_get_data_const(ev->data));
            BRK_IF_NULL(
                pkvmask = cpl_vector_get_data_const(ev->mask));
            for (int i = 0; i < n; i++) {
                if (pkvmask[i] > 0.5) {
                    pret[j++] = pkvdata[i];
                }
            }
        }
    }
    CATCH
    {
        cpl_vector_delete(ret); ret = NULL;
    }

    return ret;
}

/**
    @brief
        Assert that rejected values on both vectors are the same.

    @param kv1   1st input eris_ifu_vector
    @param kv2   2nd input eris_ifu_vector

    @return      CPL_ERROR_NONE or error code.

    It is asserted that all rejected elements in @c kv1 are also rejected in
    @c kv2 and vice versa.

    Possible cpl_error_code set in this function:

    @li CPL_ERROR_NULL_INPUT if any of the inputs is NULL
    @li CPL_ERROR_ILLEGAL_INPUT if the inputs don't have the same length
*/
cpl_error_code eris_ifu_vector_adapt_rejected(eris_ifu_vector *kv1,
                                              eris_ifu_vector *kv2)
{
    cpl_error_code  err         = CPL_ERROR_NONE;
    int             size        = 0;
    double          *pkv1mask   = NULL,
                    *pkv2mask   = NULL;

    cpl_ensure_code(kv1, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(kv2, CPL_ERROR_NULL_INPUT);

    size = cpl_vector_get_size(kv1->data);

    cpl_ensure_code(size == cpl_vector_get_size(kv2->data),
                    CPL_ERROR_ILLEGAL_INPUT);

    TRY
    {
        BRK_IF_NULL(
            pkv1mask = cpl_vector_get_data(kv1->mask));
        BRK_IF_NULL(
            pkv2mask = cpl_vector_get_data(kv2->mask));

        for (int i = 0; i < size; i++)  {
            if (pkv1mask[i] < 0.5) {
                pkv2mask[i] = 0.;
            } else if (pkv2mask[i] < 0.5) {
                pkv1mask[i] = 0.;
            }
        }
    }
    CATCH
    {
        err = cpl_error_get_code();
    }

    return err;
}

/**
    @brief
        Add a eris_ifu_vector to another.

    @param kv1      First eris_ifu_vector (modified)
    @param kv2      Second eris_ifu_vector

    @return         CPL_ERROR_NONE or the relevant @li _cpl_error_code_ on error

    The second vector is added to the first one. The input first vector is modified.

    The input vectors must have the same size.

    Possible cpl_error_code set in this function:

    @li CPL_ERROR_NULL_INPUT if any input pointer is NULL
    @li CPL_ERROR_INCOMPATIBLE_INPUT if kv1 and kv2 have different sizes
*/
cpl_error_code eris_ifu_vector_add(eris_ifu_vector *kv1, const eris_ifu_vector *kv2)
{
    int             size        = 0;
    double          *pkv1data   = NULL,
                    *pkv1mask   = NULL,
                    *pkv2data   = NULL,
                    *pkv2mask   = NULL;
    cpl_error_code  err         = CPL_ERROR_NONE;

    cpl_ensure_code(kv1, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(kv2, CPL_ERROR_NULL_INPUT);

    size = cpl_vector_get_size(kv1->data);

    cpl_ensure_code(size == cpl_vector_get_size(kv2->data),
                    CPL_ERROR_ILLEGAL_INPUT);

    TRY
    {
        BRK_IF_NULL(
            pkv1data = cpl_vector_get_data(kv1->data));
        BRK_IF_NULL(
            pkv1mask = cpl_vector_get_data(kv1->mask));
        BRK_IF_NULL(
            pkv2data = cpl_vector_get_data(kv2->data));
        BRK_IF_NULL(
            pkv2mask = cpl_vector_get_data(kv2->mask));

        for (int i = 0; i < size; i++) {
            if ((pkv1mask[i] > 0.5) && (pkv2mask[i] > 0.5)) {
                /* add elements (if element in kv1 is rejected it stays anyway
                   rejected) */
                pkv1data[i] += pkv2data[i];

                if (eris_ifu_is_nan_or_inf(pkv1data[i])) {
                    /*pkvdata[i] = NAN;*/
                    pkv1mask[i] = 0.;
                }
            } else {
                /* element of kv2 is rejected, reject also element in kv1 */
                pkv1mask[i] = 0.;
            }
        }
    }
    CATCH
    {
        err = cpl_error_get_code();
    }

    return err;
}

/**
    @brief
        Subtract two eris_ifu_vectors.

    @param kv1  First eris_ifu_vector (modified)
    @param kv2 	Second eris_ifu_vector

    @return     CPL_ERROR_NONE or the relevant @li _cpl_error_code_ on error

    The second vector is subtracted from the first one. The input first vector
    is modified.

    The input vectors must have the same size.

    Possible cpl_error_code set in this function:

    @li CPL_ERROR_NULL_INPUT if any input pointer is NULL
    @li CPL_ERROR_INCOMPATIBLE_INPUT if kv1 and kv2 have different sizes
*/
cpl_error_code eris_ifu_vector_subtract(eris_ifu_vector *kv1,
                                        const eris_ifu_vector *kv2)
{
    int             size        = 0;
    double          *pkv1data   = NULL,
                    *pkv1mask   = NULL,
                    *pkv2data   = NULL,
                    *pkv2mask   = NULL;
    cpl_error_code  err         = CPL_ERROR_NONE;

    cpl_ensure_code(kv1, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(kv2, CPL_ERROR_NULL_INPUT);

    size = cpl_vector_get_size(kv1->data);

    cpl_ensure_code(size == cpl_vector_get_size(kv2->data),
                    CPL_ERROR_ILLEGAL_INPUT);

    TRY
    {
        BRK_IF_NULL(
            pkv1data = cpl_vector_get_data(kv1->data));
        BRK_IF_NULL(
            pkv1mask = cpl_vector_get_data(kv1->mask));
        BRK_IF_NULL(
            pkv2data = cpl_vector_get_data(kv2->data));
        BRK_IF_NULL(
            pkv2mask = cpl_vector_get_data(kv2->mask));

        for (int i = 0; i < size; i++) {
            if ((pkv1mask[i] > 0.5) && (pkv2mask[i] > 0.5)) {
                /* subtract elements (if element in kv1 is rejected it stays anyway
                   rejected) */
                pkv1data[i] -= pkv2data[i];

                if (eris_ifu_is_nan_or_inf(pkv1data[i])) {
                    /*pkvdata[i] = NAN;*/
                    pkv1mask[i] = 0.;
                }
            } else {
                /* element of kv2 is rejected, reject also element in kv1 */
                pkv1mask[i] = 0.;
            }
        }
    }
    CATCH
    {
        err = cpl_error_get_code();
    }

    return err;
}

/**
    @brief
        Multiply two eris_ifu_vectors.

    @param kv1  First eris_ifu_vector (modified)
    @param kv2 	Second eris_ifu_vector

    @return         CPL_ERROR_NONE or the relevant @li _cpl_error_code_ on error

    The input first vector is modified.

    The input vectors must have the same size.

    Possible cpl_error_code set in this function:

    @li CPL_ERROR_NULL_INPUT if any input pointer is NULL
    @li CPL_ERROR_INCOMPATIBLE_INPUT if kv1 and kv2 have different sizes
*/
cpl_error_code eris_ifu_vector_multiply(eris_ifu_vector *kv1, const eris_ifu_vector *kv2)
{
    int             size        = 0;
    double          *pkv1data   = NULL,
                    *pkv1mask   = NULL,
                    *pkv2data   = NULL,
                    *pkv2mask   = NULL;
    cpl_error_code  err         = CPL_ERROR_NONE;

    cpl_ensure_code(kv1, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(kv2, CPL_ERROR_NULL_INPUT);

    size = cpl_vector_get_size(kv1->data);

    cpl_ensure_code(size == cpl_vector_get_size(kv2->data),
                    CPL_ERROR_ILLEGAL_INPUT);

    TRY
    {

        BRK_IF_NULL(
            pkv1data = cpl_vector_get_data(kv1->data));
        BRK_IF_NULL(
            pkv1mask = cpl_vector_get_data(kv1->mask));
        BRK_IF_NULL(
            pkv2data = cpl_vector_get_data(kv2->data));
        BRK_IF_NULL(
            pkv2mask = cpl_vector_get_data(kv2->mask));

        for (int i = 0; i < size; i++) {
            if ((pkv1mask[i] > 0.5) && (pkv2mask[i] > 0.5)) {
                /* multiply elements (if element in kv1 is rejected it stays anyway
                   rejected) */
                pkv1data[i] *= pkv2data[i];

                if (eris_ifu_is_nan_or_inf(pkv1data[i])) {
                    /*pkvdata[i] = NAN;*/
                    pkv1mask[i] = 0.;
                }
            } else {
                /* element of kv2 is rejected, reject also element in kv1 */
                pkv1mask[i] = 0.;
            }
        }
    }
    CATCH
    {
        err = cpl_error_get_code();
    }

    return err;
}

/**
    @brief
        Divide two eris_ifu_vectors element-wise.

    @param kv1  First eris_ifu_vector (modified)
    @param kv2 	Second eris_ifu_vector

    @return     CPL_ERROR_NONE or the relevant @li _cpl_error_code_ on error

    The first vector is divided by the second one. In opposite to
    cpl_vector_divide(), a division by zero doesn't throw an error but results
    in a rejected element of the vector

    The input vectors must have the same size.

    Possible cpl_error_code set in this function:

    @li CPL_ERROR_NULL_INPUT if any input pointer is NULL
    @li CPL_ERROR_INCOMPATIBLE_INPUT if kv1 and kv2 have different sizes
*/
cpl_error_code eris_ifu_vector_divide(eris_ifu_vector *kv1, const eris_ifu_vector *kv2)
{
    int             size        = 0;
    double          *pkv1data   = NULL,
                    *pkv1mask   = NULL,
                    *pkv2data   = NULL,
                    *pkv2mask   = NULL;
    cpl_error_code  err         = CPL_ERROR_NONE;

    cpl_ensure_code(kv1, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(kv2, CPL_ERROR_NULL_INPUT);

    size = cpl_vector_get_size(kv1->data);

    cpl_ensure_code(size == cpl_vector_get_size(kv2->data),
                    CPL_ERROR_ILLEGAL_INPUT);

    TRY
    {
        BRK_IF_NULL(
            pkv1data = cpl_vector_get_data(kv1->data));
        BRK_IF_NULL(
            pkv1mask = cpl_vector_get_data(kv1->mask));
        BRK_IF_NULL(
            pkv2data = cpl_vector_get_data(kv2->data));
        BRK_IF_NULL(
            pkv2mask = cpl_vector_get_data(kv2->mask));

        for (int i = 0; i < size; i++) {
            if ((pkv1mask[i] > 0.5) && (pkv2mask[i] > 0.5)) {
                /* divide elements (if element in kv1 is rejected it stays anyway
                   rejected) */
                pkv1data[i] /= pkv2data[i];

                if (eris_ifu_is_nan_or_inf(pkv1data[i])) {
                    /*pkvdata[i] = NAN;*/
                    pkv1mask[i] = 0.;
                }
            } else {
                /* element of kv2 is rejected, reject also element in kv1 */
                pkv1mask[i] = 0.;
            }
        }
    }
    CATCH
    {
        err = cpl_error_get_code();
    }

    return err;
}

/**
    @brief
        Elementwise addition of a scalar to a eris_ifu_vector.

    @param ev       eris_ifu_vector to modify
    @param addend 	Number to add

    @return         CPL_ERROR_NONE or the relevant @li _cpl_error_code_ on error

    Add a number to each element of the eris_ifu_vector.

    Possible cpl_error_code set in this function:

    @li CPL_ERROR_NULL_INPUT if @c ev is NULL
*/
cpl_error_code eris_ifu_vector_add_scalar(eris_ifu_vector *ev, double addend)
{
    int             n           = 0;
    double          *pkvmask    = NULL,
                    *pkvdata    = NULL;
    cpl_error_code  err         = CPL_ERROR_NONE;

    cpl_ensure_code(ev, CPL_ERROR_NULL_INPUT);

    TRY
    {
        n = cpl_vector_get_size(ev->data);
        CHECK_ERROR_STATE();

        BRK_IF_NULL(
            pkvdata = cpl_vector_get_data(ev->data));
        BRK_IF_NULL(
            pkvmask = cpl_vector_get_data(ev->mask));
        for (int i = 0; i < n; i++) {
            if (pkvmask[i] > 0.5) {
                pkvdata[i] += addend;
            }
            if (eris_ifu_is_nan_or_inf(pkvdata[i])) {
                /*pkvdata[i] = NAN;*/
                pkvmask[i] = 0.;
            }
        }
    }
    CATCH
    {
        err = cpl_error_get_code();
    }

    return err;
}

/**
    @brief
        Elementwise subtraction of a scalar from a eris_ifu_vector.

    @param ev           eris_ifu_vector to modify
    @param subtrahend 	Number to subtract

    @return         CPL_ERROR_NONE or the relevant @li _cpl_error_code_ on error

    Subtract a number from each element of the eris_ifu_vector.

    Possible cpl_error_code set in this function:

    @li CPL_ERROR_NULL_INPUT if @c ev is NULL
*/
cpl_error_code eris_ifu_vector_subtract_scalar(eris_ifu_vector *ev,
                                              double subtrahend)
{
    int             n           = 0;
    double          *pkvmask    = NULL,
                    *pkvdata    = NULL;
    cpl_error_code  err         = CPL_ERROR_NONE;

    cpl_ensure_code(ev, CPL_ERROR_NULL_INPUT);

    TRY
    {
        n = cpl_vector_get_size(ev->data);
        CHECK_ERROR_STATE();

        BRK_IF_NULL(
            pkvdata = cpl_vector_get_data(ev->data));
        BRK_IF_NULL(
            pkvmask = cpl_vector_get_data(ev->mask));
        for (int i = 0; i < n; i++) {
            if (pkvmask[i] > 0.5) {
                pkvdata[i] -= subtrahend;
            }
            if (eris_ifu_is_nan_or_inf(pkvdata[i])) {
                /*pkvdata[i] = NAN;*/
                pkvmask[i] = 0.;
            }
        }
    }
    CATCH
    {
        err = cpl_error_get_code();
    }

    return err;
}

/**
    @brief
        Elementwise multiplication of a scalar to a eris_ifu_vector.

    @param ev       eris_ifu_vector to modify
    @param factor 	Number to multiply with

    @return         CPL_ERROR_NONE or the relevant @li _cpl_error_code_ on error

    Multiply each element of the eris_ifu_vector with a number.

    Possible cpl_error_code set in this function:

    @li CPL_ERROR_NULL_INPUT if @c ev is NULL
*/
cpl_error_code eris_ifu_vector_multiply_scalar(eris_ifu_vector *ev, double factor)
{
    int             n           = 0;
    double          *pkvmask    = NULL,
                    *pkvdata    = NULL;
    cpl_error_code  err         = CPL_ERROR_NONE;

    cpl_ensure_code(ev, CPL_ERROR_NULL_INPUT);

    TRY
    {
        n = cpl_vector_get_size(ev->data);
        CHECK_ERROR_STATE();

        BRK_IF_NULL(
            pkvdata = cpl_vector_get_data(ev->data));
        BRK_IF_NULL(
            pkvmask = cpl_vector_get_data(ev->mask));
        for (int i = 0; i < n; i++) {
            if (pkvmask[i] > 0.5) {
                pkvdata[i] *= factor;
            }
            if (eris_ifu_is_nan_or_inf(pkvdata[i])) {
                /*pkvdata[i] = NAN;*/
                pkvmask[i] = 0.;
            }
        }
    }
    CATCH
    {
        err = cpl_error_get_code();
    }

    return err;
}

/**
    @brief
        Elementwise division of a eris_ifu_vector and a scalar.

    @param ev       eris_ifu_vector to modify
    @param dividend Number to divide with

    @return         CPL_ERROR_NONE or the relevant @li _cpl_error_code_ on error

    Divide each element of the eris_ifu_vector by a number.

    Possible cpl_error_code set in this function:

    @li CPL_ERROR_NULL_INPUT if @c ev is NULL
*/
cpl_error_code eris_ifu_vector_divide_scalar(eris_ifu_vector *ev, double dividend)
{
    int             n           = 0;
    double          *pkvmask    = NULL,
                    *pkvdata    = NULL;
    cpl_error_code  err         = CPL_ERROR_NONE;

    cpl_ensure_code(ev, CPL_ERROR_NULL_INPUT);

    TRY
    {
        n = cpl_vector_get_size(ev->data);
        CHECK_ERROR_STATE();

        BRK_IF_NULL(
            pkvdata = cpl_vector_get_data(ev->data));
        BRK_IF_NULL(
            pkvmask = cpl_vector_get_data(ev->mask));
        for (int i = 0; i < n; i++) {
            if (pkvmask[i] > 0.5) {
                pkvdata[i] /= dividend;
            }
            if (eris_ifu_is_nan_or_inf(pkvdata[i])) {
                /*pkvdata[i] = NAN;*/
                pkvmask[i] = 0.;
            }
        }
    }
    CATCH
    {
        err = cpl_error_get_code();
    }

    return err;
}

/**
    @brief
        Calculates the absolute of an vector inplace.

    @param ev    The vector.

    @return
        The function returns CPL_ERROR_NONE on success or a CPL error code
        otherwise.

    The vector is checked elementwise if the data is negative. If so its sign
    is changed.

    Possible cpl_error_code set in this function:

    @li CPL_ERROR_NULL_INPUT    if @c data is NULL.
*/
cpl_error_code eris_ifu_vector_abs(eris_ifu_vector *ev)
{
    cpl_error_code  ret_error   = CPL_ERROR_NONE;
    double          *pkvdata    = NULL,
                    *pkvmask    = NULL;

    cpl_ensure_code(ev, CPL_ERROR_NULL_INPUT);

    TRY
    {
        BRK_IF_NULL(
            pkvdata = cpl_vector_get_data(ev->data));
        BRK_IF_NULL(
            pkvmask = cpl_vector_get_data(ev->mask));

        for (int i = 0; i < cpl_vector_get_size(ev->data);i++) {
            if ((pkvmask[i] > 0.5) && (pkvdata[i] < 0.0)) {
                pkvdata[i] = -pkvdata[i];
            }
        }
    }
    CATCH
    {
        ret_error = cpl_error_get_code();
    }

    return ret_error;
}

/**
    @brief
        Get the size of the eris_ifu_vector.

    @param ev       Input eris_ifu_vector

    @return The size or -1 in case of an error

    Possible cpl_error_code set in this function:

    @li CPL_ERROR_NULL_INPUT if @c ev is NULL
*/
int eris_ifu_vector_get_size(const eris_ifu_vector *ev)
{
    int ret_val = 0;

    cpl_ensure_code(ev, CPL_ERROR_NULL_INPUT);

    TRY
    {
        ret_val = cpl_vector_get_size(ev->data);
        CHECK_ERROR_STATE();
    }
    CATCH
    {
        ret_val = -1;
    }

    return ret_val;
}

/**
    @brief
        Compute the mean value of non-rejected eris_ifu_vector elements.

    @param ev       Input eris_ifu_vector

    @return Mean value of vector elements or 0 on error or when all values are
            rejected.

    Possible cpl_error_code set in this function:

    @li CPL_ERROR_NULL_INPUT if @c ev is NULL
*/
double eris_ifu_vector_get_mean(const eris_ifu_vector *ev)
{
    cpl_vector  *vec    = NULL;
    double      val     = 0.;

    cpl_ensure_code(ev, CPL_ERROR_NULL_INPUT);

    TRY
    {
        vec = eris_ifu_vector_create_non_rejected(ev);
        if (vec != NULL) {
            val = cpl_vector_get_mean(vec);
            CHECK_ERROR_STATE();
        }
    }
    CATCH
    {
        val = 0.;
    }
    cpl_vector_delete(vec); vec = NULL;

    return val;
}

/**
    @brief
        Compute the median of the elements of a vector.

    @param ev    Input Vector.
    @param type  The method to calculate ther median with:
                 KMCLIPM_STATISTICAL:
                 The returned value is the lower of the two center elements of
                 the sorted input vector.
                 KMCLIPM_ARITHMETIC: The arithmetic mean of the two center
                 elements of the sorted input vector is calculated.

    @return Median value of the vector elements or 0 on error or when all values
            are rejected.

    Unlike @li cpl_vector_get_median this function doesn't modify the order of
    the elements of the input vector!

  Possible cpl_error_code set in this function:
  @li CPL_ERROR_NULL_INPUT    if @c vec is NULL.
*/
double eris_ifu_vector_get_median(const eris_ifu_vector *ev,
                                 const enum medianType type)
{
    cpl_vector  *vec    = NULL;
    double      val     = 0.;
    int         size    = 0;

    cpl_ensure_code(ev, CPL_ERROR_NULL_INPUT);

    TRY
    {
        vec = eris_ifu_vector_create_non_rejected(ev);

        if (vec != NULL) {
            size = cpl_vector_get_size(vec);
            if ((type == ERIS_IFU_STATISTICAL) && ((size % 2) == 0)) {
                /* statistical method and even size */
                cpl_vector_sort(vec, CPL_SORT_ASCENDING);

                val = cpl_vector_get(vec, size/2 - 1);
            } else {
                /* arithmetic method or uneven size */
                val = cpl_vector_get_median(vec);
            }
        }
    }
    CATCH
    {
        val = 0.;
    }
    cpl_vector_delete(vec); vec = NULL;

    return val;
}

/**
    @brief
        Remove a certain percentage of brightest pixels.

    @param ev            Input Vector.
    @param percentage    The percentage of brightest pixels to remove (between 0 and 1).

    @return The cut vector.

  Possible cpl_error_code set in this function:
  @li CPL_ERROR_NULL_INPUT    if @c vec is NULL.
*/
eris_ifu_vector* eris_ifu_vector_cut_percentian(const eris_ifu_vector *ev,
                                             double percentage)
{
    eris_ifu_vector  *ret_vec   = NULL;
    cpl_vector      *vec        = NULL,
                    *cut_vec    = NULL;
    int             size        = 0;

    cpl_ensure(ev, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure((percentage >= 0.0) && (percentage < 1.0),
               CPL_ERROR_ILLEGAL_INPUT, NULL);

    TRY
    {
        vec = eris_ifu_vector_create_non_rejected(ev);

        if (vec != NULL) {
            /* sort vector */
            cpl_vector_sort(vec, CPL_SORT_ASCENDING);
            CHECK_ERROR_STATE();

            /* cut percentage wanted */
            size = cpl_vector_get_size(vec);
            cpl_size tmp = (cpl_size)rint((1.-percentage)*(double)size-1.);
            BRK_IF_NULL(
                cut_vec = cpl_vector_extract(vec, 0, tmp, 1));
            BRK_IF_NULL(
                ret_vec = eris_ifu_vector_create(cut_vec));
            cpl_vector_delete(cut_vec);
        }
    }
    CATCH
    {
        eris_ifu_vector_delete(ret_vec); ret_vec = NULL;
    }
    cpl_vector_delete(vec); vec = NULL;

    return ret_vec;
}

/**
    @brief
        Compute the sum of non-rejected eris_ifu_vector elements.

    @param ev       Input eris_ifu_vector

    @return         Sum of vector elements or 0 on error or when all values are
                    rejected.

    Possible cpl_error_code set in this function:

    @li CPL_ERROR_NULL_INPUT if @c ev is NULL
*/
double eris_ifu_vector_get_sum(const eris_ifu_vector *ev)
{
    int             n           = 0;
    double          sum         = 0,
                    *pkvmask    = NULL,
                    *pkvdata    = NULL;

    cpl_ensure_code(ev, CPL_ERROR_NULL_INPUT);

    TRY
    {
        n = cpl_vector_get_size(ev->data);
        CHECK_ERROR_STATE();

        BRK_IF_NULL(
            pkvdata = cpl_vector_get_data(ev->data));
        BRK_IF_NULL(
            pkvmask = cpl_vector_get_data(ev->mask));
        for (int i = 0; i < n; i++) {
            if (pkvmask[i] > 0.5) {
                sum += pkvdata[i];
            }
        }
    }
    CATCH
    {
        sum = 0;
    }

    return sum;
}

/**
    @brief
        Compute the bias-corrected standard deviation of a vectors elements.

    @param ev       Input eris_ifu_vector

    @return         standard deviation of the elements or -1 on error.

    S(n-1) = sqrt[sum((xi-mean)^2) / (n-1)] (i=1 -> n)

    The length of @c ev must be at least 2.

    Possible cpl_error_code set in this function:

    @li CPL_ERROR_NULL_INPUT if @c ev is NULL
    @li CPL_ERROR_ILLEGAL_INPUT if the number of non-rejected values in @c ev is
    less than 2
*/
double eris_ifu_vector_get_stdev(const eris_ifu_vector *ev)
{
    cpl_vector  *vec    = NULL;
    double      val     = 0.;

    cpl_ensure_code(ev, CPL_ERROR_NULL_INPUT);

    TRY
    {
        vec = eris_ifu_vector_create_non_rejected(ev);
        if (vec != NULL) {
            val = cpl_vector_get_stdev(vec);
            CHECK_ERROR_STATE();
        }
    }
    CATCH
    {
        val = -1.;
    }
    cpl_vector_delete(vec); vec = NULL;

    return val;
}

/**
    @brief
        Compute the bias-corrected standard deviation of a vectors elements using median.

    @param ev       Input eris_ifu_vector

    @return         standard deviation using median of the elements or -1 on error.

    S(n-1) = sqrt[sum((xi-median)^2) / (n-1)] (i=1 -> n)

    The length of @c ev must be at least 2.

    Possible cpl_error_code set in this function:

    @li CPL_ERROR_NULL_INPUT if @c ev is NULL
    @li CPL_ERROR_ILLEGAL_INPUT if the number of non-rejected values in @c ev is
    less than 2
*/
double eris_ifu_vector_get_stdev_median(const eris_ifu_vector *ev)
{
    int             n           = 0;
    double          stdev       = 0,
                    median      = 0,
                    *pkvmask    = NULL,
                    *pkvdata    = NULL;

    cpl_ensure_code(ev, CPL_ERROR_NULL_INPUT);

    TRY
    {
        n = eris_ifu_vector_count_non_rejected(ev);
        CHECK_ERROR_STATE();

        if (n <= 1) {
            BRK_WITH_ERROR(CPL_ERROR_ILLEGAL_INPUT);
        }

        median = eris_ifu_vector_get_median(ev, ERIS_IFU_ARITHMETIC);
        CHECK_ERROR_STATE();

        BRK_IF_NULL(
            pkvdata = cpl_vector_get_data(ev->data));
        BRK_IF_NULL(
            pkvmask = cpl_vector_get_data(ev->mask));
        for (int i = 0; i < cpl_vector_get_size(ev->data); i++) {
            if (pkvmask[i] > 0.5) {
                stdev += pow(pkvdata[i]-median, 2);
            }
        }

        stdev /= (n - 1);
        stdev = sqrt(stdev);
    }
    CATCH
    {
        stdev = -1;
    }

    return stdev;
}

/**
    @brief Get the maximum of the eris_ifu_vector and its position.

    @param ev       Input eris_ifu_vector
    @param pos 	    (Output) The index of the max element (first value is 0), is
                    set to -1 if all elements are rejected. If set to NULL it
                    won't be returned.

    @return         The element value

    Possible cpl_error_code set in this function:

    @li CPL_ERROR_NULL_INPUT if @c ev is NULL
*/
double eris_ifu_vector_get_max(const eris_ifu_vector *ev, int *pos)
{
    double       val        = -DBL_MAX;
    const double *pkvdata   = NULL,
                 *pkvmask   = NULL;
    int          n          = 0;

    cpl_ensure_code(ev, CPL_ERROR_NULL_INPUT);

    TRY
    {
        BRK_IF_NULL(
            pkvdata = cpl_vector_get_data_const(ev->data));
        BRK_IF_NULL(
            pkvmask = cpl_vector_get_data_const(ev->mask));

        n = cpl_vector_get_size(ev->data);
        if (eris_ifu_vector_count_rejected(ev) == n) {
            val = 0.;
            if (pos != NULL) {
                *pos = -1;
            }
        } else {
            for (int i = 0; i < n; i++) {
                if ((pkvmask[i] > 0.5) && (val < pkvdata[i])) {
                    val = pkvdata[i];
                    if (pos != NULL) {
                        *pos = i;
                    }
                }
            }
        }
        CHECK_ERROR_STATE();
    }
    CATCH
    {
        val = 0.;
        if (pos != NULL) {
            *pos = -1;
        }
    }

    return val;
}

/**
    @brief Get the minimum of the eris_ifu_vector and its position.

    @param ev       Input eris_ifu_vector
    @param pos 	    (Output) The index of the min element (first value is 0), is
                    set to -1 if all elements are rejected. If set to NULL it
                    won't be returned.

    @return         The element value

    Possible cpl_error_code set in this function:

    @li CPL_ERROR_NULL_INPUT if @c ev is NULL
*/
double eris_ifu_vector_get_min(const eris_ifu_vector *ev, int *pos)
{
    double       val        = DBL_MAX;
    const double *pkvdata   = NULL,
                 *pkvmask   = NULL;
    int          n          = 0;

    cpl_ensure_code(ev, CPL_ERROR_NULL_INPUT);

    TRY
    {
        BRK_IF_NULL(
            pkvdata = cpl_vector_get_data_const(ev->data));
        BRK_IF_NULL(
            pkvmask = cpl_vector_get_data_const(ev->mask));

        n = cpl_vector_get_size(ev->data);
        if (eris_ifu_vector_count_rejected(ev) == n) {
            val = 0.;
            if (pos != NULL) {
                *pos = -1;
            }
        } else {
            for (int i = 0; i < n; i++) {
                if ((pkvmask[i] > 0.5) && (val > pkvdata[i])) {
                    val = pkvdata[i];
                    if (pos != NULL) {
                        *pos = i;
                    }
                }
            }
        }
        CHECK_ERROR_STATE();
    }
    CATCH
    {
        val = 0.;
        if (pos != NULL) {
            *pos = -1;
        }
    }

    return val;
}

/**
    @brief
        Compute the elementwise power of the vector.

    @param ev       The vector to be modified in place.
    @param exponent The Exponent.

    @return
        The function returns CPL_ERROR_NONE on success or a CPL error code
        otherwise.

    With a vector-element being zero, this function returns also zero.

    Possible cpl_error_code set in this function:

    @li CPL_ERROR_NULL_INPUT    if @c ev is NULL.
*/
cpl_error_code eris_ifu_vector_power(eris_ifu_vector *ev, double exponent)
{
    int             n           = 0;
    double          *pkvmask    = NULL,
                    *pkvdata    = NULL;
    cpl_error_code  err         = CPL_ERROR_NONE;

    cpl_ensure_code(ev, CPL_ERROR_NULL_INPUT);

    TRY
    {
        if (exponent == 0.0) {
            /* create vector filled with ones */
            eris_ifu_vector_multiply_scalar(ev, 0.0);
            CHECK_ERROR_STATE();

            eris_ifu_vector_add_scalar(ev, 1.0);
            CHECK_ERROR_STATE();
        } else {
            n = cpl_vector_get_size(ev->data);
            CHECK_ERROR_STATE();

            BRK_IF_NULL(
                pkvdata = cpl_vector_get_data(ev->data));
            BRK_IF_NULL(
                pkvmask = cpl_vector_get_data(ev->mask));
            for (int i = 0; i < n; i++) {
                if (pkvmask[i] > 0.5) {
                    pkvdata[i] = pow(pkvdata[i], exponent);
                }
                if (eris_ifu_is_nan_or_inf(pkvdata[i])) {
                    /*pkvdata[i] = NAN;*/
                    pkvmask[i] = 0.;
                }
            }
        }
    }
    CATCH
    {
        err = cpl_error_get_code();
    }

    return err;
}

/**
    @brief
        Fill a eris_ifu_vector.

    @param ev      The vector to be filled with the value val.
    @param val     Value used to fill the eris_ifu_vector.

    If val is an invalid value all elements of ev will be rejected.

    @return
        The function returns CPL_ERROR_NONE on success or a CPL error code
        otherwise.

    Possible cpl_error_code set in this function:

    @li CPL_ERROR_NULL_INPUT    if @c vec is NULL.
*/
cpl_error_code eris_ifu_vector_fill(eris_ifu_vector *ev, double val)
{
    cpl_error_code  err         = CPL_ERROR_NONE;

    cpl_ensure_code(ev, CPL_ERROR_NULL_INPUT);

    TRY
    {
        for (int i = 0; i < eris_ifu_vector_get_size(ev); i++) {
            eris_ifu_vector_set(ev, i, val);
        }
    }
    CATCH
    {
        err = cpl_error_get_code();
    }

    return err;
}

/**
    @brief
        Flip the values of a vector.

    @param ev    The vector.

    @return
        The function returns CPL_ERROR_NONE on success or a CPL error code
        otherwise.

    The vector is flipped elementwise from both sides.

    Possible cpl_error_code set in this function:

    @li CPL_ERROR_NULL_INPUT    if @c ev is NULL.
*/
cpl_error_code eris_ifu_vector_flip(eris_ifu_vector* ev)
{
    cpl_error_code  ret_error   = CPL_ERROR_NONE;
    double          *pkvdata    = NULL,
                    *pkvmask    = NULL,
                    tmp_dbl     = 0;
    int             half_size   = 0,
                    size        = 0;

    cpl_ensure_code(ev, CPL_ERROR_NULL_INPUT);

    TRY
    {
        BRK_IF_NULL(
            pkvdata = cpl_vector_get_data(ev->data));
        BRK_IF_NULL(
            pkvmask = cpl_vector_get_data(ev->mask));

        size = cpl_vector_get_size(ev->data);
        half_size = floor(size / 2);

        for (int i = 0; i < half_size;i++) {
            tmp_dbl = pkvdata[i];
            pkvdata[i] = pkvdata[size-1-i];
            pkvdata[size-1-i] = tmp_dbl;

            tmp_dbl = pkvmask[i];
            pkvmask[i] = pkvmask[size-1-i];
            pkvmask[size-1-i] = tmp_dbl;
        }
    }
    CATCH
    {
        ret_error = cpl_error_get_code();
    }

    return ret_error;
}

/**
    @brief
        Calculates the histogram of a vector.

    This implementation follows the one used in IDL. This means that in the
    highest bin can only be found values corresponding to the maximum value in
    the input vector.

    @param ev       Input vector.
    @param nbins    The number of bins to produce.

    @return
        A vector of size @c nbins containing the histogram of input data @c d .

    Possible cpl_error_code set in this function:

    @li CPL_ERROR_NULL_INPUT    if @c d is NULL.
    @li CPL_ERROR_ILLEGAL_INPUT if @c nbis is <= 0.

*/
eris_ifu_vector* eris_ifu_vector_histogram(const eris_ifu_vector *ev, int nbins)
{
    int             pos         = 0;
    const double    *pkvdata    = NULL,
                    *pkvmask    = NULL;
    double          *phdata     = NULL,
                    binwidth    = 0.0,
                    hmin        = 0.0,
                    hmax        = 0.0;
    eris_ifu_vector *h          = NULL;

    cpl_ensure(ev, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(nbins > 0, CPL_ERROR_NULL_INPUT, NULL);

    TRY
    {
        hmin = eris_ifu_vector_get_min(ev, NULL);
        hmax = eris_ifu_vector_get_max(ev, NULL);
        CHECK_ERROR_STATE();

        binwidth = (hmax - hmin) / (nbins - 1);

        BRK_IF_NULL(
            pkvdata = cpl_vector_get_data_const(ev->data));
        BRK_IF_NULL(
            pkvmask = cpl_vector_get_data_const(ev->mask));

        BRK_IF_NULL(
            h = eris_ifu_vector_new(nbins));
        BRK_IF_NULL(
            phdata = cpl_vector_get_data(h->data));

        for (int i = 0; i < cpl_vector_get_size(ev->data); i++) {
            if (pkvmask[i] > 0.5) {
                pos = floor((pkvdata[i] - hmin) / binwidth);
                phdata[pos]++;
            }
        }
    }
    CATCH
    {
        eris_ifu_vector_delete(h); h = NULL;
    }

    return h;
}

/**
    @brief
      Override for cpl_vector_load().

    @param filename   Name of the input file
    @param position   Extension number in the file.

    @return The function returns the newly created eris_ifu_vector or NULL if an
            error occurred.

    This is an override for cpl_vector_load(), just giving a proper error
    message if the input file isn't found. The errors returned are the same as
    with @c cpl_vector_load
*/
eris_ifu_vector* eris_ifu_vector_load(const char *filename, int position)
{
    cpl_vector      *vec    = NULL;
    eris_ifu_vector *ev     = NULL;

    TRY
    {
        vec = cpl_vector_load(filename, position);
        if (cpl_error_get_code() != CPL_ERROR_NONE) {
            if (cpl_error_get_code() == CPL_ERROR_FILE_IO) {
                cpl_msg_error("", "File not found: %s", filename);
            } else {
                cpl_msg_error("", "Problem loading file '%s' (%s --> Code %d)",
                              filename, cpl_error_get_message(),
                              cpl_error_get_code());
            }
        }
        CHECK_ERROR_STATE();

        BRK_IF_NULL(
            ev = eris_ifu_vector_create(vec));
        cpl_vector_delete(vec);
    }
    CATCH
    {
        eris_ifu_vector_delete(ev);
    }

    return ev;
}

/**
    @brief
      Override for cpl_vector_save().

    @param ev       eris_ifu_vector to write to disk or NULL
    @param filename Name of the file to write
    @param bpp      Bits per pixel
    @param pl       Property list for the output header or NULL
    @param mode     The desired output options
    @param rej_val  The rejected values are filled with this value.
                    If -1 is provided, the rejected values in the internal
                    cpl-bad pixel mask are ignored.

    @return CPL_ERROR_NONE or the relevant _cpl_error_code_ on error

    This is an override for cpl_vector_save(), just giving a proper error
    message if the input file isn't found and rejecting all infinite values
    (NaN, +/-Inf). The errors returned are the same as with @c cpl_vector_save
*/
cpl_error_code eris_ifu_vector_save(const eris_ifu_vector *ev,
                                   const char *filename,
                                   cpl_type_bpp bpp,
                                   const cpl_propertylist *pl,
                                   unsigned mode,
                                   double rej_val)
{
    cpl_error_code  err         = CPL_ERROR_NONE;
    eris_ifu_vector *ev_dup     = NULL;
    double          *pkvdata    = NULL,
                    *pkvmask    = NULL;
    int             n           = 0;

    cpl_ensure_code(ev, CPL_ERROR_NULL_INPUT);

    TRY
    {
        if ((rej_val != -1) || (eris_ifu_is_nan_or_inf(rej_val))) {
            BRK_IF_NULL(
                ev_dup = eris_ifu_vector_duplicate(ev));
            /* set rejected values to NaN */
            BRK_IF_NULL(
                pkvdata = cpl_vector_get_data(ev_dup->data));
            BRK_IF_NULL(
                pkvmask = cpl_vector_get_data(ev_dup->mask));
            n = cpl_vector_get_size(ev_dup->data);
            for (int i = 0; i < n; i++) {
                if (pkvmask[i] < 0.5) {
                    pkvdata[i] = rej_val;
                }
            }

            err = cpl_vector_save(ev_dup->data, filename, bpp, pl, mode);
            CHECK_ERROR_STATE();
        } else {
            err = cpl_vector_save(ev->data, filename, bpp, pl, mode);
            CHECK_ERROR_STATE();
        }
        eris_ifu_vector_delete(ev_dup); ev_dup = NULL;
    }
    CATCH
    {
        err = cpl_error_get_code();
    }

    return err;
}

/**
    @brief
        All values contained in @c vec are printed for debugging purposes.

    @param ev The vector to print.

    @return
        The function returns CPL_ERROR_NONE on success or a CPL error code
        otherwise.
*/
cpl_error_code eris_ifu_vector_dump(const eris_ifu_vector *ev)
{
    int             n           = 0;
    cpl_error_code  err         = CPL_ERROR_NONE;
    const double    *pkvdata   = NULL,
                    *pkvmask   = NULL;

    TRY
    {
        if (ev == NULL) {
            cpl_msg_debug("", "     ====== START KMCLIPM_VECTOR ======");
            cpl_msg_debug("", "             empty vector");
            cpl_msg_debug("", "     ====== END KMCLIPM_VECTOR ========");
        } else {
            n = cpl_vector_get_size(ev->data);

            if (n != cpl_vector_get_size(ev->mask)) {
                BRK_WITH_ERROR_MSG(CPL_ERROR_ILLEGAL_INPUT,
                                             "data and mask not of same size!");
            }

            BRK_IF_NULL(
                pkvdata = cpl_vector_get_data_const(ev->data));
            BRK_IF_NULL(
                pkvmask = cpl_vector_get_data_const(ev->mask));

            cpl_msg_debug("", "     ====== START KMCLIPM_VECTOR ======");
            cpl_msg_debug("", "     #\tdata:\tmask:");
            cpl_msg_debug("", "     ---------------------");
            for (int i = 0; i < n; i++) {
                cpl_msg_debug("", "     %d\t%g\t%g", i, pkvdata[i], pkvmask[i]);
            }
            cpl_msg_debug("", "     ====== END KMCLIPM_VECTOR ========");
            CHECK_ERROR_STATE();
        }
    }
    CATCH
    {
        err = cpl_error_get_code();
    }

    return err;
}

/**
    @brief
        Checks if a value is nan, inf or -inf.

    @param A    The value to check.

    @return     TRUE if the value is nan, inf or -inf, FALSE otherwise.
*/
int eris_ifu_is_nan_or_inf(double A)
{
    return (isnan(A) || (isinf(A)==1) || (isinf(A)==-1));
}

/**
 * @brief eris_ifu_vector_sqrt
 * @param v
 * @return
 *
 * Fixes a bug in cpl_vector_sqrt(): When a value is -nan, the function returns an ILLEGAL_INPUT error...
 * sqrt() does return just the same nan value, so let's do so...
 */
cpl_error_code eris_ifu_vector_sqrt(eris_ifu_vector *ev)
{
    cpl_size        n           = 0;
    double          *pkvmask    = NULL,
                    *pkvdata    = NULL;
    cpl_error_code  err         = CPL_ERROR_NONE;

    cpl_ensure_code(ev, CPL_ERROR_NULL_INPUT);

    TRY
    {
        n = cpl_vector_get_size(ev->data);
        CHECK_ERROR_STATE();

        BRK_IF_NULL(
            pkvdata = cpl_vector_get_data(ev->data));
        BRK_IF_NULL(
            pkvmask = cpl_vector_get_data(ev->mask));

        /* Compute the sqrt */
        for (int i = 0; i < n; i++) {
            if (pkvmask[i] > 0.5) {
                /* assert that data is >= 0
                    it can be NaN, since sqrt() handles NaN correctly:
                    sqrt(NaN) = NaN */
                cpl_ensure_code(pkvdata[i] >= 0, CPL_ERROR_ILLEGAL_INPUT);

                pkvdata[i] = sqrt(pkvdata[i]);

                if (eris_ifu_is_nan_or_inf(pkvdata[i])) {
                    /*pkvdata[i] = NAN;*/
                    pkvmask[i] = 0.;
                }
            }
        }
    }
    CATCH
    {
    err = cpl_error_get_code();
    }

    return err;
}
