/* 
 * This file is part of the ERIS Pipeline
 * Copyright (C) 2024 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include <eris_ifu_detlin_static.h>
#include <eris_ifu_error.h>
#include <eris_ifu_utils.h>
#include <eris_ifu_debug.h>
#include <eris_ifu_dfs.h>
#include <eris_utils.h>
#include <eris_ifu_functions.h>
#include <eris_pfits.h>

#include <hdrl.h>
#include <cpl.h>
#include <string.h>

/* Test helper functions */
static cpl_propertylist* create_test_propertylist(double dit, double exptime) {
    cpl_propertylist* plist = cpl_propertylist_new();
    cpl_propertylist_append_string(plist, "INSTRUME", "ERIS");
    cpl_propertylist_append_string(plist, "ESO SEQ ARM", "SPIFFIER");
    cpl_propertylist_append_string(plist, "ESO DPR TYPE", ERIS_IFU_RAW_LIN);
    cpl_propertylist_append_bool(plist, FHDR_E_AR_LAMP_ST, CPL_TRUE);
    cpl_propertylist_append_bool(plist, FHDR_E_KR_LAMP_ST, CPL_TRUE);
    cpl_propertylist_append_bool(plist, FHDR_E_NE_LAMP_ST, CPL_TRUE);
    cpl_propertylist_append_bool(plist, FHDR_E_XE_LAMP_ST, CPL_TRUE);
    cpl_propertylist_append_bool(plist, FHDR_E_QTH_LAMP_ST, CPL_TRUE);
    cpl_propertylist_append_double(plist, "ESO DET DIT", dit);
    cpl_propertylist_append_double(plist, "EXPTIME", exptime);
    return plist;
}

static hdrl_image* create_test_hdrl_image(int nx, int ny, double value, double error) {
    cpl_image* img = cpl_image_new(nx, ny, CPL_TYPE_DOUBLE);
    cpl_image* err = cpl_image_new(nx, ny, CPL_TYPE_DOUBLE);
    hdrl_image* himg = NULL;
    
    cpl_image_add_scalar(img, value);
    cpl_image_add_scalar(err, error);
    himg = hdrl_image_create(img, err);
    cpl_image_delete(img);
    cpl_image_delete(err);
    
    return himg;
}

static cpl_frameset* create_test_frameset(int num_on, int num_off, double dit) {
    cpl_frameset* frameset = cpl_frameset_new();
    char filename[256];
    cpl_propertylist* plist;
    
    /* Create ON frames */
    for (int i = 0; i < num_on; i++) {
        snprintf(filename, sizeof(filename), "test_on_%d.fits", i);
        cpl_frame* frame = cpl_frame_new();
        cpl_frame_set_filename(frame, filename);
        cpl_frame_set_tag(frame, ERIS_IFU_RAW_LIN);
        /* Set frame as ON */
        cpl_frame_set_group(frame, CPL_FRAME_GROUP_RAW);
        
        /* Create and save test FITS file with propertylist */
        plist = create_test_propertylist(dit, dit);
        hdrl_image* img = create_test_hdrl_image(10, 10, 1000.0 + i*100.0, 1.0);
        cpl_image* data = hdrl_image_get_image(img);
        cpl_image_save(data, filename, CPL_TYPE_DOUBLE, plist, CPL_IO_CREATE);
        
        hdrl_image_delete(img);
        cpl_propertylist_delete(plist);
        cpl_frameset_insert(frameset, frame);
    }
    
    /* Create OFF frames */
    for (int i = 0; i < num_off; i++) {
        snprintf(filename, sizeof(filename), "test_off_%d.fits", i);
        cpl_frame* frame = cpl_frame_new();
        cpl_frame_set_filename(frame, filename);
        cpl_frame_set_tag(frame, ERIS_IFU_RAW_LIN);
        /* Set frame as OFF */
        cpl_frame_set_group(frame, CPL_FRAME_GROUP_CALIB);
        
        /* Create and save test FITS file with propertylist */
        plist = create_test_propertylist(dit, dit);
        hdrl_image* img = create_test_hdrl_image(10, 10, 100.0 + i*10.0, 1.0);
        cpl_image* data = hdrl_image_get_image(img);
        cpl_image_save(data, filename, CPL_TYPE_DOUBLE, plist, CPL_IO_CREATE);
        
        hdrl_image_delete(img);
        cpl_propertylist_delete(plist);
        cpl_frameset_insert(frameset, frame);
    }
    
    return frameset;
}

static void cleanup_test_data(hdrl_imagelist** imglist_on, 
                            hdrl_imagelist** imglist_off,
                            cpl_vector** vec_dit_on,
                            cpl_vector** vec_dit_off) {
    if (imglist_on && *imglist_on) {
        hdrl_imagelist_delete(*imglist_on);
        *imglist_on = NULL;
    }
    if (imglist_off && *imglist_off) {
        hdrl_imagelist_delete(*imglist_off);
        *imglist_off = NULL;
    }
    if (vec_dit_on && *vec_dit_on) {
        cpl_vector_delete(*vec_dit_on);
        *vec_dit_on = NULL;
    }
    if (vec_dit_off && *vec_dit_off) {
        cpl_vector_delete(*vec_dit_off);
        *vec_dit_off = NULL;
    }
}

/* Test cases */

/* Test eris_ifu_detlin_load_frames with NULL inputs */
static void test_detlin_load_frames_null_input(void) {
    hdrl_imagelist* imglist_on = NULL;
    hdrl_imagelist* imglist_off = NULL;
    cpl_vector* vec_dit_on = NULL;
    cpl_vector* vec_dit_off = NULL;
    
    /* Test with NULL frameset */
    cpl_error_code error = eris_ifu_detlin_load_frames(NULL, 0,
        &imglist_on, &imglist_off, &vec_dit_on, &vec_dit_off);
    cpl_test_eq(error, CPL_ERROR_NULL_INPUT);
    
    /* Test with NULL output parameters */
    cpl_frameset* frameset = create_test_frameset(2, 1, 1.0);
    error = eris_ifu_detlin_load_frames(frameset, 0,
        NULL, &imglist_off, &vec_dit_on, &vec_dit_off);
    cpl_test_eq(error, CPL_ERROR_NULL_INPUT);
    
    error = eris_ifu_detlin_load_frames(frameset, 0,
        &imglist_on, NULL, &vec_dit_on, &vec_dit_off);
    cpl_test_eq(error, CPL_ERROR_NULL_INPUT);
    
    error = eris_ifu_detlin_load_frames(frameset, 0,
        &imglist_on, &imglist_off, NULL, &vec_dit_off);
    cpl_test_eq(error, CPL_ERROR_NULL_INPUT);
    
    error = eris_ifu_detlin_load_frames(frameset, 0,
        &imglist_on, &imglist_off, &vec_dit_on, NULL);
    cpl_test_eq(error, CPL_ERROR_NULL_INPUT);
    cpl_error_reset();
    
    cleanup_test_data(&imglist_on, &imglist_off, &vec_dit_on, &vec_dit_off);
    cpl_frameset_delete(frameset);
    remove("test_on_0.fits");
    remove("test_on_1.fits");
    remove("test_off_0.fits");
    remove("test_off_1.fits");
    cpl_test_error(CPL_ERROR_NONE);
}

/* Test eris_ifu_detlin_load_frames with valid inputs */
static void test_detlin_load_frames_valid_input(void) {
    hdrl_imagelist* imglist_on = NULL;
    hdrl_imagelist* imglist_off = NULL;
    cpl_vector* vec_dit_on = NULL;
    cpl_vector* vec_dit_off = NULL;
    
    /* Create test frameset with 2 ON frames and 1 OFF frame */
    cpl_frameset* frameset = create_test_frameset(2, 1, 1.0);
    
    cpl_error_code error = eris_ifu_detlin_load_frames(frameset, 0,
        &imglist_on, &imglist_off, &vec_dit_on, &vec_dit_off);
    
    /* Check that the function returns success */
    cpl_test_eq(error, CPL_ERROR_NONE);
    
    /* Verify output data structures were created */
    cpl_test_nonnull(imglist_on);
    cpl_test_nonnull(imglist_off);
    cpl_test_nonnull(vec_dit_on);
    cpl_test_nonnull(vec_dit_off);
    
    /* Verify sizes match input frameset */
    cpl_test_eq(hdrl_imagelist_get_size(imglist_on), 2);
    cpl_test_eq(hdrl_imagelist_get_size(imglist_off), 1);
    cpl_test_eq(cpl_vector_get_size(vec_dit_on), 2);
    cpl_test_eq(cpl_vector_get_size(vec_dit_off), 1);
    
    cleanup_test_data(&imglist_on, &imglist_off, &vec_dit_on, &vec_dit_off);
    cpl_frameset_delete(frameset);
    cpl_test_error(CPL_ERROR_NONE);
}

/* Test eris_ifu_detlin_compute_linearity with NULL inputs */
static void test_detlin_compute_linearity_null_input(void) {
    cpl_parameterlist* parlist = cpl_parameterlist_new();
    hdrl_imagelist* imglist = NULL;
    cpl_vector* vec_dit = NULL;
    cpl_propertylist* qclog = cpl_propertylist_new();
    
    /* Test with NULL imagelist */
    cpl_image* result = eris_ifu_detlin_compute_linearity(parlist, NULL, vec_dit, qclog);
    cpl_test_null(result);
    cpl_test_eq(cpl_error_get_code(), CPL_ERROR_NULL_INPUT);
    
    /* Test with NULL vector */
    result = eris_ifu_detlin_compute_linearity(parlist, imglist, NULL, qclog);
    cpl_test_null(result);
    cpl_test_eq(cpl_error_get_code(), CPL_ERROR_NULL_INPUT);
    cpl_error_reset();
    
    cpl_parameterlist_delete(parlist);
    cpl_propertylist_delete(qclog);
    cpl_test_error(CPL_ERROR_NONE);
}

/* Test eris_ifu_detlin_filter_mask with NULL inputs */
static void test_detlin_filter_mask_null_input(void) {
    cpl_parameterlist* parlist = cpl_parameterlist_new();
    cpl_mask* bpm = NULL;
    
    /* Test with NULL mask */
    cpl_mask* result = eris_ifu_detlin_filter_mask(NULL, parlist);
    cpl_test_null(result);
    cpl_test_eq(cpl_error_get_code(), CPL_ERROR_NULL_INPUT);
    
    /* Test with NULL parameter list */
    bpm = cpl_mask_new(10, 10);
    result = eris_ifu_detlin_filter_mask(bpm, NULL);
    cpl_test_null(result);
    cpl_test_eq(cpl_error_get_code(), CPL_ERROR_NULL_INPUT);
    cpl_error_reset();
    
    cpl_parameterlist_delete(parlist);
    cpl_mask_delete(bpm);
    cpl_test_error(CPL_ERROR_NONE);
}

/* Test eris_compute_gain with NULL input */
static void test_compute_gain_null_input(void) {
    cpl_table* result = eris_compute_gain(NULL);
    cpl_test_null(result);
    cpl_test_eq(cpl_error_get_code(), CPL_ERROR_NULL_INPUT);
    cpl_error_reset();
    cpl_test_error(CPL_ERROR_NONE);
}

/* Test eris_ifu_detlin_compute_linearity with valid inputs */
static void test_detlin_compute_linearity_valid_input(void) {
    cpl_parameterlist* parlist = cpl_parameterlist_new();
    cpl_propertylist* qclog = cpl_propertylist_new();
    
    /* Create test data */
    hdrl_imagelist* imglist = hdrl_imagelist_new();
    cpl_vector* vec_dit = cpl_vector_new(2);
    
    /* Add test images to imagelist */
    hdrl_image* img1 = create_test_hdrl_image(10, 10, 1000.0, 1.0);
    hdrl_image* img2 = create_test_hdrl_image(10, 10, 2000.0, 1.0);
    hdrl_imagelist_set(imglist, img1, 0);
    hdrl_imagelist_set(imglist, img2, 1);
    
    /* Set DITs */
    cpl_vector_set(vec_dit, 0, 1.0);
    cpl_vector_set(vec_dit, 1, 2.0);
    cpl_test_error(CPL_ERROR_NONE);
    /* Add required parameters */
    cpl_parameter* param = cpl_parameter_new_value("eris_ifu_detlin.post-filter-x",
        CPL_TYPE_INT, "x size of post filter", "pixels", 3);
    cpl_parameterlist_append(parlist, param);
    
    param = cpl_parameter_new_value("eris_ifu_detlin.post-filter-y",
        CPL_TYPE_INT, "y size of post filter", "pixels", 3);
    cpl_parameterlist_append(parlist, param);
    
    param = cpl_parameter_new_value("eris_ifu_detlin.post-filter-mode",
        CPL_TYPE_STRING, "post filter mode", "", "closing");
    cpl_parameterlist_append(parlist, param);
    cpl_test_error(CPL_ERROR_NONE);
    /* Run test */
    cpl_image* result = eris_ifu_detlin_compute_linearity(parlist, imglist, vec_dit, qclog);
    cpl_test_error(CPL_ERROR_NONE);
    /* Verify results */
    cpl_test_nonnull(result);
    cpl_test_eq(cpl_image_get_size_x(result), 10);
    cpl_test_eq(cpl_image_get_size_y(result), 10);
    
    /* Check QC parameters were added */
    cpl_test_eq(cpl_propertylist_get_size(qclog) > 0, 1);
    
    /* Cleanup */
    cpl_image_delete(result);
    cpl_parameterlist_delete(parlist);
    cpl_propertylist_delete(qclog);
    hdrl_imagelist_delete(imglist);
    cpl_vector_delete(vec_dit);
    cpl_test_error(CPL_ERROR_NONE);
}

/* Test eris_ifu_detlin_filter_mask with valid inputs */
static void test_detlin_filter_mask_valid_input(void) {
    cpl_parameterlist* parlist = cpl_parameterlist_new();
    cpl_mask* bpm = cpl_mask_new(10, 10);
    
    /* Set some test pixels as bad */
    cpl_mask_set(bpm, 5, 5, 1);
    cpl_mask_set(bpm, 6, 6, 1);
    
    /* Add required parameters */
    cpl_parameter* param = cpl_parameter_new_value("eris_ifu_detlin.post-filter-x",
        CPL_TYPE_INT, "x size of post filter", "pixels", 3);
    cpl_parameterlist_append(parlist, param);
    
    param = cpl_parameter_new_value("eris_ifu_detlin.post-filter-y",
        CPL_TYPE_INT, "y size of post filter", "pixels", 3);
    cpl_parameterlist_append(parlist, param);
    
    param = cpl_parameter_new_value("eris_ifu_detlin.post-filter-mode",
        CPL_TYPE_STRING, "post filter mode", "", "closing");
    cpl_parameterlist_append(parlist, param);
    
    /* Run test */
    cpl_mask* result = eris_ifu_detlin_filter_mask(bpm, parlist);
    
    /* Verify results */
    cpl_test_nonnull(result);
    cpl_test_eq(cpl_mask_get_size_x(result), 10);
    cpl_test_eq(cpl_mask_get_size_y(result), 10);
    
    /* Cleanup */
    cpl_mask_delete(result);
    cpl_mask_delete(bpm);
    cpl_parameterlist_delete(parlist);
    cpl_test_error(CPL_ERROR_NONE);
}

/* Test eris_compute_gain with valid inputs */
static void test_compute_gain_valid_input(void) {
    /* Create test frameset with 4 ON frames and 2 OFF frames */
    cpl_frameset* frameset = create_test_frameset(4, 2, 1.0);
    cpl_test_error(CPL_ERROR_NONE);
    /* Run test */
    cpl_table* result = eris_compute_gain(frameset);
    cpl_test_error(CPL_ERROR_NONE);
    /* Verify results */
    cpl_test_nonnull(result);
    cpl_test_eq(cpl_table_get_nrow(result), 2);
    cpl_test_eq(cpl_table_has_column(result, "gain"), 1);
    cpl_test_eq(cpl_table_has_column(result, "adu"), 1);
    
    /* Cleanup */
    remove("test_on_0.fits");
    remove("test_on_1.fits");
    remove("test_off_0.fits");
    remove("test_off_1.fits");
    cpl_table_delete(result);
    cpl_frameset_delete(frameset);
    cpl_test_error(CPL_ERROR_NONE);
}

/* Main test function */
int main(void) {
	cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_WARNING);
    
    /* Run tcpl_ests */
    test_detlin_load_frames_null_input();
    //test_detlin_load_frames_valid_input();//TODO fix seg fault
    cpl_test_error(CPL_ERROR_NONE);
    test_detlin_compute_linearity_null_input();

    //test_detlin_compute_linearity_valid_input();//TODO fix errors
    cpl_test_error(CPL_ERROR_NONE);
    test_detlin_filter_mask_null_input();
    test_detlin_filter_mask_valid_input();
    test_compute_gain_null_input();
    cpl_test_error(CPL_ERROR_NONE);
    //test_compute_gain_valid_input();//TODO fix problem separating on/off frames
    cpl_test_error(CPL_ERROR_NONE);
    
    return cpl_test_end(0);
}
