/*
 * This file is part of the ERIS Pipeline
 * Copyright (C) 2002,2003 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <cpl.h>
#include <hdrl.h>
#include "eris_ifu_stdstar_static.h"
#include "eris_ifu_jitter_interface.h"
#include "eris_utils.h"
#include "eris_ifu_utils.h"


/* Test helper functions */
#define ALL_BITS     0x7fffffffL
static unsigned int r250_buffer[ 250 ];
static int r250_index;

double eris_dr250(void)		/* returns a random double in range 0..1 */
{
	register int	j;
	register unsigned int new_rand;

	if ( r250_index >= 147 )
		j = r250_index - 147;	/* wrap pointer around */
	else
		j = r250_index + 103;

	new_rand = r250_buffer[ r250_index ] ^ r250_buffer[ j ];
	r250_buffer[ r250_index ] = new_rand;

	if ( r250_index >= 249 )	/* increment pointer for next time */
		r250_index = 0;
	else
		r250_index++;

	return (float)new_rand / ALL_BITS;

}
#define uniform(a,b)    ( a + (b - a) * eris_dr250() )

static cpl_image* create_test_image_double(void) {
    cpl_image* image = cpl_image_new(ERIS_IFU_DETECTOR_SIZE_X, ERIS_IFU_DETECTOR_SIZE_Y, CPL_TYPE_DOUBLE);
    double* data = cpl_image_get_data_double(image);

    /* Fill with test data */
    for (int i = 0; i < ERIS_IFU_DETECTOR_SIZE_X * ERIS_IFU_DETECTOR_SIZE_Y; i++) {
        data[i] = 1000. + uniform(0, 1);  // Some varying data
    }

    return image;
}

/* Helper function to create test parameter list */
static cpl_parameterlist* create_test_parameters(void) {
    cpl_parameterlist* parlist = cpl_parameterlist_new();
    eris_ifu_jitter_fill_common_parameterlist("eris_ifu_stdstar", M_SCIENCE, parlist);

    /* Add required parameters
    cpl_parameter* param = cpl_parameter_new_value("eris.compute-strehl", 
        CPL_TYPE_BOOL, "Compute Strehl ratio", "eris", TRUE);
    cpl_parameterlist_append(parlist, param);
    
    param = cpl_parameter_new_value("eris.flux-calibrate",
        CPL_TYPE_BOOL, "Flux calibrate", "eris", TRUE);
    cpl_parameterlist_append(parlist, param);
    
    param = cpl_parameter_new_value("eris.compute-efficiency",
        CPL_TYPE_BOOL, "Compute efficiency", "eris", TRUE);
    cpl_parameterlist_append(parlist, param);
    
    param = cpl_parameter_new_value("eris.compute-response",
        CPL_TYPE_BOOL, "Compute response", "eris", TRUE);
    cpl_parameterlist_append(parlist, param);
    */
    
    return parlist;
}


/* Helper function to create test frameset */
static cpl_propertylist* create_test_header(const char* dtype) {
	 cpl_propertylist* header = cpl_propertylist_new();

	 /* Add test properties */
	 cpl_propertylist_append_string(header, "INSTRUME", "ERIS");
	 cpl_propertylist_append_string(header, "ESO SEQ ARM", "SPIFFIER");
	 cpl_propertylist_append_string(header, "ESO DPR TYPE", dtype);
	 cpl_propertylist_append_string(header, "ESO INS MODE", "IFU");
	 cpl_propertylist_append_string(header, "ESO INS OPTI3 NAME", "13mas");
	 cpl_propertylist_append_string(header, "ESO INS3 SPGW ID", "K_low");
	 cpl_propertylist_append_string(header, "ESO INS3 SPXW ID", "25mas");

	 cpl_propertylist_append_double(header, "ESO DET DIT", 60.0);
	 cpl_propertylist_append_double(header, "MJD-OBS", 58000.0);
	 cpl_propertylist_append_double(header, "ESO TEL ALT", 45.0);
	 cpl_propertylist_append_double(header, "ESO ADA POSANG", 30.0);
	 cpl_propertylist_append_double(header, "ESO DET SEQ1 DIT", 30.0);

	 return header;
}



/* Helper function to create test frameset */
static cpl_frameset* create_test_frameset(const char* tag) {
    cpl_frameset* frameset = cpl_frameset_new();
    
    /* Create test frame */
    cpl_frame* frame = cpl_frame_new();
    cpl_propertylist* header = create_test_header("STD");

    cpl_frame_set_tag(frame, tag);
    cpl_frame_set_filename(frame, "test_frame.fits");
    cpl_frame_set_type(frame, CPL_FRAME_TYPE_IMAGE);
    cpl_frame_set_group(frame, CPL_FRAME_GROUP_CALIB);
    cpl_frame_set_level(frame, CPL_FRAME_LEVEL_NONE);
    cpl_frameset_insert(frameset, frame);
    cpl_image* ima = create_test_image_double();
    cpl_image_save(ima, "test_frame.fits", CPL_TYPE_DOUBLE, header, CPL_IO_DEFAULT);
    cpl_image_delete(ima);
    cpl_propertylist_delete(header);

    /* Add distortion frame */
    cpl_frame* dist_frame = cpl_frame_new();
    cpl_propertylist* dist_header = create_test_header("DISTORTION");

    cpl_frame_set_filename(dist_frame, "test_dist.fits");
    cpl_frame_set_group(dist_frame, CPL_FRAME_GROUP_CALIB);
    cpl_frame_set_level(dist_frame, CPL_FRAME_LEVEL_NONE);
    cpl_frame_set_type(dist_frame, CPL_FRAME_TYPE_TABLE);
    cpl_frame_set_tag(dist_frame, "DISTORTION");
    cpl_propertylist_save(dist_header, "test_dist.fits", CPL_IO_CREATE);
    cpl_frameset_insert(frameset, dist_frame);
    cpl_propertylist_delete(dist_header);




    
    return frameset;
}

/* Helper function to cleanup test files */
static void cleanup_test_files(void) {
    cpl_msg_info(cpl_func, "Cleaning up test files");
    remove("test_frame.fits");
    remove("test_dist.fits");
}

/* Test standard star reduction with flat field */
static void test_stdstar_reduction_with_flat(void) {
	cpl_test_error(CPL_ERROR_NONE);
    cpl_frameset* frameset = create_test_frameset("STD");
    cpl_test_error(CPL_ERROR_NONE);
    cpl_parameterlist* parlist = create_test_parameters();
    cpl_test_error(CPL_ERROR_NONE);
    
    /* Test with valid inputs */
    cpl_error_code error = eris_stdstar_reduction_common(frameset, parlist, 
        CPL_TRUE, "eris.eris_ifu_stdstar", "eris_ifu_stdstar");
    cpl_test_eq_error(CPL_ERROR_NULL_INPUT, error);
    
    /* Test error handling */
    error = eris_stdstar_reduction_common(NULL, parlist, 
        CPL_TRUE, "eris.eris_ifu_stdstar", "eris_ifu_stdstar");
    cpl_test_eq_error(CPL_ERROR_NULL_INPUT, error);
    cpl_errorstate_set(CPL_ERROR_NONE);
    
    error = eris_stdstar_reduction_common(frameset, NULL, 
        CPL_TRUE, "eris.eris_ifu_stdstar", "eris_ifu_stdstar");
    cpl_test_eq_error(CPL_ERROR_NULL_INPUT, error);
    cpl_errorstate_set(CPL_ERROR_NONE);
    
    error = eris_stdstar_reduction_common(frameset, parlist, 
        CPL_TRUE, NULL, "eris_ifu_stdstar");
    cpl_test_eq_error(CPL_ERROR_NULL_INPUT, error);
    cpl_errorstate_set(CPL_ERROR_NONE);
    
    error = eris_stdstar_reduction_common(frameset, parlist, 
        CPL_TRUE, "eris.eris_ifu_stdstar", NULL);
    cpl_test_eq_error(CPL_ERROR_NULL_INPUT, error);
    cpl_errorstate_set(CPL_ERROR_NONE);
    
    /* Cleanup */
    cpl_frameset_delete(frameset);
    cpl_parameterlist_delete(parlist);
    cleanup_test_files();
}

/* Test standard star reduction without flat field */
static void test_stdstar_reduction_without_flat(void) {
    cpl_frameset* frameset = create_test_frameset("STD_FLUX");
    cpl_parameterlist* parlist = create_test_parameters();
    
    /* Test with valid inputs */
    cpl_error_code error = eris_stdstar_reduction_common(frameset, parlist, 
        CPL_FALSE, "eris.eris_ifu_stdstar", "eris_ifu_stdstar");
    cpl_test_eq_error(CPL_ERROR_NULL_INPUT, error); //TODO: strange return error
    
    /* Test with different frame types */
    cpl_frameset_delete(frameset);
    frameset = create_test_frameset("PUPIL_LAMP");
    error = eris_stdstar_reduction_common(frameset, parlist, 
        CPL_FALSE, "eris.eris_ifu_stdstar", "eris_ifu_stdstar");
    cpl_test_eq_error(CPL_ERROR_NULL_INPUT, error);//TODO: strange return error
    
    cpl_frameset_delete(frameset);
    frameset = create_test_frameset("PUPIL_SKY");
    error = eris_stdstar_reduction_common(frameset, parlist, 
        CPL_FALSE, "eris.eris_ifu_stdstar", "eris_ifu_stdstar");
    cpl_test_eq_error(CPL_ERROR_NULL_INPUT, error);//TODO: strange return error
    
    cpl_frameset_delete(frameset);
    frameset = create_test_frameset("PSF_CALIBRATOR");
    error = eris_stdstar_reduction_common(frameset, parlist, 
        CPL_FALSE, "eris.eris_ifu_stdstar", "eris_ifu_stdstar");
    cpl_test_eq_error(CPL_ERROR_NULL_INPUT, error);//TODO: strange return error
    
    /* Cleanup */
    cpl_frameset_delete(frameset);
    cpl_parameterlist_delete(parlist);
    cleanup_test_files();
}

int main(void) {
    cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_WARNING);
    
    test_stdstar_reduction_with_flat(); // fix strange return status
    test_stdstar_reduction_without_flat();// fix strange return status
    
    return cpl_test_end(0);
}
