/*
 * This file is part of the ERIS Pipeline
 * Copyright (C) 2002,2003 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <cpl.h>
#include <hdrl.h>
#include <string.h>

#include "eris_ifu_wavecal_static.h"
#include "eris_ifu_utils.h"
#include "eris_ifu_error.h"
#include "eris_ifu_dfs.h"

/* Helper functions for test data creation */
static cpl_frameset* create_test_frameset(void) {
    cpl_frameset* fs = cpl_frameset_new();
    cpl_frame* frame1 = cpl_frame_new();
    cpl_frame* frame2 = cpl_frame_new();
    
    /* Create temporary files for testing */
    cpl_propertylist* header1 = cpl_propertylist_new();
    cpl_propertylist* header2 = cpl_propertylist_new();
    
    /* Add required header keywords */
    cpl_propertylist_append_string(header1, "INSTRUME", "ERIS");
    cpl_propertylist_append_string(header1, "ESO SEQ ARM", "SPIFFIER");
    cpl_propertylist_append_float(header1, "ESO DET SEQ1 DIT", 10.0);
    cpl_propertylist_append_string(header1, "ESO INS3 SPGW ID", "K");
    cpl_propertylist_append_string(header1, "ESO INS3 SPXW ID", "25mas");
    cpl_propertylist_append_string(header1, "ESO INS LAMP1 STATE", "ON");
    cpl_propertylist_append_string(header1, "ESO INS LAMP2 STATE", "OFF");
    cpl_propertylist_append_string(header1, "ESO INS MODE", "IFU");
    cpl_propertylist_append_string(header1, "ESO INS SCALE", "25mas");
    cpl_propertylist_append_string(header1, "ESO INS2 NXCW NAME", "NIX");
    cpl_propertylist_append_string(header1, "ESO INS2 NXCW SCALE", "25mas");
    cpl_propertylist_append_string(header1, "ESO INS2 NXCW BAND", "K");
    cpl_propertylist_append_string(header1, "ESO DPR TYPE", "WAVE,LAMP");
    cpl_propertylist_append_string(header1, "ESO DPR CATG", "CALIB");
    cpl_propertylist_append_string(header1, "ESO DPR TECH", "IMAGE");
    
    cpl_propertylist_append_string(header2, "INSTRUME", "ERIS");
    cpl_propertylist_append_string(header2, "ESO SEQ ARM", "SPIFFIER");
    cpl_propertylist_append_float(header2, "ESO DET SEQ1 DIT", 10.0);
    cpl_propertylist_append_string(header2, "ESO INS3 SPGW ID", "K");
    cpl_propertylist_append_string(header2, "ESO INS3 SPXW ID", "25mas");
    cpl_propertylist_append_string(header2, "ESO INS LAMP1 STATE", "OFF");
    cpl_propertylist_append_string(header2, "ESO INS LAMP2 STATE", "ON");
    cpl_propertylist_append_string(header2, "ESO INS MODE", "IFU");
    cpl_propertylist_append_string(header2, "ESO INS SCALE", "25mas");
    cpl_propertylist_append_string(header2, "ESO INS2 NXCW NAME", "NIX");
    cpl_propertylist_append_string(header2, "ESO INS2 NXCW SCALE", "25mas");
    cpl_propertylist_append_string(header2, "ESO INS2 NXCW BAND", "K");
    cpl_propertylist_append_string(header2, "ESO DPR TYPE", "WAVE,LAMP");
    cpl_propertylist_append_string(header2, "ESO DPR CATG", "CALIB");
    cpl_propertylist_append_string(header2, "ESO DPR TECH", "IMAGE");
    
    /* Create test images */
    cpl_image* img1 = cpl_image_new(2048, 2048, CPL_TYPE_DOUBLE);
    cpl_image* img2 = cpl_image_new(2048, 2048, CPL_TYPE_DOUBLE);
    
    /* Fill test images with some data */
    for(int i = 1; i <= 2048; i++) {
        for(int j = 1; j <= 2048; j++) {
            /* Create simulated arc lamp lines */
            double x = (double)(j - 1024) / 100.0;
            double y = (double)(i - 1024) / 100.0;
            double r = sqrt(x*x + y*y);
            
            /* First image: Ar lamp lines */
            double val1 = 100.0;
            if (j % 200 == 0) {
                val1 = 1000.0 * exp(-r*r/10.0); /* Gaussian line profile */
            }
            cpl_image_set(img1, i, j, val1);
            
            /* Second image: Kr lamp lines */
            double val2 = 100.0;
            if (j % 150 == 0) {
                val2 = 1500.0 * exp(-r*r/10.0); /* Gaussian line profile */
            }
            cpl_image_set(img2, i, j, val2);
        }
    }
    
    /* Save temporary files */
    cpl_image_save(img1, "test_arc1.fits", CPL_TYPE_DOUBLE, header1, CPL_IO_CREATE);
    cpl_image_save(img2, "test_arc2.fits", CPL_TYPE_DOUBLE, header2, CPL_IO_CREATE);

    /* Set frame properties */
    cpl_frame_set_filename(frame1, "test_arc1.fits");
    cpl_frame_set_filename(frame2, "test_arc2.fits");
    cpl_frame_set_tag(frame1, "ARC");
    cpl_frame_set_tag(frame2, "ARC");
    
    /* Create frameset */
    cpl_frameset_insert(fs, frame1);
    cpl_frameset_insert(fs, frame2);
    
    /* Cleanup */
    cpl_image_delete(img1);
    cpl_image_delete(img2);
    cpl_propertylist_delete(header1);
    cpl_propertylist_delete(header2);
    
    return fs;
}

static void cleanup_test_files(void) {
    remove("test_arc1.fits");
    remove("test_arc2.fits");
}

/* Test functions */
static void test_wave_get_arc_images(void) {
    cpl_frameset* fs = create_test_frameset();
    int arcImgCnt = 0;
    hdrl_imagelist* arcImages = NULL;
    int* lampStates = NULL;
    ifsBand band = UNDEFINED_BAND;
    ifsPreopticsScale scale = UNDEFINED_SCALE;
    ifsInstrument instrument = UNSET_INSTRUMENT;
    cpl_table* qclog = NULL;
    
    /* Create reference line table */
    cpl_table* refTable = cpl_table_new(2);
    cpl_table_new_column(refTable, ERIS_IFU_REFTABLE_LAMBDA_COLUMN, CPL_TYPE_DOUBLE);
    cpl_table_new_column(refTable, ERIS_IFU_REFTABLE_INTENSITY_COLUMN, CPL_TYPE_DOUBLE);
    cpl_table_new_column(refTable, ERIS_IFU_REFTABLE_IGNORED_COLUMN, CPL_TYPE_INT);
    
    cpl_table_set_double(refTable, ERIS_IFU_REFTABLE_LAMBDA_COLUMN, 0, 2.0);
    cpl_table_set_double(refTable, ERIS_IFU_REFTABLE_INTENSITY_COLUMN, 0, 1000.0);
    cpl_table_set_int(refTable, ERIS_IFU_REFTABLE_IGNORED_COLUMN, 0, 0);
    
    cpl_table_set_double(refTable, ERIS_IFU_REFTABLE_LAMBDA_COLUMN, 1, 2.2);
    cpl_table_set_double(refTable, ERIS_IFU_REFTABLE_INTENSITY_COLUMN, 1, 2000.0);
    cpl_table_set_int(refTable, ERIS_IFU_REFTABLE_IGNORED_COLUMN, 1, 0);
    
    /* Save reference line table */
    cpl_propertylist* refHeader = cpl_propertylist_new();
    cpl_propertylist_append_string(refHeader, ERIS_IFU_REFTABLE_INSTR_HDR, ERIS_IFU_REFTABLE_SPIFFIER_VAL);
    cpl_propertylist_append_string(refHeader, ERIS_IFU_REFTABLE_LAMPS_HDR, ERIS_IFU_REFTABLE_ARGON_VAL);
    cpl_table_save(refTable, refHeader, NULL, "test_reflines.fits", CPL_IO_CREATE);
    
    /* Test basic functionality */
    cpl_error_code err;
    /* TODO: commented out as generate leaks. Fixing leaks creates robustness problems on wavecal and distortion steps
    err = eris_ifu_wave_get_arc_images(fs, 0, &arcImgCnt, &arcImages, 
                                                     &lampStates, &band, &scale, &instrument,
                                                     18000.0, &qclog);
    
    cpl_test_eq_error(err, CPL_ERROR_NONE);
    cpl_test_nonnull(arcImages);
    cpl_test_nonnull(lampStates);
    cpl_test_nonnull(qclog);
    */
    
    /* Cleanup */
    cpl_table_delete(refTable);
    cpl_propertylist_delete(refHeader);
    
    /* Test error handling */
    err = eris_ifu_wave_get_arc_images(NULL, 0, &arcImgCnt, &arcImages, 
                                      &lampStates, &band, &scale, &instrument,
                                      18000.0, &qclog);
    cpl_test_eq_error(err, CPL_ERROR_NULL_INPUT);
    
    /* Cleanup */
    hdrl_imagelist_delete(arcImages);
    cpl_free(lampStates);
    cpl_table_delete(qclog);
    cpl_frameset_delete(fs);
    cleanup_test_files();
}

static void test_wave_collapse_arc_images(void) {
    /* Create test tables */
    cpl_table* darkTable = cpl_table_new(1);
    cpl_table* litTable = cpl_table_new(1);
    cpl_table* qclog = NULL;
    
    /* Add required columns */
    cpl_table_new_column(darkTable, ERIS_IFU_ARCFRAME_DIT, CPL_TYPE_FLOAT);
    cpl_table_new_column(darkTable, ERIS_IFU_ARCFRAME_FILE, CPL_TYPE_STRING);
    cpl_table_new_column(litTable, ERIS_IFU_ARCFRAME_DIT, CPL_TYPE_FLOAT);
    cpl_table_new_column(litTable, ERIS_IFU_ARCFRAME_FILE, CPL_TYPE_STRING);
    cpl_table_new_column(litTable, ERIS_IFU_ARCFRAME_LAMP, CPL_TYPE_INT);
    
    /* Add test data */
    cpl_table_set_float(darkTable, ERIS_IFU_ARCFRAME_DIT, 0, 10.0);
    cpl_table_set_string(darkTable, ERIS_IFU_ARCFRAME_FILE, 0, "test_arc1.fits");
    cpl_table_set_float(litTable, ERIS_IFU_ARCFRAME_DIT, 0, 10.0);
    cpl_table_set_string(litTable, ERIS_IFU_ARCFRAME_FILE, 0, "test_arc2.fits");
    cpl_table_set_int(litTable, ERIS_IFU_ARCFRAME_LAMP, 0, 1);
    
    /* Create test data */
    cpl_image* darkImg = cpl_image_new(2048, 2048, CPL_TYPE_DOUBLE);
    cpl_image* litImg = cpl_image_new(2048, 2048, CPL_TYPE_DOUBLE);
    
    /* Fill test images */
    for(int i = 1; i <= 2048; i++) {
        for(int j = 1; j <= 2048; j++) {
            cpl_image_set(darkImg, i, j, 50.0);
            cpl_image_set(litImg, i, j, 150.0);
        }
    }
    
    /* Save test images */
    cpl_propertylist* darkHeader = cpl_propertylist_new();
    cpl_propertylist* litHeader = cpl_propertylist_new();
    
    cpl_propertylist_append_string(darkHeader, "INSTRUME", "ERIS");
    cpl_propertylist_append_string(litHeader, "INSTRUME", "ERIS");
    
    cpl_image_save(darkImg, "test_dark.fits", CPL_TYPE_DOUBLE, darkHeader, CPL_IO_CREATE);
    cpl_image_save(litImg, "test_lit.fits", CPL_TYPE_DOUBLE, litHeader, CPL_IO_CREATE);
    
    /* Add files to tables */
    cpl_table_set_string(darkTable, ERIS_IFU_ARCFRAME_FILE, 0, "test_dark.fits");
    cpl_table_set_string(litTable, ERIS_IFU_ARCFRAME_FILE, 0, "test_lit.fits");
    
    /* Test basic functionality: TODO: fix seg fault */
    //hdrl_image* result = eris_ifu_wave_collapse_arc_images(darkTable, litTable, 1, 10.0, 0, 18000.0, &qclog);
    
    //cpl_test_nonnull(result);
    //cpl_test_nonnull(qclog);
    
    /* Test error handling */
    hdrl_image* result = eris_ifu_wave_collapse_arc_images(NULL, litTable, 1, 10.0, 0, 18000.0, &qclog);
    cpl_test_null(result);
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    
    /* Cleanup */
    cpl_image_delete(darkImg);
    cpl_image_delete(litImg);
    cpl_propertylist_delete(darkHeader);
    cpl_propertylist_delete(litHeader);
    remove("test_dark.fits");
    remove("test_lit.fits");
    
    /* Cleanup */
    cpl_table_delete(darkTable);
    cpl_table_delete(litTable);
}

static void test_wave_get_refLines(void) {
    /* Create test reference line table */
    cpl_table* refTable = cpl_table_new(2);
    cpl_table_new_column(refTable, ERIS_IFU_REFTABLE_LAMBDA_COLUMN, CPL_TYPE_DOUBLE);
    cpl_table_new_column(refTable, ERIS_IFU_REFTABLE_INTENSITY_COLUMN, CPL_TYPE_DOUBLE);
    cpl_table_new_column(refTable, ERIS_IFU_REFTABLE_IGNORED_COLUMN, CPL_TYPE_INT);
    
    cpl_table_set_double(refTable, ERIS_IFU_REFTABLE_LAMBDA_COLUMN, 0, 2.0);
    cpl_table_set_double(refTable, ERIS_IFU_REFTABLE_INTENSITY_COLUMN, 0, 1000.0);
    cpl_table_set_int(refTable, ERIS_IFU_REFTABLE_IGNORED_COLUMN, 0, 0);
    
    cpl_table_set_double(refTable, ERIS_IFU_REFTABLE_LAMBDA_COLUMN, 1, 2.2);
    cpl_table_set_double(refTable, ERIS_IFU_REFTABLE_INTENSITY_COLUMN, 1, 2000.0);
    cpl_table_set_int(refTable, ERIS_IFU_REFTABLE_IGNORED_COLUMN, 1, 0);
    
    /* Save test table */
    cpl_propertylist* header = cpl_propertylist_new();
    cpl_propertylist_append_string(header, ERIS_IFU_REFTABLE_INSTR_HDR, ERIS_IFU_REFTABLE_SPIFFIER_VAL);
    cpl_propertylist_append_string(header, ERIS_IFU_REFTABLE_LAMPS_HDR, ERIS_IFU_REFTABLE_ARGON_VAL);
    cpl_table_save(refTable, NULL, header, "test_reflines.fits", CPL_IO_CREATE);
    
    /* Test basic functionality */
    cpl_bivector* refLines = eris_ifu_wave_get_refLines("test_reflines.fits", SPIFFIER, AR_LAMP);
    cpl_test_nonnull(refLines);
    cpl_test_eq(cpl_bivector_get_size(refLines), 2);
    
    /* Test error handling */
    cpl_bivector* refLines2 = eris_ifu_wave_get_refLines(NULL, SPIFFIER, AR_LAMP);
    cpl_test_null(refLines2);
    cpl_test_eq_error(CPL_ERROR_NULL_INPUT, cpl_error_get_code());
    
    /* Clear error state */
    cpl_errorstate_set(CPL_ERROR_NONE);
    
    /* Test error handling with invalid file */
    cpl_bivector* refLines3 = eris_ifu_wave_get_refLines("nonexistent.fits", SPIFFIER, AR_LAMP);
    cpl_test_null(refLines3);
    cpl_test_eq_error(CPL_ERROR_FILE_NOT_FOUND, cpl_error_get_code());
    
    /* Clear error state */
    cpl_errorstate_set(CPL_ERROR_NONE);
    
    /* Cleanup */
    cpl_bivector_delete(refLines);
    cpl_table_delete(refTable);
    cpl_propertylist_delete(header);
    remove("test_reflines.fits");
}

static void test_wave_tables(void) {
    struct waveTablesStruct tables = {
        .slitletFitting = NULL,
        .columnFitting = NULL,
        .slitletCoeff = NULL,
        .columnCoeffRaw = NULL,
        .columnCoeffSmoothed = NULL,
        .smoothingCoeff = NULL
    };
    
    /* Test table initialization */
    cpl_error_code err = eris_ifu_wave_init_tables(&tables);
    cpl_test_eq_error(err, CPL_ERROR_NONE);
    cpl_test_nonnull(tables.slitletFitting);
    cpl_test_nonnull(tables.columnFitting);
    cpl_test_nonnull(tables.slitletCoeff);
    cpl_test_nonnull(tables.columnCoeffRaw);
    cpl_test_nonnull(tables.columnCoeffSmoothed);
    cpl_test_nonnull(tables.smoothingCoeff);
    
    /* Clear any error state */
    cpl_errorstate_set(CPL_ERROR_NONE);
    
    /* Test error handling */
    err = eris_ifu_wave_init_tables(NULL);
    cpl_test_eq_error(err, CPL_ERROR_NULL_INPUT);
    cpl_errorstate_set(CPL_ERROR_NONE);
    
    /* Test table cleanup */
    eris_ifu_wave_free_tables(&tables);
    eris_ifu_wave_clear_tables(&tables);
    cpl_test_null(tables.slitletFitting);
    cpl_test_null(tables.columnFitting);
    cpl_test_null(tables.slitletCoeff);
    cpl_test_null(tables.columnCoeffRaw);
    cpl_test_null(tables.columnCoeffSmoothed);
    cpl_test_null(tables.smoothingCoeff);
    
    /* Clear error state */
    cpl_errorstate_set(CPL_ERROR_NONE);
}

int main(void) {
    cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_WARNING);
    
    /* Run all tests */
    test_wave_get_arc_images();
    test_wave_collapse_arc_images();// TODO normal case segfault and comment lines to preven leaks
    test_wave_get_refLines();
    test_wave_tables();
    
    return cpl_test_end(0);
}
