/* $Id$
 *
 * This file is part of the ERIS Pipeline
 * Copyright (C) 2002,2003 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/*
 * $Author$
 * $Date$
 * $Revision$
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                Includes
 -----------------------------------------------------------------------------*/

#include <math.h>
#include <eris_nix_dfs.h>
#include <eris_nix_master_bpm.h>
#include <eris_nix_utils.h>
#include "eris_nix_test_defs.h"

/*----------------------------------------------------------------------------*/
/**
 * @defgroup eris_nix_master_bpm_test  Unit test of eris_nix_master_bpm
 *
 */
/*----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------
                                       Prototypes
 -----------------------------------------------------------------------------*/

static void eris_nix_master_bpm_test(void);

/**@{*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Textual representation of CPL frame group
  @param    group     to convert
  @return   textual representation
 */
/*----------------------------------------------------------------------------*/
static void eris_nix_master_bpm_test(void)
{
    master_bpm       * check = NULL;
    cpl_frame        * frame1 = NULL;
    cpl_frame        * frame2 = NULL;
    cpl_frameset     * frameset = NULL;
    cpl_mask         * mask =NULL;
    master_bpm       * master_bpm_frm = NULL;
    cpl_parameterlist* parlist = NULL;
    cpl_propertylist * plist = NULL;
    cpl_image        * rawdata = NULL;
    cpl_propertylist * rawdata_plist = NULL;
    cpl_frameset     * used = NULL;

    master_bpm_frm = en_master_bpm_create(NULL, NULL, BPM_DARK, NULL);
    cpl_test(!master_bpm_frm);
    cpl_test_error(CPL_ERROR_NULL_INPUT);

    /* make test master bpm */

    plist = cpl_propertylist_new();
    cpl_propertylist_update_string(plist, CPL_DFS_PRO_CATG,
                                   ERIS_NIX_MASTER_BPM_LAMP_PRO_CATG);
    cpl_propertylist_update_string(plist, "INSTRUME", "ERIS");
    cpl_propertylist_update_int(plist, "ESO DET SEQ1 WIN NX", TEST_SIZE_X);
    cpl_propertylist_update_int(plist, "ESO DET SEQ1 WIN NY", TEST_SIZE_Y);
    cpl_propertylist_update_int(plist, "ESO DET SEQ1 WIN ROT", 0);
    cpl_propertylist_update_int(plist, "ESO DET SEQ1 WIN STRX", 1);
    cpl_propertylist_update_int(plist, "ESO DET SEQ1 WIN STRY", 1);
    cpl_propertylist_update_int(plist, "ESO DET CHIP NX", TEST_SIZE_X);
    cpl_propertylist_update_int(plist, "ESO DET CHIP NY", TEST_SIZE_Y);

    mask = cpl_mask_new(TEST_SIZE_X, TEST_SIZE_Y);
    cpl_mask_set(mask, 10, 11, CPL_BINARY_1);
    master_bpm_frm = en_master_bpm_create("test", mask, BPM_DARK, plist);
    cpl_mask_set(mask, 10, 12, CPL_BINARY_1);
    en_master_bpm_set(master_bpm_frm, mask, BPM_HOT);
    cpl_mask_set(mask, 10, 13, CPL_BINARY_1);
    en_master_bpm_set(master_bpm_frm, mask, BPM_NON_LINEAR);

    /* save master bpm to a DFS-compliant MEF file */

    frameset  = cpl_frameset_new();
    frame1 = cpl_frame_new();
    rawdata =  cpl_image_new(TEST_SIZE_X, TEST_SIZE_Y, CPL_TYPE_DOUBLE);
    rawdata_plist = cpl_propertylist_new();
    cpl_image_save(rawdata, "nix_test_raw_bpm.fits", CPL_TYPE_DOUBLE,
      rawdata_plist, CPL_IO_CREATE);
    cpl_frame_set_filename(frame1, "nix_test_raw_bpm.fits");
    cpl_frame_set_group(frame1, CPL_FRAME_GROUP_RAW);
    cpl_frame_set_level(frame1, CPL_FRAME_LEVEL_NONE);
    cpl_frame_set_tag(frame1, "DARK");
    cpl_frame_set_type(frame1, CPL_FRAME_TYPE_IMAGE);
    cpl_frameset_insert(frameset, frame1);

    parlist = cpl_parameterlist_new();

    enu_dfs_save_bpm(ERIS_NIX_MASTER_BPM_LAMP_PRO_CATG, frameset, parlist,
                     frameset, master_bpm_frm, "eris_nix_flat_lamp", plist,
                     PACKAGE "/" PACKAGE_VERSION, 
                     "nix_test_master_bpm.fits");
    cpl_test_error(CPL_ERROR_NONE);

    /* Read it in again and check that it matches what was written out */

    frame2 = cpl_frame_new();
    cpl_frame_set_filename(frame2, "nix_test_master_bpm.fits");
    cpl_frame_set_tag(frame2, "MASTER_BPM");
    cpl_frame_set_group(frame2, CPL_FRAME_GROUP_CALIB);
    cpl_frame_set_level(frame2, CPL_FRAME_LEVEL_NONE);
    cpl_frame_set_type(frame2, CPL_FRAME_TYPE_IMAGE);
    cpl_frameset_insert(frameset, frame2);

    used = cpl_frameset_new();
    check = en_master_bpm_load_from_frameset(frameset, "MASTER_BPM", used,
                                             CPL_TRUE);

    int rejected;
    int val = cpl_image_get(check->bpm, 10, 13, &rejected);
    cpl_test_eq(val, 1<<(BPM_NON_LINEAR-1));

    val = cpl_image_get(check->bpm, 10, 12, &rejected);
    cpl_test_eq(val, (1<<(BPM_NON_LINEAR-1)) + (1<<(BPM_HOT-1)));

    val = cpl_image_get(check->bpm, 10, 11, &rejected);
    cpl_test_eq(val, (1<<(BPM_NON_LINEAR-1)) + (1<<(BPM_HOT-1)) + (1<<(BPM_DARK-1)));

    const char * string1 = cpl_propertylist_get_string(
                                       check->plist, CPL_DFS_PRO_CATG);
    cpl_test_eq_string(string1, ERIS_NIX_MASTER_BPM_LAMP_PRO_CATG);

    /* check that the get_mask method works */

    int flag_mask = 0;
    flag_mask = ~flag_mask; /* all flag bits */
    cpl_mask * check_mask = en_master_bpm_get_mask(check, flag_mask);
    cpl_size count = cpl_mask_count(check_mask);
    cpl_test_eq(count, 3);
    cpl_binary mask_val = cpl_mask_get(check_mask, 10, 11);
    cpl_test_eq(mask_val, CPL_BINARY_1);
    mask_val = cpl_mask_get(check_mask, 10, 12);
    cpl_test_eq(mask_val, CPL_BINARY_1);
    mask_val = cpl_mask_get(check_mask, 10, 13);
    cpl_test_eq(mask_val, CPL_BINARY_1);
    cpl_mask_delete(check_mask);

    /* pixels that are BPM_DARK */

    flag_mask = 1<<(BPM_DARK-1);
    check_mask = en_master_bpm_get_mask(check, flag_mask);
    count = cpl_mask_count(check_mask);
    cpl_test_eq(count, 1);
    mask_val = cpl_mask_get(check_mask, 10, 11);
    cpl_test_eq(mask_val, CPL_BINARY_1);
    mask_val = cpl_mask_get(check_mask, 10, 12);
    cpl_test_eq(mask_val, CPL_BINARY_0);
    mask_val = cpl_mask_get(check_mask, 10, 13);
    cpl_test_eq(mask_val, CPL_BINARY_0);
    cpl_mask_delete(check_mask);

    /* pixels that are BPM_HOT */

    flag_mask = 1<<(BPM_HOT-1);
    check_mask = en_master_bpm_get_mask(check, flag_mask);
    count = cpl_mask_count(check_mask);
    cpl_test_eq(count, 2);
    mask_val = cpl_mask_get(check_mask, 10, 11);
    cpl_test_eq(mask_val, CPL_BINARY_1);
    mask_val = cpl_mask_get(check_mask, 10, 12);
    cpl_test_eq(mask_val, CPL_BINARY_1);
    mask_val = cpl_mask_get(check_mask, 10, 13);
    cpl_test_eq(mask_val, CPL_BINARY_0);
    cpl_mask_delete(check_mask);

    /* pixels that are BPM_NON_LINEAR */

    flag_mask = 1<<(BPM_NON_LINEAR-1);
    check_mask = en_master_bpm_get_mask(check, flag_mask);
    count = cpl_mask_count(check_mask);
    cpl_test_eq(count, 3);
    mask_val = cpl_mask_get(check_mask, 10, 11);
    cpl_test_eq(mask_val, CPL_BINARY_1);
    mask_val = cpl_mask_get(check_mask, 10, 12);
    cpl_test_eq(mask_val, CPL_BINARY_1);
    mask_val = cpl_mask_get(check_mask, 10, 13);
    cpl_test_eq(mask_val, CPL_BINARY_1);
    cpl_mask_delete(check_mask);

    /* tidy up */

    en_master_bpm_delete(check);
    cpl_frameset_delete(frameset);
    cpl_mask_delete(mask);
    en_master_bpm_delete(master_bpm_frm);
    cpl_parameterlist_delete(parlist);
    cpl_propertylist_delete(plist);
    cpl_image_delete(rawdata);
    cpl_propertylist_delete(rawdata_plist);
    cpl_frameset_delete(used);
    remove("nix_test_raw_bpm.fits");
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit tests of eris_nix_master_bpm module
 */
/*----------------------------------------------------------------------------*/

int main(void)
{
    cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_WARNING);

    eris_nix_master_bpm_test();

    return cpl_test_end(0);

}

/**@}*/
