/* $Id: eris_ifu_flat.c,v 0.1 2018-07-22 06:06:06 agudo Exp $
*
* This file is part of the ERIS Pipeline
* Copyright (C) 2002,2003 European Southern Observatory
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                Includes
 -----------------------------------------------------------------------------*/
#include <string.h>

#include <cpl.h>
#include <hdrl.h>

#include "eris_ifu_error.h"
#include "eris_dfs.h"
#include "eris_utils.h"
#include "eris_ifu_utils.h"
#include "eris_pfits.h"
#include "eris_ifu_combine_static.h"

/*-----------------------------------------------------------------------------
                            Static variables
 -----------------------------------------------------------------------------*/
static const char eris_ifu_combine_description[] = "\
Addapted from sinfo_utl_cube_combine.                           \n\
This recipe performs cube combination. The input files are several cubes created \n\
by eris_ifu_jitter, their associated tags should be OBJECT_CUBE.                 \n\
The user must provide offsets in an ASCII file named offset.list located in the  \n\
same dir where the esorex command is given.                                      \n\
Its content should be two columns with the offsets of the corresponding data     \n\
cubes that can be obtained, for example, with the command: \n\
 \n\
   dfits  pro/eris_ifu_jitter_obj_cube_00*.fits | fitsort OCS.OFFSET.X OCS.OFFSET.Y > offset.list \n\
\n\
The output is a cube COMBINED_CUBE resulting from the input cubes.\n\
\n\
\n\
--compute_mode=MEDIAN should only be used when all data cubes have the same \n\
exposure time. If these differ MEAN should be used.\n\
When using MEAN the units of the wavelength axis are ADU/s, when using MEDIAN \n\
the units of the wavelength axis are ADU. \n\
\n\
-----------------------------------------------------------------------------  \n\
Input files:\n\
    DO CATG              Explanation                          Required #Frames \n\
    -------              -----------                          -------- ------- \n\
    OBJECT_CUBE          Output cube from eris_ifu_jitter        Y       [1,N] \n\
 \n\
Output files:\n\
    DO CATG              Explanation                          Product Depth \n\
    -------              ----------------------------------   ------------- \n\
    COMBINED_CUBE        Combined cube.\n\
    ----------------------------------------------------------------------------\n\
\n\
    Information on relevant parameters may be found with\n\
    esorex --params "REC_NAME_COMBINE"\n\
    esorex --help   "REC_NAME_COMBINE"\n";

/*-----------------------------------------------------------------------------
                            Private function prototypes
 -----------------------------------------------------------------------------*/

cpl_recipe_define(eris_ifu_combine, ERIS_BINARY_VERSION, "Andrea Modigliani, Y. Cao",
                  PACKAGE_BUGREPORT, "2021",
                  "This recipe combines jittered cubes into a single cube",
                  eris_ifu_combine_description);

cpl_error_code eris_ifu_combine_check_inputs(cpl_frameset *);

/*-----------------------------------------------------------------------------
                                Function code
 -----------------------------------------------------------------------------*/
/**
  @brief    Setup the recipe options
  @param    pl  the nonn-NULL parameterlist to fill
  @return   CPL_ERROR_NONE iff everything is ok

  Defining the command-line/configuration parameters for the recipe.
 */
static cpl_error_code eris_ifu_combine_fill_parameterlist(cpl_parameterlist *pl)
{
    cpl_parameter   *p              = NULL;
    char            *recname_full   = NULL;
    cpl_error_code  err             = CPL_ERROR_NONE;

    cpl_ensure_code(pl, CPL_ERROR_NULL_INPUT);

    TRY {
        recname_full = cpl_sprintf("eris.%s", REC_NAME_COMBINE);

        /* Fill the parameters list */
        p = cpl_parameter_new_value("eris.eris_ifu_combine.offset_mode",
                                    CPL_TYPE_BOOL,
                                    "Offset conventions. If TRUE: applies reference "
                                    "offset correction. If FALSE: take user offsets. "
                                    "The reference offset is computed as (min_off+max_off)/2",
                                    recname_full, CPL_TRUE) ;
        cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "offset_mode") ;
        cpl_parameterlist_append(pl, p) ;

        p = cpl_parameter_new_enum("eris.eris_ifu_combine.compute_mode",
                                   CPL_TYPE_STRING,
                                   "Coadding compute mode of the combined cube",
                                   recname_full, "MEAN", 2,
                                   "MEAN", "MEDIAN");
        cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "compute_mode");
        cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
        cpl_parameterlist_append(pl, p) ;

        p = cpl_parameter_new_value("eris.eris_ifu_combine.name_i",
                                    CPL_TYPE_STRING,
                                    "Input filename. This must be provided when offset_mode=FALSE "
                                    "Allows the user to set X  and Y cumulative offsets in "
                                    "a two column format",
                                    recname_full, "offset.list");
        cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "name_i") ;
        cpl_parameterlist_append(pl, p) ;

        p = cpl_parameter_new_value("eris.eris_ifu_combine.name_o",
                                    CPL_TYPE_STRING,
                                    "Output filename",
                                    recname_full, "out_coadd_cube.fits");
        cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "name_o") ;
        cpl_parameterlist_append(pl, p) ;

        p = cpl_parameter_new_value("eris.eris_ifu_combine.ks_clip",
                                    CPL_TYPE_BOOL,
                                    "Kappa sigma clipping",
                                    recname_full, FALSE);
        cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "ks_clip") ;
        cpl_parameterlist_append(pl, p) ;

        p = cpl_parameter_new_value("eris.eris_ifu_combine.pclip",
                                    CPL_TYPE_INT,
                                    "Apply an initial percentile clipping based on"
                                    "the absolute deviation from the median if "
                                    "ks_clip=TRUE",
                                    recname_full, 70);
        cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "pclip") ;
        cpl_parameterlist_append(pl, p) ;

        p = cpl_parameter_new_value("eris.eris_ifu_combine.kappa",
                                    CPL_TYPE_DOUBLE,
                                    "Kappa value for sigma clip",
                                    recname_full, 2.);
        cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "kappa") ;
        cpl_parameterlist_append(pl, p) ;

        p = cpl_parameter_new_value("eris.eris_ifu_combine.subtract-background",
                                    CPL_TYPE_BOOL,
                                    "Subtract median of the images chanel-by-chanel",
                                    recname_full, FALSE);
        cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "subtract-background") ;
        cpl_parameterlist_append(pl, p) ;

        p = cpl_parameter_new_value("eris.eris_ifu_combine.edge-trim",
                                    CPL_TYPE_INT,
                                    "Number or pixels to trim for each plane of the "
                                    "input frames",
                                    recname_full, 2);
        cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "edge-trim") ;
        cpl_parameterlist_append(pl, p);

        p = cpl_parameter_new_value("eris.eris_ifu_combine.weights",
                                    CPL_TYPE_STRING,
                                    "Optional input filename. The user can specify "
                                    "his own weighting scheme when compute_mode=MEAN "
                                    "A one column format is expected. ",
                                    recname_full, "");
        cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "weights") ;
        cpl_parameterlist_append(pl, p) ;
    } CATCH {
        CATCH_MSGS();
        err = cpl_error_get_code();
    }

    eris_ifu_free_string(&recname_full);

    return err;
}

/**
  @brief    Interpret the command line options and execute the data processing
  @param    frameset   the frames list
  @param    parlist    the parameters list
  @return   0 if everything is ok
 */
static int eris_ifu_combine(cpl_frameset            *frameset,
                            const cpl_parameterlist *parlist)
{
    int                 cnt                 = 0,
                        edge_trim           = 0,
                        pclip               = 0,
                        nx_out              = 0,
                        ny_out              = 0,
                        ks_clip             = 0,
                        min_size_x          = 9999,
                        min_size_y          = 9999,
                        nz                  = 0,
                        nframes             = 0;
    float               ref_offx            = 0,
                        ref_offy            = 0,
                        tmpoffx             = 0,
                        weights             = 0,
                        tmpoffy             = 0,
                        x0                  = 0,
                        y0                  = 0,
                        *offsetx            = NULL,
                        *offsety            = NULL;
    double              kappa               = 0.,
                        *exptimes           = NULL;
    bool                subtract_background = CPL_FALSE,
                        offset_mode         = CPL_FALSE;
    const char          *name_o             = NULL,
                        *name_i             = NULL,
                        *weights_i          = NULL,
                        **files             = NULL,
                        *name               = NULL,
                        *compute_mode       = NULL;
    FILE                *file_list          = NULL;
    const cpl_parameter *param              = NULL;
    cpl_propertylist    *plist              = NULL;
    const cpl_frame     *frame              = NULL;
    cpl_imagelist       *mergedCubeData     = NULL,
                        *mergedCubeError    = NULL,
                        *mergedCubeDIT      = NULL;
    cpl_image           **mergedImageData   = NULL,
                        **mergedImageError  = NULL,
                        **mergedImageDIT    = NULL;
    cpl_error_code      err                 = CPL_ERROR_NONE;

    cpl_ensure_code(parlist != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(frameset != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(eris_files_dont_exist(frameset) == CPL_ERROR_NONE, CPL_ERROR_BAD_FILE_FORMAT);

    TRY
    {
        /* Get Parameters */
        param = cpl_parameterlist_find_const(parlist,
                            "eris.eris_ifu_combine.ks_clip");
        ks_clip = cpl_parameter_get_bool(param);

        param = cpl_parameterlist_find_const(parlist,
                            "eris.eris_ifu_combine.pclip");
        pclip = cpl_parameter_get_int(param);

        param = cpl_parameterlist_find_const(parlist,
                            "eris.eris_ifu_combine.subtract-background");
        subtract_background = cpl_parameter_get_bool(param);

        param = cpl_parameterlist_find_const(parlist,
                            "eris.eris_ifu_combine.kappa");
        kappa = cpl_parameter_get_double(param);

        param = cpl_parameterlist_find_const(parlist,
                            "eris.eris_ifu_combine.name_i");
        name_i = cpl_parameter_get_string(param);

        param = cpl_parameterlist_find_const(parlist,
                            "eris.eris_ifu_combine.name_o");
        name_o = cpl_parameter_get_string(param);

        param = cpl_parameterlist_find_const(parlist,
                                "eris.eris_ifu_combine.offset_mode");
        offset_mode = cpl_parameter_get_bool(param);

        param = cpl_parameterlist_find_const(parlist,
                                "eris.eris_ifu_combine.edge-trim");
        edge_trim = cpl_parameter_get_int(param);

        param = cpl_parameterlist_find_const(parlist,
                                "eris.eris_ifu_combine.compute_mode");
        compute_mode = cpl_parameter_get_string(param);

        param = cpl_parameterlist_find_const(parlist,
                                             "eris.eris_ifu_combine.weights");
        weights_i = cpl_parameter_get_string(param);


        /* Identify the RAW and CALIB frames in the input frameset */
        BRK_IF_ERROR(
            eris_dfs_set_groups(frameset));

        /* Check the inputs frameset consistency */
        BRK_IF_ERROR(
            eris_ifu_combine_check_inputs(frameset));

        /* Check the inputs offset list consistency */
        if (!offset_mode) {
            if (NULL == (file_list = fopen (name_i, "r" )) ){
                BRK_WITH_ERROR_MSG(
                    CPL_ERROR_ILLEGAL_INPUT, "cannot open %s\n", name_i);
            }
            cnt = 0 ;
            while (fscanf(file_list, "%f %f",&tmpoffx, &tmpoffy) != EOF ){
                cnt++;
            }
            fclose(file_list);

            if (cnt != cpl_frameset_get_size(frameset)) {
                BRK_WITH_ERROR_MSG(
                    CPL_ERROR_INCOMPATIBLE_INPUT, "Input offlist is not consistent with frameset.");
            }
        }

        /* Check the inputs weights list consistency */
        if (!strcmp(compute_mode, "MEAN") && strcmp(weights_i, "")) {
            if (NULL == (file_list = fopen (weights_i, "r" )) ){
                BRK_WITH_ERROR_MSG(
                    CPL_ERROR_ILLEGAL_INPUT, "cannot open %s\n", weights_i);
            }
            cnt = 0 ;
            while (fscanf(file_list, "%f", &weights) != EOF ){
                cnt++;
            }
            fclose(file_list);

            if (cnt != cpl_frameset_get_size(frameset)) {
                BRK_WITH_ERROR_MSG(
                    CPL_ERROR_INCOMPATIBLE_INPUT, "Input list of weights is not consistent with frameset.");
            }
        }

        /* Allocate memory */
        nframes = cpl_frameset_get_size(frameset);

        BRK_IF_NULL(
            exptimes = (double*) cpl_calloc(nframes, sizeof(double)));
        BRK_IF_NULL(
            offsetx = (float*) cpl_calloc(nframes, sizeof(float)));
        BRK_IF_NULL(
            offsety = (float*) cpl_calloc(nframes, sizeof(float)));

        /* Read offset list */
        if (!offset_mode){
            cpl_msg_info(cpl_func, "Reading user-specified offset list.") ;
            file_list = fopen (name_i, "r" );

            cnt = 0;
            while (fscanf( file_list, "%f %f",&tmpoffx,&tmpoffy ) != EOF ) {
                offsetx[cnt] = tmpoffx;
                offsety[cnt] = tmpoffy;
                if (cnt == 0) {
                    x0 = offsetx[0];
                    y0 = offsety[0];
                }
                offsetx[cnt] = offsetx[cnt]-x0;
                offsety[cnt] = offsety[cnt]-y0;
                cpl_msg_info(cpl_func, "   User defined offsets RA, DEC in pixels (relative to the first) %.2f, %.2f",
                            offsetx[cnt], offsety[cnt]) ;
                cnt++;
            }
            fclose(file_list);
        } else {
            cpl_msg_info(cpl_func, "Use CUMOFFS from the header.") ;

            double  cd1_1   = 0,
                    cd1_2   = 0,
                    cd2_1   = 0,
                    cd2_2   = 0,
                    dra     = 0,
                    ddec    = 0;
            const cpl_matrix *cd = NULL;

            for (int n = 0; n < nframes; n++){
                frame = cpl_frameset_get_position_const(frameset, n);
                name = cpl_frame_get_filename(frame);

                plist = cpl_propertylist_load(name, 0);
                dra =  cpl_propertylist_get_double(plist, "ESO OCS CUMOFFS RA");
                ddec =  cpl_propertylist_get_double(plist, "ESO OCS CUMOFFS DEC");
                eris_ifu_free_propertylist(&plist);

                plist = cpl_propertylist_load(name, 1);
                cpl_wcs *tmp_wcs = cpl_wcs_new_from_propertylist(plist);
                eris_ifu_free_propertylist(&plist);

                cd = cpl_wcs_get_cd(tmp_wcs);
                cd1_1 = cpl_matrix_get(cd, 0, 0);
                cd1_2 = cpl_matrix_get(cd, 1, 0);
                cd2_1 = cpl_matrix_get(cd, 0, 1);
                cd2_2 = cpl_matrix_get(cd, 1, 1);
                cpl_wcs_delete(tmp_wcs);

                dra = -1 * dra /3600.0; //convert to degree
                ddec = -1 * ddec/3600.0;
                offsetx[n] = (float) (cd2_2 * dra - cd1_2 * ddec)/(cd1_1*cd2_2 - cd1_2*cd2_1);
                offsety[n] = (float) (cd1_1 * ddec - cd2_1 * dra)/(cd1_1*cd2_2 - cd1_2*cd2_1);

                if (n == 0) {
                    x0 = offsetx[0];
                    y0 = offsety[0];
                }
                offsetx[n] = offsetx[n]-x0;
                offsety[n] = offsety[n]-y0;

                cpl_msg_info(cpl_func, "   CUMOFFS RA, DEC in pixels (relative to the first): %.2f, %.2f",
                            offsetx[n], offsety[n]) ;

            }
        } // end if: offset mode
        CHECK_ERROR_STATE();

        /* Calculate size of new cube*/
        BRK_IF_NULL(
            files = (const char**) cpl_calloc(MAX_NAME_SIZE, sizeof(const char*)));

        for (int n = 0; n < nframes; n++){
            frame = cpl_frameset_get_position_const(frameset, n);
            files[n] = cpl_frame_get_filename(frame);
            exptimes[n] = eris_pfits_get_exptime(files[n]);

            plist = cpl_propertylist_load(files[n], 1);
            nx_out = eris_pfits_get_naxis1(plist);
            ny_out = eris_pfits_get_naxis2(plist);
            eris_ifu_free_propertylist(&plist);

            if (nx_out < min_size_x) {
                min_size_x = nx_out;
            }
            if (ny_out < min_size_y) {
                min_size_y = ny_out;
            }
        }
        CHECK_ERROR_STATE();

        /* replace exptime by weights if needed */
        if (!strcmp(compute_mode, "MEAN") && strcmp(weights_i, "")) {
            cpl_msg_info(cpl_func, "Reading user-specified weights list.") ;
            file_list = fopen (weights_i, "r" );

            cnt = 0 ;
            while (fscanf(file_list, "%f", &weights) != EOF ){
                exptimes[cnt] = weights;
                cnt++;
            }
            fclose(file_list);
        }

        cpl_msg_info(cpl_func, "Min. input cube size x = %d, y = %d", min_size_x, min_size_y);

        BRK_IF_ERROR(
            eris_ifu_combine_auto_size_cube(offsetx, offsety, nframes,
                                            &ref_offx, &ref_offy, &nx_out, &ny_out));

        if (edge_trim > 0){
            cpl_msg_info(cpl_func, "Number of pixels on edges to be trimmed: %d", edge_trim);
        }

        if(subtract_background) {
            cpl_msg_info(cpl_func,"Subtract spatial median to each cube plane");
        }

        if (ks_clip) {
            cpl_msg_info(cpl_func, "Coadding clipping mode: TRUE");
        } else {
            cpl_msg_info(cpl_func, "Coadding clipping mode: FALSE");
            kappa = -1;
            pclip = -1;
        }
        cpl_msg_info(cpl_func, "Coadding compute mode:  %s", compute_mode);

        nz = eris_ifu_combine_min_cube_size(frameset);
        CHECK_ERROR_STATE();

        BRK_IF_NULL(
            mergedImageData  = cpl_calloc(nz, sizeof(cpl_image*)));
        BRK_IF_NULL(
            mergedImageError = cpl_calloc(nz, sizeof(cpl_image*)));
        BRK_IF_NULL(
            mergedImageDIT   = cpl_calloc(nz, sizeof(cpl_image*)));
        BRK_IF_NULL(
            mergedCubeData = cpl_imagelist_new());
        BRK_IF_NULL(
            mergedCubeError = cpl_imagelist_new());
        BRK_IF_NULL(
            mergedCubeDIT = cpl_imagelist_new());
    } CATCH {
        // catch already here because OPENMP can't handle the error handling macros
        CATCH_MSGS();
        return (int)cpl_error_get_code();
    }

    /* Coadding cubes */
    cpl_msg_info(cpl_func,"   Processing %d planes...", nz);
HDRL_OMP(omp parallel for reduction(max:err))
    for (int z = 0; z < nz; z++) {
        cpl_msg_debug(cpl_func,"   Processing plane [%4.4d] of %d", z, nz);
        cpl_image **imagesData  = cpl_calloc(nframes, sizeof(cpl_image*)),
                  **imagesError = cpl_calloc(nframes, sizeof(cpl_image*));

        err = eris_ifu_combine_read_image_planes(frameset,
                                                 imagesData, imagesError,
                                                 z, edge_trim, subtract_background);

        err = eris_ifu_combine_jittered_images(imagesData,
                                              imagesError,
                                              nx_out,
                                              ny_out,
                                              &mergedImageData[z],
                                              &mergedImageError[z],
                                              &mergedImageDIT[z],
                                              nframes,
                                              offsetx, offsety,
                                              exptimes,
                                              kappa,
                                              compute_mode,
                                              pclip);

        for (int i = 0; i < nframes; i++) {
            eris_ifu_free_image(&imagesData[i]);
            eris_ifu_free_image(&imagesError[i]);
        }
        cpl_free(imagesData);
        cpl_free(imagesError);

        eris_ifu_combine_convert_0_to_NaN_img(mergedImageData[z]);
        eris_ifu_combine_convert_0_to_NaN_img(mergedImageError[z]);
        eris_ifu_combine_convert_0_to_NaN_img(mergedImageDIT[z]);
    } // end for(z)

    if (err != CPL_ERROR_NONE) {
        cpl_msg_error(__func__, "Combining jittered images failed!");
        cpl_error_set(__func__, CPL_ERROR_ILLEGAL_INPUT);
        return (int)cpl_error_get_code();
    }

    for (int z = 0; z < nz; z++) {
        if (!mergedImageData[z]) {
            mergedImageData[z] = cpl_image_new(nx_out, ny_out, CPL_TYPE_DOUBLE);
        }
        cpl_imagelist_set(mergedCubeData, mergedImageData[z], z);

        if (!mergedImageError[z]) {
            mergedImageError[z] = cpl_image_new(nx_out,ny_out,CPL_TYPE_DOUBLE);
        }
        cpl_imagelist_set(mergedCubeError, mergedImageError[z], z);

        if (!mergedImageDIT[z]) {
            mergedImageDIT[z] = cpl_image_new(nx_out,ny_out,CPL_TYPE_DOUBLE);
        }
        cpl_imagelist_set(mergedCubeDIT, mergedImageDIT[z], z);
    }

    /* Save combined cube */
    frame = cpl_frameset_get_position_const(frameset, 1);
    plist = cpl_propertylist_load(files[0], 0);
    cpl_propertylist_set_string(plist, CPL_DFS_PRO_CATG, ERIS_IFU_PRO_COMBINE);
    cpl_dfs_save_propertylist(frameset, NULL, parlist, frameset, frame, REC_NAME_COMBINE,
                                  plist, NULL, PACKAGE "/" PACKAGE_VERSION, name_o);
    eris_ifu_free_propertylist(&plist);

    plist = cpl_propertylist_load(files[0], 1);

    double crpix1 = cpl_propertylist_get_double(plist, "CRPIX1"),
           crpix2 = cpl_propertylist_get_double(plist, "CRPIX2");

    cpl_propertylist_update_double(plist, "CRPIX1", crpix1+ref_offx*2-1);
    cpl_propertylist_update_double(plist, "CRPIX2", crpix2+ref_offy*2-1);
    cpl_propertylist_update_string(plist, "EXTNAME", EXTNAME_DATA);
    cpl_propertylist_set_comment(plist, "EXTNAME", EXTNAME_DATA_COMMENT);
    cpl_propertylist_update_string(plist, "BUNIT", "adu/s");
    cpl_imagelist_save(mergedCubeData, name_o, CPL_TYPE_DOUBLE, plist, CPL_IO_EXTEND);

    cpl_propertylist_update_string(plist, "EXTNAME", EXTNAME_ERROR);
    cpl_propertylist_set_comment(plist, "EXTNAME", EXTNAME_ERROR_COMMENT);
    cpl_propertylist_update_string(plist, "BUNIT", "adu/s");
    cpl_imagelist_save(mergedCubeError, name_o, CPL_TYPE_DOUBLE, plist, CPL_IO_EXTEND);

    cpl_propertylist_update_string(plist, "EXTNAME", EXTNAME_DQ);
    cpl_propertylist_set_comment(plist, "EXTNAME", EXTNAME_DQ_COMMENT);
    cpl_propertylist_update_string(plist, "BUNIT", "sec");
    cpl_imagelist_save(mergedCubeDIT, name_o, CPL_TYPE_DOUBLE, plist, CPL_IO_EXTEND);

    eris_ifu_free_propertylist(&plist);
    for (int i = 0; i < nz; i++) {
        eris_ifu_free_image(&mergedImageData[i]);
        eris_ifu_free_image(&mergedImageError[i]);
        eris_ifu_free_image(&mergedImageDIT[i]);
    }
    cpl_free(mergedImageData);
    cpl_free(mergedImageError);
    cpl_free(mergedImageDIT);
    cpl_free(exptimes);
    cpl_free(offsetx);
    cpl_free(offsety);
    cpl_free(files);

    return (int)cpl_error_get_code();
}

/**
  @brief    Make consistency checks
  @param    frameset        Set of frames
  @return   1 if consistent, 0 if not, -1 in error case
 */
cpl_error_code eris_ifu_combine_check_inputs(cpl_frameset *frameset)
{
    const cpl_frame *frame  = NULL;
    const char      *tag    = NULL;
    int             fs_size = cpl_frameset_get_size(frameset);

    /* Check number of frames */
    if (fs_size < 2){
        cpl_error_set(cpl_func, CPL_ERROR_ILLEGAL_INPUT);
        cpl_msg_error(cpl_func, "Number of frames in SOF less than 2") ;
        return cpl_error_get_code();
    }

    /* Check tags */
    for (int i = 0; i < fs_size; i++){
        frame = cpl_frameset_get_position_const(frameset, i);
        tag = cpl_frame_get_tag(frame);
        if (strcmp(tag, ERIS_IFU_COMBINE_DOCATG)) {
            cpl_error_set(cpl_func, CPL_ERROR_ILLEGAL_INPUT);
            cpl_msg_error(cpl_func, "Wrong frame tag %s; should be %s",
                          tag, ERIS_IFU_COMBINE_DOCATG) ;
            return cpl_error_get_code();
        }
    }
    return cpl_error_get_code();
}
