/* $Id: eris_ifu_flat.c,v 0.1 2018-07-22 06:06:06 agudo Exp $
*
* This file is part of the ERIS Pipeline
* Copyright (C) 2002,2003 European Southern Observatory
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                Includes
 -----------------------------------------------------------------------------*/
#include <cpl.h>
#include "hdrl.h"
#include <string.h>
#include <stdio.h>
#include "eris_dfs.h"
#include "eris_utils.h"
#include "eris_ifu_dfs.h"
#include "eris_ifu_utils.h"
#include "eris_ifu_error.h"
#include "eris_ifu_resample.h"
#include "eris_ifu_efficiency_response.h"
#include "eris_ifu_jitter_interface.h"

/*-----------------------------------------------------------------------------
                             Plugin registration
 -----------------------------------------------------------------------------*/
/*-----------------------------------------------------------------------------
                            Private function prototypes
 -----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------
                            Static variables
 -----------------------------------------------------------------------------*/
#define RECIPE_NAME "eris_ifu_combine_hdrl"
#define ERIS_IFU_COMBINE_DOCATG "OBJECT_CUBE"
#define ERIS_IFU_COMBINE_PROCATG "COMBINED_CUBE"

#define MAX_NAME_SIZE 512
#define CONTEXT "eris.eris_ifu_combine_hdrl"
static const char eris_ifu_combine_description[] = "\
Align and combine SPIFFIER data cubes \n\
\n\
-----------------------------------------------------------------------------\n\
Input files:\n\
   DO CATG             Explanation                          Required #Frames\n\
   -------             -----------                          -------- -------\n\
   OBJECT_CUBE         Science cubes                         Y       >=2 \n\
   EXTCOEFF_TABLE      Table with atmospheric extinction  (optional, req. to fluxcal [0,1]\n\
   RESPONSE            Table with instrument response (optional, req. to fluxcal) [0,1]\n\
\n\
Output files:\n\
   PRO CATG                 Explanation\n\
   -------                  -----------\n\
    "ERIS_IFU_COMBINE_PROCATG"           Combined scince cube using hdrl library functions. \n\
\n\
Information on relevant parameters may be found with\n\
  esorex --params "RECIPE_NAME"\n\
  esorex --help   "RECIPE_NAME"\n\
---------------------------------------------------------------------------\n\
\n";
cpl_recipe_define(eris_ifu_combine_hdrl, ERIS_BINARY_VERSION, "Y. Cao",
                 PACKAGE_BUGREPORT, "2023",
				 "Combine science cubes using hdrl",
				 eris_ifu_combine_description);
/*-----------------------------------------------------------------------------
                                Function code
 -----------------------------------------------------------------------------*/
/*----------------------------------------------------------------------------*/
/**
  @brief    Setup the recipe options    
  @param    pl  the input parameterlist
  @return   0 if everything is ok

  Defining the command-line/configuration parameters for the recipe.
 */
/*----------------------------------------------------------------------------*/
static cpl_error_code eris_ifu_combine_hdrl_fill_parameterlist(cpl_parameterlist *pl)
{
   //cpl_recipe    * recipe;
   cpl_parameter * p;

    cpl_ensure_code(pl, CPL_ERROR_NULL_INPUT);

    /* Fill the parameters list */
    eris_parlist_config_add_all_recipes(pl,RECIPE_NAME);
    /* Offsets */
    p = cpl_parameter_new_value(CONTEXT".offset_mode",
                       CPL_TYPE_BOOL, "Offset conventions. If TRUE applies "
                                      "reference offset correction. If FALSE: "
                                      "take user offsets. The reference offset "
                                      "is computed as (min_off+max_off)/2",
                       CONTEXT, CPL_TRUE) ;
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "offset_mode") ;
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(pl, p) ;

    p = cpl_parameter_new_value(CONTEXT".name_i",
                    CPL_TYPE_STRING,
                    "Input filename. This must be provided and allow the user to set X "
                    "and Y cumulative offsets in a two column format",
                    CONTEXT,"offset.list");
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "name_i") ;
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(pl, p) ;

    p = cpl_parameter_new_enum(CONTEXT".offset_unit", CPL_TYPE_STRING,
        "Offset unit", CONTEXT, "PIXEL",4,
        "PIXEL", "ARCSEC", "DEGREE", "PIXDEG");
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "offset_unit");
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(pl, p);    


    /* Resampling method */
    /* --hdrldemo_resample.method */
    char            *context = NULL;
    context = cpl_sprintf("eris.%s", RECIPE_NAME);
    char            pName[256]; 
    snprintf(pName, sizeof(pName), "%s.%s", context, "method");
    p = cpl_parameter_new_enum(pName, CPL_TYPE_STRING,
        "Resampling method", context, "DRIZZLE",6,
        "NEAREST", "LINEAR", "QUADRATIC", "RENKA",
        "DRIZZLE","LANCZOS");
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "method");
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(pl, p);

    /* --hdrldemo_resample.method.loop-distance */
    snprintf(pName, sizeof(pName), "%s.%s", context, "method.loop-distance");
    p = cpl_parameter_new_value(pName, CPL_TYPE_INT,
        "Loop distance used by all (but NEAREST) "
        "methods to control the number of surrounding "
        "voxels that are taken into account. "
        "A small value allow faster re-sampling but may not give good quality",
        context, LOOP_DISTANCE);
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "method.loop-distance");
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(pl, p);

    /* --hdrldemo_resample.method.use-errorweights*/
    snprintf(pName, sizeof(pName), "%s.%s", context, "method.use-errorweights");
    p = cpl_parameter_new_value(pName,
        CPL_TYPE_BOOL,
        "Use additional weights of 1/err^2", context,
        CPL_FALSE);
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI,
            "method.use-errorweights");
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(pl, p);

    /* --hdrldemo_resample.method.renka.critical-radius */
    snprintf(pName, sizeof(pName), "%s.%s", context, "method.renka.critical-radius");
    p = cpl_parameter_new_value(pName,
        CPL_TYPE_DOUBLE,"Critical radius of the Renka "
        "method",
        context, RENKA_CRITICAL_RADIUS);
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI,
            "method.renka.critical-radius");
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(pl, p);

    /* --hdrldemo_resample.method.lanczos.kernel-size */
    snprintf(pName, sizeof(pName), "%s.%s", context, "method.lanczos.kernel-size");
    p = cpl_parameter_new_value(pName,
        CPL_TYPE_INT,"Kernel size of the Lanczos "
        "method",
        context, LANCZOS_KERNEL_SIZE);
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI,
            "method.lanczos.kernel-size");
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(pl, p);

    /* --hdrldemo_resample.method.drizzle.downscale-x */
    snprintf(pName, sizeof(pName), "%s.%s", context, "method.drizzle.downscale-x");
    p = cpl_parameter_new_value(pName,
        CPL_TYPE_DOUBLE, "Drizzle down-scaling factor "
        "in x direction",
        context, DRIZZLE_DOWN_SCALING_FACTOR_X);
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI,
            "method.drizzle.downscale-x");
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(pl, p);

    /* --hdrldemo_resample.method.drizzle.downscale-y */
    snprintf(pName, sizeof(pName), "%s.%s", context, "method.drizzle.downscale-y");
    p = cpl_parameter_new_value(pName,
        CPL_TYPE_DOUBLE, "Drizzle down-scaling factor "
        "in y direction",
        context, DRIZZLE_DOWN_SCALING_FACTOR_Y);
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI,
            "method.drizzle.downscale-y");
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(pl, p);

    /* --hdrldemo_resample.method.drizzle.downscale-z*/
    snprintf(pName, sizeof(pName), "%s.%s", context, "method.drizzle.downscale-z");
    p = cpl_parameter_new_value(pName,
        CPL_TYPE_DOUBLE, "Drizzle down-scaling factor "
        "in wavelength direction",
        context, DRIZZLE_DOWN_SCALING_FACTOR_Z);
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI,
            "method.drizzle.downscale-z");
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(pl, p);

    /* --hdrldemo_resample.outgrid.ra-min
    snprintf(pName, sizeof(pName), "%s.%s", context, "outgrid.ra-min");
    p = cpl_parameter_new_value(pName, CPL_TYPE_DOUBLE,
        "Minimum right ascension of the output "
        "image/cube in degree", context, -1.0);
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "outgrid.ra-min");
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(pl, p);
    */

    /* --hdrldemo_resample.outgrid.ra-max
    snprintf(pName, sizeof(pName), "%s.%s", context, "outgrid.ra-max");
    p = cpl_parameter_new_value(pName, CPL_TYPE_DOUBLE,
        "Maximum right ascension of the output "
        "image/cube in degree", context, -1.0);
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "outgrid.ra-max");
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(pl, p);
    */

    /* --hdrldemo_resample.outgrid.dec-min
    snprintf(pName, sizeof(pName), "%s.%s", context, "outgrid.dec-min");
    p = cpl_parameter_new_value(pName, CPL_TYPE_DOUBLE,
        "Minimum declination of the output "
        "image/cube in degree", context, -1.0);
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "outgrid.dec-min");
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(pl, p);
    */

    /* --hdrldemo_resample.outgrid.dec-max
    snprintf(pName, sizeof(pName), "%s.%s", context, "outgrid.dec-max");
    p = cpl_parameter_new_value(pName, CPL_TYPE_DOUBLE,
        "Maximum declination of the output "
        "image/cube in degree", context, -1.0);
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "outgrid.dec-max");
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(pl, p);
    */

    /* --hdrldemo_resample.lambda
    snprintf(pName, sizeof(pName), "%s.%s", context, "outgrid.lambda-min");
    p = cpl_parameter_new_value(pName,
        CPL_TYPE_DOUBLE, "Minimum wavelength of the "
        "output cube in meter", context, -1.);
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "outgrid.lambda-min");
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(pl, p);
    */

    /* --hdrldemo_resample.lambda
    snprintf(pName, sizeof(pName), "%s.%s", context, "outgrid.lambda-max");
    p = cpl_parameter_new_value(pName,
        CPL_TYPE_DOUBLE, "Maximum wavelength of the "
        "output cube in meter", context, -1.);
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "outgrid.lambda-max");
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(pl, p);
    */

    /* --hdrldemo_resample.outgrid.delta-ra
    snprintf(pName, sizeof(pName), "%s.%s", context, "outgrid.delta-ra");
    p = cpl_parameter_new_value(pName,CPL_TYPE_DOUBLE,
        "Output step-size in right ascension in degree",
        context, -1.);
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "outgrid.delta-ra");
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(pl, p);
    */

    /* --hdrldemo_resample.outgrid.delta-dec
    snprintf(pName, sizeof(pName), "%s.%s", context, "outgrid.delta-dec");
    p = cpl_parameter_new_value(pName,
        CPL_TYPE_DOUBLE, "Output step-size in "
        "declination in degree",
        context, -1.);
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "outgrid.delta-dec");
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(pl, p);
    */

    /* --hdrldemo_resample.outgrid.delta-lambda
    snprintf(pName, sizeof(pName), "%s.%s", context, "outgrid.delta-lambda");
    p = cpl_parameter_new_value(pName,
        CPL_TYPE_DOUBLE, "Output step-size in "
        "wavelength in meter",
        context, -1.);
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "outgrid.delta-lambda");
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(pl, p);
    */

    // /* --hdrldemo_resample.save-table */
    // snprintf(pName, sizeof(pName), "%s.%s", context, "save-table");
    // p = cpl_parameter_new_value(pName, CPL_TYPE_BOOL,
    //     "Save the table before resampling", context,
    //     CPL_FALSE);
    // cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "save-table");
    // cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    // cpl_parameterlist_append(pl, p);


    /* --hdrldemo_resample.subtract-background */
    snprintf(pName, sizeof(pName), "%s.%s", context, "subtract-background");
    p = cpl_parameter_new_value(pName, CPL_TYPE_BOOL,
        "Subtract median of the images chanel-by-chanel", context,
        CPL_FALSE);
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "subtract-background");
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(pl, p);

    /* EXTRA CODE TO FLUX CALIBRATE RESULTS */
    int jitterMode= M_SCIENCE;

    /* extraction params */
    eris_ifu_jitter_fill_extract_parameterlist(context, jitterMode, pl);
    /* flux calibration params */
    snprintf(pName, sizeof(pName), "%s.%s", context, "flux-calibrate");
    p = cpl_parameter_new_value(pName, CPL_TYPE_BOOL,
    		"If True flux calibrate the extracted spectrum and data cube (you need to also --extract-source=TRUE)",
			context, CPL_FALSE);
    if (jitterMode == M_SCIENCE) {
    	cpl_parameter_set_default_bool(p, CPL_FALSE);
    }
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "flux-calibrate");
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(pl, p);

    /* END EXTRA */


    /* --hdrldemo_resample.fieldmargin */
    snprintf(pName, sizeof(pName), "%s.%s", context, "fieldmargin");
    p = cpl_parameter_new_value(pName, CPL_TYPE_DOUBLE,
        "Add this margin/border (in percent) to the "
        "resampled image/cube", context, FIELDMARGIN);
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "fieldmargin");
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(pl, p);

    /* --hdrldemo_resample.edge-trim */
    snprintf(pName, sizeof(pName), "%s.%s", context, "edge-trim");
    p = cpl_parameter_new_value(pName, CPL_TYPE_INT,
        "Number or pixels to trim for each plane of the input frames. "
        "It should be smaller than half image size", context, EDGETRIM);
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "edge-trim");
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(pl, p);

    snprintf(pName, sizeof(pName), "%s.%s", context, "max-cubes-centres-dist");
    p = cpl_parameter_new_range(pName, CPL_TYPE_INT,
    "Maximum distance between cube centers to build a mosaic. Mosaic creation "
    "requires a lot of RAM. Users may trim this value to fit RAM resources",
    context, 240, 20, 10000);
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "max-cubes-centres-dist");
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(pl, p);



	snprintf(pName, sizeof(pName), "%s.%s", context, "bpc_iter");
	p = cpl_parameter_new_value(pName, CPL_TYPE_INT,
			"No. of iterations for bad pixel correction",
			context, 1);
	cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "bpc_iter");
	cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
	cpl_parameterlist_append(pl, p);




    snprintf(pName, sizeof(pName), "%s.%s", context, "chop-nan");
    p = cpl_parameter_new_value(pName, CPL_TYPE_BOOL,
    		"If true chop cube planes with more than 50% NAN pixels",
			context, CPL_FALSE);
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "chop-nan");
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(pl, p);




    eris_ifu_free_string(&context);

    return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Interpret the command line options and execute the data processing
  @param    frameset   the frames list
  @param    parlist    the parameters list
  @return   0 if everything is ok
 */
/*----------------------------------------------------------------------------*/
static int eris_ifu_combine_hdrl(
        cpl_frameset            * frameset, 
        const cpl_parameterlist * parlist)
{
    const cpl_parameter *   par ;
    //struct paramStruct params;
    struct stdParamStruct stdParams = stdParamStructInit;
    //struct sofStruct sof;
    const char* name_i = NULL;
    int offset_mode = CPL_TRUE;
    const char* offunit = NULL;
    FILE* file_list = NULL;
    int cnt = 0;
    float tmpoffx = 0;
    float tmpoffy = 0;

    int nframes = 0;


    float * offsetx = NULL;
    float * offsety =  NULL;
    float * crpix1_usr = NULL;
    float * crpix2_usr = NULL;
    float * crval1_usr = NULL;
    float * crval2_usr = NULL;
    double crpix1 = 0;
    double crpix2 = 0;
    double crval1 = 0;
    double crval2 = 0;
    double alpha = 0;
    double delta = 0;

    cpl_ensure_code(frameset, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(parlist,  CPL_ERROR_NULL_INPUT);
    cpl_error_ensure(cpl_frameset_get_size(frameset) >= 0, CPL_ERROR_ILLEGAL_INPUT,
    		0, "Missing or empty SOF file");



    /* Check entries */
    if (parlist == NULL || frameset == NULL) {
        cpl_msg_error(__func__, "Null Inputs") ;
        cpl_error_set(__func__, CPL_ERROR_NULL_INPUT) ;
        return (int) cpl_error_get_code() ;
    }
    /* check required input tags are present */
    const int ntags = 1;
    const char* required_tags[1] = {
    		ERIS_IFU_COMBINE_DOCATG
    };

    cpl_ensure_code(CPL_ERROR_NONE ==
    		eris_dfs_check_input_tags(frameset, required_tags, ntags, 1),
			CPL_ERROR_ILLEGAL_INPUT);
    const int ntags_opt = 1;
      		const char* optional_tags[2] = {
      				ERIS_IFU_CALIB_EXTCOEFF_TABLE,
  					ERIS_IFU_PRO_JITTER_RESPONSE
      };
      eris_dfs_check_input_tags(frameset, optional_tags, ntags_opt, 0);



    /* Initialise */

    /* Get Parameters */
    par = cpl_parameterlist_find_const(parlist,
                            CONTEXT".offset_mode");
    offset_mode = cpl_parameter_get_bool(par);

    par = cpl_parameterlist_find_const(parlist, 
                        CONTEXT".name_i");
    name_i = cpl_parameter_get_string(par);

    par = cpl_parameterlist_find_const(parlist, 
                        CONTEXT".offset_unit");
    offunit = cpl_parameter_get_string(par);


    /* Identify the RAW and CALIB frames in the input frameset */
    if (eris_dfs_set_groups(frameset) != 0) {
        cpl_msg_error(__func__, "Cannot identify RAW and CALIB frames") ;
        return (int) cpl_error_set(__func__, CPL_ERROR_ILLEGAL_INPUT) ;
    }

    /* Check the inputs consistency */
    cpl_ensure_code(CPL_ERROR_NONE ==
    		eris_dfs_check_input_tags(frameset, required_tags, ntags, 1),
			CPL_ERROR_ILLEGAL_INPUT);

    /* Check the inputs offset list consistency */
    cpl_frameset* cube_set;
    cube_set = eris_ifu_extract_frameset(frameset, ERIS_IFU_COMBINE_DOCATG );
    if (!offset_mode) {
        cpl_msg_info(__func__, "Use user offsets to combine.") ;

        if ( NULL == (file_list = fopen (name_i, "r" )) ){      
            cpl_msg_error(__func__, "cannot open %s\n", name_i) ;
            return (int) cpl_error_set(__func__, CPL_ERROR_ILLEGAL_INPUT) ;
        }
        cnt = 0 ;

        if (strcmp(offunit, "PIXDEG") == 0){
        	while (fscanf( file_list, "%lf %lf %lf %lf",&crpix1,&crpix2, &alpha, &delta ) != EOF ) {
        		cnt ++ ;
        	}

        } else {
        	while ( fscanf( file_list, "%f %f",&tmpoffx, &tmpoffy) != EOF ){
        		cnt ++ ;
        	}
        }
        fclose(file_list);

        if (cnt != cpl_frameset_get_size(cube_set)) {
            cpl_msg_error(__func__, "Input offlist is not consistent with frameset or offunit") ;
            if (strcmp(offunit, "PIXDEG") == 0){
            	cpl_msg_error(__func__, "If offunit is PIXDEG input ASCII file must have four columns");
            } else {
            	cpl_msg_error(__func__, "If offunit is not PIXDEG input ASCII file must have two columns");
            }
            return (int) cpl_error_set(__func__, CPL_ERROR_INCOMPATIBLE_INPUT) ;
        }
    }
    else {
        cpl_msg_info(__func__, "Use wcs in each input cube to combine.") ;
    }


    /* Read offset list */
    if (!offset_mode){
    	cpl_msg_info(__func__, "Reading offset list.") ;
    	//Allocate memories
    	nframes = cnt ;
    	offsetx = (float*) cpl_calloc(nframes, sizeof(float));
    	offsety = (float*) cpl_calloc(nframes, sizeof(float));

    	crpix1_usr = (float*) cpl_calloc(nframes, sizeof(float));
    	crpix2_usr = (float*) cpl_calloc(nframes, sizeof(float));
    	crval1_usr = (float*) cpl_calloc(nframes, sizeof(float));
    	crval2_usr = (float*) cpl_calloc(nframes, sizeof(float));
    	if (offsetx == NULL || offsety == NULL){
    		cpl_msg_error(__func__, "Could not allocate memory!") ;
    		cpl_free(offsetx);
    		cpl_free(offsety);
    		return (int) cpl_error_set(__func__, CPL_ERROR_UNSPECIFIED);
    	}

    	//Read
    	file_list = fopen (name_i, "r" );

    	cnt = 0;

    	if (strcmp(offunit, "PIXDEG") == 0){
    		while (fscanf( file_list, "%lf %lf %lf %lf",&crpix1,&crpix2, &alpha, &delta ) != EOF ) {
    			//        	cpl_frame *frame = cpl_frameset_get_position(frameset, cnt);
    			crpix1_usr[cnt] = crpix1;
    			crpix2_usr[cnt] = crpix2;
    			crval1_usr[cnt] = alpha;
    			crval2_usr[cnt] = delta;
    			cnt ++ ;
    		}
    		nframes = cnt;
    		fclose(file_list);
    	} else {
    		while (fscanf( file_list, "%f %f",&tmpoffx,&tmpoffy ) != EOF ) {
    			//        	cpl_frame *frame = cpl_frameset_get_position(frameset, cnt);
    			offsetx[cnt] = tmpoffx;
    			offsety[cnt] = tmpoffy;
    			cnt ++ ;
    		}
    		nframes = cnt;
    		fclose(file_list);
    	}
    }


    //char *combDoCatg = ERIS_IFU_COMBINE_DOCATG;
    int offcode = 0;
    double as2deg = 1.0/3600.0;
    double scalex = 1.0;
    double scaley = 1.0;
    //cpl_frameset* cube_set = NULL;

	cpl_size ncubes = cpl_frameset_get_size(cube_set);
	cpl_frame* cube_frm = NULL;
	cpl_propertylist* plist = NULL;
	cpl_propertylist* hdata = NULL;
	const char* fname = NULL;
	double dit = 0.0;
	double dit_ref = 0;
	cpl_vector* factor_vec = cpl_vector_new(ncubes);
	double* pfactor = cpl_vector_get_data(factor_vec);
	for(int i = 0; i < ncubes; i++) {

		cube_frm = cpl_frameset_get_position(cube_set, i);
		cpl_frame_set_group(cube_frm, CPL_FRAME_GROUP_RAW);
		fname = cpl_frame_get_filename(cube_frm);
		hdata = cpl_propertylist_load(fname, 0);
		dit = cpl_propertylist_get_double(hdata, FHDR_E_DIT);
		cpl_msg_info(cpl_func,"time[%d]=%g",i,dit);
		if(i == 0) {
			dit_ref = dit;
			pfactor[0] = 1;
		} else {
			pfactor[i] = dit_ref / dit ;
		}
		cpl_propertylist_delete(hdata);
	}
	if(cpl_error_get_code() != CPL_ERROR_NONE) {
		cpl_msg_error(cpl_func,"errors to get exposure time from data. Check your data. Exit.");
		cpl_vector_delete(factor_vec);
		cpl_frameset_delete(cube_set);
		return (int) cpl_error_get_code();
	}
    const char* fname_spec_cube = "cube";
    if (offunit != NULL && offsetx !=NULL && offsety !=NULL){
    	if (strcmp(offunit, "PIXEL") == 0){
    		offcode = 0;
    		cpl_msg_info(cpl_func, "User-defined offset unit in PIXEL");
    	} else if (strcmp(offunit,  "DEGREE") == 0){
    		offcode = 1;
    		cpl_msg_info(cpl_func, "User-defined offset unit is DEGREE");
    	} else if (strcmp(offunit,  "ARCSEC") == 0){
    		offcode = 2;
    		cpl_msg_info(cpl_func, "User-defined offset unit is ARCSEC");
    		scalex =  as2deg;
    		scaley =  as2deg;
    	} else if (strcmp(offunit,  "PIXDEG") == 0){
        		offcode = 3;
        		cpl_msg_info(cpl_func, "User-defined offset unit is PIXDEG");
    	} else {
    		cpl_msg_error(cpl_func, "User-defined offset unit or offset list is not correct.");
    	}




    	char* new_name = NULL;
    	cpl_imagelist* data = NULL;
    	cpl_imagelist* errs = NULL;
    	cpl_imagelist* qual = NULL;

    	cpl_propertylist* herrs = NULL;
    	cpl_propertylist* hqual = NULL;
    	for(int i = 0; i < ncubes; i++) {
    		cube_frm = cpl_frameset_get_position(cube_set, i);
    		fname = cpl_frame_get_filename(cube_frm);
    		hdata = cpl_propertylist_load(fname, 1);

    		if(offcode == 0) {          // PIXEL
    			crpix1 = cpl_propertylist_get_double(hdata, "CRPIX1");
    			crpix2 = cpl_propertylist_get_double(hdata, "CRPIX2");
    			cpl_propertylist_set_double(hdata,"CRPIX1", crpix1 + offsetx[i]);
    			cpl_propertylist_set_double(hdata,"CRPIX2", crpix2 + offsety[i]);
    		} else if (offcode == 1) {  // DEGREE
    			crval1 = cpl_propertylist_get_double(hdata, "CRVAL1");
    			crval2 = cpl_propertylist_get_double(hdata, "CRVAL2");
    			cpl_propertylist_set_double(hdata,"CRVAL1", crval1 + offsetx[i]);
    			cpl_propertylist_set_double(hdata,"CRVAL2", crval2 + offsety[i]);

    		} else if (offcode == 2) {  // ARCSEC
    			crval1 = cpl_propertylist_get_double(hdata, "CRVAL1");
    			crval2 = cpl_propertylist_get_double(hdata, "CRVAL2");
    			cpl_propertylist_set_double(hdata,"CRVAL1", crval1 + offsetx[i] * scalex);
    			cpl_propertylist_set_double(hdata,"CRVAL2", crval2 + offsety[i] * scaley);

    		} else if (offcode == 3) { //real coordinates x, y , alpha, delta

    			cpl_propertylist_set_double(hdata,"CRPIX1", crpix1_usr[i]);
    			cpl_propertylist_set_double(hdata,"CRPIX2", crpix2_usr[i]);

    			cpl_propertylist_set_double(hdata,"CRVAL1", crval1_usr[i]);
    			cpl_propertylist_set_double(hdata,"CRVAL2", crval2_usr[i]);

    		}

    		plist = cpl_propertylist_load(fname, 0);
    		herrs = cpl_propertylist_load(fname, 2);
    		hqual = cpl_propertylist_load(fname, 3);
    		data = cpl_imagelist_load(fname, CPL_TYPE_DOUBLE, 1);
    		errs = cpl_imagelist_load(fname, CPL_TYPE_DOUBLE, 2);
    		qual = cpl_imagelist_load(fname, CPL_TYPE_INT, 3);
    		/* scale cubes data and error by exposure time to exptime of 1st input frame */
    		cpl_msg_info(cpl_func,"correct[%d] by=%g",i,pfactor[i]);
    		cpl_imagelist_multiply_scalar(data,pfactor[i]);
    		cpl_imagelist_multiply_scalar(errs,sqrt(pfactor[i]));
    		new_name = cpl_sprintf("cube_%2.2d.fits",i);
    		cpl_frame_set_filename(cube_frm, new_name);
    		cpl_propertylist_save(plist, new_name, CPL_IO_CREATE);
    		cpl_imagelist_save(data, new_name, CPL_TYPE_DOUBLE, hdata, CPL_IO_EXTEND);
    		cpl_imagelist_save(errs, new_name, CPL_TYPE_DOUBLE, herrs, CPL_IO_EXTEND);
    		cpl_imagelist_save(qual, new_name, CPL_TYPE_INT, hqual, CPL_IO_EXTEND);
    		cpl_propertylist_delete(plist);
    		cpl_propertylist_delete(hdata);
    		cpl_propertylist_delete(herrs);
    		cpl_propertylist_delete(hqual);
    		cpl_imagelist_delete(data);
    		cpl_imagelist_delete(errs);
    		cpl_imagelist_delete(qual);

    		cpl_free(new_name);

    	}


    	if(eris_ifu_combine_pbp(cube_set, parlist,
    	                            ERIS_IFU_COMBINE_DOCATG, fname_spec_cube,
    	                            NULL, NULL, offunit,
    	                            RECIPE_NAME, RECIPE_NAME) != CPL_ERROR_NONE){
    	        cpl_msg_error(__func__, "Cannot resampling the cubes for combination.");
    	                            }

    } else {


    	char* new_name = NULL;
    	cpl_imagelist* data = NULL;
    	cpl_imagelist* errs = NULL;
    	cpl_imagelist* qual = NULL;

    	cpl_propertylist* herrs = NULL;
    	cpl_propertylist* hqual = NULL;
    	/* scaling by exptime */
    	for(int i = 0; i < ncubes; i++) {
    		cube_frm = cpl_frameset_get_position(cube_set, i);
    		fname = cpl_frame_get_filename(cube_frm);
    		hdata = cpl_propertylist_load(fname, 1);



    		plist = cpl_propertylist_load(fname, 0);
    		herrs = cpl_propertylist_load(fname, 2);
    		hqual = cpl_propertylist_load(fname, 3);
    		data = cpl_imagelist_load(fname, CPL_TYPE_DOUBLE, 1);
    		errs = cpl_imagelist_load(fname, CPL_TYPE_DOUBLE, 2);
    		qual = cpl_imagelist_load(fname, CPL_TYPE_INT, 3);
    		/* scale cubes data and error by exposure time to exptime of 1st input frame */
    		cpl_msg_info(cpl_func,"correct[%d] by=%g",i,pfactor[i]);
    		cpl_imagelist_multiply_scalar(data,pfactor[i]);
    		cpl_imagelist_multiply_scalar(errs,sqrt(pfactor[i]));
    		new_name = cpl_sprintf("cube_%2.2d.fits",i);
    		cpl_frame_set_filename(cube_frm, new_name);
    		cpl_propertylist_save(plist, new_name, CPL_IO_CREATE);
    		cpl_imagelist_save(data, new_name, CPL_TYPE_DOUBLE, hdata, CPL_IO_EXTEND);
    		cpl_imagelist_save(errs, new_name, CPL_TYPE_DOUBLE, herrs, CPL_IO_EXTEND);
    		cpl_imagelist_save(qual, new_name, CPL_TYPE_INT, hqual, CPL_IO_EXTEND);
    		cpl_propertylist_delete(plist);
    		cpl_propertylist_delete(hdata);
    		cpl_propertylist_delete(herrs);
    		cpl_propertylist_delete(hqual);
    		cpl_imagelist_delete(data);
    		cpl_imagelist_delete(errs);
    		cpl_imagelist_delete(qual);

    		cpl_free(new_name);

    	}


    	if(eris_ifu_combine_pbp(cube_set, parlist,
        	                            ERIS_IFU_COMBINE_DOCATG, fname_spec_cube,
        	                            NULL, NULL, offunit,
        	                            RECIPE_NAME, RECIPE_NAME) != CPL_ERROR_NONE){
        	        cpl_msg_error(__func__, "Cannot resampling the cubes for combination.");
        	                            }

    }
    cpl_vector_delete(factor_vec);

    if (offsetx != NULL)
    	cpl_free(offsetx);
    if (offsety != NULL)
    	cpl_free(offsety);

    if (crpix1_usr != NULL)
    	cpl_free(crpix1_usr);
    if (crpix2_usr != NULL)
    	cpl_free(crpix2_usr);

    if (crval1_usr != NULL)
    	cpl_free(crval1_usr);
    if (crval2_usr != NULL)
    	cpl_free(crval2_usr);


    /* add possibility to extract and flux calibrate combined data cube */
    cubeType obj_type = OBJECT_CUBE;
    cubeType resampled_obj_type = OBJECT_CUBE_COADD;
    const char *proCatg;
    cpl_frame* frame = NULL;
    if( NULL != (frame = cpl_frameset_find(frameset, ERIS_IFU_PRO_JITTER_RESPONSE ))) {
    	cpl_frameset_insert(cube_set, cpl_frame_duplicate(frame));
    	//cpl_frameset_erase(frameset, frame);

    }


    if( NULL != (frame = cpl_frameset_find(frameset, ERIS_IFU_CALIB_EXTCOEFF_TABLE ))) {
    	cpl_frameset_insert(cube_set, cpl_frame_duplicate(frame));
    	//cpl_frameset_erase(frameset, frame);

    }

    if (cpl_frameset_count_tags(cube_set, ERIS_IFU_PRO_JITTER_DAR_CUBE)
    		> 0) {
    	obj_type = DAR_CUBE;
    	resampled_obj_type = DAR_CUBE_COADD;
    	proCatg = ERIS_IFU_PRO_JITTER_DAR_CUBE;
    } else if (cpl_frameset_count_tags(cube_set, ERIS_IFU_PRO_JITTER_TWK_CUBE)
    		> 0 ) {
    	obj_type = TWEAKED_CUBE;
    	resampled_obj_type = TWEAKED_CUBE_COADD;
    	proCatg = ERIS_IFU_PRO_JITTER_TWK_CUBE;
    } else {
    	obj_type = OBJECT_CUBE;
    	resampled_obj_type = OBJECT_CUBE_COADD;
    	proCatg = ERIS_IFU_PRO_JITTER_OBJ_CUBE;
    }

    /* check if Pupil Tracking */
    char *combDoCatg =  NULL;
    char   *filenameSpec = NULL;
    eris_ifu_jitter_get_procatg_and_filename(obj_type, &combDoCatg, &filenameSpec);

    eris_ifu_free_string(&combDoCatg);
    eris_ifu_free_string(&filenameSpec);
    char* param_name = cpl_sprintf("%s.flux-calibrate", CONTEXT);
    cpl_boolean flux_calibrate = cpl_parameter_get_bool(
    		cpl_parameterlist_find_const(parlist, param_name));
    cpl_free(param_name);

    param_name = cpl_sprintf("%s.extract-source", CONTEXT);
    cpl_boolean extract_source = cpl_parameter_get_bool(
    		cpl_parameterlist_find_const(parlist, param_name));
    cpl_free(param_name);


	/* extract spectra and for STD star computes response and efficiency */
    if (extract_source && eris_can_extract(cube_set)) {
    	cpl_msg_info(cpl_func,"extracting");
    	eris_ifu_jitter_extract(cube_set, parlist, resampled_obj_type,
    			proCatg, stdParams, RECIPE_NAME, CONTEXT);
    }

    if(flux_calibrate && eris_can_flux_calibrate(cube_set)) {

    	if(extract_source) {
    		cpl_msg_info(cpl_func,"Flux calibrate extracted spectrum");
    		eris_flux_calibrate_spectra(RECIPE_NAME, RECIPE_NAME, parlist,
    				cube_set, cube_set);

    	}
    	char* cube_pro_catg = NULL;
    	if(obj_type == DAR_CUBE) {
    		cube_pro_catg = cpl_sprintf("%s",ERIS_IFU_PRO_JITTER_OBJ_DAR_CUBE_COADD);
    	} else if(obj_type == TWEAKED_CUBE) {
    		cube_pro_catg = cpl_sprintf("%s",ERIS_IFU_PRO_JITTER_TWK_CUBE_COADD);
    	} else {
    		cube_pro_catg = cpl_sprintf("%s",ERIS_IFU_PRO_JITTER_OBJ_CUBE_COADD);
    	}

    	cpl_msg_info(cpl_func,"proCatg: %s",cube_pro_catg);

    	cpl_msg_info(cpl_func,"Flux calibrate combined data cube");
    	if(extract_source) {
    		/*TODO: force to have extracted spectrum as wmin/max is taken from it
    		 * we should update this to use the wmin/wmax of the output cube instead
    		 */
    	eris_flux_calibrate_cube2(cube_pro_catg, RECIPE_NAME, RECIPE_NAME, parlist,
    			/*frameset, */cube_set);
    	}
    	cpl_free(cube_pro_catg);

    }
    cpl_frameset_erase(cube_set,ERIS_IFU_COMBINE_DOCATG);
    cpl_frameset_join(frameset,cube_set);

    char* cmd = cpl_sprintf("rm cube*.fits");
    int status = system(cmd);
    cpl_free(cmd);
    /* end addition */
    cpl_frameset_delete(cube_set);
    //Clean up
    //eris_ifu_jitter_free_sofStruct(&sof);
    //eris_ifu_free_std_param(&stdParams);
    return (int) cpl_error_get_code();
}
/**@}*/
