/* $Id$
 *
 * This file is part of the ERIS Pipeline
 * Copyright (C) 2017 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/*
 * $Author$
 * $Date$
 * $Revision$
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                Includes
 -----------------------------------------------------------------------------*/

#include <libgen.h>
#include <math.h>
#include <string.h>

#include "casu_utils.h"
#include "casu_mods.h"
#include "casu_stats.h"

#include "eris_utils.h"
#include "eris_nix_utils.h"
#include "eris_pfits.h"
#include "eris_dfs.h"
#include "eris_nix_dfs.h"
#include "eris_utils.h"
#include "eris_nix_lss_utils.h"
#include <hdrl.h>

#include <cpl.h>

/*-----------------------------------------------------------------------------
                            Static variables
 -----------------------------------------------------------------------------*/

static const char eris_nix_lss_straighten_description[] =
"This recipe uses a CPL warping routine to correct the wavelength and\n" 
"slit offset coordinates of the input ERIS/NIX long-slit spectra. The\n"
"aim for the output spectra is to have slit offset along the horizontal\n"
"axis and dispersion along the vertical.\n"
"\n"
"Areas of the detector not illuminated by the slit have their data and\n"
"confidence values set to 0.\n"
"\n"
"Input files:\n"
"\n"
"  DO CATG                         Explanation             Req.  #Frames\n"
"  -------                         -----------             ---   -------\n"
"  "ERIS_NIX_SKYSUB_OBJECT_LSS_JITTER_PRO_CATG
                          "        sky-subtracted frames    Y      1-n\n"
"                                  to be straightened.\n"
"             or\n"
"  "ERIS_NIX_SKYSUB_STD_LSS_JITTER_PRO_CATG
                       "           sky-subtracted           Y      1-n\n"
"                                  standard frames\n"
"                                  to be stacked.\n"
"  "ERIS_NIX_MASTER_STARTRACE_PRO_CATG
                  "                file containing the      Y      1\n"
"                                  warp polynomials\n"
"                                  for correcting the\n"
"                                  LSS spectrum.\n"
"\n"
"Output files:\n"
"\n"
"  DO CATG                         Explanation \n"
"  -------                         ----------- \n"
"  "ERIS_NIX_CORRECTED_OBJECT_LSS_JITTER_PRO_CATG
                             "     the straightened 2d-spectrum of the\n"
"                                  target.\n"
"             or\n"
"  "ERIS_NIX_CORRECTED_STD_LSS_JITTER_PRO_CATG
                          "        the straightened 2d-spectrum of the\n"
"                                  standard.\n"
"\n"
"  The output image will be in a FITS file named \n"
"  'corrected.<first input filename>', with extensions:\n"
"  - DATA, with the corrected data.\n"
"  - ERR, with the corrected error plane.\n"
"  - DQ, with the corrected image quality\n"
"  - CONFIDENCE, with the corrected confidence plane.\n"
"  - BKG_DATA, with corrected background spectrum.\n"
"  - BKG_ERR, with the corrected background error plane.\n"
"  - BKG_CONF, with the corrected background confidence.\n"
"\n"
"  Notes on the method.\n"
"    It is assumed that the jitter pattern moves the object up and\n"
"    down the slit-line. The reference position and the offset of\n"
"    each jitter from it are calculated as follows:\n"
"       1/ The reference is jitter 0. Often this lies near the \n"
"          middle of the pattern and the object is manually \n"
"          positioned near the middle of the slit.\n"
"       2/ Derive the slit-line PA (0 = N, increasing E) from the\n"
"          extrema of the jitter centres.\n"
"       3/ Report how far each jitter pointing lies from the\n"
"          slit-line.\n"
"       4/ Estimate the offset of each jitter relative to the\n"
"          reference jitter along the slit-line.\n"
"\n"
"    Correct the spectra to:\n"
"       Calibrate the wavelengths and straighten the spectrum by\n"
"         removing slit/dispersion curvature from the image.\n"
"       Shift spectra so that an object appears in the same column\n"
"         as it would in the reference jitter.\n"
"\n"
"    This is done for each jitter by:\n"
"       1/ Estimating the shift required to place the object in\n"
"          the same column as in the reference jitter.\n"
"          a/ Take a copy of the spectrum.\n"
"          b/ Apply the warp polynomials to straighten the copy.\n"
"          c/ Collapse the copy along the dispersion axis.\n"
"          d/ Find the position of the star.\n"
"          e/ Calculate the offset required to move the star to\n"
"             its position in the reference jitter.\n"
"       2/ Modify the warp polynomials to apply the required shift\n"
"          with the straightening and wavelength calibration.\n" 
"       3/ Apply the shifted warp polynomials to:\n"
"          ..the data plane\n"
"          ..the error plane\n"
"          ..the background spectrum\n"
"          ..the confidence array\n"
"\n"
"    Set the confidence to 0 outside the illuminated range\n"
"    of the shifted spectrum - to reduce edge effects when\n"
"    stacking them later.\n"
"\n"
"    Save the straightened spectra. For each spectrum:\n"
"       1/ Add FITS keywords to describe the WCS of the 2d\n"
"          spectrum.\n"
"       2/ Save the spectrum to FITS\n"
"\n";

 static cpl_polynomial * eris_nix_lss_shift_polynomial(
                        const cpl_polynomial * polynomial,
                        const double xshift);


#define RECIPE_NAME "eris.eris_nix_lss_straighten"

/*-----------------------------------------------------------------------------
                            Private function prototypes
 -----------------------------------------------------------------------------*/

cpl_recipe_define(eris_nix_lss_straighten, ERIS_BINARY_VERSION,
                  "John Lightfoot",
                  PACKAGE_BUGREPORT, "2017",
                  "Straighten and calibrate "
                  ERIS_NIX_SKYSUB_OBJECT_LSS_JITTER_PRO_CATG
                  " frames",
                  eris_nix_lss_straighten_description);

/*-----------------------------------------------------------------------------
                                Function code
 -----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Setup the recipe options    
  @param    self  the non-NULL parameterlist to fill
  @return   CPL_ERROR_NONE iff everything is ok

  Defining the command-line/configuration parameters for the recipe.
 */
/*----------------------------------------------------------------------------*/

static cpl_error_code eris_nix_lss_straighten_fill_parameterlist(
  cpl_parameterlist * self) {

    if (cpl_error_get_code() != CPL_ERROR_NONE) return cpl_error_get_code();

    cpl_parameter     * p = NULL;

    p = cpl_parameter_new_value(RECIPE_NAME".debug_data",
                                CPL_TYPE_BOOL,
                                "write debugging data",
                                RECIPE_NAME, CPL_FALSE);
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "debug_data");
    cpl_parameterlist_append(self, p);

    return 0;
}


/*----------------------------------------------------------------------------*/
/**
  @brief              Shift the polynomial x axis
  @param  polynomial  the polynomial to shift
  @param  xshift      the x shift to implement
  @return             the vshifted polynomial

  Shift the x-axis of the warping polynomial. If the polynomial shift is:

      poly(x,y)[perfect] ->  x[actual]

  we want:

      newpoly(x-offset, y) -> x
 */
/*----------------------------------------------------------------------------*/

cpl_polynomial * eris_nix_lss_shift_polynomial(
                                       const cpl_polynomial * polynomial,
                                       const double xshift) {

    if (cpl_error_get_code() != CPL_ERROR_NONE) return NULL;
    cpl_ensure(polynomial, CPL_ERROR_NULL_INPUT, NULL);

    cpl_polynomial * result = cpl_polynomial_duplicate(polynomial);

    cpl_size pows00[2] = {0, 0};
    double coeff00 = cpl_polynomial_get_coeff(result, pows00);
    cpl_size pows10[2] = {1, 0};
    double coeff10 = cpl_polynomial_get_coeff(result, pows10);
    cpl_size pows20[2] = {2, 0};
    double coeff20 = cpl_polynomial_get_coeff(result, pows20);

    cpl_size pows01[2] = {0, 1};
    double coeff01 = cpl_polynomial_get_coeff(result, pows01);
    cpl_size pows11[2] = {1, 1};
    double coeff11 = cpl_polynomial_get_coeff(result, pows11);
    cpl_size pows21[2] = {2, 1};
    double coeff21 = cpl_polynomial_get_coeff(result, pows21);

    cpl_size pows02[2] = {0, 2};
    double coeff02 = cpl_polynomial_get_coeff(result, pows02);
    cpl_size pows12[2] = {1, 2};
    double coeff12 = cpl_polynomial_get_coeff(result, pows12);
    cpl_size pows22[2] = {2, 2};
    double coeff22 = cpl_polynomial_get_coeff(result, pows22);

    double shift_coeff00 = coeff00 - coeff10 * xshift + coeff20 * pow(xshift,2);
    cpl_polynomial_set_coeff(result, pows00, shift_coeff00);
    double shift_coeff10 = coeff10 - coeff20 * 2.0 * xshift;
    cpl_polynomial_set_coeff(result, pows10, shift_coeff10);
    /* shift_coeff20 = coeff20 so no change needed */

    double shift_coeff01 = coeff01 - coeff11 * xshift + coeff21 * pow(xshift,2);
    cpl_polynomial_set_coeff(result, pows01, shift_coeff01);
    double shift_coeff11 = coeff11 - coeff21 * 2.0 * xshift;
    cpl_polynomial_set_coeff(result, pows11, shift_coeff11);
    /* shift_coeff21 = coeff21 so no change needed */

    double shift_coeff02 = coeff02 - coeff12 * xshift + coeff22 * pow(xshift,2);
    cpl_polynomial_set_coeff(result, pows02, shift_coeff02);
    double shift_coeff12 = coeff12 - coeff22 * 2.0 * xshift;
    cpl_polynomial_set_coeff(result, pows12, shift_coeff12);
    /* shift_coeff22 = coeff22 so no change needed */

    //cpl_polynomial_dump(result, stdout);

    /* tidy up on error */

    if (cpl_error_get_code() != CPL_ERROR_NONE) {
        cpl_polynomial_delete(result);
        result = NULL;
    }
    return result;
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Interpret the command line options and execute the data processing
  @param    frameset   the frames list
  @param    parlist    the parameters list
  @return   0 if everything is ok
 */
/*----------------------------------------------------------------------------*/

static int eris_nix_lss_straighten(cpl_frameset * frameset,
                                   const cpl_parameterlist * parlist) {

    cpl_vector            * crval1_vector = NULL;
    cpl_vector            * crval2_vector = NULL;
    located_imagelist     * jitters = NULL; 
    cpl_table             * jitter_table = NULL;
    located_imagelist     * object_jitters = NULL; 
    const char            * out_catg = NULL;
    const cpl_parameter   * p = NULL;
    cpl_vector            * profile = NULL;
    cpl_vector            * slit_offset = NULL;
    mef_extension_list    * startrace_mefs = NULL;
    cpl_propertylist      * startrace_plist = NULL;
    cpl_polynomial        * startrace_poly_i = NULL;
    cpl_polynomial        * startrace_poly_j = NULL;
    located_imagelist     * std_jitters = NULL; 
    cpl_frameset          * used = NULL;

    enu_check_error_code("%s():%d: An error is already set: %s",
                         cpl_func, __LINE__, cpl_error_get_where());

    /* Check input parameters */

    cpl_ensure_code(frameset, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(parlist, CPL_ERROR_NULL_INPUT);

    /* Set the msg verbosity level from environment variable CPL_MSG_LEVEL */

    cpl_msg_set_level_from_env();
    cpl_msg_severity severity = cpl_msg_get_level();
    cpl_msg_info(cpl_func, "level %d", (int) severity);

    /* check for invalid input */
    if(eris_files_dont_exist(frameset) != CPL_ERROR_NONE) {
    	return CPL_ERROR_BAD_FILE_FORMAT;
    }

    /* Retrieve input parameters */

    p = cpl_parameterlist_find_const(parlist, RECIPE_NAME".debug_data");
    int debug_data = cpl_parameter_get_bool(p);
    enu_check_error_code("Could not retrieve input parameters");

    /* Identify the RAW and CALIB frames in the input frameset */

    eris_nix_dfs_set_groups(frameset);
    enu_check_error_code("Could not identify RAW and CALIB frames");    

    /* read in the sky-subtracted data */

    used = cpl_frameset_new();
    object_jitters = enu_limlist_load_from_frameset(frameset,
                     ERIS_NIX_SKYSUB_OBJECT_LSS_JITTER_PRO_CATG, used);
    std_jitters = enu_limlist_load_from_frameset(frameset,
                  ERIS_NIX_SKYSUB_STD_LSS_JITTER_PRO_CATG, used);
    enu_check_error_code("Error loading frameset");

    if (object_jitters->size > 0) {
        enu_check(std_jitters->size == 0, CPL_ERROR_ILLEGAL_INPUT,
                  "SoF contains both object and std data");
        jitters = object_jitters;
        object_jitters = NULL;
        out_catg = ERIS_NIX_CORRECTED_OBJECT_LSS_JITTER_PRO_CATG;
        cpl_msg_info(cpl_func, "Read in %d "
                     ERIS_NIX_SKYSUB_OBJECT_LSS_JITTER_PRO_CATG "frames",
                     (int)(jitters->size));
    } else if (std_jitters->size > 0) {
        jitters = std_jitters;
        std_jitters = NULL;
        out_catg = ERIS_NIX_CORRECTED_STD_LSS_JITTER_PRO_CATG;
        cpl_msg_info(cpl_func, "Read in %d "
                     ERIS_NIX_SKYSUB_STD_LSS_JITTER_PRO_CATG "frames",
                     (int)(jitters->size));
    }

    /* read in the MASTER_STARTRACE file from the sof */

    cpl_frameset_iterator * frameset_iter = cpl_frameset_iterator_new(frameset);
    for (cpl_frame * frame = NULL;
         (frame = cpl_frameset_iterator_get(frameset_iter)) &&
         (cpl_error_get_code() == CPL_ERROR_NONE);
         cpl_frameset_iterator_advance(frameset_iter, 1)) {

        const char * tag = cpl_frame_get_tag(frame);

        if (!strcmp(tag, ERIS_NIX_MASTER_STARTRACE_PRO_CATG)) {
            const char * filename = cpl_frame_get_filename(frame);
            startrace_mefs = enu_load_mef_components(filename,
                                                     &startrace_plist);
            cpl_msg_info(cpl_func, "Read MASTER_STARTRACE %s", filename);
            break;
        }
    }
    enu_check(startrace_mefs != NULL, CPL_ERROR_DATA_NOT_FOUND,
              "SoF contains no "ERIS_NIX_MASTER_STARTRACE_PRO_CATG
              " data");

    /* estimate the centre of the jitter pattern
       .. The jitter centres should form a line on the sky
       .. as the object is moved up and down the slit.
       .. Find the ends of this line by looking at the extreme points. */

    cpl_msg_info(cpl_func, "Finding centre of jitter pattern");
    crval1_vector = cpl_vector_new(jitters->size);
    crval2_vector = cpl_vector_new(jitters->size);
    for (cpl_size i = 0; i < jitters->size; i++) {
        double crval1 = cpl_propertylist_get_double(
                                       jitters->limages[i]->plist,
                                       "CRVAL1");
        cpl_vector_set(crval1_vector, i, crval1);
        double crval2 = cpl_propertylist_get_double(
                                       jitters->limages[i]->plist,
                                       "CRVAL2");
        cpl_vector_set(crval2_vector, i, crval2);
        cpl_msg_info(cpl_func, "..jitter %d at RA=%10.7f Dec=%10.7f",
                     (int)i, crval1, crval2);
    }

    cpl_size crval1_maxpos = cpl_vector_get_maxpos(crval1_vector);
    double crval1_max = cpl_vector_get(crval1_vector, crval1_maxpos);
    cpl_size crval1_minpos = cpl_vector_get_minpos(crval1_vector);
    double crval1_min = cpl_vector_get(crval1_vector, crval1_minpos);

    cpl_size crval2_maxpos = cpl_vector_get_maxpos(crval2_vector);
    double crval2_max = cpl_vector_get(crval2_vector, crval2_maxpos);
    cpl_size crval2_minpos = cpl_vector_get_minpos(crval2_vector);
    double crval2_min = cpl_vector_get(crval2_vector, crval2_minpos);

    /* for the coord that changes most, assume the line ends are at
       the extreme values */
    double enda[2] = {0.0, 0.0};
    double endb[2] = {0.0, 0.0};
    if (fabs(crval1_max - crval1_min) > fabs(crval2_max - crval2_min)) {
        enda[0] = crval1_min;
        enda[1] = cpl_vector_get(crval2_vector, crval1_minpos);
        endb[0] = crval1_max;
        endb[1] = cpl_vector_get(crval2_vector, crval1_maxpos);
    } else {
        enda[0] = cpl_vector_get(crval1_vector, crval2_minpos);
        enda[1] = crval2_min;
        endb[0] = cpl_vector_get(crval1_vector, crval2_maxpos);
        endb[1] = crval2_max;
    }
    cpl_msg_info(cpl_func, "Jitter line end points (RA, DEC): "
                 "(%10.7f, %10.7f) and (%10.7f, %10.7f)",
                 enda[0], enda[1], endb[0], endb[1]);

    double centre[2] = {0.0, 0.0};

    /* define reference position as that of first jitter. 
       In the test data it seems that the object was manually 
       centred in the slit (i.e. near column 1024) for jitter 0 */

    centre[0] = cpl_vector_get(crval1_vector, 0);
    centre[1] = cpl_vector_get(crval2_vector, 0);

//    alternatively...
//    /* define the reference position as a point halfway along the
//       slit-line jitter pattern */

//    centre[0] = (enda[0] + endb[0]) / 2.0;
//    centre[1] = (enda[1] + endb[1]) / 2.0;

    cpl_msg_info(cpl_func, "Jitter centre %f %f", centre[0], centre[1]);
    double coslat = cos(centre[1] * CPL_MATH_PI / 180.0);

    /* direction of slit line, assume axis increases to left */

    double slit_line[2] = {(endb[0]-enda[0]) * coslat, endb[1]-enda[1]};
    double slit_norm = sqrt(pow(slit_line[0],2) + pow(slit_line[1],2));
    slit_line[0] /= slit_norm;
    slit_line[1] /= slit_norm;
    double slit_pa = atan2(slit_line[0], slit_line[1]); 
    cpl_msg_info(cpl_func, "..slit direction vector (long, lat) = (%f, %f)",
                 slit_line[0], slit_line[1]);
    cpl_msg_info(cpl_func, "..slit PA = %f deg", slit_pa * 180.0 / CPL_MATH_PI);
    
    double cd1_1 = cpl_propertylist_get_double(jitters->limages[0]->
                                               plist, "CD1_1");
    double cd2_1 = cpl_propertylist_get_double(jitters->limages[0]->
                                               plist, "CD2_1");
    /* pixsize in deg */
    double pixsize = sqrt(cd1_1 * cd1_1 + cd2_1 * cd2_1);
    cpl_msg_info(cpl_func, "pixel size %f deg", pixsize);

    /* how do the jitter centres line up */

    slit_offset = cpl_vector_new(jitters->size);
    for (cpl_size i = 0; i < jitters->size; i++) {
        double offset = -1000.0;
        /* be wary of 0 components in slit_line direction vector */
        if (fabs(slit_line[0]) > slit_line[1]) {
            offset = (cpl_vector_get(crval1_vector, i) - centre[0]) * 
                     coslat / slit_line[0];
        } else {
            offset = (cpl_vector_get(crval2_vector, i) - centre[1]) /
                     slit_line[1];
        }

        /* how far away is the point from the line between endpoints? */

        double distance[2] = {0.0,0.0};
        distance[0] = ((cpl_vector_get(crval1_vector, i) - centre[0]) * 
                      coslat - 
                      (offset * slit_line[0])) / pixsize;
        distance[1] = ((cpl_vector_get(crval2_vector, i) - centre[1]) - 
                      (offset * slit_line[1])) / pixsize;
        //cpl_msg_info(cpl_func, "%f %f %f %f",
        //             cpl_vector_get(crval2_vector, i),
        //             centre[1], offset, slit_line[1]);
        cpl_msg_warning(cpl_func,
                        "...distance from jitter line (%f, %f) pixels",
                        distance[0], distance[1]);

        cpl_vector_set(slit_offset, i, offset);
        cpl_msg_info(cpl_func, "..slit offset jitter %d %f deg",
                     (int)i, offset);
    }

    /* calculate nominal offset from reference position for each */
    /* jitter */

    cpl_vector * nominal_offset = cpl_vector_new(jitters->size);
    for (cpl_size i = 0; i < jitters->size; i++) {
        cpl_vector_set(nominal_offset, i,
                       cpl_vector_get(slit_offset, i) / pixsize);
    }
    cpl_msg_info(cpl_func, "Nominal offset of each jitter from "
                 "reference (pixels):");
    cpl_vector_dump(nominal_offset, NULL);
    enu_check_error_code("Error calculating nominal slit offsets");

    /* read the wavelength cal and dispersion direction polynomials */

    cpl_table * poly_table = cpl_table_duplicate(
                             (cpl_table *) startrace_mefs->mef[0]->data);
    startrace_poly_i = enlu_warp_poly_load_from_table(poly_table);
    cpl_table_delete(poly_table);

    poly_table = cpl_table_duplicate(
                 (cpl_table *) startrace_mefs->mef[1]->data);
    startrace_poly_j = enlu_warp_poly_load_from_table(poly_table);
    cpl_table_delete(poly_table);
    enu_check_error_code("Error reading warp polynomials");

    /* correct the frames for spectral and startrace distortion */

    /* profile is Lanczos2, which works well for images, not sure about
       spectral direction. ISAAC uses default profile, tanh.
 
       Also can use different profiles for spatial and spectral, should
       examine this - TBD */

    profile = cpl_vector_new(CPL_KERNEL_DEF_SAMPLES);
    cpl_vector_fill_kernel_profile(profile, CPL_KERNEL_LANCZOS,
                                   CPL_KERNEL_DEF_WIDTH);

    if (debug_data) {
  
        /* use the sky background spectrum to check the wavelength 
           calibration */

        /* ..warp the background of the first jitter, this will remove any
             curvature in the wavelength scale */

        cpl_image * bkg_copy = cpl_image_duplicate(hdrl_image_get_image_const(
                                                   jitters->limages[0]->bkg));
        cpl_image_warp_polynomial(bkg_copy,
                                  hdrl_image_get_image(
                                  jitters->limages[0]->bkg),
                                  startrace_poly_i, startrace_poly_j, 
                                  profile, CPL_KERNEL_DEF_WIDTH,
                                  profile, CPL_KERNEL_DEF_WIDTH);
        cpl_image_reject_value(bkg_copy, CPL_VALUE_NOTFINITE);

        cpl_image_save(bkg_copy, "bkg_warped.fits", CPL_TYPE_UNSPECIFIED,
                       NULL, CPL_IO_CREATE);

        /* ..collapse the 2d-spectrum along the slit direction */

        cpl_image * bkg_collapse = cpl_image_collapse_median_create(
                                       bkg_copy, 1, 600, 550);
        cpl_vector * bkg_vector = cpl_vector_new_from_image_column(
                                       bkg_collapse, 1);
        cpl_vector_save(bkg_vector,
                        "bkg_vector.fits",
                        CPL_TYPE_DOUBLE,
                        NULL,
                        CPL_IO_CREATE);

        cpl_vector_delete(bkg_vector);
        cpl_image_delete(bkg_collapse);
        cpl_image_delete(bkg_copy);
    }

    /* correct the frames */

    double actual_offset_0 = 0.0;

    for (cpl_size j = 0; j < jitters->size; j++) {

        /* first, find the object in the slit - if there is one */

        /* ..make a copy of the raw data */

        cpl_image * copy_image = cpl_image_duplicate(hdrl_image_get_image_const(
                                               jitters->limages[j]->himage));

        /* ..straighten the raw data copy back into the jitter himage */

        cpl_image_warp_polynomial(hdrl_image_get_image(
                                  jitters->limages[j]->himage), copy_image,
                                  startrace_poly_i, startrace_poly_j, 
                                  profile, CPL_KERNEL_DEF_WIDTH,
                                  profile, CPL_KERNEL_DEF_WIDTH);

        if (debug_data) {
            char* filename = cpl_sprintf("warp_%d.fits", (int)j);
            cpl_image_save(hdrl_image_get_image(jitters->limages[j]->himage),
                           filename, CPL_TYPE_UNSPECIFIED, NULL,
                           CPL_IO_CREATE);
            cpl_free(filename);
        }

        /* ..find the position of the object in the straightened result
           ....1. collapse along dispersion */

        cpl_image * collapse_image = cpl_image_collapse_create(
                                       hdrl_image_get_image(jitters->
                                       limages[j]->himage), 0);
        cpl_vector * collapse = cpl_vector_new_from_image_row(
                                       collapse_image, 1);

        /* ....2. find position of object in slit */

        /* ....3. could use nominal offset as a first guess if the 
                  reference jitter had the object in column 0 but it won't
                  - instead use a robust maximum estimate as the first
                    guess. Nominal and actual offsets should move in a
                    similar pattern if all is working as expected */

        /* double guess = cpl_vector_get(nominal_offset, j); */

        double guess = (double) cpl_vector_get_maxpos(collapse);
        cpl_msg_debug(cpl_func, "..maxpos %d", (int)guess);
        double actual_offset = enlu_linepos_1d(collapse, guess, 6);
        double nominal = cpl_vector_get(nominal_offset, j);
        if (j == 0) {
            actual_offset_0 = actual_offset;
            cpl_msg_info(cpl_func, "..object in jitter 0 is at %f pixels",
                         actual_offset_0);
        }
        cpl_msg_info(cpl_func, "multiplying nominal offset by -1 as "
                     "increasing RA decreases detector column");
        cpl_msg_info(cpl_func, "..object slit offsets (nominal, "
                     "actual_j) %f %f",
                     -1.0 * nominal, actual_offset);

        /* ..calculate the offset shift required to place the object
             at the column it is seen in jitter 0. Add the offset shift
             to the warping description and do the straighten again.
             This should remove the warp _and_ register the jitters with 
             respect to jitter 0. The aim is to avoid more resampling
             steps than are necessary as each one probably loses 
             information */

        double xshift = (actual_offset_0 - actual_offset);
        cpl_msg_info(cpl_func, "..shift to place object at jitter 0 "
                     "position is %5.2f pixels",
                     xshift);
        
        cpl_polynomial * shifted_poly_i = eris_nix_lss_shift_polynomial(
                                       startrace_poly_i, xshift);
        cpl_polynomial * shifted_poly_j = eris_nix_lss_shift_polynomial(
                                       startrace_poly_j, xshift);

        /* ..warp original data */

        cpl_image_warp_polynomial(hdrl_image_get_image(
                                  jitters->limages[j]->himage), copy_image,
                                  shifted_poly_i, shifted_poly_j, 
                                  profile, CPL_KERNEL_DEF_WIDTH,
                                  profile, CPL_KERNEL_DEF_WIDTH);
        cpl_image_delete(copy_image);

        if (debug_data) {
            char* filename = cpl_sprintf("warp2_%d.fits", (int)j);
            cpl_image_save(hdrl_image_get_image(jitters->limages[j]->himage),
                           filename, CPL_TYPE_UNSPECIFIED, NULL,
                           CPL_IO_CREATE);
            cpl_free(filename);
        }

        /* ..warp the error plane  - the resampling weights will
             not be applied correctly */

        cpl_image* copy_error = cpl_image_duplicate(hdrl_image_get_error_const(
                                   jitters->limages[j]->himage));
        cpl_image_warp_polynomial(hdrl_image_get_error(
                                  jitters->limages[j]->himage), copy_error,
                                  shifted_poly_i, shifted_poly_j, 
                                  profile, CPL_KERNEL_DEF_WIDTH,
                                  profile, CPL_KERNEL_DEF_WIDTH);
        cpl_image_delete(copy_error);

        /* ..warp background */

        cpl_image* copy_bkg = cpl_image_duplicate(hdrl_image_get_image_const(
                                   jitters->limages[j]->bkg));
        cpl_image_warp_polynomial(hdrl_image_get_image(
                                  jitters->limages[j]->bkg), copy_bkg,
                                  shifted_poly_i, shifted_poly_j, 
                                  profile, CPL_KERNEL_DEF_WIDTH,
                                  profile, CPL_KERNEL_DEF_WIDTH);
        cpl_image_delete(copy_bkg);

        /* ..warp background error - wrong weights again */

        cpl_image* copy_bkg_err = cpl_image_duplicate(hdrl_image_get_error_const(
                                   jitters->limages[j]->bkg));
        cpl_image_warp_polynomial(hdrl_image_get_error(
                                  jitters->limages[j]->bkg), copy_bkg_err,
                                  shifted_poly_i, shifted_poly_j, 
                                  profile, CPL_KERNEL_DEF_WIDTH,
                                  profile, CPL_KERNEL_DEF_WIDTH);
        cpl_image_delete(copy_bkg_err);

        /* ..warp confidence */

        cpl_image* copy_conf = cpl_image_duplicate(jitters->limages[j]->confidence);
        cpl_image_warp_polynomial(jitters->limages[j]->confidence, copy_conf,
                                  shifted_poly_i, shifted_poly_j, 
                                  profile, CPL_KERNEL_DEF_WIDTH,
                                  profile, CPL_KERNEL_DEF_WIDTH);
        cpl_image_delete(copy_conf);

        /* tidy up */

        cpl_polynomial_delete(shifted_poly_i);
        cpl_polynomial_delete(shifted_poly_j);
        enu_check_error_code("Error resampling data");

        /* last, set confidence to zero outside the illuminated x-range
           of the detector (593, 1507 are hardwired for now) */

        cpl_size left_edge = 593 + (cpl_size)xshift; 
        cpl_size right_edge = 1507 + (cpl_size)xshift; 
        for (cpl_size ii=0;
            ii < hdrl_image_get_size_x(jitters->limages[j]->himage); ii++) {

            if (ii > left_edge && ii < right_edge) continue;
            for (cpl_size jj=0;
                jj < hdrl_image_get_size_y(jitters->limages[j]->himage);
                jj++) {
                cpl_image_set(jitters->limages[j]->confidence, ii+1, jj+1,
                              0.0);
                hdrl_image_set_pixel(jitters->limages[j]->himage, ii+1, jj+1,
                                     (hdrl_value){0.0, 0.0});
                hdrl_image_reject(jitters->limages[j]->himage, ii+1, jj+1);
            }
        }
        enu_normalise_confidence(jitters->limages[j]->confidence);
        enu_check_error_code("Error zeroing confidence outside " 
                             "illuminated area");
    }

    /* save the corrected jitter spectra to FITS files */

    for (cpl_size i = 0; i < jitters->size; i++) {

        /* Set up the wcs to reflect the corrected spectrum.
           Borrows from long-slit example in 
           Calabretta and Greisen 2002 A&A 395, 1077 (example on p.
           1115-16).

           In our case the pixel coords are p1,p2 = slit-offset,wavelength
           NB this is different to the example in the paper.

           The transformation is to RA, Dec, lambda. Using a CD matrix:

           RA        CRVAL1     CD1_1  CD1_2  CD1_3     p1 - CRPIX1
           lambda  = CRVAL2  +  CD2_1  CD2_2  CD2_3  x  p2 - CRPIX2
           Dec       CRVAL3  +  CD3_1  CD3_2  CD3_3     p3 - CRPIX3

           The wavelength transformation is 3.045um at 0 to 4.107um at 2047
           so:

              CRPIX2 = 0.0
              CRVAL2 = 3.045
              CD2_1 = 0
              CD2_2 = 0
              cd2_3 = (4.107 - 3.045) / 2047

           The slit offset to RA, Dec transformation combines the RA,Dec of the
           slit centre with the P.A. of the slit rotation (assumed 0 at N, 
           increasing towrds increasing RA), so:

              CRPIX1 = 1024
              CRVAL1 = RA centre
              CD1_1 = sin(PA) * pixscale
              RA = CRVAL1 + CD1_1 * (p1 - CRPIX1)

              and

              CRVAL3 = Dec centre
              CD3_1 = cos(PA) * pixscale
              Dec = CRVAL3 + CD3_1 * (p1 - CRPIX3)

           The p3 axis is degenerate so CD entries for it can have
           arbitrary values.

              CD1_3 = CD2_3 = CD3_3 = 1
 */

        cpl_propertylist_update_int(jitters->limages[i]->plist, "NAXIS", 3);
        cpl_propertylist_update_string(jitters->limages[i]->plist, "CTYPE1",
                                       "RA---TAN");
        cpl_propertylist_update_string(jitters->limages[i]->plist, "CTYPE2",
                                       "WAVE");
        cpl_propertylist_update_string(jitters->limages[i]->plist, "CTYPE3",
                                       "DEC--TAN");
        cpl_propertylist_update_double(jitters->limages[i]->plist, "CRPIX1",
                                       1024.0);
        cpl_propertylist_update_double(jitters->limages[i]->plist, "CRPIX2",
                                       0.0);
        cpl_propertylist_update_double(jitters->limages[i]->plist, "CRPIX3",
                                       0.0);
        cpl_propertylist_update_double(jitters->limages[i]->plist, "CRVAL1",
                                       centre[0]);
        cpl_propertylist_update_double(jitters->limages[i]->plist, "CRVAL2",
                                       3.045);
        cpl_propertylist_update_double(jitters->limages[i]->plist, "CRVAL3",
                                       centre[1]);

        cpl_propertylist_update_double(jitters->limages[i]->plist, "CD1_1", 
                                       sin(slit_pa) * pixsize);
        cpl_propertylist_update_double(jitters->limages[i]->plist, "CD1_2",
                                       0.0);
        cpl_propertylist_update_double(jitters->limages[i]->plist, "CD1_3",
                                       1.0);

        cpl_propertylist_update_double(jitters->limages[i]->plist, "CD2_1",
                                       0.0);
        cpl_propertylist_update_double(jitters->limages[i]->plist, "CD2_2",
                                       (4.107 - 3.045) / 2047);
        cpl_propertylist_update_double(jitters->limages[i]->plist, "CD2_3",
                                       0.0);

        cpl_propertylist_update_double(jitters->limages[i]->plist, "CD3_1",
                                       cos(slit_pa) * pixsize);
        cpl_propertylist_update_double(jitters->limages[i]->plist, "CD3_2",
                                       0.0);
        cpl_propertylist_update_double(jitters->limages[i]->plist, "CD3_3",
                                       1.0);

        cpl_propertylist_update_string(jitters->limages[i]->plist, "CUNIT1",
                                       "DEG");
        cpl_propertylist_update_string(jitters->limages[i]->plist, "CUNIT2",
                                       "um");
        cpl_propertylist_update_string(jitters->limages[i]->plist, "CUNIT3",
                                       "DEG");

        cpl_propertylist * applist = cpl_propertylist_new();
        cpl_propertylist_update_string(applist, CPL_DFS_PRO_CATG, out_catg);
        cpl_propertylist_update_string(applist, "PRODCATG",
                                       "ANCILLARY.2DSPECTRUM");

        /* data needs to be converted from dims [nx,ny] to [nx,ny,1] to 
           save in FITS format - FITS requires the addition of a degenerate 
           axis */

        if (debug_data) {
            char* filename = cpl_sprintf("warp3_%d.fits", (int)i);
            cpl_image_save(hdrl_image_get_image(jitters->limages[i]->himage),
                           filename, CPL_TYPE_UNSPECIFIED, NULL,
                           CPL_IO_CREATE);
            cpl_free(filename);
        }

        hdrl_imagelist * himlist1 = hdrl_imagelist_new();
        hdrl_imagelist_set(himlist1, jitters->limages[i]->himage, 0);
        // FIX error expention (PIPE-12651)
        //jitters->limages[i]->himage = NULL;
        //jitters->limages[i]->himagelist = himlist1;
        //jitters->limages[i]->himage = NULL;
        jitters->limages[i]->himagelist = NULL;

        /* generate name of output file, provenance */

        char * out_fname = enu_repreface(cpl_frame_get_filename(
                                         jitters->limages[i]->frame),
                                         "straightened");

        cpl_frameset * provenance = cpl_frameset_new();
        cpl_frameset_insert(provenance, cpl_frame_duplicate(
                            jitters->limages[i]->frame));

        cpl_msg_info(cpl_func, "..writing %s", out_fname);
        enu_dfs_save_limage(frameset,
                            parlist,
                            provenance,
                            CPL_TRUE,
                            jitters->limages[i],
                            RECIPE_NAME,
                            jitters->limages[i]->frame,
                            applist,
                            PACKAGE "/" PACKAGE_VERSION,
                            out_fname);

        cpl_free(out_fname);
        cpl_frameset_delete(provenance);
        cpl_propertylist_delete(applist);
    }

cleanup:
    cpl_vector_delete(crval1_vector);
    cpl_vector_delete(crval2_vector);
    enu_located_imagelist_delete(jitters);
    cpl_table_delete(jitter_table);
    enu_located_imagelist_delete(object_jitters);
    cpl_vector_delete(profile);
    cpl_vector_delete(slit_offset);
    enu_mef_extension_list_delete(startrace_mefs);
    cpl_propertylist_delete(startrace_plist);
    cpl_polynomial_delete(startrace_poly_i);
    cpl_polynomial_delete(startrace_poly_j);
    enu_located_imagelist_delete(std_jitters);
    cpl_frameset_delete(used);

    return (int) cpl_error_get_code();
}
