/* $Id$
 *
 * This file is part of the ERIS Pipeline
 * Copyright (C) 2002,2003 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdbool.h>
#include <string.h>
#include <math.h>

#include "eris_ifu_error.h"
#include "eris_ifu_utils.h"
#include "eris_ifu_dfs.h"
#include "eris_ifu_debug.h"
#include "eris_ifu_flat_static.h"
#include "eris_ifu_dark_static.h"
#include "eris_ifu_functions.h"
#include "eris_utils.h"

/**
 * @defgroup eris_ifu_flat_static IFU Flat Field Calibration
 *
 * This module provides comprehensive flat field calibration functions for ERIS IFU data.
 * It supports multiple flat field processing modes (FAST, HDRL, SEGMENT) and handles
 * lamp-on/off frame subtraction, bad pixel mask creation, quality control parameter
 * calculation, and various image processing operations for flat field normalization.
 *
 * Key features:
 * - Multiple flat field modes: FAST (quick processing), HDRL (HDRL library-based),
 *   SEGMENT (segmented processing for specific detector regions)
 * - Bad pixel mask generation using 2D and 3D detection methods
 * - Column tilt removal for detector systematic effects
 * - Median filtering and noise reduction
 * - Quality control (QC) parameter calculation
 * - Support for both FLAT_LAMP and FLAT_NS (nodding sky) observations
 */

/*----------------------------------------------------------------------------*/
/**
  @brief    Performs the whole flat field calibration workflow
  @param    fs                   Input frameset containing raw flat frames and calibration files
  @param    parlist              Recipe parameter list with processing configuration
  @param    mode                 Flat field processing mode (FLAT_MODE_FAST, FLAT_MODE_HDRL, FLAT_MODE_SEGMENT)
  @param    productDepth         Product depth flags controlling debug output (bitwise OR of flags)
  @param    procatg              Product category string for DFS compliance
  @param    recipe_name          Name of the calling recipe (e.g., "eris_ifu_flat")
  @param    procatg_prefix       Prefix for product category strings
  @param    instrument           Instrument identifier (INSTRUMENT_ERIS or INSTRUMENT_SINFONI)
  @param    qc_list              (Output) Property list containing quality control parameters
  @param    masterFlatHdrlImg_lo (Output) Low-frequency master flat field HDRL image
  @param    qualityImage         (Output) Quality/bad pixel map image with DQI flags
  @return   CPL_ERROR_NONE on success, otherwise an error code
  @note     All output pointers must be freed by the caller after use
  @note     The function loads flat frames, applies bad pixel masks, computes the master flat,
            optionally creates additional bad pixel masks (except in SEGMENT mode), and calculates
            quality control parameters
  @note     For ERIS J-band or SINFONI data in distortion recipe mode, uses SINFONI-style
            bad pixel detection; otherwise uses HDRL-based methods
  @see      eris_ifu_flat_load_frames, eris_ifu_calc_flat, eris_ifu_calc_bpm
 */
/*----------------------------------------------------------------------------*/
cpl_error_code eris_ifu_flat_static(cpl_frameset    *fs,
                            const cpl_parameterlist *parlist,
                            const flatMode          mode,
                            int                     productDepth,
                            const char              *procatg,
                            const char              *recipe_name,
                            const char              *procatg_prefix,
                            const char              *instrument,
                            cpl_propertylist        **qc_list,
                            hdrl_image              **masterFlatHdrlImg_lo,
                            cpl_image               **qualityImage)
{
	cpl_error_code      ret                     = CPL_ERROR_NONE;
	cpl_image           *contrib_map            = NULL;
	cpl_mask            *bpm2dMask              = NULL,
                        *bpm3dMask              = NULL;
	hdrl_image          *masterFlatHdrlImg_hi   = NULL;
	hdrl_imagelist      *hdrl_imglist_on        = NULL,
                        *hdrl_imglist_off       = NULL;
    const char          *fn                     = NULL;
    char                *pc                     = NULL,
                        *tmp_str                = NULL;
	double              cleanmean               = 0.,
                        cleanstdev              = 0.;

    cpl_ensure_code(fs,                     CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(parlist,                CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(procatg,                CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(recipe_name,            CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(qc_list,                CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(masterFlatHdrlImg_lo,   CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(qualityImage,           CPL_ERROR_NULL_INPUT);

	TRY
	{
		/* Load flat frames  and mask bad pixels */
		double gain = 0., exptime = 0.;
		eris_ifu_flat_load_frames(fs,
				&gain, &exptime,
				&hdrl_imglist_on,
				&hdrl_imglist_off);

		*qc_list = cpl_propertylist_new();

		cpl_propertylist_append_string(*qc_list, CPL_DFS_PRO_CATG,
				procatg);

		/* calc QC before flat calculation*/
		eris_ifu_flat_calc_qc_pre(parlist,
				recipe_name,
				hdrl_imglist_on,
				hdrl_imglist_off,
				*qc_list);

		/* Performing data reduction */
		/* we need the band ID to be able to set proper value of threshold to
		 * detect IFU slices
		 */
		cpl_frame* raw_flat_lamp = NULL;

		if (NULL != cpl_frameset_find(fs, ERIS_IFU_RAW_FLAT_LAMP)){
			raw_flat_lamp = cpl_frameset_find(fs, ERIS_IFU_RAW_FLAT_LAMP);

		} else if (cpl_frameset_count_tags(fs, ERIS_IFU_RAW_FLAT_LAMP_ON) > 0) {

			raw_flat_lamp = cpl_frameset_find(fs, ERIS_IFU_RAW_FLAT_LAMP_ON);

		} else if (NULL != cpl_frameset_find(fs, ERIS_IFU_RAW_FLAT_LAMP_ON)){
			raw_flat_lamp = cpl_frameset_find(fs, ERIS_IFU_RAW_FLAT_LAMP_ON);

		} else if (cpl_frameset_count_tags(fs, ERIS_IFU_RAW_FLAT_NS_ON) > 0) {

			raw_flat_lamp = cpl_frameset_find(fs, ERIS_IFU_RAW_FLAT_NS_ON);
		} else {
			raw_flat_lamp = cpl_frameset_find(fs, ERIS_IFU_RAW_FLAT_NS);
		}
		const char* fname = cpl_frame_get_filename(raw_flat_lamp);
		cpl_propertylist* header = cpl_propertylist_load(fname, 0);
		ifsBand band = eris_ifu_get_band(header);
		cpl_propertylist_delete(header);
		eris_ifu_calc_flat(parlist,
				recipe_name,
				mode,
				hdrl_imglist_on,
				hdrl_imglist_off,
				productDepth,
				band,
				masterFlatHdrlImg_lo,
				&masterFlatHdrlImg_hi,
				&contrib_map);

		if (productDepth  & 4) {
			BRK_IF_NULL(
					tmp_str = cpl_sprintf("%s_dbg_flat_masterflat.fits", recipe_name));
			eris_ifu_save_hdrl_image_dbg(*masterFlatHdrlImg_lo,
					"eris_ifu_distortion_dbg_flat_02_masterflat",
					TRUE, NULL);
			eris_ifu_free_string(&tmp_str);
		}

		// do this crappy bpm-calculation only if segemtn-mode hasn't been chosen
		if (mode != FLAT_MODE_SEGMENT) {

			//        // check only for distortoin-recipe: if is (H or K) and 25mas
			//        fr = cpl_frameset_find_const(fs, ERIS_IFU_RAW_FLAT_NS);
			//        if (fr != NULL) {
			//            BRK_IF_NULL(
			//                header = cpl_propertylist_load(cpl_frame_get_filename(fr), 0));
			//            BRK_IF_NULL(
			//                tmp_band = cpl_propertylist_get_string(header, FHDR_E_GRATING_ID));
			//            BRK_IF_NULL(
			//                band = cpl_sprintf("%s", tmp_band));
			//            scaleId = eris_ifu_get_preopticsScale(header);
			//            CHECK_ERROR_STATE();
			//            eris_ifu_free_propertylist(&header);
			//        }
			//
			//        if ((scaleId == S25MAS) && (strcmp(band, "K_low") == 0)) {
			// //erw            is_K_low_and_25mas = CPL_TRUE;
			//        }
			//        eris_ifu_free_string(&band);


			/* Create & apply bad-pixel-mask */
			//erw       if ((strcmp(recipe_name, REC_NAME_DISTORTION) == 0) &&
			//            !is_K_low_and_25mas)
			if (strcmp(recipe_name, REC_NAME_DISTORTION) == 0)
			{
				// only for distortion recipe (except K_low_25mas)
				double              maxcut  = 50000.,
						mincut  = 0.1;
                char                *band_name  = NULL;
                cpl_frame           *fr         = NULL;
//                const char          *fn         = NULL;
                cpl_propertylist    *plist      = NULL;

				fr = cpl_frameset_find(fs, ERIS_IFU_RAW_FLAT_NS);
				fn = cpl_frame_get_filename(fr);
				plist = cpl_propertylist_load(fn, 0);

				if (strcmp(instrument, INSTRUMENT_ERIS) == 0) {
					// ERIS
					const char *tmp_band = NULL;
					tmp_band = cpl_propertylist_get_string(plist,
							FHDR_E_FILTER_ID);
                    band_name = cpl_sprintf("%c", tmp_band[0]);
				} else {
					// SINFONI
                    band_name = cpl_sprintf("%s", cpl_propertylist_get_string(plist,
                                        FHDR_S_FILTER_NAME));
				}

				// HDRL flatfielding can't handle slitlets with low illumination...
				if (true/*(strcmp(instrument, INSTRUMENT_SINFONI) == 0) ||
                    ((strcmp(instrument, INSTRUMENT_ERIS) == 0) &&
                                                (strcmp(band, "J") == 0))*/)
				{
					// distortion-recipe +
					//     either SINFONI-instrument
					//     or  ERIS-Instrument and J-band

					/* SINFONI-way to compute the BP-map*/
					hdrl_image          **med       = NULL,
							*original   = NULL;
					double              sigma       = 2.,
							loReject    = 0.1,
							hiReject    = 0.1;
					int                 llx         = 1350,
							lly         = 1000,
							urx         = 1390,
							ury         = 1200,
							meanfactor  = 999,
							n_iter      = 8;
					double              mjd_obs = 0.;

					mjd_obs = cpl_propertylist_get_double(plist, FHDR_MJD_OBS);
					CHECK_ERROR_STATE();

					/* special check in case of J band distortion data
					 * after 31/01/2018 */
                    if ((strcmp(band_name, "J") == 0) && (mjd_obs > 58150.)) {
						mincut = 0.05;
					}

					original = hdrl_image_duplicate(*masterFlatHdrlImg_lo);

					if (productDepth  & 4) {
						eris_ifu_save_hdrl_image_dbg(*masterFlatHdrlImg_lo,
								"eris_ifu_distortion_dbg_flat_03_bad_medIm",
								TRUE, NULL);
					}
					/* remove the intensity tilt from every column and compute the
					 * standard deviation on a rectangular zone */

					/* update BP-mask and expand it by thresholding */
					eris_ifu_flat_thresh_mask(*masterFlatHdrlImg_lo,
							mincut, maxcut);

					if (productDepth  & 4) {
						eris_ifu_save_hdrl_image_dbg(*masterFlatHdrlImg_lo,
								"eris_ifu_distortion_dbg_flat_04_bad_medIm_cut",
								TRUE, NULL);
					}

					eris_ifu_column_tilt(*masterFlatHdrlImg_lo, sigma);

					if (productDepth  & 4) {
						eris_ifu_save_hdrl_image_dbg(*masterFlatHdrlImg_lo,
								"eris_ifu_distortion_dbg_flat_05_bad_colImage",
								TRUE, NULL);
					}
					eris_ifu_flat_stats_rectangle(*masterFlatHdrlImg_lo,
							loReject, hiReject,
							llx, lly, urx, ury,
							&cleanmean,
							&cleanstdev);

					/* indicate pixels with great deviations from the clean mean as bad */
					eris_ifu_flat_thresh_mask(*masterFlatHdrlImg_lo,
							cleanmean - meanfactor*cleanstdev,
							cleanmean + meanfactor*cleanstdev);
					if (productDepth  & 4) {
						eris_ifu_save_hdrl_image_dbg(*masterFlatHdrlImg_lo,
								"eris_ifu_distortion_dbg_flat_06_bad_threshIm",
								TRUE, NULL);
					}

					med = (hdrl_image**)cpl_calloc(n_iter, sizeof(hdrl_image*));

					med[0] = eris_ifu_flat_median_image(*masterFlatHdrlImg_lo,
							-meanfactor*cleanstdev);

					for (int i = 0; i< n_iter-1; i++) {
						med[i+1] = eris_ifu_flat_median_image(med[i],
								-meanfactor*cleanstdev);
					}

					/* compare the filtered image with the input image */
					eris_ifu_sinfo_compare_images(*masterFlatHdrlImg_lo,
							med[n_iter-1],
							original);
					if (productDepth  & 4) {
						eris_ifu_save_hdrl_image_dbg(original,
								"eris_ifu_distortion_dbg_flat_07_bad_compImage",
								TRUE, NULL);
					}

					eris_ifu_free_hdrl_image(masterFlatHdrlImg_lo);
					*masterFlatHdrlImg_lo = original;

					for (int i = 0; i< n_iter; i++) {
						eris_ifu_free_hdrl_image(&med[i]);
					}
					cpl_free(med); med = NULL;
				} else {
					// distortion-recipe + ERIS-instrument

					/* HDRL-way to compute the BP-map*/
					eris_ifu_calc_bpm(parlist,
							recipe_name,
							*masterFlatHdrlImg_lo,    // needed for 2dbpm
							hdrl_imglist_on,          // needed for 3dbpm
							&bpm2dMask, &bpm3dMask); // return values

					/* update BP-mask and expand it by thresholding */
					eris_ifu_flat_thresh_mask(*masterFlatHdrlImg_lo,
							mincut, maxcut);
				}
                eris_ifu_free_string(&band_name);
				eris_ifu_free_propertylist(&plist);
			} else {
				// not distortion-recipe + any instrument

				/* HDRL-way to compute the BP-map*/
				eris_ifu_calc_bpm(parlist,
						recipe_name,
						*masterFlatHdrlImg_lo,    // needed for 2dbpm
						hdrl_imglist_on,          // needed for 3dbpm
						&bpm2dMask, &bpm3dMask); // return values
			}
		} // if (mode != FLAT_MODE_SEGMENT)

		/* calc QC after flat calculation*/
		eris_ifu_flat_calc_qc_post(parlist,
                                    recipe_name,
                                    productDepth,
                                    procatg_prefix,
                                    masterFlatHdrlImg_hi,
                                    *masterFlatHdrlImg_lo,
                                    hdrl_imglist_on,
                                    hdrl_imglist_off,
                                    contrib_map,
                                    // bpm2dImg,
                                    // bpm3dImg,
                                    bpm2dMask,
                                    bpm3dMask,
                                    fs,
                                    *qc_list,
                                    qualityImage);
    } CATCH {
		CATCH_MSGS();

		ret = cpl_error_get_code();

		eris_ifu_free_propertylist(qc_list);
		eris_ifu_free_hdrl_image(masterFlatHdrlImg_lo);
		eris_ifu_free_image(qualityImage);
	}

	eris_ifu_free_hdrl_image(&masterFlatHdrlImg_hi);
	eris_ifu_free_hdrl_imagelist(&hdrl_imglist_on);
	eris_ifu_free_hdrl_imagelist(&hdrl_imglist_off);
	eris_ifu_free_image(&contrib_map);
	//    eris_ifu_free_image(&bpm2dImg);
	//    eris_ifu_free_image(&bpm3dImg);
	eris_ifu_free_mask(&bpm2dMask);
	eris_ifu_free_mask(&bpm3dMask);
//	eris_ifu_free_string(&fn);
	eris_ifu_free_string(&pc);
	eris_check_error_code("eris_ifu_flat_static");

	return ret;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Calculate the threshold value for flat field processing
  @param    img     Input HDRL image
  @return   Threshold value (0.3 times the mean of the image), or NAN on error
  @note     Used to determine cold/hot pixel thresholds in segment mode
 */
/*----------------------------------------------------------------------------*/
double eris_ifu_getThreshold(hdrl_image *img) {
	cpl_ensure(img != NULL, CPL_ERROR_ILLEGAL_INPUT, NAN);
	double val = 0.;
	TRY {
		val = .3 * hdrl_image_get_mean(img).data;
	} CATCH {
		CATCH_MSGS();
		val = NAN;
	}
	eris_check_error_code("eris_ifu_getThreshold");
	return val;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Match a pixel row position to a block index
  @param    blocksize    Size of processing blocks in pixels
  @param    rx           X-coordinate position (pixel, 0-based)
  @return   Block index corresponding to the position, or -1 on error
  @note     Used in segment mode to map pixel positions to processing blocks
  @note     Formula: (rx-4)/blocksize + 0.5, rounded to integer
 */
/*----------------------------------------------------------------------------*/
int eris_ifu_matchRow(int blocksize, int rx) {
	cpl_ensure(blocksize > 0, CPL_ERROR_ILLEGAL_INPUT, -1);
	cpl_ensure(rx >= 0, CPL_ERROR_ILLEGAL_INPUT, -1);
	cpl_ensure(rx < ERIS_IFU_DETECTOR_SIZE_X, CPL_ERROR_ILLEGAL_INPUT, -1);

	int val = 0;
	TRY {
		val = (int)((rx-4)/blocksize + 0.5);
	} CATCH {
		CATCH_MSGS();
		val = -1;
	}
	eris_check_error_code("eris_ifu_matchRow");
	return val;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Free memory allocated for a structFlatData structure
  @param    sfd    Pointer to structFlatData structure to free
  @note     Frees all internal vectors (ry, borderAleft/right, borderBleft/right,
            borderCleft/right, borderDleft/right) but not the structure itself
  @note     Safe to call with NULL pointer or already-freed vectors
  @see      eris_ifu_flat_data_init, eris_ifu_flat_data_setBorders
 */
/*----------------------------------------------------------------------------*/
void eris_ifu_free_structFlatData(struct structFlatData* sfd) {

	if(sfd != NULL) {
		if(sfd->ry != NULL) cpl_vector_delete(sfd->ry);
		if(sfd->borderAleft != NULL) cpl_vector_delete(sfd->borderAleft);
		if(sfd->borderAright != NULL) cpl_vector_delete(sfd->borderAright);
		if(sfd->borderBleft != NULL) cpl_vector_delete(sfd->borderBleft);
		if(sfd->borderBright != NULL) cpl_vector_delete(sfd->borderBright);
		if(sfd->borderCleft != NULL) cpl_vector_delete(sfd->borderCleft);
		if(sfd->borderCright != NULL) cpl_vector_delete(sfd->borderCright);
		if(sfd->borderDleft != NULL) cpl_vector_delete(sfd->borderDleft);
		if(sfd->borderDright != NULL) cpl_vector_delete(sfd->borderDright);
	}
	eris_check_error_code("eris_ifu_free_structFlatData");
	return;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Compute the master flat field image
  @param    parlist                 Recipe parameter list containing processing configuration
  @param    recipe_name             Name of the calling recipe
  @param    mode                    Flat field mode (FLAT_MODE_FAST, FLAT_MODE_HDRL, or FLAT_MODE_SEGMENT)
  @param    hdrl_imglist_on         List of lamp-on/object flat frames
  @param    hdrl_imglist_off        List of lamp-off/sky flat frames
  @param    productDepth            Product depth flags for debug output
  @param    band                    Spectral band identifier (used in SEGMENT mode for threshold adjustment)
  @param    masterFlatHdrlImg_lo    (Output) Low-frequency master flat field image (normalized)
  @param    masterFlatHdrlImg_hi    (Output) High-frequency master flat (only if productDepth & 1 in HDRL mode)
  @param    contrib_map             (Output) Contribution map showing number of frames contributing to each pixel
  @return   CPL_ERROR_NONE on success, otherwise an error code
  @note     All output pointers must be freed by the caller
  @note     FAST mode: Subtracts off from on frames, collapses, and applies 5x3 smoothing filter
  @note     HDRL mode: Uses HDRL flat field computation with optional high-frequency output
  @note     SEGMENT mode: Processes detector in segments, detects slitlet borders, masks inter-slitlet regions
  @see      eris_ifu_flat_static, hdrl_flat_compute
 */
/*----------------------------------------------------------------------------*/
cpl_error_code eris_ifu_calc_flat(const cpl_parameterlist *parlist,
		const char              *recipe_name,
		const flatMode          mode,
		const hdrl_imagelist    *hdrl_imglist_on,
		const hdrl_imagelist    *hdrl_imglist_off,
		int                     productDepth,
		ifsBand                 band,
		hdrl_image              **masterFlatHdrlImg_lo,
		hdrl_image              **masterFlatHdrlImg_hi,
		cpl_image               **contrib_map)
{
	cpl_error_code  ret                     = CPL_ERROR_NONE;
	cpl_image       *noiseImage             = NULL;
	hdrl_image      *fastFlatImg            = NULL;
	hdrl_parameter  *pHdrlCollapse          = NULL,
			*pHdrlFlat_lo           = NULL,
			*pHdrlFlat_hi           = NULL;
	hdrl_imagelist  *tmp_hdrl_imglist       = NULL,
			*tmp_hdrl_imglist2      = NULL;
	char            *tmp_str                = NULL;

	cpl_ensure_code(hdrl_imglist_on,        CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(recipe_name,            CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(hdrl_imglist_off,       CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(masterFlatHdrlImg_lo,   CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(contrib_map,            CPL_ERROR_NULL_INPUT);

	TRY
	{
		cpl_msg_info(cpl_func, "Generating master flat image...");

		/* load collapse parameters */
		tmp_str = cpl_sprintf("eris.%s.collapse", recipe_name);
		pHdrlCollapse = hdrl_collapse_parameter_parse_parlist(parlist,
				tmp_str);
		eris_ifu_free_string(&tmp_str);

		if (mode == FLAT_MODE_FAST) {
			/* just subtract on- and off-frames and collapse */
			cpl_msg_info(cpl_func, "   Fast-mode");

			/*
			 * subtract and collapse flat images
			 */
			tmp_hdrl_imglist = hdrl_imagelist_duplicate(hdrl_imglist_on);

			hdrl_imagelist_sub_imagelist(tmp_hdrl_imglist, hdrl_imglist_off);

			hdrl_imagelist_collapse(tmp_hdrl_imglist,
					pHdrlCollapse,
					masterFlatHdrlImg_lo,
					contrib_map);

			/*
			 * smoothen masterflat & restore bpm
			 */
			cpl_image *tmpFilteredImg = cpl_image_new(
					hdrl_image_get_size_x(*masterFlatHdrlImg_lo),
					hdrl_image_get_size_y(*masterFlatHdrlImg_lo),
					cpl_image_get_type(hdrl_image_get_image(*masterFlatHdrlImg_lo)));

			cpl_matrix *filterKernel = cpl_matrix_new(5,3);
			cpl_matrix_set(filterKernel, 0, 1 ,1.0);
			cpl_matrix_set(filterKernel, 1, 1 ,2.0);
			cpl_matrix_set(filterKernel, 2, 1 ,3.0);
			cpl_matrix_set(filterKernel, 3, 1 ,2.0);
			cpl_matrix_set(filterKernel, 4, 1 ,1.0);
			cpl_matrix_set(filterKernel, 2, 0 ,1.5);
			cpl_matrix_set(filterKernel, 2, 2 ,1.5);
			cpl_image_filter(tmpFilteredImg,
					hdrl_image_get_image(*masterFlatHdrlImg_lo),
					filterKernel, CPL_FILTER_LINEAR, CPL_BORDER_FILTER);
			cpl_image_reject_from_mask(tmpFilteredImg,
					cpl_image_get_bpm(hdrl_image_get_error(*masterFlatHdrlImg_lo)));

			fastFlatImg = hdrl_image_create(tmpFilteredImg,
					hdrl_image_get_error(*masterFlatHdrlImg_lo));

			hdrl_image_delete(*masterFlatHdrlImg_lo);
			*masterFlatHdrlImg_lo = fastFlatImg;

			if (productDepth  & 4) {
				tmp_str = cpl_sprintf("%s_dbg_flat_fast.fits", recipe_name);
				eris_ifu_save_image_dbg(tmpFilteredImg, tmp_str, CPL_IO_CREATE, NULL);
				eris_ifu_free_string(&tmp_str);
			}

			eris_ifu_free_image(&tmpFilteredImg);
			eris_ifu_free_matrix(&filterKernel);

			hdrl_image_div_scalar(*masterFlatHdrlImg_lo,
					hdrl_image_get_median(*masterFlatHdrlImg_lo));
		} else if (mode == FLAT_MODE_HDRL) {
			/* the HDRL-way to create a masterflat (not well suited to
               spectroscopic raw data) */
			cpl_msg_info(cpl_func, "   HDRL-mode");

			/* load flat parameters */
			tmp_str = cpl_sprintf("eris.%s.flat_lo", recipe_name);

			pHdrlFlat_lo = hdrl_flat_parameter_parse_parlist(parlist, tmp_str);
			eris_ifu_free_string(&tmp_str);

			/* Do the actual flatfield computation, bp_mask has already been
               applied when loading images */
			tmp_hdrl_imglist = hdrl_imagelist_duplicate(hdrl_imglist_on);
			if (hdrl_imagelist_get_size(tmp_hdrl_imglist) ==
					hdrl_imagelist_get_size(hdrl_imglist_off))
			{
				hdrl_imagelist_sub_imagelist(tmp_hdrl_imglist,
						hdrl_imglist_off);
			} else {
				cpl_msg_warning(cpl_func, "#off and #on frames isn't the same! "
						"So no off-frames are subtracted at all.");
			}

			if (productDepth & 1) {
				/* calculate as well hi-freq flatfield */
				tmp_str = cpl_sprintf("eris.%s.flat_hi", recipe_name);
				pHdrlFlat_hi = hdrl_flat_parameter_parse_parlist(parlist,
						tmp_str);
				eris_ifu_free_string(&tmp_str);

				tmp_hdrl_imglist2 = hdrl_imagelist_duplicate(tmp_hdrl_imglist);
				hdrl_flat_compute(tmp_hdrl_imglist2, NULL,
						pHdrlCollapse, pHdrlFlat_hi,
						masterFlatHdrlImg_hi, contrib_map);
				eris_ifu_free_hdrl_imagelist(&tmp_hdrl_imglist2);
				eris_ifu_free_image(contrib_map);
			}

			if (productDepth  & 4) {
				tmp_str = cpl_sprintf("eris_ifu_%s_dbg_flat_01_sub", recipe_name);
				eris_ifu_save_hdrl_imagelist_dbg(hdrl_imglist_on,
						tmp_str,
						TRUE);
				eris_ifu_free_string(&tmp_str);
			}

			/* hdrl_flat_compute() gives unnecessary warning with pHdrlFlat_lo */
			cpl_msg_severity level = cpl_msg_get_level();
			cpl_msg_set_level(CPL_MSG_ERROR);

			/* calculate lo-freq flatfield */
			hdrl_flat_compute(tmp_hdrl_imglist, NULL,
					pHdrlCollapse, pHdrlFlat_lo,
					masterFlatHdrlImg_lo, contrib_map);
			cpl_msg_set_level(level);
			eris_ifu_free_hdrl_imagelist(&tmp_hdrl_imglist);

			eris_ifu_add_badpix_border(
					hdrl_image_get_image(*masterFlatHdrlImg_lo),
					false, NULL);
		} else if  (mode == FLAT_MODE_SEGMENT) {
			/* do a segmented flat-calculation*/
			cpl_msg_info(cpl_func, "   Segment-mode");

			hdrl_image          *masterFlat         = NULL,
					*subImg             = NULL,
					*subImgA            = NULL,
					*subImgD            = NULL,
					*tmpImg             = NULL;
			int                 sx                  = 0,
					sy                  = 0,
					rej                 = 0,
					blockSize           = 40,
					nLower              = 0,
					offset              = 40,
					leftEdgeA           = 0,
					leftEdgeB           = 0,
					leftEdgeC           = 0,
					leftEdgeD           = 0,
					rightEdgeA          = 0,
					rightEdgeB          = 0,
					rightEdgeC          = 0,
					rightEdgeD          = 0;
			double              columnThreshold     = .70,  //70% of mean (65% for K band)
					thresholdA          = 0.,
					thresholdD          = 0.,
					coldThreshold       = 0.,
					*pdnewcleanRowBC     = NULL,
					*pborders           = NULL,
					*psubImg            = NULL;
			const double        *pcleanRow          = NULL;
			cpl_vector          *cleanRow           = NULL,
					*dnewcleanRowBC     = NULL,
					*sample             = NULL,
					*sample2            = NULL,
					*borders            = NULL;
			hdrl_value          medianA,
			medianD;
			struct structFlatData Data;
			if(band == K_LOW ||
					band == K_SHORT ||
					band == K_MIDDLE ||
					band == K_LONG){
				columnThreshold     = .65;
			}

			/*
			 * just collapse flat-on images (don't even subtract flat_off-images)
			 */
			hdrl_imagelist_collapse(hdrl_imglist_on,
					pHdrlCollapse,
					&masterFlat,
					contrib_map);

			eris_ifu_flat_data_init(&Data, blockSize);

			// loop horizontal segments of height=blockSize (e.g. 40 rows)
			// c-notation for ry (from 4 to 2040)
			//
			// get borders of blocks A, B, C, D
			for(int ry = ERIS_IFU_DETECTOR_BP_BORDER;
					ry < ERIS_IFU_DETECTOR_SIZE_Y-ERIS_IFU_DETECTOR_BP_BORDER;
					ry += blockSize)
			{
				subImg = hdrl_image_extract(masterFlat,
						1, ry+1,
						ERIS_IFU_DETECTOR_SIZE_X,
						ry+1+blockSize-1);
				cleanRow = eris_ifu_image_collapse(
						hdrl_image_get_image_const(subImg));
				pcleanRow = cpl_vector_get_data_const(cleanRow);

				// flat field image is sliced in 4 major columns
				// col A (slitlet 9 ... 16) [1..15] (exposure: x=[50:900])
				// cal B (slitlet 1) [16]
				// col C (slitlet 32) [17]
				// col D (slitlet 17 ... 24) [18..32]
				subImgA = hdrl_image_extract(subImg, 50+1, 1,
						900+1, blockSize);
				// find right edge of cols A
				medianA = hdrl_image_get_median(subImgA);
				thresholdA = medianA.data * columnThreshold;
				nLower = 0;
				for (int cx = 920; cx <= 1100; cx++) {
					if (pcleanRow[cx] < thresholdA) {
						if (nLower == 0) rightEdgeA = cx;
						nLower += 1;
						if (nLower > 1) break;
					} else {
						nLower = 0;
					}
				}

				// find left edge of  cols A
				nLower = 0;
				leftEdgeA = 4;
				for (int cx = 60; cx >= 4; cx--) {
					if (pcleanRow[cx] < thresholdA) {
						if (nLower == 0) leftEdgeA = cx;
						nLower += 1;
						if (nLower > 3) break;
					} else {
						nLower = 0;
						leftEdgeA = 4;
					}
				}


				int Dleft  = 1150,
						Dright = 2000;
				subImgD = hdrl_image_extract(subImg, Dleft+1, 1,
						Dright+1, blockSize);
				// find right edge of  cols D
				medianD = hdrl_image_get_median(subImgD);
				thresholdD = medianD.data * columnThreshold;
				nLower = 0;
				rightEdgeD = 2044;
				for (int cx = 1980; cx <= 2047; cx++) {
					if (pcleanRow[cx] < thresholdD) {
						if (nLower == 0) rightEdgeD = cx;
						nLower += 1;
						if (nLower > 6) break;
					} else {
						nLower = 0;
						rightEdgeD = 2044;
					}
				}

				// find left edge of  cols D
				nLower = 0;
				for (int cx = 1140; cx >= 1000; cx--) {
					if (pcleanRow[cx] < thresholdD) {
						if (nLower == 0) leftEdgeD = cx;
						nLower += 1;
						if (nLower > 3) break;
					} else {
						nLower = 0;
					}
				}
				leftEdgeD -= 2;
				// get the derivative
				// and find the peak at beginning of col C (slitlet 32) [17]
				//         first mini peak of col B (slitlet 1) [16]
				//         the negative peak at the end of col C
				dnewcleanRowBC = cpl_vector_new(ERIS_IFU_DETECTOR_SIZE_X);
				pdnewcleanRowBC = cpl_vector_get_data(dnewcleanRowBC);
				for (int i = 0; i < ERIS_IFU_DETECTOR_SIZE_X-1; i++) {
					pdnewcleanRowBC[i] = pcleanRow[i+1]- pcleanRow[i];
				}

				int start = rightEdgeA + 1;
				sample = cpl_vector_extract(dnewcleanRowBC, start, leftEdgeD-1, 1);
				sample2 = cpl_vector_extract(sample, 40, cpl_vector_get_size(sample)-1, 1);
				leftEdgeC = cpl_vector_get_maxpos(sample2) + offset + 1;
				eris_ifu_free_vector(&sample2);

				sample2 = cpl_vector_extract(sample, 0, leftEdgeC-10-1, 1);
				leftEdgeB = cpl_vector_get_maxpos(sample2) - 4;
				eris_ifu_free_vector(&sample2);
				rightEdgeC = cpl_vector_get_minpos(sample);
				rightEdgeB = leftEdgeC - 2;
				leftEdgeC += start;
				leftEdgeB += start;
				rightEdgeC += start;
				rightEdgeB += start;

				eris_ifu_flat_data_setBorders(&Data, ry,
						leftEdgeA, rightEdgeA,
						leftEdgeB, rightEdgeB,
						leftEdgeC, rightEdgeC,
						leftEdgeD, rightEdgeD);

				// free memory
				eris_ifu_free_hdrl_image(&subImg);
				eris_ifu_free_hdrl_image(&subImgA);
				eris_ifu_free_hdrl_image(&subImgD);
				eris_ifu_free_vector(&cleanRow);
				eris_ifu_free_vector(&dnewcleanRowBC);
				eris_ifu_free_vector(&sample);
			} // end for ry
			CHECK_ERROR_STATE();

			// interpolate borders of blocks A, B, C, D
			for(int ry = ERIS_IFU_DETECTOR_BP_BORDER;
					ry < ERIS_IFU_DETECTOR_SIZE_Y-ERIS_IFU_DETECTOR_BP_BORDER;
					ry += blockSize)
			{
				subImg = hdrl_image_extract(masterFlat,
						1, ry+1,
						ERIS_IFU_DETECTOR_SIZE_X,
						ry+1+blockSize-1);
				psubImg = cpl_image_get_data(hdrl_image_get_image(subImg));

				borders = eris_ifu_flat_data_getBorderInterpolated(&Data, ry);
				pborders = cpl_vector_get_data(borders);

				leftEdgeA  = pborders[0];
				rightEdgeA = pborders[1];
				leftEdgeB  = pborders[2];
				rightEdgeB = pborders[3];
				leftEdgeC  = pborders[4];
				rightEdgeC = pborders[5];
				leftEdgeD  = pborders[6];
				rightEdgeD = pborders[7];

				// mark columns between major columns A..D as bad
				sx = hdrl_image_get_size_x(subImg);
				sy = hdrl_image_get_size_y(subImg);
				if (4+1 <= leftEdgeA+1) {
					for (int x = 4; x <= leftEdgeA; x++) {
						for (int y = 0; y < sy; y++) {
							BRK_IF_ERROR(
									hdrl_image_reject(subImg, x+1, y+1));
						}
					}
				}

				for (int x = rightEdgeA; x < leftEdgeB; x++) {
					for (int y = 0; y < sy; y++) {
						BRK_IF_ERROR(
								hdrl_image_reject(subImg, x+1, y+1));
					}
				}

				for (int x = rightEdgeB; x < leftEdgeC; x++) {
					for (int y = 0; y < sy; y++) {
						BRK_IF_ERROR(
								hdrl_image_reject(subImg, x+1, y+1));
					}
				}

				for (int x = rightEdgeC; x < leftEdgeD; x++) {
					for (int y = 0; y < sy; y++) {
						BRK_IF_ERROR(
								hdrl_image_reject(subImg, x+1, y+1));
					}
				}

				if (rightEdgeD+1<=2044) {
					for (int x = rightEdgeD; x < 2044; x++) {
						for (int y = 0; y < sy; y++) {
							BRK_IF_ERROR(
									hdrl_image_reject(subImg, x+1, y+1));
						}
					}
				}

				// mark pixel lower than threshold as bad in cols A..B
				BRK_IF_NULL(
						tmpImg = hdrl_image_extract(subImg, leftEdgeA+1, 1, rightEdgeA+1+1, sy));
				coldThreshold = eris_ifu_getThreshold(tmpImg);
				for (int x = leftEdgeA; x < rightEdgeA+1; x++) {
					for (int y = 0; y < sy; y++) {
						if (psubImg[x+y*sx] < coldThreshold) {
							BRK_IF_ERROR(
									hdrl_image_reject(subImg, x+1, y+1));
						}
					}
				}
				eris_ifu_free_hdrl_image(&tmpImg);

				tmpImg = hdrl_image_extract(subImg, leftEdgeB+1, 1, rightEdgeB+1+1, sy);
				coldThreshold = eris_ifu_getThreshold(tmpImg);
				for (int x = leftEdgeB; x < rightEdgeB+1; x++) {
					for (int y = 0; y < sy; y++) {
						if (psubImg[x+y*sx] < coldThreshold) {
							BRK_IF_ERROR(
									hdrl_image_reject(subImg, x+1, y+1));
						}
					}
				}
				eris_ifu_free_hdrl_image(&tmpImg);
				//cpl_msg_warning(cpl_func,"leftEdgeC: %d rightEdgeC: %d",leftEdgeC,rightEdgeC);
				tmpImg = hdrl_image_extract(subImg, leftEdgeC+1, 1, rightEdgeC+1+1, sy);
				coldThreshold = eris_ifu_getThreshold(tmpImg);
				for (int x = leftEdgeC; x < rightEdgeC+1; x++) {
					for (int y = 0; y < sy; y++) {
						if (psubImg[x+y*sx] < coldThreshold) {
							BRK_IF_ERROR(
									hdrl_image_reject(subImg, x+1, y+1));
						}
					}
				}
				eris_ifu_free_hdrl_image(&tmpImg);

				tmpImg = hdrl_image_extract(subImg, leftEdgeD+1, 1, rightEdgeD+1+1, sy);
				coldThreshold = eris_ifu_getThreshold(tmpImg);
				for (int x = leftEdgeD; x < rightEdgeD+1; x++) {
					for (int y = 0; y < sy; y++) {
						if (psubImg[x+y*sx] < coldThreshold) {
							BRK_IF_ERROR(
									hdrl_image_reject(subImg, x+1, y+1));
						}
					}
				}
				eris_ifu_free_hdrl_image(&tmpImg);

				// copy subImg back to masterFlat
				for (int x = 1; x <= ERIS_IFU_DETECTOR_SIZE_X; x++) {
					int y2 = 1;
					for (int y = ry+1; y <= ry+1+blockSize-1; y++) {
						hdrl_image_set_pixel(masterFlat, x, y, hdrl_image_get_pixel(subImg, x, y2++, &rej));
						CHECK_ERROR_STATE();
						if (rej == 1) {
							BRK_IF_ERROR(
									hdrl_image_reject(masterFlat, x, y));
						}
					}
				}

				// free memory
				eris_ifu_free_hdrl_image(&subImg);
				eris_ifu_free_vector(&borders);
			} // end for ry
			CHECK_ERROR_STATE();

			// more that 8 bad pixels in a row indicate an absorption line
			const double *pleftA  = cpl_vector_get_data_const(Data.borderAleft),
					*prightD = cpl_vector_get_data_const(Data.borderDright);
			for (int rx = 4; rx < 2044; rx++) {
				bool lineCheck = false;
				int bpCnt = 0,
						startPos = 0,
						rowIndex = eris_ifu_matchRow(blockSize, rx);

				for (int colx = pleftA[rowIndex]; colx < prightD[rowIndex]+1; colx++) {
					int rx_left = rx+3;
					if (rx_left > 2044) {
						rx_left = 2044;
					}
					int rx_right = rx-3;
					if (rx_right < 4) {
						rx_right = 4;
					}
					if (hdrl_image_is_rejected(masterFlat, rx+1, colx+1) &&
							!hdrl_image_is_rejected(masterFlat, rx_left+1, colx+1) &&
							!hdrl_image_is_rejected(masterFlat, rx_right+1, colx+1))
					{
						if (lineCheck == false) {
							startPos = colx;
							lineCheck = true;
						}
						bpCnt += 1;
					} else {
						if (lineCheck) {
							double cPix       = hdrl_image_get_pixel(masterFlat, rx+1, colx+1, &rej).data,
									cPixBefore = hdrl_image_get_pixel(masterFlat, rx+1, colx+1-1, &rej).data,
									cPixAfter  = hdrl_image_get_pixel(masterFlat, rx+1, colx+1+1, &rej).data,
									thresh     = cPix * .2;
							CHECK_ERROR_STATE();
							if ((fabs(cPix - cPixBefore) < thresh) &&
									(fabs(cPix - cPixAfter) < thresh))
							{
								bpCnt += 1;
							} else {
								if (bpCnt > 8) {
									for (int ax = startPos; ax < colx; ax++) {
										BRK_IF_ERROR(
												hdrl_image_reject(masterFlat, rx+1, ax+1));
									}
								}
								lineCheck = false;
								bpCnt = 0;
							}
						}
					}
				}
			}

			// now finally normalize
			hdrl_value a = hdrl_image_get_mean(masterFlat);
			hdrl_image_div_scalar(masterFlat, a);
			CHECK_ERROR_STATE();

			*masterFlatHdrlImg_lo = masterFlat;
			eris_ifu_free_structFlatData(&Data);
		} else {
			BRK_WITH_ERROR_MSG(CPL_ERROR_ILLEGAL_INPUT, "Wrong mode provided.");
		}
	}
	CATCH
	{
		eris_ifu_free_hdrl_image(masterFlatHdrlImg_lo);
		eris_ifu_free_image(contrib_map);
		if (productDepth & 1) {
			eris_ifu_free_hdrl_image(masterFlatHdrlImg_hi);
		}
		ret = cpl_error_get_code();
	}

	eris_ifu_free_image(&noiseImage);
	eris_ifu_free_hdrl_imagelist(&tmp_hdrl_imglist);
	eris_ifu_free_hdrl_parameter(&pHdrlCollapse);
	eris_ifu_free_hdrl_parameter(&pHdrlFlat_lo);
	eris_ifu_free_hdrl_parameter(&pHdrlFlat_hi);
	eris_ifu_free_string(&tmp_str);
	eris_check_error_code("eris_ifu_calc_flat");
	return ret;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Load flat field lamp frames into separate on/off image lists
  @param    fs                 Input frameset containing raw flat frames and calibration files
  @param    gain               (Output) Detector gain value from first on-frame header [e-/ADU]
  @param    exptime            (Output) Exposure time from first on-frame header [s]
  @param    hdrl_imglist_on    (Output) HDRL image list of all lamp-on frames
  @param    hdrl_imglist_off   (Output) HDRL image list of all lamp-off frames
  @return   CPL_ERROR_NONE on success, otherwise an error code
  @note     All output pointers must be freed by the caller
  @note     Loads and applies bad pixel masks from dark (ERIS_IFU_PRO_DARK_BPM),
            distortion (ERIS_IFU_PRO_DIST_BPM), and detector linearity (ERIS_IFU_PRO_DETLIN_BPM)
            recipes if available
  @note     Supports both FLAT_LAMP and FLAT_NS frame types, with _ON/_OFF variants
  @note     Determines lamp-on/off status using eris_ifu_frame_is_on() and eris_ifu_frame_is_sky()
  @see      eris_ifu_flat_static, eris_ifu_load_badpixel_mask, eris_ifu_load_exposure_file
 */
/*----------------------------------------------------------------------------*/
cpl_error_code eris_ifu_flat_load_frames(const cpl_frameset  *fs,
		double              *gain,
		double              *exptime,
		hdrl_imagelist      **hdrl_imglist_on,
		hdrl_imagelist      **hdrl_imglist_off)
{
	cpl_error_code      err             = CPL_ERROR_NONE;
	cpl_frameset        *fs_extracted   = NULL;
	cpl_frameset        *fs_off   = NULL;
	hdrl_image          *tmp_img        = NULL;
	cpl_propertylist    *plist          = NULL;
	cpl_mask            *bpm_dark       = NULL,
			*bpm_dist       = NULL,
			*bpm_detlin     = NULL;
	const cpl_frame     *fr             = NULL;
	const char          *fn             = NULL;
	char                *tag            = NULL;
	bool                first_on        = false;
	int                 nr_on           = 0,
			nr_off          = 0;

	cpl_ensure_code(fs,                 CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(hdrl_imglist_on,    CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(hdrl_imglist_off,   CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(gain,               CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(exptime,            CPL_ERROR_NULL_INPUT);

	TRY
	{
		/* Load the badpixel mask if present:
           flat:       yes
           distortion: no
		 */
		bpm_dark = eris_ifu_load_badpixel_mask(fs, ERIS_IFU_PRO_DARK_BPM, 0);

		if (bpm_dark != NULL) {
			if (cpl_frameset_count_tags(fs, ERIS_IFU_PRO_DIST_BPM) == 1) {
				bpm_dist = eris_ifu_load_badpixel_mask(fs,
						ERIS_IFU_PRO_DIST_BPM, 0);
				cpl_mask_or(bpm_dark, bpm_dist);
			}

			if (cpl_frameset_count_tags(fs, ERIS_IFU_PRO_DETLIN_BPM) == 1) {
				bpm_detlin = eris_ifu_load_badpixel_mask(fs,
						ERIS_IFU_PRO_DETLIN_BPM, 0);
				cpl_mask_or(bpm_dark, bpm_detlin);
			}
		}

		cpl_msg_info(cpl_func, "Loading all flat frames...");

		/* check if FLAT_LAMP or FLAT_NS frames exist */
		if (cpl_frameset_count_tags(fs, ERIS_IFU_RAW_FLAT_LAMP) > 0) {
			tag = cpl_sprintf("%s", ERIS_IFU_RAW_FLAT_LAMP);
			fs_extracted = eris_ifu_get_frameset_by_tag(fs, tag);
		} else if ( (cpl_frameset_count_tags(fs, ERIS_IFU_RAW_FLAT_LAMP_ON) > 0) &&
				    (cpl_frameset_count_tags(fs, ERIS_IFU_RAW_FLAT_LAMP_OFF) > 0) ) {
			
			fs_extracted = eris_ifu_get_frameset_by_tag(fs, ERIS_IFU_RAW_FLAT_LAMP_ON);

			fs_off = eris_ifu_get_frameset_by_tag(fs, ERIS_IFU_RAW_FLAT_LAMP_OFF);
		
            cpl_frameset_join(fs_extracted, fs_off);
			tag = cpl_sprintf("%s", ERIS_IFU_RAW_FLAT_LAMP_ON);
		} else if (cpl_frameset_count_tags(fs, ERIS_IFU_RAW_FLAT_NS) > 0) {
			tag = cpl_sprintf("%s", ERIS_IFU_RAW_FLAT_NS);
			fs_extracted = eris_ifu_get_frameset_by_tag(fs, tag);
		} else if ( (cpl_frameset_count_tags(fs, ERIS_IFU_RAW_FLAT_NS_ON) > 0) &&
				(cpl_frameset_count_tags(fs, ERIS_IFU_RAW_FLAT_NS_OFF) > 0) ) {

			fs_extracted = eris_ifu_get_frameset_by_tag(fs, ERIS_IFU_RAW_FLAT_NS_ON);

			fs_off = eris_ifu_get_frameset_by_tag(fs, ERIS_IFU_RAW_FLAT_NS_OFF);

			cpl_frameset_join(fs_extracted, fs_off);
			tag = cpl_sprintf("%s", ERIS_IFU_RAW_FLAT_NS_ON);


		} else {
			BRK_WITH_ERROR_MSG(CPL_ERROR_ILLEGAL_INPUT,
					"Neither FLAT_LAMP or FLAT_NS "
					"frames present!");
		}

		if (cpl_frameset_get_size(fs_extracted) < 1) {
			BRK_WITH_ERROR_MSG(CPL_ERROR_ILLEGAL_INPUT,
					"Not enough FLAT RAW frames!");
		}

		*hdrl_imglist_on = hdrl_imagelist_new();
		*hdrl_imglist_off = hdrl_imagelist_new();

		/*
		 * Loop through all FLAT_LAMP frames and save on- and off-frames
		 * accordingly in their cpl_imagelists
		 */
		fr = cpl_frameset_find_const(fs_extracted, tag);
		cpl_size frameCnt = cpl_frameset_get_size(fs_extracted);
                
		/* AMO: the following while need to be changed to scan all frames of the set not only the ones with frame tag tag */
		for (cpl_size i = 0 ; i < frameCnt ; i++) {
			fr = cpl_frameset_get_position_const(fs_extracted, i) ;
			fn = cpl_frame_get_filename(fr);
			eris_ifu_file_exists(fn);

			/* If the frame has a tag we process it. Else it is an object and
			 * is ignored */
			if (cpl_frame_get_tag(fr) != NULL) {
				tmp_img = eris_ifu_load_exposure_file(fn, 0, NULL);
				if (bpm_dark != NULL) {
					hdrl_image_reject_from_mask(tmp_img, bpm_dark);
				}

				if (!eris_ifu_frame_is_on(fr) || eris_ifu_frame_is_sky(fr)) {
					hdrl_imagelist_set(*hdrl_imglist_off, tmp_img,
							hdrl_imagelist_get_size(*hdrl_imglist_off));
					tmp_img = NULL;
				} else if (eris_ifu_frame_is_on(fr)  == 1) {
					hdrl_imagelist_set(*hdrl_imglist_on, tmp_img,
							hdrl_imagelist_get_size(*hdrl_imglist_on));
					tmp_img = NULL;
					if (!first_on) {
						BRK_IF_NULL(
								plist = cpl_propertylist_load(fn, 0));
						if (cpl_propertylist_has(plist, FHDR_DET_CHIP_GAIN)) {
							*gain = cpl_propertylist_get_double(plist,
									FHDR_DET_CHIP_GAIN);
						} else {
							*gain = 1.;
						}
						if (cpl_propertylist_has(plist, FHDR_EXPTIME)) {
							*exptime = cpl_propertylist_get_double(plist,
									FHDR_EXPTIME);
						} else {
							*exptime = 1.;
						}
						first_on = true;
					}
				}
			} else {
				/* if (tag == NULL) do nothing */
			}
		}

		nr_on  = (int)hdrl_imagelist_get_size(*hdrl_imglist_on);
		nr_off = (int)hdrl_imagelist_get_size(*hdrl_imglist_off);
		cpl_msg_info(cpl_func,"nr_on: %d nr_off: %d",nr_on, nr_off);
		if (nr_on != nr_off) {
			cpl_msg_warning(cpl_func,
					"Number obj/off frames differs (#obj= %d #off= %d)!",
					(int)hdrl_imagelist_get_size(*hdrl_imglist_on),
					(int)hdrl_imagelist_get_size(*hdrl_imglist_off));
		}

		if (hdrl_imagelist_get_size(*hdrl_imglist_on) < 1) {
			BRK_WITH_ERROR_MSG(CPL_ERROR_ILLEGAL_INPUT,
					"No ON raw frame in input, something wrong!");
		}

		if (hdrl_imagelist_get_size(*hdrl_imglist_on) <= 0) {
			BRK_WITH_ERROR_MSG(CPL_ERROR_ILLEGAL_INPUT,
					"input object list's size 0. Check"
					" your input data!");
		}
	}
	CATCH
	{
		CATCH_MSGS();
		eris_ifu_free_hdrl_imagelist(hdrl_imglist_on);
		eris_ifu_free_hdrl_imagelist(hdrl_imglist_off);

		err = cpl_error_get_code();
	}
        eris_ifu_free_frameset(&fs_off);
	eris_ifu_free_frameset(&fs_extracted);
	eris_ifu_free_propertylist(&plist);
	eris_ifu_free_string(&tag);
	cpl_mask_delete(bpm_dark);
	cpl_mask_delete(bpm_dist);
	cpl_mask_delete(bpm_detlin);
	hdrl_image_delete(tmp_img);
	eris_check_error_code("eris_ifu_flat_load_frames");
	return err;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Calculate quality control parameters before flat field computation
  @param    parlist            Recipe parameter list
  @param    recipe_name        Name of the calling recipe
  @param    hdrl_imglist_on    List of lamp-on flat frames
  @param    hdrl_imglist_off   List of lamp-off flat frames
  @param    qc_list            Property list to which QC parameters will be appended
  @return   CPL_ERROR_NONE on success, otherwise an error code
  @note     Calculates and appends the following QC parameters:
            - FLAT SAT NCOUNTS: Number of saturated pixels
            - SPECFLAT OFFFLUX: Average flux in off frames [ADU]
            - SPECFLAT NCNTSAVG: Average counts (on-off) [ADU]
            - SPECFLAT NCNTSSTD: Standard deviation of counts [ADU]
            - LFLAT FPN1: Fixed Pattern Noise in first rectangular region [ADU]
            - LFLAT FPN2: Fixed Pattern Noise in second rectangular region [ADU]
  @note     FPN regions are defined by recipe parameters eris.<recipe>.qc.fpn.{xmin,xmax,ymin}{1,2}
  @see      eris_ifu_flat_static, eris_ifu_get_qc_saturated
 */
/*----------------------------------------------------------------------------*/
cpl_error_code eris_ifu_flat_calc_qc_pre(
		const cpl_parameterlist *parlist,
		const char              *recipe_name,
		const hdrl_imagelist    *hdrl_imglist_on,
		const hdrl_imagelist    *hdrl_imglist_off,
		cpl_propertylist        *qc_list)
{
	cpl_error_code      err             = CPL_ERROR_NONE;
	int                 qc_sat          = 0;
	cpl_vector          *vec_off        = NULL;
	cpl_vector          *vec_diff       = NULL;
	const cpl_parameter *p              =NULL;
	hdrl_parameter      *param          = NULL;
	const hdrl_image    *img_on_const   = NULL;
	const hdrl_image    *img_off_const  = NULL;
	hdrl_image          *img_on         = NULL;
	hdrl_image          *img_off        = NULL;
	hdrl_image          *img            = NULL;
	cpl_image           *contrib        = NULL;
	int                 nr_on           = 0;
	int                 nr_off          = 0;
	int                 nr_fr           = 0;
	int                 xmin1           = 0;
	int                 xmax1           = 0;
	int                 ymin1           = 0;
	int                 ymax1           = 0;
	int                 xmin2           = 0;
	int                 xmax2           = 0;
	int                 ymin2           = 0;
	int                 ymax2           = 0;
	double              fpn_stdev1      = 0.;
	double              fpn_stdev2      = 0.;
	char                *tmp_str        = NULL;

	cpl_ensure_code(parlist,            CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(recipe_name,        CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(hdrl_imglist_on,    CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(hdrl_imglist_off,   CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(qc_list,            CPL_ERROR_NULL_INPUT);

	TRY
	{
		/* calc QC saturated pixels */
		qc_sat = eris_ifu_get_qc_saturated(hdrl_imglist_on);
		CHECK_ERROR_STATE();

		eris_ifu_append_qc_int(qc_list, "FLAT SAT NCOUNTS", qc_sat,
				"nr. saturated pixels of master flat");

		/* calc QC flux & counts */
		nr_on  = hdrl_imagelist_get_size(hdrl_imglist_on);
		nr_off = hdrl_imagelist_get_size(hdrl_imglist_off);
		CHECK_ERROR_STATE();
		nr_fr = (nr_on <= nr_off) ? nr_on : nr_off;

		if (nr_on != nr_off) {
			cpl_msg_warning(cpl_func,
					"Number obj/off frames differs (#obj= %d #off= %d)!",
					nr_on, nr_off);
		}

		vec_off = cpl_vector_new(nr_fr);
		vec_diff = cpl_vector_new(nr_fr);

		for (int i = 0; i < nr_fr; i++) {
			img_on_const  = hdrl_imagelist_get_const(hdrl_imglist_on, i);
			img_off_const = hdrl_imagelist_get_const(hdrl_imglist_off, i);

			img = hdrl_image_sub_image_create(img_on_const, img_off_const);

			cpl_vector_set(vec_diff, i, hdrl_image_get_median(img).data);
			cpl_vector_set(vec_off, i, hdrl_image_get_median(img_off_const).data);

			CHECK_ERROR_STATE();

			eris_ifu_free_hdrl_image(&img);
		}

		eris_ifu_append_qc_double(qc_list, "SPECFLAT OFFFLUX",
				cpl_vector_get_mean(vec_off),
				"[ADU] average flux off frames");
		eris_ifu_append_qc_double(qc_list, "SPECFLAT NCNTSAVG",
				cpl_vector_get_mean(vec_diff),
				"[ADU] average counts");
		if(nr_fr > 1 ) {
			eris_ifu_append_qc_double(qc_list, "SPECFLAT NCNTSSTD",
					cpl_vector_get_stdev(vec_diff),
					"[ADU] stdev counts");
		}

		/* calc fixed pattern noise */
		tmp_str = cpl_sprintf("eris.%s.qc.fpn.xmin1", recipe_name);
		p = cpl_parameterlist_find_const(parlist, tmp_str);
		xmin1 = cpl_parameter_get_int(p);
		eris_ifu_free_string(&tmp_str);
		CHECK_ERROR_STATE();

		tmp_str = cpl_sprintf("eris.%s.qc.fpn.xmax1", recipe_name);
		p = cpl_parameterlist_find_const(parlist, tmp_str);
		xmax1 = cpl_parameter_get_int(p);
		eris_ifu_free_string(&tmp_str);
		CHECK_ERROR_STATE();

		tmp_str = cpl_sprintf("eris.%s.qc.fpn.ymin1", recipe_name);
		p = cpl_parameterlist_find_const(parlist, tmp_str);
		ymin1 = cpl_parameter_get_int(p);
		eris_ifu_free_string(&tmp_str);
		CHECK_ERROR_STATE();

		tmp_str = cpl_sprintf("eris.%s.qc.fpn.ymax1", recipe_name);
		p = cpl_parameterlist_find_const(parlist, tmp_str);
		ymax1 = cpl_parameter_get_int(p);
		eris_ifu_free_string(&tmp_str);
		CHECK_ERROR_STATE();

		tmp_str = cpl_sprintf("eris.%s.qc.fpn.xmin2", recipe_name);
		p = cpl_parameterlist_find_const(parlist, tmp_str);
		xmin2 = cpl_parameter_get_int(p);
		eris_ifu_free_string(&tmp_str);
		CHECK_ERROR_STATE();

		tmp_str = cpl_sprintf("eris.%s.qc.fpn.xmax2", recipe_name);
		p = cpl_parameterlist_find_const(parlist, tmp_str);
		xmax2 = cpl_parameter_get_int(p);
		eris_ifu_free_string(&tmp_str);
		CHECK_ERROR_STATE();

		tmp_str = cpl_sprintf("eris.%s.qc.fpn.ymin2", recipe_name);
		p = cpl_parameterlist_find_const(parlist, tmp_str);
		ymin2 = cpl_parameter_get_int(p);
		eris_ifu_free_string(&tmp_str);
		CHECK_ERROR_STATE();

		tmp_str = cpl_sprintf("eris.%s.qc.fpn.ymax2", recipe_name);
		p = cpl_parameterlist_find_const(parlist, tmp_str);
		ymax2 = cpl_parameter_get_int(p);
		eris_ifu_free_string(&tmp_str);
		CHECK_ERROR_STATE();

		tmp_str = cpl_sprintf("eris.%s.collapse", recipe_name);
		param = hdrl_collapse_parameter_parse_parlist(parlist, tmp_str);
		eris_ifu_free_string(&tmp_str);

		hdrl_imagelist_collapse(hdrl_imglist_on,
				param,
				&img_on,
				&contrib);
		eris_ifu_free_image(&contrib);

		hdrl_imagelist_collapse(hdrl_imglist_off,
				param,
				&img_off,
				&contrib);
		eris_ifu_free_image(&contrib);

		hdrl_image_sub_image(img_on, img_off);
		eris_ifu_free_hdrl_image(&img_off);

		fpn_stdev1 = cpl_image_get_stdev_window(hdrl_image_get_image_const(img_on),
				xmin1, ymin1, xmax1, ymax1);
		CHECK_ERROR_STATE();

		fpn_stdev2 = cpl_image_get_stdev_window(hdrl_image_get_image_const(img_on),
				xmin2, ymin2, xmax2, ymax2);
		CHECK_ERROR_STATE();

		eris_ifu_free_hdrl_image(&img_on);

		eris_ifu_append_qc_double(qc_list, "LFLAT FPN1", fpn_stdev1,
				"[ADU] Fixed Pattern Noise of combined frames");

		eris_ifu_append_qc_double(qc_list, "LFLAT FPN2", fpn_stdev2,
				"[ADU] Fixed Pattern Noise of combined frames");
	}
	CATCH
	{
		CATCH_MSGS();
		err = cpl_error_get_code();

		eris_ifu_free_hdrl_image(&img);
	}

	eris_ifu_free_vector(&vec_off);
	eris_ifu_free_vector(&vec_diff);
	eris_ifu_free_hdrl_parameter(&param);
	eris_ifu_free_string(&tmp_str);
	eris_check_error_code("eris_ifu_flat_calc_qc_pre");
	return err;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Calculate quality control parameters after flat field computation
  @param    parlist                 Recipe parameter list
  @param    recipe_name             Name of the calling recipe
  @param    productDepth            Product depth flags controlling debug output
  @param    procatg_prefix          Prefix for product category strings
  @param    masterFlatHdrlImg_hi    High-frequency master flat (optional, can be NULL)
  @param    masterFlatHdrlImg_lo    Low-frequency master flat
  @param    hdrl_imglist_on         List of lamp-on frames
  @param    hdrl_imglist_off        List of lamp-off frames
  @param    contrib_map             Contribution map image
  @param    bpm2dMask               2D bad pixel mask (optional, can be NULL)
  @param    bpm3dMask               3D bad pixel mask (optional, can be NULL)
  @param    fs                      Input frameset
  @param    qc_list                 Property list to which QC parameters will be appended
  @param    qualityImage            (Output) Quality/bad pixel image with DQI flags
  @return   CPL_ERROR_NONE on success, otherwise an error code
  @note     Calculates and appends QC parameters including flat statistics (STDDEV, MEAN, MEDIAN)
            and bad pixel counts from various masks
  @note     Generates optional debug products if productDepth flags are set (contribution map,
            high-frequency flat, 2D/3D BPMs, on/off cubes)
  @note     Sets DQI flags in qualityImage: ERIS_DQI_BP for all bad pixels,
            ERIS_DQI_BP_BPM2D for 2D outliers, ERIS_DQI_BP_BPM3D for 3D noisy pixels
  @see      eris_ifu_flat_static, eris_ifu_get_badpix_qc_from_mask
 */
/*----------------------------------------------------------------------------*/
cpl_error_code eris_ifu_flat_calc_qc_post(const cpl_parameterlist *parlist,
		const char               *recipe_name,
		const int                productDepth,
		const char               *procatg_prefix,
		const hdrl_image         *masterFlatHdrlImg_hi,
		const hdrl_image         *masterFlatHdrlImg_lo,
		const hdrl_imagelist     *hdrl_imglist_on,
		const hdrl_imagelist     *hdrl_imglist_off,
		const cpl_image          *contrib_map,
		//                                         const cpl_image          *bpm2dImg,
		//                                         const cpl_image          *bpm3dImg,
		const cpl_mask           *bpm2dMask,
		const cpl_mask           *bpm3dMask,
		cpl_frameset             *fs,
		cpl_propertylist         *qc_list,
		cpl_image                **qualityImage)
{
	cpl_error_code      err             = CPL_ERROR_NONE;
	//int                 nBadPix         = 0;
	//double              fracBadPix      = 0;

	char                *fn             = NULL,
			*pc             = NULL;
	const cpl_mask      *bp_mask_flat   = NULL;
	cpl_image           *tmpImg         = NULL,
			*bpm2dImg       = NULL,
			*bpm3dImg       = NULL;
	//cpl_size            sx = 0;
	//cpl_size            sy = 0;
	//cpl_size            npix = 0;

	cpl_ensure_code(parlist,                CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(recipe_name,            CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(procatg_prefix,         CPL_ERROR_NULL_INPUT);
	//    cpl_ensure_code(masterFlatHdrlImg_hi,   CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(masterFlatHdrlImg_lo,   CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(hdrl_imglist_on,        CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(hdrl_imglist_off,       CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(contrib_map,            CPL_ERROR_NULL_INPUT);
	//    cpl_ensure_code(bpm2dImg,               CPL_ERROR_NULL_INPUT);
	//    cpl_ensure_code(bpm3dImg,               CPL_ERROR_NULL_INPUT);
	//    cpl_ensure_code(bpm2dMask,              CPL_ERROR_NULL_INPUT);
	//    cpl_ensure_code(bpm3dMask,              CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(fs,                     CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(qc_list,                CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(qualityImage,           CPL_ERROR_NULL_INPUT);

	TRY
	{
		/* fetch bad-pix-image */
		bp_mask_flat = hdrl_image_get_mask_const(masterFlatHdrlImg_lo);
		eris_ifu_get_badpix_qc_from_mask(bp_mask_flat, qc_list, "FLAT");

		*qualityImage = cpl_image_new_from_mask(bp_mask_flat);

		/* set "bad pixel flag" */
		cpl_image_multiply_scalar(*qualityImage, ERIS_DQI_BP);


		const cpl_image* ima = hdrl_image_get_image_const(masterFlatHdrlImg_lo);

		double stdev = cpl_image_get_stdev(ima);
		double mean = cpl_image_get_mean(ima);
		double median = cpl_image_get_median(ima);
		char* key_name;
		char* key_comm;


		key_name = cpl_sprintf("ESO QC FLAT STDDEV");
		key_comm = cpl_sprintf("[ADU] Std deviation on flat image");
		cpl_propertylist_append_double(qc_list,key_name, stdev);
		cpl_propertylist_set_comment(qc_list, key_name, key_comm);
		cpl_free(key_name);
		cpl_free(key_comm);


		key_name = cpl_sprintf("ESO QC FLAT MEAN");
		key_comm = cpl_sprintf("[ADU] Mean value on flat image");
		cpl_propertylist_append_double(qc_list,key_name, mean);
		cpl_propertylist_set_comment(qc_list, key_name, key_comm);
		cpl_free(key_name);
		cpl_free(key_comm);

		key_name = cpl_sprintf("ESO QC FLAT MEDIAN");
		key_comm = cpl_sprintf("[ADU] Median value on flat image");
		cpl_propertylist_append_double(qc_list,key_name, median);
		cpl_propertylist_set_comment(qc_list, key_name, key_comm);
		cpl_free(key_name);
		cpl_free(key_comm);


		if (bpm2dMask != NULL) {
			bpm2dImg = cpl_image_new_from_mask(bpm2dMask);
			tmpImg = cpl_image_duplicate(bpm2dImg);
			/* set "user defined flag1; outlier pixels" */
			cpl_image_multiply_scalar(tmpImg, ERIS_DQI_BP_BPM2D);
			cpl_image_add(*qualityImage, tmpImg);
			eris_ifu_free_image(&tmpImg);
		}
		if (bpm3dMask != NULL) {
			bpm3dImg = cpl_image_new_from_mask(bpm3dMask);
			tmpImg = cpl_image_duplicate(bpm3dImg);
			/* set "user defined flag1; noisy pixels" */
			cpl_image_multiply_scalar(tmpImg, ERIS_DQI_BP_BPM3D);
			cpl_image_add(*qualityImage, tmpImg);
			eris_ifu_free_image(&tmpImg);
		}

		/* save optional images */
		if ((hdrl_imglist_on != NULL) && ((productDepth & 2) != 0)) {
			fn = cpl_sprintf("%s%s", recipe_name,
					ERIS_IFU_PRO_FLAT_DBG_CUBE_ON_FN);
			eris_ifu_save_hdrl_imagelist_dbg(hdrl_imglist_on, fn, TRUE);
			eris_ifu_free_string(&fn);
		}

		if ((hdrl_imglist_off != NULL) && ((productDepth & 2) != 0)) {
			fn = cpl_sprintf("%s%s", recipe_name,
					ERIS_IFU_PRO_FLAT_DBG_CUBE_OFF_FN);
			eris_ifu_save_hdrl_imagelist_dbg(hdrl_imglist_off, fn, TRUE);
			eris_ifu_free_string(&fn);
		}

		if ((contrib_map != NULL) && ((productDepth & 1) != 0)) {
			fn = cpl_sprintf("%s%s", recipe_name,
					ERIS_IFU_PRO_FLAT_DBG_CONTRIBMAP_FN);
			pc = cpl_sprintf("%s%s", procatg_prefix,
					ERIS_IFU_PRO_FLAT_DBG_CONTRIBMAP);
			eris_ifu_save_image(fs, qc_list, parlist, recipe_name,
					pc, fn, CPL_TYPE_USHORT, contrib_map);
			eris_ifu_free_string(&fn);
			eris_ifu_free_string(&pc);
		}

		if ((masterFlatHdrlImg_hi != NULL) && ((productDepth & 1) != 0)) {
			fn = cpl_sprintf("%s%s", recipe_name,
					ERIS_IFU_PRO_FLAT_DBG_HI_FN);
			pc = cpl_sprintf("%s%s", procatg_prefix,
					ERIS_IFU_PRO_FLAT_DBG_HI);
			eris_ifu_save_image(fs, qc_list, parlist, recipe_name,
					pc, fn, CPL_TYPE_DOUBLE,
					hdrl_image_get_image_const(masterFlatHdrlImg_hi));
			eris_ifu_free_string(&fn);
			eris_ifu_free_string(&pc);
		}

		if ((bpm2dMask != NULL) && ((productDepth & 1) != 0)) {
			eris_ifu_get_badpix_qc_from_mask(bpm2dMask, qc_list, "FLAT");

			fn = cpl_sprintf("%s%s", recipe_name,
					ERIS_IFU_PRO_FLAT_DBG_BPM2D_FN);
			pc = cpl_sprintf("%s%s", procatg_prefix,
					ERIS_IFU_PRO_FLAT_DBG_BPM2D);
			eris_ifu_save_image(fs, qc_list, parlist, recipe_name,
					pc, fn, CPL_TYPE_USHORT, bpm2dImg);
			eris_ifu_free_string(&fn);
			eris_ifu_free_string(&pc);
		}

		if ((bpm3dMask != NULL) && ((productDepth & 1) != 0)) {
			eris_ifu_get_badpix_qc_from_mask(bpm3dMask, qc_list, "FLAT");

			fn = cpl_sprintf("%s%s", recipe_name,
					ERIS_IFU_PRO_FLAT_DBG_BPM3D_FN);
			pc = cpl_sprintf("%s%s", procatg_prefix,
					ERIS_IFU_PRO_FLAT_DBG_BPM3D);
			eris_ifu_save_image(fs, qc_list, parlist, recipe_name,
					pc, fn, CPL_TYPE_USHORT, bpm3dImg);
			eris_ifu_free_string(&fn);
			eris_ifu_free_string(&pc);
		}
		//cpl_propertylist_erase(qc_list, "ESO QC FLAT NBADPIX");
		CHECK_ERROR_STATE();

		cpl_propertylist_update_string(qc_list, CPL_DFS_PRO_CATG, "");
	}
	CATCH
	{
		CATCH_MSGS();
		err = cpl_error_get_code();
		eris_ifu_free_image(qualityImage);
	}

	eris_ifu_free_image(&bpm2dImg);
	eris_ifu_free_image(&bpm3dImg);
	eris_check_error_code("eris_ifu_flat_calc_qc_post");
	return err;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Get the number of saturated pixels in an image list
  @param    dataHdrl    Input HDRL image list
  @return   Number of pixels considered saturated, or -1 on error
  @note     A pixel is counted as saturated if it exceeds ERIS_IFU_FLAT_SATURATED threshold
            in at least ERIS_IFU_FLAT_SAT_MIN frames
  @note     Rejected/bad pixels are not considered in the count
  @see      eris_ifu_flat_calc_qc_pre
 */
/*----------------------------------------------------------------------------*/
int eris_ifu_get_qc_saturated(const hdrl_imagelist *dataHdrl)
{
	int                 saturated_pixels    = 0,
			tmp_sat             = 0;
	cpl_size            nx                  = 0,
			ny                  = 0,
			nz                  = 0,
			sat_min             = ERIS_IFU_FLAT_SAT_MIN;
	double              threshold           = ERIS_IFU_FLAT_SATURATED;
	const hdrl_image    *curHdrlImg         = NULL;
	const cpl_image     *curImg             = NULL;
	const double        *pcur_img           = NULL;

	cpl_ensure_code(dataHdrl, CPL_ERROR_NULL_INPUT);

	TRY
	{
		ASSURE((threshold > 0.0) &&
				(sat_min > 0),
				CPL_ERROR_ILLEGAL_INPUT,
				"threshold and sat_min must be greater than zero!");

		curHdrlImg = hdrl_imagelist_get_const(dataHdrl, 0);

		nx = hdrl_image_get_size_x(curHdrlImg);
		ny = hdrl_image_get_size_y(curHdrlImg);
		nz = hdrl_imagelist_get_size(dataHdrl);
		CHECK_ERROR_STATE();

		if (nz >= sat_min) {
			/* Loop on the pixels of the data-image */
			int ix = 0, iy = 0, iz = 0;
			for (iy = 0; iy < ny; iy++) {
				for (ix = 0; ix < nx; ix++) {
					tmp_sat = 0;
					for (iz = 0; iz < nz; iz++) {
						curHdrlImg = hdrl_imagelist_get_const(dataHdrl, iz);
						curImg = hdrl_image_get_image_const(curHdrlImg);
						pcur_img = cpl_image_get_data_double_const(curImg);

						if (!cpl_image_is_rejected(curImg, ix+1, iy+1) &&
								(pcur_img[ix+iy*nx] > threshold)) {
							tmp_sat++;
						}
					}

					if (tmp_sat >= sat_min) {
						saturated_pixels++;
					}
				}
			}
		}
	}
	CATCH
	{
		CATCH_MSG();
		saturated_pixels = -1;
	}
	eris_check_error_code("eris_ifu_get_qc_saturated");
	return saturated_pixels;
}

/**
@brief
   @name    eris_ifu_column_tilt
   @brief   Remove the column intensity tilt only pixels which lie within a
            defined noiselimit are used to fit a straight line. (from SINFONI)

   @param image , factor of sigma noise limit to determine
                        pixels that are used for the fit.
   @return image


   @doc         : first calculates statistics for each column of an image.
                  median value and standard deviation of columns are determined,
                  blank values are excluded. Fits a straight
                  line through the pixel values of each column and subtracts
                  the fit in order to remove the tilt of each column.
                  Only those pixels are used for the fit that are within
                  a defined factor of sigma noise limit. The noise is
                  calculated from pixels between the 10percentil and
                  90 percentil points.
                  if the straight line could not be determined, the median
                  of the column is subtracted from the column

   @note       :  works only for raw or averaged raw images

 */
cpl_error_code eris_ifu_column_tilt(hdrl_image *hdrl_img, double sigma)
{
	cpl_error_code  err         = CPL_ERROR_NONE;
	cpl_image       *img        = NULL;
	cpl_binary      *pmask      = NULL;
	cpl_mask        *mask       = NULL;
	int             mwt         = 0,
			nx          = 0,
			ny          = 0,
			npix        = 0,
			col_nr      = 0,
			slope       = 1000,
			saturation  = 50000;
	double          mean        = 0.,
			sum         = 0.,
			sum2        = 0.,
			sinfo_median= 0,
			noise       = 0.,
			a           = 0.,
			b           = 0.,
			siga        = 0.,
			sigb        = 0.,
			chi2        = 0.,
			q           = 0.,
			*sig        = NULL,
			*dat        = NULL,
			*column     = NULL,
			*pimg       = NULL;

	cpl_ensure_code(hdrl_img, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(sigma > 0., CPL_ERROR_ILLEGAL_INPUT);

	TRY
	{
		nx = hdrl_image_get_size_x(hdrl_img);
		ny = hdrl_image_get_size_y(hdrl_img);
		CHECK_ERROR_STATE();

		img = hdrl_image_get_image(hdrl_img);
		mask = hdrl_image_get_mask(hdrl_img);

		pimg = cpl_image_get_data_double(img);
		pmask = cpl_mask_get_data(mask);

		/* go through the columns */
		for (int i = ERIS_IFU_DETECTOR_BP_BORDER;
				i < ERIS_IFU_DETECTOR_SIZE_X-ERIS_IFU_DETECTOR_BP_BORDER ;
				i++)
		{
			col_nr = 0;
			/* initialize the buffer variables for each column */
			column = (double*)cpl_calloc(ny, sizeof(double*));
			sig    = (double*)cpl_calloc(ny, sizeof(double*));
			dat    = (double*)cpl_calloc(ny, sizeof(double*));

			/*select only non-NAN values of one column*/
			for(int j = 0; j < ny; j++) {
				if (pmask[i + j*nx] != BAD_PIX) {
					column[j] = pimg[i + j*nx];
					col_nr++;
				}
			}

			if (col_nr < 10) {
				/* column %d has almost only blank pixels
				 * and is set to blank", i+1 */
				for(int j = 0; j < ny; j++) {
					pmask[i + j*nx] = BAD_PIX;
				}
			} else {
				/* sort the data, clip off the extremes, determine the noise
				 * and get the range for the valid data. It is assumed here
				 * that most pixels are o.k.
				 */
				eris_ifu_flat_pixel_qsort(column, col_nr);

				for (int j = 0.1*col_nr + 1; j <= 0.9*col_nr; j++) {
					sum  += column[j];
					sum2 += column[j] * column[j];
					npix++;
				}

				if (npix <= 1) {
					noise = sigma * 1000.;
				} else {
					mean   = sum/(double)npix;
					noise  = sqrt((sum2 - sum*mean)/(double)(npix -1));
					noise *= sigma;
				}

				/* determine sinfo_median if colnum is odd, sinfo_median will be
				 * the colnum/2 th value, otherwise sinfo_median is the mean of
				 * colnum/2-1 th and colnum/2 th value.
				 */

				if ((col_nr % 2 == 1) || (col_nr == 0)) {
					sinfo_median = column[col_nr/2];
				} else {
					sinfo_median = (column[col_nr/2 - 1] + column[col_nr/2])/2.;
				}

				/* now select the pixels for the tilt calculation */
				col_nr = 0 ;
				for (int j = 0; j < ny; j++) {
					if ((pmask[i + j*nx] != BAD_PIX) &&
							(fabs((pimg[i+j*nx])-sinfo_median) <= noise))
					{
						column[col_nr] = pimg[i+j*nx];
						dat[col_nr] = (double)j;
						sig[col_nr] = 1.;
						col_nr++;
					}
				}

				if (col_nr == 0) {
					a = NAN;
					b = NAN;
				} else {
					mwt = 0;
					eris_ifu_my_fit(dat, column, col_nr, sig, mwt, &a,
							&b, &siga, &sigb, &chi2, &q);
				}

				if ((fabs(b) >= slope) || (fabs(a) >= saturation) ||
						isnan(b) || isnan(a))
				{
					cpl_msg_warning(cpl_func, "linear fit: slope is greater than "
							"limit: %f saturation level is "
							"reached: %f in column number %d ",
							b, a , i+1) ;
				}

				/* subtract fit or sinfo_median from data */
				for (int j = 0; j < ny; j++) {
					if ((pmask[i + j*nx] != BAD_PIX) &&
							(fabs(b) < slope) &&
							(fabs(a) < saturation))
					{
						pimg[i+j*nx] = pimg[i+j*nx] - (a + b*(double)j);
					} else if (pmask[i + j*nx] == BAD_PIX) {
						pimg[i+j*nx] = NAN ;
					} else if (((fabs(b) >= slope) ||
							(fabs(a) >= saturation) ||
							isnan(a) || isnan(b)) &&
							(pmask[i + j*nx] != BAD_PIX))
					{
						pimg[i+j*nx] -= sinfo_median;
					} else {
						cpl_msg_error(cpl_func, " case is not possible! %f %f",
								b, a);
					}
				}
			}
			eris_ifu_free_double_array(&column);
			eris_ifu_free_double_array(&sig);
			eris_ifu_free_double_array(&dat);
		} // end: for i
	}
	CATCH
	{
		CATCH_MSGS();
		err = cpl_error_get_code();
	}

	eris_ifu_free_double_array(&column);
	eris_ifu_free_double_array(&sig);
	eris_ifu_free_double_array(&dat);
	eris_check_error_code("eris_ifu_column_tilt");
	return err;
}

/**
   @name eris_ifu_dist_thresh_image
   @brief simple search for static bad pixels for a flat field or dark frame,
          values below and above the threshold values are set to NAN.

   @param image, low cut pixel value, high cut pixel value
   @return resulting image

 */
cpl_error_code eris_ifu_flat_thresh_mask(hdrl_image* hdrl_img,
		double lo_cut,
		double hi_cut)
{
	int             nx      = 0,
			ny      = 0;
	const double    *pimg   = NULL;
	cpl_image       *img    = NULL;
	cpl_mask        *mask   = NULL;
	cpl_binary      *pmask   = NULL;
	cpl_error_code  err     = CPL_ERROR_NONE;

	cpl_ensure_code(hdrl_img, CPL_ERROR_NULL_INPUT);

	TRY
	{
		nx = hdrl_image_get_size_x(hdrl_img);
		ny = hdrl_image_get_size_y(hdrl_img);
		CHECK_ERROR_STATE();

		img = hdrl_image_get_image(hdrl_img);
		mask = hdrl_image_get_mask(hdrl_img);

		pimg = cpl_image_get_data_double_const(img);
		pmask = cpl_mask_get_data(mask);

		for (int x = ERIS_IFU_DETECTOR_BP_BORDER; x < nx-ERIS_IFU_DETECTOR_BP_BORDER; x++) {
			for (int y = ERIS_IFU_DETECTOR_BP_BORDER; y < ny-ERIS_IFU_DETECTOR_BP_BORDER; y++) {
				if ((pmask[x+nx*y] == GOOD_PIX) &&
						((pimg[x+nx*y] > hi_cut) || (pimg[x+nx*y] < lo_cut)))
				{
					pmask[x+nx*y] = BAD_PIX;
				}
			}
		}
	}
	CATCH
	{
		CATCH_MSGS();
		err = cpl_error_get_code();
	}
	eris_check_error_code("eris_ifu_flat_thresh_mask");
	return err;
}

/**
 @name        sinfo_pixel_qsort
 @brief     (from SINFONI pipeline)
 @memo        Sort an array of pixels by increasing pixelvalue.
 @param    pix_arr        Array to sort.
 @param    npix        Number of pixels in the array.
 @return    void
 @doc

 Optimized implementation of a fast pixel sort. The input array is
 modified.
 */

#define PIX_SWAP(a,b) { double temp=(a); (a)=(b); (b)=temp; }
#define PIX_STACK_SIZE 50

cpl_error_code eris_ifu_flat_pixel_qsort(double *pix_arr, int npix)
{
	int i, ir, j, k, l, j_stack;
	int i_stack[PIX_STACK_SIZE * sizeof(double)];
	double a = 0.;

	cpl_error_code  ret = CPL_ERROR_NONE;

	cpl_ensure_code(pix_arr, CPL_ERROR_NULL_INPUT);

	TRY
	{
		ir = npix;
		l = 1;
		j_stack = 0;
		for (;;) {
			if (ir - l < 7) {
				for (j = l + 1; j <= ir; j++) {
					a = pix_arr[j - 1];
					for (i = j - 1; i >= 1; i--) {
						if (pix_arr[i - 1] <= a)
							break;
						pix_arr[i] = pix_arr[i - 1];
					}
					pix_arr[i] = a;
				}
				if (j_stack == 0)
					break;
				ir = i_stack[j_stack-- - 1];
				l = i_stack[j_stack-- - 1];
			} else {
				k = (l + ir) >> 1;
				PIX_SWAP(pix_arr[k - 1], pix_arr[l])
				if (pix_arr[l] > pix_arr[ir - 1]) {
					PIX_SWAP(pix_arr[l], pix_arr[ir - 1])
				}
				if (pix_arr[l - 1] > pix_arr[ir - 1]) {
					PIX_SWAP(pix_arr[l - 1], pix_arr[ir - 1])
				}
				if (pix_arr[l] > pix_arr[l - 1]) {
					PIX_SWAP(pix_arr[l], pix_arr[l - 1])
				}
				i = l + 1;
				j = ir;
				a = pix_arr[l - 1];
				for (;;) {
					do
						i++;
					while (pix_arr[i - 1] < a);
					do
						j--;
					while (pix_arr[j - 1] > a);
					if (j < i)
						break;
					PIX_SWAP(pix_arr[i - 1], pix_arr[j - 1]);
				}
				pix_arr[l - 1] = pix_arr[j - 1];
				pix_arr[j - 1] = a;
				j_stack += 2;
				if (j_stack > PIX_STACK_SIZE) {
					BRK_WITH_ERROR_MSG(CPL_ERROR_UNSPECIFIED,
							"stack too small : aborting");
				}
				if (ir - i + 1 >= j - l) {
					i_stack[j_stack - 1] = ir;
					i_stack[j_stack - 2] = i;
					ir = j - 1;
				} else {
					i_stack[j_stack - 1] = j - 1;
					i_stack[j_stack - 2] = l;
					l = i;
				}
			}
		}
	}
	CATCH
	{
		CATCH_MSGS();

		ret = cpl_error_get_code();
	}
	eris_check_error_code("eris_ifu_flat_pixel_qsort");
	return ret;
}
#undef PIX_STACK_SIZE
#undef PIX_SWAP

/**
@brief fits a straight line to a set of x, y data points by
       minimizing chi-square.
  @name sinfo_myfit()
  @param x
  @param y set of data points,
  @param ndata number of data points
  @param sig individual standard deviations,
  @param mwt
  @param a
  @param b
  @param siga
  @param sigb
  @param sinfo_chi2  chi square
  @param q     goodness of fit probability
  @return void
  @see Numeric. Rec. page 665
 */
void eris_ifu_my_fit(double x[], double y[], int ndata, double sig[], int mwt,
		double *a, double *b, double *siga, double *sigb, double *chi2,
		double *q)
{
	double   t       = 0.,
			sxoss   = 0.,
			sx      = 0.,
			sy      = 0.,
			st2     = 0.,
			ss      = 0.;

	*b = 0.;                                /*accumulate sums ...*/
	if (mwt) {
		ss = 0.;
		for (int i = 0; i < ndata; i++) {   /*... with weights*/
			double wt = 1./pow(sig[i], 2);
			ss += wt;
			sx += x[i]*wt;
			sy += y[i]*wt;
		}
	} else {
		for (int i = 0; i < ndata; i++) {   /*... or without weights*/
			sx += x[i];
			sy += y[i];
		}
		ss = ndata;
	}
	sxoss = sx/ss;

	if (mwt) {
		for (int i = 0; i < ndata; i++) {
			t = (x[i] - sxoss)/sig[i];
			st2 += t*t;
			*b += t*y[i]/sig[i];
		}
	} else {
		for (int i = 0; i < ndata; i++) {
			t = x[i] - sxoss;
			st2 += t*t;
			*b += t*y[i];
		}
	}

	*b /= st2;
	*a = (sy - sx*(*b))/ss;
	*siga = sqrt ((1.0 + sx*sx/(ss*st2))/ss);
	*sigb = sqrt (1.0/st2);
	*chi2 = 0.0;    /*calculate chi-square*/
	if (mwt == 0) {
		for (int i = 0 ; i < ndata ; i++) {
			*chi2 += pow((y[i] - (*a) - (*b)*x[i]), 2);
		}
		*q = 1.;

		/* for unweighted data evaluate typical sig using chi2, and adjust
		 * the standard deviation
		 */
		double sigdat = sqrt ((*chi2)/(ndata - 2));
		*siga *= sigdat;
		*sigb *= sigdat;
	} else {
		for (int i = 0; i < ndata; i++) {
			*chi2 += pow(((y[i] - (*a) - (*b) * x[i])/sig[i]),2);
		}
		*q = 1.; /* delete rest of lines. q is not a good value */
	}
	eris_check_error_code("eris_ifu_my_fit");
	return;
}

/**
@brief computes the mean and standard deviation of a given
       rectangle on an image by leaving the extreme intensity values.
   @name  sinfo_new_image_stats_on_rectangle()
   @param im: flatfield image to search for bad pix
   @param loReject,
   @param hiReject: percentage (0...100) of extrem values
                                            that should not be considered
   @param llx,
   @param lly: lower left pixel position of rectangle
   @param urx,
   @param ury: upper right pixel position of rectangle
   @return data structure giving the mean and standard deviation

 */
cpl_error_code eris_ifu_flat_stats_rectangle(const hdrl_image *hdrl_img,
		double     loReject,
		double     hiReject,
		int       llx,
		int       lly,
		int       urx,
		int       ury,
		double    *cleanmean,
		double    *cleanstdev)
{
	cpl_error_code  ret         = CPL_ERROR_NONE;
	const cpl_image *img        = NULL;
	int             n           = 0,
			npix        = 0,
			lo_n        = 0,
			hi_n        = 0,
			nx          = 0,
			ny          = 0;
	const double    *pimg       = NULL;
	double          *pix_array  = NULL,
			pix_sum     = 0.,
			sqr_sum     = 0.;

	cpl_ensure_code(hdrl_img, CPL_ERROR_NULL_INPUT);

	nx = hdrl_image_get_size_x(hdrl_img);
	ny = hdrl_image_get_size_y(hdrl_img);

	cpl_ensure_code(loReject+hiReject < 100., CPL_ERROR_ILLEGAL_INPUT);
	cpl_ensure_code(loReject >= 0., CPL_ERROR_ILLEGAL_INPUT);
	cpl_ensure_code(loReject < 100., CPL_ERROR_ILLEGAL_INPUT);
	cpl_ensure_code(hiReject >= 0., CPL_ERROR_ILLEGAL_INPUT);
	cpl_ensure_code(hiReject < 100., CPL_ERROR_ILLEGAL_INPUT);
	cpl_ensure_code((llx >= 0) && (lly >= 0), CPL_ERROR_ILLEGAL_INPUT);
	cpl_ensure_code((urx >= 0) && (ury >= 0), CPL_ERROR_ILLEGAL_INPUT);
	cpl_ensure_code((llx < nx) && (lly < ny), CPL_ERROR_ILLEGAL_INPUT);
	cpl_ensure_code((urx < nx) && (ury < ny), CPL_ERROR_ILLEGAL_INPUT);
	cpl_ensure_code((ury > lly) && (urx > llx), CPL_ERROR_ILLEGAL_INPUT);

	TRY
	{
		/* allocate memory */
		npix = (urx-llx+1) * (ury-lly+1);
		pix_array = (double*)cpl_calloc(npix, sizeof(double));

		/* go through the rectangle and copy the pixel values into an array */
		img = hdrl_image_get_image_const(hdrl_img);
		pimg = cpl_image_get_data_double_const(img);

		for (int row = lly; row <= ury; row++) {
			for (int col = llx; col <= urx; col++) {
				if (!isnan(pimg[col + row*nx])) {
					pix_array[n] = pimg[col + row*nx];
					n++;
				}
			}
		}

		npix = n;

		/* determining the clean mean is already done in the recipes */
		eris_ifu_flat_clean_mean(pix_array, npix, loReject, hiReject,
				cleanmean);

		/* now the clean standard deviation must be calculated */
		/* initialize sums */
		lo_n = (int)(loReject / 100. * (double)npix);
		hi_n = (int)(hiReject / 100. * (double)npix);
		pix_sum = 0.;
		sqr_sum = 0.;
		n = 0;
		for (int i = lo_n; i <= npix - hi_n; i++) {
			pix_sum += (double)pix_array[i];
			sqr_sum += ((double)pix_array[i] * (double)pix_array[i]);
			n++;
		}


		if (n == 0) {
			BRK_WITH_ERROR_MSG(CPL_ERROR_UNSPECIFIED,
					"number of clean pixels is zero!");
		}

		pix_sum /= (double)n;
		sqr_sum /= (double)n;
		*cleanstdev = sqrt(sqr_sum - pix_sum*pix_sum);
	}
	CATCH
	{
		CATCH_MSGS();

		ret = cpl_error_get_code();

		*cleanmean  = -DBL_MAX;
		*cleanstdev = -DBL_MAX;
	}

	cpl_free(pix_array);
	eris_check_error_code("eris_ifu_flat_stats_rectangle");
	return ret;
}

/**
@brief his routine computes the clean mean of a given data
   @name sinfo_new_clean_mean()
   @param array: data array to average
   @param n_elements: number of elements of the data array
   @param throwaway_low: percentage of low value elements to be
                         thrown away before averaging
   @param throwaway_high: percentage of high value elements to be
                          thrown away before averaging
   @return the clean mean of a data array
                        FLT_MAX in case of error
   @doc
   this routine computes the clean mean of a given data array that means the
   array is first sorted and a given percentage of the lowest and the
   highest values is not considered for averaging
 */
cpl_error_code eris_ifu_flat_clean_mean(double *array,
		int    n_elements,
		double throwaway_low,
		double throwaway_high,
		double *cleanmean)
{
	cpl_error_code  ret     = CPL_ERROR_NONE;
	int             n       = 0,
			lo_n    = 0,
			hi_n    = 0;
	double          sum     = 0.;

	cpl_ensure_code(array, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(n_elements > 0, CPL_ERROR_ILLEGAL_INPUT);
	cpl_ensure_code(throwaway_low > 0., CPL_ERROR_ILLEGAL_INPUT);
	cpl_ensure_code(throwaway_high > 0., CPL_ERROR_ILLEGAL_INPUT);
	cpl_ensure_code(throwaway_low + throwaway_high < 100.,
			CPL_ERROR_ILLEGAL_INPUT);

	TRY
	{
		lo_n = (int)(throwaway_low * (double)n_elements / 100.);
		hi_n = (int)(throwaway_high * (double)n_elements / 100.);

		/* sort the array */
		eris_ifu_flat_pixel_qsort(array, n_elements);

		for (int i = lo_n; i < n_elements - hi_n; i++) {
			if (!isnan(array[i])) {
				sum += array[i];
				n++;
			}
		}
		if (n == 0) {
			BRK_WITH_ERROR_MSG(CPL_ERROR_UNSPECIFIED,
					"zero values!");
		} else {
			*cleanmean = sum/(double)n;
		}
	}
	CATCH
	{
		CATCH_MSGS();

		ret = cpl_error_get_code();

		*cleanmean  = -DBL_MAX;
	}
	eris_check_error_code("eris_ifu_flat_clean_mean");
	return ret;
}

/**
   @name sinfo_new_median_image
   @brief median filter
   @param image, a sinfo_median threshold parameter
   @return resulting image
   @doc
   median filter, calculates the sinfo_median for an image
   by using the 8 closest pixels of every pixel.
   The values in the output image are determined according
   to the values of the input parameter.
   If fmedian = 0: always replace by sinfo_median
   if fmedian < 0: replace by sinfo_median if |pixel - median| > -fmedian
   if fmedian > 0: replace by sinfo_median (fmedian as a factor of
                   the square root of the median itself)
   if |pixel - median| >= fmedian * sqrt ( median )
   This can be used to consider photon noise.
   This considers a dependence of the differences on the pixel values
   themselves.
   @note it is assumed that most of the 8 nearest neighbor pixels
                        are not bad pixels!
                        blank pixels are not replaced!

 */
hdrl_image* eris_ifu_flat_median_image(const hdrl_image *hdrl_img_in,
		double fmedian)
{
	hdrl_image          *hdrl_img_out   = NULL;
	const cpl_image     *img_in         = NULL;
	cpl_image           *img_out        = NULL;
	const cpl_mask      *mask_in        = NULL;
	cpl_mask            *mask_out       = NULL;
	const cpl_binary    *pmask_in       = NULL;
	cpl_binary          *pmask_out      = NULL;
	int                 *position       = NULL,
			nposition       = 0,
			im_size         = 0,
			nx              = 0,
			ny              = 0;
	double              *value          = NULL,
			my_med          = 0,
			*pimg_out       = NULL;
	const double        *pimg_in        = NULL;

	cpl_ensure(hdrl_img_in, CPL_ERROR_NULL_INPUT, NULL);

	TRY
	{
		nx = hdrl_image_get_size_x(hdrl_img_in);
		ny = hdrl_image_get_size_y(hdrl_img_in);
		im_size = nx*ny;
		CHECK_ERROR_STATE();

		img_in = hdrl_image_get_image_const(hdrl_img_in);
		mask_in = hdrl_image_get_mask_const(hdrl_img_in);
		pimg_in = cpl_image_get_data_double_const(img_in);
		pmask_in = cpl_mask_get_data_const(mask_in);

		hdrl_img_out = hdrl_image_duplicate(hdrl_img_in);
		img_out = hdrl_image_get_image(hdrl_img_out);
		mask_out = hdrl_image_get_mask(hdrl_img_out);
		pimg_out = cpl_image_get_data_double(img_out);
		pmask_out = cpl_mask_get_data(mask_out);

		/* go through all pixels */
		for (int i = 0; i < im_size; i++) {
			/* blank pixels are not replaced */
			if (pmask_in[i] == BAD_PIX) {
				continue;
			}

			/* initialize the buffer variables for the 8 nearest neighbors */
			value = (double*)cpl_calloc(8, sizeof(double*));
			position = (int*)cpl_calloc(8, sizeof(int*));

			/* determine the pixel position of the 8 nearest neighbors
			 */
			position[0] = i + nx - 1 ; /* upper left  */
			position[1] = i + nx     ; /* upper       */
			position[2] = i + nx + 1 ; /* upper right */
			position[3] = i + 1      ; /* right       */
			position[4] = i - nx + 1 ; /* lower right */
			position[5] = i - nx     ; /* lower       */
			position[6] = i - nx - 1 ; /* lower left  */
			position[7] = i - 1      ; /* left        */

			/* determine the positions of the image margins, top positions are
               changed to low positions and vice versa. Right positions are
               changed to left positions and vice versa.
			 */
			if ((i >= 0) && (i < nx)) {
				/* bottom line */
				position[4] += 2*nx;
				position[5] += 2*nx;
				position[6] += 2*nx;
			} else if ((i >= ((int)nx*ny - nx)) && (i < (int) nx*ny)) {
				/* top line */
				position[0] -= 2*nx;
				position[1] -= 2*nx;
				position[2] -= 2*nx;
			} else if (i % nx == 0) {
				/* left side */
				position[0] += 2;
				position[6] += 2;
				position[7] += 2;
			} else if ((i % nx) == (nx - 1)) {
				/* right side */
				position[2] -= 2;
				position[3] -= 2;
				position[4] -= 2;
			}

			/* read the pixel values of the neighboring pixels,
			 * blanks are not considered
			 */
			nposition = 8;
			int n = 0;
			for (int j = 0; j < nposition; j++) {
				if((position[j] > -1) && (position[j] < im_size)) {
					if (pmask_in[position[j]] == GOOD_PIX) {
						value[n] = pimg_in[position[j]];
						n++;
					}
				}
			}
			nposition = n;

			if (nposition <= 1)  /* almost all neighbors are blank */
			{
				pmask_out[i] = BAD_PIX;
				eris_ifu_free_double_array(&value);
				eris_ifu_free_int_array(&position);
				continue;
			}

			/* sort the values and determine the sinfo_median */
			BRK_IF_ERROR(
					eris_ifu_flat_pixel_qsort(value, nposition));

			if (nposition % 2 == 1) {
				my_med = value[nposition/2];
			} else {
				my_med = (value [nposition/2 - 1] + value [nposition/2]) / 2.;
			}

			/* replace the pixel value by the sinfo_median on conditions:
			 * fmedian = 0: always replace with sinfo_median.
			 * fmedian < 0: interpret as absolute condition:
			 *              if |pixel - sinfo_median| > -fmedian
			 *              replace with sinfo_median.
			 * fmedian > 0: replace by sinfo_median (fmedian as a factor of
			 *              the square root of the sinfo_median itself)
			 *              if |pixel - sinfo_median| >= fmedian *
                                                         sqrt ( sinfo_median )
			 *              considers a dependence on the pixel value.
			 *              This can be used to consider photon noise.
			 */
			if (fmedian == 0) {
				pimg_out[i] = my_med;
			} else if ((fmedian < 0) && (fabs(my_med-pimg_in[i]) >= -fmedian)) {
				pimg_out[i] = my_med;
			} else if ((fmedian > 0) &&
					(fabs(my_med-pimg_in[i]) >=
							fmedian*sqrt(fabs(my_med))))
			{
				pimg_out[i] = my_med;
			} else {
				eris_ifu_free_double_array(&value);
				eris_ifu_free_int_array(&position);
				continue;
			}
			eris_ifu_free_double_array(&value);
			eris_ifu_free_int_array(&position);
		}
	}
	CATCH
	{
		CATCH_MSGS();
		eris_ifu_free_hdrl_image(&hdrl_img_out);
	}

	eris_ifu_free_double_array(&value);
	eris_ifu_free_int_array(&position);
	eris_check_error_code("eris_ifu_flat_median_image");
	return hdrl_img_out;
}

/**
    @name eris_ifu_sinfo_compare_images

    @brief if a pixel value of one image (im1) equals the pixel value
          of the other image keep the pixel value of the original image
          otherwise set the pixel as bad

    @param hdrl_img1     1st image
    @param hdrl_img2     2nd image
    @param hdrl_img_orig original image

    @return error in case of error
 */
cpl_error_code eris_ifu_sinfo_compare_images(const hdrl_image *hdrl_img1,
		const hdrl_image *hdrl_img2,
		hdrl_image *hdrl_img_orig)
{
	const cpl_image     *img1       = NULL,
			*img2       = NULL;
	const double        *pimg1      = NULL;
	const double        *pimg2      = NULL;
	const cpl_mask      *mask_img1  = NULL,
			*mask_img2  = NULL;
	cpl_mask            *mask_orig  = NULL;
	const cpl_binary    *pmask_img1 = NULL,
			*pmask_img2 = NULL;
	cpl_binary          *pmask_orig = NULL;
	cpl_error_code      err         = CPL_ERROR_NONE;
	int                 nx1         = 0,
			ny1         = 0,
			nx2         = 0,
			ny2         = 0;

	cpl_ensure_code(hdrl_img1, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(hdrl_img2, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(hdrl_img_orig, CPL_ERROR_NULL_INPUT);

	nx1 = hdrl_image_get_size_x(hdrl_img1);
	ny1 = hdrl_image_get_size_y(hdrl_img1);
	nx2 = hdrl_image_get_size_x(hdrl_img2);
	ny2 = hdrl_image_get_size_y(hdrl_img2);

	cpl_ensure_code(nx1 == nx2, CPL_ERROR_ILLEGAL_INPUT);
	cpl_ensure_code(ny1 == ny2, CPL_ERROR_ILLEGAL_INPUT);

	TRY
	{
		img1 = hdrl_image_get_image_const(hdrl_img1);
		img2 = hdrl_image_get_image_const(hdrl_img2);
		pimg1=cpl_image_get_data_double_const(img1);
		pimg2=cpl_image_get_data_double_const(img2);

		mask_img1 = hdrl_image_get_mask_const(hdrl_img1);
		mask_img2 = hdrl_image_get_mask_const(hdrl_img2);
		mask_orig = hdrl_image_get_mask(hdrl_img_orig);
		pmask_img1 = cpl_mask_get_data_const(mask_img1);
		pmask_img2 = cpl_mask_get_data_const(mask_img2);
		pmask_orig = cpl_mask_get_data(mask_orig);

		for (int i = 0; i < nx1*ny1; i++) {
			if ((pmask_img1[i] == BAD_PIX) ||
					(pmask_img2[i] == BAD_PIX) ||
					(pimg1[i] != pimg2[i]))
			{
				pmask_orig[i] = BAD_PIX;
			}
		}
	}
	CATCH
	{
		CATCH_MSGS();
		err = cpl_error_get_code();
	}
	eris_check_error_code("eris_ifu_sinfo_compare_images");
	return err;
}

/**
 * @brief eris_ifu_flat_fast_bpm
 * @param master_flat
 * @param parlist
 * @return
 *
 * Calculate a simple BPM for mode=fast
 * sets to bad left or right offset
 */
cpl_error_code eris_ifu_flat_fast_bpm(hdrl_image *master_flat,
		const cpl_parameterlist *parlist,
		int productDepth)
{
	int             llx             = 0,
			lly             = 0,
			urx             = 0,
			ury             = 0;
	cpl_error_code  err             = CPL_ERROR_NONE;
	cpl_mask        *bpm            = NULL;
	cpl_image       *img            = NULL;
	const cpl_image *data           = NULL;
	cpl_binary      *pbpm           = NULL;

	cpl_ensure_code(master_flat, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(parlist, CPL_ERROR_NULL_INPUT);

	TRY
	{
		// AA ToDo: implement as well right border handling

		bpm = hdrl_image_get_mask(master_flat);
		pbpm = cpl_mask_get_data(bpm);
		data = hdrl_image_get_image_const(master_flat);

		// check left side
		llx = 1,
				lly = 1,
				urx = ERIS_IFU_DETECTOR_BP_BORDER+SLITLET_WIDTH,
				ury = ERIS_IFU_DETECTOR_SIZE_Y;

		BRK_IF_NULL(
				img = cpl_image_extract(data, llx, lly, urx, ury));
		//eris_ifu_save_image_dbg(img, "sss_in.fits", CPL_IO_CREATE, NULL);

		int             nr_values           = 21,   //odd
				center_y            = 0,
				height              = 96;   // even
		cpl_vector      *profile_x          = NULL;
		double          left_edge_pos = 0.,
				right_edge_pos = 0.,
				*pprofile_x = NULL;

		//        for (int i = ERIS_IFU_DETECTOR_BP_BORDER;
		//                 y < ERIS_IFU_DETECTOR_SIZE_Y-ERIS_IFU_DETECTOR_BP_BORDER;
		//                 y++)
		//        {
		for (int i = 0; i < nr_values; i++) {
			/* collapse image in y and convert to vector */
			center_y = ERIS_IFU_DETECTOR_BP_BORDER + i*height + height/2;
			profile_x = eris_ifu_calc_centers_collapse_chunk(
					img, center_y, height);
			//            double *pimg = (double*)cpl_image_get_data_const(img);
			//eris_ifu_save_vector_dbg(profile_x, "sss_profx.fits", CPL_IO_CREATE);

			pprofile_x = cpl_vector_get_data(profile_x);

			cpl_image *imgg=NULL;
			imgg = cpl_image_wrap_double(cpl_vector_get_size(profile_x),1,
					pprofile_x);

			imgg = cpl_image_wrap_double(cpl_image_get_size_x(img), 1,
					pprofile_x);

			cpl_error_code status = eris_ifu_slitpos_gauss(imgg,
					&left_edge_pos,
					&right_edge_pos,
					0,
					productDepth);

			if (status == CPL_ERROR_EOL) {
				RECOVER();
				continue;
			} else if (status != CPL_ERROR_NONE) {
				ERIS_IFU_TRY_EXIT();
			}

			// erase bmp left
			for (int y = center_y-height/2; y < center_y+height/2; y++) {
				for (int x = 0; x <= (int)(left_edge_pos+.5); x++) {
                    pbpm[x+y*ERIS_IFU_DETECTOR_SIZE_X] = BAD_PIX;
				} //end: for(x)
			} //end: for(y)
			//eris_ifu_save_hdrl_image_dbg(master_flat, "zzz", TRUE);
			//CHECK_ERROR_STATE();
		} // end: for(i)

		// AA todo: fix
		for (int y = 2020; y < 2045; y++) {
			for (int x = 0; x <= (int)(left_edge_pos+.5); x++) {
                pbpm[x+y*ERIS_IFU_DETECTOR_SIZE_X] = BAD_PIX;
			} //end: for(x)
		} //end: for(y)

		CHECK_ERROR_STATE();
	}
	CATCH
	{
		CATCH_MSGS();
		err = cpl_error_get_code();
	}
	eris_check_error_code("eris_ifu_flat_fast_bpm");
	return err;
}

/* ---- Data collector ---- */
cpl_error_code eris_ifu_flat_data_init(struct structFlatData *self, int blocksize) {
	cpl_error_code err = CPL_ERROR_NONE;

	cpl_ensure_code(blocksize > 0, CPL_ERROR_ILLEGAL_INPUT);

	TRY
	{
		self->cnt = 0;
		self->blocksize = blocksize;
		int nr = ERIS_IFU_DETECTOR_SIZE_Y/blocksize; // is 51 for blocksize = 40
		self->ry = cpl_vector_new(nr);
		self->borderAleft = cpl_vector_new(nr);
		self->borderAright = cpl_vector_new(nr);
		self->borderBleft = cpl_vector_new(nr);
		self->borderBright = cpl_vector_new(nr);
		self->borderCleft = cpl_vector_new(nr);
		self->borderCright = cpl_vector_new(nr);
		self->borderDleft = cpl_vector_new(nr);
		self->borderDright = cpl_vector_new(nr);
	}
	CATCH
	{
		CATCH_MSGS();
		err = cpl_error_get_code();
	}
	eris_check_error_code("eris_ifu_flat_data_init");
	return err;
}

cpl_error_code eris_ifu_flat_data_setBorders(struct structFlatData *self, int ry,
		int leftEdgeA, int rightEdgeA,
		int leftEdgeB, int rightEdgeB,
		int leftEdgeC, int rightEdgeC,
		int leftEdgeD, int rightEdgeD)
{
	cpl_error_code err = CPL_ERROR_NONE;

	TRY
	{
		cpl_vector_set(self->ry, self->cnt, ry);
		cpl_vector_set(self->borderAleft, self->cnt, leftEdgeA);
		cpl_vector_set(self->borderAright, self->cnt, rightEdgeA);
		cpl_vector_set(self->borderBleft, self->cnt, leftEdgeB);
		cpl_vector_set(self->borderBright, self->cnt, rightEdgeB);
		cpl_vector_set(self->borderCleft, self->cnt, leftEdgeC);
		cpl_vector_set(self->borderCright, self->cnt, rightEdgeC);
		cpl_vector_set(self->borderDleft, self->cnt, leftEdgeD);
		cpl_vector_set(self->borderDright, self->cnt, rightEdgeD);
		self->cnt++;
	}
	CATCH
	{
		CATCH_MSGS();
		err = cpl_error_get_code();
	}
	eris_check_error_code("eris_ifu_flat_data_setBorders");
	return err;
}

cpl_vector* eris_ifu_flat_data_getBorderInterpolated(struct structFlatData *self,
		int row)
{
	cpl_vector  *borders    = NULL,
			*fit_par    = NULL;
	int         degree      = 2,
			rowx        = 0;
	double      *pfit_par   = NULL,
			*pborders    = NULL;
	TRY
	{
		rowx = row + self->blocksize/2.;

		borders = cpl_vector_new(8);
		pborders = cpl_vector_get_data(borders);

		fit_par = eris_ifu_polyfit_1d(self->ry, self->borderAleft, degree);
		pfit_par = cpl_vector_get_data(fit_par);
		pborders[0] = (int)(pfit_par[0] + pfit_par[1] * rowx + pfit_par[2] * pow(rowx, 2) + 0.5);
		eris_ifu_free_vector(&fit_par);

		fit_par = eris_ifu_polyfit_1d(self->ry, self->borderAright, degree);
		pfit_par = cpl_vector_get_data(fit_par);
		pborders[1] = (int)(pfit_par[0] + pfit_par[1] * rowx + pfit_par[2] * pow(rowx, 2) + 0.5);
		eris_ifu_free_vector(&fit_par);

		fit_par = eris_ifu_polyfit_1d(self->ry, self->borderBleft, degree);
		pfit_par = cpl_vector_get_data(fit_par);
		pborders[2] = (int)(pfit_par[0] + pfit_par[1] * rowx + pfit_par[2] * pow(rowx, 2) + 0.5);
		eris_ifu_free_vector(&fit_par);

		fit_par = eris_ifu_polyfit_1d(self->ry, self->borderBright, degree);
		pfit_par = cpl_vector_get_data(fit_par);
		pborders[3] = (int)(pfit_par[0] + pfit_par[1] * rowx + pfit_par[2] * pow(rowx, 2) + 0.5);
		eris_ifu_free_vector(&fit_par);

		fit_par = eris_ifu_polyfit_1d(self->ry, self->borderCleft, degree);
		pfit_par = cpl_vector_get_data(fit_par);
		pborders[4] = (int)(pfit_par[0] + pfit_par[1] * rowx + pfit_par[2] * pow(rowx, 2) + 0.5);
		eris_ifu_free_vector(&fit_par);

		fit_par = eris_ifu_polyfit_1d(self->ry, self->borderCright, degree);
		pfit_par = cpl_vector_get_data(fit_par);
		pborders[5] = (int)(pfit_par[0] + pfit_par[1] * rowx + pfit_par[2] * pow(rowx, 2) + 0.5);
		eris_ifu_free_vector(&fit_par);

		fit_par = eris_ifu_polyfit_1d(self->ry, self->borderDleft, degree);
		pfit_par = cpl_vector_get_data(fit_par);
		pborders[6] = (int)(pfit_par[0] + pfit_par[1] * rowx + pfit_par[2] * pow(rowx, 2) + 0.5);
		eris_ifu_free_vector(&fit_par);

		fit_par = eris_ifu_polyfit_1d(self->ry, self->borderDright, degree);
		pfit_par = cpl_vector_get_data(fit_par);
		pborders[7] = (int)(pfit_par[0] + pfit_par[1] * rowx + pfit_par[2] * pow(rowx, 2) + 0.5);
		eris_ifu_free_vector(&fit_par);
	}
	CATCH
	{
		CATCH_MSGS();
		eris_ifu_free_vector(&borders);
	}
	eris_check_error_code("eris_ifu_flat_data_getBorderInterpolated");
	return borders;
}
