/* $Id: eris_ifu_utils.h,v 1.15 2013-07-22 13:38:01 jtaylor Exp $
 *
 * This file is part of the ERIS Pipeline
 * Copyright (C) 2002,2003 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/*
 * $Author: jtaylor $
 * $Date: 2013-07-22 13:38:01 $
 * $Revision: 1.15 $
 * $Name: not supported by cvs2svn $
 */

#ifndef ERIS_IFU_UTILS_H
#define ERIS_IFU_UTILS_H

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/

#include <cpl.h>
#include <hdrl.h>
#include "eris_ifu_constants.h"
#include "eris_ifu_vector.h"
#include "eris_ifu_dfs.h"

#define MAX_STR_SIZE    512
#define FILE_NAME_SZ 256

enum hdrl_t {
    eris_hdrl_data,
    eris_hdrl_error,
    eris_hdrl_badpix};

/*-----------------------------------------------------------------------------
                                       Prototypes
 -----------------------------------------------------------------------------*/

/* ---------------------- idl-functions ------------------------------------- */
/**
    @brief IDL-like comparative operators
*/
enum idl_rel_ops {eq, ne, ge, gt, le, lt};

cpl_vector*     eris_ifu_idl_where(const cpl_vector   *data,
                                   double val,
                                   int    op);
cpl_vector*     eris_ifu_idl_values_at_indices(const cpl_vector *data,
                                               const cpl_vector *indices);

cpl_error_code  eris_ifu_cut_endings(cpl_vector** vec,
                                     int *begin,
                                     int *end,
                                     int cut);

/* --------- FREE helper functions --------- */
void eris_ifu_free_hdrl_image(hdrl_image **item);
void eris_ifu_free_hdrl_imagelist(hdrl_imagelist **item);
void eris_ifu_free_image(cpl_image **item);
void eris_ifu_free_imagelist(cpl_imagelist **item);
void eris_ifu_free_mask(cpl_mask **item);
void eris_ifu_free_matrix(cpl_matrix **item);
void eris_ifu_free_propertylist(cpl_propertylist **item);
void eris_ifu_free_vector(cpl_vector **item);
void eris_ifu_free_ifu_vector(eris_ifu_vector **item);
void eris_ifu_free_string(char **item);
void eris_ifu_free_frame(cpl_frame **item);
void eris_ifu_free_frameset(cpl_frameset **item);
void eris_ifu_free_hdrl_parameter(hdrl_parameter **item);
void eris_ifu_free_parameter(cpl_parameter **item);
void eris_ifu_free_parameterlist(cpl_parameterlist **item);
void eris_ifu_free_table(cpl_table **item);
void eris_ifu_free_double_array(double **item);
void eris_ifu_free_float_array(float **item);
void eris_ifu_free_int_array(int **item);
void eris_ifu_free_polynomial(cpl_polynomial **item);
void eris_ifu_free_bivector(cpl_bivector **item);
void eris_ifu_free_apertures(cpl_apertures **item);

/* --------- SAVE helper functions --------- */
cpl_error_code eris_ifu_save_vector_dbg(const cpl_vector *vec,
                                            const char* filename, int create,
                                            const cpl_propertylist *pl);
cpl_error_code eris_ifu_save_ifu_vector_dbg(const eris_ifu_vector *vec,
                                            const char* filename, int create);
cpl_error_code eris_ifu_save_bivector_dbg(const cpl_bivector *bivec,
                                            const char* filename, int col, int create);
cpl_error_code eris_ifu_save_image_dbg(const cpl_image *img,
                                            const char* filename, int create, const cpl_propertylist *pl);
cpl_error_code eris_ifu_save_table_dbg(const cpl_table *tbl,
                                            const char* filename, int create);
cpl_error_code eris_ifu_save_mask_dbg(const cpl_mask *mask,
                                            const char* filename, int create, const cpl_propertylist *pl);
cpl_error_code eris_ifu_save_imagelist_dbg(const cpl_imagelist *imglist,
                                            const char* filename, int create);
cpl_error_code eris_ifu_save_cpl_image_dbg(const cpl_image *img,
                                           const char* name, int singlefile,
                                           const cpl_propertylist *pl);
cpl_error_code eris_ifu_save_cpl_imagelist_dbg(const cpl_imagelist *imglist,
                                            const char* name);
cpl_error_code eris_ifu_save_hdrl_image_dbg(const hdrl_image *hdrl_img,
                                            const char* filename, int singlefile,
                                            const cpl_propertylist *pl);
cpl_error_code eris_ifu_save_hdrl_imagelist_dbg(const hdrl_imagelist *imgList,
                                            const char* filename, int singlefile);

cpl_error_code eris_ifu_save_image(cpl_frameset *fs,
                                   const cpl_propertylist *plist,
                                   const cpl_parameterlist *parlist,
                                   const char *recipe,
                                   const char *procatg,
                                   const char *filename,
                                   cpl_type type,
                                   const cpl_image *img);

cpl_error_code eris_ifu_save_imagelist(cpl_frameset *fs,
                                       cpl_frame *inherit,
                                       const cpl_propertylist *plist,
                                       const cpl_parameterlist *parlist,
                                       const char *recipe,
                                       const char *procatg,
                                       const char *filename,
                                       cpl_type type,
                                       const cpl_imagelist *imglist);

cpl_error_code eris_ifu_hdrl_image_reject_mask(hdrl_image *img,
                                               const cpl_mask *mask);

/* --------- LOAD helper functions --------- */

// agudo: commented out, should use eris_ifu_load_exposure_file()/_frame()
//hdrl_image* eris_ifu_load_image_ctg(cpl_frameset *frameset,
//                                    const char *category,
//                                    int ext);

//hdrl_image* eris_ifu_load_image(const char *filename,
//                                cpl_type im_type,
//                                int pnum,
//                                int xtnum);

cpl_mask* eris_ifu_load_badpixel_mask(const cpl_frameset *frameset,
                                      const char         *category,
                                      int                ext);

/* --------- OTHER helper functions --------- */
hdrl_imagelist* eris_ifu_get_hdrlimagelist_by_tag(cpl_frameset *frameset,
                                                  const char *tag,
                                                  int exposureCorrectionMode);

cpl_frameset* eris_ifu_get_frameset_by_tag(const cpl_frameset *frameset,
                                           const char *tag);

cpl_error_code eris_ifu_parameterlist_append_list(cpl_parameterlist *p1,
                                                  const cpl_parameterlist *p2);

cpl_error_code eris_ifu_file_exists(const char *filename);

cpl_imagelist* eris_ifu_hdrl_get_imagelist(const hdrl_imagelist *hdrl_imglist,
                                           enum hdrl_t type);

char* eris_ifu_get_lampString(int lampStatus);

const char* eris_ifu_get_bandString(ifsBand band);

const char* eris_ifu_get_instrumentString(ifsInstrument instrument);

double eris_ifu_get_band_resolution(ifsBand band);

const char* eris_ifu_get_preopticsScaleString(ifsPreopticsScale scale);

cpl_image* eris_ifu_image_create_window(const cpl_image* img);

cpl_mask* eris_ifu_mask_create_border(const cpl_mask* mask);

cpl_error_code eris_ifu_mask_nans_in_cube(cpl_imagelist *cube);
cpl_error_code eris_ifu_strehl_flag_nans(hdrl_image** hima);
//cpl_error_code eris_ifu_mask_reset(cpl_mask *mask);

cpl_mask* eris_ifu_quality2bp_mask(const cpl_image *qualityMask,
                                   deqQualityType qualityType);

cpl_imagelist* eris_ifu_interpolatedMask_to_maskZero(cpl_imagelist *bpmCube,
                                                     double threshold);

cpl_error_code eris_ifu_split_hdrl_imagelist(hdrl_imagelist *cube,
                                             cpl_imagelist *dataCube,
                                             cpl_imagelist *errorCube);

cpl_error_code eris_ifu_split3_hdrl_imagelist(hdrl_imagelist *cube,
                                             cpl_imagelist *dataCube,
											 cpl_imagelist *errorCube,
											 cpl_imagelist *qualCube);
cpl_table *
eris_qclog_init(void);


cpl_error_code
eris_qclog_add_int(cpl_table* table,
                 const char* name,
                 const int   value,
                 const char* help);


cpl_error_code
eris_qclog_add_bool(cpl_table* table,
                 const char* name,
                 const char* help);

cpl_error_code
eris_qclog_add_double(cpl_table* table,
                 const char* name,
                 const double   value,
                 const char* help);

cpl_error_code
eris_qclog_add_double_f(cpl_table* table,
                 const char*  key_name,
                 const double value,
                 const char*  key_help);
cpl_error_code
eris_qclog_add_double_format(cpl_table* table,
                 const char* name,
                 const double   value,
                 const char* help);

cpl_error_code
eris_qclog_add_string(cpl_table* table,
                 const char* name,
                 const char*   value,
                 const char* help);

cpl_propertylist* eris_qc_init(void);

cpl_error_code
eris_qc_add_int(cpl_propertylist* table,
                 const char* name,
                 const int   value,
                 const char* help,
                 const char* format);


cpl_error_code
eris_qc_add_bool(cpl_propertylist* table,
                 const char* name,
                 const char  value,
                 const char* help,
                 const char* format);


cpl_error_code
eris_qc_add_float(cpl_propertylist* table,
                 const char* name,
                 const float   value,
                 const char* help,
                 const char* format);


cpl_error_code
eris_qc_add_double(cpl_propertylist* table,
                 const char* name,
                 const double   value,
                 const char* help,
                 const char* format);


cpl_error_code
eris_qc_add_string(cpl_propertylist* table,
                 const char* name,
                 const char*   value,
                 const char* help,
                 const char* format);

cpl_error_code
eris_pfits_put_qc(cpl_propertylist *plist,
                  cpl_table *qclog);

cpl_error_code
eris_ifu_get_badpix_qc_from_mask(const cpl_mask* bp_mask,
		            cpl_propertylist* qc_list,
					const char* prefix);

cpl_error_code
eris_ifu_get_badpix_qc_from_ima(const cpl_image* image,
		            cpl_propertylist* qc_list,
					const char* prefix);
cpl_propertylist*
eris_compute_psf_qc(hdrl_image* hima, const cpl_parameterlist* parlist,
                    const char* context);

cpl_error_code
eris_gaussian_maxpos(const cpl_image * self,
                    double sigma,
                    double  * pxpos,
                    double  * pypos,
                    double  * ppeak,
                    double  * sigma_x,
                    double  * sigma_y);

cpl_mask*
eris_ifu_hima_get_obj_mask(hdrl_image* hima, const cpl_size niter,
                            const double kappa);

cpl_error_code
eris_ifu_mask_nans_in_hdrlimage(hdrl_image** hima);

cpl_error_code
eris_ifu_get_plane_cut_min_max(ifsBand band, cpl_size* zmin, cpl_size* zmax);

cpl_error_code
eris_ifu_cube_trim_nans(const cpl_size zmin,
                        const cpl_size zmax,
                        hdrl_imagelist** iml,
                        cpl_imagelist** bpm,
                        cpl_propertylist* header);

cpl_error_code
eris_ifu_cube_set_values(hdrl_imagelist** iml/*, cpl_imagelist** bpm*/);

cpl_error_code
eris_get_pupil_shift(hdrl_imagelist* iml, const int n, cpl_table** qclog_tbl);

cpl_mask*
eris_ifu_hima_get_obj_mask_percent(hdrl_image* hima, const double percent);

cpl_mask* eris_ifu_mask_from_image(const cpl_image *img_mask);

#endif
