/* $Id$
 *
 * This file is part of the ERIS/NIX Pipeline
 * Copyright (C) 2017 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

 /*
 * $Author$
 * $Date$
 * $Rev$
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/

#include <cpl.h>
#include <string.h>

#include "eris_nix_master_bpm.h"
#include "eris_nix_utils.h"

/*----------------------------------------------------------------------------*/
/**
 * @defgroup eris_nix_master_bpm      Utilities relating to MASTER_BPMs
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Delete a 'master_bpm' struct

  The function deletes the master_bpm struct and its contents. If the 
  pointer is NULL then no action is taken.
 */
/*----------------------------------------------------------------------------*/

void en_master_bpm_delete(master_bpm * target) {
    
    if (target) {
        cpl_image_delete((cpl_image *) target->bpm);
        cpl_free((char *) target->filename);
        cpl_propertylist_delete((cpl_propertylist *) target->plist);
        cpl_free(target);
    }
}

/**@}*/

/**@{*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Load a 'master_bpm' from a frameset.
  @return   Pointer to the master_bpm

  The function loads a master_bpm from a frameset.
 */
/*----------------------------------------------------------------------------*/

master_bpm * en_master_bpm_load_from_frameset(const cpl_frameset * frameset,
                                              const char * tag,
                                              cpl_frameset * used,
                                              const int required) {
    master_bpm         * result = NULL;
    cpl_image          * bpm = NULL;
    cpl_image          * full_bpm = NULL;
    cpl_propertylist   * plist = NULL;

    if (cpl_error_get_code() != CPL_ERROR_NONE) return NULL;
    cpl_ensure(frameset, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(tag, CPL_ERROR_NULL_INPUT, NULL);
 
    /* Look for tagged file in frameset */

    const cpl_frame * target_frame = cpl_frameset_find_const(frameset, tag);
    enu_check(target_frame != NULL, CPL_ERROR_DATA_NOT_FOUND,
              "SoF has no file tagged %s", tag);

    /* Read the data in as a cpl image */

    const char * filename = cpl_frame_get_filename(target_frame);
    bpm = cpl_image_load(filename, CPL_TYPE_INT, 0, 0);
    plist = cpl_propertylist_load(filename, 0);

    /* If the bpm is windowed then create a full-chip version */

    cpl_size nx = 0;
    cpl_size ny = 0;
    int rot = 0;
    cpl_size strx = 0;
    cpl_size stry = 0;
    cpl_size nx_chip = 0;
    cpl_size ny_chip = 0;
    cpl_boolean windowed = 0;
    enu_get_window_info(&nx,
                        &ny,
                        &rot,
                        &strx,
                        &stry,
                        &nx_chip,
                        &ny_chip,
                        &windowed,
                        plist);
    enu_check_error_code("Failed to read bpm window information");

    if (nx != nx_chip || ny != ny_chip) {
        cpl_msg_info(cpl_func, "windowed bpm, extending to cover full chip");
        full_bpm = cpl_image_new(nx_chip,
                                 ny_chip,
                                 CPL_TYPE_INT);
        cpl_image_copy(full_bpm,
                       bpm,
                       strx,
                       stry);
    } else {
        full_bpm = bpm; bpm = NULL;
    }

    result = en_master_bpm_create_from_image(filename, full_bpm, plist);

 cleanup:

    cpl_image_delete(bpm);
    cpl_image_delete(full_bpm);
    cpl_propertylist_delete(plist);

    if (cpl_error_get_code() == CPL_ERROR_NONE) {
        cpl_frame * dup_bpmframe = cpl_frame_duplicate(target_frame);
        cpl_frameset_insert(used, dup_bpmframe);
    } else {
        en_master_bpm_delete(result);
        result = NULL;

        if (!required) {

            /* reset the error code if something has gone wrong but the
               master_flat is not definitely required */

            cpl_msg_warning(cpl_func, "no file tagged %s in SoF", tag);
            cpl_error_reset();
        }
    }
    return result;
}

/**@}*/

/**@{*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Create a new master_bpm struct and initialise the contents
  @return   The new master_bpm struct

  The function returns a pointer to a master_bpm structure with
  components initialised to copies of input parameters. filename and
  plist can be NULL.
 */
/*----------------------------------------------------------------------------*/

master_bpm * en_master_bpm_create(const char * filename, 
                                  const cpl_mask * mask,
                                  const bpm_code flag_code,
                                  const cpl_propertylist * plist) {

    if (cpl_error_get_code() != CPL_ERROR_NONE) return NULL;
    cpl_ensure(mask && flag_code, CPL_ERROR_NULL_INPUT, NULL);

    master_bpm * result = cpl_malloc(sizeof(master_bpm)); 

    cpl_size nx = cpl_mask_get_size_x(mask);
    cpl_size ny = cpl_mask_get_size_y(mask);
    result->bpm = cpl_image_new(nx, ny, CPL_TYPE_INT);
   
    /* Set or clear the specified bit in the master_bpm */

    int * bpm_data = cpl_image_get_data_int((cpl_image *) result->bpm);
    const cpl_binary * mask_data = cpl_mask_get_data_const(mask);

    int bit_mask = (1 << (flag_code-1));
    for (cpl_size i=0; i < nx * ny; i++) {
        if (mask_data[i]) {
            bpm_data[i] |= bit_mask;
        } else {
            bpm_data[i] &= (~bit_mask);
        } 
    }  

    if (filename) {
        result->filename = cpl_strdup(filename);
    } else {
        result->filename = NULL;
    }
    if (plist) {
        result->plist = cpl_propertylist_duplicate(plist);
    } else {
        result->plist = NULL;
    }

    return result;
}

/**@}*/

/**@{*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Create a new master_bpm struct and initialise the contents
            from a CPL_TYPE_INT cpl_image
  @return   The new master_bpm struct

  The function returns a pointer to a master_bpm structure with
  components initialised to copies of the input variables. filename and
  plist can be NULL.
 */
/*----------------------------------------------------------------------------*/

master_bpm * en_master_bpm_create_from_image(const char * filename, 
                                             const cpl_image * image,
                                             const cpl_propertylist * plist) {

    if (cpl_error_get_code() != CPL_ERROR_NONE) return NULL;
    cpl_ensure(image, CPL_ERROR_NULL_INPUT, NULL);

    master_bpm * result = cpl_malloc(sizeof(master_bpm)); 

    result->bpm = cpl_image_duplicate(image);
   
    if (filename) {
        result->filename = cpl_strdup(filename);
    } else {
        result->filename = NULL;
    }
    if (plist) {
        result->plist = cpl_propertylist_duplicate(plist);
    } else {
        result->plist = NULL;
    }

    return result;
}

/**@}*/

/**@{*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Return a cpl_mask derived by applying flag_mask to the MASTER_BPM.
  @return   cpl_mask

  The function constructs a cpl_mask where 
    pixel = (flag_mask & master_bpm->bpm))

  Any returned mask must be deleted with cpl_mask_delete.
  If an error occurs NULL will be returned.
 */
/*----------------------------------------------------------------------------*/

cpl_mask * en_master_bpm_get_mask(const master_bpm * master,
                                  const int flag_mask) {

    if (cpl_error_get_code() != CPL_ERROR_NONE) return NULL;
    cpl_ensure(master, CPL_ERROR_NULL_INPUT, NULL);

    cpl_size nx = cpl_image_get_size_x(master->bpm);
    cpl_size ny = cpl_image_get_size_y(master->bpm);

    cpl_mask * result = cpl_mask_new(nx, ny);
    cpl_binary * result_data = cpl_mask_get_data(result);
    const int * bpm_data = cpl_image_get_data_int_const(master->bpm);

    for (cpl_size j=0; j<ny; j++) {
        for (cpl_size i=0; i<nx; i++) {
            result_data[nx*j+i] = (cpl_binary) ((bpm_data[nx*j+i] & flag_mask) > 0);
        }
    }

    if (cpl_error_get_code() != CPL_ERROR_NONE) {
        cpl_mask_delete(result);
        result = NULL;
    }

    return result;
}

/**@}*/

/**@{*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Set the specified plane of a MASTER_BPM.
  @return   CPL error code.

  The function sets the specified plane in the MASTER_BPM image to the value
  of the given mask.
 */
/*----------------------------------------------------------------------------*/

cpl_error_code en_master_bpm_set(master_bpm * master, 
                                 const cpl_mask * mask,
                                 bpm_code flag_code) {

    if (cpl_error_get_code() != CPL_ERROR_NONE) return cpl_error_get_code();
    cpl_ensure_code(master && mask, CPL_ERROR_NULL_INPUT);

    /* Check dimensions match */

    cpl_size nx = cpl_mask_get_size_x(mask);
    cpl_size ny = cpl_mask_get_size_y(mask);
    cpl_size nx_image = cpl_image_get_size_x(master->bpm);
    cpl_size ny_image = cpl_image_get_size_y(master->bpm);
    cpl_ensure_code((nx == nx_image) && (ny == ny_image), 
                    CPL_ERROR_INCOMPATIBLE_INPUT);

    /* Set or clear the specified bit in the master_bpm */

    int * bpm_data = cpl_image_get_data_int((cpl_image *) master->bpm);
    const cpl_binary * mask_data = cpl_mask_get_data_const(mask);

    int bit_mask = (1 << (flag_code-1));
    for (cpl_size i=0; i < nx * ny; i++) {
        if (mask_data[i]) {
            bpm_data[i] |= bit_mask;
        } else {
            bpm_data[i] &= (~bit_mask);
        } 
    }

    return cpl_error_get_code();
}

/**@}*/


