/* $Id$
 *
 * This file is part of the ERIS/NIX Pipeline
 * Copyright (C) 2017 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

 /*
 * $Author$
 * $Date$
 * $Rev$
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#include "casu_stats.h"
#endif

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/

#include <casu_mods.h>
#include <hdrl.h>
#include <libgen.h>
#include <math.h>
#include <string.h>

#include "eris_nix_defs.h"
#include "eris_nix_match.h"

/*----------------------------------------------------------------------------*/
/**
 * @defgroup eris_nix_match_utils     Miscellaneous Source Match Utilities
 */
/*----------------------------------------------------------------------------*/

/**@{*/


/*----------------------------------------------------------------------------*/
/**
  @brief    Associate image catalogue objects with standards
  @param    objtab         Table of objects in image
  @param    stdtab         Table of standards
  @param    assoc          Radius of association (pixels)
  @param    strict_classification True to only consider stellar round objects        
  @param    associated_std A table containing data for the associated objects
  @return   A CPL error code

  The routine matches catalogue objects with standards that lie within
  'assoc' pixels. If either image or standard catalogue is NULL then
  the routine returns immediately with no error and without modifying
  associated_std.

 */
/*----------------------------------------------------------------------------*/

cpl_error_code enm_associate_std(cpl_table * objtab,
                                 cpl_table * stdtab,
                                 const float assoc,
                                 const int strict_classification,
                                 cpl_table ** associated_std) {

    if (cpl_error_get_code() != CPL_ERROR_NONE) return cpl_error_get_code();
    if (!objtab) return CPL_ERROR_NONE;
    if (!stdtab) return CPL_ERROR_NONE;

    //cpl_msg_info(cpl_func, "matching");
    //cpl_table_dump(objtab, 0, 100, NULL);
    //cpl_msg_info(cpl_func, "..table of standards");
    //cpl_table_dump(stdtab, 0, 100, NULL);

    /* Check the size of the tables */

    cpl_size nstd = cpl_table_get_nrow(stdtab);
    cpl_size nobj = cpl_table_get_nrow(objtab);
    enu_check_error_code("Error accessing standard or object tables");
    if (nstd == 0) {
        cpl_msg_warning(cpl_func, "No associated sources: standards "
                        "catalogue is empty");
        *associated_std = NULL;
        return CPL_ERROR_NONE;
    } else if (nobj == 0) {
        cpl_msg_warning(cpl_func, "No associated sources: image "
                        "catalogue is empty");
        *associated_std = NULL;
        return CPL_ERROR_NONE;
    }

    /* tables look sensible - look for associations */

    cpl_table_cast_column(stdtab, "xpredict", NULL, CPL_TYPE_DOUBLE);
    double * xstd = cpl_table_get_data_double(stdtab, "xpredict");
    cpl_table_cast_column(stdtab, "ypredict", NULL, CPL_TYPE_DOUBLE);
    double * ystd = cpl_table_get_data_double(stdtab, "ypredict");
    enu_check_error_code("Error mapping table columns 1");
    cpl_table_cast_column(objtab, "X_coordinate", NULL, CPL_TYPE_DOUBLE);
    double * xobj = cpl_table_get_data_double(objtab, "X_coordinate");
    cpl_table_cast_column(objtab, "Y_coordinate", NULL, CPL_TYPE_DOUBLE);
    double * yobj = cpl_table_get_data_double(objtab, "Y_coordinate");
    cpl_table_cast_column(objtab, "Classification", NULL, CPL_TYPE_DOUBLE);
    double * classification = cpl_table_get_data_double(objtab,
                              "Classification");
    cpl_table_cast_column(objtab, "Ellipticity", NULL, CPL_TYPE_DOUBLE);
    double * ellipticity = cpl_table_get_data_double(objtab,
                           "Ellipticity");

    /* loop through table of standards, selecting rows that associate
       with an image object */

    float assoc2 = assoc * assoc;
    cpl_table_unselect_all(stdtab);

    for (cpl_size istd = 0; istd < nstd; istd++) {
        
        /* is there an image object within assoc pixels? */

        cpl_size iobj_match = -1; 
        int unique = CPL_TRUE;

        for (cpl_size iobj= 0; iobj < nobj; iobj++) {

            /* Ignore if object is weird and we're being strict.
               CASU classifications < 0 are good, >= 0 complicated or junk */
            if (strict_classification && 
                (classification[iobj] >= 0.0 || ellipticity[iobj] < 0.5)) {
                continue;
            }

            float distance2 = pow(xobj[iobj] - xstd[istd], 2.0) + 
                              pow(yobj[iobj] - ystd[istd], 2.0);
            if (distance2 < assoc2) {
                if (iobj_match != -1) {
                    unique = CPL_FALSE;
                    break;
                } else {
                    iobj_match = iobj;
                }
            }
        }

        if (iobj_match != -1 && unique) {
            cpl_table_select_row(stdtab, istd);
        }
    }

    if (cpl_table_count_selected(stdtab) > 0) {

        /* Make an associated_std table of the rows selected by association */

        *associated_std = cpl_table_extract_selected(stdtab);

        /* Go through the associated standards table, find the associated 
           object and add the object columns to it. Ignore the RA and DEC 
           columns in the catalogue as the standards table will already 
           have these */
 
        cpl_array * colnames = cpl_table_get_column_names(objtab);
        cpl_size ncol = cpl_array_get_size(colnames);

        for (cpl_size i = 0; i < ncol; i++) {
            const char * colname = cpl_array_get_string(colnames, i);
            if (strcmp(colname,"RA") && strcmp(colname,"DEC")) {
                cpl_table_new_column(*associated_std, colname,
                                     cpl_table_get_column_type(objtab, 
                                     colname));
            }
        }

        nstd = cpl_table_get_nrow(*associated_std);

        cpl_table_cast_column(*associated_std, "xpredict", NULL, 
                              CPL_TYPE_DOUBLE);
        xstd = cpl_table_get_data_double(*associated_std, "xpredict");
        cpl_table_cast_column(*associated_std, "ypredict", NULL,
                              CPL_TYPE_DOUBLE);
        ystd = cpl_table_get_data_double(*associated_std, "ypredict");
        enu_check_error_code("Error mapping matched table columns 1");

        for (cpl_size istd = 0; istd < nstd; istd++) {

            /* find the associated standard */

            cpl_size iobj_match = -1; 
            for (cpl_size iobj= 0; iobj < nobj; iobj++) {
                if (strict_classification && 
                    (classification[iobj] >= 0.0 || ellipticity[iobj] < 0.5)) {
                    continue;
                }
                float distance2 = pow(xobj[iobj] - xstd[istd], 2.0) + 
                                  pow(yobj[iobj] - ystd[istd], 2.0);
                if (distance2 < assoc2) {
                    iobj_match = iobj;
                    break;
                }
            }

            for (cpl_size i = 0; i < ncol; i++) {
                const char * colname = cpl_array_get_string(colnames, i);
                if (strcmp(colname,"RA") && strcmp(colname,"DEC")) {
                    int ignore = 0;

                    switch (cpl_table_get_column_type(objtab, colname)) {
                        case CPL_TYPE_INT:
                            cpl_table_set_int(*associated_std, colname, istd,
                                              cpl_table_get_int(objtab, colname,
                                              iobj_match, &ignore));
                            break;
                        case CPL_TYPE_FLOAT:
                            cpl_table_set_float(*associated_std, colname, istd,
                                                cpl_table_get_float(objtab,
                                                colname, iobj_match, &ignore));
                            break;
                        case CPL_TYPE_DOUBLE:
                            cpl_table_set_double(*associated_std, colname, istd,
                                                 cpl_table_get_double(objtab,
                                                 colname, iobj_match, &ignore));
                            break;
                        default:
                            cpl_table_set_float(*associated_std, colname, istd,
                                                cpl_table_get_float(objtab,
                                                colname, iobj_match, &ignore));
                            break;
                    }
                }
            }
        }
        cpl_array_delete(colnames);

    } else {
        cpl_msg_warning(cpl_func, "No sources associated");
        *associated_std = NULL;
        return CPL_ERROR_NONE;
    }

 cleanup:

    if (cpl_error_get_code() != CPL_ERROR_NONE) {
       cpl_table_delete(*associated_std);
       *associated_std = NULL;
    }
    return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Apply median offsets from matched standards table to image
  @param    wcs_method     String describing the correction method
  @param    catalogue      Name of reference catalogue
  @param    matched_stds   Table of matched standards
  @param    limage         The image to be corrected
  @param    xy_shift       Matrix containing corrections applied
  @return   A CPL error code

  The routine calculates the median shifts in x and y implicit in the
  coordinates in the matched_stds table. These shifts are applied to the 
  CRPIX1 and CRPIX2 keywords in the image to line up the object
  and standard positions.

 */
/*----------------------------------------------------------------------------*/

cpl_error_code enm_correct_crpix(const char * wcs_method,
                                 const char * catalogue,
                                 cpl_table * matched_stds,
                                 located_image * limage,
                                 cpl_matrix ** xy_shift) {

    if (cpl_error_get_code() != CPL_ERROR_NONE) return cpl_error_get_code();
    cpl_ensure_code(wcs_method, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(catalogue, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(limage, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(xy_shift, CPL_ERROR_NULL_INPUT);

    /* Get the RA,DECs and x,y  coords of the matched objects and 
       use them to Calculate the median wcs offset between expected
       and measured positions of matched catalogue objects */

    cpl_size nmatch = 0;
    if (matched_stds) {
        nmatch = cpl_table_get_nrow(matched_stds);
        //cpl_table_dump(matched_stds, 0, 100, NULL);
    }
    //cpl_msg_info(cpl_func, "nmatch %d", (int)nmatch);
 
    if (nmatch > 0) {
        /* cast columns to double if required */
        if (cpl_table_get_column_type(matched_stds, "X_coordinate") == 
            CPL_TYPE_FLOAT) {
            cpl_table_cast_column(matched_stds,
                                  "X_coordinate",
                                  NULL,
                                  CPL_TYPE_DOUBLE);
            cpl_table_cast_column(matched_stds,
                                  "Y_coordinate",
                                  NULL,
                                  CPL_TYPE_DOUBLE);
            cpl_table_cast_column(matched_stds,
                                  "xpredict",
                                  NULL,
                                  CPL_TYPE_DOUBLE);
            cpl_table_cast_column(matched_stds,
                                  "ypredict",
                                  NULL,
                                  CPL_TYPE_DOUBLE);
	}
        /* map the columns */
        const double * x_coord = cpl_table_get_data_double_const(matched_stds,
                                       "X_coordinate");
        const double * y_coord = cpl_table_get_data_double_const(matched_stds,
                                       "Y_coordinate");
        const double * xpredict = cpl_table_get_data_double_const(matched_stds,
                                       "xpredict");
        const double * ypredict = cpl_table_get_data_double_const(matched_stds,
                                       "ypredict");

        enu_check_error_code("failed to access matched table: %s",
                             cpl_error_get_message());

        double diff[nmatch];
        for (cpl_size i = 0; i < nmatch; i++) {
            diff[i] = x_coord[i] - xpredict[i];
        }
        double xoffset = casu_dmed(diff, NULL, nmatch);
        for (cpl_size i = 0; i < nmatch; i++) {
            diff[i] = y_coord[i] - ypredict[i];
        }
        double yoffset = casu_dmed(diff, NULL, nmatch);

        /* update WCS keywords with offset */

        cpl_msg_info(cpl_func, "correcting %4.2e %4.2e", xoffset, yoffset);

        double crpix1 = cpl_propertylist_get_double(limage->plist, "CRPIX1");
        cpl_propertylist_update_double(limage->plist, "CRPIX1", crpix1 + xoffset);
        double crpix2 = cpl_propertylist_get_double(limage->plist, "CRPIX2");
        cpl_propertylist_update_double(limage->plist, "CRPIX2", crpix2 + yoffset);

        /* store correction method */

        cpl_propertylist_update_string(limage->plist, "ESO WCS_METHOD", 
                                       wcs_method);
        cpl_propertylist_update_string(limage->plist, "WCS_CAT",
                                       catalogue);

        /* store the shift being used */

        *xy_shift = cpl_matrix_new(1, 2);
        cpl_matrix_set(*xy_shift, 0, 0, xoffset);
        cpl_matrix_set(*xy_shift, 0, 1, yoffset);
    } else {
        cpl_msg_info(cpl_func, "no entries in matched standards table, no "
                     "correction applied");
        *xy_shift = NULL;
    }

 cleanup:

    return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Match standards with objects by trying all possibilities and selecting best
  @param    objtab     Table of objects in image.
  @param    stdtab     Table of standards to pattern match.
  @param    match_rad  Max distance between matching obj and std (pixels).
  @param    matchtab   Pointer to table with matched standard/object pairs.
  @return   A CPL error code

  An object table from hdrl_catalogue and a standard star table, from either 
  casu_getstds or hdrl_catalogue, are given. 

  All possible associations between standards and objects are tried and a 
  metric of the pattern match is calculated for each - that is, the number
  of std/obj pairs that lie within 'match_rad' pixels of each other.

  The association with the highest metric is selected, and a table is
  constructed of the std/obj matches found for it.

 */
/*----------------------------------------------------------------------------*/

cpl_error_code enm_try_all_associations(cpl_table * objtab,
                                        cpl_table * stdtab,
                                        const double match_rad,
                                        cpl_table ** matchtab) {

    cpl_array * colnames = NULL;
    const char  * funcid = "enm_match_brute_force";
    cpl_size   * matches = NULL;
    cpl_array * match_is = NULL;
    cpl_array * match_io = NULL;
    cpl_table    * mstds = NULL;

    if (cpl_error_get_code() != CPL_ERROR_NONE) return cpl_error_get_code();
    if (!objtab) return CPL_ERROR_NONE;
    if (!stdtab) return CPL_ERROR_NONE;

    //cpl_table_dump(stdtab, 0, 100, NULL);
    //cpl_table_dump(objtab, 0, 100, NULL);

    /* Check the size of the tables */

    cpl_size nstd = cpl_table_get_nrow(stdtab);
    cpl_size nobj = cpl_table_get_nrow(objtab);
    enu_check(nstd > 0, CPL_ERROR_ILLEGAL_INPUT, "Standards catalogue"
              "contains no entries");
    enu_check(nobj > 0, CPL_ERROR_ILLEGAL_INPUT, "Image catalogue"
              "contains no entries");

    /* sort the tables in order of source flux. One aim of this that 
       if the tables have few entries with several possible matches 
       then a match with the brightest source will be found first
       and win - in such a case this may not always be the right 
       outcome but, hopefully, more often than not*/

    if (cpl_table_has_column(stdtab, "Aper_flux_3")) {
        cpl_msg_info(cpl_func, " ..sorting standards table");
        cpl_propertylist * reflist = cpl_propertylist_new();
        cpl_propertylist_update_bool(reflist, "Aper_flux_3", CPL_TRUE);
        cpl_table_sort(stdtab, reflist);
        cpl_propertylist_delete(reflist);
    }
    if (cpl_table_has_column(objtab, "Aper_flux_3")) {
        cpl_msg_info(cpl_func, " ..sorting catalogue table");
        cpl_propertylist * reflist = cpl_propertylist_new();
        cpl_propertylist_update_bool(reflist, "Aper_flux_3", CPL_TRUE);
        cpl_table_sort(objtab, reflist);
        cpl_propertylist_delete(reflist);
    }

    /* the algorithm is to trial associating each standard with each
       object then see how many other sources 'match'.
       The time taken goes as nsts*nobj (# trials) * nstd*nobj (test
       metric) so can quickly get large. This is unnecessary if
       we just to find the best shift so limit the sizes -
       but try to do this in a way that doesn't limit the spread of
       objects over the images */  

    if (nstd > 100) {
       cpl_msg_info(cpl_func, "truncating standards table to 100 entries");
       nstd = 100;
    }
    if (nobj > 100) {
       cpl_msg_info(cpl_func, "truncating catalogue table to 100 entries");
       nobj = 100;
    }

    *matchtab = NULL;

    /* tables look sensible - map positional columns */

    double * xstd = cpl_table_get_data_double(stdtab, "xpredict");
    double * ystd = cpl_table_get_data_double(stdtab, "ypredict");
    double * xobj = cpl_table_get_data_double(objtab, "X_coordinate");
    double * yobj = cpl_table_get_data_double(objtab, "Y_coordinate");

    cpl_size max_match = 0;
    cpl_size match_std = -1;
    cpl_size match_obj = -1;

    for (cpl_size is = 0; is < nstd; is++) {
        for (cpl_size io = 0; io < nobj; io++) {

            cpl_matrix * distances = NULL;
            enm_try_association(xstd, ystd, nstd,
                                xobj, yobj, nobj, 
                                is, io, &distances);

            /* match objects */

            cpl_array * match_distances = cpl_array_new(0, CPL_TYPE_DOUBLE);

            while (cpl_matrix_get_min(distances) < match_rad) {
                cpl_size istd_min = 0;
                cpl_size iobj_min = 0;
                double dist = cpl_matrix_get_min(distances);
                cpl_matrix_get_minpos(distances,
                                      &istd_min,
                                      &iobj_min);
                cpl_matrix_fill_row(distances, 10000.0, istd_min);
                cpl_matrix_fill_column(distances, 10000.0, iobj_min);
                //cpl_msg_info(cpl_func, "std %d matches obj %d dist %f",
                //             (int)istd_min, (int)iobj_min, dist);
                cpl_size next = cpl_array_get_size(match_distances);
                cpl_array_set_size(match_distances, next+1);
                cpl_array_set_double(match_distances, next, dist);
            }

            cpl_size nmatch = cpl_array_get_size(match_distances);
            if (nmatch > max_match) {
                max_match = nmatch;
                match_std = is;
                match_obj = io;                    
            }            

            cpl_array_delete(match_distances);
            cpl_matrix_delete(distances);
        }
    }
    cpl_msg_info(cpl_func, " ..best match s=%d o=%d nmatch=%d match_rad=%e",
                 (int)match_std, (int)match_obj, (int)max_match, match_rad);
    if (max_match == 1) {
        cpl_msg_warning(cpl_func, "\033[0;31m"
                        "WCS calibration is unreliable because the pattern "
                        "match is based on a single source"
                        "\033[0m");
    }

    /* recalculate the catalogue association that gave the best match.
       For this we want all the matched objects we can get so use the
       full tables */

    nstd = cpl_table_get_nrow(stdtab);
    nobj = cpl_table_get_nrow(objtab);

    cpl_matrix * distances = NULL;
    enm_try_association(xstd, ystd, nstd,
                        xobj, yobj, nobj, 
                        match_std, match_obj, &distances);

    match_is = cpl_array_new(0, CPL_TYPE_INT);
    match_io = cpl_array_new(0, CPL_TYPE_INT);

    //cpl_matrix_dump(distances, NULL);

    while (cpl_matrix_get_min(distances) < match_rad) {
        cpl_size istd_min = 0;
        cpl_size iobj_min = 0;
        cpl_matrix_get_minpos(distances,
                              &istd_min,
                              &iobj_min);
        cpl_matrix_fill_row(distances, 10000.0, istd_min);
        cpl_matrix_fill_column(distances, 10000.0, iobj_min);
        cpl_size nmatch = cpl_array_get_size(match_is);
        cpl_array_set_size(match_is, nmatch+1);
        cpl_array_set_int(match_is, nmatch, istd_min);
        cpl_array_set_size(match_io, nmatch+1);
        cpl_array_set_int(match_io, nmatch, iobj_min);
    }
    cpl_matrix_delete(distances);

    /* Make a copy of the standards table and add all the columns 
       from the object catalogue that it doesn't already have */

    mstds = cpl_table_duplicate(stdtab);
    colnames = cpl_table_get_column_names(objtab);
    cpl_size ncol = cpl_array_get_size(colnames);
    for (cpl_size k = 0; k < ncol; k++) {
        const char * colname = cpl_array_get_string(colnames, k);
        if (!cpl_table_has_column(mstds, colname)) {
            cpl_table_new_column(mstds, colname,
                                 cpl_table_get_column_type(objtab, colname));
        }
    }

    /* Now go through mstds (the matched standards catalogue) and set the 
       matches. NB: the columns in objtab
       are created by casu_imcore which only produces integer, 
       float and double columns, so we don't need to worry about 
       any other type when copying the columns to the matched 
       standards catalogue. */

    cpl_table_unselect_all(mstds);

    for (cpl_size im = 0; im < cpl_array_get_size(match_is); im++) {
        int ignore = 0;
        cpl_size is = cpl_array_get_int(match_is, im, &ignore);

        /* select the row in mstds and copy the column values except for
           RA and DEC */

        cpl_table_select_row(mstds, is);

        cpl_size io = cpl_array_get_int(match_io, im, &ignore);

        for (cpl_size k = 0; k < ncol; k++) {
            const char * colname = cpl_array_get_string(colnames, k);
            if (!strcmp(colname,"RA") || !strcmp(colname,"DEC")) { 
                continue;
            }

            cpl_type column_type = cpl_table_get_column_type(objtab,
                                                             colname);
            switch (column_type) {
                case CPL_TYPE_INT:
                    cpl_table_set_int(mstds, colname, is,
                                      cpl_table_get_int(objtab,
                                      colname, io, &ignore));
                    break;

                case CPL_TYPE_FLOAT:
                    cpl_table_set_float(mstds, colname, is,
                                        cpl_table_get_float(objtab,
                                        colname, io, &ignore));
                    break;
                case CPL_TYPE_DOUBLE:
                    cpl_table_set_double(mstds, colname, is,
                                         cpl_table_get_double(objtab,
                                         colname, io, &ignore));
                    break;
                default:
                    cpl_error_set_message(funcid,
                                          CPL_ERROR_UNSUPPORTED_MODE, 
                                          "column type not handled: %s",
                                          cpl_type_get_name(column_type));
                    break;
            }
        }
    }

    /* Now extract the selected rows into the output table */

    *matchtab = cpl_table_extract_selected(mstds);

    //cpl_table_dump(*matchtab, 0, 100, NULL);

 cleanup:

    cpl_array_delete(colnames);
    cpl_free(matches);
    cpl_array_delete(match_is);
    cpl_array_delete(match_io);
    cpl_table_delete(mstds);

    if (cpl_error_get_code() != CPL_ERROR_NONE) {
        cpl_table_delete(*matchtab);
        *matchtab = NULL;
    }

    return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Method to calculate 'distances' matrix given an object association
  @param    xstd           Array containing x-coords of standards
  @param    ystd           Array containing y-coords of standards
  @param    nstd           Number of standards
  @param    xobj           Array containing x-coords of objects
  @param    yobj           Array containing y-coords of objects
  @param    nobj           Number of objects
  @param    is             Index of match standard
  @param    io             Index of match object
  @param    distances      out cpl_matrix containing the resulting distances
                           between matched stds/objs
  @return   Matrix containing the resulting distances between matched stds/objs

  Arrays with x,y coords of standards and objects are given, and a 
  specification of which standard and which object are to be matched. The
  routine calculates the implied offset between standards and objects,
  and shifts the objects by this amount to now be in the 'standard' frame.
  A matrix is populated with the distances between each standard and each 
  object, and returned. 

 */
/*----------------------------------------------------------------------------*/

cpl_error_code enm_try_association(const double xstd[],
                                   const double ystd[],
                                   const cpl_size nstd,
                                   const double xobj[],
                                   const double yobj[],
                                   const cpl_size nobj,
                                   const cpl_size is,
                                   const cpl_size io,
                                   cpl_matrix ** distances) {

    if (cpl_error_get_code() != CPL_ERROR_NONE) return cpl_error_get_code();
    cpl_ensure_code(xstd, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(ystd, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(xobj, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(yobj, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(is>=0 && is<nstd, CPL_ERROR_ACCESS_OUT_OF_RANGE);
    cpl_ensure_code(io>=0 && io<nobj, CPL_ERROR_ACCESS_OUT_OF_RANGE);

    //cpl_msg_info(cpl_func, "associate std=%d with obj=%d", (int)is, (int)io);

    /* the implied shift if this was true */

    double xshift = xstd[is] - xobj[io];
    double yshift = ystd[is] - yobj[io]; 
 
    /* match up objects */

    *distances = cpl_matrix_new(nstd, nobj);
    cpl_matrix_fill(*distances, 10000.0);

    for (cpl_size iis = 0; iis < nstd; iis++) {
        for (cpl_size iio = 0; iio < nobj; iio++) {

            /* position object would have in std image */
            double xobj_shifted = xobj[iio] + xshift;
            double yobj_shifted = yobj[iio] + yshift;

            /* distance between them? */
            double xdist = xstd[iis] - xobj_shifted;
            double ydist = ystd[iis] - yobj_shifted;
            double dist = sqrt(xdist*xdist + ydist * ydist);                        

            cpl_matrix_set(*distances, iis, iio, dist);
        }
    }

    return cpl_error_get_code();
}

/**@}*/
