/* $Id$
 *
 * This file is part of the ERIS/NIX Pipeline
 * Copyright (C) 2017 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/*
 * $Author$
 * $Date$
 * $Rev$
 */

#ifndef ERIS_NIX_UTILS_H
#define ERIS_NIX_UTILS_H

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/

#include "eris_nix_defs.h"
#include "eris_nix_detector.h"
#include "eris_nix_master_bpm.h"
#include "eris_nix_master_dark.h"
#include "eris_nix_master_flat.h"
#include "eris_nix_gain_linearity.h"
#include "eris_nix_casu_match.h"
#include <cpl.h>

/*-----------------------------------------------------------------------------
                                   Defines
 -----------------------------------------------------------------------------*/

#define MEF_EXTENSION_CONTAINING_IMAGE "ERIS_NIX_CPL_IMAGE"
#define MEF_EXTENSION_CONTAINING_MASK "ERIS_NIX_CPL_MASK"
#define MEF_EXTENSION_CONTAINING_TABLE "ERIS_NIX_CPL_TABLE"
#define MEF_EXTENSION_CONTAINING_VECTOR "ERIS_NIX_CPL_VECTOR"

/*-----------------------------------------------------------------------------
                                       Prototypes
 -----------------------------------------------------------------------------*/

cpl_error_code enu_basic_calibrate(located_image        * limage,
                                   const int              read_offsets,
                                   const cpl_table      * refine_wcs,
                                   const master_dark    * master_dark,
                                   const gain_linearity * gain_linearity,
                                   const master_flat    * flatfield_1,
                                   const master_flat    * flatfield_2,
                                   const master_bpm     * master_bpm,
                                   const int              flag_mask,
                                   const char           * fill_rejected,
                                   const double           fill_value,
                                   const cpl_size         x_probe,
                                   const cpl_size         y_probe);

cpl_error_code enu_basic_calibrate_himage(hdrl_image           * himage,
                                          cpl_image            * confidence,
                                          cpl_propertylist     * plist,
                                          const cpl_frame      * frame,
                                          const int              read_offsets,
                                          const master_dark    * master_dark,
                                          const gain_linearity * gain_linearity,
                                          const master_flat    * flatfield_1,
                                          const master_flat    * flatfield_2,
                                          const master_bpm     * master_bpm,
                                          const int              set_confidence,
                                          const int              flag_mask,
                                          const char           * fill_rejected,
                                          const double           fill_value,
                                          const cpl_size         x_probe,
                                          const cpl_size         y_probe);

cpl_vector * enu_bracket_skys(const located_image * target,
                              const double timerange, 
                              const located_imagelist * sky_pool,
                              const int debug);

hdrl_image * enu_calc_flat(hdrl_imagelist * himlist,
                           const int min_coadds,
                           const hdrl_parameter * collapse_params,
                           const cpl_size filter_size_x,
                           const cpl_size filter_size_y,
                           const hdrl_flat_method method);

cpl_error_code enu_calc_maglim(const located_image * limage,
                               const double photzp,
                               const double fwhm_pix,
                               double * abmaglim);

cpl_error_code enu_calc_pixel_coords(cpl_table * catalogue,
                                     const cpl_propertylist * wcs_plist);

hdrl_catalogue_result * enu_catalogue_compute(const hdrl_image *,
                                              const cpl_image *,
                                              const cpl_wcs *,
                                              hdrl_parameter *);

cpl_error_code enu_catalogue_limlist(located_imagelist * limlist,
                                     hdrl_parameter * cat_params);

int enu_check_conformance(const cpl_propertylist *,
                          const cpl_propertylist *,
                          const char *);

cpl_error_code enu_check_wcs(const located_image * limage);

cpl_error_code  enu_correct_wcs(const cpl_table * refcat,
                                const char * wcs_method,
                                const char * catalogue,
                                located_image * limage,
                                const double match_rad,
                                cpl_table ** matched_stds,
                                cpl_matrix ** xy_shift);

cpl_error_code enu_debug_limlist_save(const int debug,
                                      const located_imagelist * limlist,
                                      const char * nameroot,
                                      const char * recipename,
                                      cpl_frameset * frameset,
                                      const cpl_parameterlist * parlist,
                                      const cpl_frameset * used);

cpl_error_code enu_flat_save(const char * pro_catg,
                             const hdrl_image * flat,
                             const cpl_image * confidence,
                             const cpl_mask * cold_bpm,
                             cpl_frameset * frameset,
                             const cpl_parameterlist * parlist,
                             const char * filename,
                             const char * flat_save,
                             const cpl_propertylist* qclog);

double enu_get_airmass(const cpl_propertylist * plist);

cpl_error_code enu_get_rcore_and_mesh_size(const char * context,
                                           const cpl_parameterlist * parlist,
                                           const cpl_propertylist * plist,
                                           double * core_radius,
                                           int * mesh_size);

const char * enu_get_det_mode(const cpl_propertylist * plist);

double enu_get_dit(const cpl_propertylist * plist);

const char * enu_get_filter(const cpl_propertylist * plist);

double enu_get_filter_wavelength(const char * filter);

const char * enu_get_license(void);

cpl_error_code enu_get_ra_dec(const cpl_wcs * wcs,
                              double * ra,
                              double * dec);

cpl_error_code enu_get_window_info(cpl_size * nx,
                                   cpl_size * ny,
                                   int * rot,
                                   cpl_size * strx,
                                   cpl_size * stry,
                                   cpl_size * nx_chip,
                                   cpl_size * ny_chip,
                                   cpl_boolean * windowed,
                                   const cpl_propertylist * plist);

hdrl_catalogue_result * enu_hdrl_catalogue_result_duplicate(
  const hdrl_catalogue_result *);

cpl_error_code enu_himage_load_from_fits(const char *,
                                         hdrl_image **,
                                         mef_extension_list **,
                                         cpl_propertylist **);

cpl_image * enu_load_component(mef_extension * mef,
                               const char * dps_name,
                               const char * target_hduclas1,
                               const char * target_hduclas2);
  
located_image * enu_load_limage_from_frame(const cpl_frame * frame,
                                           cpl_image **, const cpl_boolean);

mef_extension_list * enu_load_mef_components(const char * filename,
                                             cpl_propertylist ** plist);

located_image * enu_located_image_duplicate(const located_image *);

void enu_located_image_delete(located_image *);

void enu_located_imagelist_delete(located_imagelist *);

located_imagelist * enu_located_imagelist_duplicate(
        const located_imagelist *);
                                
located_imagelist * enu_limlist_load_from_frameset(cpl_frameset *,
                                                   const char *,
                                                   cpl_frameset *);

cpl_error_code enu_located_imagelist_insert(located_imagelist *,
                                            located_image *,
                                            cpl_size);

located_imagelist * enu_located_imagelist_new(cpl_size);

located_image * enu_located_image_new(hdrl_image            * himage,
                                      hdrl_imagelist        * himagelist,
                                      cpl_image             * confidence,
                                      hdrl_image            * bkg,
                                      cpl_image             * bkg_confidence,
                                      cpl_propertylist      * plist,
                                      hdrl_catalogue_result * objects,
                                      cpl_mask              * object_mask,
                                      hdrl_catalogue_result * wcs,
                                      hdrl_catalogue_result * photom,
                                      cpl_frame             * frame);

void enu_mef_extension_delete(mef_extension *);

void enu_mef_extension_list_delete(mef_extension_list *);

cpl_mask * enu_mef_extension_list_get_mask(mef_extension_list *,
                                           const char *);

mef_extension_list * enu_mef_extension_list_new(cpl_size);

mef_extension * enu_mef_new_image(const char *,
                                  const cpl_image *,
                                  const cpl_propertylist *);

mef_extension * enu_mef_new_mask(const char *,
                                 const cpl_mask *,
                                 const cpl_propertylist *);

mef_extension * enu_mef_new_table(const char *,
                                  const cpl_table *,
                                  const cpl_propertylist *);

mef_extension * enu_mef_new_vector(const char * name,
                                   const cpl_vector * vector,
                                   const cpl_propertylist * plist);

cpl_error_code enu_mef_extension_save(const mef_extension *,
                                      const char *,
                                      const cpl_propertylist *,
                                      unsigned);

cpl_error_code enu_modify_CD_matrix(located_image * limage,
                                    const cpl_table * refine_wcs);

cpl_error_code enu_normalise_confidence(cpl_image * confidence);

cpl_error_code enu_opm_limlist(const int obj_min_pixels,
                               const double obj_threshold,
                               const int bkg_mesh_size,
                               const double bkg_smooth_fwhm,
                               located_imagelist * limages);

cpl_error_code enu_opm_lss_limlist(located_imagelist * limlist,
                                   const int nsigma_cut);

cpl_error_code enu_remove_read_offsets(hdrl_image * himage,
                                       const cpl_propertylist * plist,
                                       cpl_image * confidence,
                                       const int set_confidence);

char * enu_repreface(const char * filename,
                     const char * preface);

cpl_error_code enu_sky_backgrounds(const char * method,
                                   const char * select_method,
                                   const double timerange,
                                   const located_imagelist * target_data,
                                   const located_imagelist * sky_data,
                                   const cpl_size x_probe,
                                   const cpl_size y_probe,
                                   hdrl_imagelist ** sky_himagelist,
                                   cpl_imagelist ** sky_confidence_list);

cpl_error_code enu_sky_subtract_limlist(const char * method,
                                        const char * select_method,
                                        const double timerange,
                                        const located_imagelist * sky_data,
                                        const cpl_size x_probe,
                                        const cpl_size y_probe,
                                        located_imagelist * target_data);

cpl_error_code enu_stack(located_imagelist * limages,
                         const double stk_lthr,
                         const double stk_hthr,
                         const char * stk_method,
                         const int stk_fast,
                         located_image ** result);

double enu_get_tel_alt(const cpl_propertylist * plist);

cpl_error_code eris_nix_get_badpix_qc_from_ima(const master_bpm * master_bpm,
                                               cpl_propertylist * qc_list,
                                               const char * prefix);

cpl_propertylist * enu_raw_flats_qc(located_imagelist * lamp_on_limlist,
                                    cpl_mask * bp_map_nl_mask,
                                    const cpl_parameterlist * parlist,
                                    const char * context,
                                    const double threshold_pos,
									 const cpl_boolean verbose,
									 const cpl_boolean rescale_by_dit);

#endif


