/* $Id$
 *
 * This file is part of the ERIS Pipeline
 * Copyright (C) 2002,2003 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <eris_ifu_error.h>

typedef enum {
	success, setError, setErrorMsg, checkError, checkErrorMsg,
	exitWithError,  exitWithErrorMsg, exitIfError, exitIfNull, exitIfNot, assure
} tryModeType;
typedef enum {
	none, errorCatched, errorReturned, errorReplace, errorAdd
} catchModeType;

const int error=1;

/*---------------------------------------------------------------------------*/
/**
  @brief
  @param mode:
  @return 0: execution entered Catch clause
          1: all of the Try clause was executed
 */
/*---------------------------------------------------------------------------*/
int eris_ifu_test_try_catch(tryModeType tryMode, catchModeType catchMode);
int eris_ifu_test_try_catch(tryModeType tryMode, catchModeType catchMode)
{
	int retVal = 0;
	cpl_error_code testError;
	cpl_image* testImg;

	TRY
	{
		switch (tryMode) {
			case success:
				break;
			case setError:
				SET_ERROR(CPL_ERROR_EOL);
				break;
			case setErrorMsg:
				SET_ERROR_MSG(CPL_ERROR_EOL,"oops");
				break;
			case checkError:
				SET_ERROR(CPL_ERROR_EOL);
				CHECK_ERROR_STATE();
				break;
			case checkErrorMsg:
				SET_ERROR(CPL_ERROR_EOL);
				CHECK_ERROR_STATE_MSG("oops oops");
				break;
			case exitWithError:
				BRK_WITH_ERROR(CPL_ERROR_FILE_IO);
				break;
			case exitWithErrorMsg:
				BRK_WITH_ERROR_MSG(CPL_ERROR_FILE_IO,"foo");
				break;
			case exitIfError:
				BRK_IF_ERROR(
						testError = cpl_image_abs(NULL));
				break;
			case exitIfNull:
				BRK_IF_NULL(
						testImg = cpl_image_new(0, 10, CPL_TYPE_INT));
				break;
			case assure:
				ASSURE(
						0 < -10,
						CPL_ERROR_ILLEGAL_INPUT,
						"indexes not in proper order (%d/%d)", 0, -10);
				break;
			default:
				break;
		}
		retVal = 1;
	} CATCH {
		cpl_error_code error_code = GET_NEW_ERROR();
		if (error_code == CPL_ERROR_EOL) {
			retVal |= 4;
		} else {
			retVal |= 2;
			if (catchMode == errorCatched) {
				RECOVER();
			} else if (catchMode == errorReplace) {
				RECOVER();
				cpl_error_set(__func__, CPL_ERROR_BAD_FILE_FORMAT);
			} else if (catchMode == errorAdd) {
				SET_ERROR(CPL_ERROR_BAD_FILE_FORMAT);
			}
		}
	}

	return retVal;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit tests of eris_ifu_error module
 */
/*----------------------------------------------------------------------------*/

int main(void)
{
	cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_WARNING);

	int returnValue;

	/* Test successfull case in try/catch
	 * (and the BRK_WITH_ERROR macro) */
	cpl_error_reset();
	returnValue = eris_ifu_test_try_catch(success, none);
	cpl_test_eq(returnValue, 1); /* Try clause shall be executed till its end */
	cpl_test_eq((long ) cpl_errorstate_get(), 0); /* No cpl_error is expected */
	cpl_test_error(CPL_ERROR_NONE);

	/* Test catch clause and the SET_ERROR macro */
	cpl_error_reset();
	returnValue = eris_ifu_test_try_catch(setError, none);
	cpl_test_eq(returnValue, 5); /* Catch clause shall be entered */
	cpl_test_eq((long ) cpl_errorstate_get(), 1); /* 1 cpl_error is expected */
	cpl_test_error(CPL_ERROR_EOL);

	/* Test catch clause and the SET_ERROR_MSG macro */
	cpl_error_reset();
	returnValue = eris_ifu_test_try_catch(setErrorMsg, none);
	cpl_test_eq(returnValue, 5); /* Catch clause shall be entered */
	cpl_test_eq((long ) cpl_errorstate_get(), 1); /* 1 cpl_error is expected */
	cpl_test_error(CPL_ERROR_EOL);

	/* Test catch clause and the CHECK_ERROR_STATE macro */
	cpl_error_reset();
	returnValue = eris_ifu_test_try_catch(checkError, none);
	cpl_test_eq(returnValue, 4); /* Catch clause shall be entered */
	cpl_test_eq((long ) cpl_errorstate_get(), 2); /* 2 cpl_errors are expected */
	cpl_test_error(CPL_ERROR_EOL);

	/* Test catch clause and the CHECK_ERROR_STATE_MSG macro */
	cpl_error_reset();
	returnValue = eris_ifu_test_try_catch(checkErrorMsg, none);
	cpl_test_eq(returnValue, 4); /* Catch clause shall be entered */
	cpl_test_eq((long ) cpl_errorstate_get(), 2); /* 2 cpl_errors are expected */
	cpl_test_error(CPL_ERROR_EOL);

	/* Test catch clause and the BRK_WITH_ERROR macro */
	cpl_error_reset();
	returnValue = eris_ifu_test_try_catch(exitWithError, none);
	cpl_test_eq(returnValue, 2); /* Catch clause shall be entered */
	cpl_test_eq((long ) cpl_errorstate_get(), 1); /* 1 cpl_error is expected */
	cpl_test_error(CPL_ERROR_FILE_IO);

	/* Test catch clause and the BRK_WITH_ERROR_MSG macro */
	cpl_error_reset();
	returnValue = eris_ifu_test_try_catch(exitWithErrorMsg, none);
	cpl_test_eq(returnValue, 2); /* Catch clause shall be entered */
	cpl_test_eq((long ) cpl_errorstate_get(), 1); /* 1 cpl_error is expected */
	cpl_test_error(CPL_ERROR_FILE_IO);

	/* Test catch clause and the BRK_IF_ERROR macro */
	cpl_error_reset();
	returnValue = eris_ifu_test_try_catch(exitIfError, none);
	cpl_test_eq(returnValue, 2); /* Catch clause shall be entered */
	cpl_test_eq((long ) cpl_errorstate_get(), 2); /* 1 cpl_errors are expected */
	cpl_test_error(CPL_ERROR_NULL_INPUT);

	/* Test catch clause and the BRK_IF_NULL macro */
	cpl_error_reset();
	returnValue = eris_ifu_test_try_catch(exitIfNull, none);
	cpl_test_eq(returnValue, 2); /* Catch clause shall be entered */
	cpl_test_eq((long ) cpl_errorstate_get(), 3); /* 3 cpl_errors are expected */
	cpl_test_error(CPL_ERROR_ILLEGAL_INPUT);

	/* Test catch clause and the ASSURE macro */
	cpl_error_reset();
	returnValue = eris_ifu_test_try_catch(assure, none);
	cpl_test_eq(returnValue, 2); /* Catch clause shall be entered */
	cpl_test_eq((long ) cpl_errorstate_get(), 1); /* 3 cpl_errors are expected */
	cpl_test_error(CPL_ERROR_ILLEGAL_INPUT);

	/*  Test catch clause and the RECOVER macro*/
	cpl_error_reset();
	returnValue = eris_ifu_test_try_catch(exitWithError, errorCatched);
	cpl_test_eq(returnValue, 2); /* Catch clause shall be entered */
	cpl_test_eq((long ) cpl_errorstate_get(), 0); /* No cpl_error is expected */
	cpl_test_error(CPL_ERROR_NONE);

	/* Test catch clause and replacing the original error with another one*/
	cpl_error_reset();
	returnValue = eris_ifu_test_try_catch(exitWithError, errorReplace);
	cpl_test_eq(returnValue, 2); /* Catch clause shall be entered */
	cpl_test_eq((long ) cpl_errorstate_get(), 1); /* 1 cpl_error is expected */
	cpl_test_error(CPL_ERROR_BAD_FILE_FORMAT);

	/* Test catch clause and add another error*/
	cpl_error_reset();
	returnValue = eris_ifu_test_try_catch(exitWithError, errorAdd);
	cpl_test_eq(returnValue, 2); /* Catch clause shall be entered */
	cpl_test_eq((long ) cpl_errorstate_get(), 2); /* 2 cpl_errors are expected */
	cpl_test_error(CPL_ERROR_BAD_FILE_FORMAT);

	return cpl_test_end(0);

}

