/*
 * This file is part of the ERIS Pipeline
 * Copyright (C) 2002,2003 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <cpl.h>
#include <hdrl.h>
#include <string.h>

#include <eris_ifu_strehl.h>
#include <eris_ifu_dfs.h>
#include <eris_utils.h>
#include <eris_ifu_stdstar_static.h>
#include <eris_ifu_jitter_interface.h>
/* Helper function to create test frameset */
static cpl_propertylist* create_test_header(const char* dtype) {
	 cpl_propertylist* header = cpl_propertylist_new();

	 /* Add test properties */
	 cpl_propertylist_append_string(header, "INSTRUME", "ERIS");
	 cpl_propertylist_append_string(header, "ESO SEQ ARM", "SPIFFIER");
	 cpl_propertylist_append_string(header, "ESO DPR TYPE", dtype);
	 cpl_propertylist_append_string(header, "ESO INS MODE", "IFU");
	 cpl_propertylist_append_string(header, "ESO INS OPTI3 NAME", "13mas");
	 cpl_propertylist_append_string(header, "ESO INS3 SPGW ID", "K_low");
	 cpl_propertylist_append_string(header, "ESO INS3 SPXW ID", "25mas");



	 cpl_propertylist_append_double(header, "ESO DET CHIP RON", 3.0);
	 cpl_propertylist_append_double(header, "ESO DET CHIP GAIN", 2.0);
	 cpl_propertylist_append_string(header, "ESO INS BAND", "K");
	 cpl_propertylist_append_string(header, "ESO INS SCALE", "25");

	 cpl_propertylist_append_double(header, "ESO DET DIT", 60.0);
	 cpl_propertylist_append_double(header, "MJD-OBS", 58000.0);
	 cpl_propertylist_append_double(header, "ESO TEL ALT", 45.0);
	 cpl_propertylist_append_double(header, "ESO ADA POSANG", 30.0);
	 cpl_propertylist_append_double(header, "ESO DET SEQ1 DIT", 30.0);

	 return header;
}



/* Helper function to create test data */
static cpl_frameset* create_test_frameset(const char* tag) {
    cpl_frameset* frameset = cpl_frameset_new();
    

    /* Create test FITS file */
    cpl_image* test_image = cpl_image_new(32, 32, CPL_TYPE_DOUBLE);
    cpl_image_fill_noise_uniform(test_image, 0.0, 1.0);
    
    /* Add a Gaussian PSF */
    const double sigma = 2.0;
    const double amplitude = 100.0;
    const int center_x = 16;
    const int center_y = 16;
    
    for(int y = 1; y <= 32; y++) {
        for(int x = 1; x <= 32; x++) {
            double dx = x - center_x;
            double dy = y - center_y;
            double r2 = dx*dx + dy*dy;
            double value = amplitude * exp(-r2/(2*sigma*sigma));
            cpl_image_set(test_image, x, y, value);
        }
    }
    
    /* Create FITS header */
    cpl_propertylist* header = create_test_header("STD");

    /* Save test image */
    cpl_image_save(test_image, "test_strehl.fits", CPL_TYPE_DOUBLE, header, CPL_IO_CREATE);
    
    /* Create error image */
    cpl_image* error_image = cpl_image_duplicate(test_image);
    cpl_image_multiply_scalar(error_image, 0.1); /* 10% errors */
    cpl_image_save(error_image, "test_strehl.fits", CPL_TYPE_DOUBLE, NULL, CPL_IO_EXTEND);
    
    /* Create BPM */
    cpl_image* bpm = cpl_image_new(32, 32, CPL_TYPE_INT);
    cpl_image_save(bpm, "test_strehl.fits", CPL_TYPE_INT, NULL, CPL_IO_EXTEND);
    
    /* Add frames to frameset */
    cpl_frame* frame = cpl_frame_new();
    cpl_frame_set_filename(frame, "test_strehl.fits");
    cpl_frame_set_group(frame, CPL_FRAME_GROUP_RAW);

    cpl_frame_set_tag(frame, ERIS_IFU_PRO_JITTER_CUBE);
    cpl_frameset_insert(frameset, frame);
    

    /* Add distortion frame */
    cpl_frame* dist_frame = cpl_frame_new();
    cpl_propertylist* dist_header = create_test_header("DISTORTION");

    cpl_frame_set_filename(dist_frame, "test_dist.fits");
    cpl_frame_set_group(dist_frame, CPL_FRAME_GROUP_CALIB);
    cpl_frame_set_level(dist_frame, CPL_FRAME_LEVEL_NONE);
    cpl_frame_set_type(dist_frame, CPL_FRAME_TYPE_TABLE);
    cpl_frame_set_tag(dist_frame, "DISTORTION");
    cpl_propertylist_save(dist_header, "test_dist.fits", CPL_IO_CREATE);
    cpl_frameset_insert(frameset, dist_frame);
    cpl_propertylist_delete(dist_header);


    /* Cleanup */
    cpl_propertylist_delete(header);
    cpl_image_delete(test_image);
    cpl_image_delete(error_image);
    cpl_image_delete(bpm);
    
    return frameset;
}
/* Helper function to create test parameter list */
static cpl_parameterlist* create_test_parameters(void) {
    cpl_parameterlist* parlist = cpl_parameterlist_new();
    eris_ifu_jitter_fill_common_parameterlist("eris_ifu_stdstar", M_SCIENCE, parlist);
    cpl_frameset* frameset = create_test_frameset("PSF_CALIBRATOR");
    cpl_error_code error = eris_stdstar_reduction_common(frameset, parlist,
             CPL_TRUE, "eris.eris_ifu_stdstar", "eris_ifu_stdstar");
      /* Add required parameters */
      cpl_parameter* par = cpl_parameter_new_value("eris.eris_ifu_stdstar.strehl_flux_radius",
          CPL_TYPE_DOUBLE, "Flux radius for Strehl calculation", "pixels", 3.0);
      cpl_parameterlist_append(parlist, par);

      par = cpl_parameter_new_value("eris.eris_ifu_stdstar.strehl_bkg-radius-low",
          CPL_TYPE_DOUBLE, "Inner radius for background calculation", "pixels", 5.0);
      cpl_parameterlist_append(parlist, par);

      par = cpl_parameter_new_value("eris.eris_ifu_stdstar.strehl_bkg-radius-high",
          CPL_TYPE_DOUBLE, "Outer radius for background calculation", "pixels", 7.0);
      cpl_parameterlist_append(parlist, par);

    cpl_frameset_delete(frameset);
    
    return parlist;
}


/* Helper function to cleanup test files */
static void cleanup_test_files(void) {
    remove("test_strehl.fits");
    remove("eris_ifu_strehl.fits");
    remove("test_dist.fits");
}

/* Test strehl computation */
static void test_strehl_compute(void) {
    cpl_frameset* frameset = create_test_frameset("PSF_CALIBRATOR");
    cpl_parameterlist* parlist = create_test_parameters();

    /* Test normal operation */
    cpl_error_code error = eris_ifu_strehl_compute(frameset, parlist, "eris.eris_ifu_stdstar");
    cpl_test_eq_error(CPL_ERROR_UNSPECIFIED, error);// TODO: understand returned error
    
    /* Test error handling */
    error = eris_ifu_strehl_compute(NULL, parlist, "eris.eris_ifu_stdstar");
    cpl_test_eq_error(CPL_ERROR_NULL_INPUT, error);
    cpl_errorstate_set(CPL_ERROR_NONE);
    
    error = eris_ifu_strehl_compute(frameset, NULL, "eris.eris_ifu_stdstar");
    cpl_test_eq_error(CPL_ERROR_NULL_INPUT, error);
    cpl_errorstate_set(CPL_ERROR_NONE);
    
    error = eris_ifu_strehl_compute(frameset, parlist, NULL);
    cpl_test_eq_error(CPL_ERROR_NULL_INPUT, error);
    cpl_errorstate_set(CPL_ERROR_NONE);
    
    /* Cleanup */
    cpl_frameset_delete(frameset);
    cpl_parameterlist_delete(parlist);
    cleanup_test_files();
}

/* Test stdstar strehl computation */
static void test_stdstar_strehl_compute(void) {
    cpl_frameset* frameset = create_test_frameset("PSF_CALIBRATOR");
    cpl_parameterlist* parlist = create_test_parameters();

    /* Test normal operation */
    cpl_error_code error = eris_ifu_stdstar_strehl_compute(frameset, parlist, "eris.eris_ifu_stdstar");
    cpl_test_eq_error(CPL_ERROR_DATA_NOT_FOUND, error); // TODO: strange return status
    
    /* Test error handling */
    error = eris_ifu_stdstar_strehl_compute(NULL, parlist, "eris.eris_ifu_stdstar");
    cpl_test_eq_error(CPL_ERROR_NULL_INPUT, error);
    cpl_errorstate_set(CPL_ERROR_NONE);
    
    error = eris_ifu_stdstar_strehl_compute(frameset, NULL, "eris.eris_ifu_stdstar");
    cpl_test_eq_error(CPL_ERROR_NULL_INPUT, error);
    cpl_errorstate_set(CPL_ERROR_NONE);
    
    error = eris_ifu_stdstar_strehl_compute(frameset, parlist, NULL);
    cpl_test_eq_error(CPL_ERROR_NULL_INPUT, error);
    cpl_errorstate_set(CPL_ERROR_NONE);
    
    /* Cleanup */
    cpl_frameset_delete(frameset);
    cpl_parameterlist_delete(parlist);
    cleanup_test_files();
}

int main(void) {
    cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_WARNING);
    
    test_strehl_compute();
    test_stdstar_strehl_compute(); // TODO strange return status
    
    return cpl_test_end(0);
}
