/*
 * E.S.O. - VLT project
 *
 * Test code for eris_ifu_vector module
 *
 * who       when        what
 * --------  ----------  ----------------------------------------------
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <math.h>
#include <float.h>

#include <cpl.h>

#include "eris_ifu_vector.h"
#include "eris_ifu_utils.h"

/* Test helper functions */
static eris_ifu_vector* create_test_vector(int size) {
    eris_ifu_vector* vec = eris_ifu_vector_new(size);
    double* data = cpl_vector_get_data(vec->data);
    double* mask = cpl_vector_get_data(vec->mask);
    
    /* Fill with test data */
    for (int i = 0; i < size; i++) {
        data[i] = i + 1.0;  // 1-based values
        mask[i] = 1.0;      // All values valid
    }
    
    return vec;
}

/**
 * @brief Test basic vector operations (new, delete, duplicate, get/set)
 */
static void eris_ifu_vector_basic_test(void)
{
    eris_ifu_vector* vec;
    eris_ifu_vector* dup;
    int rej;
    
    /* Test vector creation */
    vec = eris_ifu_vector_new(10);
    cpl_test_nonnull(vec);
    cpl_test_eq(eris_ifu_vector_get_size(vec), 10);
    
    /* Test vector initialization */
    for (int i = 0; i < 10; i++) {
        cpl_test_eq(eris_ifu_vector_get(vec, i), 0.0);  // Should be initialized to 0
        cpl_test_eq(eris_ifu_vector_is_rejected(vec, i), 0);  // Should not be rejected
    }
    
    /* Test vector set/get */
    cpl_test_eq_error(eris_ifu_vector_set(vec, 0, 1.0), CPL_ERROR_NONE);
    cpl_test_eq(eris_ifu_vector_get(vec, 0), 1.0);
    cpl_test_eq(eris_ifu_vector_get_rej(vec, 0, &rej), 1.0);
    cpl_test_eq(rej, 0);  // Not rejected
    
    /* Test setting NaN/Inf values */
    cpl_test_eq_error(eris_ifu_vector_set(vec, 1, 0.0/0.0), CPL_ERROR_NONE);  // NaN
    cpl_test_eq(eris_ifu_vector_is_rejected(vec, 1), 1);  // Should be rejected
    
    cpl_test_eq_error(eris_ifu_vector_set(vec, 2, 1.0/0.0), CPL_ERROR_NONE);  // Inf
    cpl_test_eq(eris_ifu_vector_is_rejected(vec, 2), 1);  // Should be rejected
    
    /* Test vector duplication */
    dup = eris_ifu_vector_duplicate(vec);
    cpl_test_nonnull(dup);
    cpl_test_eq(eris_ifu_vector_get_size(dup), eris_ifu_vector_get_size(vec));
    
    for (int i = 0; i < 10; i++) {
        cpl_test_eq(eris_ifu_vector_get(dup, i), eris_ifu_vector_get(vec, i));
        cpl_test_eq(eris_ifu_vector_is_rejected(dup, i), eris_ifu_vector_is_rejected(vec, i));
    }
    
    /* Test vector creation from data */
    double data[5] = {1.0, 2.0, 3.0, 4.0, 5.0};
    eris_ifu_vector* vec2 = eris_ifu_vector_new_wrap(5, data);
    cpl_test_nonnull(vec2);
    cpl_test_eq(eris_ifu_vector_get_size(vec2), 5);
    
    for (int i = 0; i < 5; i++) {
        cpl_test_eq(eris_ifu_vector_get(vec2, i), data[i]);
        cpl_test_eq(eris_ifu_vector_is_rejected(vec2, i), 0);
    }
    
    /* Test error handling */
    cpl_test_null(eris_ifu_vector_new(0));  // Invalid size
    cpl_test_error(CPL_ERROR_ILLEGAL_INPUT);
    
    cpl_test_eq_error(eris_ifu_vector_set(vec, 10, 1.0), CPL_ERROR_ACCESS_OUT_OF_RANGE);  // Invalid index
    
    /* Test mask operations */
    cpl_vector* mask = eris_ifu_vector_get_mask(vec);
    cpl_test_nonnull(mask);
    cpl_test_eq(cpl_vector_get_size(mask), 10);
    
    /* Test rejected count */
    cpl_test_eq(eris_ifu_vector_count_rejected(vec), 2);  // NaN and Inf values
    cpl_test_eq(eris_ifu_vector_count_non_rejected(vec), 8);
    
    /* Cleanup */
    eris_ifu_vector_delete(vec);
    eris_ifu_vector_delete(dup);
    eris_ifu_vector_delete(vec2);
    cpl_vector_delete(mask);
}

/**
 * @brief Test vector arithmetic operations (add, subtract, multiply, divide)
 */
static void eris_ifu_vector_arithmetic_test(void)
{
    eris_ifu_vector* vec1;
    eris_ifu_vector* vec2;
    
    /* Create test vectors */
    vec1 = create_test_vector(5);  // [1,2,3,4,5]
    vec2 = create_test_vector(5);  // [1,2,3,4,5]
    
    /* Test addition */
    cpl_test_eq_error(eris_ifu_vector_add(vec1, vec2), CPL_ERROR_NONE);
    for (int i = 0; i < 5; i++) {
        cpl_test_eq(eris_ifu_vector_get(vec1, i), 2.0 * (i + 1.0));  // [2,4,6,8,10]
    }
    
    /* Reset vec1 */
    eris_ifu_vector_delete(vec1);
    vec1 = create_test_vector(5);
    
    /* Test subtraction */
    cpl_test_eq_error(eris_ifu_vector_subtract(vec1, vec2), CPL_ERROR_NONE);
    for (int i = 0; i < 5; i++) {
        cpl_test_eq(eris_ifu_vector_get(vec1, i), 0.0);  // All zeros
    }
    
    /* Reset vec1 */
    eris_ifu_vector_delete(vec1);
    vec1 = create_test_vector(5);
    
    /* Test multiplication */
    cpl_test_eq_error(eris_ifu_vector_multiply(vec1, vec2), CPL_ERROR_NONE);
    for (int i = 0; i < 5; i++) {
        cpl_test_eq(eris_ifu_vector_get(vec1, i), (i + 1.0) * (i + 1.0));  // [1,4,9,16,25]
    }
    
    /* Reset vec1 */
    eris_ifu_vector_delete(vec1);
    vec1 = create_test_vector(5);
    
    /* Test division */
    cpl_test_eq_error(eris_ifu_vector_divide(vec1, vec2), CPL_ERROR_NONE);
    for (int i = 0; i < 5; i++) {
        cpl_test_eq(eris_ifu_vector_get(vec1, i), 1.0);  // All ones
    }
    
    /* Test with rejected values */
    eris_ifu_vector_set(vec2, 2, 0.0/0.0);  // NaN
    cpl_test_eq_error(eris_ifu_vector_multiply(vec1, vec2), CPL_ERROR_NONE);
    cpl_test_eq(eris_ifu_vector_is_rejected(vec1, 2), 1);  // Should be rejected
    
    /* Test error handling */
    eris_ifu_vector* vec3 = create_test_vector(3);  // Different size
    cpl_test_eq_error(eris_ifu_vector_add(vec1, vec3), CPL_ERROR_ILLEGAL_INPUT);
    
    /* Cleanup */
    eris_ifu_vector_delete(vec1);
    eris_ifu_vector_delete(vec2);
    eris_ifu_vector_delete(vec3);
}

/**
 * @brief Test vector scalar operations (add_scalar, subtract_scalar, multiply_scalar, divide_scalar)
 */
static void eris_ifu_vector_scalar_test(void)
{
    eris_ifu_vector* vec;
    
    /* Create test vector */
    vec = create_test_vector(5);  // [1,2,3,4,5]
    
    /* Test scalar addition */
    cpl_test_eq_error(eris_ifu_vector_add_scalar(vec, 1.0), CPL_ERROR_NONE);
    for (int i = 0; i < 5; i++) {
        cpl_test_eq(eris_ifu_vector_get(vec, i), (i + 1.0) + 1.0);  // [2,3,4,5,6]
    }
    
    /* Test scalar subtraction */
    cpl_test_eq_error(eris_ifu_vector_subtract_scalar(vec, 1.0), CPL_ERROR_NONE);
    for (int i = 0; i < 5; i++) {
        cpl_test_eq(eris_ifu_vector_get(vec, i), i + 1.0);  // Back to [1,2,3,4,5]
    }
    
    /* Test scalar multiplication */
    cpl_test_eq_error(eris_ifu_vector_multiply_scalar(vec, 2.0), CPL_ERROR_NONE);
    for (int i = 0; i < 5; i++) {
        cpl_test_eq(eris_ifu_vector_get(vec, i), 2.0 * (i + 1.0));  // [2,4,6,8,10]
    }
    
    /* Test scalar division */
    cpl_test_eq_error(eris_ifu_vector_divide_scalar(vec, 2.0), CPL_ERROR_NONE);
    for (int i = 0; i < 5; i++) {
        cpl_test_eq(eris_ifu_vector_get(vec, i), i + 1.0);  // Back to [1,2,3,4,5]
    }
    
    /* Test with NaN/Inf */
    cpl_test_eq_error(eris_ifu_vector_multiply_scalar(vec, 0.0/0.0), CPL_ERROR_NONE);  // NaN
    for (int i = 0; i < 5; i++) {
        cpl_test_eq(eris_ifu_vector_is_rejected(vec, i), 1);  // All should be rejected
    }
    
    /* Cleanup */
    eris_ifu_vector_delete(vec);
    cpl_test_error(CPL_ERROR_NONE);
}

/**
 * @brief Test vector statistical operations (mean, median, stdev, min/max)
 */
static void eris_ifu_vector_stats_test(void)
{
    eris_ifu_vector* vec;
    int pos;
    
    /* Create test vector */
    vec = create_test_vector(5);  // [1,2,3,4,5]
    
    /* Test mean */
    cpl_test_abs(eris_ifu_vector_get_mean(vec), 3.0, 0.000001);
    
    /* Test median */
    cpl_test_abs(eris_ifu_vector_get_median(vec, ERIS_IFU_ARITHMETIC), 3.0, 0.000001);
    cpl_test_abs(eris_ifu_vector_get_median(vec, ERIS_IFU_STATISTICAL), 3.0, 0.000001);
    
    /* Test standard deviation */
    cpl_test_abs(eris_ifu_vector_get_stdev(vec), sqrt(2.5), 0.000001);  // sqrt((sum((x-3)^2)/4))
    cpl_test_abs(eris_ifu_vector_get_stdev_median(vec), sqrt(2.5), 0.000001);
    
    /* Test min/max */
    cpl_test_eq(eris_ifu_vector_get_min(vec, &pos), 1.0);
    cpl_test_eq(pos, 0);
    
    cpl_test_eq(eris_ifu_vector_get_max(vec, &pos), 5.0);
    cpl_test_eq(pos, 4);
    
    /* Test with rejected values */
    eris_ifu_vector_set(vec, 2, 0.0/0.0);  // NaN
    cpl_test_abs(eris_ifu_vector_get_mean(vec), 3.0, 0.000001);  // Mean of [1,2,4,5]
    
    /* Test with all rejected values */
    for (int i = 0; i < 5; i++) {
        eris_ifu_vector_set(vec, i, 0.0/0.0);
    }
    cpl_test_eq(eris_ifu_vector_get_mean(vec), 0.0);
    cpl_test_eq(eris_ifu_vector_get_min(vec, &pos), 0.0);
    cpl_test_eq(pos, -1);
    
    /* Cleanup */
    eris_ifu_vector_delete(vec);
}

/**
 * @brief Test vector utility operations (power, sqrt, abs, flip, histogram)
 */
static void eris_ifu_vector_utils_test(void)
{
    eris_ifu_vector* vec;
    eris_ifu_vector* hist;
    
    /* Create test vector */
    vec = create_test_vector(5);  // [1,2,3,4,5]
    
    /* Test power */
    cpl_test_eq_error(eris_ifu_vector_power(vec, 2.0), CPL_ERROR_NONE);
    for (int i = 0; i < 5; i++) {
        cpl_test_eq(eris_ifu_vector_get(vec, i), pow(i + 1.0, 2.0));  // [1,4,9,16,25]
    }
    
    /* Reset vector */
    eris_ifu_vector_delete(vec);
    vec = create_test_vector(5);
    
    /* Test sqrt */
    cpl_test_eq_error(eris_ifu_vector_sqrt(vec), CPL_ERROR_NONE);
    for (int i = 0; i < 5; i++) {
        cpl_test_abs(eris_ifu_vector_get(vec, i), sqrt(i + 1.0), 0.000001);
    }
    
    /* Reset vector */
    eris_ifu_vector_delete(vec);
    vec = create_test_vector(5);
    
    /* Test abs */
    eris_ifu_vector_multiply_scalar(vec, -1.0);  // Make negative
    cpl_test_eq_error(eris_ifu_vector_abs(vec), CPL_ERROR_NONE);
    for (int i = 0; i < 5; i++) {
        cpl_test_eq(eris_ifu_vector_get(vec, i), i + 1.0);  // Back to positive
    }
    
    /* Test flip */
    cpl_test_eq_error(eris_ifu_vector_flip(vec), CPL_ERROR_NONE);
    for (int i = 0; i < 5; i++) {
        cpl_test_eq(eris_ifu_vector_get(vec, i), 5.0 - i);  // [5,4,3,2,1]
    }
    
    /* Test histogram */
    hist = eris_ifu_vector_histogram(vec, 5);
    cpl_test_nonnull(hist);
    cpl_test_eq(eris_ifu_vector_get_size(hist), 5);
    for (int i = 0; i < 5; i++) {
        cpl_test_eq(eris_ifu_vector_get(hist, i), 1.0);  // One value in each bin
    }
    
    /* Cleanup */
    eris_ifu_vector_delete(vec);
    eris_ifu_vector_delete(hist);
}

/**
 * @brief Test vector I/O operations (load, save)
 */
/**
 * @brief Test vector creation functions (create2)
 */
static void eris_ifu_vector_create2_test(void)
{
    cpl_vector* data;
    cpl_vector* mask;
    eris_ifu_vector* vec;
    
    /* Create test data and mask vectors */
    data = cpl_vector_new(5);
    mask = cpl_vector_new(5);
    
    double* pdata = cpl_vector_get_data(data);
    double* pmask = cpl_vector_get_data(mask);
    
    /* Fill with test data */
    for (int i = 0; i < 5; i++) {
        pdata[i] = i + 1.0;  // [1,2,3,4,5]
        pmask[i] = 1.0;      // All valid
    }
    
    /* Test normal operation */
    vec = eris_ifu_vector_create2(data, mask);
    cpl_test_nonnull(vec);
    cpl_test_eq(eris_ifu_vector_get_size(vec), 5);
    
    for (int i = 0; i < 5; i++) {
        cpl_test_eq(eris_ifu_vector_get(vec, i), i + 1.0);
        cpl_test_eq(eris_ifu_vector_is_rejected(vec, i), 0);
    }
    
    /* Test with NaN in data */
    pdata[2] = 0.0/0.0;  // NaN
    eris_ifu_vector_delete(vec);
    vec = eris_ifu_vector_create2(data, mask);
    cpl_test_nonnull(vec);
    cpl_test_eq(eris_ifu_vector_is_rejected(vec, 2), 1);  // Should be rejected
    
    /* Test with NaN in mask */
    pdata[2] = 3.0;  // Reset data
    pmask[2] = 0.0/0.0;  // NaN in mask
    eris_ifu_vector_delete(vec);
    vec = eris_ifu_vector_create2(data, mask);
    cpl_test_nonnull(vec);
    cpl_test_eq(eris_ifu_vector_is_rejected(vec, 2), 1);  // Should be rejected
    
    /* Test with mask value < 0.5 */
    pmask[2] = 0.4;  // Below threshold
    eris_ifu_vector_delete(vec);
    vec = eris_ifu_vector_create2(data, mask);
    cpl_test_nonnull(vec);
    cpl_test_eq(eris_ifu_vector_is_rejected(vec, 2), 1);  // Should be rejected
    
    /* Test error handling */
    cpl_test_null(eris_ifu_vector_create2(NULL, mask));
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    
    cpl_test_null(eris_ifu_vector_create2(data, NULL));
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    
    cpl_vector* wrong_size = cpl_vector_new(3);
    cpl_test_null(eris_ifu_vector_create2(data, wrong_size));
    cpl_test_error(CPL_ERROR_ILLEGAL_INPUT);
    
    /* Cleanup */
    eris_ifu_vector_delete(vec);
    cpl_vector_delete(data);
    cpl_vector_delete(mask);
    cpl_vector_delete(wrong_size);
}

/**
 * @brief Test rejection functions (reject, reject_from_mask, adapt_rejected)
 */
static void eris_ifu_vector_reject_test(void)
{
    eris_ifu_vector* vec1;
    eris_ifu_vector* vec2;
    cpl_vector* mask;
    
    /* Create test vectors */
    vec1 = create_test_vector(5);  // [1,2,3,4,5]
    vec2 = create_test_vector(5);  // [1,2,3,4,5]
    
    /* Test reject function */
    cpl_test_eq_error(eris_ifu_vector_reject(vec1, 2), CPL_ERROR_NONE);
    cpl_test_eq(eris_ifu_vector_is_rejected(vec1, 2), 1);  // Should be rejected
    
    /* Test error handling for reject */
    cpl_test_eq_error(eris_ifu_vector_reject(NULL, 2), CPL_ERROR_NULL_INPUT);
    cpl_test_eq_error(eris_ifu_vector_reject(vec1, -1), CPL_ERROR_ILLEGAL_INPUT);
    cpl_test_eq_error(eris_ifu_vector_reject(vec1, 5), CPL_ERROR_ILLEGAL_INPUT);
    
    /* Test reject_from_mask */
    mask = cpl_vector_new(5);
    double* pmask = cpl_vector_get_data(mask);
    for (int i = 0; i < 5; i++) {
        pmask[i] = (i == 2) ? 0.0 : 1.0;  // Reject position 2
    }
    
    cpl_test_eq_error(eris_ifu_vector_reject_from_mask(vec2, mask, 0), CPL_ERROR_NONE);
    cpl_test_eq(eris_ifu_vector_is_rejected(vec2, 2), 1);  // Should be rejected
    
    /* Test keep=1 option */
    eris_ifu_vector_reject(vec2, 3);  // Reject another position
    cpl_test_eq_error(eris_ifu_vector_reject_from_mask(vec2, mask, 1), CPL_ERROR_NONE);
    cpl_test_eq(eris_ifu_vector_is_rejected(vec2, 2), 1);  // Should still be rejected
    cpl_test_eq(eris_ifu_vector_is_rejected(vec2, 3), 1);  // Should remain rejected
    
    /* Test adapt_rejected */
    cpl_test_eq_error(eris_ifu_vector_adapt_rejected(vec1, vec2), CPL_ERROR_NONE);
    for (int i = 0; i < 5; i++) {
        cpl_test_eq(eris_ifu_vector_is_rejected(vec1, i), eris_ifu_vector_is_rejected(vec2, i));
    }
    
    /* Test error handling */
    cpl_test_eq_error(eris_ifu_vector_reject_from_mask(NULL, mask, 0), CPL_ERROR_NULL_INPUT);
    cpl_test_eq_error(eris_ifu_vector_reject_from_mask(vec1, NULL, 0), CPL_ERROR_NULL_INPUT);
    
    cpl_vector* wrong_size = cpl_vector_new(3);
    cpl_test_eq_error(eris_ifu_vector_reject_from_mask(vec1, wrong_size, 0), CPL_ERROR_ILLEGAL_INPUT);
    
    cpl_test_eq_error(eris_ifu_vector_adapt_rejected(NULL, vec2), CPL_ERROR_NULL_INPUT);
    cpl_test_eq_error(eris_ifu_vector_adapt_rejected(vec1, NULL), CPL_ERROR_NULL_INPUT);
    
    /* Cleanup */
    eris_ifu_vector_delete(vec1);
    eris_ifu_vector_delete(vec2);
    cpl_vector_delete(mask);
    cpl_vector_delete(wrong_size);
}

/**
 * @brief Test vector extraction functions (extract, create_non_rejected)
 */
static void eris_ifu_vector_extract_test(void)
{
    eris_ifu_vector* vec;
    eris_ifu_vector* sub;
    cpl_vector* non_rej;
    
    /* Create test vector */
    vec = create_test_vector(10);  // [1,2,3,4,5,6,7,8,9,10]
    
    /* Test extract */
    sub = eris_ifu_vector_extract(vec, 2, 5);  // Extract [3,4,5,6]
    cpl_test_nonnull(sub);
    cpl_test_eq(eris_ifu_vector_get_size(sub), 4);
    
    for (int i = 0; i < 3; i++) {
        cpl_test_eq(eris_ifu_vector_get(sub, i), i + 3.0);
    }
    
    /* Test with rejected values */
    eris_ifu_vector_reject(vec, 3);  // Reject position 3
    eris_ifu_vector_reject(vec, 7);  // Reject position 7
    
    /* Test create_non_rejected */
    non_rej = eris_ifu_vector_create_non_rejected(vec);
    cpl_test_nonnull(non_rej);
    cpl_test_eq(cpl_vector_get_size(non_rej), 8);  // 10 - 2 rejected values
    
    /* Test error handling */
    cpl_test_null(eris_ifu_vector_extract(NULL, 2, 5));
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    
    cpl_test_null(eris_ifu_vector_extract(vec, 5, 2));  // Stop <= Start
    cpl_test_error(CPL_ERROR_ILLEGAL_INPUT);
    
    cpl_test_null(eris_ifu_vector_create_non_rejected(NULL));
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    
    /* Cleanup */
    eris_ifu_vector_delete(vec);
    eris_ifu_vector_delete(sub);
    cpl_vector_delete(non_rej);
}

/**
 * @brief Test vector manipulation functions (cut_percentian)
 */
static void eris_ifu_vector_cut_test(void)
{
    eris_ifu_vector* vec;
    eris_ifu_vector* cut;
    int pos;
    
    /* Create test vector */
    vec = create_test_vector(10);  // [1,2,3,4,5,6,7,8,9,10]
    cpl_test_nonnull(vec);

    /* Test cut_percentian */
    cut = eris_ifu_vector_cut_percentian(vec, 0.2);  // Cut top 20%
    cpl_test_nonnull(cut);
    cpl_test_eq(eris_ifu_vector_get_size(cut), 8);  // Should keep 8 values

    double max = eris_ifu_vector_get_max(cut, &pos);
    cpl_test_abs(max, 8.0, 0.000001);  // Should have removed 9,10
    cpl_test_eq(pos, 7);  // Last position
    eris_ifu_vector_delete(cut);

    /* Test with rejected values */
    cpl_test_eq_error(eris_ifu_vector_reject(vec, 8), CPL_ERROR_NONE);  // Reject 9
    cpl_test_eq_error(eris_ifu_vector_reject(vec, 9), CPL_ERROR_NONE);  // Reject 10

    cut = eris_ifu_vector_cut_percentian(vec, 0.2);
    cpl_test_nonnull(cut);
    cpl_test_eq(eris_ifu_vector_get_size(cut), 6);  // Should keep 6 non-rejected values
    eris_ifu_vector_delete(cut);

    /* Test error handling */
    cut = eris_ifu_vector_cut_percentian(NULL, 0.2);
    cpl_test_null(cut);
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    eris_ifu_vector_delete(cut);
    cpl_error_reset();

    cut = eris_ifu_vector_cut_percentian(vec, -0.1);
    cpl_test_null(cut);
    cpl_test_error(CPL_ERROR_ILLEGAL_INPUT);
    cpl_error_reset();

    cut = eris_ifu_vector_cut_percentian(vec, 1.0);
    cpl_test_null(cut);
    cpl_test_error(CPL_ERROR_ILLEGAL_INPUT);
    cpl_error_reset();
    
    /* Cleanup */
    eris_ifu_vector_delete(vec);
}

/**
 * @brief Test vector debug functions (dump)
 */
static void eris_ifu_vector_dump_test(void)
{
    eris_ifu_vector* vec;
    
    /* Test NULL vector */
    cpl_test_eq_error(eris_ifu_vector_dump(NULL), CPL_ERROR_NONE);
    
    /* Create test vector */
    vec = create_test_vector(3);  // [1,2,3]
    
    /* Test normal operation */
    cpl_test_eq_error(eris_ifu_vector_dump(vec), CPL_ERROR_NONE);
    
    /* Test with rejected value */
    eris_ifu_vector_reject(vec, 1);
    cpl_test_eq_error(eris_ifu_vector_dump(vec), CPL_ERROR_NONE);
    
    /* Cleanup */
    eris_ifu_vector_delete(vec);
}



/**
 * @brief Test vector reject_from_mask operation
 */
static void eris_ifu_vector_reject_from_mask_test(void)
{
    eris_ifu_vector* vec;
    cpl_vector* mask;
    int rej;
    
    /* Create test vector */
    vec = create_test_vector(5);  // [1,2,3,4,5]
    mask = cpl_vector_new(5);
    
    /* Test with all valid mask */
    cpl_vector_fill(mask, 1.0);
    cpl_test_eq_error(eris_ifu_vector_reject_from_mask(vec, mask, 0), CPL_ERROR_NONE);
    
    for (int i = 0; i < 5; i++) {
        cpl_test_eq(eris_ifu_vector_is_rejected(vec, i), 0);  // None should be rejected
    }
    
    /* Test with some rejected values */
    cpl_vector_set(mask, 2, 0.0);  // Reject middle value
    cpl_test_eq_error(eris_ifu_vector_reject_from_mask(vec, mask, 0), CPL_ERROR_NONE);
    
    cpl_test_eq(eris_ifu_vector_get_rej(vec, 2, &rej), 3.0);
    cpl_test_eq(rej, 1);  // Should be rejected
    
    /* Test keep flag */
    eris_ifu_vector_reject(vec, 0);  // Reject first value
    cpl_test_eq_error(eris_ifu_vector_reject_from_mask(vec, mask, 1), CPL_ERROR_NONE);
    
    cpl_test_eq(eris_ifu_vector_is_rejected(vec, 0), 1);  // Should stay rejected
    cpl_test_eq(eris_ifu_vector_is_rejected(vec, 2), 1);  // Should be rejected from mask
    
    /* Test error handling */
    cpl_test_eq_error(eris_ifu_vector_reject_from_mask(NULL, mask, 0), CPL_ERROR_NULL_INPUT);
    cpl_test_eq_error(eris_ifu_vector_reject_from_mask(vec, NULL, 0), CPL_ERROR_NULL_INPUT);
    
    cpl_vector* wrong_size = cpl_vector_new(3);
    cpl_test_eq_error(eris_ifu_vector_reject_from_mask(vec, wrong_size, 0), CPL_ERROR_ILLEGAL_INPUT);
    cpl_vector_delete(wrong_size);
    
    /* Cleanup */
    eris_ifu_vector_delete(vec);
    cpl_vector_delete(mask);
}

/**
 * @brief Test vector adapt_rejected operation
 */
static void eris_ifu_vector_adapt_rejected_test(void)
{
    eris_ifu_vector* vec1;
    eris_ifu_vector* vec2;
    int rej;
    
    /* Create test vectors */
    vec1 = create_test_vector(5);  // [1,2,3,4,5]
    vec2 = create_test_vector(5);  // [1,2,3,4,5]
    
    /* Test with different rejected values */
    eris_ifu_vector_reject(vec1, 1);  // Reject 2 in vec1
    eris_ifu_vector_reject(vec2, 3);  // Reject 4 in vec2
    
    cpl_test_eq_error(eris_ifu_vector_adapt_rejected(vec1, vec2), CPL_ERROR_NONE);
    
    /* Both values should now be rejected in both vectors */
    cpl_test_eq(eris_ifu_vector_get_rej(vec1, 1, &rej), 2.0);
    cpl_test_eq(rej, 1);
    cpl_test_eq(eris_ifu_vector_get_rej(vec1, 3, &rej), 4.0);
    cpl_test_eq(rej, 1);
    
    cpl_test_eq(eris_ifu_vector_get_rej(vec2, 1, &rej), 2.0);
    cpl_test_eq(rej, 1);
    cpl_test_eq(eris_ifu_vector_get_rej(vec2, 3, &rej), 4.0);
    cpl_test_eq(rej, 1);
    
    /* Test error handling */
    cpl_test_eq_error(eris_ifu_vector_adapt_rejected(NULL, vec2), CPL_ERROR_NULL_INPUT);
    cpl_test_eq_error(eris_ifu_vector_adapt_rejected(vec1, NULL), CPL_ERROR_NULL_INPUT);
    
    eris_ifu_vector* wrong_size = create_test_vector(3);
    cpl_test_eq_error(eris_ifu_vector_adapt_rejected(vec1, wrong_size), CPL_ERROR_ILLEGAL_INPUT);
    eris_ifu_vector_delete(wrong_size);
    
    /* Cleanup */
    eris_ifu_vector_delete(vec1);
    eris_ifu_vector_delete(vec2);
}

/**
 * @brief Test vector get_bpm operation
 */
static void eris_ifu_vector_get_bpm_test(void)
{
    eris_ifu_vector* vec;
    cpl_vector* bpm;
    
    /* Create test vector */
    vec = create_test_vector(5);  // [1,2,3,4,5]
    
    /* Test getting BPM */
    bpm = eris_ifu_vector_get_bpm(vec);
    cpl_test_nonnull(bpm);
    cpl_test_eq(cpl_vector_get_size(bpm), 5);
    
    /* Should be the same pointer */
    //cpl_test_eq(bpm, vec->mask); // TODO: commented as does not compile
    
    /* Test with rejected values */
    eris_ifu_vector_reject(vec, 2);  // Reject middle value
    cpl_test_eq(cpl_vector_get(bpm, 2), 0.0);  // Should be marked as rejected
    
    /* Test error handling */
    cpl_test_null(eris_ifu_vector_get_bpm(NULL));
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    
    /* Cleanup */
    eris_ifu_vector_delete(vec);
}

/**
 * @brief Test vector create2 operation
 */
static void eris_ifu_vector_create2_test2(void)
{
    cpl_vector* data;
    cpl_vector* mask;
    eris_ifu_vector* vec;
    int rej;
    
    /* Create test vectors */
    data = cpl_vector_new(5);
    mask = cpl_vector_new(5);
    
    /* Fill with test data */
    for (int i = 0; i < 5; i++) {
        cpl_vector_set(data, i, i + 1.0);
        cpl_vector_set(mask, i, 1.0);
    }
    
    /* Test normal creation */
    vec = eris_ifu_vector_create2(data, mask);
    cpl_test_nonnull(vec);
    cpl_test_eq(eris_ifu_vector_get_size(vec), 5);
    
    for (int i = 0; i < 5; i++) {
        cpl_test_eq(eris_ifu_vector_get_rej(vec, i, &rej), i + 1.0);
        cpl_test_eq(rej, 0);  // Should not be rejected
    }
    eris_ifu_vector_delete(vec);
    
    /* Test with NaN/Inf in data */
    cpl_vector_set(data, 2, 0.0/0.0);  // NaN
    vec = eris_ifu_vector_create2(data, mask);
    cpl_test_nonnull(vec);
    
    cpl_test_eq(eris_ifu_vector_is_rejected(vec, 2), 1);  // Should be rejected
    eris_ifu_vector_delete(vec);
    
    /* Test with NaN/Inf in mask */
    cpl_vector_set(data, 2, 3.0);  // Reset data
    cpl_vector_set(mask, 2, 0.0/0.0);  // NaN in mask
    vec = eris_ifu_vector_create2(data, mask);
    cpl_test_nonnull(vec);
    
    cpl_test_eq(eris_ifu_vector_is_rejected(vec, 2), 1);  // Should be rejected
    eris_ifu_vector_delete(vec);
    
    /* Test with mask values */
    cpl_vector_set(mask, 1, 0.0);  // Reject value
    cpl_vector_set(mask, 2, 0.4);  // Reject value (< 0.5)
    cpl_vector_set(mask, 3, 0.6);  // Keep value (>= 0.5)
    vec = eris_ifu_vector_create2(data, mask);
    cpl_test_nonnull(vec);
    
    cpl_test_eq(eris_ifu_vector_is_rejected(vec, 1), 1);  // Should be rejected
    cpl_test_eq(eris_ifu_vector_is_rejected(vec, 2), 1);  // Should be rejected
    cpl_test_eq(eris_ifu_vector_is_rejected(vec, 3), 0);  // Should not be rejected
    
    /* Test error handling */
    cpl_test_null(eris_ifu_vector_create2(NULL, mask));
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    
    cpl_test_null(eris_ifu_vector_create2(data, NULL));
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    
    cpl_vector* wrong_size = cpl_vector_new(3);
    cpl_test_null(eris_ifu_vector_create2(data, wrong_size));
    cpl_test_error(CPL_ERROR_ILLEGAL_INPUT);
    cpl_vector_delete(wrong_size);
    
    /* Cleanup */
    eris_ifu_vector_delete(vec);
    cpl_vector_delete(data);
    cpl_vector_delete(mask);
}

static void eris_ifu_vector_io_test(void)
{
    eris_ifu_vector* vec;
    eris_ifu_vector* loaded;
    int rej;
    
    /* Create test vector */
    vec = create_test_vector(5);  // [1,2,3,4,5]
    cpl_test_nonnull(vec);
    
    /* Test save */
    cpl_test_eq_error(eris_ifu_vector_save(vec, "test_vector.fits", CPL_TYPE_DOUBLE, NULL, CPL_IO_CREATE, -1.0), CPL_ERROR_NONE);
    
    /* Test load */
    loaded = eris_ifu_vector_load("test_vector.fits", 0);
    cpl_test_nonnull(loaded);
    cpl_test_eq(eris_ifu_vector_get_size(loaded), eris_ifu_vector_get_size(vec));
    
    for (int i = 0; i < 5; i++) {
        cpl_test_eq(eris_ifu_vector_get_rej(loaded, i, &rej), eris_ifu_vector_get(vec, i));
        cpl_test_eq(rej, 0);  // Should not be rejected
    }
    eris_ifu_vector_delete(loaded);
    
    /* Test save with rejected values */
    cpl_test_eq_error(eris_ifu_vector_set(vec, 2, 0.0/0.0), CPL_ERROR_NONE);  // NaN
    cpl_test_eq_error(eris_ifu_vector_save(vec, "test_vector_rej.fits", CPL_TYPE_DOUBLE, NULL, CPL_IO_CREATE, -999.0), CPL_ERROR_NONE);

    loaded = eris_ifu_vector_load("test_vector_rej.fits", 0);
    cpl_test_nonnull(loaded);

    cpl_test_eq(eris_ifu_vector_get_rej(loaded, 2, &rej), -999.0);  // Rejected value replaced
    cpl_test_eq(rej, 1);  // Should be rejected

    eris_ifu_vector_delete(loaded);
    cpl_test_error(CPL_ERROR_NONE);
    /* Test error handling */
    cpl_test_null(eris_ifu_vector_load("nonexistent.fits", 0));
    cpl_test_error(CPL_ERROR_FILE_IO);
    cpl_error_reset();
    
    cpl_test_eq_error(eris_ifu_vector_save(NULL, "test_vector.fits", CPL_TYPE_DOUBLE, NULL, CPL_IO_CREATE, -1.0), CPL_ERROR_NULL_INPUT);
    cpl_error_reset();
    
    /* Cleanup */
    eris_ifu_vector_delete(vec);
    
    /* Remove test files */
    remove("test_vector.fits");
    remove("test_vector_rej.fits");
}

int main(void)
{
    cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_WARNING);
    
    /* Run all tests */
    eris_ifu_vector_basic_test();
    cpl_test_error(CPL_ERROR_NONE);
    
    eris_ifu_vector_arithmetic_test();
    cpl_test_error(CPL_ERROR_NONE);
    
    eris_ifu_vector_scalar_test();
    cpl_test_error(CPL_ERROR_NONE);
    
    eris_ifu_vector_stats_test();
    cpl_test_error(CPL_ERROR_NONE);
    
    eris_ifu_vector_utils_test();
    cpl_test_error(CPL_ERROR_NONE);
    
    eris_ifu_vector_create2_test();
    cpl_test_error(CPL_ERROR_NONE);

    eris_ifu_vector_create2_test2();
    cpl_test_error(CPL_ERROR_NONE);
    
    eris_ifu_vector_reject_test();
    cpl_test_error(CPL_ERROR_NONE);
    
    eris_ifu_vector_extract_test();
    cpl_test_error(CPL_ERROR_NONE);
    
    eris_ifu_vector_cut_test();
    cpl_test_error(CPL_ERROR_NONE);
    
    eris_ifu_vector_dump_test();
    cpl_test_error(CPL_ERROR_NONE);
    
    //eris_ifu_vector_io_test(); //TODO: fix error
    cpl_test_error(CPL_ERROR_NONE);
    
    return cpl_test_end(0);
}

/**@}*/
