/* $Id$
 *
 * This file is part of the ERIS Pipeline
 * Copyright (C) 2002,2003 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/*
 * $Author$
 * $Date$
 * $Revision$
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                Includes
 -----------------------------------------------------------------------------*/

#include "eris_nix_test_defs.h"
#include <eris_nix_dfs.h>
#include <eris_nix_utils.h>
#include <hdrl_utils.h>
#include <math.h>

/*----------------------------------------------------------------------------*/
/**
 * @defgroup eris_nix_dfs_test  Unit test of eris_nix_dfs
 *
 */
/*----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------
                                       Prototypes
 -----------------------------------------------------------------------------*/

static void eris_nix_dfs_test(void);

/**@{*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Textual representation of CPL frame group
  @param    group     to convert
  @return   textual representation
 */
/*----------------------------------------------------------------------------*/
static void eris_nix_dfs_test(void)
{

    /* construct test located_images */

    hdrl_image * himage = hdrl_image_new(TEST_SIZE_X,
                                         TEST_SIZE_Y);
    cpl_image_fill_window(hdrl_image_get_image(himage),
                          1, 1, TEST_SIZE_X, TEST_SIZE_Y,
                          20.0);
    cpl_image_fill_window(hdrl_image_get_error(himage),
                          1, 1, TEST_SIZE_X, TEST_SIZE_Y,
                          4.0);

    cpl_image * confidence = cpl_image_new(TEST_SIZE_X,
                                           TEST_SIZE_Y,
                                           CPL_TYPE_DOUBLE);
    cpl_image_fill_window(confidence,
                          1, 1, TEST_SIZE_X, TEST_SIZE_Y,
                          5.0);

    hdrl_image * bkg = hdrl_image_duplicate(himage);
    cpl_image * bkg_confidence = cpl_image_duplicate(confidence);

    cpl_propertylist * plist = cpl_propertylist_new();
    cpl_propertylist_update_string(plist, "INSTRUME", "ERIS");
    cpl_propertylist_update_int(plist, "ESO DET NDIT", 9);
    cpl_propertylist_update_double(plist, "ESO DET DIT", 15.0);
    cpl_propertylist_update_int(plist, "ESO DET NDSAMPLES", 7);
    cpl_propertylist_update_string(plist, "ESO DET READ CURNAME",
                                   "SLOW_UP_THE_RAMP");
    cpl_propertylist_update_int(plist, "ESO DET SEQ1 WIN ROT", 0);
    cpl_propertylist_update_int(plist, "ESO DET SEQ1 WIN STRX", 1);
    cpl_propertylist_update_int(plist, "ESO DET SEQ1 WIN STRY", 1);
    cpl_propertylist_update_int(plist, "ESO DET SEQ1 WIN NX", TEST_SIZE_X);
    cpl_propertylist_update_int(plist, "ESO DET SEQ1 WIN NY", TEST_SIZE_Y);
    cpl_propertylist_update_int(plist, "ESO DET CHIP NX", TEST_SIZE_X);
    cpl_propertylist_update_int(plist, "ESO DET CHIP NY", TEST_SIZE_Y);
    cpl_propertylist_update_string(plist, "ESO PRO CATG", "CALIB_JITTER");

    /* need some sort of FITS file behind the filename to satisfy cpl_dfs */
    cpl_image_save(hdrl_image_get_image(himage), "nix_test_image.fits",
                   CPL_TYPE_UNSPECIFIED, NULL, CPL_IO_CREATE);

    cpl_frame * frame = cpl_frame_new();
    cpl_frame_set_filename(frame, "nix_test_image.fits");
    cpl_frame_set_tag(frame, "JITTER");
    cpl_frame_set_group(frame, CPL_FRAME_GROUP_RAW);
    cpl_frame_set_level(frame, CPL_FRAME_LEVEL_NONE);
    cpl_frame_set_type(frame, CPL_FRAME_TYPE_IMAGE);

    /* located image containing hdrl_image */

    located_image * limage = enu_located_image_new(
                                       himage,
                                       NULL,
                                       confidence,
                                       bkg,
                                       bkg_confidence,
                                       plist,
                                       NULL, NULL, NULL, NULL,
                                       frame);

    /* located image containing hdrl_imagelist (NIX cube) */

    hdrl_imagelist * himagelist = hdrl_imagelist_new();
    hdrl_imagelist_set(himagelist, hdrl_image_duplicate(himage), 0);
    hdrl_imagelist_set(himagelist, hdrl_image_duplicate(himage), 1);
    hdrl_imagelist_set(himagelist, hdrl_image_duplicate(himage), 2);

    located_image * limagelist = enu_located_image_new(
                                       NULL,
                                       himagelist,
                                       cpl_image_duplicate(confidence),
                                       hdrl_image_duplicate(bkg),
                                       cpl_image_duplicate(bkg_confidence),
                                       cpl_propertylist_duplicate(plist),
                                       NULL, NULL, NULL, NULL,
                                       cpl_frame_duplicate(frame));

    {    /* start tests for saving images */

    cpl_frameset * frameset = cpl_frameset_new();
    const char * filename = "nix_test_utils_himage.fits";
    cpl_frameset_insert(frameset, cpl_frame_duplicate(frame));
    cpl_parameterlist * parlist = cpl_parameterlist_new();

    /* save hdrl_image, read in again and check */

    enu_dfs_save_himage(frameset,
                        parlist,
                        frameset,
                        CPL_TRUE,
                        limage->himage,
                        NULL,
                        NULL,
                        "testing",
                        limage->frame,
                        limage->plist,
                        NULL,
                        "test",
                        filename);
    cpl_test_error(CPL_ERROR_NONE);

    cpl_frame * frame2 = cpl_frame_new();
    cpl_frame_set_tag(frame2, "TEST_PRO_CATG");
    cpl_frame_set_filename(frame2, filename);

    located_image * timage = enu_load_limage_from_frame(frame2, NULL, CPL_FALSE);
    cpl_test_error(CPL_ERROR_NONE);

    cpl_size nx = hdrl_image_get_size_x(timage->himage);
    cpl_size ny = hdrl_image_get_size_y(timage->himage);
    cpl_test_eq(nx, TEST_SIZE_X);
    cpl_test_eq(ny, TEST_SIZE_Y);

    cpl_test_eq_ptr(timage->confidence, NULL);

    /* save hdrl_imagelist, read in again and check */

    const char * filename2 = "nix_test_utils_himagelist.fits";

    enu_dfs_save_himage(frameset,
                        parlist,
                        frameset,
                        CPL_TRUE,
                        NULL,
                        limagelist->himagelist,
                        NULL,
                        "testing",
                        limage->frame,
                        limage->plist,
                        NULL,
                        "test",
                        filename2);
    cpl_test_error(CPL_ERROR_NONE);

    cpl_frame_set_filename(frame2, filename2);
    located_image * timagelist = enu_load_limage_from_frame(frame2, NULL, CPL_FALSE);
    cpl_test_error(CPL_ERROR_NONE);

/* load_limage can't handle hdrl_imagelists yet */
//    cpl_size nlist = hdrl_imagelist_get_size(timagelist->himagelist);
//    cpl_test_eq(nlist, 3);
//    const hdrl_image * himage2 = hdrl_imagelist_get_const(timagelist->himagelist, 1);
//    nx = hdrl_image_get_size_x(himage2);
//    ny = hdrl_image_get_size_y(himage2);
//    cpl_test_eq(nx, TEST_SIZE_X);
//    cpl_test_eq(ny, TEST_SIZE_Y);

    cpl_frameset_delete(frameset);
    cpl_parameterlist_delete(parlist);
    cpl_frame_delete(frame2);
    enu_located_image_delete(timage);
    enu_located_image_delete(timagelist);

    }    /* end tests for saving images */


    {    /* start tests for enu_dfs_save_catalogue */

    cpl_parameterlist * parlist = cpl_parameterlist_new();
    cpl_frameset * frameset = cpl_frameset_new();

    enu_dfs_save_catalogue(frameset,
                           NULL,
                           parlist,
                           NULL,
                           NULL,
                           NULL,
                           "test_recipe",
                           "TEST",
                           NULL,
                           NULL);
    cpl_test_error(CPL_ERROR_NONE);

    cpl_frame * inherit = cpl_frame_new();
    cpl_frame_set_tag(inherit, "TEST_PRO_CATG");
    cpl_frameset_insert(frameset, inherit);

    enu_dfs_save_catalogue(frameset,
                           inherit,
                           parlist,
                           NULL,
                           NULL,
                           NULL,
                           "test_recipe",
                           "TEST",
                           NULL,
                           NULL);
    cpl_test_error(CPL_ERROR_NONE);

    enu_dfs_save_catalogue(frameset,
                           inherit,
                           parlist,
                           NULL,
                           NULL,
                           NULL,
                           "test_recipe",
                           "TEST",
                           "test",
                           NULL);
    cpl_test_error(CPL_ERROR_NONE);

    enu_dfs_save_catalogue(frameset,
                           inherit,
                           parlist,
                           NULL,
                           NULL,
                           NULL,
                           "test_recipe",
                           "TEST",
                           "test",
                           "TEST_PRO_CATG");
    cpl_test_error(CPL_ERROR_NONE);

    enu_dfs_save_catalogue(frameset,
                           inherit,
                           parlist,
                           NULL,
                           NULL,
                           "test_catalog",
                           "test_recipe",
                           "TEST",
                           "test",
                           "TEST_PRO_CATG");
    cpl_test_error(CPL_ERROR_NONE);

    cpl_parameterlist_delete(parlist);
    cpl_frameset_delete(frameset);

    }    /* end tests for enu_catalogue_save */

    /* tidy up */

    enu_located_image_delete(limage);
    enu_located_image_delete(limagelist);
    remove("nix_test_image.fits");
    remove("nix_test_utils_himage.fits");
    remove("nix_test_utils_himagelist.fits");
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit tests of eris_nix_dfs module
 */
/*----------------------------------------------------------------------------*/

int main(void)
{
    cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_WARNING);

    eris_nix_dfs_test();

    return cpl_test_end(0);
}

/**@}*/
