/* $Id$
 *
 * This file is part of the ERIS Pipeline
 * Copyright (C) 2002,2003 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/*
 * $Author$
 * $Date$
 * $Revision$
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                Includes
 -----------------------------------------------------------------------------*/

#include <math.h>
#include <eris_nix_dfs.h>
#include <eris_nix_master_flat.h>
#include <eris_nix_utils.h>
#include "eris_nix_test_defs.h"

/*----------------------------------------------------------------------------*/
/**
 * @defgroup eris_nix_master_flat_test  
 *           Unit test of eris_nix_master_flat
 *
 */
/*----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------
                                       Prototypes
 -----------------------------------------------------------------------------*/

static void eris_nix_master_flat_test(void);

/**@{*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Textual representation of CPL frame group
  @param    group     to convert
  @return   textual representation
 */
/*----------------------------------------------------------------------------*/
static void eris_nix_master_flat_test(void)
{
    master_flat        * check = NULL;
    cpl_image          * flat_data = NULL;
    cpl_image          * flat_error = NULL;
    cpl_frame          * frame1 = NULL;
    cpl_frame          * frame2 = NULL;
    cpl_frameset       * frameset = NULL;
    master_flat        * good_flat = NULL;
    mef_extension_list * mef_extensions = NULL;
    cpl_parameterlist  * parlist = NULL;
    cpl_propertylist   * plist = NULL;
    cpl_image          * rawdata = NULL;
    cpl_propertylist   * rawdata_plist = NULL;
    cpl_image          * testconf = NULL;
    hdrl_image         * testflat = NULL;
    cpl_mask           * testbpm = NULL;
    cpl_frameset       * used = NULL;

    good_flat = en_master_flat_create(NULL, NULL, NULL, NULL, NULL); 
    cpl_test(!good_flat);
    cpl_test_error(CPL_ERROR_NULL_INPUT);

    /* make test master flat */
  
    flat_data = cpl_image_new(TEST_SIZE_X, TEST_SIZE_Y, CPL_TYPE_DOUBLE);
    cpl_image_fill_window(flat_data, 1, 1, TEST_SIZE_X, TEST_SIZE_Y, 2.0);
    flat_error = cpl_image_new(TEST_SIZE_X, TEST_SIZE_Y, CPL_TYPE_DOUBLE);
    cpl_image_fill_window(flat_error, 1, 1, TEST_SIZE_X, TEST_SIZE_Y, 0.5);
    testflat = hdrl_image_create(flat_data, flat_error);
    testbpm = cpl_mask_new(TEST_SIZE_X, TEST_SIZE_Y);
    testconf = cpl_image_new(TEST_SIZE_X, TEST_SIZE_Y, CPL_TYPE_DOUBLE);
    cpl_image_fill_window(testconf, 1, 1, TEST_SIZE_X, TEST_SIZE_Y, 100.0);

    plist = cpl_propertylist_new();
    cpl_propertylist_append_string(plist, CPL_DFS_PRO_CATG,
                                   ERIS_NIX_MASTER_FLAT_LAMP_HIFREQ_PRO_CATG);
    cpl_propertylist_update_string(plist, "PRODCATG", "ANCILLARY.IMAGE");
    cpl_propertylist_update_string(plist, "INSTRUME", "ERIS");
    cpl_propertylist_update_int(plist, "ESO DET SEQ1 WIN NX", TEST_SIZE_X);
    cpl_propertylist_update_int(plist, "ESO DET SEQ1 WIN NY", TEST_SIZE_Y);
    cpl_propertylist_update_int(plist, "ESO DET SEQ1 WIN ROT", 0);
    cpl_propertylist_update_int(plist, "ESO DET SEQ1 WIN STRX", 1);
    cpl_propertylist_update_int(plist, "ESO DET SEQ1 WIN STRY", 1);
    cpl_propertylist_update_int(plist, "ESO DET CHIP NX", TEST_SIZE_X);
    cpl_propertylist_update_int(plist, "ESO DET CHIP NY", TEST_SIZE_Y);

    hdrl_value median = hdrl_image_get_median(testflat);
    cpl_propertylist_append_double(plist, "ESO QC FLAT MED",
                                   (double) median.data);
    hdrl_value mean = hdrl_image_get_mean(testflat);
    cpl_propertylist_append_double(plist, "ESO QC FLAT MEAN",
                                   (double) mean.data);
    double rms = hdrl_image_get_stdev(testflat);
    cpl_propertylist_append_double(plist, "ESO QC FLAT RMS", rms);
    cpl_size nbad = cpl_mask_count(testbpm); 
    cpl_propertylist_append_int(plist, "ESO QC NUMBER COLD PIXEL", (int)nbad);

    good_flat = en_master_flat_create(testflat, testbpm, testconf, 
                                      NULL, plist);

    /* save flat to FITS file */

    mef_extensions = enu_mef_extension_list_new(2);
    mef_extensions->mef[0] = enu_mef_new_mask("COLD_BPM", good_flat->cold_bpm,
                                              NULL);
    mef_extensions->mef[1] = enu_mef_new_image("CONFIDENCE", testconf, NULL);

    frameset  = cpl_frameset_new();
    frame1 = cpl_frame_new();
    rawdata =  cpl_image_new(TEST_SIZE_X, TEST_SIZE_Y, CPL_TYPE_DOUBLE);
    rawdata_plist = cpl_propertylist_new();
    cpl_image_save(rawdata, "nix_test_flat_lamp_hifreq_raw_data.fits",
                   CPL_TYPE_DOUBLE, rawdata_plist, CPL_IO_CREATE);
    cpl_frame_set_filename(frame1, "nix_test_flat_lamp_hifreq_raw_data.fits");
    cpl_frame_set_group(frame1, CPL_FRAME_GROUP_RAW);
    cpl_frame_set_level(frame1, CPL_FRAME_LEVEL_NONE);
    cpl_frame_set_tag(frame1, "FLAT_LAMP");
    cpl_frame_set_type(frame1, CPL_FRAME_TYPE_IMAGE);
    cpl_frameset_insert(frameset, frame1);

    parlist = cpl_parameterlist_new();

    /* Save the master flat to a DFS-compliant MEF file */

    enu_dfs_save_himage(frameset,
                        parlist,
                        frameset,
                        CPL_TRUE,
                        good_flat->flat,
                        NULL,
                        mef_extensions,
                        "eris_nix_flat_lamp",
                        frame1,
                        plist,
                        NULL,
                        PACKAGE "/" PACKAGE_VERSION,
                        "nix_test_master_flat_lamp_hifreq.fits");
    cpl_test_error(CPL_ERROR_NONE);

    /* Read it in again and check that it matches what was written out */

    frame2 = cpl_frame_new();
    cpl_frame_set_filename(frame2, "nix_test_master_flat_lamp_hifreq.fits");
    cpl_frame_set_tag(frame2, ERIS_NIX_MASTER_FLAT_LAMP_HIFREQ_PRO_CATG);
    cpl_frame_set_group(frame2, CPL_FRAME_GROUP_CALIB);
    cpl_frame_set_level(frame2, CPL_FRAME_LEVEL_NONE);
    cpl_frame_set_type(frame2, CPL_FRAME_TYPE_IMAGE);
    cpl_frameset_insert(frameset, frame2);

    used = cpl_frameset_new();
    int required = 1;
    check = en_master_flat_load_from_frameset(frameset,
                                       ERIS_NIX_MASTER_FLAT_LAMP_HIFREQ_PRO_CATG,
                                       used, required);

    hdrl_value val = hdrl_image_get_pixel(check->flat, 10, 10, NULL);
    cpl_test_leq(fabs(val.data - 2.0), DBL_EPSILON);
    cpl_test_leq(fabs(val.error - 0.5), DBL_EPSILON);

    const char * string1 = cpl_propertylist_get_string(check->plist,
                                                       CPL_DFS_PRO_CATG);
    cpl_test_eq_string(string1, ERIS_NIX_MASTER_FLAT_LAMP_HIFREQ_PRO_CATG);

    median = hdrl_image_get_median(check->flat);
    double dtest = cpl_propertylist_get_double(check->plist, "ESO QC FLAT MED");
    cpl_test_abs(dtest, (double) median.data, DBL_EPSILON);

    mean = hdrl_image_get_mean(check->flat);
    dtest = cpl_propertylist_get_double(check->plist,
                                        "ESO QC FLAT MEAN");
    cpl_test_abs(dtest, (double) mean.data, DBL_EPSILON);

    rms = hdrl_image_get_stdev(check->flat);
    dtest = cpl_propertylist_get_double(check->plist,
                                        "ESO QC FLAT RMS");
    cpl_test_abs(dtest, (double) rms, DBL_EPSILON);

    nbad = cpl_mask_count(check->cold_bpm); 
    int itest = cpl_propertylist_get_int(check->plist,
                                         "ESO QC NUMBER COLD PIXEL");
    cpl_test_abs(itest, (int) nbad, 0);

    double meanconf = cpl_image_get_mean(check->confidence);
    cpl_test_abs(100.0, (double) meanconf, DBL_EPSILON);

    /* tidy up */

    en_master_flat_delete(check);
    cpl_image_delete(flat_data);
    cpl_image_delete(flat_error);
    cpl_frameset_delete(frameset);
    en_master_flat_delete(good_flat);
    enu_mef_extension_list_delete(mef_extensions);
    cpl_parameterlist_delete(parlist);
    cpl_propertylist_delete(plist);
    cpl_image_delete(rawdata);
    cpl_propertylist_delete(rawdata_plist);
    cpl_image_delete(testconf);
    hdrl_image_delete(testflat);
    cpl_mask_delete(testbpm);
    cpl_frameset_delete(used);
    remove("nix_test_flat_lamp_hifreq_raw_data.fits");
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit tests of eris_nix_master_flat module
 */
/*----------------------------------------------------------------------------*/

int main(void)
{
    cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_WARNING);

    eris_nix_master_flat_test();

    return cpl_test_end(0);

}

/**@}*/
