from typing import Any, Dict

from cpl import core
from cpl import ui
from cpl import dfs
from cpl.core import Msg

# for specific python functions
import numpy as np
#import skimage as ski
from math import *
from astropy.io import fits
import scipy as sp

def area_overlap(x0, y0, r0, x1, y1, r1):
    
    # From, e.g., https://stackoverflow.com/questions/4247889/area-of-intersection-between-two-circles
    d = np.sqrt((x1 - x0)**2 + (y1 - y0)**2)
    
    if d > r1 + r0:
        return 0
    
    elif (d <= np.abs(r0 - r1)) and (r0 >= r1):
        return np.pi*r1**2
    
    elif d <= np.abs(r0 - r1) and (r0 < r1):
        return np.pi*r0**2
    
    else:
        phi = (np.arccos((r0**2 + d**2 - r1**2) / (2.0 * r0 * d))) * 2
        theta = (np.arccos((r1**2 + d**2 - r0**2) / (2.0 * r1 * d))) * 2
        area1 = 0.5 * theta * r1**2 - 0.5 * r1**2 * np.sin(theta)
        area2 = 0.5 * phi * r0**2 - 0.5 * r0**2 * np.sin(phi)
        
        return area1 + area2

def sam_mask(p, s, sam_xpos, sam_ypos, sam_size):
    
    # Create ideal SAM from list of hole coordinates relative to pupil center and
    # shift, rotation, and scale to transform from M1->pupil plane
    
    # p = [xshift, yshift, rotation, scale]
    
    model = np.zeros(s)
    xgrid, ygrid = np.meshgrid(np.arange(s[1]), np.arange(s[0]))
    
    x = sam_xpos*np.cos(p[2]) - sam_ypos*np.sin(p[2])
    y = sam_xpos*np.sin(p[2]) + sam_ypos*np.cos(p[2])
    
    x = (x*p[3]) + p[0]
    y = (y*p[3]) + p[1]
    r = sam_size * p[3]
    
    for i in range(0, len(sam_xpos)):
        rgrid = np.sqrt((xgrid-x[i])**2 + (ygrid-y[i])**2)
        model[np.where(rgrid <= r[i])] = 1.0
    
    return model


def measure_contam(p, image, model, sam_xpos, sam_ypos, sam_size):

    xgrid, ygrid = np.meshgrid(np.arange(image.shape[1]), np.arange(image.shape[0]))
    x = sam_xpos*np.cos(p[2]) - sam_ypos*np.sin(p[2])
    y = sam_xpos*np.sin(p[2]) + sam_ypos*np.cos(p[2])
    
    x = (x*p[3]) + p[0]
    y = (y*p[3]) + p[1]
    r = sam_size * p[3]
    margin=0
    kappa=2.4
    pup_id = []
    cenx = []
    ceny = []
    frac = []
	
    for i in range(0, len(sam_xpos)):
        pupil_tmp = np.copy(image)
        rgrid = np.sqrt((xgrid-x[i])**2 + (ygrid-y[i])**2)
        good = np.where(rgrid <= r[i]+margin)
        bad = np.where(rgrid > r[i]+margin)
        pupil_tmp[bad] = 0

        # max = np.nanmax(image[good])
        # min = np.nanmin(image[good])
        std = np.nanstd(image[good])
        med = np.nanmedian(image[good])
        ngood = np.sum(rgrid <= r[i]+margin)
        # thresh_min =  med - kappa * std
        thresh_max =  med + kappa * std
        # outliers =  np.where(image[good] > thresh_max )
        #pupil_tmp[outliers] = 0
        nbad = np.sum(image[good] > thresh_max)
        # pro_file='pupil_%2.2d.fits' % (i)
        # hdulist = fits.HDUList()
        # hdulist.append(fits.ImageHDU(data=pupil_tmp))
        # hdulist.writeto(pro_file, overwrite=True)
        # hdulist.close()


        # print('i:',i,'x:',sam_xpos[i],'y:',sam_ypos[i],'max:',max,'min:',min,'std:',std,'med:',med)
        if ngood > 1000 and nbad > 50:
        # if ngood > 1000 and nbad > 5:
            msk_tmp = np.copy(image)
            msk_tmp[bad] = 0
            msk_tmp[good] = 1
            
            y_indices, x_indices = np.indices(msk_tmp.shape)
            xc = np.sum(x_indices*msk_tmp)/np.sum(msk_tmp)
            yc = np.sum(y_indices*msk_tmp)/np.sum(msk_tmp)
            cen_new = [yc, xc]

            # print('thresh_min:',thresh_min,'thresh_max:',thresh_max,'nbad:',nbad,'ngood:',ngood)
            print(
                "mask: %2d centroid: [%4d,%4d] ngood: %5d nbad: %3d  nbad/ngood fraction: %3.4f"
                % (i, cen_new[1], cen_new[0], ngood, nbad, nbad / ngood)
            )
            pup_id.append(i)
            cenx.append(cen_new[1])
            ceny.append(cen_new[0])
            frac.append(nbad/ngood)


    # print('pup_id:',pup_id)
    return pup_id, cenx, ceny, frac


def sam_contamination(image, model, bpm, p, sam_xpos, sam_ypos, sam_size):
    bpm[:] = [i^1 for i in bpm[:]]
    debug = 'n'

    # basic image cleaning
    bad_pix = np.where(np.logical_or((bpm == 0), (image == np.nan),(image == np.inf)))
    image[bad_pix] = 0
 
    # further kappa-sigma cleaning
    std = np.nanstd(image)
    # mean = np.nanmean(image)
    median = np.nanmedian(image)
    # min = np.nanmin(image)
    # max = np.nanmax(image)
    thresh_lo = median - 5 * std
    thresh_up = median + 5 * std
    bad_pix = np.where(np.logical_or((image < thresh_lo),(image > thresh_up)))
    image[bad_pix] = 0

    
    # print('std',std,'mean',mean,'median',median)
    # print('min',min,'max',max)

    
    std = np.nanstd(image)
    mean = np.nanmean(image)
    median = np.nanmedian(image)
    # min = np.nanmin(image)
    # max = np.nanmax(image)
    # print('std',std,'mean',mean,'median',median)
    # print('min',min,'max',max)

    thresh_lo = median - 5 * std
    # thresh_up = median + 5 * std

    # clean only lower values
    bad_pix = np.where(image < thresh_lo)
    image[bad_pix] = 0
    # bad_pix = np.where(image > thresh_up)
    # image[bad_pix] = 0
    # print('std',std,'mean',mean,'median',median)
    # print('min',min,'max',max)

    # generate final mask to separate background from foreground
    bkg_pix = np.where(image < mean+std)
    image[bkg_pix] = 0
    good_pix = np.where(image > 0)
    label_mask = np.copy(image)
    # label_mask[good_pix] = 1

    # pro_file = "pupil_label_mask_sam.fits"
    # hdulist = fits.HDUList()
    # hdulist.append(fits.ImageHDU(data=label_mask))
    # hdulist.writeto(pro_file, overwrite=True)
    # hdulist.close()
     
    # print(sam_size)
    pup_id, cenx, ceny, frac = measure_contam(p, label_mask, model, sam_xpos, sam_ypos, sam_size)
    return pup_id, cenx, ceny, frac


def sam_objective(p, im, sam_xpos, sam_ypos, sam_size, good_indx):
    
    # Objective for minimization function
    
    model = sam_mask(p, im.shape, sam_xpos, sam_ypos, sam_size)
    res = np.nansum(im * model)
    
    res /= np.nansum(model[good_indx])
    
    return -res
    
def calculate(filename, bkg_file=None, pupil_xy=None, pupil_ref=None, sam_info=None, plot=False):
    
    # Open file
    with fits.open(filename) as hdu:
        # TODO - remove check on number of extensions in pipeline version?
        if len(hdu) > 1:
            image = hdu[1].data
            bpm = hdu[3].data
        else:
            image = hdu[0].data
        hdr = hdu[0].header
            
    # Initialize background variable, populate with array if bkg_file is not None
    background = 0.0
    if bkg_file is not None:
        with fits.open(bkg_file) as hdu:
            # TODO - remove check on number of extensions in pipeline version?
            if len(hdu) > 1:
                background = hdu[1].data
            else:
                background = hdu[0].data

        
    nxpw = hdr['HIERARCH ESO INS2 NXPW NAME']
    nxcw = hdr['HIERARCH ESO INS2 NXCW NAME']
    ibsm = hdr['HIERARCH ESO INS1 IBSM NAME']
    
    if nxcw != '13mas-LM':
        # TODO - remove this in the pipeline version as it should produce clean images
        image[np.where(np.abs(image) > 100000)] = 0.0
        
    # Define coordinate grid centered at the middle of the image
    s = image.shape
    xgrid, ygrid = np.meshgrid(np.arange(s[1]), np.arange(s[0]))
    xgrid -= s[1]//2
    ygrid -= s[0]//2
    
    '''
    The radius of the fitted circle depends on what is being fit, and the setup of the instrument
    
    radius1 - radius of calibration unit pupil
    radius2 - radius of illuminated region in image
                    * note for NXCW=13mas-LM, NXPW=LM-pupil this is smaller than the size of the CU pupil!
    '''
    
    # init radius1,2 to prevent possible error in case of iff fall in case 'pass'
    radius1 = 755.0
    radius2 = 815.0
    threshold = 0.0
    
    if nxcw == '27mas-JHK':
        threshold = 150 # ADU/s
        radius1 = 752.0
        if nxpw == 'Open1':
            radius2 = 815.0
        elif nxpw == 'JHK-pupil':
            radius2 = 780.0
        elif nxpw == 'Crosshairs':
            radius2 = 710.0
        else:
            pass
    elif nxcw == '13mas-JHK':
        threshold = 60 # ADU/s
        radius1 = 755.0
        if nxpw == 'Open1':
            radius2 = 815.0
        elif nxpw == 'JHK-pupil':
            radius2 = 785.0
        elif nxpw == 'Crosshairs':
            radius2 = 710.0
        elif nxpw == 'APP':
            radius2 = 735.0
        else:
            pass
    elif nxcw == '13mas-LM':
        threshold = 60 # ADU/s
        radius1 = 745.0
        if nxpw == 'Open1':
            radius2 = 815.0
        elif nxpw == 'ND':
            radius2 = 710.0
        elif nxpw == 'LM-pupil':
            radius2 = 710.0
        elif nxpw == 'Crosshairs':
            radius2 = 705.0
        elif nxpw == 'APP':
            radius2 = 735.0
        elif nxpw == 'Lyot':
            radius2 = 735.0
        elif nxpw == 'Lyot-ND':
            radius2 = 735.0
        elif nxpw == 'Spider':
            radius2 = 710.0
        else:
            pass
    else:
        pass
    
    
    '''
    Measure the position of the illuminated region
    SAM - minimization to fit shift, rotation, magnification
    All others - cross-correlation to fit shift
    '''

    sam_mid = []
    sam_cenx = []
    sam_ceny = []
    sam_frac = []
    
    if 'SAM-' in nxpw:
        print('case SAM',nxpw)
        # Get position and sizes of SAM holes, as well as starting position for minimization
        with fits.open(sam_info) as hdu:
            sam_xpos = hdu[nxpw].data['x']
            sam_ypos = hdu[nxpw].data['y']
            sam_size = hdu[nxpw].data['r']
            p0 = [hdu[nxpw].header['HIERARCH ESO FIT P0_{}'.format(i)] for i in range(0, 4)]
            
        # Trim image to reduce time for compute
        pad = 250
        sub_image = image[pad:-pad, pad:-pad]
        p0[0] -= pad
        p0[1] -= pad

        # Minimization
        good_indx = np.where(sub_image != 0.0)
        result = sp.optimize.minimize(sam_objective, p0,
                        (sub_image, sam_xpos, sam_ypos, sam_size, good_indx), method='Nelder-Mead')
        p = result.x
        
        # Convert coordinates to full image 
        p[0] += pad
        p[1] += pad
        xpos2, ypos2 = p[0], p[1]
        
        # Generate model
        model = sam_mask(p, image.shape, sam_xpos, sam_ypos, sam_size)
        # PIPPO
        # pro_file = "pupil_mask_sam.fits"
        # hdulist = fits.HDUList()
        # hdulist.append(fits.ImageHDU(data=model))
        # hdulist.writeto(pro_file, overwrite=True)
        # fits.append(pro_file, image)
        # hdulist.close()
        
        # Mean flux of the valid pixels within the SAM holes
        # print("SAM case")
        mask_flux = np.nanmean(image[np.where((model == 1.0) & (image != 0.0))])
        # print('mask_flux:',mask_flux)
        
        # Scale and rotation
        mask_scale = p[3]
        mask_rotation = p[2] * 180./np.pi
        sam_mid, sam_cenx, sam_ceny, sam_frac = sam_contamination(image, model, bpm, p, sam_xpos, sam_ypos, sam_size)
        # print("sam ncontam:",len(sam_mid))
	
    else:
        if nxpw == 'Crosshairs':
            # Create simple crosshair mask    
            width = 26.0
            mask = np.zeros(s) + 1.0

            # Set regions in the crosshairs or outside of the crosshair mask to zero
            indx = np.where((np.abs(xgrid) <= width/2.0) |\
                            (np.abs(ygrid) <= width/2.0) |\
                            (np.sqrt(xgrid**2 + ygrid**2) > radius2))
            mask[indx] = 0.0

            # Crosshair mask isn't perfectly aligned with the rows/columns of the detector.
            # Rotate by 0.7 degrees clockwise about the center of the image
            mask = sp.ndimage.rotate(mask, 0.7, reshape=False)
            
        elif (ibsm == 'TELBEAM') and ((nxpw == 'APP') or (nxpw == 'Lyot') or (nxpw == 'Lyot-ND') or (nxpw == 'Spider')):
            # Load reference image
            with fits.open(pupil_ref) as hdu:
                mask = hdu[nxpw].data

        else:
            # Create simple circular mask
            mask = np.zeros(s)
            r = np.sqrt(xgrid**2 + ygrid**2)
            mask[np.where(r <= radius2)] = 1.0

        # FFT convolution, the [::-1] trick is to match the behaviour of correlate(),
        # but is superfluous here as `mask` is symmetric
        res2 = sp.signal.fftconvolve(image, mask[::-1,::-1], mode='same')
        # Peak of `res2` is where the model best matches the data
        ypos2, xpos2 = np.unravel_index(np.argmax(res2), res2.shape)

        # Coordinate arrays centered at the middle of the illuminated region
        xgrid_illuminated, ygrid_illuminated = np.meshgrid(np.arange(s[1]), np.arange(s[0]))
        xgrid_illuminated -= xpos2
        ygrid_illuminated -= ypos2
        r_illuminated = np.sqrt(xgrid_illuminated**2 + ygrid_illuminated**2)
        indx_notilluminated = np.where(r_illuminated > radius2)
        
        # Median pixel value across entire illuminated region
        # print("other case")
        mask_flux = np.nanmean(image[np.where((r_illuminated <= radius2) & (image != 0.0))],dtype=np.float64)
        
        # Scale and rotation
        mask_scale = 1.0
        mask_rotation = 0.0

    
    # If defined, plot the pupil center
    if pupil_xy is not None:
        xpos1, ypos1 = pupil_xy[0], pupil_xy[1]
            
  
    '''
    Measure the position of the pupil (either calibration unit or telescope).
    Only do this when nxpw == 'Open1' (or nxpw == 'ND') to ensure a reliable measurement
    '''
    
    if (nxpw == 'Open1') | (nxpw == 'ND'):
        if ibsm == 'CALUNIT':
            # Zero all pixels outside the illuminated region
            image[indx_notilluminated] = 0.0
            # Create circular mask
            mask = np.zeros(image.shape)
            mask[np.where(r <= radius1)] = 1.0

            # Penalty term to avoid it fitting only the illuminated region when severely misaligned
            if (nxcw == '13mas-JHK') | (nxcw == '27mas-JHK'):
                mask[np.where(mask == 0.0)] -= 1.0
            elif nxcw == '13mas-LM':
                mask[np.where(mask == 0.0)] -= 1.0

            if bkg_file is not None:
                # Subtract background
                image -= background
                
                # Simple thresholding
                image[np.where(image < threshold)] = 0.0
                image[np.where(image != 0.0)] = 1
                
        elif ibsm == 'TELBEAM':
            
            # Instrument rotator angle, average of START and END values
            if 'HIERARCH ESO ADA ABSROT START' not in hdr or \
               'HIERARCH ESO ADA ABSROT END' not in hdr:
               print('Missing ESO ADA ABSROT START required for SAM modeling')
               print('Missing ESO ADA ABSROT END required for SAM modeling')
               print('Return 0 values and exit!')
               return (0, 0), (0, 0), 0, 0, 0, 0, 0, 0, 0,0 
            else:
               ada_posangle = np.average([hdr['HIERARCH ESO ADA ABSROT START'], 
                                          hdr['HIERARCH ESO ADA ABSROT END']])
            
            # Load reference image
            with fits.open(pupil_ref) as hdu:
                mask = hdu[nxpw].data

            # Rotate reference image by ADA.ABSROT.START/END counter-clockwise
            if np.abs(ada_posangle) > 0.1:
                mask = sp.ndimage.rotate(mask, -ada_posangle, reshape=False)
                

        # FFT convolution, the [::-1] trick is to match the behaviour of correlate(),
        # but is superfluous here as `mask` is symmetric
        # Subtract the background to ensure robust measurement of pupil position
        res1 = sp.signal.fftconvolve(image, mask[::-1, ::-1], mode='same')
        
        if ibsm == 'TELBEAM':
            # Constrain maximum offset to 80% of the radius of the illuminated region to prevent
            # cross-correlation peak between reference image and bright telescope structure
            mask_illuminated = np.ones(image.shape)
            mask_illuminated[np.where(r_illuminated > (0.8*radius2))] = 0
            res1 *= mask_illuminated
        
        elif ibsm == 'CALUNIT':
            # `res1` contains the product of the image and mask at each offset position
            # To constrain the result, divide `res1` by the number of pixels in the pupil that would be
            # illuminated at the corresponding offset position
            mask_illuminated = np.ones(image.shape)
            mask_illuminated[indx_notilluminated] = 0.0
            mask[np.where(mask <= 0.0)] = 0.0

            res2 = sp.signal.fftconvolve(mask_illuminated, mask[::-1, ::-1], mode='same')
            # Divide by this to get average value of illuminated pixels in the pupil
            res1/=res2
            # Multiply by this mask to prevent maximum average value being outside the illuminated region
            res1 *= mask_illuminated
            
        # Cross-correlation peak is at maximum of `res1` array
        ypos1, xpos1 = np.unravel_index(np.argmax(res1), res1.shape)

            
    # Calculate illuminated fraction    
    illuminated_area = area_overlap(xpos1, ypos1, radius1, xpos2, ypos2, radius2)
    illuminated_fraction = illuminated_area / (np.pi * radius1**2)

    # print("2sam ncontam:",len(sam_mid))

    return (xpos1, ypos1), (xpos2, ypos2), illuminated_fraction, mask_flux, mask_scale, mask_rotation, sam_mid, sam_cenx, sam_ceny, sam_frac

class PupilMonitor(ui.PyRecipe):
    # Fill in recipe information
    _name = "eris_nix_pupil_monitor"
    _version = "1.0"
    _author = "A. Modigliani"
    _email = "amodigli@eso.org"
    _copyright = "GPL-3.0-or-later"
    _synopsis = "Monitoring of VLT-UT Telescope pupil position"
    _description = (
        "The recipe input instrument signature corrected NIX pupil data and\n"
        + "corresponding reference data from the input set-of-frames\n"
        + "And determines the relative schift.\n"
        + "This is stored in dedicated QC parameters."
    )

    def __init__(self):
        super().__init__()

        # The recipe will have a single enumeration type parameter,
        # which allows the user to select the frame combination method.
        self.parameters = ui.ParameterList(
            (
                ui.ParameterEnum(
                    name="eris_nix_pupil_monitor.filter.method",
                    context="basic_science",
                    description="Name of the method used to determine\n"
                    "the input images edges",
                    default="scharr",
                    alternatives=("scharr", "sobel"),
                ),
            )
        )

    def run(self, frameset: ui.FrameSet, settings: Dict[str, Any]) -> ui.FrameSet:
        # Update the recipe paramters with the values requested by the user
        # through the settings argument
        for key, value in settings.items():
            try:
                self.parameters[key].value = value
            except KeyError:
                Msg.warning(
                    self.name,
                    f"Settings includes {key}:{value} but {self} \n"
                    "has no parameter named {key}.",
                )

        raw_frames = ui.FrameSet()
        product_frames = ui.FrameSet()
        # tst_frame = None
        # ref_frame = None


        output_file_pm = "PUPIL_MASK.fits"
        output_file_po = "PUPIL_OPEN_QC.fits"

        # Go through the list of input frames, check the tag
        # and act accordingly
        do_class = []
        files = []

        for frame in frameset:
            # print("frame_tag:>",frame.tag,"<")
            if frame.tag == "RAW":
                frame.group = ui.Frame.FrameGroup.RAW
                raw_frames.append(frame)
                do_class.append(frame.tag)
                files.append(frame.file)
                Msg.debug(self.name, f"Got raw frame: {frame.file}.")
            elif frame.tag == "PUPIL_SKY_OPEN":
                frame.group = ui.Frame.FrameGroup.CALIB
                do_class.append(frame.tag)
                files.append(frame.file)
                Msg.debug(self.name, f"Got pupil sky open frame: {frame.file}.")
            elif frame.tag == "PUPIL_SKY_CROSS":
                frame.group = ui.Frame.FrameGroup.CALIB
                do_class.append(frame.tag)
                files.append(frame.file)
                Msg.debug(self.name, f"Got pupil sky cross frame: {frame.file}.")
            elif frame.tag == "PUPIL_SKY_JHK":
                frame.group = ui.Frame.FrameGroup.CALIB
                do_class.append(frame.tag)
                files.append(frame.file)
                Msg.debug(self.name, f"Got pupil sky jhk frame: {frame.file}.")
            elif frame.tag == "PUPIL_SKY_LM":
                frame.group = ui.Frame.FrameGroup.CALIB
                do_class.append(frame.tag)
                files.append(frame.file)
                Msg.debug(self.name, f"Got pupil sky lm frame: {frame.file}.")
            elif frame.tag == "PUPIL_SKY_APP":
                frame.group = ui.Frame.FrameGroup.CALIB
                do_class.append(frame.tag)
                files.append(frame.file)
                Msg.debug(self.name, f"Got pupil sky app frame: {frame.file}.")
            elif frame.tag == "PUPIL_SKY_LYOT":
                frame.group = ui.Frame.FrameGroup.CALIB
                do_class.append(frame.tag)
                files.append(frame.file)
                Msg.debug(self.name, f"Got pupil sky lyot frame: {frame.file}.")
            elif frame.tag == "PUPIL_SKY_LYOTND":
                frame.group = ui.Frame.FrameGroup.CALIB
                do_class.append(frame.tag)
                files.append(frame.file)
                Msg.debug(self.name, f"Got pupil sky lyotnd frame: {frame.file}.")
            elif frame.tag == "PUPIL_SKY_SAM":
                frame.group = ui.Frame.FrameGroup.CALIB
                do_class.append(frame.tag)
                files.append(frame.file)
                Msg.debug(self.name, f"Got pupil sky sam frame: {frame.file}.")
            elif frame.tag == "PUPIL_SKY_SPIDER":
                frame.group = ui.Frame.FrameGroup.CALIB
                do_class.append(frame.tag)
                files.append(frame.file)
                Msg.debug(self.name, f"Got pupil sky spider frame: {frame.file}.")
            elif frame.tag == "PUPIL_LAMP_SAM":
                frame.group = ui.Frame.FrameGroup.CALIB
                do_class.append(frame.tag)
                files.append(frame.file)
                Msg.debug(self.name, f"Got pupil lamp sam frame: {frame.file}.")
            elif frame.tag == "PUPIL_LAMP_OPEN":
                frame.group = ui.Frame.FrameGroup.CALIB
                do_class.append(frame.tag)
                files.append(frame.file)
                Msg.debug(self.name, f"Got pupil lamp open frame: {frame.file}.")
            elif frame.tag == "PUPIL_LAMP_CROSS":
                frame.group = ui.Frame.FrameGroup.CALIB
                do_class.append(frame.tag)
                files.append(frame.file)
                Msg.debug(self.name, f"Got pupil lamp cross frame: {frame.file}.")
            elif frame.tag == "PUPIL_LAMP_LM":
                frame.group = ui.Frame.FrameGroup.CALIB
                do_class.append(frame.tag)
                files.append(frame.file)
                Msg.debug(self.name, f"Got pupil lamp lm frame: {frame.file}.")
            elif frame.tag == "PUPIL_LAMP_JHK":
                frame.group = ui.Frame.FrameGroup.CALIB
                do_class.append(frame.tag)
                files.append(frame.file)
                Msg.debug(self.name, f"Got pupil lamp jhk frame: {frame.file}.")
            elif frame.tag == "PUPIL_BACKGROUND_OPEN":
                frame.group = ui.Frame.FrameGroup.CALIB
                do_class.append(frame.tag)
                files.append(frame.file)
                Msg.debug(self.name, f"Got pupil bkg open frame: {frame.file}.")
            elif frame.tag == "PUPIL_BACKGROUND_CROSS":
                frame.group = ui.Frame.FrameGroup.CALIB
                do_class.append(frame.tag)
                files.append(frame.file)
                Msg.debug(self.name, f"Got pupil bkg cross frame: {frame.file}.")
            elif frame.tag == "PUPIL_BACKGROUND_LM":
                frame.group = ui.Frame.FrameGroup.CALIB
                do_class.append(frame.tag)
                files.append(frame.file)
                Msg.debug(self.name, f"Got pupil lamp bkg frame: {frame.file}.")
            elif frame.tag == "PUPIL_BACKGROUND_JHK":
                frame.group = ui.Frame.FrameGroup.CALIB
                do_class.append(frame.tag)
                files.append(frame.file)
                Msg.debug(self.name, f"Got pupil lamp jhk frame: {frame.file}.")
            elif frame.tag == "PUPIL_REF":
                frame.group = ui.Frame.FrameGroup.CALIB
                do_class.append(frame.tag)
                files.append(frame.file)
                Msg.debug(self.name, f"Got pupil ref frame: {frame.file}.")
            elif frame.tag == "SAM_INFO":
                frame.group = ui.Frame.FrameGroup.CALIB
                do_class.append(frame.tag)
                files.append(frame.file)
                Msg.debug(self.name, f"Got sam info frame: {frame.file}.")
            else:
                Msg.warning(
                    self.name,
                    f"Got frame {frame.file!r} with unexpected\n"
                    "tag {frame.tag!r}, ignoring.",
                )


        # For demonstration purposes we raise an exception here. Real world
        # recipes should rather print a message (also to have it in the
        # log file) and exit gracefully.
        # if len(raw_frames) == 0:
        #    raise core.DataNotFoundError("No raw frames in frameset.")


        # By default images are loaded as Python float data. Raw image
        # data which is usually represented as 2-byte integer data in a
        # FITS file is converted on the fly when an image is loaded from
        # a file. It is however also possible to load images without
        # performing this conversion.

        # print("do_class",do_class)

        bkg_file = None
        pupil_ref_open = None
        if 'PUPIL_LAMP_OPEN' in do_class:
            pupil_open = files[do_class.index('PUPIL_LAMP_OPEN')]

            # We need to check for a background frame, these are only taken with PUPIL_LAMP
            if 'PUPIL_BACKGROUND_OPEN' in do_class:
                bkg_file = files[do_class.index('PUPIL_BACKGROUND_OPEN')]

        elif 'PUPIL_SKY_OPEN' in do_class:
            pupil_open = files[do_class.index('PUPIL_SKY_OPEN')]
        else:
            print("ERROR both PUPIL_LAMP_OPEN and PUPIL_SKY_OPEN are missing, one of the two must be present! stop")
            return product_frames
        
        if 'PUPIL_REF' in do_class:
            pupil_ref = files[do_class.index('PUPIL_REF')]

        # print('pupil_open:', pupil_open)
        # print('pupil_ref:', pupil_ref)
        # print('PUPIL_SKY_SAM:', PUPIL_SKY_SAM)
        
        pupil_xy, open1_xy, open1_illumination, open1_mean, open1_scale, open1_rot, sam_mid, sam_cenx, sam_ceny, sam_frac = calculate(pupil_open, bkg_file=bkg_file, pupil_ref=pupil_ref, plot=False)
        open1_offset = pupil_xy[0] - open1_xy[0], pupil_xy[1] - open1_xy[1]


        # Save product with PRO.CATG=PUPIL_ALIGNEMNT containing the reduced pupil image PUPIL_OPEN (or, append keywords to input reduced file?)
        # Save product with PRO.CATG=PUPIL_ALIGNEMNT containing the reduced pupil image PUPIL_OPEN (or, append keywords to input reduced file?)
        # and the following QC parameters (note: the use of QC.MASK here is NOT a typo):

        # print("Check",open1_mean)
        with fits.open(pupil_open, mode='update') as hdu:
            ext_nb = len(fits.open(pupil_open))
            hdr = hdu[0].header
            arcfile = hdu[0].header['ARCFILE']
            pro_catg = hdu[0].header['HIERARCH ESO PRO CATG']
            file_spec = arcfile[:-5]+"_"
            #data = hdu[1].data
            hdr.set('HIERARCH ESO QC PUPIL CENX', pupil_xy[0])
            hdr.set('HIERARCH ESO QC PUPIL CENY', pupil_xy[1])
            hdr.set('HIERARCH ESO QC MASK CENX', open1_xy[0])
            hdr.set('HIERARCH ESO QC MASK CENY', open1_xy[1])
            hdr.set('HIERARCH ESO QC MASK OFFX', open1_offset[0])
            hdr.set('HIERARCH ESO QC MASK OFFY', open1_offset[1])
            hdr.set('HIERARCH ESO QC MASK IFRAC', open1_illumination * 100)
            hdr.set('HIERARCH ESO QC MASK FLUX', open1_mean)
            hdr.set('HIERARCH ESO QC MASK SCALE', open1_scale)

            
            hdr.set('HIERARCH ESO QC MASK ROT', open1_rot)

            qc_properties_open = core.PropertyList()
            qc_properties_open.append(
            core.Property("ESO QC PUPIL CENX", core.Type.DOUBLE, pupil_xy[0])
            )
            qc_properties_open.append(
            core.Property("ESO QC PUPIL CENY", core.Type.DOUBLE, pupil_xy[1])
            )
            qc_properties_open.append(
            core.Property("ESO QC MASK CENX", core.Type.DOUBLE, open1_xy[0])
            )
            qc_properties_open.append(
            core.Property("ESO QC MASK CENY", core.Type.DOUBLE, open1_xy[1])
            )
            qc_properties_open.append(
            core.Property("ESO QC MASK OFFX", core.Type.DOUBLE, open1_offset[0])
            )
            qc_properties_open.append(
            core.Property("ESO QC MASK OFFY", core.Type.DOUBLE, open1_offset[1])
            )
            qc_properties_open.append(
            core.Property("ESO QC MASK IFRAC", core.Type.DOUBLE, open1_illumination * 100)
            )
            # print("Check",open1_mean, type(open1_mean))
            if type(open1_mean) == 'numpy.float32' :
                #open1_mean = np.float64(open1_mean)
                qc_properties_open.append(
                core.Property("ESO QC MASK FLUX", core.Type.DOUBLE, np.float64(open1_mean))
                )
            else:   
                qc_properties_open.append(
                core.Property("ESO QC MASK FLUX", core.Type.DOUBLE, open1_mean)
                )
            qc_properties_open.append(
            core.Property("ESO QC MASK SCALE", core.Type.DOUBLE, open1_scale)
            )
            qc_properties_open.append(
            core.Property("ESO QC MASK ROT", core.Type.DOUBLE, open1_rot)
            )

            # print('1arcfile:',file_spec)
            # filename = file_spec + 'PUPIL_OPEN_QC.fits'
            filename = 'PUPIL_OPEN_QC.fits'
            pro_catg_new = pro_catg + '_QC'
            # print('new pro catg:', pro_catg_new)
            hdr['ESO PRO CATG'] = pro_catg_new       

            for i in range(0, ext_nb):
                  data = hdu[i].data
                  if i < 1:
                     fits.writeto(filename, data, hdr, overwrite=True)
                  else:
                     # print('append data set',pupil_open, arcfile)
                     fits.append(filename, data)

            # fits.writeto('PUPIL_OPEN_QC.fits', data, hdr, overwrite=True)
 
            hdu.flush()


        ### 2) Check for any PUPIL_LAMP_* or PUPIL_SKY_* images with the various pupil masks

        # Check for static calibrations, there should only be one PUPIl_REF and one SAM_INFO file provided
        pupil_ref = None
        sam_info = None
        n_masks = 0

        if 'PUPIL_REF' in do_class:
            pupil_ref = files[do_class.index('PUPIL_REF')]
        if 'SAM_INFO' in do_class:
            sam_info = files[do_class.index('SAM_INFO')]


        # Find images with pupil masks, either with LAMP or SKY
        mask_BACKGROUND_catgs = ('PUPIL_BACKGROUND_JHK', 'PUPIL_BACKGROUND_LM', 'PUPIL_BACKGROUND_CROSS', 'PUPIL_BACKGROUND_APP', 'PUPIL_BACKGROUND_LYOT', 'PUPIL_BACKGROUND_LYOTND', 'PUPIL_BACKGROUND_SAM7', 'PUPIL_BACKGROUND_SAM9', 'PUPIL_BACKGROUND_SAM23')
        mask_lamp_catgs = ('PUPIL_LAMP_JHK', 'PUPIL_LAMP_LM', 'PUPIL_LAMP_CROSS', 'PUPIL_LAMP_APP', 'PUPIL_LAMP_LYOT', 'PUPIL_LAMP_LYOTND', 'PUPIL_LAMP_SAM7', 'PUPIL_LAMP_SAM9', 'PUPIL_LAMP_SAM23', 'PUPIL_LAMP_SAM')
        mask_sky_catgs  = ('PUPIL_SKY_JHK',  'PUPIL_SKY_LM',  'PUPIL_SKY_CROSS',  'PUPIL_SKY_APP',  'PUPIL_SKY_LYOT',  'PUPIL_SKY_LYOTND',  'PUPIL_SKY_SAM7',  'PUPIL_SKY_SAM9',  'PUPIL_SKY_SAM23', 'PUPIL_SKY_SAM', 'PUPIL_SKY_SPIDER')

        mask_lamp_sam_catgs = ('PUPIL_LAMP_SAM7', 'PUPIL_LAMP_SAM9', 'PUPIL_LAMP_SAM23', 'PUPIL_LAMP_SAM')
        mask_sky_sam_catgs  = ('PUPIL_SKY_SAM7',  'PUPIL_SKY_SAM9',  'PUPIL_SKY_SAM23', 'PUPIL_SKY_SAM')

        # print('mask_lamp_catgs:',mask_lamp_catgs)
        # print('mask_sky_catgs:',mask_sky_catgs)
        # print('do_class',do_class)
        pupil_sam = []
	
        # Check if any element in the `do_class` list is in either `mask_lamp_catgs` or `mask_sky_catgs`
        if any(x in mask_lamp_catgs for x in do_class) or any(x in mask_sky_catgs for x in do_class):
            pupil_mask = []
            bkg_file = []
            qc_header = []

            for j in range(0, len(files)):
                # Check if this element in the list is in either `mask_lamp_catgs` or `mask_sky_catgs`
                if (do_class[j] in mask_lamp_catgs) or (do_class[j] in mask_sky_catgs):
                    pupil_mask.append(files[j])

                    # Check to see if a matching BACKGROUND exists, using replace method to go from _LAMP_ -> _BACKGROUND_ and _SKY_ -> _BACKGROUND_
                    if do_class[j].replace('LAMP','BACKGROUND').replace('SKY','BACKGROUND') in do_class:
                        # If there is, find the index of that entry in the list
                        indx = do_class.index(do_class[j].replace('LAMP','BACKGROUND').replace('SKY','BACKGROUND'))
                        # Then append that filename to the `bkg_file` list
                        bkg_file.append(files[indx])
                    else:
                        # Otherwise, append None
                        bkg_file.append(None)


            for j in range(0, len(files)):
                # Check if this element in the list is in either `mask_lamp_catgs` or `mask_sky_catgs`
                if (do_class[j] in mask_lamp_sam_catgs) or (do_class[j] in mask_sky_sam_catgs):
                    pupil_sam.append(files[j])


            n_masks = len(pupil_mask)
            # Now loop over all the pupil mask images
            # print('n_masks:',n_masks)
            for j in range(0, len(pupil_mask)):
                # print('processing pupil mask set:',j)
                _, mask_xy, mask_illumination, mask_mean, mask_scale, mask_rot, sam_mid, sam_cenx, sam_ceny, sam_frac = calculate(pupil_mask[j], bkg_file=bkg_file[j], pupil_xy=pupil_xy, pupil_ref=pupil_ref, sam_info=sam_info, plot=False)
                mask_offset = pupil_xy[0] - mask_xy[0], pupil_xy[1] - mask_xy[1]

                # Save product with PRO.CATG=PUPIL_ALIGNEMNT containing the reduced pupil image pupil_mask[j] (or, append keywords to input reduced file?)
                # and the following QC parameters:
                
                with fits.open(pupil_mask[j], mode='update') as hdu:
                    ext_nb = len(fits.open(pupil_mask[j]))
                    hdrf = hdu[0].header
                    mask_pro_catg = hdu[0].header['HIERARCH ESO PRO CATG']
                    hdrm = hdrf
                    # arcfile = hdu[0].header['ARCFILE']
                    hdrf.set('HIERARCH ESO QC PUPIL CENX', pupil_xy[0])
                    hdrm.set('HIERARCH ESO QC PUPIL CENX', pupil_xy[0])
                    hdrf.set('HIERARCH ESO QC PUPIL CENY', pupil_xy[1])
                    hdrm.set('HIERARCH ESO QC PUPIL CENY', pupil_xy[1])

                    hdrf.set('HIERARCH ESO QC MASK CENX', mask_xy[0])
                    hdrm.set('HIERARCH ESO QC MASK CENX', mask_xy[0])
                    hdrf.set('HIERARCH ESO QC MASK CENY', mask_xy[1])
                    hdrm.set('HIERARCH ESO QC MASK CENY', mask_xy[1])
		    
                    hdrf.set('HIERARCH ESO QC MASK OFFX', mask_offset[0])
                    hdrm.set('HIERARCH ESO QC MASK OFFX', mask_offset[0])
		    
                    hdrf.set('HIERARCH ESO QC MASK OFFY', mask_offset[1])
                    hdrm.set('HIERARCH ESO QC MASK OFFY', mask_offset[1])

                    hdrf.set('HIERARCH ESO QC MASK IFRAC', mask_illumination * 100)
                    hdrm.set('HIERARCH ESO QC MASK IFRAC', mask_illumination * 100)
		    
                    hdrf.set('HIERARCH ESO QC MASK FLUX', mask_mean)
                    hdrm.set('HIERARCH ESO QC MASK FLUX', mask_mean)


                    hdrf.set('HIERARCH ESO QC MASK SCALE', mask_scale)
                    hdrm.set('HIERARCH ESO QC MASK SCALE', mask_scale)
		    
                    hdrf.set('HIERARCH ESO QC MASK ROT', mask_rot)
                    hdrm.set('HIERARCH ESO QC MASK ROT', mask_rot)


                    sam_ncontam = len(sam_mid)
                    # print('sam_ncontam',sam_ncontam)
                    for i in range(0, sam_ncontam):
                        sam_id = sam_mid[i]
                        hdrf.set('HIERARCH ESO QC PUPIL SAM'+str(sam_id)+' LABEL', sam_id)
                        hdrm.set('HIERARCH ESO QC PUPIL SAM'+str(sam_id)+' CENX', sam_cenx[i])
                        hdrf.set('HIERARCH ESO QC PUPIL SAM'+str(sam_id)+' CENY', sam_ceny[i])
                        # hdrm.set('HIERARCH ESO QC PUPIL SAM'+str(sam_id)+' NGOOD', ngood)
                        # hdrf.set('HIERARCH ESO QC PUPIL SAM'+str(sam_id)+' NBAD', nbad)
                        hdrm.set('HIERARCH ESO QC PUPIL SAM'+str(sam_id)+' FRAC', sam_frac[i])


                    # PAPERO

                    qc_properties_mask = core.PropertyList()
                    qc_properties_mask.append(
                    core.Property("ESO QC PUPIL CENX", core.Type.DOUBLE, pupil_xy[0])
                    )
                    qc_properties_mask.append(
                    core.Property("ESO QC PUPIL CENY", core.Type.DOUBLE, pupil_xy[1])
                    )
                    qc_properties_mask.append(
                    core.Property("ESO QC MASK CENX", core.Type.DOUBLE, open1_xy[0])
                    )
                    qc_properties_mask.append(
                    core.Property("ESO QC MASK CENY", core.Type.DOUBLE, open1_xy[1])
                    )
                    qc_properties_mask.append(
                    core.Property("ESO QC MASK OFFX", core.Type.DOUBLE, open1_offset[0])
                    )
                    qc_properties_mask.append(
                    core.Property("ESO QC MASK OFFY", core.Type.DOUBLE, open1_offset[1])
                    )
                    qc_properties_mask.append(
                    core.Property("ESO QC MASK IFRAC", core.Type.DOUBLE, open1_illumination * 100)
                    )
                    qc_properties_mask.append(
                    core.Property("ESO QC MASK FLUX", core.Type.DOUBLE, open1_mean)
                    )
                    qc_properties_mask.append(
                    core.Property("ESO QC MASK SCALE", core.Type.DOUBLE, open1_scale)
                    )
                    qc_properties_mask.append(
                    core.Property("ESO QC MASK ROT", core.Type.DOUBLE, open1_rot)
                    )
                    qc_header.append(qc_properties_mask)
                    # print('2arcfile:',file_spec)
                    # filename = file_spec + 'PUPIL_MASK_'+str(j)+'_QC.fits'
                    filename = 'PUPIL_MASK_'+str(j)+'_QC.fits'
                   
                    # print('save data set',j, pupil_mask[j], arcfile)

                    pro_catg_new = mask_pro_catg + '_QC'
                    # print('new pro catg:', pro_catg_new)
                    hdrm['ESO PRO CATG'] = pro_catg_new 
                    for i in range(0, ext_nb):
                        data = hdu[i].data
                        hdrx = hdu[i].header
                        if i < 1:
                           fits.writeto(filename, data, hdrm, overwrite=True)
                        else:
                           # print('append data set',j, pupil_mask[j], arcfile)
                           fits.append(filename, data, hdrx)
		    
                    hdu.flush()

        # Process pupil data
        # ext_errs = 2
        ext_qual = 3
        tst_image = None
        ext_nb = len(fits.open(pupil_open))
        # print('ext_nb:',ext_nb)
        if ext_nb <= 1:
            ext_data = 0
        else:
            ext_data = 1

        # print('ext_nb=',ext_nb)
        if pupil_open:
            pupil_open_image = core.Image.load(pupil_open, core.Type.DOUBLE, ext_data)
            #pupil_open_qual = core.Image.load(pupil_open, core.Type.INT, ext_qual)
            pupil_open_header = core.PropertyList.load(pupil_open, 0)
        else:
            raise core.DataNotFoundError("No test pupil open frame in frameset.")

        Msg.info(self.name, f"Register product {output_file_po!r}.")

        # Register the created product
        # print('3arcfile:',file_spec)
        product_frames.append(
            ui.Frame(
                # file = file_spec + output_file_po,
                file = output_file_po,
                tag="PUPIL_OPEN_QC",
                group=ui.Frame.FrameGroup.PRODUCT,
                level=ui.Frame.FrameLevel.FINAL,
                frameType=ui.Frame.FrameType.IMAGE,
            )
        )
        if n_masks > 0:
            # print('n_masks:',n_masks)
            for i in range(0, n_masks):
                # print('4arcfile:',file_spec)

                # filename = file_spec + 'PUPIL_MASK_'+str(i)+'_QC.fits'
                filename = 'PUPIL_MASK_'+str(i)+'_QC.fits'
                product_frames.append(
                    ui.Frame(
                        file=filename,
                        tag="PUPIL_MASK_QC",
                        group=ui.Frame.FrameGroup.PRODUCT,
                        level=ui.Frame.FrameLevel.FINAL,
                        frameType=ui.Frame.FrameType.IMAGE,
                    )
                )

        return product_frames
