/* $Id: eris_ifu_dark.c,v 1.33 2013-03-26 17:00:45 jtaylor Exp $
 *
 * This file is part of the ERIS Pipeline
 * Copyright (C) 2002,2003 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/*
 * $Author: jtaylor $
 * $Date: 2013-03-26 17:00:45 $
 * $Revision: 1.33 $
 * $Name: not supported by cvs2svn $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                Includes
 -----------------------------------------------------------------------------*/

#include <string.h>
#include "eris_utils.h"
#include "eris_pfits.h"
#include "eris_ifu_dfs.h"
#include "eris_ifu_utils.h"
#include "eris_ifu_functions.h"
#include "eris_ifu_error.h"
#include "eris_ifu_dark_static.h"

#include <cpl.h>
#include <hdrl.h>

/*-----------------------------------------------------------------------------
                            Static variables
 -----------------------------------------------------------------------------*/

static const char eris_ifu_dark_description[] = "\
This recipe performs ERIS/SPIFFIER dark frames data reduction.\n\
It calculates the master dark frame and a bad pixel mask.\n\
\n\
It is required to provide three or more dark exposures to produce a reasonable\n\
master dark with associated noise.\n\
\n\
Two different bad pixel mask are generated selected by\n\
the –bpm.method recipe parameter:\n\
2D: For the 2D bad pixel mask the master dark image is searched for outliers.\n\
3D For the 3D bad pixel mask the provided dark exposures are stacked. Then\n\
outliers in the variance of each pixel are determined.\n\
\n\
-----------------------------------------------------------------------------\n\
Input files:\n\
   DO CATG          Explanation                              Required #Frames\n\
   -------          -----------                              -------- -------\n\
   DARK             Dark exposures                              Y       3-n  \n\
\n\
Output files:\n\
   DO CATG             Explanation                                    Product\n\
        Filename                                                        Depth\n\
   ------------------- ---------------------------------------------- -------\n\
   MASTER_DARK_IFU    Master dark frame (data, noise, DQI)             PD_SCI\n\
        eris_ifu_dark_master_dark.fits\n\
   BPM_DARK           Bad pixel mask                                   PD_SCI\n\
        eris_ifu_dark_bpm.fits\n\
   DARK_BPM2D         2D bad pixel mask                                PD_AUX\n\
        eris_ifu_dark_bpm2d.fits\n\
   DARK_DBG_BPM3D     2D bad pixel mask                                PD_AUX\n\
        eris_ifu_dark_bpm3d.fits\n\
   DARK_CONTRIBMAP    contribution map from image list collapse        PD_ALL\n\
        eris_ifu_dark_contribMap.fits\n\
                      master dark image (single FITS image)            PD_DBG\n\
        eris_ifu_dark_dbg_image.fits\n\
                      input dark images as stack & input reading BPM   PD_DBG\n\
        eris_ifu_dark_dbg_cube.fits\n\
                      Detailed output of 3D bad pixel detection        PD_DBG\n\
        eris_ifu_dark_dbg_3d.fits\n\
-----------------------------------------------------------------------------\n\
\n\
QC parameters:\n\
  QC.DARK.NBADPIX         Total number of bad pixels but border pixels\n\
  QC.DARK.BPIXFRAC        Fraction of bad pixels to total number of pixels\n\
  QC.DARK.NBADPIXSAT      Saturated pixels count\n\
  QC.DARK.NBADPIX2D       Dark 2D bad pixels count\n\
  QC.DARK.NBADPIX3D       Dark 3D bad pixels count\n\
  QC.DARK.BPIXFRAC2D      Fraction of 2D bad pixel to total number of pixels\n\
  QC.DARK.BPIXFRAC3D      Fraction of 3D bad pixel to total number of pixels\n\
  QC.MASTERDARK.MEAN      Clean mean of master dark image\n\
  QC.MASTERDARK.STDEV     Clean stdev of master dark image\n\
  QC.DARKMED.AVE          Average of raw darks medians\n\
  QC.DARKMED.STDEV        Read Out Noise\n\
  QC.DARKFPN              Fixed Pattern Noise of combined frames\n\
  QC.RON                  RMS on Read Out Noise\n\
  QC.RONRMS               STDEV of raw darks medians\n\
  QC.RONi                 Read Out Noise\n\
  qQC.NRONS                Number of RON frames d \n\
";

/*-----------------------------------------------------------------------------
                            Private function prototypes
 -----------------------------------------------------------------------------*/

cpl_recipe_define(eris_ifu_dark, ERIS_BINARY_VERSION, "Erich Wiezorrek",
                 PACKAGE_BUGREPORT, "2017",
                  "This recipe perform dark data reduction",
                  eris_ifu_dark_description);


/*-----------------------------------------------------------------------------
                                Function code
 -----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Setup the recipe options    
  @param    pl  the nonn-NULL parameter list to fill
  @return   CPL_ERROR_NONE iff everything is ok

  Defining the command-line/configuration parameters for the recipe.
 */
/*----------------------------------------------------------------------------*/
static cpl_error_code eris_ifu_dark_fill_parameterlist(cpl_parameterlist *pl)
{
    cpl_parameter   *p  = NULL;
    cpl_error_code  err = CPL_ERROR_NONE;

    cpl_ensure_code(pl, CPL_ERROR_NULL_INPUT);

    TRY
    {
        /* Fill the parameters list */

        /* --instrument & --product_depth*/
        BRK_IF_ERROR(
            eris_ifu_add_std_params(pl, REC_NAME_DARK));

        /* add common dark & flat parameters */
        BRK_IF_ERROR(
            eris_parlist_config_add_bpm(pl, REC_NAME_DARK));

        /* --QC parameters */
        BRK_IF_NULL(
            p = cpl_parameter_new_range("eris.eris_ifu_dark.qc_ron_xmin",
                                    CPL_TYPE_INT,
                                    "qc_ron_xmin",
                                    "eris.eris_ifu_dark",
                                    9,1,ERIS_IFU_DETECTOR_SIZE-4));
        BRK_IF_ERROR(
            cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI,"qc_ron_xmin"));
        BRK_IF_ERROR(
            cpl_parameterlist_append(pl, p));

        BRK_IF_NULL(
            p = cpl_parameter_new_range("eris.eris_ifu_dark.qc_ron_xmax",
                                    CPL_TYPE_INT,
                                    "qc_ron_xmax",
                                    "eris.eris_ifu_dark",
                                    2040,1,ERIS_IFU_DETECTOR_SIZE));
        BRK_IF_ERROR(
            cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI,"qc_ron_xmax"));
        BRK_IF_ERROR(
            cpl_parameterlist_append(pl, p));

        BRK_IF_NULL(
            p = cpl_parameter_new_range("eris.eris_ifu_dark.qc_ron_ymin",
                                    CPL_TYPE_INT,
                                    "qc_ron_ymin",
                                    "eris.eris_ifu_dark",
                                    9,1,ERIS_IFU_DETECTOR_SIZE));
        BRK_IF_ERROR(
            cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI,"qc_ron_ymin"));
        BRK_IF_ERROR(
            cpl_parameterlist_append(pl, p));

        BRK_IF_NULL(
            p = cpl_parameter_new_range("eris.eris_ifu_dark.qc_ron_ymax",
                                    CPL_TYPE_INT,
                                    "qc_ron_ymax",
                                    "eris.eris_ifu_dark",
                                    2040,1,ERIS_IFU_DETECTOR_SIZE));
        BRK_IF_ERROR(
            cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI,"qc_ron_ymax"));
        BRK_IF_ERROR(
            cpl_parameterlist_append(pl, p));

        BRK_IF_NULL(
            p = cpl_parameter_new_value("eris.eris_ifu_dark.qc_ron_hsize",
                        CPL_TYPE_INT,
                        "qc_ron_hsize",
                        "eris.eris_ifu_dark",
                        4));
        BRK_IF_ERROR(
            cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI,"qc_ron_hsize"));
        BRK_IF_ERROR(
            cpl_parameterlist_append(pl, p));

        BRK_IF_NULL(
            p = cpl_parameter_new_value("eris.eris_ifu_dark.qc_ron_nsamp",
                        CPL_TYPE_INT,
                        "qc_ron_nsamp",
                        "eris.eris_ifu_dark",
                        100));
        BRK_IF_ERROR(
            cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI,"qc_ron_nsamp"));
        BRK_IF_ERROR(
            cpl_parameterlist_append(pl, p));

        BRK_IF_NULL(
            p = cpl_parameter_new_range("eris.eris_ifu_dark.qc_fpn_xmin",
                                    CPL_TYPE_INT,
                                    "qc_fpn_xmin",
                                    "eris.eris_ifu_dark",
                                    7,1,ERIS_IFU_DETECTOR_SIZE));
        BRK_IF_ERROR(
            cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI,"qc_fpn_xmin"));
        BRK_IF_ERROR(
            cpl_parameterlist_append(pl, p));

        BRK_IF_NULL(
            p = cpl_parameter_new_range("eris.eris_ifu_dark.qc_fpn_xmax",
                                    CPL_TYPE_INT,
                                    "qc_fpn_xmax",
                                    "eris.eris_ifu_dark",
                                    2042,1,ERIS_IFU_DETECTOR_SIZE));
        BRK_IF_ERROR(
            cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI,"qc_fpn_xmax"));
        BRK_IF_ERROR(
            cpl_parameterlist_append(pl, p));

        BRK_IF_NULL(
            p = cpl_parameter_new_range("eris.eris_ifu_dark.qc_fpn_ymin",
                                    CPL_TYPE_INT,
                                    "qc_fpn_ymin",
                                    "eris.eris_ifu_dark",
                                    7,1,ERIS_IFU_DETECTOR_SIZE));
        BRK_IF_ERROR(
            cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI,"qc_fpn_ymin"));
        BRK_IF_ERROR(
            cpl_parameterlist_append(pl, p));

        BRK_IF_NULL(
            p = cpl_parameter_new_range("eris.eris_ifu_dark.qc_fpn_ymax",
                                    CPL_TYPE_INT,
                                    "qc_fpn_ymax",
                                    "eris.eris_ifu_dark",
                                    2042,1,ERIS_IFU_DETECTOR_SIZE));
        BRK_IF_ERROR(
            cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI,"qc_fpn_ymax"));
        BRK_IF_ERROR(
            cpl_parameterlist_append(pl, p));

        BRK_IF_NULL(
            p = cpl_parameter_new_value("eris.eris_ifu_dark.qc_fpn_hsize",
                        CPL_TYPE_INT,
                        "qc_fpn_hsize",
                        "eris.eris_ifu_dark",
                        2));
        BRK_IF_ERROR(
            cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI,"qc_fpn_hsize"));
        BRK_IF_ERROR(
            cpl_parameterlist_append(pl, p));

        BRK_IF_NULL(
            p = cpl_parameter_new_value("eris.eris_ifu_dark.qc_fpn_nsamp",
                        CPL_TYPE_INT,
                        "qc_fpn_nsamp",
                        "eris.eris_ifu_dark",
                        1000));
        BRK_IF_ERROR(
            cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI,"qc_fpn_nsamp"));
        BRK_IF_ERROR(
            cpl_parameterlist_append(pl, p));


    } CATCH {
        CATCH_MSGS();
        err = cpl_error_get_code();
    }

    return err;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Interpret the command line options and execute the data processing
  @param    frameset   the frames list
  @param    parlist    the parameters list
  @return   0 if everything is ok
 */
/*----------------------------------------------------------------------------*/
static int eris_ifu_dark(cpl_frameset            *frameset,
                         const cpl_parameterlist *parlist)
{
    hdrl_parameter      *pdarkcollapse          = NULL;
    const cpl_frame     *rawframe               = NULL;
    cpl_propertylist    *applist                = NULL;
    cpl_imagelist       *darkImageList          = NULL;
    hdrl_imagelist      *darkHdrlImageList      = NULL;
    hdrl_image          *masterDarkHdrlImage    = NULL;
    cpl_image           *contribMap             = NULL,
                        *bpMask                 = NULL,
                        *bpm2dImg               = NULL,
                        *bpm3dImg               = NULL,
                        *masterDarkImage        = NULL,
                        *errorImage             = NULL,
                        *qualityImage           = NULL;
    cpl_mask            *bpm2dMask              = NULL,
                        *bpm3dMask              = NULL;
    productDepthType    productDepth            = PD_SCIENCE;
    struct stdParamStruct stdParams             = stdParamStructInit;

    cpl_ensure_code(frameset, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(parlist,  CPL_ERROR_NULL_INPUT);

    TRY
    {
        if (cpl_frameset_get_size(frameset) == 0) {
            BRK_WITH_ERROR_MSG(CPL_ERROR_ILLEGAL_INPUT,
                "Missing or empty SOF file");
        }



        if (cpl_frameset_count_tags(frameset, ERIS_IFU_RAW_DARK) < 3) {
            BRK_WITH_ERROR_MSG(CPL_ERROR_ILLEGAL_INPUT,
                               "At least three dark frames (tagged as '"
                               ERIS_IFU_RAW_DARK
                               "') must be provided in the SOF");
        }

        cpl_msg_info(cpl_func, "Reading recipe parameters");
        BRK_IF_ERROR(
            eris_ifu_fetch_std_param(parlist, REC_NAME_DARK, &stdParams));

        productDepth = stdParams.productDepth;
        CHECK_ERROR_STATE();

        BRK_IF_NULL(
            pdarkcollapse = hdrl_collapse_parameter_parse_parlist(parlist,
                                                "eris.eris_ifu_dark.collapse"));

        /* Identify the RAW and CALIB frames in the input frameset */
        BRK_IF_ERROR(
            eris_ifu_dfs_set_groups(frameset));
        cpl_msg_info(cpl_func, "Loading set of frames (SOF)");

        rawframe = cpl_frameset_find_const(frameset, ERIS_IFU_RAW_DARK);
        if (rawframe == NULL) {
            /* cpl_frameset_find_const() does not set an error code, when a
             * frame is not found, so we will set one here. */
            return (int) cpl_error_set_message(cpl_func,
                                    CPL_ERROR_DATA_NOT_FOUND,
                                    "SOF does not have any file tagged with %s",
                                    ERIS_IFU_RAW_DARK);
        }

        BRK_IF_NULL(
            darkHdrlImageList = eris_ifu_get_hdrlimagelist_by_tag(frameset,
                    ERIS_IFU_RAW_DARK, stdParams.rawImageCorrectionMask));

        BRK_IF_NULL(
            darkImageList = eris_ifu_hdrl_get_imagelist(darkHdrlImageList,
                                                        eris_hdrl_data));

        BRK_IF_NULL(
            applist = cpl_propertylist_new());

        /* NOW PERFORMING THE DATA REDUCTION */
        BRK_IF_ERROR(
            eris_ifu_dark_static(parlist,
                    darkHdrlImageList, pdarkcollapse,
                    &masterDarkHdrlImage, &qualityImage,
                    &bpMask, &contribMap,
                    &bpm2dMask, &bpm3dMask, applist));
        //cpl_propertylist_dump(applistDark,stdout);
        masterDarkImage = cpl_image_duplicate(
            hdrl_image_get_image(masterDarkHdrlImage));

        if (masterDarkImage == NULL) {
            return (int) cpl_error_set_message(cpl_func, cpl_error_get_code(),
                "Could not load the image");
        }

        /* Save products */
        /* Add the product category  */
         BRK_IF_ERROR(
             cpl_propertylist_append_string(applist, CPL_DFS_PRO_CATG,
                 ERIS_IFU_PRO_DARK));

        /* start with master product holding data,noise,quality extensions*/
        if (masterDarkImage != NULL)
        {
            /* fetch noise image */
            BRK_IF_NULL(
                errorImage = hdrl_image_get_error(masterDarkHdrlImage));

            /* save the master product */
            BRK_IF_ERROR(
                cpl_propertylist_set_string(applist, CPL_DFS_PRO_CATG,
                     ERIS_IFU_PRO_DARK));

            BRK_IF_ERROR(
                eris_ifu_save_deq_image(frameset, NULL, parlist, frameset, NULL,
                                        REC_NAME_DARK, applist, NULL,
                                        ERIS_IFU_PRO_DARK_FN,
                                        masterDarkImage, errorImage, rmse,
                                        qualityImage, flag16bit, UNIT_ADU));
        }

        BRK_IF_ERROR(
            eris_ifu_save_image(frameset, applist, parlist, REC_NAME_DARK,
                                ERIS_IFU_PRO_DARK_BPM,
                                ERIS_IFU_PRO_DARK_BPM_FN,
                                CPL_TYPE_USHORT,
                                bpMask));

        /* save additional products if requested via productDepth argument */
        if (bpm2dMask != NULL && productDepth >= PD_AUXILLIARY) {
            BRK_IF_NULL(bpm2dImg = cpl_image_new_from_mask(bpm2dMask));
            BRK_IF_ERROR(
                eris_ifu_save_image(frameset, applist, parlist, REC_NAME_DARK,
                                    ERIS_IFU_PRO_DARK_DBG_BPM2D,
                                    ERIS_IFU_PRO_DARK_DBG_BPM2D_FN,
                                    CPL_TYPE_USHORT,
                                    bpm2dImg));
        }
        if (bpm3dMask != NULL &&  productDepth >= PD_AUXILLIARY) {
            BRK_IF_NULL(bpm3dImg = cpl_image_new_from_mask(bpm3dMask));
            BRK_IF_ERROR(
                eris_ifu_save_image(frameset, applist, parlist, REC_NAME_DARK,
                                    ERIS_IFU_PRO_DARK_DBG_BPM3D,
                                    ERIS_IFU_PRO_DARK_DBG_BPM3D_FN,
                                    CPL_TYPE_USHORT,
                                    bpm3dImg));
        }

        CHECK_ERROR_STATE();

        if (contribMap != NULL && productDepth >= PD_ALL) {
            BRK_IF_ERROR(
                eris_ifu_save_image(frameset, applist, parlist, REC_NAME_DARK,
                                    ERIS_IFU_PRO_DARK_DBG_CONTRIBMAP,
                                    ERIS_IFU_PRO_DARK_DBG_CONTRIBMAP_FN,
                                    CPL_TYPE_USHORT,
                                    contribMap));
        }

        if (masterDarkImage != NULL && productDepth >= PD_DEBUG) {
            BRK_IF_ERROR(
                eris_ifu_save_cpl_image_dbg(masterDarkImage,
                                        ERIS_IFU_PRO_DARK_DBG_IMAGE_FN, TRUE, NULL));
        }

        if (darkImageList != NULL &&  productDepth >= PD_DEBUG) {
            BRK_IF_ERROR(
                eris_ifu_save_cpl_imagelist_dbg(darkImageList,
                                        ERIS_IFU_PRO_DARK_DBG_CUBE_FN));

            cpl_image *sbpm = cpl_image_new(2048,2048,CPL_TYPE_INT);
            cpl_imagelist *cbpm = cpl_imagelist_new();
            for (int bx=0 ; bx<cpl_imagelist_get_size(darkImageList); bx++) {
                cpl_mask *tmask = cpl_image_get_bpm(
                    cpl_imagelist_get(darkImageList, bx));
                cpl_image *timg = cpl_image_new_from_mask(tmask);
                cpl_imagelist_set(cbpm, timg, bx);
//                cpl_image_save(timg,ERIS_IFU_PRO_DARK_DBG_CUBE_FN".fits",
//                        CPL_TYPE_INT,NULL,CPL_IO_EXTEND);
                cpl_image_add(sbpm, timg);
            }
            cpl_imagelist_save(cbpm, ERIS_IFU_PRO_DARK_DBG_CUBE_FN".fits",
                    CPL_TYPE_INT,NULL,CPL_IO_EXTEND);
            cpl_image_save(sbpm, ERIS_IFU_PRO_DARK_DBG_CUBE_FN".fits",
                    CPL_TYPE_INT,NULL,CPL_IO_EXTEND);
//            for (int bx=0 ; bx<cpl_imagelist_get_size(darkImageList); bx++) {
//                cpl_mask *tmask = cpl_image_get_bpm(
//                    cpl_imagelist_get(darkImageList, bx));
//                cpl_image *timg = cpl_image_new_from_mask(tmask);
//                cpl_image_save(timg,"inputBpm.fits",CPL_TYPE_INT,NULL,CPL_IO_EXTEND);
//                cpl_image_save(timg, ERIS_IFU_PRO_DARK_DBG_CUBE_FN".fits",
//                        CPL_TYPE_INT,NULL,CPL_IO_EXTEND);
//            }
            cpl_image_delete(sbpm);
            cpl_imagelist_delete(cbpm);
        }

    }
    CATCH
    {
    }

    eris_ifu_free_propertylist(&applist);
    eris_ifu_free_hdrl_image(&masterDarkHdrlImage);
    eris_ifu_free_hdrl_imagelist(&darkHdrlImageList);
    eris_ifu_free_hdrl_parameter(&pdarkcollapse);
    eris_ifu_free_image(&masterDarkImage);
    eris_ifu_free_image(&contribMap);
    eris_ifu_free_image(&bpMask);
    eris_ifu_free_image(&bpm2dImg);
    eris_ifu_free_image(&bpm3dImg);
    eris_ifu_free_imagelist(&darkImageList);
    eris_ifu_free_image(&qualityImage);
    eris_ifu_free_mask(&bpm2dMask);
    eris_ifu_free_mask(&bpm3dMask);
    eris_ifu_free_std_param(&stdParams);
    if (cpl_memory_is_empty() == 0)
    {
        cpl_memory_dump();
    }
    return (int) cpl_error_get_code();
}
