/* $Id: eris_ifu_distortion.c,v 0.1 2019-06-11 08:13:17 06:06:06 agudo Exp $
 *
 * This file is part of the ERIS Pipeline
 * Copyright (C) 2002,2003 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: agudo $
 * $Date: 2018-06-11 06:06:06  $
 * $Revision: 0.1 $
 * $Name: not supported by cvs2svn $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>          /* allows the program compilation */
#endif


/*-----------------------------------------------------------------------------
                                Includes
 ----------------------------------------------------------------------------*/
#include <strings.h>
#include <string.h>
#include <stdio.h>
#include <math.h>

#include <time.h>

#include <cpl.h>
#include <hdrl.h>
#include <irplib_utils.h>

#include "eris_ifu_dfs.h"
#include "eris_ifu_error.h"
#include "eris_ifu_debug.h"
#include "eris_ifu_functions.h"
#include "eris_ifu_utils.h"
#include "eris_ifu_distortion_static.h"
#include "eris_ifu_flat_static.h"
#include "eris_utils.h"
/*-----------------------------------------------------------------------------
                            Static variables
 -----------------------------------------------------------------------------*/
static char eris_ifu_distortion_description[] = "\
This recipe determines the optical distortions and the slitlets distances.\n\
\n\
-----------------------------------------------------------------------------\n\
Input files:\n\
   DO CATG          Explanation                              Required #Frames\n\
   -------          -----------                              -------- -------\n\
   FIBRE_NS         Fibre exposures                             Y      [1,75]\n\
   FLAT_NS          Flatlamp on/off exposures                   Y         2  \n\
   WAVE_NS          Arclamp  on/off exposures                   Y        >=2 \n\
   DARK_NS          Dark exposure                               N       [0,1]\n\
   REF_LINE_ARC     Band-specific reference arc line list       Y         1  \n\
   WAVE_SETUP       Table to configure wave calibration         Y         1  \n\
   FIRST_WAVE_FIT   Table for the first attempt to fit ARClines Y         1  \n\
\n\
Output files:\n\
   DO CATG          Explanation\n\
   -------          -----------\n\
   DISTANCES        Table with slitlet distances\n\
   BPM_DIST         Badpixel frame\n\
   SLITLET_POS      Table with slitlet edge positions\n\
   DISTORTION       Table with parameters for distortion correction\n\
   REF_LINE_ARC     Table reference arc line values\n\
   WAVE_SETUP       Table with guess wavelength calib solution values\n\
   FIRST_WAVE_FIT   Table with first fit solution\n\
-----------------------------------------------------------------------------\n\
\n\
Information on relevant parameters may be found with\n\
  esorex --params "REC_NAME_DISTORTION"\n\
  esorex --help   "REC_NAME_DISTORTION"\n";

/*-----------------------------------------------------------------------------
                            Private function prototypes
 -----------------------------------------------------------------------------*/
cpl_recipe_define(eris_ifu_distortion, ERIS_BINARY_VERSION, "Alex Agudo Berbel",
                  PACKAGE_BUGREPORT, "2019",
                  "This recipe performs distortion correction",
                  eris_ifu_distortion_description);

cpl_error_code eris_ifu_dist_save_aux_products(
        cpl_frameset *frameset,
        const cpl_parameterlist *parlist,
        cpl_propertylist *pl,
        cpl_polynomial      **distortion_poly,
        cpl_table *minmax_borders,
        hdrl_image *fibre_div_img );
/*-----------------------------------------------------------------------------
                                Functions code
 ----------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*/
/**
  @brief    Setup the recipe options    
  @param    pl  the non-NULL parameter list to fill
  @return   CPL_ERROR_NONE if everything is ok

  Defining the command-line/configuration parameters for the recipe.
 */
/*---------------------------------------------------------------------------*/
static cpl_error_code eris_ifu_distortion_fill_parameterlist(cpl_parameterlist *pl)
{
    cpl_error_code  err = CPL_ERROR_NONE;
    cpl_parameter   *p  = NULL;

    cpl_ensure_code(pl, CPL_ERROR_NULL_INPUT);

    TRY
    {
        /* --instrument & --product_depth*/
        BRK_IF_ERROR(
            eris_parlist_config_add_all_recipes(pl, REC_NAME_DISTORTION));

        /* Override: set default instrument to NONE (to be removed when
         * final detector settings are found and real data is available) */
        BRK_IF_NULL(
            p = cpl_parameterlist_find(pl,
                                   "eris.eris_ifu_distortion.instrument"));
        BRK_IF_ERROR(
            cpl_parameter_set_default_string(p, "NONE"));

        /* add common bpm-parameters */
        BRK_IF_ERROR(
            eris_parlist_config_add_bpm(pl, REC_NAME_DISTORTION));

        /* Override: set default collapse-method to MEAN */
        BRK_IF_NULL(
            p = cpl_parameterlist_find(pl,
                                   "eris.eris_ifu_distortion.collapse.method"));
        BRK_IF_ERROR(
            cpl_parameter_set_default_string(p, "MEAN"));

        /* add common flat-parameters */
        BRK_IF_ERROR(
            eris_parlist_config_add_flat(pl, REC_NAME_DISTORTION));

        /* add distortion parameters */
        BRK_IF_NULL(
            p = cpl_parameter_new_value(
                    "eris.eris_ifu_distortion.distortion.arcs.thresh_factor",
                    CPL_TYPE_DOUBLE,
                    "median_value(image)+ kappa*sigma is the minimum intensity "
                    "threshold of accepted image pixels",
                    "eris_ifu_distortion.distortion.arcs",
                    0.33333));
        BRK_IF_ERROR(
            cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI,
                                    "distortion.arcs.thresh_factor"));
        BRK_IF_ERROR(
            cpl_parameterlist_append(pl, p));

        BRK_IF_NULL(
            p = cpl_parameter_new_value(
                    "eris.eris_ifu_distortion.distortion.arcs.min_arclen_factor",
                    CPL_TYPE_DOUBLE,
                    "factor which sets minimum arc length (1.0-2)",
                    "eris_ifu_distortion.distortion.arcs",
                    1.19));
        BRK_IF_ERROR(
            cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI,
                                    "distortion.arcs.min_arclen_factor"));
        BRK_IF_ERROR(
            cpl_parameterlist_append(pl, p));

        BRK_IF_NULL(
            p = cpl_parameter_new_value(
                    "eris.eris_ifu_distortion.distortion.arcs.window_size",
                    CPL_TYPE_INT,
                    "Size of window for low pass filter used in an horizontal "
                    "low pass filter to remove unwanted arcs (5-64)",
                    "eris_ifu_distortion.distortion.arcs",
                    14));
        BRK_IF_ERROR(
            cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI,
                                    "distortion.arcs.window_size"));
        BRK_IF_ERROR(
            cpl_parameterlist_append(pl, p));

        BRK_IF_NULL(
            p = cpl_parameter_new_value(
                    "eris.eris_ifu_distortion.distortion.smooth_rad",
                    CPL_TYPE_INT,
                    "Size of smoothing factor (1-11) used to prevent for "
                    "possible intensity drops from detector electronics on "
                    "fibre illuminated slitlets (1-11)",
                    "eris_ifu_distortion.distortion",
                    3));
        BRK_IF_ERROR(
            cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "distortion.smooth_rad"));
        BRK_IF_ERROR(
            cpl_parameterlist_append(pl, p));

        BRK_IF_NULL(
            p = cpl_parameter_new_value(
                    "eris.eris_ifu_distortion.wave.div",
                    CPL_TYPE_BOOL,
                    "Divide masterflat from subtracted wave frame",
                    "eris_ifu_distortion.wave",
                    TRUE));
        BRK_IF_ERROR(
            cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI,
                                    "wave.div"));
        BRK_IF_ERROR(
            cpl_parameterlist_append(pl, p));


        BRK_IF_NULL(p = cpl_parameter_new_value("eris.eris_ifu_distortion.pixel_saturation",
                        		  CPL_TYPE_DOUBLE,"Pixel saturation level ",
								  "eris_ifu_distortion",  18000.));
        BRK_IF_ERROR(cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI,
                                                  "pixel_saturation"));
        cpl_parameterlist_append(pl, p);
    }
    CATCH
    {
        CATCH_MSGS();
        err = cpl_error_get_code();
    }

    return err;
}

/*---------------------------------------------------------------------------*/
/**
  @brief    Interpret the command line options and execute the data processing
  @param    frameset   the frames list
  @param    parlist     the parameters list

  @return   0 if everything is ok
 */
/*---------------------------------------------------------------------------*/
static int
eris_ifu_distortion(cpl_frameset *frameset, const cpl_parameterlist *parlist)
{    
    int                 productDepth            = 0;
    char                *str                    = NULL;
    const char          *instr_tmp;
    const char          *fn                     = NULL,
                        *instrument             = NULL,
                        *ins                    = NULL,
                        *band                   = NULL,
                        *dpr_type               = NULL;
    cpl_image           *qualityImage           = NULL,
                        *bp_img_flat            = NULL,
                        *errorImage             = NULL,
                        *contribMap             = NULL;
    const cpl_image     *masterFlatImage        = NULL;
    cpl_frameset        *frameset_flat          = NULL,
                        *frameset_fibre         = NULL;
    const cpl_frame     *fr                     = NULL;
    cpl_boolean         cut_off_left            = FALSE,
                        cut_off_right           = FALSE;
    int                 do_wave_flat_div        = FALSE;
    cpl_mask            *bp_mask_flat           = NULL;
    cpl_polynomial      **distortion_poly       = NULL,
                        **distortion_poly_full  = NULL;
    hdrl_image          *masterFlatHdrlImg_lo   = NULL,
                        *tmp_img                = NULL,
                        *fibre_on_img           = NULL,
                        *fibre_div_img          = NULL,
                        *dark_img               = NULL,
                        *warped_img_wave        = NULL,
                        *warped_img_full        = NULL,
                        *wave_on_img            = NULL,
                        *wave_off_img           = NULL;
    hdrl_imagelist      *fibre_imglist          = NULL,
                        *wave_imglist           = NULL,
                        *cube                   = NULL,
                        *cube_sorted            = NULL;
    hdrl_parameter      *pcollapse              = NULL;
    cpl_table           **centers_fitted        = NULL,
                        **slit_edges            = NULL,
                        *minmax_borders         = NULL,
                        *valid_arc_lines        = NULL;
    cpl_propertylist    **qc                    = NULL,
                        *pl                     = NULL,
                        *qc_list                = NULL;
    cpl_table           *qclog                  = NULL;
    cpl_imagelist       *tmpImgList             = NULL;
    flatMode            mode                    = FLAT_MODE_NONE;
    ifsPreopticsScale   scale                   = UNDEFINED_SCALE;

    cpl_ensure_code(frameset, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(parlist, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(cpl_frameset_is_empty(frameset) == FALSE,
                    CPL_ERROR_NULL_INPUT);


    /* check required input tags are present */
    /* Temporarily suppress check on required input tags
    const int ntags = 6;
    const char* required_tags[6] = {
    		ERIS_IFU_RAW_FIBRE_NS,
			ERIS_IFU_CALIB_FIRST_FIT,
			ERIS_IFU_RAW_FLAT_NS,
			ERIS_IFU_CALIB_REF_LINES,
			ERIS_IFU_RAW_WAVE_NS,
			ERIS_IFU_CALIB_WAVE_SETUP
    };

   
    cpl_ensure_code(CPL_ERROR_NONE ==
    		eris_dfs_check_input_tags(frameset, required_tags, ntags, 1),
			CPL_ERROR_ILLEGAL_INPUT);
			*/

    cpl_ensure_code(cpl_frameset_count_tags(frameset, ERIS_IFU_RAW_FIBRE_NS) >= 1,
                        CPL_ERROR_ILLEGAL_INPUT);
    cpl_ensure_code((cpl_frameset_count_tags(frameset, ERIS_IFU_RAW_FLAT_NS) == 2) ||
    		        (cpl_frameset_count_tags(frameset, ERIS_IFU_RAW_FLAT_NS_ON) == 1 &&
    				 cpl_frameset_count_tags(frameset, ERIS_IFU_RAW_FLAT_NS_OFF)),
					CPL_ERROR_ILLEGAL_INPUT);
    cpl_ensure_code((cpl_frameset_count_tags(frameset, ERIS_IFU_RAW_WAVE_NS) >= 2) ||
    		         (cpl_frameset_count_tags(frameset, ERIS_IFU_RAW_WAVE_NS_ON) >= 1 &&
    		          cpl_frameset_count_tags(frameset, ERIS_IFU_RAW_WAVE_NS_OFF) >= 1),
                    CPL_ERROR_ILLEGAL_INPUT);

    TRY
    {
        /* -----------------------------------------------------------------
         *   Setup
         * ----------------------------------------------------------------- */
        /* --instrument */
        BRK_IF_NULL(
            instrument = cpl_parameter_get_string(
                                cpl_parameterlist_find_const(parlist,
                                "eris.eris_ifu_distortion.instrument")));

        BRK_IF_NULL(
            fr = cpl_frameset_find_const(frameset, ERIS_IFU_RAW_FIBRE_NS));
        BRK_IF_NULL(
            fn = cpl_frame_get_filename(fr));
        BRK_IF_NULL(
            pl = cpl_propertylist_load(fn, 0));

        /* Override: default instrument is set to NONE (to be removed when
         * final detector settings are found and real data is available)
         * when NONE set to ERIS for all bands except H/J (there it'll be
         * SINFONI) */
        if (strcmp(instrument, INSTRUMENT_NONE) == 0) {
            BRK_IF_NULL(
                ins = cpl_propertylist_get_string(pl, "INSTRUME"));
            if (strcmp(ins, INSTRUMENT_SINFONI) == 0) {
                // we have SINFONI-data
                BRK_IF_NULL(
                    band = cpl_propertylist_get_string(pl, FHDR_S_FILTER_NAME));

                if ((strcmp(band, S_GRATING_H_BAND) == 0) ||
                    (strcmp(band, S_GRATING_J_BAND) == 0))
                {
                    instr_tmp = INSTRUMENT_SINFONI;
                } else {
                    instr_tmp = INSTRUMENT_ERIS;
                }

            } else {
                instr_tmp = INSTRUMENT_ERIS;
            }
        } else {
            // either we have ERIS-data or we don't mind what data we have and
            // we process it like ERIS-data
            instr_tmp = instrument;
        }

        cpl_msg_info(cpl_func, "Processing %s frames", instr_tmp);


        band = eris_ifu_get_bandString(eris_ifu_get_band(pl));
        scale = eris_ifu_get_preopticsScale(pl);
        cpl_msg_info(cpl_func, "Band: %s, scale: %s", band,
                     eris_ifu_get_preopticsScaleString(scale));

//        cpl_boolean is25mas = (scale == S25MAS);
//        cpl_boolean isK = ('K' == band[0]);
//        cpl_boolean isH = ('H' == band[0]);

        /* Identify the RAW and CALIB frames in the input frameset */
        BRK_IF_ERROR(eris_ifu_dfs_set_groups(frameset));

        /* save additional products if requested via productDepth argument */
        productDepth = cpl_parameter_get_int(
                                cpl_parameterlist_find_const(parlist,
                                "eris.eris_ifu_distortion.product_depth"));
        CHECK_ERROR_STATE();

        // check if collapse-method is MINMAX, not allowed here
        const char *collapse_method = NULL;
        BRK_IF_NULL(
            collapse_method = cpl_parameter_get_string(
                                cpl_parameterlist_find_const(parlist,
                                  "eris.eris_ifu_distortion.collapse.method")));
        if (strcmp(collapse_method, "MINMAX") == 0) {
            SET_ERROR_MSG(CPL_ERROR_ILLEGAL_INPUT,
                                       "collapse.method == MINMAX isn't allowed");
            CHECK_ERROR_STATE();
        }

        /* -----------------------------------------------------------------
         *   Performing master-flat calculation
         * ----------------------------------------------------------------- */

        if (cpl_frameset_count_tags(frameset, ERIS_IFU_RAW_FLAT_NS) > 0) {
        	frameset_flat = eris_ifu_get_frameset_by_tag(frameset,
        			ERIS_IFU_RAW_FLAT_NS);
        } else if ( (cpl_frameset_count_tags(frameset, ERIS_IFU_RAW_FLAT_NS_ON) > 0) &&
        		(cpl_frameset_count_tags(frameset, ERIS_IFU_RAW_FLAT_NS_OFF) > 0) ) {
        	cpl_frameset* frameset_flat_off = NULL;
        	frameset_flat = eris_ifu_get_frameset_by_tag(frameset, ERIS_IFU_RAW_FLAT_NS_ON);

        	frameset_flat_off = eris_ifu_get_frameset_by_tag(frameset, ERIS_IFU_RAW_FLAT_NS_OFF);

        	cpl_frameset_join(frameset_flat, frameset_flat_off);
        	cpl_frameset_delete(frameset_flat_off);
        }
        

        if (cpl_frameset_count_tags(frameset, ERIS_IFU_CALIB_FLATFIELD) != 0) {
//AA this goes probably away when flat-segment-procedure is implemented. Ask erw

            // overide any raw flat exposures and provide own masterflat and BPM

            cpl_msg_info(cpl_func, "Using ERIS_IFU_CALIB_FLATFIELD and"
                                   "ERIS_IFU_PRO_FLAT_BPM");
//            bp_mask_flat = eris_ifu_load_badpixel_mask(
//                frameset, ERIS_IFU_PRO_FLAT_BPM, 0);
            const char *fn1;
            fn1 = cpl_frame_get_filename(
                cpl_frameset_find_const(frameset, ERIS_IFU_CALIB_FLATFIELD));
            cpl_image *flatFieldImg = cpl_image_load(fn1, CPL_TYPE_DOUBLE, 0, 0);
            if (cpl_frameset_count_tags(frameset, ERIS_IFU_PRO_FLAT_BPM) != 0) {
                fn1 = cpl_frame_get_filename(
                    cpl_frameset_find_const(frameset, ERIS_IFU_PRO_FLAT_BPM));
                bp_mask_flat = cpl_mask_load(fn1, 0, 0);
                cpl_image_set_bpm(flatFieldImg, bp_mask_flat);
            }
            cpl_image *ffErrorImg = cpl_image_new(2048, 2048, CPL_TYPE_DOUBLE);
            masterFlatHdrlImg_lo = hdrl_image_create(flatFieldImg, ffErrorImg);
            cpl_image_delete(flatFieldImg);
            cpl_image_delete(ffErrorImg);
            BRK_IF_NULL(
                qc_list = cpl_propertylist_new());
        } else {
            // get flat-mode
            const char *tmp = cpl_parameter_get_string(
                                    cpl_parameterlist_find_const(parlist,
                                             "eris.eris_ifu_distortion.mode"));
            if (!strcmp(tmp, flatModes[0])) {
                mode = FLAT_MODE_SEGMENT;
            } else if (!strcmp(tmp, flatModes[1])) {
                mode = FLAT_MODE_HDRL;
            } else if (!strcmp(tmp, flatModes[2])) {
                mode = FLAT_MODE_FAST;
            } else {
                BRK_WITH_ERROR_MSG(CPL_ERROR_ILLEGAL_INPUT, "Wrong mode provided.");
            }
            CHECK_ERROR_STATE();

            BRK_IF_ERROR(
                eris_ifu_flat_static(frameset_flat,
                                     parlist,
                                     mode,
                                     productDepth,
                                     ERIS_IFU_PRO_DIST_DBG_MASTER_FLAT_LAMP,
                                     REC_NAME_DISTORTION,
                                     "DIST",
                                     instr_tmp,
                                     &qc_list,
                                     &masterFlatHdrlImg_lo,
                                     &qualityImage));

            /*  fetch master flat image */
            BRK_IF_NULL(
                masterFlatImage = hdrl_image_get_image_const(masterFlatHdrlImg_lo));
            /* fetch noise image */
            BRK_IF_NULL(
                errorImage = hdrl_image_get_error(masterFlatHdrlImg_lo));

            if ((masterFlatImage != NULL) && (errorImage != NULL) &&
                (qualityImage != NULL) && ((productDepth & 1) != 0))
            {
                BRK_IF_ERROR(
                    cpl_propertylist_set_string(qc_list, CPL_DFS_PRO_CATG,
                         ERIS_IFU_PRO_DIST_DBG_MASTER_FLAT_LAMP));
                // save masterflat image
                BRK_IF_ERROR(
                    eris_ifu_save_deq_image(frameset, NULL, parlist, frameset, NULL,
                                            REC_NAME_DISTORTION,
                                            qc_list, NULL,
                                            ERIS_IFU_PRO_DIST_DBG_MASTER_FLAT_LAMP_FN,
                                            masterFlatImage, errorImage, rmse,
                                            qualityImage, flag16bit, UNITLESS));
            }

            BRK_IF_NULL(
                bp_mask_flat = hdrl_image_get_mask(masterFlatHdrlImg_lo));
        }

        // save bpm image
        if (bp_mask_flat != NULL) {
            BRK_IF_NULL(
                bp_img_flat = cpl_image_new_from_mask(bp_mask_flat));
            eris_ifu_get_badpix_qc_from_ima(bp_img_flat, qc_list, "DISTORTION");

            BRK_IF_ERROR(
                    eris_ifu_save_image(frameset, qc_list, parlist,
                                REC_NAME_DISTORTION,
                                ERIS_IFU_PRO_DIST_BPM,
                                ERIS_IFU_PRO_DIST_BPM_FN,
                                CPL_TYPE_USHORT,
                                bp_img_flat));
        }

        eris_ifu_free_propertylist(&qc_list);

        /* -----------------------------------------------------------------
         *   Load dark frame if present
         * ----------------------------------------------------------------- */
        BRK_IF_NULL(
            qc_list = cpl_propertylist_new());

        // load DARK_NS frame (optional)
        fr = cpl_frameset_find_const(frameset, ERIS_IFU_RAW_DARK_NS);
        if (fr != NULL) {
            BRK_IF_NULL(
                fn = cpl_frame_get_filename(fr));
            BRK_IF_ERROR(
                eris_ifu_file_exists(fn));
            BRK_IF_NULL(
                dark_img = eris_ifu_load_exposure_file(fn, 0, NULL));
            if (bp_mask_flat != NULL) {
                BRK_IF_ERROR(
                    hdrl_image_reject_from_mask(dark_img, bp_mask_flat));
            }
            if ((productDepth & 1) != 0) {
                // save dark image
                eris_ifu_save_hdrl_image_dbg(dark_img,
                                       ERIS_IFU_PRO_DIST_DBG_DARK_FN,
                                       TRUE, NULL);
            }
        }

        /* -----------------------------------------------------------------
         *   Stack fibre frames
         * ----------------------------------------------------------------- */
        cpl_msg_info(cpl_func, "Performing fibre-exposure calculation...");

        // load all FIBRE_NS frames
        BRK_IF_NULL(
            frameset_fibre = eris_ifu_get_frameset_by_tag(frameset,
                                                        ERIS_IFU_RAW_FIBRE_NS)); 

        if (cpl_propertylist_has(pl, FHDR_DPR_TYPE)) {
            BRK_IF_NULL(
                dpr_type = cpl_propertylist_get_string(pl, FHDR_DPR_TYPE));
        }
        if ((strcmp(dpr_type, "NS,SLIT") != 0) &&
            (cpl_frameset_get_size(frameset_fibre) < 75))
        {
            // only for mask-mode we expect 75 frames
            cpl_msg_warning(cpl_func, "Normally 75 FIBRE_NS frames are provided!"
                                      " (instead of %d)",
                                    (int)cpl_frameset_get_size(frameset_fibre));
        }

        BRK_IF_NULL(
            fibre_imglist = hdrl_imagelist_new());

        BRK_IF_NULL(
            fr = cpl_frameset_find_const(frameset_fibre,
                                         ERIS_IFU_RAW_FIBRE_NS));

        while (fr != NULL) {
            BRK_IF_NULL(
                fn = cpl_frame_get_filename(fr));
            BRK_IF_ERROR(
                eris_ifu_file_exists(fn));
            BRK_IF_NULL(
                tmp_img = eris_ifu_load_exposure_file(fn, 0, NULL));
            if (bp_mask_flat != NULL) {
                BRK_IF_ERROR(
                    hdrl_image_reject_from_mask(tmp_img, bp_mask_flat));
            }
            BRK_IF_ERROR(
                hdrl_imagelist_set(fibre_imglist, tmp_img,
                        hdrl_imagelist_get_size(fibre_imglist)));

            /* next frame */
            fr = cpl_frameset_find_const(frameset_fibre, NULL);
        }

        // collapse FIBRE_NS frames
        pcollapse = hdrl_collapse_parameter_parse_parlist(
                        parlist, "eris.eris_ifu_distortion.collapse");
        BRK_IF_ERROR(
            hdrl_imagelist_collapse(fibre_imglist, pcollapse,
                                    &fibre_on_img, &contribMap));
        eris_ifu_free_hdrl_parameter(&pcollapse);
        eris_ifu_free_image(&contribMap);

        if (productDepth & 1) {
            // save fibre_on image
            eris_ifu_save_hdrl_image_dbg(fibre_on_img,
                                   ERIS_IFU_PRO_DIST_DBG_STACKED_ON_FN,
                                   TRUE, NULL);
        }

        if (dark_img != NULL) {
            // subtract fibre_on & dark_img
            BRK_IF_ERROR(
                hdrl_image_sub_image(fibre_on_img, dark_img));
            eris_ifu_free_hdrl_image(&dark_img);
        }

        BRK_IF_NULL(
            fibre_div_img = hdrl_image_duplicate(fibre_on_img));

        // divide master flat
        BRK_IF_ERROR(
            hdrl_image_div_image(fibre_div_img, masterFlatHdrlImg_lo));

        if ((productDepth & 1) != 0) {
            // save dark-subtracted and flat-divided fibre_on image
            eris_ifu_save_hdrl_image_dbg(fibre_div_img,
                                   ERIS_IFU_PRO_DIST_DBG_STACKED_FN,
                                   TRUE, NULL);
        }

        /* ---------------------------------------------------------
         *          Load and subtract wavecal
         * --------------------------------------------------------- */
        cpl_msg_info(cpl_func, "Performing wave-cal calculation...");
        cpl_frameset* frameset_wave = NULL;
        if (cpl_frameset_count_tags(frameset, ERIS_IFU_RAW_WAVE_NS) > 0) {
        	frameset_wave = eris_ifu_get_frameset_by_tag(frameset,
        			ERIS_IFU_RAW_WAVE_NS);
        } else if ( (cpl_frameset_count_tags(frameset, ERIS_IFU_RAW_WAVE_NS_ON) > 0) &&
        		(cpl_frameset_count_tags(frameset, ERIS_IFU_RAW_WAVE_NS_OFF) > 0) ) {

        	cpl_frameset* frameset_wave_off = NULL;
        	frameset_wave = eris_ifu_get_frameset_by_tag(frameset, ERIS_IFU_RAW_WAVE_NS_ON);

        	frameset_wave_off = eris_ifu_get_frameset_by_tag(frameset, ERIS_IFU_RAW_WAVE_NS_OFF);

        	cpl_frameset_join(frameset_wave, frameset_wave_off);
        	cpl_frameset_delete(frameset_wave_off);
        }
        BRK_IF_NULL(
            wave_imglist = hdrl_imagelist_new());


        
        cpl_size frameCnt = cpl_frameset_get_size(frameset_wave);
        for (cpl_size i = 0 ; i < frameCnt ; i++) {
        
        	fr = cpl_frameset_get_position_const(frameset_wave, i) ;
                fn = cpl_frame_get_filename(fr);

                eris_ifu_file_exists(fn);

                tmp_img = eris_ifu_load_exposure_file(fn, 0, NULL);
            if (bp_mask_flat != NULL) {

                    hdrl_image_reject_from_mask(tmp_img, bp_mask_flat);
            }
            if (!eris_ifu_frame_is_on(fr)) {
                if (wave_off_img != NULL) {
                    // an off-image has already been loaded
                    eris_ifu_free_hdrl_image(&wave_off_img);
                }
                wave_off_img = tmp_img;
            } else {

                    hdrl_imagelist_set(wave_imglist, tmp_img,
                            hdrl_imagelist_get_size(wave_imglist));
            }

        }
	cpl_frameset_delete(frameset_wave);

        // collapse on WAVE_NS frames
        pcollapse = hdrl_collapse_parameter_parse_parlist(
                        parlist, "eris.eris_ifu_distortion.collapse");
        BRK_IF_ERROR(
            hdrl_imagelist_collapse(wave_imglist, pcollapse,
                                    &wave_on_img, &contribMap));
        eris_ifu_free_hdrl_parameter(&pcollapse);
        eris_ifu_free_image(&contribMap);

        if ((productDepth & 2) != 0) {
            eris_ifu_save_hdrl_image_dbg(wave_on_img,
                                  "eris_ifu_distortion_dbg_wave_on",
                                  TRUE, NULL);
            eris_ifu_save_hdrl_image_dbg(wave_off_img,
                                  "eris_ifu_distortion_dbg_wave_off",
                                  TRUE, NULL);
        }

        hdrl_image_sub_image(wave_on_img, wave_off_img);

        eris_ifu_free_hdrl_image(&wave_off_img);
        eris_ifu_free_hdrl_imagelist(&wave_imglist);

        if ((productDepth & 2) != 0) {
            eris_ifu_save_hdrl_image_dbg(wave_on_img,
                                  "eris_ifu_distortion_dbg_wave_sub",
                                  TRUE, NULL);
        }

        /* save additional products if requested via productDepth argument */
        do_wave_flat_div = cpl_parameter_get_bool(
                                cpl_parameterlist_find_const(parlist,
                                "eris.eris_ifu_distortion.wave.div"));
        CHECK_ERROR_STATE();

        if (do_wave_flat_div) {
            // divide master flat
            BRK_IF_ERROR(
                hdrl_image_div_image(wave_on_img, masterFlatHdrlImg_lo));

            if ((productDepth & 2) != 0) {
                eris_ifu_save_hdrl_image_dbg(wave_on_img,
                                         "eris_ifu_distortion_dbg_wave_div",
                                         TRUE, NULL);
            }
        }

        /* -----------------------------------------------------------------
         *   Measure center positions based on fibre_on
         * ----------------------------------------------------------------- */
        cpl_msg_info(cpl_func, "Performing trace detection of slit-mask/fibre...");

        // vertical arc detection --> return centers (20 per arc)
        BRK_IF_NULL(
            centers_fitted = eris_ifu_dist_calc_centers(fibre_div_img, fibre_on_img,
                                                        /*is25mas, isK, */productDepth));

        /* -----------------------------------------------------------------
         *   Calculate slitlet positions based on wave_cal
         * ----------------------------------------------------------------- */
        cpl_msg_info(cpl_func, "Performing slitlet edge detection...");

//AAA
        cpl_image **arcImgs = NULL;
        int arcImgCnt = 0;
        BRK_IF_NULL(
            valid_arc_lines = eris_ifu_dist_wave(frameset, centers_fitted,
                productDepth, &arcImgs, &arcImgCnt, parlist, &qclog));
        // left/right edge detection --> return edges (as many as there are lines)
        BRK_IF_NULL(
            slit_edges = eris_ifu_dist_calc_slitpos(arcImgs, centers_fitted,
                                                    valid_arc_lines,
//                                                    NULL,
                                                     productDepth,
                                                    &cut_off_left, &cut_off_right/*, isH*/,
													frameset,parlist));
        for (int ix = 0; ix < arcImgCnt; ix++) {
            cpl_image *tmpImg = arcImgs[ix];
            eris_ifu_free_image(&tmpImg);
        }
        cpl_free(arcImgs);
        /* -----------------------------------------------------------------
         *   Calculate distortion
         * ----------------------------------------------------------------- */
        cpl_msg_info(cpl_func, "Performing calculation of distortion (slit)");
        BRK_IF_NULL(
            distortion_poly = eris_ifu_dist_calc_distortion(slit_edges,
                                                            centers_fitted,
                                                            productDepth,
                                                            cut_off_left,
                                                            cut_off_right,
                                                            &minmax_borders,
                                                            &qc, pl,
							    frameset, parlist));
        BRK_IF_ERROR(
            eris_ifu_dist_save_distortion(distortion_poly,
                                          minmax_borders,
                                          ERIS_IFU_PRO_DIST_DISTORTION_FN,
                                          frameset, parlist, qc));

        /* ---------------------------------------------------------
         * Apply distortion
         * --------------------------------------------------------- */

        if (productDepth == PD_AUXILLIARY || productDepth == PD_DEBUG) {
            BRK_IF_ERROR(
                eris_ifu_dist_save_aux_products(frameset, parlist, pl,
                    distortion_poly, minmax_borders, fibre_div_img));
        }
        if (productDepth & 4) {
            cpl_msg_info(cpl_func, "Distorting arc-frame by slitlet...") ;
            BRK_IF_NULL(
                warped_img_wave = eris_ifu_dist_warp_image(wave_on_img,
                                                           distortion_poly,
                                                           minmax_borders));
            BRK_IF_ERROR(
                eris_ifu_save_image(frameset, qc_list, parlist,
                                    REC_NAME_DISTORTION,
                                    ERIS_IFU_PRO_DIST_DBG_WARPED_WAVE,
                                    ERIS_IFU_PRO_DIST_DBG_WARPED_WAVE_FN,
                                    CPL_TYPE_DOUBLE,
                                    hdrl_image_get_image_const(warped_img_wave)));
        }
        if (productDepth & 16) {
            cpl_msg_info(cpl_func, "Performing calculation of distortion (full)");
            BRK_IF_NULL(
                distortion_poly_full = eris_ifu_dist_calc_distortion_full(slit_edges,
                                                                          centers_fitted,
                                                                          productDepth,
                                                                          cut_off_left,
                                                                          cut_off_right));
            BRK_IF_ERROR(
                eris_ifu_dist_save_distortion(distortion_poly_full,
                                              minmax_borders,
                                              "eris_ifu_distortion_distortion_full.fits",
                                              frameset, parlist, qc));

            cpl_msg_info(cpl_func, "Distorting fibre-frame full...") ;
            BRK_IF_NULL(
                warped_img_full = eris_ifu_dist_warp_image_full(fibre_div_img,
                                                                distortion_poly_full,
                                                                productDepth));

            BRK_IF_ERROR(
                eris_ifu_save_image(frameset, qc_list, parlist,
                                    REC_NAME_DISTORTION,
                                    ERIS_IFU_PRO_DIST_DBG_WARPED_FULL,
                                    ERIS_IFU_PRO_DIST_DBG_WARPED_FULL_FN,
                                    CPL_TYPE_DOUBLE,
                                    hdrl_image_get_image_const(warped_img_full)));

            BRK_IF_NULL(
                cube = eris_ifu_stack_warped(warped_img_full, rowIndicesDet));

            BRK_IF_NULL(
                tmpImgList = eris_ifu_hdrl_get_imagelist(cube, eris_hdrl_data));
            BRK_IF_ERROR(
                eris_ifu_save_imagelist_dbg(tmpImgList,
                    ERIS_IFU_PRO_DIST_DBG_WARPED_CUBE_FULL_FN, CPL_IO_CREATE));

            BRK_IF_NULL(
                cube_sorted = eris_ifu_stack_warped(warped_img_full, rowIndices));

            eris_ifu_free_imagelist(&tmpImgList);
            BRK_IF_NULL(
                tmpImgList = eris_ifu_hdrl_get_imagelist(cube_sorted, eris_hdrl_data));
            BRK_IF_ERROR(
                eris_ifu_save_imagelist_dbg(tmpImgList,
                    ERIS_IFU_PRO_DIST_DBG_WARPED_CUBE_FULL_FN, CPL_IO_EXTEND));
        }
    }
    CATCH
    {
       //ATCH_MSGS();
    }



    eris_ifu_free_imagelist(&tmpImgList);
    cpl_table_delete(qclog);
    //eris_ifu_free_table(&qclog);
    eris_ifu_free_propertylist(&qc_list);
    eris_ifu_free_propertylist(&pl);
    eris_ifu_free_frameset(&frameset_flat);
    eris_ifu_free_frameset(&frameset_fibre);
    eris_ifu_free_hdrl_image(&masterFlatHdrlImg_lo);
    eris_ifu_free_image(&qualityImage);
    eris_ifu_free_image(&bp_img_flat);
    eris_ifu_free_table(&valid_arc_lines);
    eris_ifu_free_hdrl_imagelist(&fibre_imglist);
    eris_ifu_free_table(&minmax_borders);
    eris_ifu_free_hdrl_imagelist(&wave_imglist);
    eris_ifu_free_hdrl_imagelist(&cube);
    eris_ifu_free_hdrl_imagelist(&cube_sorted);
    eris_ifu_free_hdrl_image(&dark_img);
    eris_ifu_free_hdrl_image(&fibre_on_img);
    eris_ifu_free_hdrl_image(&fibre_div_img);
    eris_ifu_free_hdrl_image(&warped_img_full);
    eris_ifu_free_hdrl_image(&wave_on_img);
    eris_ifu_free_hdrl_image(&wave_off_img);
    eris_ifu_free_hdrl_image(&warped_img_wave);
    eris_ifu_free_hdrl_image(&warped_img_full);
    eris_ifu_free_hdrl_parameter(&pcollapse);
    eris_ifu_free_image(&contribMap);
    eris_ifu_free_string(&str);

    if (centers_fitted != NULL) {
        for (int j = 0; j < SLITLET_CNT; j++) {
            eris_ifu_free_table(&centers_fitted[j]);
        }
        cpl_free(centers_fitted); centers_fitted = NULL;
    }
    if (slit_edges != NULL) {
        for (int j = 0; j < SLITLET_CNT; j++) {
            eris_ifu_free_table(&slit_edges[j]);
        }
        cpl_free(slit_edges); slit_edges = NULL;
    }
    if (distortion_poly != NULL) {
        for (int j = 0; j < SLITLET_CNT; j++) {
            eris_ifu_free_polynomial(&distortion_poly[j]);
        }
        cpl_free(distortion_poly); distortion_poly = NULL;
    }
    if (qc != NULL) {
        for (int i = 0; i < SLITLET_CNT; i++) {
            eris_ifu_free_propertylist(&qc[i]);
        }
        cpl_free(qc); qc = NULL;
    }

    return (int) cpl_error_get_code();
}



/*---------------------------------------------------------------------------*/
/**
  @brief    function to save additional products
  @param    frameset   the frames list
  @param    parlist     the parameters list
  @param    pl property list
  @param    distortion_poly   polynomial with distortion coefficients
  @param    minmax_borders table with information on slitlets borders min/max pos
  @param    fibre_div_img  image with fibres
  @return   0 if everything is ok
 */
/*---------------------------------------------------------------------------*/

cpl_error_code eris_ifu_dist_save_aux_products(
        cpl_frameset *frameset,
        const cpl_parameterlist *parlist,
        cpl_propertylist *pl,
        cpl_polynomial      **distortion_poly,
        cpl_table *minmax_borders,
        hdrl_image *fibre_div_img )
{
    cpl_propertylist *qc_list = NULL;
    cpl_image *tmp_cpl                = NULL;

    hdrl_image *warped_img             = NULL,
            *tmp_hdrl = NULL;
    hdrl_imagelist*cube = NULL;

    TRY {
        BRK_IF_NULL(
            qc_list = cpl_propertylist_new());

        cpl_msg_info(cpl_func, "Distorting fibre-frame by slitlet...") ;

        BRK_IF_NULL(
            warped_img = eris_ifu_dist_warp_image(fibre_div_img,
                                                  distortion_poly,
                                                  minmax_borders));
        BRK_IF_ERROR(
            eris_ifu_save_image(frameset, qc_list, parlist,
                                REC_NAME_DISTORTION,
                                ERIS_IFU_PRO_DIST_DBG_WARPED,
                                ERIS_IFU_PRO_DIST_DBG_WARPED_FN,
                                CPL_TYPE_DOUBLE,
                                hdrl_image_get_image_const(warped_img)));

        BRK_IF_NULL(
            cube = eris_ifu_stack_warped(warped_img, rowIndicesDet));

        cpl_imagelist *tmpImgList = NULL;
        BRK_IF_NULL(
            tmpImgList = eris_ifu_hdrl_get_imagelist(cube, eris_hdrl_data));
        BRK_IF_ERROR(
            eris_ifu_save_imagelist_dbg(tmpImgList,
                                        ERIS_IFU_PRO_DIST_DBG_WARPED_CUBE_FN,
                                        CPL_IO_CREATE));
        eris_ifu_free_imagelist(&tmpImgList);

        BRK_IF_ERROR(
            hdrl_imagelist_collapse_mean(cube, &tmp_hdrl, &tmp_cpl));
        //PIPPO
                cpl_propertylist_set_string(pl,FHDR_PRO_CATG,
                		ERIS_IFU_PRO_DIST_DBG_WARPED_CUBE_FLAT_MEAN);
                cpl_propertylist_set_string(pl,"PIPEFILE",
                                		ERIS_IFU_PRO_DIST_DBG_WARPED_CUBE_FLAT_MEAN_FN);
        BRK_IF_ERROR(
        		eris_ifu_save_image_dbg(hdrl_image_get_image(tmp_hdrl),
        				ERIS_IFU_PRO_DIST_DBG_WARPED_CUBE_FLAT_MEAN_FN,
						CPL_IO_CREATE, pl));
        eris_ifu_free_hdrl_image(&tmp_hdrl);
        eris_ifu_free_image(&tmp_cpl);


        eris_ifu_free_hdrl_imagelist(&cube);
        BRK_IF_NULL(
            cube = eris_ifu_stack_warped(warped_img, rowIndices));

        BRK_IF_NULL(
            tmpImgList = eris_ifu_hdrl_get_imagelist(cube, eris_hdrl_data));
        BRK_IF_ERROR(
            eris_ifu_save_imagelist_dbg(tmpImgList,
                                        ERIS_IFU_PRO_DIST_DBG_WARPED_CUBE_FN,
                                        CPL_IO_EXTEND));
        eris_ifu_free_imagelist(&tmpImgList);


        BRK_IF_ERROR(
            hdrl_imagelist_collapse_mean(cube, &tmp_hdrl, &tmp_cpl));

        BRK_IF_ERROR(
            eris_ifu_save_image_dbg(hdrl_image_get_image(tmp_hdrl),
                                 ERIS_IFU_PRO_DIST_DBG_WARPED_CUBE_FLAT_MEAN_FN,
								 CPL_IO_EXTEND, pl));
        eris_ifu_free_hdrl_image(&tmp_hdrl);
        eris_ifu_free_image(&tmp_cpl);

        BRK_IF_ERROR(
                eris_ifu_dist_warp_stats(warped_img, qc_list, pl, frameset, parlist));

    } CATCH {
    }
    eris_ifu_free_propertylist(&qc_list);
    eris_ifu_free_hdrl_image(&warped_img);
    eris_ifu_free_hdrl_imagelist(&cube);

    return cpl_error_get_code();
}

