/* $Id$
 *
 * This file is part of the ERIS Pipeline
 * Copyright (C) 2017 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/*
 * $Author$
 * $Date$
 * $Revision$
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                Includes
 -----------------------------------------------------------------------------*/

#include <string.h>
#include <eris_nix_dfs.h>
#include <eris_nix_scired.h>
#include <hdrl.h>

#include <cpl.h>

/*-----------------------------------------------------------------------------
                            Static variables
 -----------------------------------------------------------------------------*/

static const char eris_nix_img_cal_phot_description[] =
"This recipe calibrates the photometry of ERIS/NIX frames. It does this \n"
"in one of 3 ways. \n"
"\n"
"1. If a frame that has already been photometrically calibrated is\n"
"   present in the input set of files (SoF), then the photometric\n"
"   zeropoint is transferred from that to the target images.\n"
"\n"
"   Input files:\n"
"    DO CATG                    Explanation        Req.  #Frames\n"
"    -------                    -----------        ---   -------\n"
"    "ERIS_NIX_IMG_STD_COMBINED_PRO_CATG
                    "           photometrically     Y        1\n"
"                               calibrated file\n"
"    "ERIS_NIX_CAL_WCS_OBJECT_JITTER_PRO_CATG
                         "      files to be         Y       1-n\n"
"                               calibrated\n"
"\n"
"   Output files:\n"
"    DO CATG                    Explanation      Name\n"
"    -------                    -----------      ----\n"
"    "ERIS_NIX_CAL_PHOT_OBJECT_JITTER_PRO_CATG
                          "     Calibrated       cal_phot.<in_name>\n"
"                               files\n"
"\n"
"    The structure of the output files is the same as the input\n"
"    except for some additional keywords defining the zeropoint,\n"
"    stored in the primary header and the header of the extension\n"
"    holding catalogued objects, if present:\n"
"    --- ESO.QC.MAGZPT = The zeropoint transferred from the input\n"
"                        calibrated image.\n"
"    --- ESO.QC.MAGZERR = The zeropoint sigma transferred from\n"
"                         the input calibrated image.\n"
"    --- PHOTZP = The zeropoint taking into account the exposure\n"
"                 time and extinction for this image. This\n"
"                 correction can be applied directly to the image\n"
"                 data.\n"
"    --- PHOTZPER = The uncertainty of PHOTZP.\n"
"    --- ZPMETHOD = 'separate_STD_star'\n"
"    --- ZP_CAT = <value of parameter cdssearch_astron>\n"
"\n"
"2. If there is no photometrically calibrated frame in the SoF \n"
"   then the recipe will search for a standard star in each frame\n"
"   by trying to associate an object found there with an entry in a \n"
"   catalogue of photometric standards, specified using parameter\n"
"   cdssearch-photom. Currently, the method works only if there is\n"
"   a single standard star in the field.\n"
"\n"
"   Input files:\n"
"    DO CATG                    Explanation        Req.  #Frames\n"
"    -------                    -----------        ---   -------\n"
"    "ERIS_NIX_CAL_WCS_OBJECT_JITTER_PRO_CATG
                         "      target frames with  Y      1-n\n"
"                               absolutely calibrated\n"
"                               wcs, that is with\n" 
"                               ESO.WCS_METHOD = \n"
"                               'science_field_standard'\n"
"    "ERIS_NIX_PHOT_DATA_PRO_CATG
             "                  a photometry data\n"
"                               file with colour\n"
"                               equations relating\n"
"                               the catalogue\n"
"                               filters to those\n"
"                               of NIX\n"
"\n"
"   Output files:\n"
"    DO CATG                    Explanation      Name\n"
"    -------                    -----------      ----\n"
"    "ERIS_NIX_CAL_PHOT_OBJECT_JITTER_PRO_CATG
                          "     Calibrated       cal_phot.<in_name>\n"
"                               files\n"
"\n"
"    The structure of each output file is the same as the input\n"
"    except for some added keywords defining the zeropoint stored\n"
"    in the FITS headers, and the addition of a new extension\n"
"    named MATCHSTD_PHOT with details of the matched photometric\n"
"    standards. The new header items are:\n"
"    -- ESO.QC.MAGZPT    The zeropoint calculated through aperture 3\n"
"                        using objects from the current SoF images.\n"
"    -- ESO.QC.MAGZERR   The zeropoint sigma calculated through\n"
"                        aperture 3 using objects from the current\n"
"                        SoF images.\n"
"    -- ESO.QC.MAGNZPT   The number of objects used to calculate the\n"
"                        zeropoint.\n"
"    -- ESO.QC.LIMITING_MAG The 5 sigma limiting magnitude through\n"
"                        aperture 3.\n"
"    -- PHOTZP           The zeropoint taking into account the\n"
"                        exposure time and extinction for this image.\n"
"    -- PHOTZPER         The uncertainty of PHOTZP.\n"
"    -- ZPMETHOD         = '2MASS'\n"
"\n"
"   Calculation of the photometric zeropoint for each frame is done\n"
"   using the same CASU routines as are used for HAWK-I. The images\n"
"   in the SoF are treated together, as if they were from the 4 \n"
"   HAWKI-I detectors.\n"
"\n"
"3. If there is no photometrically calibrated frame in the SoF\n"
"   and either their astrometric calibration is not absolute or no\n"
"   standard stars are found in the target frames, then the\n"
"   assigned zeropoint will be the default value for this filter,\n"
"   copied from the photometric data file.\n"
"\n"
"   The inputs required are the same as for case 2. The outputs\n"
"   will be the same as for case 1, except for:\n"
"   -- ZPMETHOD          = 'DEFAULT'\n"
"\n";

#define RECIPE_NAME "eris_nix_img_cal_phot"
#define CONTEXT "eris."RECIPE_NAME

/*-----------------------------------------------------------------------------
                            Private function prototypes
 -----------------------------------------------------------------------------*/

cpl_recipe_define(eris_nix_img_cal_phot, ERIS_BINARY_VERSION,
                  "John Lightfoot",
                  PACKAGE_BUGREPORT, "2017",
                  "Calibrate photometry of ERIS/NIX frames",
                  eris_nix_img_cal_phot_description);

/*-----------------------------------------------------------------------------
                                Function code
 -----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Setup the recipe options    
  @param    self  the non-NULL parameterlist to fill
  @return   CPL_ERROR_NONE iff everything is ok

  Defining the command-line/configuration parameters for the recipe.
 */
/*----------------------------------------------------------------------------*/

static cpl_error_code eris_nix_img_cal_phot_fill_parameterlist(
  cpl_parameterlist * self) {

    if (cpl_error_get_code() != CPL_ERROR_NONE) return cpl_error_get_code();

    hdrl_parameter    * catalogue_defaults = NULL;
    cpl_parameterlist * catalogue_parlist = NULL;
    cpl_parameter     * p = NULL;

    /* Flag to decide how we get the photometric standard star information. */
       
    p = cpl_parameter_new_enum(CONTEXT".cdssearch_photom",
                               CPL_TYPE_STRING,
                               "CDS photometric catalogue",
                               CONTEXT,
                               "2MASS", 2, "none", "2MASS");
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "cdssearch_photom");
    cpl_parameterlist_append(self, p);

    /* parameter list for object catalogue generation */
    /* ..generate the default list first */

    catalogue_defaults = hdrl_catalogue_parameter_create(20, 3.0, CPL_TRUE,
                         10.0, CPL_TRUE, 128, 5.0, 3.0,
                         ERIS_NIX_SATURATION_DEFAULT, HDRL_CATALOGUE_ALL);
    catalogue_parlist = hdrl_catalogue_parameter_create_parlist(CONTEXT,
                        "catalogue", catalogue_defaults);

    /* ..subset the parameters actually used */

    for (p = cpl_parameterlist_get_first(catalogue_parlist);
         p != NULL;
         p = cpl_parameterlist_get_next(catalogue_parlist)) {

        const char * pname = cpl_parameter_get_name(p);
        if (strstr(pname, "minmax") == NULL) {
            cpl_parameter * duplicate = cpl_parameter_duplicate(p);
            cpl_parameterlist_append(self, duplicate);
        }
    }

    /* ..add ao-params which specifies how to set params depending on AO */

    p = cpl_parameter_new_enum(CONTEXT".catalogue.ao-params",
                               CPL_TYPE_STRING,
                                "Default catalogue.core-radius and "
                                "catalogue.mesh-size depending on "
                                "AOMODE, or not",
                                CONTEXT,
                                "auto", 2, "auto", "user");
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "catalogue.ao-params");
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, p);

    /* parameters for object-catalogue matching algorithm */

    p = cpl_parameter_new_value(CONTEXT".pixel_radius",
                                CPL_TYPE_DOUBLE,
                                "Max. distance between object and catalogue "
                                "entry for association (pixels)",
                                CONTEXT, 5.0);
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "pixel-radius");
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, p);

    p = cpl_parameter_new_range(CONTEXT".minphotom",
                                CPL_TYPE_INT,
                                "Min number of matched stars for photometric "
                                "calibration", CONTEXT, 1, 1, 100000);
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "minphotom");
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, p);

    p = cpl_parameter_new_value(CONTEXT".magerrcut",
                                CPL_TYPE_DOUBLE,
                                "Matched stars with magnitude error above "
                                "this cutoff will not be used.",
                                CONTEXT, 0.5);
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "magerrcut");
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, p);

    p = cpl_parameter_new_value(CONTEXT".debug-data",
                                CPL_TYPE_BOOL, "true to save interim results",
                                CONTEXT, CPL_FALSE);
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "debug-data");
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, p);

    /* tidy up */

    hdrl_parameter_delete(catalogue_defaults);
    cpl_parameterlist_delete(catalogue_parlist);

    return 0;
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Interpret the command line options and execute the data processing
  @param    frameset   the frames list
  @param    parlist    the parameters list
  @return   0 if everything is ok
 */
/*----------------------------------------------------------------------------*/

static int eris_nix_img_cal_phot(cpl_frameset * frameset,
                                 const cpl_parameterlist * parlist) {

    eris_nix_scired_cal_phot(frameset, parlist, RECIPE_NAME, CONTEXT);

    return (int) cpl_error_get_code();
}
