/* $Id$
 *
 * This file is part of the ERIS Pipeline
 * Copyright (C) 2017 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/*
 * $Author$
 * $Date$
 * $Revision$
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                Includes
 -----------------------------------------------------------------------------*/

#include <string.h>
#include "eris_nix_utils.h"
#include "eris_nix_dfs.h"
#include "eris_nix_scired.h"
#include <hdrl.h>

#include <cpl.h>

/*-----------------------------------------------------------------------------
                            Static variables
 -----------------------------------------------------------------------------*/

static const char eris_nix_img_cal_wcs_description[] =
"This recipe calibrates the astrometry of ERIS/NIX frames.\n"
"\n"
" Input files:\n"
"   DO CATG                      Explanation        Req.  #Frames\n"
"   -------                      -----------        ---   -------\n"
"   "ERIS_NIX_SKYSUB_OBJECT_JITTER_PRO_CATG
                       "         sky-subtracted     Y      1-n\n"
"                                target frames\n"
"                  or\n"
"   "ERIS_NIX_SKYSUB_STD_JITTER_PRO_CATG
                    "            sky-subtracted     Y      1-n\n"
"                                STD frames\n"
"   "ERIS_NIX_WCS_MATCHED_CATALOGUE_PRO_CATG
                        "        manual match of    N      0-1\n"
"                                stars in first\n"
"                                jitter to RA, Dec\n"
"\n"
" Output files:\n"
"\n"
"   DO CATG                      Explanation        Name\n"
"   -------                      -----------        ----\n"
"   "ERIS_NIX_CAL_WCS_OBJECT_JITTER_PRO_CATG           
                        "        The wcs            cal_wcs.<in_name>\n"
"                                calibrated\n"
"                                target frame\n"
"                  or\n"
"   "ERIS_NIX_CAL_WCS_STD_JITTER_PRO_CATG
                     "           The wcs            cal_wcs.<in_name>\n"
"                                calibrated\n"
"                                STD frame\n"
"\n"
" There are two stages to the calibration. \n"
"\n"
" First, the jitter nearest the centre of the jitter pattern is \n"
" identified as the 'reference' jitter. The wcs of the 'reference'\n"
" jitter is calibrated in one of 3 ways as described further below.\n"
" The aim is to calibrate this jitter against an absolute\n"
" reference such as Gaia.\n"
"\n"
" Second, the wcs calibration is propagated to other jitters in\n"
" turn, adding them to the calibrated pool as we go. At each step\n"
" the next jitter to calibrate is the one with the largest overlap\n"
" with a jitter in the calibrated pool, so the most likely to have\n"
" sources in common. The calibration is achieved by comparing their\n"
" source catalogues.\n"
"\n"
"\n"
" Catalogues are matched by trying all possible associations between \n"
" their objects and calculating a metric of the pattern match for \n"
" each - that is, the number of object pairs that lie within a \n"
" small distance of each other. The association with the highest \n"
" number of matches is selected.\n"
"\n"
" The catalogues are sorted in order of decreasing brightness \n"
" using column 'Aper_flux_3', if it exists. If few sources are \n"
" present this means that matches between the brightest \n"
" will be favoured.\n"
"\n"
" When a match is found the wcs of the image to be calibrated is \n"
" recalculated to make the matched sources coincide.\n"
"\n"
"\n"
" The 'reference' jitter can be calibrated in one of 3 ways:\n"
"\n"
"1. If an online astrometric catalogue is specified by parameter\n"
"   cdssearch-astrom it will be searched over the area of the\n"
"   jitter image for astrometric standards. \n"
"\n"
"   In this case, if the calibration succeeds the headers of the \n"
"   output files will have keywords:\n"
"   -- ESO.WCS_METHOD = '"ERIS_NIX_WCS_CATALOGUE_MATCH"'\n"
"   -- WCS_CAT = <the name of the catalogue used>\n"
"\n"
"2. If no reference catalogue is given, cdssearch-astrom is \n"
"   'none', then the wcs cannot be calibrated absolutely and \n"
"   the recipe can only align the jitters relative to the \n"
"   'reference'. \n"
"\n"
"   If the calibration succeeds the output files will have \n"
"   keywords:\n"
"   -- ESO.WCS_METHOD = '"ERIS_NIX_WCS_JITTER_RELATIVE"'\n"
"   -- WCS_CAT = 'none'\n"
"\n"
"3. The user can directly specify a position in the 'reference'\n"
"   jitter. This is done by putting the required information \n"
"   in a FITS table and including that file in the SoF with tag\n"
"   "ERIS_NIX_WCS_MATCHED_CATALOGUE_PRO_CATG".\n"
"\n"
"   The format of the FITS table should be:\n"
"\n"
"Filename  RA         Dec       X_coordinate  Y_coordinate  Catalogue\n"
"xyz.fits  99.791...  8.736...  544           599.5         Gaia\n"
"abc.fits  99.791...  8.736...  561.5         595.2         Gaia\n"
"pqr.fits  99.791...  8.736...  557.2         496.8         Gaia\n"
"   .         .          .        .             .              .\n"
"\n"
"   where:\n"
"     Filename holds the name of the 'reference' jitter.\n"
"     RA and Dec hold the coords in degrees of the index position.\n"
"     X_coordinate and Y_coordinate hold the x,y coords of the \n"
"       index position.\n"
"     Catalogue has the name of the source catalogue of RA, Dec.\n"
"\n"
"   If the 'Catalogue' entry is 'Gaia' or '2mass' then the output\n"
"   file will have keywords:\n"
"   -- ESO.WCS_METHOD = '"ERIS_NIX_WCS_CATALOGUE_MATCH"'\n"
"   -- WCS_CAT = <the name of the catalogue used>\n"
"   otherwise:\n"
"   -- ESO.WCS_METHOD = '"ERIS_NIX_WCS_JITTER_RELATIVE"'\n"
"   -- WCS_CAT = 'none'\n"
"\n"
"\n"
" If the calibration fails then the wcs of the file is not modified\n"
" and the relevant header keywords are set thus:\n"
"   -- ESO.WCS_METHOD = '"ERIS_NIX_WCS_TEL_POINTING"'\n"
"   -- WCS_CAT = 'none'\n";

#define RECIPE_NAME "eris_nix_img_cal_wcs"
#define CONTEXT "eris."RECIPE_NAME
/*-----------------------------------------------------------------------------
                            Private function prototypes
 -----------------------------------------------------------------------------*/

cpl_recipe_define(eris_nix_img_cal_wcs, ERIS_BINARY_VERSION,
                  "John Lightfoot",
                  PACKAGE_BUGREPORT, "2017",
                  "Calibrate wcs of ERIS/NIX frames",
                  eris_nix_img_cal_wcs_description);

/*-----------------------------------------------------------------------------
                                Function code
 -----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Setup the recipe options    
  @param    self  the non-NULL parameterlist to fill
  @return   CPL_ERROR_NONE iff everything is ok

  Defining the command-line/configuration parameters for the recipe.
 */
/*----------------------------------------------------------------------------*/

static cpl_error_code eris_nix_img_cal_wcs_fill_parameterlist(
                      cpl_parameterlist * self) {

    if (cpl_error_get_code() != CPL_ERROR_NONE) return cpl_error_get_code();

    hdrl_parameter    * catalogue_defaults = NULL;
    cpl_parameterlist * catalogue_parlist = NULL;
    cpl_parameter     * p = NULL;

    /* generate the general parameter list for the catalogue */

    catalogue_defaults = hdrl_catalogue_parameter_create(20, 3.0, CPL_TRUE,
                         10.0, CPL_TRUE, 128, 5.0, 3.0,
                         ERIS_NIX_SATURATION_DEFAULT, HDRL_CATALOGUE_ALL);
    catalogue_parlist = hdrl_catalogue_parameter_create_parlist(CONTEXT,
                        "catalogue", catalogue_defaults);

    /* subset of parameters to be used */

    for (p = cpl_parameterlist_get_first(catalogue_parlist);
         p != NULL;
         p = cpl_parameterlist_get_next(catalogue_parlist)) {

        const char * pname = cpl_parameter_get_name(p);
        if (strstr(pname, "minmax") == NULL) {
            cpl_parameter * duplicate = cpl_parameter_duplicate(p);
            cpl_parameterlist_append(self, duplicate);
        }
    }

    /* ..add ao-params which specifies how to set params depending on AO */

    p = cpl_parameter_new_enum(CONTEXT".catalogue.ao-params",
                               CPL_TYPE_STRING,
                                "Default catalogue.core-radius and "
                                "catalogue.mesh-size depending on "
                                "AOMODE, or not",
                                CONTEXT,
                                "auto", 2, "auto", "user");
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "catalogue.ao-params");
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, p);

    /* Flag to decide how we get the astrometric standard star information.
       If "none", then use the local catalogues specified in the sof. If not,
       then use one of the selection of catalogues available from CDS -
       only gaia has sufficient accuracy */

    p = cpl_parameter_new_enum(CONTEXT".cdssearch_astrom",
                               CPL_TYPE_STRING,
                               "CDS astrometric catalogue",
                               CONTEXT,
                               "none", 3, "none", "2mass", "gaiadr3");
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "cdssearch-astrom");
    cpl_parameterlist_append(self, p);

    p = cpl_parameter_new_value(CONTEXT".debug-data",
                                CPL_TYPE_BOOL, "true to save interim results",
                                CONTEXT, CPL_FALSE);
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "debug-data");
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, p);

    p = cpl_parameter_new_value(CONTEXT".edges-trim",
                                CPL_TYPE_STRING,
								"Trim image edges in the source detection: "
								"[txmin,tymin,txmax,tymax]. t means trim "
								"the user can trim differently from the min and "
								"max image edge values along X and Y axes",
                                CONTEXT, "30,30,30,30");
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "edges-trim");
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, p);

    /* tidy up */

    hdrl_parameter_delete(catalogue_defaults);
    cpl_parameterlist_delete(catalogue_parlist);

    return 0;
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Interpret the command line options and execute the data processing
  @param    frameset   the frames list
  @param    parlist    the parameters list
  @return   0 if everything is ok
 */
/*----------------------------------------------------------------------------*/

static int eris_nix_img_cal_wcs(cpl_frameset * frameset,
                                const cpl_parameterlist * parlist) {

	eris_nix_scired_cal_wcs(frameset, parlist, RECIPE_NAME, CONTEXT);


    return (int) cpl_error_get_code();
}
