/* $Id: eris_star_index.c,v 1.9 2012-03-03 10:18:26 amodigli Exp $
 *
 * This file is part of the X-Shooter Pipeline
 * Copyright (C) 2002,2003 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
/*
 * $Author: amodigli $
 * $Date: 2012-03-03 10:18:26 $
 * $Revision: 1.9 $
 * $Name: not supported by cvs2svn $
 */



#ifdef HAVE_CONFIG_H
#include <config.h>          /* allows the program compilation */
#endif

#include <cpl.h>
#include <string.h>
#include <math.h>
#include "eris_ifu_star_index.h"
#include "eris_utils.h"

/*----------------------------------------------------------------------------*/
/**
 * @defgroup eris_ifu_star_index     Star Index Management
 *
 * This module provides functions for managing a star index structure that
 * stores reference star spectra in a FITS file format with caching support.
 *
 * The star index maintains a table of reference stars with their coordinates
 * (RA, DEC), names, and associated spectrum data. Recent entries are cached
 * in memory for faster access, while older entries are stored on disk.
 *
 * @par Synopsis:
 * @code
 *   #include "eris_ifu_star_index.h"
 * @endcode
 */
/*----------------------------------------------------------------------------*/

struct _star_index_
{
    cpl_table* index_table;
    char* fits_file_name;
    int index_size;
    cpl_table** cache;
    int cache_size;
    int* cache_index;
};

//typedef struct _star_index_ star_index;
static const char* COL_NAME_EXTID 	= "ext_id";
static const char* COL_NAME_NAME 	= "name";
static const char* COL_NAME_RA 	= "ra";
static const char* COL_NAME_DEC 	= "dec";

static star_index* star_index_construct(const char* fits_file);
static void star_index_destruct(star_index* pindex);
// private functions
/*----------------------------------------------------------------------------*/
/**
  @brief    Create a new star index structure
  @param    fits_file   Optional FITS file name (can be NULL)
  @return   Pointer to newly allocated star index structure, or NULL on error
  
  Creates and initializes a new star index structure. If fits_file is provided,
  the filename is stored for later use. The structure must be freed using
  star_index_delete().
  
  @note This is an internal function. Use star_index_create() or star_index_load()
        to create a star index structure.
 */
/*----------------------------------------------------------------------------*/
static star_index* star_index_construct(const char* fits_file)
{
    star_index* pret = cpl_malloc(sizeof(star_index));
    pret->index_size = 0;
    pret->index_table = 0;
    pret->cache_size = 0;
    pret->cache = 0;
    pret->cache_index = 0;
    if (fits_file)
    {
        size_t bt = strlen(fits_file) * sizeof(*fits_file)+1;
        pret->fits_file_name = cpl_malloc(bt);
        strcpy(pret->fits_file_name, fits_file);
    }
    else
    {
        pret->fits_file_name = 0;
    }
    return pret;
}
/*----------------------------------------------------------------------------*/
/**
  @brief    Destroy a star index structure and free all associated memory
  @param    pindex   Pointer to star index structure to destroy
  
  Frees all memory associated with the star index, including cached tables,
  the index table, and the filename string. Safe to call with NULL pointer.
  
  @note This is an internal function. Use star_index_delete() to destroy
        a star index structure.
 */
/*----------------------------------------------------------------------------*/
static void star_index_destruct(star_index* pindex)
{
    if(pindex)
    {
        if (pindex->cache)
        {
            int i = 0;
            for ( i = 0; i < pindex->cache_size; i++)
            {
                cpl_table_delete(pindex->cache[i]);
            }
            cpl_free(pindex->cache);
            pindex->cache = 0;
            pindex->cache_size = 0;
        }
        cpl_table_delete(pindex->index_table);
        if(pindex->fits_file_name)
        {
            cpl_free(pindex->fits_file_name);
        }
        cpl_free(pindex->cache_index);
        cpl_free(pindex);
    }

}

/**@{*/
/*----------------------------------------------------------------------------*/
/**
  @brief    Create a new empty star index structure
  @return   Pointer to newly allocated star index structure, or NULL on error
  
  Creates a new star index with an empty index table containing columns:
  ext_id, name, ra, and dec. The structure must be freed using star_index_delete().
  
  @note On error, the structure is automatically freed and NULL is returned.
 */
/*----------------------------------------------------------------------------*/
star_index* star_index_create(void)
{
    star_index* pret = star_index_construct(0);
    // initialize table
    pret->index_table = cpl_table_new(pret->index_size);
    // create columns ext_id, name, ra, dec
    cpl_table_new_column(pret->index_table, COL_NAME_EXTID, CPL_TYPE_INT);
    cpl_table_new_column(pret->index_table, COL_NAME_NAME, CPL_TYPE_STRING);
    cpl_table_new_column(pret->index_table, COL_NAME_RA, CPL_TYPE_DOUBLE);
    cpl_table_new_column(pret->index_table, COL_NAME_DEC, CPL_TYPE_DOUBLE);

    if(cpl_error_get_code() == CPL_ERROR_NONE) {
    	return pret;
    } else {
    	star_index_destruct(pret);
    	return 0;
    }

}
/*----------------------------------------------------------------------------*/
/**
  @brief    Load a star index structure from a FITS file
  @param    fits_file   Path to the FITS file containing the star index
  @return   Pointer to loaded star index structure, or NULL on error
  
  Loads the star index table from the first extension of the specified FITS file.
  The file must exist and be readable. The structure must be freed using
  star_index_delete().
  
  @note On error, the structure is automatically freed and NULL is returned.
        The function checks for file existence before attempting to load.
 */
/*----------------------------------------------------------------------------*/
star_index* star_index_load(const char* fits_file)
{

	cpl_ensure(fits_file, CPL_ERROR_NULL_INPUT, NULL);
	if (access(fits_file, F_OK)) {
		cpl_msg_error(cpl_func, "File %s was not found",
				fits_file);
		cpl_error_set(cpl_func, CPL_ERROR_FILE_NOT_FOUND);
		cpl_ensure(CPL_FALSE, CPL_ERROR_FILE_NOT_FOUND, NULL);
	}
    star_index* pret = star_index_construct(fits_file);
    // load index table from the file
    cpl_table* pindex = 0;
    pindex = cpl_table_load(fits_file,1,0);
    // TODO check_nomsg the structure of the table
    pret->index_table = pindex;
    pret->index_size = cpl_table_get_nrow(pindex);

    if(cpl_error_get_code() == CPL_ERROR_NONE) {
    	return pret;
    } else {
    	star_index_destruct(pret);
    	//cpl_error_reset();
    	return 0;
    }
}
/*----------------------------------------------------------------------------*/
/**
  @brief    Delete a star index structure
  @param    pindex   Pointer to star index structure to delete
  
  Frees all memory associated with the star index structure. Safe to call
  with NULL pointer.
 */
/*----------------------------------------------------------------------------*/
void star_index_delete(star_index* pindex)
{
    star_index_destruct(pindex);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Add a new star entry to the index
  @param    pindex      Pointer to star index structure
  @param    RA          Right Ascension in degrees
  @param    DEC         Declination in degrees
  @param    star_name   Name of the reference standard star
  @param    ptable      Pointer to table containing the star spectrum data
  @return   The index position (1-based) of the added entry, or 0 on error
  
  Adds a new entry to the star index with the specified coordinates and name.
  The table is cached in memory. The ext_id is automatically assigned.
  
  @note The function expands the cache as needed. The returned index is
        1-based (first entry returns 1).
 */
/*----------------------------------------------------------------------------*/
int star_index_add(star_index* pindex, double RA, double DEC,
		const char* star_name, cpl_table* ptable)
{
	cpl_ensure(pindex, CPL_ERROR_NULL_INPUT, 0);
	cpl_ensure(star_name, CPL_ERROR_NULL_INPUT, 0);
	cpl_ensure(ptable, CPL_ERROR_NULL_INPUT, 0);

    int retval = 0;
    if (pindex)
    {
        // expand the index table
        cpl_table_insert_window(pindex->index_table, pindex->index_size++, 1);
        if (!pindex->cache)
        {
            pindex->cache_size = 1;
            pindex->cache = cpl_malloc(sizeof(cpl_table*) * pindex->cache_size);
            pindex->cache_index = cpl_malloc(sizeof(pindex->cache_index[0]) * pindex->cache_size);
        }
        else
        {
            // add new entry
            pindex->cache_size++;
            pindex->cache = cpl_realloc(pindex->cache, sizeof(cpl_table*) * pindex->cache_size);
        }
        pindex->cache[pindex->cache_size - 1] = cpl_table_duplicate(ptable);
        // fill the index table with values
        cpl_table_set_string(pindex->index_table, COL_NAME_NAME, pindex->index_size - 1 ,star_name);
        cpl_table_set(pindex->index_table, COL_NAME_RA, pindex->index_size - 1 ,RA);
        cpl_table_set(pindex->index_table, COL_NAME_DEC, pindex->index_size - 1,DEC);
        cpl_table_set_int(pindex->index_table, COL_NAME_EXTID, pindex->index_size - 1 ,pindex->index_size + 1);
        retval = pindex->index_size;
    }

    if(cpl_error_get_code() == CPL_ERROR_NONE) {
    	return retval;
    }  else {
    	eris_print_rec_status(0);
    	return 0;
    }
}
/*----------------------------------------------------------------------------*/
/**
  @brief    Remove an entry from the star index by star name
  @param    pindex    Pointer to star index structure
  @param    starname  Name of the star to remove
  @return   The index position of the removed entry, or -1 if not found
  
  @note This function is currently not used.
  
  Marks the entry with the given star name as deleted by setting its ext_id
  to -1. The cached table is also freed if present. The actual data removal
  occurs during the save operation.
 */
/*----------------------------------------------------------------------------*/

/* not used */
int star_index_remove_by_name(star_index* pindex, const char* starname)
{
	cpl_ensure(pindex, CPL_ERROR_NULL_INPUT, 0);
	cpl_ensure(starname, CPL_ERROR_NULL_INPUT, 0);
    int i = 0;
    int index_pos = -1;
    for (i = 0; i < pindex->index_size; i++)
    {
        const char* curr_star_name = 0;
        curr_star_name = cpl_table_get_string(pindex->index_table, COL_NAME_NAME, i);
        if (strcmp(curr_star_name, starname) == 0)
        {
            index_pos = i;
            break;
        }
    }
    if (index_pos >= 0)
    {
        // star is found
        // clear only the index table, real data would be cleaned during save operation
        cpl_table_set_int(pindex->index_table, COL_NAME_EXTID, index_pos, -1);
        if (index_pos - pindex->index_size + pindex->cache_size  >= 0)
        {
            // clear cache
            int cache_index = index_pos - pindex->index_size + pindex->cache_size;
            cpl_table_delete(pindex->cache[cache_index]);
            pindex->cache[cache_index] = 0;
        }
    }

    return index_pos;
}
/*----------------------------------------------------------------------------*/
/**
  @brief    Save the star index to a FITS file
  @param    pindex     Pointer to star index structure
  @param    fits_file  Path to the output FITS file
  @return   Number of rows saved, or 0 on error
  
  Saves the star index table and all associated star data tables to a FITS file.
  Deleted entries (ext_id == -1) are excluded. The index table is saved as
  extension 1, and star data tables follow as subsequent extensions.
  
  @note The function filters out deleted entries before saving. Cached entries
        are saved directly, while non-cached entries are loaded from the original
        FITS file if available.
 */
/*----------------------------------------------------------------------------*/

int star_index_save(star_index* pindex, const char* fits_file)
{

	cpl_ensure(pindex, CPL_ERROR_NULL_INPUT, 0);
	cpl_ensure(fits_file, CPL_ERROR_NULL_INPUT, 0);

    int i  = 0;
    int inull = 0;
    cpl_table* pnew_index = 0;
    int nrows = 0;
    // firstly save the index table - deleted entries should be removed firstly
    cpl_table_unselect_all(pindex->index_table);
    cpl_table_or_selected_int(pindex->index_table, COL_NAME_EXTID, CPL_EQUAL_TO, -1);
    // inverse selection
    cpl_table_not_selected(pindex->index_table);
    pnew_index = cpl_table_extract_selected(pindex->index_table);

    nrows = cpl_table_get_nrow(pnew_index);
    //	printf("rows to save[%d]\n", nrows);
    for (i = 0; i < nrows; i++)
    {
        cpl_table_set_int(pnew_index, COL_NAME_EXTID, i, i+2); // ext in fits starts from 1, and another 1 is used by index_table
    }
    //	printf("writing index [%s]\n", fits_file);
    cpl_table_save(pnew_index, NULL, NULL, fits_file, CPL_IO_CREATE);
    cpl_table_delete(pnew_index);
    pnew_index = 0;
    // save the data
    for (i = 0;i < pindex->index_size; i++)
    {
        //		printf("saving ext [%d]\n", i);
        // 2. save cache
        int saved_ext = cpl_table_get_int(pindex->index_table, COL_NAME_EXTID, i, &inull);
        //		printf("saving 1\n");
        if (saved_ext > 0) // check_nomsg that was not removed
        {
            cpl_table* ptable = 0;
            //			printf("saving 2\n");
            if (i < pindex->index_size - pindex->cache_size)
            {
                //				printf("saving 3\n");
                ptable = cpl_table_load(pindex->fits_file_name, saved_ext, 0);
            }
            else
            {
                //				printf("saving 4\n");
                ptable = cpl_table_duplicate(pindex->cache[i - pindex->index_size + pindex->cache_size ]);
            }
            //			printf("saving 5\n");
            cpl_table_save(ptable, NULL, NULL, fits_file, CPL_IO_EXTEND);
            //			printf("saving 6\n");
            cpl_table_delete(ptable);
            //			printf("saving 7\n");
        }
        //		printf("saving 8\n");
    }
    //	printf("saving exit\n");

    if(cpl_error_get_code() == CPL_ERROR_NONE) {
    	return nrows;
    } else {
    	eris_print_rec_status(0);
    	return 0;
    }
}
/*----------------------------------------------------------------------------*/
/**
  @brief    Get a star spectrum table by coordinates
  @param    pindex      Pointer to star index structure
  @param    RA          Right Ascension in degrees
  @param    DEC         Declination in degrees
  @param    RA_EPS      Tolerance for RA matching in degrees
  @param    DEC_EPS     Tolerance for DEC matching in degrees
  @param    pstar_name  Pointer to store the star name (can be NULL)
  @return   Pointer to table containing the star spectrum, or NULL if not found
  
  Searches the index for a star matching the given coordinates within the
  specified tolerances. If found, returns a duplicate of the cached table
  or loads it from disk. The caller is responsible for freeing the returned table.
  
  @note There is a potential memory leak if this function is called in a loop
        without freeing the returned table. The returned table must be freed
        using cpl_table_delete().
 */
/*----------------------------------------------------------------------------*/


cpl_table* star_index_get(star_index* pindex, double RA, double DEC,
		double RA_EPS, double DEC_EPS, const char** pstar_name)
{

	cpl_ensure(pindex, CPL_ERROR_NULL_INPUT, NULL);
	cpl_ensure(pstar_name, CPL_ERROR_NULL_INPUT, NULL);

    int i = 0;
    cpl_table* pret = 0;
    int inull = 0;

    for (i = 0; i < pindex->index_size; i++)
    {
        double curr_ra = 0;
        double curr_dec = 0;
        int ext_id = 0;

        ext_id = cpl_table_get_int(pindex->index_table, COL_NAME_EXTID, i ,&inull);
        curr_ra = cpl_table_get(pindex->index_table, COL_NAME_RA, i,&inull);
        curr_dec = cpl_table_get(pindex->index_table, COL_NAME_DEC, i,&inull);
        if ((ext_id > 0) && (fabs(curr_ra - RA) < RA_EPS) && (fabs(curr_dec - DEC) < DEC_EPS))
        {
            // found
            // retrieve the data
            if (i - pindex->index_size + pindex->cache_size  >= 0)
            {
                // data is in cache    TODO: this generates a leak as there is a loop and the created table is not deleated
                pret = cpl_table_duplicate(pindex->cache[i - pindex->index_size + pindex->cache_size ]);
            }
            else
            {
                // data is on disk
                pret = cpl_table_load(pindex->fits_file_name, ext_id, 0);
            }
            if (pret && pstar_name)
            {
                *pstar_name = cpl_table_get_string(pindex->index_table, COL_NAME_NAME, i);
            }
            break;
        }
    }
    if(cpl_error_get_code() == CPL_ERROR_NONE) {
       return pret;
    } else {
    	cpl_table_delete(pret);
    	return NULL;
    }
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Parse catalog and load reference star table
  @param    cat        Input frame catalog
  @param    dRA        Right Ascension in degrees
  @param    dDEC       Declination in degrees
  @param    EPSILON    Tolerance for coordinate matching in degrees
  @param    pptable    Pointer to store the loaded table (output parameter)
  @return   CPL_ERROR_NONE on success, otherwise an error code
  
  Loads a star index from the catalog file and searches for a star matching
  the given coordinates. If found, the table is returned via pptable.
  
  @note The function automatically manages the star index lifecycle. The returned
        table must be freed by the caller using cpl_table_delete().
 */
/*----------------------------------------------------------------------------*/

cpl_error_code
eris_parse_catalog_std_stars(cpl_frame* cat,
		double dRA,
		double dDEC,
		double EPSILON,
		cpl_table** pptable)
{

	cpl_ensure(cat, CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT);
	cpl_ensure(pptable, CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT);
	if (cat != NULL) {

		const char* name = cpl_frame_get_filename(cat);

		if (name) {
			star_index* pstarindex = star_index_load(name);
			if (pstarindex) {
				const char* star_name = 0;
				cpl_msg_info(cpl_func,
						"The catalog is loaded, looking for star in RA[%f] DEC[%f] tolerance[%f]",
						dRA, dDEC, EPSILON);
				*pptable = star_index_get(pstarindex, dRA, dDEC, EPSILON,
						EPSILON, &star_name);
				if (*pptable && star_name) {
					cpl_msg_info(cpl_func,
							"REF table is found in the catalog, star name is [%s]",
							star_name);
				}
				else {
					cpl_msg_info(cpl_func,
							"ERROR - REF table could not be found in the catalog");
				}
			}
			else {
				cpl_msg_info(cpl_func, "ERROR - could not load the catalog");
			}
			star_index_delete(pstarindex);

		}
	}


	return cpl_error_get_code();
}
/*----------------------------------------------------------------------------*/
/**
  @brief    Dump the star index table to a file
  @param    pindex   Pointer to star index structure
  @param    pfile    Output file stream (must be open for writing)
  
  Writes a text representation of the index table to the specified file stream.
  All rows are dumped from the beginning to the end of the table.
 */
/*----------------------------------------------------------------------------*/

void star_index_dump(star_index* pindex, FILE* pfile)
{
    cpl_table_dump(pindex->index_table, 0, 	cpl_table_get_nrow(pindex->index_table), pfile);
}
/**@}*/
