/* $Id: eris_ifu_strehl.c,v 1.9 2013-03-26 17:00:44 jtaylor Exp $
 *
 * This file is part of the ERIS Pipeline
 * Copyright (C) 2002,2003 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */


#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/

#include <eris_ifu_strehl.h>
#include <eris_ifu_jitter_interface.h>
#include <eris_ifu_wavecal_static.h>
#include <eris_utils.h>
#include <eris_pfits.h>
#include <eris_ifu_utils.h>
#include <string.h>
#include <cpl.h>
#include <hdrl.h>

#define M1_RADIUS 8.15
#define M2_RADIUS 0.9
#define ERIS_IFU_RAW_BPM                    "RAW_BPM"
#define ERIS_IFU_RAW_ERROR                  "RAW_ERROR"
/*----------------------------------------------------------------------------*/
/**
 * @defgroup eris_ifu_strehl   Strehl Ratio Computation and Image Quality Assessment
 *
 * This module provides functions for computing the Strehl ratio of point spread
 * functions (PSF) from IFU observations, typically using PSF calibrator stars.
 * 
 * ## What is the Strehl Ratio?
 * 
 * The Strehl ratio is a measure of optical system quality, defined as the ratio
 * of the peak intensity of the actual PSF to the peak intensity of a theoretical
 * diffraction-limited PSF. It ranges from 0 (poor) to 1 (perfect diffraction limit).
 * 
 * For adaptive optics (AO) systems like ERIS, the Strehl ratio is a key
 * performance metric indicating:
 * - AO correction quality
 * - Atmospheric turbulence effects
 * - Overall image sharpness
 * 
 * ## Computation Method:
 * 
 * The Strehl ratio computation uses the HDRL library and follows these steps:
 * 
 * 1. **Load collapsed cube image** (wavelength-averaged) from:
 *    - PSF_CALIBRATOR observations (preferred)
 *    - STD star observations (alternative)
 *    - STD_FLUX observations (alternative)
 * 
 * 2. **Determine measurement parameters** from FWHM QC parameters:
 *    - flux_radius: aperture for stellar flux integration (default: 3σ of PSF)
 *    - bkg_radius_low: inner radius for background annulus
 *    - bkg_radius_high: outer radius for background annulus
 * 
 * 3. **Locate PSF center** within image (avoiding edges)
 * 
 * 4. **Compute Strehl ratio** using HDRL:
 *    - Integrates flux within aperture
 *    - Estimates background from annulus
 *    - Compares to theoretical Airy disk for given telescope parameters
 * 
 * 5. **Save QC parameters**:
 *    - ESO QC STREHL: Strehl ratio value
 *    - ESO QC STREHL ERROR: Uncertainty estimate
 * 
 * ## Telescope Parameters:
 * 
 * - Primary mirror radius (M1): 8.15 m (VLT)
 * - Secondary mirror radius (M2): 0.9 m
 * - Wavelength: Central wavelength of observation band
 * - Pixel scale: Depends on SPIFFIER pre-optics (25, 100, or 250 mas)
 * 
 * ## Adaptive Parameter Selection:
 * 
 * The module automatically adjusts measurement radii if:
 * - PSF is near image edge (reduces radii to fit)
 * - FWHM information is available (uses 3σ rule)
 * - User provides -1 for parameters (triggers automatic mode)
 * 
 * @note This module depends on the HDRL (High-level Data Reduction Library)
 * @note Computation requires collapsed cube images (mean or median along wavelength)
 * @note Error images are generated using detector model if not provided
 */
/*----------------------------------------------------------------------------*/

/**@{*/

#define HDRL_PARAMETER_HEAD void * base
/** @cond PRIVATE */
typedef struct {
    HDRL_PARAMETER_HEAD;
    cpl_size    llx ;
    cpl_size    lly ;
    cpl_size    urx ;
    cpl_size    ury ;
} hdrl_rect_region_parameter;

/*----------------------------------------------------------------------------*/
/**
  @brief    Save propertylist to FITS file with DFS compliance
  @param    procatg     Product category string (e.g., "ERIS_IFU_STREHL")
  @param    filename    Output filename
  @param    header      FITS header propertylist to save
  @param    parlist     Recipe parameter list
  @param    frameset    Input frameset for DFS
  @return   CPL_ERROR_NONE on success, error code otherwise
  
  Saves a propertylist as a DFS-compliant FITS file with proper product
  category and pipeline provenance information.
  
  @note     This is a helper function for saving Strehl QC parameters
  @note     Uses cpl_dfs_save_propertylist for DFS compliance
 */
/*----------------------------------------------------------------------------*/
static cpl_error_code eris_ifu_propertylist_save(
        const char              *   procatg,
        const char              *   filename,
        const cpl_propertylist  *   header,
        const cpl_parameterlist *   parlist,
        cpl_frameset            *   frameset)
{
    /* Add a QC parameter  */
    cpl_propertylist * applist = cpl_propertylist_new();

    /* Add the product category and save image */
    cpl_propertylist_update_string(applist, CPL_DFS_PRO_CATG, procatg);

    cpl_dfs_save_propertylist(frameset, NULL, parlist, frameset, NULL,
            "ERIS_IFU_STREHL", header, NULL, PACKAGE "/" PACKAGE_VERSION,
            filename);
    cpl_propertylist_delete(applist);
    eris_check_error_code("eris_ifu_propertylist_save");
    return cpl_error_get_code();
}



/*----------------------------------------------------------------------------*/
/**
  @brief    Fix negative or zero coordinates in rect region by wrapping to image size
  @param    rect_region     Rectangle region parameter to fix (modified in-place)
  @param    nx              Image size in X (pixels), added to values < 1
  @param    ny              Image size in Y (pixels), added to values < 1
  @return   CPL_ERROR_NONE on success, error code otherwise
  
  Allows reverse indexing similar to Python: 0 becomes nx, -2 becomes nx-2, etc.
  Wrapping is based on FITS convention: pixel 1 is first, pixel nx is last (inclusive).
  
  ## Examples:
  - llx = -5, nx = 100  →  llx = 95  (5 pixels from right edge)
  - urx = 0, nx = 100   →  urx = 100 (last pixel)
  - lly = 1, ny = 100   →  lly = 1   (unchanged, already valid)
  
  @note     Only values < 1 are adjusted
  @note     After adjustment, verifies region is valid within image bounds
  
  Possible errors:
  - CPL_ERROR_NULL_INPUT if rect_region is NULL
  - CPL_ERROR_ILLEGAL_INPUT if parameter is not a rect region type
  - Errors from hdrl_rect_region_parameter_verify if region is invalid
 */
/*----------------------------------------------------------------------------*/
static cpl_error_code hdrl_rect_region_fix_negatives(
        hdrl_parameter    *     rect_region,
        const cpl_size          nx,
        const cpl_size          ny)
{
    hdrl_rect_region_parameter * rr_loc =
        (hdrl_rect_region_parameter *)rect_region ;

    cpl_error_ensure(rect_region != 0, CPL_ERROR_NULL_INPUT,
            return CPL_ERROR_NULL_INPUT, "region input must not be NULL");
    cpl_error_ensure(hdrl_rect_region_parameter_check(rect_region),
            CPL_ERROR_ILLEGAL_INPUT, return CPL_ERROR_ILLEGAL_INPUT,
            "Expected Rect Region parameter") ;

    if (nx > 0 && rr_loc->llx < 1) rr_loc->llx += nx;
    if (ny > 0 && rr_loc->lly < 1) rr_loc->lly += ny;
    if (nx > 0 && rr_loc->urx < 1) rr_loc->urx += nx;
    if (ny > 0 && rr_loc->ury < 1) rr_loc->ury += ny;

    return hdrl_rect_region_parameter_verify(rect_region, nx, ny);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Compute photon shot noise error image using detector model
  @param    ima_data    Input image data in ADU (must be bias/overscan corrected)
  @param    gain        Detector gain in e⁻/ADU
  @param    ron         Read-out noise in ADU
  @param    ima_errs    Output: error image in ADU (allocated by function)
  @return   CPL_ERROR_NONE on success, error code otherwise
  
  Computes error image using standard photon statistics formula:
  
  \f$ \sigma_{ADU} = \sqrt{ \frac{counts}{gain} + RON^2 } \f$
  
  The first term is the photon shot noise (Poisson statistics converted to ADU),
  and the second term is the read-out noise.
  
  ## Algorithm:
  1. Duplicate input image
  2. Set negative values to RON (no measurable electrons → only RON contributes)
  3. Divide by gain to get electron counts
  4. Add RON² in quadrature
  5. Take square root
  
  @note     Output image must be deallocated by caller
  @note     Input image must be bias and overscan corrected (no offsets)
  @note     Negative values are set to RON error
  @note     This is a simplified detector model suitable for quick estimates
  
  Possible errors:
  - CPL_ERROR_NULL_INPUT if ima_data or ima_errs is NULL
  - CPL_ERROR_ILLEGAL_INPUT if gain or ron <= 0
 */
/*----------------------------------------------------------------------------*/
static cpl_error_code
eris_ifu_detector_shotnoise_model(const cpl_image* ima_data, const double gain,
                              const double ron, cpl_image ** ima_errs)
{
    cpl_ensure_code(ima_data, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(ima_errs, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(gain > 0., CPL_ERROR_ILLEGAL_INPUT);
    cpl_ensure_code(ron > 0., CPL_ERROR_ILLEGAL_INPUT);

    *ima_errs = cpl_image_duplicate(ima_data);
    /* set negative values (= zero measurable electrons) to read out noise */
    cpl_image_threshold(*ima_errs, 0., INFINITY, ron, ron);

    /* err_ADU = sqrt(counts/gain + ron * ron)*/

    cpl_image_divide_scalar(*ima_errs, gain);
    cpl_image_add_scalar(*ima_errs, ron * ron);
    cpl_image_power(*ima_errs, 0.5);
    eris_check_error_code("eris_ifu_detector_shotnoise_model");
    return cpl_error_get_code();
}



/* ---------------------------------------------------------------------------*/
/**
 * @brief    Load frame header and fix negative region parameters by image size
 *
 * @param    frm         Frame to load header from
 * @param    extnum      Extension number to load header from
 * @param    par         Region parameter to be adjusted by image dimensions
 * @return   CPL_ERROR_NONE on success, error code otherwise
 * 
 * Reads NAXIS1 and NAXIS2 from FITS header and uses them to wrap negative
 * region coordinates using hdrl_rect_region_fix_negatives().
 * 
 * @note     This enables Python-style negative indexing from FITS headers
 * @see      hdrl_rect_region_fix_negatives
 */
/* ---------------------------------------------------------------------------*/
static cpl_error_code
eris_ifu_fix_neg_region(const cpl_frame * frm, cpl_size extnum,
                        hdrl_parameter * par)
{
    cpl_propertylist * plist =
        cpl_propertylist_load(cpl_frame_get_filename(frm), extnum);
    if (!plist) {
        return cpl_error_get_code();
    }
    cpl_size nx = eris_pfits_get_naxis1(plist);
    cpl_size ny = eris_pfits_get_naxis2(plist);
    hdrl_rect_region_fix_negatives(par, nx, ny);
    cpl_propertylist_delete(plist);
    eris_check_error_code("eris_ifu_fix_neg_region");
    return cpl_error_get_code();
}

/* ---------------------------------------------------------------------------*/
/**
 * @brief    Load image from MEF file, optionally loading only a subregion
 *
 * @param    frm             Frame to load from
 * @param    eidx            Extension number to load
 * @param    region_params   Region parameters (if NULL, load full image)
 * @return   Loaded image, or NULL on error
 * 
 * Loads either the full extension or a windowed region based on region_params.
 * If region_params is provided, negative coordinates are fixed relative to
 * image dimensions before loading.
 * 
 * @note     Returned image must be deallocated by caller
 * @note     Always loads as CPL_TYPE_DOUBLE
 */
/* ---------------------------------------------------------------------------*/
static cpl_image * eris_ifu_image_mef_load_region(
        cpl_frame       *   frm,
        cpl_size            eidx,
        hdrl_parameter  *   region_params)
{
    cpl_image   *   out = NULL ;

    if (region_params == NULL) {
        out = cpl_image_load(cpl_frame_get_filename(frm),
                CPL_TYPE_DOUBLE, 0, eidx);
    } else {
        eris_ifu_fix_neg_region(frm, eidx, region_params);
        out = cpl_image_load_window(cpl_frame_get_filename(frm),
                CPL_TYPE_DOUBLE, 0, eidx,
                hdrl_rect_region_get_llx(region_params),
                hdrl_rect_region_get_lly(region_params),
                hdrl_rect_region_get_urx(region_params),
                hdrl_rect_region_get_ury(region_params)) ;
    }
    eris_check_error_code("eris_ifu_image_mef_load_region");
    return out ;
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Load hdrl_image from frames with optional error and BPM
  @param    in              Input frame containing data image
  @param    ext_num         Extension number for data
  @param    in_err          Input frame containing error image (optional, can be NULL)
  @param    ext_num_err     Extension number for error
  @param    in_bpm          Input frame containing bad pixel mask (optional, can be NULL)
  @param    ext_num_bpm     Extension number for BPM
  @param    region_params   Region to load (if NULL, load full image)
  @param    ron             Read-out noise in ADU (for error model if in_err is NULL)
  @param    gain            Detector gain in e⁻/ADU (for error model if in_err is NULL)
  @param    out_hdrl_ima    Output: hdrl_image object (allocated by function)
  @return   CPL_ERROR_NONE on success, error code otherwise
  
  Constructs an hdrl_image from separate data, error, and BPM extensions.
  
  ## Error Handling Options:
  1. **in_err provided**: Loads error from specified frame/extension
  2. **in_err NULL, ron and gain provided**: Generates error using detector model
  3. **in_err NULL, ron and gain negative**: Creates hdrl_image with no error
  
  ## BPM Handling:
  - If in_bpm provided: Loads BPM, thresholds to create mask, applies to data image
  - BPM value 0 = good pixel, non-zero = bad pixel
  - Bad pixels are rejected in the output hdrl_image
  
  @note     Output hdrl_image must be deallocated with hdrl_image_delete()
  @note     Error model assumes bias/overscan corrected data
  @note     BPM is inverted (0→good becomes 1→good for CPL mask)
  
  Possible errors:
  - CPL_ERROR_NULL_INPUT if in or out_hdrl_ima is NULL
  - CPL_ERROR_ILLEGAL_INPUT if image cannot be loaded
 */
/*----------------------------------------------------------------------------*/
static cpl_error_code eris_ifu_hdrl_image_load(
        cpl_frame       *   in,
        cpl_size            ext_num,
        cpl_frame       *   in_err,
        cpl_size            ext_num_err,
        cpl_frame       *   in_bpm,
        cpl_size            ext_num_bpm,
        hdrl_parameter  *   region_params,
        double              ron,
        double              gain,
        hdrl_image      **  out_hdrl_ima)
{
    cpl_image       *   img ;

    /* Check Entries */
    cpl_ensure_code(in, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(out_hdrl_ima, CPL_ERROR_NULL_INPUT);

    /* Load in image */
    img = eris_ifu_image_mef_load_region(in, ext_num, region_params);
    if (img == NULL) return CPL_ERROR_ILLEGAL_INPUT ;

    /* Load BPM */
    if (in_bpm != NULL) {
        cpl_image * bpm = eris_ifu_image_mef_load_region(in_bpm, ext_num_bpm, region_params);
        cpl_mask * bpm_mask = cpl_mask_threshold_image_create(bpm,
                -0.5, 0.5) ;
        cpl_image_delete(bpm) ;
        cpl_mask_not(bpm_mask) ;
        cpl_image_reject_from_mask(img, bpm_mask);
        cpl_mask_delete(bpm_mask) ;
    }

    /* Load error */
    cpl_image * err =NULL;
    if (in_err != NULL) {
        err = eris_ifu_image_mef_load_region(in_err, ext_num_err, region_params);
    } else if (ron < 0. && gain < 0.) {
        /* no error */
        err = NULL;
    } else {
        /* No passed error -> Create uniform error */
        eris_ifu_detector_shotnoise_model(img, gain, ron, &err);
    }

    /* Create out himage */
    *out_hdrl_ima = hdrl_image_create(img, err);

    /* Cleanup */
    cpl_image_delete(img);
    cpl_image_delete(err);
    eris_check_error_code("eris_ifu_hdrl_image_load");
    return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Compute Strehl ratio from raw/calibrated image (basic version)
  @param    frameset        Input frameset containing image
  @param    parlist         Recipe parameters
  @param    recipe_name     Name of calling recipe
  @return   CPL_ERROR_NONE on success, error code otherwise
  
  Computes Strehl ratio using basic approach with parameters from parlist.
  This function searches for JITTER_CUBE products and uses extension 0/1/2
  for data/error/BPM.
  
  ## Processing Steps:
  1. Parse Strehl parameters from parlist (wavelength, M1/M2 radii, pixel scale, radii)
  2. Load JITTER_CUBE frame
  3. Read detector parameters (RON, gain) from header
  4. Create full-image region
  5. Load hdrl_image with error and BPM
  6. Mask NaN values
  7. Compute Strehl using hdrl_strehl_compute()
  8. Save QC parameters to FITS file
  
  @note     This is a simpler version compared to eris_ifu_stdstar_strehl_compute()
  @note     Expects Strehl parameters to be fully specified in parlist
  @note     TODO: Currently not used - eris_ifu_stdstar_strehl_compute() is preferred
  
  Possible errors:
  - CPL_ERROR_NULL_INPUT if inputs are NULL
  - CPL_ERROR_DATA_NOT_FOUND if JITTER_CUBE not found
  - CPL_ERROR_UNSPECIFIED if parameter parsing fails
  - CPL_ERROR_FILE_NOT_FOUND if image cannot be loaded
 */
/*----------------------------------------------------------------------------*/
/*TODO: NOT USED
static cpl_error_code
eris_ifu_strehl_calculate(hdrl_image* hima,
		const double wavelength,
		const double m1_radius,
		const double m2_radius,
		const double pixel_scale_x,
		const double pixel_scale_y,
		const double flux_radius,
		const double bkg_radius_low,
		const double bkg_radius_high)
{


	hdrl_parameter * strehl_p = hdrl_strehl_parameter_create(wavelength, m1_radius,
			m2_radius, pixel_scale_x, pixel_scale_y, flux_radius, bkg_radius_low,
			bkg_radius_high);
	return cpl_error_get_code();
}
*/
/*----------------------------------------------------------------------------*/
/**
  @brief    Compute Strehl ratio from cube (basic interface, currently unused)
  @param    frameset        Input frameset
  @param    parlist         Recipe parameters
  @param    recipe_name     Recipe name
  @return   CPL_ERROR_NONE on success, error code otherwise
  
  Basic Strehl computation interface. Searches for JITTER_CUBE and computes
  Strehl ratio using parameters from parlist.
  
  @note     Currently NOT USED - eris_ifu_stdstar_strehl_compute() is the
            preferred function for standard star processing
  @note     Uses fixed extension numbers: 0=data, 1=error, 2=BPM
  @note     Requires all Strehl parameters in parlist (no automatic computation)
  
  @see      eris_ifu_stdstar_strehl_compute
 */
/*----------------------------------------------------------------------------*/

cpl_error_code
eris_ifu_strehl_compute(cpl_frameset * frameset, const cpl_parameterlist * parlist,
		const char* recipe_name)
{

    int                         extnum_raw = 0;
    int                         extnum_err = 1;
    int                         extnum_bpm = 2;
    hdrl_strehl_result          res;

    cpl_frame               *   in_frm ;
    cpl_frame               *   err_frm ;
    cpl_frame               *   bpm_frm ;
    hdrl_image              *   hima ;
    hdrl_parameter          *   region_params = NULL;

    cpl_ensure_code(frameset, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(parlist, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(recipe_name, CPL_ERROR_NULL_INPUT);
    /* Check initial Entries */
    if (eris_ifu_dfs_set_groups(frameset) != CPL_ERROR_NONE) {
    	return cpl_error_get_code();
    }

    /* Get parameters*/
    /* those params are not needed
     *
     *  char* param_name = NULL;
     *  const cpl_parameter     *   par = NULL;
    param_name = cpl_sprintf("%s.ext-nb-raw", recipe_name);
    par = cpl_parameterlist_find_const(parlist, param_name);
    extnum_raw = cpl_parameter_get_int(par);
    cpl_free(param_name);

    param_name = cpl_sprintf("%s.ext-nb-raw-err", recipe_name);
    par = cpl_parameterlist_find_const(parlist, param_name);
    extnum_err = cpl_parameter_get_int(par);
    cpl_free(param_name);

    param_name = cpl_sprintf("%s.ext-nb-raw-bpm", recipe_name);
    par = cpl_parameterlist_find_const(parlist, param_name);
    extnum_bpm = cpl_parameter_get_int(par);
    cpl_free(param_name);
    */

    /* Parse the Strehl Parameters */
    hdrl_parameter * p = hdrl_strehl_parameter_parse_parlist(parlist,
                                                             recipe_name);
    if (p == NULL) {
        return cpl_error_set_message(cpl_func, CPL_ERROR_UNSPECIFIED,
                "Parsing of the strehl parameters failed");
    }

    /* Parse the Region Parameters
    region_params=hdrl_rect_region_parameter_parse_parlist(parlist,
            recipe_name, "region-") ;
    if (region_params == NULL) {
        hdrl_parameter_delete(p);
        return cpl_error_set_message(cpl_func, CPL_ERROR_UNSPECIFIED,
                "Parsing of the region parameters failed");
    }
    */

    /* Load INPUT Data */
    /* Get the first Required frame */
    if ((in_frm = cpl_frameset_find(frameset, ERIS_IFU_PRO_JITTER_CUBE))==NULL) {
        //if (region_params) hdrl_parameter_delete(region_params) ;
        if (p) hdrl_parameter_delete(p) ;
        return cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                "Missing RAW file");
    }
    //bpm_frm = cpl_frameset_find(frameset, ERIS_IFU_RAW_BPM) ;
    //err_frm = cpl_frameset_find(frameset, ERIS_IFU_RAW_ERROR) ;
    int sx = 0;
    int sy = 0;
    const char* fname = cpl_frame_get_filename(in_frm);
    cpl_image* in_ima = cpl_image_load(fname,CPL_TYPE_FLOAT, 0, 0);
    sx = cpl_image_get_size_x(in_ima);
    sy = cpl_image_get_size_y(in_ima);
    bpm_frm = in_frm;
    err_frm = in_frm;
    cpl_propertylist* plist = cpl_propertylist_load(fname,0);
    double ron = cpl_propertylist_get_double(plist,"ESO DET CHIP RON");
    double gain = cpl_propertylist_get_double(plist,"ESO DET CHIP GAIN");
    cpl_propertylist_delete(plist);
    cpl_image_delete(in_ima);
    region_params = hdrl_rect_region_parameter_create(1, 1, sx, sy);
    /* Load the image */
    if (eris_ifu_hdrl_image_load(in_frm, extnum_raw, err_frm, extnum_err,
                bpm_frm, extnum_bpm, region_params, ron, gain,
                &hima) != CPL_ERROR_NONE) {
        if (region_params) hdrl_parameter_delete(region_params) ;
        if (p) hdrl_parameter_delete(p) ;
        return cpl_error_set_message(cpl_func, CPL_ERROR_FILE_NOT_FOUND,
                "Cannot load RAW image");
    }
    if (region_params) hdrl_parameter_delete(region_params) ;


    /*  This is not needed as ERRORS are provided by the input data
    if(err_frm == NULL) {
        // Replace the error image created by the ron/gain by the mad-scaled
        // rms of the image
       double dmad = 0.;
       cpl_image * img = cpl_image_load(cpl_frame_get_filename(in_frm),
    		   	   	   	   	   	   	    CPL_TYPE_DOUBLE, 0, extnum_raw);
       cpl_image_get_mad(img, &dmad);
       cpl_image_delete(img);

       cpl_image_multiply_scalar(hdrl_image_get_error(hima), 0.);
       cpl_image_add_scalar(hdrl_image_get_error(hima),
                            (dmad * CPL_MATH_STD_MAD));
    }
    */
    cpl_msg_info(cpl_func,"Handles NANs");
    eris_ifu_mask_nans_in_hdrlimage(&hima);
    /* Strehl COMPUTATION */
    res = hdrl_strehl_compute(hima, p);
    hdrl_parameter_delete(p);
    cpl_msg_info(cpl_func,"strehl=%g+-%g", res.strehl_value.data,
                 res.strehl_value.error);
    /* expected difference sqrt(pi / 2)  due to median */
    cpl_msg_info(cpl_func, "star peak at %g/%g: %g +- %g", res.star_x,
                 res.star_y, res.star_peak.data, res.star_peak.error);
    cpl_msg_info(cpl_func, "star flux %g+-%g", res.star_flux.data,
                 res.star_flux.error);
    cpl_msg_info(cpl_func,"median estimated background=%g+-%g "
                 "(computed error %g)",
                 res.star_background.data, res.star_background.error,
                 res.computed_background_error);
    cpl_propertylist* header=cpl_propertylist_new();
    cpl_propertylist_update_double(header, "ESO QC STREHL",
                                   res.strehl_value.data);
    cpl_propertylist_update_double(header, "ESO QC STREHL ERROR",
                                   res.strehl_value.error);
    cpl_propertylist_update_string(header, CPL_DFS_PRO_CATG, "ERIS_IFU_STREHL");

    eris_ifu_propertylist_save("ERIS_IFU_STREHL","eris_ifu_strehl.fits", header,
                         parlist,frameset);


    /* Cleanup */
    cpl_propertylist_delete(header);
    hdrl_image_delete(hima);
    /* In case of Poisson error model: */
    eris_check_error_code("eris_ifu_strehl_compute");
    return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Compute Strehl ratio from standard star collapsed cube with adaptive parameters
  @param    frameset        Input frameset containing standard star cubes
  @param    parlist         Recipe parameters
  @param    context         Recipe context for parameter lookup (e.g., "eris")
  @return   CPL_ERROR_NONE on success, error code otherwise
  
  Main function for computing Strehl ratio from PSF/STD calibrator observations.
  Uses wavelength-collapsed cube (mean along spectral axis) and automatically
  determines optimal measurement parameters from FWHM QC keywords.
  
  ## Input Data Search Priority:
  1. PSF_CUBE_COADD_MEDIAN (for FWHM determination)
  2. STD_CUBE_COADD_MEDIAN (alternative)
  3. STD_FLUX_CUBE_COADD_MEDIAN (alternative)
  Then uses corresponding *_MEAN cube for actual Strehl computation.
  
  ## Automatic Parameter Determination:
  
  ### Pixel Scale:
  - Determined from pre-optics scale setting:
    * S250MAS → 0.250 arcsec/pixel
    * S100MAS → 0.100 arcsec/pixel
    * S25MAS  → 0.025 arcsec/pixel
    * PUPIL   → 0.100 arcsec/pixel (TBD)
  
  ### Wavelength:
  - Central wavelength of observation band (converted to meters)
  
  ### Flux Radius (aperture):
  - **User = -1 (automatic)**: 3σ of PSF = 3 × FWHM / 2.355
  - **User provides value**: Used as-is (in pixels), converted to arcsec
  
  ### Background Radii:
  - **User = -1 (automatic)**:
    * bkg_radius_low  = 1.5 × flux_radius
    * bkg_radius_high = 2.0 × flux_radius
  - **User provides values**: Converted from pixels to arcsec
  
  ## Adaptive Boundary Handling:
  
  If PSF center is near image edge (lines 747-774):
  1. Locate PSF maximum (avoiding 4-pixel margin)
  2. Check if flux_radius fits within image
  3. If not: reduce flux_radius to largest possible value
  4. Adjust background radii accordingly
  5. Skip if bkg_radius_low becomes ≤ 0
  
  ## Extension Structure:
  - Extension 1: Data (collapsed mean image)
  - Extension 2: Error (propagated uncertainty)
  - Extension 3: Quality/BPM (bad pixel mask)
  
  ## Output:
  - Updates input cube FITS file with QC parameters:
    * ESO QC STREHL: Strehl ratio value
    * ESO QC STREHL ERROR: Uncertainty estimate
  - Also logs:
    * Star peak position and intensity
    * Star flux
    * Background estimate
  
  ## Error Handling:
  - If Strehl computation fails: temporarily suppresses error and returns success
  - If Strehl value is NaN: sets QC values to 999.0 as error indicator
  - Validates parameters before computation
  
  @note     Uses HDRL library for actual Strehl computation
  @note     Requires collapsed cube products (generated by cube collapse step)
  @note     FWHM keywords must be present in MEDIAN cube for automatic mode
  @note     Modifies input FITS file in-place with QC parameters
  @note     VLT telescope parameters: M1=8.15m, M2=0.9m
  
  @see      hdrl_strehl_compute
  @see      eris_ifu_cube_collapse_mean_and_save
  
  Possible errors:
  - CPL_ERROR_NULL_INPUT if inputs are NULL
  - CPL_ERROR_DATA_NOT_FOUND if no suitable cube products found
  - CPL_ERROR_FILE_NOT_FOUND if cube cannot be loaded
  - Warnings if automatic parameter adjustment needed
 */
/*----------------------------------------------------------------------------*/

cpl_error_code
eris_ifu_stdstar_strehl_compute(cpl_frameset * frameset,
                                const cpl_parameterlist * parlist,
                                const char* context/*,
                                const char* recipe_name*/)
{
    //const cpl_parameter     *   par = NULL;
    int                         extnum_raw = 1;
    int                         extnum_err = 2;
    int                         extnum_bpm = 3;
    hdrl_strehl_result          res;

    cpl_frame               *   in_frm = NULL ;
    cpl_frame               *   err_frm ;
    cpl_frame               *   bpm_frm ;
    hdrl_image              *   hima ;
    hdrl_parameter          *   region_params = NULL;
    char* param_name = NULL;

    double wavelength = 0;
    //ifsPreopticsScale = eris_ifu_get_spiffier_preoptics_scale(sof->band);
    double pixel_scale_x = 0;
    double pixel_scale_y = 0;
    double flux_radius_pix = 1;
    double bkg_radius_low = 1;
    double bkg_radius_high = 1.5;

    cpl_ensure_code(frameset,CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(parlist,CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(context,CPL_ERROR_NULL_INPUT);

    param_name = cpl_sprintf("%s.strehl_flux_radius", context);
    flux_radius_pix = cpl_parameter_get_double(
    		cpl_parameterlist_find_const(parlist, param_name));
    cpl_free(param_name);


    param_name = cpl_sprintf("%s.strehl_bkg-radius-low", context);
    bkg_radius_low = cpl_parameter_get_double(
    		cpl_parameterlist_find_const(parlist, param_name));
    cpl_free(param_name);


    param_name = cpl_sprintf("%s.strehl_bkg-radius-high", context);
    bkg_radius_high = cpl_parameter_get_double(
    		cpl_parameterlist_find_const(parlist, param_name));
    cpl_free(param_name);


    ifsBand             band = UNDEFINED_BAND;
    ifsPreopticsScale   scale = UNDEFINED_SCALE;
    ifsInstrument       instrument = UNSET_INSTRUMENT;
    eris_ifu_jitter_get_instrument_settings(frameset, &instrument, &band, &scale);

    switch(scale) {
    case S250MAS:
    	pixel_scale_x = 0.250;
    	break;

    case S100MAS:
    	pixel_scale_x = 0.100;
    	break;

    case S25MAS:
    	pixel_scale_x = 0.025;
    	break;

    case UNDEFINED_SCALE:
    	pixel_scale_x = 0.025;
    	break;
    //TODO: FOLLOWING TO BE VERIFIED
    case PUPIL:
       	pixel_scale_x = 0.100;
       	break;
    }
    eris_ifu_get_central_lambda(band, &wavelength);
    wavelength *= 1.e-6; /* convert to [m] */
    pixel_scale_y = pixel_scale_x;
    cpl_msg_info(cpl_func,"pixel_scale: %g [arcsec]",pixel_scale_x);
    /* Determine flux_radius, bkg_radius_low, bkg_radius_high from QC.FWHM
     * on cube collapsed median*/
    cpl_frame* std_frm;
    cpl_frame* std_flux_frm;
    cpl_frame* psf_frm;
    //cpl_frameset_dump(frameset,stdout);
    std_frm = cpl_frameset_find(frameset, ERIS_IFU_PRO_JITTER_STD_CUBE_COADD_MEDIAN);
    std_flux_frm = cpl_frameset_find(frameset, ERIS_IFU_PRO_JITTER_STD_FLUX_CUBE_COADD_MEDIAN);
    psf_frm = cpl_frameset_find(frameset, ERIS_IFU_PRO_JITTER_PSF_CUBE_COADD_MEDIAN);
    if ( (std_frm == NULL)  && (psf_frm == NULL) && (std_flux_frm == NULL)) {
    	return cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
    			"Missing ERIS_IFU_PRO_JITTER_PSF_CUBE_COADD_MEDIAN file, Cannot compute Strehl,");
    } else {
    	if(std_frm != NULL) {
    		in_frm = std_frm;
    	} else if(psf_frm != NULL) {
    		in_frm = psf_frm;
    	} else if(std_flux_frm != NULL) {
    		in_frm = std_flux_frm;
    	}
    }

    cpl_propertylist* plist = cpl_propertylist_load(cpl_frame_get_filename(in_frm),0);
    double fwhm_maj = 9;
    double fwhm_min = 9;

    if(cpl_propertylist_has(plist, "ESO QC FWHM MAJ")) {
        fwhm_maj = cpl_propertylist_get_double(plist, "ESO QC FWHM MAJ");
        if(fwhm_maj == 0) {
        	if(cpl_propertylist_has(plist, "ESO QC FWHMX")) {
        		fwhm_maj = cpl_propertylist_get_double(plist, "ESO QC FWHMX");
        	}
        }
    }

    if(cpl_propertylist_has(plist, "ESO QC FWHM MIN")) {
    	fwhm_min = cpl_propertylist_get_double(plist, "ESO QC FWHM MIN");
    	if(fwhm_min == 0) {
    		if(cpl_propertylist_has(plist, "ESO QC FWHMY")) {
    			fwhm_min = cpl_propertylist_get_double(plist, "ESO QC FWHMY");
    		}
    	}
    }

    cpl_propertylist_delete(plist);
    //cpl_msg_info(cpl_func,"flux_radius_pix: %g",flux_radius_pix);
    //cpl_msg_info(cpl_func,"fwhm_min: %g fwhm_maj: %g",fwhm_min, fwhm_maj);
    double fwhm_sup = (fwhm_maj > fwhm_min) ? fwhm_maj : fwhm_min;
    if(flux_radius_pix == -1) {
    	cpl_msg_info(cpl_func,"pipeline setting flux_radius_pix: 3 * PSF sigmas");
        flux_radius_pix = 3. * fwhm_sup / CPL_MATH_FWHM_SIG; // 3 sigmas
    }
    cpl_msg_info(cpl_func,"flux_radius_pix: %g",flux_radius_pix);
    double flux_radius_arc = flux_radius_pix * pixel_scale_x;
    //cpl_msg_info(cpl_func,"bkg_radius_low: %g",bkg_radius_low);
    if(bkg_radius_low == -1) {
    	cpl_msg_info(cpl_func,"pipeline setting bkg_radius_low: 1.5 * flux_radius");
        bkg_radius_low = 1.5 * flux_radius_arc;
    } else {
    	bkg_radius_low *= pixel_scale_x;
    }
    //cpl_msg_info(cpl_func,"bkg_radius_high: %g",bkg_radius_high);
    if(bkg_radius_high == -1) {
    	cpl_msg_info(cpl_func,"pipeline setting bkg_radius_high: 2.0 * flux_radius");
        bkg_radius_high = 2.0 * flux_radius_arc;
    } else {
    	bkg_radius_high *= pixel_scale_x;
    }

    cpl_msg_info(cpl_func,"flux_radius: %g [arcsec] bkg_radius_low: %g [arcsec] bkg_radius_high: %g [arcsec]",
        		flux_radius_arc, bkg_radius_low,bkg_radius_high);

    cpl_msg_info(cpl_func,"flux_radius: %g [pix] bkg_radius_low: %g [pix] bkg_radius_high: %g [pix]",
    		flux_radius_arc / pixel_scale_x,bkg_radius_low / pixel_scale_x,bkg_radius_high / pixel_scale_x);


    /* Load INPUT Data for Strehl Computation: we use the cube collapsed mean */
    std_frm = cpl_frameset_find(frameset, ERIS_IFU_PRO_JITTER_STD_CUBE_MEAN);
    psf_frm = cpl_frameset_find(frameset, ERIS_IFU_PRO_JITTER_PSF_CUBE_MEAN);
    std_flux_frm = cpl_frameset_find(frameset, ERIS_IFU_PRO_JITTER_STD_FLUX_CUBE_MEAN);
    if ( (std_frm == NULL) && (std_flux_frm == NULL)  && (psf_frm == NULL) ) {

        return cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                "Missing ERIS_IFU_PRO_JITTER_PSF_CUBE_MEAN file, Cannot compute Strehl,");
    } else {
    	if(std_frm != NULL) {
    		in_frm = std_frm;
    	} else if(psf_frm != NULL) {
    		in_frm = psf_frm;
    	}
    }

    int sx = 0;
    int sy = 0;
    const char* fname = cpl_frame_get_filename(in_frm);
    cpl_msg_info(cpl_func,"strehl computed on: %s",fname);
    cpl_image* in_ima = cpl_image_load(fname,CPL_TYPE_FLOAT, 0, 1);

    sx = cpl_image_get_size_x(in_ima);
    sy = cpl_image_get_size_y(in_ima);
    bpm_frm = in_frm;
    err_frm = in_frm;

    plist = cpl_propertylist_load(fname,0);
    double ron = cpl_propertylist_get_double(plist,"ESO DET CHIP RON");
    double gain = cpl_propertylist_get_double(plist,"ESO DET CHIP GAIN");

    //cpl_propertylist_delete(plist);
    cpl_image_delete(in_ima);
    region_params = hdrl_rect_region_parameter_create(1, 1, sx, sy);

    /* Load the image */
    if (eris_ifu_hdrl_image_load(in_frm, extnum_raw, err_frm, extnum_err,
                bpm_frm, extnum_bpm, region_params, ron, gain,
                &hima) != CPL_ERROR_NONE) {
        if (region_params) hdrl_parameter_delete(region_params) ;

        return cpl_error_set_message(cpl_func, CPL_ERROR_FILE_NOT_FOUND,
                "Cannot load RAW image");
    }
    if (region_params) hdrl_parameter_delete(region_params) ;
    

    //cpl_msg_info(cpl_func,"Handles NANs");
    eris_ifu_mask_nans_in_hdrlimage(&hima);

    cpl_size cx = 0;
    cpl_size cy = 0;
    cpl_size margin_x = 4;
    cpl_size margin_y = 4;
    cpl_size llx = margin_x;
    cpl_size lly = margin_y;
    cpl_size urx = sx - margin_x;
    cpl_size ury = sy - margin_y;

    cpl_image_get_maxpos_window(hdrl_image_get_image(hima), llx, lly, urx, ury, &cx, &cy);
    cpl_msg_info(cpl_func,"Image max detected at cx: %lld, cy: %lld",cx, cy);
    /*
    cpl_image_save(hdrl_image_get_image(hima),"ima.fits", CPL_TYPE_DOUBLE, NULL,
    		CPL_IO_DEFAULT);
    */
    cpl_msg_info(cpl_func,"check Strehl parameters");
    /*
    cpl_msg_info(cpl_func,"flux_radius[pix]: %g bkg_radius_low[pix]: %g bkg_radius_high[pix]: %g",
        	        		flux_radius_pix ,bkg_radius_low / pixel_scale_x,bkg_radius_high / pixel_scale_x );
    */
    res.strehl_value.data = 0;
    res.strehl_value.error = 0;
    hdrl_parameter * p;
    cpl_msg_info(cpl_func,"wavelength: %g M1: %g M2: %g",wavelength, M1_RADIUS,M2_RADIUS);
    cpl_msg_info(cpl_func,"flux_radius_pix: %g cx: %lld cy: %lld sx: %d sy:%d",flux_radius_pix, cx, cy, sx, sy);
    
    /* Adaptive parameter adjustment if PSF near edge */
    if( (flux_radius_pix + cx) > (sx -2) ||
    	(flux_radius_pix + cy) > (sy -2) ||
		(cx-flux_radius_pix) < 2 ||
		(cy-flux_radius_pix) < 2){
    	//cpl_msg_info(cpl_func,"flux_radius_pix: %g",flux_radius_pix);
    	cpl_msg_info(cpl_func,"correct Strehl parameters");
    	flux_radius_pix = ( (flux_radius_pix + cx) > (sx -2) ) ?  flux_radius_pix : sx -2;
    	flux_radius_pix = ( (flux_radius_pix + cy) > (sy -2) ) ?  flux_radius_pix : sy -2;
    	flux_radius_pix = ( (cx - flux_radius_pix) < 2 ) ?  cx -2 : flux_radius_pix;
    	flux_radius_pix = ( (cy - flux_radius_pix) < 2 ) ?  cy -2 : flux_radius_pix;
    	//cpl_msg_info(cpl_func,"flux_radius_pix %g pixel_scale_x %g", flux_radius_pix, pixel_scale_x);
    	flux_radius_arc = flux_radius_pix * pixel_scale_x;
    	bkg_radius_low = flux_radius_arc;
    	bkg_radius_high = (flux_radius_pix + 1) * pixel_scale_x;
    	cpl_msg_info(cpl_func,"flux_radius: %g bkg_radius_low: %g bkg_radius_high: %g",
    	        		flux_radius_arc ,bkg_radius_low ,bkg_radius_high );

    	if(bkg_radius_low > 0) {
    	p = hdrl_strehl_parameter_create(wavelength,
    	    		M1_RADIUS, M2_RADIUS, pixel_scale_x, pixel_scale_y,
    				flux_radius_arc, bkg_radius_low, bkg_radius_high) ;
    	  /* Strehl COMPUTATION */
    	    res = hdrl_strehl_compute(hima, p);
    	    hdrl_parameter_delete(p);
    	} else {
    		cpl_msg_warning(cpl_func,"bkg_radius_low: %g. Skip Strehl computation.", bkg_radius_low);
    	}


    } else {
    	//cpl_msg_info(cpl_func,"fwhm_sup: %g",fwhm_sup);
    	//for (flux_radius_pix=fwhm_sup; flux_radius_pix < 3 * fwhm_sup; flux_radius_pix += 0.2) {
    		flux_radius_arc = flux_radius_pix * pixel_scale_x;
    		cpl_msg_info(cpl_func,"flux_radius_pix: %g",flux_radius_pix);
    		if(bkg_radius_low > 0) {
    		p = hdrl_strehl_parameter_create(wavelength,
    				M1_RADIUS, M2_RADIUS, pixel_scale_x, pixel_scale_y,
					flux_radius_arc, bkg_radius_low, bkg_radius_high) ;

    		res = hdrl_strehl_compute(hima, p);
    		hdrl_parameter_delete(p);
    		cpl_msg_info(cpl_func,"strehl=%g+-%g", res.strehl_value.data,
    		    		    			res.strehl_value.error);
    		} else {
    			cpl_msg_warning(cpl_func,"bkg_radius_low: %g. Skip Strehl computation.", bkg_radius_low);
    		}

    	//}
    }



    if(cpl_error_get_code() != CPL_ERROR_NONE) {
       cpl_msg_error(cpl_func,"Error computing Strehl. Temporarily suppress it.");
       cpl_error_set(cpl_func,CPL_ERROR_NONE);
       hdrl_image_delete(hima);
       return CPL_ERROR_NONE;
    } else {
    	if (res.strehl_value.data > 0) {

    		cpl_msg_info(cpl_func,"strehl=%g+-%g", res.strehl_value.data,
    				res.strehl_value.error);
    		/* expected difference sqrt(pi / 2)  due to median */
    		cpl_msg_info(cpl_func, "star peak at %g/%g: %g +- %g", res.star_x,
    				res.star_y, res.star_peak.data, res.star_peak.error);
    		cpl_msg_info(cpl_func, "star flux %g+-%g", res.star_flux.data,
    				res.star_flux.error);
    		cpl_msg_info(cpl_func,"median estimated background=%g+-%g "
    				"(computed error %g)",
					res.star_background.data, res.star_background.error,
					res.computed_background_error);

    		double qc_strehl = 0;
    		double qc_strehl_err = 0;
    		if(isnan(res.strehl_value.data)){
    			qc_strehl=999.;
    			qc_strehl_err=999.;
    		} else {
    			qc_strehl=res.strehl_value.data;
    			qc_strehl_err=res.strehl_value.error;
    		}

    		cpl_image* data = cpl_image_load(fname, CPL_TYPE_DOUBLE, 0, extnum_raw);

    		cpl_image* errs = cpl_image_load(fname, CPL_TYPE_DOUBLE, 0, extnum_err);
    		cpl_propertylist* eheader = cpl_propertylist_load(fname, extnum_err);


    		cpl_image* qual = cpl_image_load(fname, CPL_TYPE_INT, 0, extnum_bpm);
    		cpl_propertylist* qheader = cpl_propertylist_load(fname, extnum_bpm);



    		cpl_propertylist_append_double(plist, "ESO QC STREHL", qc_strehl);
    		cpl_propertylist_append_double(plist, "ESO QC STREHL ERROR", qc_strehl_err);
    		cpl_image_save(data, fname, CPL_TYPE_DOUBLE, plist, CPL_IO_DEFAULT);
    		//cpl_propertylist_dump(plist,stdout);
    		/*
        const char* pcatg = cpl_propertylist_get_string(plist, FHDR_PRO_CATG);
    	eris_ifu_save_image(frameset, plist, parlist, recipe_name, pcatg,
    			fname, CPL_TYPE_DOUBLE, data);
    		 */

    		cpl_image_save(errs, fname, CPL_TYPE_DOUBLE, eheader, CPL_IO_EXTEND);
    		cpl_image_save(qual, fname, CPL_TYPE_INT, qheader, CPL_IO_EXTEND);
    		cpl_image_delete(data);
    		cpl_image_delete(errs);
    		cpl_image_delete(qual);

    		cpl_propertylist_delete(eheader);
    		cpl_propertylist_delete(qheader);
    		//cpl_msg_info(cpl_func,"ok5");
    	}
    }
    cpl_propertylist_delete(plist);
    /* Cleanup */

    hdrl_image_delete(hima);
    eris_check_error_code("eris_ifu_stdstar_strehl_compute");

    return cpl_error_get_code();
}

/** @endcond */

/**@}*/
