/* $Id$
 *
 * This file is part of the ERIS/NIX Pipeline
 * Copyright (C) 2017 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

 /*
 * $Author$
 * $Date$
 * $Rev$
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/

#include "eris_nix_utils.h"
#include "eris_utils.h"
#include "eris_nix_casu_utils.h"
#include "eris_nix_dfs.h"
#include "eris_nix_match.h"

#include <casu_mods.h>
#include <casu_stats.h>
#include <hdrl.h>
#include <libgen.h>
#include <math.h>
#include <string.h>

/*----------------------------------------------------------------------------*/
/**
 * @defgroup eris_nix_utils     Miscellaneous Utilities
 *
 * @par Synopsis:
 * @code
 *   #include <eris_nix_utils.h>
 * @endcode
 *
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Do basic calibration of located_image (single or cube)
  @param    limage         The image to be calibrated
  @param    read_offsets   If true then remove read offsets and set blank areas
  @param    refine_wcs     refined wcs table or NULL
  @param    mdark    the master dark or NULL
  @param    gain_lin gain/linearity data or NULL
  @param    flatfield_1    flatfield data, or NULL
  @param    flatfield_2    flatfield data, or NULL
  @param    mbad_pix_map     master bad pixel mask or NULL
  @param    flag_mask      Result bpm set to mbad_pix_map & flag_mask
  @param    fill_rejected  How treat reject pixels: 'set_value', 'set_NaN' or 'noop'
  @param    fill_value     value to set reject pixels if 'set_value'
  @param    x_probe        x-coord of pixel to print debug info 
  @param    y_probe        y-coord of pixel to print debug info 
  @return  CPL_ERROR_NONE if all goes well, otherwise a CPL error code

  The function performs basic calibration on the given located_image.
  If the pointer to this is NULL the routine will return immediately
  with no error.

  'Basic' calibration comprises:
    - remove read offsets, reject and zero confidence of blank pixels
    - correct the wcs CD matrix, if refine_wcs not NULL
    - subtract MASTER_DARK, if mdark not NULL
    - linearize, if gain_lin not NULL
    - populate error plane, if gain_lin not NULL
    - divide by flatfield_1 (MASTER_FLAT_HIFREQ), if not NULL
    - divide by flatfield_2 (e.g. MASTER_FLAT_LOFREQ), if not NULL
    - associate with mask from mbad_pix_map & flag_mask, if
      mbad_pix_map not NULL. flag_mask should have bits set to 1 to
      specify flag types that count
 
 */
/*----------------------------------------------------------------------------*/

cpl_error_code enu_basic_calibrate(located_image        * limage,
                                   const int              read_offsets,
                                   const cpl_table      * refine_wcs,
                                   const master_dark    * mdark,
                                   const gain_linearity * gain_lin,
                                   const master_flat    * flatfield_1,
                                   const master_flat    * flatfield_2,
                                   const master_bpm     * mbad_pix_map,
                                   const int              flag_mask,
                                   const char           * fill_rejected,
                                   const double           fill_value,
                                   const cpl_size         x_probe,
                                   const cpl_size         y_probe) {

    if (cpl_error_get_code() != CPL_ERROR_NONE) return cpl_error_get_code();

    if (limage == NULL) return CPL_ERROR_NONE;

    /* correct the CD matrix if required, same for 'single' or 'cube' data */

    enu_modify_CD_matrix(limage, refine_wcs);

    /* for single images, reduce the image
       for cube data, reduce each image plane in turn */

    if (limage->himage) {
        enu_basic_calibrate_himage(limage->himage,
                                   limage->confidence,
                                   limage->plist,
                                   limage->frame,
                                   read_offsets,
                                   mdark,
                                   gain_lin,
                                   flatfield_1,
                                   flatfield_2,
                                   mbad_pix_map,
                                   CPL_TRUE,
                                   flag_mask,
                                   fill_rejected,
                                   fill_value,
                                   x_probe,
                                   y_probe);
        enu_check_error_code("failure in basic calibration");

    } else if (limage->himagelist) {
        const cpl_size nplanes = hdrl_imagelist_get_size(limage->himagelist);
        cpl_error_code didfail = CPL_ERROR_NONE;

#ifdef _OPENMP
        cpl_errorstate cleanstate = cpl_errorstate_get();

#pragma omp parallel for
#endif
        for (cpl_size plane = 0; plane < nplanes; plane++) {

            cpl_error_code errori = cpl_error_get_code();

            /* The total number of iterations must be pre-determined for the
               parallelism to work. In case of an error we can therefore not
               break, so instead we skip immediately to the next iteration.
               FIXME: This check on didfail does not guarantee that only one
               iteration can cause an error to be dumped, but it is not
               worse than checking on the thread-local state, errori. */
            if (didfail) continue;

            do {

            cpl_msg_info(cpl_func, "basic calibrate cube: plane %d",
                         (int)plane);

            /* the confidence array applies to all planes in the cube alike,
               so only modify it on the first call */

            if (enu_basic_calibrate_himage(hdrl_imagelist_get(limage->himagelist,
                                                              plane),
                                           limage->confidence,
                                           limage->plist,
                                           limage->frame,
                                           read_offsets,
                                           mdark,
                                           gain_lin,
                                           flatfield_1,
                                           flatfield_2,
                                           mbad_pix_map,
                                           (plane == 0),
                                           flag_mask,
                                           fill_rejected,
                                           fill_value,
                                           x_probe,
                                           y_probe)) {
                errori = cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                               "Cannot process plane %d/%d",
                                               (int)plane+1, (int)nplanes);
                break;
            }
            } while (0);

            if (errori) {
#ifdef _OPENMP
                /* Cannot access these errors after the join,
                   so dump them now. :-(((((((((((((((((((( */
                cpl_errorstate_dump(cleanstate, CPL_FALSE, NULL);
                cpl_errorstate_set(cleanstate);

#pragma omp critical(enu_basic_calibrate)
#endif
                didfail = errori;
            }
        }
        if (didfail) (void)cpl_error_set_message(cpl_func, didfail,
                                                 "Cannot process %d plane(s)",
                                                 (int)nplanes);
    }

 cleanup:
    eris_check_error_code("enu_basic_calibrate");
    return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Do basic calibration of an hdrl_image
  @param    himage         The image to be calibrated
  @param    confidence     The confidence array associated with the image or NULL
  @param    plist          The propertylist associated with the image
  @param    frame          The frame associated with the image
  @param    read_offsets   If true then remove read offsets and set blank areas
  @param    mdark    the master dark or NULL
  @param    gain_lin gain/linearity data or NULL
  @param    flatfield_1    flatfield data, or NULL
  @param    flatfield_2    flatfield data, or NULL
  @param    mbad_pix_map     master bad pixel mask or NULL
  @param    set_confidence If true then fill the confidence array 
  @param    flag_mask      Result bpm set to mbad_pix_map & flag_mask
  @param    fill_rejected  How treat reject pixels: 'set_value', 'set_NaN' or 'noop'
  @param    fill_value     value to set reject pixels if 'set_value'
  @param    x_probe        x-coord of pixel to print debug info 
  @param    y_probe        y-coord of pixel to print debug info 
  @return  CPL_ERROR_NONE if all goes well, otherwise a CPL error code

  The function performs basic calibration on the given located_image.
  If the pointer to this is NULL the routine will return immediately
  with no error.

  'Basic' calibration comprises:
    - remove read offsets, reject and zero confidence of blank pixels
    - subtract MASTER_DARK, if mdark not NULL
    - linearize, if gain_lin not NULL
    - populate error plane, if gain_lin not NULL
    - divide by flatfield_1 (MASTER_FLAT_HIFREQ), if not NULL
    - divide by flatfield_2 (e.g. MASTER_FLAT_LOFREQ), if not NULL
    - associate with mask from mbad_pix_map & flag_mask, if
      mbad_pix_map not NULL. flag_mask should have bits set to 1 to
      specify flag types that count
 */
/*----------------------------------------------------------------------------*/

cpl_error_code enu_basic_calibrate_himage(hdrl_image           * himage,
                                          cpl_image            * confidence,
                                          cpl_propertylist     * plist,
                                          const cpl_frame      * frame,
                                          const int              read_offsets,
                                          const master_dark    * mdark,
                                          const gain_linearity * gain_lin,
                                          const master_flat    * flatfield_1,
                                          const master_flat    * flatfield_2,
                                          const master_bpm     * mbad_pix_map,
                                          const int              set_confidence,
                                          const int              flag_mask,
                                          const char           * fill_rejected,
                                          const double           fill_value,
                                          const cpl_size         x_probe,
                                          const cpl_size         y_probe) {

    if (cpl_error_get_code() != CPL_ERROR_NONE) return cpl_error_get_code();

    cpl_ensure_code(himage, CPL_ERROR_NULL_INPUT);

    /* window information */

    cpl_size nx = 0;
    cpl_size ny = 0;
    int rot = 0;
    cpl_size strx = 0;
    cpl_size stry = 0;
    cpl_size nx_chip = 0;
    cpl_size ny_chip = 0;
    cpl_boolean windowed = 0;
    enu_get_window_info(&nx,
                        &ny,
                        &rot,
                        &strx,
                        &stry,
                        &nx_chip,
                        &ny_chip,
                        &windowed,
                        plist);
    enu_check_error_code("failed to read detector window information");
    if (windowed) {
        cpl_msg_info(cpl_func, "detector is windowed!");
    }

    /* is the instrument ERIS or NACO */

    const char * instrume = cpl_propertylist_get_string(plist, "INSTRUME");
    const cpl_boolean eris = !strcmp(instrume, "ERIS");

    /* probe pixel specified? */

    int probe = CPL_FALSE;
    if ((x_probe >= 0) && (y_probe >= 0)) {
        if (x_probe < nx && y_probe < ny) {
            probe = CPL_TRUE;
        }
    }

    if (probe) {
        int reject = 0;
        hdrl_value val = hdrl_image_get_pixel(himage,
                                              x_probe,
                                              y_probe,
                                              &reject);
        cpl_msg_info(cpl_func,
                     "..basic_calibrate probe (%d,%d) entry(v,e,r)=%5.3e %5.3e %d",
                     (int) x_probe, (int)y_probe, val.data, val.error,
                     reject);
        val = hdrl_image_get_median(himage);
        cpl_msg_info(cpl_func,
                     "..basic_calibrate probe (median) entry(v,e)=%5.3e %5.3e",
                     val.data, val.error);
    }

    if (windowed) {

        /* correct wcs centre, this will have to been set assuming the data
           are full frame */

        cpl_msg_info(cpl_func, "basic calibrate: correcting wcs for windowed data");
        const double crpix1 = cpl_propertylist_get_double(plist, "CRPIX1");
        cpl_propertylist_update_double(plist, "CRPIX1",
                                       crpix1 - (double)strx + 1.0);
        const double crpix2 = cpl_propertylist_get_double(plist, "CRPIX2");
        cpl_propertylist_update_double(plist, "CRPIX2",
                                       crpix2 - (double)stry + 1.0);
        cpl_msg_info(cpl_func, "..crpix %f %f -> %f %f", crpix1, crpix2,
                     crpix1 - (double)strx + 1.0,
                     crpix2 - (double)stry + 1.0);
    }

    /* read offset removal only applies to ERIS */

    if (read_offsets && eris) {
        cpl_msg_info(cpl_func, "basic calibrate: removing read-offsets and "
                             "blanking edges");
        enu_remove_read_offsets(himage, plist, confidence, set_confidence);
        enu_check_error_code("error removing read-offsets");

        if (probe) {
            int reject = 0;
            hdrl_value val = hdrl_image_get_pixel(himage,
                                                  x_probe,
                                                  y_probe,
                                                  &reject);
            cpl_msg_info(cpl_func,
                         "..basic_calibrate probe (%d,%d) after "
                         "read_offsets(v,e,r)=%5.3e %5.3e %d",
                         (int) x_probe, (int)y_probe,
                         val.data, val.error, reject);
            val = hdrl_image_get_median(himage);
            cpl_msg_info(cpl_func,
                         "..basic_calibrate probe (median) after "
                         "read_offsets(v,e)=%5.3e %5.3e", val.data, val.error);
        }
    }

    if (mdark) {
        cpl_msg_info(cpl_func, "basic calibrate: subtracting master dark");

        /* Check that DIT, detector mode and windowing are compatible.
           Should work for both ERIS and NACO-style keywords */

        int compatible = enu_check_conformance(
                         plist,
                         mdark->plist,
                         "^ESO DET SEQ1 DIT$|"
                         "^ESO DET READ CURNAME$|"
                         "^ESO DET SEQ1 WIN NX$|"
                         "^ESO DET SEQ1 WIN NY$|"
                         "^ESO DET SEQ1 WIN ROT$|"
                         "^ESO DET SEQ1 WIN STRX$|"
                         "^ESO DET SEQ1 WIN STRY$|"
                         "^ESO DET DIT$|"
                         "^ESO DET NCORRS NAME$|"
                         "^ESO DET WIN NX$|"
                         "^ESO DET WIN NY$|"
                         "^ESO DET WIN STARTX$|"
                         "^ESO DET WIN STARTY$");
        const char * filename = NULL;
        if (frame) {
            filename = cpl_frame_get_filename(frame);
        }
        if (filename == NULL) {
            enu_check(compatible, CPL_ERROR_INCOMPATIBLE_INPUT,
                      "MASTER_DARK does not match frame");
        } else {
            enu_check(compatible, CPL_ERROR_INCOMPATIBLE_INPUT,
                      "MASTER_DARK does not match %s", filename);
        }

        /* subtract the dark and copy its gain value to propertylist */

        hdrl_image_sub_image(himage, mdark->dark);
        double gain = cpl_propertylist_get_double(mdark->plist,
                                                  "ESO QC GAIN");
        cpl_propertylist_update_double(plist, "ESO DARK GAIN", gain);
        cpl_propertylist_set_comment(plist, "ESO DARK GAIN",
                                     "GAIN from master_dark [e-/ADU]");
        enu_check_error_code("error subtracting DARK");

        if (probe) {
            int reject = 0;
            hdrl_value val = hdrl_image_get_pixel(himage,
                                                  x_probe,
                                                  y_probe,
                                                  &reject);
            int reject2 = 0;
            hdrl_value dark = hdrl_image_get_pixel(mdark->dark,
                                                   x_probe,
                                                   y_probe,
                                                   &reject2);
            cpl_msg_info(cpl_func,
                         "..basic_calibrate probe (%d,%d) after dark_subtract "
                         "dark(v,e,r)=%5.3e %5.3e %d "
                         "data(v,e,r)=%5.3e %5.3e %d",
                         (int) x_probe, (int)y_probe,
                         dark.data, dark.error, reject2,
                         val.data, val.error, reject);
            val = hdrl_image_get_median(himage);
            dark = hdrl_image_get_median(mdark->dark);
            cpl_msg_info(cpl_func,
                         "..basic_calibrate probe (median) after dark_subtract "
                         "dark(v,e)=%5.3e %5.3e data(v,e)=%5.3e %5.3e",
                         dark.data, dark.error,
                         val.data, val.error);
        }
    }
        
    if (gain_lin) {
        hdrl_image * himage_copy = NULL;

        /* Apply linearization correction */

        cpl_msg_info(cpl_func, "basic calibrate: linearizing and estimating "
                     "frame variance");

        himage_copy = end_linearize_and_variance_detmon(gain_lin,
                                                        mdark,
                                                        himage,
                                                        plist,
                                                        x_probe,
                                                        y_probe);
        hdrl_image_copy(himage, himage_copy, 1, 1);
        hdrl_image_delete(himage_copy);
        enu_check_error_code("error estimating variance");

        /* add saturation level associated with linearisation to the
           frames propertylists */

        cpl_propertylist_update_double(plist,
                                       "ESO DETMON SATURATION",
                                       gain_lin->saturation_limit);
        cpl_propertylist_set_comment(plist, 
                                     "ESO DETMON SATURATION",
                                     "saturation level from "
                                     "linearisation [ADU]");

        /* as part of error calculation the data are divided by DIT, so 
           BUNIT -> adu/s */

        cpl_propertylist_update_string(plist, "BUNIT", "adu/s");

        if (probe) {
            int reject = 0;
            hdrl_value val = hdrl_image_get_pixel(himage,
                                                  x_probe,
                                                  y_probe,
                                                  &reject);
            cpl_msg_info(cpl_func,
                         "..basic_calibrate probe (%d,%d) "
                         "after linearize/calc_variance "
                         "data(v,e,r)=%5.3e %5.3e %d",
                         (int) x_probe, (int)y_probe, val.data, val.error,
                         reject);
            val = hdrl_image_get_median(himage);
            cpl_msg_info(cpl_func,
                         "..basic_calibrate probe (median) "
                         "after linearize/calc_variance "
                         "data(v,e)=%5.3e %5.3e", val.data, val.error);
        }
    }

    if (flatfield_1) {
        cpl_msg_info(cpl_func, "basic calibrate: dividing by flatfield 1");

        /* the flatfield will cover the full detector, get the part it
           that covers the detector window of the image */

        /* extract the window */

        hdrl_image * flat_window = !windowed ? NULL :
            hdrl_image_extract(flatfield_1->flat,
                               strx, stry, 
                               strx - 1 + nx,
                               stry - 1 + ny);
        const hdrl_image * flat_use = !windowed ? flatfield_1->flat
            : flat_window;

        enu_check_error_code("error extracting window from flatfield 1");

        /* divide by the flat */

        hdrl_image_div_image(himage, flat_use);

        /* update the confidence */

        if (set_confidence) {

            /* copy the confidence array from the flatfield. Set to 0
               confidence pixels where the image itself is bad. The
               confidence array itself should have no mask - it and the 
               image mask are 'parallel' descriptions of the bad pixels */

            cpl_image * flat_conf_window = !windowed ? NULL :
                cpl_image_extract(flatfield_1->confidence,
                                  strx, stry, 
                                  strx - 1 + nx,
                                  stry - 1 + ny);
            const cpl_image * flat_conf_use = !windowed ? flatfield_1->confidence
                : flat_conf_window;

            enu_check_error_code("error extracting window from flatfield 1 "
                                 "confidence");
            cpl_image_copy(confidence, flat_conf_use, 1, 1);
            cpl_image_delete(flat_conf_window);
  
            /* set confidence to 0 where the image mask is already bad */
            if (cpl_image_get_bpm_const(hdrl_image_get_image(himage))) {
                cpl_mask * old_mask =
                    cpl_image_set_bpm(confidence,
                                      cpl_image_get_bpm(hdrl_image_get_image(himage)));
                cpl_mask_delete(old_mask);
                cpl_image_fill_rejected(confidence, 0.0);
                (void)cpl_image_unset_bpm(confidence);
            } else {
                cpl_image_accept_all(confidence); /* FIXME: Is this reachable ? */
            }
        }

        enu_check_error_code("error dividing by flatfield 1");

        if (probe) {
            int reject = 0;
            hdrl_value val = hdrl_image_get_pixel(himage,
                                                  x_probe,
                                                  y_probe,
                                                  &reject);
            int reject2 = 0;
            hdrl_value flat = hdrl_image_get_pixel(flat_use,
                                                   x_probe,
                                                   y_probe,
                                                   &reject2);
            cpl_msg_info(cpl_func,
                         "..basic_calibrate probe (%d,%d) after flatfield1 "
                         "flat(v,e,r)=%5.3e %5.3e %d "
                         "data(v,e,r)=%5.3e %5.3e %d",
                         (int) x_probe, (int)y_probe,
                         flat.data, flat.error, reject2,
                         val.data, val.error, reject);
            val = hdrl_image_get_median(himage);
            flat = hdrl_image_get_median(flat_use);
            cpl_msg_info(cpl_func,
                         "..basic_calibrate probe (median) after flatfield1 "
                         "flat(v,e)=%5.3e %5.3e data(v,e)=%5.3e %5.3e",
                         flat.data, flat.error, 
                         val.data, val.error);
        }

        hdrl_image_delete(flat_window);
    }

    if (flatfield_2) {
        cpl_msg_info(cpl_func, "basic calibrate: dividing by flatfield 2");

        hdrl_image * flat_window = !windowed ? NULL :
            hdrl_image_extract(flatfield_2->flat,
                               strx, stry, 
                               strx - 1 + nx,
                               stry - 1 + ny);
        const hdrl_image * flat_use = !windowed ? flatfield_2->flat
            : flat_window;

        enu_check_error_code("error extracting window from flatfield 2");

        hdrl_image_div_image(himage, flat_use);
 
        if (set_confidence) {
            cpl_image * flat_conf_window = !windowed ? NULL :
                cpl_image_extract(flatfield_2->confidence,
                                  strx, stry, 
                                  strx - 1 + nx,
                                  stry - 1 + ny);
            const cpl_image * flat_conf_use = !windowed ? flatfield_2->confidence
                : flat_conf_window;

            enu_check_error_code("error extracting window from flatfield 1 "
                                 "confidence");

            cpl_image_copy(confidence, flat_conf_use, 1, 1);
            cpl_image_delete(flat_conf_window);
  
            /* set confidence to 0 where the image mask is already bad */
            if (cpl_image_get_bpm_const(hdrl_image_get_image(himage))) {
                cpl_mask * old_mask =
                    cpl_image_set_bpm(confidence,
                                      cpl_image_get_bpm(hdrl_image_get_image(himage)));
                cpl_mask_delete(old_mask);
                cpl_image_fill_rejected(confidence, 0.0);
                (void)cpl_image_unset_bpm(confidence);
            } else {
                cpl_image_accept_all(confidence); /* FIXME: Is this reachable ? */
            }
        }

        enu_check_error_code("error dividing by flatfield 2");

        if (probe) {
            int reject = 0;
            hdrl_value val = hdrl_image_get_pixel(himage,
                                                  x_probe,
                                                  y_probe,
                                                  &reject);
            int reject2 = 0;
            hdrl_value flat = hdrl_image_get_pixel(flat_use,
                                                   x_probe,
                                                   y_probe,
                                                   &reject2);
            cpl_msg_info(cpl_func,
                         "..basic_calibrate probe (%d,%d) after flatfield2 "
                         "flat(v,e,r)=%5.3e %5.3e %d "
                         "data(v,e,r)=%5.3e %5.3e %d",
                         (int) x_probe, (int)y_probe,
                         flat.data, flat.error, reject2,
                         val.data, val.error, reject);
            val = hdrl_image_get_median(himage);
            flat = hdrl_image_get_median(flat_use);
            cpl_msg_info(cpl_func,
                         "..basic_calibrate probe (median) after flatfield2 "
                         "flat(v,e)=%5.3e %5.3e data(v,e)=%5.3e %5.3e",
                         flat.data, flat.error, 
                         val.data, val.error);
        }
        hdrl_image_delete(flat_window);
    }

    if (mbad_pix_map) {
        cpl_msg_info(cpl_func, "basic calibrate: associating with master_bpm");
        cpl_mask * new_mask = en_master_bpm_get_mask(mbad_pix_map, flag_mask);

        /* window the bpm mask appropriate to the image */

        cpl_mask * new_mask_window = !windowed ? NULL :
            cpl_mask_extract(new_mask,
                             strx, stry, 
                             strx - 1 + nx,
                             stry - 1 + ny);
        const cpl_mask * new_mask_use = !windowed ? new_mask : new_mask_window;

        enu_check_error_code("error extracting window from bpm mask");

        /* 'or' the mbad_pix_map mask with that already on the image */

        cpl_mask * current_mask = hdrl_image_get_mask(himage);
        cpl_mask_or(current_mask, new_mask_use);
        enu_check_error_code("error setting new mask");

        if (set_confidence) {

            /* set rejected confidence to zero */

            double * conf_data = cpl_image_get_data(confidence);
            cpl_binary * mask_data = cpl_mask_get_data(current_mask);
            cpl_size nx_mask = cpl_mask_get_size_x(current_mask);
            cpl_size ny_mask = cpl_mask_get_size_y(current_mask);
            for (cpl_size i = 0; i < nx_mask * ny_mask; i++) {
                if (mask_data[i] == CPL_BINARY_1) {
                    conf_data[i] = 0.0;
                }
            }
        }            

        if (probe) {
            int reject = 0;
            hdrl_value val = hdrl_image_get_pixel(himage,
                                                  x_probe,
                                                  y_probe,
                                                  &reject);
            int ignore = 0;
            double conf = cpl_image_get(confidence, x_probe, y_probe, &ignore);
            cpl_msg_info(cpl_func, "..basic_calibrate probe (%d,%d) "
                         "after associate_mask(v,e,r,c)=%5.3e %5.3e %d %5.3e",
                         (int) x_probe, (int)y_probe, val.data, val.error,
                         reject, conf);
        }
        cpl_mask_delete(new_mask);
        cpl_mask_delete(new_mask_window);
    }

    if (!strcmp(fill_rejected, "noop")) {
        cpl_msg_info(cpl_func, "not modifying rejected pixels");
    } else {
        double fill = 0.0;
        if (!strcmp(fill_rejected, "set_NaN")) {
            fill = NAN;
            cpl_msg_info(cpl_func, "basic calibrate: setting rejected pixels "
                         "to NaN");
        } else if (!strcmp(fill_rejected, "set_value")) {
            fill = fill_value;
            cpl_msg_info(cpl_func, "basic calibrate: setting rejected pixels "
                         "to %f", fill);
        } else {
            enu_check(CPL_FALSE, CPL_ERROR_UNSPECIFIED,
                      "bad fill-rejected value: programming error");
        }
        cpl_image_fill_rejected(hdrl_image_get_image(himage), fill);
        cpl_image_fill_rejected(hdrl_image_get_error(himage), fill);
        enu_check_error_code("error setting rejected pixels to %f",
                             fill_value);

        if (probe) {
            int reject = 0;
            hdrl_value val = hdrl_image_get_pixel(himage,
                                                  x_probe,
                                                  y_probe,
                                                  &reject);
            int ignore = 0;
            double conf = cpl_image_get(confidence, x_probe, y_probe, &ignore);
            cpl_msg_info(cpl_func,
                         "..basic_calibrate probe (%d,%d) after "
                         "fill-rejected(v,e,r,c)=%5.3e %5.3e %d %5.3e",
                         (int) x_probe, (int)y_probe, val.data, val.error,
                         reject, conf);
        }
    }

 cleanup:
    eris_check_error_code("enu_basic_calibrate_himage");
    return cpl_error_get_code();;
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Calculate a flatfield result.
  @param    himlist         The hdrl_imagelist of flat measurements
  @param    min_coadds      The minimum number of results required to be combined
  @param    collapse_params Parameter controlling the collapse algorithm
  @param    filter_size_x   The size of the median filter in x
  @param    filter_size_y   The size of the median filter in y
  @param    method          HDRL_FLAT_FREQ_HIGH or HRDL_FLAT_FREQ_LOW
  @return   An hdrl_image with the flatfield result, or NULL
 */
/*----------------------------------------------------------------------------*/

hdrl_image * enu_calc_flat(hdrl_imagelist * himlist,
                           const int min_coadds,
                           const hdrl_parameter * collapse_params,
                           const cpl_size filter_size_x,
                           const cpl_size filter_size_y,
                           const hdrl_flat_method method) {

    /* entry check */

    if (cpl_error_get_code() != CPL_ERROR_NONE) return NULL;
    cpl_ensure(himlist, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(collapse_params, CPL_ERROR_NULL_INPUT, NULL);

    cpl_image      * contribution = NULL;
    hdrl_parameter * flat_params = NULL;
    hdrl_image     * result = NULL;

    /* verify that we have enough difference results */

    const cpl_size size = hdrl_imagelist_get_size(himlist);
    enu_check(size >= min_coadds, CPL_ERROR_ILLEGAL_INPUT, 
              "insufficient frames for flat (%d<min:%d)",
              (int) size, min_coadds);

    /* calculate the flatfield */

    flat_params = hdrl_flat_parameter_create(filter_size_x,
                                             filter_size_y,
                                             method);
    if (method == HDRL_FLAT_FREQ_HIGH) {
        cpl_msg_info(cpl_func, "Calculating high freq flatfield");
    } else if (method == HDRL_FLAT_FREQ_LOW) {
        cpl_msg_info(cpl_func, "Calculating low freq flatfield");
    }
    cpl_msg_info(cpl_func, "..smoothing filter sizes %d %d",
                 (int)filter_size_x, (int)filter_size_y);
    hdrl_flat_compute(himlist, NULL, collapse_params, 
                      flat_params, &result, &contribution);
    enu_check_error_code("error computing flat-field");

 cleanup:
    cpl_image_delete(contribution);
    hdrl_parameter_delete(flat_params);    
    if (cpl_error_get_code() != CPL_ERROR_NONE) {
        hdrl_image_delete(result);
        result = NULL;
    }
    eris_check_error_code("enu_calc_flat");
    return result;
}


/*----------------------------------------------------------------------------*/
/**
  @brief   Calculate magnitude limit of image
  @param   limage      The image in question
  @param   photzp      input photometric zero point
  @param   fwhm_pix    input FWHM in pixel units
  @param   abmaglim    double* to contain result
  @return  CPL_ERROR_NONE if all goes well, otherwise cpl_error_get_code()

  This routine is a wrapper for hdrl_maglim_compute.
  */
/*----------------------------------------------------------------------------*/

cpl_error_code enu_calc_maglim(const located_image * limage,
                               const double photzp,
                               const double fwhm_pix,
                               double * abmaglim) {

    if (cpl_error_get_code() != CPL_ERROR_NONE) return cpl_error_get_code();
    cpl_ensure_code(limage, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(abmaglim, CPL_ERROR_NULL_INPUT);

    /* bad data in gives bad result */
    if (fwhm_pix <= 0.0) {
        *abmaglim = -1.0;
        return CPL_ERROR_NONE;
    }

    /* calculate magnitude limit using HDRL */
    /* ..set mode calculation following HDRL docs example */

    double histo_min = 10.;
    double histo_max = 1.;
    histo_min=0;
    histo_max=0;
    double bin_size = 0.;
    cpl_size error_niter = 0;
    hdrl_mode_type mode_method = HDRL_MODE_MEDIAN;
    hdrl_parameter * mode_parameter = NULL;
    mode_parameter = hdrl_collapse_mode_parameter_create(histo_min,
                                                         histo_max,
                                                         bin_size,
                                                         mode_method,
                                                         error_niter);
    /* AMO: the following is NOT required */
    double* pdata = cpl_image_get_data_double(hdrl_image_get_image(limage->himage));
    double* pconf = cpl_image_get_data_double(limage->confidence);
    cpl_size sx = hdrl_image_get_size_x(limage->himage);
    cpl_size sy = hdrl_image_get_size_y(limage->himage);
    cpl_mask* mask_data = cpl_image_get_bpm(hdrl_image_get_image(limage->himage));
    cpl_mask* mask_errs = cpl_image_get_bpm(hdrl_image_get_error(limage->himage));
    cpl_binary* pbpm_data = cpl_mask_get_data(mask_data);
    cpl_binary* pbpm_errs = cpl_mask_get_data(mask_errs);
    for(cpl_size indexj = 0; indexj < sy; indexj++) {
    	for(cpl_size indexi = 0; indexi < sx; indexi++) {
    		if(!isfinite(pdata[indexi+indexj*sx]) ) {
    			pbpm_data[indexi+indexj*sx] = CPL_BINARY_1;
    			pbpm_errs[indexi+indexj*sx] = CPL_BINARY_1;
    			pconf[indexi+indexj*sx] = CPL_BINARY_1;
//cpl_msg_warning(cpl_func,"found bad pixel at : [%lld,%lld]",indexi,indexj);
    		}
    	}
    }
   

    hdrl_maglim_compute(hdrl_image_get_image(limage->himage),
                        photzp,
                        fwhm_pix,
                        ((int) (3.0 * fwhm_pix) / 2) * 2 + 1,
                        ((int) (3.0 * fwhm_pix) / 2) * 2 + 1,
                        HDRL_IMAGE_EXTEND_MIRROR,
                        mode_parameter,
                        abmaglim);
    hdrl_parameter_delete(mode_parameter);

    eris_check_error_code("enu_calc_maglim");
    return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
  @brief   Calculate predicted positions of catalogue objects for given wcs
  @param   catalogue   The catalogue of objects
  @param   wcs_plist   Propertylist containing the wcs info
  @return  CPL_ERROR_NONE if all goes well, otherwise cpl_error_get_code()

  This routine takes an object catalogue table and a wcs (read from a
  propertylist), calculates the pixel coords of objects in the catalogue,
  and adds them to the table in columns "X_coordinate" and "Y_coordinate".
  */
/*----------------------------------------------------------------------------*/

cpl_error_code enu_calc_pixel_coords(cpl_table * catalogue,
                                     const cpl_propertylist * wcs_plist) {

    if (cpl_error_get_code() != CPL_ERROR_NONE) return cpl_error_get_code();
    cpl_ensure_code(catalogue, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(wcs_plist, CPL_ERROR_NULL_INPUT);

    cpl_array        * conv_status = NULL;
    cpl_matrix       * ref_phys = NULL;
    cpl_matrix       * ref_world = NULL;
    cpl_wcs          * wcs = NULL;

    /* write files with inputs for regression testing - uncomment if required
    cpl_propertylist * wcs_plist_copy = cpl_propertylist_duplicate(wcs_plist);
    cpl_propertylist_erase_regexp(wcs_plist_copy, CPL_WCS_REGEXP, 1);
    cpl_propertylist_save(wcs_plist_copy, "enu_calc_pixel_coords_test_wcs.fits",
                          CPL_IO_CREATE);
    cpl_table_save(catalogue, NULL, NULL, 
                   "enu_calc_pixel_coords_test_input_table.fits",
                   CPL_IO_CREATE);
    */

    cpl_size nobj = cpl_table_get_nrow(catalogue);
    const double * ra = cpl_table_get_data_double_const(catalogue, "RA");
    const double * dec = NULL;
//    if (cpl_table_has_column(catalogue, "Dec")) {
//        dec = cpl_table_get_data_double_const(catalogue, "Dec");
//    } else if (cpl_table_has_column(catalogue, "DEC")) {
    if (cpl_table_has_column(catalogue, "DEC")) {
        dec = cpl_table_get_data_double_const(catalogue, "DEC");
    }
    ref_world = cpl_matrix_new(nobj, 2);
    for (cpl_size iobj = 0; iobj < nobj; iobj++) {
        cpl_matrix_set(ref_world, iobj, 0, ra[iobj]);
        cpl_matrix_set(ref_world, iobj, 1, dec[iobj]);
    }

    wcs = cpl_wcs_new_from_propertylist(wcs_plist);

    cpl_wcs_convert(wcs, ref_world, &ref_phys, &conv_status,
                    CPL_WCS_WORLD2PHYS);
    enu_check_error_code("failure calculating predicted positions");

    /* Store the predicted positions in columns X_coordinate and
       Y_coordinate, create thes if necessary */

    if (!cpl_table_has_column(catalogue, "X_coordinate")) {
        cpl_table_new_column(catalogue, "X_coordinate", CPL_TYPE_DOUBLE); 
    }
    if (!cpl_table_has_column(catalogue, "Y_coordinate")) {
        cpl_table_new_column(catalogue, "Y_coordinate", CPL_TYPE_DOUBLE); 
    }

    for (cpl_size iobj = 0; iobj < nobj; iobj++) {
        cpl_table_set_double(catalogue, "X_coordinate", iobj, 
                             cpl_matrix_get(ref_phys, iobj, 0));
        cpl_table_set_double(catalogue, "Y_coordinate", iobj, 
                             cpl_matrix_get(ref_phys, iobj, 1));
    }

    /* write file with output for regression testing - uncomment if required
    cpl_table_save(catalogue, NULL, NULL,
                   "enu_calc_pixel_coords_test_output_table.fits",
                   CPL_IO_CREATE);
    */

 cleanup:
    cpl_array_delete(conv_status);
    cpl_matrix_delete(ref_phys);
    cpl_matrix_delete(ref_world);
    cpl_wcs_delete(wcs);
    eris_check_error_code("enu_calc_pixel_coords");
    return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
  @brief   Wrapper for hdrl_catalogue_compute.
  @param   himage      Input image
  @param   confidence  Confidence map
  @param   wcs         wcs information
  @param   params      parameter struct controlling catalogue determination
  @return  Catalogue of detected objects or NULL if none found

  The function checks that the confidence array is not all 0 before calling
  hdrl_catalogue_compute. Calling that routine with a bad confidence array
  gives rise to a failure with a confusing error report.

  cat = enu_catalogue_compute(himage, confidence, wcs, cat_params);
 */
/*----------------------------------------------------------------------------*/

hdrl_catalogue_result * enu_catalogue_compute(
        const hdrl_image * himage,
        const cpl_image * confidence,
        const cpl_wcs * wcs,
        hdrl_parameter * params) {

    if (cpl_error_get_code() != CPL_ERROR_NONE) return NULL;

    cpl_ensure(himage, CPL_ERROR_NULL_INPUT, NULL);
/*
    cpl_ensure(confidence, CPL_ERROR_NULL_INPUT, NULL);
*/
    cpl_ensure(wcs, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(params, CPL_ERROR_NULL_INPUT, NULL);

    hdrl_catalogue_result * result = NULL;

    if (confidence) {

       /* Check that the confidence array for the image is not zero -
          this causes a non-obvious error to be generated deep in the
          casu routines */

       cpl_ensure(cpl_image_get_max(confidence) > 0.0, CPL_ERROR_ILLEGAL_INPUT,
                  NULL);
    }

    result = hdrl_catalogue_compute(hdrl_image_get_image_const(himage),
                                    confidence, wcs, params);

    /* REMOVE THE FOLLOWING FOR NOW. IT WAS A FUDGE TO TRY AND IMPROVE
       SOURCE MATCHING BETWEN JITTERS - BUT BETTER TO LEAVE IT IN AND
       FILTER LATER AS REQUIRED */
    /* Remove catalogue entries with low Av_conf.
       CASU documentation: Av_conf = the average confidence level within the 
                                     default rcore aperture useful for
                                     spotting spurious outliers in various
                                     parameter selection spaces.
       The NIX detector has patches with a low density of working pixels
       which equate to areas of low confidence */
/*
    cpl_table_and_selected_double(result->catalogue, "Av_conf", CPL_LESS_THAN,
                                  90.);
    cpl_table_erase_selected(result->catalogue);
*/

    if (cpl_error_get_code() != CPL_ERROR_NONE) {

        /* Return NULL on error */

        hdrl_catalogue_result_delete(result);
        result = NULL;

        /* but trap case where simply no objects were present 
           - for our purposes this is not an error */

        const char * message = cpl_error_get_message();
        cpl_msg_info(cpl_func, "catalogue %s", message);

        if (strstr(message, "No objects found in image")) {
            cpl_msg_warning(cpl_func, "No objects found in image");
            cpl_error_reset();
        }
    }
    eris_check_error_code("enu_catalogue_compute");
    return result;
}


/*----------------------------------------------------------------------------*/
/**
  @brief   Find images taken within a given time of a target image
  @param   target      The target image
  @param   timerange   The time range centred on target for suitable images 
  @param   pool        List of candidate images
  @param   debug       True to print debug information
  @return  A vector with the indeces of suitable images in the candidate pool
  */
/*----------------------------------------------------------------------------*/

cpl_vector * enu_bracket_skys(const located_image * target,
                              const double timerange,
                              const located_imagelist * pool,
                              const int debug) {

    if (cpl_error_get_code() != CPL_ERROR_NONE) return NULL;

    cpl_ensure(target, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(timerange > 0.0, CPL_ERROR_ILLEGAL_INPUT, NULL);
    cpl_ensure(pool, CPL_ERROR_NULL_INPUT, NULL);

    cpl_vector * result = NULL;
    cpl_vector * result_diff = NULL;

    double target_mjd = cpl_propertylist_get_double(target->plist, "MJD-OBS");

    for (cpl_size j = 0; j < pool->size; j++) {
        double mjd_j = cpl_propertylist_get_double((pool->limages[j])->
                                                   plist, "MJD-OBS");
        double diff = (target_mjd - mjd_j) * 3600.0 * 24.0;

        if (fabs(diff) < timerange/2.0) {
            if (result == NULL) {
                result = cpl_vector_new(1);
                result_diff = cpl_vector_new(1);
                cpl_vector_set(result, 0, (double)j);
                cpl_vector_set(result_diff, 0, diff);
            } else {
                cpl_size size = cpl_vector_get_size(result);
                cpl_vector_set_size(result, size + 1);
                cpl_vector_set_size(result_diff, size + 1);
                cpl_vector_set(result, size, (double)j);
                cpl_vector_set(result_diff, size, diff);
            }
        }
    }
    enu_check_error_code("error in enu_bracket_skys: %s",
                         cpl_error_get_message());
    enu_check(result && cpl_vector_get_size(result) > 0,
              CPL_ERROR_INCOMPATIBLE_INPUT,
              "no sky frames found within %f sec of target frame", timerange/2.0);

    if (debug) {
        cpl_size size = cpl_vector_get_size(result);
        if (size > 0) {
            cpl_msg_info(" ", ".. enu_bracket_skys: timerange %5.1f", timerange);
            cpl_msg_info(" ", "..  j#    diff(sec)");
        }
        for (cpl_size i = 0; i < size; i++) {
            int j = cpl_vector_get(result, i);
            double diff = cpl_vector_get(result_diff, i);
            cpl_msg_info(" ", ".. %3d  %7.1f", j, diff);
        }
    }
    
  cleanup:
    if (cpl_error_get_code() != CPL_ERROR_NONE) {
        cpl_vector_delete(result);
        result = NULL;
    }
    cpl_vector_delete(result_diff);
    eris_check_error_code("enu_bracket_skys");
    return result;
}


/*----------------------------------------------------------------------------*/
/**
  @brief   Calculate object catalogues for a list of images
  @param   limlist     List of input images
  @param   params      parameter struct controlling catalogue determination
  @return  CPL_ERROR_NONE if all goes well, otherwise a cpl error code

 */
/*----------------------------------------------------------------------------*/

cpl_error_code enu_catalogue_limlist(located_imagelist * limlist,
                                     hdrl_parameter * params) {

    cpl_ensure_code(cpl_error_get_code() == CPL_ERROR_NONE,
                    cpl_error_get_code());
    cpl_ensure_code(limlist, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(params, CPL_ERROR_NULL_INPUT);

    cpl_size njitters = limlist->size;
    cpl_msg_info(cpl_func, "cataloguing %d images", (int) njitters);

    for (cpl_size i = 0; i < njitters; i++) {
        cpl_wcs * wcs = cpl_wcs_new_from_propertylist
            (limlist->limages[i]->plist);
        cpl_msg_info(cpl_func, "cataloguing image %d", (int) i);
        /* AMO the following is required to handle bad pixels that were not all flagged */
        double* pdata = cpl_image_get_data_double(hdrl_image_get_image((limlist->limages[i])->himage));
        //double* perrs = cpl_image_get_data_double(hdrl_image_get_error((limlist->limages[i])->himage));
        cpl_size sx = hdrl_image_get_size_x(limlist->limages[i]->himage);
        cpl_size sy = hdrl_image_get_size_y(limlist->limages[i]->himage);
        cpl_mask* mask = cpl_image_get_bpm(hdrl_image_get_image(limlist->limages[i]->himage));
        cpl_mask* mskerr = cpl_image_get_bpm(hdrl_image_get_error(limlist->limages[i]->himage));
        double* pconf = cpl_image_get_data_double(limlist->limages[i]->confidence);
        cpl_binary* pbpm_data = cpl_mask_get_data(mask);
        cpl_binary* pbpm_errs = cpl_mask_get_data(mskerr);
        for(cpl_size indexj = 0; indexj < sy; indexj++) {
        	for(cpl_size indexi = 0; indexi < sx; indexi++) {
        		if(!isfinite(pdata[indexi+indexj*sx]) ) {
        			pbpm_data[indexi+indexj*sx] = CPL_BINARY_1;
        			pbpm_errs[indexi+indexj*sx] = CPL_BINARY_1;
        			pconf[indexi+indexj*sx] = 0;
        		}
        	}
        }
       
        //TODO AMO: the following call generates a memory leak
        limlist->limages[i]->objects = enu_catalogue_compute(
                                       limlist->limages[i]->himage,
                                       limlist->limages[i]->confidence,
                                       wcs,
                                       params);
        cpl_wcs_delete(wcs);

        if (cpl_error_get_code() != CPL_ERROR_NONE) break;
    }
    eris_check_error_code("enu_catalogue_limlist");
    return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Check that the specified subset of two propertylists match
  @param    plist1  The first property list
  @param    plist2  The second property list
  @param    regexp  A regular expression specifying the properties to match
  @return   1 for a match, 0 otherwise

  The function checks that a given subset of two propertylists is identical.
  For example, if you want to check that the DIT and detector mode
  are the same in two lists you would call:

  match = enu_check_conformance(plist1, plist2,
                                "^ESO DET DIT$|^ESO DET READ CURNAME$");
 */
/*----------------------------------------------------------------------------*/

int enu_check_conformance(const cpl_propertylist * plist1,
                          const cpl_propertylist * plist2,
                          const char * regexp) {

    if (cpl_error_get_code() != CPL_ERROR_NONE) return 0;

    /* Check parameters not NULL */

    cpl_ensure(plist1 && plist2 && regexp, CPL_ERROR_NULL_INPUT, 0);

    int result = CPL_TRUE;
    char * reason = NULL;

    /* Create the two subsets to be compared */

    cpl_propertylist * subset1 = cpl_propertylist_new();
    cpl_propertylist_copy_property_regexp(subset1, plist1, regexp, 0);
    cpl_propertylist * subset2 = cpl_propertylist_new();
    cpl_propertylist_copy_property_regexp(subset2, plist2, regexp, 0);
    enu_check_error_code("error deriving subset of propertylist");

    /* Compare them */

    cpl_size size1 = cpl_propertylist_get_size(subset1);
    cpl_size size2 = cpl_propertylist_get_size(subset2);
    if (size1 != size2) {
        result = CPL_FALSE;
        reason = cpl_sprintf("propertylist sizes do not match %s", " ");
    } else {
        for (cpl_size i=0; i<size1; i++) {
            const cpl_property * prop1 = cpl_propertylist_get(subset1, i);
            const char * name = cpl_property_get_name(prop1);

            /* does the second list have a property of that name? */

            if (!cpl_propertylist_has(subset2, name)) {
                result = CPL_FALSE;
                reason = cpl_sprintf("name mismatch: %s", name);
                break;
            }

            /* do the values match? */

            const cpl_property * prop2 = cpl_propertylist_get_property_const(
                                         subset2, name);

            const cpl_type type1 = cpl_property_get_type(prop1);
            const cpl_type type2 = cpl_property_get_type(prop2);
            if (type1 != type2) {
                result = CPL_FALSE;
                reason = cpl_sprintf("type mismatch: name %s: types %s and %s",
                         name,
                         cpl_type_get_name(cpl_property_get_type(prop1)),
                         cpl_type_get_name(cpl_property_get_type(prop2)));
                break;
            }
      
            int ival1 = 0;
            int ival2 = 0;
            double dval1 = 0.0;
            double dval2 = 0.0;
            const char * sval1 = NULL;
            const char * sval2 = NULL;

            switch (type1) {
            case CPL_TYPE_INT:
                ival1 = cpl_property_get_int(prop1);
                ival2 = cpl_property_get_int(prop2);
                if (ival1 != ival2) {
                    result = CPL_FALSE;
                    reason = cpl_sprintf("value mismatch: name %s: values "
                                         "%d and %d",
                                         name, ival1, ival2);
                }
                break;

            case CPL_TYPE_DOUBLE:
                dval1 = cpl_property_get_double(prop1);
                dval2 = cpl_property_get_double(prop2);
                if (dval1 != dval2) {
                    result = CPL_FALSE;
                    reason = cpl_sprintf("value mismatch: name %s: values "
                                         "%4.2e and %4.2e",
                                         name, dval1, dval2);
                }
                break;

            case CPL_TYPE_STRING:
                sval1 = cpl_property_get_string(prop1);
                sval2 = cpl_property_get_string(prop2);
                if (strcmp(sval1, sval2)) {
                    result = CPL_FALSE;
                    reason = cpl_sprintf("value mismatch: name %s: values "
                                         "%s, %s", name, sval1, sval2);
                }
                break;
            default:
                cpl_error_set_message(cpl_func, CPL_ERROR_INVALID_TYPE,
                                      "name %s, type not handled %s",
                                      name, cpl_type_get_name(
                                      cpl_property_get_type(prop1))); 
                break;

            if (result == CPL_FALSE) break;
            }        
        }
    }

cleanup:

    cpl_propertylist_delete(subset1);
    cpl_propertylist_delete(subset2);


    /* Always return False on error */

    if (cpl_error_get_code() != CPL_ERROR_NONE) {
        result = CPL_FALSE;
    } else {
        if (result == CPL_FALSE) {
            cpl_msg_info(cpl_func, "propertylist match failed: %s", reason);
        }
    }
    cpl_free(reason);
    eris_check_error_code("enu_check_conformance");
    return result;
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Check that the image has valid WCS keywords
  @param    limage  The image to be checked
  @return   A cpl_error_code, CPL_ERROR_NONE if all is well.

  The function checks that the given image has keywords CTYPE1 and 
  CTYPE2 and that neither of them is set to "PIXEL". This would signify
  that no astronomical WCS has been set - as happened during the early
  ERIS commissioning runs, requiring that the WCS be 'fixed' before running
  through the pipeline.
 */
/*----------------------------------------------------------------------------*/

cpl_error_code enu_check_wcs(const located_image * limage) {

    if (cpl_error_get_code() != CPL_ERROR_NONE) return cpl_error_get_code();
    cpl_ensure_code(limage, CPL_ERROR_NULL_INPUT);

    cpl_boolean bad = CPL_FALSE;

    if (!cpl_propertylist_has(limage->plist, "CTYPE1")) {
        bad = CPL_TRUE;;
    }
    if (!bad) {
        const char * ctype1 = cpl_propertylist_get_string(
                              limage->plist, "CTYPE1");
        if (strstr(ctype1, "PIXEL")) {
            bad = CPL_TRUE;
        }
    }

    if (!bad) {
        if (!cpl_propertylist_has(limage->plist, "CTYPE2")) {
            bad = CPL_TRUE;
        }
    }
    if (!bad) {
        const char * ctype2 = cpl_propertylist_get_string(limage->plist,
                                                          "CTYPE2");
        if (strstr(ctype2, "PIXEL")) {
            bad = CPL_TRUE;
        } 
    }

    cpl_error_code result = cpl_error_get_code();
    if (result == CPL_ERROR_NONE && bad) {
        result = cpl_error_set_message(cpl_func, CPL_ERROR_ILLEGAL_INPUT, 
                                       "missing or bad WCS keywords in "
                                       "input file %s", 
                                       cpl_frame_get_filename(limage->frame));
    }
    eris_check_error_code("enu_check_wcs");
    return result;
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Save a list of intermediate image results for use in debugging
  @param    debug       True if the debug file are to be written
  @param    limlist     The list of images to be written
  @param    nameroot    The preface added to the filename associated with each image
  @param    recipename  The recipe name
  @param    frameset    The list of input frames for the recipe
  @param    parlist     The list of input parameters
  @param    used        The list of raw/calibration frames used for this product 
  @return   CPL_ERROR_NONE if all goes well, otherwise a cpl error code

 */
/*----------------------------------------------------------------------------*/

cpl_error_code enu_debug_limlist_save(const int debug, 
                                      const located_imagelist * limlist,
                                      const char * nameroot, 
                                      const char * recipename,
                                      cpl_frameset * frameset,
                                      const cpl_parameterlist * parlist,
                                      const cpl_frameset * used) {

    if (cpl_error_get_code() != CPL_ERROR_NONE) return cpl_error_get_code();
    if (!debug) return CPL_ERROR_NONE;
    cpl_ensure_code(limlist, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(nameroot, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(recipename, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(frameset, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(parlist, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(used, CPL_ERROR_NULL_INPUT);

    char             * fname_copy = NULL;     
    char             * out_fname = NULL;     

    /* loop through images */

    cpl_size nimages = limlist->size;
    for (cpl_size j = 0; j < nimages; j++) {

        /* generate name of output file */

        if (limlist->limages[j]->frame) {
            const char * fname = cpl_frame_get_filename(
                                 limlist->limages[j]->frame);
            fname_copy = cpl_strdup(fname);
        } else {
            const char * fname = "debug.fits";
            fname_copy = cpl_strdup(fname);
        }
        out_fname = cpl_sprintf("%s_%s", nameroot, basename(fname_copy));

        cpl_msg_info(cpl_func, "..(debug) writing %s", out_fname);

        /* some dummy info to keep dfs happy */

        cpl_propertylist * applist = cpl_propertylist_new();
        cpl_propertylist_update_string(applist,
                                       CPL_DFS_PRO_CATG,
                                       "debug");
        cpl_propertylist_update_string(applist, "PRODCATG", "ANCILLARY.IMAGE");

        cpl_frameset * provenance = cpl_frameset_new();
        cpl_frameset_insert(provenance, cpl_frame_duplicate(
                            limlist->limages[j]->frame));

        enu_dfs_save_limage(frameset,
                            parlist,
                            provenance,
                            CPL_TRUE,
                            limlist->limages[j],
                            recipename,
                            limlist->limages[j]->frame,
                            applist,
                            "test",
                            out_fname);

        cpl_frameset_delete(provenance);
        cpl_propertylist_delete(applist);
    }

    /* tidy up */

    cpl_free(fname_copy);
    cpl_free(out_fname);
    eris_check_error_code("enu_debug_limlist_save");
    return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Save a flatfield result.
  @param    pro_catg    The value of the ESO.PRO.CATG keyword
  @param    flat        The flatfield result
  @param    confidence  The confidence array
  @param    cold_bpm    The cold pixel mask, may be NULL
  @param    frameset    The list of input frames for the recipe
  @param    parlist     The list of input parameters
  @param    filename    The name of the output file
  @param    recipe_name The recipe name
  @return   CPL_ERROR_NONE if all goes well, otherwise an error code
 */
/*----------------------------------------------------------------------------*/

cpl_error_code enu_flat_save(const char * pro_catg,
                             const hdrl_image * flat,
                             const cpl_image * confidence,
                             const cpl_mask * cold_bpm,
                             cpl_frameset * frameset,
                             const cpl_parameterlist * parlist,
                             const char * filename,
                             const char * recipe_name,
                             const cpl_propertylist* qclog) {

    /* checks */
   
    if (cpl_error_get_code() != CPL_ERROR_NONE) return cpl_error_get_code();
    cpl_ensure_code(pro_catg, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(flat, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(frameset, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(parlist, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(filename, CPL_ERROR_NULL_INPUT);

    mef_extension_list   * mefs = NULL;
    cpl_propertylist     * plist = NULL;

    /* make output propertylist */

    plist = cpl_propertylist_new();
    cpl_propertylist_append_string(plist, CPL_DFS_PRO_CATG, pro_catg);
    cpl_propertylist_update_string(plist, "PRODCATG", "ANCILLARY.IMAGE");

    /* add QC parameters  */

    hdrl_value flat_median = hdrl_image_get_median(flat);
    cpl_propertylist_append_double(plist, "ESO QC FLAT MED",
                                   (double) flat_median.data);
    cpl_propertylist_set_comment(plist,"ESO QC FLAT MED",
    		"[ADU] Median flat value");
    hdrl_value flat_mean = hdrl_image_get_mean(flat);
    cpl_propertylist_append_double(plist, "ESO QC FLAT MEAN", 
                                   (double) flat_mean.data);
    cpl_propertylist_set_comment(plist,"ESO QC FLAT MEAN",
        		"[ADU] Mean flat value");
    double flat_rms = hdrl_image_get_stdev(flat);
    cpl_propertylist_append_double(plist, "ESO QC FLAT RMS", flat_rms);
    cpl_propertylist_set_comment(plist,"ESO QC FLAT RMS",
           		"[ADU] RMS flat value");
    /* the normalisation factor is not returned by hdrl_flat_compute,
       and is cumbersome to obtain otherwise */

    cpl_msg_warning(cpl_func, "TBD: calculate normalisation value");
    double norm = -1.0;
    cpl_propertylist_append_double(plist, "ESO QC FLAT NORM", norm);
    cpl_propertylist_set_comment(plist,"ESO QC FLAT NORM",
               		"Flat normalisation value");
    if(cold_bpm) {
        cpl_size ncold = cpl_mask_count(cold_bpm);
        cpl_propertylist_append_double(plist, "ESO QC NUMBER COLD PIXELS",
                                       ncold);
        cpl_propertylist_set_comment(plist,"ESO QC NUMBER COLD PIXELS",
                       		"Number of cold pixels");
    }
    enu_check_error_code("error constructing output propertylist");

    /* save the flat to a DFS-compliant MEF file */

    if (!cold_bpm) {
        mefs = enu_mef_extension_list_new(1);
        mefs->mef[0] = enu_mef_new_image("CONFIDENCE", confidence, NULL);
    } else {
        mefs = enu_mef_extension_list_new(2);
        mefs->mef[0] = enu_mef_new_mask("COLD_BPM", cold_bpm, NULL);
        mefs->mef[1] = enu_mef_new_image("CONFIDENCE", confidence, NULL);
    }
    if(qclog != NULL) {
	   cpl_propertylist_append(plist, qclog);
    }
    enu_dfs_save_himage(frameset,
                        parlist,
                        frameset,
                        CPL_TRUE,
                        flat,
                        NULL,
                        mefs,
                        recipe_name,
                        NULL,
                        plist,
                        NULL,
                        PACKAGE "/" PACKAGE_VERSION,
                        filename);

 cleanup:
    enu_mef_extension_list_delete(mefs);
    cpl_propertylist_delete(plist);
    eris_check_error_code("enu_flat_save");
    return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Get the mean airmass of an observation
  @param    plist    The observation file propertylist
  @return   The airmass

  The function returns the observation airmass. For NACO files this is read
  from keyword AIRMASS, for NIX it is the mean of keywords ESO.TEL.AIRM.START
  and ESO.TEL.AIRM.END.

  If none of these keywords is defined it will return 0 and set an error code.

  If an error is set on entry, the routine will return 0 immediately.
 */
/*----------------------------------------------------------------------------*/
double enu_get_airmass(const cpl_propertylist * plist) {

    if (cpl_error_get_code() != CPL_ERROR_NONE) return 0;

    double airmass = 0.0;
    if (cpl_propertylist_has(plist, "AIRMASS")) {
        /* NACO */
        airmass = cpl_propertylist_get_double(plist, "AIRMASS");
    } else if (cpl_propertylist_has(plist, "ESO TEL AIRM START")) {
        /* NIX */
        const double astart = cpl_propertylist_get_double(plist,
                              "ESO TEL AIRM START");
        const double aend = cpl_propertylist_get_double(plist,
                            "ESO TEL AIRM END");
        airmass = (astart + aend) / 2.0;

        /* check for failure to set AIRM keywords correctly */
        const double alt = cpl_propertylist_get_double(plist,
                           "ESO TEL ALT");
        double airmass_alt = 1.0 /cos ((90.0 - alt) * CPL_MATH_PI / 180.0);
        if (fabs(airmass - airmass_alt) > 1.0e-5) {
            cpl_msg_warning(cpl_func, "discrepency - airmass %f -> %f", airmass,
                            airmass_alt);
            airmass = airmass_alt;
        } 
    } else {
        cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND, 
                              "unable to find airmass information");
    }
    eris_check_error_code("enu_get_airmass");
    return airmass;
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Get catalogue core-radius and mesh-size appropriate to AO mode
  @param    context         input recipe context (depends on recipe name)
  @param    parlist         input recipe parameters
  @param    plist           the observation file propertylist
  @param    obj_core_radius address of double to contain value
  @param    bkg_mesh_size   address of into to contain value
  @return   CPL_ERROR_NONE if all goes well, otherwise an error code

  If parlist parameter ao-params is "auto" the routine will put values
  into core_radius and mesh_size that are appropriate to the AO mode in
  property ESO.OBS.AOMODE.

  If ao-params is "user" then the routine will read these values from
  parlist parameters.

  If an error is set on entry, the routine will return immediately.
 */
/*----------------------------------------------------------------------------*/

cpl_error_code enu_get_rcore_and_mesh_size(const char * context,
                                           const cpl_parameterlist * parlist,
                                           const cpl_propertylist * plist,
                                           double * obj_core_radius,
                                           int * bkg_mesh_size) {

    char * param_name = NULL;
    const cpl_parameter * p = NULL;
    cpl_parameter * par = NULL;

    cpl_error_code error_code = CPL_ERROR_NONE;

    if (cpl_error_get_code() != CPL_ERROR_NONE) return CPL_ERROR_NONE;
    cpl_ensure_code(context, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(parlist, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(plist, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(obj_core_radius, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(bkg_mesh_size, CPL_ERROR_NULL_INPUT);

    *obj_core_radius = -1.0;
    *bkg_mesh_size = -1;

    /* find out if to set catalogue.core-radius and mesh-size to defaults
       depending on AOMODE, or to read them from parameters */

    param_name = cpl_sprintf("%s.catalogue.ao-params", context);
    cpl_msg_info(cpl_func, "%s", param_name);
    p = cpl_parameterlist_find_const(parlist, param_name);
    const char * ao_params = cpl_parameter_get_string(p);
    cpl_free(param_name);
    enu_check_error_code("failed to read ao-params");

    if (!strcmp(ao_params, "auto") && 
        cpl_propertylist_has(plist, "ESO OBS AOMODE")) {
        const char * aomode = cpl_propertylist_get_string(plist,
                                                          "ESO OBS AOMODE");
        cpl_msg_info(cpl_func, "AOMODE: %s", aomode);
      
        if (!strcmp(aomode, "FULL_AO")) {
            *obj_core_radius = 10.0;
            *bkg_mesh_size = 64;
            if(!strstr(context,"skysub")) {

            	param_name = cpl_sprintf("%s.catalogue.obj.core-radius", context);
            	par = (cpl_parameter*) cpl_parameterlist_find_const(parlist, param_name);
            	cpl_parameter_set_double(par, *obj_core_radius);
            	cpl_free(param_name);

            }

            param_name = cpl_sprintf("%s.catalogue.bkg.mesh-size", context);
            par = (cpl_parameter*) cpl_parameterlist_find_const(parlist, param_name);
            cpl_parameter_set_int(par, *bkg_mesh_size);
            cpl_free(param_name);

        } else if (!strcmp(aomode, "NO_AO")) {
            *obj_core_radius = 25.0;
            *bkg_mesh_size = 128;
            if(!strstr(context,"skysub")) {

            	param_name = cpl_sprintf("%s.catalogue.obj.core-radius", context);
            	par = (cpl_parameter*) cpl_parameterlist_find_const(parlist, param_name);
            	cpl_parameter_set_double(par, *obj_core_radius);
            	cpl_free(param_name);

            }
            param_name = cpl_sprintf("%s.catalogue.bkg.mesh-size", context);
            par = (cpl_parameter*) cpl_parameterlist_find_const(parlist, param_name);
            cpl_parameter_set_int(par, *bkg_mesh_size);
            cpl_free(param_name);
        } else {
            error_code = CPL_ERROR_DATA_NOT_FOUND;
        }

    } else {
        if(!strstr(context,"skysub")) {
            param_name = cpl_sprintf("%s.catalogue.obj.core-radius", context);
            p = cpl_parameterlist_find_const(parlist, param_name);
            *obj_core_radius = cpl_parameter_get_double(p);
            cpl_free(param_name);
        }

        param_name = cpl_sprintf("%s.catalogue.bkg.mesh-size", context);
        p = cpl_parameterlist_find_const(parlist, param_name);
        *bkg_mesh_size = cpl_parameter_get_int(p);
        cpl_free(param_name);
    }
    cpl_msg_info(cpl_func, "catalogue AO-related params: %s %f %d",
                 ao_params, *obj_core_radius, *bkg_mesh_size);

 cleanup:
    eris_check_error_code("enu_get_rcore_and_mesh_size");
    return error_code;
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Get telescope altitude of an observation
  @param    plist    The observation file propertylist
  @return   The airmass

  The function returns the telescope altitide. 
  If the keyword is defined it will return 0 and set an error code.

  If an error is set on entry, the routine will return 0 immediately.
 */
/*----------------------------------------------------------------------------*/
double enu_get_tel_alt(const cpl_propertylist * plist) {

    if (cpl_error_get_code() != CPL_ERROR_NONE) return 0;

    double tel_alt = 0.0;
    if (cpl_propertylist_has(plist, "ESO TEL ALT")) {
        /* NACO */
        tel_alt = cpl_propertylist_get_double(plist, "ESO TEL ALT");
    } else {
        cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                              "unable to find ESO TEL ALT information");
    }
    eris_check_error_code("enu_get_tel_alt");
    return tel_alt;
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Get the DIT of an integration
  @param    plist    The observation file propertylist
  @return   The DIT

  The function returns the integration DIT. For NACO files this is read
  from keyword ESO DET DIT, for NIX from ESO DET SEQ1 DIT.

  If a problem occurs the function will return 0 and with an error code set.
  If an error is set on entry, the routine will return 0 immediately.
 */
/*----------------------------------------------------------------------------*/
double enu_get_dit(const cpl_propertylist * plist) {

    if (cpl_error_get_code() != CPL_ERROR_NONE) return 0;

    double dit = 0.0;
    if (cpl_propertylist_has(plist, "ESO DET DIT")) {
        dit = cpl_propertylist_get_double(plist, "ESO DET DIT");
    } else if (cpl_propertylist_has(plist, "ESO DET SEQ1 DIT")) {
        dit = cpl_propertylist_get_double(plist, "ESO DET SEQ1 DIT");
    } else {
      cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                            "no keyword ESO.DET.DIT or ESO.DET.SEQ1.DIT");
    }
    eris_check_error_code("enu_get_dit");
    return dit;
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Get the detector mode of an integration
  @param    plist    The observation file propertylist
  @return   The DIT

  The function returns the integration detector mode. For NACO files this is
  read from keyword ESO DET NCORRS NAME, for NIX from ESO DET READ CURNAME.

  If a problem occurs the function will return NULL and with an error code set.
  If an error is set on entry, the routine will return NULL immediately.
 */
/*----------------------------------------------------------------------------*/
const char * enu_get_det_mode(const cpl_propertylist * plist) {

    if (cpl_error_get_code() != CPL_ERROR_NONE) return NULL;

    const char * result = NULL;
    if (cpl_propertylist_has(plist, "ESO DET NCORRS NAME")) {
        result = cpl_propertylist_get_string(plist, "ESO DET NCORRS NAME");
    } else if (cpl_propertylist_has(plist, "ESO DET READ CURNAME")) {
        result = cpl_propertylist_get_string(plist, "ESO DET READ CURNAME");
    } else {
      cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                            "no keyword ESO.DET.NCORRS.NAME or "
                            "ESO.DET.READ.CURNAME");
    }
    eris_check_error_code("enu_get_det_mode");
    return result;
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Get the filter used in an observation
  @param    plist    The observation file propertylist
  @return   The filter name

  The function returns the observation filter. This is read from keyword
  FILTER, if present. If not, for NACO files it is read from keywords
  ESO.INS.OPTI4 | ESO.INS.OPTI5.NAME | ESO.INS.OPTI6.NAME, or for NIX from 
  ESO.INS2.NXFW.NAME.

  If an error occurs or the information cannot be found, then an error will
  be set , and NULL returned.

  If an error is set on entry, the routine will return NULL immediately.
 */
/*----------------------------------------------------------------------------*/
const char * enu_get_filter(const cpl_propertylist * plist) {

    if (cpl_error_get_code() != CPL_ERROR_NONE) return NULL;
    cpl_ensure(plist, CPL_ERROR_NULL_INPUT, NULL);

    const char * filter = NULL;

    if (cpl_propertylist_has(plist, "FILTER")) {
        filter = cpl_propertylist_get_string(plist, "FILTER");
    } else if (cpl_propertylist_has(plist, "INSTRUME")) {
        const char * instrume = cpl_propertylist_get_string(plist, "INSTRUME");

        if (!strcmp(instrume, "NAOS+CONICA")) {
            filter = cpl_propertylist_get_string(plist, "ESO INS OPTI4 NAME");
            cpl_msg_info(cpl_func, "OPTI4 %s", filter);
            if (!strcmp(filter, "clear") || !strcmp(filter, "empty")) {
                filter = cpl_propertylist_get_string(plist, "ESO INS OPTI5 NAME");
                cpl_msg_info(cpl_func, "OPTI5 %s", filter);
            }
            if (!strcmp(filter, "clear") || !strcmp(filter, "empty")) {
                filter = cpl_propertylist_get_string(plist, "ESO INS OPTI6 NAME");
                cpl_msg_info(cpl_func, "OPTI6 %s", filter);
            }
        } else if (!strcmp(instrume, "ERIS")) {
            filter = cpl_propertylist_get_string(plist, "ESO INS2 NXFW NAME");
        } else {
            cpl_error_set(cpl_func, CPL_ERROR_INCOMPATIBLE_INPUT);
        }
    } else {
        filter = "unknown";
    }
    eris_check_error_code("enu_get_filter");
    return filter;
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Get the effective wavelength of a filter
  @param    filter    The name of the filter
  @return   The filter wavelength

  The function returns the wavelength of the filter.

  If an error occurs or the information cannot be found, then an error will
  be set , and 1.0 returned.

  If an error is set on entry, the routine will return 1.0 immediately.
 */
/*----------------------------------------------------------------------------*/

double enu_get_filter_wavelength(const char * filter) {

    if (cpl_error_get_code() != CPL_ERROR_NONE) return -1.0;
    cpl_ensure(filter, CPL_ERROR_NULL_INPUT, -1.0);

    double lambda = 1.0;
    if (!strcmp(filter, "J")) {
        lambda = 1.28;
    } else if (!strcmp(filter, "H")) {
        lambda = 1.66;
    } else if (!strcmp(filter, "Ks")) {
        lambda = 2.18;
    } else if (!strcmp(filter, "Short-Lp")) {
        lambda = 3.32;
    } else if (!strcmp(filter, "L-Broad")) {
        lambda = 3.57;
    } else if (!strcmp(filter, "Lp")) {
        lambda = 3.79;
    } else if (!strcmp(filter, "Mp")) {
        lambda = 4.78;
    } else if (!strcmp(filter, "Pa-b")) {
        lambda = 1.282;
    } else if (!strcmp(filter, "Fe-II")) {
        lambda = 1.644;
    } else if (!strcmp(filter, "H2-cont")) {
        lambda = 2.068;
    } else if (!strcmp(filter, "H2-1-0S")) {
        lambda = 2.120;
    } else if (!strcmp(filter, "Br-g")) {
        lambda = 2.172;
    } else if (!strcmp(filter, "K-peak")) {
        lambda = 2.198;
    } else if (!strcmp(filter, "IB-2.42")) {
        lambda = 2.420;
    } else if (!strcmp(filter, "IB-2.48")) {
        lambda = 2.479;
    } else if (!strcmp(filter, "Br-a-cont")) {
        lambda = 3.965;
    } else if (!strcmp(filter, "Br-a")) {
        lambda = 4.051;
    } else {
        cpl_msg_warning(cpl_func, "filter %s not recognized", filter);
    }
    eris_check_error_code("enu_get_filter_wavelength");
    return lambda;
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Get the pipeline copyright and license
  @return   The copyright and license string

  The function returns a pointer to the statically allocated license string.
  This string should not be modified using the returned pointer.
 */
/*----------------------------------------------------------------------------*/
const char * enu_get_license(void) {

    const char * eris_nix_license = 
        "This file is part of the ERIS/NIX Instrument Pipeline\n"
        "Copyright (C) 2017 European Southern Observatory\n"
        "\n"
        "This program is free software; you can redistribute it and/or modify\n"
        "it under the terms of the GNU General Public License as published by\n"
        "the Free Software Foundation; either version 2 of the License, or\n"
        "(at your option) any later version.\n"
        "\n"
        "This program is distributed in the hope that it will be useful,\n"
        "but WITHOUT ANY WARRANTY; without even the implied warranty of\n"
        "MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the\n"
        "GNU General Public License for more details.\n"
        "\n"
        "You should have received a copy of the GNU General Public License\n"
        "along with this program; if not, write to the Free Software\n"
        "Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, \n"
        "MA  02110-1301  USA";
    return eris_nix_license;
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Get RA and Dec at centre of image with given wcs
  @return   A CPL error code, CPL_ERROR_NONE if all goes well

  The function returns the RA and Dec at the axis mid-point of the given wcs.
 */
/*----------------------------------------------------------------------------*/

cpl_error_code enu_get_ra_dec(const cpl_wcs * wcs, double * ra, double * dec) {

    if (cpl_error_get_code() != CPL_ERROR_NONE) return cpl_error_get_code(); 
    cpl_ensure(wcs, CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT);
    cpl_ensure(ra, CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT);
    cpl_ensure(dec, CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT);

    /* get the axis centre */

    const cpl_array * dims = cpl_wcs_get_image_dims(wcs);
    cpl_matrix * from = cpl_matrix_new(1, 2);
    int ignore = 0;
    double dim = cpl_array_get(dims, 0, &ignore);
    cpl_matrix_set(from, 0, 0, dim / 2.0);
    dim = cpl_array_get(dims, 1, &ignore);
    cpl_matrix_set(from, 0, 1, dim / 2.0);

    /* convert to world */

    cpl_matrix * to = NULL;
    cpl_array * status = NULL;
    cpl_wcs_convert(wcs, from, &to, &status, CPL_WCS_PHYS2WORLD);

    *ra = cpl_matrix_get(to, 0, 0);
    *dec = cpl_matrix_get(to, 0, 1);

    /* tidy up */

    cpl_matrix_delete(from);
    cpl_matrix_delete(to);
    cpl_array_delete(status);
    eris_check_error_code("enu_get_ra_dec");
    return cpl_error_get_code();
}

#define MJD_OBJ_REF_WINDOW_STRY_POS_FAST_UNCORR 61014.7
/*----------------------------------------------------------------------------*/
/**
  @brief    Get the detector 'window' information
  @param    nx       Pointer to variable for window nx
  @param    ny       Pointer to variable for window ny
  @param    rot      Pointer to variable for window rotation
  @param    strx     Pointer to variable for window x start index
  @param    stry     Pointer to variable for window y start index
  @param    nx_chip  Pointer to variable for chip nx
  @param    ny_chip  Pointer to variable for chip ny
  @param    windowed CPL_FALSE if whole chip used, CPL_TRUE if windowed
  @param    plist    The observation file propertylist
  @return   A CPL error code, CPL_ERROR_NONE if all goes well

  The function fills variables with information on the detector windowing used,
  read from the given parameter list. Works for both ERIS and NACO-style
  keywords.
 */
/*----------------------------------------------------------------------------*/
cpl_error_code enu_get_window_info(cpl_size * nx, cpl_size * ny, int * rot,
                                   cpl_size * strx, cpl_size * stry,
                                   cpl_size * nx_chip, cpl_size * ny_chip,
                                   cpl_boolean * windowed,
                                   const cpl_propertylist * plist) {

    if (cpl_error_get_code() != CPL_ERROR_NONE) return cpl_error_get_code(); 

    cpl_ensure_code(plist, CPL_ERROR_NULL_INPUT);
    enu_check(cpl_propertylist_has(plist, "INSTRUME") == 1,
              CPL_ERROR_DATA_NOT_FOUND,
              "propertylist does not contain INSTRUME keyword");

    /* get window information */

    const char * instrume = cpl_propertylist_get_string(plist, "INSTRUME");

    cpl_size nxw = 0;
    cpl_size nyw = 0;
    int rotw = 0;
    cpl_size strxw = 0;
    cpl_size stryw = 0;

    if (!strcmp(instrume, "ERIS")) {
        nxw = cpl_propertylist_get_int(plist, "ESO DET SEQ1 WIN NX");
        nyw = cpl_propertylist_get_int(plist, "ESO DET SEQ1 WIN NY");
        rotw = cpl_propertylist_get_int(plist, "ESO DET SEQ1 WIN ROT");
        strxw = cpl_propertylist_get_int(plist, "ESO DET SEQ1 WIN STRX");
        stryw = cpl_propertylist_get_int(plist, "ESO DET SEQ1 WIN STRY");
    } else {
        /* for NACO */
        /* STARTX,Y can be stored as double or int */
        nxw = cpl_propertylist_get_int(plist, "ESO DET WIN NX");
        nyw = cpl_propertylist_get_int(plist, "ESO DET WIN NY");
        if (cpl_propertylist_get_type(plist, "ESO DET WIN STARTX") == 
            CPL_TYPE_DOUBLE) {
            strxw = (int) cpl_propertylist_get_double(plist,
                                       "ESO DET WIN STARTX");
            stryw = (int) cpl_propertylist_get_double(plist,
                                       "ESO DET WIN STARTY");
        } else {
            strxw = cpl_propertylist_get_int(plist, "ESO DET WIN STARTX");
            stryw = cpl_propertylist_get_int(plist, "ESO DET WIN STARTY");
        }
    }

    cpl_size nx_chipw = 2048;
    cpl_size ny_chipw = 2048;

    if(cpl_propertylist_has(plist, "ESO DET CHIP NX")) {
        nx_chipw = cpl_propertylist_get_int(plist,"ESO DET CHIP NX");
    } else {
        nx_chipw = cpl_propertylist_get_int(plist,"ESO DET CHIP1 NX");
    }

    if(cpl_propertylist_has(plist, "ESO DET CHIP NY")) {
        ny_chipw = cpl_propertylist_get_int(plist,"ESO DET CHIP NY");
    } else {
        ny_chipw = cpl_propertylist_get_int(plist,"ESO DET CHIP1 NY");
    }

    enu_check_error_code("failed to read detector mode information");
    enu_check(rotw == 0, CPL_ERROR_UNSUPPORTED_MODE,
              "detector window rot must be 0");

    cpl_boolean windowedw = (rotw != 0) || 
                            (strxw != 1) ||
                            (stryw != 1) ||
                            (nxw != nx_chipw) ||
                            (nyw != ny_chipw);
    cpl_msg_info(cpl_func, "flatfield rot=%d strx=%d stry=%d nx=%d ny=%d windowedw: %d",
                 (int)rotw, (int)strxw, (int)stryw, (int)nxw, (int)nyw, windowedw);

    if (!strcmp(instrume, "ERIS")) {
    	/* Handle special case for windowed mode and FAST_UNCORR */
    	double mjd_obs = MJD_OBJ_REF_WINDOW_STRY_POS_FAST_UNCORR;
    	if(cpl_propertylist_has(plist,"ESO DET SEQ1 WIN NAME")) {
    		mjd_obs = cpl_propertylist_get_double(plist,"MJD-OBS");
    	}
    	
    	const char * det_mode = "SLOW_GR_UTR";
    	if(cpl_propertylist_has(plist,"ESO DET READ CURNAME")) {
    		det_mode = enu_get_det_mode(plist);
    	}
    	
    	const char * win_name = "windowF";
    	if(cpl_propertylist_has(plist,"ESO DET SEQ1 WIN NAME")) {
    		win_name = cpl_propertylist_get_string(plist,"ESO DET SEQ1 WIN NAME");
    	}
    	
    	if ( windowedw && (strcmp(win_name, "windowF") != 0) &&
    			(strcmp(det_mode, "FAST_UNCORR") == 0) && ((stryw % 2) == 0) && (mjd_obs < MJD_OBJ_REF_WINDOW_STRY_POS_FAST_UNCORR) ){
    		stryw +=1;
    		cpl_msg_warning(cpl_func,"windowing special case windowedw: %d win_name: %s", windowedw, win_name);
    	} else {
    		cpl_msg_warning(cpl_func,"windowing normal case windowedw: %d win_name: %s", windowedw, win_name);
    	}
    	
    }
    
    /* only now copy to external variables in case these are
       all the same - which would have screwed up the logic above */

    *nx = nxw;
    *ny = nyw;
    *rot = rotw;
    *strx = strxw;
    *stry = stryw;
    *nx_chip = nx_chipw;
    *ny_chip = ny_chipw;
    *windowed = windowedw;

 cleanup:
    eris_check_error_code("enu_get_window_info");
    return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
 * @brief  Return a deep copy of the given hdrl_catalogue_result object
 * @param    target  The structure to be copied
 * @return  A copy of the target hdrl_catalogue_result
 */
/*----------------------------------------------------------------------------*/
hdrl_catalogue_result * enu_hdrl_catalogue_result_duplicate(
  const hdrl_catalogue_result * target) {
    cpl_ensure(target, CPL_ERROR_NULL_INPUT, NULL);

    hdrl_catalogue_result * result = cpl_malloc(sizeof(hdrl_catalogue_result));

    result->catalogue = cpl_table_duplicate(target->catalogue);
    result->background = cpl_image_duplicate(target->background);
    result->segmentation_map = cpl_image_duplicate(target->segmentation_map);
    result->qclist = cpl_propertylist_duplicate(target->qclist);

    return result;
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Load an hdrl_image from a multi-extension FITS file
  @param    filename        The name of the file to be loaded
  @param    result          The loaded himage
  @param    mef_extensions  A list with data from any excess FITS extensions found
  @param    plist           The property list from the primary FITS HDU
  @return   CPL_ERROR_NONE if all goes well, otherwise an error code

  The function loads an hdrl_image from a multi-extension FITS file.
 */
/*----------------------------------------------------------------------------*/

cpl_error_code enu_himage_load_from_fits(const char * filename,
                                         hdrl_image ** result,
                                         mef_extension_list ** mef_extensions,
                                         cpl_propertylist ** plist) {

    if (cpl_error_get_code() != CPL_ERROR_NONE) return cpl_error_get_code();

    cpl_propertylist * plist1 = NULL; 
    cpl_propertylist * plist2 = NULL; 
    cpl_propertylist * plist3 = NULL;
    cpl_image        * data = NULL;
    cpl_image        * error = NULL;
    cpl_mask         * bpm = NULL;
    cpl_mask         * old_bpm = NULL;

    /* check parameters */

    cpl_ensure_code(filename, CPL_ERROR_NULL_INPUT);

    /* check FITS files has enough extensions */

    cpl_size size = cpl_fits_count_extensions(filename);
    cpl_ensure_code(size >= 3, CPL_ERROR_BAD_FILE_FORMAT); 

    /* Get main propertylist from HDU 0 in the file */

    *plist = cpl_propertylist_load(filename, 0);
    enu_check_error_code("accessing HDU 0 header");

    /* Access HDU 1 in the file, this should contain the data */

    plist1 = cpl_propertylist_load(filename, 1);
    enu_check_error_code("accessing HDU 1 header");

    const char * extname = cpl_propertylist_get_string(plist1, "EXTNAME");
    const char * hduclass = cpl_propertylist_get_string(plist1, "HDUCLASS");
    const char * hdudoc = cpl_propertylist_get_string(plist1, "HDUDOC");
    //const char * hduvers = cpl_propertylist_get_string(plist1, "HDUVERS");
    const char * hduclas1 = cpl_propertylist_get_string(plist1, "HDUCLAS1");
    const char * hduclas2 = cpl_propertylist_get_string(plist1, "HDUCLAS2");
    enu_check_error_code("accessing HDU 1 propertylist");
    enu_check(strstr(extname, "DATA") != NULL &&
              strstr(hduclass, "ESO") != NULL &&
              strstr(hdudoc, "SDP") != NULL &&
              strstr(hduclas1, "IMAGE") != NULL &&
              strstr(hduclas2, "DATA") != NULL,
              CPL_ERROR_BAD_FILE_FORMAT, "bad HD1 format");
    data = cpl_image_load(filename, HDRL_TYPE_DATA, 0, 1);
    enu_check_error_code("accessing HDU 1 data");

    /* the wcs items are stored with the image planes, merge
       them back into plist  - order is import for the NAXIS*
       keywords so the properties are inserted after NAXIS
       in reverse order*/

    const char * wcs_items = "NAXIS WCSAXES CRPIX1 CRPIX2 "
                             "CD1_1 CD1_2 CD2_1 CD2_2 CUNIT1 CUNIT2 "
                             "CTYPE1 CTYPE2 CRVAL1 CRVAL2 LONPOLE "
                             "LATPOLE CSYER1 CSYER2 EQUINOX MJD-OBS "
                             "DATE-OBS NAXIS2 NAXIS1";
    cpl_size plist1_size = cpl_propertylist_get_size(plist1);
    for (cpl_size ip=plist1_size; ip>0; ip--) {
        const cpl_property * newprop = cpl_propertylist_get_const(plist1, ip-1);
        const char * name = cpl_property_get_name(newprop);
        if (strstr(wcs_items, name) != NULL) {
            if (cpl_propertylist_has(*plist, name)) {
                cpl_propertylist_copy_property(*plist, plist1, name);
            } else {
                cpl_propertylist_insert_after_property(*plist, "NAXIS",
                                                       newprop);
            } 
        }
    }

    /* Access HDU 2 in the file, this should contain the data error */

    plist2 = cpl_propertylist_load(filename, 2);
    enu_check_error_code("accessing HDU 2");
    extname = cpl_propertylist_get_string(plist2, "EXTNAME");
    hduclass = cpl_propertylist_get_string(plist2, "HDUCLASS");
    hdudoc = cpl_propertylist_get_string(plist2, "HDUDOC");
    //hduvers = cpl_propertylist_get_string(plist2, "HDUVERS");
    hduclas1 = cpl_propertylist_get_string(plist2, "HDUCLAS1");
    hduclas2 = cpl_propertylist_get_string(plist2, "HDUCLAS2");
    enu_check_error_code("accessing HDU 2 propertylist");
    enu_check(strstr(extname, "ERR") != NULL &&
              strstr(hduclass, "ESO") != NULL &&
              strstr(hdudoc, "SDP") != NULL &&
              strstr(hduclas1, "IMAGE") != NULL &&
              strstr(hduclas2, "ERROR") != NULL,
              CPL_ERROR_BAD_FILE_FORMAT, "bad HD2 format");
    error = cpl_image_load(filename, HDRL_TYPE_ERROR, 0, 2);
    enu_check_error_code("accessing HDU 1 data error");

    /* Access HDU 3 in the file, this should contain the data quality */

    plist3 = cpl_propertylist_load(filename, 3);
    enu_check_error_code("accessing HDU 3");
    extname = cpl_propertylist_get_string(plist3, "EXTNAME");
    hduclass = cpl_propertylist_get_string(plist3, "HDUCLASS");
    hdudoc = cpl_propertylist_get_string(plist3, "HDUDOC");
    //hduvers = cpl_propertylist_get_string(plist3, "HDUVERS");
    hduclas1 = cpl_propertylist_get_string(plist3, "HDUCLAS1");
    hduclas2 = cpl_propertylist_get_string(plist3, "HDUCLAS2");
    enu_check_error_code("accessing HDU 3 propertylist");
    enu_check(strstr(extname, "DQ") != NULL &&
              strstr(hduclass, "ESO") != NULL &&
              strstr(hdudoc, "SDP") != NULL &&
              strstr(hduclas1, "IMAGE") != NULL &&
              strstr(hduclas2, "QUALITY") != NULL,
              CPL_ERROR_BAD_FILE_FORMAT, "bad HD3 format");

    /* attach the mask to the data, construct the hdrl image */

    bpm = cpl_mask_load(filename, 0, 3);
    enu_check_error_code("accessing HDU 3 data mask");
    old_bpm = cpl_image_set_bpm(data, bpm);
    cpl_mask_delete(old_bpm); old_bpm = NULL;
    *result = hdrl_image_create(data, error);

    /* Read any further extensions into a mef_extension_list */

    int n_ext = size - 3;
    *mef_extensions = enu_mef_extension_list_new(n_ext);
    for (int ext=0; ext<n_ext; ext++) {

        /* get the extension name... */

        cpl_propertylist * ext_plist = cpl_propertylist_load(filename, ext+4);
        const char * name = cpl_propertylist_get_string(ext_plist, "EXTNAME");
        const char * mef_type = cpl_propertylist_get_string(ext_plist, 
                                "ERIS_NIX_MEF_TYPE");

        /* ...and value, type keyed on extension name */

        if (!strcmp(mef_type, MEF_EXTENSION_CONTAINING_MASK)) {
            cpl_mask * mask = cpl_mask_load(filename, 0, ext+4);
            mef_extension * new = enu_mef_new_mask(name, mask, NULL);
            (*mef_extensions)->mef[ext] = new;
            cpl_mask_delete(mask);
        } else if (!strcmp(mef_type, MEF_EXTENSION_CONTAINING_IMAGE)) {
            cpl_image * image = cpl_image_load(filename, HDRL_TYPE_DATA, 0,
              ext+4);
            mef_extension * new = enu_mef_new_image(name, image, NULL);
            (*mef_extensions)->mef[ext] = new;
            cpl_image_delete(image);
        } else if (!strcmp(mef_type, MEF_EXTENSION_CONTAINING_TABLE)) {
            cpl_table * table = cpl_table_load(filename, ext+4, 0);
            mef_extension * new = enu_mef_new_table(name, table, ext_plist);
            (*mef_extensions)->mef[ext] = new;
            cpl_table_delete(table);
        } else {
            cpl_error_set_message("enu_himage_load_from_fits",
              CPL_ERROR_UNSUPPORTED_MODE,
              "unsupported extension name: %s", name); 
        }
 
        cpl_propertylist_delete(ext_plist);
    }

 cleanup:
    cpl_propertylist_delete(plist1);
    cpl_propertylist_delete(plist2);
    cpl_propertylist_delete(plist3);
    cpl_image_delete(data);
    cpl_image_delete(error);
    cpl_mask_delete(old_bpm);
    if (cpl_error_get_code() != CPL_ERROR_NONE) {
        cpl_propertylist_delete(*plist);
        *plist = NULL;
        hdrl_image_delete(*result);
        *result = NULL;
        enu_mef_extension_list_delete(*mef_extensions);
        *mef_extensions = NULL;
    }
    eris_check_error_code("enu_himage_load_from_fits");
    return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Load components of a located_image from a frame
  @param    frame      The frame containing the FITS file
  @param    pcopyconf  A confidence image to use and reset, or NULL
  @param    collapse_cube user switch to collapse the input data cube format
  @return   located_image containing the result

  The function loads a located_image structure from the components of the
  FITS file referenced in the given frame.

  If the input confidence image is provided and has the right size,
  then it is inserted into the created object and set to NULL on return.

 */
/*----------------------------------------------------------------------------*/

located_image * enu_load_limage_from_frame(const cpl_frame * frame,
                                           cpl_image ** pcopyconf,
                                           const cpl_boolean collapse_cube) {

    if (cpl_error_get_code() != CPL_ERROR_NONE) return NULL;

    cpl_ensure(frame, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(cpl_frame_get_filename(frame), CPL_ERROR_NULL_INPUT, NULL);

    located_image * result = NULL;
    mef_extension_list * mefs = NULL;
    cpl_image * data = NULL;
    cpl_image * err = NULL;
    cpl_image * qual = NULL;
    cpl_image * conf = NULL;
    cpl_image * bkg = NULL;
    cpl_image * bkg_err = NULL;
    cpl_image * bkg_conf = NULL;

    const char * filename = cpl_frame_get_filename(frame);
    cpl_msg_info(cpl_func, "..reading %s", filename);

    /* treatment depends on number of extensions, name of first extension, and
       ESO DET FRAM FORMAT (single image or cube).
       Get this information */

    cpl_size next = cpl_fits_count_extensions(filename);

    cpl_propertylist * plist = cpl_propertylist_load(filename, 0);

    /* is this an ERIS file or NACO */
    const char * instrume = cpl_propertylist_get_string(plist, "INSTRUME");
    const cpl_boolean eris = !strcmp(instrume, "ERIS");

    char * first_ext = NULL;
    if (eris && (next > 0)) {
        cpl_propertylist * plist1 = cpl_propertylist_load(filename, 1);
        if (cpl_propertylist_has(plist1, "EXTNAME")) {
        	first_ext = cpl_sprintf("%s", cpl_propertylist_get_string(plist1, "EXTNAME"));
        }
        cpl_propertylist_delete(plist1);
        //cpl_msg_info(cpl_func, "extname %s", first_ext);
    }

    char * frame_format = NULL;
    if (cpl_propertylist_has(plist, "ESO DET FRAM FORMAT")) {
        frame_format = cpl_sprintf("%s", cpl_propertylist_get_string(plist, "ESO DET FRAM FORMAT"));
        cpl_msg_info(cpl_func, "..format %s", frame_format);
    }
                    
    /* Now read the file appropriately */

    if (!eris) {
        cpl_msg_info(cpl_func, "NACO data");

        /* It is a raw FITS file with header and 2d image data, 
           plus extensions to be ignored */

        data = cpl_image_load(filename, HDRL_TYPE_DATA, 0, 0);
        enu_check_error_code("failed to read file: %s", filename);
        hdrl_image * himage = hdrl_image_create(data, NULL);
        cpl_image_delete(data); data = NULL;
        const cpl_size nx = hdrl_image_get_size_x(himage);
        const cpl_size ny = hdrl_image_get_size_y(himage);

        /* confidence set to 100 everywhere */

        cpl_image * confidence;
        if (pcopyconf != NULL && *pcopyconf != NULL &&
            cpl_image_get_size_x(*pcopyconf) == nx &&
            cpl_image_get_size_y(*pcopyconf) == ny) {
            confidence = *pcopyconf;
            *pcopyconf = NULL;
        } else {
            confidence = cpl_image_new(nx, ny, HDRL_TYPE_DATA);
            cpl_image_fill_window(confidence, 1, 1, nx, ny, 100.0); 
        }

        cpl_frame_dump(frame, NULL);
        result = enu_located_image_new(himage,
                                       NULL,
                                       confidence,
                                       NULL,
                                       NULL,
                                       plist,
                                       NULL,
                                       NULL,
                                       NULL,
                                       NULL,
                                       cpl_frame_duplicate(frame));
        plist = NULL;

    } else if ((next == 0) ||
               (next == 1 &&
               frame_format &&
               !strcmp(first_ext, "ASM_DATA"))) {

        if (next == 1)  {

            /* read IA_FWHM from the ASM table and store it in the
               FITS header */

            cpl_table * asm_table = cpl_table_load(filename, 1, CPL_TRUE);
            cpl_size nrow = cpl_table_get_nrow(asm_table);
            if(cpl_table_has_column(asm_table,"IA_FWHM")) {
            	double * ia_fwhm_data = cpl_table_get_data_double(asm_table,
            			"IA_FWHM");
            	cpl_vector * ia_fwhm = cpl_vector_wrap(nrow, ia_fwhm_data);
            	double ia_fwhm_mean = cpl_vector_get_mean(ia_fwhm);

            	const char * filter = enu_get_filter(plist);
            	double lambda = enu_get_filter_wavelength(filter);

            	/* correct the FWHM for wavelength, check PIPE-10981 docs for
               explanation of formula */

            	double ia_fwhm_corr = ia_fwhm_mean * pow(0.5/lambda, 1./5.);

            	cpl_propertylist_update_double(plist, "PSF_FWHM", ia_fwhm_corr);
            	cpl_propertylist_set_comment(plist, "PSF_FWHM", "PSF fwhm derived "
            			"from AO ASM table [arcsec]");

            	cpl_vector_unwrap(ia_fwhm);
            } else {
            	cpl_msg_warning(cpl_func,"ASM table miss IA_FWHM column. PSF_FWHM not computed");
            }
            cpl_table_delete(asm_table);

        } else {

            /* NO ASM table is present, AO was not used. Estimate the
               fwhm from DIMM instead. Again, see PIPE-10981 for background */

            if (cpl_propertylist_has(plist, "ESO TEL AMBI FWHM START") &&
                cpl_propertylist_has(plist, "ESO TEL AMBI FWHM END") &&
                cpl_propertylist_has(plist, "ESO TEL AIRM START") &&
                cpl_propertylist_has(plist, "ESO TEL AIRM END")) {

                double dimm1 = cpl_propertylist_get_double(plist,
                                       "ESO TEL AMBI FWHM START");
                double dimm2 = cpl_propertylist_get_double(plist,
                                       "ESO TEL AMBI FWHM END");
                double airmass1 = cpl_propertylist_get_double(plist,
                                       "ESO TEL AIRM START");
                double airmass2 = cpl_propertylist_get_double(plist,
                                       "ESO TEL AIRM END");

                const char * filter = enu_get_filter(plist);
                double lambda = enu_get_filter_wavelength(filter);

                /* correct DIMM values for wavelength, then average */

                dimm1 = dimm1 * 
                            pow(0.5/lambda, 1./5.) * 
                            pow(airmass1, 3./5.) * 
                            (1. - 78.08 * 
                                (pow(lambda/1.e6, 2./5.) * 
                                 pow(airmass1, -1./5.) / 
                                 pow(dimm1, 1./3.)
                                )
                            );
                dimm2 = dimm2 * 
                            pow(0.5/lambda, 1./5.) * 
                            pow(airmass2, 3./5.) * 
                            (1. - 78.08 * 
                                (pow(lambda/1.e6, 2./5.) * 
                                 pow(airmass2, -1./5.) / 
                                 pow(dimm2, 1./3.)
                                )
                            );

                cpl_propertylist_update_double(plist, "PSF_FWHM",
                                               (dimm1 + dimm2) / 2.0);
                cpl_propertylist_set_comment(plist, "PSF_FWHM",
                                             "PSF fwhm derived "
                                             "from DIMM [arcsec]");
            }
        }

        if (!strcmp(frame_format, "single")) {

            /* It is a raw FITS file with header and 2d image data */

            data = cpl_image_load(filename, HDRL_TYPE_DATA, 0, 0);
            enu_check_error_code("failed to read file: %s", filename);
            hdrl_image * himage = hdrl_image_create(data, NULL);
            cpl_image_delete(data); data = NULL;
            const cpl_size nx = hdrl_image_get_size_x(himage);
            const cpl_size ny = hdrl_image_get_size_y(himage);

            /* confidence set to 100 everywhere */

            cpl_image * confidence;
            if (pcopyconf != NULL && *pcopyconf != NULL &&
                cpl_image_get_size_x(*pcopyconf) == nx &&
                cpl_image_get_size_y(*pcopyconf) == ny) {
                confidence = *pcopyconf;
                *pcopyconf = NULL;
            } else {
                confidence = cpl_image_new(nx, ny, HDRL_TYPE_DATA);
                cpl_image_fill_window(confidence, 1, 1, nx, ny, 100.0); 
            }

            result = enu_located_image_new(himage,
                                           NULL,
                                           confidence,
                                           NULL,
                                           NULL,
                                           plist,
                                           NULL,
                                           NULL,
                                           NULL,
                                           NULL,
                                           cpl_frame_duplicate(frame));
            plist = NULL;

        } else if (!strcmp(frame_format, "cube")) {

            /* It is a raw FITS file with header and 3d image data (cube) */

            cpl_imagelist * datalist = cpl_imagelist_load(filename,
                                       HDRL_TYPE_DATA, 0);
            enu_check_error_code("failed to read file: %s", filename);
            hdrl_imagelist * himagelist = hdrl_imagelist_create(datalist,
                                                                NULL);
            cpl_imagelist_delete(datalist); datalist = NULL;
            const cpl_size nx = hdrl_imagelist_get_size_x(himagelist);
            const cpl_size ny = hdrl_imagelist_get_size_y(himagelist);

            /* confidence set to 100 everywhere */

            cpl_image * confidence;
            if (pcopyconf != NULL && *pcopyconf != NULL &&
                cpl_image_get_size_x(*pcopyconf) == nx &&
                cpl_image_get_size_y(*pcopyconf) == ny) {
                confidence = *pcopyconf;
                *pcopyconf = NULL;
            } else {
                confidence = cpl_image_new(nx, ny, HDRL_TYPE_DATA);
                cpl_image_fill_window(confidence, 1, 1, nx, ny, 100.0); 
            }

            if(collapse_cube) {
                hdrl_image * himage = NULL;
                cpl_image* conf_ima = NULL;
                if(collapse_cube == 1) {
                    hdrl_imagelist_collapse_mean(himagelist,&himage,&conf_ima);
                } else if(collapse_cube == 2) {
                    hdrl_imagelist_collapse_median(himagelist,&himage,&conf_ima);
                } else if(collapse_cube == 3) {
                    hdrl_imagelist_collapse_weighted_mean(himagelist,&himage,&conf_ima);
                }
                cpl_image_delete(conf_ima);
                result = enu_located_image_new(himage,
                                               NULL,
                                               confidence,
                                               NULL,
                                               NULL,
                                               plist,
                                               NULL,
                                               NULL,
                                               NULL,
                                               NULL,
                                               cpl_frame_duplicate(frame));
            } else {
                result = enu_located_image_new(NULL,
                                               himagelist,
                                               confidence,
                                               NULL,
                                               NULL,
                                               plist,
                                               NULL,
                                               NULL,
                                               NULL,
                                               NULL,
                                               cpl_frame_duplicate(frame));
            }
            plist = NULL;
        }
    } else {

        /* It is a mef: it should be in ESO Data Products Standard format */

        cpl_propertylist_delete(plist);
        plist = NULL;
        mefs = enu_load_mef_components(filename, &plist);
        enu_check_error_code("Unable to load components for file");

        /* look in first extension for HDU keywords */

        const char * hduclass = cpl_propertylist_get_string(
                                mefs->mef[0]->plist, "HDUCLASS");
        const char * hdudoc = cpl_propertylist_get_string(
                              mefs->mef[0]->plist, "HDUDOC");
        const char * hduvers = cpl_propertylist_get_string(
                               mefs->mef[0]->plist, "HDUVERS");
        enu_check_error_code("Unable to read ESO DPS keywords from header");

        enu_check(strstr(hduclass, "ESO") != NULL &&
                  strstr(hdudoc, "SDP") != NULL &&
                  strstr(hduvers, "SDP version") != NULL,
                  CPL_ERROR_BAD_FILE_FORMAT, "file not in ESO DPS format");

        /* Read the names of the extensions holding standard data
           components */ 

        const char * scidata = NULL;
        if (mefs->size > 0 &&
            cpl_propertylist_has(mefs->mef[1]->plist, "SCIDATA")){
            scidata = cpl_propertylist_get_string(mefs->mef[1]->plist,
                                                  "SCIDATA");
        }
        const char * errdata = NULL;
        if (cpl_propertylist_has(mefs->mef[0]->plist, "ERRDATA")){
            errdata = cpl_propertylist_get_string(mefs->mef[0]->plist,
                                                  "ERRDATA");
        }
        const char * qualdata = NULL;
        if (cpl_propertylist_has(mefs->mef[0]->plist, "QUALDATA")){
            qualdata = cpl_propertylist_get_string(mefs->mef[0]->plist,
                                                   "QUALDATA");
        }
        const char * confdata = NULL;
        if (cpl_propertylist_has(mefs->mef[0]->plist, "CONFDATA")){
            confdata = cpl_propertylist_get_string(mefs->mef[0]->plist,
                                                   "CONFDATA");
        }
        const char * bkgdata = NULL;
        if (cpl_propertylist_has(mefs->mef[0]->plist, "BKGDATA")){
            bkgdata = cpl_propertylist_get_string(mefs->mef[0]->plist,
                                                  "BKGDATA");
        }
        const char * bkgerr = NULL;
        if (cpl_propertylist_has(mefs->mef[0]->plist, "BKGERR")){
            bkgerr = cpl_propertylist_get_string(mefs->mef[0]->plist,
                                                 "BKGERR");
        }
        const char * bkgconf = NULL;
        if (cpl_propertylist_has(mefs->mef[0]->plist, "BKGCONF")){
            bkgconf = cpl_propertylist_get_string(mefs->mef[0]->plist,
                                                  "BKGCONF");
        }
        enu_check_error_code("accessing DPS standard extensions");

        /* now loop through extensions filling standard data elements */

        for (cpl_size i=0; i<next; i++) {
            const char * extname = cpl_propertylist_get_string(
                                   mefs->mef[i]->plist, "EXTNAME");

            /* the returned plist is a merge of the plist for the
               science image and the header plist.*/

            if (scidata && !strcmp(scidata, extname)) {
                data = enu_load_component(mefs->mef[i], "SCIDATA", 
                                          "IMAGE", "DATA");
                enu_check_error_code("accessing SCIDATA extension");

                /* Start with the science image plist (for the data and 
                   wcs keywords)*/

                cpl_propertylist * primary_plist = plist;

                /* remove the SDP7 keywords which were appropriate to
                   the science data extension but will
                   confuse the enu_limage_save routine if left in the
                   main header */

                plist = cpl_propertylist_duplicate(mefs->mef[i]->plist);
                const char * sdp7_items = " EXTNAME HDUCLAS1 HDUCLAS2"
                                          " HDUCLAS3 ";
                cpl_size psize = cpl_propertylist_get_size(plist);
                for (cpl_size ip = psize; ip > 0; ip--) {
                    const cpl_property * prop = cpl_propertylist_get_const(
                                                plist, ip-1);
                    const char * name = cpl_property_get_name(prop);

                    /* pad the keyword name with spaces which will
                       stop strstr misidentifying a keyword that
                       matches _part_ of an SDP7 keyword */

                    char * padded_name = cpl_sprintf(" %s ", name);
                    if (strstr(sdp7_items, padded_name) != NULL) {
                        cpl_propertylist_erase(plist, name);
                    }
                    cpl_free(padded_name);
                }

                /* merge info from the primary header */

                psize = cpl_propertylist_get_size(primary_plist);
                for (cpl_size ip = psize; ip > 0; ip--) {
                    const cpl_property * prop = cpl_propertylist_get_const(
                                                primary_plist, ip-1);
                    const char * name = cpl_property_get_name(prop);
                    if (!cpl_propertylist_has(plist, name)) {
                        cpl_propertylist_copy_property(plist, primary_plist,
                                                       name);
                        }
                    }
                cpl_propertylist_delete(primary_plist);

            } else if (errdata && !strcmp(errdata, extname)) {
                err = enu_load_component(mefs->mef[i], "SCIERR", 
                                         "IMAGE", "ERROR");
                enu_check_error_code("accessing SCIERR extension");
            } else if (qualdata && !strcmp(qualdata, extname)) {
                qual = enu_load_component(mefs->mef[i], "QUALDATA", 
                                          "IMAGE", "QUALITY");
                enu_check_error_code("accessing QUALDATA extension");
            } else if (confdata && !strcmp(confdata, extname)) {
                conf = enu_load_component(mefs->mef[i], "CONFDATA", 
                                          "IMAGE", "CONF");
                enu_check_error_code("accessing CONFDATA extension");
            } else if (bkgdata && !strcmp(bkgdata, extname)) {
                bkg = enu_load_component(mefs->mef[i], "BKGDATA", 
                                         "IMAGE", "BKG_DATA");
                enu_check_error_code("accessing BKGDATA extension");
            } else if (bkgerr && !strcmp(bkgerr, extname)) {
                bkg_err = enu_load_component(mefs->mef[i], "BKGERR", 
                                             "IMAGE", "BKG_ERR");
                enu_check_error_code("accessing BKGERR extension");
            } else if (bkgconf && !strcmp(bkgconf, extname)) {
                bkg_conf = enu_load_component(mefs->mef[i], "BKGCONF", 
                                              "IMAGE", "BKG_CONF");
                enu_check_error_code("accessing BKGCONF extension");
            }
        }

        /* set the data mask from quality, construct the hdrl image */

        enu_check(data != NULL, CPL_ERROR_BAD_FILE_FORMAT, "no data in file");
        cpl_mask * bpm  = cpl_image_get_bpm(data);
        cpl_mask_threshold_image(bpm, qual, 0.9, 1.1, CPL_BINARY_1);
        bpm  = cpl_image_get_bpm(err);
        cpl_mask_threshold_image(bpm, qual, 0.9, 1.1, CPL_BINARY_1);
        hdrl_image * hdata = hdrl_image_create(data, err);
        hdrl_image * hbkg = NULL;
        if (bkg) {
            hbkg = hdrl_image_create(bkg, bkg_err);
        }

        result = enu_located_image_new(hdata,
                                       NULL,
                                       conf,
                                       hbkg,
                                       bkg_conf,
                                       plist,
                                       NULL,
                                       NULL,
                                       NULL,
                                       NULL,
                                       cpl_frame_duplicate(frame));

        plist = NULL;
        cpl_image_delete(data);
        cpl_image_delete(err);
        cpl_image_delete(qual);
        cpl_image_delete(bkg);
        cpl_image_delete(bkg_err);
        data = NULL;
        err = NULL;
        qual = NULL;
        conf = NULL;
        bkg = NULL;
        bkg_err = NULL;
        bkg_conf = NULL;
    }

 cleanup:
    cpl_free(frame_format);
    cpl_free(first_ext);
    cpl_propertylist_delete(plist);
    cpl_image_delete(data);
    cpl_image_delete(err);
    cpl_image_delete(qual);
    cpl_image_delete(conf);
    cpl_image_delete(bkg);
    cpl_image_delete(bkg_err);
    cpl_image_delete(bkg_conf);
    enu_mef_extension_list_delete(mefs);
    if (cpl_error_get_code() != CPL_ERROR_NONE) {
        enu_located_image_delete(result);
        result = NULL;
    }

    eris_check_error_code("enu_load_limage_from_frame");
    return result;
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Load components of a multi-extension FITS file
  @param    filename   The name of the MEF file
  @param    plist      Property list of keywords from the primary HDU
  @return   mef_extension_list containing components of file

  The function loads the components from a multi-extension FITS file.
  Extensions are indexed 0 to n, corresponding to HDUs 1 to n+1.
 */
/*----------------------------------------------------------------------------*/

mef_extension_list * enu_load_mef_components(const char * filename, 
                                             cpl_propertylist ** plist) {

    if (cpl_error_get_code() != CPL_ERROR_NONE) return NULL;

    cpl_ensure(filename, CPL_ERROR_NULL_INPUT, NULL);

    /* How many extensions are there, construct result of the correct
       size */

    cpl_size next = cpl_fits_count_extensions(filename);
    mef_extension_list * result = enu_mef_extension_list_new(next);
    mef_extension * new = NULL;
    cpl_propertylist * eplist = NULL;

    /* Read primary HDU header */

    *plist = cpl_propertylist_load(filename, 0);
    enu_check_error_code("accessing HDU 0 header");

    /* Now loop through extensions, loading them in turn */

    for (int i=0; i<next; i++) {
        int hdu = i + 1;
  
        /* get the extension name... */

        cpl_propertylist_delete(eplist);
        eplist = cpl_propertylist_load(filename, hdu);
        const char * name = cpl_propertylist_get_string(eplist, "EXTNAME");
        const char * hduclas1 = cpl_propertylist_get_string(eplist,
                                                            "HDUCLAS1");
        enu_check_error_code("error accessing keywords in HDU %d", hdu);

        /* ...type and value, create appropriate mef_extension */

        if (!strcmp(hduclas1, MEF_EXTENSION_CONTAINING_MASK)) {
            cpl_mask * mask = cpl_mask_load(filename, 0, hdu);
            new = enu_mef_new_mask(name, mask, eplist);
            cpl_mask_delete(mask);
        } else if (!strcmp(hduclas1, "IMAGE")) {
            cpl_image * image = cpl_image_load(filename, HDRL_TYPE_DATA, 0, hdu);
            new = enu_mef_new_image(name, image, eplist);
            cpl_image_delete(image);
        } else if (!strcmp(hduclas1, MEF_EXTENSION_CONTAINING_TABLE)) {
            cpl_table * table = cpl_table_load(filename, hdu, 0);
            new = enu_mef_new_table(name, table, eplist);
            cpl_table_delete(table);
        } else {
            cpl_error_set_message("enu_load_mef_components",
              CPL_ERROR_UNSUPPORTED_MODE,
              "extension has unsupported HDUCLAS1: %s %s", name, hduclas1); 
        }
        enu_check_error_code("error accessing HDU %d", hdu);

        result->mef[i] = new;
        new = NULL;
    }

 cleanup:
    enu_mef_extension_delete(new);
    cpl_propertylist_delete(eplist);
    if (cpl_error_get_code() != CPL_ERROR_NONE) {
        enu_mef_extension_list_delete(result);
        result = NULL;
        cpl_propertylist_delete(*plist);
        *plist = NULL;
    }
    eris_check_error_code("enu_load_mef_components");
    return result;
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Delete a located_image and its contents

  The function deletes the contents of a located_image and then the
  located_image itself.
 */
/*----------------------------------------------------------------------------*/

void enu_located_image_delete(located_image * limage) {

    if (limage) {
        hdrl_image_delete(limage->himage);
        hdrl_imagelist_delete(limage->himagelist);
        cpl_image_delete(limage->confidence);
        hdrl_image_delete(limage->bkg);
        cpl_image_delete(limage->bkg_confidence);
        cpl_propertylist_delete(limage->plist);
        hdrl_catalogue_result_delete(limage->objects);
        cpl_mask_delete(limage->object_mask);
        hdrl_catalogue_result_delete(limage->matchstd_wcs);
        hdrl_catalogue_result_delete(limage->matchstd_phot);
        cpl_frame_delete(limage->frame);
        cpl_free(limage);
    }
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Make a deep copy of a located_image and its contents
  @param    limage   The located_image to copy
  @return   A duplicate located_image

  The function returns a copy of the given located image.
 */
/*----------------------------------------------------------------------------*/

located_image * enu_located_image_duplicate(const located_image * limage) {

    cpl_ensure(limage, CPL_ERROR_NULL_INPUT, NULL);

    /* Watch out for NULL components */

    hdrl_image * himage = NULL;
    if (limage->himage) {
        himage = hdrl_image_duplicate(limage->himage);
    }
    hdrl_imagelist * himagelist = NULL;
    if (limage->himagelist) {
        himagelist = hdrl_imagelist_duplicate(limage->himagelist);
    }
    cpl_image * confidence = NULL;
    if (limage->confidence) { 
        confidence = cpl_image_duplicate(limage->confidence);
    }
    hdrl_image * bkg = NULL;
    if (limage->bkg) {
        bkg = hdrl_image_duplicate(limage->bkg);
    }
    cpl_image * bkg_confidence = NULL;
    if (limage->bkg_confidence) { 
        bkg_confidence = cpl_image_duplicate(limage->bkg_confidence);
    }
    cpl_propertylist * plist = NULL;
    if (limage->plist) {
        plist = cpl_propertylist_duplicate(limage->plist);
    }
    hdrl_catalogue_result * objects = NULL;
    if (limage->objects) {
        objects = enu_hdrl_catalogue_result_duplicate(limage->objects);
    }
    cpl_mask * object_mask = NULL;
    if (limage->object_mask) {
        object_mask = cpl_mask_duplicate(limage->object_mask);
    }  
    hdrl_catalogue_result * matchstd_wcs = NULL;
    if (limage->matchstd_wcs) {
        matchstd_wcs = enu_hdrl_catalogue_result_duplicate(
                          limage->matchstd_wcs);
    }
    hdrl_catalogue_result * matchstd_phot = NULL;
    if (limage->matchstd_phot) {
        matchstd_phot = enu_hdrl_catalogue_result_duplicate(
                          limage->matchstd_phot);
    }  
    cpl_frame * frame = NULL;
    if (limage->frame) {
        frame = cpl_frame_duplicate(limage->frame);
    }

    located_image * result = enu_located_image_new(himage,
                                                   himagelist,
                                                   confidence,
                                                   bkg,
                                                   bkg_confidence,
                                                   plist,
                                                   objects,
                                                   object_mask,
                                                   matchstd_wcs,
                                                   matchstd_phot,
                                                   frame);
    eris_check_error_code("enu_located_image_duplicate");

    return result;
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Create a located_image structure and initialise the contents
  @param    himage         An hdrl_image with 2d data or
  @param    himagelist     An hdrl_imagelist with cube data
  @param    confidence     The confidence map
  @param    bkg            The hdrl_image with the associated background 
  @param    bkg_confidence The confidence map of the background
  @param    plist          The associated property_list
  @param    objects        struct with objects catalogued in image
  @param    object_mask    A mask to blot out the catalogued objects
  @param    wcs            struct with catalogue of matched wcs calibrators
  @param    photom         struct with catalogue of matched photometric calibrators
  @param    frame          Frame associated with image
  @return   The new located image structure

  The function returns a pointer to a located_image structure with
  components initialised to the call parameters. The located_image
  becomes responsible for deleting the components.
 */
/*----------------------------------------------------------------------------*/

located_image * enu_located_image_new(hdrl_image            * himage,
                                      hdrl_imagelist        * himagelist,
                                      cpl_image             * confidence,
                                      hdrl_image            * bkg,
                                      cpl_image             * bkg_confidence,
                                      cpl_propertylist      * plist,
                                      hdrl_catalogue_result * objects,
                                      cpl_mask              * object_mask,
                                      hdrl_catalogue_result * wcs,
                                      hdrl_catalogue_result * photom,
                                      cpl_frame             * frame) {

    if (cpl_error_get_code() != CPL_ERROR_NONE) return NULL;

    /* store either image or cube (imagelist) */
    cpl_ensure(!(himage && himagelist), CPL_ERROR_ILLEGAL_INPUT, NULL);

    located_image * result = cpl_malloc(sizeof(located_image)); 
    result->himage = himage;
    result->himagelist = himagelist;
    result->confidence = confidence;
    result->bkg = bkg;
    result->bkg_confidence = bkg_confidence;
    result->plist = plist;
    result->objects = objects;
    result->object_mask = object_mask;
    result->matchstd_wcs = wcs;
    result->matchstd_phot = photom;
    result->frame = frame;

    return result;
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Load tagged data from a frameset into a located_imagelist
  @param    frameset  A frame set
  @param    tag       The frame tag to look for
  @param    used      The list of raw/calibration frames used for this product
  @return   The filled located_imagelist

  The function returns a pointer to a located_imagelist structure whose
  components have been filled from the given frameset.
 */
/*----------------------------------------------------------------------------*/

located_imagelist * enu_limlist_load_from_frameset(
                                          cpl_frameset * frameset,
                                          const char * tag, 
                                          cpl_frameset * used) {

    if (cpl_error_get_code() != CPL_ERROR_NONE) return NULL;
    cpl_ensure(frameset, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(tag, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(used, CPL_ERROR_NULL_INPUT, NULL);

    located_imagelist     * result = NULL;
    cpl_frameset_iterator * frameset_iter = NULL;
    mef_extension_list    * mefs = NULL;
    cpl_image * data = NULL;
    cpl_image * err = NULL;
    cpl_image * qual = NULL;
    cpl_image * conf = NULL;
    cpl_image * bkg = NULL;
    cpl_image * bkg_err = NULL;
    cpl_image * bkg_conf = NULL;
    cpl_propertylist * plist = NULL;
   
    /* first, find how big the located_imagelist needs to be */

    frameset_iter = cpl_frameset_iterator_new(frameset);
    cpl_size size = 0;

    for (cpl_frame * frame = NULL;
         (frame = cpl_frameset_iterator_get(frameset_iter)) &&
         (cpl_error_get_code() == CPL_ERROR_NONE);
         cpl_frameset_iterator_advance(frameset_iter, 1)) {

        const char * frametag = cpl_frame_get_tag(frame);
        if (!strcmp(tag, frametag)) {
            size++;
        }
    }
    result = enu_located_imagelist_new(size);

    /* go through the frameset again and populate the container */

    cpl_frameset_iterator_reset(frameset_iter);
    cpl_size position = 0; 
    for (cpl_frame * frame = NULL;
         (frame = cpl_frameset_iterator_get(frameset_iter)) &&
         (cpl_error_get_code() == CPL_ERROR_NONE);
         cpl_frameset_iterator_advance(frameset_iter, 1)) {

        /* if frametag is among those wanted, create a located image with
           the data and insert it into the located image list */

        const char * frametag = cpl_frame_get_tag(frame);

        if (!strcmp(tag, frametag)) {

            /* we want this file, treatment depends on number of extensions 
               it has and on the name of the first extension if present */

            located_image * limage = enu_load_limage_from_frame(frame,
                                                                NULL,
                                                                CPL_FALSE);

/*
            const char * filename = cpl_frame_get_filename(frame);
            cpl_msg_info(cpl_func, "..reading %s", filename);
            cpl_size next = cpl_fits_count_extensions(filename);
            located_image * limage = NULL;

            plist = cpl_propertylist_load(filename, 0);

            char * first_ext = NULL;
            if (next > 0) {
                cpl_propertylist * plist1 = cpl_propertylist_load(filename, 1);
                if (cpl_propertylist_has(plist1, "EXTNAME")) {
                    first_ext = cpl_sprintf("%s", cpl_propertylist_get_string(plist1, "EXTNAME"));
                }
                cpl_propertylist_delete(plist1);
            }

            char * frame_format = NULL;
            if (cpl_propertylist_has(plist, "ESO DET FRAM FORMAT")) {
                frame_format = cpl_sprintf("%s", cpl_propertylist_get_string(plist, "ESO DET FRAM FORMAT"));
            }
                     
            if ((next == 0) ||
                (next == 1 &&
                 !strcmp(first_ext, "ASM_DATA") &&
                 !strcmp(frame_format, "single"))) {
*/
                /* raw FITS file with header and 2d image data */
/*
                data = cpl_image_load(filename, HDRL_TYPE_DATA, 0, 0);
                enu_check_error_code("failed to read file: %s", filename);
                hdrl_image * himage = hdrl_image_create(data, NULL);
                cpl_image_delete(data); data = NULL;
                const cpl_size nx = hdrl_image_get_size_x(himage);
                const cpl_size ny = hdrl_image_get_size_y(himage);
*/
                /* confidence set to 100 everywhere */
/*
                cpl_image * confidence = cpl_image_new(nx, ny, HDRL_TYPE_DATA);
                cpl_image_fill_window(confidence, 1, 1, nx, ny, 100.0); 

                limage = enu_located_image_new(himage, NULL, confidence, NULL,
                                               NULL, plist, NULL, NULL, NULL,
                                               NULL,
                                               cpl_frame_duplicate(frame));
                plist = NULL;

            } else if (next == 1 &&
                       !strcmp(first_ext, "ASM_DATA") &&
                       !strcmp(frame_format, "cube")) {
*/
                /* raw FITS file with header and 3d image data (cube) */
/*
                cpl_imagelist * datalist = cpl_imagelist_load(filename,
                                                              HDRL_TYPE_DATA,
                                                              0);
                enu_check_error_code("failed to read file: %s", filename);
                hdrl_imagelist * himagelist = hdrl_imagelist_create(datalist,
                                                                    NULL);
                cpl_imagelist_delete(datalist); datalist = NULL;
                const cpl_size nx = hdrl_imagelist_get_size_x(himagelist);
                const cpl_size ny = hdrl_imagelist_get_size_y(himagelist);
*/
                /* confidence set to 100 everywhere */
/*
                cpl_image * confidence = cpl_image_new(nx, ny, HDRL_TYPE_DATA);
                cpl_image_fill_window(confidence, 1, 1, nx, ny, 100.0); 

                limage = enu_located_image_new(NULL, himagelist, confidence,
                                               NULL, NULL, plist, NULL, NULL,
                                               NULL, NULL,
                                               cpl_frame_duplicate(frame));
                plist = NULL;

            } else {
*/
                /* If it's a mef it should be in ESO Data Products Standard 
                   format */
/*
                cpl_propertylist_delete(plist);
                plist = NULL;
                enu_mef_extension_list_delete(mefs);
                mefs = enu_load_mef_components(filename, &plist);
                enu_check_error_code("Unable to load components for file");
*/
                /* look in first extension for HDU keywords */
/*
                const char * hduclass = cpl_propertylist_get_string(
                                        mefs->mef[0]->plist, "HDUCLASS");
                const char * hdudoc = cpl_propertylist_get_string(
                                      mefs->mef[0]->plist, "HDUDOC");
                //const char * hduvers = cpl_propertylist_get_string(
                //                       mefs->mef[0]->plist, "HDUVERS");
                enu_check_error_code("Unable to read ESO DPS keywords from header");

                enu_check(strstr(hduclass, "ESO") != NULL &&
                         // strstr(hduvers, "SDP version 8") != NULL &&
                          strstr(hdudoc, "SDP") != NULL, 
                          CPL_ERROR_BAD_FILE_FORMAT, "file not in ESO DPS format");
*/
                /* now loop through extensions filling standard data elements */
/*
                for (cpl_size i=0; i<next; i++) {
                    const char * hduclas2 = cpl_propertylist_get_string(
                                           mefs->mef[i]->plist, "HDUCLAS2");
*/
                    /* the returned plist is a merge of the plist for the
                       science image and the header plist.*/
/*
                    if (hduclas2 && !strcmp(hduclas2, "DATA")) {
                        data = enu_load_component(mefs->mef[i], "SCIDATA", 
                                                 "IMAGE", "DATA");
                        enu_check_error_code("accessing SCIDATA extension");
*/
                        /* Start with the science image plist (for the data and 
                           wcs keywords)*/
/*
                        cpl_propertylist * primary_plist = plist;
*/
                        /* remove the SDP7 keywords which were appropriate to
                           the science data extension but will
                           confuse the enu_limage_save routine if left in the
                           main header */
/*
                        plist = cpl_propertylist_duplicate(
                                       mefs->mef[i]->plist);
                        const char * sdp7_items = " EXTNAME HDUCLAS1 HDUCLAS2"
                                                  " HDUCLAS3 ";
                        cpl_size psize = cpl_propertylist_get_size(plist);
                        for (cpl_size ip=psize; ip>0; ip--) {
                            const cpl_property * prop = 
                                       cpl_propertylist_get_const(
                                       plist, ip-1);
                            const char * name = cpl_property_get_name(prop);
*/
                            /* pad the keyword name with spaces which will
                               stop strstr misidentifying a keyword that
                               matches _part_ of an SDP7 keyword */
/*
                            char * padded_name = cpl_sprintf(" %s ", name);
                            if (strstr(sdp7_items, padded_name) != NULL) {
                                cpl_propertylist_erase(plist, name);
                            }
                            cpl_free(padded_name);
                        }
*/
                        /* merge info from the primary header */
/*
                        psize = cpl_propertylist_get_size(primary_plist);
                        for (cpl_size ip=psize; ip>0; ip--) {
                            const cpl_property * prop = 
                                                 cpl_propertylist_get_const(
                                                 primary_plist, ip-1);
                            const char * name = cpl_property_get_name(prop);
                            if (!cpl_propertylist_has(plist, name)) {
                                cpl_propertylist_copy_property(plist,
                                                               primary_plist,
                                                               name);
                            }
                        }
                        cpl_propertylist_delete(primary_plist);

                    } else if (hduclas2 && !strcmp(hduclas2, "ERROR")) {
                        err = enu_load_component(mefs->mef[i], "SCIERR", 
                                                 "IMAGE", "ERROR");
                        enu_check_error_code("accessing SCIERR extension");
                    } else if (hduclas2 && !strcmp(hduclas2, "QUALITY")) {
                        qual = enu_load_component(mefs->mef[i], "QUALDATA", 
                                                 "IMAGE", "QUALITY");
                        enu_check_error_code("accessing QUALDATA extension");
                    } else if (hduclas2 && !strcmp(hduclas2, "CONF")) {
                        conf = enu_load_component(mefs->mef[i], "CONFDATA", 
                                                 "IMAGE", "CONF");
                        enu_check_error_code("accessing CONFDATA extension");
                    } else if (hduclas2 && !strcmp(hduclas2, "BKG_DATA")) {
                        bkg = enu_load_component(mefs->mef[i], "BKGDATA", 
                                                 "IMAGE", "BKG_DATA");
                        enu_check_error_code("accessing BKGDATA extension");
                    } else if (hduclas2 && !strcmp(hduclas2, "BKG_ERR")) {
                        bkg_err = enu_load_component(mefs->mef[i], "BKGERR", 
                                                     "IMAGE", "BKG_ERR");
                        enu_check_error_code("accessing BKGERR extension");
                    } else if (hduclas2 && !strcmp(hduclas2, "BKG_CONF")) {
                        bkg_conf = enu_load_component(mefs->mef[i], "BKGCONF", 
                                                      "IMAGE", "BKG_CONF");
                        enu_check_error_code("accessing BKGCONF extension");
                    }
                }
*/
                /* set the data mask from quality, construct the hdrl image */
/*
                enu_check(data != NULL, CPL_ERROR_BAD_FILE_FORMAT, 
                          "no data in file");
                cpl_mask * bpm  = cpl_image_get_bpm(data);
                cpl_mask_threshold_image(bpm, qual, 0.9, 1.1, CPL_BINARY_1);
                bpm  = cpl_image_get_bpm(err);
                cpl_mask_threshold_image(bpm, qual, 0.9, 1.1, CPL_BINARY_1);
                hdrl_image * hdata = hdrl_image_create(data, err);
                hdrl_image * hbkg = NULL;
                if (bkg) {
                    hbkg = hdrl_image_create(bkg, bkg_err);
                }

                limage = enu_located_image_new(hdata, NULL, conf, hbkg,
                                               bkg_conf,
                                               plist, NULL, NULL, NULL, NULL,
                                               cpl_frame_duplicate(frame));
                plist = NULL;
                cpl_image_delete(data);
                cpl_image_delete(err);
                cpl_image_delete(qual);
                cpl_image_delete(bkg);
                cpl_image_delete(bkg_err);
                data = NULL;
                err = NULL;
                qual = NULL;
                conf = NULL;
                bkg = NULL;
                bkg_err = NULL;
                bkg_conf = NULL;
            }
            if(frame_format != NULL) {
            	cpl_free(frame_format);
            }
            if(first_ext != NULL) {
            	cpl_free(first_ext);
            }
*/
            /* insert new located_image into list */

            enu_located_imagelist_insert(result, limage, position);

            /* Update used_frameset */

            cpl_frame * dup_frame = cpl_frame_duplicate(frame);
            cpl_frameset_insert(used, dup_frame); 

            position++;
        }
    }

 //cleanup:

    cpl_propertylist_delete(plist);
    cpl_image_delete(data);
    cpl_image_delete(err);
    cpl_image_delete(qual);
    cpl_image_delete(conf);
    cpl_image_delete(bkg);
    cpl_image_delete(bkg_err);
    cpl_image_delete(bkg_conf);
    enu_mef_extension_list_delete(mefs);
    cpl_frameset_iterator_delete(frameset_iter);
    if (cpl_error_get_code() != CPL_ERROR_NONE) {
        enu_located_imagelist_delete(result);
        result = NULL;
    }
    eris_check_error_code("enu_limlist_load_from_frameset");
    return result;
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Load data from a MEF component of specified type
  @param    mef       The mef_extension whose content is to be loaded
  @param    dps_name  The name of the ESO Data Product Standard element, e.g. "SCIDATA"
  @param    hduclas1  The check value for keyword HDUCLAS1
  @param    hduclas2  The check value for keyword HDUCLAS2
  @return   A cpl_image containing the data 

  The function removes the cpl_image from the input, thus passes ownership
  of the image to the caller.
 */
/*----------------------------------------------------------------------------*/

cpl_image * enu_load_component(mef_extension * mef,
                               const char * dps_name,
                               const char * hduclas1,
                               const char * hduclas2) {
 
    if (cpl_error_get_code() != CPL_ERROR_NONE) return NULL;
    cpl_ensure(mef, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(dps_name, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(hduclas1, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(hduclas2, CPL_ERROR_NULL_INPUT, NULL);

    cpl_image * result = NULL;
                                               
    cpl_propertylist * plist = mef->plist;
    const char * hduclass = cpl_propertylist_get_string(plist, "HDUCLASS");
    const char * hdudoc = cpl_propertylist_get_string(plist, "HDUDOC");
    //const char * hduvers = cpl_propertylist_get_string(plist, "HDUVERS");
    const char * my_hduclas1 = cpl_propertylist_get_string(plist, "HDUCLAS1");
    const char * my_hduclas2 = cpl_propertylist_get_string(plist, "HDUCLAS2");

    enu_check(strstr(hduclass, "ESO") != NULL &&
              strstr(hdudoc, "SDP") != NULL &&
              //strstr(hduvers, "SDP version 8") != NULL &&
              strstr(my_hduclas1, hduclas1) != NULL &&
              strstr(my_hduclas2, hduclas2) != NULL,
              CPL_ERROR_BAD_FILE_FORMAT, "bad %s format", dps_name);

    result = mef->data;

 cleanup:
    if (cpl_error_get_code() != CPL_ERROR_NONE) {
        result = NULL;
    }
    if (result != NULL) mef->data = NULL;
    eris_check_error_code("enu_load_component");
    return result;
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Delete a located_imagelist and its contents
  @param    limlist  The located_imagelist to delete

  The function deletes the contents of a located_imagelist and then the
  located_imagelist itself.
 */
/*----------------------------------------------------------------------------*/

void enu_located_imagelist_delete(located_imagelist * limlist) {

    if (limlist) {
        for (cpl_size i = 0; i < limlist->size; i++) {
            enu_located_image_delete(limlist->limages[i]);
        }
        cpl_free(limlist->limages);
        cpl_free(limlist);
    }
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Make a deep copy of a located_imagelist and its contents
  @param    limlist  The located_imagelist to be copied
  @return   A copy of the located_imagelist

  The function returns a duplicate of the given located_imagelist.
 */
/*----------------------------------------------------------------------------*/

located_imagelist * enu_located_imagelist_duplicate(
                     const located_imagelist * limlist) {

    cpl_ensure(limlist, CPL_ERROR_NULL_INPUT, NULL);

    located_imagelist * result = enu_located_imagelist_new(limlist->size);
    for (cpl_size i = 0; i < limlist->size; i++) {
        enu_located_imagelist_insert(result,
                                     enu_located_image_duplicate(
                                     limlist->limages[i]), i);
    }

    return result;
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Insert a located_image at a specified point in a located_imagelist
  @param    limlist   The destination list 
  @param    limage    The image to insert
  @param    position  The insertion position (0 based)
  @return   CPL_ERROR_NONE if all goes well, a cpl error code otherwise

  The function inserts a located_image at a specified position in a 
  located_imagelist.
 */
/*----------------------------------------------------------------------------*/

cpl_error_code enu_located_imagelist_insert(located_imagelist * limlist,
                                            located_image * limage,
                                            cpl_size position) {

    if (cpl_error_get_code() != CPL_ERROR_NONE) return cpl_error_get_code();

    /* position valid? */

    cpl_ensure_code(position >= 0 && position < limlist->size,
      CPL_ERROR_ACCESS_OUT_OF_RANGE);
    limlist->limages[position] = limage;

    return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Return a pointer to a new located_imagelist
  @param    size    The number of images that the list is to hold
  @return   The pointer to the new located_imagelist

  The function returns a pointer to a new located_imagelist struct with
  space to hold the specified number of located_images.
 */
/*----------------------------------------------------------------------------*/

located_imagelist * enu_located_imagelist_new(cpl_size size) {

    if (cpl_error_get_code() != CPL_ERROR_NONE) return NULL;

    located_imagelist * limlist = cpl_malloc(sizeof(located_imagelist)); 
    limlist->size = size;
    limlist->limages = cpl_calloc(size, sizeof(located_image));
    for (cpl_size i = 0; i < limlist->size; i++) {
        limlist->limages[i] = NULL;
    }
    return limlist;
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Delete a mef_extension and its contents
  @param    mef   The extension to delete

  The function deletes the contents of a mef_extension and then the
  mef_extension itself.
 */
/*----------------------------------------------------------------------------*/

void enu_mef_extension_delete(mef_extension * mef) {

    if (mef) {
        if (!strcmp(mef->data_type, MEF_EXTENSION_CONTAINING_IMAGE)) {
            cpl_image_delete((cpl_image *) (mef->data));
        } else if (!strcmp(mef->data_type, MEF_EXTENSION_CONTAINING_MASK)) {
            cpl_mask_delete((cpl_mask *) mef->data);
        } else if (!strcmp(mef->data_type, MEF_EXTENSION_CONTAINING_TABLE)) {
            cpl_table_delete((cpl_table *) mef->data);
        } else if (!strcmp(mef->data_type, MEF_EXTENSION_CONTAINING_VECTOR)) {
            cpl_vector_delete((cpl_vector *) mef->data);
        } else {
            cpl_error_set("enu_mef_extension_delete", 
              CPL_ERROR_UNSUPPORTED_MODE);
        }
        cpl_propertylist_delete(mef->plist);
        cpl_free(mef->name);
        cpl_free(mef);
    }
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Delete a mef_extension_list and its contents
  @param    list    The mef_extension_list to delete

  The function deletes the contents of a mef_extension_list and then the
  mef_extension_list itself.
 */
/*----------------------------------------------------------------------------*/

void enu_mef_extension_list_delete(mef_extension_list * list) {

    if (list) {
        for (cpl_size i = 0; i < list->size; i++) {
            enu_mef_extension_delete(list->mef[i]);
        }
        cpl_free(list->mef);
        cpl_free(list);
    }
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Get a cpl_mask from a named mef_extension in a list
  @param    meflist    The list of extensions
  @param    target     The name of the extension containing the mask
  @return   Pointer to the cpl_mask

  The searches a mef_extension_list for a named mef_extension, and returns
  the cpl_mask it contains.
 */
/*----------------------------------------------------------------------------*/

cpl_mask * enu_mef_extension_list_get_mask(mef_extension_list * meflist, 
                                           const char * target) {

    if (cpl_error_get_code() != CPL_ERROR_NONE) return NULL;

    cpl_ensure(meflist, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(target, CPL_ERROR_NULL_INPUT, NULL);

    /* Search for extension */

    cpl_mask * result = NULL;
    for (int i=0; i<meflist->size; i++) {
        if (!strcmp(meflist->mef[i]->name, target)) {
            cpl_ensure(!strcmp(meflist->mef[i]->data_type,
                       MEF_EXTENSION_CONTAINING_MASK),
                       CPL_ERROR_INCOMPATIBLE_INPUT, NULL);
            result = cpl_mask_duplicate((const cpl_mask *) meflist->mef[i]->data);
            break;
        }
    }

    /* Check that the extensions was found */

    cpl_ensure(result, CPL_ERROR_DATA_NOT_FOUND, NULL);

    return result;
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Construct a new mef_extension_list 
  @param    size    The maximum number of extensions the list can hold
  @return   A pointer to the new mef_extension_list

  The function returns a pointer to a new mef_extension_list struct with
  space to hold the specified number of mef_extension structs.
 */
/*----------------------------------------------------------------------------*/

mef_extension_list * enu_mef_extension_list_new(cpl_size size) {

    if (cpl_error_get_code() != CPL_ERROR_NONE) return NULL;

    mef_extension_list * mef_list = cpl_malloc(sizeof(mef_extension_list)); 
    mef_list->size = size;
    mef_list->mef = cpl_calloc(size, sizeof(mef_extension));
    for (cpl_size i = 0; i < size; i++) {
        mef_list->mef[i] = NULL;
    }

    return mef_list;
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Create a mef_extension to hold a cpl_image
  @param    name    The name to give the extension       
  @param    data    The cpl_image to place in the extension
  @param    plist   The propertylist to place in the extension
  @return   The new mef_extension

  The function returns a pointer to a mef_extension structure 
  holding a copy of the cpl_image.

  The mef_extension and its contents should be deleted by a call to
  enu_mef_extension_delete.
 */
/*----------------------------------------------------------------------------*/

mef_extension * enu_mef_new_image(const char * name,
                                  const cpl_image * data,
                                  const cpl_propertylist * plist) {

    if (cpl_error_get_code() != CPL_ERROR_NONE) return NULL;

    /* check parameters */

    cpl_ensure(name && data, CPL_ERROR_NULL_INPUT, NULL);

    mef_extension * result = cpl_malloc(sizeof(mef_extension)); 
    result->name = cpl_strdup(name);
    result->data = (void *) cpl_image_duplicate(data);
    result->data_type = MEF_EXTENSION_CONTAINING_IMAGE;
    if (plist) {
        result->plist = cpl_propertylist_duplicate(plist);
    } else {
        result->plist = NULL;
    }
    return result;
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Create a mef_extension struct holding a cpl_mask
  @param    name    The name to give the extension       
  @param    data    The cpl_mask to place in the extension
  @param    plist   The propertylist to place in the extension
  @return   The new mef_extension struct

  The function returns a pointer to a mef_extension structure 
  holding a copy of the cpl_mask.

  The mef_extension and its contents should be deleted by a call to
  enu_mef_extension_delete.
 */
/*----------------------------------------------------------------------------*/

mef_extension * enu_mef_new_mask(const char * name,
                                 const cpl_mask * data,
                                 const cpl_propertylist * plist) {

    if (cpl_error_get_code() != CPL_ERROR_NONE) return NULL;

    /* check parameters */

    cpl_ensure(name && data, CPL_ERROR_NULL_INPUT, NULL);

    mef_extension * result = cpl_malloc(sizeof(mef_extension)); 
    result->name = cpl_strdup(name);
    result->data = (void *) cpl_mask_duplicate(data);
    result->data_type = MEF_EXTENSION_CONTAINING_MASK;
    if (plist) {
        result->plist = cpl_propertylist_duplicate(plist);
    } else {
        result->plist = NULL;
    }
    return result;
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Create a mef_extension struct holding a cpl_table
  @param    name    The name to give the extension       
  @param    table   The cpl_table to place in the extension
  @param    plist   The propertylist to place in the extension
  @return   The new mef_extension struct

  The function returns a pointer to a mef_extension structure 
  holding a copy of the cpl_table.

  The mef_extension and its contents should be deleted by a call to
  enu_mef_extension_delete.
 */
/*----------------------------------------------------------------------------*/

mef_extension * enu_mef_new_table(const char * name,
                                  const cpl_table * table,
                                  const cpl_propertylist * plist) {

    if (cpl_error_get_code() != CPL_ERROR_NONE) return NULL;
    cpl_ensure(name, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(table, CPL_ERROR_NULL_INPUT, NULL);

    mef_extension * result = cpl_malloc(sizeof(mef_extension)); 
    result->name = cpl_strdup(name);
    result->data = (void *) cpl_table_duplicate(table);
    result->plist = NULL;
    if (plist) {
        result->plist = cpl_propertylist_duplicate(plist);
    } else {
        result->plist = NULL;
    }
    result->data_type = MEF_EXTENSION_CONTAINING_TABLE;

    return result;
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Create a mef_extension struct holding a cpl_vector
  @param    name    The name to give the extension
  @param    vector  The cpl_vector to place in the extension
  @param    plist   The propertylist to place in the extension
  @return   The new mef_extension struct

  The function returns a pointer to a mef_extension structure 
  holding a copy of the cpl_vector.

  The mef_extension and its contents should be deleted by a call to
  enu_mef_extension_delete.
 */
/*----------------------------------------------------------------------------*/

mef_extension * enu_mef_new_vector(const char * name,
                                   const cpl_vector * vector,
                                   const cpl_propertylist * plist) {

    if (cpl_error_get_code() != CPL_ERROR_NONE) return NULL;
    cpl_ensure(name, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(vector, CPL_ERROR_NULL_INPUT, NULL);

    mef_extension * result = cpl_malloc(sizeof(mef_extension)); 
    result->name = cpl_strdup(name);
    result->data = (void *) cpl_vector_duplicate(vector);
    result->plist = NULL;
    if (plist) {
        result->plist = cpl_propertylist_duplicate(plist);
    } else {
        result->plist = NULL;
    }
    result->data_type = MEF_EXTENSION_CONTAINING_VECTOR;

    return result;
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Save a mef_extension struct to FITS file
  @param    mef      The mef_extension  to save      
  @param    filename The name of the file
  @param    plist    The propertylist to save
  @param    mode     The desired output options
  @return   CPL_ERROR_NONE on success, a cpl error code otherwise

  The function writes the mef extension to the named multi-extension FITS file.

 */
/*----------------------------------------------------------------------------*/

cpl_error_code enu_mef_extension_save(const mef_extension * mef,
                                      const char * filename,
                                      const cpl_propertylist * plist,
                                      unsigned mode) {

    if (cpl_error_get_code() != CPL_ERROR_NONE) return cpl_error_get_code();

    cpl_msg_debug(cpl_func, "enu_mef_extension_save entry");

    cpl_propertylist * plist_copy = NULL;

    /* check parameters, only one of the data types can be set */

    cpl_ensure_code(mef, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(filename, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(plist, CPL_ERROR_NULL_INPUT);

    plist_copy = cpl_propertylist_duplicate(plist);
    cpl_propertylist_update_string(plist_copy, "ERIS_NIX_MEF_TYPE",
                                   mef->data_type);

    if (!strcmp(mef->data_type, MEF_EXTENSION_CONTAINING_IMAGE)) {
       if (cpl_propertylist_has(plist, "CD3_3")) {
           //double cd3_3 = cpl_propertylist_get_double(plist, "CD3_3");

           /* the existence of CD3_3 in the header means that this
              data has ndim=3. In the LSS case that 3rd
              dimension is degenerate but allows the wcs decription to
              work.

              To get CPL to store NDIM=3 we have to store the
              2d data as one image in an imagelist. */

           cpl_msg_info(cpl_func, "..saving data with NDIM=3");

           cpl_imagelist * imlist = cpl_imagelist_new();
           cpl_imagelist_set(imlist, mef->data, 0);
           cpl_imagelist_save(imlist,
                              filename, 
                              CPL_TYPE_UNSPECIFIED,
                              plist_copy,
                              mode);
           /* let image data be deleted with mef */
           cpl_imagelist_unset(imlist, 0);
           cpl_imagelist_delete(imlist);

       } else {
           cpl_image_save((const cpl_image *) (mef->data),
                          filename,
                          CPL_TYPE_UNSPECIFIED,
                          plist_copy,
                          mode);
       }
    } else if (!strcmp(mef->data_type, MEF_EXTENSION_CONTAINING_MASK)) {
        cpl_mask_save((const cpl_mask *) (mef->data),
                      filename,
                      plist_copy, 
                      mode);
    } else if (!strcmp(mef->data_type, MEF_EXTENSION_CONTAINING_TABLE)) {
        cpl_table_save((const cpl_table *) (mef->data),
                       NULL,
                       plist_copy, 
                       filename,
                       mode);
    } else if (!strcmp(mef->data_type, MEF_EXTENSION_CONTAINING_VECTOR)) {
        cpl_vector_save((const cpl_vector *) (mef->data),
                        filename, 
                        CPL_TYPE_DOUBLE,
                        plist_copy,
                        mode);
    } else {
        cpl_error_set_message("enu_mef_extension_save",
          CPL_ERROR_UNSUPPORTED_MODE,
          "unsupported extension type: %s", mef->data_type);
    }

    cpl_propertylist_delete(plist_copy);
    eris_check_error_code("enu_mef_extension_save");
    return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Calculate object masks for images in a located_imagelist.
  @param    obj_min_pixels  The minimum allowable size of an object
  @param    obj_threshold   The detection threshold in sigma above sky
  @param    bkg_mesh_size   The smoothing box size for background map estimation
  @param    bkg_smooth_fwhm The FWHM of the smoothing kernel in the detection algorithm
  @param    limlist         The list of images requiring object masks
  @return   CPL error code, CPL_ERROR_NONE on success

  The function uses CASU routines to calculate object masks for images in a 
  located_imagelist.

 */
/*----------------------------------------------------------------------------*/

cpl_error_code enu_opm_limlist(const int obj_min_pixels,
                               const double obj_threshold,
                               const int bkg_mesh_size,
                               const double bkg_smooth_fwhm,
                               located_imagelist * limlist) {

    if (cpl_error_get_code() != CPL_ERROR_NONE) return cpl_error_get_code();

    /* parameter check */

    cpl_ensure_code(limlist, CPL_ERROR_NULL_INPUT);

    casu_fits ** indata = NULL;
    casu_fits ** inconf = NULL;
    casu_fits ** invar = NULL;

    /* translate located_imagelist to casu_fits components */

    encu_limlist_to_casu_fits(limlist, &indata, &inconf, &invar);

    /* loop through the located_images, calculate the object mask
       for each, copy that mask to the object_mask in the located_image */

    int casu_status = CASU_OK;
    int niter = 3;

    cpl_msg_info(cpl_func, ".. calculating object masks");
    for (cpl_size j = 0; j < limlist->size; j++) {
        casu_opm(indata[j], inconf[j], obj_min_pixels, obj_threshold,
                 bkg_mesh_size, bkg_smooth_fwhm, niter, &casu_status);
        enu_check(casu_status == CASU_OK, CPL_ERROR_UNSPECIFIED,
                  "CASU error calculating object mask");
        limlist->limages[j]->object_mask = cpl_mask_duplicate(
                                           cpl_image_get_bpm(indata[j]->image));
    }

  cleanup:

    for (cpl_size j = 0; j < limlist->size; j++) {
        casu_fits_delete(indata[j]);
        casu_fits_delete(inconf[j]);
        casu_fits_delete(invar[j]);
    }
    cpl_free(indata);
    cpl_free(inconf);
    cpl_free(invar);
    eris_check_error_code("enu_opm_limlist");
    return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Calculate object masks for LSS images in a located_imagelist.
  @param    limlist     The list of images to be masked
  @param    nsigma_cut  TBD
  @return   CPL error code, CPL_ERROR_NONE on success.

  The function calculates object masks for LSS images in a 
  located_imagelist. The method used is to collapse the spectrum in the
  dispersion direction, then look for objects in the resulting vector.

  This tends to underestimate the object width in cases where the object
  is brighter at one end of the spectrum - could try a 2d thresholding
  as well?

 */
/*----------------------------------------------------------------------------*/

cpl_error_code enu_opm_lss_limlist(located_imagelist * limlist,
                                   const int nsigma_cut) {

    if (cpl_error_get_code() != CPL_ERROR_NONE) return cpl_error_get_code();
    cpl_ensure_code(limlist, CPL_ERROR_NULL_INPUT);

    cpl_msg_info(cpl_func, "..calculating LSS object pixel mask");

    /* loop through the located_images, calculate the object mask
       for each, copy that mask to the object_mask in the located_image */

    for (cpl_size j = 0; j < limlist->size; j++) {
        cpl_size nx = hdrl_image_get_size_x(limlist->limages[j]->himage);
        cpl_size ny = hdrl_image_get_size_y(limlist->limages[j]->himage);
        cpl_image * collapsed = cpl_image_collapse_median_create(
                                hdrl_image_get_image(
                                limlist->limages[j]->himage), 0, 0, 0);

        /* mask for pixels that are unusually high */

        double csigma = 0.0;
        double cmedian = cpl_image_get_median_dev(collapsed, &csigma);
        double maxval = cpl_image_get_max(collapsed);
        cpl_mask * collapsed_opm = cpl_mask_threshold_image_create(
                                   collapsed, cmedian + nsigma_cut * csigma,
                                   2.0 * maxval);

        /* 'grow' the masked areas to try and catch lower level emission */

        cpl_mask * kernel = cpl_mask_new(11, 1);
        cpl_mask_not(kernel);
        cpl_mask * dilated_collapsed_opm = cpl_mask_new(nx, 1);
        cpl_mask_filter(dilated_collapsed_opm, collapsed_opm, kernel,
                        CPL_FILTER_DILATION, CPL_BORDER_NOP);
        cpl_mask_delete(kernel);

        /* grow the 1d mask to a 2d image */

        const cpl_binary * collapsed_opm_data = cpl_mask_get_data_const(
                                                dilated_collapsed_opm);
        cpl_mask * opm = cpl_mask_new(nx, ny);
        cpl_binary * opm_data = cpl_mask_get_data(opm);
        for (cpl_size iy=0; iy < ny; iy++) {
            memcpy(opm_data + iy * (nx * sizeof(*opm_data)),
                   collapsed_opm_data,
                   nx * sizeof(*opm_data));
        }

        /* set the object_mask of the located_image */

        cpl_mask_delete(limlist->limages[j]->object_mask);
        limlist->limages[j]->object_mask = opm;

        cpl_image_delete(collapsed);
        cpl_mask_delete(collapsed_opm);
        cpl_mask_delete(dilated_collapsed_opm);
    }
    eris_check_error_code("enu_opm_lss_limlist");
    return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Correct the wcs of an image
  @param    refcat       The catalogue of reference positions
  @param    wcs_method   String describing the correction method
  @param    catalogue    Name of reference catalogue
  @param    limage       The lmage with object catalogues to be corrected
  @param    match_rad    Max distance for match between obj and std.       
  @param    matched_stds Table containing standards matched to image objects
  @param    xy_shift     Shift required in x and y to correct the wcs
  @return   CPL_ERROR_NONE if all goes well, otherwise a cpl error code

 */
/*----------------------------------------------------------------------------*/

cpl_error_code  enu_correct_wcs(const cpl_table * refcat,
                                const char * wcs_method,
                                const char * catalogue,
                                located_image * limage,
                                const double match_rad,
                                cpl_table ** matched_stds,
                                cpl_matrix ** xy_shift) {

    cpl_matrix       * celestial = NULL;
    cpl_array        * colnames = NULL;
    cpl_array        * conv_status = NULL;
    cpl_table        * refcat_copy = NULL;
    cpl_matrix       * ref_phys = NULL;
    cpl_matrix       * ref_world = NULL;
    cpl_matrix       * xy = NULL;
    cpl_wcs          * wcs = NULL;

    if (cpl_error_get_code() != CPL_ERROR_NONE) return cpl_error_get_code();

    cpl_msg_info(cpl_func, "correcting %s", cpl_frame_get_filename(limage->frame));

    *xy_shift = NULL;

    cpl_msg_info(cpl_func, " ..%s", cpl_frame_get_filename(limage->frame));

    /* calculate predicted positions of refcat objects for this wcs */

    cpl_size nref = cpl_table_get_nrow(refcat);
    const double * ra = cpl_table_get_data_double_const(refcat, "RA");
    const double * dec = NULL;
    if (cpl_table_has_column(refcat, "DEC")) {
        dec = cpl_table_get_data_double_const(refcat, "DEC");
    }
    ref_world = cpl_matrix_new(nref, 2);
    for (cpl_size iobj = 0; iobj < nref; iobj++) {
        cpl_matrix_set(ref_world, iobj, 0, ra[iobj]);
        cpl_matrix_set(ref_world, iobj, 1, dec[iobj]);
    }
    wcs = cpl_wcs_new_from_propertylist(limage->plist);
    cpl_wcs_convert(wcs, ref_world, &ref_phys, &conv_status,
                    CPL_WCS_WORLD2PHYS);
    enu_check_error_code("failure calculating predicted positions");

    /* Store the predicted positions in columns of a duplicate refcat
       but with all columns removed except RA, DEC, 'Std index', 
       Aper_flux_3 */

    refcat_copy = cpl_table_duplicate(refcat);
    colnames = cpl_table_get_column_names(refcat_copy); 
    cpl_size ncol = cpl_array_get_size(colnames);
    for (cpl_size ic = 0; ic < ncol; ic++) {
        const char * colname = cpl_array_get_string(colnames, ic);
        if (strcmp(colname, "RA") && 
            strcmp(colname, "DEC") &&
            strcmp(colname, "Aper_flux_3") &&
            strcmp(colname, "Std index")) {
            cpl_table_erase_column(refcat_copy, colname);
        }
    }

    /* add xpredict and ypredict columns */

    cpl_table_new_column(refcat_copy, "xpredict", CPL_TYPE_DOUBLE);
    cpl_table_new_column(refcat_copy, "ypredict", CPL_TYPE_DOUBLE);
    for (cpl_size iobj = 0; iobj < nref; iobj++) {
        cpl_table_set_double(refcat_copy, "xpredict", iobj, 
                             cpl_matrix_get(ref_phys, iobj, 0));
        cpl_table_set_double(refcat_copy, "ypredict", iobj, 
                             cpl_matrix_get(ref_phys, iobj, 1));
    }
    enu_check_error_code("failure while creating predicted positions");

    /* Match detected sources with refcat_copy */ 

    cpl_msg_info(cpl_func, " ..matching sources");
    cpl_size nobj = 0;
    if (limage->objects && limage->objects->catalogue) {
        nobj = cpl_table_get_nrow(limage->objects->catalogue);
    }

    if (min(nref, nobj) > 0) {

        cpl_msg_info(cpl_func, " ..calling enm_try_all_associations");
        
        enm_try_all_associations(limage->objects->catalogue,
                                 refcat_copy,
                                 match_rad,
                                 matched_stds);
        enu_check_error_code("enu_correct_wcs failure at "
                             "try_all_associations");

        enm_correct_crpix(wcs_method,
                          catalogue,
                          *matched_stds,
                          limage,
                          xy_shift);
        enu_check_error_code("failure to correct wcs");

    } else {
        cpl_msg_info(cpl_func, " ..no catalogued objects, cannot correct wcs");
    }

  cleanup:
    cpl_matrix_delete(celestial);
    cpl_array_delete(colnames);
    cpl_array_delete(conv_status);
    cpl_table_delete(refcat_copy);
    cpl_matrix_delete(ref_phys);
    cpl_matrix_delete(ref_world);
    cpl_matrix_delete(xy);
    cpl_wcs_delete(wcs);
    eris_check_error_code("enu_correct_wcs");
    return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Update the CD matrix to reduce image distortion and rotation
  @param    limage     The jitter whose wcs is to be modified
  @param    refine_wcs The table holding corrected wcs info, or NULL
  @return   A cpl_error_code, CPL_ERROR_NONE if all is well.

  Updates the CD matrix entries in the FITS wcs to reduce image distortion
  and rotation.
 */
/*----------------------------------------------------------------------------*/

cpl_error_code enu_modify_CD_matrix(located_image * limage,
                                    const cpl_table * refine_wcs) {

    if (cpl_error_get_code() != CPL_ERROR_NONE) return cpl_error_get_code();
    cpl_ensure_code(limage, CPL_ERROR_NULL_INPUT);

    if (!refine_wcs) return CPL_ERROR_NONE;

    const char * camera = cpl_propertylist_get_string(limage->plist,
                                                      "ESO INS2 NXCW NAME");
    enu_check(camera != NULL, 
              CPL_ERROR_ILLEGAL_INPUT,
              "failed to read NIX camera from ESO.INS2.NXCW.NAME");

    cpl_size nrow = cpl_table_get_nrow(refine_wcs);
    cpl_boolean match = CPL_FALSE;

    double cd1_1 = 0.0;
    double cd1_2 = 0.0;
    double cd2_1 = 0.0;
    double cd2_2 = 0.0;

    for (cpl_size row = 0; row < nrow; row++) {
        const char * row_cam = cpl_table_get_string(refine_wcs, "camera", row);

        if (!strcmp(camera, row_cam)) {
            match = CPL_TRUE;

            int ignore = 0;
            cd1_1 = cpl_table_get_double(refine_wcs, "CD1_1", row, &ignore);
            cd1_2 = cpl_table_get_double(refine_wcs, "CD1_2", row, &ignore);
            cd2_1 = cpl_table_get_double(refine_wcs, "CD2_1", row, &ignore);
            cd2_2 = cpl_table_get_double(refine_wcs, "CD2_2", row, &ignore);
            /*
            cpl_msg_info(cpl_func, 
                         "camera %s %s cd1_1 %10.8e cd1_2 %10.8e cd2_1 %10.8e cd2_2 %10.8e",
                         camera, row_cam, cd1_1, cd1_2, cd2_1, cd2_2);
            */
            cpl_propertylist_update_double(limage->plist, "CD1_1", cd1_1);
            cpl_propertylist_update_double(limage->plist, "CD1_2", cd1_2);
            cpl_propertylist_update_double(limage->plist, "CD2_1", cd2_1);
            cpl_propertylist_update_double(limage->plist, "CD2_2", cd2_2);
        }
    }

    if (!match) {
        cpl_error_set_message(cpl_func,
                              CPL_ERROR_ILLEGAL_INPUT,
                              "camera name not matched: %s",
                              camera);
    } else {
        cd1_1 = cpl_propertylist_get_double(limage->plist, "CD1_1");
        cd1_2 = cpl_propertylist_get_double(limage->plist, "CD1_2");
        cd2_1 = cpl_propertylist_get_double(limage->plist, "CD2_1");
        cd2_2 = cpl_propertylist_get_double(limage->plist, "CD2_2");
        cpl_msg_info(cpl_func, "..set CD matrix cd1_1=%10.8e cd1_2=%10.8e "
                     "cd_2_1=%10.8e cd2_2=%10.8e",
                     cd1_1, cd1_2, cd2_1, cd2_2);
    }

 cleanup:
    eris_check_error_code("enu_modify_CD_matrix");
    return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Normalise confidence array so that mean of good pixels is 100.
  @param    confidence  The confidence array
  @return   CPL_ERROR_NONE on success, otherwise a cpl error code

  Fixes problmes in confidence array; bad pixels, negative pixels set to 0.
  Normalises non-zero values to 100. 

 */
/*----------------------------------------------------------------------------*/

cpl_error_code enu_normalise_confidence(cpl_image * confidence) {

    if (cpl_error_get_code() != CPL_ERROR_NONE) return cpl_error_get_code();
    cpl_ensure_code(confidence, CPL_ERROR_NULL_INPUT);

    const cpl_size nx = cpl_image_get_size_x(confidence);
    const cpl_size ny = cpl_image_get_size_y(confidence);
    double * data = cpl_image_get_data_double(confidence);
    /* Avoid creation + checking of empty bpm */
    const cpl_mask * bpm = cpl_image_get_bpm_const(confidence);
    cpl_mask * mask = bpm ? cpl_image_get_bpm(confidence) : NULL;
    cpl_binary * mask_data = bpm ? cpl_mask_get_data(mask) : NULL;

    double sum = 0.0;
    int nonzero = 0; 

    for (cpl_size i = 0; i < nx*ny; i++) {
        if (mask_data != NULL && mask_data[i] == CPL_TRUE) {
            mask_data[i] = CPL_FALSE;
            data[i] = 0.0;
        }
        if (data[i] < 0.0) data[i] = 0.0;
        sum += data[i];
        if (data[i] > 0.0) nonzero++;
    }

    if (sum > 0.0) {
        cpl_image_multiply_scalar(confidence, 100.0 * (double)nonzero / sum); 
    }
    eris_check_error_code("enu_normalise_confidence");
    return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Function to remove read offsets from an himage and set blank pixels  
  @param    himage         The hdrl_image to have offsets removed
  @param    plist          The propertylist associated with the image
  @param    confidence     The confidence array associated with the image or NULL
  @param    set_confidence If true set confidence of blank areas to 0

  @return   CPL_ERROR_NONE if all goes well, otherwise an error code

  There is a small offset on results depending on the readout channel of the
  detector. These 'channels' are 64 pixel wide columns over the detector.

  The 4 pixels at the edge of the detector do not detect photons. Their
  values can be used to estimate the readout channel offset if they fall
  in the readout window. 

  This method estimates the offset for each readout channel and subtracts it
  from the results for that channel. The offset for each channel is the
  average of the edge pixels in the window for that channel, and the error
  on the offset is their standard deviation. There do appear to be some
  hot pixels in the edges so a simple sigma-rejection of outliers is performed
  before calculating the results.

  A border 4 pixels wide around the detector edge is set to 'invalid'
  where this falls inside the detector readout window.

  For the commissioning data there was a DIT sequencing problem that meant the
  bottom borde rows were corrupted for short DITs. See
  https://jira.eso.org/browse/PIPE-10417. This was fixed on about 1-Mar-2023
  so the method depends on the date of the observation, MJD-OBS before of after
  60004.

 */
/*----------------------------------------------------------------------------*/

        
cpl_error_code enu_remove_read_offsets(hdrl_image * himage,
                                       const cpl_propertylist * plist,
                                       cpl_image * confidence,
                                       const int set_confidence) {

    cpl_image * himg  = hdrl_image_get_image(himage);
    cpl_image * eimg  = hdrl_image_get_error(himage);
    double    * hdata = cpl_image_get_data_double(himg);
    double    * edata = cpl_image_get_data_double(eimg);
    cpl_binary* hbpm  = cpl_mask_get_data(cpl_image_get_bpm(himg));
    cpl_binary* ebpm  = cpl_mask_get_data(cpl_image_get_bpm(eimg));
    const int   mx    = cpl_image_get_size_x(himg);
    cpl_boolean is_fast_uncorr = CPL_FALSE;
    if (cpl_error_get_code() != CPL_ERROR_NONE) return cpl_error_get_code();

    /* check that the image data is double */
    cpl_ensure_code(hdata != NULL, CPL_ERROR_UNSUPPORTED_MODE);
    cpl_ensure_code(edata != NULL, CPL_ERROR_UNSUPPORTED_MODE);
    cpl_ensure_code(hbpm  != NULL, CPL_ERROR_UNSUPPORTED_MODE);
    cpl_ensure_code(ebpm  != NULL, CPL_ERROR_UNSUPPORTED_MODE);

    cpl_ensure_code(plist, CPL_ERROR_NULL_INPUT);

    /* get window information */

    cpl_size nx = 0;
    cpl_size ny = 0;
    int rot = 0;
    cpl_size strx = 0;
    cpl_size stry = 0;
    cpl_size nx_chip = 0;
    cpl_size ny_chip = 0;
    cpl_boolean windowed = 0;
    enu_get_window_info(&nx,
                        &ny,
                        &rot,
                        &strx,
                        &stry,
                        &nx_chip,
                        &ny_chip,
                        &windowed,
                        plist);
    enu_check_error_code("failed to read detector window information");

    /* determine whether bottom edge of detector is corrupted by 
       problem with short DITDELAY. Base this on the MJD of the 
       observation - the fix was applied around 1-Mar-2023 = MJD 60004 */

    double mjd = cpl_propertylist_get_double(plist, "MJD-OBS");
    enu_check_error_code("failed to read MJD-OBS");
    if (mjd <= 60004) {
        cpl_msg_info(cpl_func, "PIPE-10417 DITDELAY bug, "
                     "remove_read_offsets will ignore bottom 4 rows "
                     "of chip");   
    }
    const char * curr_name = cpl_propertylist_get_string(plist, "ESO DET READ CURNAME");
    if (strcmp(curr_name, "FAST_UNCORR") == 0) {
    	is_fast_uncorr = CPL_TRUE;
    }
    cpl_size top_width = 4;
    cpl_size bottom_width = 4;
    cpl_msg_debug(cpl_func, 
                  "..offsets from %d top rows and %d bottom rows of chip",
                  (int) top_width, (int) bottom_width);

    cpl_size nread_chan = nx_chip / 64;

    /* Calculate and remove the read offsets */
    /* ..loop through the read channels */
    cpl_vector* voff = cpl_vector_new(nread_chan);
    cpl_size ndata = 0;
    double* pvoff = cpl_vector_get_data(voff);
    for (cpl_size read_chan = 0; read_chan < nread_chan; read_chan++) {

        /* calculate the offsets */

        cpl_array * offset_data = cpl_array_new(0, CPL_TYPE_DOUBLE);
        ndata = 0;

        for (cpl_size x = 64 * read_chan; x < 64 * (read_chan+1); x++) {

            /* are we in the x window */
            /* window/pixel coords are 1-based, C index 0-based  */

            if (x+1 >= strx && x+1 < strx+nx) {

                for (cpl_size y = 0; y < bottom_width; y++) {
 
                    /* are we in the y window */

                    if (y+1 >= stry && y+1 < stry+ny) {
                        const int ipos = x-strx+1 + mx * (y-stry+1);

                        if (!hbpm[ipos]) {
                            if (mjd > 60004) {
                                const double pixval = hdata[ipos];
                                /* bottom channels OK, include them in 
                                   offsets */
                                ndata++;
                                cpl_array_set_size(offset_data, ndata);
                                cpl_array_set(offset_data, ndata-1, pixval);
                            }
                            /* reject the bottom channels for subsequent use */
                            hbpm[ipos] = CPL_BINARY_1;
                            ebpm[ipos] = CPL_BINARY_1;

                            if (set_confidence && confidence) {
                                cpl_image_set(confidence,
                                              x-strx+2,
                                              y-stry+2,
                                              0.0);
                            }
                        }
                    }
                }

                for (cpl_size y = ny_chip-top_width; y < ny_chip; y++) {
                    if (y+1 >= stry && y+1 < stry+ny) {
                        const int ipos = x-strx+1 + mx * (y-stry+1);

                        if (!hbpm[ipos]) {
                            const double pixval = hdata[ipos];
                            ndata++;
                            cpl_array_set_size(offset_data, ndata);
                            cpl_array_set(offset_data, ndata-1, pixval);
                            hbpm[ipos] = CPL_BINARY_1;
                            ebpm[ipos] = CPL_BINARY_1;
                            if (set_confidence && confidence) {
                                cpl_image_set(confidence,
                                              x-strx+2,
                                              y-stry+2,
                                              0.0);
                            }
                        }
                    }
                }
            }
        }

        /* ..Remove the offset */

        hdrl_value offset = {0.0, 0.0};
        if (ndata > 0) {
            cpl_msg_debug(cpl_func, "read_chan %d", (int)read_chan);
            cpl_msg_debug(cpl_func, "ndata %d", (int)ndata);
            cpl_msg_debug(cpl_func, "median %f", cpl_array_get_median(offset_data));
            cpl_msg_debug(cpl_func, "mean %f", cpl_array_get_mean(offset_data));
            cpl_msg_debug(cpl_func, "std %f", cpl_array_get_stdev(offset_data));

            /* simple rejection of biggest outliers, there seem to be some
               hot pixels even in the unused borders of the chip */

            double mean = cpl_array_get_mean(offset_data);
            double stdev = cpl_array_get_stdev(offset_data);
            int nflagged = 0;
            for (cpl_size i = 0; i < cpl_array_get_size(offset_data); i++) {
                int invalid = 0;
                double datum = cpl_array_get_double(offset_data, i, &invalid);
                if (!invalid && fabs(datum - mean) > 5 * stdev) {
                    cpl_array_set_invalid(offset_data, i);
                    nflagged++;
                }
            }

            offset.data = cpl_array_get_mean(offset_data);
            offset.error = cpl_array_get_stdev(offset_data);
            pvoff[read_chan] = offset.data;

            cpl_msg_debug(cpl_func, "..removing offset %f (%f) for read "
                          "channel %d", offset.data, offset.error,
                          (int) read_chan);

            for (cpl_size x = 64 * read_chan; x < 64 * (read_chan+1); x++) {
                for (cpl_size y = 0; y < ny_chip; y++) {

                    if (x+1 >= strx && x+1 < strx+nx &&
                        y+1 >= stry && y+1 < stry+ny) {
                        const int ipos = x-strx+1 + mx * (y-stry+1);
 
                        if (!hbpm[ipos]) {
                            hdata[ipos] -= offset.data;
                            edata[ipos] = hypot(edata[ipos], offset.error);
                        }
                    }
                }
            }
        }
        cpl_array_delete(offset_data);
    }


    /* if fast mode add back offset of offsets */
    if(is_fast_uncorr && ndata > 0) {
    	cpl_image_add_scalar(himg,cpl_vector_get_mean(voff));
    }

    /* reject blanked pixels at left/right edges of detector */

    for (cpl_size y = 0; y < ny_chip; y++) {
        for (cpl_size x = 0; x < 4; x++) {
            /* in window? */
            if (x+1 >= strx && x+1 < strx+nx &&
                y+1 >= stry && y+1 < stry+ny) {
                const int ipos = x-strx+1 + mx * (y-stry+1);

                hbpm[ipos] = CPL_BINARY_1;
                ebpm[ipos] = CPL_BINARY_1;

                if (set_confidence && confidence) {
                    cpl_image_set(confidence,
                                  x-strx+2,
                                  y-stry+2,
                                  0.0);
                }
            }
        }
    }

    for (cpl_size y = 0; y < ny_chip; y++) {
        for (cpl_size x = nx_chip-4; x < nx_chip; x++) {
            /* in window? */
            if (x+1 >= strx && x+1 < strx+nx &&
                y+1 >= stry && y+1 < stry+ny) {

                hdrl_image_reject(himage, 
                                  x-strx+2,
                                  y-stry+2);
                if (set_confidence && confidence) {
                    cpl_image_set(confidence,
                                  x-strx+2,
                                  y-stry+2,
                                  0.0);
                }
            }
        }
    }

    cpl_vector_delete(voff);
    /* re-normalise the confidence array after the changes */

    if (set_confidence && confidence) {
        enu_normalise_confidence(confidence);
    }

 cleanup:
    eris_check_error_code("enu_remove_read_offsets");
    return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Preface a raw filename with a string
  @param    filename     The filename to be re-prefaced
  @param    preface      The new preface
  @return
 */
/*----------------------------------------------------------------------------*/

char * enu_repreface(const char * filename,
                     const char * preface) {

    if (cpl_error_get_code() != CPL_ERROR_NONE) return NULL;

    char * in_name = cpl_strdup(filename);

    /* expect data files to start with either ERIS.20xx or NACO.20xx */
    char * pch = strstr(filename, "ERIS.20");
    if (!pch) {
        pch = strstr(filename, "NACO.20");
    }

    char * outname = cpl_sprintf("%s.%s", preface, pch);
    cpl_free(in_name);

    return outname;
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Get sky backgrounds for a list of target images. 
  @param    method              Method for calculating background from selected sky images
  @param    select_method       Method for selecting sky images relevant to target
  @param    timerange           For method "bracket", the width of the time window
  @param    target_list         List of target images requiring sky backgrounds
  @param    sky_list            List of candidate sky images for calculating backgrounds
  @param    x_probe             x-coord of pixel to print debug info 
  @param    y_probe             y-coord of pixel to print debug info 
  @param    sky_himagelist      Returned list of sky background estimates
  @param    sky_confidence_list Confidence maps for sky backgrounds
  @return   CPL_ERROR_NONE on success, otherwise a cpl error code

 */
/*----------------------------------------------------------------------------*/


cpl_error_code enu_sky_backgrounds(const char * method,
                                   const char * select_method,
                                   const double timerange,
                                   const located_imagelist * target_list,
                                   const located_imagelist * sky_list,
                                   const cpl_size x_probe,
                                   const cpl_size y_probe,
                                   hdrl_imagelist ** sky_himagelist,
                                   cpl_imagelist ** sky_confidence_list) {

    if (cpl_error_get_code() != CPL_ERROR_NONE) return cpl_error_get_code();

    cpl_image      * contrib = NULL;
    hdrl_imagelist * skylist = NULL;
    cpl_image      * sky_confidence = NULL;
    hdrl_image     * sky_himage = NULL;
    cpl_vector     * sky_indeces = NULL;    

    cpl_msg_debug(cpl_func, "enu_sky_backgrounds entry");

    /* parameter validation */

    cpl_ensure_code(method, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(select_method, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(target_list, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(sky_list, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(!strcmp(select_method, "bracket") &&
                    (timerange > 0.0), CPL_ERROR_ILLEGAL_INPUT);

    *sky_himagelist = hdrl_imagelist_new();
    *sky_confidence_list = cpl_imagelist_new();

    /* some variables used for probe pixel debugging */

    int debug = CPL_FALSE;
    cpl_vector * data = NULL;
    cpl_vector * error = NULL;
    cpl_vector * reject = NULL;
    cpl_vector * confidence = NULL;

    cpl_msg_info(cpl_func, "..calculating skys");

    /* loop through target images */

    cpl_size njitters = target_list->size;
    for (cpl_size j = 0; j < njitters; j++) {

        /* Get the indices of relevant sky images */ 

        if (strstr(select_method, "bracket") != NULL) {
            sky_indeces = enu_bracket_skys(target_list->limages[j], timerange,
                                           sky_list, x_probe > -1);
        }
        enu_check_error_code("failure in enu_bracket_skys");

        cpl_size skysize = cpl_vector_get_size(sky_indeces);
        debug = CPL_FALSE;
        if (x_probe > 0 && skysize > 0) {
            const cpl_size x_size = hdrl_image_get_size_x(
                                    sky_list->limages[0]->himage);
            const cpl_size y_size = hdrl_image_get_size_y(
                                    sky_list->limages[0]->himage);
            if (x_probe > 0 && x_probe <= x_size && 
                y_probe > 0 && y_probe <= y_size) {
                debug = CPL_TRUE;
                if (skysize > 0) {
                    data = cpl_vector_new(skysize);
                    error = cpl_vector_new(skysize);
                    reject = cpl_vector_new(skysize);
                    confidence = cpl_vector_new(skysize);
                }
            }
        }

        if (strstr(method, "collapse-median") != NULL) {

            /* construct an hdrl_imagelist of the jitters to be combined,
               accumulate coadd of confidence arrays, apply object_masks 
               if such defined in the located_images */

            skylist = hdrl_imagelist_new();
            for (int k = 0; k < (int)skysize; k++) {
                int i = (int)cpl_vector_get(sky_indeces, k);
                hdrl_image * imcopy = hdrl_image_duplicate(sky_list->
                                      limages[i]->himage);
                cpl_image * confcopy = cpl_image_duplicate(sky_list->
                                       limages[i]->confidence);
                if (sky_list->limages[i]->object_mask != NULL) {
                    cpl_mask_or(cpl_image_get_bpm(hdrl_image_get_image(imcopy)),
                                sky_list->limages[i]->object_mask);
                    cpl_mask_or(cpl_image_get_bpm(confcopy),
                                sky_list->limages[i]->object_mask);
                    cpl_image_fill_rejected(confcopy, 0.0);
                    cpl_image_accept_all(confcopy);
                }

                if (debug) {
                    int rejected = 1;
                    hdrl_value himval = hdrl_image_get_pixel(imcopy,
                                        x_probe, y_probe, &rejected);
                    int ignore = 1;
                    double conf = cpl_image_get(confcopy,
                                                x_probe, y_probe, 
                                                &ignore);
                    cpl_vector_set(data, k, himval.data);
                    cpl_vector_set(error, k, himval.error);
                    cpl_vector_set(reject, k, (double)rejected);
                    cpl_vector_set(confidence, k, conf);
                }

                hdrl_imagelist_set(skylist, imcopy, 
                                   hdrl_imagelist_get_size(skylist));

                if (sky_confidence == NULL) {
                    sky_confidence = confcopy;
                } else {
                    cpl_image_add(sky_confidence, confcopy);
                    cpl_image_delete(confcopy);
                }
            }
            cpl_msg_debug(cpl_func, "after skylist %s",
                          cpl_error_get_message());

            /* Combine the images to give the result */

            hdrl_imagelist_collapse(skylist, HDRL_COLLAPSE_MEDIAN, &sky_himage,
                                    &contrib);
            cpl_image_delete(contrib);
            contrib = NULL;
            cpl_msg_debug(cpl_func, "after collapse %s",
                          cpl_error_get_message());

            /* Re-normalise the confidence array */

            enu_normalise_confidence(sky_confidence);

            if (debug) {
                cpl_msg_info(" ",
                             ".. enu_sky_backgrounds: probe pixel %d,%d",
                             (int)x_probe, (int)y_probe);
                cpl_msg_info(" ", ".. jitter %d" , (int)j);
                cpl_msg_info(" ", "...  j#   q  c       d   (e)");

                for (cpl_size k = 0; k < skysize; k++) { 
                    cpl_msg_info(" ", "... %3d %3d %3d %8.2f(%4.2f)",
                                 (int)cpl_vector_get(sky_indeces, k),
                                 (int)cpl_vector_get(reject, k),
                                 (int)cpl_vector_get(confidence, k),
                                 cpl_vector_get(data, k),
                                 cpl_vector_get(error, k));
                }
                int rejected = 0;
                int ignore = 0;
                hdrl_value hval = hdrl_image_get_pixel(sky_himage,
                                  x_probe, y_probe, &rejected);
                double cval = cpl_image_get(sky_confidence, x_probe,
                                            y_probe, &ignore);
                cpl_msg_info(" ", "... ->  %3d %3d %8.2f(%4.2f)",
                             rejected, (int)cval, hval.data, hval.error);

                cpl_vector_delete(data);
                cpl_vector_delete(error);
                cpl_vector_delete(reject);
                cpl_vector_delete(confidence);
            }

            hdrl_imagelist_delete(skylist); skylist=NULL;
            enu_check_error_code("failure in collapse-median");

        } else if (strstr(method, "median-median") != NULL) {

            /* construct a CPL array of the median of the jitters
               selected */

            cpl_array * sky_medians = cpl_array_new(0, CPL_TYPE_DOUBLE);

            for (int k = 0; k < (int)skysize; k++) {
                int i = (int)cpl_vector_get(sky_indeces, k);

                /* get a copy of the image with the object mask applied */

                hdrl_image * imcopy = hdrl_image_duplicate(
                                      sky_list->limages[i]->himage);
                if (sky_list->limages[i]->object_mask != NULL) {
                    cpl_mask_or(cpl_image_get_bpm(hdrl_image_get_image(imcopy)),
                                sky_list->limages[i]->object_mask);
                }
                hdrl_value bkg = hdrl_image_get_median(imcopy);
                hdrl_image_delete(imcopy);

                cpl_size asize = cpl_array_get_size(sky_medians);
                cpl_array_set_size(sky_medians, asize+1);
                cpl_array_set_double(sky_medians, asize, bkg.data);

                if (debug) {
                    cpl_vector_set(data, k, bkg.data);
                    cpl_vector_set(error, k, bkg.error);
                    cpl_vector_set(confidence, k, 100.0);
                    cpl_vector_set(reject, k, 0.0);
                }
            }

            /* The result is an image filled with the median of sky_medians
               ..the error plane is the stdev of sky_medians */

            double sky_med_med = cpl_array_get_median(sky_medians);
            double sky_med_stdev = cpl_array_get_stdev(sky_medians);
            cpl_size nx = hdrl_image_get_size_x(sky_list->limages[0]->himage);
            cpl_size ny = hdrl_image_get_size_y(sky_list->limages[0]->himage);
            cpl_image * sky_data = cpl_image_new(nx, ny, HDRL_TYPE_DATA);
            cpl_image_fill_window(sky_data, 1, 1, nx, ny, sky_med_med);
            cpl_image * sky_error = cpl_image_new(nx, ny, HDRL_TYPE_ERROR);
            cpl_image_fill_window(sky_error, 1, 1, nx, ny, sky_med_stdev);
            sky_himage = hdrl_image_create(sky_data, sky_error);
            cpl_image_delete(sky_data);
            cpl_image_delete(sky_error);
	    
            sky_confidence = cpl_image_new(nx, ny, CPL_TYPE_DOUBLE);
            cpl_image_fill_window(sky_confidence, 1, 1, nx, ny, 100.0);

            cpl_array_delete(sky_medians);

            if (debug) {
                cpl_msg_info(" ",
                             ".. enu_sky_backgrounds: median of images");
                cpl_msg_info(" ", ".. jitter %d" , (int)j);
                cpl_msg_info(" ", "...  j#       c       d   (e)");

                for (cpl_size k = 0; k < skysize; k++) { 
                    cpl_msg_info(" ", "... %3d     %3d %8.2f(%4.2f)",
                                 (int)cpl_vector_get(sky_indeces, k),
                                 (int)cpl_vector_get(confidence, k),
                                 cpl_vector_get(data, k),
                                 cpl_vector_get(error, k));
                }
                int rejected = 0;
                int ignore = 0;
                hdrl_value hval = hdrl_image_get_pixel(sky_himage,
                                  1, 1, &rejected);
                double cval = cpl_image_get(sky_confidence, 1, 1, &ignore);
                cpl_msg_info(" ", "... ->  %3d %3d %8.2f(%4.2f)",
                             rejected, (int)cval, hval.data, hval.error);

                cpl_vector_delete(data);
                cpl_vector_delete(error);
                cpl_vector_delete(confidence);
            }
            enu_check_error_code("failure in  median-median");
        }

        /* append to lists */

        hdrl_imagelist_set(*sky_himagelist, sky_himage, 
                           hdrl_imagelist_get_size(*sky_himagelist));
        cpl_imagelist_set(*sky_confidence_list, sky_confidence,
                          cpl_imagelist_get_size(*sky_confidence_list));
        cpl_vector_delete(sky_indeces); sky_indeces = NULL;
    }

  cleanup:
    hdrl_imagelist_delete(skylist);
    cpl_vector_delete(sky_indeces);
    if (cpl_error_get_code() != CPL_ERROR_NONE) {
        hdrl_imagelist_delete(*sky_himagelist);
        *sky_himagelist = NULL;
        cpl_imagelist_delete(*sky_confidence_list);
        *sky_confidence_list = NULL;
    }
    eris_check_error_code("enu_sky_backgrounds");
    return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Stack a set of images. 
  @param    limages      List of images to be stacked
  @param    stk_lthr     The lower rejection threshold in call to casu_imstack
  @param    stk_hthr     The upper rejection threshold in call to casu_imstack
  @param    stk_method   The interpolation method in call to casu_imstack
  @param    stk_fast     The 'fast' parameter in call to casu_imstack
  @param    result       The stacked result image
  @return   CPL_ERROR_NONE on success, otherwise a cpl error code

 */
/*----------------------------------------------------------------------------*/

cpl_error_code enu_stack(located_imagelist * limages,
                         const double stk_lthr, const double stk_hthr,
                         const char * stk_method, const int stk_fast,
                         located_image ** result) {
    
    cpl_ensure_code(cpl_error_get_code() == CPL_ERROR_NONE, 
                    cpl_error_get_code());

    casu_fits ** indata = NULL;
    casu_fits ** inconf = NULL;
    casu_fits ** invar = NULL;
    casu_fits  * out = NULL;
    casu_fits  * outc = NULL;
    casu_fits  * outv = NULL;

    /* parameter validation */

    cpl_ensure_code(limages, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code((stk_fast == 0) || (stk_fast == 1),
                    CPL_ERROR_ILLEGAL_INPUT);

    /* parameter translation */

    const int casu_stk_method = (!strcmp(stk_method, "nearest") ? 0 : 1);

    /* translate located_imagelist to casu_fits components */

    encu_limlist_to_casu_fits(limages, &indata, &inconf, &invar);

    /* do the stacking */

    int njitters = (int) limages->size;
    int casu_status = CASU_OK;
    casu_imstack(indata, inconf, invar, NULL, (int)njitters, (int)njitters,
                 stk_lthr, stk_hthr, casu_stk_method, 0, stk_fast, 0,
                 "ESO DET DIT", &out, &outc, &outv, &casu_status);
    *result = enu_located_image_new(hdrl_image_create(out->image, outv->image),
                                    NULL,
                                    cpl_image_duplicate(outc->image), 
                                    NULL, NULL,
                                    out->ehu,
                                    NULL, NULL, NULL, NULL, NULL);
    freespace(out);
    freespace(outc);
    freespace(outv);

    return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Estimate and subtract sky backgrounds for a list of target images 
  @param    method              Method for calculating background from selected sky images
  @param    select_method       Method for selecting sky images relevant to target
  @param    timerange           For method "bracket", the width of the time window
  @param    sky_data            List of candidate sky images for calculating backgrounds
  @param    x_probe             x-coord of pixel to print debug info 
  @param    y_probe             y-coord of pixel to print debug info 
  @param    target_data         List of target images requiring sky background subtraction
  @return   CPL_ERROR_NONE on success, otherwise a cpl error code

 */
/*----------------------------------------------------------------------------*/

cpl_error_code enu_sky_subtract_limlist(const char * method,
                                        const char * select_method,
                                        const double timerange,
                                        const located_imagelist * sky_data,
                                        const cpl_size x_probe,
                                        const cpl_size y_probe,
                                        located_imagelist * target_data) {

    cpl_ensure_code(cpl_error_get_code() == CPL_ERROR_NONE,
                    cpl_error_get_code());
    cpl_ensure_code(method, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(select_method, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(target_data, CPL_ERROR_NULL_INPUT);

    cpl_imagelist         * sky_conf_list = NULL;
    hdrl_imagelist        * sky_himlist = NULL;

    cpl_msg_info(cpl_func, 
                 "enu_sky_subtract called: method=%s, sky_selector=%s, "
                 "bracket=%6.2f",
                 method, select_method, timerange);
    cpl_size njitters = target_data->size;
    cpl_msg_info(cpl_func, "number of jitters to be reduced %d",
                 (int) njitters);

    /* Derive sky backgrounds */

    enu_sky_backgrounds(method, select_method, timerange, target_data,
                        sky_data, x_probe, y_probe, &sky_himlist,
                        &sky_conf_list);
    enu_check_error_code("failure in enu_sky_backgrounds");

    /* Subtract the sky backgrounds */

    hdrl_value before = {0.0, 0.0};
    double bconf = 0.0;
    hdrl_value bkg = {0.0, 0.0};
    double bkgconf = 0.0;
    hdrl_value after = {0.0, 0.0};
    double aconf = 0.0;
    int brejected = 1;
    int bkgrejected = 1;
    int arejected = 1;
    int ignore = 1;
    int header_printed = CPL_FALSE;

    for (cpl_size i = 0; i < njitters; i++) {

        /* probe pixel debug stuff */

        const cpl_size x_size = hdrl_image_get_size_x(
                                target_data->limages[i]->himage);
        const cpl_size y_size = hdrl_image_get_size_y(
                                target_data->limages[i]->himage);
        if (x_probe > 0 && x_probe <= x_size && 
            y_probe > 0 && y_probe <= y_size) {
            before = hdrl_image_get_pixel(target_data->limages[i]->himage,
                                          x_probe, y_probe, &brejected);
            bconf = cpl_image_get(target_data->limages[i]->confidence,
                                  x_probe, y_probe, &ignore);            
        }

        /* this bit does the actual subtraction. */

        hdrl_image_sub_image(target_data->limages[i]->himage,
                             hdrl_imagelist_get_const(sky_himlist, i));
        target_data->limages[i]->bkg = hdrl_image_duplicate(
                                       hdrl_imagelist_get(sky_himlist, i));
        target_data->limages[i]->bkg_confidence = cpl_image_duplicate(
                                       cpl_imagelist_get(sky_conf_list, i));

        /* Confidence array remains that of the target, on the assumption
           that the S/N of the 'sky' is much higher */

        /* Or could do it this way... */
        /* Confidence arrays are combined as if inverse variances */
        /* COMMENTED OUT
        cpl_image * temp = cpl_image_add_create(
                           target_data->limages[i]->confidence,
                           target_data->limages[i]->bkg_confidence);
        cpl_image_multiply(target_data->limages[i]->confidence,
                           target_data->limages[i]->bkg_confidence);
        cpl_image_divide(target_data->limages[i]->confidence, temp);
        cpl_image_delete(temp);
        */

        /* normalise the new confidence */

        cpl_image_fill_rejected(target_data->limages[i]->confidence, 0.0);
        cpl_image_accept_all(target_data->limages[i]->confidence);
        enu_normalise_confidence(target_data->limages[i]->confidence);

        /* more probe pixel debug stuff */

        if (x_probe > 0 && x_probe <= x_size && 
            y_probe > 0 && y_probe <= y_size) {
            after = hdrl_image_get_pixel(target_data->limages[i]->himage,
                                         x_probe, y_probe, &arejected);
            aconf = cpl_image_get(target_data->limages[i]->confidence,
                                  x_probe, y_probe, &ignore);
            bkg = hdrl_image_get_pixel(target_data->limages[i]->bkg,
                                       x_probe, y_probe, &bkgrejected);
            bkgconf = cpl_image_get(target_data->limages[i]->bkg_confidence,
                                    x_probe, y_probe, &ignore);

            if (!header_printed) {
                header_printed = CPL_TRUE;
                cpl_msg_info(" ", "..Subtracting sky backgrounds: probe pixel %d,%d",
                             (int)x_probe, (int)y_probe);
                cpl_msg_info(" ", "   j#           before                       sky                      after");
                cpl_msg_info(" ", "        q  c      d        e        q   c     d        e       q  c       d         e");
            }
            cpl_msg_info(" ", 
                   "..%3d   %d %3.0f %4.2e(%4.2e))   %d %3.0f %4.2e(%4.2e)   %d %3.0f %10.2e(%4.2e)",
                         (int)i,
                         brejected, bconf, before.data, before.error,
                         bkgrejected, bkgconf, bkg.data, bkg.error, 
                         arejected, aconf, after.data, after.error); 
        }
    }

  cleanup:
    cpl_imagelist_delete(sky_conf_list);
    hdrl_imagelist_delete(sky_himlist);

    return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Compute QC on input master bpm
  @param    mbad_pix_map          input bad pixel mask
  @param    qc_list             output qc parameter list
  @param    prefix             input prefix (depends on recipe name)
  @return   cpl_propertylist* with the computed QC params

 */
/*----------------------------------------------------------------------------*/

cpl_error_code
eris_nix_get_badpix_qc_from_ima(const master_bpm* mbad_pix_map,
		            cpl_propertylist* qc_list,
					const char* prefix)
{
	cpl_ensure_code(mbad_pix_map != NULL, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(qc_list != NULL, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(prefix != NULL, CPL_ERROR_NULL_INPUT);

	int flag_mask = 0;
	cpl_mask * bpm_mask = NULL;

	flag_mask = ~flag_mask;
	bpm_mask = en_master_bpm_get_mask(mbad_pix_map, flag_mask);
	int nbad = cpl_mask_count(bpm_mask);
	cpl_propertylist_append_int(qc_list, "ESO QC NUMBER BAD PIXELS", nbad);
	cpl_size sx = cpl_mask_get_size_x(bpm_mask);
	cpl_size sy = cpl_mask_get_size_y(bpm_mask);
	double fraction = (double) nbad / (sx * sy);
	cpl_propertylist_append_double(qc_list, "ESO QC FRACTION BAD PIXELS",fraction);
	return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Compute QC on input raw flats
  @param    lamp_on_limlist     Input raw data
  @param    bp_map_nl_mask      input bad pixel mask (BP_MAP_NL)
  @param    parlist             input recipe parameters
  @param    context             input recipe context (depends on recipe name)
  @param    threshold_pos       input positive threshold to flag pixels above it
  @param    verbose             input parameter to log QC on each input frame
  @return   cpl_propertylist* with the computed QC params

 */
/*----------------------------------------------------------------------------*/

cpl_propertylist*
enu_raw_flats_qc(located_imagelist* lamp_on_limlist,
                 cpl_mask* bp_map_nl_mask,
                 const cpl_parameterlist* parlist,
                 const char* context,
		 const double threshold_pos,
		 const cpl_boolean verbose,
		 const cpl_boolean rescale_by_dit){

    cpl_ensure(lamp_on_limlist != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(bp_map_nl_mask != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(parlist != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(context != NULL, CPL_ERROR_NULL_INPUT, NULL);


    cpl_propertylist* qc;
    //located_image* loc_img;
    cpl_mask* mask;
    double mean = 0;
    double median = 0;
    double rms = 0;
    char* key_name;
    char* key_comm;

    qc = cpl_propertylist_new();
    double threshold_neg = - 4.3e7;
    threshold_neg = 300.;
    long nthresh_pos = 0;
    long nthresh_neg = 0;

    const cpl_parameter* param;
    char* pname;

    /* negative saturation maybe not to be checked as no pixels are sat */
    pname = cpl_sprintf("%s%s",context,".saturation_neg");
    cpl_msg_warning(cpl_func,"pname: %s",pname);
    param = cpl_parameterlist_find_const(parlist,pname);
    threshold_neg = cpl_parameter_get_double(param);
    cpl_free(pname);
    //cpl_msg_info(cpl_func,"threshold_neg: %g",threshold_neg);
    //cpl_msg_info(cpl_func,"threshold_pos: %g",threshold_pos);
    //char* fname;
    cpl_image* ima;
    double mean_min = DBL_MAX;
    double median_min = DBL_MAX;

    double mean_max = -DBL_MIN;
    double median_max = -DBL_MIN;
    double nthresh_pos_max = 0;
    double nthresh_neg_max = 0;
    /*
    cpl_msg_info(cpl_func,"mean_min: %g",mean_min);
    cpl_msg_info(cpl_func,"mean_max: %g",mean_max);
    cpl_msg_info(cpl_func,"median_min: %g",median_min);
    cpl_msg_info(cpl_func,"median_max: %g",median_max);
    */
    double dit = 1;
    cpl_propertylist* plist = NULL;
    for(cpl_size i = 0; i < lamp_on_limlist->size; i++) {
        //fname = cpl_sprintf("raw_ima_%lld.fits",i);
        mask = hdrl_image_get_mask(lamp_on_limlist->limages[i]->himage);
        /*
        cpl_image_save(hdrl_image_get_image(lamp_on_limlist->limages[i]->himage),
                       fname,CPL_TYPE_DOUBLE,NULL,CPL_IO_DEFAULT);
        */

        cpl_mask_or(mask, bp_map_nl_mask);

        hdrl_image_reject_from_mask(lamp_on_limlist->limages[i]->himage, mask);
        ima = hdrl_image_get_image(lamp_on_limlist->limages[i]->himage);
        plist = lamp_on_limlist->limages[i]->plist;
        if(cpl_propertylist_has(plist,"ESO DET SEQ1 DIT")) {
           dit = cpl_propertylist_get_double(plist,"ESO DET SEQ1 DIT");
        }
        nthresh_neg = eris_image_get_threshpix(ima, threshold_neg, CPL_FALSE);
        nthresh_pos = eris_image_get_threshpix(ima, threshold_pos, CPL_TRUE);

        eris_image_flag_threshpix(&ima, threshold_neg, CPL_FALSE);
        mean = cpl_image_get_mean(ima);
        median = cpl_image_get_median(ima);
        rms = cpl_image_get_stdev(ima);
        //cpl_mask_delete(mask);

        //nsat = nthresh_neg + nthresh_pos;
        //nsat = nthresh_neg;
        if(rescale_by_dit) {
        	mean *= dit;
        	median *= dit;
        	rms *= dit;
        }
        if(verbose) {
        	key_name = cpl_sprintf("ESO QC FLAT_ON%lld NPOSSAT",i);
        	key_comm = cpl_sprintf("Number of flat pixels above threshold");
        	cpl_propertylist_append_int(qc, key_name, nthresh_pos);
        	cpl_propertylist_set_comment(qc, key_name, key_comm);
        	cpl_free(key_name);
        	cpl_free(key_comm);

        	key_name = cpl_sprintf("ESO QC FLAT_ON%lld NNEGSAT",i);
        	key_comm = cpl_sprintf("Number of flat pixels below neg threshold");
        	cpl_propertylist_append_int(qc, key_name, nthresh_neg);
        	cpl_propertylist_set_comment(qc, key_name, key_comm);
        	cpl_free(key_name);
        	cpl_free(key_comm);

        	key_name = cpl_sprintf("ESO QC FLAT_ON%lld MEAN",i);
        	key_comm = cpl_sprintf("[ADU] Mean of flat");
        	cpl_propertylist_append_double(qc, key_name, mean);
        	cpl_propertylist_set_comment(qc,key_name,key_comm);
        	cpl_free(key_name);
        	cpl_free(key_comm);

        	key_name = cpl_sprintf("ESO QC FLAT_ON%lld MEDIAN",i);
        	key_comm = cpl_sprintf("[ADU] Median of flat");
        	cpl_propertylist_append_double(qc, key_name, median);
        	cpl_propertylist_set_comment(qc,key_name,key_comm);
        	cpl_free(key_name);
        	cpl_free(key_comm);

        	key_name = cpl_sprintf("ESO QC FLAT_ON%lld STDEV",i);
        	key_comm = cpl_sprintf("[ADU] Stdev of flat");
        	cpl_propertylist_append_double(qc, key_name, rms);
        	cpl_propertylist_set_comment(qc,key_name,key_comm);
        	cpl_free(key_name);
        	cpl_free(key_comm);
        }

        if(mean > mean_max) {
        	mean_max = mean;
        }
        if(median > median_max) {
        	median_max = median;
        }


        if(mean < mean_min) {
        	mean_min = mean;
        }
        if(median < median_min) {
        	median_min = median;
        }


        if(nthresh_pos > nthresh_pos_max) {
        	nthresh_pos_max = nthresh_pos;
        }
        if(nthresh_neg > nthresh_neg_max) {
        	nthresh_neg_max = nthresh_neg;
        }
    }
    cpl_msg_info(cpl_func,"mean_min: %g",mean_min);
    cpl_msg_info(cpl_func,"mean_max: %g",mean_max);
    cpl_msg_info(cpl_func,"median_min: %g",median_min);
    cpl_msg_info(cpl_func,"median_max: %g",median_max);
    cpl_msg_info(cpl_func,"f1: %g",mean_min / mean_max);
    cpl_msg_info(cpl_func,"f2: %g",median_min / median_max);

    key_name = cpl_sprintf("ESO QC FLAT_ON MEAN MIN");
    key_comm = cpl_sprintf("[ADU] Min of Means of flat");
    cpl_propertylist_append_double(qc, key_name, mean_min);
    cpl_propertylist_set_comment(qc,key_name,key_comm);
    cpl_free(key_name);
    cpl_free(key_comm);

    key_name = cpl_sprintf("ESO QC FLAT_ON MEDIAN MIN");
    key_comm = cpl_sprintf("[ADU] Min of Medians of flat");
    cpl_propertylist_append_double(qc, key_name, median_min);
    cpl_propertylist_set_comment(qc,key_name,key_comm);
    cpl_free(key_name);
    cpl_free(key_comm);

    key_name = cpl_sprintf("ESO QC FLAT_ON MEAN MAX");
    key_comm = cpl_sprintf("[ADU] Max of Means of flat");
    cpl_propertylist_append_double(qc, key_name, mean_max);
    cpl_propertylist_set_comment(qc,key_name,key_comm);
    cpl_free(key_name);
    cpl_free(key_comm);

    key_name = cpl_sprintf("ESO QC FLAT_ON MEDIAN MAX");
    key_comm = cpl_sprintf("[ADU] Max of Medians of flat");
    cpl_propertylist_append_double(qc, key_name, median_max);
    cpl_propertylist_set_comment(qc,key_name,key_comm);
    cpl_free(key_name);
    cpl_free(key_comm);


    key_name = cpl_sprintf("ESO QC FLAT_ON MEAN FRAC");
    key_comm = cpl_sprintf("Min/Max of Means of flat");
    cpl_propertylist_append_double(qc, key_name, mean_min / mean_max);
    cpl_propertylist_set_comment(qc,key_name,key_comm);
    cpl_free(key_name);
    cpl_free(key_comm);


    key_name = cpl_sprintf("ESO QC FLAT_ON MEDIAN FRAC");
    key_comm = cpl_sprintf("Min/Max of Medians of flat");
    cpl_propertylist_append_double(qc, key_name, median_min / median_max);
    cpl_propertylist_set_comment(qc,key_name,key_comm);
    cpl_free(key_name);
    cpl_free(key_comm);



    key_name = cpl_sprintf("ESO QC FLAT_ON NPOSSAT MAX");
    key_comm = cpl_sprintf("Max of Number of flat pixels above threshold");
    cpl_propertylist_append_int(qc, key_name, nthresh_pos_max);
    cpl_propertylist_set_comment(qc, key_name, key_comm);
    cpl_free(key_name);
    cpl_free(key_comm);

    key_name = cpl_sprintf("ESO QC FLAT_ON NNEGSAT MAX");
    key_comm = cpl_sprintf("MAX Number of flat pixels below neg threshold");
    cpl_propertylist_append_int(qc, key_name, nthresh_neg_max);
    cpl_propertylist_set_comment(qc, key_name, key_comm);
    cpl_free(key_name);
    cpl_free(key_comm);



    eris_check_error_code("enu_raw_flats_qc");
    return qc;
}
/**@}*/
