
/*                                                                           *
 *   This file is part of the ESO X-shooter Pipeline                         *
 *   Copyright (C) 2006 European Southern Observatory                        *
 *                                                                           *
 *   This library is free software; you can redistribute it and/or modify    *
 *   it under the terms of the GNU General Public License as published by    *
 *   the Free Software Foundation; either version 2 of the License, or       *
 *   (at your option) any later version.                                     *
 *                                                                           *
 *   This program is distributed in the hope that it will be useful,         *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of          *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the           *
 *   GNU General Public License for more details.                            *
 *                                                                           *
 *   You should have received a copy of the GNU General Public License       *
 *   along with this program; if not, write to the Free Software             *
 *   Foundation, 51 Franklin St, Fifth Floor, Boston, MA  02111-1307  USA    *
 *                                                                           */

/*
 * $Author: amodigli $
 * $Date: 2013-03-13 12:35:04 $
 * $Revision: 1.87 $
 * $Name: not supported by cvs2svn $
 */

#include <eris_utils_image.h>
/**
 * @brief  Compute X Scharr filter transformation.
 * @param  in input image
 * @return X scarr filter image  
 * @ref http://en.wikipedia.org/wiki/Sobel_operator
*/
cpl_image*
eris_scharr_x(cpl_image* in) {

  cpl_image* scharr_x = NULL;
  double* pscharr_x = NULL;
  double* pin = NULL;


  scharr_x = cpl_image_duplicate(in);
  pscharr_x = cpl_image_get_data_double(scharr_x);
  pin = cpl_image_get_data_double(in);
  cpl_size sx = cpl_image_get_size_x(in);
  cpl_size sy = cpl_image_get_size_y(in);

  for(cpl_size i = 1; i < sx - 1;i++) {
    for(cpl_size j = 1; j < sy - 1;j++) {
      pscharr_x[i + j*sx] = 3 * pin[i-1 + (j+1)*sx] - 3 * pin[i + 1+ (j+1)*sx]+
                10 * pin[i - 1 + j*sx] - 10 * pin[i + 1 + j*sx]+
	          3 * pin[i - 1 + (j-1) * sx] - 3 * pin[i + 1 + (j-1) * sx];
    }
  }


  return scharr_x;

}



/**
 * @brief  Compute Y Scharr filter transformation.
 * @param  in input image
 * @return Y scharr filter image  
 * @ref http://en.wikipedia.org/wiki/Sobel_operator
*/
cpl_image*
eris_scharr_y(cpl_image* in) {

  cpl_image* scharr_y=NULL;
  double* pscharr_y=NULL;
  double* pin=NULL;

  scharr_y = cpl_image_duplicate(in);
  pscharr_y = cpl_image_get_data_double(scharr_y) ;
  pin = cpl_image_get_data_double(in);
  cpl_size sx = cpl_image_get_size_x(in) ;
  cpl_size sy = cpl_image_get_size_y(in);

  for(cpl_size i = 1;i < sx-1;i++) {
    for(cpl_size j = 1;j < sy-1;j++) {
      pscharr_y[i+j*sx]=3*pin[i-1+(j+1)*sx]+10*pin[i+(j+1)*sx]+3*pin[i+1+(j+1)*sx]+
	         -3*pin[i-1+(j-1)*sx]-10*pin[i+(j-1)*sx]-3*pin[i+1+(j-1)*sx];
    }
  }

  return scharr_y;

}


/**
 * @brief  Compute X Sobel filter transformation.
 * @param  in input image
 * @return X scharr filter image  
 * @ref http://en.wikipedia.org/wiki/Sobel_operator
*/

cpl_image*
eris_sobel_lx(cpl_image* in) {

  cpl_image* lx = NULL;
  double* plx = NULL;
  double* pin = NULL;

  lx = cpl_image_duplicate(in);
  plx = cpl_image_get_data_double(lx);
  pin = cpl_image_get_data_double(in);
  cpl_size sx = cpl_image_get_size_x(in);
  cpl_size sy = cpl_image_get_size_y(in);

  for(cpl_size i=1;i<sx-1;i++) {
    for(cpl_size j=1;j<sy-1;j++) {
      plx[i+j*sx]=pin[i-1+(j+1)*sx]-pin[i+1+(j+1)*sx]+
                2*pin[i-1+j*sx]-2*pin[i+1+j*sx]+
	          pin[i-1+(j-1)*sx]-pin[i+1+(j-1)*sx];
    }
  }

  return lx;

}



/**
 * @brief  Compute Y Sobel filter transformation.
 * @param  in input image
 * @return Y scharr filter image  
 * @ref http://en.wikipedia.org/wiki/Sobel_operator
*/

cpl_image*
eris_sobel_ly(cpl_image* in) {



  cpl_image* ly=NULL;
  double* ply=NULL;
  double* pin=NULL;



  ly = cpl_image_duplicate(in);
  ply = cpl_image_get_data_double(ly);
  pin = cpl_image_get_data_double(in);
  cpl_size sx = cpl_image_get_size_x(in);
  cpl_size sy = cpl_image_get_size_y(in);

  for(cpl_size i=1;i<sx-1;i++) {
    for(cpl_size j=1;j<sy-1;j++) {
      ply[i+j*sx]=pin[i-1+(j+1)*sx]+2*pin[i+(j+1)*sx]+pin[i+1+(j+1)*sx]+
	         -pin[i-1+(j-1)*sx]-2*pin[i+(j-1)*sx]-pin[i+1+(j-1)*sx];
    }
  }

  return ly;

}
