/*                                                                              *
 *   This file is part of the ESO IRPLIB package                                *
 *   Copyright (C) 2004,2005 European Southern Observatory                      *
 *                                                                              *
 *   This library is free software; you can redistribute it and/or modify       *
 *   it under the terms of the GNU General Public License as published by       *
 *   the Free Software Foundation; either version 2 of the License, or          *
 *   (at your option) any later version.                                        *
 *                                                                              *
 *   This program is distributed in the hope that it will be useful,            *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of             *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the              *
 *   GNU General Public License for more details.                               *
 *                                                                              *
 *   You should have received a copy of the GNU General Public License          *
 *   along with this program; if not, write to the Free Software                *
 *   Foundation, 51 Franklin St, Fifth Floor, Boston, MA  02111-1307  USA       *
 *                                                                              */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*----------------------------------------------------------------------------
                                Includes
 ----------------------------------------------------------------------------*/

#include <detmon.h>
#include <detmon_lg.h>
#include <detmon_lg_impl.h>


#include <math.h>
/*---------------------------------------------------------------------------*/
/*
 * @defgroup detmon_test   Testing of the IRPLIB utilities
 */
/*---------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------
                                Defines
 ----------------------------------------------------------------------------*/

#ifndef IMAGESIZE
#define IMAGESIZE       64
#endif

#define IMAGELISTSIZE   10

#define NBADPIXS        10

#define ON_LEVEL        300.0
#define OFF_LEVEL       100.0

#define NOISE_MIN       10.0
#define NOISE_MAX       40.0

#define ON TRUE
#define OFF FALSE


/*----------------------------------------------------------------------------
                            Private Function prototypes
 ----------------------------------------------------------------------------*/

static void detmon_lg_tests(void);
static void detmon_check_order_test(void);
cpl_frameset * detmon_tests_fill_frameset(const char     * tag_on,
                                                 const char     * tag_off,
                                                 int              non,
                                                 cpl_boolean      opt_nir,
                                                 cpl_boolean      with_exts,
                                                 int              nexts,
                                                 int              nsettings);

cpl_frameset * detmon_tests_fill_frameset_oct(const char         * tag_on,
                                                 const char     * tag_off,
                                                 int              non,
                                                 cpl_boolean      opt_nir,
                                                 cpl_boolean      with_exts,
                                                     int              nexts);

int compare_settings(const cpl_frame * frame1, const cpl_frame * frame2);

cpl_imagelist * detmon_tests_imlist(int ndits, cpl_boolean onoff);

/*---------------------------------------------------------------------------*/
/*
 * @brief   Unit tests of fit module
 */
/*---------------------------------------------------------------------------*/

int main(void)
{
    /* Initialize CPL + IRPLIB */
    cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_WARNING);

    detmon_lg_tests();
    detmon_check_order_test();
    return cpl_test_end(0);
}

static void detmon_lg_tests(void)
{

    /* Variables used for correlation function */
    cpl_size              px, py;
    cpl_image           * image1;
#ifdef HAVE_SFFTW_NOT_USED
    cpl_image           * image2;
#endif
    cpl_image           * result;

    cpl_parameterlist   * parlist;
    cpl_frameset        * frameset;
    cpl_frameset        * set;
    cpl_propertylist    * prolist;

    /* Variables used for detmon_ir_lg function */
    const char                * tag_on;
    const char                * tag_off;


#ifdef HAVE_SFFTW_NOT_USED
    /* CORRELATION */

    /* 1. trial: all inputs illegal, NULL pointers */
    result = detmon_image_correlate(NULL, NULL, 0, 0);
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    cpl_error_reset();
    cpl_test_null(result);
    cpl_image_delete(result);

    /* 2. trial: 2 equal and symmetrical images */
    image1 = cpl_image_new(IMAGESIZE, IMAGESIZE, CPL_TYPE_FLOAT);
    cpl_image_fill_gaussian(image1, (IMAGESIZE + 1)/2, (IMAGESIZE + 1)/2, 25, 40, 40);

    image2 = cpl_image_new(IMAGESIZE, IMAGESIZE, CPL_TYPE_FLOAT);
    cpl_image_fill_gaussian(image2, (IMAGESIZE + 1)/2, (IMAGESIZE + 1)/2, 25, 40, 40);

    px = -1;
    py = -1;

    result = detmon_image_correlate(image1, image2, 13, 13);

    cpl_image_get_maxpos(result, &px, &py);
    cpl_test(px == 13+1 && py == 13+1);

    cpl_image_delete(image1);
    cpl_image_delete(image2);
    cpl_image_delete(result);

    /* 3. trial: 2 equal but not symmetrical images */
    image1 = cpl_image_new(IMAGESIZE, IMAGESIZE, CPL_TYPE_FLOAT);
    cpl_image_fill_gaussian(image1, (IMAGESIZE + 1)/2+30, (IMAGESIZE + 1)/2+30, 25, 40, 40);

    image2 = cpl_image_new(IMAGESIZE, IMAGESIZE, CPL_TYPE_FLOAT);
    cpl_image_fill_gaussian(image2, (IMAGESIZE + 1)/2+30, (IMAGESIZE + 1)/2+30, 25, 40, 40);

    px = -1;
    py = -1;

    result = detmon_image_correlate(image1, image2, 13, 13);

    cpl_image_get_maxpos(result, &px, &py);
    cpl_test(px == 13+1 && py == 13+1);

    cpl_image_delete(image1);
    cpl_image_delete(image2);
    cpl_image_delete(result);

    /* 4. trial: 2 different images */
    image1 = cpl_image_new(IMAGESIZE, IMAGESIZE, CPL_TYPE_FLOAT);
    cpl_image_fill_gaussian(image1, (IMAGESIZE + 1)/2+30, (IMAGESIZE + 1)/2+30, 25, 40, 40);

    image2 = cpl_image_new(IMAGESIZE, IMAGESIZE, CPL_TYPE_FLOAT);
    cpl_image_fill_gaussian(image2, (IMAGESIZE + 1)/2+40, (IMAGESIZE + 1)/2+40, 25, 40, 40);

    px = -1;
    py = -1;

    result = detmon_image_correlate(image1, image2, 13, 13);

    cpl_image_get_maxpos(result, &px, &py);

    /*
     * FIXME:
     * cpl_test(px == 13+11 && py == 13+11);
     */
    cpl_image_delete(image1);
    cpl_image_delete(image2);
    cpl_image_delete(result);
#else
    /* CORRELATION with cpl_image_fft() */

    /* 1. trial: all inputs illegal, NULL pointers */
    result = detmon_autocorrelate(NULL, 0, 0);
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    cpl_error_reset();
    cpl_test_null(result);
    cpl_image_delete(result);

    /* 2. trial: 2 equal and symmetrical images */
    image1 = cpl_image_new(IMAGESIZE, IMAGESIZE, CPL_TYPE_FLOAT);
    cpl_image_fill_gaussian(image1, (IMAGESIZE + 1)/2, (IMAGESIZE + 1)/2, 25, 40, 40);

    px = -1;
    py = -1;

    result = detmon_autocorrelate(image1, 13, 13);

    cpl_image_get_maxpos(result, &px, &py);
    cpl_test(px == 13+1 && py == 13+1);

    cpl_image_delete(image1);
    cpl_image_delete(result);

#endif

    /* HIGH LEVEL FUNCTION FOR LINEARITY/GAIN RECIPE */

    tag_on  = "FRAME_ON";
    tag_off = "FRAME_OFF";

    set = detmon_tests_fill_frameset(tag_on, tag_off,
					    20, TRUE, FALSE, 0, 1);

    prolist = detmon_fill_prolist("PROCATG", "TYPE", "TECH", CPL_TRUE);

    /* 1. test: All inputs illegal, NULL pointers */
    detmon_lg(NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,
		     NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, TRUE);
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    cpl_error_reset();


    /* 2. test: Default use, NIR case */
    parlist = cpl_parameterlist_new();
    detmon_lg_fill_parlist_nir_default(parlist,
					      "detmon_ir_lg", "detmon");

    frameset = cpl_frameset_duplicate(set);

    detmon_lg(frameset, parlist, tag_on, tag_off, "detmon_ir_lg",
		     "detmon", REGEXP, prolist, prolist, prolist,
		     prolist, prolist, prolist, "DETMON / 1.0",
		     NULL, NULL, TRUE);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_error_reset();

    cpl_frameset_delete(frameset);
    cpl_parameterlist_delete(parlist);


    /* 3. test: NIR case, with different llx, lly, urx, ury */
    parlist = cpl_parameterlist_new();
    detmon_lg_fill_parlist(parlist, "detmon_ir_lg", "detmon",
			      "PTC",
                            3,                /* --order         */
                            3,                /* --kappa         */
                            25,               /* --niter         */
                            8,               /* --llx           */
                            8,               /* --lly           */
                            56,               /* --urx           */
                            56,               /* --ury           */
                            10000,            /* --ref_level     */
                            "CPL_FALSE",      /* --intermediate  */
                            "CPL_FALSE",      /* --autocorr      */
                            "CPL_FALSE",      /* --collapse      */
                            "CPL_TRUE",       /* --rescale       */
                            "CPL_TRUE",       /* --pix2pix       */
                            "CPL_FALSE",       /* --bpmbin       */
                              -1,             /* --filter        */
                            26,               /* --m             */
                            26,               /* --n             */
				  1e-3,
                            "CPL_TRUE",       /* --pafgen       */
				  "detmon_ir_lg",
                            -1,               /* --llx1          */
                            -1,               /* --lly1          */
                            -1,               /* --urx1          */
                            -1,               /* --ury1          */
                            -1,               /* --llx2          */
                            -1,               /* --lly2          */
                            -1,               /* --urx2          */
                            -1,               /* --ury2          */
                            -1,               /* --llx3          */
                            -1,               /* --lly3          */
                            -1,               /* --urx3          */
                            -1,               /* --ury3          */
                            -1,               /* --llx4          */
                            -1,               /* --lly4          */
                            -1,               /* --urx4          */
                            -1,               /* --ury4          */
                            -1,               /* --llx5          */
                            -1,               /* --lly5          */
                            -1,               /* --urx5          */
                            -1,               /* --ury5          */
				  0,               /* --exts          */
    NIR);

    frameset = cpl_frameset_duplicate(set);

    detmon_lg(frameset, parlist, tag_on, tag_off, "detmon_ir_lg",
		     "detmon", REGEXP, prolist, prolist, prolist,
		     prolist, prolist, prolist, "DETMON / 1.0",
		     NULL, NULL, TRUE);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_error_reset();

    cpl_frameset_delete(frameset);
    cpl_parameterlist_delete(parlist);

    /* 3. test: NIR case, MED method applied for GAIN */
    parlist = cpl_parameterlist_new();
    detmon_lg_fill_parlist(parlist, "detmon_ir_lg", "detmon",
			      "MED",
                            3,                /* --order         */
                            3,                /* --kappa         */
                            25,               /* --niter         */
                            -1,               /* --llx           */
                            -1,               /* --lly           */
                            -1,               /* --urx           */
                            -1,               /* --ury           */
                            10000,            /* --ref_level     */
                            "CPL_FALSE",      /* --intermediate  */
                            "CPL_FALSE",      /* --autocorr      */
                            "CPL_FALSE",      /* --collapse      */
                            "CPL_TRUE",       /* --rescale       */
                            "CPL_TRUE",       /* --pix2pix       */
                            "CPL_FALSE",       /* --bpmbin       */
                              -1,             /* --filter        */
                            26,               /* --m             */
                            26,               /* --n             */
				  1e-3,
                            "CPL_TRUE",       /* --pafgen       */
				  "detmon_ir_lg",
                            -1,               /* --llx1          */
                            -1,               /* --lly1          */
                            -1,               /* --urx1          */
                            -1,               /* --ury1          */
                            -1,               /* --llx2          */
                            -1,               /* --lly2          */
                            -1,               /* --urx2          */
                            -1,               /* --ury2          */
                            -1,               /* --llx3          */
                            -1,               /* --lly3          */
                            -1,               /* --urx3          */
                            -1,               /* --ury3          */
                            -1,               /* --llx4          */
                            -1,               /* --lly4          */
                            -1,               /* --urx4          */
                            -1,               /* --ury4          */
                            -1,               /* --llx5          */
                            -1,               /* --lly5          */
                            -1,               /* --urx5          */
                            -1,               /* --ury5          */
				  0,               /* --exts          */
    NIR);

    frameset = cpl_frameset_duplicate(set);

    detmon_lg(frameset, parlist, tag_on, tag_off, "detmon_ir_lg",
		     "detmon", REGEXP, prolist, prolist, prolist,
		     prolist, prolist, prolist, "DETMON / 1.0",
		     NULL, NULL, TRUE);
    //cpl_test_error(CPL_ERROR_NONE);
    cpl_error_reset();

    cpl_frameset_delete(frameset);
    cpl_parameterlist_delete(parlist);

    /*FIXME: The products should be checked to see that MED has been in fact
      applied (not only that it finished without errors) */

    /* 4. test: NIR case, --intermediate activated */
    parlist = cpl_parameterlist_new();
    detmon_lg_fill_parlist(parlist, "detmon_ir_lg", "detmon",
			      "PTC",
                            3,                /* --order         */
                            3,                /* --kappa         */
                            25,               /* --niter         */
                            -1,               /* --llx           */
                            -1,               /* --lly           */
                            -1,               /* --urx           */
                            -1,               /* --ury           */
                            10000,            /* --ref_level     */
                            "CPL_TRUE",      /* --intermediate  */
                            "CPL_FALSE",      /* --autocorr      */
                            "CPL_FALSE",      /* --collapse      */
                            "CPL_TRUE",       /* --rescale       */
                            "CPL_TRUE",       /* --pix2pix       */
                            "CPL_FALSE",       /* --bpmbin       */
                              -1,             /* --filter        */
                            26,               /* --m             */
                            26,               /* --n             */
				  1e-3,
                            "CPL_TRUE",       /* --pafgen       */
				  "detmon_ir_lg",
                            -1,               /* --llx1          */
                            -1,               /* --lly1          */
                            -1,               /* --urx1          */
                            -1,               /* --ury1          */
                            -1,               /* --llx2          */
                            -1,               /* --lly2          */
                            -1,               /* --urx2          */
                            -1,               /* --ury2          */
                            -1,               /* --llx3          */
                            -1,               /* --lly3          */
                            -1,               /* --urx3          */
                            -1,               /* --ury3          */
                            -1,               /* --llx4          */
                            -1,               /* --lly4          */
                            -1,               /* --urx4          */
                            -1,               /* --ury4          */
                            -1,               /* --llx5          */
                            -1,               /* --lly5          */
                            -1,               /* --urx5          */
                            -1,               /* --ury5          */
				  0,               /* --exts          */
    NIR);

    frameset = cpl_frameset_duplicate(set);

    detmon_lg(frameset, parlist, tag_on, tag_off, "detmon_ir_lg",
		     "detmon", REGEXP, prolist, prolist, prolist,
		     prolist, prolist, prolist, "DETMON / 1.0",
		     NULL, NULL, TRUE);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_error_reset();

    cpl_frameset_delete(frameset);
    cpl_parameterlist_delete(parlist);

    /*FIXME: Check that the new intermediate have been in fact created */

    /* 5. test: NIR case, --rescale deactivated */
    parlist = cpl_parameterlist_new();
    detmon_lg_fill_parlist(parlist, "detmon_ir_lg", "detmon",
			      "PTC",
                            3,                /* --order         */
                            3,                /* --kappa         */
                            25,               /* --niter         */
                            -1,               /* --llx           */
                            -1,               /* --lly           */
                            -1,               /* --urx           */
                            -1,               /* --ury           */
                            10000,            /* --ref_level     */
                            "CPL_FALSE",      /* --intermediate  */
                            "CPL_FALSE",      /* --autocorr      */
                            "CPL_FALSE",      /* --collapse      */
                            "CPL_FALSE",       /* --rescale       */
                            "CPL_TRUE",       /* --pix2pix       */
                            "CPL_FALSE",       /* --bpmbin       */
                              -1,             /* --filter        */
                            26,               /* --m             */
                            26,               /* --n             */
				  1e-3,
                            "CPL_TRUE",       /* --pafgen       */
				  "detmon_ir_lg",
                            -1,               /* --llx1          */
                            -1,               /* --lly1          */
                            -1,               /* --urx1          */
                            -1,               /* --ury1          */
                            -1,               /* --llx2          */
                            -1,               /* --lly2          */
                            -1,               /* --urx2          */
                            -1,               /* --ury2          */
                            -1,               /* --llx3          */
                            -1,               /* --lly3          */
                            -1,               /* --urx3          */
                            -1,               /* --ury3          */
                            -1,               /* --llx4          */
                            -1,               /* --lly4          */
                            -1,               /* --urx4          */
                            -1,               /* --ury4          */
                            -1,               /* --llx5          */
                            -1,               /* --lly5          */
                            -1,               /* --urx5          */
                            -1,               /* --ury5          */
				  0,               /* --exts          */
    NIR);

    frameset = cpl_frameset_duplicate(set);

    detmon_lg(frameset, parlist, tag_on, tag_off, "detmon_ir_lg",
		     "detmon", REGEXP, prolist, prolist, prolist,
		     prolist, prolist, prolist, "DETMON / 1.0",
		     NULL, NULL, TRUE);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_error_reset();

    cpl_frameset_delete(frameset);
    cpl_parameterlist_delete(parlist);

    /* 5a. test: NIR case, --bpmbin activated */
    parlist = cpl_parameterlist_new();
    detmon_lg_fill_parlist(parlist, "detmon_ir_lg", "detmon",
			      "PTC",
                            3,                /* --order         */
                            3,                /* --kappa         */
                            25,               /* --niter         */
                            -1,               /* --llx           */
                            -1,               /* --lly           */
                            -1,               /* --urx           */
                            -1,               /* --ury           */
                            10000,            /* --ref_level     */
                            "CPL_FALSE",      /* --intermediate  */
                            "CPL_FALSE",      /* --autocorr      */
                            "CPL_FALSE",      /* --collapse      */
                            "CPL_TRUE",       /* --rescale       */
                            "CPL_TRUE",       /* --pix2pix       */
                            "CPL_TRUE",       /* --bpmbin       */
                              -1,             /* --filter        */
                            26,               /* --m             */
                            26,               /* --n             */
				  1e-3,
                            "CPL_TRUE",       /* --pafgen       */
				  "detmon_ir_lg",
                            -1,               /* --llx1          */
                            -1,               /* --lly1          */
                            -1,               /* --urx1          */
                            -1,               /* --ury1          */
                            -1,               /* --llx2          */
                            -1,               /* --lly2          */
                            -1,               /* --urx2          */
                            -1,               /* --ury2          */
                            -1,               /* --llx3          */
                            -1,               /* --lly3          */
                            -1,               /* --urx3          */
                            -1,               /* --ury3          */
                            -1,               /* --llx4          */
                            -1,               /* --lly4          */
                            -1,               /* --urx4          */
                            -1,               /* --ury4          */
                            -1,               /* --llx5          */
                            -1,               /* --lly5          */
                            -1,               /* --urx5          */
                            -1,               /* --ury5          */
				  0,               /* --exts          */
    NIR);

    frameset = cpl_frameset_duplicate(set);

    detmon_lg(frameset, parlist, tag_on, tag_off, "detmon_ir_lg",
		     "detmon", REGEXP, prolist, prolist, prolist,
		     prolist, prolist, prolist, "DETMON / 1.0",
		     NULL, NULL, TRUE);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_error_reset();

    {
	/* Check that the image is in fact binary */
	cpl_image * bpm = cpl_image_load("detmon_ir_lg_bpm.fits",
					 CPL_TYPE_INT, 0, 0);

	cpl_mask * mask = cpl_mask_threshold_image_create(bpm, 0.5, 1.5);

	int nbpixs = cpl_mask_count(mask);

	int flux = (int)cpl_image_get_flux(bpm);

	cpl_test_eq(flux, nbpixs);

	cpl_image_delete(bpm);
	cpl_mask_delete(mask);
    }

    cpl_frameset_delete(frameset);
    cpl_parameterlist_delete(parlist);

    /* Is there any way, looking into the products, to know if the --rescale
       has been in fact deactivated? Other health checkings should be done
       here to prove that the deactivation doesn't compromise correctness */

    /* 6. test: NIR case, --pix2pix deactivated */
    parlist = cpl_parameterlist_new();
    detmon_lg_fill_parlist(parlist, "detmon_ir_lg", "detmon",
			      "PTC",
                            3,                /* --order         */
                            3,                /* --kappa         */
                            25,               /* --niter         */
                            -1,               /* --llx           */
                            -1,               /* --lly           */
                            -1,               /* --urx           */
                            -1,               /* --ury           */
                            10000,            /* --ref_level     */
                            "CPL_FALSE",      /* --intermediate  */
                            "CPL_FALSE",      /* --autocorr      */
                            "CPL_FALSE",      /* --collapse      */
                            "CPL_TRUE",       /* --rescale       */
                            "CPL_FALSE",       /* --pix2pix       */
                            "CPL_FALSE",       /* --bpmbin       */
                              -1,             /* --filter        */
                            26,               /* --m             */
                            26,               /* --n             */
				  1e-3,
                            "CPL_TRUE",       /* --pafgen       */
				  "detmon_ir_lg",
                            -1,               /* --llx1          */
                            -1,               /* --lly1          */
                            -1,               /* --urx1          */
                            -1,               /* --ury1          */
                            -1,               /* --llx2          */
                            -1,               /* --lly2          */
                            -1,               /* --urx2          */
                            -1,               /* --ury2          */
                            -1,               /* --llx3          */
                            -1,               /* --lly3          */
                            -1,               /* --urx3          */
                            -1,               /* --ury3          */
                            -1,               /* --llx4          */
                            -1,               /* --lly4          */
                            -1,               /* --urx4          */
                            -1,               /* --ury4          */
                            -1,               /* --llx5          */
                            -1,               /* --lly5          */
                            -1,               /* --urx5          */
                            -1,               /* --ury5          */
				  0,               /* --exts          */
    NIR);

    frameset = cpl_frameset_duplicate(set);

/*    detmon_lg(frameset, parlist, tag_on, tag_off, "detmon_ir_lg",
		     "detmon", REGEXP, prolist, prolist, prolist,
		     prolist, prolist, prolist, "DETMON / 1.0",
		     NULL, NULL, TRUE);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_error_reset();
*/
    cpl_frameset_delete(frameset);
    cpl_parameterlist_delete(parlist);

    /* Should check at least that there is no data cube */

    /* 7. test: NIR case, --filter activated (4000) */
    parlist = cpl_parameterlist_new();
    detmon_lg_fill_parlist(parlist, "detmon_ir_lg", "detmon",
			      "PTC",
                            3,                /* --order         */
                            3,                /* --kappa         */
                            25,               /* --niter         */
                            -1,               /* --llx           */
                            -1,               /* --lly           */
                            -1,               /* --urx           */
                            -1,               /* --ury           */
                            10000,            /* --ref_level     */
                            "CPL_FALSE",      /* --intermediate  */
                            "CPL_FALSE",      /* --autocorr      */
                            "CPL_FALSE",      /* --collapse      */
                            "CPL_TRUE",       /* --rescale       */
                            "CPL_TRUE",       /* --pix2pix       */
                            "CPL_FALSE",       /* --bpmbin       */
                              4000,             /* --filter        */
                            26,               /* --m             */
                            26,               /* --n             */
				  1e-3,
                            "CPL_TRUE",       /* --pafgen       */
				  "detmon_ir_lg",
                            -1,               /* --llx1          */
                            -1,               /* --lly1          */
                            -1,               /* --urx1          */
                            -1,               /* --ury1          */
                            -1,               /* --llx2          */
                            -1,               /* --lly2          */
                            -1,               /* --urx2          */
                            -1,               /* --ury2          */
                            -1,               /* --llx3          */
                            -1,               /* --lly3          */
                            -1,               /* --urx3          */
                            -1,               /* --ury3          */
                            -1,               /* --llx4          */
                            -1,               /* --lly4          */
                            -1,               /* --urx4          */
                            -1,               /* --ury4          */
                            -1,               /* --llx5          */
                            -1,               /* --lly5          */
                            -1,               /* --urx5          */
                            -1,               /* --ury5          */
				  0,               /* --exts          */
    NIR);

    frameset = cpl_frameset_duplicate(set);

    detmon_lg(frameset, parlist, tag_on, tag_off, "detmon_ir_lg",
		     "detmon", REGEXP, prolist, prolist, prolist,
		     prolist, prolist, prolist, "DETMON / 1.0",
		     NULL, NULL, TRUE);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_error_reset();

    cpl_frameset_delete(frameset);
    cpl_parameterlist_delete(parlist);

    /*FIXME: Move this deletion more to the end if standard set used
      for more tests */
    cpl_frameset_delete(set);

    /* Check that there are less entries into the table.
     We should know for sure how many entries are going to be!
    To avoid other errors as not having enough points for fitting!*/

    /* 8. test: Check that it fails when not enough frames for order-degree
       fitting */
    parlist = cpl_parameterlist_new();
    detmon_lg_fill_parlist(parlist, "detmon_ir_lg", "detmon",
			      "PTC",
                            3,                /* --order         */
                            3,                /* --kappa         */
                            25,               /* --niter         */
                            -1,               /* --llx           */
                            -1,               /* --lly           */
                            -1,               /* --urx           */
                            -1,               /* --ury           */
                            10000,            /* --ref_level     */
                            "CPL_FALSE",      /* --intermediate  */
                            "CPL_FALSE",      /* --autocorr      */
                            "CPL_FALSE",      /* --collapse      */
                            "CPL_TRUE",       /* --rescale       */
                            "CPL_TRUE",       /* --pix2pix       */
                            "CPL_FALSE",       /* --bpmbin       */
                              -1,             /* --filter        */
                            26,               /* --m             */
                            26,               /* --n             */
				  1e-3,
                            "CPL_TRUE",       /* --pafgen       */
				  "detmon_ir_lg",
                            -1,               /* --llx1          */
                            -1,               /* --lly1          */
                            -1,               /* --urx1          */
                            -1,               /* --ury1          */
                            -1,               /* --llx2          */
                            -1,               /* --lly2          */
                            -1,               /* --urx2          */
                            -1,               /* --ury2          */
                            -1,               /* --llx3          */
                            -1,               /* --lly3          */
                            -1,               /* --urx3          */
                            -1,               /* --ury3          */
                            -1,               /* --llx4          */
                            -1,               /* --lly4          */
                            -1,               /* --urx4          */
                            -1,               /* --ury4          */
                            -1,               /* --llx5          */
                            -1,               /* --lly5          */
                            -1,               /* --urx5          */
                            -1,               /* --ury5          */
				  0,               /* --exts          */
    NIR);

    frameset = detmon_tests_fill_frameset(tag_on, tag_off,
						 6, TRUE, FALSE, 0, 1);

/*    detmon_lg(frameset, parlist, tag_on, tag_off, "detmon_ir_lg",
		     "detmon", REGEXP, prolist, prolist, prolist,
		     prolist, prolist, prolist, "DETMON / 1.0",
		     NULL, NULL, TRUE);
    cpl_test_error(CPL_ERROR_DATA_NOT_FOUND);
    cpl_error_reset();
*/
    cpl_frameset_delete(frameset);
    cpl_parameterlist_delete(parlist);


    /* 9. test: Default use, OPT case */
    parlist = cpl_parameterlist_new();
    detmon_lg_fill_parlist_opt_default(parlist,
					      "detmon_opt_lg", "detmon");

    frameset = detmon_tests_fill_frameset(tag_on, tag_off,
						 20, FALSE, FALSE, 0, 1);

/*    detmon_lg(frameset, parlist, tag_on, tag_off, "detmon_opt_lg",
		     "detmon", REGEXP, prolist, prolist, prolist,
		     prolist, prolist, prolist, "DETMON / 1.0",
		     NULL, NULL, FALSE);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_error_reset();
*/
    cpl_frameset_delete(frameset);
    cpl_parameterlist_delete(parlist);

    /* 4. test: Passing a frameset where frames have extensions */
    parlist = cpl_parameterlist_new();
    detmon_lg_fill_parlist(parlist, "detmon_ir_lg", "detmon",
			      "PTC",
                            3,                /* --order         */
                            3,                /* --kappa         */
                            25,               /* --niter         */
                            -1,               /* --llx           */
                            -1,               /* --lly           */
                            -1,               /* --urx           */
                            -1,               /* --ury           */
                            10000,            /* --ref_level     */
                            "CPL_FALSE",      /* --intermediate  */
                            "CPL_FALSE",      /* --autocorr      */
                            "CPL_FALSE",      /* --collapse      */
                            "CPL_TRUE",       /* --rescale       */
                            "CPL_TRUE",       /* --pix2pix       */
                            "CPL_FALSE",       /* --bpmbin       */
                              -1,             /* --filter        */
                            26,               /* --m             */
                            26,               /* --n             */
				  1e-3,
                            "CPL_TRUE",       /* --pafgen       */
				  "detmon_ir_lg",
                            -1,               /* --llx1          */
                            -1,               /* --lly1          */
                            -1,               /* --urx1          */
                            -1,               /* --ury1          */
                            -1,               /* --llx2          */
                            -1,               /* --lly2          */
                            -1,               /* --urx2          */
                            -1,               /* --ury2          */
                            -1,               /* --llx3          */
                            -1,               /* --lly3          */
                            -1,               /* --urx3          */
                            -1,               /* --ury3          */
                            -1,               /* --llx4          */
                            -1,               /* --lly4          */
                            -1,               /* --urx4          */
                            -1,               /* --ury4          */
                            -1,               /* --llx5          */
                            -1,               /* --lly5          */
                            -1,               /* --urx5          */
                            -1,               /* --ury5          */
				  -1,               /* --exts          */
    NIR);
    /*
                                    * Unique change of this call with
                                    * respect of the default call:
                                    * last parameter --exts = -1
                                    * to tell the recipe to compute on
                                    * all extensions.
                                    */

    frameset = detmon_tests_fill_frameset(tag_on, tag_off,
                                                 20, TRUE, TRUE, 8, 1);

    detmon_lg(frameset, parlist, tag_on, tag_off, "detmon_ir_lg", "detmon", REGEXP,
                     prolist, prolist, prolist, prolist, prolist, prolist, "DETMON / 1.0", NULL, NULL, TRUE);
    cpl_test(cpl_error_get_code() == CPL_ERROR_NONE ||
                cpl_error_get_code() == CPL_ERROR_UNSUPPORTED_MODE);
    cpl_error_reset();

    cpl_parameterlist_delete(parlist);
    cpl_frameset_delete(frameset);

    /* 2 Trial: passing a frameset where frames have extensions,
       but computing on a single extension */
    parlist = cpl_parameterlist_new();
    detmon_lg_fill_parlist(parlist, "detmon_ir_lg", "detmon",
			      "PTC",
                            3,                /* --order         */
                            3,                /* --kappa         */
                            25,               /* --niter         */
                            -1,               /* --llx           */
                            -1,               /* --lly           */
                            -1,               /* --urx           */
                            -1,               /* --ury           */
                            10000,            /* --ref_level     */
                            "CPL_FALSE",      /* --intermediate  */
                            "CPL_FALSE",      /* --autocorr      */
                            "CPL_FALSE",      /* --collapse      */
                            "CPL_TRUE",       /* --rescale       */
                            "CPL_TRUE",       /* --pix2pix       */
                            "CPL_FALSE",       /* --bpmbin       */
                              -1,             /* --filter        */
                            26,               /* --m             */
                            26,               /* --n             */
				  1e-3,
                            "CPL_TRUE",       /* --pafgen       */
				  "detmon_ir_lg",
                            -1,               /* --llx1          */
                            -1,               /* --lly1          */
                            -1,               /* --urx1          */
                            -1,               /* --ury1          */
                            -1,               /* --llx2          */
                            -1,               /* --lly2          */
                            -1,               /* --urx2          */
                            -1,               /* --ury2          */
                            -1,               /* --llx3          */
                            -1,               /* --lly3          */
                            -1,               /* --urx3          */
                            -1,               /* --ury3          */
                            -1,               /* --llx4          */
                            -1,               /* --lly4          */
                            -1,               /* --urx4          */
                            -1,               /* --ury4          */
                            -1,               /* --llx5          */
                            -1,               /* --lly5          */
                            -1,               /* --urx5          */
                            -1,               /* --ury5          */
				  2,               /* --exts          */
    NIR);
    /*
                                    * Main change of this call with
                                    * respect of the default call:
                                    * last parameter --exts = 2
                                    * to tell the recipe to compute only on
                                    * the second extension.
                                    */

    frameset = detmon_tests_fill_frameset(tag_on, tag_off,
                                                 20, TRUE, TRUE, 8, 1);

    detmon_lg(frameset, parlist, tag_on, tag_off, "detmon_ir_lg", "detmon", REGEXP,
                     prolist, prolist, prolist, prolist, prolist, prolist, "DETMON / 1.0", NULL, NULL, TRUE);
    cpl_test(cpl_error_get_code() == CPL_ERROR_NONE ||
                cpl_error_get_code() == CPL_ERROR_UNSUPPORTED_MODE);
    cpl_error_reset();

    cpl_parameterlist_delete(parlist);
    cpl_frameset_delete(frameset);


    /* New trial. With a octatett instead of a quartett. */
    parlist = cpl_parameterlist_new();
    detmon_lg_fill_parlist_nir_default(parlist, "detmon_ir_lg", "detmon");

    frameset = detmon_tests_fill_frameset_oct(tag_on, tag_off, 20, TRUE, FALSE, 0);

/*    detmon_lg(frameset, parlist, tag_on, tag_off, "detmon_ir_lg", "detmon", REGEXP,
                     prolist, prolist, prolist, prolist, prolist, prolist, "DETMON / 1.0", NULL, NULL, TRUE);
    cpl_test(cpl_error_get_code() == CPL_ERROR_NONE ||
                cpl_error_get_code() == CPL_ERROR_UNSUPPORTED_MODE);
    cpl_error_reset();
*/
    cpl_parameterlist_delete(parlist);
    cpl_frameset_delete(frameset);

    /* Test with more than a single setting */
    parlist = cpl_parameterlist_new();
    detmon_lg_fill_parlist_nir_default(parlist, "detmon_ir_lg", "detmon");

    frameset = detmon_tests_fill_frameset(tag_on, tag_off,
                                                 20, TRUE, FALSE, 0, 3);

    detmon_lg(frameset, parlist, tag_on, tag_off, "detmon_ir_lg", "detmon", REGEXP,
                     prolist, prolist, prolist, prolist, prolist, prolist, "DETMON / 1.0", compare_settings, NULL, TRUE);

    cpl_test_error(CPL_ERROR_NONE);

    cpl_parameterlist_delete(parlist);
    cpl_frameset_delete(frameset);

    cpl_propertylist_delete(prolist);

    cpl_test_zero(system("rm *.paf detmon_lg-dummy*.fits"));

    /* TESTS for detmon_gain() */
    {
	/* Variable used for detmon_gain() testing */
	cpl_table * gain_table;
	cpl_imagelist * imlist_on;
	cpl_imagelist * imlist_off;
	cpl_imagelist * diff;
	cpl_imagelist * corr;
	cpl_vector * exptimes;
	cpl_vector * ndits;
	cpl_propertylist * qclist;
	int i;
	double gain;

	/* Test with NULL inputs */
	gain_table = detmon_gain(NULL, NULL, NULL, NULL, 0, 0, 0, 0, 0, 0.0,
					0, 0, 0, NULL, 0, NULL, NULL);
	cpl_test_null(gain_table);
	cpl_test_error(CPL_ERROR_NULL_INPUT);

	imlist_on  = detmon_tests_imlist(8, ON);
	imlist_off = detmon_tests_imlist(8, OFF);

	exptimes = cpl_vector_new(16);
	ndits = cpl_vector_new(16);

	for(i = 0; i < 16; i++) {
	    cpl_vector_set(exptimes, i, (double)(i/2 + 1));
	    cpl_vector_set(ndits, i,  1.);
	}

	qclist = cpl_propertylist_new();

	gain_table = detmon_gain(imlist_on, imlist_off, exptimes, ndits,
					0.0,
					1, 1,
					IMAGESIZE, IMAGESIZE, 3.0, 25, 26, 26,
					qclist, IRPLIB_GAIN_NO_COLLAPSE |
					IRPLIB_GAIN_NIR,
					NULL, NULL);


	cpl_test_nonnull(gain_table);
	cpl_test_error(CPL_ERROR_NONE);

	/* Check correctness */
	gain = cpl_propertylist_get_double(qclist, DETMON_QC_GAIN);
	cpl_test(gain > 220.0 && gain < 240.0);

	cpl_table_delete(gain_table);
	cpl_propertylist_delete(qclist);


	qclist = cpl_propertylist_new();

	gain_table = detmon_gain(imlist_on, imlist_off, exptimes, ndits,
                                        0.0,
					1, 1,
					IMAGESIZE, IMAGESIZE, 3.0, 25, 26, 26,
					qclist, IRPLIB_GAIN_NO_COLLAPSE |
					IRPLIB_GAIN_NIR |
					IRPLIB_GAIN_PTC,
					NULL, NULL);
	cpl_test_nonnull(gain_table);
	cpl_test_error(CPL_ERROR_NONE);

	/* Check correctness */
	gain = cpl_propertylist_get_double(qclist, DETMON_QC_GAIN);
	cpl_test(gain > 160.0 && gain < 180.0);

	cpl_table_delete(gain_table);
	cpl_propertylist_delete(qclist);

	qclist = cpl_propertylist_new();

	diff = NULL;
	corr = NULL;

	gain_table = detmon_gain(imlist_on, imlist_off, exptimes, ndits,
                                        0.0,
					1, 1,
					IMAGESIZE, IMAGESIZE, 3.0, 25, 26, 26,
					qclist, IRPLIB_GAIN_NO_COLLAPSE |
					IRPLIB_GAIN_NIR |
					IRPLIB_GAIN_WITH_AUTOCORR, &diff, &corr);
	cpl_test_nonnull(gain_table);
	cpl_test_nonnull(diff);
	cpl_test_nonnull(corr);
	cpl_test_error(CPL_ERROR_NONE);

	cpl_imagelist_delete(diff);
	cpl_imagelist_delete(corr);
	cpl_table_delete(gain_table);
	cpl_propertylist_delete(qclist);

	qclist = cpl_propertylist_new();

	gain_table = detmon_gain(imlist_on, imlist_off, exptimes, ndits,
                                        0.0,
					1, 1,
					IMAGESIZE, IMAGESIZE, 3.0, 25, 26, 26,
					qclist, IRPLIB_GAIN_NO_COLLAPSE |
					IRPLIB_GAIN_NIR |
					IRPLIB_GAIN_WITH_AUTOCORR, NULL, NULL);
	cpl_test_nonnull(gain_table);
	cpl_test_error(CPL_ERROR_NONE);

	cpl_imagelist_delete(imlist_on);
	cpl_imagelist_delete(imlist_off);
	cpl_propertylist_delete(qclist);
	cpl_vector_delete(exptimes);
	cpl_vector_delete(ndits);
	cpl_table_delete(gain_table);
    }

    /* TESTS for detmon_lin() */
    {
	cpl_table     * lin_table;
	cpl_imagelist * imlist_on;
	cpl_imagelist * imlist_off;
	cpl_vector    * exptimes;
	cpl_propertylist * qclist;
	cpl_imagelist * coeffs_cube =NULL;
	cpl_image     * bpm =NULL;
	double lincoeff1, lincoeff2, lincoeff3;
	int i;

	/* TESTS with NULL inputs */
	lin_table = detmon_lin(NULL, NULL, NULL, 0.0, 0, 0, 0, 0,
                                      0, 10000, 3, FALSE, NULL, 0, NULL, NULL);

	cpl_test_null(lin_table);
	cpl_test_error(CPL_ERROR_NULL_INPUT);

	/* Basic setting for testing */
	imlist_on  = detmon_tests_imlist(8, ON);
	imlist_off = detmon_tests_imlist(8, OFF);

	exptimes = cpl_vector_new(16);

	for(i = 0; i < 16; i++) {
	    cpl_vector_set(exptimes, i, (double)(i/2 + 1));
	}

	qclist = cpl_propertylist_new();

	lin_table = detmon_lin(imlist_on, imlist_off, exptimes, 0.0,
				      1, 1, IMAGESIZE, IMAGESIZE,
				      3, 10000, 3, FALSE, qclist,
                                      IRPLIB_LIN_NO_COLLAPSE |
				      IRPLIB_LIN_NIR,
	                              NULL, NULL);

	cpl_test_nonnull(lin_table);
	cpl_test_error(CPL_ERROR_NONE);

	/* Check correctness */
	lincoeff1 = cpl_propertylist_get_double(qclist, "ESO QC LIN COEF1");
	lincoeff2 = cpl_propertylist_get_double(qclist, "ESO QC LIN COEF2");
	lincoeff3 = cpl_propertylist_get_double(qclist, "ESO QC LIN COEF3");

	cpl_test(lincoeff1 > 195.0 && lincoeff1 < 205.0);
	cpl_test(abs(lincoeff2) < 1);
	cpl_test(abs(lincoeff3) < 1);

	cpl_table_delete(lin_table);

	lin_table = detmon_lin(imlist_on, imlist_off, exptimes, 0.0,
				      1, 1, IMAGESIZE, IMAGESIZE,
				      3, 10000, 3, FALSE, qclist,
                                      IRPLIB_LIN_NO_COLLAPSE |
				      IRPLIB_LIN_NIR | IRPLIB_LIN_PIX2PIX,
	                              &coeffs_cube, &bpm);

	cpl_test_nonnull(lin_table);
	cpl_test_nonnull(coeffs_cube);
	cpl_test_nonnull(bpm);
	cpl_test_error(CPL_ERROR_NONE);



	cpl_test_zero(system("rm detmon_ir_lg*.fits"));

	cpl_table_delete(lin_table);
	cpl_imagelist_delete(coeffs_cube);
	cpl_image_delete(bpm);

	cpl_imagelist_delete(imlist_on);
	cpl_imagelist_delete(imlist_off);
	cpl_vector_delete(exptimes);
	cpl_propertylist_delete(qclist);
    }
}

/*---------------------------------------------------------------------------*/
/*
 * @brief  Detect bad pixels of a selected type
 * @param  in           Input image
 * @param  mean         Mean level of the image - used to compute threshold
 * @param  sigma        Stdev level of the image - used to compute threshold
 * @param  flag         Flag - represents selected type of bad pixels (see enum)
 * @return The detected bad pixels in a cpl_mask.
 */
/*---------------------------------------------------------------------------*/

cpl_frameset * detmon_tests_fill_frameset(const char     * tag_on,
                                                 const char     * tag_off,
                                                 int              non,
                                                 cpl_boolean      opt_nir,
                                                 cpl_boolean      with_exts,
                                                 int              nexts,
                                                 int              nsettings)
{
    cpl_frameset                * frameset;
    cpl_frame                   * frame;
    int i, h;
    char                        frame_name[200];
    cpl_propertylist            * plist;
    cpl_image                   * image;
    frameset = cpl_frameset_new();

    for (h = 0; h < nsettings; h++) {
	for (i=0; i<non; i++) {
	    frame = cpl_frame_new();
	    sprintf(frame_name, "detmon_lg-dummyon%d%d.fits",i,h);
	    cpl_frame_set_filename(frame, frame_name);
	    cpl_frame_set_tag(frame, tag_on);
	    cpl_frame_set_type(frame, CPL_FRAME_TYPE_IMAGE);
	    cpl_frame_set_group(frame, CPL_FRAME_GROUP_RAW);

	    cpl_frameset_insert(frameset, frame);

	    plist = cpl_propertylist_new();
	    if(opt_nir == TRUE) {
		cpl_propertylist_append_double(plist, "ESO DET DIT", (double)(i/2+1));
		cpl_propertylist_append_int(plist, "ESO DET NDIT", 1);
	    } else {
		cpl_propertylist_append_double(plist, "ESO DET WIN1 UIT1", (double)((i+1)/2));
	    }

	    cpl_propertylist_append_double(plist, "EXPTIME", (double)(i/2 + 1) );
	    cpl_propertylist_append_double(plist, "MJD-OBS", 0);

	    cpl_propertylist_append_double(plist, "SETTING", h+1);
	    if(with_exts == TRUE) {
		int j;
		cpl_image_save(NULL, frame_name, CPL_BPP_IEEE_FLOAT, plist, CPL_IO_DEFAULT);
		for(j = 0; j < nexts; j++) {
		    image = cpl_image_new(IMAGESIZE, IMAGESIZE, CPL_TYPE_FLOAT);
		    cpl_image_fill_noise_uniform(image, 5 - i, 10 + i);
		    cpl_image_add_scalar(image,300* i);
		    cpl_ensure(cpl_error_get_code()== CPL_ERROR_NONE, CPL_ERROR_NULL_INPUT, NULL);

		    cpl_image_save(image, frame_name, CPL_BPP_IEEE_FLOAT, plist, CPL_IO_EXTEND);
		    cpl_image_delete(image);
		}
	    } else {
		image = cpl_image_new(IMAGESIZE, IMAGESIZE, CPL_TYPE_FLOAT);
		cpl_image_fill_noise_uniform(image, 5 - i, 10 + i);
		cpl_image_add_scalar(image,300* i);
		cpl_ensure(cpl_error_get_code()== CPL_ERROR_NONE, CPL_ERROR_NULL_INPUT, NULL);

		cpl_image_save(image, frame_name, CPL_BPP_IEEE_FLOAT, plist, CPL_IO_DEFAULT);


		cpl_image_delete(image);
	    }
	    cpl_propertylist_delete(plist);
	}
	for (i=0; i<non; i++) {


	    frame = cpl_frame_new();
	    sprintf(frame_name, "detmon_lg-dummyoff%d%d.fits",i,h);
	    cpl_frame_set_filename(frame, frame_name);
	    cpl_frame_set_tag(frame, tag_off);
	    cpl_frame_set_type(frame, CPL_FRAME_TYPE_IMAGE);
	    cpl_frame_set_group(frame, CPL_FRAME_GROUP_RAW);

	    cpl_frameset_insert(frameset, frame);

	    plist = cpl_propertylist_new();
	    if(opt_nir == TRUE) {
		cpl_propertylist_append_double(plist, "ESO DET DIT", (double)(i/2+1));
		cpl_propertylist_append_double(plist, "ESO DET NDIT", 1);
	    } else {
		cpl_propertylist_append_double(plist, "ESO DET WIN1 UIT1", (double)((i+1)/2));
	    }
	    cpl_propertylist_append_double(plist, "MJD-OBS", 0);
	    cpl_propertylist_append_double(plist, "EXPTIME", (double)(i/2 + 1));
	    cpl_propertylist_append_double(plist, "SETTING", h+1);

	    if(with_exts == TRUE) {
		int j;
		cpl_image_save(NULL, frame_name, CPL_BPP_IEEE_FLOAT, plist, CPL_IO_DEFAULT);
		for(j = 0; j < nexts; j++) {
		    image = cpl_image_new(IMAGESIZE, IMAGESIZE, CPL_TYPE_FLOAT);
		    cpl_image_fill_noise_uniform(image, 5 - i, 10 + i);
		    cpl_image_add_scalar(image, i);
		    cpl_ensure(cpl_error_get_code()== CPL_ERROR_NONE, CPL_ERROR_NULL_INPUT, NULL);

		    cpl_image_save(image, frame_name, CPL_BPP_IEEE_FLOAT, plist, CPL_IO_EXTEND);
		    cpl_image_delete(image);
		}
	    } else {
		image = cpl_image_new(IMAGESIZE, IMAGESIZE, CPL_TYPE_FLOAT);
		cpl_image_fill_noise_uniform(image, 5 - i, 10 + i);
		cpl_image_add_scalar(image, i);
		cpl_ensure(cpl_error_get_code()== CPL_ERROR_NONE, CPL_ERROR_NULL_INPUT, NULL);

		cpl_image_save(image, frame_name, CPL_BPP_IEEE_FLOAT, plist, CPL_IO_DEFAULT);


		cpl_image_delete(image);
	    }
	    cpl_propertylist_delete(plist);
	}
    }

    return frameset;
}


cpl_frameset * detmon_tests_fill_frameset_oct(const char         * tag_on,
                                                 const char     * tag_off,
                                                 int              non,
                                                 cpl_boolean      opt_nir,
                                                 cpl_boolean      with_exts,
                                                 int              nexts)
{
    cpl_frameset                * frameset;
    cpl_frame                   * frame;
    int i;
    char                        frame_name[200];
    cpl_propertylist            * plist;
    cpl_image                   * image;
    frameset = cpl_frameset_new();

    for (i=0; i<non; i++) {
        frame = cpl_frame_new();
        sprintf(frame_name, "detmon_lg-dummyon%d.fits",i);
        cpl_frame_set_filename(frame, frame_name);
        cpl_frame_set_tag(frame, tag_on);
        cpl_frame_set_type(frame, CPL_FRAME_TYPE_IMAGE);
        cpl_frame_set_group(frame, CPL_FRAME_GROUP_RAW);

        cpl_frameset_insert(frameset, frame);

        plist = cpl_propertylist_new();
        if(opt_nir == TRUE) {
          if( i == 2 || i == 3) {
            cpl_propertylist_append_double(plist, "ESO DET DIT", (double)(1));
            cpl_propertylist_append_double(plist, "ESO DET NDIT", 1);
            cpl_propertylist_append_double(plist, "EXPTIME", (double)(1));
            cpl_propertylist_append_double(plist, "MJD-OBS", 0);
          } else {
            cpl_propertylist_append_double(plist, "ESO DET DIT", (double)(i/2+1));
            cpl_propertylist_append_double(plist, "ESO DET NDIT", 1);
            cpl_propertylist_append_double(plist, "EXPTIME", (double)(i/2+1));
            cpl_propertylist_append_double(plist, "MJD-OBS", 0);
          }
        } else {
            cpl_propertylist_append_double(plist, "ESO DET WIN1 UIT1", (double)((i+1)/2));
            cpl_propertylist_append_double(plist, "EXPTIME", (double)((i+1)/2));
            cpl_propertylist_append_double(plist, "MJD-OBS", 0);
        }

        if(with_exts == TRUE) {
          int j;
          cpl_image_save(NULL, frame_name, CPL_BPP_IEEE_FLOAT, plist, CPL_IO_DEFAULT);
          for(j = 0; j < nexts; j++) {
            image = cpl_image_new(IMAGESIZE, IMAGESIZE, CPL_TYPE_FLOAT);
            cpl_image_fill_noise_uniform(image, 5 - i, 10 + i);
	    if(i == 0 || i == 1 || i== 2 || i == 3) {
		cpl_image_add_scalar(image,300);
	    } else
		cpl_image_add_scalar(image,300* i);
            cpl_ensure(cpl_error_get_code()== CPL_ERROR_NONE, CPL_ERROR_NULL_INPUT, NULL);

            cpl_image_save(image, frame_name, CPL_BPP_IEEE_FLOAT, plist, CPL_IO_EXTEND);
            cpl_image_delete(image);
          }
        } else {
            image = cpl_image_new(IMAGESIZE, IMAGESIZE, CPL_TYPE_FLOAT);
            cpl_image_fill_noise_uniform(image, 5 - i, 10 + i);
	    if(i == 0 || i == 1 || i== 2 || i == 3) {
		cpl_image_add_scalar(image,300);
	    } else
		cpl_image_add_scalar(image,300* i);

            cpl_ensure(cpl_error_get_code()== CPL_ERROR_NONE, CPL_ERROR_NULL_INPUT, NULL);

            cpl_image_save(image, frame_name, CPL_BPP_IEEE_FLOAT, plist, CPL_IO_DEFAULT);


            cpl_image_delete(image);
        }
        cpl_propertylist_delete(plist);
    }

    for (i=0; i<non; i++) {


        frame = cpl_frame_new();
        sprintf(frame_name, "detmon_lg-dummyoff%d.fits",i);
        cpl_frame_set_filename(frame, frame_name);
        cpl_frame_set_tag(frame, tag_off);
        cpl_frame_set_type(frame, CPL_FRAME_TYPE_IMAGE);
        cpl_frame_set_group(frame, CPL_FRAME_GROUP_RAW);

        cpl_frameset_insert(frameset, frame);

        plist = cpl_propertylist_new();
        if(opt_nir == TRUE) {
          if( i == 2 || i == 3) {
            cpl_propertylist_append_double(plist, "ESO DET DIT", (double)(1));
            cpl_propertylist_append_double(plist, "ESO DET NDIT", 1);
            cpl_propertylist_append_double(plist, "EXPTIME", (double)(1));
            cpl_propertylist_append_double(plist, "MJD-OBS", 0);
          } else {
            cpl_propertylist_append_double(plist, "ESO DET DIT", (double)(i/2+1));
            cpl_propertylist_append_double(plist, "ESO DET NDIT", 1);
            cpl_propertylist_append_double(plist, "EXPTIME", (double)(i/2+1));
            cpl_propertylist_append_double(plist, "MJD-OBS", 0);
          }
        } else {
            cpl_propertylist_append_double(plist, "ESO DET WIN1 UIT1", (double)((i+1)/2));
            cpl_propertylist_append_double(plist, "EXPTIME", (double)((i+1)/2));
            cpl_propertylist_append_double(plist, "MJD-OBS", 0);
        }

        if(with_exts == TRUE) {
          int j;
          cpl_image_save(NULL, frame_name, CPL_BPP_IEEE_FLOAT, plist, CPL_IO_DEFAULT);
          for(j = 0; j < nexts; j++) {
            image = cpl_image_new(IMAGESIZE, IMAGESIZE, CPL_TYPE_FLOAT);
            cpl_image_fill_noise_uniform(image, 5 - i, 10 + i);
	    if(i == 0 || i == 1 || i== 2 || i == 3) {
		cpl_image_add_scalar(image,300);
	    } else
		cpl_image_add_scalar(image,300* i);

            cpl_ensure(cpl_error_get_code()== CPL_ERROR_NONE, CPL_ERROR_NULL_INPUT, NULL);

            cpl_image_save(image, frame_name, CPL_BPP_IEEE_FLOAT, plist, CPL_IO_EXTEND);
            cpl_image_delete(image);
          }
        } else {
            image = cpl_image_new(IMAGESIZE, IMAGESIZE, CPL_TYPE_FLOAT);
            cpl_image_fill_noise_uniform(image, 5 - i, 10 + i);
	    if(i == 0 || i == 1 || i== 2 || i == 3) {
		cpl_image_add_scalar(image,300);
	    } else
		cpl_image_add_scalar(image,300* i);

            cpl_ensure(cpl_error_get_code()== CPL_ERROR_NONE, CPL_ERROR_NULL_INPUT, NULL);

            cpl_image_save(image, frame_name, CPL_BPP_IEEE_FLOAT, plist, CPL_IO_DEFAULT);


            cpl_image_delete(image);
        }
        cpl_propertylist_delete(plist);
    }

    return frameset;
}


int compare_settings(const cpl_frame * frame1, const cpl_frame * frame2)
{
    int                     comparison;
    cpl_propertylist       *plist1;
    cpl_propertylist       *plist2;
    double                  dval1, dval2;

    /* Test entries */
    if(frame1 == NULL || frame2 == NULL)
        return -1;

    /* Get property lists */
    if((plist1 = cpl_propertylist_load(cpl_frame_get_filename(frame1),
                                       0)) == NULL) {
        cpl_msg_error(cpl_func, "getting header from reference frame");
        return -1;
    }
    if((plist2 = cpl_propertylist_load(cpl_frame_get_filename(frame2),
                                       0)) == NULL) {
        cpl_msg_error(cpl_func, "getting header from reference frame");
        cpl_propertylist_delete(plist1);
        return -1;
    }

    /* Test status */
    if(cpl_error_get_code()) {
        cpl_propertylist_delete(plist1);
        cpl_propertylist_delete(plist2);
        return -1;
    }

    /* Compare exposure time */
    comparison = 1;
    dval1 = cpl_propertylist_get_double(plist1, "SETTING");
    dval2 = cpl_propertylist_get_double(plist2, "SETTING");
    if(cpl_error_get_code()) {
        cpl_msg_error(cpl_func, "cannot get exposure time");
        cpl_propertylist_delete(plist1);
        cpl_propertylist_delete(plist2);
        return -1;
    }
    if( dval1 != dval2)
        comparison = 0;

    /* Free and return */
    cpl_propertylist_delete(plist1);
    cpl_propertylist_delete(plist2);
    return comparison;

}

cpl_imagelist * detmon_tests_imlist(int ndits, cpl_boolean onoff)
{
  cpl_imagelist * imlist = cpl_imagelist_new();
  int i;

  double levelstep = onoff ? ON_LEVEL : OFF_LEVEL;
  double factor    = onoff ? 1.0 : 0.5;

  for (i = 1; i <= ndits; i++) {
    cpl_image * im1 = cpl_image_new(IMAGESIZE, IMAGESIZE, CPL_TYPE_FLOAT);
    cpl_image * im2 = cpl_image_new(IMAGESIZE, IMAGESIZE, CPL_TYPE_FLOAT);
    double noise_min = NOISE_MIN * sqrt(i) / (double) ndits * factor;
    double noise_max = NOISE_MAX * sqrt(i) / (double) ndits * factor;
    cpl_image_fill_noise_uniform(im1, noise_min, noise_max);
    cpl_image_fill_noise_uniform(im2, noise_min, noise_max);
    cpl_image_add_scalar(im1, levelstep * i);
    cpl_image_add_scalar(im2, levelstep * i);
    cpl_imagelist_set(imlist, im1, 2 * (i - 1));
    cpl_imagelist_set(imlist, im2, 2 * (i - 1) + 1);
  }

  return imlist;
}

static void detmon_check_order_test(void)
{
	const double TOLERANCE = 1E-7;
//	const int SZ = 9;
	const double exptime[][9] = {
			{ 1,2,3,4,5,6,7,8,9} ,
			{ 1,1,1,4,5,6,7,8,9} ,
			{ 1,2,3,4,5,5,5,8,9} ,
			{ 1,2,3,4,5,6,8,8,8} ,
			{ 1,1,1,1,1,6,6,6,6} ,
	};
	const int order[] = { 8, 6, 6, 6, 1 };

	int sz = sizeof(exptime) / sizeof(exptime[0]);
	int i;
	for (i = 0; i < sz; i++)
	{
		const double* ex = exptime[i];
		cpl_error_code err = detmon_check_order(ex, 9, TOLERANCE, order[i]);
		cpl_test(err == CPL_ERROR_NONE);
	};
}
