/* $Id$
 *
 * This file is part of the ERIS Pipeline
 * Copyright (C) 2002,2003 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <cpl.h>
#include <math.h>
#include <string.h>
#include "eris_ifu_combine_static.h"
#include "eris_ifu_error.h"
#include "eris_ifu_functions.h"
#include "eris_ifu_utils.h"

/* Test helper functions */
#define ALL_BITS     0x7fffffffL
static unsigned int r250_buffer[ 250 ];
static int r250_index;

double eris_dr250(void)		/* returns a random double in range 0..1 */
{
	register int	j;
	register unsigned int new_rand;

	if ( r250_index >= 147 )
		j = r250_index - 147;	/* wrap pointer around */
	else
		j = r250_index + 103;

	new_rand = r250_buffer[ r250_index ] ^ r250_buffer[ j ];
	r250_buffer[ r250_index ] = new_rand;

	if ( r250_index >= 249 )	/* increment pointer for next time */
		r250_index = 0;
	else
		r250_index++;

	return (double)new_rand / ALL_BITS;

}
#define uniform(a,b)    ( a + (b - a) * eris_dr250() )


static cpl_image* create_test_image(int nx, int ny, double value) {
    cpl_image* img = cpl_image_new(nx, ny, CPL_TYPE_DOUBLE);
    double* data = cpl_image_get_data_double(img);
    for (int i = 0; i < nx * ny; i++) {
        data[i] = value;
    }
    return img;
}

static void test_nearest_int(void) {
    /* Test positive numbers */
    cpl_test_eq(eris_ifu_combine_nearest_int(1.2), 1);
    cpl_test_eq(eris_ifu_combine_nearest_int(1.7), 2);
    cpl_test_eq(eris_ifu_combine_nearest_int(1.5001), 2);
    cpl_test_eq(eris_ifu_combine_nearest_int(1.5), 1);

    /* Test negative numbers */
    cpl_test_eq(eris_ifu_combine_nearest_int(-1.2), -1);
    cpl_test_eq(eris_ifu_combine_nearest_int(-1.7), -2);
    cpl_test_eq(eris_ifu_combine_nearest_int(-1.5), -2);
    cpl_test_eq(eris_ifu_combine_nearest_int(-1.499999), -1);
    
    /* Test zero */
    cpl_test_eq(eris_ifu_combine_nearest_int(0.0), 0);
    cpl_test_eq(eris_ifu_combine_nearest_int(-0.0), 0);
    cpl_test_error(CPL_ERROR_NONE);
}

static void test_get_xy_min_max(void) {
    const int nframes = 3;
    float offsetx[] = {1.0, -2.0, 0.5};
    float offsety[] = {-1.0, 2.0, 0.0};
    float min_offx, max_offx, min_offy, max_offy;
    
    eris_ifu_combine_get_xy_min_max(nframes, offsetx, offsety,
                                   &min_offx, &max_offx,
                                   &min_offy, &max_offy);
    
    cpl_test_eq(min_offx, -2.0);
    cpl_test_eq(max_offx, 1.0);
    cpl_test_eq(min_offy, -1.0);
    cpl_test_eq(max_offy, 2.0);
    cpl_test_error(CPL_ERROR_NONE);
}

static void test_combine_jittered_images_null_inputs(void) {
    cpl_error_code error;
    
    /* Test with NULL inputs */
    error = eris_ifu_combine_jittered_images(NULL, NULL, 64, 64, NULL, NULL, NULL,
                                           0, NULL, NULL, NULL, 0.0, NULL, 0);
    cpl_test_eq(error, CPL_ERROR_NULL_INPUT);
    cpl_error_reset();
}

static void test_combine_jittered_images_basic(void) {
    const int nx = 64;
    const int ny = 64;
    const int n_cubes = 2;
    cpl_test_error(CPL_ERROR_NONE);
    /* Create test images */
    cpl_image** imagesData = cpl_malloc(n_cubes * sizeof(cpl_image*));
    cpl_image** imagesError = cpl_malloc(n_cubes * sizeof(cpl_image*));
    
    for (int i = 0; i < n_cubes; i++) {
        imagesData[i] = create_test_image(nx, ny, 1.0);
        imagesError[i] = create_test_image(nx, ny, 0.1);
    }
    cpl_test_error(CPL_ERROR_NONE);
    /* Create offsets and exposure times */
    float offsetx[] = {0.0, 1.0};
    float offsety[] = {0.0, 1.0};
    double exptimes[] = {1.0, 1.0};
    cpl_test_error(CPL_ERROR_NONE);
    /* Output images */
    cpl_image* mergedImageData = NULL;
    cpl_image* mergedImageError = NULL;
    cpl_image* mergedImageDIT = NULL;
    cpl_test_error(CPL_ERROR_NONE);
    /* Test combining images */

    cpl_error_code error = eris_ifu_combine_jittered_images(
        imagesData, imagesError, nx, ny,
        &mergedImageData, &mergedImageError, &mergedImageDIT,
        n_cubes, offsetx, offsety, exptimes,
        3.0, "MEAN", -1);

    cpl_test_eq(error, CPL_ERROR_NONE);
    cpl_test_nonnull(mergedImageData);
    cpl_test_nonnull(mergedImageError);
    cpl_test_nonnull(mergedImageDIT);
    
    /* Cleanup */
    for (int i = 0; i < n_cubes; i++) {
        cpl_image_delete(imagesData[i]);
        cpl_image_delete(imagesError[i]);
    }
    cpl_free(imagesData);
    cpl_free(imagesError);
    cpl_image_delete(mergedImageData);
    cpl_image_delete(mergedImageError);
    cpl_image_delete(mergedImageDIT);
    cpl_test_error(CPL_ERROR_NONE);
}

static void test_combine_divide_DIT(void) {
    const int n_cubes = 2;
    const int nx = 64;
    const int ny = 64;
    
    /* Create test images */
    cpl_imagelist** cubesData = cpl_malloc(n_cubes * sizeof(cpl_imagelist*));
    for (int i = 0; i < n_cubes; i++) {
        cubesData[i] = cpl_imagelist_new();
        cpl_image* img = create_test_image(nx, ny, 10.0 * (i + 1));  // Different values for each cube
        cpl_imagelist_set(cubesData[i], img, 0);
    }
    
    /* Create exposure times */
    double exptimes[] = {2.0, 4.0};  // Different exposure times
    
    /* Test dividing by DIT */
    cpl_error_code error = eris_ifu_combine_divide_DIT(cubesData, n_cubes, exptimes);
    cpl_test_eq(error, CPL_ERROR_NONE);
    
    /* Verify results */
    for (int i = 0; i < n_cubes; i++) {
        cpl_image* img = cpl_imagelist_get(cubesData[i], 0);
        double* data = cpl_image_get_data_double(img);
        double expected = 10.0 * (i + 1) / exptimes[i];
        
        /* Check first pixel (all pixels should have same value) */
        cpl_test_abs(data[0], expected, 1e-10);
    }
    
    /* Test with NULL inputs */
    error = eris_ifu_combine_divide_DIT(NULL, n_cubes, exptimes);
    cpl_test_eq(error, CPL_ERROR_NULL_INPUT);
    cpl_error_reset();
    /* Cleanup */
    for (int i = 0; i < n_cubes; i++) {
        cpl_imagelist_delete(cubesData[i]);
    }
    cpl_free(cubesData);
    cpl_test_error(CPL_ERROR_NONE);
}

static void test_combine_auto_size_cube(void) {
    const int nframes = 3;
    float offsetx[] = {0.0, 2.5, -1.5};  // Mix of positive and negative offsets
    float offsety[] = {0.0, -1.5, 2.0};
    float ref_offx, ref_offy;
    int size_x = 64;  // Initial size
    int size_y = 64;
    
    /* Test normal operation */
    cpl_error_code error = eris_ifu_combine_auto_size_cube(
        offsetx, offsety, nframes, &ref_offx, &ref_offy, &size_x, &size_y);
    
    cpl_test_eq(error, CPL_ERROR_NONE);
    
    /* Verify size increases based on offsets */
    cpl_msg_warning(cpl_func,"size_x: %d",size_x);
    cpl_msg_warning(cpl_func,"size_y: %d",size_y);

    //cpl_test_gt(size_x, 64);
    //cpl_test_gt(size_y, 64);
    
    /* Test reference offset calculation */
    float expected_ref_x = (2.5 + (-1.5)) / 2.0;  // (max + min) / 2
    float expected_ref_y = (2.0 + (-1.5)) / 2.0;
    cpl_test_abs(ref_offx, expected_ref_x, 1e-6);
    cpl_test_abs(ref_offy, expected_ref_y, 1e-6);
    
    /* Test error cases */
    error = eris_ifu_combine_auto_size_cube(NULL, offsety, nframes, 
                                          &ref_offx, &ref_offy, &size_x, &size_y);
    cpl_test_eq(error, CPL_ERROR_ILLEGAL_INPUT);
    
    error = eris_ifu_combine_auto_size_cube(offsetx, NULL, nframes,
                                          &ref_offx, &ref_offy, &size_x, &size_y);
    cpl_test_eq(error, CPL_ERROR_ILLEGAL_INPUT);
    
    /* Test with invalid size */
    size_x = 32;  // Too small
    error = eris_ifu_combine_auto_size_cube(offsetx, offsety, nframes,
                                          &ref_offx, &ref_offy, &size_x, &size_y);
    cpl_test_eq(error, CPL_ERROR_ILLEGAL_INPUT);
    cpl_error_reset();
    cpl_test_error(CPL_ERROR_NONE);
}

static void test_combine_build_mask(void) {
    const int n_cubes = 2;
    const int nx_in = 32;
    const int ny_in = 32;
    const int nx_out = 64;
    const int ny_out = 64;
    
    /* Create test images */
    cpl_imagelist** cubesDataShifted = cpl_malloc(n_cubes * sizeof(cpl_imagelist*));
    for (int i = 0; i < n_cubes; i++) {
        cubesDataShifted[i] = cpl_imagelist_new();
        cpl_image* img = create_test_image(nx_in, ny_in, 1.0);
        cpl_imagelist_set(cubesDataShifted[i], img, 0);
    }
    
    /* Create output mask */
    cpl_image* mergedImgDIT = cpl_image_new(nx_out, ny_out, CPL_TYPE_DOUBLE);
    
    /* Create offsets and exposure times */
    int llx[] = {16, 24};  // Different positions for each cube
    int lly[] = {16, 24};
    double exptimes[] = {1.0, 2.0};
    
    /* Test building mask */
    cpl_error_code error = eris_ifu_combine_build_mask(
        cubesDataShifted, mergedImgDIT, n_cubes, llx, lly, exptimes);
    
    cpl_test_eq(error, CPL_ERROR_NONE);
    
    /* Verify mask values */
    double* mask_data = cpl_image_get_data_double(mergedImgDIT);
    
    /* Check overlapping region - should have sum of exposure times */
    int overlap_x = 24;
    int overlap_y = 24;
    cpl_test_abs(mask_data[overlap_x + overlap_y * nx_out], 
                 exptimes[0] + exptimes[1], 1e-10);
    
    /* Check non-overlapping region - should have single exposure time */
    int single_x = 16;
    int single_y = 16;
    cpl_test_abs(mask_data[single_x + single_y * nx_out], exptimes[0], 1e-10);
    
    /* Test error cases */
    error = eris_ifu_combine_build_mask(NULL, mergedImgDIT, n_cubes, llx, lly, exptimes);
    cpl_test_eq(error, CPL_ERROR_NULL_INPUT);
    
    error = eris_ifu_combine_build_mask(cubesDataShifted, NULL, n_cubes, llx, lly, exptimes);
    cpl_test_eq(error, CPL_ERROR_NULL_INPUT);
    cpl_error_reset();
    /* Cleanup */
    for (int i = 0; i < n_cubes; i++) {
        cpl_imagelist_delete(cubesDataShifted[i]);
    }
    cpl_free(cubesDataShifted);
    cpl_image_delete(mergedImgDIT);
    cpl_test_error(CPL_ERROR_NONE);
}

static void test_combine_build_mask_cube(void) {
    const int n_cubes = 2;
    const int nx_in = 32;
    const int ny_in = 32;
    const int nx_out = 64;
    const int ny_out = 64;
    
    /* Create test images */
    cpl_image** imagesDataShifted = cpl_malloc(n_cubes * sizeof(cpl_image*));
    for (int i = 0; i < n_cubes; i++) {
        imagesDataShifted[i] = create_test_image(nx_in, ny_in, 1.0);
    }
    
    /* Create output mask */
    cpl_image* mergedImgDIT = NULL;
    
    /* Create offsets and exposure times */
    int llx[] = {16, 24};  // Different positions for each cube
    int lly[] = {16, 24};
    double exptimes[] = {1.0, 2.0};
    
    /* Test building mask */
    cpl_error_code error = eris_ifu_combine_build_mask_cube(
        imagesDataShifted, &mergedImgDIT, llx, lly, exptimes, n_cubes, nx_out, ny_out);
    
    cpl_test_eq(error, CPL_ERROR_NONE);
    cpl_test_nonnull(mergedImgDIT);
    
    /* Verify mask values */
    double* mask_data = cpl_image_get_data_double(mergedImgDIT);
    
    /* Check overlapping region - should have sum of exposure times */
    int overlap_x = 24;
    int overlap_y = 24;
    cpl_test_abs(mask_data[overlap_x + overlap_y * nx_out], 
                 exptimes[0] + exptimes[1], 1e-10);
    
    /* Check non-overlapping region - should have single exposure time */
    int single_x = 16;
    int single_y = 16;
    cpl_test_abs(mask_data[single_x + single_y * nx_out], exptimes[0], 1e-10);
    
    /* Test error cases */
    error = eris_ifu_combine_build_mask_cube(NULL, &mergedImgDIT, llx, lly, exptimes, 
                                           n_cubes, nx_out, ny_out);
    cpl_test_eq(error, CPL_ERROR_NULL_INPUT);
    cpl_error_reset();
    /* Cleanup */
    for (int i = 0; i < n_cubes; i++) {
        cpl_image_delete(imagesDataShifted[i]);
    }
    cpl_free(imagesDataShifted);
    cpl_image_delete(mergedImgDIT);
    cpl_test_error(CPL_ERROR_NONE);
}

static void test_combine_coadd_ks_clip(void) {
    const int n_cubes = 5;
    const int nx_in = 32;
    const int ny_in = 32;
    const int nx_out = 64;
    const int ny_out = 64;
    const double kappa = 3.0;
    
    /* Create test images */
    cpl_image** imagesDataShifted = cpl_malloc(n_cubes * sizeof(cpl_image*));
    cpl_image** imagesErrorShifted = cpl_malloc(n_cubes * sizeof(cpl_image*));
    
    /* Create images with different values to test kappa-sigma clipping */
    double values[] = {10.0, 11.0, 10.5, 9.5, 20.0};  // Last value should be clipped
    double errors[] = {0.1, 0.1, 0.1, 0.1, 0.1};
    
    for (int i = 0; i < n_cubes; i++) {
        imagesDataShifted[i] = create_test_image(nx_in, ny_in, values[i]);
        imagesErrorShifted[i] = create_test_image(nx_in, ny_in, errors[i]);
    }
    
    /* Create output images */
    cpl_image* mergedImageData = NULL;
    cpl_image* mergedImageError = NULL;
    cpl_image* mergedImageDIT = cpl_image_new(nx_out, ny_out, CPL_TYPE_DOUBLE);
    
    /* Create offsets and exposure times */
    int llx[] = {16, 16, 16, 16, 16};  // Same position to test clipping
    int lly[] = {16, 16, 16, 16, 16};
    double exptimes[] = {1.0, 1.0, 1.0, 1.0, 1.0};
    
    /* Test mean mode */
    cpl_error_code error = eris_ifu_combine_coadd_ks_clip(
        n_cubes, kappa, llx, lly, exptimes,
        &mergedImageData, &mergedImageError, mergedImageDIT,
        imagesDataShifted, imagesErrorShifted,
        "MEAN", 0, nx_out, ny_out);
    
    cpl_test_eq(error, CPL_ERROR_NONE);
    cpl_test_nonnull(mergedImageData);
    cpl_test_nonnull(mergedImageError);
    
    /* Verify results - the outlier (20.0) should be clipped */
    double* data = cpl_image_get_data_double(mergedImageData);
    double expected_mean = (values[0] + values[1]) / 2.0;  // Mean of first two values
    //cpl_image_save(mergedImageData, "mergedImageData.fits", CPL_TYPE_DOUBLE, NULL, CPL_IO_CREATE);
    cpl_test_abs(data[16 + 16 * nx_out], expected_mean, 1e-10);
    
    /* Test median mode */
    error = eris_ifu_combine_coadd_ks_clip(
        n_cubes, kappa, llx, lly, exptimes,
        &mergedImageData, &mergedImageError, mergedImageDIT,
        imagesDataShifted, imagesErrorShifted,
        "MEDIAN", 0, nx_out, ny_out);
    
    cpl_test_eq(error, CPL_ERROR_NONE);
    
    /* Verify results - should get middle value */
    data = cpl_image_get_data_double(mergedImageData);
    cpl_test_abs(data[16 + 16 * nx_out], values[1], 1e-10);
    
    /* Test error cases */
    error = eris_ifu_combine_coadd_ks_clip(
        n_cubes, kappa, NULL, lly, exptimes,
        &mergedImageData, &mergedImageError, mergedImageDIT,
        imagesDataShifted, imagesErrorShifted,
        "MEAN", 0, nx_out, ny_out);
    cpl_test_eq(error, CPL_ERROR_NULL_INPUT);
    
    /* Cleanup */
    for (int i = 0; i < n_cubes; i++) {
        cpl_image_delete(imagesDataShifted[i]);
        cpl_image_delete(imagesErrorShifted[i]);
    }
    cpl_free(imagesDataShifted);
    cpl_free(imagesErrorShifted);
    cpl_image_delete(mergedImageData);
    cpl_image_delete(mergedImageError);
    cpl_image_delete(mergedImageDIT);
    cpl_test_error(CPL_ERROR_NONE);
}

static void test_combine_coadd(void) {
    const int n_cubes = 3;
    const int nx_in = 32;
    const int ny_in = 32;
    const int nx_out = 64;
    const int ny_out = 64;
    
    /* Create test images */
    cpl_image** imagesDataShifted = cpl_malloc(n_cubes * sizeof(cpl_image*));
    cpl_image** imagesErrorShifted = cpl_malloc(n_cubes * sizeof(cpl_image*));
    
    double values[] = {10.0, 11.0, 12.0};
    double errors[] = {0.1, 0.1, 0.1};
    
    for (int i = 0; i < n_cubes; i++) {
        imagesDataShifted[i] = create_test_image(nx_in, ny_in, values[i]);
        imagesErrorShifted[i] = create_test_image(nx_in, ny_in, errors[i]);
    }
    
    /* Create output images */
    cpl_image* mergedImageData = NULL;
    cpl_image* mergedImageError = NULL;
    cpl_image* mergedImageDIT = cpl_image_new(nx_out, ny_out, CPL_TYPE_DOUBLE);
    
    /* Create offsets and exposure times */
    int llx[] = {16, 16, 16};  // Same position to test averaging
    int lly[] = {16, 16, 16};
    double exptimes[] = {1.0, 1.0, 1.0};
    
    /* Test mean mode */
    cpl_error_code error = eris_ifu_combine_coadd(
        n_cubes, &mergedImageData, &mergedImageError, mergedImageDIT,
        imagesDataShifted, imagesErrorShifted, exptimes,
        llx, lly, "MEAN", nx_out, ny_out);
    
    cpl_test_eq(error, CPL_ERROR_NONE);
    cpl_test_nonnull(mergedImageData);
    cpl_test_nonnull(mergedImageError);
    
    /* Verify results - should get mean of all values */
    double* data = cpl_image_get_data_double(mergedImageData);
    double expected_mean = (values[0] + values[1] + values[2]) / 3.0;
    cpl_test_abs(data[16 + 16 * nx_out], expected_mean, 1e-10);
    
    /* Test median mode */
    error = eris_ifu_combine_coadd(
        n_cubes, &mergedImageData, &mergedImageError, mergedImageDIT,
        imagesDataShifted, imagesErrorShifted, exptimes,
        llx, lly, "MEDIAN", nx_out, ny_out);
    
    cpl_test_eq(error, CPL_ERROR_NONE);
    
    /* Verify results - should get middle value */
    data = cpl_image_get_data_double(mergedImageData);
    cpl_test_abs(data[16 + 16 * nx_out], values[1], 1e-10);
    
    /* Test error cases */
    error = eris_ifu_combine_coadd(
        n_cubes, &mergedImageData, &mergedImageError, NULL,
        imagesDataShifted, imagesErrorShifted, exptimes,
        llx, lly, "MEAN", nx_out, ny_out);
    cpl_test_eq(error, CPL_ERROR_NULL_INPUT);
    
    /* Cleanup */
    for (int i = 0; i < n_cubes; i++) {
        cpl_image_delete(imagesDataShifted[i]);
        cpl_image_delete(imagesErrorShifted[i]);
    }
    cpl_free(imagesDataShifted);
    cpl_free(imagesErrorShifted);
    cpl_image_delete(mergedImageData);
    cpl_image_delete(mergedImageError);
    cpl_image_delete(mergedImageDIT);
}

int main(void) {
    cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_WARNING);
    
    /* Test helper functions */
    test_nearest_int();
    test_get_xy_min_max();

    /* Test main functions */
    test_combine_jittered_images_null_inputs();
    test_combine_jittered_images_basic();
    test_combine_divide_DIT();
    test_combine_auto_size_cube();
    test_combine_build_mask();
    test_combine_build_mask_cube();
    //test_combine_coadd_ks_clip(); //TODO fix errors
    cpl_test_error(CPL_ERROR_NONE);
    //est_combine_coadd(); //TODO fix errors
    cpl_test_error(CPL_ERROR_NONE);
    
    return cpl_test_end(0);
}
