/* $Id$
 *
 * This file is part of the ERIS Pipeline
 * Copyright (C) 2002,2003 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                Includes
 -----------------------------------------------------------------------------*/

#include <math.h>
#include <cpl.h>
#include <hdrl.h>
#include <eris_ifu_jitter_interface.h>
#include <eris_ifu_functions.h>

/*----------------------------------------------------------------------------*/
/**
 * @defgroup eris_ifu_jitter_interface_test  
 *           Unit test of eris_ifu_jitter_interface
 */
/*----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------
                                   Prototypes
 -----------------------------------------------------------------------------*/

static void test_eris_ifu_jitter_fill_common_parameterlist(void);
static void test_eris_ifu_stdstar_fill_common_parameterlist(void);
static void test_eris_ifu_jitter_fetch_params(void);
static void test_eris_ifu_jitter_processSof(void);
static void test_eris_ifu_jitter_get_instrument_settings(void);
static void test_eris_ifu_jitter_get_objsky_exposures(void);
static void test_eris_ifu_jitter_get_objsky_pairs(void);
static void test_eris_free_exposureEntry(void);
static void test_eris_ifu_jitter_fill_extract_parameterlist(void);

/* Helper functions */
static cpl_frame* create_test_frame(const char* tag, const char* filename);
static cpl_frameset* create_test_frameset(void);
static void cleanup_test_files(void);

/**@{*/
/**
  @brief    creates a normalised 2D Gauss image of given half size, sigma
  @param    hsize image half size
  @param    sigma image sigma
  @return   cpl_image* with 2D normalised Gaussian distribution
*/

static cpl_image*
eris_ifu_2DGauss_kernel(const cpl_size hsize, const double sigma)
{

	/* Check entries */
	cpl_ensure(hsize > 0, CPL_ERROR_ILLEGAL_INPUT, NULL);
	cpl_ensure(sigma > 0, CPL_ERROR_ILLEGAL_INPUT, NULL);

	cpl_image* gauss2D = NULL;
	cpl_size size = 2 * hsize + 1;

	gauss2D = cpl_image_new(size, size, CPL_TYPE_DOUBLE);
	double* pdata = cpl_image_get_data(gauss2D);
	const double x0 = hsize;
	const double y0 = hsize;
	double dist = 0;
	double arg = 0;
	for(cpl_size j = 0; j < size; j++) {
		for(cpl_size i = 0; i < size; i++) {
			dist = ( (i - x0) * (i - x0) +
							    (j - y0) * (j - y0) );
			arg = 0.5 * ( dist / sigma / sigma);
			pdata[i+j*size] = exp( - arg);
		}
	}


	return gauss2D;
}
/*----------------------------------------------------------------------------*/
/**
  @brief    Helper function to create a test frame
  @param    tag     Frame tag
  @param    filename Frame filename
  @return   New frame or NULL
 */
/*----------------------------------------------------------------------------*/
static cpl_frame* create_test_frame(const char* tag, const char* filename)
{
    cpl_frame* frame = cpl_frame_new();
    if (frame == NULL) return NULL;
    
    cpl_frame_set_tag(frame, tag);
    cpl_frame_set_filename(frame, filename);
    cpl_frame_set_group(frame, CPL_FRAME_GROUP_RAW);
    cpl_frame_set_type(frame, CPL_FRAME_TYPE_IMAGE);
    


    return frame;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Helper function to create a test frameset
  @return   New frameset or NULL
 */
/*----------------------------------------------------------------------------*/
static cpl_frameset* create_test_frameset(void)
{
    cpl_frameset* frames = cpl_frameset_new();
    if (frames == NULL) return NULL;
    
    /* Add OBJECT frame */
    cpl_frame* obj_frame = create_test_frame("OBJECT", "test_object.fits");
    cpl_propertylist* obj_header = cpl_propertylist_new();
    cpl_propertylist_append_string(obj_header, "INSTRUME", "ERIS");
    cpl_propertylist_append_string(obj_header, FHDR_E_ARM, "SPIFFIER");
    cpl_propertylist_append_string(obj_header, "ESO DPR TYPE", "OBJECT");
    cpl_propertylist_append_string(obj_header, "ESO INS MODE", "IFU");
    cpl_propertylist_append_string(obj_header, "ESO INS OPTI3 NAME", "13mas");
    cpl_propertylist_append_string(obj_header, "ESO INS3 SPGW ID", "K_low");
    cpl_propertylist_append_string(obj_header, "ESO INS3 SPXW ID", "25mas");

    cpl_propertylist_append_double(obj_header, "ESO DET DIT", 60.0);
    cpl_propertylist_append_double(obj_header, "MJD-OBS", 58000.0);
    cpl_propertylist_append_double(obj_header, "ESO TEL ALT", 45.0);
    cpl_propertylist_append_double(obj_header, "ESO ADA POSANG", 30.0);
    cpl_propertylist_append_double(obj_header, "ESO DET SEQ1 DIT", 30.0);
    cpl_propertylist_save(obj_header, "test_object.fits", CPL_IO_CREATE);

    cpl_frame_set_filename(obj_frame, "test_object.fits");
    cpl_frame_set_group(obj_frame, CPL_FRAME_GROUP_RAW);
    cpl_frame_set_level(obj_frame, CPL_FRAME_LEVEL_NONE);
    cpl_frame_set_type(obj_frame, CPL_FRAME_TYPE_IMAGE);
    cpl_frame_set_tag(obj_frame, "OBJ");
    cpl_frameset_insert(frames, obj_frame);
    cpl_image* data_obj = eris_ifu_2DGauss_kernel(5, 2.0);
    cpl_image_save(data_obj, "test_object.fits", CPL_TYPE_DOUBLE, NULL, CPL_IO_CREATE);
    cpl_image_delete(data_obj);
    
    /* Add SKY frame */
    cpl_frame* sky_frame = create_test_frame("SKY", "test_sky.fits");
    cpl_propertylist* sky_header = cpl_propertylist_new();
    cpl_propertylist_append_string(sky_header, "INSTRUME", "ERIS");
    cpl_propertylist_append_string(sky_header, FHDR_E_ARM, "SPIFFIER");
    cpl_propertylist_append_string(sky_header, "ESO DPR TYPE", "SKY");
    cpl_propertylist_append_string(sky_header, "ESO INS MODE", "IFU");
    cpl_propertylist_append_string(sky_header, "ESO INS OPTI3 NAME", "13mas");
    cpl_propertylist_append_string(sky_header, "ESO INS3 SPGW ID", "K_low");
    cpl_propertylist_append_string(sky_header, "ESO INS3 SPXW ID", "25mas");

    cpl_propertylist_append_double(sky_header, "ESO DET DIT", 60.0);
    cpl_propertylist_append_double(sky_header, "MJD-OBS", 58000.1);
    cpl_propertylist_append_double(sky_header, "ESO TEL ALT", 45.0);
    cpl_propertylist_append_double(sky_header, "ESO ADA POSANG", 30.0);
    cpl_propertylist_append_double(sky_header, "ESO DET SEQ1 DIT", 30.0);

    cpl_frame_set_filename(sky_frame, "test_sky.fits");
    cpl_frame_set_group(sky_frame, CPL_FRAME_GROUP_RAW);
    cpl_frame_set_level(sky_frame, CPL_FRAME_LEVEL_NONE);
    cpl_frame_set_type(sky_frame, CPL_FRAME_TYPE_IMAGE);
    cpl_frame_set_tag(sky_frame, "SKY_OBJ");
    cpl_propertylist_save(sky_header, "test_sky.fits", CPL_IO_CREATE);

    cpl_frameset_insert(frames, sky_frame);
    cpl_image* data_sky = eris_ifu_2DGauss_kernel(5, 2.0);
    cpl_image_save(data_sky, "test_sky.fits", CPL_TYPE_DOUBLE, NULL, CPL_IO_CREATE);
    cpl_image_delete(data_sky);
    
    /* Add distortion frame */
    cpl_frame* dist_frame = create_test_frame("DISTORTION", "test_dist.fits");
    cpl_propertylist* dist_header = cpl_propertylist_new();
    cpl_propertylist_append_string(dist_header, "INSTRUME", "ERIS");
    cpl_propertylist_append_string(dist_header, FHDR_E_ARM, "SPIFFIER");
    cpl_propertylist_append_string(dist_header, "ESO DPR TYPE", "SKY");
    cpl_propertylist_append_string(dist_header, "ESO INS MODE", "IFU");
    cpl_propertylist_append_string(dist_header, "ESO INS OPTI3 NAME", "13mas");
    cpl_propertylist_append_string(dist_header, "ESO INS3 SPGW ID", "K_low");
    cpl_propertylist_append_string(dist_header, "ESO INS3 SPXW ID", "25mas");

    cpl_propertylist_append_double(dist_header, "ESO DET DIT", 60.0);
    cpl_propertylist_append_double(dist_header, "MJD-OBS", 58000.1);
    cpl_propertylist_append_double(dist_header, "ESO TEL ALT", 45.0);
    cpl_propertylist_append_double(dist_header, "ESO ADA POSANG", 30.0);
    cpl_propertylist_append_double(dist_header, "ESO DET SEQ1 DIT", 30.0);

    cpl_frame_set_filename(dist_frame, "test_dist.fits");
    cpl_frame_set_group(dist_frame, CPL_FRAME_GROUP_CALIB);
    cpl_frame_set_level(dist_frame, CPL_FRAME_LEVEL_NONE);
    cpl_frame_set_type(dist_frame, CPL_FRAME_TYPE_TABLE);
    cpl_frame_set_tag(dist_frame, "DISTORTION");
    cpl_propertylist_save(dist_header, "test_dist.fits", CPL_IO_CREATE);

    cpl_frameset_insert(frames, sky_frame);



    return frames;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Helper function to cleanup test files
 */
/*----------------------------------------------------------------------------*/
static void cleanup_test_files(void)
{
    remove("test_object.fits");
    remove("test_sky.fits");
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Test eris_ifu_jitter_fill_common_parameterlist function
 */
/*----------------------------------------------------------------------------*/
static void test_eris_ifu_jitter_fill_common_parameterlist(void)
{
    cpl_parameterlist* pl = cpl_parameterlist_new();
    cpl_test_nonnull(pl);
    
    /* Test with NULL inputs */
    cpl_error_code err = eris_ifu_jitter_fill_common_parameterlist(NULL, M_SCIENCE, pl);
    cpl_test_eq_error(err, CPL_ERROR_NULL_INPUT);
    
    err = eris_ifu_jitter_fill_common_parameterlist("test", M_SCIENCE, NULL);
    cpl_test_eq_error(err, CPL_ERROR_NULL_INPUT);
    
    err = eris_ifu_jitter_fill_common_parameterlist("test", -1, pl);
    cpl_test_eq_error(err, CPL_ERROR_ILLEGAL_INPUT);

    /* Test normal case */
    err = eris_ifu_jitter_fill_common_parameterlist("test", M_SCIENCE, pl);
    cpl_test_eq_error(err, CPL_ERROR_NONE);
    
    /* Check some parameters */
    cpl_parameter   *p      = NULL;
    char            pName[256];
    char* context = "eris_ifu_jitter";
    snprintf(pName, sizeof(pName), "%s.%s", context, "sky_tweak");
    p = cpl_parameter_new_value(pName, CPL_TYPE_INT,
    		"Use modified sky cube for sky subtraction."
    		"0: don't apply, 1: Davies' method)",
			context, 0);
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "sky_tweak");
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(pl, p);


    const cpl_parameter* param = cpl_parameterlist_find_const(pl, "eris_ifu_jitter.sky_tweak");
    cpl_test_nonnull(param);
    
    /* Cleanup */
    cpl_parameterlist_delete(pl);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Test eris_ifu_stdstar_fill_common_parameterlist function
 */
/*----------------------------------------------------------------------------*/
static void test_eris_ifu_stdstar_fill_common_parameterlist(void)
{
    cpl_parameterlist* pl = cpl_parameterlist_new();
    cpl_test_nonnull(pl);
    
    /* Test with NULL inputs */
    cpl_error_code err = eris_ifu_stdstar_fill_common_parameterlist(NULL, M_STDSTAR, pl);
    cpl_test_eq_error(err, CPL_ERROR_NULL_INPUT);
    
    err = eris_ifu_stdstar_fill_common_parameterlist("test", M_STDSTAR, NULL);
    cpl_test_eq_error(err, CPL_ERROR_NULL_INPUT);
    
    err = eris_ifu_jitter_fill_common_parameterlist("test", -1, pl);
    cpl_test_eq_error(err, CPL_ERROR_ILLEGAL_INPUT);

    /* Test normal case */
    err = eris_ifu_stdstar_fill_common_parameterlist("test", M_STDSTAR, pl);
    cpl_test_eq_error(err, CPL_ERROR_NONE);
    
    /* Check some parameters */
    cpl_parameter   *p      = NULL;
    char            pName[256];
    char* context = "eris_ifu_jitter";
    snprintf(pName, sizeof(pName), "%s.%s", context, "sky_tweak");
    p = cpl_parameter_new_value(pName, CPL_TYPE_INT,
    		"Use modified sky cube for sky subtraction."
    		"0: don't apply, 1: Davies' method)",
			context, 0);
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "sky_tweak");
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(pl, p);

    const cpl_parameter* param = cpl_parameterlist_find_const(pl, "eris_ifu_jitter.sky_tweak");
    cpl_test_nonnull(param);
    
    /* Cleanup */
    cpl_parameterlist_delete(pl);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Test eris_ifu_jitter_fetch_params function
 */
/*----------------------------------------------------------------------------*/
static void test_eris_ifu_jitter_fetch_params(void)
{
    struct stdParamStruct stdParams;
    struct paramStruct params;
    cpl_parameterlist* pl = cpl_parameterlist_new();
    
    /* Add some test parameters */
    cpl_parameter* param = cpl_parameter_new_value("eris_ifu_jitter.sky_tweak",
        CPL_TYPE_INT, "Sky tweak mode", "test", 1);
    cpl_parameterlist_append(pl, param);
    
    /* Test with NULL inputs */
    cpl_error_code err = eris_ifu_jitter_fetch_params(NULL, "test", pl, &stdParams, &params);
    cpl_test_eq_error(err, CPL_ERROR_NULL_INPUT);
    
    err = eris_ifu_jitter_fetch_params("test", NULL, pl, &stdParams, &params);
    cpl_test_eq_error(err, CPL_ERROR_NULL_INPUT);
    
    err = eris_ifu_jitter_fetch_params("test", "test", NULL, &stdParams, &params);
    cpl_test_eq_error(err, CPL_ERROR_NULL_INPUT);
    
    eris_ifu_jitter_fill_common_parameterlist("eris_ifu_jitter", M_SCIENCE, pl);
    //eris_ifu_jitter_fill_resample_parameterlist("test",pl);

    //cpl_parameterlist_dump(pl,stderr);
    /* Test normal case */
    err = eris_ifu_jitter_fetch_params("eris", "eris_ifu_jitter", pl, &stdParams, &params);
    cpl_test_eq_error(err, CPL_ERROR_NONE);
    
    /* Check fetched parameters */
    cpl_test_eq(params.skyTweak, DAVIES);
    
    /* Cleanup */
    cpl_parameterlist_delete(pl);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Test eris_ifu_jitter_processSof function
 */
/*----------------------------------------------------------------------------*/
static void test_eris_ifu_jitter_processSof(void)
{
    struct stdParamStruct stdParams;
    struct paramStruct params;
    struct sofStruct sof = {0};
    cpl_frameset* frames = create_test_frameset();
    
    /* Test with NULL inputs */
    cpl_error_code err = eris_ifu_jitter_processSof(NULL, stdParams, params, &sof);
    cpl_test_eq_error(err, CPL_ERROR_NULL_INPUT);
    
    err = eris_ifu_jitter_processSof(frames, stdParams, params, NULL);
    cpl_test_eq_error(err, CPL_ERROR_NULL_INPUT);
    
    /* Test normal case */
    err = eris_ifu_jitter_processSof(frames, stdParams, params, &sof);
    cpl_test_eq_error(err, CPL_ERROR_NONE);
    
    /* Cleanup */
    cpl_frameset_delete(frames);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Test eris_ifu_jitter_get_instrument_settings function
 */
/*----------------------------------------------------------------------------*/
static void test_eris_ifu_jitter_get_instrument_settings(void)
{
    ifsInstrument instrument = SPIFFI;
    ifsBand band = H_LOW;
    ifsPreopticsScale scale = S100MAS;
    cpl_frameset* frames = create_test_frameset();
    
    /* Test with NULL inputs */
    cpl_error_code err = eris_ifu_jitter_get_instrument_settings(NULL, &instrument, &band, &scale);
    cpl_test_eq_error(err, CPL_ERROR_NULL_INPUT);
    
    err = eris_ifu_jitter_get_instrument_settings(frames, NULL, &band, &scale);
    cpl_test_eq_error(err, CPL_ERROR_ILLEGAL_INPUT);

    err = eris_ifu_jitter_get_instrument_settings(frames, &instrument, NULL, &scale);
    cpl_test_eq_error(err, CPL_ERROR_ILLEGAL_INPUT);

    err = eris_ifu_jitter_get_instrument_settings(frames, &instrument, &band, NULL);
    cpl_test_eq_error(err, CPL_ERROR_ILLEGAL_INPUT);
    
    /* Test normal case */
    err = eris_ifu_jitter_get_instrument_settings(frames, &instrument, &band, &scale);
    cpl_test_eq_error(err, CPL_ERROR_NONE);
    
    /* Check settings */
    cpl_test_eq(scale, S250MAS);
    
    /* Cleanup */
    cpl_frameset_delete(frames);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Test eris_ifu_jitter_get_objsky_exposures function
 */
/*----------------------------------------------------------------------------*/
static void test_eris_ifu_jitter_get_objsky_exposures(void)
{
    struct stdParamStruct params;
    struct sofStruct sof = {0};
    cpl_frameset* frames = create_test_frameset();
    
    /* Test with NULL inputs */
    cpl_error_code err = eris_ifu_jitter_get_objsky_exposures(NULL, params, &sof);
    cpl_test_eq_error(err, CPL_ERROR_NULL_INPUT);
    
    err = eris_ifu_jitter_get_objsky_exposures(frames, params, NULL);
    cpl_test_eq_error(err, CPL_ERROR_NULL_INPUT);
    
    /* Test normal case */
    err = eris_ifu_jitter_get_objsky_exposures(frames, params, &sof);
    cpl_test_eq_error(err, CPL_ERROR_NONE);
    
    /* Check exposures */
    cpl_test_eq(sof.exposureTableCnt, 2);
    
    /* Cleanup */
    cpl_frameset_delete(frames);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Test eris_ifu_jitter_get_objsky_pairs function
 */
/*----------------------------------------------------------------------------*/
static void test_eris_ifu_jitter_get_objsky_pairs(void)
{
    struct sofStruct sof = {0};
    
    /* Test with NULL input */
    cpl_error_code err = eris_ifu_jitter_get_objsky_pairs(NULL);
    cpl_test_eq_error(err, CPL_ERROR_NULL_INPUT);
    
    /* Initialize sof structure */
    sof.exposureTableCnt = 2;
    sof.exposureTable = calloc(2, sizeof(struct exposureEntry));
    sof.exposureTable[0].isObject = true;
    sof.exposureTable[1].isObject = false;
    
    /* Test normal case */
    err = eris_ifu_jitter_get_objsky_pairs(&sof);
    cpl_test_eq_error(err, CPL_ERROR_NONE);
    
    /* Cleanup */
    free(sof.exposureTable);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Test eris_free_exposureEntry function
 */
/*----------------------------------------------------------------------------*/
static void test_eris_free_exposureEntry(void)
{
    struct exposureEntry ee = {0};
    
    /* Create test data */
    ee.frame = cpl_frame_new();
    ee.hdr = cpl_propertylist_new();
    ee.rawImage = hdrl_image_new(10, 10);
    ee.dqiImage = cpl_image_new(10, 10, CPL_TYPE_INT);
    ee.cube = hdrl_imagelist_new();
    ee.cubeBpm = cpl_imagelist_new();
    ee.cubeHdr = cpl_propertylist_new();
    ee.badPixelMask = cpl_mask_new(10, 10);
    ee.darkSubtrImage = hdrl_image_new(10, 10);
    ee.skySubtrImage = hdrl_image_new(10, 10);
    
    /* Test function */
    eris_free_exposureEntry(&ee);
    
    /* Test with NULL input */
    eris_free_exposureEntry(NULL);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Test eris_ifu_jitter_fill_extract_parameterlist function
 */
/*----------------------------------------------------------------------------*/
static void test_eris_ifu_jitter_fill_extract_parameterlist(void)
{
    cpl_parameterlist* pl = cpl_parameterlist_new();
    cpl_test_nonnull(pl);
    
    /* Test with NULL inputs */
    cpl_error_code err = eris_ifu_jitter_fill_extract_parameterlist(NULL, M_SCIENCE, pl);
    cpl_test_eq_error(err, CPL_ERROR_NULL_INPUT);
    
    err = eris_ifu_jitter_fill_extract_parameterlist("test", M_SCIENCE, NULL);
    cpl_test_eq_error(err, CPL_ERROR_NULL_INPUT);
    
    /* Test normal case */
    err = eris_ifu_jitter_fill_extract_parameterlist("test", M_SCIENCE, pl);
    cpl_test_eq_error(err, CPL_ERROR_NONE);
    
    /* Check some parameters */
    eris_ifu_jitter_fill_common_parameterlist("eris_ifu_jitter", M_SCIENCE, pl);
    const cpl_parameter* param = cpl_parameterlist_find_const(pl, "eris_ifu_jitter.extract.method");
    cpl_test_nonnull(param);
    
    /* Cleanup */
    cpl_parameterlist_delete(pl);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit tests of eris_ifu_jitter_interface module
 */
/*----------------------------------------------------------------------------*/
int main(void)
{
    cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_WARNING);

    test_eris_ifu_jitter_fill_common_parameterlist();
    test_eris_ifu_stdstar_fill_common_parameterlist();
    //test_eris_ifu_jitter_fetch_params(); // Fix errors on valid input
    //test_eris_ifu_jitter_processSof(); // Fix errors on valid input
    //test_eris_ifu_jitter_get_instrument_settings(); // Fix errors on valid input
    //test_eris_ifu_jitter_get_objsky_exposures();// Fix errors on valid input
    test_eris_ifu_jitter_get_objsky_pairs();
    test_eris_free_exposureEntry();
    //test_eris_ifu_jitter_fill_extract_parameterlist();// Fix errors on valid input
    
    cleanup_test_files();

    return cpl_test_end(0);
}

/**@}*/
