/* $Id$
 *
 * This file is part of the ERIS Pipeline
 * Copyright (C) 2002,2003 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                Includes
 -----------------------------------------------------------------------------*/

#include <math.h>
#include <cpl.h>
#include <hdrl.h>
#include <eris_ifu_jitter_static.h>
#include <eris_ifu_functions.h>
#include <eris_ifu_jitter_interface.h>

/*----------------------------------------------------------------------------*/
/**
 * @defgroup eris_ifu_jitter_static_test  
 *           Unit test of eris_ifu_jitter_static
 */
/*----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------
                                   Prototypes
 -----------------------------------------------------------------------------*/

static void test_eris_ifu_jitter_get_obj_type(void);
static void test_eris_ifu_jitter_get_sky_type(void);
static void test_eris_ifu_jitter_get_coadd_obj_type(void);
static void test_eris_ifu_jitter_get_procatg_and_filename(void);
static void test_eris_ifu_jitter_reconstruct_cube(void);
static void test_eris_ifu_jitter_subtract_background(void);
static void test_eris_ifu_jitter_build_cube(void);
static void test_eris_ifu_dar_correction(void);
static void test_eris_ifu_dar_cpl_shift(void);
static void test_eris_ifu_dar_gsl_shift(void);
static void test_eris_frameset_count_tag(void);

/* Helper functions */
static cpl_image* create_test_image(cpl_size nx, cpl_size ny);
static hdrl_image* create_test_hdrl_image(cpl_size nx, cpl_size ny);
static cpl_frameset* create_test_frameset(void);
static void cleanup_test_files(void);

/**@{*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Helper function to create a test frameset
  @return   New frameset or NULL
 */
/*----------------------------------------------------------------------------*/
static cpl_propertylist* create_test_header(void)
{

    cpl_propertylist* header = cpl_propertylist_new();
    cpl_propertylist_append_string(header, "INSTRUME", "ERIS");
    cpl_propertylist_append_string(header, FHDR_E_ARM, "SPIFFIER");
    cpl_propertylist_append_string(header, "ESO DPR TYPE", "OBJECT");
    cpl_propertylist_append_string(header, "ESO INS MODE", "IFU");
    cpl_propertylist_append_string(header, "ESO INS OPTI3 NAME", "13mas");
    cpl_propertylist_append_string(header, "ESO INS3 SPGW ID", "K_low");
    cpl_propertylist_append_string(header, "ESO INS3 SPXW ID", "25mas");

    cpl_propertylist_append_double(header, "ESO DET DIT", 60.0);
    cpl_propertylist_append_double(header, "MJD-OBS", 58000.0);
    cpl_propertylist_append_double(header, "ESO TEL ALT", 45.0);
    cpl_propertylist_append_double(header, "ESO ADA POSANG", 30.0);
    cpl_propertylist_append_double(header, "ESO DET SEQ1 DIT", 30.0);
    cpl_propertylist_append_double(header, "ESO TEL AIRM START", 1.123);
    cpl_propertylist_append_double(header, "ESO TEL AIRM END", 1.122);
    cpl_propertylist_append_double(header, "ESO TEL PARANG START", -169.208);
    cpl_propertylist_append_double(header, "ESO TEL PARANG END", -170.094);
    cpl_propertylist_append_double(header, "ESO TEL AMBI TEMP", 14.43);
    cpl_propertylist_append_double(header, "ESO TEL AMBI RHUM", 21.5);
    cpl_propertylist_append_double(header, "ESO TEL AMBI PRES START", 744.70);
    cpl_propertylist_append_double(header, "ESO TEL AMBI PRES END", 744.70);
    cpl_propertylist_append_double(header, "CRVAL1", 187.277891762885);
    cpl_propertylist_append_double(header, "CRVAL2", 2.05223023599808);
    cpl_propertylist_append_double(header, "CRVAL3", 1.93395);

    cpl_propertylist_append_int(header, "NAXIS1", 64);
    cpl_propertylist_append_int(header, "NAXIS2", 64);
    cpl_propertylist_append_int(header, "NAXIS3", 2150);

    cpl_propertylist_append_double(header, "CRPIX1", 32);
    cpl_propertylist_append_double(header, "CRPIX2", 32);
    cpl_propertylist_append_double(header, "CRPIX3", 1);

    cpl_propertylist_append_double(header, "CDELT3", 0.00001);

    //cpl_propertylist_save(header, "test_header.fits", CPL_IO_CREATE);

    return header;
}



/*----------------------------------------------------------------------------*/
/**
  @brief    Helper function to create a test image
  @param    nx      X dimension
  @param    ny      Y dimension
  @return   New image or NULL
 */
/*----------------------------------------------------------------------------*/
static cpl_image* create_test_image(cpl_size nx, cpl_size ny)
{
    cpl_image* img = cpl_image_new(nx, ny, CPL_TYPE_DOUBLE);
    if (img == NULL) return NULL;
    
    /* Fill with test pattern */
    for (cpl_size y = 1; y <= ny; y++) {
        for (cpl_size x = 1; x <= nx; x++) {
            double val = (double)(x + y);
            cpl_image_set(img, x, y, val);
        }
    }
    
    return img;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Helper function to create a test HDRL image
  @param    nx      X dimension
  @param    ny      Y dimension
  @return   New HDRL image or NULL
 */
/*----------------------------------------------------------------------------*/
static hdrl_image* create_test_hdrl_image(cpl_size nx, cpl_size ny)
{
    cpl_image* data = create_test_image(nx, ny);
    cpl_image* error = cpl_image_new(nx, ny, CPL_TYPE_DOUBLE);
    cpl_mask* mask = cpl_mask_new(nx, ny);
    
    /* Fill error with sqrt of data */
    for (cpl_size y = 1; y <= ny; y++) {
        for (cpl_size x = 1; x <= nx; x++) {
            double val = cpl_image_get(data, x, y, NULL);
            cpl_image_set(error, x, y, sqrt(val));
        }
    }
    cpl_image_set_bpm(data,mask);
    
    hdrl_image* img = hdrl_image_create(data, error);
    cpl_image_delete(data);
    cpl_image_delete(error);

    return img;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Helper function to create a test frameset
  @return   New frameset or NULL
 */
/*----------------------------------------------------------------------------*/
static cpl_frameset* create_test_frameset(void)
{
    cpl_frameset* frames = cpl_frameset_new();
    if (frames == NULL) return NULL;
    
    /* Create test image */
    cpl_image* img = create_test_image(10, 10);
    cpl_image_save(img, "test_cube.fits", CPL_TYPE_DOUBLE, NULL, CPL_IO_CREATE);
    cpl_image_delete(img);
    
    /* Add OBJECT frame */
    cpl_frame* obj_frame = cpl_frame_new();
    cpl_frame_set_filename(obj_frame, "test_cube.fits");
    cpl_frame_set_tag(obj_frame, "OBJECT_CUBE");
    cpl_frame_set_group(obj_frame, CPL_FRAME_GROUP_RAW);
    cpl_frame_set_type(obj_frame, CPL_FRAME_TYPE_IMAGE);
    cpl_frameset_insert(frames, obj_frame);
    
    return frames;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Helper function to cleanup test files
 */
/*----------------------------------------------------------------------------*/
static void cleanup_test_files(void)
{
    remove("test_cube.fits");
    //remove("test_output.fits");
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Test eris_ifu_jitter_get_obj_type function
 */
/*----------------------------------------------------------------------------*/
static void test_eris_ifu_jitter_get_obj_type(void)
{
    /* Test different modes */
    cubeType type = eris_ifu_jitter_get_obj_type(SCIENCE);
    cpl_test_eq(type, OBJECT_CUBE);
    
    type = eris_ifu_jitter_get_obj_type(PSF);
    cpl_test_eq(type, PSF_CUBE);
    
    type = eris_ifu_jitter_get_obj_type(STD);
    cpl_test_eq(type, STD_CUBE);
    
    type = eris_ifu_jitter_get_obj_type(STD_FLUX);
    cpl_test_eq(type, STD_FLUX_CUBE);
    
    type = eris_ifu_jitter_get_obj_type(MODE_PUPIL);
    cpl_test_eq(type, OBJECT_CUBE);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Test eris_ifu_jitter_get_sky_type function
 */
/*----------------------------------------------------------------------------*/
static void test_eris_ifu_jitter_get_sky_type(void)
{
    /* Test different modes */
    cubeType type = eris_ifu_jitter_get_sky_type(SCIENCE);
    cpl_test_eq(type, SKY_OBJECT_CUBE);
    
    type = eris_ifu_jitter_get_sky_type(PSF);
    cpl_test_eq(type, SKY_PSF_CUBE);
    
    type = eris_ifu_jitter_get_sky_type(STD);
    cpl_test_eq(type, SKY_STD_CUBE);
    
    type = eris_ifu_jitter_get_sky_type(STD_FLUX);
    cpl_test_eq(type, SKY_STD_FLUX_CUBE);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Test eris_ifu_jitter_get_coadd_obj_type function
 */
/*----------------------------------------------------------------------------*/
static void test_eris_ifu_jitter_get_coadd_obj_type(void)
{
    /* Test different types */
    cubeType type = eris_ifu_jitter_get_coadd_obj_type(OBJECT_CUBE);
    cpl_test_eq(type, OBJECT_CUBE_COADD);
    
    type = eris_ifu_jitter_get_coadd_obj_type(PSF_CUBE);
    cpl_test_eq(type, PSF_CUBE_COADD);
    
    type = eris_ifu_jitter_get_coadd_obj_type(STD_CUBE);
    cpl_test_eq(type, STD_CUBE_COADD);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Test eris_ifu_jitter_get_procatg_and_filename function
 */
/*----------------------------------------------------------------------------*/
static void test_eris_ifu_jitter_get_procatg_and_filename(void)
{
    char* proCatg = NULL;
    char* filenamePrefix = NULL;
    
    /* Test with NULL inputs */
    cpl_error_code err = eris_ifu_jitter_get_procatg_and_filename(OBJECT_CUBE, NULL, &filenamePrefix);
    cpl_test_eq_error(err, CPL_ERROR_NULL_INPUT);
    
    err = eris_ifu_jitter_get_procatg_and_filename(OBJECT_CUBE, &proCatg, NULL);
    cpl_test_eq_error(err, CPL_ERROR_NULL_INPUT);
    
    /* Test normal case */
    err = eris_ifu_jitter_get_procatg_and_filename(OBJECT_CUBE, &proCatg, &filenamePrefix);
    cpl_test_eq_error(err, CPL_ERROR_NONE);
    cpl_test_nonnull(proCatg);
    cpl_test_nonnull(filenamePrefix);
    
    /* Cleanup */
    cpl_free(proCatg);
    cpl_free(filenamePrefix);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Test eris_ifu_jitter_reconstruct_cube function
 */
/*----------------------------------------------------------------------------*/
static void test_eris_ifu_jitter_reconstruct_cube(void)
{
	cpl_size sx = 10;
	cpl_size sy = 10;
    cpl_image* img = create_test_image(sx, sy);
    cpl_vector* distances = cpl_vector_new(SLITLET_CNT-1);
    cpl_bivector* positions = cpl_bivector_new(SLITLET_CNT-1);
    
    /* Test with NULL inputs */
    cpl_imagelist* cube;
    cube = eris_ifu_jitter_reconstruct_cube(NULL, DISTANCES, 0, 1.0, distances, positions);
    cpl_test_null(cube);
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    
    cube = eris_ifu_jitter_reconstruct_cube(img, DISTANCES, 0, 1.0, NULL, positions);
    cpl_test_null(cube);
    //cpl_test_error(CPL_ERROR_NULL_INPUT);

    cube = eris_ifu_jitter_reconstruct_cube(img, DISTANCES, 0, 1.0, distances, NULL);
    cpl_test_null(cube);
    //cpl_test_error(CPL_ERROR_NULL_INPUT);

    cube = eris_ifu_jitter_reconstruct_cube(img, 5, 0, 1.0, distances, positions);
    cpl_test_null(cube);
    cpl_test_error(CPL_ERROR_ILLEGAL_INPUT);

    cube = eris_ifu_jitter_reconstruct_cube(img, DISTANCES, 0, -1.0, distances, positions);
    cpl_test_null(cube);
    cpl_test_error(CPL_ERROR_ILLEGAL_INPUT);

    cube = eris_ifu_jitter_reconstruct_cube(img, DISTANCES, -6, 1.0, distances, positions);
    cpl_test_null(cube);
    cpl_test_error(CPL_ERROR_ILLEGAL_INPUT);


    /* Test normal case */
    cube = eris_ifu_jitter_reconstruct_cube(img, DISTANCES, 0, 1.0, distances, positions);
    cpl_test_nonnull(cube);
    
    /* Cleanup */
    cpl_image_delete(img);
    cpl_vector_delete(distances);
    cpl_bivector_delete(positions);
    cpl_imagelist_delete(cube);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Test eris_ifu_jitter_subtract_background function
 */
/*----------------------------------------------------------------------------*/
static void test_eris_ifu_jitter_subtract_background(void)
{
    struct sofStruct sof = {0};
    sof.exposureTableCnt = 1;
    sof.exposureTable = calloc(1, sizeof(struct exposureEntry));
    sof.exposureTable[0].rawImage = create_test_hdrl_image(10, 10);
    
    /* Test with NULL inputs */
    hdrl_image* result = eris_ifu_jitter_subtract_background(DAVIES, 0, NULL, 1);
    cpl_test_null(result);
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    
    /* Test normal case */
    result = eris_ifu_jitter_subtract_background(DAVIES, 0, &sof, 1);
    cpl_test_nonnull(result);
    
    /* Cleanup */
    hdrl_image_delete(result);
    hdrl_image_delete(sof.exposureTable[0].rawImage);
    free(sof.exposureTable);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Test eris_ifu_jitter_build_cube function
 */
/*----------------------------------------------------------------------------*/
static void test_eris_ifu_jitter_build_cube(void)
{
    struct sofStruct sof = {0};
    struct paramStruct params = {0};
    hdrl_image* img = create_test_hdrl_image(10, 10);
    cpl_polynomial *ohLambdaCorrection = cpl_polynomial_new(3);
    
    /* Test with NULL inputs */
    hdrl_imagelist* cube = eris_ifu_jitter_build_cube(NULL, 0, &sof, params, false, NULL, 1);
    cpl_test_null(cube);
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    
    cube = eris_ifu_jitter_build_cube(img, 0, &sof, params, false, NULL, 1);
    cpl_test_null(cube);
    cpl_test_error(CPL_ERROR_NULL_INPUT);


    /* Test normal case */
    //cube = eris_ifu_jitter_build_cube(img, 0, &sof, params, false, ohLambdaCorrection, 1); //TODO fix seg fault
    //cpl_test_nonnull(cube);
    
    /* Cleanup */
    hdrl_image_delete(img);
    hdrl_imagelist_delete(cube);
    cpl_polynomial_delete(ohLambdaCorrection);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Test eris_ifu_dar_correction function
 */
/*----------------------------------------------------------------------------*/
static void test_eris_ifu_dar_correction(void)
{
    hdrl_imagelist* cube = hdrl_imagelist_new();
    cpl_size sx = 64;
    cpl_size sy = 64;
    hdrl_image* img = create_test_hdrl_image(sx, sy);
    hdrl_imagelist_set(cube, img, 0);
    cpl_propertylist* hdr = create_test_header();
    
    /* Test with NULL inputs */
    cpl_error_code err;
    err = eris_ifu_dar_correction(NULL, hdr, 1, 1.0, 10);
    cpl_test_eq_error(err, CPL_ERROR_NULL_INPUT);
    
    err = eris_ifu_dar_correction(cube, NULL, 1, -1.0, 10);
    cpl_test_eq_error(err, CPL_ERROR_NULL_INPUT);

    err = eris_ifu_dar_correction(cube, hdr, 1, 1.0, -10);
    cpl_test_eq_error(err, CPL_ERROR_ILLEGAL_INPUT);

    /* Test normal case */
    err = eris_ifu_dar_correction(cube, hdr, 1, 1.0, 10);
    cpl_test_eq_error(err, CPL_ERROR_NONE);
    
    /* Cleanup */
    hdrl_imagelist_delete(cube);
    cpl_propertylist_delete(hdr);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Test eris_ifu_dar_cpl_shift function
 */
/*----------------------------------------------------------------------------*/
static void test_eris_ifu_dar_cpl_shift(void)
{
    cpl_image* in = create_test_image(10, 10);
    cpl_image* out1 = cpl_image_new(10, 10, CPL_TYPE_DOUBLE);
    cpl_image* out2 = cpl_image_new(10, 10, CPL_TYPE_DOUBLE);
    
    /* Test with NULL inputs */
    cpl_error_code err;
    err = eris_ifu_dar_cpl_shift(NULL, out1, out2, 1.0, 1.0, CPL_KERNEL_DEFAULT, 1.0, 3);
    cpl_test_eq_error(err, CPL_ERROR_NULL_INPUT);
    
    err = eris_ifu_dar_cpl_shift(in, NULL, out2, 1.0, 1.0, CPL_KERNEL_DEFAULT, 1.0, 3);
    cpl_test_eq_error(err, CPL_ERROR_NULL_INPUT);

    err = eris_ifu_dar_cpl_shift(in, out1, NULL, 1.0, 1.0, CPL_KERNEL_DEFAULT, 1.0, 3);
    cpl_test_eq_error(err, CPL_ERROR_NULL_INPUT);

    /* Test normal case */
    err = eris_ifu_dar_cpl_shift(in, out1, out2, 1.0, 1.0, CPL_KERNEL_DEFAULT, 1.0, 3);
    cpl_test_eq_error(err, CPL_ERROR_NONE);
    
    /* Cleanup */
    cpl_image_delete(in);
    cpl_image_delete(out1);
    cpl_image_delete(out2);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Test eris_ifu_dar_gsl_shift function
 */
/*----------------------------------------------------------------------------*/
static void test_eris_ifu_dar_gsl_shift(void)
{
    cpl_image* in = create_test_image(10, 10);
    cpl_image* out = cpl_image_new(10, 10, CPL_TYPE_DOUBLE);
    
    /* Test with NULL inputs */
    //eris_ifu_dar_gsl_shift(NULL, out, 1.0, 1.0, gsl_interp2d_bilinear);
    //cpl_test_error(CPL_ERROR_NULL_INPUT);

    //eris_ifu_dar_gsl_shift(in, NULL, 1.0, 1.0, gsl_interp2d_bilinear);
    //cpl_test_error(CPL_ERROR_NULL_INPUT);

    //eris_ifu_dar_gsl_shift(in, out, 1.0, 1.0, NULL);
    //cpl_test_error(CPL_ERROR_NULL_INPUT);

    /* Test normal case */
    eris_ifu_dar_gsl_shift(in, out, 1.0, 1.0, gsl_interp2d_bilinear);
    cpl_test_error(CPL_ERROR_NONE);
    
    /* Cleanup */
    cpl_image_delete(in);
    cpl_image_delete(out);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Test eris_frameset_count_tag function
 */
/*----------------------------------------------------------------------------*/
static void test_eris_frameset_count_tag(void)
{
    cpl_frameset* frames = create_test_frameset();
    
    /* Test with NULL inputs */
    int count = eris_frameset_count_tag(NULL, "OBJECT_CUBE");
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    cpl_test_eq(count, -1);
    
    count = eris_frameset_count_tag(frames, NULL);
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    cpl_test_eq(count, -1);
    
    /* Test normal case */
    count = eris_frameset_count_tag(frames, "OBJECT_CUBE");
    cpl_test_eq(count, 1);
    
    /* Test non-existent tag */
    count = eris_frameset_count_tag(frames, "NONEXISTENT");
    cpl_test_eq(count, 0);
    
    /* Cleanup */
    cpl_frameset_delete(frames);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit tests of eris_ifu_jitter_static module
 */
/*----------------------------------------------------------------------------*/
int main(void)
{
    cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_WARNING);

    test_eris_ifu_jitter_get_obj_type();
    test_eris_ifu_jitter_get_sky_type();
    test_eris_ifu_jitter_get_coadd_obj_type();
    test_eris_ifu_jitter_get_procatg_and_filename();
    test_eris_ifu_jitter_reconstruct_cube();
    test_eris_ifu_jitter_subtract_background();
    //test_eris_ifu_jitter_build_cube();// seg fault with valid input
    //test_eris_ifu_dar_correction(); //error with proper input
    test_eris_ifu_dar_cpl_shift();
    test_eris_ifu_dar_gsl_shift();
    test_eris_frameset_count_tag();
    
    cleanup_test_files();

    return cpl_test_end(0);
}

/**@}*/
