/* $Id: eris_utils_image-test.c,v 1.1 2024-01 amodigli Exp $
 *
 * This file is part of the ERIS Pipeline
 * Copyright (C) 2002,2003 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/*
 * $Author: amodigli $
 * $Date: 2024-01 $
 * $Revision: 1.1 $
 * $Name: not supported by cvs2svn $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/

#include <eris_utils_image.h>

/*----------------------------------------------------------------------------*/
/**
 * @defgroup eris_utils_image_test  Unit test of eris_utils_image
 *
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/*-----------------------------------------------------------------------------
                              Function prototypes
 -----------------------------------------------------------------------------*/

static cpl_image* create_test_image(void);
static void verify_edge_detection(const cpl_image* result, const double* expected);

/*-----------------------------------------------------------------------------
                                Function code
 -----------------------------------------------------------------------------*/

/**
 * @brief Create a test image for edge detection
 * @return A new test image
 */
static cpl_image* create_test_image(void)
{
    cpl_image* img;
    double* data;
    
    /* Create 5x5 test image with a vertical edge */
    img = cpl_image_new(5, 5, CPL_TYPE_DOUBLE);
    data = cpl_image_get_data_double(img);
    
    /* Fill left half with 0s and right half with 1s */
    for(int j = 0; j < 5; j++) {
        for(int i = 0; i < 5; i++) {
            data[i + j*5] = (i < 2) ? 0.0 : 1.0;
        }
    }
    
    return img;
}

/**
 * @brief Verify edge detection results
 * @param result The result image to verify
 * @param expected Array of expected values for the central 3x3 region
 */
static void verify_edge_detection(const cpl_image* result, const double* expected)
{
    const double* data;
    int status;
    
    data = cpl_image_get_data_double_const(result);
    
    /* Verify central 3x3 region (excluding borders) */
    for(int j = 1; j < 4; j++) {
        for(int i = 1; i < 4; i++) {
            cpl_test_abs(data[i + j*5], expected[(i-1) + (j-1)*3], 0.000001);
        }
    }
}

/**@}*/

/**
 * @brief Test eris_scharr_x function
 */
/**
 * @brief Test eris_scharr_y function
 */
/**
 * @brief Test eris_sobel_lx function
 */
/**
 * @brief Test eris_sobel_ly function
 */
static void eris_sobel_ly_test(void)
{
    cpl_image* img;
    cpl_image* result;
    
    /* Create test image */
    img = create_test_image();
    
    /* Expected values for central 3x3 region after Sobel Y filter */
    const double expected[9] = {
        0.0, 0.0, 0.0,  /* Row 1 */
        0.0, 0.0, 0.0,  /* Row 2 */
        0.0, 0.0, 0.0   /* Row 3 */
    };
    
    /* Test normal operation */
    result = eris_sobel_ly(img);
    cpl_test_nonnull(result);
    verify_edge_detection(result, expected);
    cpl_image_delete(result);
    
    /* Test null input */
    result = eris_sobel_ly(NULL);
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    cpl_test_null(result);
    
    /* Clean up */
    cpl_image_delete(img);
    cpl_image_delete(result);
}

/**
 * @brief Test eris_sobel_lx function
 */
static void eris_sobel_lx_test(void)
{
    cpl_image* img;
    cpl_image* result;
    
    /* Create test image */
    img = create_test_image();
    
    /* Expected values for central 3x3 region after Sobel X filter */
    const double expected[9] = {
        -6.0, -8.0, -6.0,  /* Row 1 */
        -6.0, -8.0, -6.0,  /* Row 2 */
        -6.0, -8.0, -6.0   /* Row 3 */
    };
    
    /* Test normal operation */
    result = eris_sobel_lx(img);
    cpl_test_nonnull(result);
    verify_edge_detection(result, expected);
    cpl_image_delete(result);
    
    /* Test null input */
    result = eris_sobel_lx(NULL);
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    cpl_test_null(result);
    
    /* Clean up */
    cpl_image_delete(img);
    cpl_image_delete(result);
}

/**
 * @brief Test eris_scharr_y function
 */
static void eris_scharr_y_test(void)
{
    cpl_image* img;
    cpl_image* result;
    
    /* Create test image */
    img = create_test_image();
    
    /* Expected values for central 3x3 region after Scharr Y filter */
    const double expected[9] = {
        0.0, 0.0, 0.0,  /* Row 1 */
        0.0, 0.0, 0.0,  /* Row 2 */
        0.0, 0.0, 0.0   /* Row 3 */
    };
    
    /* Test normal operation */
    result = eris_scharr_y(img);
    cpl_test_nonnull(result);
    verify_edge_detection(result, expected);
    cpl_image_delete(result);
    
    /* Test null input */
    result = eris_scharr_y(NULL);
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    cpl_test_null(result);
    
    /* Clean up */
    cpl_image_delete(img);
    cpl_image_delete(result);
}

/**
 * @brief Test eris_scharr_x function
 */
static void eris_scharr_x_test(void)
{
    cpl_image* img;
    cpl_image* result;
    
    /* Create test image */
    img = create_test_image();
    
    /* Expected values for central 3x3 region after Scharr X filter */
    const double expected[9] = {
        -30.0, -32.0, -30.0,  /* Row 1 */
        -30.0, -32.0, -30.0,  /* Row 2 */
        -30.0, -32.0, -30.0   /* Row 3 */
    };
    cpl_image_save(img, "img.fits", CPL_TYPE_DOUBLE, NULL, CPL_IO_CREATE);
    /* Test normal operation */
    result = eris_scharr_x(img);
    cpl_test_nonnull(result);
    verify_edge_detection(result, expected);
    cpl_image_delete(result);
    
    /* Test null input */
    result = eris_scharr_x(NULL);
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    cpl_test_null(result);
    
    /* Clean up */
    cpl_image_delete(img);
    cpl_image_delete(result);
}

int main(void)
{
    cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_WARNING);
    
    //eris_scharr_x_test();  //TODO: fix error
    cpl_test_error(CPL_ERROR_NONE);
    
    eris_scharr_y_test();
    cpl_test_error(CPL_ERROR_NONE);
    
    //eris_sobel_lx_test(); //TODO: fix error
    cpl_test_error(CPL_ERROR_NONE);
    
    eris_sobel_ly_test();
    cpl_test_error(CPL_ERROR_NONE);
    
    return cpl_test_end(0);
}
