/* $Id: eris_ifu_recipe.c,v 1.33 2013-03-26 17:00:45 jtaylor Exp $
 *
 * This file is part of the ERIS Pipeline
 * Copyright (C) 2002,2003 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/*
 * $Author: jtaylor $
 * $Date: 2013-03-26 17:00:45 $
 * $Revision: 1.33 $
 * $Name: not supported by cvs2svn $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                Includes
 -----------------------------------------------------------------------------*/
#include <cpl.h>

#include "eris_utils.h"
#include "eris_pfits.h"
#include "eris_dfs.h"
#include "eris_ifu_dfs.h"
#include "eris_ifu_utils.h"
#include "eris_ifu_error.h"
#include "eris_ifu_wavecal_static.h"
#include "eris_ifu_distortion_static.h"

/*-----------------------------------------------------------------------------
                            Static variables
 -----------------------------------------------------------------------------*/
static const char eris_ifu_wavecal_description[] = "\
This recipe performs ERIS/SPIFFIER wavelength calibration data reduction.\n\
\n\
-----------------------------------------------------------------------------\n\
Input files:\n\
   DO CATG          Explanation                              Required #Frames\n\
   -------          -----------                              -------- -------\n\
   WAVE_LAMP        Arclamp on/off exposures                    Y         2  \n\
   DISTORTION       Table with distortion correction parameters Y         1  \n\
   REF_LINE_ARC     Band-specific reference arc line list       Y         1  \n\
   WAVE_SETUP       Table to configure wave calibration         Y         1  \n\
   FIRST_WAVE_FIT   Table for the first attempt to fit ARClines Y         1  \n\
   SLITLET_POS      Table with slitlet edge positions           N       [0,1]\n\
   MASTER_FLAT      Optional master flat image                  N       [0,1]\n\
\n\
Output files:\n\
   DO CATG          Explanation\n\
   -------          -----------\n\
   WAVE_MAP         Wavelength calibrarion map\n\
----------------------------------------------------------------------------\n\
\n\
Information on relevant parameters may be found with\n\
  esorex --params "REC_NAME_WAVECAL"\n\
  esorex --help   "REC_NAME_WAVECAL"\n";

/*-----------------------------------------------------------------------------
                            Private function prototypes
 -----------------------------------------------------------------------------*/

cpl_recipe_define(eris_ifu_wavecal, ERIS_BINARY_VERSION, "Erich Wiezorrek",
                  PACKAGE_BUGREPORT, "2018",
                  "This recipe performs the wavelength calibration",
                  eris_ifu_wavecal_description);

cpl_error_code eris_ifu_wavecal_fetch_params(
		const cpl_parameterlist * parlist,
		struct stdParamStruct *stdParams );

cpl_error_code eris_ifu_wavecal_processSof(
        cpl_frameset* frames,
        int exposureCorrectionMode,
		double saturation_threhold,
        int *arcImgCnt,
        hdrl_imagelist **arcImages,
        int **lampStates,
        ifsBand *band,
        ifsPreopticsScale *scale,
        ifsInstrument *instrument,
        cpl_bivector **slitPos,
		cpl_table** qclog
		);

static cpl_propertylist * eris_ifu_wave_get_qc_params(const ifsBand band,
		cpl_table* qclog, struct waveTablesStruct *tables);

cpl_error_code eris_ifu_wave_save_products(
        cpl_image *waveCalImg,
        cpl_propertylist *applist,
        cpl_frameset *frameset,
        const cpl_parameterlist *parlist);

/*-----------------------------------------------------------------------------
                                Function code
 -----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Setup the recipe options    
  @param    pl  the nonn-NULL parameterlist to fill
  @return   CPL_ERROR_NONE iff everything is ok

  Defining the command-line/configuration parameters for the recipe.
 */
/*----------------------------------------------------------------------------*/
static cpl_error_code eris_ifu_wavecal_fill_parameterlist(cpl_parameterlist *pl)
{
    cpl_error_code  err = CPL_ERROR_NONE;

    TRY
    {
        BRK_IF_ERROR(
            eris_ifu_add_std_params(pl, REC_NAME_WAVECAL));
    } CATCH {
        CATCH_MSGS();
        err = cpl_error_get_code();
    }

    return err;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Interpret the command line options and execute the data processing
  @param    frameset   the frames list
  @param    parlist    the parameters list
  @return   0 if everything is ok
 */
/*----------------------------------------------------------------------------*/
static int eris_ifu_wavecal(cpl_frameset            * frameset,
                    const cpl_parameterlist * parlist)
{
    int arcImgCnt = 0;
    int *lampStates = NULL;
    ifsBand band = UNDEFINED_BAND;
    ifsPreopticsScale scale = UNDEFINED_SCALE;
    ifsInstrument instrument = UNSET_INSTRUMENT;
    struct stdParamStruct stdParams = stdParamStructInit;

    hdrl_imagelist *arcImages = NULL;
    const char *refLineTableFileName = NULL;
    const char *firstFitTableFileName = NULL;
    const char *waveSetupFileName = NULL;
    cpl_bivector *slitPos = NULL;
    cpl_image *waveCalImg = NULL;
    cpl_propertylist *applist = NULL;
    struct waveTablesStruct tables;
    struct waveSetupStruct waveSetup;
    cpl_table* qclog = NULL;
    double saturation_threshold = 0;


    /* check required input tags are present */
    /* temporarily suppressed check on required input tags
      const int ntags = 6;
      const char* required_tags[6] = {
      		ERIS_IFU_PRO_DIST_DISTORTION,
			ERIS_IFU_CALIB_FIRST_FIT,
  			ERIS_IFU_CALIB_REF_LINES,
  			ERIS_IFU_CALIB_WAVE_SETUP,
  			ERIS_IFU_CALIB_REF_LINES,
			ERIS_IFU_RAW_ARC_LAMP
      };
      
      
      cpl_ensure_code(CPL_ERROR_NONE ==
      		eris_dfs_check_input_tags(frameset, required_tags, ntags, 1),
  			CPL_ERROR_ILLEGAL_INPUT);
  			*/

       const int nopt_tags = 1;
       const char* optional_tags[1] = {ERIS_IFU_CALIB_FLATFIELD
       };
      		eris_dfs_check_input_tags(frameset, optional_tags, nopt_tags, 0);


    TRY
    {
        // make sure there is no nonsense in the table structure
        // in case it is written before filled properly
        eris_ifu_wave_clear_tables(&tables);

        cpl_msg_info(cpl_func, "Reading recipe parameters");
        BRK_IF_ERROR(
        	eris_ifu_wavecal_fetch_params(parlist, &stdParams));
        cpl_msg_info(cpl_func,
        		"Instrument is %d, requested product level is %d",
				stdParams.instrument, stdParams.productDepth);

        eris_print_rec_status(0);
        saturation_threshold = cpl_parameter_get_double(
        cpl_parameterlist_find_const(parlist, "eris.eris_ifu_wavecal.pixel_saturation"));
        eris_print_rec_status(1);
        BRK_IF_ERROR(
        	eris_ifu_wavecal_processSof(
        		frameset,
        		stdParams.rawImageCorrectionMask,
				saturation_threshold,
        		&arcImgCnt,
        		&arcImages,
				&lampStates,
				&band,
				&scale,
				&instrument,
				&slitPos,
				&qclog
				));

        refLineTableFileName = cpl_frame_get_filename(
        	cpl_frameset_find(frameset, ERIS_IFU_CALIB_REF_LINES));
		firstFitTableFileName = cpl_frame_get_filename(
        	cpl_frameset_find(frameset, ERIS_IFU_CALIB_FIRST_FIT));
        waveSetupFileName = cpl_frame_get_filename(
            cpl_frameset_find(frameset, ERIS_IFU_CALIB_WAVE_SETUP));
		CHECK_ERROR_STATE();

		BRK_IF_ERROR(
		    eris_ifu_read_wave_setup(waveSetupFileName, band, &waveSetup));
        BRK_IF_ERROR(
            eris_ifu_wave_init_tables(&tables));

        BRK_IF_NULL(
        	waveCalImg = eris_ifu_wave_get_calImg(
        		arcImgCnt,
	       		arcImages,
				lampStates,
				band,
				instrument,
                waveSetup,
				refLineTableFileName,
				firstFitTableFileName,
				slitPos,
                &tables,
                stdParams.productDepth,
                frameset,
                parlist,
				qclog)
        );

        if ((stdParams.productDepth & 1) != 0) {
            BRK_IF_ERROR(
                eris_ifu_wave_save_fitting_tables(&tables, instrument,  band,
						  waveSetup, frameset, parlist,
						  "eris_ifu_wavecal"));
        }


        applist = eris_ifu_wave_get_qc_params(band, qclog, &tables);

        eris_ifu_wave_save_products(waveCalImg, applist, frameset, parlist);

        eris_ifu_wave_free_tables(&tables);
    } CATCH
 	{
        if ((stdParams.productDepth & 1) != 0) {
            eris_ifu_wave_save_fitting_tables(&tables, instrument,  band,
					      waveSetup, frameset, parlist,
					      "eris_ifu_wavecal");
        }
//        CATCH_MSGS();
 	}
    cpl_table_delete(qclog);
    eris_ifu_free_bivector(&slitPos);
    eris_ifu_free_int_array(&lampStates);
    eris_ifu_free_hdrl_imagelist(&arcImages);
    eris_ifu_free_image(&waveCalImg);
    eris_ifu_free_propertylist(&applist);
    eris_ifu_wave_free_tables(&tables);
    eris_ifu_free_std_param(&stdParams);

    return (int)cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
  @brief    fetch parameter from CPL parameter structure and put in ERIS
  dedicated one

  @param    parlist    the parameters list
  @param    stdParams  parameters for STD star data reduction

  @return   0 if everything is ok
 */
/*----------------------------------------------------------------------------*/
cpl_error_code eris_ifu_wavecal_fetch_params(
		const cpl_parameterlist * parlist,
        struct stdParamStruct *stdParams )
{
    TRY
    {
        BRK_IF_ERROR(
            eris_ifu_fetch_std_param(parlist, REC_NAME_WAVECAL, stdParams));
    } CATCH
	{
    	CATCH_MSGS();
	}

    return cpl_error_get_code();
}
/*----------------------------------------------------------------------------*/
/**
  @brief  process sof

  @param    frameset   input frameset
  @param    exposureCorrectionMode exposure
  @param    saturation_threhold  saturation threshold
  @param    arcImgCnt  counter of arc line images
  @param    arcImages  arc images
  @param    lampStates lamp states
  @param    band       observed band
  @param    scale
  @param    stdParams  parameters for STD star data reduction
  @param    instrument  instrume value
  @param    slitPos     slit(let) positions
  @param    qclog        qclog parameters
  @return   0 if everything is ok
 */
/*----------------------------------------------------------------------------*/
cpl_error_code eris_ifu_wavecal_processSof(
		cpl_frameset* frames,
		int exposureCorrectionMode,
		double saturation_threhold,
		int *arcImgCnt,
		hdrl_imagelist **arcImages,
		int **lampStates,
		ifsBand *band,
		ifsPreopticsScale *scale,
		ifsInstrument *instrument,
		cpl_bivector **slitPos,
		cpl_table** qclog
		)
{
	cpl_frame *frame = NULL;
	hdrl_image *flatfieldImage = NULL;
	cpl_image *flatfieldMask;
	cpl_polynomial *poly_u = NULL;
	cpl_polynomial *poly_v = NULL;

    TRY
    {
    	if (frames == NULL) {
    		BRK_WITH_ERROR_MSG(CPL_ERROR_NULL_INPUT,
    			"missing frameset");
    	}
    	if (cpl_frameset_is_empty(frames)) {
    		BRK_WITH_ERROR_MSG(CPL_ERROR_NULL_INPUT,
    		    			"SOF file is empty or missing");
    	}
    	CHECK_ERROR_STATE();

    	BRK_IF_ERROR(
    		eris_ifu_dfs_set_groups(frames));

    	// get arc lamp image
    	cpl_frameset *arcFrames = NULL;

    	if (cpl_frameset_count_tags(frames, ERIS_IFU_RAW_ARC_LAMP) > 0) {
    		arcFrames = eris_ifu_get_frameset_by_tag(frames,
    				ERIS_IFU_RAW_ARC_LAMP);
    	} else if ( (cpl_frameset_count_tags(frames, ERIS_IFU_RAW_ARC_LAMP_ON) > 0) &&
    			(cpl_frameset_count_tags(frames, ERIS_IFU_RAW_ARC_LAMP_OFF) > 0) ) {

    		cpl_frameset* arcFrames_off = NULL;
    		arcFrames = eris_ifu_get_frameset_by_tag(frames, ERIS_IFU_RAW_ARC_LAMP_ON);

    		arcFrames_off = eris_ifu_get_frameset_by_tag(frames, ERIS_IFU_RAW_ARC_LAMP_OFF);

    		cpl_frameset_join(arcFrames, arcFrames_off);
    		cpl_frameset_delete(arcFrames_off);
    	}
    	
    	BRK_IF_ERROR(
    			eris_ifu_wave_get_arc_images(arcFrames,
    			    exposureCorrectionMode,
    				arcImgCnt,
    				arcImages,
					lampStates,
					band,
					scale,
					instrument,
					saturation_threhold,
					qclog));
    	cpl_frameset_delete(arcFrames);
        //PIPPO
    	// get reference lines
    	frame = cpl_frameset_find(frames, ERIS_IFU_CALIB_REF_LINES);
    	CHECK_ERROR_STATE();
    	if (frame == NULL) {
    		BRK_WITH_ERROR_MSG(CPL_ERROR_NULL_INPUT,
    			"missing \"%s\" tag in the SOF, arc lamp reference lines",
				ERIS_IFU_CALIB_REF_LINES);
    	}

    	// get flat field calibration file (which is not required)
    	frame = cpl_frameset_find(frames, ERIS_IFU_CALIB_FLATFIELD);
    	if (frame != NULL) {
    	    deqQualityType qualityType;
    		BRK_IF_NULL(
    			flatfieldImage = eris_ifu_load_cal_image_frame(frame,
    				*band, *scale, &flatfieldMask, &qualityType));
    		cpl_msg_info(cpl_func,"Apply master flat to each input arc frame");
    		BRK_IF_ERROR(
    			hdrl_imagelist_div_image(*arcImages, flatfieldImage));
    		eris_ifu_free_hdrl_image(&flatfieldImage);
    		eris_ifu_free_image(&flatfieldMask);
     	}

    	frame = cpl_frameset_find(frames, ERIS_IFU_CALIB_DISTORTION);
    	CHECK_ERROR_STATE();
    	if (frame == NULL) {
    		BRK_WITH_ERROR_MSG(CPL_ERROR_NULL_INPUT,
    			"missing \"%s\" tag in the SOF, distortion polynomials",
				ERIS_IFU_CALIB_DISTORTION);
    	}
        if (cpl_fits_count_extensions(cpl_frame_get_filename(frame)) > 2) {
            cpl_polynomial **distortion;
            cpl_table      *borders;
            BRK_IF_ERROR(
                eris_ifu_load_distortion_polynomials(
                    cpl_frame_get_filename(frame),
                    &distortion, &borders));
            cpl_msg_info(cpl_func,"Correct distortions on each input arc frame");
            for (cpl_size pos=0; pos<*arcImgCnt; pos++) {
                hdrl_image *inImage = hdrl_imagelist_get(*arcImages, pos);
                hdrl_image *tmpImage = eris_ifu_dist_warp_image(
                                    inImage, distortion, borders);
                hdrl_imagelist_set(*arcImages, tmpImage, pos);
                CHECK_ERROR_STATE();
            }
            for (cpl_size sc=0; sc < SLITLET_CNT; sc++) {
                cpl_polynomial_delete(distortion[sc]);
            }
            cpl_free(distortion);
            cpl_table_delete(borders);

        } else {
            BRK_IF_ERROR(
                eris_ifu_load_distortion_polynomials_old(
                    cpl_frame_get_filename(frame), &poly_u, &poly_v));
            //    	cpl_polynomial_dump(poly_u,stdout);
            //      cpl_polynomial_dump(poly_v,stdout);
            for (cpl_size pos=0; pos<*arcImgCnt; pos++) {
                hdrl_image *inImage = hdrl_imagelist_get(*arcImages, pos);
                //       		cpl_image_save(hdrl_image_get_image(inImage),"warp.fits",
                //       			CPL_TYPE_UNSPECIFIED, NULL, CPL_IO_CREATE);
                hdrl_image *tmpImage = eris_ifu_warp_polynomial_image(
                    inImage, poly_u, poly_v);
                //       		cpl_image_save(hdrl_image_get_image(tmpImage),"warp.fits",
                //       			CPL_TYPE_UNSPECIFIED, NULL, CPL_IO_EXTEND);
                hdrl_imagelist_set(*arcImages, tmpImage, pos);
                CHECK_ERROR_STATE();
            }

            frame = cpl_frameset_find(frames, ERIS_IFU_PRO_DIST_SLIT_POS);
            CHECK_ERROR_STATE();
            BRK_IF_NULL(
                    *slitPos = eris_ifu_load_slit_positions(
                        cpl_frame_get_filename(frame)));
        }

    } CATCH
	{
//    	CATCH_MSGS();
	}
    eris_ifu_free_polynomial(&poly_u);
    eris_ifu_free_polynomial(&poly_v);
    return cpl_error_get_code();

}
/*----------------------------------------------------------------------------*/
/**
  @brief    functions that get QC parameters

  @param    tables    table with QC parameters


  @return  waveTablesStruct structure  if everything is ok
 */
/*----------------------------------------------------------------------------*/
static cpl_propertylist * eris_ifu_wave_get_qc_params(const ifsBand band,
		cpl_table* qclog, struct waveTablesStruct *tables)
{
    cpl_propertylist *appList = NULL;
    const cpl_array *tmpArray = NULL;
    cpl_vector *coeffVec = NULL;
    cpl_size coeffDim = 0;
    cpl_size nRow = 0;
    char * qcName = NULL;

    TRY
    {

        appList = cpl_propertylist_new();
        eris_pfits_put_qc(appList, qclog);
        // calculate average and median of wavecal coeffs

        coeffDim = cpl_table_get_int(tables->columnCoeffSmoothed, "degree", 0,
            NULL);

        nRow = cpl_table_get_nrow(tables->columnCoeffSmoothed);
        BRK_IF_NULL(
            coeffVec = cpl_vector_new(nRow));
        for (cpl_size ix=0; ix<=coeffDim; ix++) {
            for (cpl_size row=0; row<nRow; row++) {
                tmpArray = cpl_table_get_array(
                                tables->columnCoeffSmoothed, "coeffs", row);


                cpl_vector_set(coeffVec, row,
                            cpl_array_get_double(tmpArray, ix, NULL));
            }

            qcName = cpl_sprintf("COEF%lld AVG", ix);

            eris_ifu_append_qc_double(appList, qcName,
                    cpl_vector_get_mean(coeffVec),
                    "Average wavecal Coef");
            eris_ifu_free_string(&qcName);

            qcName = cpl_sprintf("COEF%lld MED", ix);

            eris_ifu_append_qc_double(appList, qcName,
                    cpl_vector_get_median(coeffVec),
                    "Median wavecal Coef");
            eris_ifu_free_string(&qcName);
        }

        // calculate average and median of FWHM of arc lines

        cpl_table_unselect_all(tables->columnFitting);

        nRow = cpl_table_or_selected_int(tables->columnFitting, ERIS_IFU_FITTABLE_ERRORCODE,
            CPL_EQUAL_TO, 0);

        if (nRow > 0) {
            cpl_table *tmpTable;

            tmpTable= cpl_table_extract_selected(tables->columnFitting);

            double sigmaMed = cpl_table_get_column_median(tmpTable, "sigma");
            double sigmaAvg = cpl_table_get_column_mean(tmpTable, "sigma");
            double sigmaRms = cpl_table_get_column_stdev(tmpTable, "sigma");
            double resolMed = cpl_table_get_column_median(tmpTable, "resol");
            double resolAvg = cpl_table_get_column_mean(tmpTable, "resol");
            double resolRms = cpl_table_get_column_stdev(tmpTable, "resol");
            cpl_size nRows = cpl_table_get_nrow(tmpTable);


            double convFactor = 2. * sqrt(2. * log(2.));

            eris_ifu_append_qc_double(appList, "FWHM AVG",
            		sigmaAvg * convFactor,
					"[pix] Average FWHM of found lines");

            eris_ifu_append_qc_double(appList, "FWHM MED",
            		sigmaMed * convFactor,
					"[pix] Median FWHM of found lines");

            eris_ifu_append_qc_double(appList, "FWHM STD",
            		sigmaRms * convFactor,
					"[pix] Stdev FWHM of found lines");
            eris_ifu_append_qc_int(appList, "NFITLINES",
                        		nRows,
            					"Number of lines used tor the fit");
            double centralLambda = 0;
            eris_ifu_get_central_lambda(band, &centralLambda);
            double dispersion = 0;
            eris_ifu_get_dispersion(band, &dispersion);
            /*
            eris_ifu_append_qc_double(appList, "NOMRESOL AVG",
                        centralLambda / (sigmaAvg * convFactor) / dispersion,
                        					"Nominal average resolution power");
            eris_ifu_append_qc_double(appList, "NOMRESOL MED",
                        centralLambda / (sigmaMed * convFactor) / dispersion,
                        					"Nominal median resolution power");
            eris_ifu_append_qc_double(appList, "NOMRESOL STD",
                        centralLambda / (sigmaRms * convFactor) / dispersion,
                        					"Nominal stdev resolution power");
            */
            eris_ifu_append_qc_double(appList, "RESOL AVG", resolAvg,
            		"average resolution power");
            eris_ifu_append_qc_double(appList, "RESOL MED", resolMed,
            		"median resolution power");
            eris_ifu_append_qc_double(appList, "RESOL STD", resolRms,
            		"stdev resolution power");

            //cpl_table_dump_structure(tmpTable, stdout);
            for(cpl_size i = 0; i < 32; i++) {
            	char* keyname;
				char* keycomm;
            	cpl_table *extTable;
            	cpl_table_and_selected_int(tmpTable,"slitlet", CPL_EQUAL_TO, i);

            	extTable = cpl_table_extract_selected(tmpTable);
            	resolMed = cpl_table_get_column_median(extTable, "resol");
            	resolAvg = cpl_table_get_column_mean(extTable, "resol");
            	sigmaMed = cpl_table_get_column_median(extTable, "sigma");
            	sigmaAvg = cpl_table_get_column_mean(extTable, "sigma");
            	nRows = cpl_table_get_nrow(extTable);
            	//cpl_msg_warning(cpl_func, "Slitlet: %lld, nRows: %lld", i, nRows);
            	if(nRows > 0) {
            		resolRms = cpl_table_get_column_stdev(extTable, "resol");
            		sigmaRms = cpl_table_get_column_stdev(extTable, "sigma");
            	} else {
            		resolRms = 0;
            		sigmaRms = 0;
            	}

            	keyname = cpl_sprintf("SLITLET%lld FWHM AVG", i);
            	keycomm = cpl_sprintf("[pix] Average FWHM of found lines on slitlet %lld" ,i);
            	eris_ifu_append_qc_double(appList, keyname, sigmaAvg * convFactor,
            			keycomm);
            	cpl_free(keyname); cpl_free(keycomm);
            	keyname = cpl_sprintf("SLITLET%lld FWHM MED", i);
            	keycomm = cpl_sprintf("[pix] Median FWHM of found lines on slitlet %lld" ,i);
            	eris_ifu_append_qc_double(appList, keyname, sigmaMed * convFactor,
						keycomm);
            	cpl_free(keyname); cpl_free(keycomm);
            	keyname = cpl_sprintf("SLITLET%lld FWHM STD", i);
            	keycomm = cpl_sprintf("[pix] Stdev FWHM of found lines on slitlet %lld" ,i);
            	eris_ifu_append_qc_double(appList, keyname, sigmaRms * convFactor,
						keycomm);
            	cpl_free(keyname); cpl_free(keycomm);
            	keyname = cpl_sprintf("SLITLET%lld NFITLINES", i);
            	keycomm = cpl_sprintf("Number of lines used for the fit on slitlet %lld" ,i);
            	eris_ifu_append_qc_int(appList, keyname, nRows, keycomm);
            	cpl_free(keyname); cpl_free(keycomm);

            	/*
            	keyname = cpl_sprintf("SLITLET%lld NOMRESOL AVG", i);
            	keycomm = cpl_sprintf("Nominal average resolution power");
            	eris_ifu_append_qc_double(appList, keyname,
            	          centralLambda / (sigmaAvg * convFactor) / dispersion,
            	                        				keycomm);
            	cpl_free(keyname); cpl_free(keycomm);

            	keyname = cpl_sprintf("SLITLET%lld NOMRESOL MED", i);
            	keycomm = cpl_sprintf("Nominal median resolution power");
            	eris_ifu_append_qc_double(appList, keyname,
            	          centralLambda / (sigmaMed * convFactor) / dispersion,
            	                        					keycomm);
            	cpl_free(keyname); cpl_free(keycomm);
            	keyname = cpl_sprintf("SLITLET%lld NOMRESOL STD", i);
            	keycomm = cpl_sprintf("Nominal stdev resolution power");
            	eris_ifu_append_qc_double(appList, keyname,
            	          centralLambda / (sigmaRms * convFactor) / dispersion,
            	                        					keycomm);
            	cpl_free(keyname); cpl_free(keycomm);
                */

            	keyname = cpl_sprintf("SLITLET%lld RESOL AVG", i);
            	keycomm = cpl_sprintf("average resolution power");
            	eris_ifu_append_qc_double(appList, keyname, resolAvg, keycomm);
            	cpl_free(keyname); cpl_free(keycomm);
            	keyname = cpl_sprintf("SLITLET%lld RESOL MED", i);
            	keycomm = cpl_sprintf("median resolution power");
            	eris_ifu_append_qc_double(appList, keyname, resolMed, keycomm);
            	cpl_free(keyname); cpl_free(keycomm);
            	keyname = cpl_sprintf("SLITLET%lld RESOL STD", i);
            	keycomm = cpl_sprintf("stdev resolution power");
            	eris_ifu_append_qc_double(appList, keyname, resolRms, keycomm);
            	cpl_free(keyname); cpl_free(keycomm);

            	cpl_table_delete(extTable);
            	cpl_table_select_all(tmpTable);
            }
            cpl_table_delete(tmpTable);
        }

        //calculate average and median of wavepos error
        if (nRow > 0) {
            cpl_table *tmpTable;
            BRK_IF_NULL(
                tmpTable= cpl_table_extract_selected(tables->columnFitting));
            cpl_vector *tWavePosErr =cpl_vector_wrap(nRow,
                cpl_table_get_data_double(tmpTable, "wavelengthError"));
            cpl_vector *vWavePosErr =cpl_vector_duplicate(tWavePosErr);
            cpl_vector_unwrap(tWavePosErr);
            eris_ifu_free_table(&tmpTable);
            CHECK_ERROR_STATE();

            cpl_vector_sort(vWavePosErr, CPL_SORT_ASCENDING);
            double *data = cpl_vector_get_data(vWavePosErr);
            cpl_vector *cvWavePosErr = cpl_vector_wrap(
                (cpl_size) ((double)nRow * .8),
                &data[(int) ((double)nRow * .1)]);
            double wavePosErrAvg = cpl_vector_get_mean(vWavePosErr);
            double wavePosErrMed = cpl_vector_get_median(vWavePosErr);
            double cwavePosErrAvg = cpl_vector_get_mean(cvWavePosErr);
            double cwavePosErrMed = cpl_vector_get_median(cvWavePosErr);
            //convert all vector elements to their absolute values
            double *vdata = cpl_vector_get_data(vWavePosErr);
            for (cpl_size ix=0; ix<cpl_vector_get_size(vWavePosErr); ix++) {
                vdata[ix] =fabs(vdata[ix]);
            }
            vdata = cpl_vector_get_data(cvWavePosErr);
            for (cpl_size ix=0; ix<cpl_vector_get_size(cvWavePosErr); ix++) {
                vdata[ix] =fabs(vdata[ix]);
            }
            double wavePosErrAvgAbs = cpl_vector_get_mean(vWavePosErr);
            double wavePosErrMedAbs  = cpl_vector_get_median(vWavePosErr);
            double cwavePosErrAvgAbs  = cpl_vector_get_mean(cvWavePosErr);
            double cwavePosErrMedAbs  = cpl_vector_get_median(cvWavePosErr);

            CHECK_ERROR_STATE();
            cpl_vector_unwrap(cvWavePosErr);
            cpl_vector_delete(vWavePosErr);

            eris_ifu_append_qc_double(appList, "POSERR AVG",
            		wavePosErrAvg,
					"[um] Average of reference line position errors");

            eris_ifu_append_qc_double(appList, "POSERR MED",
            		wavePosErrMed,
					"[um] Median of reference line position errors");

            eris_ifu_append_qc_double(appList, "POSERR CLEAN AVG",
            		cwavePosErrAvg,
					"[um] Clean average of reference line position errors");

            eris_ifu_append_qc_double(appList, "POSERR CLEAN MED",
            		cwavePosErrMed,
					"[um] Clean median of reference line position errors");

            eris_ifu_append_qc_double(appList, "POSERR AVG ABS",
            		wavePosErrAvgAbs ,
					"[um] Average of reference line position absolute errors");

            eris_ifu_append_qc_double(appList, "POSERR MED ABS",
            		wavePosErrMedAbs ,
					"[um] Median of reference line position absolute errors");

            eris_ifu_append_qc_double(appList, "POSERR CLEAN AVG ABS",
            		cwavePosErrAvgAbs ,
					"[um] Clean average of reference line position absolute errors");

            eris_ifu_append_qc_double(appList, "POSERR CLEAN MED ABS",
            		cwavePosErrMedAbs ,
					"[um] Clean median of reference line position absolute errors");

        }

    } CATCH
    {
        CATCH_MSGS();
        eris_ifu_free_propertylist(&appList);
    }
    eris_ifu_free_vector(&coeffVec);
    return appList;
}
/*----------------------------------------------------------------------------*/
/**
  @brief    function to save products of wavecal recipe

  @param    waveCalImg  wavemap image
  @param    applist     propertylist with parameters
  @param    frameset    set of frames
  @param    parlist     cpl parameterlist


  @return   CPL_ERROR_NONE if everything is ok
 */
/*----------------------------------------------------------------------------*/
cpl_error_code eris_ifu_wave_save_products(
        cpl_image *waveCalImg,
        cpl_propertylist *applist,
        cpl_frameset *frameset,
        const cpl_parameterlist *parlist)
{
    TRY
    {
        BRK_IF_ERROR(
            eris_ifu_save_image(frameset, applist, parlist, REC_NAME_WAVECAL,
                ERIS_IFU_PRO_WAVE_MAP, ERIS_IFU_PRO_WAVE_MAP_FN,
                CPL_TYPE_UNSPECIFIED, waveCalImg));
    } CATCH
    {
        CATCH_MSGS();
    }

    return cpl_error_get_code();
}

