/* $Id$
 *
 * This file is part of the ERIS Pipeline
 * Copyright (C) 2002,2003 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/**
 * @defgroup eris_ifu_distortion_static   IFU Geometric Distortion Correction
 *
 * This module provides comprehensive functions for geometric distortion correction
 * in IFU spectrograph data. The distortion correction process involves:
 * 
 * - Detection and fitting of slitlet centers across the detector
 * - Calculation of 2D polynomial distortion maps for each slitlet
 * - Image warping to correct geometric distortions
 * - Bad pixel mask transformation
 * - Quality control statistics computation
 * 
 * The main workflow includes:
 * 1. Calculate slitlet centers from fiber illumination frames
 * 2. Detect slit edges from arc lamp images
 * 3. Fit 2D polynomials to map distorted to rectified coordinates
 * 4. Warp science images using the distortion maps
 * 5. Transform bad pixel masks accordingly
 * 6. Compute QC statistics on warped images
 * 
 * @note This module handles both single-fiber and triple-fiber slitlet configurations
 * @note Special handling is implemented for edge slitlets that may be cut off
 * @note Memory management: Caller is responsible for freeing returned objects
 */
/**@{*/

#include <string.h>
#include "eris_ifu_distortion_static.h"
#include "eris_ifu_wavecal_static.h"
#include "eris_ifu_dfs.h"
#include "eris_ifu_error.h"
#include "eris_ifu_utils.h"
#include "eris_ifu_debug.h"
#include "eris_utils.h"

/**
 * @brief Detect peak positions from a collapsed profile
 * @param profile       Input profile vector (collapsed in Y direction)
 * @param lowerCutLevel If TRUE, use lower threshold for peak detection
 * @return Vector containing detected peak center positions, or NULL on error
 *
 * This function analyzes a horizontal profile and detects peaks by:
 * - Computing a dynamic cut level based on profile mean
 * - Finding left and right edges of peaks above the threshold
 * - Calculating center positions as (left+right)/2
 * - Iterating with decreasing thresholds if expected peak count not found
 * - Sorting peaks by height to remove low-amplitude spurious detections
 *
 * @note The function expects to find either SLITLET_CNT (32) or 3*SLITLET_CNT (96) peaks
 * @note Returned vector must be freed by caller using cpl_vector_delete()
 * @note Border pixels (ERIS_IFU_DETECTOR_BP_BORDER) are ignored in the detection
 */
cpl_vector* eris_ifu_dist_calc_centers_profile(const cpl_vector* profile,
                                               cpl_boolean lowerCutLevel)
{
    int             cnt             = 0,
                    est_cnt         = 0;
    cpl_size        nx;
    double          mean            = 0.,
                    cut_level       = 0.,
                    cut_factor      = 1.4,
                    left            = 0.,
                    right           = 0.,
                    width           = 0.,
                    *ppos           = NULL,
                    *pheight        = NULL;
    const double    *pprofile       = NULL;
    cpl_vector      *pos            = NULL,
                    *height         = NULL,
                    *cen_estimates  = NULL;

    cpl_ensure(profile != NULL, CPL_ERROR_NULL_INPUT, NULL);

    TRY
    {
        nx = cpl_vector_get_size(profile);

        if (lowerCutLevel) {
            cut_factor = 1.0;
        }
        /* determine the mean of the row data, multiply scalar to get a
         * level around half-max just to be sure to omit low-level outliers */
        eris_ifu_vector *collapsed_vec2 = eris_ifu_vector_create(profile);
        mean = eris_ifu_vector_get_mean(collapsed_vec2);
        cut_level = cut_factor * mean;

        //cut_level = 2 * eris_ifu_vector_get_mean(collapsed_vec2) * 0.8;
        eris_ifu_free_ifu_vector(&collapsed_vec2);

        /* search for left/right edge of slitlet,
         * subtract to get center and store
         * (ignoring bad-pix-border on left and right side of detector)
         * (max. size would be 3*SLITLET_CNT, but allow for more to prevent seg. fault)
         */
        BRK_IF_NULL(
            pos = cpl_vector_new(3*SLITLET_CNT+20));
        BRK_IF_NULL(
            height = cpl_vector_new(3*SLITLET_CNT+20));
        BRK_IF_ERROR(cpl_vector_multiply_scalar(height, 0.0));
        BRK_IF_NULL(
            ppos = cpl_vector_get_data(pos));
        BRK_IF_NULL(
            pheight = cpl_vector_get_data(height));

        BRK_IF_NULL(
            pprofile = cpl_vector_get_data_const(profile));

        for (cut_factor = 1.4; cut_factor > .1; cut_factor -= 0.2) {
            cut_level = cut_factor * mean;
            cnt = 0;

            for (int i = ERIS_IFU_DETECTOR_BP_BORDER;
                    i < nx-ERIS_IFU_DETECTOR_BP_BORDER;
                    i++)
            {
                /* find left edge */
                if ((left == 0) && (pprofile[i] > cut_level)) {
                    left = i;
                }
                /* find right edge */
                if (((left != 0) && (right == 0)) &&
                        ((pprofile[i] < cut_level) || (i == nx-ERIS_IFU_DETECTOR_BP_BORDER-1))) {
                    right = i-1;
                }
                /* found left/right-edge, calc center, reset & continue */
                if ((left != 0) && (right != 0)) {
                    width = right-left+1;
                    ppos[cnt] = (double)left + width/2.0;
                    pheight[cnt] = pprofile[(int) (ppos[cnt]+.5) ];
                    cnt++;
                    left = 0;
                    right = 0;
                }
                if (cnt >= cpl_vector_get_size(pos)) {
                    break;
                }
            }
            if (cnt == SLITLET_CNT || cnt == 3 * SLITLET_CNT) {
                est_cnt = cnt;
                break;
            }
            if (cnt > SLITLET_CNT && cnt < SLITLET_CNT + 20) {
                est_cnt = SLITLET_CNT;
                break;
            }
            if (cnt > 3 * SLITLET_CNT) {
                est_cnt = 3 * SLITLET_CNT;
                break;
            }
        }

        // if there are too many peaks, sort the arrays to remove the lowest peaks
        if (cnt  > est_cnt) {
            cpl_bivector *toBeSorted = NULL;
            BRK_IF_NULL(toBeSorted = cpl_bivector_wrap_vectors(pos, height));
            BRK_IF_ERROR(cpl_bivector_sort(toBeSorted, toBeSorted,
                CPL_SORT_DESCENDING, CPL_SORT_BY_Y));
            cpl_bivector_unwrap_vectors(toBeSorted);
            cnt = est_cnt;
        }

        /* now we know the number of detected traces, extract the significant
         * part of the vector to return */
        BRK_IF_NULL(
            cen_estimates = cpl_vector_extract(pos, 0, cnt-1, 1));
        //
        BRK_IF_ERROR(cpl_vector_sort(cen_estimates, CPL_SORT_ASCENDING));

//        cpl_msg_debug(cpl_func,"  est. profile: cutlevel: %g, #peaks: %d", cut_level, (int)cpl_vector_get_size(cen_estimates));
    }
    CATCH
    {
        CATCH_MSGS();
        eris_ifu_free_vector(&cen_estimates);
    }

    eris_ifu_free_vector(&pos);
    eris_ifu_free_vector(&height);

    return cen_estimates;
}

/**
 * @brief Estimate position of missing low slitlet
 * @param est_centers Input vector with estimated center positions
 * @return New vector with corrected center positions including estimated missing slitlet, or NULL on error
 *
 * This function handles cases where peak detection misses a slitlet by:
 * - Checking if the correct number of slitlets is detected (32 or 96)
 * - If missing, estimating the position based on neighboring slitlets
 * - Special handling for the known bad slitlet (#15 in 25mas mode)
 * - Removing trailing zeros caused by detector edge cutoff
 *
 * @note For 25mas scale: looks for missing slitlet around position 15
 * @note For 100mas/250mas scales: checks distances between consecutive traces
 * @note Returned vector must be freed by caller using cpl_vector_delete()
 */
cpl_vector* eris_ifu_dist_estimate_low_slitlet(const cpl_vector *est_centers)
{
    cpl_vector      *est_new        = NULL,
                    *est_new2       = NULL;
    double          *pest_new       = NULL;
    const double    *pest_centers   = NULL;
    cpl_size        cnt             = 0,
                    ind             = 0;

    cpl_ensure(est_centers, CPL_ERROR_NULL_INPUT, NULL);

    TRY
    {
        BRK_IF_NULL(
            pest_centers = cpl_vector_get_data_const(est_centers));

        cnt = cpl_vector_get_size(est_centers);
        if ((cnt == SLITLET_CNT) || (cnt == 3*SLITLET_CNT)) {
            // all slitlet seem to be detected correctly
            BRK_IF_NULL(
                est_new = cpl_vector_duplicate(est_centers));
            return est_new;
        } else if (cnt < SLITLET_CNT) {
            // 25mas: slitlet missing
            BRK_IF_NULL(
                est_new = cpl_vector_new(SLITLET_CNT));
            cpl_vector_fill(est_new, 0.);
        } else if ((cnt > SLITLET_CNT+10) && (cnt < 3*SLITLET_CNT)) {
            // 100mas/250mas: slitlet missing
            BRK_IF_NULL(
                est_new = cpl_vector_new(3*SLITLET_CNT));
            cpl_vector_fill(est_new, 0.);
        } else {
            // more than 3*32 slitlets or 33-43 slitlets
            // there are too many slitlets, handle elsewhere
            BRK_IF_NULL(
                est_new = cpl_vector_duplicate(est_centers));
            return est_new;
        }
        BRK_IF_ERROR(
            cpl_vector_multiply_scalar(est_new, 0.));
        BRK_IF_NULL(
            pest_new = cpl_vector_get_data(est_new));

        if (cnt < SLITLET_CNT) {
            // 25mas
            ind = 11;

            // locate position of missing slitlet (not always on pos. 15 because a slitlet could be cut off on left of right of detector)
            // because of this look up to 3 slitlets to left and right
            while ((fabs(pest_centers[ind+1]-pest_centers[ind]-2*SLITLET_WIDTH) > 10) && (ind < 18)) {
                ind++;

            }

            for (int i = 0; i < ind+1; i++) {
                pest_new[i] = pest_centers[i];
            }
            // add in missing value
            pest_new[15] = (pest_centers[ind+1] - pest_centers[ind])/2 + pest_centers[ind];
            for (cpl_size i = ind+1; i < SLITLET_CNT; i++) {
                pest_new[i+1] = pest_centers[i];
            }
        } else {
            // 100mas or 250mas
            // start checking from slitlet #5 on until #18 (#14 and #16 are bad)
            int start_slit = 5,
                end_slit = 18;
            ind = 3*start_slit;
            cpl_size i = 0;
            // just copy, don't check
            for (i = 0; i < ind+1; i++) {
                pest_new[i] = pest_centers[i];
            }

            // check distances, if correct: copy, if not: estimate & copy
            ind = 3*start_slit + 1;
            i = ind;
            cpl_boolean found = CPL_FALSE;
            int decr_found = 0,
                est_trace_distance = 20;
            while (ind < 3*end_slit) {
                double val = fabs(pest_centers[ind]-pest_centers[ind-3]-SLITLET_WIDTH);
                if (found && (decr_found > 0)) {
                    val -= est_trace_distance;
                    decr_found--;
                }
                if (val < 5) {
                    // position is correct
                    pest_new[i++] = pest_centers[ind++];
                } else {
                    // position is missing
                    pest_new[i] = pest_centers[ind-3]+SLITLET_WIDTH;
                    i++;
                    pest_new[i] = pest_centers[ind++];
                    i++;
                    found = CPL_TRUE;
                    decr_found = 2;
                }
            }

            // just copy, don't check
            for (; i < 3*SLITLET_CNT; i++) {
                if (ind < cnt) {
                    pest_new[i] = pest_centers[ind++];
                }
            }
        }

        // check if there are trailing zeros.
        // If yes cut them. This is indicating that the profile is cut off at the left or right border
        cnt = cpl_vector_get_size(est_new);
        ind = cnt-1;
        while (pest_new[ind] < 0.0001 && ind > 0) {
            ind--;
        }
        if (ind != cnt-1) {
            BRK_IF_NULL(
                est_new2 = cpl_vector_extract(est_new, 0, ind, 1));
            eris_ifu_free_vector(&est_new);
            est_new = est_new2;
        }
    }
    CATCH
    {
        CATCH_MSGS();
        eris_ifu_free_vector(&est_new);
    }

    return est_new;
}

/**
 * @brief Fit Gaussian profiles to refine center positions
 * @param profile      The profile in X across the detector
 * @param est_centers  Estimated center positions from peak detection
 * @param do_fix_cnt   If TRUE, correct the count to exactly 32 or 96 traces
 * @return Vector with exactly 32 or 96 fitted center values, or NULL on error
 *
 * This function performs Gaussian fitting to refine center estimates:
 * - First call (do_fix_cnt=TRUE): Determines correct number of traces (32 or 96)
 * - Fits Gaussian to each estimated center to get sub-pixel precision
 * - If more traces detected than expected, uses statistical outlier rejection
 * - Subsequent calls (do_fix_cnt=FALSE): Handles edge cases where traces run off detector
 *
 * Outlier rejection strategy:
 * - Check leftmost and rightmost slitlets separately using median and stddev
 * - Reject peaks differing by more than thresh*stddev from local median
 * - Apply stricter global rejection if needed
 *
 * @note Returned vector must be freed by caller using cpl_vector_delete()
 * @note Function exits with error if less than 32 or 96 valid traces found
 */
cpl_vector* eris_ifu_dist_calc_centers_fit(const cpl_vector *profile,
                                           const cpl_vector *est_centers,
                                           cpl_boolean      do_fix_cnt)
{
    int                     cnt             = 0,
                            cnt2            = 0,
                            cen             = 0,
                            start           = 0,
                            end             = 0,
                            j               = 0;
    double                  med             = 0.,
                            std             = 0.,
                            thresh1         = 1.5,
                            thresh2         = 2.5,
                            lastValidOffset = 0.,
                            *pfit_centers   = NULL,
                            *pfit_peaks     = NULL;
    cpl_vector              *fit_centers    = NULL,
                            *fit_peaks      = NULL,
                            *est_centers2   = NULL;
    cpl_boolean             fix_cnt         = FALSE;
    struct gaussParStruct   gaussPar;
    eris_ifu_vector         *v              = NULL,
                            *ve             = NULL;

    cpl_ensure(profile, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(cpl_vector_get_size(profile) == ERIS_IFU_DETECTOR_SIZE,
               CPL_ERROR_ILLEGAL_INPUT, NULL);
    cpl_ensure(est_centers, CPL_ERROR_NULL_INPUT, NULL);

    TRY
    {
        BRK_IF_NULL(
            est_centers2 = eris_ifu_dist_estimate_low_slitlet(est_centers));

        cnt = (int) cpl_vector_get_size(est_centers2);

        if (do_fix_cnt) {
            // check if less than 32 traces
            if (cnt < SLITLET_CNT) {
                    cpl_msg_error(cpl_func, "Detected %d traces instead of at least %d!",
                                  cnt2, SLITLET_CNT);
                    SET_ERROR(CPL_ERROR_ILLEGAL_INPUT);
            }

            // check if less than 96 traces (and more than 32 of course)
            if ((cnt > 2*SLITLET_CNT) && (cnt < 3*SLITLET_CNT)) {
                cpl_msg_error(cpl_func, "Detected %d traces instead of at least %d!",
                              cnt2, 3*SLITLET_CNT);
                SET_ERROR(CPL_ERROR_ILLEGAL_INPUT);
            }
        }

        // we have more than 32 or 96 traces, has to fixed!
        if (do_fix_cnt && (cnt != SLITLET_CNT) && (cnt != 3*SLITLET_CNT)) {
            fix_cnt = CPL_TRUE;
        }

        // fit gaussians to all estimated centers in order to get position and
        // amplitude. If needed we fix the number of detected traces using
        // statistics on the amplitude
        BRK_IF_NULL(
            fit_centers = cpl_vector_new(cnt));
        BRK_IF_NULL(
            pfit_centers = cpl_vector_get_data(fit_centers));
        if (fix_cnt) {
            BRK_IF_NULL(
                fit_peaks = cpl_vector_new(cnt));
            BRK_IF_NULL(
                pfit_peaks = cpl_vector_get_data(fit_peaks));
        }

        for (int i = 0; i < cnt; i++) {
            cen = (int) (cpl_vector_get(est_centers2, i) + 0.5);
            BRK_IF_ERROR(
                eris_ifu_line_gauss_fit(profile, cen,
                                        NS_EST_SLIT_DIST/3, &gaussPar));

            if (gaussPar.errorCode == 0) {
                pfit_centers[i] = gaussPar.x0;
                lastValidOffset =
                        pfit_centers[i] - cpl_vector_get(est_centers2, i);
            } else if (i != 0) {
                pfit_centers[i] =  cpl_vector_get(est_centers2, i) +
                        lastValidOffset;
            }
            if (fix_cnt) {
                pfit_peaks[i] = gaussPar.area
                                  / sqrt(2*CPL_MATH_PI*pow(gaussPar.sigma,2))
                                     + gaussPar.offset;
            }
        }

        if (fix_cnt) {
            BRK_IF_NULL(
                v = eris_ifu_vector_create(fit_peaks));

            // check if there are any outliers on the left side of detector
            // (especially some wrong edge on the very left border,
            // therefore check the leftmost slitlets)
            start = 0;
            end = cnt/10*2;
            BRK_IF_NULL(
                ve = eris_ifu_vector_extract(v, start, end));
            med = eris_ifu_vector_get_median(ve, ERIS_IFU_ARITHMETIC);
            std = eris_ifu_vector_get_stdev_median(ve);
            if (std > 0.1) {
                // reject only if stddev is significant
                j = 0;
                for (int i = start; i <= end; i++) {
                    double val = eris_ifu_vector_get(ve, j++);
                    if ((val < med-thresh1*std) || (val > med+thresh1*std)) {
                        eris_ifu_vector_reject(v, i);
                    }
                }
            }
            eris_ifu_free_ifu_vector(&ve);

            // check if there are any outliers on the right side of detector
            start = cnt-1-cnt/10*2;
            end = cnt-1;
            BRK_IF_NULL(
                ve = eris_ifu_vector_extract(v, start, end));
            med = eris_ifu_vector_get_median(ve, ERIS_IFU_ARITHMETIC);
            std = eris_ifu_vector_get_stdev_median(ve);
            if (std > 0.1) {
                // reject only if stddev is significant
                j = 0;
                for (int i = start; i <= end; i++) {
                    double val = eris_ifu_vector_get(ve, j++);
                    if ((val < med-thresh1*std) || (val > med+thresh1*std)) {
                        eris_ifu_vector_reject(v, i);
                    }
                }
            }
            eris_ifu_free_ifu_vector(&ve);

            // still too many slitlets ?
            cnt2 = eris_ifu_vector_count_non_rejected(v);
            if ((cnt2 != SLITLET_CNT) && (cnt2 != 3*SLITLET_CNT)) {
                // now do a median across whole array and reject
                med = eris_ifu_vector_get_median(v, ERIS_IFU_ARITHMETIC);
                std = eris_ifu_vector_get_stdev_median(v);
                for (int i = 0; i <= end; i++) {
                    if (!eris_ifu_vector_is_rejected(v, i)) {
                        double val = eris_ifu_vector_get(v, i);
                        if ((val < med-thresh2*std) || (val > med+thresh2*std)) {
                            eris_ifu_vector_reject(v, i);
                        }
                    }
                }
            }

            // last check
            cnt2 = eris_ifu_vector_count_non_rejected(v);
            if ((cnt2 != SLITLET_CNT) && (cnt2 != 3*SLITLET_CNT)) {
                int cnt3 = 0;
                if (abs(SLITLET_CNT-cnt2) < 10) {
                    cnt3 = SLITLET_CNT;
                }
                if (abs(3*SLITLET_CNT-cnt2) < 10) {
                    cnt3 = 3*SLITLET_CNT;
                }

                cpl_msg_error(cpl_func, "Detected %d traces instead of %d!", cnt2, cnt3);
                SET_ERROR(CPL_ERROR_ILLEGAL_OUTPUT);
            } else {
                // we have now 32 or 3*32 traces!
                cpl_vector  *fit_centers2 = NULL;
                double      *pfit_centers2 = NULL;
                BRK_IF_NULL(
                    fit_centers2 = cpl_vector_new(cnt2));
                BRK_IF_NULL(
                    pfit_centers2 = cpl_vector_get_data(fit_centers2));
                j = 0;
                for (int i = 0; i < cnt; i++) {
                    if (!eris_ifu_vector_is_rejected(v, i)) {
                        pfit_centers2[j++] = pfit_centers[i];
                    }
                }
                cpl_vector_delete(fit_centers);
                fit_centers = fit_centers2;
            }
        } // end: fix_cnt
    }
    CATCH
    {
        CATCH_MSGS();
        eris_ifu_free_vector(&fit_centers);
    }

    eris_ifu_free_vector(&est_centers2);
    eris_ifu_free_ifu_vector(&ve);
    eris_ifu_free_ifu_vector(&v);
    eris_ifu_free_vector(&fit_peaks);

    return fit_centers;
}

/**
 * @brief Copy fitted centers to the output table array
 * @param fit_centers  Vector of fitted center positions
 * @param y_index      Row index in output tables
 * @param y_value      Y coordinate value to store
 * @param cen_array    Array of output tables (one per slitlet)
 * @return CPL_ERROR_NONE on success, error code otherwise
 *
 * This function distributes fitted centers into per-slitlet tables:
 * - For 32 traces: stores center position in "x" column
 * - For 96 traces: stores left, center, and right traces in "x_l", "x", "x_r" columns
 * - Handles edge cases where traces run off detector by comparing with neighboring rows
 *
 * @note Requires cen_array to be pre-allocated with SLITLET_CNT tables
 * @note Special logic handles partial traces at detector edges
 */
cpl_error_code eris_ifu_dist_calc_centers_copy(const cpl_vector *fit_centers,
                                               int              y_index,
                                               double           y_value,
                                               cpl_table        **cen_array)
{
    int             actual_size      = 0,
                    index_to_compare = 0,
                    tmp             = 0;
    double          last_val        = 0.,
                    actual_val      = 0.;
    cpl_error_code  err             = CPL_ERROR_NONE;
    const double    *pfit_centers   = NULL;

    cpl_ensure_code(fit_centers, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(cen_array, CPL_ERROR_NULL_INPUT);

    TRY
    {
        actual_size = (int) cpl_vector_get_size(fit_centers);
        BRK_IF_NULL(
            pfit_centers = cpl_vector_get_data_const(fit_centers));

        if (actual_size == SLITLET_CNT) {
            for (int i = 0; i < SLITLET_CNT; i++) {
                BRK_IF_ERROR(
                    cpl_table_set_double(cen_array[i], "y", y_index, y_value));
                BRK_IF_ERROR(
                    cpl_table_set_double(cen_array[i], "x", y_index, pfit_centers[i]));
            }
        } else if (actual_size == 3*SLITLET_CNT) {
            for (int i = 0; i < SLITLET_CNT; i++) {
                BRK_IF_ERROR(
                    cpl_table_set_double(cen_array[i], "y", y_index, y_value));

                // process 1st line
                BRK_IF_ERROR(
                    cpl_table_set_double(cen_array[i], "x_l", y_index, pfit_centers[3*i]));

                // process 2nd line
                BRK_IF_ERROR(
                    cpl_table_set_double(cen_array[i], "x", y_index, pfit_centers[3*i+1]));

                // process 3rd line
                BRK_IF_ERROR(
                    cpl_table_set_double(cen_array[i], "x_r", y_index, pfit_centers[3*i+2]));
            }
        } else {
            cpl_boolean special_handling = CPL_FALSE;

            if (y_index < cpl_table_get_nrow(cen_array[0])/2) {
                // lower detector part: compare more to the middle
                index_to_compare = y_index+1;
            } else {
                // higher detector part: compare more to the middle
                index_to_compare = y_index-1;
            }

            if (!special_handling && (actual_size < 3*SLITLET_CNT)) {
                // check if left edge is cut off from image
                last_val = cpl_table_get_double(cen_array[0], "x_l", index_to_compare, &tmp);
                actual_val = pfit_centers[0];
                if (!special_handling && (fabs(last_val-actual_val) < 5.)) {
                    // left edge is here!
                    // fill from left to right until end of vector with actual_vals is reached
                    // (it is supposed that max 1-2 traces are missing)
                    tmp = 0;
                    for (int i = 0; i < SLITLET_CNT; i++) {
                        BRK_IF_ERROR(
                            cpl_table_set_double(cen_array[i], "y", y_index, y_value));

                        // process 1st line
                        BRK_IF_ERROR(
                            cpl_table_set_double(cen_array[i], "x_l", y_index, pfit_centers[3*i]));
                        if (++tmp == actual_size) break;

                        // process 2nd line
                        BRK_IF_ERROR(
                            cpl_table_set_double(cen_array[i], "x", y_index, pfit_centers[3*i+1]));
                        if (++tmp == actual_size) break;

                        // process 3rd line
                        BRK_IF_ERROR(
                            cpl_table_set_double(cen_array[i], "x_r", y_index, pfit_centers[3*i+2]));
                        if (++tmp == actual_size) break;
                    }
                    special_handling = CPL_TRUE;
                }

                // check if right edge is cut off from image
                last_val = cpl_table_get_double(cen_array[SLITLET_CNT-1], "x_r", index_to_compare, &tmp);
                actual_val = pfit_centers[actual_size-1];
                if (!special_handling && (fabs(last_val-actual_val) < 5.)) {
                    // right edge is here!
                    // fill from rihgt to left until end of vector with actual_vals is reached
                    // (it is supposed that max 1-2 traces are missing)
                    tmp = actual_size;
                    for (int i = SLITLET_CNT-1; i >= 0; i--) {
                        BRK_IF_ERROR(
                            cpl_table_set_double(cen_array[i], "y", y_index, y_value));

                        // process 3rd line
                        BRK_IF_ERROR(
                            cpl_table_set_double(cen_array[i], "x_r", y_index, pfit_centers[tmp-1]));
                        if (--tmp == actual_size) break;

                        // process 2nd line
                        BRK_IF_ERROR(
                            cpl_table_set_double(cen_array[i], "x", y_index, pfit_centers[tmp-1]));
                        if (--tmp == actual_size) break;

                        // process 1st line
                        BRK_IF_ERROR(
                            cpl_table_set_double(cen_array[i], "x_l", y_index, pfit_centers[tmp-1]));
                        if (--tmp == actual_size) break;
                    }
                    special_handling = CPL_TRUE;
                }
            } // if (actual_size < 3*SLITLET_CNT)

            if (!special_handling && (actual_size < SLITLET_CNT)) {
                // check if left edge is cut off from image
                last_val = cpl_table_get_double(cen_array[0], "x", index_to_compare, &tmp);
                actual_val = pfit_centers[0];
                if (!special_handling && (fabs(last_val-actual_val) < 5.)) {
                    // left edge is here!
                    // fill from left to right until end of vector with actual_vals is reached
                    // (it is supposed that max 1-2 traces are missing)
                    tmp = 0;
                    for (int i = 0; i < SLITLET_CNT; i++) {
                        BRK_IF_ERROR(
                            cpl_table_set_double(cen_array[i], "y", y_index, y_value));
                        BRK_IF_ERROR(
                            cpl_table_set_double(cen_array[i], "x", y_index, pfit_centers[i]));
                        if (++tmp == actual_size) break;
                    }
                    special_handling = CPL_TRUE;
                }

                // check if right edge is cut off from image
                last_val = cpl_table_get_double(cen_array[SLITLET_CNT-1], "x", index_to_compare, &tmp);
                actual_val = pfit_centers[actual_size-1];
                if (!special_handling && (fabs(last_val-actual_val) < 5.)) {
                    // right edge is here!
                    // fill from right to left until end of vector with actual_vals is reached
                    // (it is supposed that max 1-2 traces are missing)
                    tmp = actual_size;
                    for (int i = SLITLET_CNT-1; i >= 0; i--) {
                        BRK_IF_ERROR(
                            cpl_table_set_double(cen_array[i], "y", y_index, y_value));
                        BRK_IF_ERROR(
                            cpl_table_set_double(cen_array[i], "x", y_index, pfit_centers[tmp-1]));
                        if (--tmp == actual_size) break;
                    }
                    special_handling = CPL_TRUE;
                }
            } // if (actual_size < SLITLET_CNT)

            if (!special_handling) {
                BRK_WITH_ERROR_MSG(CPL_ERROR_ILLEGAL_OUTPUT,
                                                 "Handling of trace running out of "
                                                 "detector image failed!!!");
            }
        } // if actual_size = ...
    }
    CATCH
    {
        CATCH_MSGS();
        for (int j = 0; j < SLITLET_CNT; j++) {
            cpl_table_set_double(cen_array[j], "x", y_index, 0.);
            cpl_table_set_double(cen_array[j], "y", y_index, 0.);
        }

        err = cpl_error_get_code();
    }

    return err;
}

/**
 * @brief Calculate slitlet centers across the detector
 * @param fibre_div      Divided fiber flat field image
 * @param fibre_on       Fiber illumination image
 * @param productDepth   Debug output level (bit flags)
 * @return Array of SLITLET_CNT tables containing fitted center positions, or NULL on error
 *
 * This is the main function for determining slitlet center positions:
 * 1. Takes horizontal chunk in middle of detector
 * 2. Collapses in Y to get profile
 * 3. Estimates and fits centers to determine number of traces (32 or 96)
 * 4. Allocates output tables accordingly
 * 5. Processes full detector in chunks from center outward
 * 6. Stores results in per-slitlet tables with columns:
 *    - Single trace: "x", "y"
 *    - Triple trace: "x_l", "x", "x_r", "y"
 *
 * @note Each element of returned array must be freed with cpl_table_delete()
 * @note Array itself must be freed with cpl_free()
 * @note Debug files saved if productDepth & 4
 */
cpl_table** eris_ifu_dist_calc_centers(const hdrl_image* fibre_div,
                                       const hdrl_image *fibre_on,
//                                       cpl_boolean is25mas,
//                                       cpl_boolean isK,
                                       int productDepth)
{
    int             cnt                 = 0,
                    nr_values           = 21,   //odd
 //                   ny                  = 0,
                    center_y            = 0,
                    height              = 96,   // even
//                    ix_middle           = nr_values / 2 + 1;
                    ix_middle           = 9;
    const cpl_image *img                = NULL,
                    *img_div            = NULL,
                    *img_on             = NULL;
    cpl_vector      *profile_x          = NULL,
                    *profile_x_div      = NULL,
                    *profile_x_on       = NULL,
                    *est_centers        = NULL,
                    *est_centers_div    = NULL,
                    *est_centers_on     = NULL,
                    *fit_centers        = NULL,
                    *fit_centers_middle = NULL;
    cpl_table       **cen_array         = NULL;


    cpl_ensure(fibre_div, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(fibre_on, CPL_ERROR_NULL_INPUT, NULL);
    // nr_values should be odd, since we process first the middle value
    // and the the lower part resp. upper part
    cpl_ensure(nr_values % 2 == 1, CPL_ERROR_ILLEGAL_INPUT, NULL);
    cpl_ensure(height % 2 == 0, CPL_ERROR_ILLEGAL_INPUT, NULL);

    TRY
    {

        BRK_IF_NULL(
            img_on = hdrl_image_get_image_const(fibre_on));
        BRK_IF_NULL(
            img_div = hdrl_image_get_image_const(fibre_div));

//        ny = (int) cpl_image_get_size_y(img_div);

        BRK_IF_NULL(
            cen_array = cpl_calloc(SLITLET_CNT, sizeof(cpl_table*)));

        /* - take horizontal chunk of height 96 in the middle of detector in x
         * - collapse in y
         * - get profile
         * - estimate centers
         * - allocate arrays accordingly (32 or 3*32 tables)
         * - fit centers
         * - store in the middle line of whole array
         */
        center_y = ERIS_IFU_DETECTOR_BP_BORDER + ix_middle*height + height/2;
        BRK_IF_NULL(
            profile_x_div = eris_ifu_calc_centers_collapse_chunk(
                                                        img_div, center_y, height));
        BRK_IF_NULL(
            profile_x_on = eris_ifu_calc_centers_collapse_chunk(
                                                        img_on, center_y, height));

        BRK_IF_NULL(
            est_centers_div = eris_ifu_dist_calc_centers_profile(profile_x_div, CPL_FALSE));
        BRK_IF_NULL(
            est_centers_on = eris_ifu_dist_calc_centers_profile(profile_x_on, CPL_FALSE));

        // will return a vector with 32 or 3*32 center values (errors are catched)
        BRK_IF_NULL(
            fit_centers_middle = eris_ifu_dist_calc_centers_fit(
                profile_x_on, est_centers_on, CPL_TRUE));

        if (productDepth  & 4) {
            cpl_propertylist *pl = NULL;
            BRK_IF_NULL(
                pl = cpl_propertylist_new());
            eris_ifu_save_vector_dbg(NULL, ERIS_IFU_DIST_DBG_FN, CPL_IO_CREATE, pl);
            BRK_IF_ERROR(
                cpl_propertylist_update_string(pl, "EXTNAME", "profile_x_div"));
            eris_ifu_save_vector_dbg(profile_x_div,
                    ERIS_IFU_DIST_DBG_FN, CPL_IO_EXTEND, pl);
            BRK_IF_ERROR(
                cpl_propertylist_update_string(pl, "EXTNAME", "profile_x_on"));
            eris_ifu_save_vector_dbg(profile_x_on,
                    ERIS_IFU_DIST_DBG_FN, CPL_IO_EXTEND, pl);
            BRK_IF_ERROR(
                cpl_propertylist_update_string(pl, "EXTNAME", "est_centers_div"));
            eris_ifu_save_vector_dbg(est_centers_div,
                    ERIS_IFU_DIST_DBG_FN, CPL_IO_EXTEND, pl);
            BRK_IF_ERROR(
                cpl_propertylist_update_string(pl, "EXTNAME", "est_centers_on"));
            eris_ifu_save_vector_dbg(est_centers_on,
                    ERIS_IFU_DIST_DBG_FN, CPL_IO_EXTEND,pl );
            BRK_IF_ERROR(
                cpl_propertylist_update_string(pl, "EXTNAME", "fit_centers_middle"));
            eris_ifu_save_vector_dbg(fit_centers_middle,
                    ERIS_IFU_DIST_DBG_FN, CPL_IO_EXTEND,pl);
            eris_ifu_free_propertylist(&pl);
        }
        eris_ifu_free_vector(&profile_x_div);
        eris_ifu_free_vector(&profile_x_on);
        eris_ifu_free_vector(&est_centers_div);
        eris_ifu_free_vector(&est_centers_on);
        eris_ifu_free_vector(&fit_centers);

        /* in the 1st iteration we know now the number of fibre traces
         * (one or three per slitlet)
         * --> Allocate images to hold all centers*/
        cnt = (int) cpl_vector_get_size(fit_centers_middle);

        if (cnt == SLITLET_CNT) {
            // one trace per slitlet
            for (int j = 0; j < SLITLET_CNT; j++) {
                BRK_IF_NULL(
                    cen_array[j] = cpl_table_new(nr_values));
                BRK_IF_ERROR(
                    cpl_table_new_column(cen_array[j], "x", CPL_TYPE_DOUBLE));
                BRK_IF_ERROR(
                    cpl_table_new_column(cen_array[j], "y", CPL_TYPE_DOUBLE));
                for (int i = 0; i < nr_values; i++) {
                    // init to NaN
                    cpl_table_set(cen_array[j], "x", i, NAN);
                    cpl_table_set(cen_array[j], "y", i, NAN);
                }
            }
        } else {
            // three traces per slitlet
            for (int j = 0; j < SLITLET_CNT; j++) {
                BRK_IF_NULL(
                    cen_array[j] = cpl_table_new(nr_values));
                BRK_IF_ERROR(
                    cpl_table_new_column(cen_array[j], "x_l", CPL_TYPE_DOUBLE));
                BRK_IF_ERROR(
                    cpl_table_new_column(cen_array[j], "x", CPL_TYPE_DOUBLE));
                BRK_IF_ERROR(
                    cpl_table_new_column(cen_array[j], "x_r", CPL_TYPE_DOUBLE));
                BRK_IF_ERROR(
                    cpl_table_new_column(cen_array[j], "y", CPL_TYPE_DOUBLE));
                for (int i = 0; i < nr_values; i++) {
                    // init to NaN
                    cpl_table_set(cen_array[j], "x_l", i, NAN);
                    cpl_table_set(cen_array[j], "x", i, NAN);
                    cpl_table_set(cen_array[j], "x_r", i, NAN);
                    cpl_table_set(cen_array[j], "y", i, NAN);
                }
            }
        }

//erw        if (isK/*25mas*/) {
        if (CPL_TRUE) {
            img = img_on;
        } else {
            img = img_div;
        }
        // now do the same for the whole detectors in two blocks
        // center to top and center to bottom
        CHECK_ERROR_STATE();
        BRK_IF_NULL(est_centers = cpl_vector_duplicate(fit_centers_middle));
        for (int i = ix_middle; i < nr_values; i++) {
            center_y = ERIS_IFU_DETECTOR_BP_BORDER + i*height + height/2;
            /* collapse image in y and convert to vector */
            BRK_IF_NULL(
                    profile_x = eris_ifu_calc_centers_collapse_chunk(
                        img, center_y, height));
            /* do least square fit using a Gaussian in order to get exact centers */
            BRK_IF_NULL(
                    fit_centers = eris_ifu_dist_calc_centers_fit(profile_x, est_centers, CPL_FALSE));

            /* copy all values into image-array to hold all values */
            BRK_IF_ERROR(
                    eris_ifu_dist_calc_centers_copy(fit_centers, i, center_y,
                        cen_array));

            if ((productDepth & 4) != 0) {
                char *extname = NULL;
                cpl_propertylist *pl = NULL;
                BRK_IF_NULL(
                    pl = cpl_propertylist_new());
                extname = cpl_sprintf("Sx_%02d profile_x", i+1);
                cpl_propertylist_update_string(pl, "EXTNAME", extname);
                eris_ifu_free_string(&extname);
                eris_ifu_save_vector_dbg(profile_x,
                        ERIS_IFU_DIST_DBG_FN, CPL_IO_EXTEND, pl);
                extname = cpl_sprintf("Sx_%02d est_centers", i+1);
                cpl_propertylist_update_string(pl, "EXTNAME", extname);
                eris_ifu_free_string(&extname);
                eris_ifu_save_vector_dbg(est_centers,
                        ERIS_IFU_DIST_DBG_FN, CPL_IO_EXTEND, pl);
                extname = cpl_sprintf("Sx_%02d fit_centers", i+1);
                cpl_propertylist_update_string(pl, "EXTNAME", extname);
                eris_ifu_free_string(&extname);
                eris_ifu_save_vector_dbg(fit_centers,
                        ERIS_IFU_DIST_DBG_FN, CPL_IO_EXTEND ,pl);
                eris_ifu_free_propertylist(&pl);
            }

            BRK_IF_ERROR(cpl_vector_copy(est_centers, fit_centers));
            eris_ifu_free_vector(&profile_x);
            eris_ifu_free_vector(&fit_centers);
        } // end: i = nr_values

        BRK_IF_ERROR(cpl_vector_copy(est_centers, fit_centers_middle));
        for (int i = ix_middle-1; i >= 0; i--) {
            center_y = ERIS_IFU_DETECTOR_BP_BORDER + i*height + height/2;
            /* collapse image in y and convert to vector */
            BRK_IF_NULL(
                profile_x = eris_ifu_calc_centers_collapse_chunk(
                                                            img, center_y, height));
            /* do least square fit using a Gaussian in order to get exact centers */
            BRK_IF_NULL(
                fit_centers = eris_ifu_dist_calc_centers_fit(profile_x, est_centers, CPL_FALSE));

            /* copy all values into image-array to hold all values */
            BRK_IF_ERROR(
                eris_ifu_dist_calc_centers_copy(fit_centers, i, center_y,
                                                cen_array));

            if ((productDepth & 4) != 0) {
                char *extname = NULL;
                cpl_propertylist *pl = NULL;
                BRK_IF_NULL(
                    pl = cpl_propertylist_new());
                extname = cpl_sprintf("Sx_%02d profile_x", i+1);
                cpl_propertylist_update_string(pl, "EXTNAME", extname);
                eris_ifu_free_string(&extname);
                eris_ifu_save_vector_dbg(profile_x,
                        ERIS_IFU_DIST_DBG_FN, CPL_IO_EXTEND, pl);
                extname = cpl_sprintf("Sx_%02d est_centers", i+1);
                cpl_propertylist_update_string(pl, "EXTNAME", extname);
                eris_ifu_free_string(&extname);
                eris_ifu_save_vector_dbg(est_centers,
                        ERIS_IFU_DIST_DBG_FN, CPL_IO_EXTEND, pl);
                extname = cpl_sprintf("Sx_%02d fit_centers", i+1);
                cpl_propertylist_update_string(pl, "EXTNAME", extname);
                eris_ifu_free_string(&extname);
                eris_ifu_save_vector_dbg(fit_centers,
                        ERIS_IFU_DIST_DBG_FN, CPL_IO_EXTEND, pl);
                eris_ifu_free_propertylist(&pl);
            }

            BRK_IF_ERROR(cpl_vector_copy(est_centers, fit_centers));
            eris_ifu_free_vector(&profile_x);
            eris_ifu_free_vector(&fit_centers);
      } // end: i = nr_values

        if (productDepth  & 4) {
            /* save centers */
            BRK_IF_ERROR(
                cpl_propertylist_save(NULL, "eris_ifu_distortion_dbg_centers_fitted.fits", CPL_IO_CREATE));
            BRK_IF_ERROR(
                cpl_propertylist_save(NULL, "eris_ifu_distortion_dbg_centers_fitted_visual.fits", CPL_IO_CREATE));
            for (int j = 0; j < SLITLET_CNT; j++) {
                cpl_table_save(cen_array[j], NULL, NULL, "eris_ifu_distortion_dbg_centers_fitted.fits", CPL_IO_EXTEND);
            }
        }
    }
    CATCH
    {
        CATCH_MSGS();
        for (int i = 0; i < SLITLET_CNT; i++) {
             eris_ifu_free_table(&cen_array[i]);
        }
        cpl_free(cen_array); cen_array = NULL;
    }

    eris_ifu_free_vector(&profile_x_div);
    eris_ifu_free_vector(&profile_x_on);
    eris_ifu_free_vector(&est_centers_div);
    eris_ifu_free_vector(&est_centers_on);
    eris_ifu_free_vector(&est_centers);
    eris_ifu_free_vector(&fit_centers);
    eris_ifu_free_vector(&fit_centers_middle);

    return cen_array;
}

int eris_ifu_distortion_calc_y(int n, int i) {
    // split ERIS_IFU_DETECTOR_SIZE_Y in n parts
    // shift every point half down
    // round to int with adding 0.5 for proper rounding
    return (int)((double)ERIS_IFU_DETECTOR_SIZE_Y/n*(i+.5)+.5);
}

int eris_ifu_distortion_target_left_edge(int i) {
    return i * SLITLET_WIDTH/* + 1*/;
}
int eris_ifu_distortion_target_right_edge(int i) {
    return (i + 1) * SLITLET_WIDTH - 1;
}

/**
 * @brief Determine number of arc traces to process for a slitlet
 * @param i               Slitlet index
 * @param triple_traces   TRUE if 3 traces per slitlet, FALSE if 1
 * @param cut_off_left    TRUE if leftmost slitlet is cut off
 * @param cut_off_right   TRUE if rightmost slitlet is cut off
 * @return Number of arc traces (3 or 5 for single-fiber, 5 or 7 for triple-fiber)
 *
 * Counts: left edge + right edge + center line(s)
 * Decrements count if edge slitlet is cut off at detector boundary.
 */
int eris_ifu_distortion_get_narcs(int           i,
                                  cpl_boolean   triple_traces ,
                                  cpl_boolean   cut_off_left,
                                  cpl_boolean   cut_off_right)
{
    int n_arcs = 0;
    // get number of vertical traces (arcs) to process
    if (triple_traces) {
        n_arcs = 5; // left + right edge + 3x center line
    } else {
        n_arcs = 3; // left + right edge + 1x center line
    }

    if (((i == 0) && cut_off_left) ||
        ((i == SLITLET_CNT-1) && cut_off_right))
    {
        // if left or right edge is cut off, omit it
        n_arcs--;
    }
    return n_arcs;
}

/**
 * @brief Calculate distortion polynomials for all slitlets
 * @param slit_edges      Array of tables with detected slit edge positions
 * @param centers         Array of tables with fitted slitlet centers
 * @param productDepth    Debug output level
 * @param cut_off_left    Indicates if leftmost slitlet is cut off
 * @param cut_off_right   Indicates if rightmost slitlet is cut off
 * @param minmax_borders  Output: table with min/max border positions per slitlet
 * @param qc              Output: array of QC property lists (one per slitlet)
 * @param pl              Property list for FITS headers
 * @param frameset        Input frameset
 * @param parlist         Parameter list
 * @return Array of SLITLET_CNT distortion polynomials, or NULL on error
 *
 * This function computes 2D polynomial distortion maps:
 * 1. Fits polynomials to edge positions as function of Y
 * 2. Estimates cut-off edges using neighboring slitlet widths
 * 3. Defines target positions in rectified coordinate system
 * 4. Builds 2D grid of (target_x, target_y) -> measured_x
 * 5. Fits 2D polynomial to this mapping
 * 6. Computes QC statistics (slitlet widths, center positions, etc.)
 *
 * @note Polynomial degree is 2 in X and 3 in Y
 * @note Each polynomial must be freed with cpl_polynomial_delete()
 * @note Array must be freed with cpl_free()
 * @note QC property lists contain statistics on fitted edges
 */
cpl_polynomial** eris_ifu_dist_calc_distortion(cpl_table        **slit_edges,
                                               cpl_table        **centers,
                                               int              productDepth,
                                               cpl_boolean      cut_off_left,
                                               cpl_boolean      cut_off_right,
                                               cpl_table        **minmax_borders,
                                               cpl_propertylist ***qc,
                                               cpl_propertylist *pl,
					       cpl_frameset* frameset,
					       const cpl_parameterlist* parlist)
{
    int                 n_arcs          = 0,
                        n_calib_wave    = 0,
                        n_calib_cen     = 0,
                        n_size          = 100,
// AA expect results that are more forgiving
//                        fit_order       = 3,
                        fit_order       = 2,
                        arc_cnt         = 0,
                        pos             = 0;
    double              o_l5            = 0.,
                        o_c5            = 0.,
                        o_r5            = 0.,
                        o_l             = 0.,
                        o_c             = 0.,
                        o_r             = 0.,
                        o_c_l           = 0.,
                        o_c_r           = 0.,
                        t_o             = 0,
                        t_l             = 0,
                        t_r             = 0,
                        t_c             = 0,
                        t_c_l           = 0,
                        t_c_r           = 0,
                        m_c             = 0,
                        m_c_l           = 0,
                        m_c_r           = 0,
//                        m_t_t           = 0,
                        std             = 0.,
                        median          = 0.,
                        l_min           = 0.,
                        *py             = NULL;
    const double        *pwave_y        = NULL,
                        *pcenter_y      = NULL;
    cpl_polynomial      **poly_u        = NULL;
    cpl_bivector        *grid           = NULL;
    cpl_vector          *val_to_fit     = NULL,
                        *tmp_vec        = NULL,
                        *y              = NULL;
    eris_ifu_vector     *y_wave         = NULL,
                        *y_cen          = NULL,
                        *t              = NULL;
    cpl_boolean         triple_traces   = FALSE;
    cpl_table           *tbl            = NULL,
                        *dbg_polynomials = NULL;
    struct arcstruct    *arc_tbl        = NULL;


    cpl_ensure(slit_edges, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(centers, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(minmax_borders, CPL_ERROR_NULL_INPUT, NULL);

    TRY
    {
        BRK_IF_NULL( dbg_polynomials = cpl_table_new(SLITLET_CNT));
        cpl_table_new_column(dbg_polynomials, ERIS_IFU_DIST_DBG_SLITLET, CPL_TYPE_INT);
        cpl_table_new_column_array(dbg_polynomials, ERIS_IFU_DIST_DBG_EDGE_LEFT, CPL_TYPE_DOUBLE, fit_order+1);
        cpl_table_new_column_array(dbg_polynomials, ERIS_IFU_DIST_DBG_EDGE_RIGHT, CPL_TYPE_DOUBLE, fit_order+1);
        cpl_table_new_column_array(dbg_polynomials, ERIS_IFU_DIST_DBG_CENTER_LEFT, CPL_TYPE_DOUBLE, fit_order+1);
        cpl_table_new_column_array(dbg_polynomials, ERIS_IFU_DIST_DBG_CENTER, CPL_TYPE_DOUBLE, fit_order+1);
        cpl_table_new_column_array(dbg_polynomials, ERIS_IFU_DIST_DBG_CENTER_RIGHT, CPL_TYPE_DOUBLE, fit_order+1);
        CHECK_ERROR_STATE();

        BRK_IF_NULL(
            *minmax_borders = cpl_table_new(SLITLET_CNT));
        BRK_IF_ERROR(
            cpl_table_new_column(*minmax_borders, ERIS_IFU_POLY_EDGE_L, CPL_TYPE_DOUBLE));
        BRK_IF_ERROR(
            cpl_table_new_column(*minmax_borders, ERIS_IFU_POLY_EDGE_R, CPL_TYPE_DOUBLE));

        BRK_IF_NULL(
            poly_u = cpl_calloc(SLITLET_CNT, sizeof(cpl_polynomial*)));

        if (cpl_table_get_ncol(centers[5]) > 2) {
            triple_traces = TRUE;
        }

        if (productDepth & 1) {
        	cpl_propertylist_append_string(pl, CPL_DFS_PRO_CATG,
        	        			 "DIST_FIT_TABLE");
        	 cpl_propertylist_save(pl, ERIS_IFU_PRO_DIST_QC_FIT_FN,
        			 CPL_IO_CREATE);
        	 eris_setup_product_header(ERIS_IFU_PRO_DIST_QC_FIT_FN,
        			 "DIST_FIT_TABLE", CPL_FRAME_TYPE_TABLE,
					 "eris_ifu_distortion", frameset, parlist, pl);
        }
        BRK_IF_NULL(
            arc_tbl = cpl_calloc(SLITLET_CNT, sizeof(struct arcstruct)));

        // calculate all slitlets that are not cut off
        for (int i = 0; i < SLITLET_CNT; i++) {
            //cpl_msg_debug(cpl_func, "i: %d\n", i);
            // get number of vertical traces (arcs) to process
            n_arcs = eris_ifu_distortion_get_narcs(i, triple_traces,
                                                   cut_off_left, cut_off_right);
            // get number of detected wave_calib-lines
            n_calib_wave = (int) cpl_table_get_nrow(slit_edges[i]);
            // get number of centerpoints
            n_calib_cen = (int) cpl_table_get_nrow(centers[i]);

            // get y-coordinates
            BRK_IF_NULL(
                pwave_y = cpl_table_get_data_double_const(
                                             slit_edges[i], "y_pos"));
            BRK_IF_NULL(
                pcenter_y = cpl_table_get_data_double_const(
                                             centers[i], "y"));
            BRK_IF_NULL(
                y_wave = eris_ifu_vector_new_wrap(n_calib_wave, pwave_y));
            BRK_IF_NULL(
                y_cen = eris_ifu_vector_new_wrap(n_calib_cen, pcenter_y));

            // fit all edges (left, right, centers)
            if (!((i == 0) && cut_off_left)) {
                // process only if it is not the 1st slitlet and cutoff
                BRK_IF_NULL(
                    arc_tbl[i].fit_l = eris_ifu_dist_calc_distortion_fitedge(
                                            slit_edges[i], "edge_left", y_wave,
                                            n_calib_wave, n_size, fit_order,
                                            i, dbg_polynomials));
            }
            if (!((i == SLITLET_CNT-1) && cut_off_right)) {
                // process only if it is not the last slitlet and cutoff
                BRK_IF_NULL(
                    arc_tbl[i].fit_r = eris_ifu_dist_calc_distortion_fitedge(
                                            slit_edges[i], "edge_right", y_wave,
                                            n_calib_wave, n_size, fit_order,
                                            i, dbg_polynomials));
            }
            BRK_IF_NULL(
                arc_tbl[i].fit_c = eris_ifu_dist_calc_distortion_fitedge(
                                            centers[i], "x", y_cen,
                                            n_calib_cen, n_size, fit_order,
                                            i, dbg_polynomials));
            if (triple_traces) {
                if (!((i == 0) && cut_off_left)) {
                    // process if it is not the 1st slitlet and not cutoff
                    BRK_IF_NULL(
                        arc_tbl[i].fit_c_l = eris_ifu_dist_calc_distortion_fitedge(
                                                centers[i], "x_l", y_cen,
                                                n_calib_cen, n_size, fit_order,
                                                i, dbg_polynomials));
                } else {
                    // if it is the last slitlet and cutoff, then
                    // check first if, and how many, values are nan
                    // If there are too many NaN's do nothing and copy the shape of c to c_l
                    if (cpl_table_count_invalid(centers[i], "x_l") <= 0.1 * n_calib_cen) {
                        // calculate fitted edge only if less tahn 10% of the values are NaN
                        BRK_IF_NULL(
                            arc_tbl[i].fit_c_l = eris_ifu_dist_calc_distortion_fitedge(
                                                    centers[i], "x_l", y_cen,
                                                    n_calib_cen, n_size, fit_order,
                                                    i, dbg_polynomials));
                    }
                }

                if (!((i == SLITLET_CNT-1) && cut_off_right)) {
                    // process if it is not the last slitlet and not cutoff
                    BRK_IF_NULL(
                        arc_tbl[i].fit_c_r = eris_ifu_dist_calc_distortion_fitedge(
                                                centers[i], "x_r", y_cen,
                                                n_calib_cen, n_size, fit_order,
                                                i, dbg_polynomials));
                } else {
                    // if it is the last slitlet and cutoff, then
                    // check first if, and how many, values are nan
                    // If there are too many NaN's do nothing and copy the shape of c to c_r
                    if (cpl_table_count_invalid(centers[i], "x_r") <= 0.1 * n_calib_cen) {
                        // calculate fitted edge only if less than 10% of the values are NaN
                        BRK_IF_NULL(
                            arc_tbl[i].fit_c_r = eris_ifu_dist_calc_distortion_fitedge(
                                                    centers[i], "x_r", y_cen,
                                                    n_calib_cen, n_size, fit_order,
                                                    i, dbg_polynomials));
                    }
                }
            } // end: if (triple_traces)
            eris_ifu_free_ifu_vector(&y_wave);
            eris_ifu_free_ifu_vector(&y_cen);
        } // end: i = SLITLET_CNT

        BRK_IF_ERROR( cpl_table_save(dbg_polynomials, NULL, NULL,
                "eris_ifu_distortion_dbg_polynomials.fits", CPL_IO_CREATE));

        BRK_IF_NULL(
            *qc = cpl_calloc(SLITLET_CNT, sizeof(cpl_propertylist*)));
        for (int i = 0; i < SLITLET_CNT; i++) {
            BRK_IF_NULL(
                (*qc)[i] = cpl_propertylist_new());
        }

        // estimate positions of cut off edges with neighbouring slitlet width
        if (cut_off_left) {
            if (arc_tbl[0].fit_l == NULL) {
                /* The leftmost edge of slitlet #0 is cut off. Therefore subtract the
                 * width of slitlet #1 from the right edge of slitlet #0
                 */
                BRK_IF_ERROR(
                    cpl_propertylist_append_string((*qc)[0],
                                                    "ESO EDGE L", "artificial"));
                BRK_IF_NULL(
                    arc_tbl[0].fit_l = eris_ifu_vector_duplicate(arc_tbl[0].fit_r));
                BRK_IF_ERROR(
                    eris_ifu_vector_subtract(arc_tbl[0].fit_l, arc_tbl[1].fit_r));
                BRK_IF_ERROR(
                    eris_ifu_vector_add(arc_tbl[0].fit_l, arc_tbl[1].fit_l));
            }
            if (triple_traces && (arc_tbl[SLITLET_CNT-1].fit_c_l == NULL)) {
                /* The c_l-edge of slitlet #0 is cut off. Therefore subtract the
                 * width of c-c_l of slitlet #31 to the left edge of slitlet #32
                 */
                BRK_IF_ERROR(
                    cpl_propertylist_append_string((*qc)[0],
                                                    "ESO EDGE C-L", "artificial"));
                BRK_IF_NULL(
                    arc_tbl[0].fit_c_l = eris_ifu_vector_duplicate(arc_tbl[0].fit_c));
                BRK_IF_ERROR(
                    eris_ifu_vector_subtract(arc_tbl[0].fit_c_l, arc_tbl[1].fit_c));
                BRK_IF_ERROR(
                    eris_ifu_vector_add(arc_tbl[0].fit_c_l, arc_tbl[1].fit_c_l));
            }
            cut_off_left = CPL_FALSE;
        }

        if (cut_off_right) {
            if (arc_tbl[SLITLET_CNT-1].fit_r == NULL) {
                /* The rightmost edge of slitlet #32 is cut off. Therefore add the
                 * width of slitlet #31 to the left edge of slitlet #32
                 */
                BRK_IF_ERROR(
                    cpl_propertylist_append_string((*qc)[SLITLET_CNT-1],
                                                    "ESO EDGE R", "artificial"));
                BRK_IF_NULL(
                    arc_tbl[SLITLET_CNT-1].fit_r = eris_ifu_vector_duplicate(arc_tbl[SLITLET_CNT-1].fit_l));
                BRK_IF_ERROR(
                    eris_ifu_vector_add(arc_tbl[SLITLET_CNT-1].fit_r, arc_tbl[SLITLET_CNT-2].fit_r));
                BRK_IF_ERROR(
                    eris_ifu_vector_subtract(arc_tbl[SLITLET_CNT-1].fit_r, arc_tbl[SLITLET_CNT-2].fit_l));
            }
            if (triple_traces && (arc_tbl[SLITLET_CNT-1].fit_c_r == NULL)) {
                /* The c_r-edge of slitlet #32 is cut off. Therefore add the
                 * width of c_r-c of slitlet #31 to the left center edge of slitlet #32
                 */
                BRK_IF_ERROR(
                    cpl_propertylist_append_string((*qc)[0],
                                                    "ESO EDGE C-R", "artificial"));
                BRK_IF_NULL(
                    arc_tbl[SLITLET_CNT-1].fit_c_r = eris_ifu_vector_duplicate(arc_tbl[SLITLET_CNT-1].fit_c));
                BRK_IF_ERROR(
                    eris_ifu_vector_add(arc_tbl[SLITLET_CNT-1].fit_c_r, arc_tbl[SLITLET_CNT-2].fit_c_r));
                BRK_IF_ERROR(
                    eris_ifu_vector_subtract(arc_tbl[SLITLET_CNT-1].fit_c_r, arc_tbl[SLITLET_CNT-2].fit_c));
            }
            cut_off_right = CPL_FALSE;
        }

        // take fifth slitlet as master one
        int ms = 4;
        o_l5 = eris_ifu_vector_get_mean(arc_tbl[ms].fit_l);
        o_c5 = eris_ifu_vector_get_mean(arc_tbl[ms].fit_c);
        o_r5 = eris_ifu_vector_get_mean(arc_tbl[ms].fit_r);
        m_c = 64. * (o_c5 - o_l5) / (o_r5 - o_l5);
        if (triple_traces) {
            o_c_l = eris_ifu_vector_get_mean(arc_tbl[ms].fit_c_l);
            o_c_r = eris_ifu_vector_get_mean(arc_tbl[ms].fit_c_r);
            m_c_l = 64. * (o_c_l - o_l5) / (o_r5 - o_l5);
            m_c_r = 64. * (o_c_r - o_l5) / (o_r5 - o_l5);
//            m_t_t = (o_c_r - o_c_l) / (m_c_r - m_c_l);
//            printf("REF: %6.1f %6.1f %6.1f %6.1f %6.1f  width: %.1f  r: %.1f %.1f %.1f %.1f\n",
//                o_l5, o_c_l, o_c5, o_r5, o_c_r, o_r5-o_l5, m_c_l, m_c, m_c_r, m_t_t);
        } else {
//            printf("REF: %.1f %.1f %.1f  width: %.1f  r: %.1f\n",
//                o_l5, o_c5, o_r5, o_r5-o_l5, m_c);
        }

        if (productDepth >= PD_DEBUG) {
            BRK_IF_ERROR(
                cpl_propertylist_update_string(pl, CPL_DFS_PRO_CATG,
                		ERIS_IFU_PRO_DIST_DBG_DIST_FIT_GRID));
            BRK_IF_ERROR(
                cpl_image_save(NULL, ERIS_IFU_PRO_DIST_DBG_DIST_FIT_GRID_FN,
                		CPL_TYPE_FLOAT, pl, CPL_IO_CREATE));
        }

        // estimate mean origin positions of the arc-lines and calculate
        // target positions for the edges, fill the grid, perform polynomial-fit
        for (int i = 0; i < SLITLET_CNT; i++) {
            cpl_msg_debug(cpl_func, "i: %d", i);
            // get number of vertical traces (arcs) to process
            n_arcs = eris_ifu_distortion_get_narcs(i, triple_traces,
                                                   cut_off_left, cut_off_right);

            /*
             * estimate mean origin positions of the arc-lines (left, center, right)
             */
            o_l = eris_ifu_vector_get_mean(arc_tbl[i].fit_l);
            o_c = eris_ifu_vector_get_mean(arc_tbl[i].fit_c);
            o_r = eris_ifu_vector_get_mean(arc_tbl[i].fit_r);

//            ASSURE(o_c_l < o_c, CPL_ERROR_ILLEGAL_OUTPUT, "Slitlet #%d: o_l >= o_c (o_l: %g, o_c: %g)", i+1, o_l, o_c);
//            ASSURE(o_c_l < o_c, CPL_ERROR_ILLEGAL_OUTPUT, "Slitlet #%d: o_c >= o_r (o_c: %g, o_r: %g)", i+1, o_c, o_r);
            if (triple_traces) {
                o_c_l = eris_ifu_vector_get_mean(arc_tbl[i].fit_c_l);
                o_c_r = eris_ifu_vector_get_mean(arc_tbl[i].fit_c_r);
                cpl_msg_debug(cpl_func, "  o_l: %g, o_c_l: %g, o_c: %g, o_c_r: %g, o_r: %g", o_l, o_c_l, o_c, o_c_r, o_r);
                ASSURE(o_c_l < o_c, CPL_ERROR_ILLEGAL_OUTPUT, "Slitlet #%d: o_c_l >= o_c (o_c_l: %g, o_c: %g)", i+1, o_c_l, o_c);
                ASSURE(o_c < o_c_r, CPL_ERROR_ILLEGAL_OUTPUT, "Slitlet #%d: o_c >= o_c_r (o_c: %g, o_c_r: %g)", i+1, o_c, o_c_r);
            } else {
                cpl_msg_debug(cpl_func, "  o_l: %g, o_c: %g, o_r: %g", o_l, o_c, o_r);
            }

            /*
             * from these estimates define target positions for the arc-lines (left, center, right)
             */
            // left+right edge are simple: a slitlet should have 64pix width
            t_o = eris_ifu_distortion_target_left_edge(i);
            t_l = eris_ifu_distortion_target_left_edge(i);
            t_r = eris_ifu_distortion_target_right_edge(i);
            t_l = t_o + m_c - (o_c-o_l) / (o_c5-o_l5) * m_c;
//            t_l = t_o + m_c_r - (o_c_r - o_l) / m_t_t;
            // the center positions are calculated with the sentence of three...
            t_c = (int)((o_c-o_l)*(t_r-t_l)/(o_r-o_l) + t_l + .5);
            t_c = t_o + m_c;
            if (triple_traces) {
                t_c_l = (int)((o_c_l-o_l)*(t_r-t_l)/(o_r-o_l) + t_l + .5);
                t_c_r = (int)((o_c_r-o_l)*(t_r-t_l)/(o_r-o_l) + t_l + .5);
                t_c_l = t_o + m_c_l;
                t_c_r = t_o + m_c_r;
                cpl_msg_debug(cpl_func, "  t_l: %f, t_c_l: %f, t_c: %f, t_c_r: %f, t_r: %f", t_l, t_c_l, t_c, t_c_r, t_r);
            } else {
                cpl_msg_debug(cpl_func, "  t_l: %f, t_c: %f, t_r: %f", t_l, t_c, t_r);
            }
            if (i == 0) {
                t_r -= 2.7;
            } else if (i == 1){
                t_r -= 1.5;
            }
            double tx = t_o;
//            printf(" %2.2d: %6.1f %6.1f %6.1f %6.1f %6.1f  width: %.1f  t_o: %6.1f  t %6.1f %6.1f %6.1f %6.1f %6.1f  tg %6.1f %6.1f %6.1f %6.1f %6.1f\n",
//                i, o_l, o_c_l, o_c, o_r, o_c_r, o_r-o_l, t_o, t_l, t_c_l, t_c, t_c_r, t_r, t_l-tx, t_c_l-tx, t_c-tx, t_c_r-tx, t_r-tx);
//            cpl_msg_debug(cpl_func, "i: %02d, o_l:    %7.2f, o_c_l:    %7.2f,"
//                                    "o_c:   %7.2f, o_c_r:   %7.2f, o_r:  %7.2f",
//                                    i, o_l, o_c_l, o_c, o_c_r, o_r);
//            cpl_msg_debug(cpl_func, "       t_l:    %04d   , t_c_l:    %04d   ,"
//                                    " t_c:   %04d   , t_c_r:   %04d   , t_r:  %04d",
//                          t_l, t_c_l, t_c, t_c_r, t_r);
            // create grid to fit to (contains target-coordinates)
            BRK_IF_NULL(
                grid = cpl_bivector_new(n_arcs * n_size));

            // create vector with values to fit (contains origin-coordinates)
            BRK_IF_NULL(
                val_to_fit = cpl_vector_new(n_arcs * n_size));

            arc_cnt = 0;
            // fill in edges and subtract always in a manner that destination_x(target) is 0 for left edge
            // fill in left edge
            BRK_IF_ERROR(
                eris_ifu_dist_calc_distortion_fillgrid(grid, arc_tbl[i].fit_l, val_to_fit,
                                                       t_l-tx, n_size, arc_cnt++));
            if (triple_traces) {
                // fill in center_l
                BRK_IF_ERROR(
                    eris_ifu_dist_calc_distortion_fillgrid(grid, arc_tbl[i].fit_c_l, val_to_fit,
                                                           t_c_l-tx, n_size, arc_cnt++));
            }
            // fill in center
//cpl_msg_debug(cpl_func, ">>> %d", i);
//if (i==31) {
//    int a=0;
//}
            BRK_IF_ERROR(
                eris_ifu_dist_calc_distortion_fillgrid(grid, arc_tbl[i].fit_c, val_to_fit,
                                                       t_c-tx, n_size, arc_cnt++));
            if (triple_traces) {
                // fill in center_l
                BRK_IF_ERROR(
                    eris_ifu_dist_calc_distortion_fillgrid(grid, arc_tbl[i].fit_c_r, val_to_fit,
                                                           t_c_r-tx, n_size, arc_cnt++));
            }
            // fill in right edge
            BRK_IF_ERROR(
                eris_ifu_dist_calc_distortion_fillgrid(grid, arc_tbl[i].fit_r, val_to_fit,
                                                       t_r-tx, n_size, arc_cnt++));

            // save l_min and r_max
            l_min = floor(eris_ifu_vector_get_min(arc_tbl[i].fit_l, &pos));
            BRK_IF_ERROR(
                cpl_table_set_double(*minmax_borders, ERIS_IFU_POLY_EDGE_L, i,
                                     l_min));
            BRK_IF_ERROR(
                cpl_table_set_double(*minmax_borders, ERIS_IFU_POLY_EDGE_R, i,
                                     ceil(eris_ifu_vector_get_max(arc_tbl[i].fit_r, &pos))));

            // fill in edges and subtract always in a manner that destination_x(measured) is 0.0 for left edge at beginning
            BRK_IF_ERROR(
                cpl_vector_subtract_scalar(val_to_fit, l_min));

            if (productDepth >= PD_DEBUG) {
                char *extName;
                extName = cpl_sprintf("GRIDX_%2.2d", i);
                BRK_IF_ERROR(
                    cpl_propertylist_update_string(pl, "EXTNAME", extName));
                BRK_IF_ERROR(
                    cpl_vector_save(cpl_bivector_get_x(grid),
                    		ERIS_IFU_PRO_DIST_DBG_DIST_FIT_GRID_FN,
                        CPL_TYPE_FLOAT, pl, CPL_IO_EXTEND));
                cpl_free(extName);
                extName = cpl_sprintf("GRIDY_%2.2d", i);
                BRK_IF_ERROR(
                    cpl_propertylist_update_string(pl, "EXTNAME", extName));
                BRK_IF_ERROR(
                    cpl_vector_save(cpl_bivector_get_y(grid),
                    		ERIS_IFU_PRO_DIST_DBG_DIST_FIT_GRID_FN,
                        CPL_TYPE_FLOAT, pl, CPL_IO_EXTEND));
                 cpl_free(extName);
                extName = cpl_sprintf("VAL2FIT_%2.2d", i);
                BRK_IF_ERROR(
                    cpl_propertylist_update_string(pl, "EXTNAME", extName));
                BRK_IF_ERROR(
                    cpl_vector_save(val_to_fit,
                    		ERIS_IFU_PRO_DIST_DBG_DIST_FIT_GRID_FN,
                        CPL_TYPE_FLOAT, pl, CPL_IO_EXTEND));
                cpl_free(extName);
           }
//if (i==16) {
//    int aa=0;
//}
            /* Perform the fit */
            BRK_IF_NULL(
                poly_u[i] = eris_ifu_dist_poly_fit_2d_create(grid, val_to_fit, NULL));

            eris_ifu_free_bivector(&grid);
            eris_ifu_free_vector(&val_to_fit);
        } // end: i = SLITLET_CNT

        // save values of all fitted edges and some stats
        if (productDepth & 1) {
            // for the ease of stats: create the y-vector with the used values
            BRK_IF_NULL(
                y = cpl_vector_new(n_size));
            BRK_IF_NULL(
                py = cpl_vector_get_data(y));
            for (int j = 0; j < n_size; j++) {
                py[j] = eris_ifu_distortion_calc_y(n_size, j);
            }
            char* keyname;
            char* keycomm;
            for (int i = 0; i < SLITLET_CNT; i++) {

            	tbl = cpl_table_new(n_size);
            	if (arc_tbl[i].fit_l != NULL) {

            		cpl_table_new_column(tbl, "l", CPL_TYPE_DOUBLE);
                    tmp_vec = eris_ifu_vector_get_data(arc_tbl[i].fit_l);
            		cpl_table_copy_data_double(tbl, "l", cpl_vector_get_data_const(tmp_vec));
            		eris_ifu_free_vector(&tmp_vec);
            	}
            	if (arc_tbl[i].fit_c_l != NULL) {
            		cpl_table_new_column(tbl, "c_l", CPL_TYPE_DOUBLE);
                    tmp_vec = eris_ifu_vector_get_data(arc_tbl[i].fit_c_l);
            		cpl_table_copy_data_double(tbl, "c_l", cpl_vector_get_data_const(tmp_vec));
            		eris_ifu_free_vector(&tmp_vec);
            	}
            	if (arc_tbl[i].fit_c != NULL) {
            		std = eris_ifu_vector_get_stdev(arc_tbl[i].fit_c);
            		keyname = cpl_sprintf("ESO QC SLITLET%2.2d C STDEV",i);
            		keycomm = cpl_sprintf("[pix] center slitlet %2.2d",i);
            		if (!eris_ifu_is_nan_or_inf(std)) {
            			cpl_propertylist_append_double((*qc)[i], keyname, std);
            		} else {
            			cpl_propertylist_append_string((*qc)[i], keyname, "NaN");
            		}
            		cpl_propertylist_set_comment((*qc)[i], keyname,keycomm);
            		cpl_free(keyname);
            		cpl_free(keycomm);
            		cpl_table_new_column(tbl, "c", CPL_TYPE_DOUBLE);
                    tmp_vec = eris_ifu_vector_get_data(arc_tbl[i].fit_c);
            		cpl_table_copy_data_double(tbl, "c", cpl_vector_get_data_const(tmp_vec));
            		eris_ifu_free_vector(&tmp_vec);
            	}
            	if (arc_tbl[i].fit_c_r != NULL) {
            		cpl_table_new_column(tbl, "c_r", CPL_TYPE_DOUBLE);
                    tmp_vec = eris_ifu_vector_get_data(arc_tbl[i].fit_c_r);
            		cpl_table_copy_data_double(tbl, "c_r", cpl_vector_get_data_const(tmp_vec));
            		eris_ifu_free_vector(&tmp_vec);
            	}
            	if (arc_tbl[i].fit_r != NULL) {
            		cpl_table_new_column(tbl, "r", CPL_TYPE_DOUBLE);
                    tmp_vec = eris_ifu_vector_get_data(arc_tbl[i].fit_r);
            		cpl_table_copy_data_double(tbl, "r", cpl_vector_get_data_const(tmp_vec));
            		eris_ifu_free_vector(&tmp_vec);
            	}

            	// calc stddev of differences of distances between single traces
            	if ((arc_tbl[i].fit_l != NULL) && (arc_tbl[i].fit_r != NULL)) {
            		t = eris_ifu_vector_duplicate(arc_tbl[i].fit_r);
            		eris_ifu_vector_subtract(t, arc_tbl[i].fit_l);
            		std = eris_ifu_vector_get_stdev(t);
            		keyname = cpl_sprintf("ESO QC SLITLET%2.2d R-L STDEV",i);
            		keycomm = cpl_sprintf("[pix] right-left distance slitlet %2.2d",i);
            		if (!eris_ifu_is_nan_or_inf(std)) {
            			cpl_propertylist_append_double((*qc)[i], keyname, std);
            		} else {
            			cpl_propertylist_append_string((*qc)[i], keyname, "NaN");
            		}
            		cpl_propertylist_set_comment((*qc)[i], keyname, keycomm);
            		cpl_free(keyname);
            		cpl_free(keycomm);

            		median = eris_ifu_vector_get_median(t, ERIS_IFU_ARITHMETIC);
            		keyname = cpl_sprintf("ESO QC SLITLET%2.2d R-L MEDIAN",i);
            		keycomm = cpl_sprintf("[pix] right-left distance median slitlet %2.2d",i);
            		if (!eris_ifu_is_nan_or_inf(median)) {
            			cpl_propertylist_append_double((*qc)[i], keyname, median);
            		} else {
            			cpl_propertylist_append_string((*qc)[i], "MEDIAN R-L", "NaN");
            		}
            		cpl_propertylist_set_comment((*qc)[i], keyname, keycomm);
            		cpl_free(keyname);
            		cpl_free(keycomm);

            		// for the ease of stats: save the slit-width & y
            		cpl_table_new_column(tbl, "slit_width", CPL_TYPE_DOUBLE);
                    tmp_vec = eris_ifu_vector_get_data(t);
            		cpl_table_copy_data_double(tbl, "slit_width", cpl_vector_get_data_const(tmp_vec));
            		cpl_table_new_column(tbl, "y", CPL_TYPE_DOUBLE);
            		cpl_table_copy_data_double(tbl, "y", cpl_vector_get_data_const(y));
            		eris_ifu_free_vector(&tmp_vec);
            		eris_ifu_free_ifu_vector(&t);
            	}
            	if ((arc_tbl[i].fit_l != NULL) && (arc_tbl[i].fit_c != NULL)) {
            		t = eris_ifu_vector_duplicate(arc_tbl[i].fit_c);
            		eris_ifu_vector_subtract(t, arc_tbl[i].fit_l);
            		std = eris_ifu_vector_get_stdev(t);
            		keyname = cpl_sprintf("ESO QC SLITLET%2.2d C-L STDEV",i);
            		keycomm = cpl_sprintf("[pix] center-left distance stdev slitlet %2.2d",i);
            		if (!eris_ifu_is_nan_or_inf(std)) {
            			cpl_propertylist_append_double((*qc)[i], keyname, std);
            		} else {
            			cpl_propertylist_append_string((*qc)[i], keyname, "NaN");
            		}
            		cpl_propertylist_set_comment((*qc)[i], keyname, keycomm);
            		cpl_free(keyname);
            		cpl_free(keycomm);
            		median = eris_ifu_vector_get_median(t, ERIS_IFU_ARITHMETIC);
            		keyname = cpl_sprintf("ESO QC SLITLET%2.2d C-L MEDIAN",i);
            		keycomm = cpl_sprintf("[pix] center-left distance median slitlet %2.2d",i);
            		if (!eris_ifu_is_nan_or_inf(median)) {
            			cpl_propertylist_append_double((*qc)[i], keyname, median);
            		} else {
            			cpl_propertylist_append_string((*qc)[i], keyname, "NaN");
            		}
            		cpl_propertylist_set_comment((*qc)[i], keyname, keycomm);
            		cpl_free(keyname);
            		cpl_free(keycomm);
            		eris_ifu_free_ifu_vector(&t);
            	}
            	if ((arc_tbl[i].fit_c != NULL) && (arc_tbl[i].fit_r != NULL)) {
            		t = eris_ifu_vector_duplicate(arc_tbl[i].fit_r);
            		eris_ifu_vector_subtract(t, arc_tbl[i].fit_c);
            		std = eris_ifu_vector_get_stdev(t);
            		keyname = cpl_sprintf("ESO QC SLITLET%2.2d R-C STDEV",i);
            		keycomm = cpl_sprintf("[pix] right-center distance stdev slitlet %2.2d",i);
            		if (!eris_ifu_is_nan_or_inf(std)) {
            			cpl_propertylist_append_double((*qc)[i], keyname, std);
            		} else {
            			cpl_propertylist_append_string((*qc)[i], keyname, "NaN");
            		}
            		cpl_propertylist_set_comment((*qc)[i], keyname, keycomm);
            		cpl_free(keyname);
            		cpl_free(keycomm);
            		median = eris_ifu_vector_get_median(t, ERIS_IFU_ARITHMETIC);
            		keyname = cpl_sprintf("ESO QC SLITLET%2.2d R-C MEDIAN",i);
            		keycomm = cpl_sprintf("[pix] right-center distance median slitlet %2.2d",i);
            		if (!eris_ifu_is_nan_or_inf(median)) {
            			cpl_propertylist_append_double((*qc)[i], keyname, median);
            		} else {
            			cpl_propertylist_append_string((*qc)[i], keyname, "NaN");
            		}
            		cpl_propertylist_set_comment((*qc)[i], keyname, keycomm);
            		cpl_free(keyname);
            		cpl_free(keycomm);
            		eris_ifu_free_ifu_vector(&t);
            	}
            	if ((arc_tbl[i].fit_c != NULL) && (arc_tbl[i].fit_c_l != NULL)) {
            		t = eris_ifu_vector_duplicate(arc_tbl[i].fit_c);
            		eris_ifu_vector_subtract(t, arc_tbl[i].fit_c_l);
            		std = eris_ifu_vector_get_stdev(t);
            		keyname = cpl_sprintf("ESO QC SLITLET%2.2d C-CL STDEV",i);
            		keycomm = cpl_sprintf("[pix] center-centerleft distance stdev slitlet %2.2d",i);
            		if (!eris_ifu_is_nan_or_inf(std)) {
            			cpl_propertylist_append_double((*qc)[i], keyname, std);
            		} else {
            			cpl_propertylist_append_string((*qc)[i], keyname, "NaN");
            		}
            		cpl_propertylist_set_comment((*qc)[i], keyname, keycomm);
            		cpl_free(keyname);
            		cpl_free(keycomm);
            		median = eris_ifu_vector_get_median(t, ERIS_IFU_ARITHMETIC);
            		keyname = cpl_sprintf("ESO QC SLITLET%2.2d C-CL MEDIAN",i);
            		keycomm = cpl_sprintf("[pix] center-centerleft distance median slitlet %2.2d",i);
            		if (!eris_ifu_is_nan_or_inf(median)) {
            			cpl_propertylist_append_double((*qc)[i], keyname, median);
            		} else {
            			cpl_propertylist_append_string((*qc)[i], keyname, "NaN");
            		}
            		cpl_propertylist_set_comment((*qc)[i], keyname, keycomm);
            		cpl_free(keyname);
            		cpl_free(keycomm);
            		eris_ifu_free_ifu_vector(&t);
            	}
            	if ((arc_tbl[i].fit_c_r != NULL) && (arc_tbl[i].fit_c != NULL)) {
            		t = eris_ifu_vector_duplicate(arc_tbl[i].fit_c_r);
            		eris_ifu_vector_subtract(t, arc_tbl[i].fit_c);
            		std = eris_ifu_vector_get_stdev(t);
            		keyname = cpl_sprintf("ESO QC SLITLET%2.2d CR-C STDEV",i);
            		keycomm = cpl_sprintf("[pix] centerright-center distance stdev slitlet %2.2d",i);
            		if (!eris_ifu_is_nan_or_inf(std)) {
            			cpl_propertylist_append_double((*qc)[i], keyname, std);
            		} else {
            			cpl_propertylist_append_string((*qc)[i], keyname, "NaN");
            		}
            		cpl_propertylist_set_comment((*qc)[i], keyname, keycomm);
            		cpl_free(keyname);
            		cpl_free(keycomm);
            		median = eris_ifu_vector_get_median(t, ERIS_IFU_ARITHMETIC);
            		keyname = cpl_sprintf("ESO QC SLITLET%2.2d CR-C MEDIAN",i);
            		keycomm = cpl_sprintf("[pix] centerright-center distance median slitlet %2.2d",i);
            		if (!eris_ifu_is_nan_or_inf(median)) {
            			cpl_propertylist_append_double((*qc)[i], keyname, median);
            		} else {
            			cpl_propertylist_append_string((*qc)[i], keyname, "NaN");
            		}
            		cpl_propertylist_set_comment((*qc)[i], keyname, keycomm);
            		cpl_free(keyname);
            		cpl_free(keycomm);
            		eris_ifu_free_ifu_vector(&t);
            	}
                BRK_IF_ERROR(
                    cpl_table_save(tbl, NULL, (*qc)[i], ERIS_IFU_PRO_DIST_QC_FIT_FN, CPL_IO_EXTEND));

                eris_ifu_free_table(&tbl);
            } // end: i = SLITLET_CNT
        } // end:  if (productDepth & 1)
    }
    CATCH
    {
        CATCH_MSGS();
        for (int j = 0; j < SLITLET_CNT; j++) {
            cpl_polynomial_delete(poly_u[j]); poly_u[j] = NULL;
        }
        cpl_free(poly_u); poly_u = NULL;

        eris_ifu_free_table(minmax_borders);

        if (*qc != NULL && **qc != NULL) {
            for (int i = 0; i < SLITLET_CNT; i++) {
                eris_ifu_free_propertylist(&(*qc)[i]);
            }
        }
        if (*qc != NULL) {
            cpl_free(**qc); **qc = NULL;
        }
    }
    eris_ifu_free_table(&dbg_polynomials);
    eris_ifu_free_bivector(&grid);
    eris_ifu_free_vector(&val_to_fit);
    eris_ifu_free_vector(&y);
    eris_ifu_free_ifu_vector(&y_wave);
    for (int i = 0; i < SLITLET_CNT; i++) {
        eris_ifu_free_ifu_vector(&arc_tbl[i].fit_l);
        eris_ifu_free_ifu_vector(&arc_tbl[i].fit_c);
        eris_ifu_free_ifu_vector(&arc_tbl[i].fit_c_l);
        eris_ifu_free_ifu_vector(&arc_tbl[i].fit_c_r);
        eris_ifu_free_ifu_vector(&arc_tbl[i].fit_r);
    }
    cpl_free(arc_tbl); arc_tbl = NULL;
    eris_check_error_code("eris_ifu_dist_calc_distortion");
    return poly_u;
}

/**
 * @brief Calculate distortion polynomials (alternative method)
 * @param slit_edges      Array of tables with detected slit edge positions
 * @param centers         Array of tables with fitted slitlet centers
 * @param productDepth    Debug output level
 * @param cut_off_left    Indicates if leftmost slitlet is cut off
 * @param cut_off_right   Indicates if rightmost slitlet is cut off
 * @return Array of SLITLET_CNT distortion polynomials, or NULL on error
 *
 * Alternative distortion calculation with different polynomial fitting approach.
 * Uses same input data but different target coordinate calculation.
 *
 * @note Polynomial degree is 3 in both X and Y
 * @note Each polynomial must be freed with cpl_polynomial_delete()
 * @note Array must be freed with cpl_free()
 */
cpl_polynomial** eris_ifu_dist_calc_distortion_full(cpl_table   **slit_edges,
                                                    cpl_table   **centers,
                                                    int         productDepth,
                                                    cpl_boolean cut_off_left,
                                                    cpl_boolean cut_off_right)
{
    int                 n_arcs          = 0,
                        n_calib_wave    = 0,
                        n_calib_cen     = 0,
                        n_size          = 100,
                        fit_order       = 3,
                        t_l             = 0,
                        t_r             = 0,
                        t_c             = 0,
                        t_c_l           = 0,
                        t_c_r           = 0,
                        arc_cnt         = 0;
    double              o_l             = 0.,
                        o_c             = 0.,
                        o_r             = 0.,
                        o_c_l           = 0.,
                        o_c_r           = 0.,
                        std             = 0.;
    const double        *pwave_y        = NULL,
                        *pcenter_y      = NULL;
    cpl_polynomial      **poly_u        = NULL;
    cpl_bivector        *grid           = NULL;
    cpl_vector          *val_to_fit     = NULL,
                        *tt             = NULL;
    eris_ifu_vector     *y_wave         = NULL,
                        *y_cen          = NULL,
                        *t              = NULL;
    cpl_boolean         triple_traces   = FALSE;
    cpl_table           *tbl            = NULL,
                        *dbg_polynomials = NULL;
    cpl_propertylist    **pl            = NULL;
    struct arcstruct    *arc_tbl        = NULL;

    cpl_ensure(slit_edges, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(centers, CPL_ERROR_NULL_INPUT, NULL);

    TRY
    {
        BRK_IF_NULL( dbg_polynomials = cpl_table_new(SLITLET_CNT));
        cpl_table_new_column(dbg_polynomials, ERIS_IFU_DIST_DBG_SLITLET, CPL_TYPE_INT);
        cpl_table_new_column_array(dbg_polynomials, ERIS_IFU_DIST_DBG_EDGE_LEFT, CPL_TYPE_DOUBLE, fit_order+1);
        cpl_table_new_column_array(dbg_polynomials, ERIS_IFU_DIST_DBG_EDGE_RIGHT, CPL_TYPE_DOUBLE, fit_order+1);
        cpl_table_new_column_array(dbg_polynomials, ERIS_IFU_DIST_DBG_CENTER_LEFT, CPL_TYPE_DOUBLE, fit_order+1);
        cpl_table_new_column_array(dbg_polynomials, ERIS_IFU_DIST_DBG_CENTER, CPL_TYPE_DOUBLE, fit_order+1);
        cpl_table_new_column_array(dbg_polynomials, ERIS_IFU_DIST_DBG_CENTER_RIGHT, CPL_TYPE_DOUBLE, fit_order+1);
        CHECK_ERROR_STATE();

       BRK_IF_NULL(
            poly_u = cpl_calloc(SLITLET_CNT, sizeof(cpl_polynomial*)));

        if (cpl_table_get_ncol(centers[5]) > 2) {
            triple_traces = TRUE;
        }

        BRK_IF_ERROR(
            cpl_propertylist_save(NULL, ERIS_IFU_PRO_DIST_QC_FIT_FN, CPL_IO_CREATE));

        BRK_IF_NULL(
            arc_tbl = cpl_calloc(SLITLET_CNT, sizeof(struct arcstruct)));

        // calculate all slitlets taht are not cut off
        for (int i = 0; i < SLITLET_CNT; i++) {
            //cpl_msg_debug(cpl_func, "i: %d\n", i);
            // get number of vertical traces (arcs) to process
            n_arcs = eris_ifu_distortion_get_narcs(i, triple_traces,
                                                   cut_off_left, cut_off_right);
            // get number of detected wave_calib-lines
            n_calib_wave = (int) cpl_table_get_nrow(slit_edges[i]);
            // get number of centerpoints
            n_calib_cen = (int) cpl_table_get_nrow(centers[i]);

            // get y-coordinates
            BRK_IF_NULL(
                pwave_y = cpl_table_get_data_double_const(
                                             slit_edges[i], "y_pos"));
            BRK_IF_NULL(
                pcenter_y = cpl_table_get_data_double_const(
                                             centers[i], "y"));
            BRK_IF_NULL(
                y_wave = eris_ifu_vector_new_wrap(n_calib_wave, pwave_y));
            BRK_IF_NULL(
                y_cen = eris_ifu_vector_new_wrap(n_calib_cen, pcenter_y));

            // fit all edges (left, right, centers)
            if (!((i == 0) && cut_off_left)) {
                // process only if it is not the 1st slitlet and cutoff
                BRK_IF_NULL(
                    arc_tbl[i].fit_l = eris_ifu_dist_calc_distortion_fitedge(
                                            slit_edges[i], "edge_left", y_wave,
                                            n_calib_wave, n_size, fit_order,
                                            i, dbg_polynomials));
            }
            if (!((i == SLITLET_CNT-1) && cut_off_right)) {
                // process only if it is not the last slitlet and cutoff
                BRK_IF_NULL(
                    arc_tbl[i].fit_r = eris_ifu_dist_calc_distortion_fitedge(
                                            slit_edges[i], "edge_right", y_wave,
                                            n_calib_wave, n_size, fit_order,
                                            i, dbg_polynomials));
            }
            BRK_IF_NULL(
                arc_tbl[i].fit_c = eris_ifu_dist_calc_distortion_fitedge(
                                            centers[i], "x", y_cen,
                                            n_calib_cen, n_size, fit_order,
                                            i, dbg_polynomials));
            if (triple_traces) {
                if (!((i == 0) && cut_off_left)) {
                    // process if it is not the 1st slitlet and not cutoff
                    BRK_IF_NULL(
                        arc_tbl[i].fit_c_l = eris_ifu_dist_calc_distortion_fitedge(
                                                centers[i], "x_l", y_cen,
                                                n_calib_cen, n_size, fit_order,
                                                i, dbg_polynomials));
                } else {
                    // if it is the last slitlet and cutoff, then
                    // check first if, and how many, values are nan
                    // If there are too many NaN's do nothing and copy the shape of c to c_l
                    if (cpl_table_count_invalid(centers[i], "x_l") <= 0.1 * n_calib_cen) {
                        // calculate fitted edge only if less tahn 10% of the values are NaN
                        BRK_IF_NULL(
                            arc_tbl[i].fit_c_l = eris_ifu_dist_calc_distortion_fitedge(
                                                    centers[i], "x_l", y_cen,
                                                    n_calib_cen, n_size, fit_order,
                                                    i, dbg_polynomials));
                    }
                }

                if (!((i == SLITLET_CNT-1) && cut_off_right)) {
                    // process if it is not the last slitlet and not cutoff
                    BRK_IF_NULL(
                        arc_tbl[i].fit_c_r = eris_ifu_dist_calc_distortion_fitedge(
                                                centers[i], "x_r", y_cen,
                                                n_calib_cen, n_size, fit_order,
                                                i, dbg_polynomials));
                } else {
                    // if it is the last slitlet and cutoff, then
                    // check first if, and how many, values are nan
                    // If there are too many NaN's do nothing and copy the shape of c to c_r
                    if (cpl_table_count_invalid(centers[i], "x_r") <= 0.1 * n_calib_cen) {
                        // calculate fitted edge only if less than 10% of the values are NaN
                        BRK_IF_NULL(
                            arc_tbl[i].fit_c_r = eris_ifu_dist_calc_distortion_fitedge(
                                                    centers[i], "x_r", y_cen,
                                                    n_calib_cen, n_size, fit_order,
                                                    i, dbg_polynomials));
                    }
                }
            } // end: if (triple_traces)
            eris_ifu_free_ifu_vector(&y_wave);
            eris_ifu_free_ifu_vector(&y_cen);
        } // end: i = SLITLET_CNT

        BRK_IF_NULL(
            pl = cpl_calloc(SLITLET_CNT, sizeof(cpl_propertylist*)));
        for (int i = 0; i < SLITLET_CNT; i++) {
            BRK_IF_NULL(
                pl[i] = cpl_propertylist_new());
        }

        // estimate positions of cut off edges with neighbouring slitlet width
        if (cut_off_left) {
            BRK_IF_ERROR(
                cpl_propertylist_append_string(pl[0],
                                                "EDGE L", "artificial"));
            if (arc_tbl[0].fit_l == NULL) {
                /* The leftmost edge of slitlet #0 is cut off. Therefore subtract the
                 * width of slitlet #1 from the right edge of slitlet #0
                 */
                BRK_IF_NULL(
                    arc_tbl[0].fit_l = eris_ifu_vector_duplicate(arc_tbl[0].fit_r));
                BRK_IF_ERROR(
                    eris_ifu_vector_subtract(arc_tbl[0].fit_l, arc_tbl[1].fit_r));
                BRK_IF_ERROR(
                    eris_ifu_vector_add(arc_tbl[0].fit_l, arc_tbl[1].fit_l));
            }
            if (triple_traces && (arc_tbl[SLITLET_CNT-1].fit_c_l == NULL)) {
                /* The c_l-edge of slitlet #0 is cut off. Therefore subtract the
                 * width of c-c_l of slitlet #31 to the left edge of slitlet #32
                 */
                BRK_IF_NULL(
                    arc_tbl[0].fit_c_l = eris_ifu_vector_duplicate(arc_tbl[0].fit_c));
                BRK_IF_ERROR(
                    eris_ifu_vector_subtract(arc_tbl[0].fit_c_l, arc_tbl[1].fit_c));
                BRK_IF_ERROR(
                    eris_ifu_vector_add(arc_tbl[0].fit_c_l, arc_tbl[1].fit_c_l));
            }
            cut_off_left = CPL_FALSE;
        }

        if (cut_off_right) {
            BRK_IF_ERROR(
                cpl_propertylist_append_string(pl[SLITLET_CNT-1],
                                                "EDGE R", "artificial"));
            if (arc_tbl[SLITLET_CNT-1].fit_r == NULL) {
                /* The rightmost edge of slitlet #32 is cut off. Therefore add the
                 * width of slitlet #31 to the left edge of slitlet #32
                 */
                BRK_IF_NULL(
                    arc_tbl[SLITLET_CNT-1].fit_r = eris_ifu_vector_duplicate(arc_tbl[SLITLET_CNT-1].fit_l));
                BRK_IF_ERROR(
                    eris_ifu_vector_add(arc_tbl[SLITLET_CNT-1].fit_r, arc_tbl[SLITLET_CNT-2].fit_r));
                BRK_IF_ERROR(
                    eris_ifu_vector_subtract(arc_tbl[SLITLET_CNT-1].fit_r, arc_tbl[SLITLET_CNT-2].fit_l));
            }
            if (triple_traces && (arc_tbl[SLITLET_CNT-1].fit_c_r == NULL)) {
                /* The c_r-edge of slitlet #32 is cut off. Therefore add the
                 * width of c_r-c of slitlet #31 to the left center edge of slitlet #32
                 */
                BRK_IF_NULL(
                    arc_tbl[SLITLET_CNT-1].fit_c_r = eris_ifu_vector_duplicate(arc_tbl[SLITLET_CNT-1].fit_c));
                BRK_IF_ERROR(
                    eris_ifu_vector_add(arc_tbl[SLITLET_CNT-1].fit_c_r, arc_tbl[SLITLET_CNT-2].fit_c_r));
                BRK_IF_ERROR(
                    eris_ifu_vector_subtract(arc_tbl[SLITLET_CNT-1].fit_c_r, arc_tbl[SLITLET_CNT-2].fit_c));
            }
            cut_off_right = CPL_FALSE;
        }

        const char *fn = "eris_ifu_distortion_dbg_calcdistor_grid_xy_val2fit.fits";
        if (productDepth & 4) {
            eris_ifu_save_vector_dbg(NULL, fn, CPL_IO_CREATE, NULL);
        }
        // estimate mean origin positions of the arc-lines and calculate
        // target positions for the edges, fill the grid, perform polynomial-fit
        for (int i = 0; i < SLITLET_CNT; i++) {
            // get number of vertical traces (arcs) to process
            n_arcs = eris_ifu_distortion_get_narcs(i, triple_traces,
                                                   cut_off_left, cut_off_right);

            /*
             * estimate mean origin positions of the arc-lines (left, center, right)
             */
            o_l = eris_ifu_vector_get_mean(arc_tbl[i].fit_l);
            o_c = eris_ifu_vector_get_mean(arc_tbl[i].fit_c);
            o_r = eris_ifu_vector_get_mean(arc_tbl[i].fit_r);
            if (triple_traces) {
                o_c_l = eris_ifu_vector_get_mean(arc_tbl[i].fit_c_l);
                o_c_r = eris_ifu_vector_get_mean(arc_tbl[i].fit_c_r);
            }

            /*
             * from these estimates define target positions for the arc-lines (left, center, right)
             */
            // left+right edge are simple: a slitlet should have 64pix width
            t_l = eris_ifu_distortion_target_left_edge(i);
            t_r = eris_ifu_distortion_target_right_edge(i);

            // the center positions are calculated with the sentence of three...
            t_c = (int)((o_c-o_l)*(t_r-t_l)/(o_r-o_l) + t_l + .5);
            if (triple_traces) {
                t_c_l = (int)((o_c_l-o_l)*(t_r-t_l)/(o_r-o_l) + t_l + .5);
                t_c_r = (int)((o_c_r-o_l)*(t_r-t_l)/(o_r-o_l) + t_l + .5);
            }

//            cpl_msg_debug(cpl_func, "i: %02d, o_l:    %7.2f, o_c_l:    %7.2f,"
//                                    "o_c:   %7.2f, o_c_r:   %7.2f, o_r:  %7.2f",
//                                    i, o_l, o_c_l, o_c, o_c_r, o_r);
//            cpl_msg_debug(cpl_func, "       t_l:    %04d   , t_c_l:    %04d   ,"
//                                    " t_c:   %04d   , t_c_r:   %04d   , t_r:  %04d",
//                          t_l, t_c_l, t_c, t_c_r, t_r);
            // create grid to fit to (contains target-coordinates)
            BRK_IF_NULL(
                grid = cpl_bivector_new(n_arcs * n_size));

            // create vector with values to fit (contains origin-coordinates)
            BRK_IF_NULL(
                val_to_fit = cpl_vector_new(n_arcs * n_size));

            arc_cnt = 0;
            // fill in left edge
            BRK_IF_ERROR(
                eris_ifu_dist_calc_distortion_fillgrid(grid, arc_tbl[i].fit_l, val_to_fit,
                                                       t_l, n_size, arc_cnt++));
            if (triple_traces) {
                // fill in center_l
                BRK_IF_ERROR(
                    eris_ifu_dist_calc_distortion_fillgrid(grid, arc_tbl[i].fit_c_l, val_to_fit,
                                                           t_c_l, n_size, arc_cnt++));
            }
            // fill in center
            BRK_IF_ERROR(
                eris_ifu_dist_calc_distortion_fillgrid(grid, arc_tbl[i].fit_c, val_to_fit,
                                                       t_c, n_size, arc_cnt++));
            if (triple_traces) {
                // fill in center_l
                BRK_IF_ERROR(
                    eris_ifu_dist_calc_distortion_fillgrid(grid, arc_tbl[i].fit_c_r, val_to_fit,
                                                           t_c_r, n_size, arc_cnt++));
            }
            // fill in right edge
            BRK_IF_ERROR(
                eris_ifu_dist_calc_distortion_fillgrid(grid, arc_tbl[i].fit_r, val_to_fit,
                                                       t_r, n_size, arc_cnt++));

            if (productDepth & 4) {
//                cpl_propertylist *pl = NULL;
//                BRK_IF_NULL(
//                    pl = cpl_propertylist_new());
                char *extname = NULL;

                extname = cpl_sprintf("xVector Sx_%02d", i+1);
                cpl_propertylist_update_string(pl[i], "EXTNAME", extname);
                eris_ifu_free_string(&extname);
                eris_ifu_save_vector_dbg(cpl_bivector_get_x(grid), fn, CPL_IO_EXTEND, pl[i]);

                extname = cpl_sprintf("yVector Sx_%02d", i+1);
                cpl_propertylist_update_string(pl[i], "EXTNAME", extname);
                eris_ifu_free_string(&extname);
                eris_ifu_save_vector_dbg(cpl_bivector_get_y(grid), fn, CPL_IO_EXTEND, pl[i]);

                extname = cpl_sprintf("val2fit Sx_%02d", i+1);
                cpl_propertylist_update_string(pl[i], "EXTNAME", extname);
                eris_ifu_free_string(&extname);
                eris_ifu_save_vector_dbg(val_to_fit, fn, CPL_IO_EXTEND, pl[i]);

//                eris_ifu_free_propertylist(&pl);
            }

            /* Apply the fitting */
            BRK_IF_NULL(
                poly_u[i] = eris_ifu_dist_poly_fit_2d_create(grid, val_to_fit, NULL));

            eris_ifu_free_bivector(&grid);
            eris_ifu_free_vector(&val_to_fit);
        } // end: i = SLITLET_CNT

        // save values of all fitted edges and some stats
        if (productDepth & 2) {
            for (int i = 0; i < SLITLET_CNT; i++) {
                BRK_IF_NULL(
                    tbl = cpl_table_new(n_size));
                if (arc_tbl[i].fit_l != NULL) {
                    BRK_IF_ERROR(
                        cpl_table_new_column(tbl, "l", CPL_TYPE_DOUBLE));
                    BRK_IF_NULL(
                        tt = eris_ifu_vector_get_data(arc_tbl[i].fit_l));
                    BRK_IF_ERROR(
                        cpl_table_copy_data_double(tbl, "l", cpl_vector_get_data_const(tt)));
                    eris_ifu_free_vector(&tt);
                }
                if (arc_tbl[i].fit_c_l != NULL) {
                    BRK_IF_ERROR(
                        cpl_table_new_column(tbl, "c_l", CPL_TYPE_DOUBLE));
                    BRK_IF_NULL(
                        tt = eris_ifu_vector_get_data(arc_tbl[i].fit_c_l));
                    BRK_IF_ERROR(
                        cpl_table_copy_data_double(tbl, "c_l", cpl_vector_get_data_const(tt)));
                    eris_ifu_free_vector(&tt);
                }
                if (arc_tbl[i].fit_c != NULL) {
                    BRK_IF_ERROR(
                        cpl_table_new_column(tbl, "c", CPL_TYPE_DOUBLE));
                    BRK_IF_NULL(
                        tt = eris_ifu_vector_get_data(arc_tbl[i].fit_c));
                    BRK_IF_ERROR(
                        cpl_table_copy_data_double(tbl, "c", cpl_vector_get_data_const(tt)));
                    eris_ifu_free_vector(&tt);
                }
                if (arc_tbl[i].fit_c_r != NULL) {
                    BRK_IF_ERROR(
                        cpl_table_new_column(tbl, "c_r", CPL_TYPE_DOUBLE));
                    BRK_IF_NULL(
                        tt = eris_ifu_vector_get_data(arc_tbl[i].fit_c_r));
                    BRK_IF_ERROR(
                        cpl_table_copy_data_double(tbl, "c_r", cpl_vector_get_data_const(tt)));
                    eris_ifu_free_vector(&tt);
                }
                if (arc_tbl[i].fit_r != NULL) {
                    BRK_IF_ERROR(
                        cpl_table_new_column(tbl, "r", CPL_TYPE_DOUBLE));
                    BRK_IF_NULL(
                        tt = eris_ifu_vector_get_data(arc_tbl[i].fit_r));
                    BRK_IF_ERROR(
                        cpl_table_copy_data_double(tbl, "r", cpl_vector_get_data_const(tt)));
                    eris_ifu_free_vector(&tt);
                }

                // calc stddev of differences of distances between single traces
                if ((arc_tbl[i].fit_l != NULL) && (arc_tbl[i].fit_r != NULL)) {
                    t = eris_ifu_vector_duplicate(arc_tbl[i].fit_r);
                    BRK_IF_ERROR(
                        eris_ifu_vector_subtract(t, arc_tbl[i].fit_l));
                    std = eris_ifu_vector_get_stdev(t);
                    if (!eris_ifu_is_nan_or_inf(std)) {
                        BRK_IF_ERROR(
                            cpl_propertylist_append_double(pl[i], "STDEV R-L", std));
                    } else {
                        BRK_IF_ERROR(
                            cpl_propertylist_append_string(pl[i], "STDEV R-L", "NaN"));
                    }

                    // for the ease of stats: save the slit-width
                    BRK_IF_ERROR(
                        cpl_table_new_column(tbl, "slit_width", CPL_TYPE_DOUBLE));
                    BRK_IF_NULL(
                        tt = eris_ifu_vector_get_data(t));
                    BRK_IF_ERROR(
                        cpl_table_copy_data_double(tbl, "slit_width", cpl_vector_get_data_const(tt)));

                    eris_ifu_free_vector(&tt);
                    eris_ifu_free_ifu_vector(&t);
                }
                if ((arc_tbl[i].fit_l != NULL) && (arc_tbl[i].fit_c != NULL)) {
                    t = eris_ifu_vector_duplicate(arc_tbl[i].fit_c);
                    BRK_IF_ERROR(
                        eris_ifu_vector_subtract(t, arc_tbl[i].fit_l));
                    std = eris_ifu_vector_get_stdev(t);
                    if (!eris_ifu_is_nan_or_inf(std)) {
                        BRK_IF_ERROR(
                            cpl_propertylist_append_double(pl[i], "STDEV C-L", std));
                    } else {
                        BRK_IF_ERROR(
                            cpl_propertylist_append_string(pl[i], "STDEV C-L", "NaN"));
                    }
                    eris_ifu_free_ifu_vector(&t);
                }
                if ((arc_tbl[i].fit_c != NULL) && (arc_tbl[i].fit_r != NULL)) {
                    t = eris_ifu_vector_duplicate(arc_tbl[i].fit_r);
                    BRK_IF_ERROR(
                        eris_ifu_vector_subtract(t, arc_tbl[i].fit_c));
                    std = eris_ifu_vector_get_stdev(t);
                    if (!eris_ifu_is_nan_or_inf(std)) {
                        BRK_IF_ERROR(
                            cpl_propertylist_append_double(pl[i], "STDEV R-C", std));
                    } else {
                        BRK_IF_ERROR(
                            cpl_propertylist_append_string(pl[i], "STDEV R-C", "NaN"));
                    }
                    eris_ifu_free_ifu_vector(&t);
                }
                if ((arc_tbl[i].fit_c != NULL) && (arc_tbl[i].fit_c_l != NULL)) {
                    t = eris_ifu_vector_duplicate(arc_tbl[i].fit_c);
                    BRK_IF_ERROR(
                        eris_ifu_vector_subtract(t, arc_tbl[i].fit_c_l));
                    std = eris_ifu_vector_get_stdev(t);
                    if (!eris_ifu_is_nan_or_inf(std)) {
                        BRK_IF_ERROR(
                            cpl_propertylist_append_double(pl[i], "STDEV C-CL", std));
                    } else {
                        BRK_IF_ERROR(
                            cpl_propertylist_append_string(pl[i], "STDEV C-CL", "NaN"));
                    }
                    eris_ifu_free_ifu_vector(&t);
                }
                if ((arc_tbl[i].fit_c_r != NULL) && (arc_tbl[i].fit_c != NULL)) {
                    t = eris_ifu_vector_duplicate(arc_tbl[i].fit_c_r);
                    BRK_IF_ERROR(
                        eris_ifu_vector_subtract(t, arc_tbl[i].fit_c));
                    std = eris_ifu_vector_get_stdev(t);
                    if (!eris_ifu_is_nan_or_inf(std)) {
                        BRK_IF_ERROR(
                            cpl_propertylist_append_double(pl[i], "STDEV CR-C", std));
                    } else {
                        BRK_IF_ERROR(
                            cpl_propertylist_append_string(pl[i], "STDEV CR-C", "NaN"));
                    }
                    eris_ifu_free_ifu_vector(&t);
                }
                if ((productDepth & 2) != 0) {
                    BRK_IF_ERROR(
                        cpl_table_save(tbl, NULL, pl[i], ERIS_IFU_PRO_DIST_QC_FIT_FN, CPL_IO_EXTEND));
                }

                eris_ifu_free_table(&tbl);
            } // end: i = SLITLET_CNT
        } // end:  if (productDepth & 2)
    }
    CATCH
    {
        CATCH_MSGS();
        for (int j = 0; j < SLITLET_CNT; j++) {
            cpl_polynomial_delete(poly_u[j]); poly_u[j] = NULL;
        }
        cpl_free(poly_u); poly_u = NULL;
    }

    eris_ifu_free_bivector(&grid);
    eris_ifu_free_vector(&val_to_fit);
    eris_ifu_free_ifu_vector(&y_wave);
    for (int i = 0; i < SLITLET_CNT; i++) {
        eris_ifu_free_ifu_vector(&arc_tbl[i].fit_l);
        eris_ifu_free_ifu_vector(&arc_tbl[i].fit_c);
        eris_ifu_free_ifu_vector(&arc_tbl[i].fit_c_l);
        eris_ifu_free_ifu_vector(&arc_tbl[i].fit_c_r);
        eris_ifu_free_ifu_vector(&arc_tbl[i].fit_r);
    }
    if (pl != NULL) {
        for (int i = 0; i < SLITLET_CNT; i++) {
            eris_ifu_free_propertylist(&pl[i]);
        }
    }
    cpl_free(arc_tbl); arc_tbl = NULL;
    cpl_free(pl); pl = NULL;

    return poly_u;
}

/**
 * @brief Fit polynomial to single edge as function of Y
 * @param edge          Table with edge positions
 * @param col_name      Column name containing X positions
 * @param y             Y coordinate vector
 * @param n_calib       Number of calibration points
 * @param n_size        Output vector size (number of Y positions)
 * @param fit_order     Polynomial order (typically 2 or 3)
 * @param slitlet       Slitlet index for debugging
 * @param dbg_tbl       Debug table to store polynomial coefficients
 * @return Vector with fitted edge positions at n_size Y coordinates, or NULL on error
 *
 * Fits polynomial of form: x(y) = a0 + a1*y + a2*y^2 + ...
 * Returns evaluated positions at regular Y intervals.
 *
 * @note Returned vector must be freed with eris_ifu_vector_delete()
 */
eris_ifu_vector* eris_ifu_dist_calc_distortion_fitedge(const cpl_table *edge,
                                                       const char *col_name,
                                                       const eris_ifu_vector *y,
                                                       int n_calib,
                                                       int n_size,
                                                       int fit_order,
                                                       int slitlet,
                                                       cpl_table *dbg_tbl)
{
    const double        *px         = NULL;
    double              *pfit_par   = NULL;
    eris_ifu_vector     *x          = NULL,
                        *fit        = NULL;
    cpl_vector          *fit_par    = NULL;
    cpl_array           *tmp_array  = NULL;

    cpl_ensure(edge, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(col_name, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(y, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(n_calib > 0, CPL_ERROR_ILLEGAL_INPUT, NULL);
    cpl_ensure(n_size > 0, CPL_ERROR_ILLEGAL_INPUT, NULL);
    cpl_ensure(fit_order > 0, CPL_ERROR_ILLEGAL_INPUT, NULL);

    TRY
    {
        BRK_IF_NULL(
            px = cpl_table_get_data_double_const(edge, col_name));

        BRK_IF_NULL(
            x = eris_ifu_vector_new_wrap(n_calib, px));

        // fit polynomial to edge
        BRK_IF_NULL(
            fit_par = eris_ifu_polyfit_edge(y, x, fit_order));
        eris_ifu_free_ifu_vector(&x);

        BRK_IF_NULL(
            pfit_par = cpl_vector_get_data(fit_par));

        // save polynomials to table to support remote debugging
        tmp_array = cpl_array_wrap_double(pfit_par, cpl_vector_get_size(fit_par));
        cpl_table_set_int(dbg_tbl, ERIS_IFU_DIST_DBG_SLITLET, slitlet, slitlet);
        cpl_table_set_array(dbg_tbl, col_name, slitlet, tmp_array);
        cpl_array_unwrap(tmp_array);
        CHECK_ERROR_STATE();

        // create vector with fitted values
        BRK_IF_NULL(
            fit = eris_ifu_vector_new(n_size));
        for (int j = 0; j < n_size; j++) {
            int yy = eris_ifu_distortion_calc_y(n_size, j);
            double v = pfit_par[0] +
                       pfit_par[1] * yy +
                       pfit_par[2] * pow(yy, 2);
//                       + pfit_par[3] * pow(yy, 3);
            BRK_IF_ERROR(
                eris_ifu_vector_set(fit, j, v));
        }
    }
    CATCH
    {
        CATCH_MSGS();
        eris_ifu_free_ifu_vector(&fit);
    }
    eris_ifu_free_vector(&fit_par);

    return fit;
}

/**
 * @brief Fill grid with distortion mapping data
 * @param grid         Output: 2D grid of target coordinates (x, y)
 * @param data         Input: fitted edge positions
 * @param val_to_fit   Output: measured X positions to fit
 * @param x_pos        Target X position for this arc
 * @param n_size       Number of data points
 * @param arc_cnt      Arc index (for offset in grid)
 * @return CPL_ERROR_NONE on success, error code otherwise
 *
 * Populates grid and value arrays for 2D polynomial fitting:
 * - Grid X: all set to x_pos (target coordinate)
 * - Grid Y: distributed Y coordinates
 * - Values: measured X positions from fitted data
 *
 * @note Rejected values in data are set to NAN in val_to_fit
 */
cpl_error_code eris_ifu_dist_calc_distortion_fillgrid(cpl_bivector *grid,
                                                      const eris_ifu_vector *data,
                                                      cpl_vector *val_to_fit,
                                                      double x_pos,
                                                      int n_size,
                                                      int arc_cnt)
{
    cpl_error_code      err             = CPL_ERROR_NONE;
    double              *pgridx         = NULL,
                        *pgridy         = NULL,
                        *pval_to_fit    = NULL;

    cpl_ensure_code(grid, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(data, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(val_to_fit, CPL_ERROR_NULL_INPUT);
    if (x_pos == -1){
        x_pos=0;
    }
    cpl_ensure_code(x_pos >= -64, CPL_ERROR_ILLEGAL_INPUT);
    cpl_ensure_code(arc_cnt >= 0, CPL_ERROR_ILLEGAL_INPUT);
    cpl_ensure_code(n_size >= 0, CPL_ERROR_ILLEGAL_INPUT);

    TRY
    {
        BRK_IF_NULL(
            pgridx = cpl_bivector_get_x_data(grid));
        BRK_IF_NULL(
            pgridy = cpl_bivector_get_y_data(grid));
        BRK_IF_NULL(
            pval_to_fit = cpl_vector_get_data(val_to_fit));

        for (int j = arc_cnt*n_size; j < (arc_cnt+1)*n_size; j++) {
            ASSURE(j < cpl_bivector_get_size(grid),
                                CPL_ERROR_ILLEGAL_INPUT,
                                "index overrun");

            int jj = j % n_size;

            // same target-x for all y-positions
            pgridx[j] = x_pos;

            // calc target-y
            pgridy[j] = eris_ifu_distortion_calc_y(n_size, jj);

            // insert fitted x-value at y-position
            if (eris_ifu_vector_is_rejected(data, jj)) {
                pval_to_fit[j] = NAN;
            } else {
                pval_to_fit[j] = eris_ifu_vector_get(data, jj);
            }
        }
    }
    CATCH
    {
        CATCH_MSGS();
        err = cpl_error_get_code();
    }

    return err;
}

/**
 * @brief Create 2D polynomial fit from grid data
 * @param xy_pos  2D grid of (x,y) positions
 * @param values  Measured values to fit
 * @param msee    Output: mean squared error estimate (optional, can be NULL)
 * @return Fitted 2D polynomial, or NULL on error
 *
 * Performs 2D polynomial fit with maximum degrees:
 * - X dimension: degree 2
 * - Y dimension: degree 3
 *
 * The polynomial can be evaluated at any (x,y) to get the fitted value.
 *
 * @note Returned polynomial must be freed with cpl_polynomial_delete()
 * @note NaN values in input are handled by CPL fitting routine
 */
cpl_polynomial* eris_ifu_dist_poly_fit_2d_create(cpl_bivector     *xy_pos,
                                                 const cpl_vector *values,
                                                 double           *msee)
{
    int             xy_size         = 0;
    double          rechisq         = 0.,
                    value           = 0.;
    cpl_size        degree          = NS_FIT_DEGREE;
    cpl_matrix      *samppos2d      = NULL;
    cpl_vector      *fitresidual    = NULL;
    cpl_polynomial  *fit2d          = NULL;

    typedef double* (*get_data)(cpl_bivector*);
    get_data data_extractor[2] = { &cpl_bivector_get_x_data,
                                   &cpl_bivector_get_y_data };

    cpl_ensure(xy_pos, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(values, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(degree > 0, CPL_ERROR_ILLEGAL_INPUT, NULL);

    TRY
    {
        xy_size = (int) cpl_bivector_get_size(xy_pos);
        CHECK_ERROR_STATE();

        BRK_IF_NULL(
            fit2d = cpl_polynomial_new(2));
        BRK_IF_NULL(
            samppos2d = cpl_matrix_new(2, xy_size));

        for (int i = 0; i < 2; i++) {
            for (int j = 0; j < xy_size; j++) {
                value = data_extractor[i](xy_pos)[j];
                BRK_IF_ERROR(
                    cpl_matrix_set(samppos2d, i, j, value));
            }
        }
        const cpl_size    maxdeg2d[]  = {2,3};
        BRK_IF_ERROR(
            cpl_polynomial_fit(fit2d, samppos2d, NULL, values, NULL, CPL_TRUE,
                                NULL, maxdeg2d));
//        BRK_IF_ERROR(
//            cpl_polynomial_fit(fit2d, samppos2d, NULL, values, NULL, CPL_FALSE,
//                                NULL, &degree));

        BRK_IF_NULL(
            fitresidual = cpl_vector_new(xy_size));
        BRK_IF_ERROR(
            cpl_vector_fill_polynomial_fit_residual(fitresidual, values, NULL,
                                                  fit2d, samppos2d, &rechisq));
        if (msee) {
            *msee = cpl_vector_product(fitresidual, fitresidual)
                                 / (double) cpl_vector_get_size(fitresidual);
        }
    }
    CATCH
    {
        CATCH_MSGS();
        eris_ifu_free_polynomial(&fit2d);
    }

    eris_ifu_free_matrix(&samppos2d);
    eris_ifu_free_vector(&fitresidual);

    return fit2d;
}

/**
 * @brief Save distortion polynomials to FITS file
 * @param poly2d          Array of 2D polynomials (one per slitlet)
 * @param minmax_borders  Table with min/max border positions
 * @param fn              Output filename
 * @param frameset        Input frameset for provenance
 * @param parlist         Parameter list for provenance
 * @param qc              Array of QC property lists
 * @return CPL_ERROR_NONE on success, error code otherwise
 *
 * Saves polynomials in FITS table format with:
 * - One extension per slitlet
 * - Columns: degx, degy, coeff
 * - QC parameters in extension headers
 * - Final extension contains minmax_borders table
 *
 * @note Output file can be loaded by eris_ifu_load_distortion_polynomials()
 */
cpl_error_code eris_ifu_dist_save_distortion(cpl_polynomial          **poly2d,
                                             const cpl_table         *minmax_borders,
                                             const char              *fn,
                                             cpl_frameset            *frameset,
                                             const cpl_parameterlist *parlist,
                                             cpl_propertylist        **qc)
{
    double      coeff/*,
                xshift          = 0.*/;
//    char        key_name[FILE_NAME_SZ];
    cpl_size         max_degree;
    cpl_size         coef_pow[2];
    cpl_table        *poly_tbl      = NULL;
    cpl_error_code   err            = CPL_ERROR_NONE;

    cpl_ensure_code(poly2d, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(frameset, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(parlist, CPL_ERROR_NULL_INPUT);
    char* extname;
    TRY
    {
        for (int i = 0; i < SLITLET_CNT; i++) {
            if (poly2d[i] == NULL) {
                cpl_propertylist_save(NULL, fn, CPL_IO_EXTEND);
            } else {
                max_degree = cpl_polynomial_get_degree(poly2d[i]);
                BRK_IF_NULL(
                    poly_tbl = cpl_table_new(max_degree * max_degree));
                BRK_IF_ERROR(
                    cpl_table_new_column(poly_tbl,"degx", CPL_TYPE_INT));
                BRK_IF_ERROR(
                    cpl_table_new_column(poly_tbl,"degy", CPL_TYPE_INT));
                BRK_IF_ERROR(
                    cpl_table_new_column(poly_tbl,"coeff", CPL_TYPE_DOUBLE));

                cpl_size tx = 0;
                for (cpl_size cy=0; cy < max_degree; cy++) {
                    for (cpl_size cx=0; cx < max_degree; cx++) {
                        coef_pow[0]=cx;
                        coef_pow[1]=cy;
                        coeff = cpl_polynomial_get_coeff(poly2d[i], coef_pow);
                        if (fabs(coeff) > 1e-30) {  //Here DBL_ZERO_TOLERANCE 1e-10 is too high
                            BRK_IF_ERROR(
                                cpl_table_set_int(poly_tbl,"degx",tx,(int) cx));
                            BRK_IF_ERROR(
                                    cpl_table_set_int(poly_tbl,"degy",tx,(int) cy));
                            BRK_IF_ERROR(
                                    cpl_table_set_double(poly_tbl,"coeff", tx,
                                        coeff));
                            tx++;
                        }
                    }
                }
                BRK_IF_ERROR(
                        cpl_table_set_size(poly_tbl, tx));

//                /* QC LOG */
//                snprintf(key_name, FILE_NAME_SZ-1, "%s%d%d", "QC COEFF", 0, 0);
//                BRK_IF_ERROR(
//                    eris_ifu_dist_qclog_add_double(qc[i], key_name, pcf[0][0],
//                                                "Polynomial distortion coefficient"));
//
//                snprintf(key_name, FILE_NAME_SZ-1, "%s%d%d", "QC COEFF", 1,0);
//                BRK_IF_ERROR(
//                    eris_ifu_dist_qclog_add_double(qc[i], key_name, pcf[1][0],
//                                                "Polynomial distortion coefficient"));
//
//                snprintf(key_name, FILE_NAME_SZ-1, "%s%d%d", "QC COEFF", 0, 1);
//                BRK_IF_ERROR(
//                    eris_ifu_dist_qclog_add_double(qc[i], key_name, pcf[0][1],
//                                                "Polynomial distortion coefficient"));
//
//                snprintf(key_name, FILE_NAME_SZ-1, "%s%d%d", "QC COEFF", 1, 1);
//                BRK_IF_ERROR(
//                    eris_ifu_dist_qclog_add_double(qc[i], key_name, pcf[1][1],
//                                                "Polynomial distortion coefficient"));
//
//                snprintf(key_name, FILE_NAME_SZ-1, "%s%d%d", "QC COEFF", 2, 0);
//                BRK_IF_ERROR(
//                    eris_ifu_dist_qclog_add_double(qc[i], key_name, pcf[2][0],
//                                                "Polynomial distortion coefficient"));
//
//                snprintf(key_name, FILE_NAME_SZ-1, "%s%d%d", "QC COEFF", 0, 2);
//                BRK_IF_ERROR(
//                    eris_ifu_dist_qclog_add_double(qc[i], key_name, pcf[0][2],
//                                                "Polynomial distortion coefficient"));
//
//                snprintf(key_name, FILE_NAME_SZ-1, "%s%d%d", "QC COEFF", 2, 1);
//                BRK_IF_ERROR(
//                    eris_ifu_dist_qclog_add_double(qc[i], key_name, pcf[2][1],
//                                                "Polynomial distortion coefficient"));
//
//                snprintf(key_name, FILE_NAME_SZ-1, "%s%d%d", "QC COEFF", 1, 2);
//                BRK_IF_ERROR(
//                    eris_ifu_dist_qclog_add_double(qc[i], key_name, pcf[1][2],
//                                                "Polynpoly2domial distortion coefficient"));
//
//                snprintf(key_name, FILE_NAME_SZ-1, "%s%d%d","QC COEFF", 3, 0);
//                BRK_IF_ERROR(
//                    eris_ifu_dist_qclog_add_double(qc[i], key_name, pcf[3][0],
//                                                "Polynomial distortion coefficient"));
//
//                snprintf(key_name, FILE_NAME_SZ-1, "%s%d%d","QC COEFF", 0, 3);
//                BRK_IF_ERROR(
//                    eris_ifu_dist_qclog_add_double(qc[i], key_name, pcf[0][3],
//                                                "Polynomial distortion coefficient"));
//
//                xshift = eris_ifu_dist_compute_shift( x_c, y_c,
//                                            pcf[0][0], pcf[1][0], pcf[0][1],
//                                            pcf[1][1], pcf[2][0], pcf[0][2],
//                                            pcf[2][1], pcf[1][2], pcf[3][0], pcf[0][3]);
//                BRK_IF_ERROR(
//                    eris_ifu_dist_qclog_add_double(qc[i], "QC XSHIFT CC", xshift,
//                                                   "X shift in x_c,y_c"));
//
//                xshift = eris_ifu_dist_compute_shift( x_l, y_l,
//                                            pcf[0][0], pcf[1][0], pcf[0][1],
//                                            pcf[1][1], pcf[2][0], pcf[0][2],
//                                            pcf[2][1], pcf[1][2], pcf[3][0], pcf[0][3]);
//                BRK_IF_ERROR(
//                    eris_ifu_dist_qclog_add_double(qc[i], "QC XSHIFT LL", xshift,
//                                                   "X shift in x_l,y_l"));
//
//                xshift = eris_ifu_dist_compute_shift(x_l,y_u,pcf[0][0],pcf[1][0],pcf[0][1],
//                                pcf[1][1],pcf[2][0],pcf[0][2],
//                                pcf[2][1],pcf[1][2],pcf[3][0],pcf[0][3]);
//                BRK_IF_ERROR(
//                    eris_ifu_dist_qclog_add_double(qc[i],"QC XSHIFT UL",xshift,
//                                "X shift in x_l,y_u"));
//
//                xshift = eris_ifu_dist_compute_shift(x_u,y_u,pcf[0][0],pcf[1][0],pcf[0][1],
//                                pcf[1][1],pcf[2][0],pcf[0][2],
//                                pcf[2][1],pcf[1][2],pcf[3][0],pcf[0][3]);
//                BRK_IF_ERROR(
//                    eris_ifu_dist_qclog_add_double(qc[i],"QC XSHIFT UR",xshift,
//                                "X shift in x_u,y_u"));
//
//                xshift = eris_ifu_dist_compute_shift(x_u,y_l,pcf[0][0],pcf[1][0],pcf[0][1],
//                                pcf[1][1],pcf[2][0],pcf[0][2],
//                                pcf[2][1],pcf[1][2],pcf[3][0],pcf[0][3]);
//                BRK_IF_ERROR(
//                    eris_ifu_dist_qclog_add_double(qc[i],"QC XSHIFT LR",xshift,
//                                "X shift in x_u,y_l"));

                extname = cpl_sprintf("SLITLET%2.2d",i);
                cpl_propertylist_append_string(qc[i],"EXTNAME",extname);
                if (i == 0) {
                    BRK_IF_ERROR(
                        eris_ifu_save_table(frameset, NULL, parlist, frameset, NULL,
                                            REC_NAME_DISTORTION,
                                            ERIS_IFU_PRO_DIST_DISTORTION,
                                            qc[i], "EXTNAME",
                                            fn,
                                            poly_tbl));
                } else {
                    BRK_IF_ERROR(
                        cpl_table_save(poly_tbl, NULL, qc[i], fn, CPL_IO_EXTEND));
                }
                cpl_free(extname);
                eris_ifu_free_table(&poly_tbl);
            } // if (poly2d[ii] != NULL)
        } // end: i = SLITLET_CNT

        cpl_propertylist* ext = cpl_propertylist_new();
        extname = cpl_sprintf("SLITLET%2.2d",SLITLET_CNT);
                               cpl_propertylist_append_string(ext,"EXTNAME",extname);
        // save minmax borders as well
        BRK_IF_ERROR(
            cpl_table_save(minmax_borders, NULL, ext, fn, CPL_IO_EXTEND));
        cpl_propertylist_delete(ext);
	cpl_free(extname);
    }
    CATCH
    {
        CATCH_MSGS();
        err = cpl_error_get_code();
    }

    return err;
}

/**
 * @brief Warp full image using distortion polynomials (alternative method)
 * @param hdrl_img_in  Input HDRL image to warp
 * @param poly_u       Array of X-distortion polynomials
 * @param productDepth Debug output level
 * @return Warped HDRL image, or NULL on error
 *
 * Alternative warping method that processes entire image at once.
 * Uses polynomial_warp with identity Y polynomial.
 *
 * @note Returned image must be freed with hdrl_image_delete()
 * @note Less efficient than slitlet-wise warping
 */
hdrl_image* eris_ifu_dist_warp_image_full(const hdrl_image *hdrl_img_in,
                                     cpl_polynomial   **poly_u,
                                     int              productDepth)
{
    cpl_size        pows[2];
    cpl_image       *img_warped             = NULL,
                    *img_warped_extr        = NULL;
    hdrl_image      *hdrl_img_warped        = NULL,
                    *hdrl_img_warped_extr   = NULL;
    cpl_polynomial  *poly_v                 = NULL;
    char            *fn                     = NULL;

    cpl_ensure(hdrl_img_in, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(poly_u,   CPL_ERROR_NULL_INPUT, NULL);

    TRY
    {
        BRK_IF_NULL(
            poly_v = cpl_polynomial_new(2));

        pows[0] = 0;
        pows[1] = 0;
        BRK_IF_ERROR(
            cpl_polynomial_set_coeff(poly_v, pows, 0.0));
        pows[0] = 0;
        pows[1] = 1;
        BRK_IF_ERROR(
            cpl_polynomial_set_coeff(poly_v, pows, 1.0));

        BRK_IF_NULL(
            hdrl_img_warped = hdrl_image_new(hdrl_image_get_size_x(hdrl_img_in),
                                             hdrl_image_get_size_y(hdrl_img_in)));

        BRK_IF_NULL(
            img_warped = hdrl_image_get_image(hdrl_img_warped));

        for (int i = 0; i < SLITLET_CNT; i++) {
            if (poly_u[i] != NULL) {
//                cpl_msg_info(cpl_func, "Warp slitlet #%02d", i+1);

                BRK_IF_NULL(
                    hdrl_img_warped_extr = eris_ifu_warp_polynomial_image(hdrl_img_in,
                                                                          poly_u[i], poly_v));

                // target left + right edge
                int t_l = eris_ifu_distortion_target_left_edge(i),
                    t_r = eris_ifu_distortion_target_right_edge(i);
                BRK_IF_NULL(
                    img_warped_extr = hdrl_image_get_image(hdrl_img_warped_extr));

                if (productDepth & 8) {
                    // save warped intermediate images uncut
                    fn = cpl_sprintf("eris_ifu_distortion_dbg_warp_out_%02d.fits", i+1);
                    BRK_IF_ERROR(
                        eris_ifu_save_image_dbg(img_warped_extr, fn,
                                                CPL_IO_CREATE, NULL));
                }

                if (i != 0) {
                    // set to zero left of slitlet (except for first slitlet)
                    BRK_IF_ERROR(
                        cpl_image_fill_window(img_warped_extr, 1, 1, t_l,
                                              ERIS_IFU_DETECTOR_SIZE_Y, 0.));
                }
                if (i != SLITLET_CNT-1) {
                    // set to zero right of slitlet (except for last slitlet)
                    BRK_IF_ERROR(
                        cpl_image_fill_window(img_warped_extr, t_r+2, 1,
                                              ERIS_IFU_DETECTOR_SIZE_X,
                                              ERIS_IFU_DETECTOR_SIZE_Y, 0.));
                }

                if (productDepth & 8) {
                    // save warped intermediate images cut
                    BRK_IF_ERROR(
                        eris_ifu_save_image_dbg(img_warped_extr, fn,
                                                CPL_IO_EXTEND, NULL));
                    cpl_free(fn); fn = NULL;
                }

                BRK_IF_ERROR(
                    cpl_image_add(img_warped, img_warped_extr));

                eris_ifu_free_hdrl_image(&hdrl_img_warped_extr);
            }
        } // end: i = SLITLET_CNT
    }
    CATCH
    {
        CATCH_MSGS();
        eris_ifu_free_hdrl_image(&hdrl_img_warped);
    }

    eris_ifu_free_polynomial(&poly_v);

    return hdrl_img_warped;
}

/**
 * @brief Warp single slitlet
 * @param imgIn      Input HDRL image
 * @param poly_u     X-distortion polynomial
 * @param poly_v     Y-distortion polynomial (typically identity)
 * @param l_min      Minimum X coordinate of slitlet
 * @param r_max      Maximum X coordinate of slitlet
 * @param slitletNr  Slitlet index for diagnostics
 * @return Warped slitlet image (SLITLET_WIDTH x ERIS_IFU_DETECTOR_SIZE_Y), or NULL on error
 *
 * Extracts and warps a single slitlet:
 * 1. Extracts region from l_min to r_max
 * 2. Applies polynomial warping
 * 3. Trims or pads result to standard SLITLET_WIDTH (64 pixels)
 *
 * Special handling for slitlets extending beyond detector boundaries.
 *
 * @note Returned image must be freed with hdrl_image_delete()
 * @note Output width is always SLITLET_WIDTH regardless of input
 */
hdrl_image* eris_ifu_dist_warp_slitlet(const hdrl_image     *imgIn,
                                       const cpl_polynomial *poly_u,
                                       const cpl_polynomial *poly_v,
                                       double l_min,
                                       double r_max,
                                       int slitletNr)
{
    int             llx             = 0,
                    urx             = 0;
    hdrl_image      *imgOutSlit     = NULL,
                    *imgInSlit      = NULL,
                    *imgOutSlitTrim = NULL;

    cpl_ensure(imgIn, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(poly_u,      CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(poly_u,      CPL_ERROR_NULL_INPUT, NULL);


    TRY
    {
        // get left and right nominal border of slitlet
        llx   = (int)(l_min)+1;
        urx   = (int)(r_max)+1;

        if ((llx > 0) && (urx <= ERIS_IFU_DETECTOR_SIZE_X)) {
            BRK_IF_NULL(
                imgInSlit = hdrl_image_extract(imgIn,
                                               llx, 1,
                                               urx, ERIS_IFU_DETECTOR_SIZE_Y));
        } else {
            BRK_IF_NULL(
                imgInSlit = hdrl_image_new(urx-llx+1, ERIS_IFU_DETECTOR_SIZE_Y));
            // mask all pixel as bad, the good ones will be flagged later
            cpl_mask *tmpMask;
            BRK_IF_NULL(
                tmpMask = cpl_mask_threshold_image_create(
                    hdrl_image_get_image(imgInSlit),-1,1));
            hdrl_image_reject_from_mask(imgInSlit, tmpMask);
            cpl_mask_delete(tmpMask);
            hdrl_image *tmpImg;
            if (llx < 1) {
                BRK_IF_NULL(
                    tmpImg = hdrl_image_extract(imgIn,
                         1, 1, urx, ERIS_IFU_DETECTOR_SIZE_Y));
                int pos1 = -llx + 2;
                BRK_IF_ERROR(
                    hdrl_image_copy(imgInSlit,tmpImg, pos1, 1));
            } else {
                BRK_IF_NULL(
                    tmpImg = hdrl_image_extract(imgIn,
                        llx, 1, ERIS_IFU_DETECTOR_SIZE_X, ERIS_IFU_DETECTOR_SIZE_Y));
                BRK_IF_ERROR(
                    hdrl_image_copy(imgInSlit,tmpImg, 1, 1));
            }
            eris_ifu_free_hdrl_image(&tmpImg);
        }

        BRK_IF_NULL(
            imgOutSlit = eris_ifu_warp_polynomial_image(imgInSlit,
                                                        poly_u, poly_v));

        if (hdrl_image_get_size_x(imgOutSlit) > SLITLET_WIDTH) {
            // extract left 64 pixels of slitlet (all data to the right is invalid)
            BRK_IF_NULL(
                imgOutSlitTrim = hdrl_image_extract(imgOutSlit,
                                                    1, 1,
                                                    SLITLET_WIDTH,
                                                    ERIS_IFU_DETECTOR_SIZE_Y));
        } else {
            // if slitlet is less than 64 pix wide, paste it into a slitlet of 64pix width
            cpl_msg_warning(cpl_func, "Slitlet #%d: width only %d pix !!!", slitletNr+1, (int)hdrl_image_get_size_x(imgOutSlit));
            BRK_IF_NULL(
                imgOutSlitTrim = hdrl_image_new(SLITLET_WIDTH, ERIS_IFU_DETECTOR_SIZE_Y));
            // will be aligned to the right side --> better
            int offset = SLITLET_WIDTH - (int) hdrl_image_get_size_x(imgOutSlit);
            // will be aligned to the left side --> worse
            //int offset = 0;
            BRK_IF_ERROR(
                eris_ifu_image_add_slit(imgOutSlitTrim, imgOutSlit, offset));
        }
    }
    CATCH
    {
        CATCH_MSGS();
        eris_ifu_free_hdrl_image(&imgOutSlitTrim);
    }

    eris_ifu_free_hdrl_image(&imgOutSlit);
    eris_ifu_free_hdrl_image(&imgInSlit);

    return imgOutSlitTrim;
}

/**
 * @brief Warp full detector image by warping each slitlet
 * @param imgIn    Input HDRL image
 * @param poly_u   Array of X-distortion polynomials (one per slitlet)
 * @param borders  Table with l_min/r_max for each slitlet
 * @return Warped full detector image, or NULL on error
 *
 * Main warping function that:
 * 1. Creates output image of same size as input
 * 2. Warps each slitlet independently
 * 3. Pastes warped slitlets into output at standard positions
 *
 * This approach is more accurate than full-image warping as it handles
 * discontinuities between slitlets properly.
 *
 * @note Returned image must be freed with hdrl_image_delete()
 * @note Can be parallelized (warping of different slitlets is independent)
 */
hdrl_image* eris_ifu_dist_warp_image(const hdrl_image  *imgIn,
                                     cpl_polynomial   **poly_u,
                                     const cpl_table   *borders)
{
    int             tmp             = 0;
    cpl_size        pows[2];
    cpl_polynomial  *poly_v         = NULL;
    hdrl_image      *imgOut         = NULL,
                    *imgOutSlitTrim = NULL;

    cpl_ensure(imgIn,  CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(poly_u, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(borders, CPL_ERROR_NULL_INPUT, NULL);

    TRY
    {
        BRK_IF_NULL(
            poly_v = cpl_polynomial_new(2));
        pows[0] = 0; pows[1] = 0;
        BRK_IF_ERROR(
            cpl_polynomial_set_coeff(poly_v, pows, 0.0));
        pows[0] = 0; pows[1] = 1;
        BRK_IF_ERROR(
            cpl_polynomial_set_coeff(poly_v, pows, 1.0));

        BRK_IF_NULL(
            imgOut = hdrl_image_new(hdrl_image_get_size_x(imgIn),
                                    hdrl_image_get_size_y(imgIn)));

        for (int i = 0; i < SLITLET_CNT; i++) {
            if (poly_u[i] != NULL) {
//                cpl_msg_debug(cpl_func, "Warp slitlet #%02d", i+1);

                // Extract, warp and trim slitlet
                // (can be parallelized)
                BRK_IF_NULL(
                    imgOutSlitTrim = eris_ifu_dist_warp_slitlet(imgIn,
                                                                poly_u[i],
                                                                poly_v,
                                                                cpl_table_get_double(borders, ERIS_IFU_POLY_EDGE_L, i, &tmp),
                                                                cpl_table_get_double(borders, ERIS_IFU_POLY_EDGE_R, i, &tmp),
                                                                i));

                // add- in slitlet to synthetic full warped image
                // (can be parallelized)
                BRK_IF_ERROR(
                    eris_ifu_image_add_slit(imgOut,
                                            imgOutSlitTrim,
                                            eris_ifu_distortion_target_left_edge(i)));

                eris_ifu_free_hdrl_image(&imgOutSlitTrim);

            }
        } // end: i = SLITLET_CNT
    }
    CATCH
    {
        CATCH_MSGS();
        eris_ifu_free_hdrl_image(&imgOut);
    }
    eris_ifu_free_polynomial(&poly_v);
    eris_ifu_free_hdrl_image(&imgOutSlitTrim);

    return imgOut;
}

/**
 * @brief Warp bad pixel mask using distortion polynomials
 * @param bpmIn        Input bad pixel mask as HDRL image
 * @param poly_u       Array of X-distortion polynomials
 * @param borders      Table with slitlet boundaries
 * @param productDepth Debug output level
 * @return Warped bad pixel mask, or NULL on error
 *
 * Warps bad pixel mask with special handling:
 * 1. Creates column contribution map (each pixel stores its original column number)
 * 2. Warps the contribution map
 * 3. Interpolates BPM values based on fractional contributions
 * 4. Marks border regions as bad where data comes from outside valid detector area
 *
 * This ensures bad pixels are correctly propagated through geometric transformation.
 *
 * @note Returned image must be freed with hdrl_image_delete()
 * @note Algorithm accounts for fractional pixel contributions at boundaries
 */
hdrl_image* eris_ifu_dist_warp_bpm(const hdrl_image  *bpmIn,
                                   cpl_polynomial   **poly_u,
                                   const cpl_table   *borders,
                                   productDepthType   productDepth)
{
    int             tmp             = 0;
    cpl_size        pows[2];
    cpl_polynomial  *poly_v         = NULL;
    hdrl_image      *bpmOut         = NULL,
                    *imgOutSlitTrim = NULL;

    cpl_ensure(bpmIn,  CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(poly_u, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(borders, CPL_ERROR_NULL_INPUT, NULL);

    /* Create contribution map */
    cpl_size size_x = hdrl_image_get_size_x(bpmIn);
    cpl_size size_y = hdrl_image_get_size_y(bpmIn);

    hdrl_image *col_hmap = hdrl_image_new(size_x, size_y);

    //Initialize each pixel to its column number; start from 1. 
    for (cpl_size i = 1; i <= size_x; i++){
        hdrl_value col_num;
        col_num.data = i * 1.0; 
        col_num.error = 0.0;
        for (cpl_size j = 1; j <= size_y; j++){
            hdrl_image_set_pixel(col_hmap, i, j,col_num); 
        }
    }
    if (productDepth >= PD_DEBUG) {
        eris_ifu_save_hdrl_image_dbg(col_hmap, "eris_ifu_distortion_dbg_col_hmap", 1, NULL);
    }
    poly_v = cpl_polynomial_new(2);
    pows[0] = 0; pows[1] = 0;
    cpl_polynomial_set_coeff(poly_v, pows, 0.0);
    pows[0] = 0; pows[1] = 1;
    cpl_polynomial_set_coeff(poly_v, pows, 1.0);

    /* Warp the contribution map */      
    hdrl_image *col_hmap_warped = hdrl_image_new(size_x, size_y);
    for (int i = 0; i < SLITLET_CNT; i++) {
        // Extract, warp and trim slitlet   
        imgOutSlitTrim = eris_ifu_dist_warp_slitlet(col_hmap,
                                                    poly_u[i],
                                                    poly_v,
                                                    cpl_table_get_double(borders, ERIS_IFU_POLY_EDGE_L, i, &tmp),
                                                    cpl_table_get_double(borders, ERIS_IFU_POLY_EDGE_R, i, &tmp),
                                                    i);

        // add-in slitlet to synthetic full warped image
        eris_ifu_image_add_slit(col_hmap_warped,
                                imgOutSlitTrim,
                                eris_ifu_distortion_target_left_edge(i));

        eris_ifu_free_hdrl_image(&imgOutSlitTrim);
    
    }
    hdrl_image_delete(col_hmap);

    if (productDepth >= PD_DEBUG) {
        eris_ifu_save_hdrl_image_dbg(col_hmap_warped, "eris_ifu_distortion_dbg_col_hmap_warped", 1, NULL);
    }

    /* Interpolate the bpm according to the contribution*/
    // TODO: do this on slitlet-wise, do mask columns beyond l_max, r_max. 
    bpmOut = hdrl_image_new(size_x, size_y);
    int slitwidth = 0;
    cpl_size col_n_0 = 0;
    /* TODO: AMO: limit the loop to SLITLET_CNT -1 as the last iteration generates infinite loop processing science data */
    for (cpl_size k = 0; k < SLITLET_CNT; k++){
        if (k + 1 == 1) //slitlet 1
            slitwidth = 61;
        else if (k + 1 == 2)
            slitwidth = 62;
        else if (k + 1== 10)
            slitwidth = 63;
        else
            slitwidth = SLITLET_WIDTH;

        for (cpl_size y = 1; y <= size_y; y++){
            bool fake_col_n_0 = false;
            for (cpl_size x = 1; x <= SLITLET_WIDTH; x++){
                if (x >= SLITLET_WIDTH/2) {
                    // We are already at the right side of the slitlet and don't expect that there will
                    // be any bad pixels/zero- or NaN-values at the left border.
                    // But there could still be such bad pixels at the right side. In this case we WILL NOT calculate a fake_col_n_0.
                    // So we set it to true and we will step over the calculation of it
                    fake_col_n_0 = true;
                }
                hdrl_value pix_contr = hdrl_image_get_pixel(col_hmap_warped, x+k*SLITLET_WIDTH, y, NULL);
                cpl_size col_n = (cpl_size) pix_contr.data;
                double contri_n =  pix_contr.data - col_n;
                if (((pix_contr.data == 0) || hdrl_image_is_rejected(col_hmap_warped, x+k*SLITLET_WIDTH, y)) && !fake_col_n_0) {
                    col_n = -1;
                    // try to extrapolate a reasonable col_n_0
                    // go as log to the right until we get a valid pixel, keep this value
                    // get the pixel right of this one, subtract and then subtract as often
                    int g = x+1;
                    hdrl_value ggg = hdrl_image_get_pixel(col_hmap_warped, g+k*SLITLET_WIDTH, y, NULL);
                    while (isnan(ggg.data) || (ggg.data == 0)) {
                        g++;
                        ggg = hdrl_image_get_pixel(col_hmap_warped, g+k*SLITLET_WIDTH, y, NULL);
                    }
                    // go even one more to the right, because the first difference is smaller than the other differences
                    g++;
                    ggg = hdrl_image_get_pixel(col_hmap_warped, g+k*SLITLET_WIDTH, y, NULL);
                    hdrl_value kkk = hdrl_image_get_pixel(col_hmap_warped, (g+1)+k*SLITLET_WIDTH, y, NULL);
                    double diff = kkk.data - ggg.data;
                    col_n_0 = ggg.data-(g-1)*diff;
                    // we had to calculate a fake/estimated col_n_0, set a flag accordingly
                    fake_col_n_0 = true;
                }
                if ((x==1) && !fake_col_n_0)
                    col_n_0 = col_n;
                hdrl_value pix_bpm;
                if ((col_n < 1) || isnan(contri_n))
                {
                    // either left border
                    // or pix_contr.data is nan
                    pix_bpm.data = 1;
                    pix_bpm.error = 1;
                } else if (col_n > ERIS_IFU_DETECTOR_SIZE_X || col_n-col_n_0+1 > slitwidth+2) //right border or right edge
                {
                    pix_bpm.data = 1;
                    pix_bpm.error = 1;
                } else if (col_n == ERIS_IFU_DETECTOR_SIZE_X || col_n-col_n_0+1 == slitwidth+2)
                {
                    pix_bpm.data = hdrl_image_get_pixel(bpmIn, col_n, y, NULL).data * contri_n + 1.0 * (1.0-contri_n);
                }
                else {
                    pix_bpm.data = hdrl_image_get_pixel(bpmIn, col_n, y, NULL).data * contri_n +
                                   hdrl_image_get_pixel(bpmIn, col_n+1, y, NULL).data * (1.0-contri_n);
                }

                hdrl_image_set_pixel(bpmOut, x+k*SLITLET_WIDTH, y, pix_bpm);
            }
        }
    }

    hdrl_image_delete(col_hmap_warped);

    eris_ifu_free_polynomial(&poly_v);

    return bpmOut;
}

/**
 * @brief Stack warped image into cube format
 * @param imgIn  Input warped full-detector image
 * @param rowIx  LUT for slitlet ordering
 * @return Image list representing cube (z-axis is spectral), or NULL on error
 *
 * Transforms 2D warped image into 3D cube by stacking slitlets:
 * - Each plane in Z corresponds to one wavelength (detector Y coordinate)
 * - Each plane contains all 32 slitlets side-by-side
 * - Slitlet order is determined by rowIx mapping
 *
 * @note Returned imagelist must be freed with hdrl_imagelist_delete()
 * @note No wavelength calibration applied; still in pixel coordinates
 */
hdrl_imagelist* eris_ifu_stack_warped(const hdrl_image *imgIn,
                                      const int *rowIx)
{
    cpl_ensure(imgIn, CPL_ERROR_NULL_INPUT, NULL);

    int             cnt             = 0;
    double          *pImgTmpData    = NULL,
                    *pImgTmpErr     = NULL;
    const double    *pImgInData     = NULL,
                    *pImgInErr      = NULL;
    hdrl_image      *imgTmp         = NULL;
    hdrl_imagelist  *cube           = NULL;

    TRY
    {
        BRK_IF_NULL(
            cube = hdrl_imagelist_new());

        BRK_IF_NULL(
            pImgInData = cpl_image_get_data_double_const(hdrl_image_get_image_const(imgIn)));
        BRK_IF_NULL(
            pImgInErr = cpl_image_get_data_double_const(hdrl_image_get_error_const(imgIn)));


        for (int z = 0; z < ERIS_IFU_DETECTOR_SIZE_Y; z++) {
            // a slitlet is 64pix wide, but we have only 32 slitlets
            // In order to get a square cube, at the end each slitlet is doubled in y-direction...
            BRK_IF_NULL(
                imgTmp = hdrl_image_new(SLITLET_WIDTH, SLITLET_CNT));
            BRK_IF_NULL(
                pImgTmpData = cpl_image_get_data_double(hdrl_image_get_image(imgTmp)));
            BRK_IF_NULL(
                pImgTmpErr = cpl_image_get_data_double(hdrl_image_get_error(imgTmp)));


            for (int y = 0; y < SLITLET_CNT; y++) {
                for (int x = 0; x < SLITLET_WIDTH; x++) {
                    pImgTmpData[x+rowIx[y]*SLITLET_WIDTH] = pImgInData[cnt++];
                }
            }
            BRK_IF_ERROR(
                hdrl_imagelist_set(cube, imgTmp, z));
        }
    }
    CATCH
    {
        CATCH_MSGS();
        eris_ifu_free_hdrl_imagelist(&cube);
    }

    return cube;
}

/**
 * @brief Compute QC statistics on warped image
 * @param hdrlWarpedImg  Warped image to analyze
 * @param qc_list        Output: QC parameters to add to product
 * @param pl             Property list for output file
 * @param frameset       Input frameset
 * @param parlist        Parameter list
 * @return CPL_ERROR_NONE on success, error code otherwise
 *
 * Computes quality control parameters on warped image:
 * - Re-detects slitlet centers to verify distortion correction
 * - Calculates center position statistics (stdev, median)
 * - Measures slitlet widths
 * - Saves fitted centers to QC file
 *
 * QC parameters added per slitlet:
 * - C STDEV: standard deviation of center position
 * - C MEDIAN: median center position
 * - R-L STDEV/MEDIAN: slitlet width statistics
 * - For triple traces: additional center-left, center-right statistics
 *
 * @note Creates intermediate QC file with fitted centers
 */
cpl_error_code eris_ifu_dist_warp_stats(const hdrl_image *hdrlWarpedImg,
                                        cpl_propertylist *qc_list,
                                        cpl_propertylist *pl,
										cpl_frameset* frameset,
										const cpl_parameterlist* parlist)
{
    int                 nr_values           = 21, //odd
                        height              = 90, //even
                        center_y            = 0,
                        actual_size         = 0,
//                        ix_middle           = nr_values / 2 + 1;
                        ix_middle           = 9;
    const double        *pfit_centers       = NULL;
    cpl_error_code      err                 = CPL_ERROR_NONE;
    cpl_vector          *profile_x          = NULL,
                        *est_centers        = NULL,
                        *fit_centers        = NULL,
                        *fit_centers_middle = NULL;
   const cpl_image     *warpedImg          = NULL;
    cpl_table           **arr               = NULL;
    cpl_propertylist    *pl_my              = NULL;
    double swidth = 64;
    cpl_ensure_code(hdrlWarpedImg, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(qc_list, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(pl, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(frameset, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(parlist, CPL_ERROR_NULL_INPUT);

    TRY
    {
        BRK_IF_NULL(
            arr = cpl_calloc(SLITLET_CNT, sizeof(cpl_table*)));

        BRK_IF_NULL(
            warpedImg = hdrl_image_get_image_const(hdrlWarpedImg));

        // find out number of fibre-tarces per slitlet (one or three)
        center_y = ERIS_IFU_DETECTOR_BP_BORDER + ix_middle*height + height/2;
        BRK_IF_NULL(
            profile_x = eris_ifu_calc_centers_collapse_chunk(
                                                        warpedImg, center_y, height));
//AA: evtl instead of CPL_FALSE put in (isK & !is25mas)
        BRK_IF_NULL(
            est_centers = eris_ifu_dist_calc_centers_profile(profile_x, CPL_FALSE));
        // will return a vector with 32 or 3*32 center values (errors are catched)
        BRK_IF_NULL(
            fit_centers_middle = eris_ifu_dist_calc_centers_fit(
                profile_x, est_centers, CPL_TRUE));
        actual_size = (int) cpl_vector_get_size(est_centers);
        eris_ifu_free_vector(&profile_x);
        eris_ifu_free_vector(&est_centers);

        // create table to hold values
        if (actual_size <= SLITLET_CNT) {
            for (int i = 0; i < SLITLET_CNT; i++) {
                    arr[i] = cpl_table_new(nr_values);
                    cpl_table_new_column(arr[i], "c", CPL_TYPE_DOUBLE);
                    cpl_table_new_column(arr[i], "left", CPL_TYPE_DOUBLE);
                    cpl_table_new_column(arr[i], "right", CPL_TYPE_DOUBLE);
            }
        } else {
            for (int i = 0; i < SLITLET_CNT; i++) {
                    arr[i] = cpl_table_new(nr_values);
                    cpl_table_new_column(arr[i], "c_l", CPL_TYPE_DOUBLE);
                    cpl_table_new_column(arr[i], "c", CPL_TYPE_DOUBLE);
                    cpl_table_new_column(arr[i], "c_r", CPL_TYPE_DOUBLE);
                    cpl_table_new_column(arr[i], "r_l", CPL_TYPE_DOUBLE);
                    cpl_table_new_column(arr[i], "c_l/r_l", CPL_TYPE_DOUBLE);
                    cpl_table_new_column(arr[i], "left", CPL_TYPE_DOUBLE);
                    cpl_table_new_column(arr[i], "right", CPL_TYPE_DOUBLE);

                    cpl_table_new_column(arr[i], "slitlet_tl", CPL_TYPE_DOUBLE);
                    cpl_table_new_column(arr[i], "slitlet_tc", CPL_TYPE_DOUBLE);
                    cpl_table_new_column(arr[i], "slitlet_tr", CPL_TYPE_DOUBLE);
            }
        }

        // now do the same for the whole detectors in two blocks
        // center to top and center to bottom
        CHECK_ERROR_STATE();
        est_centers = cpl_vector_duplicate(fit_centers_middle);
        for (int j = ix_middle; j < nr_values; j++) {
            center_y = ERIS_IFU_DETECTOR_BP_BORDER + j*height + height/2;
            /* collapse image in y and convert to vector */
            profile_x = eris_ifu_calc_centers_collapse_chunk(warpedImg, center_y,
            		height);
            /* do least square fit using a Gaussian in order to get exact centers */
            fit_centers = eris_ifu_dist_calc_centers_fit(profile_x, est_centers,
            		CPL_FALSE);

            /* copy values into table */
            actual_size = (int) cpl_vector_get_size(fit_centers);
            pfit_centers = cpl_vector_get_data_const(fit_centers);

            if (actual_size == SLITLET_CNT) {
                // one trace per slitlet
                for (int i = 0; i < actual_size; i++) {
                    cpl_table_set_double(arr[i], "c", j, pfit_centers[i]);
                    if (i > 0) {
                       cpl_table_set_double(arr[i], "left", j, (i-1) * swidth);
                       cpl_table_set_double(arr[i], "left", j, i * swidth);
                    }
                }
            } else {
                // three traces per slitlet
                int ii = 0;
                for (int i = 0; i < actual_size; i+=3) {
                    if (i >= 3*SLITLET_CNT) {
                        cpl_msg_warning(cpl_func, "Detected more than 96 traces (%d). Please check output!", actual_size);
                    } else {
                    	cpl_table_set_double(arr[ii], "left", j, ii * swidth);
                    	cpl_table_set_double(arr[ii], "right", j, (ii+1) * swidth);
                            cpl_table_set_double(arr[ii], "c_l", j, pfit_centers[i+1]-ii * swidth);
                            cpl_table_set_double(arr[ii], "c",   j, pfit_centers[i+1]);
                            cpl_table_set_double(arr[ii], "c_r", j, (ii+1) * swidth - pfit_centers[i+1]);
                            cpl_table_set_double(arr[ii], "r_l", j, swidth);
                            cpl_table_set_double(arr[ii], "c_l/r_l", j,
                            		(pfit_centers[i+1]-ii * swidth)/swidth);
                            /* OLD code */
                            cpl_table_set_double(arr[ii], "slitlet_tl", j, pfit_centers[i]);
                            cpl_table_set_double(arr[ii], "slitlet_tc",   j, pfit_centers[i+1]);
                            cpl_table_set_double(arr[ii], "slitlet_tr", j, pfit_centers[i+2]);
                            /* end old code */
                            ii++;
                    }
                }
            }
            BRK_IF_ERROR(cpl_vector_copy(est_centers, fit_centers));
            eris_ifu_free_vector(&profile_x);
            eris_ifu_free_vector(&fit_centers);
       }

        BRK_IF_ERROR(cpl_vector_copy(est_centers, fit_centers_middle));
        for (int j = ix_middle-1; j >= 0; j--) {
            center_y = ERIS_IFU_DETECTOR_BP_BORDER + j*height + height/2;
            /* collapse image in y and convert to vector */
            BRK_IF_NULL(
                    profile_x = eris_ifu_calc_centers_collapse_chunk(
                        warpedImg, center_y, height));
            /* do least square fit using a Gaussian in order to get exact centers */
            BRK_IF_NULL(
                    fit_centers = eris_ifu_dist_calc_centers_fit(profile_x, est_centers, CPL_FALSE));

            BRK_IF_NULL(
                pfit_centers = cpl_vector_get_data_const(fit_centers));

            /* copy values into table */
            if (actual_size == SLITLET_CNT) {
                // one trace per slitlet
                for (int i = 0; i < actual_size; i++) {
                        cpl_table_set_double(arr[i], "c", j, pfit_centers[i]);
                }
            } else {
                // three traces per slitlet
                int ii = 0;
                for (int i = 0; i < actual_size; i+=3) {
                    if (i >= 3*SLITLET_CNT) {
                        cpl_msg_warning(cpl_func, "Detected more than 96 traces (%d). Please check output!", actual_size);
                    } else {
                    	    cpl_table_set_double(arr[ii], "left", j, ii * swidth);
                    	    cpl_table_set_double(arr[ii], "right", j, (ii+1) * swidth);
                            cpl_table_set_double(arr[ii], "c_l", j, pfit_centers[i+1]-ii * swidth);
                            cpl_table_set_double(arr[ii], "c",   j, pfit_centers[i+1]);
                            cpl_table_set_double(arr[ii], "c_r", j, (ii+1) * swidth - pfit_centers[i+1]);
                            cpl_table_set_double(arr[ii], "r_l", j, swidth);
                            cpl_table_set_double(arr[ii], "c_l/r_l", j,
                                                        		(pfit_centers[i+1]-ii * swidth) / swidth);

                            /* OLD code */
                            cpl_table_set_double(arr[ii], "slitlet_tl", j, pfit_centers[i]);
                            cpl_table_set_double(arr[ii], "slitlet_tc",   j, pfit_centers[i+1]);
                            cpl_table_set_double(arr[ii], "slitlet_tr", j, pfit_centers[i+2]);
                            /* end old code */
                        ii++;
                    }
                }
            }
            BRK_IF_ERROR(cpl_vector_copy(est_centers, fit_centers));
            eris_ifu_free_vector(&profile_x);
            eris_ifu_free_vector(&fit_centers);

        }

        char* extname;
        char* keyname;
        char* keycomm;

        // Save warped image as final product
        cpl_propertylist_append_string(pl, CPL_DFS_PRO_CATG,
        		ERIS_IFU_PRO_DIST_DBG_WARPED_FIT);
        cpl_propertylist_set_string(pl,"PIPEFILE",
        		ERIS_IFU_PRO_DIST_QC_WARPED_FIT_FN);
        cpl_propertylist_save(pl, ERIS_IFU_PRO_DIST_QC_WARPED_FIT_FN,
        		CPL_IO_CREATE);

        eris_setup_product_header(ERIS_IFU_PRO_DIST_QC_WARPED_FIT_FN,
        		ERIS_IFU_PRO_DIST_DBG_WARPED_FIT, CPL_FRAME_TYPE_TABLE,
				"eris_ifu_distortion", frameset, parlist, pl);




        for (int i = 0; i < SLITLET_CNT; i++) {

            BRK_IF_NULL(
                pl_my = cpl_propertylist_new());
            extname = cpl_sprintf("SLITLET%2.2d",i);
            cpl_propertylist_append_string(pl_my,"EXTNAME",extname);
//            if (cpl_table_get_ncol(arr[i]) == 1) {
            if (actual_size <= SLITLET_CNT) {
                keyname = cpl_sprintf("ESO QC SLITLET%2.2d C STDEV",i);
                cpl_propertylist_append_double(pl_my, keyname, cpl_table_get_column_stdev(arr[i], "c"));
                keycomm = cpl_sprintf("[pix] center stdev");
                cpl_propertylist_set_comment(pl_my,keyname,keycomm);
                cpl_free(keyname);
                cpl_free(keycomm);


                keyname = cpl_sprintf("ESO QC SLITLET%2.2d C MEDIAN",i);
                cpl_propertylist_append_double(pl_my, keyname, cpl_table_get_column_median(arr[i], "c"));
                keycomm = cpl_sprintf("[pix] center median");
                cpl_propertylist_set_comment(pl_my,keyname,keycomm);
                cpl_free(keyname);
                cpl_free(keycomm);
            } else {
            	keyname = cpl_sprintf("ESO QC SLITLET%2.2d R-L STDEV",i);
                cpl_propertylist_append_double(pl_my, keyname, cpl_table_get_column_stdev(arr[i], "r_l"));
                keycomm = cpl_sprintf("[pix] center-left stdev");
                cpl_propertylist_set_comment(pl_my,keyname,keycomm);
                cpl_free(keyname);
                cpl_free(keycomm);

                keyname = cpl_sprintf("ESO QC SLITLET%2.2d R-L MEDIAN",i);
                cpl_propertylist_append_double(pl_my, keyname, cpl_table_get_column_median(arr[i], "r_l"));
                keycomm = cpl_sprintf("[pix] center-left median");
                cpl_propertylist_set_comment(pl_my,keyname,keycomm);
		        cpl_free(keyname);
		        cpl_free(keycomm);

		        keyname = cpl_sprintf("ESO QC SLITLET%2.2d C-L STDEV",i);
                cpl_propertylist_append_double(pl_my, keyname, cpl_table_get_column_stdev(arr[i], "c_l"));
                keycomm = cpl_sprintf("[pix] center stdev");
                cpl_propertylist_set_comment(pl_my,keyname,keycomm);
                cpl_free(keyname);
                cpl_free(keycomm);

                keyname = cpl_sprintf("ESO QC SLITLET%2.2d C-L MEDIAN",i);
                cpl_propertylist_append_double(pl_my, keyname, cpl_table_get_column_median(arr[i], "c_l"));
                keycomm = cpl_sprintf("[pix] center median");
                cpl_propertylist_set_comment(pl_my,keyname,keycomm);
                cpl_free(keyname);
                cpl_free(keycomm);

                keyname = cpl_sprintf("ESO QC SLITLET%2.2d R-C STDEV",i);
                cpl_propertylist_append_double(pl_my, keyname, cpl_table_get_column_stdev(arr[i], "c_r"));
                keycomm = cpl_sprintf("[pix] center-right stdev");
                cpl_propertylist_set_comment(pl_my,keyname,keycomm);
                cpl_free(keyname);
                cpl_free(keycomm);

                keyname = cpl_sprintf("ESO QC SLITLET%2.2d TL MEDIAN",i);
                cpl_propertylist_append_double(pl_my, keyname, cpl_table_get_column_median(arr[i], "slitlet_tl"));
                keycomm = cpl_sprintf("[pix] trace left median");
                cpl_propertylist_set_comment(pl_my,keyname,keycomm);
                cpl_free(keyname);
                cpl_free(keycomm);

                keyname = cpl_sprintf("ESO QC SLITLET%2.2d TL STDEV",i);
                cpl_propertylist_append_double(pl_my, keyname, cpl_table_get_column_stdev(arr[i], "slitlet_tl"));
                keycomm = cpl_sprintf("[pix] trace left stdev");
                cpl_propertylist_set_comment(pl_my,keyname,keycomm);
                cpl_free(keyname);
                cpl_free(keycomm);

                keyname = cpl_sprintf("ESO QC SLITLET%2.2d TC MEDIAN",i);
                cpl_propertylist_append_double(pl_my, keyname, cpl_table_get_column_median(arr[i], "slitlet_tc"));
                keycomm = cpl_sprintf("[pix] trace center median");
                cpl_propertylist_set_comment(pl_my,keyname,keycomm);
                cpl_free(keyname);
                cpl_free(keycomm);

                keyname = cpl_sprintf("ESO QC SLITLET%2.2d TC STDEV",i);
                cpl_propertylist_append_double(pl_my, keyname, cpl_table_get_column_stdev(arr[i], "slitlet_tc"));
                keycomm = cpl_sprintf("[pix] trace center stdev");
                cpl_propertylist_set_comment(pl_my,keyname,keycomm);
                cpl_free(keyname);
                cpl_free(keycomm);

                keyname = cpl_sprintf("ESO QC SLITLET%2.2d TR MEDIAN",i);
                cpl_propertylist_append_double(pl_my, keyname, cpl_table_get_column_median(arr[i], "slitlet_tr"));
                keycomm = cpl_sprintf("[pix] trace right median");
                cpl_propertylist_set_comment(pl_my,keyname,keycomm);
                cpl_free(keyname);
                cpl_free(keycomm);

                keyname = cpl_sprintf("ESO QC SLITLET%2.2d TR STDEV",i);
                cpl_propertylist_append_double(pl_my, keyname, cpl_table_get_column_stdev(arr[i], "slitlet_tr"));
                keycomm = cpl_sprintf("[pix] trace right stdev");
                cpl_propertylist_set_comment(pl_my,keyname,keycomm);
                cpl_free(keyname);
                cpl_free(keycomm);

                keyname = cpl_sprintf("ESO QC SLITLET%2.2d CL_RL FRAC",i);
                cpl_propertylist_append_double(pl_my, keyname, cpl_table_get_column_median(arr[i], "c_l/r_l"));
                keycomm = cpl_sprintf("(center-left)/(right-left) ratio");
                cpl_propertylist_set_comment(pl_my,keyname,keycomm);
                cpl_free(keyname);
                cpl_free(keycomm);

                keyname = cpl_sprintf("ESO QC SLITLET%2.2d R-C MEDIAN",i);
                cpl_propertylist_append_double(pl_my, keyname, cpl_table_get_column_median(arr[i], "c_r"));
                keycomm = cpl_sprintf("[pix] center-right median");
                cpl_propertylist_set_comment(pl_my,keyname,keycomm);
                cpl_free(keyname);
                cpl_free(keycomm);


            }

            BRK_IF_ERROR(
                cpl_table_save(arr[i], NULL, pl_my,
                               ERIS_IFU_PRO_DIST_QC_WARPED_FIT_FN,
                               CPL_IO_EXTEND));
            eris_ifu_free_propertylist(&pl_my);
            cpl_free(extname);
        }
    }
    CATCH
    {
        CATCH_MSGS();
        err = cpl_error_get_code();
    }

    for (int i = 0; i < SLITLET_CNT; i++) {
         eris_ifu_free_table(&arr[i]);
    }
    cpl_free(arr); arr = NULL;
    eris_ifu_free_vector(&profile_x);
    eris_ifu_free_vector(&est_centers);
    eris_ifu_free_vector(&fit_centers);
    eris_ifu_free_vector(&fit_centers_middle);
    eris_ifu_free_propertylist(&pl_my);
    eris_check_error_code("eris_ifu_dist_warp_stats");
    return err;
}

/**
 * @brief Paste slitlet image into full detector image
 * @param hdrlImgFull  Full detector image (modified in place)
 * @param hdrlImgSlit  Slitlet image to paste
 * @param offset       X offset for paste position
 * @return CPL_ERROR_NONE on success, error code otherwise
 *
 * Copies data, error, and mask from slitlet image into full image at specified offset.
 * Useful for assembling warped image from individual slitlets.
 *
 * @note Can be parallelized if different offsets ensure non-overlapping regions
 * @note Both images must have same Y dimension (ERIS_IFU_DETECTOR_SIZE_Y)
 */
cpl_error_code eris_ifu_image_add_slit(hdrl_image       *hdrlImgFull,
                                       const hdrl_image *hdrlImgSlit,
                                       int              offset)
{
    cpl_error_code  err             = CPL_ERROR_NONE;
    cpl_image        *tmpImgFull        = NULL;
    const cpl_image  *tmpImgSlit        = NULL;
    cpl_mask         *tmpMaskFull       = NULL;
    const cpl_mask   *tmpMaskSlit       = NULL;
    double           *pimgFullData   = NULL,
                     *pimgFullErr    = NULL;
    const double     *pimgSlitData   = NULL,
                     *pimgSlitErr    = NULL;
    cpl_binary       *pimgFullMask   = NULL;
    const cpl_binary *pimgSlitMask   = NULL;
    int              widthFull       = 0,
                     widthSlit       = 0;

    cpl_ensure_code(hdrlImgFull, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(hdrlImgSlit, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(hdrl_image_get_size_y(hdrlImgFull) == ERIS_IFU_DETECTOR_SIZE_Y,
                    CPL_ERROR_ILLEGAL_INPUT);
    cpl_ensure_code(hdrl_image_get_size_y(hdrlImgSlit) == ERIS_IFU_DETECTOR_SIZE_Y,
                    CPL_ERROR_ILLEGAL_INPUT);
    cpl_ensure_code(hdrl_image_get_size_x(hdrlImgFull) >= hdrl_image_get_size_x(hdrlImgSlit),
                    CPL_ERROR_ILLEGAL_INPUT);
    cpl_ensure_code(offset >= 0, CPL_ERROR_ILLEGAL_INPUT);

    TRY
    {
        tmpImgFull = hdrl_image_get_image(hdrlImgFull);
        tmpMaskFull = cpl_image_get_bpm(tmpImgFull);
        BRK_IF_NULL(
            pimgFullData = cpl_image_get_data(tmpImgFull));
        BRK_IF_NULL(
            pimgFullErr = cpl_image_get_data(hdrl_image_get_error(hdrlImgFull)));
        if (tmpMaskFull != NULL) {
            BRK_IF_NULL(
                pimgFullMask = cpl_mask_get_data(tmpMaskFull));
        }
        widthFull = (int) hdrl_image_get_size_x(hdrlImgFull);

        tmpImgSlit = hdrl_image_get_image_const(hdrlImgSlit);
        tmpMaskSlit = cpl_image_get_bpm_const(tmpImgSlit);
        BRK_IF_NULL(
            pimgSlitData = cpl_image_get_data_const(tmpImgSlit));
        BRK_IF_NULL(
            pimgSlitErr = cpl_image_get_data_const(hdrl_image_get_error_const(hdrlImgSlit)));
        if (tmpMaskSlit != NULL) {
            BRK_IF_NULL(
                pimgSlitMask = cpl_mask_get_data_const(tmpMaskSlit));
        }

        widthSlit = (int) hdrl_image_get_size_x(hdrlImgSlit);

        for (int y = 0; y < ERIS_IFU_DETECTOR_SIZE_Y; y++) {
            for (int x = 0; x < widthSlit; x++) {
                pimgFullData[x+offset+y*widthFull] = pimgSlitData[x+y*widthSlit];
                pimgFullErr[x+offset+y*widthFull]  = pimgSlitErr[x+y*widthSlit];
                if ((tmpMaskFull != NULL) && (tmpMaskSlit != NULL)) {
                    pimgFullMask[x+offset+y*widthFull] = pimgSlitMask[x+y*widthSlit];
                }
            }
        }
    }
    CATCH
    {
        CATCH_MSGS();
        err = cpl_error_get_code();
    }

    return err;
}

/**
 * @brief Calculate slitlet edge positions from arc lamp images
 * @param arcImg          Array of arc lamp images
 * @param centers_array   Array of tables with fitted slitlet centers
 * @param valid_arc_lines Table of valid arc lines (from wavelength calibration)
 * @param productDepth    Debug output level
 * @param cut_off_left    Output: TRUE if leftmost slitlet is cut off
 * @param cut_off_right   Output: TRUE if rightmost slitlet is cut off
 * @param frameset        Input frameset
 * @param parlist         Parameter list
 * @return Array of SLITLET_CNT tables with detected edge positions, or NULL on error
 *
 * Detects slit edges for each arc line:
 * 1. For each slitlet, extracts center position
 * 2. For each valid arc line, creates horizontal profile
 * 3. Fits Gaussian to find left and right edges
 * 4. Applies statistical outlier rejection based on slit width
 * 5. Handles edge cases where first/last slitlet extends beyond detector
 *
 * Output tables contain columns:
 * - edge_left: X coordinate of left edge
 * - edge_right: X coordinate of right edge
 * - slit_width: Distance between edges
 * - y_pos: Y coordinate of arc line
 *
 * @note Each table must be freed with cpl_table_delete()
 * @note Array must be freed with cpl_free()
 * @note Cut-off flags help subsequent processing handle edge slitlets correctly
 */
cpl_table** eris_ifu_dist_calc_slitpos(cpl_image   **arcImg,
                                       cpl_table   **centers_array,
                                       cpl_table   *valid_arc_lines,
                                       int         productDepth,
                                       cpl_boolean *cut_off_left,
                                       cpl_boolean *cut_off_right,
									   const cpl_frameset* frameset,
									   const cpl_parameterlist* parlist)
{
    int                 max_lines       = 200,
                        middle          = 0,
                        half_height     =0,
                        urx, ury, llx, lly = 0;
//                        max_pos         = 0;
    double              arc_center      = 0.,
                        percentage      = 0.,
//                        cut_level       = 0.,
//                        max             = -DBL_MAX,
                        median          = 0,
                        stddev          = 0.,
//                        *pprofile_y     = NULL,
                        *pleft_edge_tmp = NULL,
                        *pright_edge_tmp= NULL,
                        *pypos_tmp      = NULL,
                        *pleft_edge     = NULL,
                        *pright_edge    = NULL,
                        *pypos          = NULL;
    char                *extname        = NULL;
    const char *fn = "eris_ifu_distortion_dbg_slitpos_slit_width.fits";
    const char *fnx = "eris_ifu_distortion_dbg_slitpos_profile_xy.fits";
//    cpl_boolean         cut_off_fully   = CPL_FALSE;
    cpl_image           *profile_x      = NULL;
//                        *img_data       = NULL;
    cpl_table           **tbl           = NULL,
                        *valid_arc_lines_slitlet = NULL;
    cpl_vector          *left_edge_tmp  = NULL,
                        *right_edge_tmp = NULL,
                        *ypos_tmp       = NULL,
                        *left_edge      = NULL,
                        *right_edge     = NULL,
                        *ypos           = NULL;
    cpl_propertylist    *pl_tbl         = NULL;
    eris_ifu_vector     *slit_width     = NULL,
                        *slit_width_rej = NULL,
                        *left_edge_rej  = 0,
                        *right_edge_rej = 0;

    cpl_ensure(arcImg, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(*centers_array, CPL_ERROR_NULL_INPUT, NULL);

    TRY
    {
        *cut_off_right = FALSE;
        *cut_off_right = FALSE;

        BRK_IF_NULL(
            left_edge_tmp = cpl_vector_new(max_lines));
        BRK_IF_NULL(
            right_edge_tmp = cpl_vector_new(max_lines));
        BRK_IF_NULL(
            ypos_tmp = cpl_vector_new(max_lines));

        BRK_IF_NULL(
            pleft_edge_tmp = cpl_vector_get_data(left_edge_tmp));
        BRK_IF_NULL(
            pright_edge_tmp = cpl_vector_get_data(right_edge_tmp));
        BRK_IF_NULL(
            pypos_tmp = cpl_vector_get_data(ypos_tmp));

        /* write empty header */
        cpl_frameset* fs = (cpl_frameset*) frameset;
		char* pipe_id = cpl_sprintf("%s%s%s", PACKAGE, "/", PACKAGE_VERSION);
		cpl_propertylist *applist = cpl_propertylist_new();
		cpl_propertylist_update_string(applist, CPL_DFS_PRO_CATG,
				 ERIS_IFU_PRO_DIST_SLIT_POS);
		cpl_dfs_save_propertylist(fs, NULL, parlist, frameset, NULL,
				REC_NAME_DISTORTION, applist, NULL, pipe_id,
											  ERIS_IFU_PRO_DIST_SLIT_POS_FN);
        cpl_propertylist_delete(applist);
        cpl_free(pipe_id);
        CHECK_ERROR_STATE();
       // BRK_IF_ERROR(
       //     cpl_propertylist_save(pl_tbl, ERIS_IFU_PRO_DIST_SLIT_POS_FN, CPL_IO_CREATE));

        BRK_IF_NULL(
            tbl = cpl_calloc(SLITLET_CNT, sizeof(cpl_table*)));

        if (productDepth & 8) {
            eris_ifu_save_image_dbg(NULL, fn, CPL_IO_CREATE, NULL);
            eris_ifu_save_image_dbg(NULL, fnx, CPL_IO_CREATE, NULL);
       }
        /* loop all slitlets */
        for (int i = 0; i < SLITLET_CNT; i++) {
            cpl_msg_debug(cpl_func, "i: %d", i);
            int ttt = 0;
            // take middle value from "x"
            middle = (int) cpl_table_get_nrow(centers_array[i])/2;
            arc_center = cpl_table_get_double(centers_array[i], "x", middle, &ttt);
            CHECK_ERROR_STATE();

            if ((productDepth & 3) != 0) {
                cpl_msg_debug(cpl_func, "   Slitlet #%i: center pos: %g", i+1,
                              arc_center);
            }

            /* extract list of valid lines */
            if (valid_arc_lines != NULL) {
                cpl_table_unselect_all(valid_arc_lines);
                cpl_table_or_selected_int(valid_arc_lines, ERIS_IFU_FITTABLE_SLITLET, CPL_EQUAL_TO, i);
                valid_arc_lines_slitlet = cpl_table_extract_selected(valid_arc_lines);
            }
            /* loop all lines (brighter than a tenth of the brightest line)
             * in vertical profile and detect edges */
            int y        = 0,
                arcImgIx = 0,
                nr_lines = 0;
//            while (y < ERIS_IFU_DETECTOR_SIZE_Y) {
            for (int iii = 0; iii < cpl_table_get_nrow(valid_arc_lines_slitlet); iii++) {
// AA simplify all this!
                y = (int) ( 0.5 + cpl_table_get_double(
                    valid_arc_lines_slitlet, "x0", iii, NULL));
                arcImgIx = cpl_table_get_int(
                    valid_arc_lines_slitlet, "imgIdx", iii, NULL);
//                if (pprofile_y[y] > cut_level) {
                        pypos_tmp[nr_lines] = y;

                        /* create horizontal profile at bright line
                         * (height ~5pix, width 64 + 15pix left and right) */
                        llx = (int) (arc_center+0.5) -64/2-25+1,
                        urx = (int) (arc_center+0.5) +64/2+25+1;
//AA check if better for bad Lines
                        {
                            //old
                            // int half_height = 3 / 2;
                            half_height = 5 / 2,
                            lly         = y - half_height + 1,
                            ury         = y + half_height + 1;
    // y ist the first value, that is above the threshold
    //HIER  in y gaussfit, evtl am besten entlang llx-urx

                            if (llx < 1)
                                llx = 1;
                            if (urx > 2048)
                                urx = 2048;

                            BRK_IF_NULL(
                                profile_x = cpl_image_collapse_window_create(arcImg[arcImgIx],
                                                                         llx, lly,
                                                                         urx, ury, 0));
                        }
//                        {//new
//                            // identify actual ypos in valid_lines2
//                            int valid_line = 0;
//                            if (a > 0) {
//                                valid_line = y;
//                            } else if (b > 0){
//                                valid_line = y+1;
//                            }
//
//                            // get leftmost and rightmost column of this slitlet
//                            cpl_table_unselect_all(valid_arc_lines_slitlet);
//                            cpl_table_or_selected_int(valid_arc_lines_slitlet, ERIS_IFU_FITTABLE_POSITION, CPL_EQUAL_TO, valid_line);
////                            cpl_table *valid_arc_lines3 = cpl_table_extract_selected(valid_arc_lines_slitlet);
//
////                            int min = cpl_table_get_column_min(),
////                                max = cpl_table_get_column_min()
//
//
//                            // get the according y-values
//
//                            // create a line accross the detector from llx to urx
//
//                            // create a profile in a way that +/-3or5 pix vertically around the line are averaged and put into profile_x
//                        }

                        cpl_error_code status = eris_ifu_slitpos_gauss(
                            profile_x,
                            &(pleft_edge_tmp[nr_lines]),
                            &(pright_edge_tmp[nr_lines]),
                            llx-1,
                            productDepth);
                        if (status == CPL_ERROR_EOL) {
                            RECOVER();
                        }

                        pleft_edge_tmp[nr_lines] -= 1.;
                        pright_edge_tmp[nr_lines] -= 1.;

                        if (productDepth & 8) {
                            cpl_propertylist    *pl_line    = NULL;
                            BRK_IF_NULL(
                                pl_line = cpl_propertylist_new());
                            extname = cpl_sprintf("Sx_%02d  Lx_%02d", i+1, nr_lines+1);
                            cpl_propertylist_append_string(pl_line, "EXTNAME", extname);
                            eris_ifu_free_string(&extname);
                            cpl_propertylist_append_double(pl_line, "lineypos",      y);
                            cpl_propertylist_append_double(pl_line, "llx",    llx);
                            cpl_propertylist_append_double(pl_line, "lly",    lly);
                            cpl_propertylist_append_double(pl_line, "urx",   urx);
                            cpl_propertylist_append_double(pl_line, "ury",   ury);
                            cpl_propertylist_append_double(pl_line, "edge_l",  pleft_edge_tmp[nr_lines]);
                            cpl_propertylist_append_double(pl_line, "edge_r",  pright_edge_tmp[nr_lines]);
                            eris_ifu_save_image_dbg(profile_x, fnx, CPL_IO_EXTEND, pl_line);
                            eris_ifu_free_propertylist(&pl_line);
                        }
                        eris_ifu_free_image(&profile_x);

                        if (status == CPL_ERROR_EOL) {
                            continue;
                        } else if (status != CPL_ERROR_NONE) {
                            ERIS_IFU_TRY_EXIT();
                        }
                        nr_lines++;
                        if (nr_lines >= max_lines) {
                            SET_ERROR_MSG(CPL_ERROR_ILLEGAL_INPUT,
                                                       "more than 100 lines detected!");
/*erw
    //                    double est_pos = pleft_edge[nr_lines] + (pright_edge[nr_lines]-pleft_edge[nr_lines])/2.0;
    //                    cpl_msg_debug(cpl_func, "        left_pos: %g, right_pos: %g", pleft_edge[nr_lines], pright_edge[nr_lines]);
    //                    cpl_msg_debug(cpl_func, "        est. pos: %g", est_pos);
    //                    // this is the eff. offset from slitlet center to fibre-center
    //                    cpl_msg_debug(cpl_func, "        diff: %g", arc_center-est_pos);

                        eris_ifu_free_image(&profile_x);

                        // get off surroundings of already processed line
                        y+=5;
                        // found line
                        nr_lines++;
                        if (nr_lines >= max_lines) {
                            SET_ERROR_MSG(CPL_ERROR_ILLEGAL_INPUT,
                                                       "more than 100 lines detected!");
*/
                        }

//                } // end: if detected line
//                y++;
            } // end: while y

            eris_ifu_free_table(&valid_arc_lines_slitlet);

            /* cut vectors to their correct length */
            BRK_IF_NULL(
                left_edge = cpl_vector_extract(left_edge_tmp, 0, nr_lines-1, 1));
            BRK_IF_NULL(
                right_edge = cpl_vector_extract(right_edge_tmp, 0, nr_lines-1, 1));
            BRK_IF_NULL(
                ypos = cpl_vector_extract(ypos_tmp, 0, nr_lines-1, 1));
            BRK_IF_NULL(
                pleft_edge = cpl_vector_get_data(left_edge));
            BRK_IF_NULL(
                pright_edge = cpl_vector_get_data(right_edge));
            BRK_IF_NULL(
                pypos = cpl_vector_get_data(ypos));

            slit_width = eris_ifu_vector_new(nr_lines);
            for (int ty = 0; ty < nr_lines; ty++) {
                eris_ifu_vector_set(slit_width, ty, pright_edge[ty]-pleft_edge[ty]);
            }

            /*
             * omit outliers in a multi process step
            */
            BRK_IF_NULL(
                pl_tbl = cpl_propertylist_new());

            /* status before rejecting */
            median = eris_ifu_vector_get_median(slit_width, ERIS_IFU_ARITHMETIC);
            stddev = eris_ifu_vector_get_stdev_median(slit_width);
            CHECK_ERROR_STATE();
            BRK_IF_ERROR(
                cpl_propertylist_append_double(pl_tbl, "bef_med", median));
            BRK_IF_ERROR(
                cpl_propertylist_append_double(pl_tbl, "bef_sdv", stddev));
            cpl_msg_debug(cpl_func, "before: median: %g, stddev: %g", median, stddev);
            if (productDepth & 8) {
                cpl_propertylist *pl = NULL;
                BRK_IF_NULL(
                    pl = cpl_propertylist_new());
                extname = cpl_sprintf("Sx_%02d ypos", i+1);
                cpl_propertylist_append_string(pl, "EXTNAME", extname);
                eris_ifu_free_string(&extname);
                BRK_IF_ERROR(
                    eris_ifu_save_vector_dbg(ypos, fn, CPL_IO_EXTEND, pl));
                extname = cpl_sprintf("Sx_%02d left_edge", i+1);
                cpl_propertylist_append_string(pl, "EXTNAME", extname);
                eris_ifu_free_string(&extname);
                BRK_IF_ERROR(
                    eris_ifu_save_vector_dbg(left_edge, fn, CPL_IO_EXTEND, pl));
                extname = cpl_sprintf("Sx_%02d right_edge", i+1);
                cpl_propertylist_append_string(pl, "EXTNAME", extname);
                eris_ifu_free_string(&extname);
                BRK_IF_ERROR(
                    eris_ifu_save_vector_dbg(right_edge, fn, CPL_IO_EXTEND, pl));
                extname = cpl_sprintf("Sx_%02d slit_width", i+1);
                cpl_propertylist_append_string(pl_tbl, "EXTNAME", extname);
                eris_ifu_free_string(&extname);
                BRK_IF_ERROR(
                    eris_ifu_vector_save(slit_width, fn, CPL_TYPE_DOUBLE,
                        pl_tbl, CPL_IO_EXTEND, 70));
                eris_ifu_free_propertylist(&pl);
            }

            BRK_IF_NULL(
                left_edge_rej = eris_ifu_vector_create(left_edge));
            BRK_IF_NULL(
                right_edge_rej = eris_ifu_vector_create(right_edge));
            BRK_IF_NULL(
                slit_width_rej = eris_ifu_vector_duplicate(slit_width));

            if ((median < 58) && (stddev > 3) &&
                ((i == 0) || (i == SLITLET_CNT-1)))
            {
                // median is obviously so wrong, that many values indicate that slitlet is cut off!
                // reject values < SLITLET_WIDTH-stddev
                for (int ty = 0; ty < nr_lines; ty++) {
                    double val = eris_ifu_vector_get(slit_width_rej, ty);
                    if (val < SLITLET_WIDTH-stddev) {
                        eris_ifu_vector_reject(slit_width_rej, ty);
                        eris_ifu_vector_reject(left_edge_rej, ty);
                        eris_ifu_vector_reject(right_edge_rej, ty);
                    }
                }
if (eris_ifu_vector_count_non_rejected(slit_width_rej) <= 1) {
//    cut_off_fully = CPL_TRUE;
    eris_ifu_free_ifu_vector(&left_edge_rej);
    eris_ifu_free_ifu_vector(&right_edge_rej);
    eris_ifu_free_ifu_vector(&slit_width_rej);
    BRK_IF_NULL(
        left_edge_rej = eris_ifu_vector_create(left_edge));
    BRK_IF_NULL(
        right_edge_rej = eris_ifu_vector_create(right_edge));
    BRK_IF_NULL(
        slit_width_rej = eris_ifu_vector_duplicate(slit_width));
} else {
                median = eris_ifu_vector_get_median(slit_width_rej, ERIS_IFU_ARITHMETIC);
                stddev = eris_ifu_vector_get_stdev_median(slit_width_rej);
                CHECK_ERROR_STATE();

                BRK_IF_ERROR(
                    cpl_propertylist_append_double(pl_tbl, "med_rej1", median));
                BRK_IF_ERROR(
                    cpl_propertylist_append_double(pl_tbl, "sdv_rej1", stddev));
                cpl_msg_debug(cpl_func, "median: %g, stddev: %g", median, stddev);
                if (productDepth & 8) {
                    extname = cpl_sprintf("Sx_%02d slit_width_rej", i+1);
                    cpl_propertylist_append_string(pl_tbl, "EXTNAME", extname);
                    eris_ifu_free_string(&extname);
                    BRK_IF_ERROR(
                        eris_ifu_vector_save(slit_width_rej, fn, CPL_TYPE_DOUBLE, pl_tbl, CPL_IO_EXTEND, 70));
                }
}
                if (i == 0) {
                    *cut_off_left = CPL_TRUE;
                    BRK_IF_ERROR(
                        cpl_propertylist_append_string(pl_tbl, "cutoff", "left"));
                }
                if (i == SLITLET_CNT-1) {
                    *cut_off_right = CPL_TRUE;
                    BRK_IF_ERROR(
                        cpl_propertylist_append_string(pl_tbl, "cutoff", "right"));
                }
            }

            /* omit outliers devying more than 2% (15% for problematic slitlet #16)*/
            percentage = 1.02;
            if (i == 15 || i == 17) {
                percentage = 1.15;
            }
            for (int ty = 0; ty < nr_lines; ty++) {
                double val = eris_ifu_vector_get(slit_width_rej, ty);
                if ((val > median*percentage) || (val < median/percentage)) {
                    eris_ifu_vector_reject(slit_width_rej, ty);
                    eris_ifu_vector_reject(left_edge_rej, ty);
                    eris_ifu_vector_reject(right_edge_rej, ty);
                }
            }

            median = eris_ifu_vector_get_median(slit_width_rej, ERIS_IFU_ARITHMETIC);
            BRK_IF_ERROR(
                cpl_propertylist_append_double(pl_tbl, "aft_med", median));
            if (eris_ifu_vector_count_non_rejected(slit_width_rej) >= 3) {
                stddev = eris_ifu_vector_get_stdev_median(slit_width_rej);
                BRK_IF_ERROR(
                    cpl_propertylist_append_double(pl_tbl, "aft_sdv", stddev));
            }
            CHECK_ERROR_STATE();
            cpl_msg_debug(cpl_func, "after:  median: %g, stddev: %g", median, stddev);
            if (productDepth & 8) {
                extname = cpl_sprintf("Sx_%02d slit_width_rej2", i+1);
                cpl_propertylist_append_string(pl_tbl, "EXTNAME", extname);
                eris_ifu_free_string(&extname);
                BRK_IF_ERROR(
                    eris_ifu_vector_save(slit_width_rej, fn, CPL_TYPE_DOUBLE, pl_tbl, CPL_IO_EXTEND, 70));
            }

            /*
             * check if first or last slitlet is off detector (partially cut off)
             * (2nd test)
             */
            if ((i == 0) && ((*cut_off_left == CPL_TRUE) ||
                 (eris_ifu_vector_get_min(left_edge_rej, NULL) < ERIS_IFU_DETECTOR_BP_BORDER)))
            {
                *cut_off_left = CPL_TRUE;
                if (!cpl_propertylist_has(pl_tbl, "cutoff")) {
                    BRK_IF_ERROR(
                        cpl_propertylist_append_string(pl_tbl, "cutoff", "left"));
                }

                // reset slit_width
                eris_ifu_free_ifu_vector(&slit_width_rej);
                BRK_IF_NULL(
                    slit_width_rej = eris_ifu_vector_duplicate(slit_width));

                eris_ifu_vector *x = NULL;
                x = eris_ifu_vector_duplicate(slit_width_rej);
                for (int j = 0; j < nr_lines; j++) {
                    eris_ifu_vector_set(x, j, j+1);
                }
                cpl_vector *fff = eris_ifu_polyfit_edge(x, slit_width_rej, 1);
                eris_ifu_free_ifu_vector(&x);

                double fp0 = cpl_vector_get(fff, 0),
                       fp1 = cpl_vector_get(fff, 1);
                eris_ifu_free_vector(&fff);
                stddev = eris_ifu_vector_get_stdev_median(slit_width_rej);
                for (int ty = 0; ty < nr_lines; ty++) {
                    double val = eris_ifu_vector_get(slit_width_rej, ty);
                    double val_calc = fp0+fp1*(ty+1);
                    if (fabs(val - val_calc) > stddev*0.7) { // stddev is a bit too high
                        eris_ifu_vector_reject(slit_width_rej, ty);
                    }
                }
                if (productDepth & 8) {
                    extname = cpl_sprintf("Sx_%02d slit_width_rej3", i+1);
                    cpl_propertylist_append_string(pl_tbl, "EXTNAME", extname);
                    eris_ifu_free_string(&extname);
                   BRK_IF_ERROR(
                        eris_ifu_vector_save(slit_width_rej, fn, CPL_TYPE_DOUBLE, pl_tbl,
                                         CPL_IO_EXTEND, 70));
                }
            } // if left_edge cut off

            if ((i == SLITLET_CNT-1) && ((*cut_off_right == CPL_TRUE) ||
                 (eris_ifu_vector_get_max(right_edge_rej, NULL) > ERIS_IFU_DETECTOR_SIZE_X-ERIS_IFU_DETECTOR_BP_BORDER)))
            {
                *cut_off_right = TRUE;
                if (!cpl_propertylist_has(pl_tbl, "cutoff")) {
                    BRK_IF_ERROR(
                        cpl_propertylist_append_string(pl_tbl, "cutoff", "right"));
                }

                // reset slit_width
                eris_ifu_free_ifu_vector(&slit_width_rej);
                BRK_IF_NULL(
                    slit_width_rej = eris_ifu_vector_duplicate(slit_width));

                eris_ifu_vector *x = NULL;
                x = eris_ifu_vector_duplicate(slit_width_rej);
                for (int j = 0; j < nr_lines; j++) {
                    eris_ifu_vector_set(x, j, j+1);
                }
                cpl_vector *fff = eris_ifu_polyfit_edge(x, slit_width_rej, 1);
                eris_ifu_free_ifu_vector(&x);

                double fp0 = cpl_vector_get(fff, 0),
                       fp1 = cpl_vector_get(fff, 1);
                eris_ifu_free_vector(&fff);
                stddev = eris_ifu_vector_get_stdev_median(slit_width_rej);
                CHECK_ERROR_STATE();
                for (int ty = 0; ty < nr_lines; ty++) {
                    double val = eris_ifu_vector_get(slit_width_rej, ty);
                    double val_calc = fp0+fp1*(ty+1);
                    if (fabs(val - val_calc) > stddev*0.7) { // stddev is a bit too high
                        BRK_IF_ERROR(
                            eris_ifu_vector_reject(slit_width_rej, ty));
                    }
                }
                if (productDepth & 8) {
                    extname = cpl_sprintf("Sx_%02d slit_width_rej4", i+1);
                    cpl_propertylist_append_string(pl_tbl, "EXTNAME", extname);
                    eris_ifu_free_string(&extname);
                    BRK_IF_ERROR(
                        eris_ifu_vector_save(slit_width_rej, fn, CPL_TYPE_DOUBLE, pl_tbl,
                                             CPL_IO_EXTEND, 70));
                }
            } // if right_edge cut off

            eris_ifu_free_ifu_vector(&left_edge_rej);
            eris_ifu_free_ifu_vector(&right_edge_rej);
            CHECK_ERROR_STATE();

            /* setup table to save */
            BRK_IF_NULL(
                tbl[i] = cpl_table_new(eris_ifu_vector_count_non_rejected(slit_width_rej)));
            BRK_IF_ERROR(
                cpl_table_new_column(tbl[i], ERIS_IFU_DIST_EDGE_L, CPL_TYPE_DOUBLE));
            BRK_IF_ERROR(
                cpl_table_new_column(tbl[i], ERIS_IFU_DIST_EDGE_R, CPL_TYPE_DOUBLE));
            BRK_IF_ERROR(
                cpl_table_new_column(tbl[i], ERIS_IFU_DIST_SLIT, CPL_TYPE_DOUBLE));
            BRK_IF_ERROR(
                cpl_table_new_column(tbl[i], ERIS_IFU_DIST_YPOS, CPL_TYPE_DOUBLE));

            // copy vectors to table
            int cnt = 0;
            for (int ty = 0; ty < cpl_vector_get_size(ypos); ty++) {
                if (!eris_ifu_vector_is_rejected(slit_width_rej, ty)) {
                    // add to table only if line hasn't been rejected
                    BRK_IF_ERROR(
                        cpl_table_set_double(tbl[i], ERIS_IFU_DIST_YPOS,   cnt, pypos[ty]));
                    BRK_IF_ERROR(
                        cpl_table_set_double(tbl[i], ERIS_IFU_DIST_EDGE_L, cnt, pleft_edge[ty]));
                    BRK_IF_ERROR(
                        cpl_table_set_double(tbl[i], ERIS_IFU_DIST_EDGE_R, cnt, pright_edge[ty]));
                    BRK_IF_ERROR(
                        cpl_table_set_double(tbl[i], ERIS_IFU_DIST_SLIT,   cnt, eris_ifu_vector_get(slit_width_rej, ty)));

                    if ((i == 0) && (*cut_off_left == TRUE)) {
                        // set invalid values to nan on left side
                        if (pleft_edge[ty] < ERIS_IFU_DETECTOR_BP_BORDER) {
                            BRK_IF_ERROR(
                                cpl_table_set_double(tbl[i], ERIS_IFU_DIST_EDGE_L, cnt, NAN));
                            BRK_IF_ERROR(
                                cpl_table_set_double(tbl[i], ERIS_IFU_DIST_SLIT,   cnt, pright_edge[ty]-ERIS_IFU_DETECTOR_BP_BORDER));
                        }
                    }
                    if ((i == SLITLET_CNT-1) && (*cut_off_right == TRUE)) {
                        // set invalid values to nan on right side
                        if (pright_edge[ty] > ERIS_IFU_DETECTOR_SIZE_X-ERIS_IFU_DETECTOR_BP_BORDER) {
                            BRK_IF_ERROR(
                                cpl_table_set_double(tbl[i], ERIS_IFU_DIST_EDGE_R, cnt, NAN));
                            BRK_IF_ERROR(
                                cpl_table_set_double(tbl[i], ERIS_IFU_DIST_SLIT,   cnt, ERIS_IFU_DETECTOR_SIZE_X - ERIS_IFU_DETECTOR_BP_BORDER - pleft_edge[ty]));
                        }
                    }

                    cnt++;
                }
            }

            extname = cpl_sprintf("Sx_%02d", i+1);
            cpl_propertylist_append_string(pl_tbl, "EXTNAME", extname);
            eris_ifu_free_string(&extname);
            BRK_IF_ERROR(
                cpl_table_save(tbl[i], NULL, pl_tbl, ERIS_IFU_PRO_DIST_SLIT_POS_FN, CPL_IO_EXTEND));

            eris_ifu_free_propertylist(&pl_tbl);
            eris_ifu_free_ifu_vector(&slit_width);
            eris_ifu_free_ifu_vector(&slit_width_rej);
            eris_ifu_free_vector(&left_edge);
            eris_ifu_free_vector(&right_edge);
            eris_ifu_free_vector(&ypos);
        } // end: for i = SLITLET_CNT
    }
    CATCH
    {
        CATCH_MSGS();

        eris_ifu_free_image(&profile_x);
        for (int i = 0; i < SLITLET_CNT; i++) {
            eris_ifu_free_table(&tbl[i]);
        }
        cpl_free(tbl); tbl = NULL;
    }

    eris_ifu_free_vector(&left_edge_tmp);
    eris_ifu_free_vector(&right_edge_tmp);
    eris_ifu_free_vector(&ypos_tmp);
    eris_ifu_free_vector(&left_edge);
    eris_ifu_free_vector(&right_edge);
    eris_ifu_free_vector(&ypos);
    eris_ifu_free_propertylist(&pl_tbl);
    eris_ifu_free_ifu_vector(&left_edge_rej);
    eris_ifu_free_ifu_vector(&right_edge_rej);

    return tbl;
}

/**
 * @brief Fit Gaussian to find peak center and width
 * @param x       X coordinates
 * @param y       Y values (intensities)
 * @param x0      Output: center position
 * @param sigma   Output: Gaussian width (positive on success)
 * @param area    Output: Gaussian area (positive on success)
 * @param offset  Output: background offset level
 * @return CPL_ERROR_NONE on success, error code otherwise
 *
 * Performs iterative Gaussian fitting:
 * 1. First attempt: fit all 4 parameters (x0, sigma, area, offset)
 * 2. If first fit doesn't converge, fix sigma and area, refit x0 and offset
 * 3. Returns best-fit parameters even if convergence not achieved
 *
 * Gaussian model: y = offset + (area / sqrt(2*pi*sigma^2)) * exp(-(x-x0)^2 / (2*sigma^2))
 *
 * @note Returns -1 for all parameters if fitting completely fails
 * @note Convergence issues are handled gracefully; partial results may be returned
 */
cpl_error_code eris_ifu_fit_gauss(const cpl_vector *x,
                                  const cpl_vector *y,
                                  double *x0,
                                  double *sigma,
                                  double *area,
                                  double *offset)
{
    cpl_error_code  ret_error   = CPL_ERROR_NONE,
                    fit_error   = CPL_ERROR_NONE;


    cpl_ensure_code(x,      CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(y,      CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(x0,     CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(area,   CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(sigma,  CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(offset, CPL_ERROR_NULL_INPUT);

    TRY
    {
        *x0     = 0.;
        *sigma  = 0.;
        *area   = 0.;
        *offset = 0.;

        fit_error = cpl_vector_fit_gaussian(x, NULL, y, NULL,
                                            CPL_FIT_ALL,
                                            x0,
                                            sigma,
                                            area,
                                            offset,
                                            NULL, NULL, NULL);

//        cpl_msg_debug(cpl_func, "             --- 1st FIT ---------------------"
//                                "-------------");
//        cpl_msg_debug(cpl_func, "             center: %g  (sigma: %g, area: %g,"
//                                " offset: %g)", *x0, *sigma, *area, *offset);

        // this happens only once in obscure test data...
        if ((fit_error == CPL_ERROR_NONE) &&
            (cpl_error_get_code() == CPL_ERROR_SINGULAR_MATRIX))
        {
            cpl_error_reset();
            fit_error = CPL_ERROR_CONTINUE;
        }

        if (fit_error == CPL_ERROR_CONTINUE) {
            // if first fit doesn't convert, try it again with fixed
            // area- and sigma-parameter
            RECOVER();

            fit_error = cpl_vector_fit_gaussian(x, NULL, y, NULL,
                                                CPL_FIT_CENTROID | CPL_FIT_OFFSET,
                                                x0,
                                                sigma,
                                                area,
                                                offset,
                                                NULL, NULL, NULL);

//            cpl_msg_debug(cpl_func, "             --- 2nd FIT -----------------"
//                                    "-----------------");
//            cpl_msg_debug(cpl_func, "             center: %g  (sigma: %g, "
//                                    "area: %g, offset: %g)",
//                                    *x0, *sigma, *area, *offset);

            if (fit_error == CPL_ERROR_CONTINUE) {
                // if it didn't convert again, give up and take the
                // estimated value
                RECOVER();
            }
        }
        CHECK_ERROR_STATE();
    }
    CATCH
    {
        ret_error = cpl_error_get_code();

        *x0  = -1;
        *sigma  = -1;
        *area  = -1;
        *offset  = -1;
    }

    return ret_error;
}

/**
 * @brief Iterative polynomial fitting with outlier rejection
 * @param x          X coordinates
 * @param y          Y values to fit
 * @param fit_order  Polynomial order (1, 2, or 3)
 * @return Vector of polynomial coefficients [a0, a1, a2, ...], or NULL on error
 *
 * Performs robust polynomial fitting:
 * 1. Fit polynomial to all data
 * 2. Calculate residuals
 * 3. Reject points with residuals > median + 5*stddev
 * 4. Repeat 3 times total
 *
 * Polynomial form: y(x) = a0 + a1*x + a2*x^2 + a3*x^3
 *
 * @note Handles rejected values in input eris_ifu_vector structures
 * @note Returned vector must be freed with cpl_vector_delete()
 * @note Rejection mask in x and y is synchronized
 */
cpl_vector* eris_ifu_polyfit_edge(const eris_ifu_vector *x,
                                  const eris_ifu_vector *y,
                                  int                   fit_order)
{
    eris_ifu_vector *x_dup              = NULL,
                    *y_dup              = NULL,
                    *fit_vec            = NULL;
    cpl_vector      *fit_par            = NULL,
                    *x_good             = NULL,
                    *y_good             = NULL;
    double          *pfit_par           = NULL,
                    *pfdata             = NULL,
                    *pfmask             = NULL,
                    stddev              = 0.0,
                    median              = 0.0,
                    *pxdata             = NULL,
                    *pxmask             = NULL,
                    *pymask             = NULL;
    int             size                = 0,
                    iter                = 3,
                    i                   = 0,
                    j                   = 0;

    cpl_ensure(x, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(y, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(fit_order >= 1, CPL_ERROR_ILLEGAL_INPUT, NULL);
    cpl_ensure(cpl_vector_get_size(x->data) == cpl_vector_get_size(y->data), CPL_ERROR_ILLEGAL_INPUT, NULL);

    TRY
    {
        // assert that rejected values in x are also rejected in y and vice versa
        BRK_IF_NULL(
            x_dup = eris_ifu_vector_duplicate(x));
        BRK_IF_NULL(
            y_dup = eris_ifu_vector_duplicate(y));
        BRK_IF_ERROR(
            eris_ifu_vector_adapt_rejected(x_dup, y_dup));

        size = (int) cpl_vector_get_size(x_dup->data);

        BRK_IF_NULL(
            pxdata = cpl_vector_get_data(x_dup->data));
        BRK_IF_NULL(
            pxmask = cpl_vector_get_data(x_dup->mask));
        BRK_IF_NULL(
            pymask = cpl_vector_get_data(y_dup->mask));

        BRK_IF_NULL(
            fit_vec = eris_ifu_vector_new(size));
        BRK_IF_NULL(
            pfdata = cpl_vector_get_data(fit_vec->data));
        BRK_IF_NULL(
            pfmask = cpl_vector_get_data(fit_vec->mask));

        // iterate fitting
        for (i = 0; i < iter; i++)
        {
            cpl_vector_delete(fit_par); fit_par = NULL;

            // fit
            BRK_IF_NULL(
                x_good = eris_ifu_vector_create_non_rejected(x_dup));
            BRK_IF_NULL(
                y_good = eris_ifu_vector_create_non_rejected(y_dup));

            BRK_IF_NULL(
                fit_par = eris_ifu_polyfit_1d(x_good, y_good, fit_order));
            BRK_IF_NULL(
                pfit_par = cpl_vector_get_data(fit_par));

            eris_ifu_free_vector(&x_good);
            eris_ifu_free_vector(&y_good);

            // create fitted vector
            for (j = 0; j < size; j++) {
                if (eris_ifu_vector_is_rejected(x_dup, j)) {
                    BRK_IF_ERROR(
                        eris_ifu_vector_reject(fit_vec, j));
                } else {
                    double fit_val = -pfit_par[0];
                    if (fit_order >= 1) {
                        fit_val -= pfit_par[1] * pxdata[j];
                    }
                    if (fit_order >= 2) {
                        fit_val -= pfit_par[2] * pow(pxdata[j], 2);
                    }
                    if (fit_order >= 3) {
                        fit_val -= pfit_par[3] * pow(pxdata[j], 3);
                    }

                    BRK_IF_ERROR(
                        eris_ifu_vector_set(fit_vec, j, fit_val));
                }
            }

            // fit_vec = y_good - fit_vec
            BRK_IF_ERROR(
                eris_ifu_vector_add(fit_vec, y_dup));

            median = eris_ifu_vector_get_median(fit_vec, ERIS_IFU_ARITHMETIC);
            stddev = eris_ifu_vector_get_stdev(fit_vec);
            CHECK_ERROR_STATE();

            BRK_IF_ERROR(
                eris_ifu_vector_abs(fit_vec));

            // clip values larger than median + 5 * stddev
            double clip_val = median + 5 * stddev;
            for (j = 0; j < size; j++) {
                if ((pfmask[j] >= 0.5) && (pfdata[j] > clip_val)) {
                    pfmask[j] = 0.;
                    pxmask[j] = 0.;
                    pymask[j] = 0.;
                }
            }
        } // for i = iter
    }
    CATCH
    {
        CATCH_MSGS();
        eris_ifu_free_vector(&fit_par);
    }

    eris_ifu_free_ifu_vector(&fit_vec);
    eris_ifu_free_ifu_vector(&x_dup);
    eris_ifu_free_ifu_vector(&y_dup);
    eris_ifu_free_vector(&x_good);
    eris_ifu_free_vector(&y_good);

    return fit_par;
}

/**
 * @brief Process arc lamp images for distortion calibration
 * @param frames                  Input frameset
 * @param exposureCorrectionMode  Exposure correction flags
 * @param arcImgCnt              Output: number of arc images
 * @param arcImages              Output: HDRL imagelist of arc images
 * @param lampStates             Output: array of lamp state flags
 * @param band                   Output: spectral band
 * @param scale                  Output: preoptics scale
 * @param instrument             Output: instrument identifier
 * @param saturation_threhold    Saturation threshold for QC
 * @param qclog                  Output: QC log table
 * @return CPL_ERROR_NONE on success, error code otherwise
 *
 * Loads and processes arc lamp frames:
 * - Identifies ON and OFF frames
 * - Performs exposure corrections
 * - Combines multiple lamp images if needed
 * - Extracts instrument configuration
 *
 * @note Similar to eris_ifu_wave_get_arc_images() but tailored for distortion
 * @note Caller must free arcImages, lampStates
 */
cpl_error_code eris_ifu_wavecal_processSof_dist(
        cpl_frameset* frames,
        int exposureCorrectionMode,
        int *arcImgCnt,
        hdrl_imagelist **arcImages,
        int **lampStates,
        ifsBand *band,
        ifsPreopticsScale *scale,
        ifsInstrument *instrument,
		double saturation_threhold,
		cpl_table** qclog
        )
{
    cpl_frame *frame = NULL;

    TRY
    {
        if (frames == NULL) {
            BRK_WITH_ERROR_MSG(CPL_ERROR_NULL_INPUT,
                "missing frameset");
        }
        if (cpl_frameset_is_empty(frames)) {
            BRK_WITH_ERROR_MSG(CPL_ERROR_NULL_INPUT,
                            "SOF file is empty or missing");
        }
        CHECK_ERROR_STATE();

//AA only change: ERIS_IFU_RAW_WAVE_NS instead of ERIS_IFU_RAW_WAVE_LAMP
        // get arc lamp image
        cpl_frameset *arcFrames = NULL;
        if (cpl_frameset_count_tags(frames, ERIS_IFU_RAW_WAVE_NS) > 0) {
        	arcFrames = eris_ifu_get_frameset_by_tag(frames, ERIS_IFU_RAW_WAVE_NS);
        } else if ( (cpl_frameset_count_tags(frames, ERIS_IFU_RAW_WAVE_NS_ON) > 0) &&
        		(cpl_frameset_count_tags(frames, ERIS_IFU_RAW_WAVE_NS_OFF) > 0) ) {
        	cpl_frameset *arcFrames_off = NULL;
        	arcFrames = eris_ifu_get_frameset_by_tag(frames, ERIS_IFU_RAW_WAVE_NS_ON);

        	arcFrames_off = eris_ifu_get_frameset_by_tag(frames, ERIS_IFU_RAW_WAVE_NS_OFF);

        	cpl_frameset_join(arcFrames, arcFrames_off);

        	cpl_frameset_delete(arcFrames_off);
        }
        BRK_IF_ERROR(
                eris_ifu_wave_get_arc_images(arcFrames,
                    exposureCorrectionMode,
                    arcImgCnt,
                    arcImages,
                    lampStates,
                    band,
                    scale,
                    instrument,
					saturation_threhold,
					qclog));
        cpl_frameset_delete(arcFrames);

        // get reference lines
        frame = cpl_frameset_find(frames, ERIS_IFU_CALIB_REF_LINES);
        CHECK_ERROR_STATE();
        if (frame == NULL) {
            BRK_WITH_ERROR_MSG(CPL_ERROR_NULL_INPUT,
                "missing \"%s\" tag in the SOF, arc lamp reference lines",
                ERIS_IFU_CALIB_REF_LINES);
        }
    } CATCH
    {
//    	CATCH_MSGS();
    }

    return cpl_error_get_code();
}

/**
 * @brief Filter valid arc lines by intensity and fit quality
 * @param tbl       Table of fitted arc lines (modified in place)
 * @param band      Spectral band (affects thresholds)
 * @param nr_cols   Number of columns per slitlet
 * @return CPL_ERROR_NONE on success, error code otherwise
 *
 * Applies multiple filtering criteria:
 * 1. Removes lines with area below band- and slitlet-specific threshold
 * 2. Removes lines with intensity below threshold
 * 3. Removes lines that failed fitting in >50% of columns
 * 4. Removes lines that appear only once (likely spurious)
 * 5. Keeps only center column for multi-column fits
 * 6. Removes lines with non-zero error codes
 *
 * @note Modifies table in place by erasing rows
 * @note Thresholds vary by band and slitlet (slitlet #16 has lower thresholds)
 */
cpl_error_code eris_ifu_distortion_reduce_lines(cpl_table *tbl,
                                                ifsBand band,
                                                int nr_cols)
{
    cpl_error_code  err                 = CPL_ERROR_NONE;
    int             cut_level_area      = 0,
                    cut_level_intensity = 0;
    const char      *bandStr            = NULL;

    cpl_ensure_code(tbl != NULL, CPL_ERROR_NULL_INPUT);

    TRY
    {
        BRK_IF_NULL(
            bandStr = eris_ifu_get_bandString(band));


//        // cut again lower (at a tenth) for slitlet #16
//        CHECK_ERROR_STATE();
//
//        CHECK_ERROR_STATE();

        // get rid of lines where for a specific wavelength in all columns the fit is bad
        cpl_table   *tSlit  = NULL,
                    *tCol   = NULL,
                    *tPos   = NULL;
        cpl_msg_debug(cpl_func, "Removing bad lines");
        for (int sIdx = 0; sIdx < SLITLET_CNT; sIdx++) {
cpl_msg_debug(cpl_func, "  sIdx: %d", sIdx);

            // get rid of lines where area is smaller than a band- and slitlet specific cut-level
            if ('J' == bandStr[0]) {
                if (sIdx != 16-1) {
                    cut_level_area = 1000;
                    cut_level_intensity = 600;
                } else {
                    cut_level_area = 90; // ??
                    cut_level_intensity = 30; // ??
                }
            } else if ('H' == bandStr[0]) {
                if (sIdx != 16-1) {
                    cut_level_area = 300; // ??
                    cut_level_intensity = 150; // ??
                } else {
                    cut_level_area = 300/10; // ??
                    cut_level_intensity = 150/10; // ??
                }
            } else if ('K' == bandStr[0]) {
                if (sIdx != 16-1) {
                    cut_level_area = 200; // ??
                    cut_level_intensity = 100; // ??
                } else {
                    cut_level_area = 200/10; // ??
                    cut_level_intensity = 100/10; // ??
                }
            } else {
                SET_ERROR_MSG(CPL_ERROR_ILLEGAL_INPUT, "Band must be H, K or J");
            }

            cpl_table_unselect_all(tbl);
            cpl_table_or_selected_int(tbl, "slitlet", CPL_EQUAL_TO, sIdx);
            cpl_table_and_selected_double(tbl, "area", CPL_LESS_THAN, cut_level_area);
            cpl_table_erase_selected(tbl);

            cpl_table_unselect_all(tbl);
            cpl_table_or_selected_int(tbl, "slitlet", CPL_EQUAL_TO, sIdx);
            cpl_table_and_selected_double(tbl, "intensity", CPL_LESS_THAN, cut_level_intensity);
            cpl_table_erase_selected(tbl);

            // now get subtable width slitlet #sIdx
            cpl_table_unselect_all(tbl);
            cpl_table_or_selected_int(tbl, "slitlet", CPL_EQUAL_TO, sIdx);
            CHECK_ERROR_STATE();
            BRK_IF_NULL(
                tSlit = cpl_table_extract_selected(tbl));

            // get subtable of 1st column
            int index_first = cpl_table_get_int(tSlit, "index", 0, NULL);
            cpl_table_unselect_all(tSlit);
            cpl_table_or_selected_int(tSlit, "index", CPL_EQUAL_TO, index_first);
            CHECK_ERROR_STATE();
            BRK_IF_NULL(
                tCol = cpl_table_extract_selected(tSlit));

            // loop through all positions in 1st column and check if a position has been fitted mostly bad or not
            // > 50% bad fits --> reject
            for (int i = 0; i < cpl_table_get_nrow(tCol); i++) {
//cpl_msg_debug(cpl_func, "    pos: %d", i);
                int pos = cpl_table_get_int(tCol, "position", i, NULL);
                cpl_table_unselect_all(tSlit);
                cpl_table_or_selected_int(tSlit, "position", CPL_EQUAL_TO, pos);
                CHECK_ERROR_STATE();
                BRK_IF_NULL(
                    tPos = cpl_table_extract_selected(tSlit));
                int good = 0;
                if (cpl_table_get_nrow(tPos) == nr_cols) {
                    for (int j = 0; j < cpl_table_get_nrow(tPos); j++) {
                        int code = cpl_table_get_int(tPos, "errorcode", j, NULL);
                        CHECK_ERROR_STATE();
                        if (code == 0) {
                            good++;
                        }
                    }
                }
                if (good < cpl_table_get_nrow(tPos)/2) {
                    // reject position
                    cpl_table_unselect_all(tbl);
                    cpl_table_or_selected_int(tbl, "slitlet", CPL_EQUAL_TO, sIdx);
                    cpl_table_and_selected_int(tbl, "position", CPL_EQUAL_TO, pos);
                    cpl_table_erase_selected(tbl);
                    CHECK_ERROR_STATE();
                }

                eris_ifu_free_table(&tPos);
            }

            // check for errors on positions which exist only once, delete them
            eris_ifu_free_table(&tSlit);
            cpl_table_unselect_all(tbl);
            cpl_table_or_selected_int(tbl, "slitlet", CPL_EQUAL_TO, sIdx);
            CHECK_ERROR_STATE();
            BRK_IF_NULL(
                tSlit = cpl_table_extract_selected(tbl));

            cpl_table_unselect_all(tSlit);
            cpl_table_or_selected_int(tSlit, "errorcode", CPL_NOT_EQUAL_TO, 0);
            CHECK_ERROR_STATE();
            BRK_IF_NULL(
                tPos = cpl_table_extract_selected(tSlit));

            for (int i = 0; i < cpl_table_get_nrow(tPos); i++) {
                cpl_table_unselect_all(tSlit);
                int pos_bad = cpl_table_get_int(tPos, "position", i, NULL);
                cpl_table_or_selected_int(tSlit, "position", CPL_EQUAL_TO, pos_bad);
                if (cpl_table_count_selected(tSlit) == 1) {
                    cpl_table_unselect_all(tbl);
                    cpl_table_or_selected_int(tbl, "slitlet", CPL_EQUAL_TO, sIdx);
                    cpl_table_and_selected_int(tbl, "position", CPL_EQUAL_TO, pos_bad);
                    cpl_table_erase_selected(tbl);
                }
            }

            eris_ifu_free_table(&tPos);
            eris_ifu_free_table(&tSlit);
            eris_ifu_free_table(&tCol);
        }
        CHECK_ERROR_STATE();

    //    // now get rid of lines which have been bad at least once
    //        cpl_table_select_all(tbl);
    //        while (cpl_table_get_column_mean(tbl, ERIS_IFU_FITTABLE_ERRORCODE) < 0.) {
    //            // there is still some errorcode (line) to eliminate

    //            // search 1st error
    //            int ii = 0;
    //            int n = 0;
    //            cpl_boolean found = CPL_FALSE;
    //            while (ii < cpl_table_get_nrow(tbl) && !found) {
    //                if (cpl_table_is_selected(tbl, ii) &&
    //                    (cpl_table_get_int(tbl, ERIS_IFU_FITTABLE_ERRORCODE, ii, &n) == 0))
    //                {
    //                    ii++;
    //                } else {
    //                    found = CPL_TRUE;
    //                }
    //            }

    //            if (found) {
    //                double wavelength = cpl_table_get_double(tbl, ERIS_IFU_FITTABLE_WAVELENGTH, ii, &n);
    //                // mark the entries with this wavelength to be eliminated
    //                cpl_table_and_selected_double(tbl, ERIS_IFU_FITTABLE_WAVELENGTH, CPL_NOT_EQUAL_TO, wavelength);
    //            }

    //            cpl_table_not_selected(tbl);
    //            cpl_table_erase_selected(tbl);
    //            cpl_table_select_all(tbl);
    //        }

        // we have multiple columns per slitlet. Get rid of all except the center one
        for (int sIdx = 0; sIdx < SLITLET_CNT; sIdx++) {
            // get subtable width slitlet #sIdx
            cpl_table_unselect_all(tbl);
            cpl_table_or_selected_int(tbl, "slitlet", CPL_EQUAL_TO, sIdx);
            CHECK_ERROR_STATE();
            BRK_IF_NULL(
                tSlit = cpl_table_extract_selected(tbl));

            // get index of first column
            int index_first = cpl_table_get_int(tSlit, "index", 0, NULL);
            cpl_table_and_selected_int(tbl, "index", CPL_NOT_EQUAL_TO, index_first+nr_cols/2);
            cpl_table_erase_selected(tbl);
            CHECK_ERROR_STATE();

            eris_ifu_free_table(&tSlit);
        }
        CHECK_ERROR_STATE();

        // get rid of last errors
        cpl_table_unselect_all(tbl);
        cpl_table_or_selected_int(tbl, "errorcode", CPL_NOT_EQUAL_TO, 0);
        cpl_table_erase_selected(tbl);
        CHECK_ERROR_STATE();

        // delete unneeded columns
//            cpl_table_erase_column(tbl, "index");
        cpl_table_erase_column(tbl, "imgIdx");
//            cpl_table_erase_column(tbl, ERIS_IFU_FITTABLE_WAVELENGTH);
//            cpl_table_erase_column(tbl, ERIS_IFU_FITTABLE_ERRORCODE);
//            cpl_table_erase_column(tbl, "x0");
//            cpl_table_erase_column(tbl, "sigma");
//            cpl_table_erase_column(tbl, "area");
//            cpl_table_erase_column(tbl, "offset");
        cpl_table_erase_column(tbl, "mse");
        cpl_table_erase_column(tbl, "wavelengthFit");
        cpl_table_erase_column(tbl, "wavelengthError");
        cpl_table_erase_column(tbl, "range");
        cpl_table_erase_column(tbl, "xdata");
        cpl_table_erase_column(tbl, "ydata");
        CHECK_ERROR_STATE();
    }
    CATCH
    {
        CATCH_MSGS();
    }
    return err;
}

/**
 * @brief Remove duplicate or overlapping arc lines
 * @param tbl  Table of arc lines (modified in place)
 * @return CPL_ERROR_NONE on success, error code otherwise
 *
 * Removes lines with identical or very similar wavelengths (< 0.001 difference).
 * When duplicates found, removes both entries to avoid ambiguity.
 *
 * @note Processes each slitlet separately
 * @note Modifies table in place by erasing rows
 */
cpl_error_code eris_ifu_distortion_reduce_identical_lines(cpl_table *tbl)
{
    cpl_error_code  err     = CPL_ERROR_NONE;
    cpl_table       *tSlit  = NULL;

    TRY
    {
        cpl_msg_debug(cpl_func, "Removing identical/overlapping lines");
        for (int sIdx = 0; sIdx < SLITLET_CNT; sIdx++) {
            cpl_msg_debug(cpl_func, "  sIdx: %d", sIdx);

            // get subtable of slitlet
            cpl_table_unselect_all(tbl);
            cpl_table_or_selected_int(tbl, "slitlet", CPL_EQUAL_TO, sIdx);
            CHECK_ERROR_STATE();
            BRK_IF_NULL(
                tSlit = cpl_table_extract_selected(tbl));

            for (int i = 0; i < cpl_table_get_nrow(tSlit)-1; i++) {
                double wl1 = cpl_table_get_double(tSlit, "wavelength", i+1, NULL),
                       wl2 = cpl_table_get_double(tSlit, "wavelength", i, NULL);
                CHECK_ERROR_STATE();
                if (wl1-wl2 < 0.001) {
                    cpl_msg_debug(cpl_func, "Remove:");
                    cpl_msg_debug(cpl_func, "  wl: %g, x0: %g, sigma: %g, area: %g, offset: %g", wl1, cpl_table_get_double(tSlit, "x0", i+1, NULL), cpl_table_get_double(tSlit, "sigma", i+1, NULL), cpl_table_get_double(tSlit, "area", i+1, NULL), cpl_table_get_double(tSlit, "offset", i+1, NULL));
                    cpl_msg_debug(cpl_func, "  wl: %g, x0: %g, sigma: %g, area: %g, offset: %g", wl2, cpl_table_get_double(tSlit, "x0", i, NULL), cpl_table_get_double(tSlit, "sigma", i, NULL), cpl_table_get_double(tSlit, "area", i, NULL), cpl_table_get_double(tSlit, "offset", i, NULL));

                    int pos1 = cpl_table_get_int(tSlit, "position", i+1, NULL),
                        pos2 = cpl_table_get_int(tSlit, "position", i, NULL);
                    CHECK_ERROR_STATE();
                    cpl_table_and_selected_int(tbl, "position", CPL_EQUAL_TO, pos1);
                    cpl_table_erase_selected(tbl);
                    CHECK_ERROR_STATE();
                    cpl_table_unselect_all(tbl);
                    cpl_table_or_selected_int(tbl, "slitlet", CPL_EQUAL_TO, sIdx);
                    cpl_table_and_selected_int(tbl, "position", CPL_EQUAL_TO, pos2);
                    cpl_table_erase_selected(tbl);
                    CHECK_ERROR_STATE();
                }
                CHECK_ERROR_STATE();
            }

            eris_ifu_free_table(&tSlit);
        }

        CHECK_ERROR_STATE();
    }
    CATCH
    {
        CATCH_MSGS();
    }
    return err;
}

/**
 * @brief Perform wavelength calibration to identify valid arc lines
 * @param fs                Frameset with arc images and reference data
 * @param centers_fitted    Array of tables with fitted slitlet centers
 * @param productDepth      Debug output level
 * @param arcImgs           Output: array of arc lamp images
 * @param imgCnt            Output: number of images
 * @param parlist           Parameter list
 * @param qclog             Output: QC log table
 * @return Table of valid arc lines with positions, or NULL on error
 *
 * Performs simplified wavelength calibration to identify good arc lines:
 * 1. Loads arc lamp images and reference line lists
 * 2. Collapses spectra at slitlet centers
 * 3. Performs first-fit wavelength calibration
 * 4. Fits all lines and computes quality metrics
 * 5. Filters lines by intensity and fit quality
 * 6. Returns table with columns: slitlet, position, wavelength, fit parameters
 *
 * This provides the list of valid arc lines needed for distortion calibration.
 *
 * @note Returned table must be freed with cpl_table_delete()
 * @note Uses wavelength calibration to ensure only real arc lines are used
 */
cpl_table* eris_ifu_dist_wave(cpl_frameset *fs,
                              cpl_table **centers_fitted,
                              int productDepth,
                              cpl_image ***arcImgs,
                              int *imgCnt,
                              const cpl_parameterlist* parlist,
							  cpl_table** qclog)
{

    int                 corrMask    = LINE_EXPOSURE_CORRECTION | COLUMN_EXPOSURE_CORRECTION,
                        arcImgCnt   = 0,
                        *lampStates = NULL,
//                        nr_cols     = 3,
                        center      = 0;
    cpl_size            nRows       = 0;
    ifsBand             band        = UNDEFINED_BAND;
    ifsPreopticsScale   scale       = UNDEFINED_SCALE;
    ifsInstrument       instrument  = UNSET_INSTRUMENT;
    cpl_image     **dataImg   = NULL;
    const hdrl_image    *tmpImg     = NULL;
    hdrl_imagelist      *arcImages  = NULL;
    cpl_image **collapsedImgDetOrder = NULL;
    cpl_propertylist    *pl         = NULL;
    cpl_image           **collapsedSpectra      = NULL;
    cpl_vector          *collapsedSpectrum      = NULL,
                        *centerss               = NULL;
    cpl_bivector        **refLines              = NULL;
    const char          *refLineTableFileName   = NULL,
                        *firstFitTableFileName  = NULL,
                        *waveSetupFileName      = NULL;
    cpl_table           *firstFitTable          = NULL;
    cpl_polynomial      *firstFit               = NULL;
    cpl_polynomial *allFits[ERIS_IFU_DETECTOR_SIZE_Y];
    struct waveTablesStruct tables;
    struct waveSetupStruct  waveSetup;
    double saturation_threshold = 0;
    TRY
    {
        // make sure there is no nonsense in the table structure
        // in case it is written before filled properly
        eris_ifu_wave_clear_tables(&tables);

        for (int ix=0; ix<ERIS_IFU_DETECTOR_SIZE_Y; ix++) {
            allFits[ix] = NULL;
        }

        saturation_threshold = cpl_parameter_get_double(
        cpl_parameterlist_find_const(parlist, "eris.eris_ifu_distortion.pixel_saturation"));

        // preparations
        BRK_IF_ERROR(
            eris_ifu_wavecal_processSof_dist(
                fs,
                corrMask,
                &arcImgCnt,
                &arcImages,
                &lampStates,
                &band,
                &scale,
                &instrument,
				saturation_threshold,
				qclog));

        // allocations
        BRK_IF_NULL(
            refLines = cpl_calloc(arcImgCnt, sizeof(cpl_bivector *)));
        BRK_IF_NULL(
            collapsedSpectra = cpl_calloc(arcImgCnt, sizeof(cpl_image *)));
        BRK_IF_NULL(
            dataImg = cpl_calloc(arcImgCnt, sizeof(cpl_image *)));
        *imgCnt = arcImgCnt;

        // load arc images
        for (int i = 0; i < arcImgCnt; i++) {
            BRK_IF_NULL(
                tmpImg = hdrl_imagelist_get_const(arcImages, i));
            BRK_IF_NULL(
                dataImg[i] = cpl_image_duplicate(
                    hdrl_image_get_image_const(tmpImg)));
        }
        nRows = cpl_image_get_size_y(dataImg[0]);

        // load reference lines
        refLineTableFileName = cpl_frame_get_filename(
                                   cpl_frameset_find(fs, ERIS_IFU_CALIB_REF_LINES));
        for (int ix = 0; ix < arcImgCnt; ix++) {
            BRK_IF_NULL(
                refLines[ix] = eris_ifu_wave_get_refLines(
                                            refLineTableFileName, instrument,
                                            lampStates[ix]));
        }

        // prepare centers-vector
        // we need a vector of length SLITLET_CNT with one averaged center value
        centerss = cpl_vector_new(SLITLET_CNT);
        for (int sIdx = 0; sIdx < SLITLET_CNT; sIdx++) {
            // get a sensible center value (mean of the center in this slit)
            cpl_table *centers = centers_fitted[sIdx];
            double *dd = cpl_table_get_data_double(centers, "x");
            cpl_vector *vcen = cpl_vector_wrap (cpl_table_get_nrow(centers), dd);
            double ddd = cpl_vector_get_mean(vcen);
            cpl_vector_set(centerss,sIdx, ddd);
            cpl_vector_unwrap(vcen);
        }

        // prepare slitletStart/End-vectors
        // width from center add/subtract number of pixels
//        cpl_vector *vStart = cpl_vector_duplicate(centerss);
//        cpl_vector_subtract_scalar(vStart, 1.);
//        cpl_vector *vEnd = cpl_vector_duplicate(centerss);
//        cpl_vector_add_scalar(vEnd, 1.);
//        const double *slitletStart = cpl_vector_get_data_const(vStart);
//        const double *slitletEnd = cpl_vector_get_data_const(vEnd);

        //
        for (int aIdx = 0; aIdx < arcImgCnt; aIdx++) {
            for (int sIdx = 0; sIdx < SLITLET_CNT; sIdx++) {
                center = (int) cpl_vector_get (centerss, sIdx);
                // collapse along y at center-pos (in x)
                collapsedSpectrum = eris_ifu_wave_collapse_slitlet(
                                            dataImg[aIdx], center);

                eris_ifu_wave_save_spectrum(collapsedSpectra, aIdx,
                        collapsedSpectrum, sIdx, nRows,
                        lampStates[aIdx], band, instrument,
                        refLines[aIdx], productDepth);

                eris_ifu_free_vector(&collapsedSpectrum);
            }
        }

        // load first-fit-file
        firstFitTableFileName = cpl_frame_get_filename(
                                    cpl_frameset_find(fs, ERIS_IFU_CALIB_FIRST_FIT));
        if (firstFitTableFileName == NULL) {
            BRK_WITH_ERROR_MSG(CPL_ERROR_NULL_INPUT,
                "missing \"ERIS_IFU_CALIB_FIRST_FIT\" tag in the SOF, wavecal first fit table");
        }
        BRK_IF_NULL(
                firstFitTable = eris_ifu_wave_get_firstFitTable(
                                    firstFitTableFileName, instrument,  band));

        waveSetupFileName = cpl_frame_get_filename(
            cpl_frameset_find(fs, ERIS_IFU_CALIB_WAVE_SETUP));
        if (waveSetupFileName == NULL) {
            BRK_WITH_ERROR_MSG(CPL_ERROR_NULL_INPUT,
                "missing \"ERIS_IFU_CALIB_WAVE_SETUP\" tag in the SOF, wavecal setup table");
        }
        BRK_IF_ERROR(
            eris_ifu_wave_init_tables(&tables));
        BRK_IF_ERROR(
            eris_ifu_read_wave_setup(waveSetupFileName, band, &waveSetup));

        const int slitlet2collapsed[SLITLET_CNT] = {
                 0, 15,  1, 16,  2, 17,  3, 18,  4, 19,  5, 20,  6, 21,  7,
                30, 31,
                22,  8, 23,  9, 24, 10, 25, 11, 26, 12, 27, 13, 28, 14, 29};

        collapsedImgDetOrder =
                cpl_calloc(arcImgCnt, sizeof(cpl_image *));
        for (int aIdx = 0; aIdx < arcImgCnt; aIdx++) {
            cpl_size xSize = cpl_image_get_size_x(collapsedSpectra[aIdx]);
            cpl_size ySize = cpl_image_get_size_y(collapsedSpectra[aIdx]);
            collapsedImgDetOrder[aIdx] =
                    cpl_image_new(xSize, ySize, CPL_TYPE_DOUBLE);
            double *in = cpl_image_get_data_double(collapsedSpectra[aIdx]);
            double *out = cpl_image_get_data_double(collapsedImgDetOrder[aIdx]);
            for (int sx=0; sx < SLITLET_CNT; sx++) {
                int ix = slitlet2collapsed[sx];
                for (int iy = 0; iy < ySize; iy++){
                    out[sx+iy*xSize] = in[ix+iy*xSize];
                }
            }

        }

        int firstFitOffset;
        int firstFitMasterOffset = 0;
        BRK_IF_NULL(
            firstFit = eris_ifu_get_first_fit(collapsedSpectra,
                lampStates, arcImgCnt, 16, 0, &firstFitMasterOffset,
                waveSetup, firstFitTable,
                tables.slitletFitting, tables.slitletCoeff));
        cpl_msg_info(__func__,"FirstFit Master Offset %d", firstFitMasterOffset);
        eris_ifu_free_polynomial(&firstFit);
        for (int sIdx = 0; sIdx < SLITLET_CNT; sIdx++) {
                BRK_IF_NULL(
                    firstFit = eris_ifu_get_first_fit(collapsedSpectra,
                        lampStates, arcImgCnt, sIdx, firstFitMasterOffset, &firstFitOffset,
                        waveSetup, firstFitTable,
                        tables.slitletFitting, tables.slitletCoeff));

//                eris_ifu_fit_all_lines(REC_NAME_DISTORTION, sIdx, slitletStart, slitletEnd,
//                        arcImgCnt, dataImg, refLines,
//                        waveSetup, firstFit, allFits,
//                        tables.columnFitting, tables.columnCoeffRaw,
//                        tables.columnCoeffSmoothed, tables.smoothingCoeff);
                const double column[] = {0.,1.,2.,3.,4.,5.,6.,7.,8.,9.,
                        10.,11.,12.,13.,14.,15.,16.,17.,18.,19.,
                        20.,21.,22.,23.,24.,25.,26.,27.,28.,29.,30.,31,};
                BRK_IF_ERROR(
                    eris_ifu_fit_all_lines(REC_NAME_DISTORTION, sIdx, column, column,
                         arcImgCnt, (const cpl_image **) collapsedImgDetOrder, refLines,
                         waveSetup, firstFit, allFits,
                         tables.columnFitting, tables.columnCoeffRaw,
                         tables.columnCoeffSmoothed, tables.smoothingCoeff));
                CHECK_ERROR_STATE();
                cpl_table_unselect_all(tables.columnFitting);
                cpl_table_or_selected_int(tables.columnFitting, ERIS_IFU_FITTABLE_SLITLET, CPL_EQUAL_TO, sIdx);
                for (int rx = 0; rx < cpl_table_get_nrow(tables.columnFitting); rx++) {
                    if (cpl_table_is_selected(tables.columnFitting, rx) == 1) {
                        cpl_table_set_int(tables.columnFitting, ERIS_IFU_FITTABLE_INDEX, rx, (int) cpl_vector_get (centerss, sIdx));
                    }
                }

                eris_ifu_free_polynomial(&firstFit);
        }
        if (productDepth > 3 ) {
            BRK_IF_ERROR(
                eris_ifu_wave_save_fitting_tables(&tables, instrument,  band,
						  waveSetup, fs, parlist,
						  "eris_ifu_distortion"));
        }
        cpl_table_unselect_all(tables.columnFitting);
        cpl_table_or_selected_int(tables.columnFitting, ERIS_IFU_FITTABLE_ERRORCODE, CPL_NOT_EQUAL_TO, 0);
        BRK_IF_ERROR(cpl_table_erase_selected(tables.columnFitting));

        // add intensity of lines
        cpl_table_unselect_all(tables.columnFitting);
        cpl_table_new_column(tables.columnFitting, "intensity", CPL_TYPE_DOUBLE);
        for (int i = 0; i < cpl_table_get_nrow(tables.columnFitting); i++) {
            double sigma = cpl_table_get_double(tables.columnFitting, "sigma", i, NULL),
                   area = cpl_table_get_double(tables.columnFitting, "area", i, NULL),
                   offset = cpl_table_get_double(tables.columnFitting, "offset", i, NULL);

            double inten = area / sqrt(2.*(double)CX_PI*pow(sigma,2)) + offset;
            if ((inten - offset) > 20. && sigma < 2.) {
//            if ((inten - offset) > 20.) {
                cpl_table_set_double(tables.columnFitting, "intensity", i, inten);
            } else {
                //printf("Deleted due less intensity:  %7.1f %f\n",  offset, inten);
                cpl_table_or_selected_window(tables.columnFitting, i, 1);
            }
            CHECK_ERROR_STATE();
        }
        BRK_IF_ERROR(cpl_table_erase_selected(tables.columnFitting));

//erw        BRK_IF_ERROR(
//            eris_ifu_distortion_reduce_lines(tables.columnFitting, band, nr_cols));

        // sort positions
        BRK_IF_NULL(
            pl =  cpl_propertylist_new());
        BRK_IF_ERROR(
            cpl_propertylist_append_bool(pl, "position", CPL_TRUE));
        BRK_IF_ERROR(
            cpl_table_sort(tables.columnFitting, pl));

//erw        BRK_IF_ERROR(
//            eris_ifu_distortion_reduce_identical_lines(tables.columnFitting));

        if ((productDepth & 3) != 0) {
            int c = 0;
            for (int sIdx = 0; sIdx < SLITLET_CNT; sIdx++) {
                cpl_table_unselect_all(tables.columnFitting);
                cpl_table_or_selected_int(tables.columnFitting, ERIS_IFU_FITTABLE_SLITLET, CPL_EQUAL_TO, sIdx);
                cpl_table *t = cpl_table_extract_selected(tables.columnFitting);
                if (sIdx == 0) {
                    c = CPL_IO_CREATE;
                } else {
                    c = CPL_IO_EXTEND;
                }
                BRK_IF_ERROR(
                    cpl_table_save(t, NULL, NULL, "eris_ifu_distortion_dbg_wave_fit_lines.fits", c));
                eris_ifu_free_table(&t);
            }
        }
    }
    CATCH
    {
//        eris_ifu_free_table(&tbl);
    }
    cpl_free(lampStates);
    eris_ifu_free_vector(&centerss);
    eris_ifu_free_propertylist(&pl);
    eris_ifu_free_hdrl_imagelist(&arcImages);
    for (int ix=0; ix<arcImgCnt; ix++) {
        eris_ifu_free_bivector(&refLines[ix]);
        eris_ifu_free_image(&collapsedSpectra[ix]);
        eris_ifu_free_image(&collapsedImgDetOrder[ix]);
    }
    cpl_free(refLines);
    cpl_free(collapsedSpectra);
    cpl_free(collapsedImgDetOrder);
    for (int ix=0; ix<ERIS_IFU_DETECTOR_SIZE_Y; ix++) {
        eris_ifu_free_polynomial(&allFits[ix]);
    }
    eris_ifu_free_table(&firstFitTable);
    eris_ifu_free_table(&tables.slitletFitting);
    eris_ifu_free_table(&tables.slitletCoeff);
    eris_ifu_free_table(&tables.columnCoeffRaw);
    eris_ifu_free_table(&tables.columnCoeffSmoothed);
    eris_ifu_free_table(&tables.smoothingCoeff);
    *arcImgs = dataImg;
    return tables.columnFitting;
}
